!-----------------------------------------------------------------------
!     Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!-----------------------------------------------------------------------

module PV_analy
! 渦位解析に関するモジュール
  use Algebra
  use Derivation

contains

subroutine Ertel_PV( x, y, z, u, v, w, rho, pt, cor, pv, undeff, sx, sy, sz, cord )
! エルテルのポテンシャル渦度を計算する
  implicit none
  real, intent(in) :: x(:)  ! x 方向の座標変数 [m]
  real, intent(in) :: y(:)  ! y 方向の座標変数 [m]
  real, intent(in) :: z(:)  ! z 方向の座標変数 [m]
  real, intent(in) :: u(size(x),size(y),size(z))  ! 速度場の x 成分 [m/s]
  real, intent(in) :: v(size(x),size(y),size(z))  ! 速度場の y 成分 [m/s]
  real, intent(in) :: w(size(x),size(y),size(z))  ! 速度場の z 成分 [m/s]
  real, intent(in) :: rho(size(z))  ! 密度場 [kg/m^3]
  real, intent(in) :: pt(size(x),size(y),size(z))  ! 温位場 [K]
  real, intent(in) :: cor(size(x),size(y))  ! コリオリパラメータ [/s]
  real, intent(inout) :: pv(size(x),size(y),size(z))  ! PV [Km^2/kgs]
  real, intent(in), optional :: undeff
  real, intent(in), optional :: sx(size(x),size(y),size(z))  ! スケーリングファクター
  real, intent(in), optional :: sy(size(x),size(y),size(z))  ! スケーリングファクター
  real, intent(in), optional :: sz(size(x),size(y),size(z))  ! スケーリングファクター
  character(1), intent(in), optional :: cord   ! 鉛直座標系
                              ! 'z', 'p' = 幾何高度 [m], 圧力 [Pa]
                              ! デフォルト = 'z'
  real :: tmp1(size(x),size(y),size(z))
  real :: tmp2(size(x),size(y),size(z))
  real :: tmp3(size(x),size(y),size(z))
  real :: tmp4(size(x),size(y),size(z))
  real :: tmp5(size(x),size(y),size(z))
  real :: tmp6(size(x),size(y),size(z))
  real :: tmp7(size(x),size(y),size(z))
  integer :: i, j, k
  integer :: nx  ! 第 1 要素数
  integer :: ny  ! 第 2 要素数
  integer :: nz  ! 第 3 要素数
  real :: scalex(size(x),size(y),size(z))
  real :: scaley(size(x),size(y),size(z))
  real :: scalez(size(x),size(y),size(z))
  character(1) :: zp_flag

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(present(sx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=sx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(sy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=sy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(sz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=sz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(cord))then
     zp_flag=cord
  else
     zp_flag='z'
  end if

  if(present(undeff))then
!  温位の空間勾配を計算.
     call grad_3d( x, y, z, pt, tmp1, tmp2, tmp3, undeff,  &
  &                    scalex, scaley, scalez )
!  3 次元 rotation を計算.
     call curl_3d( x, y, z, u, v, w, tmp4, tmp5, tmp6, undeff,  & 
  &               scalex, scaley, scalez )
!  omega と grad pt の内積を計算
     call dot_prod_3d( tmp4, tmp5, tmp6, tmp1, tmp2, tmp3, tmp7, undeff )
  else
!  温位の空間勾配を計算.
     call grad_3d( x, y, z, pt, tmp1, tmp2, tmp3,  &
  &                    hx=scalex, hy=scaley, hz=scalez )
!  3 次元 rotation を計算.
     call curl_3d( x, y, z, u, v, w, tmp4, tmp5, tmp6,  & 
  &               hx=scalex, hy=scaley, hz=scalez )
!  omega と grad pt の内積を計算
     call dot_prod_3d( tmp4, tmp5, tmp6, tmp1, tmp2, tmp3, tmp7 )
  end if

  if(present(undeff))then
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)

     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(tmp3(i,j,k)==undeff.or.tmp7(i,j,k)==undeff.or.  &
  &              rho(k)==undeff)then
                 pv(i,j,k)=undeff
              else
                 pv(i,j,k)=(tmp7(i,j,k)+cor(i,j)*tmp3(i,j,k))/rho(k)
                 ! 計算するのは絶対渦度なので, コリオリを足し合わせ.
              end if
           end do
        end do
     end do

!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)

     do k=1,nz
        do j=1,ny
           do i=1,nx
              pv(i,j,k)=(tmp7(i,j,k)+cor(i,j)*tmp3(i,j,k))/rho(k)
           end do
        end do
     end do

!$omp end do
!$omp end parallel

  end if

end subroutine

!--------------------------------------------------------
!--------------------------------------------------------

subroutine HEPV( x, y, z, u, v, rhoc, pt, cor, pv, undeff, sx, sy )
! 静力学近似が成り立つ系での, 任意の鉛直座標系における
! エルテルのポテンシャル渦度を計算する.
  implicit none
  real, intent(in) :: x(:)  ! x 方向の座標変数 [MKS unit]
  real, intent(in) :: y(:)  ! y 方向の座標変数 [MKS unit]
  real, intent(in) :: z(:)  ! z 方向の座標変数 [MKS unit]
  real, intent(in) :: u(size(x),size(y),size(z))  ! 速度場の x 成分 [m/s]
  real, intent(in) :: v(size(x),size(y),size(z))  ! 速度場の y 成分 [m/s]
  real, intent(in) :: rhoc(size(x),size(y),size(z))  !重み付き密度 [MKS unit]
  real, intent(in) :: pt(size(x),size(y),size(z))  ! スカラー保存量
  real, intent(in) :: cor(size(x),size(y))  ! コリオリパラメータ [/s]
  real, intent(inout) :: pv(size(x),size(y),size(z))  ! PV [Km^2/kgs]
  real, intent(in), optional :: undeff
  real, intent(in), optional :: sx(size(x),size(y))  ! スケーリングファクター
  real, intent(in), optional :: sy(size(x),size(y))  ! スケーリングファクター
  real :: tmp1(size(x),size(y),size(z))
  real :: tmp2(size(x),size(y),size(z))
  real :: tmp3(size(x),size(y),size(z))
  real :: tmp4(size(x),size(y),size(z))
  real :: tmp5(size(x),size(y),size(z))
  real :: tmp6(size(x),size(y),size(z))
  real :: tmp7(size(x),size(y),size(z))
  integer :: i, j, k
  integer :: nx  ! 第 1 要素数
  integer :: ny  ! 第 2 要素数
  integer :: nz  ! 第 3 要素数
  real :: scalex(size(x),size(y))
  real :: scaley(size(x),size(y))

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(present(sx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=sx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0
        end do
     end do
  end if

  if(present(sy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=sy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0
        end do
     end do
  end if

  if(present(undeff))then
  !  スカラー保存量の空間勾配, rotation を計算.
!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        call grad_2d( x, y, pt(:,:,k), tmp1(:,:,k), tmp2(:,:,k), undeff,  &
  &                   hx=scalex, hy=scaley )
        call curl( x, y, u(:,:,k), v(:,:,k), tmp6(:,:,k),  &
  &                undeff, hx=scalex, hy=scaley )
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)
     do j=1,ny
        do i=1,nx
           call grad_1d( z, pt(i,j,:), tmp3(i,j,:), undeff )
           call grad_1d( z, v(i,j,:), tmp4(i,j,:), undeff )
           call grad_1d( z, u(i,j,:), tmp5(i,j,:), undeff )
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(tmp4(i,j,k)/=undeff)then
                 tmp4(i,j,k)=-tmp4(i,j,k)
              end if
              if(tmp6(i,j,k)/=undeff)then
                 tmp6(i,j,k)=tmp6(i,j,k)+cor(i,j)
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  !  omega と grad pt の内積を計算
     call dot_prod_3d( tmp4, tmp5, tmp6, tmp1, tmp2, tmp3, tmp7, undeff )

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)

     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(tmp4(i,j,k)==undeff.or.rhoc(i,j,k)==undeff)then
                 pv(i,j,k)=undeff
              else
                 pv(i,j,k)=tmp7(i,j,k)*rhoc(i,j,k)
              end if
           end do
        end do
     end do

!$omp end do
!$omp end parallel

  else

  !  スカラー保存量の空間勾配, rotation を計算.
!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        call grad_2d( x, y, pt(:,:,k), tmp1(:,:,k), tmp2(:,:,k),  &
  &                   hx=scalex, hy=scaley )
        call curl( x, y, u(:,:,k), v(:,:,k), tmp6(:,:,k),  &
  &                undeff, hx=scalex, hy=scaley )
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)
     do j=1,ny
        do i=1,nx
           call grad_1d( z, pt(i,j,:), tmp3(i,j,:) )
           call grad_1d( z, v(i,j,:), tmp4(i,j,:) )
           call grad_1d( z, u(i,j,:), tmp5(i,j,:) )
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              tmp4(i,j,k)=-tmp4(i,j,k)
              tmp6(i,j,k)=tmp6(i,j,k)+cor(i,j)
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  !  omega と grad pt の内積を計算
     call dot_prod_3d( tmp4, tmp5, tmp6, tmp1, tmp2, tmp3, tmp7 )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)

     do k=1,nz
        do j=1,ny
           do i=1,nx
              pv(i,j,k)=tmp7(i,j,k)*rhoc(i,j,k)
           end do
        end do
     end do

!$omp end do
!$omp end parallel

  end if

end subroutine

!--------------------------------------------------------
!--------------------------------------------------------

subroutine HQGPV( x, y, z, phi, t_ref, cor, qgpv, undef, hx, hy, rhoc )
  ! Holton (2004), Holton and Hakim (2012) より QG 系での PV を計算する.
  use Thermo_Const
  use Phys_Const
  use Math_Const

  implicit none

  real, intent(in) :: x(:)  ! x 方向の座標変数 [MKS unit]
  real, intent(in) :: y(:)  ! y 方向の座標変数 [MKS unit]
  real, intent(in) :: z(:)  ! z 方向の座標変数 [MKS unit] (Z or P)
  real, intent(in) :: t_ref(size(z))   ! (potential) temperature for basic [K]
                      ! p-coord : temp, z-coord : potential temp.
  real, intent(in) :: cor(size(x),size(y))  ! coriolis parameter [s-1]
  real, intent(in) :: phi(size(x),size(y),size(z))
                      ! p-coord : geopotential [J/kg], z-coord : pressure [Pa].
  real, intent(inout) :: qgpv(size(x),size(y),size(z))   ! QGPV [s-1]
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x),size(y))  ! スケーリングファクター
  real, intent(in), optional :: hy(size(x),size(y))  ! スケーリングファクター
  real, intent(in), optional :: rhoc(size(z))  ! basic den. for z-coord. [kg/m3]

  integer :: nx, ny, nz, i, j, k
  real :: kp, p_inv, fc, undeff
  real, dimension(size(x),size(y)) :: sx, sy
  real, dimension(size(x),size(y),size(z)) :: tmp
  real, dimension(size(z)) :: sigma, dsdp, dtdp, d2tdp2, tmpz
  logical :: z_flag

  nx=size(x)
  ny=size(y)
  nz=size(z)

  kp=Rd/Cpd

  if(present(rhoc))then
     z_flag=.true.
  else
     z_flag=.false.
  end if

  if(present(undef))then
     undeff=undef
  else
     undeff=-999.0
  end if

  if(present(hx).and.present(hy))then
     sx=hx
     sy=hy
  else
     sx=1.0
     sy=1.0
  end if

  fc=cor(nx/2,ny/2)

  !-- calculating vertical stability
  call grad_1d( z, t_ref, dtdp )
  call laplacian_1d( z, t_ref, d2tdp2 )

  if(z_flag.eqv..true.)then  ! -> Z-coord.

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        sigma(k)=g*dtdp(k)/t_ref(k)
        tmpz(k)=1.0/(rhoc(k)*sigma(k))
     end do
!$omp end do
!$omp end parallel

     call grad_1d( z, tmpz, dsdp )

  else  ! -> P-coord.

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        sigma(k)=(Rd/z(k))*(kp*t_ref(k)/z(k)-dtdp(k))
        dsdp(k)=(-sigma(k)+Rd*(kp*(dtdp(k)-t_ref(k)/z(k))/z(k)-d2tdp2(k)))/z(k)
     end do
!$omp end do
!$omp end parallel

  end if

  !-- calculating derivation for longitude

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j)

  do k=1,nz
     call laplacian_2d( x, y, phi(:,:,k), tmp(:,:,k), undef=undeff,  &
  &                     hx=sx, hy=sy )
  end do

!$omp end do

!$omp barrier

!$omp do schedule(runtime) private(i,j,k)

  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(z_flag.eqv..true.)then
              qgpv(i,j,k)=tmp(i,j,k)/(fc*rhoc(k))
           else
              qgpv(i,j,k)=tmp(i,j,k)/(fc)
           end if
        end do
     end do
  end do

!$omp end do

  !-- calculating derivation for altitude

!$omp barrier

!$omp do schedule(runtime) private(i,j)

  do j=1,ny
     do i=1,nx
        call laplacian_1d( z, phi(i,j,:), tmp(i,j,:) )
     end do
  end do

!$omp end do

!$omp barrier

!$omp do schedule(runtime) private(i,j,k)

  do j=1,ny
     do i=1,nx
        do k=1,nz
           if(z_flag.eqv..true.)then
              qgpv(i,j,k)=qgpv(i,j,k)+tmp(i,j,k)*fc/(rhoc(k)*sigma(k))
           else
              qgpv(i,j,k)=qgpv(i,j,k)+tmp(i,j,k)*fc/sigma(k)
           end if
        end do
     end do
  end do

!$omp end do

!$omp barrier

!$omp do schedule(runtime) private(i,j)

  do j=1,ny
     do i=1,nx
        call grad_1d( z, phi(i,j,:), tmp(i,j,:) )
     end do
  end do

!$omp end do

!$omp barrier

!$omp do schedule(runtime) private(i,j,k)

  do j=1,ny
     do i=1,nx
        do k=1,nz
           if(z_flag.eqv..true.)then
              qgpv(i,j,k)=qgpv(i,j,k)+tmp(i,j,k)*fc*dsdp(k)
           else
              qgpv(i,j,k)=qgpv(i,j,k)-tmp(i,j,k)*fc*dsdp(k)/(sigma(k)*sigma(k))
           end if
        end do
     end do
  end do

!$omp end do

!$omp barrier

!$omp do schedule(runtime) private(i,j,k)

  do k=1,nz
     do j=1,ny
        do i=1,nx
           qgpv(i,j,k)=qgpv(i,j,k)+cor(i,j)
        end do     
     end do     
  end do

!$omp end do
!$omp end parallel

end subroutine HQGPV

!--------------------------------------------------------
!-- 以降は原式の再チェックとコーディングの最適化必要
!-- マニュアル化はまだ.
!-- 関数としては利用してはいけない.
!--------------------------------------------------------

!subroutine QG_PV( lon, lat, p, phi, t_ref, qgpv )
!  ! Yoshino et al. (2003) : modified for (2).
!  use Thermo_Const
!  use Phys_Const
!  use Math_Const
!
!  implicit none
!
!  real, intent(in) :: lon(:)  ! longitude [rad]
!  real, intent(in) :: lat(:)  ! latitude [rad]
!  real, intent(in) :: p(:)    ! pressure [Pa]
!  real, intent(in) :: t_ref(size(p))   ! horizontal averaged temperature [K]
!  real, intent(in) :: phi(size(lon),size(lat),size(p))
!                              ! geopotential [J/kg]
!  real, intent(inout) :: qgpv(size(lon),size(lat),size(p))   ! QGPV [s-1]
!
!  integer :: nx, ny, nz, i, j, k
!  real :: kp, p_inv, fc
!  real, dimension(size(lon),size(lat),size(p)) :: tmp
!  real, dimension(size(lat)) :: f0
!  real, dimension(size(p)) :: sigma, dsdp, dtdp, d2tdp2
!
!  nx=size(lon)
!  ny=size(lat)
!  nz=size(p)
!
!  kp=Rd/Cpd
!  qgpv=0.0
!
!  !-- calculating coriolis parameter
!  do j=1,ny
!     f0(j)=2.0*omega*sin(lat(j))
!  end do
!  fc=f0(ny/2)
!
!  !-- calculating vertical stability
!  call grad_1d( p, t_ref, dtdp )
!  call sub_lapsi_1d( p, t_ref, d2tdp2 )
!
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(k,p_inv)
!
!  do k=1,nz
!     p_inv=1.0/p(k)
!     sigma(k)=(Rd*p_inv)*(kp*t_ref(k)*p_inv-dtdp(k))
!     dsdp(k)=(-sigma(k)+Rd*(kp*p_inv*(dtdp(k)-t_ref(k)*p_inv)-d2tdp2(k)))*p_inv
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!  !-- calculating derivation for longitude
!
!!$omp do schedule(runtime) private(j,k)
!
!  do k=1,nz
!     do j=1,ny
!        call sub_lapsi_1d( lon, phi(:,j,k), tmp(:,j,k) )
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j,k)
!
!  do k=1,nz
!     do j=1,ny
!        do i=1,nx
!           qgpv(i,j,k)=qgpv(i,j,k)+tmp(i,j,k)/(fc*(radius*cos(lat(j)))**2)
!        end do
!     end do
!  end do
!
!!$omp end do
!
!  !-- calculating derivation for latitude
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,k)
!
!  do k=1,nz
!     do i=1,nx
!        call sub_lapsi_1d( lat, phi(i,:,k), tmp(i,:,k) )
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j,k)
!
!  do k=1,nz
!     do i=1,nx
!        do j=1,ny
!           qgpv(i,j,k)=qgpv(i,j,k)+tmp(i,j,k)/(fc*(radius**2))
!        end do
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,k)
!
!  do k=1,nz
!     do i=1,nx
!        call grad_1d( lat, phi(i,:,k), tmp(i,:,k) )
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j,k)
!
!  do k=1,nz
!     do j=1,ny
!        do i=1,nx
!           qgpv(i,j,k)=qgpv(i,j,k)-tmp(i,j,k)*tan(lat(j))/(fc*(radius**2))
!        end do
!     end do
!  end do
!
!!$omp end do
!
!  !-- calculating derivation for altitude
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j)
!
!  do j=1,ny
!     do i=1,nx
!        call sub_lapsi_1d( p, phi(i,j,:), tmp(i,j,:) )
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j,k)
!
!  do j=1,ny
!     do i=1,nx
!        do k=1,nz
!           qgpv(i,j,k)=qgpv(i,j,k)+tmp(i,j,k)*fc/sigma(k)
!        end do
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j)
!
!  do j=1,ny
!     do i=1,nx
!        call grad_1d( p, phi(i,j,:), tmp(i,j,:) )
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j,k)
!
!  do j=1,ny
!     do i=1,nx
!        do k=1,nz
!           qgpv(i,j,k)=qgpv(i,j,k)-tmp(i,j,k)*fc*dsdp(k)/(sigma(k)*sigma(k))
!        end do
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j,k)
!
!  do k=1,nz
!     do j=1,ny
!        do i=1,nx
!           qgpv(i,j,k)=qgpv(i,j,k)+f0(j)
!        end do     
!     end do     
!  end do
!
!!$omp end do
!!$omp end parallel
!
!end subroutine QG_PV
!
!!------------------------------------
!!------------------------------------
!
!subroutine E_PV( lon, lat, p, u, v, pt, epv )
!  ! Yoshino et al. (2003) : modified for (2).
!  ! On normalized pressure coordinate, Ertel's PV.
!  use Thermo_Const
!  use Phys_Const
!  use Math_Const
!
!  implicit none
!
!  real, intent(in) :: lon(:)  ! longitude [rad]
!  real, intent(in) :: lat(:)  ! latitude [rad]
!  real, intent(in) :: p(:)    ! pressure [Pa]
!  real, intent(in) :: u(size(lon),size(lat),size(p))   ! zonal wind [ms-1]
!  real, intent(in) :: v(size(lon),size(lat),size(p))   ! meridional wind [ms-1]
!  real, intent(in) :: pt(size(lon),size(lat),size(p))  ! potential temp. [K]
!  real, intent(inout) :: epv(size(lon),size(lat),size(p))   ! EPV [10^{-6}PVU]
!
!  integer :: nx, ny, nz, i, j, k
!  real :: kp
!  real, dimension(size(p)) :: pex
!  real, dimension(size(lat)) :: f0
!  real, dimension(size(lon),size(lat)) :: sx, sy
!  real, dimension(size(lon),size(lat),size(p)) :: dptdp, dudp, dvdp, zeta
!  real, dimension(size(lon),size(lat),size(p)) :: dptdlon, dptdlat
!
!  nx=size(lon)
!  ny=size(lat)
!  nz=size(p)
!
!  kp=Rd/Cpd
!  epv=0.0
!  pex=(p/p0)**kp
!
!  !-- calculating coriolis parameter
!  do j=1,ny
!     f0(j)=2.0*omega*sin(lat(j))
!     sx(:,j)=radius*cos(lat(j))
!  end do
!  sy=radius
!
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(i,j)
!
!  do j=1,ny
!     do i=1,nx
!        call grad_1d( pex, pt(i,j,:), dptdp(i,j,:) )
!        call grad_1d( pex, u(i,j,:), dudp(i,j,:) )
!        call grad_1d( pex, v(i,j,:), dvdp(i,j,:) )
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(k)
!
!  do k=1,nz
!     call curl( lon, lat, u(:,:,k), v(:,:,k), zeta(:,:,k), hx=sx, hy=sy )
!     call grad_2d( lon, lat, pt(:,:,k),  &
!  &                dptdlon(:,:,k), dptdlat(:,:,k), hx=sx, hy=sy )
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j,k)
!
!  do k=1,nz
!     do j=1,ny
!        do i=1,nx
!           epv(i,j,k)=-(g*kp*pex(k)/p(k))  &
!  &                   *((f0(j)+zeta(i,j,k))*dptdp(i,j,k)  &
!  &                     -(dptdlon(i,j,k)*dvdp(i,j,k)  &
!  &                      -dptdlat(i,j,k)*dudp(i,j,k)))
!        end do     
!     end do     
!  end do
!
!!$omp end do
!!$omp end parallel
!
!end subroutine E_PV
!
!!------------------------------------
!!------------------------------------
!
!subroutine QGPV_inv( x, y, z, t_ref, eps, qgpv, coord, cor, phi,  &
!  &                  t, ug, vg, psi, ini_flag, rhoc )
!  ! Yoshino et al. (2003) : modified for (2).
!  use Thermo_Const
!  use Phys_Const
!  use Math_Const
!
!  implicit none
!
!  real, intent(in) :: x(:)    ! x-coord [rad or m]
!  real, intent(in) :: y(:)    ! y-coord [rad or m]
!  real, intent(in) :: z(:)    ! z-coord [Pa or m]
!  real, intent(in) :: t_ref(size(z))   ! P-coord : basic temp. [K]
!                              ! Z-coord : basic potential temp. [K]
!  real, intent(in) :: eps     ! converging value
!  real, intent(in) :: qgpv(size(x),size(y),size(z))   ! QGPV-f [s-1]
!  character(2), intent(in) :: coord   ! 'xy' = Cartecian, 'll' = lon-lat.
!  real, intent(in) :: cor(size(x),size(y))   ! coriolis parameter [s-1]
!  real, intent(inout) :: phi(size(x),size(y),size(z))
!                              ! P-coord : geopotential [J/kg]
!                              ! Z-coord : pressure [Pa]
!  real, intent(inout) :: t(size(x),size(y),size(z))   ! temperature [K]
!  real, intent(inout) :: ug(size(x),size(y),size(z))  ! geostrophy-x [ms-1]
!  real, intent(inout) :: vg(size(x),size(y),size(z))  ! geostrophy-y [ms-1]
!  real, intent(inout) :: psi(size(x),size(y),size(z))  ! stream func. [Js/kg]
!  logical, intent(in), optional :: ini_flag   ! .true. = initializing, default = .true.
!  real, intent(in), optional :: rhoc(size(z))  ! basic den. for z-coord. [kg/m3]
!
!  integer :: nx, ny, nz, i, j, k, counter
!  real :: kp, errmax, fc, radinv, rad2, rad2inv
!  real, dimension(size(x),size(y)) :: sx, sy
!  real, dimension(size(z)) :: dtdp, d2tdp2, sigma, dsdp, tmpz
!  real, dimension(size(x),size(y),size(z)) :: a, b, c, d, e, rho, bo, tmp
!  logical :: z_flag
!
!  nx=size(x)
!  ny=size(y)
!  nz=size(z)
!
!  if(coord(1:2)=='ll')then
!     rad2=radius*radius
!     radinv=1.0/radius
!     rad2inv=radinv*radinv
!     do j=1,ny
!        do i=1,nx
!           sx(i,j)=radius*cos(y(j))
!           sy(i,j)=radius
!        end do
!     end do
!  else
!     rad2=1.0
!     radinv=1.0
!     rad2inv=1.0
!     sx=1.0
!     sy=1.0
!  end if
!
!  kp=Rd/Cpd
!  bo=0.0
!  tmp=0.0
!
!  !-- setting lateral boundary conditions
!  do k=1,nz
!     do j=1,ny
!        do i=1,nx
!           if(i==1.or.i==nx.or.j==1.or.j==ny.or.k==1.or.k==nz)then
!              bo(i,j,k)=phi(i,j,k)
!           else
!              bo(i,j,k)=0.0
!           end if
!        end do
!     end do
!  end do
!
!  !-- setting vertical boundary condition
!  do j=2,ny-1
!     do i=2,nx-1
!        if(z_flag.eqv..true.)then
!           bo(i,j,1)=(rhoc(1)/t_ref(1))*g*t(i,j,1)
!           bo(i,j,nz)=(rhoc(nz)/t_ref(nz))*g*t(i,j,nz)
!        else
!           bo(i,j,1)=-Rd*t(i,j,1)/z(1)
!           bo(i,j,nz)=-Rd*t(i,j,nz)/z(nz)
!        end if
!!        bo(i,j,1)=t(i,j,1)
!!        bo(i,j,nz)=t(i,j,nz)
!     end do
!  end do
!
!  if(present(ini_flag))then
!     if(ini_flag.eqv..true.)then
!        phi=0.0
!        psi=0.0
!     end if
!  else
!     phi=0.0
!     psi=0.0
!  end if
!
!  fc=cor(nx/2,ny/2)
!
!  !-- calculating vertical stability
!  call grad_1d( z, t_ref, dtdp )
!  call sub_lapsi_1d( z, t_ref, d2tdp2 )
!
!  !-- calculating vertical coefficients for Ellip_Slv
!  if(z_flag.eqv..true.)then  ! Z-coord.
!     do k=1,nz
!        sigma(k)=g*dtdp(k)/t_ref(k)
!        tmpz(k)=1.0/(rhoc(k)*sigma(k))
!     end do
!
!     call grad_1d( z, tmpz, dsdp )
!
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(i,j,k)
!     do k=1,nz
!        do j=1,ny
!           do i=1,nx
!              d(i,j,k)=(fc**2)/(rhoc(k)*sigma(k))
!              e(i,j,k)=(fc**2)*dsdp(k)
!           end do
!        end do
!     end do
!!$omp end do
!!$omp end parallel
!
!  else  ! P-coord.
!
!     do k=1,nz
!        sigma(k)=(Rd/z(k))*(kp*t_ref(k)/z(k)-dtdp(k))
!        dsdp(k)=(-sigma(k)+Rd*(kp*(dtdp(k)-t_ref(k)/z(k))/z(k)-d2tdp2(k)))/z(k)
!     end do
!
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(i,j,k)
!     do k=1,nz
!        do j=1,ny
!           do i=1,nx
!              d(i,j,k)=(fc**2)/sigma(k)
!              e(i,j,k)=-d(i,j,k)*dsdp(k)/sigma(k)
!           end do
!        end do
!     end do
!!$omp end do
!!$omp end parallel
!
!  end if
!
!  if(coord(1:2)=='ll')then
!
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(i,j,k)
!     do k=1,nz
!        do j=1,ny
!           do i=1,nx
!              a(i,j,k)=rad2inv
!              b(i,j,k)=-rad2inv*cos(y(j))*sin(y(j))
!              c(i,j,k)=rad2inv*(cos(y(j))**2)
!              d(i,j,k)=d(i,j,k)*(cos(y(j))**2)
!              e(i,j,k)=e(i,j,k)*(cos(y(j))**2)
!              rho(i,j,k)=qgpv(i,j,k)*cor(i,j)*(cos(y(j))**2)
!!              rho(i,j,k)=(qgpv(i,j,k)-f0(j))*f0(j)*(radius*cos(lat(j)))**2
!           end do
!        end do
!     end do
!!$omp end do
!!$omp end parallel
!
!     counter=0
!     errmax=eps
!
!     call Ellip_Jacobi_3d( x, y, z, rho, eps, '111122', phi,  &
!  &                        bound_opt=bo, xa=a, ya=c, za=d, e=b, f=e )
!
!  else
!
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(i,j,k)
!     do k=1,nz
!        do j=1,ny
!           do i=1,nx
!              rho(i,j,k)=qgpv(i,j,k)*cor(i,j)*(cos(y(j))**2)
!!              rho(i,j,k)=(qgpv(i,j,k)-f0(j))*f0(j)*(radius*cos(lat(j)))**2
!           end do
!        end do
!     end do
!!$omp end do
!!$omp end parallel
!
!     counter=0
!     errmax=eps
!
!     call Ellip_Jacobi_3d( x, y, z, rho, eps, '111122', phi,  &
!  &                        bound_opt=bo, za=d, f=e )
!
!  end if
!
!  !-- calculating final variables
!
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(i,j)
!
!  do j=1,ny
!     do i=1,nx
!        call grad_1d( z, phi(i,j,:), t(i,j,:) )
!     end do
!  end do
!
!!$omp end do
!!$omp barrier
!!$omp do schedule(runtime) private(j,k)
!
!  do k=1,nz
!     do j=1,ny
!        call grad_1d( x, phi(:,j,k), vg(:,j,k), hx=sx(:,j) )
!     end do
!  end do
!
!!$omp end do
!!$omp barrier
!!$omp do schedule(runtime) private(i,k)
!
!  do k=1,nz
!     do i=1,nx
!        call grad_1d( y, phi(i,:,k), ug(i,:,k), hx=sy(i,:) )
!     end do
!  end do
!
!!$omp end do
!!$omp barrier
!!$omp do schedule(runtime) private(i,j,k)
!
!  do k=1,nz
!     do j=1,ny
!        do i=1,nx
!           if(z_flag.eqv..true.)then
!              t(i,j,k)=t(i,j,k)*t_ref(k)/(rhoc(k)*g)
!              ug(i,j,k)=-ug(i,j,k)/(fc*rhoc(k))
!              vg(i,j,k)=vg(i,j,k)/(fc*rhoc(k))
!              psi(i,j,k)=phi(i,j,k)/(fc*rhoc(k))
!           else
!              t(i,j,k)=-t(i,j,k)*z(k)/Rd
!              ug(i,j,k)=-ug(i,j,k)/fc
!              vg(i,j,k)=vg(i,j,k)/fc
!              psi(i,j,k)=phi(i,j,k)/fc
!           end if
!        end do
!     end do
!  end do
!
!!$omp end do
!!$omp end parallel
!
!end subroutine QGPV_inv
!
!!------------------------------------
!!------------------------------------
!
!subroutine QGOMG_inv( lon, lat, p, t_ref, eps, phi, qgomg,  &
!  &                   diaq, ini_flag )
!  ! Yoshino et al. (2003) : modified for (2).
!  use Thermo_Const
!  use Phys_Const
!  use Math_Const
!
!  implicit none
!
!  real, intent(in) :: lon(:)  ! longitude [rad]
!  real, intent(in) :: lat(:)  ! latitude [rad]
!  real, intent(in) :: p(:)    ! pressure [Pa]
!  real, intent(in) :: t_ref(size(p))   ! horizontal averaged temperature [K]
!  real, intent(in) :: eps     ! converging value
!  real, intent(in) :: phi(size(lon),size(lat),size(p))
!                           ! geopotential [J/kg]
!  real, intent(inout) :: qgomg(size(lon),size(lat),size(p))  ! QG vertical wind. [Pa/s]
!  real, intent(in), optional  :: diaq(size(lon),size(lat),size(p))
!                           ! diabatic heating [K/s] (based on dT/dt)
!  logical, intent(in), optional :: ini_flag   ! .true. = initializing, default = .true.
!
!  integer :: nx, ny, nz, i, j, k, counter
!  real :: kp, p_inv, errmax, radinv, rad2inv, fc
!  real, dimension(size(lat)) :: f0, coslat
!  real, dimension(size(p)) :: dtdp, d2tdp2, sigma
!  real, dimension(size(lon),size(lat)) :: sx, sy
!  real, dimension(size(lon),size(lat),size(p)) :: a, b, c, d, rho, bo, hq
!  real, dimension(size(lon),size(lat),size(p)) :: t, ug, vg, zeta, JQ
!  real, dimension(size(lon),size(lat),size(p)) :: ADVT, ADV1, ADV2
!  real, dimension(size(lon),size(lat),size(p)) :: term1, term2
!
!  nx=size(lon)
!  ny=size(lat)
!  nz=size(p)
!
!  kp=Rd/Cpd
!  radinv=1.0/radius
!  rad2inv=radinv*radinv
!  bo=0.0
!
!  if(present(ini_flag))then
!     if(ini_flag.eqv..true.)then
!        qgomg=0.0
!     end if
!  else
!     qgomg=0.0
!  end if
!
!  if(present(diaq))then
!     hq=diaq
!  else
!     hq=0.0
!  end if
!
!  !-- calculating coriolis parameter
!  do j=1,ny
!     f0(j)=2.0*omega*sin(lat(j))
!     coslat(j)=cos(lat(j))
!     do i=1,nx
!        sx(i,j)=radius*coslat(j)
!        sy(i,j)=radius
!     end do
!  end do
!  fc=f0(ny/2)
!
!  !-- calculating vertical stability
!  call grad_1d( p, t_ref, dtdp )
!  call sub_lapsi_1d( p, t_ref, d2tdp2 )
!
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(i,j)
!
!  do j=1,ny
!     do i=1,nx
!        call grad_1d( p, phi(i,j,:), t(i,j,:) )
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(k,p_inv)
!
!  do k=1,nz
!     p_inv=1.0/p(k)
!     sigma(k)=(Rd*p_inv)*(kp*t_ref(k)*p_inv-dtdp(k))
!     call sub_lapsi_2d( lon, lat, phi(:,:,k), zeta(:,:,k) )
!     call sub_lapsi_2d( lon, lat, hq(:,:,k), JQ(:,:,k) )
!     call grad_2d( lon, lat, phi(:,:,k), vg(:,:,k), ug(:,:,k), hx=sx, hy=sy )
!     call grad_2d( lon, lat, t(:,:,k), ADV1(:,:,k), ADV2(:,:,k), hx=sx, hy=sy )
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j,k)
!
!  do k=1,nz
!     do j=1,ny
!        do i=1,nx
!           zeta(i,j,k)=zeta(i,j,k)*rad2inv/(fc*(coslat(j))**2)+f0(j)
!           JQ(i,j,k)=-kp*JQ(i,j,k)*rad2inv/(sigma(k)*p(k))
!           ug(i,j,k)=-ug(i,j,k)/fc
!           vg(i,j,k)=vg(i,j,k)/fc
!           ADVT(i,j,k)=-(ug(i,j,k)*ADV1(i,j,k)+vg(i,j,k)*ADV2(i,j,k))
!        end do
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(k)
!
!  do k=1,nz
!     call grad_2d( lon, lat, zeta(:,:,k), ADV1(:,:,k), ADV2(:,:,k),  &
!  &                hx=sx, hy=sy )
!     call sub_lapsi_2d( lon, lat, ADVT(:,:,k), term2(:,:,k) )
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j,k)
!
!  do k=1,nz
!     do j=1,ny
!        do i=1,nx
!           ADVT(i,j,k)=ug(i,j,k)*ADV1(i,j,k)+vg(i,j,k)*ADV2(i,j,k)
!        end do
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j)
!
!  do j=1,ny
!     do i=1,nx
!        call grad_1d( p, ADVT(i,j,:), term1(i,j,:) )
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j,k)
!
!  do k=1,nz
!     do j=1,ny
!        do i=1,nx
!           a(i,j,k)=rad2inv
!           b(i,j,k)=rad2inv*(coslat(j)**2)
!           d(i,j,k)=-sin(lat(j))*coslat(j)*rad2inv
!           c(i,j,k)=((coslat(j)*fc)**2)/sigma(k)
!           rho(i,j,k)=JQ(i,j,k)+term2(i,j,k)*rad2inv/sigma(k)  &
!  &                            +(coslat(j)**2)*(fc/sigma(k))*term1(i,j,k)
!        end do
!     end do
!  end do
!
!!$omp end do
!!$omp end parallel
!
!  counter=0
!  errmax=eps
!
!!!  do while (eps<=errmax)
!!!     errmax=0.0
!     !-- calculating phi
!     call Ellip_Jacobi_3d( lon, lat, p, rho, eps, '222211', qgomg,  &
!  &                        bound_opt=bo, xa=a, ya=b, za=c, e=d )!, ln=1,  &
!!  &                        init_flag=.false. )
!!!counter=counter+1
!!!write(*,*) "loop num", counter
!     !-- calculating boundary variables
!!!$omp parallel default(shared)
!!!$omp do schedule(runtime) private(i,j)
!
!!     do j=1,ny
!!        do i=1,nx
!!           call grad_1d( p, phi(i,j,:), t(i,j,:) )
!!        end do
!!     end do
!
!!!$omp end do
!!
!!!$omp barrier
!!
!!!$omp do schedule(runtime) private(j,k)
!!
!!     do k=1,nz
!!        do j=1,ny
!!           call grad_1d( lon, phi(:,j,k), vg(:,j,k) )
!!        end do
!!     end do
!!
!!!$omp end do
!!
!!!$omp barrier
!!
!!!$omp do schedule(runtime) private(i,k)
!!
!!!     do k=1,nz
!!!        do i=1,nx
!!!           call grad_1d( lat, phi(i,:,k), ug(i,:,k) )
!!!        end do
!!!     end do
!
!!!$omp end do
!!!$omp end parallel
!
!!!     do k=1,nz
!!!        do j=1,ny
!!!           do i=1,nx
!!!              if(abs(phi(i,j,k)-tmp(i,j,k))>errmax)then
!!!                 errmax=abs(phi(i,j,k)-tmp(i,j,k))
!!!              end if
!!!           end do
!!!        end do
!!!     end do
!
!!!     tmp=phi
!
!!!write(*,*) "err check", errmax
!!!  end do
!
!end subroutine QGOMG_inv
!
!!------------------------------------
!!------------------------------------
!
!subroutine EPV_inv( lon, lat, pex, pv, phi, t, ug, vg, psi,  &
!  &                 ini_flag )
!  ! Yoshino et al. (2003) : modified for (6) and (8).
!  use Thermo_Const
!  use Phys_Const
!  use Math_Const
!
!  implicit none
!
!  real, intent(in) :: lon(:)  ! longitude [rad]
!  real, intent(in) :: lat(:)  ! latitude [rad]
!  real, intent(in) :: pex(:)  ! pressure by exner function [1]
!  real, intent(in) :: pv(size(lon),size(lat),size(pex))   ! EPV [10^{-6}PVU]
!  real, intent(inout) :: phi(size(lon),size(lat),size(pex))
!                              ! geopotential [J/kg]
!  real, intent(inout) :: t(size(lon),size(lat),size(pex))   ! potential temperature [K]
!  real, intent(inout) :: ug(size(lon),size(lat),size(pex))  ! x-wind [ms-1]
!  real, intent(inout) :: vg(size(lon),size(lat),size(pex))  ! y-wind [ms-1]
!  real, intent(inout) :: psi(size(lon),size(lat),size(pex))  ! stream func. [10^6PVUs]
!  logical, intent(in), optional :: ini_flag   ! .true. = initializing, default = .true.
!
!  integer :: nx, ny, nz, i, j, k, l, counter
!  real :: kp, errmaxh, errmaxs, radinv, rad2, rad2inv, alome, alome2
!  real :: leng, height, strat, FF, zmin, zmax, phis, psis
!  real :: epss, epsh, lengrat, accel
!  real :: lond(size(lon)), latd(size(lat))
!  real, dimension(size(pex)) :: pres, pvc, boundiv, boundivv
!  real, dimension(size(lat)) :: f0, coslat
!  real, dimension(size(lon),size(pex)) :: bzon
!  real, dimension(size(lat),size(pex)) :: bmer
!  real, dimension(size(lon),size(lat)) :: sx, sy
!  real, dimension(size(lon),size(lat),size(pex)) :: apsi, bpsi, cpsi, dphi
!  real, dimension(size(lon),size(lat),size(pex)) :: rho, tmp1, tmp2
!  real, dimension(size(lon),size(lat),size(pex)) :: dpsi_ll, dpsi_lon2, dpsi_lat2
!  real, dimension(size(lon),size(lat),size(pex)) :: dpsi_lonp, dpsi_latp, dpsi_lat
!  real, dimension(size(lon),size(lat),size(pex)) :: dphi_lonp, dphi_latp, dphi_p2
!  real, dimension(size(lon),size(lat),size(pex)) :: laphi, lapsi
!  real, dimension(size(lon),size(lat),size(pex)) :: bo_phi, bo_psi, pvu
!  real, dimension(size(lon),size(lat),size(pex),15) :: dmpval
!  real, parameter :: acceld=1.0e-1
!  real, parameter :: epssr=1.0e4    ! converging value for psi
!  real, parameter :: epshr=1.0    ! converging value for phi
!  logical, parameter :: forced_NPV=.true.   ! forced Negative PV to +1.0e-2 PVU.
!
!  nx=size(lon)
!  ny=size(lat)
!  nz=size(pex)
!
!  !-- calclating scale variables
!  call Mean_2d( t(:,:,1), zmin )
!  call Mean_2d( t(:,:,nz), zmax )
!
!  FF=2.0*omega
!  strat=cpd*(zmax-zmin)
!  height=1.0
!  leng=sqrt(strat)*height/FF
!  lengrat=leng/radius
!  psis=FF/(strat*height*height)
!  phis=1.0/(strat*height*height)
!  epsh=epshr*phis
!  epss=epssr*psis
!  accel=acceld
!
!  write(*,*) "*** MESSAGE (EPV_inv) ***"
!  write(*,'(a12,1P4E14.6)') "(S,F,L,H) : ", strat, FF, leng, height
!  write(*,'(a12,1P2E14.6)') "(phi,psi) : ", phis, psis
!  write(*,'(a20,1P3E14.6)') "(phie,psie,accel) : ", epsh, epss, accel
!
!  kp=Rd/Cpd
!  bo_phi=phi
!  bo_psi=psi
!  tmp1=0.0
!  tmp2=0.0
!  rad2=radius*radius
!  radinv=1.0/radius
!  rad2inv=1.0/rad2
!  alome=2.0*omega*radius
!  alome2=alome**2
!
!  !-- nondimensionalization
!
!  if(present(ini_flag))then
!     if(ini_flag.eqv..true.)then
!        phi=0.0
!        psi=0.0
!     else  ! scaling each variable
!        phi=phi*phis
!        psi=psi*psis
!        tmp1=phi
!        tmp2=psi
!     end if
!  else
!     phi=0.0
!     psi=0.0
!  end if
!
!  do k=1,nz
!     pres(k)=p0*(pex(k)**(Cpd/Rd))
!     pvc(k)=pres(k)*Cpd/(FF*strat*g*kp*pex(k))
!  end do
!  do i=1,nx
!     lond(i)=lon(1)+(lon(i)-lon(1))/lengrat
!  end do
!  do j=1,ny
!     latd(j)=lat(1)+(lat(j)-lat(1))/lengrat
!  end do
!
!  !-- forced -PV to +1.0e-2 PVU
!  if(forced_NPV.eqv..true.)then
!     write(*,*) "*** MESSAGE : EPV_inv ***"
!     write(*,*) "Forced Negative PV to +1.0e-2 PVU."
!     do k=1,nz
!        do j=1,ny
!           do i=1,nx
!              if(pv(i,j,k)<=0.0)then
!                 pvu(i,j,k)=1.0e-8*pvc(k)
!              else
!                 pvu(i,j,k)=pv(i,j,k)*pvc(k)
!              end if
!           end do
!        end do
!     end do
!  end if
!
!  !-- calculating coriolis parameter
!  do j=1,ny
!     f0(j)=2.0*omega*sin(lat(j))
!     coslat(j)=cos(lat(j))
!     do i=1,nx
!        sx(i,j)=radius*coslat(j)
!        sy(i,j)=radius
!     end do
!  end do
!
!  !-- calculating integral
!
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(j,k)
!
!  do k=1,nz
!     do j=1,ny
!        bmer(j,k)=ug(1,j,k)+ug(nx,j,k)
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,k)
!
!  do k=1,nz
!     do i=1,nx
!        bzon(i,k)=vg(i,1,k)*coslat(1)+vg(i,ny,k)*coslat(ny)
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(k)
!
!  do k=1,nz
!     call rectangle_int( lon, bzon(:,k), lon(1), lon(nx), boundiv(k) )
!     call rectangle_int( lat, bmer(:,k), lat(1), lat(ny), boundivv(k) )
!     boundiv(k)=(boundiv(k)+boundivv(k))  &
!  &             /((coslat(1)+coslat(ny))*(lon(nx)-lon(1))+2.0*(lat(ny)-lat(1)))
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j,k)
!
!  do k=1,nz
!     do j=1,ny
!        do i=1,nx
!           bpsi(i,j,k)=-coslat(j)*sin(lat(j))
!           cpsi(i,j,k)=coslat(j)**2
!        end do
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j)
!
!  !-- setting boundary condition
!  do j=1,ny
!     do i=1,nx
!        bo_psi(i,j,1)=-Cpd*t(i,j,1)*phis/sin(lat(j))
!        bo_psi(i,j,nz)=-Cpd*t(i,j,nz)*phis/sin(lat(j))
!        if(i/=1.and.i/=nx.and.j/=1.and.j/=ny)then
!           bo_phi(i,j,1)=-Cpd*t(i,j,1)*phis
!           bo_phi(i,j,nz)=-Cpd*t(i,j,nz)*phis
!!           bo_phi(i,j,1)=phi(i,j,1)*phis
!!           bo_phi(i,j,nz)=phi(i,j,nz)*phis
!        else
!           bo_phi(i,j,1)=phi(i,j,1)
!           bo_phi(i,j,nz)=phi(i,j,nz)
!        end if
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!!!$omp do schedule(runtime) private(i,k)
!!!
!!!  do k=2,nz-1
!!!     do i=1,nx
!!!        bo_psi(i,1,k)=-radius*ug(i,1,k)*psis
!!!        bo_psi(i,ny,k)=-radius*ug(i,ny,k)*psis
!!!        bo_psi(i,1,k)=0.0
!!!        bo_psi(i,ny,k)=0.0
!!!        bo_phi(i,1,k)=phi(i,1,k)
!!!        bo_phi(i,ny,k)=phi(i,ny,k)
!!!     end do
!!!  end do
!!!
!!!!$omp end do
!
!!!!$omp barrier
!
!!!!$omp do schedule(runtime) private(j,k)
!!!
!!!  do k=2,nz-1
!!!     do j=1,ny
!!!        bo_psi(1,j,k)=radius*coslat(j)*vg(1,j,k)*psis
!!!        bo_psi(nx,j,k)=radius*coslat(j)*vg(nx,j,k)*psis
!!!        bo_psi(1,j,k)=0.0
!!!        bo_psi(nx,j,k)=0.0
!!!        bo_phi(1,j,k)=phi(1,j,k)
!!!        bo_phi(nx,j,k)=phi(nx,j,k)
!!!     end do
!!!  end do
!!!
!!!!$omp end do
!
!!!!$omp barrier
!
!  !-- calculating rho for the first step.
!
!!$omp do schedule(runtime) private(k)
!
!  do k=1,nz
!     call sub_cross2_2d( lond, latd, psi(:,:,k), dpsi_ll(:,:,k) )
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(j)
!
!  do j=1,ny
!     call sub_cross2_2d( lond, pex, psi(:,j,:), dpsi_lonp(:,j,:) )
!  end do
!
!!$Omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i)
!
!  do i=1,nx
!     call sub_cross2_2d( latd, pex, psi(i,:,:), dpsi_latp(i,:,:) )
!  end do
!
!!$Omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(j,k)
!
!  do k=1,nz
!     do j=1,ny
!        call sub_lapsi_1d( lond, psi(:,j,k), dpsi_lon2(:,j,k) )
!     end do
!  end do
!
!!$Omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,k)
!
!  do k=1,nz
!     do i=1,nx
!        call sub_lapsi_1d( latd, psi(i,:,k), dpsi_lat2(i,:,k) )
!        call grad_1d( latd, psi(i,:,k), dpsi_lat(i,:,k) )
!     end do
!  end do
!
!!$Omp end do
!!$omp end parallel
!
!  counter=0
!  errmaxh=epsh
!  errmaxs=epss
!
!  dmpval=0.0
!  dmpval(:,:,:,14)=phi(:,:,:)
!  dmpval(:,:,:,15)=psi(:,:,:)
!
!do l=1,15
!call write_file_3d( 'tmpdmp'//trim(adjustl(i2c_convert(0,forma='(i2.2)'))),  &
!  &                 nx, ny, nz, 1+nz*(l-1), dmpval(:,:,:,l) )
!end do
!
!  do while (epsh<=errmaxh.and.epss<=errmaxs)
!     errmaxh=0.0
!     errmaxs=0.0
!
!     !-- calculating each coefficient for phi
!
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(k)
!
!     do k=1,nz
!        call sub_lapsi_2d( lond, latd, phi(:,:,k), laphi(:,:,k) )
!     end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(j)
!
!     do j=1,ny
!        call sub_cross2_2d( lond, pex, phi(:,j,:), dphi_lonp(:,j,:) )
!     end do
!
!!$omp end do
!
!!$omp barrier
!!$omp do schedule(runtime) private(i)
!
!     do i=1,nx
!        call sub_cross2_2d( latd, pex, phi(i,:,:), dphi_latp(i,:,:) )
!     end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j)
!
!     do j=1,ny
!        do i=1,nx
!           call sub_lapsi_1d( pex, phi(i,j,:), dphi_p2(i,j,:) )
!        end do
!     end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j,k)
!
!     do k=1,nz
!        do j=1,ny
!           do i=1,nx
!              rho(i,j,k)=((coslat(j)**2)*(  &
!  &                       pvu(i,j,k)/((height)**2)  &
!  &                       -sin(lat(j))*dphi_p2(i,j,k)  &
!  &                       -lengrat*coslat(j)*dpsi_lat(i,j,k))  &
!  &                      +laphi(i,j,k)  &
!  &                      -2.0*(dpsi_lon2(i,j,k)*dpsi_lat2(i,j,k)  &
!  &                           -dpsi_ll(i,j,k)**2)  &
!  &                      +dpsi_lonp(i,j,k)*dphi_lonp(i,j,k)  &
!  &                      +(coslat(j)**2)*dpsi_latp(i,j,k)*dphi_latp(i,j,k))  &
!  &                      /(sin(lat(j))+dphi_p2(i,j,k))
!!if(i==1.or.i==nx.or.j==1.or.j==ny.or.k==1.or.k==nz)then
!!write(*,*) "bo_phi check", t(i,j,k), bo_phi(i,j,k), bo_psi(i,j,k), i, j, k
!!end if
!dmpval(i,j,k,1)=rho(i,j,k)
!dmpval(i,j,k,2)=laphi(i,j,k)
!dmpval(i,j,k,3)=(coslat(j)**2)*pvu(i,j,k)/(height**2)
!dmpval(i,j,k,4)=-2.0*(dpsi_lon2(i,j,k)*dpsi_lat2(i,j,k))
!dmpval(i,j,k,5)=2.0*(dpsi_ll(i,j,k)**2)
!dmpval(i,j,k,6)=dpsi_lon2(i,j,k)
!dmpval(i,j,k,7)=dpsi_lat2(i,j,k)
!dmpval(i,j,k,8)=dpsi_ll(i,j,k)
!dmpval(i,j,k,9)=dpsi_lonp(i,j,k)*dphi_lonp(i,j,k)  &
!  &             +(coslat(j)**2)*dpsi_latp(i,j,k)*dphi_latp(i,j,k)
!dmpval(i,j,k,10)=sin(lat(j))+dphi_p2(i,j,k)
!           end do
!        end do
!     end do
!
!!$omp end do
!!$omp end parallel
!
!     !-- calculating psi
!     do k=2,nz-1
!        call Ellip_Jacobi_2d( lond, latd, rho(:,:,k), epss, '2222',  &
!  &                           psi(:,:,k), bound_opt=bo_psi(:,:,k),  &
!  &                           c=cpsi(:,:,k), e=bpsi(:,:,k),  &
!  &                           ln=1, init_flag=.false., accel=accel )
!     end do
!
!     !-- calculating bottom and top boundary conditions
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(i,j)
!     do j=1,ny
!        do i=1,nx
!           psi(i,j,1)=psi(i,j,2)-(pex(2)-pex(1))*bo_psi(i,j,1)
!           psi(i,j,nz)=psi(i,j,nz-1)+(pex(nz)-pex(nz-1))*bo_psi(i,j,nz)
!        end do
!     end do
!!$omp end do
!!$omp end parallel
!
!!!     !-- calculating lateral conditions
!!!     do k=2,nz-1
!!!        psi(1,1,k)=psi(2,2,k)
!!!        do i=2,nx
!!!           !-- E->W, S
!!!           psi(i,1,k)=psi(i-1,1,k)+radius*coslat(1)*(vg(i,1,k)+boundiv(k))  &
!!!  &                                *(lon(i)-lon(i-1))
!!!        end do
!!!        do j=2,ny
!!!           !-- W, S->N
!!!           psi(nx,j,k)=psi(nx,j-1,k)+radius*(-ug(nx,j,k)+boundiv(k))  &
!!!  &                                *(lat(j)-lat(j-1))
!!!        end do
!!!        do i=nx-1,1,-1
!!!           !-- W->E, N
!!!           psi(i,ny,k)=psi(i+1,ny,k)+radius*coslat(ny)*(-vg(i,ny,k)+boundiv(k))  &
!!!  &                                *(lon(i+1)-lon(i))
!!!        end do
!!!        do j=ny-1,1,-1
!!!           !-- E, N->S
!!!           psi(1,j,k)=psi(1,j+1,k)+radius*(ug(1,j,k)+boundiv(k))  &
!!!  &                                *(lat(j+1)-lat(j))
!!!        end do
!!!     end do
!!!
!!!     do k=2,nz-1
!!!        do i=1,nx
!!!           bo_psi(i,1,k)=psi(i,1,k)
!!!           bo_psi(i,ny,k)=psi(i,ny,k)
!!!        end do
!!!        do j=1,ny
!!!           bo_psi(1,j,k)=psi(1,j,k)
!!!           bo_psi(nx,j,k)=psi(nx,j,k)
!!!        end do
!!!     end do
!
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(k)
!     !-- calculating each coefficient for psi
!
!     do k=1,nz
!        call sub_lapsi_2d( lond, latd, psi(:,:,k), lapsi(:,:,k) )
!        call sub_cross2_2d( lond, latd, psi(:,:,k), dpsi_ll(:,:,k) )
!     end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(j)
!
!     do j=1,ny
!        call sub_cross2_2d( lond, pex, psi(:,j,:), dpsi_lonp(:,j,:) )
!     end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i)
!
!     do i=1,nx
!        call sub_cross2_2d( latd, pex, psi(i,:,:), dpsi_latp(i,:,:) )
!     end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,k)
!
!     do k=1,nz
!        do i=1,nx
!           call sub_lapsi_1d( latd, psi(i,:,k), dpsi_lat2(i,:,k) )
!           call grad_1d( latd, psi(i,:,k), dpsi_lat(i,:,k) )
!        end do
!     end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(j,k)
!
!     do k=1,nz
!        do j=1,ny
!           call sub_lapsi_1d( lond, psi(:,j,k), dpsi_lon2(:,j,k) )
!        end do
!     end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j,k)
!
!     do k=1,nz
!        do j=1,ny
!           do i=1,nx
!              rho(i,j,k)=(coslat(j)**2)*(  &
!  &                       pvu(i,j,k)/(height**2)  &
!  &                       +lengrat*coslat(j)*dpsi_lat(i,j,k))  &
!  &                      +sin(lat(j))*lapsi(i,j,k)  &
!  &                      +2.0*(dpsi_lon2(i,j,k)*dpsi_lat2(i,j,k)  &
!  &                           -dpsi_ll(i,j,k)**2)  &
!  &                      +dpsi_lonp(i,j,k)*dphi_lonp(i,j,k)  &
!  &                       +(coslat(j)**2)*dpsi_latp(i,j,k)*dphi_latp(i,j,k)
!              dphi(i,j,k)=sin(lat(j))*(coslat(j)**2)+lapsi(i,j,k)
!dmpval(i,j,k,11)=rho(i,j,k)
!dmpval(i,j,k,12)=lapsi(i,j,k)
!dmpval(i,j,k,13)=dphi(i,j,k)
!              if(dphi(i,j,k)==0.0)then
!                 write(*,*) "Detect z is zero", sin(lat(j))*(coslat(j)**2), lapsi(i,j,k)
!              end if
!           end do
!        end do
!     end do
!
!!$omp end do
!!$omp end parallel
!
!     !-- calculating phi
!     call Ellip_Jacobi_3d( lond, latd, pex, rho, epsh, '111122', phi,  &
!  &                        bound_opt=bo_phi, ya=cpsi, za=dphi,  &
!  &                        e=bpsi, ln=1,  &
!  &                        init_flag=.false., accel=accel )
!
!counter=counter+1
!write(*,*) "loop num", counter
!
!     do k=2,nz-1
!        do j=2,ny-1
!           do i=2,nx-1
!              if(abs(phi(i,j,k)-tmp1(i,j,k))>errmaxh)then
!                 errmaxh=abs(phi(i,j,k)-tmp1(i,j,k))
!              end if
!              if(abs(psi(i,j,k)-tmp2(i,j,k))>errmaxs)then
!                 errmaxs=abs(psi(i,j,k)-tmp2(i,j,k))
!              end if
!           end do
!        end do
!     end do
!
!     tmp1=phi
!     tmp2=psi
!
!dmpval(:,:,:,14)=phi(:,:,:)
!dmpval(:,:,:,15)=psi(:,:,:)
!write(*,*) "err check [h,s]", errmaxh, errmaxs
!do l=1,15
!call write_file_3d( 'tmpdmp'//trim(adjustl(i2c_convert(counter,forma='(i2.2)'))),  &
!  &                 nx, ny, nz, 1+nz*(l-1), dmpval(:,:,:,l) )
!end do
!  end do
!
!  !-- reversing phi and psi
!
!  do k=1,nz
!     do j=1,ny
!        do i=1,nx
!           phi(i,j,k)=phi(i,j,k)/phis
!           psi(i,j,k)=psi(i,j,k)/psis
!        end do
!     end do
!  end do
!
!  !-- calculating final variables
!
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(i,j)
!
!  do j=1,ny
!     do i=1,nx
!        call grad_1d( pex(2:nz-1), phi(i,j,2:nz-1), t(i,j,2:nz-1) )
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(j,k)
!
!  do k=1,nz
!     do j=2,ny-1
!        call grad_1d( lon(2:nx-1), psi(2:nx-1,j,k), vg(2:nx-1,j,k),  &
!  &                   hx=sx(2:nx-1,j) )
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,k)
!
!  do k=1,nz
!     do i=2,nx-1
!        call grad_1d( lat(2:ny-1), psi(i,2:ny-1,k), ug(i,2:ny-1,k),  &
!  &                   hx=sy(i,2:ny-1) )
!     end do
!  end do
!
!!$omp end do
!
!!$omp barrier
!
!!$omp do schedule(runtime) private(i,j,k)
!
!  do k=1,nz
!     do j=1,ny
!        do i=1,nx
!           if(k/=1.and.k/=nz)then
!              t(i,j,k)=-t(i,j,k)/Cpd
!           end if
!           if(i/=1.and.i/=nx.and.j/=1.and.j/=ny)then
!              ug(i,j,k)=-ug(i,j,k)
!           end if
!        end do
!     end do
!  end do
!
!!$omp end do
!!$omp end parallel
!
!end subroutine EPV_inv

!------------------------------------
!------------------------------------



end module
