!---------------------------------------------------------------
! Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!---------------------------------------------------------------
module Map_Function
!  地図座標系におけるいくらかの変換関数

interface ll2rt
  module procedure ll2rt_f, ll2rt_d
end interface ll2rt

interface ll2radi
  module procedure ll2radi_f, ll2radi_d
end interface ll2radi

interface dis2lon
  module procedure dis2lon_f, dis2lon_d
end interface dis2lon

interface dis2lat
  module procedure dis2lat_f, dis2lat_d
end interface dis2lat

interface rt2ll
  module procedure rt2ll_f, rt2ll_d
end interface rt2ll

interface lat2mery
  module procedure lat2mery_f, lat2mery_d
end interface lat2mery

interface ll2lamdis
  module procedure ll2lamdis_f, ll2lamdis_d
end interface ll2lamdis

interface ll2lamxy
  module procedure ll2lamxy_f, ll2lamxy_d
end interface ll2lamxy

interface ll2merdis
  module procedure ll2merdis_f, ll2merdis_d
end interface ll2merdis

interface lon2merx
  module procedure lon2merx_f, lon2merx_d
end interface lon2merx

interface vec2lam
  module procedure vec2lam_f, vec2lam_d
end interface vec2lam

interface x2merlon
  module procedure x2merlon_f, x2merlon_d
end interface x2merlon

interface xy2lamll
  module procedure xy2lamll_f, xy2lamll_d
end interface xy2lamll

interface y2merlat
  module procedure y2merlat_f, y2merlat_d
end interface y2merlat

interface Lambert_coe
  module procedure Lambert_coe_f, Lambert_coe_d
end interface Lambert_coe

contains

subroutine ll2rt_f( lon0, lat0, lon1, lat1, r, theta )
! 地球の球面座標系において, (lon0, lat0) の位置を原点とした (r,\theta) 座標を
! 展開したとき, (lon1, lat1) の位置における距離 r, 同位角 theta を
! 計算する. 距離 r は球面上の円弧距離とし, theta = 90, 270 deg が球面の
! 子午面上を通るように theta は配置される.
! rt2ll の逆関数, 距離だけ求める場合は ll2radi を用いればよい.
! 地図投影としては, 正距方位図法に該当する.
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: lon0          ! 極座標原点の経度 [rad]
  real, intent(in) :: lat0          ! 極座標原点の緯度 [rad]
  real, intent(in) :: lon1          ! 計算された経度 [rad]
  real, intent(in) :: lat1          ! 計算された緯度 [rad]
  real, intent(inout) :: r          ! lon0 からの円弧距離 [m]
  real, intent(inout) :: theta      ! lon0 での同位角 [rad]
  real :: tmpcos, tmpsin, tmptan

  r=ll2radi_f( lon0, lat0, lon1, lat1 )

  if(r>0.0)then
     tmpcos=cos(lat1)*sin(lon1-lon0)
     tmpsin=sin(lat1)*cos(lat0)-cos(lat1)*sin(lat0)*cos(lon1-lon0)
     if(tmpcos==0.0.and.tmpsin==0.0)then
        theta=0.0
     else if(tmpcos==0.0.and.tmpsin>0.0)then
        theta=0.5*pi
     else if(tmpcos==0.0.and.tmpsin<0.0)then
        theta=1.5*pi
     else if(tmpcos>0.0.and.tmpsin==0.0)then
        theta=0.0
     else if(tmpcos<0.0.and.tmpsin==0.0)then
        theta=pi
     else
        if(tmpcos>0.0.and.tmpsin>0.0)then
           tmptan=tmpsin/tmpcos
           theta=atan(tmptan)
        else if(tmpcos<0.0.and.tmpsin>0.0)then
           tmptan=tmpsin/abs(tmpcos)
           theta=pi-atan(tmptan)
        else if(tmpcos>0.0.and.tmpsin<0.0)then
           tmptan=abs(tmpsin)/tmpcos
           theta=2.0*pi-atan(tmptan)
        else if(tmpcos<0.0.and.tmpsin<0.0)then
           tmptan=tmpsin/tmpcos
           theta=pi+atan(tmptan)
        end if
     end if
  else
     theta=0.0
  end if

end subroutine ll2rt_f

!-----------------------------------------------
!-----------------------------------------------

subroutine ll2rt_d( lon0, lat0, lon1, lat1, r, theta )
! 地球の球面座標系において, (lon0, lat0) の位置を原点とした (r,\theta) 座標を
! 展開したとき, (lon1, lat1) の位置における距離 r, 同位角 theta を
! 計算する. 距離 r は球面上の円弧距離とし, theta = 90, 270 deg が球面の
! 子午面上を通るように theta は配置される.
! rt2ll の逆関数, 距離だけ求める場合は ll2radi を用いればよい.
! 地図投影としては, 正距方位図法に該当する.
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: lon0          ! 極座標原点の経度 [rad]
  double precision, intent(in) :: lat0          ! 極座標原点の緯度 [rad]
  double precision, intent(in) :: lon1          ! 計算された経度 [rad]
  double precision, intent(in) :: lat1          ! 計算された緯度 [rad]
  double precision, intent(inout) :: r          ! lon0 からの円弧距離 [m]
  double precision, intent(inout) :: theta      ! lon0 での同位角 [rad]
  double precision :: tmpcos, tmpsin, tmptan

  r=ll2radi_d( lon0, lat0, lon1, lat1 )

  if(r>0.0d0)then
     tmpcos=dcos(lat1)*dsin(lon1-lon0)
     tmpsin=dsin(lat1)*dcos(lat0)-dcos(lat1)*dsin(lat0)*dcos(lon1-lon0)
     if(tmpcos==0.0d0.and.tmpsin==0.0d0)then
        theta=0.0d0
     else if(tmpcos==0.0d0.and.tmpsin>0.0d0)then
        theta=0.5d0*pi_dp
     else if(tmpcos==0.0d0.and.tmpsin<0.0d0)then
        theta=1.5d0*pi_dp
     else if(tmpcos>0.0d0.and.tmpsin==0.0d0)then
        theta=0.0d0
     else if(tmpcos<0.0d0.and.tmpsin==0.0d0)then
        theta=pi_dp
     else
        if(tmpcos>0.0d0.and.tmpsin>0.0d0)then
           tmptan=tmpsin/tmpcos
           theta=datan(tmptan)
        else if(tmpcos<0.0d0.and.tmpsin>0.0d0)then
           tmptan=tmpsin/dabs(tmpcos)
           theta=pi_dp-datan(tmptan)
        else if(tmpcos>0.0d0.and.tmpsin<0.0d0)then
           tmptan=dabs(tmpsin)/tmpcos
           theta=2.0d0*pi_dp-datan(tmptan)
        else if(tmpcos<0.0d0.and.tmpsin<0.0d0)then
           tmptan=tmpsin/tmpcos
           theta=pi_dp+datan(tmptan)
        end if
     end if
  else
     theta=0.0d0
  end if

end subroutine ll2rt_d

!-----------------------------------------------
!-----------------------------------------------

subroutine rt2ll_f( r, theta, lon0, lat0, lon, lat )
! 地球の球面座標系において, (lon0, lat0) の位置を原点とした (r,\theta) 座標を
! 展開したとき, 距離 r, 同位角 theta の位置における球面上での経度を
! 計算する. 距離 r は球面上の円弧距離とし, theta = 90, 270 deg が球面の
! 子午面上を通るように theta は配置される.
! 地図投影としては, 正距方位図法に該当する.
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: r          ! lon0 からの円弧距離 [m]
  real, intent(in) :: theta      ! lon0 での同位角 [rad]
  real, intent(in) :: lon0       ! 極座標原点の経度 [rad]
  real, intent(in) :: lat0       ! 極座標原点の緯度 [rad]
  real, intent(inout) :: lon     ! 計算された経度 [rad]
  real, intent(inout) :: lat     ! 計算された緯度 [rad]
  real :: thetad, lond, latd, tmplon, tmplat, rratio

  thetad=180.0*theta/pi
  lond=180.0*lon0/pi
  latd=180.0*lat0/pi
  rratio=r/radius

  do while(thetad>360.0)
     thetad=thetad-360.0
  end do

  if(thetad==-90.0.or.thetad==270.0)then
     lon=lon0
     lat=lat0-rratio
  else if(thetad==90.0)then
     lon=lon0
     lat=lat0+rratio
  else if((-90.0<thetad.and.90.0>thetad).or.  &
  &  (270.0<thetad.and.360.0>=thetad))then
     tmplat=cos(lat0)*sin(rratio)*sin(theta)+sin(lat0)*cos(rratio)
     lat=asin(tmplat)
     tmplon=sin(rratio)*cos(theta)/cos(asin(tmplat))
     lon=lon0+asin(tmplon)
  else if((90.0<thetad.and.270.0>thetad).or.  &
  &       (-180.0<=thetad.and.-90.0>thetad))then
     tmplat=cos(lat0)*sin(rratio)*sin(theta)+sin(lat0)*cos(rratio)
     lat=asin(tmplat)
     tmplon=-sin(rratio)*cos(theta)/cos(asin(tmplat))
     lon=lon0-asin(tmplon)
  else
     write(*,*) "### ERROR : (rt2ll:Map_Function)"
     write(*,*) "argument 'theta' is not valid : ", theta
     write(*,*) "STOP."
     stop
  end if

end subroutine rt2ll_f

!-----------------------------------------------
!-----------------------------------------------

subroutine rt2ll_d( r, theta, lon0, lat0, lon, lat )
! 地球の球面座標系において, (lon0, lat0) の位置を原点とした (r,\theta) 座標を
! 展開したとき, 距離 r, 同位角 theta の位置における球面上での経度を
! 計算する. 距離 r は球面上の円弧距離とし, theta = 90, 270 deg が球面の
! 子午面上を通るように theta は配置される.
! 地図投影としては, 正距方位図法に該当する.
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: r          ! lon0 からの円弧距離 [m]
  double precision, intent(in) :: theta      ! lon0 での同位角 [rad]
  double precision, intent(in) :: lon0       ! 極座標原点の経度 [rad]
  double precision, intent(in) :: lat0       ! 極座標原点の緯度 [rad]
  double precision, intent(inout) :: lon     ! 計算された経度 [rad]
  double precision, intent(inout) :: lat     ! 計算された緯度 [rad]
  double precision :: thetad, lond, latd, tmplon, tmplat, rratio

  thetad=180.0d0*theta/pi_dp
  lond=180.0d0*lon0/pi_dp
  latd=180.0d0*lat0/pi_dp
  rratio=r/radius_dp

  do while(thetad>360.0d0)
     thetad=thetad-360.0d0
  end do

  if(thetad==-90.0d0.or.thetad==270.0d0)then
     lon=lon0
     lat=lat0-rratio
  else if(thetad==90.0d0)then
     lon=lon0
     lat=lat0+rratio
  else if((-90.0d0<thetad.and.90.0d0>thetad).or.  &
  &  (270.0d0<thetad.and.360.0d0>=thetad))then
     tmplat=cos(lat0)*sin(rratio)*sin(theta)+sin(lat0)*cos(rratio)
     lat=asin(tmplat)
     tmplon=sin(rratio)*cos(theta)/cos(asin(tmplat))
     lon=lon0+asin(tmplon)
  else if((90.0d0<thetad.and.270.0d0>thetad).or.  &
  &       (-180.0d0<=thetad.and.-90.0d0>thetad))then
     tmplat=cos(lat0)*sin(rratio)*sin(theta)+sin(lat0)*cos(rratio)
     lat=asin(tmplat)
     tmplon=-sin(rratio)*cos(theta)/cos(asin(tmplat))
     lon=lon0-asin(tmplon)
  else
     write(*,*) "### ERROR : (rt2ll:Map_Function)"
     write(*,*) "argument 'theta' is not valid : ", theta
     write(*,*) "STOP."
     stop
  end if

end subroutine rt2ll_d

!-----------------------------------------------
!-----------------------------------------------

real function ll2radi_f( lon1, lat1, lon2, lat2 )
! ある球面上での 2 点での緯度, 経度を元に, それらを終始点とする円弧の距離を
! 計算する.
  use Phys_Const
  implicit none
  real, intent(in) :: lon1    ! 点 1 での経度 [rad]
  real, intent(in) :: lat1    ! 点 1 での緯度 [rad]
  real, intent(in) :: lon2    ! 点 2 での経度 [rad]
  real, intent(in) :: lat2    ! 点 2 での緯度 [rad]
  double precision :: lond1, lond2, latd1, latd2, tmp

  lond1=dble(lon1)
  lond2=dble(lon2)
  latd1=dble(lat1)
  latd2=dble(lat2)

  if(lond1==lond2.and.latd1==latd2)then
     ll2radi_f=0.0
  else
     tmp=sin(latd1)*sin(latd2)+cos(latd1)*cos(latd2)*cos(lond2-lond1)
     if(tmp<-1.0d0.or.tmp>1.0d0)then
        write(*,*) "*** ERROR (ll2radi) *** : Detect error", tmp, latd1, latd2, lond1, lond2
        stop
     end if
     ll2radi_f=real(acos(tmp))*radius
  end if

  return
end function ll2radi_f

!-----------------------------------------------
!-----------------------------------------------

real function ll2radi_d( lon1, lat1, lon2, lat2 )
! ある球面上での 2 点での緯度, 経度を元に, それらを終始点とする円弧の距離を
! 計算する.
  use Phys_Const
  implicit none
  double precision, intent(in) :: lon1    ! 点 1 での経度 [rad]
  double precision, intent(in) :: lat1    ! 点 1 での緯度 [rad]
  double precision, intent(in) :: lon2    ! 点 2 での経度 [rad]
  double precision, intent(in) :: lat2    ! 点 2 での緯度 [rad]
  double precision :: lond1, lond2, latd1, latd2, tmp

  lond1=lon1
  lond2=lon2
  latd1=lat1
  latd2=lat2

  if(lond1==lond2.and.latd1==latd2)then
     ll2radi_d=0.0d0
  else
     tmp=sin(latd1)*sin(latd2)+cos(latd1)*cos(latd2)*cos(lond2-lond1)
     if(tmp<-1.0d0.or.tmp>1.0d0)then
        write(*,*) "*** ERROR (ll2radi) *** : Detect error", tmp, latd1, latd2, lond1, lond2
        stop
     end if
     ll2radi_d=real(acos(tmp))*radius_dp
  end if

  return
end function ll2radi_d

!-----------------------------------------------
!-----------------------------------------------

real function dis2lon_f( x, lon0, phi0 )
!  基準経度 lon0 から空間距離 x [m] 離れた位置における経度 [rad].
!  ただし, 基準緯度 phi0 [rad] から同緯度上で計測した距離を用いる.
!  x は東に計算する際は正値, 西に計算する際は負値を与えれば計算可能.
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: x     ! 基準経度 lon0 からの空間距離 [m] (東向きが正)
  real, intent(in) :: lon0  ! 基準経度 [rad]
  real, intent(in) :: phi0  ! 基準緯度 [rad]

  dis2lon_f=x/(radius*cos(phi0))+lon0

  return
end function

!-----------------------------------------------

double precision function dis2lon_d( x, lon0, phi0 )
!  基準経度 lon0 から空間距離 x [m] 離れた位置における経度 [rad].
!  ただし, 基準緯度 phi0 [rad] から同緯度上で計測した距離を用いる.
!  x は東に計算する際は正値, 西に計算する際は負値を与えれば計算可能.
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: x     ! 基準経度 lon0 からの空間距離 [m] (東向きが正)
  double precision, intent(in) :: lon0  ! 基準経度 [rad]
  double precision, intent(in) :: phi0  ! 基準緯度 [rad]

  dis2lon_d=x/(radius_dp*cos(phi0))+lon0

  return
end function

!---------------------------------

!---------------------------------

real function dis2lat_f( y, phi0 )
!  基準緯度 phi0 から空間距離 y [m] 離れた位置における緯度 [rad].
!  y は北に計算する際は正値, 南に計算する際は負値を与えれば計算可能.
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: y     ! 基準緯度 phi0 からの空間距離 [m] (北向きが正)
  real, intent(in) :: phi0  ! 基準緯度 [rad]

  dis2lat_f=y/radius+phi0

  return
end function

!---------------------------------

!---------------------------------

double precision function dis2lat_d( y, phi0 )
!  基準緯度 phi0 から空間距離 y [m] 離れた位置における緯度 [rad].
!  y は北に計算する際は正値, 南に計算する際は負値を与えれば計算可能.
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: y     ! 基準緯度 phi0 からの空間距離 [m] (北向きが正)
  double precision, intent(in) :: phi0  ! 基準緯度 [rad]

  dis2lat_d=y/radius_dp+phi0

  return
end function

! 以降, 緯度経度座標と地図投影座標における変換関数
!---------------------------------
! 1. 以降はメルカトル座標
!---------------------------------

real function x2merlon_f(x,lam0)
!  基準経度 lam0 から地図距離 x [m] 離れた位置における経度 [rad].
!  ただし, x は東方向に正, 西方向に負を与えれば計算可能.
  use Phys_Const
  implicit none
  real, intent(in) :: x     ! 基準経度 lam0 からの空間距離 [m] (東向きが正)
  real, intent(in) :: lam0  ! 基準経度 [rad]

  x2merlon_f=x/radius+lam0

  return
end function

!---------------------------------

double precision function x2merlon_d(x,lam0)
!  基準経度 lam0 から地図距離 x [m] 離れた位置における経度 [rad].
!  ただし, x は東方向に正, 西方向に負を与えれば計算可能.
  use Phys_Const
  implicit none
  double precision, intent(in) :: x     ! 基準経度 lam0 からの空間距離 [m] (東向きが正)
  double precision, intent(in) :: lam0  ! 基準経度 [rad]

  x2merlon_d=x/radius_dp+lam0

  return
end function

!---------------------------------

real function y2merlat_f(y,phi0)
!  基準緯度 phi0 から空間距離 y [m] 離れた位置における緯度 [rad].
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: y     ! 基準緯度 phi0 からの空間距離 [rad] (北向き正).
  real, intent(in) :: phi0  ! 基準緯度 [rad]

  y2merlat_f=2.0*atan(exp(y/radius)*abs(tan(0.25*pi+0.5*phi0)))-0.5*pi

  return
end function

!---------------------------------

double precision function y2merlat_d(y,phi0)
!  基準緯度 phi0 から空間距離 y [m] 離れた位置における緯度 [rad].
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: y     ! 基準緯度 phi0 からの空間距離 [rad] (北向き正).
  double precision, intent(in) :: phi0  ! 基準緯度 [rad]

  y2merlat_d=2.0d0*datan(dexp(y/radius_dp)*dabs(dtan(0.25d0*pi_dp+0.5d0*phi0)))-0.5d0*pi_dp

  return
end function

!---------------------------------
!---------------------------------

real function lon2merx_f(lam,lam0)
!  基準経度 lam0 から経度 lam までの地図距離 x [m].
!  ただし, x は東方向に正, 西方向に負を与えれば計算可能.
  use Phys_Const
  implicit none
  real, intent(in) :: lam   ! 計算したい経度 [rad]
  real, intent(in) :: lam0  ! 基準経度 [rad]

  lon2merx_f=radius*(lam-lam0)

  return
end function

!---------------------------------
!---------------------------------

double precision function lon2merx_d(lam,lam0)
!  基準経度 lam0 から経度 lam までの地図距離 x [m].
!  ただし, x は東方向に正, 西方向に負を与えれば計算可能.
  use Phys_Const
  implicit none
  double precision, intent(in) :: lam   ! 計算したい経度 [rad]
  double precision, intent(in) :: lam0  ! 基準経度 [rad]

  lon2merx_d=radius_dp*(lam-lam0)

  return
end function

!---------------------------------
!---------------------------------

real function lat2mery_f(phi,phi0)
!  基準緯度 phi0 から緯度 phi までの地図距離 y [m].
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: phi   ! 計算したい緯度 [rad]
  real, intent(in) :: phi0  ! 基準緯度 [rad]

  lat2mery_f=radius*log(abs(tan(0.25*pi+0.5*phi)/tan(0.25*pi+0.5*phi0)))

  return
end function

!---------------------------------
!---------------------------------

double precision function lat2mery_d(phi,phi0)
!  基準緯度 phi0 から緯度 phi までの地図距離 y [m].
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: phi   ! 計算したい緯度 [rad]
  double precision, intent(in) :: phi0  ! 基準緯度 [rad]

  lat2mery_d=radius_dp*dlog(dabs(dtan(0.25d0*pi_dp+0.5d0*phi)/dtan(0.25d0*pi_dp+0.5d0*phi0)))

  return
end function

!---------------------------------
!---------------------------------

real function ll2merdis_f(lambda1,phi1,lambda2,phi2,lambda0,phi0)
!  lambda1, phi1 から lambda2, phi2 までの距離 [m].
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: lambda1  ! 任意点 1 での経度 [rad]
  real, intent(in) :: phi1     ! 任意点 1 での緯度 [rad]
  real, intent(in) :: lambda2  ! 任意点 2 での経度 [rad]
  real, intent(in) :: phi2     ! 任意点 2 での緯度 [rad]
  real, intent(in) :: lambda0  ! 投影面座標の原点に対応する経度 [rad]
  real, intent(in) :: phi0     ! 投影面座標の原点に対応する緯度 [rad]

  real :: x1, x2, y1, y2

  x1=lon2merx_f( lambda1, lambda0 )
  x2=lon2merx_f( lambda2, lambda0 )
  y1=lat2mery_f( phi1, phi0 )
  y2=lat2mery_f( phi2, phi0 )

  ll2merdis_f=sqrt((x2-x1)**2+(y2-y1)**2)

  return
end function

!---------------------------------
!---------------------------------

double precision function ll2merdis_d(lambda1,phi1,lambda2,phi2,lambda0,phi0)
!  lambda1, phi1 から lambda2, phi2 までの距離 [m].
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: lambda1  ! 任意点 1 での経度 [rad]
  double precision, intent(in) :: phi1     ! 任意点 1 での緯度 [rad]
  double precision, intent(in) :: lambda2  ! 任意点 2 での経度 [rad]
  double precision, intent(in) :: phi2     ! 任意点 2 での緯度 [rad]
  double precision, intent(in) :: lambda0  ! 投影面座標の原点に対応する経度 [rad]
  double precision, intent(in) :: phi0     ! 投影面座標の原点に対応する緯度 [rad]

  double precision :: x1, x2, y1, y2

  x1=lon2merx_d( lambda1, lambda0 )
  x2=lon2merx_d( lambda2, lambda0 )
  y1=lat2mery_d( phi1, phi0 )
  y2=lat2mery_d( phi2, phi0 )

  ll2merdis_d=dsqrt((x2-x1)**2+(y2-y1)**2)

  return
end function

!---------------------------------
! 2. 以降はランベルト座標
!---------------------------------

subroutine xy2lamll_f( x, y, lambda, phi, lambda0, phi0, phi1, phi2,  &
  &                    nopt, uopt, ropt )
!  基準経度 lambda0, 基準緯度 phi1, phi2 の投影面において,
!  経度 lambda0, 緯度 phi0 を地図座標の原点としたとき,
!  地図座標 (x, y) [m] での緯度経度 [rad].
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: x          ! 地図座標第一成分 [m]
  real, intent(in) :: y          ! 地図座標第二成分 [m]
  real, intent(inout) :: lambda  ! (x,y) に対応する経度 [rad]
  real, intent(inout) :: phi     ! (x,y) に対応する緯度 [rad]
  real, intent(in) :: lambda0    ! 基準経度 [rad]
  real, intent(in) :: phi0       ! 基準緯度 [rad]
  real, intent(in) :: phi1       ! ランベルト投影基準緯度 [rad]
  real, intent(in) :: phi2       ! ランベルト投影基準緯度 [rad]
  real, intent(in), optional :: nopt       ! ncoe in Lambert_coe_f
  real, intent(in), optional :: uopt(0:2)  ! uphi(0:2) in Lambert_coe_f
  real, intent(in), optional :: ropt       ! rhophi0 in Lambert_coe_f

  real :: ncoe, u1, up, cos1, cos2, tan1, tan2, tan0, w0

  cos1=cos(phi1)
  cos2=cos(phi2)

  if(present(nopt))then
     tan0=uopt(0)
     tan1=uopt(1)
     tan2=uopt(2)
     ncoe=nopt
     w0=ropt
  else
     tan0=tan(0.25*pi+0.5*phi0)
     tan1=tan(0.25*pi+0.5*phi1)
     tan2=tan(0.25*pi+0.5*phi2)
     ncoe=log(cos1/cos2)/log(tan2/tan1)
     w0=radius*(cos1/ncoe)*(exp(ncoe*log(tan1/tan0)))
  end if

  lambda=lambda0+atan(-x/(y-w0))/ncoe
  phi=2.0*atan(exp((1.0/ncoe)*log(radius*cos1/(ncoe*sqrt(x**2+(y-w0)**2))))*tan1)-0.5*pi

end subroutine

!---------------------------------
!---------------------------------

subroutine xy2lamll_d( x, y, lambda, phi, lambda0, phi0, phi1, phi2,  &
  &                    nopt, uopt, ropt )
!  基準経度 lambda0, 基準緯度 phi1, phi2 の投影面において,
!  経度 lambda0, 緯度 phi0 を地図座標の原点としたとき,
!  地図座標 (x, y) [m] での緯度経度 [rad].
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: x          ! 地図座標第一成分 [m]
  double precision, intent(in) :: y          ! 地図座標第二成分 [m]
  double precision, intent(inout) :: lambda  ! (x,y) に対応する経度 [rad]
  double precision, intent(inout) :: phi     ! (x,y) に対応する緯度 [rad]
  double precision, intent(in) :: lambda0    ! 基準経度 [rad]
  double precision, intent(in) :: phi0       ! 基準緯度 [rad]
  double precision, intent(in) :: phi1       ! ランベルト投影基準緯度 [rad]
  double precision, intent(in) :: phi2       ! ランベルト投影基準緯度 [rad]
  double precision, intent(in), optional :: nopt       ! ncoe in Lambert_coe_f
  double precision, intent(in), optional :: uopt(0:2)  ! uphi(0:2) in Lambert_coe_f
  double precision, intent(in), optional :: ropt       ! rhophi0 in Lambert_coe_f

  double precision :: ncoe, u1, up, cos1, cos2, tan1, tan2, tan0, w0

  cos1=cos(phi1)
  cos2=cos(phi2)

  if(present(nopt))then
     tan0=uopt(0)
     tan1=uopt(1)
     tan2=uopt(2)
     ncoe=nopt
     w0=ropt
  else
     tan0=dtan(0.25d0*pi_dp+0.5d0*phi0)
     tan1=dtan(0.25d0*pi_dp+0.5d0*phi1)
     tan2=dtan(0.25d0*pi_dp+0.5d0*phi2)
     ncoe=dlog(cos1/cos2)/dlog(tan2/tan1)
     w0=radius_dp*(cos1/ncoe)*(dexp(ncoe*dlog(tan1/tan0)))
  end if

  lambda=lambda0+datan(-x/(y-w0))/ncoe
  phi=2.0d0*datan(dexp((1.0d0/ncoe)*dlog(radius_dp*cos1/(ncoe*dsqrt(x**2+(y-w0)**2))))*tan1)-0.5d0*pi_dp

end subroutine

!---------------------------------
!---------------------------------

subroutine ll2lamxy_f( lambda, phi, x, y, lambda0, phi0, phi1, phi2,  &
  &                    nopt, uopt, ropt )
!  基準経度 lambda0, 基準緯度 phi1, phi2 の投影面において,
!  経度 lambda0, 緯度 phi0 を地図座標の原点としたとき,
!  球面上での緯度経度 (phi,lambda) が対応する地図座標 (x, y) [m].
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: lambda   ! (x,y) に対応する経度 [rad]
  real, intent(in) :: phi      ! (x,y) に対応する緯度 [rad]
  real, intent(inout) :: x     ! 地図座標第一成分 [m]
  real, intent(inout) :: y     ! 地図座標第二成分 [m]
  real, intent(in) :: lambda0  ! 基準経度 [rad]
  real, intent(in) :: phi0       ! 基準緯度 [rad]
  real, intent(in) :: phi1       ! ランベルト投影基準緯度 [rad]
  real, intent(in) :: phi2       ! ランベルト投影基準緯度 [rad]
  real, intent(in), optional :: nopt       ! ncoe in Lambert_coe_f
  real, intent(in), optional :: uopt(0:2)  ! uphi(0:2) in Lambert_coe_f
  real, intent(in), optional :: ropt       ! rhophi0 in Lambert_coe_f

  real :: ncoe, u1, up, cos1, cos2, tan1, tan2, tan0, tanx, w0, w

  cos1=cos(phi1)
  cos2=cos(phi2)

  if(present(nopt))then
     tan0=uopt(0)
     tan1=uopt(1)
     tan2=uopt(2)
     ncoe=nopt
     w0=ropt
  else
     tan0=tan(0.25*pi+0.5*phi0)
     tan1=tan(0.25*pi+0.5*phi1)
     tan2=tan(0.25*pi+0.5*phi2)
     ncoe=log(cos1/cos2)/log(tan2/tan1)
     w0=radius*(cos1/ncoe)*(exp(ncoe*log(tan1/tan0)))
  end if

  tanx=tan(0.25*pi+0.5*phi)
  w=radius*(cos1/ncoe)*(exp(ncoe*log(tan1/tanx)))

  x=w*sin(ncoe*(lambda-lambda0))
  y=w0-w*cos(ncoe*(lambda-lambda0))

end subroutine

!---------------------------------
!---------------------------------

subroutine ll2lamxy_d( lambda, phi, x, y, lambda0, phi0, phi1, phi2,  &
  &                    nopt, uopt, ropt )
!  基準経度 lambda0, 基準緯度 phi1, phi2 の投影面において,
!  経度 lambda0, 緯度 phi0 を地図座標の原点としたとき,
!  球面上での緯度経度 (phi,lambda) が対応する地図座標 (x, y) [m].
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: lambda   ! (x,y) に対応する経度 [rad]
  double precision, intent(in) :: phi      ! (x,y) に対応する緯度 [rad]
  double precision, intent(inout) :: x     ! 地図座標第一成分 [m]
  double precision, intent(inout) :: y     ! 地図座標第二成分 [m]
  double precision, intent(in) :: lambda0  ! 基準経度 [rad]
  double precision, intent(in) :: phi0       ! 基準緯度 [rad]
  double precision, intent(in) :: phi1       ! ランベルト投影基準緯度 [rad]
  double precision, intent(in) :: phi2       ! ランベルト投影基準緯度 [rad]
  double precision, intent(in), optional :: nopt       ! ncoe in Lambert_coe_f
  double precision, intent(in), optional :: uopt(0:2)  ! uphi(0:2) in Lambert_coe_f
  double precision, intent(in), optional :: ropt       ! rhophi0 in Lambert_coe_f

  double precision :: ncoe, u1, up, cos1, cos2, tan1, tan2, tan0, tanx, w0, w

  cos1=dcos(phi1)
  cos2=dcos(phi2)

  if(present(nopt))then
     tan0=uopt(0)
     tan1=uopt(1)
     tan2=uopt(2)
     ncoe=nopt
     w0=ropt
  else
     tan0=dtan(0.25d0*pi_dp+0.5d0*phi0)
     tan1=dtan(0.25d0*pi_dp+0.5d0*phi1)
     tan2=dtan(0.25d0*pi_dp+0.5d0*phi2)
     ncoe=dlog(cos1/cos2)/dlog(tan2/tan1)
     w0=radius_dp*(cos1/ncoe)*(dexp(ncoe*dlog(tan1/tan0)))
  end if

  tanx=dtan(0.25d0*pi_dp+0.5d0*phi)

  w=radius_dp*(cos1/ncoe)*(dexp(ncoe*dlog(tan1/tanx)))

  x=w*dsin(ncoe*(lambda-lambda0))
  y=w0-w*dcos(ncoe*(lambda-lambda0))

end subroutine

!---------------------------------
!---------------------------------

real function ll2lamdis_f( lambda1, phi1, lambda2, phi2,  &
  &                        lambda0, phit1, phit2, phi0,  &
  &                        nopt, uopt, ropt )
!  基準緯度 phit1, phit2, 基準経度 lambda0 から地図座標の原点が
!  緯度phi0, 経度 lambda 0 としたとき, 緯度経度座標で定義される 2 点
!  (lambda1,phi1), (lambda2,phi2) 間の地図上での距離 [m].
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: lambda1 ! 求めたい点 1 の経度 [rad].
  real, intent(in) :: phi1    ! 求めたい点 1 の緯度 [rad].
  real, intent(in) :: lambda2 ! 求めたい点 2 の経度 [rad].
  real, intent(in) :: phi2    ! 求めたい点 2 の緯度 [rad].
  real, intent(in) :: lambda0 ! 基準経度かつ地図座標基準経度 [rad]
  real, intent(in) :: phit1   ! 基準緯度 1 [rad]
  real, intent(in) :: phit2   ! 基準緯度 2 [rad]
  real, intent(in) :: phi0    ! 地図座標基準緯度 0 [rad].
  real, intent(in), optional :: nopt       ! ncoe in Lambert_coe_f
  real, intent(in), optional :: uopt(0:2)  ! uphi(0:2) in Lambert_coe_f
  real, intent(in), optional :: ropt       ! rhophi0 in Lambert_coe_f
  real :: ncoe, tan1, tan2, tan0, w0, cos1, cos2, x1, y1, x2, y2

  cos1=cos(phit1)
  cos2=cos(phit2)

  if(present(nopt))then
     tan0=uopt(0)
     tan1=uopt(1)
     tan2=uopt(2)
     ncoe=nopt
     w0=ropt
  else
     tan0=tan(0.25*pi+0.5*phi0)
     tan1=tan(0.25*pi+0.5*phit1)
     tan2=tan(0.25*pi+0.5*phit2)
     ncoe=log(cos1/cos2)/log(tan2/tan1)
     w0=radius*(cos1/ncoe)*(exp(ncoe*log(tan1/tan0)))
  end if

  call ll2lamxy_f( lambda1, phi1, x1, y1, lambda0, phi0, phit1, phit2,  &
  &                nopt=ncoe, uopt=(/tan0,tan1,tan2/), ropt=w0 )
  call ll2lamxy_f( lambda2, phi2, x2, y2, lambda0, phi0, phit1, phit2,  &
  &                nopt=ncoe, uopt=(/tan0,tan1,tan2/), ropt=w0 )

  ll2lamdis_f=sqrt((x2-x1)**2+(y2-y1)**2)

  return

end function

!---------------------------------
!---------------------------------

double precision function ll2lamdis_d( lambda1, phi1, lambda2, phi2,  &
  &                                    lambda0, phit1, phit2, phi0,  &
  &                                    nopt, uopt, ropt )
!  基準緯度 phit1, phit2, 基準経度 lambda0 から地図座標の原点が
!  緯度phi0, 経度 lambda 0 としたとき, 緯度経度座標で定義される 2 点
!  (lambda1,phi1), (lambda2,phi2) 間の地図上での距離 [m].
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: lambda1 ! 求めたい点 1 の経度 [rad].
  double precision, intent(in) :: phi1    ! 求めたい点 1 の緯度 [rad].
  double precision, intent(in) :: lambda2 ! 求めたい点 2 の経度 [rad].
  double precision, intent(in) :: phi2    ! 求めたい点 2 の緯度 [rad].
  double precision, intent(in) :: lambda0 ! 基準経度かつ地図座標基準経度 [rad]
  double precision, intent(in) :: phit1   ! 基準緯度 1 [rad]
  double precision, intent(in) :: phit2   ! 基準緯度 2 [rad]
  double precision, intent(in) :: phi0    ! 地図座標基準緯度 0 [rad].
  double precision, intent(in), optional :: nopt       ! ncoe in Lambert_coe_f
  double precision, intent(in), optional :: uopt(0:2)  ! uphi(0:2) in Lambert_coe_f
  double precision, intent(in), optional :: ropt       ! rhophi0 in Lambert_coe_f
  double precision:: ncoe, tan1, tan2, tan0, w0, cos1, cos2, x1, y1, x2, y2

  cos1=dcos(phit1)
  cos2=dcos(phit2)

  if(present(nopt))then
     tan0=uopt(0)
     tan1=uopt(1)
     tan2=uopt(2)
     ncoe=nopt
     w0=ropt
  else
     tan0=dtan(0.25d0*pi_dp+0.5d0*phi0)
     tan1=dtan(0.25d0*pi_dp+0.5d0*phit1)
     tan2=dtan(0.25d0*pi_dp+0.5d0*phit2)
     ncoe=dlog(cos1/cos2)/dlog(tan2/tan1)
     w0=radius*(cos1/ncoe)*(dexp(ncoe*dlog(tan1/tan0)))
  end if

  call ll2lamxy_d( lambda1, phi1, x1, y1, lambda0, phi0, phit1, phit2,  &
  &                nopt=ncoe, uopt=(/tan0,tan1,tan2/), ropt=w0 )
  call ll2lamxy_d( lambda2, phi2, x2, y2, lambda0, phi0, phit1, phit2,  &
  &                nopt=ncoe, uopt=(/tan0,tan1,tan2/), ropt=w0 )

  ll2lamdis_d=dsqrt((x2-x1)**2+(y2-y1)**2)

  return

end function

!---------------------------------
!---------------------------------

logical function check_close( xposi, yposi, ival )
! xposi, yposi で囲まれた閉曲線領域内に ival 点があるかどうかをチェックする.
! これらはすべて格子点上で処理される.

  implicit none

  integer, intent(in) :: xposi(:)  ! 閉曲線の x 方向格子点番号
  integer, intent(in) :: yposi(size(xposi))  ! 閉曲線の y 方向格子点番号
  integer, intent(in) :: ival(2)  ! チェックする点の x, y 格子点番号
  integer :: ii, kx
  double precision :: total, dtheta, x1, y1, x2, y2, r1, r2
  double precision :: r1r2, r1dr2

  kx=size(xposi)

  if(xposi(1)/=xposi(kx).or.yposi(1)/=yposi(kx))then
     write(*,*) "*** ERROR (check_close) ***: xposi(1) is not xposi(nx). "
     stop
  end if

  total=0.0d0

  do ii=2,kx
     x1=dble(xposi(ii-1)-ival(1))
     y1=dble(yposi(ii-1)-ival(2))
     x2=dble(xposi(ii)-ival(1))
     y2=dble(yposi(ii)-ival(2))
     r1=dsqrt(x1**2+y1**2)
     r2=dsqrt(x2**2+y2**2)
     r1r2=r1*r2
     r1dr2=x1*x2+y1*y2

     if(r1==0.0.or.r2==0.0)then
        check_close=.true.
        return
     end if

     if(r1dr2>r1r2)then
        dtheta=dacos(1.0d0)
     else
        dtheta=dacos(r1dr2/r1r2)
     end if

     if(x1*y2-y1*x2>0.0d0)then
        total=total+dtheta
     else
        total=total-dtheta
     end if

  end do

  if(dabs(total)>0.1d0)then
     check_close=.true.
  else
     check_close=.false.
  end if

  return

end function check_close


!---------------------------------
!---------------------------------

subroutine vec2lam_f( lambda0, phi1, phi2, lambda, phi,  &
  &                   ui, vi, uo, vo, conv_flag, undef )
! 水平ベクトル成分を緯度経度座標から地図座標に変換する (ランベルト座標).
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: lambda0     ! ランベルト投影基準経度 [rad]
  real, intent(in) :: phi1        ! ランベルト投影基準緯度 1 [rad]
  real, intent(in) :: phi2        ! ランベルト投影基準緯度 2 [rad]
  real, intent(in) :: lambda(:,:) ! 速度が定義されている経度 [rad]
  real, intent(in) :: phi(:,:)    ! 速度が定義されている緯度 [rad]
  real, intent(in) :: ui(size(phi,1),size(phi,2))
                      ! 経度方向 or 地図座標 x 方向の速度成分 [m/s] (変換前)
  real, intent(in) :: vi(size(phi,1),size(phi,2))
                      ! 緯度方向 or 地図座標 y 方向の速度成分 [m/s] (変換前)
  real, intent(inout) :: uo(size(phi,1),size(phi,2))
                      ! 経度方向 or 地図座標 x 方向の速度成分 [m/s] (変換後)
  real, intent(inout) :: vo(size(phi,1),size(phi,2))
                      ! 緯度方向 or 地図座標 y 方向の速度成分 [m/s] (変換後)
  character(5), intent(in) :: conv_flag  ! 変換の方向
                              ! 'll2xy' = 緯度経度座標からランベルト投影座標へ
                              ! 'xy2ll' = ランベルト投影座標から緯度経度座標へ

  real, intent(in), optional :: undef   ! 未定義値

  integer :: ix, jy, i, j
  real :: rundef, cos1, cos2, tan1, tan2, ncoe, cosp, sinp

  ix=size(phi,1)
  jy=size(phi,2)

  cos1=cos(phi1)
  cos2=cos(phi2)
  tan1=tan(0.25*pi+0.5*phi1)
  tan2=tan(0.25*pi+0.5*phi2)
  ncoe=log(cos1/cos2)/log(tan2/tan1)

  if(present(undef))then

     uo=undef
     vo=undef

     if(conv_flag(1:5)=='ll2xy')then  ! ulon, vlat -> ux, vy

        do j=1,jy
           do i=1,ix
              if(ui(i,j)/=undef.and.vi(i,j)/=undef)then
                 sinp=sin(ncoe*(lambda(i,j)-lambda0))
                 cosp=cos(ncoe*(lambda(i,j)-lambda0))

                 uo(i,j)=ui(i,j)*cosp-vi(i,j)*sinp
                 vo(i,j)=ui(i,j)*sinp+vi(i,j)*cosp
              end if
           end do
        end do

     else if(conv_flag(1:5)=='xy2ll')then  ! ux, vy -> ulon, vlat

        do j=1,jy
           do i=1,ix
              if(ui(i,j)/=undef.and.vi(i,j)/=undef)then
                 sinp=sin(ncoe*(lambda(i,j)-lambda0))
                 cosp=cos(ncoe*(lambda(i,j)-lambda0))

                 uo(i,j)=ui(i,j)*cosp+vi(i,j)*sinp
                 vo(i,j)=-ui(i,j)*sinp+vi(i,j)*cosp
              end if
           end do
        end do

     end if

  else

     if(conv_flag(1:5)=='ll2xy')then  ! ulon, vlat -> ux, vy

        do j=1,jy
           do i=1,ix
              sinp=sin(ncoe*(lambda(i,j)-lambda0))
              cosp=cos(ncoe*(lambda(i,j)-lambda0))

              uo(i,j)=ui(i,j)*cosp-vi(i,j)*sinp
              vo(i,j)=ui(i,j)*sinp+vi(i,j)*cosp
           end do
        end do

     else if(conv_flag(1:5)=='xy2ll')then  ! ux, vy -> ulon, vlat

        do j=1,jy
           do i=1,ix
              sinp=sin(ncoe*(lambda(i,j)-lambda0))
              cosp=cos(ncoe*(lambda(i,j)-lambda0))

              uo(i,j)=ui(i,j)*cosp+vi(i,j)*sinp
              vo(i,j)=-ui(i,j)*sinp+vi(i,j)*cosp
           end do
        end do

     end if

  end if

end subroutine

!---------------------------------
!---------------------------------

subroutine vec2lam_d( lambda0, phi1, phi2, lambda, phi,  &
  &                   ui, vi, uo, vo, conv_flag, undef )
! 水平ベクトル成分を緯度経度座標から地図座標に変換する (ランベルト座標).
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: lambda0     ! ランベルト投影基準経度 [rad]
  double precision, intent(in) :: phi1        ! ランベルト投影基準緯度 1 [rad]
  double precision, intent(in) :: phi2        ! ランベルト投影基準緯度 2 [rad]
  double precision, intent(in) :: phi(:,:)    ! 速度が定義されている緯度 [rad]
  double precision, intent(in) :: lambda(:,:) ! 速度が定義されている経度 [rad]
  double precision, intent(in) :: ui(size(phi,1),size(phi,2))
                      ! 経度方向 or 地図座標 x 方向の速度成分 [m/s] (変換前)
  double precision, intent(in) :: vi(size(phi,1),size(phi,2))
                      ! 緯度方向 or 地図座標 y 方向の速度成分 [m/s] (変換前)
  double precision, intent(inout) :: uo(size(phi,1),size(phi,2))
                      ! 経度方向 or 地図座標 x 方向の速度成分 [m/s] (変換後)
  double precision, intent(inout) :: vo(size(phi,1),size(phi,2))
                      ! 緯度方向 or 地図座標 y 方向の速度成分 [m/s] (変換後)
  character(5), intent(in) :: conv_flag  ! 変換の方向
                              ! 'll2xy' = 緯度経度座標からランベルト投影座標へ
                              ! 'xy2ll' = ランベルト投影座標から緯度経度座標へ

  double precision, intent(in), optional :: undef   ! 未定義値

  integer :: ix, jy, i, j
  double precision :: rundef, cos1, cos2, tan1, tan2, ncoe, cosp, sinp

  ix=size(phi,1)
  jy=size(phi,2)

  cos1=dcos(phi1)
  cos2=dcos(phi2)
  tan1=dtan(0.25d0*pi_dp+0.5d0*phi1)
  tan2=dtan(0.25d0*pi_dp+0.5d0*phi2)
  ncoe=dlog(cos1/cos2)/dlog(tan2/tan1)

  if(present(undef))then

     uo=undef
     vo=undef

     if(conv_flag(1:5)=='ll2xy')then  ! ulon, vlat -> ux, vy

        do j=1,jy
           do i=1,ix
              if(ui(i,j)/=undef.and.vi(i,j)/=undef)then
                 sinp=dsin(ncoe*(lambda(i,j)-lambda0))
                 cosp=dcos(ncoe*(lambda(i,j)-lambda0))

                 uo(i,j)=ui(i,j)*cosp-vi(i,j)*sinp
                 vo(i,j)=ui(i,j)*sinp+vi(i,j)*cosp
              end if
           end do
        end do

     else if(conv_flag(1:5)=='xy2ll')then  ! ux, vy -> ulon, vlat

        do j=1,jy
           do i=1,ix
              if(ui(i,j)/=undef.and.vi(i,j)/=undef)then
                 sinp=dsin(ncoe*(lambda(i,j)-lambda0))
                 cosp=dcos(ncoe*(lambda(i,j)-lambda0))

                 uo(i,j)=ui(i,j)*cosp+vi(i,j)*sinp
                 vo(i,j)=-ui(i,j)*sinp+vi(i,j)*cosp
              end if
           end do
        end do

     end if

  else

     if(conv_flag(1:5)=='ll2xy')then  ! ulon, vlat -> ux, vy

        do j=1,jy
           do i=1,ix
              sinp=dsin(ncoe*(lambda(i,j)-lambda0))
              cosp=dcos(ncoe*(lambda(i,j)-lambda0))

              uo(i,j)=ui(i,j)*cosp-vi(i,j)*sinp
              vo(i,j)=ui(i,j)*sinp+vi(i,j)*cosp
           end do
        end do

     else if(conv_flag(1:5)=='xy2ll')then  ! ux, vy -> ulon, vlat

        do j=1,jy
           do i=1,ix
              sinp=dsin(ncoe*(lambda(i,j)-lambda0))
              cosp=dcos(ncoe*(lambda(i,j)-lambda0))

              uo(i,j)=ui(i,j)*cosp+vi(i,j)*sinp
              vo(i,j)=-ui(i,j)*sinp+vi(i,j)*cosp
           end do
        end do

     end if

  end if

end subroutine

!---------------------------------
!---------------------------------

subroutine lambert_coe_f( lambda0, phi0, phi1, phi2, ncoe, uphi, rhophi0 )
!  基準緯度 phi1, phi2, 基準経度 lambda0, 投影座標の原点基準緯度 phi0
!  の場合における, 各種 Lambert 系変数を計算する際の必要パラメータ
!  n, U, rho を求める.
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: lambda0        ! 基準経度かつ地図座標基準経度 [rad]
  real, intent(in) :: phi1           ! 基準緯度 1 [rad]
  real, intent(in) :: phi2           ! 基準緯度 2 [rad]
  real, intent(in) :: phi0           ! 地図座標基準緯度 0 [rad].
  real, intent(inout) :: ncoe        ! 縮尺因子 N
  real, intent(inout) :: uphi(0:2)   ! 正接係数 U(phi0,phi1,phi2)
  real, intent(inout) :: rhophi0     ! 扇半径 r(phi0)

  real :: cos1, cos2

  cos1=cos(phi1)
  cos2=cos(phi2)

  uphi(0)=tan(0.25*pi+0.5*phi0)
  uphi(1)=tan(0.25*pi+0.5*phi1)
  uphi(2)=tan(0.25*pi+0.5*phi2)
  ncoe=log(cos1/cos2)/log(uphi(2)/uphi(1))
  rhophi0=radius*(cos1/ncoe)*exp(ncoe*log(uphi(1)/uphi(0)))

end subroutine

!---------------------------------
!---------------------------------

subroutine lambert_coe_d( lambda0, phi0, phi1, phi2, ncoe, uphi, rhophi0 )
!  基準緯度 phi1, phi2, 基準経度 lambda0, 投影座標の原点基準緯度 phi0
!  の場合における, 各種 Lambert 系変数を計算する際の必要パラメータ
!  n, U, rho を求める.
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: lambda0        ! 基準経度かつ地図座標基準経度 [rad]
  double precision, intent(in) :: phi1           ! 基準緯度 1 [rad]
  double precision, intent(in) :: phi2           ! 基準緯度 2 [rad]
  double precision, intent(in) :: phi0           ! 地図座標基準緯度 0 [rad].
  double precision, intent(inout) :: ncoe        ! 縮尺因子 N
  double precision, intent(inout) :: uphi(0:1)   ! 正接係数 U(phi0,phi1)
  double precision, intent(inout) :: rhophi0     ! 扇半径 r(phi0)

  double precision :: cos1, cos2, tanx

  cos1=dcos(phi1)
  cos2=dcos(phi2)
  tanx=dtan(0.25d0*pi_dp+0.5d0*phi2)

  uphi(0)=dtan(0.25d0*pi_dp+0.5d0*phi0)
  uphi(1)=dtan(0.25d0*pi_dp+0.5d0*phi1)
  ncoe=dlog(cos1/cos2)/dlog(tanx/uphi(1))
  rhophi0=radius_dp*(cos1/ncoe)*dexp(ncoe*dlog(uphi(1)/uphi(0)))

end subroutine

!---------------------------------
!---------------------------------


end module
