! netcdf_slice.f90 - abstraction of partial array access
! vi: set sw=4 ts=8:
! Copyright (C) TOYODA Eizi, 2000.  All rights reserved.

! 
!
! netcdf_slice  netCDF 饤֥꤬󶡤ʬϤ
! ݲǤ롣桼ʬϤΰꤷ
! б netCDF 饤֥ƤӽФˡ
! NC_SLICE ¤ΤƤϤϤԤ
! 
! ѿλļ 7 ʲǤʤФʤʤ

module netcdf_slice

    use iso_varying_string
    use netcdf_file
    use netcdf_variable
    implicit none

    integer, parameter::	NC_SLICE_DIMS = 7

    type NC_SLICE_BODY
	private
	type(NC_DIMENSION)::			dim
	! ϰ
	integer::				start, stride, count
	! ƥ졼
	integer::				size
    end type

    type NC_SLICE
	private
	type(NC_SLICE_BODY), pointer::		body
    end type

    type NC_LIMIT
	private
	type(NC_SLICE_BODY), pointer::		slices(:)
	logical::				scalar
    end type

    interface assignment(=)
	module procedure NetcdfLimitSetWholeVariable
	module procedure NetcdfSliceSetIndex
	module procedure NetcdfSliceSetList
    end interface

    interface new
	module procedure NetcdfLimitGetWholeVariable
	module procedure NetcdfSliceDimension
    end interface

    interface Slice
	module procedure NetcdfSliceDimension
	module procedure NetcdfSliceDimensionOrd
	module procedure NetcdfSliceDimensionByName
	module procedure NetcdfSliceDimensionSetIndex
	module procedure NetcdfSliceDimensionOrdSetIndex
    end interface

    interface WholeVariable
	module procedure NetcdfLimitGetWholeVariable
    end interface

    interface operator(.error.)
	module procedure NetcdfLimitError
	module procedure NetcdfSliceError
    end interface

    interface Dispose; module procedure NetcdfLimitDispose; end interface

    interface Size
	module procedure NetcdfLimitSize, NetcdfSliceSize
    end interface

    interface String; module procedure NetcdfLimitString; end interface

    interface Count
	module procedure NetcdfLimitCount, NetcdfSliceSize
    end interface

    interface Next; module procedure NetcdfLimitNext; end interface

    interface End; module procedure NetcdfSliceEnd; end interface

    interface Start
	module procedure NetcdfLimitStart
	module procedure NetcdfSliceStart
    end interface

    interface Stride
	module procedure NetcdfLimitStride
	module procedure NetcdfSliceStride
    end interface

contains

    !
    ! --- Limit κѴ ---
    !

    ! generic WholeVariable/new
    function NetcdfLimitGetWholeVariable(var) result(result)
	type(NC_LIMIT)::			result
	type(NC_VARIABLE), intent(in)::		var
	type(NC_DIMENSION), pointer::		dims(:)
	integer::				i, vdims
    continue
	vdims = DimensionsNumber(var)
	if (vdims < 0) then
	    nullify(result%slices)
	    result%scalar = .false.
	    return
	else if (vdims == 0) then
	    nullify(result%slices)
	    result%scalar = .true.
	    return
	endif
	dims => Dimensions(var)
	result%scalar = .false.
	allocate(result%slices(vdims))
	result%slices(:)%start = 1
	result%slices(:)%stride = 1
	do, i = 1, vdims
	    result%slices(i)%dim = dims(i)
	    result%slices(i)%size = len(dims(i))
	    result%slices(i)%count = result%slices(i)%size
	enddo
	deallocate(dims)
    end function

    ! generic dispose
    subroutine NetcdfLimitDispose(limit)
	type(NC_LIMIT), intent(inout)::		limit
    continue
	limit%scalar = .false.
	if (.not. associated(limit%slices)) return
	deallocate(limit%slices)
	nullify(limit%slices)
    end subroutine

    !
    ! --- GENERIC Slice ---
    !

    function NetcdfSliceDimension(limit, dim) result(result)
	type(NC_SLICE)::			result
	type(NC_LIMIT), intent(in)::		limit
	type(NC_DIMENSION), intent(in)::	dim
	integer::		i
    continue
	if (.not. associated(limit%slices)) goto 999
	do, i = 1, size(limit%slices)
	    if (limit%slices(i)%dim == dim) then
		result%body => limit%slices(i)
		return
	    endif
	enddo
999	continue
	nullify(result%body)
    end function

    function NetcdfSliceDimensionOrd(limit, dimord) result(result)
	type(NC_SLICE)::			result
	type(NC_LIMIT), intent(in)::		limit
	integer, intent(in)::			dimord
	integer::		i
    continue
	if (.not. associated(limit%slices)) goto 999
	if (dimord > size(limit%slices)) goto 999
	result%body => limit%slices(dimord)
	return
999	continue
	nullify(result%body)
    end function

    function NetcdfSliceDimensionByName(limit, dimname) result(result)
	type(NC_SLICE)::			result
	type(NC_LIMIT), intent(in)::		limit
	character(len = *), intent(in)::	dimname
	type(NC_DIMENSION)::			dim
    continue
	if (.not. associated(limit%slices)) goto 999
	dim = Dimension(limit%slices(1)%dim%file, dimname)
	if (.error. dim) goto 999
	result = Slice(limit, dim)
	return
999	continue
	nullify(result%body)
	return
    end function

    logical function NetcdfSliceDimensionSetIndex(limit, dimname, start, &
    & stride, count) result(result)
	type(NC_LIMIT), intent(in)::		limit
	character(len = *), intent(in)::	dimname
	integer, intent(in)::			start
	integer, intent(in), optional::		stride, count
	type(NC_SLICE)::			it
    continue
	it = Slice(limit, dimname)
	if (.not. associated(it%body)) then
	    result = .FALSE.; return
	endif
	if (present(count) .and. present(stride)) then
	    it = (/start, count, stride/)
	else if (present(count)) then
	    it = (/start, count/)
	else
	    it = start
	endif
	result = .TRUE.
    end function

    logical function NetcdfSliceDimensionOrdSetIndex(limit, dimord, start, &
    & stride, count) result(result)
	type(NC_LIMIT), intent(in)::		limit
	integer, intent(in)::			dimord
	integer, intent(in)::			start
	integer, intent(in), optional::		stride, count
	type(NC_SLICE)::			it
    continue
	it = Slice(limit, dimord)
	if (.not. associated(it%body)) then
	    result = .FALSE.; return
	endif
	if (present(count) .and. present(stride)) then
	    it = (/start, count, stride/)
	else if (present(count)) then
	    it = (/start, count/)
	else
	    it = start
	endif
	result = .TRUE.
    end function

    !
    ! --- GENERIC ASSIGNMENT ---
    !

    ! generic assignment(NC_LIMIT = NC_VARIABLE)
    subroutine NetcdfLimitSetWholeVariable(limit, var)
	type(NC_LIMIT), intent(inout)::		limit
	type(NC_VARIABLE), intent(in)::		var
    continue
	limit = WholeVariable(var)
    end subroutine

    ! generic assignment(NC_SLICE = integer)
    subroutine NetcdfSliceSetIndex(slice, idx)
	type(NC_SLICE), intent(inout)::		slice
	integer, intent(in)::			idx
    continue
	if (.not. associated(slice%body)) return
	slice%body%start = idx
	slice%body%count = 1
    end subroutine

    ! generic assignment(NC_SLICE = integer(:))
    subroutine NetcdfSliceSetList(slice, idx)
	type(NC_SLICE), intent(inout)::		slice
	integer, intent(in)::			idx(:)
	type(NC_SLICE_BODY), pointer::		body
    continue
	if (.not. associated(slice%body)) return
	body => slice%body
	body%start = 1
	if (size(idx) > 1) body%start = idx(1)
	body%count = 1
	if (size(idx) > 2) body%count = idx(2)
	body%stride = 1
	if (size(idx) > 3) body%stride = idx(3)
    end subroutine

    !
    ! --- ITERATOR ---
    !

    ! generic next
    ! limit disposed when loop is done.
    !
    subroutine NetcdfLimitNext(limit)
	type(NC_LIMIT), intent(inout)::		limit
	integer::		i, upper
	type(NC_SLICE_BODY), pointer::		it
    continue
	if (.not. associated(limit%slices)) return
	do, i = 1, size(limit%slices)
	    it => limit%slices(i)
	    upper = it%start + it%stride * it%count
	    if (upper < it%size .or. it%size == 0) then
		it%start = upper + it%stride
		return
	    endif
	enddo
	call Dispose(limit)
    end subroutine

    !
    ! --- MISC. FUNCTION ---
    !

    ! generic start
    integer function NetcdfSliceStart(slice) result(result)
	type(NC_SLICE), intent(in)::		slice
    continue
	if (.not. associated(slice%body)) then
	    result = 0; return
	endif
	result = slice%body%start
    end function

    ! generic end
    integer function NetcdfSliceEnd(slice) result(result)
	type(NC_SLICE), intent(in)::		slice
    continue
	if (.not.associated(slice%body)) then
	    result = 1; return
	endif
	result = slice%body%start + slice%body%stride * slice%body%count
    end function

    ! generic size
    integer function NetcdfSliceSize(slice) result(result)
	type(NC_SLICE), intent(in)::		slice
    continue
	if (.not.associated(slice%body)) then
	    result = 1; return
	endif
	result = slice%body%count
    end function

    ! generic stride
    integer function NetcdfSliceStride(slice) result(result)
	type(NC_SLICE), intent(in)::		slice
    continue
	if (.not.associated(slice%body)) then
	    result = 1; return
	endif
	result = slice%body%stride
    end function

    ! generic size
    integer function NetcdfLimitSize(limit) result(result)
	type(NC_LIMIT), intent(in)::		limit
	integer::		i, len
    continue
	if (.error. limit) then
	    result = 0; return
	endif
	if (limit%scalar) then
	    result = 1; return
	endif
	result = 1
	do, i = 1, size(limit%slices)
	    len = limit%slices(i)%count
	    if (len < 0) then
		result = 0; return
	    endif
	    if (len == 0) len = 1
	    result = result * len
	enddo
    end function

    function NetcdfLimitString(limit) result(result)
	type(VARYING_STRING)::			result
	type(NC_LIMIT), intent(in)::		limit
	integer::				i
	character(len = 80)::			buf
    continue
	if (.error. limit) then
	    result = 'limit(error)'; return
	endif
	if (limit%scalar) then
	    result = 'limit(scalar)'; return
	endif
	result = 'limit('
	do, i = 1, size(limit%slices)
	    write(unit=buf, fmt="('(', 3i4, ')')") limit%slices(i)%start, &
		& limit%slices(i)%count, limit%slices(i)%stride
	    result = result // trim(buf)
	enddo
	result = result // ')'
    end function

    function NetcdfLimitCount(limit) result(result)
	integer::				result(NC_SLICE_DIMS)
	type(NC_LIMIT), intent(in)::		limit
	integer::				i, defined
    continue
	if (.not. associated(limit%slices)) then
	    result(:) = 1; return
	endif
	defined = min(NC_SLICE_DIMS, size(limit%slices))
	do, i = 1, defined
	    result(i) = limit%slices(i)%count
	enddo
	result(defined + 1: ) = 1
    end function

    function NetcdfLimitStart(limit) result(result)
	integer::				result(NC_SLICE_DIMS)
	type(NC_LIMIT), intent(in)::		limit
	integer::				i, defined
    continue
	if (.not. associated(limit%slices)) then
	    result(:) = 1; return
	endif
	defined = min(NC_SLICE_DIMS, size(limit%slices))
	do, i = 1, defined
	    result(i) = limit%slices(i)%start
	enddo
	result(defined + 1: ) = 1
    end function

    function NetcdfLimitStride(limit) result(result)
	integer::				result(NC_SLICE_DIMS)
	type(NC_LIMIT), intent(in)::		limit
	integer::				i, defined
    continue
	if (.not. associated(limit%slices)) then
	    result(:) = 1; return
	endif
	defined = min(NC_SLICE_DIMS, size(limit%slices))
	do, i = 1, defined
	    result(i) = limit%slices(i)%stride
	enddo
	result(defined + 1: ) = 1
    end function

    ! generic operator(.error.)
    logical function NetcdfLimitError(limit) result(result)
	type(NC_LIMIT), intent(in)::		limit
	integer::		i
    continue
	if (limit%scalar) then
	    result = .FALSE.; return
	endif
	if (.not. associated(limit%slices)) then
	    result = .TRUE.; return
	endif
	result = .FALSE.
	do, i = 1, size(limit%slices)
	    if (limit%slices(i)%size < 0) then
		result = .TRUE.; return
	    endif
	enddo
    end function

    ! generic operator(.error.)
    logical function NetcdfSliceError(slice) result(result)
	type(NC_SLICE), intent(in)::		slice
    continue
	if (.not. associated(slice%body)) then
	    result = .TRUE.; return
	endif
	result = (slice%body%size < 0)
    end function

end module
