! gtool_variable.f90 - gtool4 file interface
! Copyright (C) by TOYODA Eizi, 2000.  All rights reserved.
! vi: set sw=4 ts=8:

module gtool_variable 

    use netcdf
    use dc_string
    implicit none

    type GT_VARIABLE
	type(NC_VARIABLE), pointer::		ncvar;
	type(NC_DIMENSION), pointer::		ncdim;
    end type

    interface Create
	module procedure GtoolVariableCreate
    end interface

    interface Open
	module procedure GtoolVariableOpen
    end interface

    interface Dimension
	module procedure GtoolVariableToDimension
    end interface

    interface Close
	module procedure GtoolVariableClose
    end interface

    integer, parameter:: GT_VARIABLE_UNKNOWN = 0
    integer, parameter:: GT_VARIABLE_NETCDF = 1
    integer, parameter:: GT_VARIABLE_NETCDF_DIM = 2
    integer, parameter:: GT_VARIABLE_GTOOL3 = 3

    interface Class
	module procedure GtoolVariableClass
    end interface

    interface Name
	module procedure GtoolVariableName
    end interface

    interface Fullname
	module procedure GtoolVariableFullname
    end interface

    interface MaxVal
	module procedure GtoolVariableMaxVal
    end interface

    interface MinVal
	module procedure GtoolVariableMinVal
    end interface

contains

    ! internal
    ! 
    subroutine FVASplit(fullname, file, var, attr)
	type(VARYING_STRING), intent(in)::	fullname
	type(VARYING_STRING), intent(out), optional::	file, var, attr
	type(VARYING_STRING):: varpart
	integer:: hash, colon
    continue
	hash = index(char(fullname), '#', back=.TRUE.)
	if (hash == 0) then
	    if (present(file)) file = fullname
	    if (present(var)) var = ''
	    if (present(attr)) attr = ''
	    return
	endif
	if (present(file)) file = extract(fullname, 1, hash - 1)
	varpart = extract(fullname, hash + 1)
	colon = index(var, ':')
	if (colon == 0) then
	    if (present(var)) var = varpart
	    if (present(attr)) attr = ''
	    varpart = ''
	    return
	endif
	if (present(var)) var = extract(varpart, 1, colon - 1)
	if (present(attr)) attr = extract(varpart, colon + 1)
	varpart = ''
    end subroutine

    subroutine GtoolVariableCreate(var, fullname, fail)
	type(GT_VARIABLE), intent(inout)::		var
	type(VARYING_STRING), intent(in)::		fullname
	type(VARYING_STRING):: 		filename, varname
	type(NC_FILE)::					ncfile
	logical, intent(out), optional:: fail
    continue
	if (present(fail)) fail = .FALSE.
	! try to open netCDF file
	call FVASplit(fullname, filename, varname)
	allocate(var%ncvar)
	ncfile = NetcdfOpenFile(char(filename), writable=.TRUE.)
	if (.error. ncfile) ncfile = NetcdfCreateFile(char(fullname))
	if (.not..error. ncfile) then
	    var%ncvar = Variable(ncfile, char(varname))
	    if (.not..error. (var%ncvar)) then
		return
	    endif
	endif
	! if opening netCDF fail
	deallocate(var%ncvar)
	nullify(var%ncvar)
	if (present(fail)) then
	    fail = .TRUE.
	else
	    call NetcdfAssert()
	endif
    end subroutine

    ! fake
    type(VARYING_STRING) function FirstVariable(file) result(result)
	type(NC_FILE), intent(in)::		file
	integer:: i
	type(NC_VARIABLE):: v
    continue
	v%file = file
	v%id = 1
	result = Name(v)
    end function

    subroutine GtoolVariableOpen(var, fullname, fail)
	type(GT_VARIABLE), intent(inout)::		var
	type(VARYING_STRING), intent(in)::		fullname
	type(VARYING_STRING):: filename, varname
	type(NC_FILE)::					ncfile
	logical, intent(out), optional:: fail
    continue
	if (present(fail)) fail = .FALSE.
	! try to open netCDF file
	call FVASplit(fullname, filename, varname)
	allocate(var%ncvar)
	ncfile = NetcdfOpenFile(char(filename))
	if (.not..error. ncfile) then
	    if (varname == '') varname = FirstVariable(ncfile)
	    var%ncvar = Variable(ncfile, char(varname))
	    if (.not..error. var%ncvar) then
		return
	    endif
	endif
	! if opening netCDF fail
	deallocate(var%ncvar)
	nullify(var%ncvar)
	if (present(fail)) then
	    fail = .TRUE.
	else
	    call NetcdfAssert()
	endif
    end subroutine

    function GtoolVariableToDimension(var, dim_ord) result(result)
	type(GT_VARIABLE):: result
	type(GT_VARIABLE), intent(in):: var
	integer, intent(in):: dim_ord
	type(NC_DIMENSION), pointer:: alldims(:)
	type(NC_VARIABLE):: dimvar
    continue
	nullify(result%ncvar)
	alldims => Dimensions(var%ncvar)
	dimvar = Variable(var%ncvar%file, char(Name(alldims(dim_ord))))
	if (.error. dimvar) then
    	    allocate(result%ncdim)
	    result%ncdim = alldims(dim_ord)
	else
	    allocate(result%ncvar)
	    result%ncvar = dimvar
	endif
	deallocate(alldims)
    end function

    integer function GtoolVariableClass(var) result(result)
	type(GT_VARIABLE), intent(in)::		var
    continue
	if (associated(var%ncvar)) then
	    result = GT_VARIABLE_NETCDF; return
	else if (associated(var%ncdim)) then
	    result = GT_VARIABLE_NETCDF_DIM; return
	endif
	result = GT_VARIABLE_UNKNOWN
    end function

    subroutine GtoolVariableClose(var, fail)
	type(GT_VARIABLE), intent(inout)::		var
	logical, optional, intent(out):: fail
    continue
	if (Class(var) == GT_VARIABLE_NETCDF) then
	    if (associated(var%ncvar)) deallocate(var%ncvar)
	    if (present(fail)) fail = .FALSE.
	else if (Class(var) == GT_VARIABLE_NETCDF_DIM) then
	    deallocate(var%ncdim)
	    if (present(fail)) fail = .FALSE.
	else
	    if (present(fail)) then
		fail = .TRUE.
	    else
		print "('Close: invalid GT_VARIABLE handle')"
		stop
	    endif
	endif
    end subroutine

    type(VARYING_STRING) function GtoolVariableName(var) result(result)
	type(GT_VARIABLE), intent(in):: var
    continue
	if (Class(var) == GT_VARIABLE_NETCDF) then
	    result = Name(var%ncvar)
	else if (Class(var) == GT_VARIABLE_NETCDF_DIM) then
	    result = Name(var%ncdim)
	else
	    result = ''
	endif
    end function

    type(VARYING_STRING) function GtoolVariableFullname(var) result(result)
	type(GT_VARIABLE), intent(in):: var
    continue
	if (Class(var) == GT_VARIABLE_NETCDF) then
	    result = Fullname(var%ncvar)
	else if (Class(var) == GT_VARIABLE_NETCDF_DIM) then
	    result = Name(var%ncdim)
	else
	    result = ''
	endif
    end function

    real function GtoolVariableMaxVal(var) result(result)
	type(GT_VARIABLE), intent(in):: var
	real, pointer:: buffer(:, :, :, :, :, :, :)
    continue
	if (Class(var) == GT_VARIABLE_NETCDF) then
	    buffer => get_real(var%ncvar)
	    if (.not.associated(buffer)) then
		result = 1.0; return
	    endif
	    result = maxval(buffer)
	    deallocate(buffer)
	else if (Class(var) == GT_VARIABLE_NETCDF_DIM) then
	    result = len(var%ncdim)
	else
	    result = 0
	endif
    end function

    real function GtoolVariableMinVal(var) result(result)
	type(GT_VARIABLE), intent(in):: var
	real, pointer:: buffer(:, :, :, :, :, :, :)
    continue
	if (Class(var) == GT_VARIABLE_NETCDF) then
	    buffer => get_real(var%ncvar)
	    if (.not.associated(buffer)) then
		result = 0; return
	    endif
	    result = minval(buffer)
	    deallocate(buffer)
	else if (Class(var) == GT_VARIABLE_NETCDF_DIM) then
	    result = 1
	else
	    result = 0
	endif
    end function

end module
