!= 補助的な変数を計算するサブルーチン・関数群
!
!= Subroutines and functions for calculating auxiliary variables
!
! Authors::   Yoshiyuki O. Takahashi, Yasuhiro MORIKAWA, Yukiko YAMADA
! Version::   $Id: auxiliary.f90,v 1.18 2015/02/06 11:25:12 yot Exp $ 
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008-2009. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module auxiliary
  !
  != 補助的な変数を計算するサブルーチン・関数群
  !
  != Subroutines and functions for calculating auxiliary variables
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! このモジュールでは, 補助的な変数を計算するサブルーチンや関数を提供します. 
  !
  ! This module provides subroutines and functions for calculating auxiliary variables. 
  !
  !== Procedures List
  !
  ! AuxVars          :: 温度の半整数 $ \sigma $ レベルの補間と
  !                     気圧と高度の算出
  ! AuxVarsInit      ::
  ! AuxVarsFinalize  :: 終了処理 (モジュール内部の変数の割り付け解除)
  ! ---------------  :: ---------------
  ! AuxVars          :: Interpolate temperature on half $ \sigma $ level, 
  !                     and calculate pressure and height
  ! AuxVarsInit      ::
  ! AuxVarsFinalize  :: Termination (deallocate variables in this module)
  !
  !--
  !== NAMELIST
  !
  ! NAMELIST#auxiliary_nml
  !++

  ! モジュール引用 ; USE statements
  !

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数. 
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数. 
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数. 
                             ! Number of vertical level

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING     ! 文字列.       Strings. 

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public :: AuxVars
  public :: AuxVarsFinalize
  public :: AuxVarsInit

  ! 公開変数
  ! Public variables
  !
  real(DP), save, public:: RefPress
                              ! 基準気圧. 
                              ! Reference air pressure

  ! 非公開変数
  ! Private variables
  !
  logical, save :: auxiliary_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag

  real(DP), save, allocatable:: z_TempInpolM (:)
                              ! 温度の補間用の係数. 
                              ! Coefficient for interpolation of temperature
  real(DP), save, allocatable:: z_TempInpol (:)
                              ! 温度の補間用の係数. 
                              ! Coefficient for interpolation of temperature

  character(*), parameter:: module_name = 'auxiliary'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: auxiliary.f90,v 1.18 2015/02/06 11:25:12 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

contains

  !--------------------------------------------------------------------------------------

  recursive subroutine AuxVars(                           &
    & xy_Ps, xyz_Temp, xyz_QVap,                          &  ! (in )
    & xyr_Temp, xyz_VirTemp, xyr_VirTemp, xy_SurfVirTemp, &  ! (out) optional
    & xyz_Press,  xyr_Press,                              &  ! (out) optional
    & xy_SurfHeight, xy_SurfTemp,                         &  ! (in ) optional
    & xyz_Height, xyr_Height,                             &  ! (out) optional
    & xyz_Exner,  xyr_Exner                               &  ! (out) optional
    & )
    !
    ! 温度の半整数σレベルの補間, 気圧と高度の算出, 
    ! エクスナー関数の計算を行います. 
    !
    ! Interpolate temperature on half sigma level, 
    ! and calculate pressure and height, 
    ! and calculate exner function. 

    ! モジュール引用 ; USE statements
    !

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & z_Sigma, &
                              ! $ \sigma $ レベル (整数). 
                              ! Full $ \sigma $ level
      & r_Sigma, &
                              ! $ \sigma $ レベル (半整数). 
                              ! Half $ \sigma $ level
      & z_DelSigma, &
                              ! $ \Delta \sigma $ (整数). 
                              ! $ \Delta \sigma $ (Full)
      & r_DelSigma
                              ! $ \Delta \sigma $ (半整数). 
                              ! $ \Delta \sigma $ (Half)

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Grav, &               ! $ g $ [m s-2]. 
                              ! 重力加速度. 
                              ! Gravitational acceleration
      & GasRDry, &
                              ! $ R $ [J kg-1 K-1]. 
                              ! 乾燥大気の気体定数. 
                              ! Gas constant of air
      & CpDry, &
                              ! $ C_p $ [J kg-1 K-1]. 
                              ! 乾燥大気の定圧比熱. 
                              ! Specific heat of air at constant pressure
      & EpsV
                              ! $ \epsilon_v $ .
                              ! 水蒸気分子量比.
                              ! Molecular weight of water vapor

    ! 時刻管理
    ! Time control
    !
    use timeset, only: TimesetClockStart, TimesetClockStop

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xy_Ps       (0:imax-1, 1:jmax)
                              ! $ p_s (t-\Delta t) $ . 地表面気圧. Surface pressure
    real(DP), intent(in):: xyz_Temp    (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ .     温度. Temperature
    real(DP), intent(in):: xyz_QVap    (0:imax-1, 1:jmax, 1:kmax)
                              ! $ q $ .     比湿. Specific humidity
    real(DP), intent(out), optional:: xyr_Temp   (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{T} $ . 温度 (半整数レベル). 
                              ! Temperature (half level)
    real(DP), intent(out), optional:: xyz_VirTemp(0:imax-1, 1:jmax, 1:kmax)
                              ! $ \hat{T} $ . 仮温度
                              ! Virtual temperature
    real(DP), intent(out), optional:: xyr_VirTemp(0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{T} $ . 仮温度 (半整数レベル). 
                              ! Virtual temperature (half level)
    real(DP), intent(out), optional:: xy_SurfVirTemp(0:imax-1, 1:jmax)
                              ! $ \hat{T} $ . 仮温度 (惑星表面). 
                              ! Virtual temperature (surface)
    real(DP), intent(out), optional:: xyz_Press  (0:imax-1, 1:jmax, 1:kmax)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)
    real(DP), intent(out), optional:: xyr_Press  (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ . 気圧 (半整数レベル). 
                              ! Air pressure (half level)
    real(DP), intent(in ), optional:: xy_SurfHeight(0:imax-1,1:jmax)
                              ! $ z_s $ . 地表面高度. 
                              ! Surface height. 
    real(DP), intent(in ), optional:: xy_SurfTemp(0:imax-1,1:jmax)
                              ! $ T_s $ . 惑星表面温度. 
                              ! Surface temperature. 
    real(DP), intent(out), optional:: xyz_Height (0:imax-1, 1:jmax, 1:kmax)
                              ! 高度 (整数レベル). 
                              ! Height (full level)
    real(DP), intent(out), optional:: xyr_Height (0:imax-1, 1:jmax, 0:kmax)
                              ! 高度 (半整数レベル). 
                              ! Height (half level)
    real(DP), intent(out), optional:: xyz_Exner (0:imax-1, 1:jmax, 1:kmax)
                              ! Exner 関数 (整数レベル). 
                              ! Exner function (full level)
    real(DP), intent(out), optional:: xyr_Exner (0:imax-1, 1:jmax, 0:kmax)
                              ! Exner 関数 (半整数レベル). 
                              ! Exner function (half level)


    ! 作業変数
    ! Work variables
    !
    real(DP):: xyz_PressWork  (0:imax-1, 1:jmax, 1:kmax)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)
    real(DP):: xyr_PressWork  (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ . 気圧 (半整数レベル). 
                              ! Air pressure (half level)
    real(DP):: xyr_TempWork   (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{T} $ . 温度 (半整数レベル). 
                              ! Temperature (half level)
    real(DP):: xyz_VirTempWork(0:imax-1, 1:jmax, 1:kmax)
                              ! $ T_v $ . 仮温度.
                              ! Virtual temperature
    real(DP):: xyr_VirTempWork(0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{T}_v $ . 仮温度 (半整数レベル). 
                              ! Virtual temperature (half level)

    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction

    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. auxiliary_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    ! 温度の補間
    ! Interpolate temperature
    !
    if ( present( xyr_Temp ) ) then
      call AuxVarsInternalRoutine(   &
        & xy_Ps,      xyz_Temp,      &  ! (in )
        & xyr_Temp = xyr_Temp  &  ! (out) optional
        & )
    end if

    ! 仮温度の計算
    ! Calculate virtual temperature
    !
    if ( present( xy_SurfVirTemp ) ) then
      if ( .not. present( xy_SurfTemp ) ) then
        call MessageNotify( 'E', module_name, 'xy_SurfTemp has to be given in arguments to calculate xy_SurfVirTemp.' )
      end if
      xy_SurfVirTemp = xy_SurfTemp * ( 1.0_DP + ((( 1.0_DP / EpsV ) - 1.0_DP ) * xyz_QVap(:,:,1) ) )
    end if

    ! 仮温度の計算
    ! Calculate virtual temperature
    !
    if ( present( xyz_VirTemp ) ) then
      xyz_VirTemp = xyz_Temp * ( 1.0_DP + ((( 1.0_DP / EpsV ) - 1.0_DP ) * xyz_QVap) )
    end if

    ! 仮温度の補間
    ! Interpolate virtual temperature
    !
    if ( present( xyr_VirTemp ) ) then
      if ( present( xyz_VirTemp ) ) then
        xyz_VirTempWork = xyz_VirTemp
      else
        xyz_VirTempWork = &
          & xyz_Temp * ( 1.0_DP + ((( 1.0_DP / EpsV ) - 1.0_DP ) * xyz_QVap) )
      end if
      call AuxVarsInternalRoutine(   &
        & xy_Ps, xyz_VirTempWork,    &  ! (in )
        & xyr_Temp = xyr_VirTemp     &  ! (out) optional
        & )
    end if

    ! 気圧の計算
    ! Calculate pressure
    !
    if ( present( xyz_Press ) ) then
      call AuxVarsInternalRoutine(   &
        & xy_Ps,      xyz_Temp,      &  ! (in )
        & xyz_Press = xyz_Press  &  ! (out) optional
        & )
    end if

    if ( present( xyr_Press ) ) then
      call AuxVarsInternalRoutine(   &
        & xy_Ps,      xyz_Temp,      &  ! (in )
        & xyr_Press = xyr_Press  &  ! (out) optional
        & )
    end if

    ! 高度の計算
    ! Calculate height
    !
    if ( present( xyz_Height ) ) then
      if ( .not. present( xy_SurfHeight ) ) then
        call MessageNotify( 'E', module_name, 'xy_SurfHeight has to be given in arguments to calculate xyz_Height.' )
      end if

!!$      if ( present( xyr_Temp ) ) then
!!$        xyr_TempWork = xyr_Temp
!!$      else
!!$        call AuxVarsInternalRoutine( &
!!$          & xy_Ps, xyz_Temp,         &  ! (in )
!!$          & xyr_Temp = xyr_TempWork  &  ! (out) optional
!!$          & )
!!$      end if
      if ( present( xyz_VirTemp ) .and. present( xyr_VirTemp ) ) then
        xyz_VirTempWork = xyz_VirTemp
        xyr_VirTempWork = xyr_VirTemp
      else
        ! 計算時間計測一時停止
        ! Pause measurement of computation time
        call TimesetClockStop( module_name )
        call AuxVars(                      &
          & xy_Ps, xyz_Temp, xyz_QVap,     &  ! (in )
          & xyz_VirTemp = xyz_VirTempWork, &  ! (out) optional
          & xyr_VirTemp = xyr_VirTempWork  &  ! (out) optional
          & )
        ! 計算時間計測開始
        ! Start measurement of computation time
        call TimesetClockStart( module_name )
      end if

!!$      xyz_Height(:,:,1) = &
!!$        & xy_SurfHeight(:,:) &
!!$        & + GasRDry / Grav * xyz_Temp(:,:,1) * ( 1. - z_Sigma(1) )
!!$      do k = 2, kmax
!!$        xyz_Height(:,:,k) = &
!!$          &   xyz_Height(:,:,k-1) &
!!$          & + GasRDry / Grav * xyr_TempWork(:,:,k-1) &
!!$          &   * r_DelSigma(k-1) / r_Sigma(k-1)
!!$      end do

      xyz_Height(:,:,1) =                          &
        & xy_SurfHeight                            &
        & + GasRDry / Grav * xyz_VirTempWork(:,:,1) * ( 1.0_DP - z_Sigma(1) )
      do k = 2, kmax
        xyz_Height(:,:,k) = xyz_Height(:,:,k-1)         &
          & + GasRDry / Grav * xyr_VirTempWork(:,:,k-1) &
          &   * r_DelSigma(k-1) / r_Sigma(k-1)
      end do
    end if

    if ( present( xyr_Height ) ) then
      if ( .not. present( xy_SurfHeight ) ) then
        call MessageNotify( 'E', module_name, 'xy_SurfHeight has to be given in arguments to calculate xyr_Height.' )
      end if

      if ( present( xyz_VirTemp ) ) then
        xyz_VirTempWork = xyz_VirTemp
      else
        ! 計算時間計測一時停止
        ! Pause measurement of computation time
        call TimesetClockStop( module_name )
        call AuxVars(                      &
          & xy_Ps, xyz_Temp, xyz_QVap,     &  ! (in )
          & xyz_VirTemp = xyz_VirTempWork  &  ! (out) optional
          & )
        ! 計算時間計測開始
        ! Start measurement of computation time
        call TimesetClockStart( module_name )
      end if

      xyr_Height(:,:,0) = xy_SurfHeight
      do k = 1, kmax
!!$        xyr_Height(:,:,k) = xyr_Height(:,:,k-1)        &
!!$          & + GasRDry / Grav * xyz_Temp(:,:,k)     &
!!$          &   * z_DelSigma(k) / z_Sigma(k)
        xyr_Height(:,:,k) = xyr_Height(:,:,k-1)        &
          & + GasRDry / Grav * xyz_VirTempWork(:,:,k)  &
          &   * z_DelSigma(k) / z_Sigma(k)
      end do
    end if

    ! エクスナー関数の計算
    ! Calculate exner function
    !
    if ( present( xyz_Exner ) ) then
      if ( present( xyz_Press ) ) then
        xyz_PressWork = xyz_Press
      else
        call AuxVarsInternalRoutine(   &
          & xy_Ps,      xyz_Temp,      &  ! (in )
          & xyz_Press = xyz_PressWork  &  ! (out) optional
          & )
      end if

      xyz_Exner = ( xyz_PressWork / RefPress ) ** ( GasRDry / CpDry )
    end if


    if ( present( xyr_Exner ) ) then
      if ( present( xyr_Press ) ) then
        xyr_PressWork = xyr_Press
      else
        call AuxVarsInternalRoutine(   &
          & xy_Ps,      xyz_Temp,      &  ! (in )
          & xyr_Press = xyr_PressWork  &  ! (out) optional
          & )
      end if

      xyr_Exner = ( xyr_PressWork / RefPress ) ** ( GasRDry / CpDry )
    end if

    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine AuxVars

  !--------------------------------------------------------------------------------------

  subroutine AuxVarsInternalRoutine( &
    & xy_Ps,      xyz_Temp,   &  ! (in )
    & xyr_Temp,               &  ! (out) optional
    & xyz_Press,  xyr_Press   &  ! (out) optional
    & )
    !
    ! 温度の半整数σレベルの補間, 気圧の算出, 
    ! エクスナー関数の計算を行います. 
    !
    ! Interpolate temperature on half sigma level, 
    ! and calculate pressure, 
    ! and calculate exner function. 

    ! モジュール引用 ; USE statements
    !

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & z_Sigma, &
                              ! $ \sigma $ レベル (整数). 
                              ! Full $ \sigma $ level
      & r_Sigma
                              ! $ \sigma $ レベル (半整数). 
                              ! Half $ \sigma $ level

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xy_Ps       (0:imax-1, 1:jmax)
                              ! $ p_s (t-\Delta t) $ . 地表面気圧. Surface pressure
    real(DP), intent(in):: xyz_Temp    (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ .     温度. Temperature
    real(DP), intent(out), optional:: xyr_Temp   (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{T} $ . 温度 (半整数レベル). 
                              ! Temperature (half level)
    real(DP), intent(out), optional:: xyz_Press  (0:imax-1, 1:jmax, 1:kmax)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)
    real(DP), intent(out), optional:: xyr_Press  (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ . 気圧 (半整数レベル). 
                              ! Air pressure (half level)


    ! 作業変数
    ! Work variables
    !
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction

    ! 実行文 ; Executable statement
    !


    ! 初期化確認
    ! Initialization check
    !
    if ( .not. auxiliary_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! 温度の補間
    ! Interpolate temperature
    !
    if ( present( xyr_Temp ) ) then
      do k = 2, kmax
        xyr_Temp(:,:,k-1) =   z_TempInpolM(k) * xyz_Temp(:,:,k-1)  &
          &                 + z_TempInpol (k) * xyz_Temp(:,:,k)
      end do

      xyr_Temp(:,:,kmax) = xyz_Temp(:,:,kmax)
      xyr_Temp(:,:,0)    = xyz_Temp(:,:,1)

    end if

    ! 気圧の計算
    ! Calculate pressure
    !
    if ( present( xyz_Press ) ) then
      do k = 1, kmax
        xyz_Press(:,:,k) = xy_Ps * z_Sigma(k)
      end do
    end if

    if ( present( xyr_Press ) ) then
      do k = 0, kmax
        xyr_Press(:,:,k) = xy_Ps * r_Sigma(k)
      end do
    end if


  end subroutine AuxVarsInternalRoutine

  !--------------------------------------------------------------------------------------

  subroutine AuxVarsInit
    !
    ! auxiliary モジュールの初期化を行います. 
    ! NAMELIST#auxiliary_nml の読み込みはこの手続きで行われます. 
    !
    ! "auxiliary" module is initialized. 
    ! "NAMELIST#auxiliary_nml" is loaded in this procedure. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & z_Sigma, &
                              ! $ \sigma $ レベル (整数). 
                              ! Full $ \sigma $ level
      & r_Sigma
                              ! $ \sigma $ レベル (半整数). 
                              ! Half $ \sigma $ level

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! 種別型パラメタ
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! 文字列操作
    ! Character handling
    !
    use dc_string, only: StoA

    ! 宣言文 ; Declaration statements
    !
    implicit none

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
                              ! IOSTAT of NAMELIST read

    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /auxiliary_nml/ &
      & RefPress
          !
          ! デフォルト値については初期化手続 "auxiliary#AuxVarsInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "auxiliary#AuxVarsInit" for the default values. 
          !

    ! 実行文 ; Executable statement
    !

    if ( auxiliary_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !
    RefPress        =   1.0e5_DP

    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml, &           ! (in)
        & nml = auxiliary_nml, &  ! (out)
        & iostat = iostat_nml )   ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
    end if

    !  補間用の係数の計算
    !  Calculate coefficient for interpolation
    !
    allocate( z_TempInpolM(1:kmax) )
    allocate( z_TempInpol(1:kmax) )

    z_TempInpolM(1) = 0.0_DP
    z_TempInpol (1) = 0.0_DP

    do k = 2, kmax
      z_TempInpolM(k) = &
        &    ( log( r_Sigma( k-1 ) ) - log( z_Sigma( k ) ) ) &
        &  / ( log( z_Sigma( k-1 ) ) - log( z_Sigma( k ) ) )
      z_TempInpol(k) = &
        &    ( log( z_Sigma( k-1 ) ) - log( r_Sigma( k-1 ) ) ) &
        &  / ( log( z_Sigma( k-1 ) ) - log( z_Sigma( k )   ) )
    end  do

    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '  z_TempInpolM = %f', d = (/ z_TempInpolM /), n = (/ kmax /) )
    call MessageNotify( 'M', module_name, '  z_TempInpol  = %f', d = (/ z_TempInpol  /), n = (/ kmax /) )
    call MessageNotify( 'M', module_name, '  RefPress     = %f', d = (/ RefPress /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    auxiliary_inited = .true.

  end subroutine AuxVarsInit

  !--------------------------------------------------------------------------------------

  subroutine AuxVarsFinalize
    !
    ! モジュール内部の変数の割り付け解除を行います. 
    !
    ! Deallocate variables in this module. 
    !

    ! 宣言文 ; Declaration statements
    !
    implicit none

    ! 実行文 ; Executable statement
    !

    if ( .not. auxiliary_inited ) return

    ! 割り付け解除
    ! Deallocation
    !
    if ( allocated( z_TempInpolM ) ) deallocate( z_TempInpolM )
    if ( allocated( z_TempInpol  ) ) deallocate( z_TempInpol  )

    auxiliary_inited = .false.

  end subroutine AuxVarsFinalize

  !--------------------------------------------------------------------------------------

end module auxiliary
