!= Nakajima et al. (1992) を用いた飽和比湿の算出
!
!= Evaluate saturation specific humidity with Nakajima et al. (1992)
!
! Authors::   Masaki ISHIWATARI, Yasuhiro MORIKAWA, Y. O. Takahashi
! Version::   $Id: saturate_nha1992.f90,v 1.9 2015/01/29 12:07:16 yot Exp $
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module saturate_nha1992
  !
  != Nakajima et al. (1992) を用いた飽和比湿の算出
  !
  != Evaluate saturation specific humidity with Nakajima et al. (1992)
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! Nakajima et al. (1992) で用いられた飽和蒸気圧 $ p^{*} $ に関する以下の式
  ! を用い, 飽和比湿および飽和比湿の温度微分の値を算出します. 
  !
  ! Saturation specific humidity and temperature derivative of it 
  ! are calculated with a folloing formula for saturation watar vapor pressure 
  ! $ p^{*} $ in Nakajima et al. (1992). 
  ! 
  ! \[
  !    p^{*} (T) = p_0^{*} \exp \left( - \frac{l}{RT} \right). 
  ! \]
  !
  ! ここで, $ T $ は温度, $ R $ は普遍気体定数です. 
  ! $ R $ および潜熱 $ l $ , 
  ! 水蒸気飽和曲線の定数 $ p_0^{*} $ は以下のように与えられます. 
  !
  ! where $ T $ is temperature, $ R $ is the gas constant. 
  ! $ R $ and latent heat $ l $ , 
  ! constant for the water vapor saturation curve $ p_0^{*} $ are as follows.
  ! 
  ! * $ R = 8.314 $ [J mol-1 K-1]
  ! * $ l = 43655 $ [J mol-1]
  ! * $ p_0^{*} = 1.4 \times 10^{11} $ [Pa]
  !
  ! 飽和水蒸気圧から飽和比湿 $ q^{*} $ を求める際には, 以下の式を用います. 
  !
  ! Saturation specific humidity $ q^{*} $ is calculated from 
  ! saturation watar vapor pressure as follows.
  !
  ! \[
  !    q^{*} (T, p) = \varepsilon \frac{p^{*} (T)}{p}
  ! \]
  !
  ! ここで $ \varepsilon $ は凝結成分と大気の分子量比, 
  ! $ p $ は気圧です. 
  !
  ! where $ \varepsilon $ is molecular weight ratio of water vapor to air, 
  ! $ p $ is air pressure. 
  ! 
  ! 従って, 飽和比湿, 飽和比湿の温度変化を求める式は以下のように
  ! なります. 
  !
  ! Therefore, saturation specific humidity and temperature derivative of it 
  ! are calculated as follows. 
  !
  ! \[
  !    q^{*} (T, p) = \varepsilon \frac{p_0^{*}}{p} \exp \left( - \frac{l}{RT} \right), \] \[
  !    \DP{q^{*} (T, p)}{T} = \varepsilon \frac{p_0^{*}}{p} \frac{l}{RT^2} \exp \left( - \frac{l}{RT} \right) 
  !    = q^{*} \frac{l}{RT^2}
  ! \]
  !
  !== References
  !
  ! * Nakajima, S., Hayashi, Y.-Y., Abe, Y., 1992: 
  !   A study on the "runaway greenhouse effect" with a 
  !   one dimensional radiative convective equilibrium model. 
  !   <i>J. Atmos. Sci.</i>, <b>49</b>, 2256--2266.
  !
  !== Procedures List
  !
  ! CalcQVapSat            :: 飽和比湿の計算
  ! CalcDQVapSatDTemp      :: 飽和比湿の温度微分の計算
  ! ------------  :: ------------
  ! CalcQVapSat            :: Calculate saturation specific humidity
  ! CalcDQVapSatDTemp      :: Calculate temperature derivative of saturation specific humidity
  !
  !--
  !== NAMELIST
  !
  ! NAMELIST#saturate_nha1992_nml
  !++

  ! モジュール引用 ; USE statements
  !

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING     ! 文字列.       Strings. 

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public:: xyz_CalcQVapSatOnLiq
  public:: xyz_CalcDQVapSatDTempOnLiq
  public:: xyz_CalcQVapSatOnSol
  public:: xyz_CalcDQVapSatDTempOnSol
  public:: SaturateInit

  ! 公開変数
  ! Public variables
  !
  logical, save, public:: saturate_nha1992_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag

  ! 非公開変数
  ! Private variables
  !
  real(DP), parameter:: LatHeat = 43655_DP
                              ! $ l $ [J mol-1].
                              ! 水の凝結の潜熱. Latent heat of condensation of water vapor
  real(DP), parameter:: P0 = 1.4e+11_DP
                              ! $ p_0^{*} $ [Pa].
                              ! 水蒸気飽和曲線の定数. constant for water vapor saturation curve

  character(*), parameter:: module_name = 'saturate_nha1992'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: saturate_nha1992.f90,v 1.9 2015/01/29 12:07:16 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

contains

  !--------------------------------------------------------------------------------------

  function xyz_CalcQVapSatOnLiq( xyz_Temp, xyz_Press ) result( xyz_QVapSat )
    !
    ! 温度 *Temp* と気圧 *Press* を用い, 
    ! 飽和比湿 *QVapSat* を求めます. 
    !
    ! Calculate saturation specific humidity *QVapSat* using
    ! temperature *Temp* and air pressure *Press*. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 物理・数学定数設定
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & GasRUniv              ! $ R^{*} $ [J K-1 mol-1].
                              ! 普遍気体定数.  Universal gas constant

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & EpsV                  ! $ \epsilon_v $ . 
                              ! 水蒸気分子量比. 
                              ! Molecular weight of water vapor

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xyz_Temp (:,:,:)
                              ! $ T $ . 温度. Temperature
    real(DP), intent(in):: xyz_Press(:,:,:)
                              ! $ p $ . 気圧. Air pressure
    real(DP):: xyz_QVapSat(size(xyz_Temp,1), size(xyz_Temp,2), size(xyz_Temp,3))
                              ! $ q^{*} $ . 飽和比湿. Saturation specific humidity

    ! 作業変数
    ! Work variables
    !

    ! 実行文 ; Executable statement
    !

    if ( .not. saturate_nha1992_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    ! 飽和比湿の計算
    ! Calculate saturation specific humidity
    !
    xyz_QVapSat = EpsV * ( P0 / xyz_Press ) * exp ( - LatHeat / ( GasRUniv * xyz_Temp ) )


  end function xyz_CalcQVapSatOnLiq

  !--------------------------------------------------------------------------------------

  function xyz_CalcDQVapSatDTempOnLiq( xyz_Temp, xyz_QVapSat ) result( xyz_DQVapSatDTemp )
    !
    ! 温度 *Temp* と飽和比湿 *QVapSat* を用い, 
    ! 飽和比湿の温度微分 *DQVapSatDTemp* を求めます. 
    !
    ! Calculate temperature derivative of saturation specific humidity 
    ! *DQVapSatDTemp* using
    ! temperature *Temp* and saturation specific humidity *QVapSat*. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 物理・数学定数設定
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & GasRUniv              ! $ R^{*} $ [J K-1 mol-1].
                              ! 普遍気体定数.  Universal gas constant

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xyz_Temp   (:,:,:)
                              ! $ T $ . 温度. Temperature
    real(DP), intent(in):: xyz_QVapSat(:,:,:)
                              ! $ q^{*} $ . 飽和比湿. Saturation specific humidity
    real(DP):: xyz_DQVapSatDTemp(size(xyz_Temp,1), size(xyz_Temp,2), size(xyz_Temp,3))
                              ! $ \DP{q^{*}}{T} $ . 飽和比湿の温度微分. 
                              ! Temperature derivative of saturation specific humidity. 

    ! 作業変数
    ! Work variables
    !

    ! 実行文 ; Executable statement
    !

    if ( .not. saturate_nha1992_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    ! 飽和比湿の温度微分の計算
    ! Calculate temperature derivative of saturation specific humidity
    !
    xyz_DQVapSatDTemp = xyz_QVapSat * LatHeat / ( GasRUniv * xyz_Temp**2 )


  end function xyz_CalcDQVapSatDTempOnLiq

  !--------------------------------------------------------------------------------------

  function xyz_CalcQVapSatOnSol( xyz_Temp, xyz_Press ) result( xyz_QVapSat )
    !
    ! 温度 *Temp* と気圧 *Press* を用い, 
    ! 飽和比湿 *QVapSat* を求めます. 
    !
    ! Calculate saturation specific humidity *QVapSat* using
    ! temperature *Temp* and air pressure *Press*. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xyz_Temp (:,:,:)
                              ! $ T $ . 温度. Temperature
    real(DP), intent(in):: xyz_Press(:,:,:)
                              ! $ p $ . 気圧. Air pressure
    real(DP):: xyz_QVapSat(size(xyz_Temp,1), size(xyz_Temp,2), size(xyz_Temp,3))
                              ! $ q^{*} $ . 飽和比湿. Saturation specific humidity

    ! 作業変数
    ! Work variables
    !

    ! 実行文 ; Executable statement
    !

    if ( .not. saturate_nha1992_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    ! 飽和比湿の計算
    ! Calculate saturation specific humidity
    !
    xyz_QVapSat = xyz_CalcQVapSatOnLiq( xyz_Temp, xyz_Press )


  end function xyz_CalcQVapSatOnSol

  !--------------------------------------------------------------------------------------

  function xyz_CalcDQVapSatDTempOnSol( xyz_Temp, xyz_QVapSat ) result( xyz_DQVapSatDTemp )
    !
    ! 温度 *Temp* と飽和比湿 *QVapSat* を用い, 
    ! 飽和比湿の温度微分 *DQVapSatDTemp* を求めます. 
    !
    ! Calculate temperature derivative of saturation specific humidity 
    ! *DQVapSatDTemp* using
    ! temperature *Temp* and saturation specific humidity *QVapSat*. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xyz_Temp   (:,:,:)
                              ! $ T $ . 温度. Temperature
    real(DP), intent(in):: xyz_QVapSat(:,:,:)
                              ! $ q^{*} $ . 飽和比湿. Saturation specific humidity
    real(DP):: xyz_DQVapSatDTemp(size(xyz_Temp,1), size(xyz_Temp,2), size(xyz_Temp,3))
                              ! $ \DP{q^{*}}{T} $ . 飽和比湿の温度微分. 
                              ! Temperature derivative of saturation specific humidity. 

    ! 作業変数
    ! Work variables
    !

    ! 実行文 ; Executable statement
    !

    if ( .not. saturate_nha1992_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if

    ! 飽和比湿の温度微分の計算
    ! Calculate temperature derivative of saturation specific humidity
    !
    xyz_DQVapSatDTemp = xyz_CalcDQVapSatDTempOnLiq( xyz_Temp, xyz_QVapSat )


  end function xyz_CalcDQVapSatDTempOnSol

  !--------------------------------------------------------------------------------------

  subroutine SaturateInit
    !
    ! saturate_nha1992 モジュールの初期化を行います. 
    !--
    ! NAMELIST#saturate_nha1992_nml の読み込みはこの手続きで行われます. 
    !++
    !
    ! "saturate_nha1992" module is initialized. 
    !--
    ! "NAMELIST#saturate_nha1992_nml" is loaded in this procedure. 
    !++

    ! モジュール引用 ; USE statements
    !

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! 種別型パラメタ
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! 文字列操作
    ! Character handling
    !
    use dc_string, only: StoA

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! 宣言文 ; Declaration statements
    !
    implicit none

!!$    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
!!$                              ! Unit number for NAMELIST file open
!!$    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
!!$                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
!!$    namelist /saturate_nha1992_nml/ 
          !
          ! デフォルト値については初期化手続 "saturate_nha1992#SaturateInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "saturate_nha1992#SaturateInit" for the default values. 
          !

    ! 実行文 ; Executable statement
    !

    if ( saturate_nha1992_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !


!!$    ! NAMELIST の読み込み
!!$    ! NAMELIST is input
!!$    !
!!$    if ( trim(namelist_filename) /= '' ) then
!!$      call FileOpen( unit_nml, &          ! (out)
!!$        & namelist_filename, mode = 'r' ) ! (in)
!!$
!!$      rewind( unit_nml )
!!$      read( unit_nml, &           ! (in)
!!$        & nml = saturate_nha1992_nml, &  ! (out)
!!$        & iostat = iostat_nml )   ! (out)
!!$      close( unit_nml )
!!$
!!$      call NmlutilMsg( iostat_nml, module_name ) ! (in)
!!$    end if

    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    saturate_nha1992_inited = .true.

  end subroutine SaturateInit

  !--------------------------------------------------------------------------------------

end module saturate_nha1992
