!= アルベド, 粗度長の設定, 陸面と海洋の差のみ考慮
!
!= Set surface albedo and roughness length, only considering land-ocean contrast
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: surface_properties_lo.f90,v 1.2 2014/05/07 09:39:23 murashin Exp $
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008-2009. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module surface_properties_lo

  ! モジュール引用 ; USE statements
  !

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision.
    &                 STRING     ! 文字列.       Strings.

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify


  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public :: SetAlbedoLO
  public :: SetRoughLenLO
  public :: SurfacePropertiesLOInit


  ! 公開変数
  ! Public variables
  !

  ! 非公開変数
  ! Private variables
  !
  real(DP), save :: AlbedoLand
  real(DP), save :: AlbedoOcean
  real(DP), save :: RoughLenLand
  real(DP), save :: RoughLenOcean


  logical, save :: surface_properties_lo_inited = .false.
                              ! 初期設定フラグ.
                              ! Initialization flag

  character(*), parameter:: module_name = 'surface_properties_lo'
                              ! モジュールの名称.
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: surface_properties_lo.f90,v 1.2 2014/05/07 09:39:23 murashin Exp $'
                              ! モジュールのバージョン
                              ! Module version


contains

  !--------------------------------------------------------------------------------------

  subroutine SetAlbedoLO( &
    & xy_SurfCond,        &
    & xy_SurfAlbedo       &
    )

    ! モジュール引用 ; USE statements
    !

    ! 格子点設定
    ! Grid points settings
    !
    use gridset, only: imax, & ! 経度格子点数.
                               ! Number of grid points in longitude
      &                jmax, & ! 緯度格子点数.
                               ! Number of grid points in latitude
      &                kmax    ! 鉛直層数.
                               ! Number of vertical level


    integer , intent(in   ) :: xy_SurfCond  ( 0:imax-1, 1:jmax )
    real(DP), intent(inout) :: xy_SurfAlbedo( 0:imax-1, 1:jmax )

    ! 作業変数
    ! Work variables
    !
    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. surface_properties_lo_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    do j = 1, jmax
      do i = 0, imax-1

        if( xy_SurfCond(i,j) == 0 ) then
          xy_SurfAlbedo(i,j) = AlbedoOcean
        else
          xy_SurfAlbedo(i,j) = AlbedoLand
        end if

      end do
    end do


  end subroutine SetAlbedoLO

  !--------------------------------------------------------------------------------------

  subroutine SetRoughLenLO( &
    & xy_SurfCond,          &
    & xy_RoughLen           &
    )

    ! モジュール引用 ; USE statements
    !

    ! 格子点設定
    ! Grid points settings
    !
    use gridset, only: imax, & ! 経度格子点数.
                               ! Number of grid points in longitude
      &                jmax, & ! 緯度格子点数.
                               ! Number of grid points in latitude
      &                kmax    ! 鉛直層数.
                               ! Number of vertical level


    integer , intent(in   ) :: xy_SurfCond( 0:imax-1, 1:jmax )
    real(DP), intent(inout) :: xy_RoughLen( 0:imax-1, 1:jmax )

    ! 作業変数
    ! Work variables
    !
    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. surface_properties_lo_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    do j = 1, jmax
      do i = 0, imax-1

        if( xy_SurfCond(i,j) == 0 ) then
          xy_RoughLen(i,j) = RoughLenOcean
        else
          xy_RoughLen(i,j) = RoughLenLand
        end if

      end do
    end do


  end subroutine SetRoughLenLO

  !--------------------------------------------------------------------------------------

  subroutine SurfacePropertiesLOInit

    ! モジュール引用 ; USE statements
    !

    ! 種別型パラメタ
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg

    ! 作業変数
    ! Work variables
    !
    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号.
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT.
                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /surface_properties_lo_nml/ &
      & AlbedoLand,    &
      & AlbedoOcean,   &
      & RoughLenLand,  &
      & RoughLenOcean

          ! デフォルト値については初期化手続 "surface_properties_lo#SurfacePropertiesLOInit"
          ! のソースコードを参照のこと.
          !
          ! Refer to source codes in the initialization procedure
          ! "surface_properties_lo#SurfacePropertiesLOInit" for the default values.
          !

    if ( surface_properties_lo_inited ) return



    ! デフォルト値の設定
    ! Default values settings
    !

    AlbedoLand    = 0.3_DP
    AlbedoOcean   = 0.1_DP
    RoughLenLand  = 1.0e-1_DP
    RoughLenOcean = 1.0e-4_DP


    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml,                   &  ! (in)
        & nml = surface_properties_lo_nml, &  ! (out)
        & iostat = iostat_nml           &  ! (out)
        & )
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
      if ( iostat_nml == 0 ) write( STDOUT, nml = surface_properties_lo_nml )
    end if


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, 'AlbedoLand    = %f', d = (/ AlbedoLand /) )
    call MessageNotify( 'M', module_name, 'AlbedoOcean   = %f', d = (/ AlbedoOcean /) )
    call MessageNotify( 'M', module_name, 'RoughLenLand  = %f', d = (/ RoughLenLand /) )
    call MessageNotify( 'M', module_name, 'RoughLenOcean = %f', d = (/ RoughLenOcean /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )


    surface_properties_lo_inited = .true.

  end subroutine SurfacePropertiesLOInit

  !--------------------------------------------------------------------------------------

end module surface_properties_lo

