module opt_prop

  use vtype_module

  use opt_prop_kd, only : OptPropKDGetPFIntedRatio

  implicit none

  private

  public :: OptPropInit
  public :: OptPropFinalize
  public :: OptPropGetNWN
  public :: OptPropGetNPress
  public :: OptPropGetNMol
  public :: OptPropGetNBand
  public :: OptPropGetDelWN
  public :: OptPropGetWN
  public :: OptPropGetBandNum
  public :: OptPropGetBandWNBnds
  public :: OptPropGetBandBinIndexBnds
!!$  public :: OptPropGetTotalStrFlux
  public :: OptPropGetIntpolSize
  public :: OptPropCalcIndexAndWeight
  public :: OptPropGetAbsCoefProf
  public :: OptPropGetAbsCoefProfForKD
  public :: OptPropGetRayScatCoef
  public :: OptPropGetPtclParam
  public :: OptPropGetPFInted
  public :: OptPropGetStrPFInted
  public :: OptPropGetStrBandAvePF
  public :: OptPropGetPFIntedRatio
  public :: OptPropGetPFIntedRatioForKD
  public :: OptPropGetStrPFIntedRatio
  public :: OptPropPFTblGetPFDPFDT
  public :: OptPropPFTblChkBandIntPFRatio
  public :: OptPropPFTblChkBandIntStrFluxRatio


  integer, save               :: IDMethod
  integer, parameter          :: IDMethodLBL        =  1
  integer, parameter          :: IDMethodLBLConstPF =  2
  integer, parameter          :: IDMethodKDDevelop  = 10
  integer, parameter          :: IDMethodKD         = 11


  ! Common
  character(extstr*4)  , save :: OptPropNcFNSave
  character(extstr*4)  , save :: StrSpeNcFNSave
  !
  character(128)       , save :: PressAxisName
  character(128)       , save :: MolAxisName
  !
  integer              , save :: NPressSave
  integer              , save :: NMolSave
  !
  real(DP), allocatable, save :: p_PressSave(:)
  real(DP), allocatable, save :: m_MolNumSave(:)

  ! LBL
  integer              , save :: NWaveNumSave
  real(DP), allocatable, save :: w_WaveNumSave(:)
  real(DP), allocatable, save :: w_StrSpeSave(:)

  real(DP), allocatable, save :: b_BandAveStrPFSave(:)

  real(DP), allocatable, save :: aa_BandWaveNumBndsSave     (:,:)

  ! KDDevelop
  integer , parameter         :: NBndSave = 2
  integer              , save :: NBinSave
  integer              , save :: NBandSave
  integer , allocatable, save :: a_BinNumSave(:)
!!$  integer , allocatable, save :: aa_BandWaveNumIndexBndsSave(:,:)
  integer , allocatable, save :: aa_BandBinIndexBndsSave    (:,:)
  !
  character(128), save :: WaveNumAxisName
  character(128), save :: AbsCoefVarName
  character(128), save :: AtmPFRatioVarName
  character(128), save :: StrPFRatioVarName
  character(128), save :: StrSpeVarName
  !
  real(DP), allocatable, save :: a_DelWaveNum(:)
  !
  ! KDDevelop only
  real(DP), allocatable, save :: za_AtmPFRatio      (:,:)
  real(DP), allocatable, save :: zma_AbsCoef     (:,:,:)
  integer , allocatable, save :: ma_IDAbsCoefType(:,:)

  real(DP), allocatable, save :: a_StrPFRatio(:)

  character(128), save :: ModuleName = 'opt_prop'

  logical, save :: FlagInited = .false.


  interface OptPropGetPtclParam
    module procedure &
      & OptPropGetPtclParam1D, &
      & OptPropGetPtclParam2D
  end interface OptPropGetPtclParam

  interface OptPropCalcIndexAndWeight
    module procedure &
      & OptPropCalcIndexAndWeight1D, &
      & OptPropCalcIndexAndWeight2D
  end interface OptPropCalcIndexAndWeight

  interface OptPropGetPFIntedRatio
    module procedure &
      & OptPropGetPFIntedRatioLBLKDDevelop, &
      & OptPropKDGetPFIntedRatio
  end interface OptPropGetPFIntedRatio

  interface OptPropPFTblGetPFDPFDT
    module procedure   &
      OptPropPFTblGetPFDPFDT2DSpecRange, &
      OptPropPFTblGetPFDPFDT1DSpecRange, &
      OptPropPFTblGetPFDPFDT1D, &
      OptPropPFTblGetPFDPFDT0D
  end interface OptPropPFTblGetPFDPFDT

  interface OptPropGetAbsCoefProfForKD
    module procedure &
      & OptPropGetAbsCoefProfForKD1D, &
      & OptPropGetAbsCoefProfForKD2D
  end interface OptPropGetAbsCoefProfForKD

  interface OptPropGetPFIntedRatioForKD
    module procedure &
      &  OptPropGetPFIntedRatioForKD1D, &
      &  OptPropGetPFIntedRatioForKD2D
  end interface OptPropGetPFIntedRatioForKD


  !----------------------------------------------------------------------------

contains

  !----------------------------------------------------------------------------

  subroutine OptPropInit( &
    & Type, &
    & OptPropNcFN, RayScatCoefNcFN, StrSpeNcFN &
    & )

#ifdef KDONLY
#else
    use opt_prop_lbl       , only : OptPropLBLInit
    use opt_prop_lblconstpf, only : OptPropLBLConstPFInit
    use opt_prop_kddevelop , only : OptPropKDDevelopInit
#endif
    use opt_prop_kd        , only : OptPropKDInit

    character(*), intent(in) :: Type
    character(*), intent(in) :: OptPropNcFN
    character(*), intent(in) :: RayScatCoefNcFN
    character(*), intent(in) :: StrSpeNcFN

    if ( FlagInited ) return


    select case ( Type )
    case ( 'LBL' )
      IDMethod = IDMethodLBL
    case ( 'LBLConstPF' )
      IDMethod = IDMethodLBLConstPF
    case ( 'KDDevelop' )
      IDMethod = IDMethodKDDevelop
    case ( 'KD' )
      IDMethod = IDMethodKD
    case default
      write( 6, * ) 'In module, ', trim( ModuleName )
      write( 6, * ) 'Unexpected type: ', trim( Type )
      stop
    end select


    select case ( IDMethod )
    case ( IDMethodLBL )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropLBLInit( &
        & OptPropNcFN, RayScatCoefNcFN, StrSpeNcFN &
        & )
#endif
    case ( IDMethodLBLConstPF )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropLBLConstPFInit( &
        & OptPropNcFN, RayScatCoefNcFN, StrSpeNcFN &
        & )
#endif
    case ( IDMethodKDDevelop )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropKDDevelopInit( &
        & OptPropNcFN, RayScatCoefNcFN, StrSpeNcFN &
        & )
#endif
    case ( IDMethodKD )
      call OptPropKDInit( &
        & OptPropNcFN &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


    FlagInited = .true.

  end subroutine OptPropInit

  !----------------------------------------------------------------------------

  subroutine OptPropFinalize

#ifdef KDONLY
#else
    use opt_prop_lbl, only : OptPropLBLFinalize
    use opt_prop_lblconstpf, only : OptPropLBLConstPFFinalize
    use opt_prop_kddevelop, only : OptPropKDDevelopFinalize
#endif
    use opt_prop_kd, only : OptPropKDFinalize

    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropLBLFinalize
#endif
    case ( IDMethodLBLConstPF )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropLBLConstPFFinalize
#endif
    case ( IDMethodKDDevelop )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropKDDevelopFinalize
#endif
    case ( IDMethodKD )
      call OptPropKDFinalize
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


    FlagInited = .false.

  end subroutine OptPropFinalize

  !----------------------------------------------------------------------------

  function OptPropGetNWN() result( NWN )

#ifdef KDONLY
#else
    use opt_prop_lbl       , only : OptPropLBLGetNWN
    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetNWN
    use opt_prop_kddevelop , only : OptPropKDDevelopGetNWN
#endif
    use opt_prop_kd        , only : OptPropKDGetNWN

    integer :: NWN

    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      NWN = OptPropLBLGetNWN()
#endif
    case ( IDMethodLBLConstPF )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      NWN = OptPropLBLConstPFGetNWN()
#endif
    case ( IDMethodKDDevelop )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      NWN = OptPropKDDevelopGetNWN()
#endif
    case ( IDMethodKD )
      NWN = OptPropKDGetNWN()
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end function OptPropGetNWN

  !----------------------------------------------------------------------------

  function OptPropGetNPress() result( NPress )

#ifdef KDONLY
#else
    use opt_prop_lbl, only : OptPropLBLGetNPress
    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetNPress
    use opt_prop_kddevelop, only : OptPropKDDevelopGetNPress
#endif
    use opt_prop_kd, only : OptPropKDGetNPress

    integer :: NPress

    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      NPress = OptPropLBLGetNPress()
#endif
    case ( IDMethodLBLConstPF )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      NPress = OptPropLBLConstPFGetNPress()
#endif
    case ( IDMethodKDDevelop )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      NPress = OptPropKDDevelopGetNPress()
#endif
    case ( IDMethodKD )
      NPress = OptPropKDGetNPress()
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end function OptPropGetNPress

  !----------------------------------------------------------------------------

  function OptPropGetNMol() result( NMol )

#ifdef KDONLY
#else
    use opt_prop_lbl, only : OptPropLBLGetNMol
    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetNMol
    use opt_prop_kddevelop, only : OptPropKDDevelopGetNMol
#endif
    use opt_prop_kd, only : OptPropKDGetNMol

    integer :: NMol

    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      NMol = OptPropLBLGetNMol()
#endif
    case ( IDMethodLBLConstPF )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      NMol = OptPropLBLConstPFGetNMol()
#endif
    case ( IDMethodKDDevelop )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      NMol = OptPropKDDevelopGetNMol()
#endif
    case ( IDMethodKD )
      NMol = OptPropKDGetNMol()
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end function OptPropGetNMol

  !----------------------------------------------------------------------------

  function OptPropGetNBand() result( NBand )

#ifdef KDONLY
#else
    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetNBand
    use opt_prop_kddevelop, only : OptPropKDDevelopGetNBand
#endif
    use opt_prop_kd, only : OptPropKDGetNBand

    integer :: NBand

    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
      write( 6, * ) "OptPropGetNband cannot be used when IDMethod is IDMethodLBL."
      stop
    case ( IDMethodLBLConstPF )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      NBand = OptPropLBLConstPFGetNBand()
#endif
    case ( IDMethodKDDevelop )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      NBand = OptPropKDDevelopGetNBand()
#endif
    case ( IDMethodKD )
      NBand = OptPropKDGetNBand()
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end function OptPropGetNBand

  !----------------------------------------------------------------------------

  function OptPropGetDelWN( iWaveNum ) result( DelWaveNum )

#ifdef KDONLY
#else
    use opt_prop_lbl, only : OptPropLBLGetDelWN
    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetDelWN
    use opt_prop_kddevelop, only : OptPropKDDevelopGetDelWN
#endif
    use opt_prop_kd, only : OptPropKDGetDelWN

    integer, intent(in) :: iWaveNum


    real(DP) :: DelWaveNum

    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      DelWaveNum = OptPropLBLGetDelWN()
#endif
    case ( IDMethodLBLConstPF )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      DelWaveNum = OptPropLBLConstPFGetDelWN()
#endif
    case ( IDMethodKDDevelop )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      DelWaveNum = OptPropKDDevelopGetDelWN( iWaveNum )
#endif
    case ( IDMethodKD )
      DelWaveNum = OptPropKDGetDelWN( iWaveNum )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select



  end function OptPropGetDelWN

  !----------------------------------------------------------------------------

  function OptPropGetWN( iWaveNum ) result( WaveNum )

#ifdef KDONLY
#else
    use opt_prop_lbl, only : OptPropLBLGetWN
    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetWN
#endif

    integer, intent(in) :: iWaveNum

    real(DP) :: WaveNum

    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      WaveNum = OptPropLBLGetWN( iWaveNum )
#endif
    case ( IDMethodLBLConstPF )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      WaveNum = OptPropLBLConstPFGetWN( iWaveNum )
#endif
    case ( IDMethodKDDevelop )
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  function OptPropGetWN cannot be used when IDMethod is IDMethodKDDevelop'
      stop
    case ( IDMethodKD )
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  function OptPropGetWN cannot be used when IDMethod is IDMethodKD'
      stop
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end function OptPropGetWN

  !----------------------------------------------------------------------------

  function OptPropGetBandNum( iWaveNum ) result( BandNum )

#ifdef KDONLY
#else
!!$    use opt_prop_lbl, only : OptPropLBLFinalize
    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetBandNum
    use opt_prop_kddevelop , only : OptPropKDDevelopGetBandNum
#endif
    use opt_prop_kd        , only : OptPropKDGetBandNum

    integer, intent(in ) :: iWaveNum

    integer :: BandNum


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
      write( 6, * ) "OptPropGetBandNum cannot be used when IDMethod is IDMethodLBL"
      stop
    case ( IDMethodLBLConstPF )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      BandNum = OptPropLBLConstPFGetBandNum( iWaveNum )
#endif
    case ( IDMethodKDDevelop )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      BandNum = OptPropKDDevelopGetBandNum( iWaveNum )
#endif
    case ( IDMethodKD )
      BandNum = OptPropKDGetBandNum( iWaveNum )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end function OptPropGetBandNum

  !----------------------------------------------------------------------------

  subroutine OptPropGetBandWNBnds( &
    & NBand, &
    & aa_BandWNBnds &
    & )

#ifdef KDONLY
#else
    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetBandWNBnds
    use opt_prop_kddevelop, only : OptPropKDDevelopGetBandWNBnds
#endif
    use opt_prop_kd, only : OptPropKDGetBandWNBnds

    integer , intent(in ) :: NBand
    real(DP), intent(out) :: aa_BandWNBnds(2,NBand)


    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
      write( 6, * ) "OptPropGetBandWNBnds cannot be called when IDMethod is IDMethod LBL."
      stop
    case ( IDMethodLBLConstPF )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropLBLConstPFGetBandWNBnds( &
        & NBand, &
        & aa_BandWNBnds &
        & )
#endif
    case ( IDMethodKDDevelop )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropKDDevelopGetBandWNBnds( &
        & NBand, &
        & aa_BandWNBnds &
        & )
#endif
    case ( IDMethodKD )
      call OptPropKDGetBandWNBnds( &
        & NBand, &
        & aa_BandWNBnds &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end subroutine OptPropGetBandWNBnds

  !----------------------------------------------------------------------------

  subroutine OptPropGetBandBinIndexBnds( &
    & NBand, &
    & aa_BandBinIndexBnds &
    & )

#ifdef KDONLY
#else
    use opt_prop_kddevelop, only : OptPropKDDevelopGetBandBinIndexBnds
#endif
    use opt_prop_kd, only : OptPropKDGetBandBinIndexBnds

    integer, intent(in ) :: NBand
    integer, intent(out) :: aa_BandBinIndexBnds(2,NBand)


    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
      write( 6, * ) "OptPropGetBandBinIndexBnds cannot be called when IDMethod is IDMethodLBL."
      stop
    case ( IDMethodLBLConstPF )
      write( 6, * ) "OptPropGetBandBinIndexBnds cannot be called when IDMethod is IDMethodLBLConstPF."
      stop
    case ( IDMethodKDDevelop )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropKDDevelopGetBandBinIndexBnds( &
        & NBand, &
        & aa_BandBinIndexBnds &
        & )
#endif
    case ( IDMethodKD )
      call OptPropKDGetBandBinIndexBnds( &
        & NBand, &
        & aa_BandBinIndexBnds &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end subroutine OptPropGetBandBinIndexBnds

  !----------------------------------------------------------------------------

!!$  subroutine OptPropGetTotalStrFlux( &
!!$    & TotalStrFlux &
!!$    & )
!!$
!!$    use opt_prop_lbl       , only : OptPropLBLGetTotalStrFlux
!!$    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetTotalStrFlux
!!$    use opt_prop_kddevelop , only : OptPropKDDevelopGetTotalStrFlux
!!$    use opt_prop_kd        , only : OptPropKDGetTotalStrFlux
!!$
!!$    real(DP), intent(out) :: TotalStrFlux
!!$
!!$
!!$    ! local variables
!!$    !
!!$    real(DP) :: DelWaveNum
!!$
!!$    integer  :: l
!!$
!!$
!!$    select case ( IDMethod )
!!$    case ( IDMethodLBL )
!!$
!!$      call OptPropLBLGetTotalStrFlux( &
!!$        & TotalStrFlux &
!!$        & )
!!$
!!$    case ( IDMethodLBLConstPF )
!!$
!!$      call OptPropLBLConstPFGetTotalStrFlux( &
!!$        & TotalStrFlux &
!!$        & )
!!$
!!$    case ( IDMethodKDDevelop )
!!$
!!$      call OptPropKDDevelopGetTotalStrFlux( &
!!$        & TotalStrFlux &
!!$        & )
!!$
!!$    case ( IDMethodKD )
!!$
!!$      call OptPropKDGetTotalStrFlux( &
!!$        & TotalStrFlux &
!!$        & )
!!$
!!$    case default
!!$      write( 6, * ) 'In ', trim( ModuleName )
!!$      write( 6, * ) '  Unexpected case default'
!!$      stop
!!$    end select
!!$
!!$
!!$  end subroutine OptPropGetTotalStrFlux

  !----------------------------------------------------------------------------

  function OptPropGetIntpolSize( TblAxisNum, DefIntpolSize ) result( IntpolSize )

    use opt_prop_kd, only : OptPropKDGetIntpolSize

    integer, intent(in) :: TblAxisNum
    integer, intent(in) :: DefIntpolSize

    integer :: IntpolSize


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodKD )
      IntpolSize = OptPropKDGetIntpolSize( TblAxisNum, DefIntpolSize )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end function OptPropGetIntpolSize

  !----------------------------------------------------------------------------

  subroutine OptPropCalcIndexAndWeight1D( &
    & kmax, NMol, m_MolNum, r_Press, r_Temp, rm_VMR, SurfTemp, &
    & NIntpol, &
    & NIntpol1Use, NIntpol2Use, NIntpol3Use, NIntpol4Use, NIntpol5Use, &
    & r_KDTblPressIndex, r_KDTblTempIndex, &
    & r_KDTblVMRMol1Index, r_KDTblVMRMol2Index, r_KDTblVMRMol3Index, &
    & rptuvw_Weight, &
    & r_KDTblPressIndexSfc, r_KDTblTempIndexSfc, &
    & r_KDTblVMRMol1IndexSfc, r_KDTblVMRMol2IndexSfc, r_KDTblVMRMol3IndexSfc, &
    & ptuvw_WeightSfc &
    & )

    use opt_prop_kd, only : OptPropKDCalcIndexAndWeight

    integer     , intent(in ) :: kmax
    integer     , intent(in ) :: NMol
    integer     , intent(in ) :: m_MolNum  (1:NMol)
    real(DP)    , intent(in ) :: r_Press   (1:kmax)
    real(DP)    , intent(in ) :: r_Temp    (1:kmax)
    real(DP)    , intent(in ) :: rm_VMR    (1:kmax,1:NMol)
    real(DP)    , intent(in ) :: SurfTemp
    integer     , intent(in ) :: NIntpol
    integer     , intent(in ) :: NIntpol1Use
    integer     , intent(in ) :: NIntpol2Use
    integer     , intent(in ) :: NIntpol3Use
    integer     , intent(in ) :: NIntpol4Use
    integer     , intent(in ) :: NIntpol5Use
    integer     , intent(out) :: r_KDTblPressIndex  (1:kmax)
    integer     , intent(out) :: r_KDTblTempIndex   (1:kmax)
    integer     , intent(out) :: r_KDTblVMRMol1Index(1:kmax)
    integer     , intent(out) :: r_KDTblVMRMol2Index(1:kmax)
    integer     , intent(out) :: r_KDTblVMRMol3Index(1:kmax)
    real(DP)    , intent(out) :: rptuvw_Weight(1:kmax,NIntpol,NIntpol,NIntpol,NIntpol,NIntpol)
    integer     , intent(out) :: r_KDTblPressIndexSfc  (1)
    integer     , intent(out) :: r_KDTblTempIndexSfc   (1)
    integer     , intent(out) :: r_KDTblVMRMol1IndexSfc(1)
    integer     , intent(out) :: r_KDTblVMRMol2IndexSfc(1)
    integer     , intent(out) :: r_KDTblVMRMol3IndexSfc(1)
    real(DP)    , intent(out) :: ptuvw_WeightSfc(NIntpol,NIntpol,NIntpol,NIntpol,NIntpol)


    ! local variables
    !


    select case ( IDMethod )
    case ( IDMethodKD )
      call OptPropKDCalcIndexAndWeight( &
        & kmax, NMol, m_MolNum, r_Press, r_Temp, rm_VMR, SurfTemp, &
        & NIntpol, &
        & NIntpol1Use, NIntpol2Use, NIntpol3Use, NIntpol4Use, NIntpol5Use, &
        & r_KDTblPressIndex, r_KDTblTempIndex, &
        & r_KDTblVMRMol1Index, r_KDTblVMRMol2Index, r_KDTblVMRMol3Index, &
        & rptuvw_Weight, &
        & r_KDTblPressIndexSfc, r_KDTblTempIndexSfc, &
        & r_KDTblVMRMol1IndexSfc, r_KDTblVMRMol2IndexSfc, r_KDTblVMRMol3IndexSfc, &
        & ptuvw_WeightSfc &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  In OptPropCalcIndexAndWeight'
      write( 6, * ) '    Unexpected case default'
      stop
    end select


  end subroutine OptPropCalcIndexAndWeight1D

  !----------------------------------------------------------------------------

  subroutine OptPropCalcIndexAndWeight2D( &
    & imax, kmax, NMol, m_MolNum, &
    & xr_Press, xr_Temp, xrm_VMR, x_SurfTemp, &
    & NIntpol, &
    & NIntpol1Use, NIntpol2Use, NIntpol3Use, NIntpol4Use, NIntpol5Use, &
    & xr_KDTblPressIndex, xr_KDTblTempIndex, &
    & xr_KDTblVMRMol1Index, xr_KDTblVMRMol2Index, xr_KDTblVMRMol3Index, &
    & ptuvwxr_Weight, &
    & xr_KDTblPressIndexSfc, xr_KDTblTempIndexSfc, &
    & xr_KDTblVMRMol1IndexSfc, xr_KDTblVMRMol2IndexSfc, xr_KDTblVMRMol3IndexSfc, &
    & ptuvwx_WeightSfc &
    & )

    use opt_prop_kd, only : OptPropKDCalcIndexAndWeight

    integer     , intent(in ) :: imax
    integer     , intent(in ) :: kmax
    integer     , intent(in ) :: NMol
    integer     , intent(in ) :: m_MolNum  (1:NMol)
    real(DP)    , intent(in ) :: xr_Press  (1:imax,1:kmax)
    real(DP)    , intent(in ) :: xr_Temp   (1:imax,1:kmax)
    real(DP)    , intent(in ) :: xrm_VMR   (1:imax,1:kmax,1:NMol)
    real(DP)    , intent(in ) :: x_SurfTemp(1:imax)
    integer     , intent(in ) :: NIntpol
    integer     , intent(in ) :: NIntpol1Use
    integer     , intent(in ) :: NIntpol2Use
    integer     , intent(in ) :: NIntpol3Use
    integer     , intent(in ) :: NIntpol4Use
    integer     , intent(in ) :: NIntpol5Use
    integer     , intent(out) :: xr_KDTblPressIndex  (1:imax,1:kmax)
    integer     , intent(out) :: xr_KDTblTempIndex   (1:imax,1:kmax)
    integer     , intent(out) :: xr_KDTblVMRMol1Index(1:imax,1:kmax)
    integer     , intent(out) :: xr_KDTblVMRMol2Index(1:imax,1:kmax)
    integer     , intent(out) :: xr_KDTblVMRMol3Index(1:imax,1:kmax)
    real(DP)    , intent(out) :: ptuvwxr_Weight(NIntpol,NIntpol,NIntpol,NIntpol,NIntpol,1:imax,1:kmax)
    integer     , intent(out) :: xr_KDTblPressIndexSfc  (1:imax,1)
    integer     , intent(out) :: xr_KDTblTempIndexSfc   (1:imax,1)
    integer     , intent(out) :: xr_KDTblVMRMol1IndexSfc(1:imax,1)
    integer     , intent(out) :: xr_KDTblVMRMol2IndexSfc(1:imax,1)
    integer     , intent(out) :: xr_KDTblVMRMol3IndexSfc(1:imax,1)
    real(DP)    , intent(out) :: ptuvwx_WeightSfc(NIntpol,NIntpol,NIntpol,NIntpol,NIntpol,1:imax)


    ! local variables
    !


    select case ( IDMethod )
    case ( IDMethodKD )
      call OptPropKDCalcIndexAndWeight( &
        & imax, kmax, NMol, m_MolNum, &
        & xr_Press, xr_Temp, xrm_VMR, x_SurfTemp, &
        & NIntpol, &
        & NIntpol1Use, NIntpol2Use, NIntpol3Use, NIntpol4Use, NIntpol5Use, &
        & xr_KDTblPressIndex, xr_KDTblTempIndex, &
        & xr_KDTblVMRMol1Index, xr_KDTblVMRMol2Index, xr_KDTblVMRMol3Index, &
        & ptuvwxr_Weight, &
        & xr_KDTblPressIndexSfc, xr_KDTblTempIndexSfc, &
        & xr_KDTblVMRMol1IndexSfc, xr_KDTblVMRMol2IndexSfc, xr_KDTblVMRMol3IndexSfc, &
        & ptuvwx_WeightSfc &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  In OptPropCalcIndexAndWeight2D'
      write( 6, * ) '    Unexpected case default'
      stop
    end select


  end subroutine OptPropCalcIndexAndWeight2D

  !----------------------------------------------------------------------------

  subroutine OptPropGetAbsCoefProf( &
    & iWaveNum, kmax, NMol, m_MolNum, r_Press, r_Temp, rm_VMR, &
    & rm_AbsCoef &
    & )

#ifdef KDONLY
#else
    use opt_prop_lbl       , only : OptPropLBLGetAbsCoefProf
    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetAbsCoefProf
    use opt_prop_kddevelop , only : OptPropKDDevelopGetAbsCoefProf
#endif
    use opt_prop_kd        , only : OptPropKDGetAbsCoefProf

    integer , intent(in ) :: iWaveNum
    integer , intent(in ) :: kmax
    integer , intent(in ) :: NMol
    integer , intent(in ) :: m_MolNum  (1:NMol)
    real(DP), intent(in ) :: r_Press   (1:kmax)
    real(DP), intent(in ) :: r_Temp    (1:kmax)
    real(DP), intent(in ) :: rm_VMR    (1:kmax,1:NMol)
    real(DP), intent(out) :: rm_AbsCoef(1:kmax,1:NMol)


    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropLBLGetAbsCoefProf( &
        & iWaveNum, kmax, NMol, m_MolNum, rm_VMR, &
        & rm_AbsCoef &
        & )
#endif
    case ( IDMethodLBLConstPF )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropLBLConstPFGetAbsCoefProf( &
        & iWaveNum, kmax, NMol, m_MolNum, rm_VMR, &
        & rm_AbsCoef &
        & )
#endif
    case ( IDMethodKDDevelop )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropKDDevelopGetAbsCoefProf( &
        & iWaveNum, kmax, NMol, m_MolNum, rm_VMR, &
        & rm_AbsCoef &
        & )
#endif
    case ( IDMethodKD )
      call OptPropKDGetAbsCoefProf( &
        & iWaveNum, kmax, NMol, m_MolNum, r_Press, r_Temp, rm_VMR, &
        & rm_AbsCoef &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end subroutine OptPropGetAbsCoefProf

  !----------------------------------------------------------------------------

  subroutine OptPropGetAbsCoefProfForKD1D( &
    & iWaveNum, kmax, NMol, m_MolNum, r_Press, r_Temp, rm_VMR, &
    & NIntpol, &
    & NIntpol1Use, NIntpol2Use, NIntpol3Use, NIntpol4Use, NIntpol5Use, &
    & r_KDTblPressIndex, r_KDTblTempIndex, &
    & r_KDTblVMRMol1Index, r_KDTblVMRMol2Index, r_KDTblVMRMol3Index, &
    & rptuvw_Weight, &
    & rm_AbsCoef &
    & )

    use opt_prop_kd        , only : &
      & OptPropKDGetAbsCoefProfWithIndex

    integer , intent(in ) :: iWaveNum
    integer , intent(in ) :: kmax
    integer , intent(in ) :: NMol
    integer , intent(in ) :: m_MolNum  (1:NMol)
    real(DP), intent(in ) :: r_Press   (1:kmax)
    real(DP), intent(in ) :: r_Temp    (1:kmax)
    real(DP), intent(in ) :: rm_VMR    (1:kmax,1:NMol)
    integer , intent(in ) :: NIntpol
    integer , intent(in ) :: NIntpol1Use
    integer , intent(in ) :: NIntpol2Use
    integer , intent(in ) :: NIntpol3Use
    integer , intent(in ) :: NIntpol4Use
    integer , intent(in ) :: NIntpol5Use
    integer , intent(in ) :: r_KDTblPressIndex  (1:kmax)
    integer , intent(in ) :: r_KDTblTempIndex   (1:kmax)
    integer , intent(in ) :: r_KDTblVMRMol1Index(1:kmax)
    integer , intent(in ) :: r_KDTblVMRMol2Index(1:kmax)
    integer , intent(in ) :: r_KDTblVMRMol3Index(1:kmax)
    real(DP), intent(in ) :: rptuvw_Weight(1:kmax,NIntpol,NIntpol,NIntpol,NIntpol,NIntpol)
    real(DP), intent(out) :: rm_AbsCoef(1:kmax,1:NMol)


    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodKD )
      call OptPropKDGetAbsCoefProfWithIndex( &
        & iWaveNum, kmax, NMol, m_MolNum, r_Press, r_Temp, rm_VMR, &
        & rm_AbsCoef, &
        & NIntpol, &
        & NIntpol1Use, NIntpol2Use, NIntpol3Use, NIntpol4Use, NIntpol5Use, &
        & r_KDTblPressIndex, r_KDTblTempIndex, &
        & r_KDTblVMRMol1Index, r_KDTblVMRMol2Index, r_KDTblVMRMol3Index, &
        & rptuvw_Weight &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  In OptPropGetAbsCoefProfForKD'
      write( 6, * ) '    Unexpected case default'
      stop
    end select


  end subroutine OptPropGetAbsCoefProfForKD1D

  !----------------------------------------------------------------------------

  subroutine OptPropGetAbsCoefProfForKD2D( &
    & iWaveNum, imax, kmax, NMol, m_MolNum, &
    & xr_Press, xr_Temp, xrm_VMR, &
    & NIntpol, &
    & NIntpol1Use, NIntpol2Use, NIntpol3Use, NIntpol4Use, NIntpol5Use, &
    & xr_KDTblPressIndex, xr_KDTblTempIndex, &
    & xr_KDTblVMRMol1Index, xr_KDTblVMRMol2Index, xr_KDTblVMRMol3Index, &
    & ptuvwxr_Weight, &
    & xrm_AbsCoef &
    & )

    use opt_prop_kd        , only : &
      & OptPropKDGetAbsCoefProfWithIndex

    integer     , intent(in ) :: iWaveNum
    integer     , intent(in ) :: imax
    integer     , intent(in ) :: kmax
    integer     , intent(in ) :: NMol
    integer     , intent(in ) :: m_MolNum   (1:NMol)
    real(DP)    , intent(in ) :: xr_Press   (1:imax,1:kmax)
    real(DP)    , intent(in ) :: xr_Temp    (1:imax,1:kmax)
    real(DP)    , intent(in ) :: xrm_VMR    (1:imax,1:kmax,1:NMol)
    integer     , intent(in ) :: NIntpol
    integer     , intent(in ) :: NIntpol1Use
    integer     , intent(in ) :: NIntpol2Use
    integer     , intent(in ) :: NIntpol3Use
    integer     , intent(in ) :: NIntpol4Use
    integer     , intent(in ) :: NIntpol5Use
    integer     , intent(in ) :: xr_KDTblPressIndex  (1:imax,1:kmax)
    integer     , intent(in ) :: xr_KDTblTempIndex   (1:imax,1:kmax)
    integer     , intent(in ) :: xr_KDTblVMRMol1Index(1:imax,1:kmax)
    integer     , intent(in ) :: xr_KDTblVMRMol2Index(1:imax,1:kmax)
    integer     , intent(in ) :: xr_KDTblVMRMol3Index(1:imax,1:kmax)
    real(DP)    , intent(in ) :: ptuvwxr_Weight(NIntpol,NIntpol,NIntpol,NIntpol,NIntpol,1:imax,1:kmax)
    real(DP)    , intent(out) :: xrm_AbsCoef(1:imax,1:kmax,1:NMol)


    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodKD )
      call OptPropKDGetAbsCoefProfWithIndex( &
        & iWaveNum, imax, kmax, NMol, m_MolNum, &
        & xr_Press, xr_Temp, xrm_VMR, &
        & xrm_AbsCoef, &
        & NIntpol, &
        & NIntpol1Use, NIntpol2Use, NIntpol3Use, NIntpol4Use, NIntpol5Use, &
        & xr_KDTblPressIndex, xr_KDTblTempIndex, &
        & xr_KDTblVMRMol1Index, xr_KDTblVMRMol2Index, xr_KDTblVMRMol3Index, &
        & ptuvwxr_Weight &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  In OptPropGetAbsCoefProfForKD2D'
      write( 6, * ) '    Unexpected case default'
      stop
    end select


  end subroutine OptPropGetAbsCoefProfForKD2D

  !----------------------------------------------------------------------------

  subroutine OptPropGetRayScatCoef( &
    & iWaveNum, &
    & NMol, m_MolNum, &
    & RayScatCoefNonRadAct, m_RayScatCoef &
    & )

#ifdef KDONLY
#else
    use opt_prop_lbl       , only : OptPropLBLGetRayScatCoef
    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetRayScatCoef
    use opt_prop_kddevelop , only : OptPropKDDevelopGetRayScatCoef
#endif
    use opt_prop_kd        , only : OptPropKDGetRayScatCoef


    integer     , intent(in ) :: iWaveNum
    integer     , intent(in ) :: NMol
    integer     , intent(in ) :: m_MolNum(NMol)
    real(DP)    , intent(out) :: RayScatCoefNonRadAct
    real(DP)    , intent(out) :: m_RayScatCoef(NMol)


    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropLBLGetRayScatCoef( &
        & iWaveNum, &
        & NMol, m_MolNum, &
        & RayScatCoefNonRadAct, m_RayScatCoef &
        & )
#endif
    case ( IDMethodLBLConstPF )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropLBLConstPFGetRayScatCoef( &
        & iWaveNum, &
        & NMol, m_MolNum, &
        & RayScatCoefNonRadAct, m_RayScatCoef &
        & )
#endif
    case ( IDMethodKDDevelop )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropKDDevelopGetRayScatCoef( &
        & iWaveNum, &
        & NMol, m_MolNum, &
        & RayScatCoefNonRadAct, m_RayScatCoef &
        & )
#endif
!!$      RayScatCoef = 0.0d0
    case ( IDMethodKD )
      call OptPropKDGetRayScatCoef( &
        & iWaveNum, &
        & NMol, m_MolNum, &
        & RayScatCoefNonRadAct, m_RayScatCoef &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end subroutine OptPropGetRayScatCoef

  !----------------------------------------------------------------------------

  subroutine OptPropGetPtclParam1D( &
    & iWaveNum, &
    & kmax, NPtcl, a_PtclName, za_PtclRadius, &
    & za_QExt, za_SSA, za_AF &
    & )

#ifdef KDONLY
#else
    use opt_prop_lbl       , only : OptPropLBLGetPtclParam
    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetPtclParam
    use opt_prop_kddevelop , only : OptPropKDDevelopGetPtclParam
#endif
    use opt_prop_kd        , only : OptPropKDGetPtclParam


    integer     , intent(in ) :: iWaveNum
    integer     , intent(in ) :: kmax
    integer     , intent(in ) :: NPtcl
    character(*), intent(in ) :: a_PtclName(NPtcl)
    real(DP)    , intent(in ) :: za_PtclRadius(kmax,NPtcl)
    real(DP)    , intent(out) :: za_QExt(kmax,NPtcl)
    real(DP)    , intent(out) :: za_SSA (kmax,NPtcl)
    real(DP)    , intent(out) :: za_AF  (kmax,NPtcl)


    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropLBLGetPtclParam( &
        & iWaveNum, &
        & kmax, NPtcl, a_PtclName, za_PtclRadius, &
        & za_QExt, za_SSA, za_AF &
        & )
#endif
    case ( IDMethodLBLConstPF )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropLBLConstPFGetPtclParam( &
        & iWaveNum, &
        & kmax, NPtcl, a_PtclName, za_PtclRadius, &
        & za_QExt, za_SSA, za_AF &
        & )
#endif
    case ( IDMethodKDDevelop )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropKDDevelopGetPtclParam( &
        & iWaveNum, &
        & kmax, NPtcl, a_PtclName, za_PtclRadius, &
        & za_QExt, za_SSA, za_AF &
        & )
#endif
!!$      za_QExt = 0.0d0
!!$      za_SSA  = 0.0d0
!!$      za_AF   = 0.0d0
    case ( IDMethodKD )
      call OptPropKDGetPtclParam( &
        & iWaveNum, &
        & kmax, NPtcl, a_PtclName, za_PtclRadius, &
        & za_QExt, za_SSA, za_AF &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end subroutine OptPropGetPtclParam1D

  !----------------------------------------------------------------------------

  subroutine OptPropGetPtclParam2D( &
    & iWaveNum, &
    & imax, kmax, NPtcl, a_PtclName, xza_PtclRadius, &
    & xza_QExt, xza_SSA, xza_AF &
    & )

    use opt_prop_kd, only : OptPropKDGetPtclParam2D


    integer     , intent(in ) :: iWaveNum
    integer     , intent(in ) :: imax
    integer     , intent(in ) :: kmax
    integer     , intent(in ) :: NPtcl
    character(*), intent(in ) :: a_PtclName(NPtcl)
    real(DP)    , intent(in ) :: xza_PtclRadius(imax,kmax,NPtcl)
    real(DP)    , intent(out) :: xza_QExt      (imax,kmax,NPtcl)
    real(DP)    , intent(out) :: xza_SSA       (imax,kmax,NPtcl)
    real(DP)    , intent(out) :: xza_AF        (imax,kmax,NPtcl)


    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL, IDMethodLBLConstPF, IDMethodKDDevelop )
      stop 'This is not supported.'
    case ( IDMethodKD )
      call OptPropKDGetPtclParam2D( &
        & iWaveNum, &
        & imax, kmax, NPtcl, a_PtclName, xza_PtclRadius, &
        & xza_QExt, xza_SSA, xza_AF &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end subroutine OptPropGetPtclParam2D

  !----------------------------------------------------------------------------

  subroutine OptPropGetPFInted( &
    & iWaveNum, &
    & kmax, &
    & r_Temp, SurfTemp, &
    & r_AtmPFInted, SurfPFInted, SurfDPFDTInted &
    & )

#ifdef KDONLY
#else
    use opt_prop_lbl, only : OptPropLBLGetPFInted
!!$    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetPFInted
#endif

    integer , intent(in ) :: iWaveNum
    integer , intent(in ) :: kmax
    real(DP), intent(in ) :: r_Temp   (1:kmax)
    real(DP), intent(in ) :: SurfTemp
    real(DP), intent(out) :: r_AtmPFInted(1:kmax)
    real(DP), intent(out) :: SurfPFInted
    real(DP), intent(out) :: SurfDPFDTInted


    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropLBLGetPFInted( &
        & iWaveNum, &
        & kmax, &
        & r_Temp, SurfTemp, &
        & r_AtmPFInted, SurfPFInted, SurfDPFDTInted &
        & )
#endif
    case ( IDMethodLBLConstPF )
      write( 6, * ) 'OptPropGetPFInted cannot be used when IDMethod is IDMethodLBLConstPF'
      stop
!!$      call OptPropLBLConstPFGetPFInted( &
!!$        & iWaveNum, &
!!$        & kmax, &
!!$        & r_Temp, SurfTemp, &
!!$        & r_AtmPFInted, SurfPFInted, SurfDPFDTInted &
!!$        & )
    case ( IDMethodKDDevelop )
      write( 6, * ) 'OptPropGetPFInted cannot be used when IDMethod is IDMethodKDDevelop'
      stop
!!$      call OptPropGetPFIntedCore( &
!!$        & iWaveNum, &
!!$        & kmax, &
!!$        & r_Temp, SurfTemp, &
!!$        & r_AtmPFInted, SurfPFInted, SurfDPFDTInted &
!!$        & )
    case ( IDMethodKD )
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  function OptPropGetPFInted cannot be used when IDMethod is IDMethodKD'
      stop
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end subroutine OptPropGetPFInted

  !----------------------------------------------------------------------------

  subroutine OptPropGetStrPFInted( &
    & iWaveNum, StrFluxTOA, &
    & StrPFInted &
    & )

#ifdef KDONLY
#else
    use opt_prop_lbl, only : OptPropLBLGetStrPFInted
!!$    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetStrPFInted
#endif

    integer , intent(in ) :: iWaveNum
    real(DP), intent(in ) :: StrFluxTOA
    real(DP), intent(out) :: StrPFInted


    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropLBLGetStrPFInted( &
        & iWaveNum, StrFluxTOA, &
        & StrPFInted &
        & )
#endif
    case ( IDMethodLBLConstPF )
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  function OptPropGetPFInted cannot be used when IDMethod is IDMethodLBLConstPF'
      stop
!!$      call OptPropLBLConstPFGetStrPFInted( &
!!$        & iWaveNum, &
!!$        & StrPFInted &
!!$        & )
    case ( IDMethodKDDevelop )
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  function OptPropGetPFInted cannot be used when IDMethod is IDMethodKDDevelop'
      stop
!!$      call OptPropGetStrPFIntedCore( &
!!$        & iWaveNum, &
!!$        & StrPFInted &
!!$        & )
    case ( IDMethodKD )
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  function OptPropGetPFInted cannot be used when IDMethod is IDMethodKD'
      stop
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end subroutine OptPropGetStrPFInted

  !----------------------------------------------------------------------------

  subroutine OptPropGetStrBandAvePF( &
    & iBand, StrFluxTOA, &
    & StrBandAvePF &
    & )

#ifdef KDONLY
#else
    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetStrBandAvePF
    use opt_prop_kddevelop , only : OptPropKDDevelopGetStrBandAvePF
#endif
    use opt_prop_kd        , only : OptPropKDGetStrBandAvePF

    integer , intent(in ) :: iBand
    real(DP), intent(in ) :: StrFluxTOA
    real(DP), intent(out) :: StrBandAvePF


    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  function OptPropGetPFInted cannot be used when IDMethod is IDMethodLBL'
      stop
    case ( IDMethodLBLConstPF )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropLBLConstPFGetStrBandAvePF( &
        & iBand, StrFluxTOA, &
        & StrBandAvePF &
        & )
#endif
    case ( IDMethodKDDevelop )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropKDDevelopGetStrBandAvePF( &
        & iBand, StrFluxTOA, &
        & StrBandAvePF &
        & )
#endif
    case ( IDMethodKD )
      call OptPropKDGetStrBandAvePF( &
        & iBand, StrFluxTOA, &
        & StrBandAvePF &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end subroutine OptPropGetStrBandAvePF

  !----------------------------------------------------------------------------

!!$  subroutine OptPropGetPFIntedRatio( &
  subroutine OptPropGetPFIntedRatioLBLKDDevelop( &
    & iWaveNum, kmax, r_Temp, SurfTemp, &
    & r_AtmPFRatio, SurfPFRatio &
    & )

#ifdef KDONLY
#else
    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetPFIntedRatio
    use opt_prop_kddevelop , only : OptPropKDDevelopGetPFIntedRatio
#endif
!!$    use opt_prop_kd        , only : OptPropKDGetPFIntedRatio

    integer     , intent(in ) :: iWaveNum
    integer     , intent(in ) :: kmax
    real(DP)    , intent(in ) :: r_Temp   (1:kmax)
    real(DP)    , intent(in ) :: SurfTemp
    real(DP)    , intent(out) :: r_AtmPFRatio(1:kmax)
    real(DP)    , intent(out) :: SurfPFRatio


    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  function OptPropGetPFIntedRatio cannot be used when IDMethod is IDMethodLBL'
      stop
    case ( IDMethodLBLConstPF )
!!$      write( 6, * ) 'In ', trim( ModuleName )
!!$      write( 6, * ) '  function OptPropGetPFIntedRatio cannot be used when IDMethod is IDMethodLBLConst'
!!$      stop
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropLBLConstPFGetPFIntedRatio( &
        & iWaveNum, kmax, &
        & r_AtmPFRatio, SurfPFRatio &
        & )
#endif
    case ( IDMethodKDDevelop )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropKDDevelopGetPFIntedRatio( &
        & iWaveNum, kmax, &
        & r_AtmPFRatio, SurfPFRatio &
        & )
#endif
    case ( IDMethodKD )
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  function OptPropGetPFIntedRatio cannot be used when IDMethod is IDMethodKD'
      stop
!!$      call OptPropKDGetPFIntedRatio( &
!!$        & iWaveNum, kmax, r_Temp, SurfTemp, &
!!$        & r_AtmPFRatio, SurfPFRatio &
!!$        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


!!$  end subroutine OptPropGetPFIntedRatio
  end subroutine OptPropGetPFIntedRatioLBLKDDevelop

  !----------------------------------------------------------------------------

  subroutine OptPropGetPFIntedRatioForKD1D( &
    & iWaveNum, kmax, NMol, m_MolNum, r_Press, r_Temp, SurfTemp, rm_VMR, &
    & r_AtmPFRatio, SurfPFRatio, &
    & NIntpol, &
    & NIntpol1Use, NIntpol2Use, NIntpol3Use, NIntpol4Use, NIntpol5Use, &
    & r_KDTblPressIndex, r_KDTblTempIndex, &
    & r_KDTblVMRMol1Index, r_KDTblVMRMol2Index, r_KDTblVMRMol3Index, &
    & rptuvw_Weight, &
    & r_KDTblPressIndexSfc, r_KDTblTempIndexSfc, &
    & r_KDTblVMRMol1IndexSfc, r_KDTblVMRMol2IndexSfc, r_KDTblVMRMol3IndexSfc, &
    & ptuvw_WeightSfc &
    & )

    use opt_prop_kd, only : OptPropKDGetPFIntedRatioWithIndex

    integer     , intent(in ) :: iWaveNum
    integer     , intent(in ) :: kmax
    integer     , intent(in ) :: NMol
    integer     , intent(in ) :: m_MolNum(NMol)
    real(DP)    , intent(in ) :: r_Press  (1:kmax)
    real(DP)    , intent(in ) :: r_Temp   (1:kmax)
    real(DP)    , intent(in ) :: SurfTemp
    real(DP)    , intent(in ) :: rm_VMR    (1:kmax,1:NMol)
    real(DP)    , intent(out) :: r_AtmPFRatio(1:kmax)
    real(DP)    , intent(out) :: SurfPFRatio
    integer     , intent(in ) :: NIntpol
    integer     , intent(in ) :: NIntpol1Use
    integer     , intent(in ) :: NIntpol2Use
    integer     , intent(in ) :: NIntpol3Use
    integer     , intent(in ) :: NIntpol4Use
    integer     , intent(in ) :: NIntpol5Use
    integer     , intent(in ) :: r_KDTblPressIndex  (1:kmax)
    integer     , intent(in ) :: r_KDTblTempIndex   (1:kmax)
    integer     , intent(in ) :: r_KDTblVMRMol1Index(1:kmax)
    integer     , intent(in ) :: r_KDTblVMRMol2Index(1:kmax)
    integer     , intent(in ) :: r_KDTblVMRMol3Index(1:kmax)
    real(DP)    , intent(in ) :: rptuvw_Weight(1:kmax,NIntpol,NIntpol,NIntpol,NIntpol,NIntpol)
    integer     , intent(in ) :: r_KDTblPressIndexSfc  (1)
    integer     , intent(in ) :: r_KDTblTempIndexSfc   (1)
    integer     , intent(in ) :: r_KDTblVMRMol1IndexSfc(1)
    integer     , intent(in ) :: r_KDTblVMRMol2IndexSfc(1)
    integer     , intent(in ) :: r_KDTblVMRMol3IndexSfc(1)
    real(DP)    , intent(in ) :: ptuvw_WeightSfc(NIntpol,NIntpol,NIntpol,NIntpol,NIntpol)



    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodKD )
      call OptPropKDGetPFIntedRatioWithIndex( &
        & iWaveNum, kmax, NMol, m_MolNum, r_Press, r_Temp, SurfTemp, rm_VMR, &
        & r_AtmPFRatio, SurfPFRatio, &
        & NIntpol, &
        & NIntpol1Use, NIntpol2Use, NIntpol3Use, NIntpol4Use, NIntpol5Use, &
        & r_KDTblPressIndex, r_KDTblTempIndex, &
        & r_KDTblVMRMol1Index, r_KDTblVMRMol2Index, r_KDTblVMRMol3Index, &
        & rptuvw_Weight, &
        & r_KDTblPressIndexSfc, r_KDTblTempIndexSfc, &
        & r_KDTblVMRMol1IndexSfc, r_KDTblVMRMol2IndexSfc, r_KDTblVMRMol3IndexSfc, &
        & ptuvw_WeightSfc &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  In OptPropGetPFIntedRatioForKD'
      write( 6, * ) '    Unexpected case default'
      stop
    end select


  end subroutine OptPropGetPFIntedRatioForKD1D

  !----------------------------------------------------------------------------

  subroutine OptPropGetPFIntedRatioForKD2D( &
    & iWaveNum, imax, kmax, NMol, m_MolNum, &
    & xr_Press, xr_Temp, x_SurfTemp, xrm_VMR, &
    & xr_AtmPFRatio, x_SurfPFRatio, &
    & NIntpol, &
    & NIntpol1Use, NIntpol2Use, NIntpol3Use, NIntpol4Use, NIntpol5Use, &
    & xr_KDTblPressIndex, xr_KDTblTempIndex, &
    & xr_KDTblVMRMol1Index, xr_KDTblVMRMol2Index, xr_KDTblVMRMol3Index, &
    & ptuvwxr_Weight, &
    & xr_KDTblPressIndexSfc, xr_KDTblTempIndexSfc, &
    & xr_KDTblVMRMol1IndexSfc, xr_KDTblVMRMol2IndexSfc, xr_KDTblVMRMol3IndexSfc, &
    & ptuvwx_WeightSfc &
    & )

    use opt_prop_kd, only : OptPropKDGetPFIntedRatioWithIndex

    integer     , intent(in ) :: iWaveNum
    integer     , intent(in ) :: imax
    integer     , intent(in ) :: kmax
    integer     , intent(in ) :: NMol
    integer     , intent(in ) :: m_MolNum(NMol)
    real(DP)    , intent(in ) :: xr_Press     (1:imax,1:kmax)
    real(DP)    , intent(in ) :: xr_Temp      (1:imax,1:kmax)
    real(DP)    , intent(in ) :: x_SurfTemp   (1:imax)
    real(DP)    , intent(in ) :: xrm_VMR      (1:imax,1:kmax,1:NMol)
    real(DP)    , intent(out) :: xr_AtmPFRatio(1:imax,1:kmax)
    real(DP)    , intent(out) :: x_SurfPFRatio(1:imax)
    integer     , intent(in ) :: NIntpol
    integer     , intent(in ) :: NIntpol1Use
    integer     , intent(in ) :: NIntpol2Use
    integer     , intent(in ) :: NIntpol3Use
    integer     , intent(in ) :: NIntpol4Use
    integer     , intent(in ) :: NIntpol5Use
    integer     , intent(in ) :: xr_KDTblPressIndex  (1:imax,1:kmax)
    integer     , intent(in ) :: xr_KDTblTempIndex   (1:imax,1:kmax)
    integer     , intent(in ) :: xr_KDTblVMRMol1Index(1:imax,1:kmax)
    integer     , intent(in ) :: xr_KDTblVMRMol2Index(1:imax,1:kmax)
    integer     , intent(in ) :: xr_KDTblVMRMol3Index(1:imax,1:kmax)
    real(DP)    , intent(in ) :: ptuvwxr_Weight(NIntpol,NIntpol,NIntpol,NIntpol,NIntpol,1:imax,1:kmax)
    integer     , intent(in ) :: xr_KDTblPressIndexSfc  (1:imax,1)
    integer     , intent(in ) :: xr_KDTblTempIndexSfc   (1:imax,1)
    integer     , intent(in ) :: xr_KDTblVMRMol1IndexSfc(1:imax,1)
    integer     , intent(in ) :: xr_KDTblVMRMol2IndexSfc(1:imax,1)
    integer     , intent(in ) :: xr_KDTblVMRMol3IndexSfc(1:imax,1)
    real(DP)    , intent(in ) :: ptuvwx_WeightSfc(NIntpol,NIntpol,NIntpol,NIntpol,NIntpol,1:imax)


    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodKD )
      call OptPropKDGetPFIntedRatioWithIndex( &
        & iWaveNum, imax, kmax, NMol, m_MolNum, &
        & xr_Press, xr_Temp, x_SurfTemp, xrm_VMR, &
        & xr_AtmPFRatio, x_SurfPFRatio, &
        & NIntpol, &
        & NIntpol1Use, NIntpol2Use, NIntpol3Use, NIntpol4Use, NIntpol5Use, &
        & xr_KDTblPressIndex, xr_KDTblTempIndex, &
        & xr_KDTblVMRMol1Index, xr_KDTblVMRMol2Index, xr_KDTblVMRMol3Index, &
        & ptuvwxr_Weight, &
        & xr_KDTblPressIndexSfc, xr_KDTblTempIndexSfc, &
        & xr_KDTblVMRMol1IndexSfc, xr_KDTblVMRMol2IndexSfc, xr_KDTblVMRMol3IndexSfc, &
        & ptuvwx_WeightSfc &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  In OptPropGetPFIntedRatioForKD'
      write( 6, * ) '    Unexpected case default'
      stop
    end select


  end subroutine OptPropGetPFIntedRatioForKD2D

  !----------------------------------------------------------------------------

  subroutine OptPropGetStrPFIntedRatio( &
    & iWaveNum, &
    & StrPFRatio &
    & )

#ifdef KDONLY
#else
    use opt_prop_lblconstpf, only : OptPropLBLConstPFGetStrPFIntedRatio
    use opt_prop_kddevelop , only : OptPropKDDevelopGetStrPFIntedRatio
#endif
    use opt_prop_kd        , only : OptPropKDGetStrPFIntedRatio

    integer     , intent(in ) :: iWaveNum
    real(DP)    , intent(out) :: StrPFRatio


    ! local variables
    !


    if ( .not. FlagInited ) then
      write( 6, * ) trim( ModuleName ), " is not initialized."
      stop
    end if


    select case ( IDMethod )
    case ( IDMethodLBL )
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  function OptPropGetPFIntedRatio cannot be used when IDMethod is IDMethodLBL'
      stop
    end select


    select case ( IDMethod )
    case ( IDMethodLBLConstPF )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropLBLConstPFGetStrPFIntedRatio( &
        & iWaveNum, &
        & StrPFRatio &
        & )
#endif
    case ( IDMethodKDDevelop )
#ifdef KDONLY
      stop 'This is not supported.'
#else
      call OptPropKDDevelopGetStrPFIntedRatio( &
        & iWaveNum, &
        & StrPFRatio &
        & )
#endif
    case ( IDMethodKD )
      call OptPropKDGetStrPFIntedRatio( &
        & iWaveNum, &
        & StrPFRatio &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end subroutine OptPropGetStrPFIntedRatio

  !----------------------------------------------------------------------------

  subroutine OptPropPFTblGetPFDPFDT2DSpecRange( &
    & iBand, &
    & imax, &
    & is, ie, &
    & NLev, xa_Temp, &
    & xa_PF, &
    & FlagDPFDT &
    & )

    use opt_prop_kd, only : OptPropKDPFTblGetPFDPFDT

    integer , intent(in ) :: iBand
    integer , intent(in ) :: imax
    integer , intent(in ) :: is
    integer , intent(in ) :: ie
    integer , intent(in ) :: NLev
    real(dp), intent(in ) :: xa_Temp(imax,NLev)
    real(dp), intent(out) :: xa_PF  (imax,NLev)
    logical , intent(in ), optional :: FlagDPFDT


    ! Local variables
    !


    select case ( IDMethod )
    case ( IDMethodLBL, IDMethodLBLConstPF, IDMethodKDDevelop )
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  function OptPropPFTblGetPFDPFDT1D cannot be used ' &
        & //' when IDMethod is IDMethodLBL, IDMethodLBLConstPF, IDMethodKDDevelop'
      stop
    case ( IDMethodKD )
      call OptPropKDPFTblGetPFDPFDT( &
        & iBand, &
        & imax, &
        & is, ie, &
        & NLev, xa_Temp, &
        & xa_PF, &
        & FlagDPFDT &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end subroutine OptPropPFTblGetPFDPFDT2DSpecRange

  !----------------------------------------------------------------------------

  subroutine OptPropPFTblGetPFDPFDT1DSpecRange( &
    & iBand, &
    & imax, &
    & is, ie, &
    & x_Temp, &
    & x_PF, &
    & FlagDPFDT &
    & )

    use opt_prop_kd, only : OptPropKDPFTblGetPFDPFDT

    integer , intent(in ) :: iBand
    integer , intent(in ) :: imax
    integer , intent(in ) :: is
    integer , intent(in ) :: ie
    real(dp), intent(in ) :: x_Temp(imax)
    real(dp), intent(out) :: x_PF  (imax)
    logical , intent(in ), optional :: FlagDPFDT


    ! Local variables
    !


    select case ( IDMethod )
    case ( IDMethodLBL, IDMethodLBLConstPF, IDMethodKDDevelop )
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  function OptPropPFTblGetPFDPFDT1D cannot be used ' &
        & //' when IDMethod is IDMethodLBL, IDMethodLBLConstPF, IDMethodKDDevelop'
      stop
    case ( IDMethodKD )
      call OptPropKDPFTblGetPFDPFDT( &
        & iBand, &
        & imax, &
        & is, ie, &
        & x_Temp, &
        & x_PF, &
        & FlagDPFDT &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end subroutine OptPropPFTblGetPFDPFDT1DSpecRange

  !----------------------------------------------------------------------------

  subroutine OptPropPFTblGetPFDPFDT1D( &
    & iBand, NLev, a_Temp, &
    & a_PF, &
    & FlagDPFDT &
    & )

    use opt_prop_kd, only : OptPropKDPFTblGetPFDPFDT

    integer , intent(in ) :: iBand
    integer , intent(in ) :: NLev
    real(dp), intent(in ) :: a_Temp(NLev)
    real(dp), intent(out) :: a_PF  (NLev)
    logical , intent(in ), optional :: FlagDPFDT


    ! Local variables
    !


    select case ( IDMethod )
    case ( IDMethodLBL, IDMethodLBLConstPF, IDMethodKDDevelop )
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  function OptPropPFTblGetPFDPFDT1D cannot be used ' &
        & //' when IDMethod is IDMethodLBL, IDMethodLBLConstPF, IDMethodKDDevelop'
      stop
    case ( IDMethodKD )
      call OptPropKDPFTblGetPFDPFDT( &
        & iBand, NLev, a_Temp, &
        & a_PF, &
        & FlagDPFDT &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end subroutine OptPropPFTblGetPFDPFDT1D

  !----------------------------------------------------------------------------

  subroutine OptPropPFTblGetPFDPFDT0D( &
    & iBand, Temp, &
    & PF, &
    & FlagDPFDT &
    & )

    use opt_prop_kd, only : OptPropKDPFTblGetPFDPFDT

    integer , intent(in ) :: iBand
    real(dp), intent(in ) :: Temp
    real(dp), intent(out) :: PF
    logical , intent(in ), optional :: FlagDPFDT


    ! Local variables
    !


    select case ( IDMethod )
    case ( IDMethodLBL, IDMethodLBLConstPF, IDMethodKDDevelop )
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  function OptPropPFTblGetPFDPFDT0D cannot be used ' &
        & //' when IDMethod is IDMethodLBL, IDMethodLBLConstPF, IDMethodKDDevelop'
      stop
    case ( IDMethodKD )
      call OptPropKDPFTblGetPFDPFDT( &
        & iBand, Temp, &
        & PF, &
        & FlagDPFDT &
        & )
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select



  end subroutine OptPropPFTblGetPFDPFDT0D

  !----------------------------------------------------------------------------

  subroutine OptPropPFTblChkBandIntPFRatio( &
    & iBand, ThresholdBandIntPFRatio, &
    & TempMax, &
    & FlagBandIntPFRatio &
    & )

    use opt_prop_kd, only : OptPropKDPFTblChkBandIntPFRatio

    integer , intent(in ) :: iBand
    real(dp), intent(in ) :: ThresholdBandIntPFRatio
    real(dp), intent(in ) :: TempMax
    logical , intent(out) :: FlagBandIntPFRatio


    ! Local variables
    !


    select case ( IDMethod )
    case ( IDMethodKD )
      call OptPropKDPFTblChkBandIntPFRatio( &
        & iBand, ThresholdBandIntPFRatio, &
        & TempMax, &
        & FlagBandIntPFRatio &
        & )
    case ( IDMethodLBL, IDMethodLBLConstPF, IDMethodKDDevelop )
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  function OptPropPFTblChkBandIntPFRatio cannot be used ' &
        & //' when IDMethod is IDMethodLBL, IDMethodLBLConstPF, IDMethodKDDevelop'
      stop
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end subroutine OptPropPFTblChkBandIntPFRatio

  !----------------------------------------------------------------------------

  subroutine OptPropPFTblChkBandIntStrFluxRatio( &
    & iBand, ThresholdBandIntStrFluxRatio, &
    & FlagBandIntStrFluxRatio &
    & )

    use opt_prop_kd, only : OptPropKDPFTblChkBandIntStrFluxRatio

    integer , intent(in ) :: iBand
    real(dp), intent(in ) :: ThresholdBandIntStrFluxRatio
    logical , intent(out) :: FlagBandIntStrFluxRatio


    ! Local variables
    !


    select case ( IDMethod )
    case ( IDMethodKD )
      call OptPropKDPFTblChkBandIntStrFluxRatio( &
        & iBand, ThresholdBandIntStrFluxRatio, &
        & FlagBandIntStrFluxRatio &
        & )
    case ( IDMethodLBL, IDMethodLBLConstPF, IDMethodKDDevelop )
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  function OptPropPFTblChkBandIntStrFluxRatio cannot be used ' &
        & //' when IDMethod is IDMethodLBL, IDMethodLBLConstPF, IDMethodKDDevelop'
      stop
    case default
      write( 6, * ) 'In ', trim( ModuleName )
      write( 6, * ) '  Unexpected case default'
      stop
    end select


  end subroutine OptPropPFTblChkBandIntStrFluxRatio

  !----------------------------------------------------------------------------

end module opt_prop

