!= 主成分雲モデル
!
!= Atmospheric major component cloud
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: major_comp_phase_change.f90,v 1.3 2014/05/07 09:39:21 murashin Exp $ 
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module cloud_major_comp
  !
  != 主成分雲モデル
  !
  != Atmospheric major component cloud
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  !== Procedures List
  ! 
!!$  ! DryConvAdjust :: 乾燥対流調節
!!$  ! ------------  :: ------------
!!$  ! DryConvAdjust :: Dry convective adjustment
  !
  !== NAMELIST
  !
  ! NAMELIST#cloud_major_comp_nml
  !

  ! モジュール引用 ; USE statements
  !

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数. 
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数. 
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数. 
                             ! Number of vertical level

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING     ! 文字列.       Strings. 

  ! NAMELIST ファイル入力に関するユーティリティ
  ! Utilities for NAMELIST file input
  !
  use namelist_util, only: MaxNmlArySize
                              ! NAMELIST から読み込む配列の最大サイズ. 
                              ! Maximum size of arrays loaded from NAMELIST

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private


  logical, save :: FlagSedimentationInfty


  ! 公開手続き
  ! Public procedure
  !
  public :: CloudMajorComp
  public :: CloudMajorCompNoCloud
  public :: CloudMajorCompInit
!!$  public :: MajorCompPhaseChangeOnGround
!!$  public :: MajorCompPhaseChangeInit


  ! 公開変数
  ! Public variables
  !
  logical, save, public:: cloud_major_comp_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag

  ! 非公開変数
  ! Private variables
  !
  integer,              save :: NThreads
  integer, allocatable, save :: a_ls(:)
  integer, allocatable, save :: a_le(:)

  logical, save :: FlagMajCompPhaseChange
  logical, save :: FlagModMom
  logical, save :: FlagModTemp = .false.

  real(DP), save :: MajCompIceDen
  real(DP), save :: CloudNumRatio
  real(DP), save :: CloudNuclRad
  real(DP), save :: CloudNuclDen

  real(DP), save :: TimeStepSmall


  character(*), parameter:: module_name = 'cloud_major_comp'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: major_comp_phase_change.f90,v 1.3 2014/05/07 09:39:21 murashin Exp $'
                              ! モジュールのバージョン
                              ! Module version


contains

  !--------------------------------------------------------------------------------------

  subroutine CloudMajorComp(                     &
    & xy_SurfHeight,                                 & ! (in)
    & xy_Ps, xyz_Temp, xyzf_QMix, xyz_U, xyz_V,      & ! (inout)
    & xy_SurfMajCompIce                              & ! (inout)
    & )
    !
    ! CO2 相変化
    !
    ! CO2 phase change
    !

    ! モジュール引用 ; USE statements
    !

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & DelTime, &            ! $ \Delta t $
      & TimeN, &              ! ステップ $ t $ の時刻. Time of step $ t $. 
      & TimesetClockStart, TimesetClockStop

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 組成に関わる配列の設定
    ! Settings of array for atmospheric composition
    !
    use composition, only : &
      & ncmax, &
      & IndexH2OVap, &
      & IndexTKE, &
      & IndexMajCompIce

    ! 温度の半整数σレベルの補間, 気圧と高度の算出
    ! Interpolate temperature on half sigma level, 
    ! and calculate pressure and height
    !
    use auxiliary, only: AuxVars


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in   ):: xy_SurfHeight    (0:imax-1, 1:jmax)
    real(DP), intent(inout):: xy_Ps            (0:imax-1, 1:jmax)
                              ! $ T $ .     温度. Temperature
    real(DP), intent(inout):: xyz_Temp         (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ .     温度. Temperature
    real(DP), intent(inout):: xyzf_QMix        (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
    real(DP), intent(inout):: xyz_U            (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout):: xyz_V            (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout):: xy_SurfMajCompIce(0:imax-1, 1:jmax)
                              !
                              ! Surface major component ice amount

    ! 作業変数
    ! Work variables
    !
    real(DP) :: TimeStep
    integer  :: iSubLoop
    integer  :: NSubLoop
    real(DP) :: TimeStepSubLoop

    real(DP) :: xyz_TempB           (0:imax-1, 1:jmax, 1:kmax)
                              ! 調節前の温度. 
                              ! Temperature before adjustment
    real(DP) :: xyzf_QMixB          (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)

    real(DP) :: xyr_Press (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ . 気圧 (半整数レベル). 
                              ! Air pressure (half level)
    real(DP) :: xyz_Press (0:imax-1, 1:jmax, 1:kmax)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)
    real(DP) :: xyr_Height(0:imax-1, 1:jmax, 0:kmax)

    real(DP) :: xyz_PartRad         (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_PartDen         (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xyz_DTempDt         (0:imax-1, 1:jmax, 1:kmax)
                              ! 温度変化率. 
                              ! Temperature tendency
    real(DP):: xyzf_DQMixDt        (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
    real(DP):: xy_DSurfMajCompIceDt(0:imax-1, 1:jmax)


    ! 実行文 ; Executable statement
    !

    ! 初期化
    ! Initialization
    !
    if ( .not. cloud_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    if ( .not. FlagMajCompPhaseChange ) return

    if ( IndexMajCompIce <= 0 ) then
      call MessageNotify( 'E', module_name, 'Major component ice is not contained in xyzf_QMix.' )
    end if

    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    ! Store variables
    !
    xyz_TempB  = xyz_Temp
    xyzf_QMixB = xyzf_QMix


    TimeStep = 2.0_DP * DelTime
    NSubLoop = max( int( TimeStep / TimeStepSmall ), 1 )
    TimeStepSubLoop = TimeStep / dble( NSubLoop )

    do iSubLoop = 1, NSubLoop

      ! 温度の半整数σレベルの補間, 気圧と高度の算出
      ! Interpolate temperature on half sigma level,
      ! and calculate pressure and height
      !
      call AuxVars( &
        & xy_Ps, xyz_Temp, xyzf_QMix(:,:,:,IndexH2OVap),   & ! (in )
        & xyr_Press     = xyr_Press,                       & ! (out) optional
        & xyz_Press     = xyz_Press,                       & ! (out) optional
        & xy_SurfHeight = xy_SurfHeight,                   & ! (in ) optional
        & xyr_Height    = xyr_Height                       & ! (out) optional
        & )

      call CloudMajorCompCore(                     &
        & TimeStepSubLoop,                               & ! (in)
        & xyr_Press, xyz_Press, xyr_Height,              & ! (in)
        & xy_Ps, xyz_Temp, xyzf_QMix, xyz_U, xyz_V,      & ! (inout)
        & xy_SurfMajCompIce,                             & ! (inout)
        & xyz_PartRad, xyz_PartDen,                      & ! (out)
        & xy_DSurfMajCompIceDt                           & ! (out)
        & )

    end do



    ! ヒストリデータ出力
    ! History data output
    !
    xyz_DTempDt = ( xyz_Temp - xyz_TempB ) / ( 2.0_DP * DelTime )
    call HistoryAutoPut( TimeN, 'DTempDtMajCompPhaseChange', xyz_DTempDt )

    xyzf_DQMixDt = ( xyzf_QMix - xyzf_QMixB ) / ( 2.0_DP * DelTime )
    call HistoryAutoPut( TimeN, 'DQMajCompIceDtMajCompPhaseChange', xyzf_DQMixDt(:,:,:,IndexMajCompIce) )

    call HistoryAutoPut( TimeN, 'MajCompCloudRadiusForGravSed', xyz_PartRad )


    call HistoryAutoPut( TimeN, 'MajCompCloudRadius', xyz_PartRad )
    call HistoryAutoPut( TimeN, 'DSurfMajCompIceDtGravSed', xy_DSurfMajCompIceDt )


    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine CloudMajorComp

  !--------------------------------------------------------------------------------------

  subroutine CloudMajorCompCore(                     &
    & TimeStep,                                      & ! (in)
    & xyr_Press, xyz_Press, xyr_Height,              & ! (in)
    & xy_Ps, xyz_Temp, xyzf_QMix, xyz_U, xyz_V,      & ! (inout)
    & xy_SurfMajCompIce,                             & ! (inout)
    & xyz_PartRad, xyz_PartDen,                      & ! (out)
    & xy_DSurfMajCompIceDt                           & ! (out)
    & )
    !
    ! CO2 相変化
    !
    ! CO2 phase change
    !

    ! モジュール引用 ; USE statements
    !

    ! 組成に関わる配列の設定
    ! Settings of array for atmospheric composition
    !
    use composition, only : &
      & ncmax, &
      & IndexTKE, &
      & IndexMajCompIce

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Grav, &               ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration
      & CpDry
                              ! $ C_p $ [J kg-1 K-1].
                              ! 乾燥大気の定圧比熱.
                              ! Specific heat of air at constant pressure

    ! 温度の半整数σレベルの補間, 気圧と高度の算出
    ! Interpolate temperature on half sigma level, 
    ! and calculate pressure and height
    !
    use auxiliary, only: AuxVars

    ! 主成分相変化
    ! Phase change of atmospheric major component
    !
    use saturate_major_comp, only :    &
      & SaturateMajorCompCondTemp,     &
      & SaturateMajorCompInqLatentHeat

    ! 質量の補正
    ! Mass fixer
    !
    use mass_fixer, only: MassFixerColumn

    ! 雲関系ルーチン
    ! Cloud-related routines
    !
    use cloud_utils, only : CloudUtils1BinCalcPartProp1D


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in   ):: TimeStep

    real(DP), intent(in   ):: xyr_Press (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ . 気圧 (半整数レベル). 
                              ! Air pressure (half level)
    real(DP), intent(in   ):: xyz_Press (0:imax-1, 1:jmax, 1:kmax)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)
    real(DP), intent(in   ):: xyr_Height(0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(inout):: xy_Ps            (0:imax-1, 1:jmax)
                              ! $ T $ .     温度. Temperature
    real(DP), intent(inout):: xyz_Temp         (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ .     温度. Temperature
    real(DP), intent(inout):: xyzf_QMix        (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
    real(DP), intent(inout):: xyz_U            (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout):: xyz_V            (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout):: xy_SurfMajCompIce(0:imax-1, 1:jmax)
                              !
                              ! Surface major component ice amount
    real(DP), intent(out  ):: xyz_PartRad         (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(out  ):: xyz_PartDen         (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(out  ):: xy_DSurfMajCompIceDt(0:imax-1, 1:jmax)
                              ! 
                              ! Surface major component ice tendency


    ! 作業変数
    ! Work variables
    !
    real(DP):: LatentHeatMajCompSubl

    real(DP):: xyr_MajCompIceGravSedMassFlux(0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyr_DPPress         (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_MajCompIceMass (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: r_Press           (0:kmax)
    real(DP):: z_Press           (1:kmax)
    real(DP):: z_Temp            (1:kmax)
    real(DP):: r_Height          (0:kmax)

    real(DP):: z_DelAtmMass      (1:kmax)
    real(DP):: z_DelNuclNum      (1:kmax)

    real(DP):: z_TempCond        (1:kmax)
    real(DP):: z_QMajCompIce     (1:kmax)
    real(DP):: z_MajCompIceMass  (1:kmax)

    real(DP):: z_PartRad         (1:kmax)
    real(DP):: z_PartDen         (1:kmax)

    real(DP):: r_DPPress                  (0:kmax)
    real(DP):: r_MajCompIceGravSedMassFlux(0:kmax)


    real(DP):: xy_PsB              (0:imax-1, 1:jmax)
    real(DP):: xy_PsA              (0:imax-1, 1:jmax)
    real(DP):: xyr_PressB          (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_PressA          (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_TempB           (0:imax-1, 1:jmax, 1:kmax)
                              ! 調節前の温度. 
                              ! Temperature before adjustment
    real(DP):: xyzf_QMixB          (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)

    real(DP):: xyz_DelAtmMass      (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xyz_TempTmp         (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_QH2OVapTmp      (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xyz_DTempDt         (0:imax-1, 1:jmax, 1:kmax)
                              ! 温度変化率. 
                              ! Temperature tendency
    real(DP):: xy_DPsDt            (0:imax-1, 1:jmax)

    real(DP):: xyr_DPressDt         (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: z_DelTempSubl(1:kmax)
    real(DP):: z_DelTempCond(1:kmax)

    integer :: mmax
    real(DP):: xyza_Array       (0:imax-1, 1:jmax, 1:kmax, 1:ncmax+1+1+1)
    logical :: a_FlagSurfaceSink                          (1:ncmax+1+1+1)
    real(DP):: xyra_MassFlow    (0:imax-1, 1:jmax, 0:kmax, 1:ncmax+1+1+1)

    real(DP):: xyrf_MassFlow    (0:imax-1, 1:jmax, 0:kmax, 1:ncmax)
    real(DP):: xyr_TempFlow     (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_MomXFlow     (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_MomYFlow     (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_DelAtmMassB   (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_DelAtmMassA   (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xy_ColumnLatEneB(0:imax-1, 1:jmax)
    real(DP):: xy_ColumnIntEneB(0:imax-1, 1:jmax)
    real(DP):: xy_ColumnLatEneA(0:imax-1, 1:jmax)
    real(DP):: xy_ColumnIntEneA(0:imax-1, 1:jmax)
    real(DP):: xy_LostLatEneA  (0:imax-1, 1:jmax)
    real(DP):: xy_LostIntEneA  (0:imax-1, 1:jmax)
    real(DP):: TotalEneB
    real(DP):: TotalEneA
    real(DP):: Ratio

    real(DP) :: CondMaterialDen
    real(DP) :: CloudRad

    integer:: iThread

    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction
    integer:: l
    integer:: m
    integer:: n


    ! 実行文 ; Executable statement
    !

    ! 初期化
    ! Initialization
    !
    if ( .not. cloud_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    if ( .not. FlagMajCompPhaseChange ) return

    if ( IndexMajCompIce <= 0 ) then
      call MessageNotify( 'E', module_name, 'Major component ice is not contained in xyzf_QMix.' )
    end if


    ! Store variables
    !
    xyz_TempB  = xyz_Temp
    xyzf_QMixB = xyzf_QMix


    ! Set latent heat
    LatentHeatMajCompSubl = SaturateMajorCompInqLatentHeat()


    CondMaterialDen = MajCompIceDen
    CloudRad        = -1.0_DP

    !$OMP PARALLEL DEFAULT(PRIVATE) &
    !$OMP SHARED( &
    !$OMP         NThreads, TimeStep, a_ls, a_le, &
    !$OMP         imax, jmax, kmax, &
    !$OMP         IndexMajCompIce, &
    !$OMP         Grav, CpDry, &
    !$OMP         LatentHeatMajCompSubl, &
    !$OMP         CloudNumRatio, &
    !$OMP         CondMaterialDen, CloudRad, CloudNuclRad, CloudNuclDen, &
    !$OMP         xyr_Press, xyz_Press, xyr_Height, &
    !$OMP         xyz_Temp, xyzf_QMix, &
    !$OMP         xyz_DelAtmMass, &
    !$OMP         xyz_PartRad, xyz_PartDen, &
    !$OMP         xyz_MajCompIceMass, &
    !$OMP         xyr_DPPress, xyr_MajCompIceGravSedMassFlux &
    !$OMP       )

    !$OMP DO
    do iThread = 0, NThreads-1

      do l = a_ls(iThread), a_le(iThread)

        i = mod( (l-1), imax ) + 1 - 1
        j = int( (l-1) / imax ) + 1


        do k = 0, kmax
          r_Press (k) = xyr_Press (i,j,k)
          r_Height(k) = xyr_Height(i,j,k)
        end do
        do k = 1, kmax
          z_Press         (k) = xyz_Press         (i,j,k)
          z_Temp          (k) = xyz_Temp          (i,j,k)
          z_QMajCompIce   (k) = xyzf_QMix         (i,j,k,IndexMajCompIce)
        end do


        do k = 1, kmax
          z_DelAtmMass(k) = ( r_Press(k-1) - r_Press(k) ) / Grav
        end do

        z_MajCompIceMass = z_DelAtmMass * z_QMajCompIce

        call SaturateMajorCompCondTemp( &
          & z_Press,                    & ! (in)
          & z_TempCond                  & ! (out)
          & )

        ! Phase change
        !   (This is not needed?)
        !
        !   sublimation of all cloud ice
        z_DelTempSubl = &
          & - LatentHeatMajCompSubl * z_MajCompIceMass &
          &   / ( CpDry * z_DelAtmMass )
        z_Temp = z_Temp + z_DelTempSubl
        z_MajCompIceMass = 0.0_DP
        !
        !   condensation
        z_DelTempCond = max( z_TempCond - z_Temp, 0.0_DP )
        z_Temp = z_Temp + z_DelTempCond
        z_MajCompIceMass = z_MajCompIceMass    &
          & + CpDry * z_DelAtmMass * z_DelTempCond  &
          &   / LatentHeatMajCompSubl


        z_DelNuclNum    = z_DelAtmMass * CloudNumRatio

        ! update mixing ratio, temporarily
        z_QMajCompIce   = z_MajCompIceMass / z_DelAtmMass
        !
        call CloudUtils1BinCalcPartProp1D(             &
          & CondMaterialDen, CloudRad,                 & ! (in )
          & CloudNuclRad, CloudNuclDen,                & ! (in )
          & z_DelAtmMass, z_QMajCompIce, z_DelNuclNum, & ! (in )
          & z_PartRad, z_PartDen                       & ! (out)
          & )

        ! 重力沈降過程
        ! Gravitational sedimentation process
        !
        call CloudMajorCompCalcSedMassFlux1D( &
          & TimeStep,                        & ! (in   )
          & LatentHeatMajCompSubl,           & ! (in   )
          & r_Press, r_Height,               & ! (in   )
          & z_PartDen, z_PartRad,            & ! (in   )
          & z_TempCond,                      & ! (in   )
          & z_Temp,                          & ! (inout)
          & z_MajCompIceMass,                & ! (inout)
          & r_MajCompIceGravSedMassFlux,     & ! (out  )
          & r_DPPress                        & ! (out  )
          & )

        do k = 1, kmax
          xyz_DelAtmMass    (i,j,k) = z_DelAtmMass    (k)
        end do
        do k = 1, kmax
          xyz_PartDen       (i,j,k) = z_PartDen       (k)
          xyz_PartRad       (i,j,k) = z_PartRad       (k)
          xyz_Temp          (i,j,k) = z_Temp          (k)
          xyz_MajCompIceMass(i,j,k) = z_MajCompIceMass(k)
        end do
        do k = 0, kmax
          xyr_DPPress                  (i,j,k) = r_DPPress(k)
          xyr_MajCompIceGravSedMassFlux(i,j,k) = r_MajCompIceGravSedMassFlux(k)
        end do

      end do
    end do
    !$OMP END DO
    !$OMP END PARALLEL


    xyr_DPressDt         = xyr_MajCompIceGravSedMassFlux * Grav
    xy_DPsDt             = xyr_DPressDt(:,:,0)
    xy_DSurfMajCompIceDt = - xy_DPsDt / Grav


!!$    ! update mixing ratio
!!$    xyz_QMajCompIce = xyz_MajCompIceMass / xyz_DelAtmMass
    ! NOTE
    !  xyz_QMajCompIce and xyzf_QMix need not be updated, here, because 
    !  xyzf_QMix(:,:,:,IndexMajCompIce) will be updated with xyz_MajCompIceMass
    !  below.


    ! packing
    mmax = ncmax
    if ( FlagModTemp ) then
      mmax = mmax + 1
    end if
    if ( FlagModMom ) then
      mmax = mmax + 2
    end if
    do m = 1, ncmax
      n = m
      xyza_Array(:,:,:,m) = xyzf_QMix(:,:,:,n)
      if ( ( n == IndexTKE ) .or. ( n == IndexMajCompIce ) ) then
        a_FlagSurfaceSink(m) = .true.
      else
        a_FlagSurfaceSink(m) = .false.
      end if
    end do
    m = ncmax
    if ( FlagModTemp ) then
      m = m + 1
      xyza_Array(:,:,:,m) = xyz_Temp
      a_FlagSurfaceSink(m) = .true.
    end if
    if ( FlagModMom ) then
      m = m + 1
      xyza_Array(:,:,:,m) = xyz_U
      a_FlagSurfaceSink(m) = .true.
      m = m + 1
      xyza_Array(:,:,:,m) = xyz_V
      a_FlagSurfaceSink(m) = .true.
    end if

    call CloudMajorCompCalcFlow(         &
      & xyr_Press, xyr_DPPress,                                      & ! (in)
      & mmax, a_FlagSurfaceSink(1:mmax), xyza_Array(:,:,:,1:mmax),   & ! (in)
      & xyra_MassFlow(:,:,:,1:mmax)                                  & ! (out)
      & )

    ! unpacking
    do m = 1, ncmax
      xyrf_MassFlow(:,:,:,m) = xyra_MassFlow(:,:,:,m)
    end do
    m = ncmax
    if ( FlagModTemp ) then
      m = m + 1
      xyr_TempFlow = xyra_MassFlow(:,:,:,m)
    else
      xyr_TempFlow = 0.0_DP
    end if
    if ( FlagModMom ) then
      m = m + 1
      xyr_MomXFlow = xyra_MassFlow(:,:,:,m)
      m = m + 1
      xyr_MomYFlow = xyra_MassFlow(:,:,:,m)
    else
      xyr_MomXFlow = 0.0_DP
      xyr_MomYFlow = 0.0_DP
    end if


    ! Adjustment
    !   preparation
    xy_PsB = xy_Ps
!!$    xy_PsA = xy_PsB + xy_DPsDt * ( 2.0_DP * DelTime )
    xy_PsA = xy_PsB + xy_DPsDt * TimeStep

    ! 温度の半整数σレベルの補間, 気圧と高度の算出
    ! Interpolate temperature on half sigma level, 
    ! and calculate pressure and height
    !
    xyz_TempTmp    = 300.0_DP
    xyz_QH2OVapTmp =   0.0_DP
    call AuxVars( &
      & xy_PsB, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressB                  & ! (out) optional
      & )
    call AuxVars( &
      & xy_PsA, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressA                  & ! (out) optional
      & )
    do k = 1, kmax
      xyz_DelAtmMassB(:,:,k) = ( xyr_PressB(:,:,k-1) - xyr_PressB(:,:,k) ) / Grav
      xyz_DelAtmMassA(:,:,k) = ( xyr_PressA(:,:,k-1) - xyr_PressA(:,:,k) ) / Grav
    end do
    !   Atmospheric composition
    do n = 1, ncmax
      if ( n == IndexMajCompIce ) then
        ! major component is treated differently from others
        xyzf_QMix(:,:,:,n) = xyz_MajCompIceMass / xyz_DelAtmMassA
      else
        do k = 1, kmax
          xyzf_QMix(:,:,k,n) =                                              &
            &   (   xyz_DelAtmMassB(:,:,k) * xyzf_QMix(:,:,k,n)             &
            &     - ( xyrf_MassFlow(:,:,k,n) - xyrf_MassFlow(:,:,k-1,n) ) ) &
            & / xyz_DelAtmMassA(:,:,k)
        end do
      end if
    end do
    if ( FlagModTemp ) then
      do k = 1, kmax
        ! Temperature
        xyz_Temp(:,:,k) =                                           &
          &   (   xyz_DelAtmMassB(:,:,k) * xyz_Temp(:,:,k)          &
          &     - ( xyr_TempFlow(:,:,k) - xyr_TempFlow(:,:,k-1) ) ) &
          & / xyz_DelAtmMassA(:,:,k)
      end do
    end if
    if ( FlagModMom ) then
      do k = 1, kmax
        ! Zonal wind
        xyz_U(:,:,k) =                                              &
          &   (   xyz_DelAtmMassB(:,:,k) * xyz_U(:,:,k)             &
          &     - ( xyr_MomXFlow(:,:,k) - xyr_MomXFlow(:,:,k-1) ) ) &
          & / xyz_DelAtmMassA(:,:,k)
        ! Meridional wind
        xyz_V(:,:,k) =                                              &
          &   (   xyz_DelAtmMassB(:,:,k) * xyz_V(:,:,k)             &
          &     - ( xyr_MomYFlow(:,:,k) - xyr_MomYFlow(:,:,k-1) ) ) &
          & / xyz_DelAtmMassA(:,:,k)
      end do
    end if


    ! Surface major component ice adjustment
!!$    xy_SurfMajCompIce = xy_SurfMajCompIce + xy_DSurfMajCompIceDt * ( 2.0_DP * DelTime )
    xy_SurfMajCompIce = xy_SurfMajCompIce + xy_DSurfMajCompIceDt * TimeStep
    ! Surface pressure adjustment
    xy_Ps = xy_PsA


    call AuxVars( &
      & xy_PsA, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressA                  & ! (out) optional
      & )

    ! 成分の質量の補正
    ! Fix masses of constituents
    !
    call MassFixerColumn( &
      & xyr_PressA, & ! (in)
      & xyzf_QMix,  & ! (inout)
      & Text = "In CloudMajorComp"  & ! (in)
      & )

    ! Check
    call CloudMajorCompConsChk(       &
      & a_FlagSurfaceSink,            & ! (in)
      & xyz_DelAtmMassB, xyzf_QMixB,  & ! (in)
      & xyz_DelAtmMassA, xyzf_QMix    & ! (in)
      & )

    ! Check major component ice
    xy_ColumnLatEneB = 0.0_DP
    xy_ColumnIntEneB = 0.0_DP
    !
    xy_ColumnLatEneA = 0.0_DP
    xy_ColumnIntEneA = 0.0_DP
    do k = kmax, 1, -1
      xy_ColumnLatEneB = xy_ColumnLatEneB &
        & + LatentHeatMajCompSubl &
        &   * xyzf_QMixB(:,:,k,IndexMajCompIce) * xyz_DelAtmMassB(:,:,k)
      xy_ColumnIntEneB = xy_ColumnIntEneB &
        & + CpDry * xyz_TempB(:,:,k) * xyz_DelAtmMassB(:,:,k)
      !
      xy_ColumnLatEneA = xy_ColumnLatEneA &
        & + LatentHeatMajCompSubl &
        &   * xyzf_QMix (:,:,k,IndexMajCompIce) * xyz_DelAtmMassA(:,:,k)
        ! xyz_DelAtmMassA should be used. But, in order to do that,
        ! we need some modifications, maybe loss of internal energy
        ! should be considered. 
      xy_ColumnIntEneA = xy_ColumnIntEneA &
!        & + CpDry * xyz_Temp (:,:,k) * xyz_DelAtmMassA(:,:,k)
        & + CpDry * xyz_Temp (:,:,k) * xyz_DelAtmMassB(:,:,k)
    end do
    !
    xy_LostLatEneA = &
      &   LatentHeatMajCompSubl &
!!$      &   * xy_DSurfMajCompIceDt * ( 2.0_DP * DelTime )
      &   * xy_DSurfMajCompIceDt * TimeStep
    xy_LostIntEneA = &
      & - CpDry * xyr_TempFlow(:,:,0)
    do j = 1, jmax
      do i = 0, imax-1
        TotalEneB =   xy_ColumnIntEneB(i,j) &
          &         - xy_ColumnLatEneB(i,j)
        TotalEneA =   xy_ColumnIntEneA(i,j) + xy_LostIntEneA(i,j)   &
          &         - ( xy_ColumnLatEneA(i,j) + xy_LostLatEneA(i,j) )
        Ratio = abs( TotalEneB - TotalEneA ) / TotalEneB
        if ( Ratio > 1.0e-10_DP ) then
          call MessageNotify( 'M', module_name, &
            & 'Major component ice phase change does not conserve energy, %f, at (%d,%d).', &
            & d = (/ Ratio /), i = (/ i, j /) )
        end if
      end do
    end do


    ! Calculation of radius and density for output
    do j = 1, jmax
      do i = 0, imax-1
        do k = 0, kmax
          r_Press (k) = xyr_Press (i,j,k)
        end do
        do k = 1, kmax
          z_QMajCompIce   (k) = xyzf_QMix         (i,j,k,IndexMajCompIce)
        end do
        do k = 1, kmax
          z_DelAtmMass(k) = ( r_Press(k-1) - r_Press(k) ) / Grav
        end do
        z_DelNuclNum = z_DelAtmMass * CloudNumRatio
        call CloudUtils1BinCalcPartProp1D(             &
          & CondMaterialDen, CloudRad,                 & ! (in )
          & CloudNuclRad, CloudNuclDen,                & ! (in )
          & z_DelAtmMass, z_QMajCompIce, z_DelNuclNum, & ! (in )
          & z_PartRad, z_PartDen                       & ! (out)
          & )
        do k = 1, kmax
          xyz_PartDen       (i,j,k) = z_PartDen       (k)
          xyz_PartRad       (i,j,k) = z_PartRad       (k)
        end do
      end do
    end do


  end subroutine CloudMajorCompCore

  !--------------------------------------------------------------------------------------

  subroutine BKNoSubStep_CloudMajorComp(                         &
    & xyr_Press, xyz_Press, xyr_Height,              & ! (in)
    & xy_Ps, xyz_Temp, xyzf_QMix, xyz_U, xyz_V,      & ! (inout)
    & xy_SurfMajCompIce                              & ! (inout)
    & )
    !
    ! CO2 相変化
    !
    ! CO2 phase change
    !

    ! モジュール引用 ; USE statements
    !

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & DelTime, &            ! $ \Delta t $
      & TimeN, &              ! ステップ $ t $ の時刻. Time of step $ t $. 
      & TimesetClockStart, TimesetClockStop

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 組成に関わる配列の設定
    ! Settings of array for atmospheric composition
    !
    use composition, only : &
      & ncmax, &
      & IndexTKE, &
      & IndexMajCompIce

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Grav, &               ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration
      & CpDry
                              ! $ C_p $ [J kg-1 K-1].
                              ! 乾燥大気の定圧比熱.
                              ! Specific heat of air at constant pressure

    ! 温度の半整数σレベルの補間, 気圧と高度の算出
    ! Interpolate temperature on half sigma level, 
    ! and calculate pressure and height
    !
    use auxiliary, only: AuxVars

    ! 主成分相変化
    ! Phase change of atmospheric major component
    !
    use saturate_major_comp, only :    &
      & SaturateMajorCompCondTemp,     &
      & SaturateMajorCompInqLatentHeat

    ! 質量の補正
    ! Mass fixer
    !
    use mass_fixer, only: MassFixerColumn

    ! 雲関系ルーチン
    ! Cloud-related routines
    !
    use cloud_utils, only : CloudUtils1BinCalcPartProp1D


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in   ):: xyr_Press (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ . 気圧 (半整数レベル). 
                              ! Air pressure (half level)
    real(DP), intent(in   ):: xyz_Press (0:imax-1, 1:jmax, 1:kmax)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)
    real(DP), intent(in   ):: xyr_Height(0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(inout):: xy_Ps            (0:imax-1, 1:jmax)
                              ! $ T $ .     温度. Temperature
    real(DP), intent(inout):: xyz_Temp         (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ .     温度. Temperature
    real(DP), intent(inout):: xyzf_QMix        (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
    real(DP), intent(inout):: xyz_U            (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout):: xyz_V            (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout):: xy_SurfMajCompIce(0:imax-1, 1:jmax)
                              !
                              ! Surface major component ice amount

    ! 作業変数
    ! Work variables
    !
    real(DP):: LatentHeatMajCompSubl

    real(DP):: TimeStep

    real(DP):: xyr_MajCompIceGravSedMassFlux(0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_PartRad         (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_PartDen         (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyr_DPPress         (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_MajCompIceMass (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: r_Press           (0:kmax)
    real(DP):: z_Press           (1:kmax)
    real(DP):: z_Temp            (1:kmax)
    real(DP):: r_Height          (0:kmax)

    real(DP):: z_DelAtmMass      (1:kmax)
    real(DP):: z_DelNuclNum      (1:kmax)

    real(DP):: z_TempCond        (1:kmax)
    real(DP):: z_QMajCompIce     (1:kmax)
    real(DP):: z_MajCompIceMass  (1:kmax)

    real(DP):: z_PartRad         (1:kmax)
    real(DP):: z_PartDen         (1:kmax)

    real(DP):: r_DPPress                  (0:kmax)
    real(DP):: r_MajCompIceGravSedMassFlux(0:kmax)


    real(DP):: xy_PsB              (0:imax-1, 1:jmax)
    real(DP):: xy_PsA              (0:imax-1, 1:jmax)
    real(DP):: xyr_PressB          (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_PressA          (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_TempB           (0:imax-1, 1:jmax, 1:kmax)
                              ! 調節前の温度. 
                              ! Temperature before adjustment
    real(DP):: xyzf_QMixB          (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)

    real(DP):: xyz_DelAtmMass      (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xyz_TempTmp         (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_QH2OVapTmp      (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xyz_DTempDt         (0:imax-1, 1:jmax, 1:kmax)
                              ! 温度変化率. 
                              ! Temperature tendency
    real(DP):: xy_DSurfMajCompIceDt(0:imax-1, 1:jmax)
                              ! 
                              ! Surface major component ice tendency
    real(DP):: xy_DPsDt            (0:imax-1, 1:jmax)

    real(DP):: xyr_DPressDt         (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: z_DelTempSubl(1:kmax)
    real(DP):: z_DelTempCond(1:kmax)

    integer :: mmax
    real(DP):: xyza_Array       (0:imax-1, 1:jmax, 1:kmax, 1:ncmax+1+1+1)
    logical :: a_FlagSurfaceSink                          (1:ncmax+1+1+1)
    real(DP):: xyra_MassFlow    (0:imax-1, 1:jmax, 0:kmax, 1:ncmax+1+1+1)

    real(DP):: xyrf_MassFlow    (0:imax-1, 1:jmax, 0:kmax, 1:ncmax)
    real(DP):: xyr_TempFlow     (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_MomXFlow     (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_MomYFlow     (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_DelAtmMassB   (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_DelAtmMassA   (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xy_ColumnLatEneB(0:imax-1, 1:jmax)
    real(DP):: xy_ColumnIntEneB(0:imax-1, 1:jmax)
    real(DP):: xy_ColumnLatEneA(0:imax-1, 1:jmax)
    real(DP):: xy_ColumnIntEneA(0:imax-1, 1:jmax)
    real(DP):: xy_LostLatEneA  (0:imax-1, 1:jmax)
    real(DP):: xy_LostIntEneA  (0:imax-1, 1:jmax)
    real(DP):: TotalEneB
    real(DP):: TotalEneA
    real(DP):: Ratio

    real(DP) :: CondMaterialDen
    real(DP) :: CloudRad

    integer:: iThread

    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction
    integer:: l
    integer:: m
    integer:: n


    ! 実行文 ; Executable statement
    !

    ! 初期化
    ! Initialization
    !
    if ( .not. cloud_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    if ( .not. FlagMajCompPhaseChange ) return

    if ( IndexMajCompIce <= 0 ) then
      call MessageNotify( 'E', module_name, 'Major component ice is not contained in xyzf_QMix.' )
    end if


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )

    ! Store variables
    !
    xyz_TempB  = xyz_Temp
    xyzf_QMixB = xyzf_QMix


    ! Set latent heat
    LatentHeatMajCompSubl = SaturateMajorCompInqLatentHeat()


    TimeStep = 2.0_DP * DelTime

    CondMaterialDen = MajCompIceDen
    CloudRad        = -1.0_DP

    !$OMP PARALLEL DEFAULT(PRIVATE) &
    !$OMP SHARED( &
    !$OMP         NThreads, TimeStep, a_ls, a_le, &
    !$OMP         imax, jmax, kmax, &
    !$OMP         IndexMajCompIce, &
    !$OMP         Grav, CpDry, &
    !$OMP         LatentHeatMajCompSubl, &
    !$OMP         CloudNumRatio, &
    !$OMP         CondMaterialDen, CloudRad, CloudNuclRad, CloudNuclDen, &
    !$OMP         xyr_Press, xyz_Press, xyr_Height, &
    !$OMP         xyz_Temp, xyzf_QMix, &
    !$OMP         xyz_DelAtmMass, &
    !$OMP         xyz_PartRad, xyz_PartDen, &
    !$OMP         xyz_MajCompIceMass, &
    !$OMP         xyr_DPPress, xyr_MajCompIceGravSedMassFlux &
    !$OMP       )

    !$OMP DO
    do iThread = 0, NThreads-1

      do l = a_ls(iThread), a_le(iThread)

        i = mod( (l-1), imax ) + 1 - 1
        j = int( (l-1) / imax ) + 1


        do k = 0, kmax
          r_Press (k) = xyr_Press (i,j,k)
          r_Height(k) = xyr_Height(i,j,k)
        end do
        do k = 1, kmax
          z_Press         (k) = xyz_Press         (i,j,k)
          z_Temp          (k) = xyz_Temp          (i,j,k)
          z_QMajCompIce   (k) = xyzf_QMix         (i,j,k,IndexMajCompIce)
        end do


        do k = 1, kmax
          z_DelAtmMass(k) = ( r_Press(k-1) - r_Press(k) ) / Grav
        end do

        z_MajCompIceMass = z_DelAtmMass * z_QMajCompIce

        call SaturateMajorCompCondTemp( &
          & z_Press,                    & ! (in)
          & z_TempCond                  & ! (out)
          & )

        ! Phase change
        !   (This is not needed?)
        !
        !   sublimation of all cloud ice
        z_DelTempSubl = &
          & - LatentHeatMajCompSubl * z_MajCompIceMass &
          &   / ( CpDry * z_DelAtmMass )
        z_Temp = z_Temp + z_DelTempSubl
        z_MajCompIceMass = 0.0_DP
        !
        !   condensation
        z_DelTempCond = max( z_TempCond - z_Temp, 0.0_DP )
        z_Temp = z_Temp + z_DelTempCond
        z_MajCompIceMass = z_MajCompIceMass    &
          & + CpDry * z_DelAtmMass * z_DelTempCond  &
          &   / LatentHeatMajCompSubl


        z_DelNuclNum    = z_DelAtmMass * CloudNumRatio

        ! update mixing ratio, temporarily
        z_QMajCompIce   = z_MajCompIceMass / z_DelAtmMass
        !
        call CloudUtils1BinCalcPartProp1D(             &
          & CondMaterialDen, CloudRad,                 & ! (in )
          & CloudNuclRad, CloudNuclDen,                & ! (in )
          & z_DelAtmMass, z_QMajCompIce, z_DelNuclNum, & ! (in )
          & z_PartRad, z_PartDen                       & ! (out)
          & )

        ! 重力沈降過程
        ! Gravitational sedimentation process
        !
        call CloudMajorCompCalcSedMassFlux1D( &
          & TimeStep,                        & ! (in   )
          & LatentHeatMajCompSubl,           & ! (in   )
          & r_Press, r_Height,               & ! (in   )
          & z_PartDen, z_PartRad,            & ! (in   )
          & z_TempCond,                      & ! (in   )
          & z_Temp,                          & ! (inout)
          & z_MajCompIceMass,                & ! (inout)
          & r_MajCompIceGravSedMassFlux,     & ! (out  )
          & r_DPPress                        & ! (out  )
          & )

        do k = 1, kmax
          xyz_DelAtmMass    (i,j,k) = z_DelAtmMass    (k)
        end do
        do k = 1, kmax
          xyz_PartDen       (i,j,k) = z_PartDen       (k)
          xyz_PartRad       (i,j,k) = z_PartRad       (k)
          xyz_Temp          (i,j,k) = z_Temp          (k)
          xyz_MajCompIceMass(i,j,k) = z_MajCompIceMass(k)
        end do
        do k = 0, kmax
          xyr_DPPress                  (i,j,k) = r_DPPress(k)
          xyr_MajCompIceGravSedMassFlux(i,j,k) = r_MajCompIceGravSedMassFlux(k)
        end do

      end do
    end do
    !$OMP END DO
    !$OMP END PARALLEL


    xyr_DPressDt         = xyr_MajCompIceGravSedMassFlux * Grav
    xy_DPsDt             = xyr_DPressDt(:,:,0)
    xy_DSurfMajCompIceDt = - xy_DPsDt / Grav


!!$    ! update mixing ratio
!!$    xyz_QMajCompIce = xyz_MajCompIceMass / xyz_DelAtmMass
    ! NOTE
    !  xyz_QMajCompIce and xyzf_QMix need not be updated, here, because 
    !  xyzf_QMix(:,:,:,IndexMajCompIce) will be updated with xyz_MajCompIceMass
    !  below.


    ! packing
    mmax = ncmax
    if ( FlagModTemp ) then
      mmax = mmax + 1
    end if
    if ( FlagModMom ) then
      mmax = mmax + 2
    end if
    do m = 1, ncmax
      n = m
      xyza_Array(:,:,:,m) = xyzf_QMix(:,:,:,n)
      if ( ( n == IndexTKE ) .or. ( n == IndexMajCompIce ) ) then
        a_FlagSurfaceSink(m) = .true.
      else
        a_FlagSurfaceSink(m) = .false.
      end if
    end do
    m = ncmax
    if ( FlagModTemp ) then
      m = m + 1
      xyza_Array(:,:,:,m) = xyz_Temp
      a_FlagSurfaceSink(m) = .true.
    end if
    if ( FlagModMom ) then
      m = m + 1
      xyza_Array(:,:,:,m) = xyz_U
      a_FlagSurfaceSink(m) = .true.
      m = m + 1
      xyza_Array(:,:,:,m) = xyz_V
      a_FlagSurfaceSink(m) = .true.
    end if

    call CloudMajorCompCalcFlow(         &
      & xyr_Press, xyr_DPPress,                                      & ! (in)
      & mmax, a_FlagSurfaceSink(1:mmax), xyza_Array(:,:,:,1:mmax),   & ! (in)
      & xyra_MassFlow(:,:,:,1:mmax)                                  & ! (out)
      & )

    ! unpacking
    do m = 1, ncmax
      xyrf_MassFlow(:,:,:,m) = xyra_MassFlow(:,:,:,m)
    end do
    m = ncmax
    if ( FlagModTemp ) then
      m = m + 1
      xyr_TempFlow = xyra_MassFlow(:,:,:,m)
    else
      xyr_TempFlow = 0.0_DP
    end if
    if ( FlagModMom ) then
      m = m + 1
      xyr_MomXFlow = xyra_MassFlow(:,:,:,m)
      m = m + 1
      xyr_MomYFlow = xyra_MassFlow(:,:,:,m)
    else
      xyr_MomXFlow = 0.0_DP
      xyr_MomYFlow = 0.0_DP
    end if


    ! Adjustment
    !   preparation
    xy_PsB = xy_Ps
    xy_PsA = xy_PsB + xy_DPsDt * ( 2.0_DP * DelTime )

    ! 温度の半整数σレベルの補間, 気圧と高度の算出
    ! Interpolate temperature on half sigma level, 
    ! and calculate pressure and height
    !
    xyz_TempTmp    = 300.0_DP
    xyz_QH2OVapTmp =   0.0_DP
    call AuxVars( &
      & xy_PsB, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressB                  & ! (out) optional
      & )
    call AuxVars( &
      & xy_PsA, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressA                  & ! (out) optional
      & )
    do k = 1, kmax
      xyz_DelAtmMassB(:,:,k) = ( xyr_PressB(:,:,k-1) - xyr_PressB(:,:,k) ) / Grav
      xyz_DelAtmMassA(:,:,k) = ( xyr_PressA(:,:,k-1) - xyr_PressA(:,:,k) ) / Grav
    end do
    !   Atmospheric composition
    do n = 1, ncmax
      if ( n == IndexMajCompIce ) then
        ! major component is treated differently from others
        xyzf_QMix(:,:,:,n) = xyz_MajCompIceMass / xyz_DelAtmMassA
      else
        do k = 1, kmax
          xyzf_QMix(:,:,k,n) =                                              &
            &   (   xyz_DelAtmMassB(:,:,k) * xyzf_QMix(:,:,k,n)             &
            &     - ( xyrf_MassFlow(:,:,k,n) - xyrf_MassFlow(:,:,k-1,n) ) ) &
            & / xyz_DelAtmMassA(:,:,k)
        end do
      end if
    end do
    if ( FlagModTemp ) then
      do k = 1, kmax
        ! Temperature
        xyz_Temp(:,:,k) =                                           &
          &   (   xyz_DelAtmMassB(:,:,k) * xyz_Temp(:,:,k)          &
          &     - ( xyr_TempFlow(:,:,k) - xyr_TempFlow(:,:,k-1) ) ) &
          & / xyz_DelAtmMassA(:,:,k)
      end do
    end if
    if ( FlagModMom ) then
      do k = 1, kmax
        ! Zonal wind
        xyz_U(:,:,k) =                                              &
          &   (   xyz_DelAtmMassB(:,:,k) * xyz_U(:,:,k)             &
          &     - ( xyr_MomXFlow(:,:,k) - xyr_MomXFlow(:,:,k-1) ) ) &
          & / xyz_DelAtmMassA(:,:,k)
        ! Meridional wind
        xyz_V(:,:,k) =                                              &
          &   (   xyz_DelAtmMassB(:,:,k) * xyz_V(:,:,k)             &
          &     - ( xyr_MomYFlow(:,:,k) - xyr_MomYFlow(:,:,k-1) ) ) &
          & / xyz_DelAtmMassA(:,:,k)
      end do
    end if


    ! Surface major component ice adjustment
    xy_SurfMajCompIce = xy_SurfMajCompIce + xy_DSurfMajCompIceDt * ( 2.0_DP * DelTime )
    ! Surface pressure adjustment
    xy_Ps = xy_PsA


    call AuxVars( &
      & xy_PsA, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressA                  & ! (out) optional
      & )

    ! 成分の質量の補正
    ! Fix masses of constituents
    !
    call MassFixerColumn( &
      & xyr_PressA, & ! (in)
      & xyzf_QMix,  & ! (inout)
      & Text = "In CloudMajorComp"  & ! (in)
      & )

    ! Check
    call CloudMajorCompConsChk(       &
      & a_FlagSurfaceSink,            & ! (in)
      & xyz_DelAtmMassB, xyzf_QMixB,  & ! (in)
      & xyz_DelAtmMassA, xyzf_QMix    & ! (in)
      & )

    ! Check major component ice
    xy_ColumnLatEneB = 0.0_DP
    xy_ColumnIntEneB = 0.0_DP
    !
    xy_ColumnLatEneA = 0.0_DP
    xy_ColumnIntEneA = 0.0_DP
    do k = kmax, 1, -1
      xy_ColumnLatEneB = xy_ColumnLatEneB &
        & + LatentHeatMajCompSubl &
        &   * xyzf_QMixB(:,:,k,IndexMajCompIce) * xyz_DelAtmMassB(:,:,k)
      xy_ColumnIntEneB = xy_ColumnIntEneB &
        & + CpDry * xyz_TempB(:,:,k) * xyz_DelAtmMassB(:,:,k)
      !
      xy_ColumnLatEneA = xy_ColumnLatEneA &
        & + LatentHeatMajCompSubl &
        &   * xyzf_QMix (:,:,k,IndexMajCompIce) * xyz_DelAtmMassA(:,:,k)
        ! xyz_DelAtmMassA should be used. But, in order to do that,
        ! we need some modifications, maybe loss of internal energy
        ! should be considered. 
      xy_ColumnIntEneA = xy_ColumnIntEneA &
!        & + CpDry * xyz_Temp (:,:,k) * xyz_DelAtmMassA(:,:,k)
        & + CpDry * xyz_Temp (:,:,k) * xyz_DelAtmMassB(:,:,k)
    end do
    !
    xy_LostLatEneA = &
      &   LatentHeatMajCompSubl &
      &   * xy_DSurfMajCompIceDt * ( 2.0_DP * DelTime )
    xy_LostIntEneA = &
      & - CpDry * xyr_TempFlow(:,:,0)
    do j = 1, jmax
      do i = 0, imax-1
        TotalEneB =   xy_ColumnIntEneB(i,j) &
          &         - xy_ColumnLatEneB(i,j)
        TotalEneA =   xy_ColumnIntEneA(i,j) + xy_LostIntEneA(i,j)   &
          &         - ( xy_ColumnLatEneA(i,j) + xy_LostLatEneA(i,j) )
        Ratio = abs( TotalEneB - TotalEneA ) / TotalEneB
        if ( Ratio > 1.0e-10_DP ) then
          call MessageNotify( 'M', module_name, &
            & 'Major component ice phase change does not conserve energy, %f, at (%d,%d).', &
            & d = (/ Ratio /), i = (/ i, j /) )
        end if
      end do
    end do


    ! ヒストリデータ出力
    ! History data output
    !
    xyz_DTempDt = ( xyz_Temp - xyz_TempB ) / ( 2.0_DP * DelTime )
    call HistoryAutoPut( TimeN, 'DTempDtMajCompPhaseChange', xyz_DTempDt )

    call HistoryAutoPut( TimeN, 'MajCompCloudRadiusForGravSed', xyz_PartRad )


    do j = 1, jmax
      do i = 0, imax-1
        do k = 0, kmax
          r_Press (k) = xyr_Press (i,j,k)
        end do
        do k = 1, kmax
          z_QMajCompIce   (k) = xyzf_QMix         (i,j,k,IndexMajCompIce)
        end do
        do k = 1, kmax
          z_DelAtmMass(k) = ( r_Press(k-1) - r_Press(k) ) / Grav
        end do
        z_DelNuclNum = z_DelAtmMass * CloudNumRatio
        call CloudUtils1BinCalcPartProp1D(             &
          & CondMaterialDen, CloudRad,                 & ! (in )
          & CloudNuclRad, CloudNuclDen,                & ! (in )
          & z_DelAtmMass, z_QMajCompIce, z_DelNuclNum, & ! (in )
          & z_PartRad, z_PartDen                       & ! (out)
          & )
        do k = 1, kmax
          xyz_PartDen       (i,j,k) = z_PartDen       (k)
          xyz_PartRad       (i,j,k) = z_PartRad       (k)
        end do
      end do
    end do

    call HistoryAutoPut( TimeN, 'MajCompCloudRadius', xyz_PartRad )

    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine BKNoSubStep_CloudMajorComp

  !--------------------------------------------------------------------------------------

  subroutine BKSerial_CloudMajorComp(                         &
    & xyr_Press, xyz_Press, xyr_Height,              & ! (in)
    & xy_Ps, xyz_Temp, xyzf_QMix, xyz_U, xyz_V,      & ! (inout)
    & xy_SurfMajCompIce                              & ! (inout)
    & )
    !
    ! CO2 相変化
    !
    ! CO2 phase change
    !

    ! モジュール引用 ; USE statements
    !

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & DelTime, &            ! $ \Delta t $
      & TimeN, &              ! ステップ $ t $ の時刻. Time of step $ t $. 
      & TimesetClockStart, TimesetClockStop

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 組成に関わる配列の設定
    ! Settings of array for atmospheric composition
    !
    use composition, only : &
      & ncmax, &
      & IndexTKE, &
      & IndexMajCompIce

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Grav, &               ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration
      & CpDry
                              ! $ C_p $ [J kg-1 K-1].
                              ! 乾燥大気の定圧比熱.
                              ! Specific heat of air at constant pressure

    ! 温度の半整数σレベルの補間, 気圧と高度の算出
    ! Interpolate temperature on half sigma level, 
    ! and calculate pressure and height
    !
    use auxiliary, only: AuxVars

    ! 主成分相変化
    ! Phase change of atmospheric major component
    !
    use saturate_major_comp, only :    &
      & SaturateMajorCompCondTemp,     &
      & SaturateMajorCompInqLatentHeat

    ! 重力沈降過程
    ! Gravitational sedimentation process
    !
!!$    use grav_sed, only : GravSedAdjust

    ! 質量の補正
    ! Mass fixer
    !
    use mass_fixer, only: MassFixerColumn

    ! 雲関系ルーチン
    ! Cloud-related routines
    !
    use cloud_utils, only : CloudUtils1BinCalcPartProp


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in   ):: xyr_Press (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ . 気圧 (半整数レベル). 
                              ! Air pressure (half level)
    real(DP), intent(in   ):: xyz_Press (0:imax-1, 1:jmax, 1:kmax)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)
    real(DP), intent(in   ):: xyr_Height(0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(inout):: xy_Ps            (0:imax-1, 1:jmax)
                              ! $ T $ .     温度. Temperature
    real(DP), intent(inout):: xyz_Temp         (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ .     温度. Temperature
    real(DP), intent(inout):: xyzf_QMix        (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
    real(DP), intent(inout):: xyz_U            (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout):: xyz_V            (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout):: xy_SurfMajCompIce(0:imax-1, 1:jmax)
                              !
                              ! Surface major component ice amount

    ! 作業変数
    ! Work variables
    !
    real(DP):: LatentHeatMajCompSubl

    real(DP):: TimeStep

    real(DP):: xyz_QMajCompIce   (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xyr_MajCompIceGravSedMassFlux(0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_PartRad         (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_PartDen         (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyr_DPPress         (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_MajCompIceMass (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: r_Press           (0:kmax)
    real(DP):: z_Press           (1:kmax)
    real(DP):: z_Temp            (1:kmax)
    real(DP):: r_Height          (0:kmax)

    real(DP):: z_TempCond        (1:kmax)
    real(DP):: z_MajCompIceMass  (1:kmax)

    real(DP):: z_PartRad         (1:kmax)
    real(DP):: z_PartDen         (1:kmax)

    real(DP):: r_DPPress                  (0:kmax)
    real(DP):: r_MajCompIceGravSedMassFlux(0:kmax)


    real(DP):: xy_PsB              (0:imax-1, 1:jmax)
    real(DP):: xy_PsA              (0:imax-1, 1:jmax)
    real(DP):: xyr_PressB          (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_PressA          (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_TempB           (0:imax-1, 1:jmax, 1:kmax)
                              ! 調節前の温度. 
                              ! Temperature before adjustment
    real(DP):: xyzf_QMixB          (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)

    real(DP):: xyz_DelAtmMass      (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xy_DelTempCond      (0:imax-1, 1:jmax)

    real(DP):: xyz_TempTmp         (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_QH2OVapTmp      (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xyz_DTempDt         (0:imax-1, 1:jmax, 1:kmax)
                              ! 温度変化率. 
                              ! Temperature tendency
    real(DP):: xy_DSurfMajCompIceDt(0:imax-1, 1:jmax)
                              ! 
                              ! Surface major component ice tendency
    real(DP):: xy_DPsDt            (0:imax-1, 1:jmax)

    real(DP):: xyr_DPressDt         (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xy_DelTempSubl (0:imax-1, 1:jmax)
    real(DP):: xyz_TempCond   (0:imax-1, 1:jmax, 1:kmax)

    integer :: mmax
    real(DP):: xyza_Array       (0:imax-1, 1:jmax, 1:kmax, 1:ncmax+1+1+1)
    logical :: a_FlagSurfaceSink                          (1:ncmax+1+1+1)
    real(DP):: xyra_MassFlow    (0:imax-1, 1:jmax, 0:kmax, 1:ncmax+1+1+1)

    real(DP):: xyrf_MassFlow    (0:imax-1, 1:jmax, 0:kmax, 1:ncmax)
    real(DP):: xyr_TempFlow     (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_MomXFlow     (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_MomYFlow     (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_DelAtmMassB   (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_DelAtmMassA   (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xy_ColumnLatEneB(0:imax-1, 1:jmax)
    real(DP):: xy_ColumnIntEneB(0:imax-1, 1:jmax)
    real(DP):: xy_ColumnLatEneA(0:imax-1, 1:jmax)
    real(DP):: xy_ColumnIntEneA(0:imax-1, 1:jmax)
    real(DP):: xy_LostLatEneA  (0:imax-1, 1:jmax)
    real(DP):: xy_LostIntEneA  (0:imax-1, 1:jmax)
    real(DP):: TotalEneB
    real(DP):: TotalEneA
    real(DP):: Ratio

    real(DP) :: CondMaterialDen
    real(DP) :: CloudRad
    real(DP) :: xyz_DelNuclNum (0:imax-1, 1:jmax, 1:kmax)

    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction
    integer:: m
    integer:: n


    ! 実行文 ; Executable statement
    !

    ! 初期化
    ! Initialization
    !
    if ( .not. cloud_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    if ( .not. FlagMajCompPhaseChange ) return

    if ( IndexMajCompIce <= 0 ) then
      call MessageNotify( 'E', module_name, 'Major component ice is not contained in xyzf_QMix.' )
    end if


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )

    ! Store variables
    !
    xyz_TempB  = xyz_Temp
    xyzf_QMixB = xyzf_QMix


    ! Set latent heat
    LatentHeatMajCompSubl = SaturateMajorCompInqLatentHeat()

    do k = 1, kmax
      xyz_DelAtmMass(:,:,k) = ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k) ) / Grav
    end do

    xyz_QMajCompIce    = xyzf_QMix(:,:,:,IndexMajCompIce)
    xyz_MajCompIceMass = xyz_DelAtmMass * xyz_QMajCompIce

    call SaturateMajorCompCondTemp( &
      & xyz_Press,                  & ! (in)
      & xyz_TempCond                & ! (inout)
      & )


    TimeStep = 2.0_DP * DelTime


    do k = 1, kmax
      ! sublimation of all cloud ice
      xy_DelTempSubl = &
        & - LatentHeatMajCompSubl * xyz_MajCompIceMass(:,:,k) &
        &   / ( CpDry * xyz_DelAtmMass(:,:,k) )
      xyz_Temp(:,:,k) = xyz_Temp(:,:,k) + xy_DelTempSubl
      xyz_MajCompIceMass(:,:,k) = 0.0_DP

      ! condensation
      xy_DelTempCond = max( xyz_TempCond(:,:,k) - xyz_Temp(:,:,k), 0.0_DP )
      xyz_Temp(:,:,k) = xyz_Temp(:,:,k) + xy_DelTempCond
      xyz_MajCompIceMass(:,:,k) = xyz_MajCompIceMass(:,:,k)    &
        & + CpDry * xyz_DelAtmMass(:,:,k) * xy_DelTempCond     &
        &   / LatentHeatMajCompSubl
    end do


    CondMaterialDen = MajCompIceDen
    CloudRad        = -1.0_DP
    xyz_DelNuclNum  = xyz_DelAtmMass * CloudNumRatio


    ! update mixing ratio, temporarily
    xyz_QMajCompIce = xyz_MajCompIceMass / xyz_DelAtmMass
    !
    call CloudUtils1BinCalcPartProp( &
      & CondMaterialDen, CloudRad,                 & ! (in )
      & CloudNuclRad, CloudNuclDen,                & ! (in )
      & xyr_Press,                                 & ! (in )
      & xyz_DelNuclNum,                            & ! (in )
      & xyz_QMajCompIce,                           & ! (in )
      & xyz_PartRad, xyz_PartDen                   & ! (out)
      & )

    do j = 1, jmax
      do i = 0, imax-1

        do k = 0, kmax
          r_Press (k) = xyr_Press (i,j,k)
          r_Height(k) = xyr_Height(i,j,k)
        end do
        do k = 1, kmax
          z_Press         (k) = xyz_Press         (i,j,k)
          z_MajCompIceMass(k) = xyz_MajCompIceMass(i,j,k)
          z_PartRad       (k) = xyz_PartRad       (i,j,k)
          z_PartDen       (k) = xyz_PartDen       (i,j,k)
          z_TempCond      (k) = xyz_TempCond      (i,j,k)
          z_Temp          (k) = xyz_Temp          (i,j,k)
        end do

        ! 重力沈降過程
        ! Gravitational sedimentation process
        !
        call CloudMajorCompCalcSedMassFlux1D( &
          & TimeStep,                        & ! (in   )
          & LatentHeatMajCompSubl,           & ! (in   )
          & r_Press, r_Height,               & ! (in   )
          & z_PartDen, z_PartRad,            & ! (in   )
          & z_TempCond,                      & ! (in   )
          & z_Temp,                          & ! (inout)
          & z_MajCompIceMass,                & ! (inout)
          & r_MajCompIceGravSedMassFlux,     & ! (out  )
          & r_DPPress                        & ! (out  )
          & )

        do k = 1, kmax
          xyz_PartDen       (i,j,k) = z_PartDen       (k)
          xyz_PartRad       (i,j,k) = z_PartRad       (k)
          xyz_Temp          (i,j,k) = z_Temp          (k)
          xyz_MajCompIceMass(i,j,k) = z_MajCompIceMass(k)
        end do
        do k = 0, kmax
          xyr_DPPress                  (i,j,k) = r_DPPress(k)
          xyr_MajCompIceGravSedMassFlux(i,j,k) = r_MajCompIceGravSedMassFlux(k)
        end do

      end do
    end do

    xyr_DPressDt         = xyr_MajCompIceGravSedMassFlux * Grav
    xy_DPsDt             = xyr_DPressDt(:,:,0)
    xy_DSurfMajCompIceDt = - xy_DPsDt / Grav


!!$    ! update mixing ratio
!!$    xyz_QMajCompIce = xyz_MajCompIceMass / xyz_DelAtmMass
    ! NOTE
    !  xyz_QMajCompIce and xyzf_QMix need not be updated, here, because 
    !  xyzf_QMix(:,:,:,IndexMajCompIce) will be updated with xyz_MajCompIceMass
    !  below.


    ! packing
    mmax = ncmax
    if ( FlagModTemp ) then
      mmax = mmax + 1
    end if
    if ( FlagModMom ) then
      mmax = mmax + 2
    end if
    do m = 1, ncmax
      n = m
      xyza_Array(:,:,:,m) = xyzf_QMix(:,:,:,n)
      if ( ( n == IndexTKE ) .or. ( n == IndexMajCompIce ) ) then
        a_FlagSurfaceSink(m) = .true.
      else
        a_FlagSurfaceSink(m) = .false.
      end if
    end do
    m = ncmax
    if ( FlagModTemp ) then
      m = m + 1
      xyza_Array(:,:,:,m) = xyz_Temp
      a_FlagSurfaceSink(m) = .true.
    end if
    if ( FlagModMom ) then
      m = m + 1
      xyza_Array(:,:,:,m) = xyz_U
      a_FlagSurfaceSink(m) = .true.
      m = m + 1
      xyza_Array(:,:,:,m) = xyz_V
      a_FlagSurfaceSink(m) = .true.
    end if

    call CloudMajorCompCalcFlow(         &
      & xyr_Press, xyr_DPPress,                                      & ! (in)
      & mmax, a_FlagSurfaceSink(1:mmax), xyza_Array(:,:,:,1:mmax),   & ! (in)
      & xyra_MassFlow(:,:,:,1:mmax)                                  & ! (out)
      & )

    ! unpacking
    do m = 1, ncmax
      xyrf_MassFlow(:,:,:,m) = xyra_MassFlow(:,:,:,m)
    end do
    m = ncmax
    if ( FlagModTemp ) then
      m = m + 1
      xyr_TempFlow = xyra_MassFlow(:,:,:,m)
    else
      xyr_TempFlow = 0.0_DP
    end if
    if ( FlagModMom ) then
      m = m + 1
      xyr_MomXFlow = xyra_MassFlow(:,:,:,m)
      m = m + 1
      xyr_MomYFlow = xyra_MassFlow(:,:,:,m)
    else
      xyr_MomXFlow = 0.0_DP
      xyr_MomYFlow = 0.0_DP
    end if


    ! Adjustment
    !   preparation
    xy_PsB = xy_Ps
    xy_PsA = xy_PsB + xy_DPsDt * ( 2.0_DP * DelTime )

    ! 温度の半整数σレベルの補間, 気圧と高度の算出
    ! Interpolate temperature on half sigma level, 
    ! and calculate pressure and height
    !
    xyz_TempTmp    = 300.0_DP
    xyz_QH2OVapTmp =   0.0_DP
    call AuxVars( &
      & xy_PsB, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressB                  & ! (out) optional
      & )
    call AuxVars( &
      & xy_PsA, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressA                  & ! (out) optional
      & )
    do k = 1, kmax
      xyz_DelAtmMassB(:,:,k) = ( xyr_PressB(:,:,k-1) - xyr_PressB(:,:,k) ) / Grav
      xyz_DelAtmMassA(:,:,k) = ( xyr_PressA(:,:,k-1) - xyr_PressA(:,:,k) ) / Grav
    end do
    !   Atmospheric composition
    do n = 1, ncmax
      if ( n == IndexMajCompIce ) then
        ! major component is treated differently from others
        xyzf_QMix(:,:,:,n) = xyz_MajCompIceMass / xyz_DelAtmMassA
      else
        do k = 1, kmax
          xyzf_QMix(:,:,k,n) =                                              &
            &   (   xyz_DelAtmMassB(:,:,k) * xyzf_QMix(:,:,k,n)             &
            &     - ( xyrf_MassFlow(:,:,k,n) - xyrf_MassFlow(:,:,k-1,n) ) ) &
            & / xyz_DelAtmMassA(:,:,k)
        end do
      end if
    end do
    if ( FlagModTemp ) then
      do k = 1, kmax
        ! Temperature
        xyz_Temp(:,:,k) =                                           &
          &   (   xyz_DelAtmMassB(:,:,k) * xyz_Temp(:,:,k)          &
          &     - ( xyr_TempFlow(:,:,k) - xyr_TempFlow(:,:,k-1) ) ) &
          & / xyz_DelAtmMassA(:,:,k)
      end do
    end if
    if ( FlagModMom ) then
      do k = 1, kmax
        ! Zonal wind
        xyz_U(:,:,k) =                                              &
          &   (   xyz_DelAtmMassB(:,:,k) * xyz_U(:,:,k)             &
          &     - ( xyr_MomXFlow(:,:,k) - xyr_MomXFlow(:,:,k-1) ) ) &
          & / xyz_DelAtmMassA(:,:,k)
        ! Meridional wind
        xyz_V(:,:,k) =                                              &
          &   (   xyz_DelAtmMassB(:,:,k) * xyz_V(:,:,k)             &
          &     - ( xyr_MomYFlow(:,:,k) - xyr_MomYFlow(:,:,k-1) ) ) &
          & / xyz_DelAtmMassA(:,:,k)
      end do
    end if


    ! Surface major component ice adjustment
    xy_SurfMajCompIce = xy_SurfMajCompIce + xy_DSurfMajCompIceDt * ( 2.0_DP * DelTime )
    ! Surface pressure adjustment
    xy_Ps = xy_PsA


    call AuxVars( &
      & xy_PsA, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressA                  & ! (out) optional
      & )

    ! 成分の質量の補正
    ! Fix masses of constituents
    !
    call MassFixerColumn( &
      & xyr_PressA, & ! (in)
      & xyzf_QMix,  & ! (inout)
      & Text = "In CloudMajorComp"  & ! (in)
      & )

    ! Check
    call CloudMajorCompConsChk(       &
      & a_FlagSurfaceSink,            & ! (in)
      & xyz_DelAtmMassB, xyzf_QMixB,  & ! (in)
      & xyz_DelAtmMassA, xyzf_QMix    & ! (in)
      & )

    ! Check major component ice
    xy_ColumnLatEneB = 0.0_DP
    xy_ColumnIntEneB = 0.0_DP
    !
    xy_ColumnLatEneA = 0.0_DP
    xy_ColumnIntEneA = 0.0_DP
    do k = kmax, 1, -1
      xy_ColumnLatEneB = xy_ColumnLatEneB &
        & + LatentHeatMajCompSubl &
        &   * xyzf_QMixB(:,:,k,IndexMajCompIce) * xyz_DelAtmMassB(:,:,k)
      xy_ColumnIntEneB = xy_ColumnIntEneB &
        & + CpDry * xyz_TempB(:,:,k) * xyz_DelAtmMassB(:,:,k)
      !
      xy_ColumnLatEneA = xy_ColumnLatEneA &
        & + LatentHeatMajCompSubl &
        &   * xyzf_QMix (:,:,k,IndexMajCompIce) * xyz_DelAtmMassA(:,:,k)
        ! xyz_DelAtmMassA should be used. But, in order to do that,
        ! we need some modifications, maybe loss of internal energy
        ! should be considered. 
      xy_ColumnIntEneA = xy_ColumnIntEneA &
!        & + CpDry * xyz_Temp (:,:,k) * xyz_DelAtmMassA(:,:,k)
        & + CpDry * xyz_Temp (:,:,k) * xyz_DelAtmMassB(:,:,k)
    end do
    !
    xy_LostLatEneA = &
      &   LatentHeatMajCompSubl &
      &   * xy_DSurfMajCompIceDt * ( 2.0_DP * DelTime )
    xy_LostIntEneA = &
      & - CpDry * xyr_TempFlow(:,:,0)
    do j = 1, jmax
      do i = 0, imax-1
        TotalEneB =   xy_ColumnIntEneB(i,j) &
          &         - xy_ColumnLatEneB(i,j)
        TotalEneA =   xy_ColumnIntEneA(i,j) + xy_LostIntEneA(i,j)   &
          &         - ( xy_ColumnLatEneA(i,j) + xy_LostLatEneA(i,j) )
        Ratio = abs( TotalEneB - TotalEneA ) / TotalEneB
        if ( Ratio > 1.0e-10_DP ) then
          call MessageNotify( 'M', module_name, &
            & 'Major component ice phase change does not conserve energy, %f, at (%d,%d).', &
            & d = (/ Ratio /), i = (/ i, j /) )
        end if
      end do
    end do


    ! ヒストリデータ出力
    ! History data output
    !
    xyz_DTempDt = ( xyz_Temp - xyz_TempB ) / ( 2.0_DP * DelTime )
    call HistoryAutoPut( TimeN, 'DTempDtMajCompPhaseChange', xyz_DTempDt )
    call HistoryAutoPut( TimeN, 'MajCompCloudRadius', xyz_PartRad )


    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine BKSerial_CloudMajorComp

  !--------------------------------------------------------------------------------------

  subroutine TEST_CloudMajorComp(                         &
    & xyr_Press, xyz_Press, xyr_Height,              & ! (in)
    & xy_Ps, xyz_Temp, xyzf_QMix, xyz_U, xyz_V,      & ! (inout)
    & xy_SurfMajCompIce                              & ! (inout)
    & )
    !
    ! CO2 相変化
    !
    ! CO2 phase change
    !

    ! モジュール引用 ; USE statements
    !

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & DelTime, &            ! $ \Delta t $
      & TimeN, &              ! ステップ $ t $ の時刻. Time of step $ t $. 
      & TimesetClockStart, TimesetClockStop

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 組成に関わる配列の設定
    ! Settings of array for atmospheric composition
    !
    use composition, only : &
      & ncmax, &
      & IndexTKE, &
      & IndexMajCompIce

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Grav, &               ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration
      & CpDry
                              ! $ C_p $ [J kg-1 K-1].
                              ! 乾燥大気の定圧比熱.
                              ! Specific heat of air at constant pressure

    ! 温度の半整数σレベルの補間, 気圧と高度の算出
    ! Interpolate temperature on half sigma level, 
    ! and calculate pressure and height
    !
    use auxiliary, only: AuxVars

    ! 主成分相変化
    ! Phase change of atmospheric major component
    !
    use saturate_major_comp, only :    &
      & SaturateMajorCompCondTemp,     &
      & SaturateMajorCompInqLatentHeat

    ! 重力沈降過程
    ! Gravitational sedimentation process
    !
!!$    use grav_sed, only : GravSedAdjust

    ! 質量の補正
    ! Mass fixer
    !
    use mass_fixer, only: MassFixerColumn


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in   ):: xyr_Press (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ . 気圧 (半整数レベル). 
                              ! Air pressure (half level)
    real(DP), intent(in   ):: xyz_Press (0:imax-1, 1:jmax, 1:kmax)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)
    real(DP), intent(in   ):: xyr_Height(0:imax-1, 1:jmax, 0:kmax)
    real(DP), intent(inout):: xy_Ps            (0:imax-1, 1:jmax)
                              ! $ T $ .     温度. Temperature
    real(DP), intent(inout):: xyz_Temp         (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ .     温度. Temperature
    real(DP), intent(inout):: xyzf_QMix        (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
    real(DP), intent(inout):: xyz_U            (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout):: xyz_V            (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout):: xy_SurfMajCompIce(0:imax-1, 1:jmax)
                              !
                              ! Surface major component ice amount

    ! 作業変数
    ! Work variables
    !
    real(DP):: TimeStep

    real(DP):: LatentHeatMajCompSubl

    real(DP):: xyr_GravSedCldMassFlux(0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_CldRad          (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_CldDen          (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyr_DPPress         (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_MajCompIceMass (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: r_Press           (0:kmax)
    real(DP):: z_Press           (1:kmax)
    real(DP):: z_Temp            (1:kmax)
    real(DP):: r_Height          (0:kmax)

    real(DP):: z_CldRad          (1:kmax)
    real(DP):: z_CldDen          (1:kmax)
    real(DP):: z_CldMMR          (1:kmax)

    real(DP):: r_GravSedCldMassFlux(0:kmax)

    real(DP):: xy_PsB              (0:imax-1, 1:jmax)
    real(DP):: xy_PsA              (0:imax-1, 1:jmax)
    real(DP):: xyr_PressB          (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_PressA          (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_TempB           (0:imax-1, 1:jmax, 1:kmax)
                              ! 調節前の温度. 
                              ! Temperature before adjustment
    real(DP):: xyzf_QMixB          (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)

    real(DP):: xyz_DelAtmMass      (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xyz_TempTmp         (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_QH2OVapTmp      (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xyz_DTempDt         (0:imax-1, 1:jmax, 1:kmax)
                              ! 温度変化率. 
                              ! Temperature tendency
    real(DP):: xy_DSurfMajCompIceDt(0:imax-1, 1:jmax)
                              ! 
                              ! Surface major component ice tendency
    real(DP):: xy_DPsDt            (0:imax-1, 1:jmax)

    real(DP):: xyr_DPressDt         (0:imax-1, 1:jmax, 0:kmax)

    integer :: mmax
    real(DP):: xyza_Array       (0:imax-1, 1:jmax, 1:kmax, 1:ncmax+1+1+1)
    logical :: a_FlagSurfaceSink                          (1:ncmax+1+1+1)
    real(DP):: xyra_MassFlow    (0:imax-1, 1:jmax, 0:kmax, 1:ncmax+1+1+1)

    real(DP):: xyrf_MassFlow    (0:imax-1, 1:jmax, 0:kmax, 1:ncmax)
    real(DP):: xyr_TempFlow     (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_MomXFlow     (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_MomYFlow     (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_DelAtmMassB   (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_DelAtmMassA   (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xy_ColumnLatEneB(0:imax-1, 1:jmax)
    real(DP):: xy_ColumnIntEneB(0:imax-1, 1:jmax)
    real(DP):: xy_ColumnLatEneA(0:imax-1, 1:jmax)
    real(DP):: xy_ColumnIntEneA(0:imax-1, 1:jmax)
    real(DP):: xy_LostLatEneA  (0:imax-1, 1:jmax)
    real(DP):: xy_LostIntEneA  (0:imax-1, 1:jmax)
    real(DP):: TotalEneB
    real(DP):: TotalEneA
    real(DP):: Ratio

    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction
    integer:: m
    integer:: n


    ! 実行文 ; Executable statement
    !

    ! 初期化
    ! Initialization
    !
    if ( .not. cloud_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    if ( .not. FlagMajCompPhaseChange ) return

    if ( IndexMajCompIce <= 0 ) then
      call MessageNotify( 'E', module_name, 'Major component ice is not contained in xyzf_QMix.' )
    end if


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )

    ! Store variables
    !
    xyz_TempB  = xyz_Temp
    xyzf_QMixB = xyzf_QMix

    TimeStep = DelTime * 2.0_DP

    ! Set latent heat
    LatentHeatMajCompSubl = SaturateMajorCompInqLatentHeat()

    do k = 1, kmax
      xyz_DelAtmMass(:,:,k) = ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k) ) / Grav
    end do

    do j = 1, jmax
      do i = 0, imax-1

        do k = 0, kmax
          r_Press (k) = xyr_Press (i,j,k)
          r_Height(k) = xyr_Height(i,j,k)
        end do
        do k = 1, kmax
          z_Press (k) = xyz_Press(i,j,k)
          z_Temp  (k) = xyz_Temp (i,j,k)
          z_CldMMR(k) = xyzf_QMix(i,j,k,IndexMajCompIce)
        end do

        call TEST_CloudMajorComp1D( &
          & Grav, CpDry, CloudNumRatio, LatentHeatMajCompSubl, TimeStep, & ! (in   )
          & kmax, r_Press, z_Press, r_Height,                       & ! (in   )
          & z_Temp, z_CldMMR,                                       & ! (inout)
          & z_CldDen, z_CldRad,                                     & ! (out  )
          & r_GravSedCldMassFlux                                    & ! (out  )
          & )

        do k = 1, kmax
          xyz_Temp  (i,j,k)                 = z_Temp  (k)
          xyzf_QMix (i,j,k,IndexMajCompIce) = z_CldMMR(k)
          xyz_CldDen(i,j,k)                 = z_CldDen(k)
          xyz_CldRad(i,j,k)                 = z_CldRad(k)
        end do
        do k = 0, kmax
          xyr_GravSedCldMassFlux(i,j,k) = r_GravSedCldMassFlux(k)
        end do

      end do
    end do

    xyr_DPressDt         = xyr_GravSedCldMassFlux * Grav
    xyr_DPPress          = xyr_Press + xyr_DPressDt * TimeStep
    xy_DPsDt             = xyr_DPressDt(:,:,0)
    xy_DSurfMajCompIceDt = - xy_DPsDt / Grav

    xyz_MajCompIceMass = xyz_DelAtmMass * xyzf_QMix(:,:,:,IndexMajCompIce)


    ! packing
    mmax = ncmax
    if ( FlagModTemp ) then
      mmax = mmax + 1
    end if
    if ( FlagModMom ) then
      mmax = mmax + 2
    end if
    do m = 1, ncmax
      n = m
      xyza_Array(:,:,:,m) = xyzf_QMix(:,:,:,n)
      if ( ( n == IndexTKE ) .or. ( n == IndexMajCompIce ) ) then
        a_FlagSurfaceSink(m) = .true.
      else
        a_FlagSurfaceSink(m) = .false.
      end if
    end do
    m = ncmax
    if ( FlagModTemp ) then
      m = m + 1
      xyza_Array(:,:,:,m) = xyz_Temp
      a_FlagSurfaceSink(m) = .true.
    end if
    if ( FlagModMom ) then
      m = m + 1
      xyza_Array(:,:,:,m) = xyz_U
      a_FlagSurfaceSink(m) = .true.
      m = m + 1
      xyza_Array(:,:,:,m) = xyz_V
      a_FlagSurfaceSink(m) = .true.
    end if

    call CloudMajorCompCalcFlow(         &
      & xyr_Press, xyr_DPPress,                                      & ! (in)
      & mmax, a_FlagSurfaceSink(1:mmax), xyza_Array(:,:,:,1:mmax),   & ! (in)
      & xyra_MassFlow(:,:,:,1:mmax)                                  & ! (out)
      & )

    ! unpacking
    do m = 1, ncmax
      xyrf_MassFlow(:,:,:,m) = xyra_MassFlow(:,:,:,m)
    end do
    m = ncmax
    if ( FlagModTemp ) then
      m = m + 1
      xyr_TempFlow = xyra_MassFlow(:,:,:,m)
    else
      xyr_TempFlow = 0.0_DP
    end if
    if ( FlagModMom ) then
      m = m + 1
      xyr_MomXFlow = xyra_MassFlow(:,:,:,m)
      m = m + 1
      xyr_MomYFlow = xyra_MassFlow(:,:,:,m)
    else
      xyr_MomXFlow = 0.0_DP
      xyr_MomYFlow = 0.0_DP
    end if


    ! Adjustment
    !   preparation
    xy_PsB = xy_Ps
    xy_PsA = xy_PsB + xy_DPsDt * ( 2.0_DP * DelTime )

    ! 温度の半整数σレベルの補間, 気圧と高度の算出
    ! Interpolate temperature on half sigma level, 
    ! and calculate pressure and height
    !
    xyz_TempTmp    = 300.0_DP
    xyz_QH2OVapTmp =   0.0_DP
    call AuxVars( &
      & xy_PsB, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressB                  & ! (out) optional
      & )
    call AuxVars( &
      & xy_PsA, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressA                  & ! (out) optional
      & )
    do k = 1, kmax
      xyz_DelAtmMassB(:,:,k) = ( xyr_PressB(:,:,k-1) - xyr_PressB(:,:,k) ) / Grav
      xyz_DelAtmMassA(:,:,k) = ( xyr_PressA(:,:,k-1) - xyr_PressA(:,:,k) ) / Grav
    end do
    !   Atmospheric composition
    do n = 1, ncmax
      if ( n == IndexMajCompIce ) then
        ! major component is treated differently from others
        xyzf_QMix(:,:,:,n) = xyz_MajCompIceMass / xyz_DelAtmMassA
      else
        do k = 1, kmax
          xyzf_QMix(:,:,k,n) =                                              &
            &   (   xyz_DelAtmMassB(:,:,k) * xyzf_QMix(:,:,k,n)             &
            &     - ( xyrf_MassFlow(:,:,k,n) - xyrf_MassFlow(:,:,k-1,n) ) ) &
            & / xyz_DelAtmMassA(:,:,k)
        end do
      end if
    end do
    if ( FlagModTemp ) then
      do k = 1, kmax
        ! Temperature
        xyz_Temp(:,:,k) =                                           &
          &   (   xyz_DelAtmMassB(:,:,k) * xyz_Temp(:,:,k)          &
          &     - ( xyr_TempFlow(:,:,k) - xyr_TempFlow(:,:,k-1) ) ) &
          & / xyz_DelAtmMassA(:,:,k)
      end do
    end if
    if ( FlagModMom ) then
      do k = 1, kmax
        ! Zonal wind
        xyz_U(:,:,k) =                                              &
          &   (   xyz_DelAtmMassB(:,:,k) * xyz_U(:,:,k)             &
          &     - ( xyr_MomXFlow(:,:,k) - xyr_MomXFlow(:,:,k-1) ) ) &
          & / xyz_DelAtmMassA(:,:,k)
        ! Meridional wind
        xyz_V(:,:,k) =                                              &
          &   (   xyz_DelAtmMassB(:,:,k) * xyz_V(:,:,k)             &
          &     - ( xyr_MomYFlow(:,:,k) - xyr_MomYFlow(:,:,k-1) ) ) &
          & / xyz_DelAtmMassA(:,:,k)
      end do
    end if


    ! Surface major component ice adjustment
    xy_SurfMajCompIce = xy_SurfMajCompIce + xy_DSurfMajCompIceDt * ( 2.0_DP * DelTime )
    ! Surface pressure adjustment
    xy_Ps = xy_PsA


    call AuxVars( &
      & xy_PsA, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressA                  & ! (out) optional
      & )

    ! 成分の質量の補正
    ! Fix masses of constituents
    !
    call MassFixerColumn( &
      & xyr_PressA, & ! (in)
      & xyzf_QMix,  & ! (inout)
      & Text = "In CloudMajorComp"  & ! (in)
      & )

    ! Check
    call CloudMajorCompConsChk(       &
      & a_FlagSurfaceSink,            & ! (in)
      & xyz_DelAtmMassB, xyzf_QMixB,  & ! (in)
      & xyz_DelAtmMassA, xyzf_QMix    & ! (in)
      & )

    ! Check major component ice
    xy_ColumnLatEneB = 0.0_DP
    xy_ColumnIntEneB = 0.0_DP
    !
    xy_ColumnLatEneA = 0.0_DP
    xy_ColumnIntEneA = 0.0_DP
    do k = kmax, 1, -1
      xy_ColumnLatEneB = xy_ColumnLatEneB &
        & + LatentHeatMajCompSubl &
        &   * xyzf_QMixB(:,:,k,IndexMajCompIce) * xyz_DelAtmMassB(:,:,k)
      xy_ColumnIntEneB = xy_ColumnIntEneB &
        & + CpDry * xyz_TempB(:,:,k) * xyz_DelAtmMassB(:,:,k)
      !
      xy_ColumnLatEneA = xy_ColumnLatEneA &
        & + LatentHeatMajCompSubl &
        &   * xyzf_QMix (:,:,k,IndexMajCompIce) * xyz_DelAtmMassA(:,:,k)
        ! xyz_DelAtmMassA should be used. But, in order to do that,
        ! we need some modifications, maybe loss of internal energy
        ! should be considered. 
      xy_ColumnIntEneA = xy_ColumnIntEneA &
!        & + CpDry * xyz_Temp (:,:,k) * xyz_DelAtmMassA(:,:,k)
        & + CpDry * xyz_Temp (:,:,k) * xyz_DelAtmMassB(:,:,k)
    end do
    !
    xy_LostLatEneA = &
      &   LatentHeatMajCompSubl &
      &   * xy_DSurfMajCompIceDt * ( 2.0_DP * DelTime )
    xy_LostIntEneA = &
      & - CpDry * xyr_TempFlow(:,:,0)
    do j = 1, jmax
      do i = 0, imax-1
        TotalEneB =   xy_ColumnIntEneB(i,j) &
          &         - xy_ColumnLatEneB(i,j)
        TotalEneA =   xy_ColumnIntEneA(i,j) + xy_LostIntEneA(i,j)   &
          &         - ( xy_ColumnLatEneA(i,j) + xy_LostLatEneA(i,j) )
        Ratio = abs( TotalEneB - TotalEneA ) / TotalEneB
        if ( Ratio > 1.0e-10_DP ) then
          call MessageNotify( 'M', module_name, &
            & 'Major component ice phase change does not conserve energy, %f, at (%d,%d).', &
            & d = (/ Ratio /), i = (/ i, j /) )
        end if
      end do
    end do


    ! ヒストリデータ出力
    ! History data output
    !
    xyz_DTempDt = ( xyz_Temp - xyz_TempB ) / ( 2.0_DP * DelTime )
    call HistoryAutoPut( TimeN, 'DTempDtMajCompPhaseChange', xyz_DTempDt )

    call HistoryAutoPut( TimeN, 'MajCompCloudRadius', xyz_CldRad )


    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine TEST_CloudMajorComp

  !----------------------------------------------------------------------------

  subroutine TEST_CloudMajorComp1D( &
    & Grav, Cp, CloudNumRatio, LatentHeatMajCompSubl, TimeStep, & ! (in   )
    & kmax, r_Press, z_Press, r_Height,                         & ! (in   )
    & z_Temp, z_CldMMR,                                         & ! (inout)
    & z_CldDen, z_CldRad,                                       & ! (out  )
    & r_GravSedCldMassFlux                                      & ! (out  )
    & )

    ! 雲関系ルーチン
    ! Cloud-related routines
    !
    use cloud_utils, only : CloudUtils1BinCalcPartProp1D

    ! 重力沈降過程
    ! Gravitational sedimentation process
    !
    use grav_sed, only : &
      & GravSedCalcSedVel1D, &
      & GravSedTransport1D


    real(dp), intent(in   ) :: Grav
    real(dp), intent(in   ) :: Cp
    real(dp), intent(in   ) :: CloudNumRatio
    real(DP), intent(in   ) :: LatentHeatMajCompSubl
    real(dp), intent(in   ) :: TimeStep
    integer , intent(in   ) :: kmax
    real(dp), intent(in   ) :: r_Press     (0:kmax)
    real(dp), intent(in   ) :: z_Press     (1:kmax)
    real(dp), intent(in   ) :: r_Height    (0:kmax)
    real(dp), intent(inout) :: z_Temp      (1:kmax)
    real(dp), intent(inout) :: z_CldMMR    (1:kmax)
    real(dp), intent(out  ) :: z_CldDen    (1:kmax)
    real(dp), intent(out  ) :: z_CldRad    (1:kmax)
    real(dp), intent(out  ) :: r_GravSedCldMassFlux(0:kmax)


    ! Local variables
    !
    real(DP) :: DelTimeSmall = 1.0e100_DP

    real(dp) :: CondMaterialDen
    real(dp) :: CloudRad
    real(dp) :: z_DelNuclNum   (1:kmax)

    real(DP) :: r_PartDen(0:kmax)
    real(DP) :: r_PartRad(0:kmax)

    real(dp) :: r_SedVel     (0:kmax)
    real(dp) :: z_DelAtmMass (1:kmax)
    real(dp) :: z_DelCldMass (1:kmax)
    real(dp) :: z_DelCldMassA(1:kmax)

    real(dp) :: r_GravSedCldMassFluxEach(0:kmax)

    real(dp) :: CourantNum
    real(dp) :: TimeStepSubLoop
    integer  :: NLoop
    integer  :: iLoop

    integer  :: k


    CondMaterialDen = MajCompIceDen
    CloudRad        = -1.0_DP


    do k = 1, kmax
      z_DelAtmMass(k) = ( r_Press(k-1) - r_Press(k) ) / Grav
    end do
    z_DelNuclNum = z_DelAtmMass * CloudNumRatio


    call CloudMajorCompPhaseChangeInAtm( &
      & Cp, LatentHeatMajCompSubl, &
      & kmax, z_Press, z_DelAtmMass, &
      & z_CldMMR, z_Temp &
      & )
    call CloudUtils1BinCalcPartProp1D(       &
      & CondMaterialDen, CloudRad,                 & ! (in )
      & CloudNuclRad, CloudNuclDen,                & ! (in )
      & z_DelAtmMass, z_CldMMR, z_DelNuclNum,      & ! (in )
      & z_CldRad, z_CldDen                         & ! (out)
      & )


    r_GravSedCldMassFlux = 0.0_DP

    ! sub-loop
    NLoop = max( int( TimeStep / DelTimeSmall ), 1 )
    TimeStepSubLoop = TimeStep / dble( NLoop )
    do iLoop = 1, NLoop

      r_PartDen(0:kmax-1) = z_CldDen
      r_PartDen(kmax)     = z_CldDen(kmax)
      r_PartRad(0:kmax-1) = z_CldRad
      r_PartRad(kmax)     = z_CldRad(kmax)
      call GravSedCalcSedVel1D(                  &
        & kmax+1, r_Press, r_PartDen, r_PartRad, & ! (in )
        & r_SedVel                               & ! (out)
        & )

      if ( iLoop == 1 ) then
        ! check
        do k = 0, kmax-1
          CourantNum =   ( abs( r_SedVel(k) ) * TimeStepSubLoop ) &
            &          / ( r_Height(k+1) - r_Height(k) )
          if ( CourantNum > 1.0d0 ) then
            write( 6, * ) 'warning: CFL', k, CourantNum
          end if
        end do
      end if


      z_DelCldMass = z_DelAtmMass * z_CldMMR
      call GravSedTransport1D( &
        & TimeStepSubLoop,                                 & ! (in )
        & kmax, r_Press, r_Height, r_SedVel, z_DelCldMass, & ! (in )
        & z_DelCldMassA,                                   & ! (out)
        & r_GravSedCldMassFluxEach                         & ! (out)
        & )
      z_DelCldMass = z_DelCldMassA
      z_CldMMR = z_DelCldMass / z_DelAtmMass

      r_GravSedCldMassFlux = r_GravSedCldMassFlux + r_GravSedCldMassFluxEach

      call CloudMajorCompPhaseChangeInAtm( &
        & Cp, LatentHeatMajCompSubl, &
        & kmax, z_Press, z_DelAtmMass, &
        & z_CldMMR, z_Temp &
        & )
      call CloudUtils1BinCalcPartProp1D(       &
        & CondMaterialDen, CloudRad,                 & ! (in )
        & CloudNuclRad, CloudNuclDen,                & ! (in )
        & z_DelAtmMass, z_CldMMR, z_DelNuclNum,      & ! (in )
        & z_CldRad, z_CldDen                         & ! (out)
        & )

    end do

    r_GravSedCldMassFlux = r_GravSedCldMassFlux / TimeStep


  end subroutine TEST_CloudMajorComp1D

  !----------------------------------------------------------------------------

  subroutine CloudMajorCompPhaseChangeInAtm( &
    & Cp, MajCompLatentHeat, &
    & kmax, z_Press, z_DelAtmMass, &
    & z_CldMMR, z_Temp &
    & )

    use constants0, only : PI

    ! 主成分相変化
    ! Phase change of atmospheric major component
    !
    use saturate_major_comp, only : SaturateMajorCompCondTemp

    real(dp), intent(in   ) :: Cp
    real(dp), intent(in   ) :: MajCompLatentHeat
    integer , intent(in   ) :: kmax
    real(dp), intent(in   ) :: z_Press     (1:kmax)
    real(dp), intent(in   ) :: z_DelAtmMass(1:kmax)
    real(dp), intent(inout) :: z_CldMMR    (1:kmax)
    real(dp), intent(inout) :: z_Temp      (1:kmax)


    ! Local variables
    !
    real(dp) :: z_DelCldMass(1:kmax)
    real(dp) :: z_TempSat   (1:kmax)

    integer  :: k


    z_DelCldMass = z_DelAtmMass * z_CldMMR


    ! Sublimation
    z_Temp = z_Temp - z_DelCldMass * MajCompLatentHeat / ( Cp * z_DelAtmMass )
    z_DelCldMass = 0.0d0
    !
    call SaturateMajorCompCondTemp(  &
      & z_Press,                     &  ! (in)
      & z_TempSat                    &  ! (out)
      & )
    do k = 1, kmax
      if ( z_Temp(k) < z_TempSat(k) ) then ! Condensation
        z_DelCldMass(k) = ( z_TempSat(k) - z_Temp(k) ) * Cp * z_DelAtmMass(k) / MajCompLatentHeat
        z_Temp(k) = z_TempSat(k)
!!$        write( 6, * ) 'Atmosphere condenses at k = ', k, &
!!$          & ' : ', r_Press(k), r_Temp(k), TempSat
      end if
    end do

    z_CldMMR = z_DelCldMass / z_DelAtmMass


  end subroutine CloudMajorCompPhaseChangeInAtm

  !------------------------------------------------------------------------------------

  subroutine CloudMajorCompCalcSedMassFlux1D( &
    & TimeStep,                        & ! (in   )
    & LatentHeatMajCompSubl,           & ! (in   )
    & r_Press, r_Height,               & ! (in   )
    & z_PartDen, z_PartRad,            & ! (in   )
    & z_TempCond,                      & ! (in   )
    & z_Temp,                          & ! (inout)
    & z_MajCompIceMass,                & ! (inout)
    & r_GravSedMassFlux,               & ! (out  )
    & r_DPPress                        & ! (out  )
    & )

    ! 物理・数学定数設定
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI                    ! $ \pi $ .
                              ! 円周率.  Circular constant

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Grav, &               ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration
      & CpDry
                              ! $ C_p $ [J kg-1 K-1].
                              ! 乾燥大気の定圧比熱.
                              ! Specific heat of air at constant pressure

    ! 重力沈降過程
    ! Gravitational sedimentation process
    !
    use grav_sed, only : GravSedCalcSedVel1D

    real(DP)    , intent(in   ) :: TimeStep
    real(DP)    , intent(in   ) :: LatentHeatMajCompSubl
    real(DP)    , intent(in   ) :: r_Press   (0:kmax)
    real(DP)    , intent(in   ) :: r_Height  (0:kmax)
    real(DP)    , intent(in   ) :: z_PartDen (1:kmax)
    real(DP)    , intent(in   ) :: z_PartRad (1:kmax)
    real(DP)    , intent(in   ) :: z_TempCond(1:kmax)
    real(DP)    , intent(inout) :: z_Temp    (1:kmax)
    real(DP)    , intent(inout) :: z_MajCompIceMass (1:kmax)
    real(DP)    , intent(out  ) :: r_GravSedMassFlux(0:kmax)
    real(DP)    , intent(out  ) :: r_DPPress        (0:kmax)


    !
    ! local variables
    !
    real(DP) :: r_PartDen(0:kmax)
    real(DP) :: r_PartRad(0:kmax)

    real(DP) :: z_DelAtmMass    (1:kmax)
    real(DP) :: z_DelCompMass   (1:kmax)
    real(DP) :: FallingCompMass
    real(DP) :: FallingCompMassSubl
    real(DP) :: z_CompMassSink  (1:kmax)

    real(DP) :: r_SedVel        (0:kmax)
    real(DP) :: r_GravSedMass   (0:kmax)

    real(DP) :: r_DPressDt      (0:kmax)

    real(DP) :: APHeight
    real(DP) :: DPHeight

    real(DP) :: MassOfPartialLayer
    real(DP) :: MassConsumed

    real(DP) :: Press1
    real(DP) :: Press2
    real(DP) :: Press

    real(DP) :: DelTempSubl1
    real(DP) :: DelTempSubl2
    real(DP) :: DelTempSubl

    integer  :: k
    integer  :: kk
    integer  :: kkk


    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. cloud_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! Calculation of mass in each layer and layer thickness in unit of meter
    !   Layer thickness is calculated by using mass of a layer.
    do k = 1, kmax
      z_DelAtmMass(k) = ( r_Press(k-1) - r_Press(k) ) / Grav
    end do

    z_DelCompMass = z_MajCompIceMass

    !
    ! calculation of sedimentation terminal velocity
    !
    r_PartDen(0:kmax-1) = z_PartDen
    r_PartDen(kmax)     = z_PartDen(kmax)
    r_PartRad(0:kmax-1) = z_PartRad
    r_PartRad(kmax)     = z_PartRad(kmax)
    if ( FlagSedimentationInfty ) then
      r_SedVel = -1.0d100
    else
      call GravSedCalcSedVel1D(                  &
        & kmax+1, r_Press, r_PartDen, r_PartRad, & ! (in )
        & r_SedVel                               & ! (out)
        & )
    end if


    ! arbitrary upper (lower) limit
!!$        r_SedVel = sign( 1.0_DP, r_SedVel ) * max( abs(r_SedVel), 1.0e-20_DP )


    ! Evaluation of gravitational sedimentation mass

    !   Initialization
    r_GravSedMass   = 0.0_DP
    z_CompMassSink  = 0.0_DP

    ! kk : layer index
    do kk = 1, kmax

      ! Initialization
      FallingCompMass = z_DelCompMass(kk)

      ! k : level index
      loop_level_k : do k = kk-1, 0, -1

        ! mass is sublimed in (kkk)th = (k+1)th layer during sedimentation
        !   kkk : layer index
        kkk = k+1
        !   sublimation
        !     temperature decrease when only part of ice sublime
        DelTempSubl1 = max( z_Temp(kkk) - z_TempCond(kkk), 0.0_DP )
        !     temperature decrease when all of ice sublime
        DelTempSubl2 = &
          &   LatentHeatMajCompSubl * FallingCompMass &
          &   / ( CpDry * z_DelAtmMass(kkk) )
        DelTempSubl = min( DelTempSubl1, DelTempSubl2 )
        z_Temp(kkk) = z_Temp(kkk) - DelTempSubl
        FallingCompMassSubl =                         &
          &   CpDry * z_DelAtmMass(kkk) * DelTempSubl &
          &   / LatentHeatMajCompSubl
        FallingCompMass     = FallingCompMass     - FallingCompMassSubl
        z_CompMassSink(kkk) = z_CompMassSink(kkk) + FallingCompMassSubl


        ! check whether (kk-1)th level crosses kth level
        !   note : (kk-1)th level is a bottom of (kk)th layer
        !   note : APHeight : height of arrival point
        APHeight = r_Height(kk-1) + r_SedVel(kk-1) * TimeStep
        if ( r_Height(k) <= APHeight ) then
          ! (kk-1)th level does not cross kth level
          FallingCompMass = 0.0_DP
          exit loop_level_k
        else
          ! (kk-1)th level crosses kth level

          ! check whether (kk)th level crosses kth level
          !   note : (kk)th level is a top of (kk)th layer
          !   Here, sedimentation velocity in (kk)th layer is assumed to be
          !   constant with a velocity at (kk-1)th level (bottom level of a
          !   layer).
          !   So, sedimentation velocity at (kk-1)th level is used.
          !   note : r_SedVel is negative
          !   note : APHeight : height of arrival point
          APHeight = r_Height(kk) + r_SedVel(kk-1) * TimeStep
          if ( APHeight <= r_Height(k) ) then
            ! All mass in (kk)th layer cross kth level

            ! do nothing
          else
            ! Part of mass in (kk)th layer crosses kth level
            ! note : r_SedVel is negative

            ! note : DPHeight : height of departure point, i.e., height which
            !                   reaches kth level at end of a time step
            DPHeight = r_Height(k) - r_SedVel(kk-1) * TimeStep

            ! evaluate pressure at the departure point
            !   Press1 : pressure at bottom of (kk)th layer
            !   Press2 : pressure at top    of (kk)th layer
            Press1 = r_Press(kk-1)
            if ( r_Press(kk) == 0.0_DP ) then
              Press2 = r_Press(kk-1) * 1.0e-1_DP
            else
              Press2 = r_Press(kk  )
            end if
            !   Press  : pressure at the departure point
            Press = &
              &   log( Press2 / Press1 )              &
              & / ( r_Height  (kk) - r_Height(kk-1) ) &
              & * ( DPHeight       - r_Height(kk-1) ) &
              & + log( Press1 )
            Press = exp( Press )
!!$            DelCompMass =              &
!!$              &   ( Press1 - Press  )  &
!!$              & / ( Press1 - Press2 )  &
!!$              & * z_DelCompMass(kk)
            ! MassOfPartialLayer : Mass between Press1 and Press, i.e., mass
            !                    : which crosses kth level
            MassOfPartialLayer =       &
              &   ( Press1 - Press  )  &
              & / ( Press1 - Press2 )  &
              & * z_DelCompMass(kk)
            ! MassConsumed : Mass of (kk)th layer which is consumed (sublimed)
            ! in layers above
            MassConsumed = z_DelCompMass(kk) - FallingCompMass
            ! Concept:
            !   MassOfPartialLayer is mass of bottom part of (kk)th layer.
            !   Mass of bottom part of layer will be consumed (sublimed) in
            !   layers above, first (preferentially).
            FallingCompMass = max( MassOfPartialLayer - MassConsumed, 0.0_DP )
          end if

        end if

        ! Gravitational sedimentation mass is increased by contribution of 
        ! (kk)th layer (remnant of (kk)th layer mass). 
        r_GravSedMass(k) = r_GravSedMass(k) + FallingCompMass

      end do loop_level_k

    end do


!!$    ! Modification of sedimentation mass to avoid negative mass
!!$    do k = kmax, 1, -1
!!$      ! note: negative sign on r_GravSedMass means downward transport
!!$      DelCompMassTentative = z_DelCompMass(k) &
!!$        & - ( ( -r_GravSedMass(k) ) - ( -r_GravSedMass(k-1) ) ) &
!!$        & - z_CompMassSink(k)
!!$      if ( DelCompMassTentative < 0.0_DP ) then
!!$        r_GravSedMass(k-1) = z_DelCompMass(k) + r_GravSedMass(k) - z_CompMassSink(k)
!!$        z_DelCompMass(k) = 0.0_DP
!!$      else
!!$        z_DelCompMass(k) = DelCompMassTentative
!!$      end if
!!$    end do
    do k = 1, kmax
      ! note: negative sign on r_GravSedMass means downward transport
      z_DelCompMass(k) = z_DelCompMass(k)                       &
        & - ( ( -r_GravSedMass(k) ) - ( -r_GravSedMass(k-1) ) ) &
        & - z_CompMassSink(k)
    end do

    ! note: negative sign on r_GravSedMass means downward transport
    r_GravSedMassFlux = - r_GravSedMass / TimeStep

    z_MajCompIceMass = z_DelCompMass

    r_DPressDt = - r_GravSedMassFlux * Grav
    r_DPPress = r_Press + r_DPressDt * TimeStep


  end subroutine CloudMajorCompCalcSedMassFlux1D

  !--------------------------------------------------------------------------------------

  subroutine CloudMajorCompCalcSedMassFluxInternal(       &
    & TimeStep,                                           & ! (in)
    & LatentHeatMajCompSubl,                              & ! (in)
    & r_Press, r_KIndex, r_DPPress,                       & ! (in)
    & z_TempCond,                                         & ! (in)
    & z_DelAtmMass, z_DelCompMass,                        & ! (in)
    & z_Temp,                                             & ! (inout)
    & r_QMixFlux                                          & ! (out)
    & )

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & CpDry
                              ! $ C_p $ [J kg-1 K-1].
                              ! 乾燥大気の定圧比熱.
                              ! Specific heat of air at constant pressure

    real(DP), intent(in   ) :: TimeStep
    real(DP), intent(in   ) :: LatentHeatMajCompSubl
    real(DP), intent(in   ) :: r_Press      (0:kmax)
    integer , intent(in   ) :: r_KIndex     (0:kmax)
    real(DP), intent(in   ) :: r_DPPress    (0:kmax)
    real(DP), intent(in   ) :: z_TempCond   (1:kmax)
    real(DP), intent(in   ) :: z_DelAtmMass (1:kmax)
    real(DP), intent(in   ) :: z_DelCompMass(1:kmax)
    real(DP), intent(inout) :: z_Temp       (1:kmax)
    real(DP), intent(out  ) :: r_QMixFlux   (0:kmax)


    ! Local variables
    !
    real(DP) :: DelCompMass
    real(DP) :: DelTempSubl1
    real(DP) :: DelTempSubl2
    real(DP) :: DelTempSubl
    real(DP) :: r_TotMassFlow (0:kmax)
    real(DP) :: r_IntMassFlow (0:kmax)
    real(DP) :: r_FracMassFlow(0:kmax)
    real(DP) :: Press
    real(DP) :: Press1
    real(DP) :: Press2

!!$    real(DP) :: z_DelCompMassA(1:kmax)

    integer :: k
    integer :: kk
    integer :: kkk


    !
    ! Calculation of mass contained in two departure points
    !
    ! schematic figure of indexing
    !
    !          arrival                     departure
    !
    ! k+4 ===================         ===================
    !                            ---- -------------------  KIndex(k+1)
    !                           |
    !                           | --- -------------------  KIndex(k  )
    ! k+3 ===================   ||    =================== (KIndex(k  )-1)
    !                           ||
    !                           ||        (from KIndex(k-1)+1 to KIndex(k)-1)
    !                           ||
    ! k+2 ===================   ||    =================== (KIndex(k-1))
    !                           ||
    !                           || -- -------------------  KIndex(k-1)
    !                           |||
    ! k+1 =================== <- ||   ===================
    !                            ||
    !                            ||
    !                            ||
    ! k   =================== <-- |   ===================
    !                             |
    !                             |
    !                             |
    ! k-1 =================== <---    ===================
    !

!!$    z_DelCompMassA = 0.0_DP

    !
    ! Calculation of mass crossing layer interfaces
    !

    k = kmax
    r_IntMassFlow (k) = 0.0_DP
    r_FracMassFlow(k) = 0.0_DP
    !
    do k = kmax-1, 0, -1

      ! integer mass flow
      r_IntMassFlow(k)  = 0.0_DP
      !
      do kk = k+1, r_KIndex(k)-1
        DelCompMass = z_DelCompMass(kk)

        ! mass is sublimed during sedimentation
        !   check downward
        do kkk = kk-1, k+1, -1
          ! sublimation
          !   temperature decrease when only part of ice sublime
          DelTempSubl1 = max( z_Temp(kkk) - z_TempCond(kkk), 0.0_DP )
          !   temperature decrease when all of ice sublime
          DelTempSubl2 = &
            &   LatentHeatMajCompSubl * DelCompMass &
            &   / ( CpDry * z_DelAtmMass(kkk) )
          DelTempSubl = min( DelTempSubl1, DelTempSubl2 )
          z_Temp(kkk) = z_Temp(kkk) - DelTempSubl
          DelCompMass = DelCompMass                     &
            & - CpDry * z_DelAtmMass(kkk) * DelTempSubl &
            &   / LatentHeatMajCompSubl
        end do
        r_IntMassFlow(k) = r_IntMassFlow(k) + DelCompMass

      end do

      ! fractional mass flow
      r_FracMassFlow(k) = 0.0_DP
      !
      kk = r_KIndex(k)
      !-----
      !   Method considering exponential distribution of mass with height
      Press  = r_DPPress(k)
      Press1 = r_Press(kk-1)
      if ( r_Press(kk) == 0.0_DP ) then
        Press2 = r_Press(kk-1) * 1.0e-1_DP
      else
        Press2 = r_Press(kk  )
      end if
      DelCompMass =              &
        &   ( Press1 - Press  )  &
        & / ( Press1 - Press2 )  &
        & * z_DelCompMass(kk)
      !-----
      !   mass is sublimed during sedimentation
      !   check downward
      do kkk = kk-1, k+1, -1
        ! sublimation
        !   temperature decrease when only part of ice sublime
        DelTempSubl1 = max( z_Temp(kkk) - z_TempCond(kkk), 0.0_DP )
        !   temperature decrease when all of ice sublime
        DelTempSubl2 = &
          &   LatentHeatMajCompSubl * DelCompMass &
          &   / ( CpDry * z_DelAtmMass(kkk) )
        DelTempSubl = min( DelTempSubl1, DelTempSubl2 )
        z_Temp(kkk) = z_Temp(kkk) - DelTempSubl
        DelCompMass = DelCompMass                     &
          & - CpDry * z_DelAtmMass(kkk) * DelTempSubl &
          &   / LatentHeatMajCompSubl
      end do
      r_FracMassFlow(k) = DelCompMass

    end do

    r_TotMassFlow = r_IntMassFlow + r_FracMassFlow

    ! Add sign of sedimentation velocity.
    ! This is equivalent to mulplying -1.
!!$    r_TotMassFlow = sign( 1.0_DP, r_SedVel ) * r_TotMassFlow
    r_TotMassFlow = - r_TotMassFlow

    ! calculation of flux
    r_QMixFlux = r_TotMassFlow / TimeStep


!!$    do k = 1, kmax
!!$      z_DelCompMassA(k) = z_DelCompMassA(k) &
!!$        & + ( )
!!$    end do


  end subroutine CloudMajorCompCalcSedMassFluxInternal

  !--------------------------------------------------------------------------------------

  subroutine CloudMajorCompNoCloud(                  &
    & xyr_Press, xyz_Press,                          & ! (in)
    & xy_Ps, xyz_Temp, xyzf_QMix, xyz_U, xyz_V,      & ! (inout)
    & xy_SurfMajCompIce                              & ! (inout)
    & )
    !
    ! CO2 相変化
    !
    ! CO2 phase change
    !

    ! モジュール引用 ; USE statements
    !

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & DelTime, &            ! $ \Delta t $
      & TimeN, &              ! ステップ $ t $ の時刻. Time of step $ t $. 
      & TimesetClockStart, TimesetClockStop

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 組成に関わる配列の設定
    ! Settings of array for atmospheric composition
    !
    use composition, only : &
      & ncmax, &
      & IndexTKE

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Grav, &               ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration
      & CpDry
                              ! $ C_p $ [J kg-1 K-1].
                              ! 乾燥大気の定圧比熱.
                              ! Specific heat of air at constant pressure

    ! 温度の半整数σレベルの補間, 気圧と高度の算出
    ! Interpolate temperature on half sigma level, 
    ! and calculate pressure and height
    !
    use auxiliary, only: AuxVars

    ! 主成分相変化
    ! Phase change of atmospheric major component
    !
    use saturate_major_comp, only :    &
      & SaturateMajorCompCondTemp,     &
      & SaturateMajorCompInqLatentHeat

    ! 質量の補正
    ! Mass fixer
    !
    use mass_fixer, only: MassFixerColumn

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in   ):: xyr_Press (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ . 気圧 (半整数レベル). 
                              ! Air pressure (half level)
    real(DP), intent(in   ):: xyz_Press (0:imax-1, 1:jmax, 1:kmax)
                              ! $ p $ . 気圧 (整数レベル). 
                              ! Air pressure (full level)
    real(DP), intent(inout):: xy_Ps            (0:imax-1, 1:jmax)
                              ! $ T $ .     温度. Temperature
    real(DP), intent(inout):: xyz_Temp         (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ .     温度. Temperature
    real(DP), intent(inout):: xyzf_QMix        (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
    real(DP), intent(inout):: xyz_U            (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout):: xyz_V            (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout):: xy_SurfMajCompIce(0:imax-1, 1:jmax)
                              !
                              ! Surface major component ice amount

    ! 作業変数
    ! Work variables
    !
    real(DP):: LatentHeatMajCompSubl

    real(DP):: xy_PsB              (0:imax-1, 1:jmax)
    real(DP):: xy_PsA              (0:imax-1, 1:jmax)
    real(DP):: xyr_PressB          (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_PressA          (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyzf_QMixB          (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
    real(DP):: xyzf_QMixA          (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)

    real(DP):: xy_PsSave           (0:imax-1, 1:jmax)
    real(DP):: xyz_TempSave        (0:imax-1, 1:jmax, 1:kmax)
                              ! 調節前の温度. 
                              ! Temperature before adjustment
    real(DP):: xyzf_QMixSave       (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)

    real(DP):: xyz_DelAtmMass      (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xy_TempTmp          (0:imax-1, 1:jmax)
    real(DP):: xy_DTempDtSubl      (0:imax-1, 1:jmax)
    real(DP):: xy_DTempDtCond      (0:imax-1, 1:jmax)
    real(DP):: xyr_AtmMassFallFlux (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_TempTmp         (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_QH2OVapTmp      (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xyz_DTempDt         (0:imax-1, 1:jmax, 1:kmax)
                              ! 温度変化率. 
                              ! Temperature tendency
    real(DP):: xy_DSurfMajCompIceDt(0:imax-1, 1:jmax)
                              ! 
                              ! Surface major component ice tendency
    real(DP):: xy_DPsDt            (0:imax-1, 1:jmax)

    real(DP):: xy_DSurfMajCompIceDtOneLayer(0:imax-1, 1:jmax)
    real(DP):: xyr_DPressDt                (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_DPPress                 (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_TempCond   (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: DelTimeSmall
    real(DP):: DPPress
    integer :: iLoop
    integer :: NLoop

    integer :: mmax
    real(DP):: xyza_Array       (0:imax-1, 1:jmax, 1:kmax, 1:ncmax+1+1+1)
    logical :: a_FlagSurfaceSink                          (1:ncmax+1+1+1)
    real(DP):: xyra_MassFlow    (0:imax-1, 1:jmax, 0:kmax, 1:ncmax+1+1+1)

    real(DP):: xyrf_MassFlow    (0:imax-1, 1:jmax, 0:kmax, 1:ncmax)
    real(DP):: xyr_TempFlow     (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_MomXFlow     (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_MomYFlow     (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_DelAtmMassB   (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_DelAtmMassA   (0:imax-1, 1:jmax, 1:kmax)

    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction
    integer:: m
    integer:: n

!!$    logical :: FlagCheckPs


    ! 実行文 ; Executable statement
    !

    ! 初期化
    ! Initialization
    !
    if ( .not. cloud_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    if ( .not. FlagMajCompPhaseChange ) return


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    ! Set latent heat
    LatentHeatMajCompSubl = SaturateMajorCompInqLatentHeat()


!!$    FlagCheckPs = .false.
!!$    do j = 1, jmax
!!$      do i = 0, imax-1
!!$        if ( xyr_Press(i,j,0) > 1.0e4_DP ) then
!!$          FlagCheckPs = .true.
!!$        end if
!!$      end do
!!$    end do
!!$    if ( FlagCheckPs ) then
!!$      call MessageNotify( 'W', module_name, 'Surface pressure is greater than 10000 Pa.' )
!!$    end if


    ! Store variables
    !
    xy_PsSave     = xy_Ps
    xyz_TempSave  = xyz_Temp
    xyzf_QMixSave = xyzf_QMix


    call SaturateMajorCompCondTemp( &
      & xyz_Press,                  & ! (in)
      & xyz_TempCond                & ! (inout)
      & )


    do k = 1, kmax
      xyz_DelAtmMass(:,:,k) = ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k) ) / Grav
    end do

    xyr_AtmMassFallFlux(:,:,kmax) = 0.0_DP
    do k = kmax, 1, -1
      ! sublimation of falling condensate
      xy_DTempDtSubl = &
        & - LatentHeatMajCompSubl * xyr_AtmMassFallFlux(:,:,k) &
        &   / ( CpDry * xyz_DelAtmMass(:,:,k) )
      xyz_Temp(:,:,k) = xyz_Temp(:,:,k) + xy_DTempDtSubl * ( 2.0_DP * DelTime )
      xyr_AtmMassFallFlux(:,:,k) = 0.0_DP

      ! condensation
      xy_TempTmp = xyz_Temp(:,:,k)
      xyz_Temp(:,:,k) = max( xyz_TempCond(:,:,k), xyz_Temp(:,:,k) )
      xy_DTempDtCond = ( xyz_Temp(:,:,k) - xy_TempTmp ) / ( 2.0_DP * DelTime )
      xy_DSurfMajCompIceDtOneLayer =                           &
        &   CpDry * xyz_DelAtmMass(:,:,k) * xy_DTempDtCond     &
        &   / LatentHeatMajCompSubl
      xyr_AtmMassFallFlux(:,:,k-1) = xyr_AtmMassFallFlux(:,:,k) &
        & + xy_DSurfMajCompIceDtOneLayer
    end do

    xyr_DPressDt = - xyr_AtmMassFallFlux * Grav


    xy_DPsDt             = xyr_DPressDt(:,:,0)
    xy_DSurfMajCompIceDt = - xy_DPsDt / Grav



    NLoop = 1
    do k = 0, kmax-1
      do j = 1, jmax
        do i = 0, imax-1

          do
            DelTimeSmall = ( 2.0_DP * DelTime ) / dble( NLoop )

            DPPress = xyr_Press(i,j,k) + xyr_DPressDt(i,j,k) * DelTimeSmall
            if ( DPPress < xyr_Press(i,j,k+1) ) then
              NLoop = NLoop + 1
            else
              exit
            end if
            if ( NLoop > 100 ) then
              call MessageNotify( 'E', module_name, &
                & 'Number of loop to search NLoop is greater than 100.' )
            end if
          end do

        end do
      end do
    end do


    DelTimeSmall = ( 2.0_DP * DelTime ) / dble( NLoop )
    !
    do iLoop = 1, NLoop
      xyz_TempTmp    = 300.0_DP
      xyz_QH2OVapTmp =   0.0_DP
      call AuxVars( &
        & xy_Ps, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
        & xyr_Press = xyr_PressB                 & ! (out) optional
        & )
      xyr_DPPress = xyr_PressB + xyr_DPressDt * DelTimeSmall

      ! packing
      mmax = ncmax
      if ( FlagModTemp ) then
        mmax = mmax + 1
      end if
      if ( FlagModMom ) then
        mmax = mmax + 2
      end if
      do m = 1, ncmax
        n = m
        xyza_Array(:,:,:,m) = xyzf_QMix(:,:,:,n)
        if ( n == IndexTKE ) then
          a_FlagSurfaceSink(m) = .true.
        else
          a_FlagSurfaceSink(m) = .false.
        end if
      end do
      m = ncmax
      if ( FlagModTemp ) then
        m = m + 1
        xyza_Array(:,:,:,m) = xyz_Temp
        a_FlagSurfaceSink(m) = .true.
      end if
      if ( FlagModMom ) then
        m = m + 1
        xyza_Array(:,:,:,m) = xyz_U
        a_FlagSurfaceSink(m) = .true.
        m = m + 1
        xyza_Array(:,:,:,m) = xyz_V
        a_FlagSurfaceSink(m) = .true.
      end if

      call CloudMajorCompCalcFlow(       &
        & xyr_PressB, xyr_DPPress,                                     & ! (in)
        & mmax, a_FlagSurfaceSink(1:mmax), xyza_Array(:,:,:,1:mmax),   & ! (in)
        & xyra_MassFlow(:,:,:,1:mmax)                                  & ! (out)
        & )

      ! unpacking
      do m = 1, ncmax
        xyrf_MassFlow(:,:,:,m) = xyra_MassFlow(:,:,:,m)
      end do
      m = ncmax
      if ( FlagModTemp ) then
        m = m + 1
        xyr_TempFlow = xyra_MassFlow(:,:,:,m)
      else
        xyr_TempFlow = 0.0_DP
      end if
      if ( FlagModMom ) then
        m = m + 1
        xyr_MomXFlow = xyra_MassFlow(:,:,:,m)
        m = m + 1
        xyr_MomYFlow = xyra_MassFlow(:,:,:,m)
      else
        xyr_MomXFlow = 0.0_DP
        xyr_MomYFlow = 0.0_DP
      end if


      ! Adjustment
      !   preparation
      xy_PsB = xy_Ps
      xy_PsA = xy_PsB + xy_DPsDt * DelTimeSmall

      ! 温度の半整数σレベルの補間, 気圧と高度の算出
      ! Interpolate temperature on half sigma level, 
      ! and calculate pressure and height
      !
      xyz_TempTmp    = 300.0_DP
      xyz_QH2OVapTmp =   0.0_DP
      call AuxVars( &
        & xy_PsB, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
        & xyr_Press = xyr_PressB                  & ! (out) optional
        & )
      call AuxVars( &
        & xy_PsA, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
        & xyr_Press = xyr_PressA                  & ! (out) optional
        & )
      do k = 1, kmax
        xyz_DelAtmMassB(:,:,k) = ( xyr_PressB(:,:,k-1) - xyr_PressB(:,:,k) ) / Grav
        xyz_DelAtmMassA(:,:,k) = ( xyr_PressA(:,:,k-1) - xyr_PressA(:,:,k) ) / Grav
      end do
      !   Atmospheric composition
      do n = 1, ncmax
        do k = 1, kmax
          xyzf_QMix(:,:,k,n) =                                              &
            &   (   xyz_DelAtmMassB(:,:,k) * xyzf_QMix(:,:,k,n)             &
            &     - ( xyrf_MassFlow(:,:,k,n) - xyrf_MassFlow(:,:,k-1,n) ) ) &
            & / xyz_DelAtmMassA(:,:,k)
        end do
      end do
      if ( FlagModTemp ) then
        do k = 1, kmax
          ! Temperature
          xyz_Temp(:,:,k) =                                           &
            &   (   xyz_DelAtmMassB(:,:,k) * xyz_Temp(:,:,k)          &
            &     - ( xyr_TempFlow(:,:,k) - xyr_TempFlow(:,:,k-1) ) ) &
            & / xyz_DelAtmMassA(:,:,k)
        end do
      end if
      if ( FlagModMom ) then
        do k = 1, kmax
          ! Zonal wind
          xyz_U(:,:,k) =                                              &
            &   (   xyz_DelAtmMassB(:,:,k) * xyz_U(:,:,k)             &
            &     - ( xyr_MomXFlow(:,:,k) - xyr_MomXFlow(:,:,k-1) ) ) &
            & / xyz_DelAtmMassA(:,:,k)
          ! Meridional wind
          xyz_V(:,:,k) =                                              &
            &   (   xyz_DelAtmMassB(:,:,k) * xyz_V(:,:,k)             &
            &     - ( xyr_MomYFlow(:,:,k) - xyr_MomYFlow(:,:,k-1) ) ) &
            & / xyz_DelAtmMassA(:,:,k)
        end do
      end if

      ! Surface pressure adjustment
      xy_Ps = xy_PsA

      call AuxVars( &
        & xy_PsA, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
        & xyr_Press = xyr_PressA                  & ! (out) optional
        & )
      ! 成分の質量の補正
      ! Fix masses of constituents
      !
      call MassFixerColumn( &
        & xyr_PressA, & ! (in)
        & xyzf_QMix   & ! (inout)
        & )

    end do

    ! Surface major component ice adjustment
    xy_SurfMajCompIce = xy_SurfMajCompIce + xy_DSurfMajCompIceDt * ( 2.0_DP * DelTime )


    ! Check
    !   preparation
    xy_PsB = xy_PsSave
    xy_PsA = xy_PsB + xy_DPsDt * ( 2.0_DP * DelTime )

    xyzf_QMixB = xyzf_QMixSave
    xyzf_QMixA = xyzf_QMix

    ! 温度の半整数σレベルの補間, 気圧と高度の算出
    ! Interpolate temperature on half sigma level, 
    ! and calculate pressure and height
    !
    xyz_TempTmp    = 300.0_DP
    xyz_QH2OVapTmp =   0.0_DP
    call AuxVars( &
      & xy_PsB, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressB                  & ! (out) optional
      & )
    call AuxVars( &
      & xy_PsA, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressA                  & ! (out) optional
      & )
    do k = 1, kmax
      xyz_DelAtmMassB(:,:,k) = ( xyr_PressB(:,:,k-1) - xyr_PressB(:,:,k) ) / Grav
      xyz_DelAtmMassA(:,:,k) = ( xyr_PressA(:,:,k-1) - xyr_PressA(:,:,k) ) / Grav
    end do
    call CloudMajorCompConsChk(      &
      & a_FlagSurfaceSink,           & ! (in)
      & xyz_DelAtmMassB, xyzf_QMixB, & ! (in)
      & xyz_DelAtmMassA, xyzf_QMixA  & ! (in)
      & )

    ! ヒストリデータ出力
    ! History data output
    !
    xyz_DTempDt = ( xyz_Temp - xyz_TempSave ) / ( 2.0_DP * DelTime )

    call HistoryAutoPut( TimeN, 'DTempDtMajCompPhaseChange', xyz_DTempDt )


    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine CloudMajorCompNoCloud

  !--------------------------------------------------------------------------------------

  subroutine MajorCompPhaseChangeOnGround(      &
    & xy_DPsDt, xy_DSurfMajCompIceDt,           & ! (in)
    & xy_Ps, xyzf_QMix, xyz_Temp, xyz_U, xyz_V, & ! (inout)
    & xy_SurfMajCompIce                         & ! (inout)
    & )
    !
    ! CO2 相変化
    !
    ! CO2 phase change
    !

    ! モジュール引用 ; USE statements
    !

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & DelTime, &            ! $ \Delta t $
      & TimesetClockStart, TimesetClockStop

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 組成に関わる配列の設定
    ! Settings of array for atmospheric composition
    !
    use composition, only : &
      & ncmax, &
      & IndexTKE

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Grav, &               ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration
      & CpDry
                              ! $ C_p $ [J kg-1 K-1].
                              ! 乾燥大気の定圧比熱.
                              ! Specific heat of air at constant pressure

    ! 温度の半整数σレベルの補間, 気圧と高度の算出
    ! Interpolate temperature on half sigma level, 
    ! and calculate pressure and height
    !
    use auxiliary, only: AuxVars

    ! 質量の補正
    ! Mass fixer
    !
    use mass_fixer, only: MassFixerColumn


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in   ):: xy_DPsDt            (0:imax-1, 1:jmax)
    real(DP), intent(in   ):: xy_DSurfMajCompIceDt(0:imax-1, 1:jmax)
    real(DP), intent(inout):: xy_Ps            (0:imax-1, 1:jmax)
    real(DP), intent(inout):: xyzf_QMix        (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
    real(DP), intent(inout):: xyz_Temp         (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout):: xyz_U            (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout):: xyz_V            (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(inout):: xy_SurfMajCompIce   (0:imax-1, 1:jmax)
                              !
                              ! Surface major component ice amount

    ! 作業変数
    ! Work variables
    !
    real(DP):: xy_PsB              (0:imax-1, 1:jmax)
    real(DP):: xy_PsA              (0:imax-1, 1:jmax)
    real(DP):: xyr_PressB          (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_PressA          (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyzf_QMixB          (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)

    real(DP):: xyz_TempTmp         (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_QH2OVapTmp      (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xyr_DPressDt        (0:imax-1, 1:jmax, 0:kmax)

    integer :: mmax
    real(DP):: xyza_Array       (0:imax-1, 1:jmax, 1:kmax, 1:ncmax+1+1+1)
    logical :: a_FlagSurfaceSink                          (1:ncmax+1+1+1)
    real(DP):: xyra_MassFlow    (0:imax-1, 1:jmax, 0:kmax, 1:ncmax+1+1+1)

    real(DP):: xyrf_MassFlow    (0:imax-1, 1:jmax, 0:kmax, 1:ncmax)
    real(DP):: xyr_TempFlow     (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_MomXFlow     (0:imax-1, 1:jmax, 0:kmax)
    real(DP):: xyr_MomYFlow     (0:imax-1, 1:jmax, 0:kmax)

    real(DP):: xyz_DelAtmMassB   (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_DelAtmMassA   (0:imax-1, 1:jmax, 1:kmax)

    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction
    integer:: m
    integer:: n


    ! 実行文 ; Executable statement
    !

    ! 初期化
    ! Initialization
    !
    if ( .not. cloud_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    if ( .not. FlagMajCompPhaseChange ) return


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    xy_PsB = xy_Ps
    xy_PsA = xy_PsB + xy_DPsDt * ( 2.0_DP * DelTime )

    xyzf_QMixB = xyzf_QMix

    xyz_TempTmp    = 300.0_DP
    xyz_QH2OVapTmp =   0.0_DP

    ! 温度の半整数σレベルの補間, 気圧と高度の算出
    ! Interpolate temperature on half sigma level, 
    ! and calculate pressure and height
    !
    call AuxVars( &
      & xy_PsB, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressB                  & ! (out) optional
      & )
    call AuxVars( &
      & xy_PsA, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressA                  & ! (out) optional
      & )


    xyr_DPressDt = ( xyr_PressA - xyr_PressB ) / ( 2.0_DP * Deltime )


    ! packing
    mmax = ncmax
    if ( FlagModTemp ) then
      mmax = mmax + 1
    end if
    if ( FlagModMom ) then
      mmax = mmax + 2
    end if
    do m = 1, ncmax
      n = m
      xyza_Array(:,:,:,m) = xyzf_QMix(:,:,:,n)
      if ( n == IndexTKE ) then
        a_FlagSurfaceSink(m) = .true.
      else
        a_FlagSurfaceSink(m) = .false.
      end if
    end do
    m = ncmax
    if ( FlagModTemp ) then
      m = m + 1
      xyza_Array(:,:,:,m) = xyz_Temp
      a_FlagSurfaceSink(m) = .true.
    end if
    if ( FlagModMom ) then
      m = m + 1
      xyza_Array(:,:,:,m) = xyz_U
      a_FlagSurfaceSink(m) = .true.
      m = m + 1
      xyza_Array(:,:,:,m) = xyz_V
      a_FlagSurfaceSink(m) = .true.
    end if

!!$    call CloudMajorCompCalcFlowBK(                                   &
    call CloudMajorCompCalcFlow(                                   &
      & xyr_PressB, xyr_DPressDt,                                    & ! (in)
      & mmax, a_FlagSurfaceSink(1:mmax), xyza_Array(:,:,:,1:mmax),   & ! (in)
      & xyra_MassFlow(:,:,:,1:mmax)                                  & ! (out)
      & )

    ! unpacking
    do m = 1, ncmax
      xyrf_MassFlow(:,:,:,m) = xyra_MassFlow(:,:,:,m)
    end do
    m = ncmax
    if ( FlagModTemp ) then
      m = m + 1
      xyr_TempFlow = xyra_MassFlow(:,:,:,m)
    else
      xyr_TempFlow = 0.0_DP
    end if
    if ( FlagModMom ) then
      m = m + 1
      xyr_MomXFlow = xyra_MassFlow(:,:,:,m)
      m = m + 1
      xyr_MomYFlow = xyra_MassFlow(:,:,:,m)
    else
      xyr_MomXFlow = 0.0_DP
      xyr_MomYFlow = 0.0_DP
    end if


    ! Adjustment
    !   preparation
    do k = 1, kmax
      xyz_DelAtmMassB(:,:,k) = ( xyr_PressB(:,:,k-1) - xyr_PressB(:,:,k) ) / Grav
      xyz_DelAtmMassA(:,:,k) = ( xyr_PressA(:,:,k-1) - xyr_PressA(:,:,k) ) / Grav
    end do
    !   Atmospheric composition
    do n = 1, ncmax
      do k = 1, kmax
        xyzf_QMix(:,:,k,n) =                                              &
          &   (   xyz_DelAtmMassB(:,:,k) * xyzf_QMix(:,:,k,n)             &
          &     - ( xyrf_MassFlow(:,:,k,n) - xyrf_MassFlow(:,:,k-1,n) ) ) &
          & / xyz_DelAtmMassA(:,:,k)
      end do
    end do
    if ( FlagModTemp ) then
      do k = 1, kmax
        ! Temperature
        xyz_Temp(:,:,k) =                                           &
          &   (   xyz_DelAtmMassB(:,:,k) * xyz_Temp(:,:,k)          &
          &     - ( xyr_TempFlow(:,:,k) - xyr_TempFlow(:,:,k-1) ) ) &
          & / xyz_DelAtmMassA(:,:,k)
      end do
    end if
    if ( FlagModMom ) then
      do k = 1, kmax
        ! Zonal wind
        xyz_U(:,:,k) =                                              &
          &   (   xyz_DelAtmMassB(:,:,k) * xyz_U(:,:,k)             &
          &     - ( xyr_MomXFlow(:,:,k) - xyr_MomXFlow(:,:,k-1) ) ) &
          & / xyz_DelAtmMassA(:,:,k)
        ! Meridional wind
        xyz_V(:,:,k) =                                              &
          &   (   xyz_DelAtmMassB(:,:,k) * xyz_V(:,:,k)             &
          &     - ( xyr_MomYFlow(:,:,k) - xyr_MomYFlow(:,:,k-1) ) ) &
          & / xyz_DelAtmMassA(:,:,k)
      end do
    end if

    !   Surface major component ice
    xy_SurfMajCompIce = xy_SurfMajCompIce + xy_DSurfMajCompIceDt * ( 2.0_DP * DelTime )
    !   Surface pressure
    xy_Ps = xy_PsA


    call AuxVars( &
      & xy_PsA, xyz_TempTmp, xyz_QH2OVapTmp,    & ! (in )
      & xyr_Press = xyr_PressA                  & ! (out) optional
      & )
    ! 成分の質量の補正
    ! Fix masses of constituents
    !
    call MassFixerColumn( &
      & xyr_PressA, & ! (in)
      & xyzf_QMix   & ! (inout)
      & )


    ! Check
    call CloudMajorCompConsChk(       &
      & a_FlagSurfaceSink,            & ! (in)
      & xyz_DelAtmMassB, xyzf_QMixB,  & ! (in)
      & xyz_DelAtmMassA, xyzf_QMix    & ! (in)
      & )


    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )


  end subroutine MajorCompPhaseChangeOnGround

  !--------------------------------------------------------------------------------------

  subroutine CloudMajorCompCalcFlow(       &
    & xyr_Press, xyr_DPPress,              & ! (in)
    & mmax, a_FlagSurfaceSink, xyza_Array, & ! (in)
    & xyra_MassFlow                        & ! (out)
    & )
    !
    ! CO2 相変化
    !
    ! CO2 phase change
    !

    ! モジュール引用 ; USE statements
    !

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Grav                  ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration

    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in ):: xyr_Press    (0:imax-1, 1:jmax, 0:kmax)
                              ! pressure
    real(DP), intent(in ):: xyr_DPPress  (0:imax-1, 1:jmax, 0:kmax)
    integer , intent(in ):: mmax
    logical , intent(in ):: a_FlagSurfaceSink(1:mmax)
    real(DP), intent(in ):: xyza_Array   (0:imax-1, 1:jmax, 1:kmax, 1:mmax)
    real(DP), intent(out):: xyra_MassFlow(0:imax-1, 1:jmax, 0:kmax, 1:mmax)

    ! 作業変数
    ! Work variables
    !
    real(DP):: DelAtmMass
    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction
    integer:: k2              ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction
    integer:: m


    ! 実行文 ; Executable statement
    !

    ! 初期化
    ! Initialization
    !
    if ( .not. cloud_major_comp_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    if ( .not. FlagMajCompPhaseChange ) return


    ! check
    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          if ( xyr_DPPress(i,j,k-1) < xyr_DPPress(i,j,k) ) then
            call MessageNotify( 'E', module_name, 'Order of departure points are inappropriate, P(k=%d)=%f < P(k=%d)=%f.', &
              & i = (/ k-1, k /), d = (/ xyr_DPPress(i,j,k-1), xyr_DPPress(i,j,k) /) )
          end if
        end do
      end do
    end do

    xyra_MassFlow = 0.0_DP
    do k = 0, kmax-1
      do j = 1, jmax
        do i = 0, imax-1

!!$          if ( xyr_DPressDt(i,j,k) >= 0.0_DP ) then
          if ( xyr_DPPress(i,j,k) > xyr_Press(i,j,k) ) then

            sum_upward_mass_transport : do k2 = k, 1, -1
              if ( xyr_DPPress(i,j,k) > xyr_Press(i,j,k2-1) ) then
                DelAtmMass = ( xyr_Press(i,j,k2-1) - xyr_Press(i,j,k2) ) / Grav
                do m = 1, mmax
                  xyra_MassFlow(i,j,k,m) = xyra_MassFlow(i,j,k,m) &
                    & + xyza_Array(i,j,k2,m) * DelAtmMass
                end do
              else
                DelAtmMass = ( xyr_DPPress(i,j,k) - xyr_Press(i,j,k2) ) / Grav
                do m = 1, mmax
                  xyra_MassFlow(i,j,k,m) = xyra_MassFlow(i,j,k,m) &
                    & + xyza_Array(i,j,k2,m) * DelAtmMass
                end do
                exit sum_upward_mass_transport
              end if
            end do sum_upward_mass_transport

          else

            sum_downward_mass_transport : do k2 = k+1, kmax
              if ( xyr_DPPress(i,j,k) < xyr_Press(i,j,k2  ) ) then
                DelAtmMass = ( xyr_Press(i,j,k2-1) - xyr_Press(i,j,k2) ) / Grav
                do m = 1, mmax
                  xyra_MassFlow(i,j,k,m) = xyra_MassFlow(i,j,k,m) &
                    & - xyza_Array(i,j,k2,m) * DelAtmMass
                end do
              else
                DelAtmMass = ( xyr_Press(i,j,k2-1) - xyr_DPPress(i,j,k) ) / Grav
                do m = 1, mmax
                  xyra_MassFlow(i,j,k,m) = xyra_MassFlow(i,j,k,m) &
                    & - xyza_Array(i,j,k2,m) * DelAtmMass
                end do
                exit sum_downward_mass_transport
              end if
            end do sum_downward_mass_transport

          end if

        end do
      end do

    end do


    do k = 0, 0
      do j = 1, jmax
        do i = 0, imax-1

          ! not surface sink
!!!$          if ( xyr_DPressDt(i,j,k) <= 0.0_DP ) then
!!$          if ( xyr_DPPress(i,j,k) <= xyr_Press(i,j,k) ) then
            do m = 1, mmax
              if ( .not. a_FlagSurfaceSink(m) ) then
                xyra_MassFlow(i,j,k,m) = 0.0_DP
              end if
            end do
!!$          end if

        end do
      end do

    end do



  end subroutine CloudMajorCompCalcFlow

  !--------------------------------------------------------------------------------------

  subroutine CloudMajorCompConsChk(       &
    & a_FlagSurfaceSink,                  & ! (in)
    & xyz_DelAtmMassB, xyzf_QMixB,        & ! (in)
    & xyz_DelAtmMassA, xyzf_QMixA         & ! (in)
    & )

    ! 組成に関わる配列の設定
    ! Settings of array for atmospheric composition
    !
    use composition, only : ncmax

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Grav, &
                              ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration
      & CpDry
                              ! $ C_p $ [J kg-1 K-1].
                              ! 乾燥大気の定圧比熱.
                              ! Specific heat of air at constant pressure

    logical , intent(in) :: a_FlagSurfaceSink(1:ncmax)
    real(DP), intent(in) :: xyz_DelAtmMassB(0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in) :: xyzf_QMixB     (0:imax-1, 1:jmax, 1:kmax, ncmax)
    real(DP), intent(in) :: xyz_DelAtmMassA(0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in) :: xyzf_QMixA     (0:imax-1, 1:jmax, 1:kmax, ncmax)

    ! Local variables
    !
    real(DP) :: ValB
    real(DP) :: ValA
    real(DP) :: xyf_SumB(0:imax-1, 1:jmax, 1:ncmax)
    real(DP) :: xyf_SumA(0:imax-1, 1:jmax, 1:ncmax)
    real(DP) :: Ratio
    integer  :: i
    integer  :: j
    integer  :: k
    integer  :: n


    xyf_SumB = 0.0_DP
    xyf_SumA = 0.0_DP
    do n = 1, ncmax
      do k = kmax, 1, -1
        xyf_SumB(:,:,n) = xyf_SumB(:,:,n) &
          & + xyz_DelAtmMassB(:,:,k) * xyzf_QMixB(:,:,k,n)
        xyf_SumA(:,:,n) = xyf_SumA(:,:,n) &
          & + xyz_DelAtmMassA(:,:,k) * xyzf_QMixA(:,:,k,n)
      end do
    end do
    do n = 1, ncmax
      if ( .not. a_FlagSurfaceSink(n) ) then
        do j = 1, jmax
          do i = 0, imax-1
            ValB = xyf_SumB(i,j,n)
            ValA = xyf_SumA(i,j,n)

            Ratio = ( ValA - ValB ) / ( ValA + 1.0d-100 )
            if ( abs( Ratio ) > 1.0d-10 ) then
              if ( ( ValB < 0.0_DP ) .and. ( abs( ValB ) < 1.0e-20_DP ) ) then
                ! Do nothing
              else
                call MessageNotify( 'M', module_name, 'Mass No. %d is not conserved, %f.', i = (/ n /), d = (/ Ratio /) )
              end if
            end if
          end do
        end do
      end if
    end do


  end subroutine CloudMajorCompConsChk

  !--------------------------------------------------------------------------------------

  subroutine CloudMajorCompInit(                   &
    & ArgFlagMajCompPhaseChange, CondMajCompName   & ! (in)
    & )
    !
    ! major_comp_phase_change モジュールの初期化を行います. 
    ! NAMELIST#major_comp_phase_change_nml の読み込みはこの手続きで行われます. 
    !
    ! "major_comp_phase_change" module is initialized. 
    ! "NAMELIST#major_comp_phase_change_nml" is loaded in this procedure. 
    !

    ! モジュール引用 ; USE statements
    !

    ! OpenMP
    !
    !$ use omp_lib

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! 種別型パラメタ
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! 文字列操作
    ! Character handling
    !
    use dc_string, only: StoA

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & AxnameX, &
      & AxnameY, &
      & AxnameZ, &
      & AxnameR, &
      & AxnameT

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only : ConstantsInit

    ! 補助的な変数を計算するサブルーチン・関数群
    ! Subroutines and functions for calculating auxiliary variables
    !
    use auxiliary, only : AuxVarsInit

    ! 主成分相変化
    ! Phase change of atmospheric major component
    !
    use saturate_major_comp, only : &
      & SaturateMajorCompInit

    ! 重力沈降過程
    ! Gravitational sedimentation process
    !
    use grav_sed, only : GravSedInit

    ! 短波入射 (太陽入射)
    ! Short wave (insolation) incoming
    !
    use rad_short_income, only : RadShortIncomeInit


    use set_Mars_dust, only : SetMarsDustInit

    ! 質量の補正
    ! Mass fixer
    !
    use mass_fixer, only : MassFixerInit

    ! 雲関系ルーチン
    ! Cloud-related routines
    !
    use cloud_utils, only : CloudUtilsInit


    ! 宣言文 ; Declaration statements
    !
    implicit none

    logical     , intent(in) :: ArgFlagMajCompPhaseChange
    character(*), intent(in) :: CondMajCompName


    ! Local variables
    !
    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
                              ! IOSTAT of NAMELIST read

    integer:: iThread


    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /cloud_major_comp_nml/ &
      & FlagSedimentationInfty,   &
      & FlagModMom,               &
      & MajCompIceDen,            &
!!$      & CloudRad,                 &
      & CloudNumRatio,            &
      & CloudNuclRad,             &
      & CloudNuclDen,             &
      & TimeStepSmall


       ! デフォルト値については初期化手続 "cloud_major_comp#CloudMajorCompInit"
       ! のソースコードを参照のこと. 
       !
       ! Refer to source codes in the initialization procedure
       ! "cloud_major_comp#CloudMajorCompInit" for the default values. 
       !

    ! 実行文 ; Executable statement
    !

    if ( cloud_major_comp_inited ) return


    FlagMajCompPhaseChange = ArgFlagMajCompPhaseChange


    ! デフォルト値の設定
    ! Default values settings
    !
    FlagSedimentationInfty = .false.
    FlagModMom       = .false.
    MajCompIceDen    = 1.62e3     ! density at 150 K (Mangan et al., 2017)
         ! Mangan et al. (2017)
         ! https://www.sciencedirect.com/science/article/pii/S0019103516306832
!!$    CloudRad         =  2.0d-6
    CloudNumRatio    = -1.0d0
    CloudNuclRad     =  0.0d0
    CloudNuclDen     =  0.0e3_DP   ! dummy

    TimeStepSmall    = 1.0e10_DP


    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml,                 &  ! (in)
        & nml = cloud_major_comp_nml, &  ! (out)
        & iostat = iostat_nml )          ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
!      if ( iostat_nml == 0 ) write( STDOUT, nml = cumulus_adjust_nml )
    end if


    ! Preparation for OpenMP
    !
    NThreads = 1
    !$ NThreads  = omp_get_max_threads()
!    !$ write( 6, * ) "Number of processors : ", omp_get_num_procs()
!    !$ write( 6, * ) "Number of threads    : ", nthreads

    allocate( a_ls(0:NThreads-1) )
    allocate( a_le(0:NThreads-1) )

    do iThread = 0, NThreads-1
      if ( iThread == 0 ) then
        a_ls(iThread) = 1
      else
        a_ls(iThread) = a_le(iThread-1) + 1
      end if
      a_le(iThread) = a_ls(iThread) + imax*jmax / NThreads - 1
      if ( iThread + 1 <= mod( imax*jmax, NThreads ) ) then
        a_le(iThread) = a_le(iThread) + 1
      end if
    end do


    ! 物理定数設定
    ! Physical constants settings
    !
    call ConstantsInit

    if ( FlagMajCompPhaseChange ) then
      ! 主成分相変化
      ! Phase change of atmospheric major component
      !
      call SaturateMajorCompInit(  &
        & CondMajCompName          & ! (in)
        & )
    end if

    ! 補助的な変数を計算するサブルーチン・関数群
    ! Subroutines and functions for calculating auxiliary variables
    !
    call AuxVarsInit

    ! 質量の補正
    ! Mass fixer
    !
    call MassFixerInit

    ! 重力沈降過程
    ! Gravitational sedimentation process
    !
    call GravSedInit

    ! 短波入射 (太陽入射)
    ! Short wave (insolation) incoming
    !
    call RadShortIncomeInit

    call SetMarsDustInit

    ! 雲関系ルーチン
    ! Cloud-related routines
    !
    call CloudUtilsInit


    ! ヒストリデータ出力のためのへの変数登録
    ! Register of variables for history data output
    !
!!$    call HistoryAutoAddVariable( 'DSurfTempDtMajCompPhaseChange', &
!!$      & (/ AxnameX, AxnameY, AxnameT /),                             &
!!$      & 'heating by major component phase change', 'K s-1' )
    call HistoryAutoAddVariable( 'DTempDtMajCompPhaseChange',     &
      & (/ AxnameX, AxnameY, AxnameZ, AxnameT /),                 &
      & 'heating by major component phase change', 'K s-1' )
    call HistoryAutoAddVariable( 'DQMajCompIceDtMajCompPhaseChange', &
      & (/ AxnameX, AxnameY, AxnameZ, AxnameT /),                    &
      & 'tendency of mixing ratio by major component cloud', 'K s-1' )

    call HistoryAutoAddVariable( 'MajCompCloudRadiusForGravSed', &
      & (/ AxnameX, AxnameY, AxnameZ, AxnameT /), &
      & 'Major component cloud radius for gravitational sedimentation', &
      & 'm' )
    call HistoryAutoAddVariable( 'MajCompCloudRadius', &
      & (/ AxnameX, AxnameY, AxnameZ, AxnameT /), &
      & 'Major component cloud radius', &
      & 'm' )
    call HistoryAutoAddVariable( 'DSurfMajCompIceDtGravSed', &
      & (/ AxnameX, AxnameY, AxnameT /), &
      & 'Tendency of surface major component ice by sedimentation', &
      & 'kg m-2 s-1' )


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, ' FlagSedimentationInfty = %b', l = (/ FlagModMom /) )
    call MessageNotify( 'M', module_name, ' FlagModMom             = %b', l = (/ FlagModMom /) )
    call MessageNotify( 'M', module_name, ' MajCompIceDen          = %f', d = (/ MajCompIceDen /) )
!!$    call MessageNotify( 'M', module_name, ' CloudRad         = %f', d = (/ CloudRad /) )
    call MessageNotify( 'M', module_name, ' CloudNumRatio          = %f', d = (/ CloudNumRatio /) )
    call MessageNotify( 'M', module_name, ' CloudNuclRad           = %f', d = (/ CloudNuclRad /) )
    call MessageNotify( 'M', module_name, ' CloudNuclDen           = %f', d = (/ CloudNuclDen /) )
    call MessageNotify( 'M', module_name, ' TimeStepSmall          = %f', d = (/ TimeStepSmall /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    cloud_major_comp_inited = .true.

  end subroutine CloudMajorCompInit

  !-------------------------------------------------------------------

end module cloud_major_comp
