!= 初期値データ提供
!
!= Prepare initial data for tropical cyclone (Reed and Jablonowski, 2011)
!
! Authors::   Yoshiyuki O. TAKAHASHI
! Version::   $Id: initial_data.f90,v 1.16 2015/02/17 23:50:42 yot Exp $ 
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module initial_data_rj2011
  !
  != 初期値データ提供
  !
  != Prepare initial data
  !
  ! 初期値データのサンプルを提供します. 
  !
  ! 現在は以下のデータを提供します. 
  !
  ! ...
  !
  ! Prepare sample data of initial data (restart data)
  !
  ! Now, following data are provided. 
  !
  ! * Analytical initial condition proposed by Reed and Jablonowski (2011)
  !
  !== Procedures List
  !
  ! SetInitData   :: 初期値データ取得
  ! ------------  :: ------------
  ! SetInitData   :: Get initial data
  !
  !== NAMELIST
  !
  ! NAMELIST#initial_data_rw2011_nml
  !

  ! モジュール引用 ; USE statements
  !

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数. 
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数. 
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数. 
                             ! Number of vertical level

  ! 組成に関わる配列の設定
  ! Settings of array for atmospheric composition
  !
  use composition, only: &
    &                    ncmax, &
                              ! 成分の数
                              ! Number of composition
    &                    IndexH2OVap

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING     ! 文字列.       Strings. 

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP     ! 倍精度実数型. Double precision. 

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public :: InitialDataRJ2011SetData
  public :: InitialDataRJ2011Init


  ! 公開変数
  ! Public variables
  !
  real(DP), save :: LonCDeg
  real(DP), save :: LatCDeg

  real(DP), save :: Press0
  real(DP), save :: HeightTrop
  !
  real(DP), save :: DelPress
  real(DP), save :: RadiusPress
  real(DP), save :: HeightPress
  !
  real(DP), save :: QH2OVap0
  real(DP), save :: QH2OVapTrop
  real(DP), save :: HeightQ1
  real(DP), save :: HeightQ2
  !
  real(DP), save :: Temp0
  real(DP), save :: Gamma


  logical, save, public:: initial_data_rj2011_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag


  ! 非公開変数
  ! Private variables
  !

  character(*), parameter:: module_name = 'initial_data_rj2011'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: initial_data.f90,v 1.16 2015/02/17 23:50:42 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

contains

  !--------------------------------------------------------------------------------------

  subroutine BK_InitialDataRJ2011SetData(           &
    & xyz_U, xyz_V, xyz_Temp, xyz_QH2OVap, xy_Ps &   ! (out)
    & )
    !
    ! Reed and Jablonowski (2011) の初期値
    !
    ! initial condition by Reed and Jablonowski (2011)
    !

    ! モジュール引用 ; USE statements
    !

    ! MPI 関連ルーチン
    ! MPI related routines
    !
    use mpi_wrapper, only : NProcs

    ! 物理・数学定数設定
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI                    ! $ \pi $.
                              ! 円周率. Circular constant

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & RPlanet, &            ! $ a $ [m].
                              ! 惑星半径.
                              ! Radius of planet
      & Omega, &              ! $ \Omega $ [s-1].
                              ! 回転角速度.
                              ! Angular velocity
      & GasRDry, &            ! $ R $ [J kg-1 K-1]. 
                              ! 乾燥大気の気体定数. 
                              ! Gas constant of air
      & Grav, &               ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration
      & EpsV                  ! $ \epsilon_v $ .
                              ! 水蒸気分子量比.
                              ! Molecular weight of water vapor

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & x_Lon, &
                              ! $ \lambda $ [rad.] . 経度. Longitude
      & y_Lat, &
                              ! $ \varphi $ [rad.] . 緯度. Latitude
      & z_Sigma, &
                              ! $ \sigma $ レベル (整数). 
                              ! Full $ \sigma $ level
      & y_Lat_Weight
                              ! $ \Delta \varphi $ [rad.] .
                              ! 緯度座標重み.
                              ! Weight of latitude

    ! 宣言文 ; Declaration statements
    !

    real(DP), intent(out):: xyz_U  (0:imax-1, 1:jmax, 1:kmax)
                              ! $ u $ .   東西風速. Eastward wind
    real(DP), intent(out):: xyz_V  (0:imax-1, 1:jmax, 1:kmax)
                              ! $ v $ .   南北風速. Northward wind
    real(DP), intent(out):: xyz_Temp  (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ .   温度. Temperature
    real(DP), intent(out):: xyz_QH2OVap(0:imax-1, 1:jmax, 1:kmax)
                              ! $ q $ .   比湿. Specific humidity
    real(DP), intent(out):: xy_Ps (0:imax-1, 1:jmax)
                              ! $ p_s $ . 地表面気圧. Surface pressure


    ! 作業変数
    ! Work variables
    !

    ! パラメタ設定
    ! 
    !
    !
    real(DP) :: LonC
    real(DP) :: LatC

    real(DP) :: xy_Dist       (0:imax-1, 1:jmax)
    real(DP) :: xyz_Press     (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_Height    (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_VirTemp   (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_VirTempDev(0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_TempDev   (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_TangSpeed (0:imax-1, 1:jmax, 1:kmax)


    real(DP) :: VirTemp0
    real(DP) :: VirTempTrop
    real(DP) :: PressTrop

    real(DP) :: CorParamC
    real(DP) :: PressDev
    real(DP) :: DPressDR
    real(DP) :: TmpVal

    real(DP) :: d1
    real(DP) :: d2
    real(DP) :: d

    real(DP) :: Epsilon = 1.0e-25_DP

    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction


    ! 実行文 ; Executable statement

    if ( .not. initial_data_rj2011_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    LonC        = LonCDeg * PI / 180.0_DP
    LatC        = LatCDeg * PI / 180.0_DP

    CorParamC   = 2.0_DP * Omega * sin( LatC )

    VirTemp0    = Temp0 * ( 1.0_DP + ( 1.0_DP / EpsV - 1.0_DP ) * QH2OVap0 )
    VirTempTrop = VirTemp0 - Gamma * HeightTrop
    PressTrop   = Press0 * ( VirTempTrop / VirTemp0 )**( Grav/(GasRDry*Gamma) )


    ! 水平距離の計算
    ! Calculate horizontal distance
    !
    do j = 1, jmax
      do i = 0, imax-1
        TmpVal = &
          &   sin( LatC ) * sin( y_Lat(j) ) &
          & + cos( LatC ) * cos( y_Lat(j) ) &
          &   * cos( x_Lon(i) - LonC ) 
        xy_Dist(i,j) = RPlanet * acos( TmpVal )
      end do
    end do

    ! 圧力の計算
    ! Calculate pressure
    !
    xy_Ps = Press0 - DelPress * exp( - ( xy_Dist / RadiusPress )**(3.0d0/2.0d0) )
    !
    do k = 1, kmax
      xyz_Press(:,:,k) = xy_Ps * z_Sigma(k)
    end do


    ! 高度の計算
    ! Calculate height
    !
    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          if ( xyz_Press(i,j,k) >= PressTrop ) then
            xyz_Height(i,j,k) = &
              &   VirTemp0 / Gamma &
              & * ( 1.0_DP - ( xyz_Press(i,j,k) / xy_Ps(i,j) )**(GasRDry*Gamma/Grav) )
          else
            xyz_Height(i,j,k) = &
              & HeightTrop &
              & + GasRDry * VirTempTrop / Grav &
              &   * log( PressTrop / xyz_Press(i,j,k) )
          end if
        end do
      end do
    end do


    ! specific humidity
    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          if ( xyz_Height(i,j,k) <= HeightTrop ) then
            xyz_QH2OVap(i,j,k) = QH2OVap0 * exp( - xyz_Height(i,j,k) / HeightQ1 ) * exp( - ( xyz_Height(i,j,k) / HeightQ2 )**2 )
          else
            xyz_QH2OVap(i,j,k) = QH2OVapTrop
          end if
        end do
      end do
    end do
    ! temperature
    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          if ( xyz_Height(i,j,k) <= HeightTrop ) then
            xyz_VirTemp(i,j,k) = VirTemp0 - Gamma * xyz_Height(i,j,k)
          else
            xyz_VirTemp(i,j,k) = VirTempTrop
          end if
        end do
      end do
    end do
    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          if ( xyz_Height(i,j,k) <= HeightTrop ) then
            xyz_VirTempDev(i,j,k) = &
              &   ( VirTemp0 - Gamma * xyz_Height(i,j,k) ) &
              & * ( &
              &     - 2.0_DP * GasRDry &
              &       * ( VirTemp0 - Gamma * xyz_Height(i,j,k) ) * xyz_Height(i,j,k) &
              &   ) &
              & / ( &
              &       2.0_DP * GasRDry * ( VirTemp0 - Gamma * xyz_Height(i,j,k) ) * xyz_Height(i,j,k) &
              &     + Grav * HeightPress**2 &
              &       * ( &
              &             1.0_DP &
              &           - Press0 / DelPress &
              &             * exp( ( xy_Dist(i,j) / RadiusPress )**(3.0_DP/2.0_DP) ) &
              &             * exp( ( xyz_Height(i,j,k) / HeightPress )**2 ) &
              &         ) &
              &   )
          else
            xyz_VirTempDev(i,j,k) = 0.0_DP
          end if
        end do
      end do
    end do
    xyz_VirTemp = xyz_VirTemp + xyz_VirTempDev
    !
    xyz_Temp    = xyz_VirTemp    / ( 1.0_DP + ( 1.0_DP / EpsV - 1.0_DP ) * xyz_QH2OVap )
    ! This is not used.
    xyz_TempDev = xyz_VirTempDev / ( 1.0_DP + ( 1.0_DP / EpsV - 1.0_DP ) * xyz_QH2OVap )


    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          if ( xyz_Height(i,j,k) <= HeightTrop ) then
            PressDev = &
              & - DelPress &
              &   * exp( - ( xy_Dist(i,j) / RadiusPress )**(3.0_DP/2.0_DP) ) &
              &   * exp( - ( xyz_Height(i,j,k) / HeightPress )**2 ) &
              &   * ( ( VirTemp0 - Gamma * xyz_Height(i,j,k) ) / VirTemp0 )**( Grav/(GasRDry*Gamma) )
            DPressDR = &
              &   PressDev &
              &   * ( - 3.0_DP / 2.0_DP ) &
              &   * ( xy_Dist(i,j) / RadiusPress )**(1.0_DP/2.0_DP) / RadiusPress
            xyz_TangSpeed(i,j,k) = &
              & - CorParamC * xy_Dist(i,j) / 2.0_DP &
              & + sqrt( &
              &           CorParamC**2 * xy_Dist(i,j)**2 / 4.0_DP &
              &         + GasRDry * xyz_VirTemp(i,j,k) * xy_Dist(i,j) / xyz_Press(i,j,k) &
              &           * DPressDR &
              &       )
          else
            xyz_TangSpeed(i,j,k) = 0.0_DP
          end if
        end do
      end do
    end do
    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          d1 =   sin( LatC ) * cos( y_Lat(j) ) &
            &  - cos( LatC ) * sin( y_Lat(j) ) &
            &    * cos( x_Lon(i) - LonC )
          d2 =   cos( LatC ) &
            &    * sin( x_Lon(i) - LonC )
          d  = max( Epsilon, sqrt( d1**2 + d2**2 ) )
          xyz_U(i,j,k) = xyz_TangSpeed(i,j,k) * d1 / d
          xyz_V(i,j,k) = xyz_TangSpeed(i,j,k) * d2 / d
        end do
      end do
    end do


  end subroutine BK_InitialDataRJ2011SetData

  !--------------------------------------------------------------------------------------

  subroutine InitialDataRJ2011SetData(           &
    & xyz_U, xyz_V, xyz_Temp, xyz_QH2OVap, xy_Ps &   ! (out)
    & )
    !
    ! Reed and Jablonowski (2011) の初期値
    !
    ! initial condition by Reed and Jablonowski (2011)
    !

    ! モジュール引用 ; USE statements
    !

    ! MPI 関連ルーチン
    ! MPI related routines
    !
    use mpi_wrapper, only : NProcs

    ! 物理・数学定数設定
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI                    ! $ \pi $.
                              ! 円周率. Circular constant

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & RPlanet, &            ! $ a $ [m].
                              ! 惑星半径.
                              ! Radius of planet
      & Omega, &              ! $ \Omega $ [s-1].
                              ! 回転角速度.
                              ! Angular velocity
      & GasRDry, &            ! $ R $ [J kg-1 K-1]. 
                              ! 乾燥大気の気体定数. 
                              ! Gas constant of air
      & Grav, &               ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration
      & EpsV                  ! $ \epsilon_v $ .
                              ! 水蒸気分子量比.
                              ! Molecular weight of water vapor

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & x_Lon, &
                              ! $ \lambda $ [rad.] . 経度. Longitude
      & y_Lat, &
                              ! $ \varphi $ [rad.] . 緯度. Latitude
      & z_Sigma, &
                              ! $ \sigma $ レベル (整数). 
                              ! Full $ \sigma $ level
      & y_Lat_Weight
                              ! $ \Delta \varphi $ [rad.] .
                              ! 緯度座標重み.
                              ! Weight of latitude

    ! 宣言文 ; Declaration statements
    !

    implicit none

    real(DP), intent(out):: xyz_U  (0:imax-1, 1:jmax, 1:kmax)
                              ! $ u $ .   東西風速. Eastward wind
    real(DP), intent(out):: xyz_V  (0:imax-1, 1:jmax, 1:kmax)
                              ! $ v $ .   南北風速. Northward wind
    real(DP), intent(out):: xyz_Temp  (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ .   温度. Temperature
    real(DP), intent(out):: xyz_QH2OVap(0:imax-1, 1:jmax, 1:kmax)
                              ! $ q $ .   比湿. Specific humidity
    real(DP), intent(out):: xy_Ps (0:imax-1, 1:jmax)
                              ! $ p_s $ . 地表面気圧. Surface pressure


    ! 作業変数
    ! Work variables
    !

    ! パラメタ設定
    ! 
    !
    !
    real(DP) :: LonC
    real(DP) :: LatC

    real(DP) :: xy_Dist       (0:imax-1, 1:jmax)

    real(DP) :: xyz_PressModel(0:imax-1, 1:jmax, 1:kmax)

    real(DP) :: xyz_Height    (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_VirTemp   (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_TangSpeed (0:imax-1, 1:jmax, 1:kmax)


    real(DP) :: VirTemp0
    real(DP) :: VirTempTrop
    real(DP) :: PressTrop

    real(DP) :: CorParamC
    real(DP) :: xyz_Press   (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_PressDev(0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DPressDZ(0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: DPressDR

    real(DP) :: TmpVal

    integer  :: iItr
    real(DP) :: xyz_HeightA (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: AbsDiff

    real(DP) :: d1
    real(DP) :: d2
    real(DP) :: d

    real(DP) :: Epsilon       = 1.0e-25_DP
    ! In Reed and Jablonowski (2011), a value below is 2e-13.
    ! But, the value is increased by about a factor of 10, because 2e-13
    ! is not achieved by our model. 
    real(DP) :: ConvCriterion = 2.0e-12_DP

    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction


    ! 実行文 ; Executable statement

    if ( .not. initial_data_rj2011_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    LonC        = LonCDeg * PI / 180.0_DP
    LatC        = LatCDeg * PI / 180.0_DP

    CorParamC   = 2.0_DP * Omega * sin( LatC )

    VirTemp0    = Temp0 * ( 1.0_DP + ( 1.0_DP / EpsV - 1.0_DP ) * QH2OVap0 )
    VirTempTrop = VirTemp0 - Gamma * HeightTrop
    PressTrop   = Press0 * ( VirTempTrop / VirTemp0 )**( Grav/(GasRDry*Gamma) )


    ! 水平距離の計算
    ! Calculate horizontal distance
    !
    do j = 1, jmax
      do i = 0, imax-1
        TmpVal = &
          &   sin( LatC ) * sin( y_Lat(j) ) &
          & + cos( LatC ) * cos( y_Lat(j) ) &
          &   * cos( x_Lon(i) - LonC ) 
        xy_Dist(i,j) = RPlanet * acos( TmpVal )
      end do
    end do


    ! 圧力の計算
    ! Calculate pressure
    !
    xy_Ps = Press0 - DelPress * exp( - ( xy_Dist / RadiusPress )**(3.0d0/2.0d0) )
    !
    do k = 1, kmax
      xyz_PressModel(:,:,k) = xy_Ps * z_Sigma(k)
    end do


    ! 高度の計算
    ! Calculate height
    !
    !   Initial value
    xyz_Press = xyz_PressModel
    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          if ( xyz_Press(i,j,k) >= PressTrop ) then
            xyz_Height(i,j,k) = &
              &   VirTemp0 / Gamma &
              & * ( 1.0_DP - ( xyz_Press(i,j,k) / xy_Ps(i,j) )**(GasRDry*Gamma/Grav) )
          else
            xyz_Height(i,j,k) = &
              & HeightTrop &
              & + GasRDry * VirTempTrop / Grav &
              &   * log( PressTrop / xyz_Press(i,j,k) )
          end if
        end do
      end do
    end do
    !
    iItr = 1
    do
      call InitialDataRJ2011CalcPress(              &
        & Press0, PressTrop,                        & ! (in)
        & Gamma, VirTemp0, VirTempTrop, HeightTrop, & ! (in)
        & DelPress, HeightPress, RadiusPress,       & ! (in)
        & xy_Dist, xyz_Height,                      & ! (in)
        & xyz_Press, xyz_PressDev                   & ! (out)
        & )
      call InitialDataRJ2011CalcDPressDZ(                        &
        & HeightTrop, Gamma, VirTemp0, VirTempTrop, HeightPress, & ! (in)
        & xyz_Height, xyz_Press, xyz_PressDev,                   & ! (in)
        & xyz_DPressDZ                                           & ! (out)
        & )
      xyz_HeightA = xyz_Height + ( xyz_PressModel - xyz_Press ) / xyz_DPressDZ

      AbsDiff = 0.0_DP
      do k = 1, kmax
        do j = 1, jmax
          do i = 0, imax-1
            AbsDiff = max( AbsDiff, abs( xyz_HeightA(i,j,k) - xyz_Height(i,j,k) ) / abs( xyz_HeightA(i,j,k) ) )
          end do
        end do
      end do
      if ( AbsDiff < ConvCriterion ) exit

      xyz_Height = xyz_HeightA

      iItr = iItr + 1
      if ( iItr > 1000 ) then
        call MessageNotify( 'E', module_name, &
          & 'Iteration loop is not convergent, iItr = %d.', i = (/ iItr /) )
      end if
    end do

    xyz_Height = xyz_HeightA


    ! If xyz_Height is determined, other fields are calculated below. 

    ! specific humidity
    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          if ( xyz_Height(i,j,k) <= HeightTrop ) then
            xyz_QH2OVap(i,j,k) = QH2OVap0 * exp( - xyz_Height(i,j,k) / HeightQ1 ) * exp( - ( xyz_Height(i,j,k) / HeightQ2 )**2 )
          else
            xyz_QH2OVap(i,j,k) = QH2OVapTrop
          end if
        end do
      end do
    end do
    ! temperature
    call InitialDataRJ2011CalcPress(              &
      & Press0, PressTrop,                        & ! (in)
      & Gamma, VirTemp0, VirTempTrop, HeightTrop, & ! (in)
      & DelPress, HeightPress, RadiusPress,       & ! (in)
      & xy_Dist, xyz_Height,                      & ! (in)
      & xyz_Press, xyz_PressDev                   & ! (out)
      & )
    call InitialDataRJ2011CalcDPressDZ(                        &
      & HeightTrop, Gamma, VirTemp0, VirTempTrop, HeightPress, & ! (in)
      & xyz_Height, xyz_Press, xyz_PressDev,                   & ! (in)
      & xyz_DPressDZ                                           & ! (out)
      & )
    xyz_VirTemp = - Grav * xyz_Press / ( GasRDry * xyz_DPressDZ )
    xyz_Temp    = &
      & xyz_VirTemp / ( 1.0_DP + ( 1.0_DP / EpsV - 1.0_DP ) * xyz_QH2OVap )


    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          if ( xyz_Height(i,j,k) <= HeightTrop ) then
            DPressDR = &
              &   xyz_PressDev(i,j,k) &
              &   * ( - 3.0_DP / 2.0_DP ) &
              &   * ( xy_Dist(i,j) / RadiusPress )**(1.0_DP/2.0_DP) / RadiusPress
            xyz_TangSpeed(i,j,k) = &
              & - CorParamC * xy_Dist(i,j) / 2.0_DP &
              & + sqrt( &
              &           CorParamC**2 * xy_Dist(i,j)**2 / 4.0_DP &
              &         + GasRDry * xyz_VirTemp(i,j,k) * xy_Dist(i,j) &
              &           / xyz_Press(i,j,k) &
              &           * DPressDR &
              &       )
          else
            xyz_TangSpeed(i,j,k) = 0.0_DP
          end if
        end do
      end do
    end do
    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          d1 =   sin( LatC ) * cos( y_Lat(j) ) &
            &  - cos( LatC ) * sin( y_Lat(j) ) &
            &    * cos( x_Lon(i) - LonC )
          d2 =   cos( LatC ) &
            &    * sin( x_Lon(i) - LonC )
          d  = max( Epsilon, sqrt( d1**2 + d2**2 ) )
          xyz_U(i,j,k) = xyz_TangSpeed(i,j,k) * d1 / d
          xyz_V(i,j,k) = xyz_TangSpeed(i,j,k) * d2 / d
        end do
      end do
    end do


  end subroutine InitialDataRJ2011SetData

  !--------------------------------------------------------------------------------------

  subroutine InitialDataRJ2011CalcPress(        &
    & Press0, PressTrop,                        & ! (in)
    & Gamma, VirTemp0, VirTempTrop, HeightTrop, & ! (in)
    & DelPress, HeightPress, RadiusPress,       & ! (in)
    & xy_Dist, xyz_Height,                      & ! (in)
    & xyz_Press, xyz_PressDev                   & ! (out)
    & )

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & GasRDry, &            ! $ R $ [J kg-1 K-1]. 
                              ! 乾燥大気の気体定数. 
                              ! Gas constant of air
      & Grav                  ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration

    real(DP), intent(in ) :: Press0
    real(DP), intent(in ) :: PressTrop
    real(DP), intent(in ) :: Gamma
    real(DP), intent(in ) :: VirTemp0
    real(DP), intent(in ) :: VirTempTrop
    real(DP), intent(in ) :: HeightTrop
    real(DP), intent(in ) :: DelPress
    real(DP), intent(in ) :: HeightPress
    real(DP), intent(in ) :: RadiusPress
    real(DP), intent(in ) :: xy_Dist     (0:imax-1, 1:jmax)
    real(DP), intent(in ) :: xyz_Height  (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(out) :: xyz_Press   (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(out) :: xyz_PressDev(0:imax-1, 1:jmax, 1:kmax)


    ! 作業変数
    ! Work variables
    !
    real(DP) :: PressBar
    real(DP) :: PressDev

    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction

    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          if ( xyz_Height(i,j,k) <= HeightTrop ) then
            PressBar = &
              & Press0 &
              & * ( ( VirTemp0 - Gamma * xyz_Height(i,j,k) ) / VirTemp0 )**( Grav/(GasRDry*Gamma) )
            PressDev = &
              & - DelPress &
              &   * exp( - ( xy_Dist(i,j) / RadiusPress )**(3.0_DP/2.0_DP) ) &
              &   * exp( - ( xyz_Height(i,j,k) / HeightPress )**2 ) &
              &   * ( ( VirTemp0 - Gamma * xyz_Height(i,j,k) ) / VirTemp0 )**( Grav/(GasRDry*Gamma) )
          else
            PressBar = &
              &   PressTrop &
              & * exp( - Grav * ( xyz_Height(i,j,k) - HeightTrop ) / ( GasRDry * VirTempTrop ) )
            PressDev = 0.0_DP
          end if
          xyz_Press   (i,j,k) = PressBar + PressDev
          xyz_PressDev(i,j,k) =            PressDev
        end do
      end do
    end do


  end subroutine InitialDataRJ2011CalcPress

  !--------------------------------------------------------------------------------------

  subroutine InitialDataRJ2011CalcDPressDZ(                  &
    & HeightTrop, Gamma, VirTemp0, VirTempTrop, HeightPress, & ! (in)
    & xyz_Height, xyz_Press, xyz_PressDev,                   & ! (in)
    & xyz_DPressDZ                                           & ! (out)
    & )

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & GasRDry, &            ! $ R $ [J kg-1 K-1]. 
                              ! 乾燥大気の気体定数. 
                              ! Gas constant of air
      & Grav                  ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration

    real(DP), intent(in ) :: HeightTrop
    real(DP), intent(in ) :: Gamma
    real(DP), intent(in ) :: VirTemp0
    real(DP), intent(in ) :: VirTempTrop
    real(DP), intent(in ) :: HeightPress
    real(DP), intent(in ) :: xyz_Height  (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in ) :: xyz_Press   (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in ) :: xyz_PressDev(0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(out) :: xyz_DPressDZ(0:imax-1, 1:jmax, 1:kmax)


    ! 作業変数
    ! Work variables
    !
    real(DP) :: xyz_VirTempBar(0:imax-1, 1:jmax, 1:kmax)

    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction


    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          if ( xyz_Height(i,j,k) <= HeightTrop ) then
            xyz_VirTempBar(i,j,k) = VirTemp0 - Gamma * xyz_Height(i,j,k)
          else
            xyz_VirTempBar(i,j,k) = VirTempTrop
          end if
        end do
      end do
    end do

    xyz_DPressDZ =                                        &
      & - Grav / ( GasRDry * xyz_VirTempBar ) * xyz_Press &
      & - 2.0_DP * xyz_Height / HeightPress**2 * xyz_PressDev


  end subroutine InitialDataRJ2011CalcDPressDZ

  !--------------------------------------------------------------------------------------

  subroutine InitialDataRJ2011SetTQ(             &
    & LonC, LatC,                                & ! (in)
    & Press0, HeightTrop,                        & ! (in)
    & QH2OVap0, QH2OVapTrop, HeightQ1, HeightQ2, & ! (in)
    & Temp0, Gamma,                              & ! (in)
    & DelPress, RadiusPress, HeightPress,        & ! (in)
    & xy_Dist, xyz_Height,                       & ! (in)
    & xyz_U, xyz_V, xyz_Temp, xyz_QH2OVap, xy_Ps & ! (out)
    & )
    !
    ! Reed and Jablonowski (2011) の初期値
    !
    ! initial condition by Reed and Jablonowski (2011)
    !

    ! モジュール引用 ; USE statements
    !

    ! MPI 関連ルーチン
    ! MPI related routines
    !
    use mpi_wrapper, only : NProcs

    ! 物理・数学定数設定
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI                    ! $ \pi $.
                              ! 円周率. Circular constant

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & RPlanet, &            ! $ a $ [m].
                              ! 惑星半径.
                              ! Radius of planet
      & Omega, &              ! $ \Omega $ [s-1].
                              ! 回転角速度.
                              ! Angular velocity
      & GasRDry, &            ! $ R $ [J kg-1 K-1]. 
                              ! 乾燥大気の気体定数. 
                              ! Gas constant of air
      & Grav, &               ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration
      & EpsV                  ! $ \epsilon_v $ .
                              ! 水蒸気分子量比.
                              ! Molecular weight of water vapor

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & x_Lon, &
                              ! $ \lambda $ [rad.] . 経度. Longitude
      & y_Lat, &
                              ! $ \varphi $ [rad.] . 緯度. Latitude
      & z_Sigma
                              ! $ \sigma $ レベル (整数). 
                              ! Full $ \sigma $ level

    ! 宣言文 ; Declaration statements
    !

    implicit none

    real(DP), intent(in ) :: LonC
    real(DP), intent(in ) :: LatC
    real(DP), intent(in ) :: Press0
    real(DP), intent(in ) :: HeightTrop
    real(DP), intent(in ) :: QH2OVap0
    real(DP), intent(in ) :: QH2OVapTrop
    real(DP), intent(in ) :: HeightQ1
    real(DP), intent(in ) :: HeightQ2
    real(DP), intent(in ) :: Temp0
    real(DP), intent(in ) :: Gamma
    real(DP), intent(in ) :: DelPress
    real(DP), intent(in ) :: RadiusPress
    real(DP), intent(in ) :: HeightPress
    real(DP), intent(in ) :: xy_Dist    (0:imax-1, 1:jmax)
    real(DP), intent(in ) :: xyz_Height (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(out) :: xyz_U      (0:imax-1, 1:jmax, 1:kmax)
                              ! $ u $ .   東西風速. Eastward wind
    real(DP), intent(out) :: xyz_V      (0:imax-1, 1:jmax, 1:kmax)
                              ! $ v $ .   南北風速. Northward wind
    real(DP), intent(out) :: xyz_Temp   (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ .   温度. Temperature
    real(DP), intent(out) :: xyz_QH2OVap(0:imax-1, 1:jmax, 1:kmax)
                              ! $ q $ .   比湿. Specific humidity
    real(DP), intent(out) :: xy_Ps      (0:imax-1, 1:jmax)
                              ! $ p_s $ . 地表面気圧. Surface pressure


    ! 作業変数
    ! Work variables
    !

    ! パラメタ設定
    ! 
    !
    !
    real(DP) :: VirTemp0
    real(DP) :: VirTempTrop
    real(DP) :: PressTrop

    real(DP) :: xyz_QH2OVapBar(0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_VirTempBar(0:imax-1, 1:jmax, 1:kmax)

    real(DP) :: xyz_Press     (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_VirTemp   (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_VirTempDev(0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_TempDev   (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_TangSpeed (0:imax-1, 1:jmax, 1:kmax)

    real(DP) :: CorParamC
    real(DP) :: Press
    real(DP) :: PressBar
    real(DP) :: PressDev
    real(DP) :: DPressDR
    real(DP) :: TmpVal

    real(DP) :: d1
    real(DP) :: d2
    real(DP) :: d

    real(DP) :: Epsilon = 1.0e-25_DP

    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction


    ! 実行文 ; Executable statement

    if ( .not. initial_data_rj2011_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    CorParamC = 2.0_DP * Omega * sin( LatC )

    VirTemp0    = Temp0 * ( 1.0_DP + ( 1.0_DP / EpsV - 1.0_DP ) * QH2OVap0 )
    VirTempTrop = VirTemp0 - Gamma * HeightTrop
    PressTrop   = Press0 * ( VirTempTrop / VirTemp0 )**( Grav/(GasRDry*Gamma) )

    ! specific humidity
    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          if ( xyz_Height(i,j,k) <= HeightTrop ) then
            xyz_QH2OVapBar(i,j,k) = QH2OVap0 * exp( - xyz_Height(i,j,k) / HeightQ1 ) * exp( - ( xyz_Height(i,j,k) / HeightQ2 )**2 )
          else
            xyz_QH2OVapBar(i,j,k) = QH2OVapTrop
          end if
        end do
      end do
    end do
    ! virtual temperature
    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          if ( xyz_Height(i,j,k) <= HeightTrop ) then
            xyz_VirTempBar(i,j,k) = VirTemp0 - Gamma * xyz_Height(i,j,k)
          else
            xyz_VirTempBar(i,j,k) = VirTempTrop
          end if
        end do
      end do
    end do


    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          if ( xyz_Height(i,j,k) <= HeightTrop ) then
            xyz_VirTempDev(i,j,k) = &
              &   ( VirTemp0 - Gamma * xyz_Height(i,j,k) ) &
              & * ( &
              &     - 2.0_DP * GasRDry &
              &       * ( VirTemp0 - Gamma * xyz_Height(i,j,k) ) * xyz_Height(i,j,k) &
              &   ) &
              & / ( &
              &       2.0_DP * GasRDry * ( VirTemp0 - Gamma * xyz_Height(i,j,k) ) * xyz_Height(i,j,k) &
              &     + Grav * HeightPress**2 &
              &       * ( &
              &             1.0_DP &
              &           - Press0 / DelPress &
              &             * exp( ( xy_Dist(i,j) / RadiusPress )**(3.0_DP/2.0_DP) ) &
              &             * exp( ( xyz_Height(i,j,k) / HeightPress )**2 ) &
              &         ) &
              &   )
          else
            xyz_VirTempDev(i,j,k) = 0.0_DP
          end if
        end do
      end do
    end do
    ! This is not used.
    xyz_TempDev = xyz_VirTempDev / ( 1.0_DP + ( 1.0_DP / EpsV - 1.0_DP ) * xyz_QH2OVap )

    xyz_QH2OVap = xyz_QH2OVapBar
    !
    xyz_VirTemp = xyz_VirTempBar + xyz_VirTempDev
    !
    xyz_Temp    = xyz_VirTemp / ( 1.0_DP + ( 1.0_DP / EpsV - 1.0_DP ) * xyz_QH2OVap )


    ! tangential velocity
    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1

          if ( xyz_Height(i,j,k) <= HeightTrop ) then
            PressBar = &
              & Press0 &
              & * ( ( VirTemp0 - Gamma * xyz_Height(i,j,k) ) / VirTemp0 )**( Grav/(GasRDry*Gamma) )
            PressDev = &
              & - DelPress &
              &   * exp( - ( xy_Dist(i,j) / RadiusPress )**(3.0_DP/2.0_DP) ) &
              &   * exp( - ( xyz_Height(i,j,k) / HeightPress )**2 ) &
              &   * ( ( VirTemp0 - Gamma * xyz_Height(i,j,k) ) / VirTemp0 )**( Grav/(GasRDry*Gamma) )
            DPressDR = &
              &   PressDev &
              &   * ( - 3.0_DP / 2.0_DP ) &
              &   * ( xy_Dist(i,j) / RadiusPress )**(1.0_DP/2.0_DP) / RadiusPress
            Press = PressBar + PressDev
            xyz_TangSpeed(i,j,k) = &
              & - CorParamC * xy_Dist(i,j) / 2.0_DP &
              & + sqrt( &
              &           CorParamC**2 * xy_Dist(i,j)**2 / 4.0_DP &
              &         + GasRDry * xyz_VirTemp(i,j,k) * xy_Dist(i,j) / Press &
              &           * DPressDR &
              &       )
          else
            xyz_TangSpeed(i,j,k) = 0.0_DP
          end if
        end do
      end do
    end do
    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          d1 =   sin( LatC ) * cos( y_Lat(j) ) &
            &  - cos( LatC ) * sin( y_Lat(j) ) &
            &    * cos( x_Lon(i) - LonC )
          d2 =   cos( LatC ) &
            &    * sin( x_Lon(i) - LonC )
          d  = max( Epsilon, sqrt( d1**2 + d2**2 ) )
          xyz_U(i,j,k) = xyz_TangSpeed(i,j,k) * d1 / d
          xyz_V(i,j,k) = xyz_TangSpeed(i,j,k) * d2 / d
        end do
      end do
    end do


  end subroutine InitialDataRJ2011SetTQ

  !--------------------------------------------------------------------------------------

  subroutine InitialDataRJ2011Init

    ! モジュール引用 ; USE statements
    !

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! 宣言文 ; Declaration statements
    !
    implicit none

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /initial_data_rj2011_nml/ &
      & LonCDeg, LatCDeg, &
      !
      & Press0, HeightTrop, &
      !
      & DelPress, RadiusPress, HeightPress, &
      !
      & QH2OVap0, QH2OVapTrop, HeightQ1, HeightQ2, &
      !
      & Temp0, Gamma
          !
          ! デフォルト値については初期化手続 "initial_data_rj2011#InitDataInit"
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "initial_data_rj2011#InitDataInit" for the default values. 
          !


    ! 実行文 ; Executable statement

    if ( initial_data_rj2011_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !
    LonCDeg     = 180.0_DP
    LatCDeg     =  10.0_DP
    !
    Press0      = 1015.00e2_DP ! p_0
    HeightTrop  = 15000_DP
    !
    DelPress    =   11.15e2_DP ! \Delta p
    RadiusPress = 282.0e3_DP   ! r_p
    HeightPress = 7000.0_DP    ! z_p
    !
    QH2OVap0    = 21.0e-3_DP   ! q_0
    QH2OVapTrop = 1.0e-8_DP    ! q_t
    HeightQ1    = 3000_DP      ! z_{q1}
    HeightQ2    = 8000_DP      ! z_{q2}
    !
    Temp0       = 302.15_DP    ! T_0
    Gamma       = 0.007_DP     ! \Gamma, virtual temperature lapse rate



    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml,                    & ! (in)
        & nml = initial_data_rj2011_nml, & ! (out)
        & iostat = iostat_nml )            ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
    end if


    ! Initialization of modules used in this module
    !


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----'       )
    call MessageNotify( 'M', module_name, '  LonCDeg     = %f', d = (/ LonCDeg     /) )
    call MessageNotify( 'M', module_name, '  LatCDeg     = %f', d = (/ LatCDeg     /) )
    call MessageNotify( 'M', module_name, '  Press0      = %f', d = (/ Press0      /) )
    call MessageNotify( 'M', module_name, '  HeightTrop  = %f', d = (/ HeightTrop  /) )
    call MessageNotify( 'M', module_name, '  DelPress    = %f', d = (/ DelPress    /) )
    call MessageNotify( 'M', module_name, '  RadiusPress = %f', d = (/ RadiusPress /) )
    call MessageNotify( 'M', module_name, '  HeightPress = %f', d = (/ HeightPress /) )
    call MessageNotify( 'M', module_name, '  QH2OVap0    = %f', d = (/ QH2OVap0    /) )
    call MessageNotify( 'M', module_name, '  QH2OVapTrop = %f', d = (/ QH2OVapTrop /) )
    call MessageNotify( 'M', module_name, '  HeightQ1    = %f', d = (/ HeightQ1    /) )
    call MessageNotify( 'M', module_name, '  HeightQ2    = %f', d = (/ HeightQ2    /) )
    call MessageNotify( 'M', module_name, '  Temp0       = %f', d = (/ Temp0       /) )
    call MessageNotify( 'M', module_name, '  Gamma       = %f', d = (/ Gamma       /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version)       )

    initial_data_rj2011_inited = .true.

  end subroutine InitialDataRJ2011Init

  !--------------------------------------------------------------------------------------

end module initial_data_rj2011
