=begin

  Program:   Visualization Toolkit
  Module:    $RCSfile: finance.rb,v $
  Language:  Ruby
  Date:      $Date: 2005/03/11 07:50:08 $
  Version:   $Revision: 1.2 $

  Copyright (c) 1993-2002 Ken Martin, Will Schroeder, Bill Lorensen 
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notice for more information.

=end
require "vtk"

def ReadFinancialData(x, y, z, s)
  filename = ENV["VTK_DATA_ROOT"]

  
  filename || raise("ERROR: Please define environment variable VTK_DATA_ROOT")

  filename += "/Data/financial.txt"
  
  file = File.open(filename,"r")
  file || raise("ERROR: Can't read file: #{filename}")
                
  tag,npts = file.readline.split # read number of points
  npts = npts.to_i
  
  dataSet = Vtk::UnstructuredGrid.new
  xV = Array.new(npts)
  yV = Array.new(npts)
  zV = Array.new(npts)
  sV = Array.new(npts)

  if !(xV=ParseFile(file, x)) || !(yV=ParseFile(file, y)) ||
  !(zV=ParseFile(file, z)) || !(sV=ParseFile(file, s))
    raise "Couldn't read data!"
  end

  newPts = Vtk::Points.new
  newScalars = Vtk::FloatArray.new

  xyz = Array.new(3)
  npts.times{|i|
    xyz[0] = xV[i]
    xyz[1] = yV[i]
    xyz[2] = zV[i]
    newPts.InsertPoint(i, xyz)
    newScalars.InsertValue(i, sV[i])
    }

  dataSet.SetPoints(newPts)
  dataSet.GetPointData.SetScalars(newScalars)

  return dataSet
end

def ParseFile(file, label)
  min = Vtk::VTK_LARGE_FLOAT
  max = -Vtk::VTK_LARGE_FLOAT

  return 0 if (!file || !label)

  file.rewind
  
  tag,npts = file.readline.split
  npts = npts.to_i
  while ( tag=file.readline.chop! )
    if tag==label
      readData = 1
      data = Array.new
      while (data.length<npts-1)
        data.concat file.readline.split.collect{|s| s.to_f}
      end
      for i in 0...npts
        min = data[i] if data[i] < min
        max = data[i] if data[i] > min
      end
      # normalize data
      for i in 0...npts
        data[i] = min + data[i]/(max-min)
      end
      return data
    end
  end
  return nil
end


  
# read data
dataSet = ReadFinancialData("MONTHLY_PAYMENT","INTEREST_RATE",
                              "LOAN_AMOUNT","TIME_LATE")
exit until dataSet

# construct pipeline for original population
popSplatter = Vtk::GaussianSplatter.new
popSplatter.SetInput(dataSet)
popSplatter.SetSampleDimensions(50,50,50)
popSplatter.SetRadius(0.05)
popSplatter.ScalarWarpingOff
popSurface = Vtk::ContourFilter.new
popSurface.SetInput(popSplatter.GetOutput)
popSurface.SetValue(0.01,0.1)
#popSurface.SetValue(0,0.01)
popMapper = Vtk::PolyDataMapper.new
popMapper.SetInput(popSurface.GetOutput)
popMapper.ScalarVisibilityOff
popActor = Vtk::Actor.new
popActor.SetMapper(popMapper)
popActor.GetProperty.SetOpacity(0.3)
popActor.GetProperty.SetColor(0.9,0.9,0.9)

# construct pipeline for delinquent population
lateSplatter = Vtk::GaussianSplatter.new
lateSplatter.SetInput(dataSet)
lateSplatter.SetSampleDimensions(50,50,50)
lateSplatter.SetRadius(0.05)
lateSplatter.SetScaleFactor(0.005)
lateSurface = Vtk::ContourFilter.new
lateSurface.SetInput(lateSplatter.GetOutput)
lateSurface.SetValue(0,0.01)
lateMapper = Vtk::PolyDataMapper.new
lateMapper.SetInput(lateSurface.GetOutput)
lateMapper.ScalarVisibilityOff
lateActor = Vtk::Actor.new
lateActor.SetMapper(lateMapper)
lateActor.GetProperty.SetColor(1.0,0.0,0.0)

# create axes
popSplatter.Update
bounds = popSplatter.GetOutput.GetBounds
axes = Vtk::Axes.new
axes.SetOrigin(bounds[0], bounds[2], bounds[4])
axes.SetScaleFactor(popSplatter.GetOutput.GetLength/5)
axesTubes = Vtk::TubeFilter.new
axesTubes.SetInput(axes.GetOutput)
axesTubes.SetRadius(axes.GetScaleFactor/25.0)
axesTubes.SetNumberOfSides(6)
axesMapper = Vtk::PolyDataMapper.new
axesMapper.SetInput(axesTubes.GetOutput)
axesActor = Vtk::Actor.new
axesActor.SetMapper(axesMapper)

# graphics stuff
renderer = Vtk::Renderer.new
renWin = Vtk::RenderWindow.new
renWin.AddRenderer(renderer)
iren = Vtk::RenderWindowInteractor.new
iren.SetRenderWindow(renWin)

# read data  //set up renderer
renderer.AddActor(lateActor)
renderer.AddActor(axesActor)
renderer.AddActor(popActor)
renderer.SetBackground(1,1,1)
renWin.SetSize(300,300)

# interact with data
iren.Initialize

renWin.Render
iren.Start
