require "rbconfig"
require "gtkglext"
require "vtk"


class GtkGLExtVTKRenderWindowBase < Gtk::DrawingArea

  def initialize
    super
    glconfig = Gdk::GLConfig.new(Gdk::GLConfig::MODE_RGB|
                                   Gdk::GLConfig::MODE_DEPTH|
                                   Gdk::GLConfig::MODE_DOUBLE)
    if !glconfig
      glconfig = Gdk::GLConfig.new(Gdk::GLConfig::MODE_RGB|
                                   Gdk::GLConfig::MODE_DEPTH)
    end
    set_gl_capability(glconfig)
    @RenderWindow = Vtk::RenderWindow.new
    # private attributes
    @Created = false

    # used by the LOD actors
    @DesiredUpdateRate = 15
    @StillUpdateRate = 0.0001

    self.ConnectSignals
        
    # need this to be able to handle key_press events.
    set_flags(Gtk::Window::CAN_FOCUS)
    # default size
    set_size_request(300, 300)
  end

  def ConnectSignals
    signal_connect("realize"){|wid,event| OnRealize(wid, event) }
    signal_connect("expose_event"){|wid,event| OnExpose(wid, event) }
    signal_connect("configure_event"){|wid,event| OnConfigure(wid, event) }
    signal_connect("button_press_event"){|wid,event| OnButtonDown(wid, event) }
    signal_connect("button_release_event"){|wid,event| OnButtonUp(wid, event) }
    signal_connect("motion_notify_event"){|wid,event| OnMouseMove(wid, event) }
    signal_connect("enter_notify_event"){|wid,event| OnEnter(wid, event) }
    signal_connect("leave_notify_event"){|wid,event| OnLeave(wid, event) }
    signal_connect("key_press_event"){|wid,event| OnKeyPress(wid, event) }
    signal_connect("delete_event"){|wid,event| OnDestroy(wid, event) }
    add_events(Gdk::Event::EXPOSURE_MASK|
                 Gdk::Event::BUTTON_PRESS_MASK |
                 Gdk::Event::BUTTON_RELEASE_MASK |
                 Gdk::Event::KEY_PRESS_MASK |
                 Gdk::Event::POINTER_MOTION_MASK |
                 Gdk::Event::POINTER_MOTION_HINT_MASK |
                 Gdk::Event::ENTER_NOTIFY_MASK |
                 Gdk::Event::LEAVE_NOTIFY_MASK)
  end
        
  def GetRenderWindow
    return @RenderWindow
  end

  def GetRenderer
    @RenderWindow.GetRenderers.InitTraversal
    return @RenderWindow.GetRenderers.GetNextItem
  end

  def SetDesiredUpdateRate(rate)
    @DesiredUpdateRate = rate
  end

  def GetDesiredUpdateRate
    return @DesiredUpdateRate
  end

  def SetStillUpdateRate(rate)
    @StillUpdateRate = rate
  end

  def GetStillUpdateRate
    return @StillUpdateRate
  end

  def Render
    if @Created
      return @RenderWindow.Render
    else
      return nil
    end
  end

  def Created
    return @Created
  end
    
  def OnRealize(wid, event)
    if !@Created
      # you can't get the xid without the window being realized.
      realize
      if Config::CONFIG["host_os"] =~ /win32/
#        win_id = @widget.window.handle.to_s
        win_id = window.handle.to_s
      else
#        win_id = @widget.window.xid.to_s
        win_id = window.xid.to_s
        @RenderWindow.SetWindowInfo(win_id)
        @Created = true
      end
      return true
    else
      return false
    end
  end

  def OnConfigure(wid, event)
#    @widget=wid
    @RenderWindow.SetSize(event.width, event.height)
    self.Render
    return true
  end

  def OnExpose(wid, event)
    self.Render
    return true
  end

  def OnDestroy(wid, event)
    hide
    del(@RenderWindow)
    destroy
    return true
  end

  def OnButtonDown(wid, event)
    @RenderWindow.SetDesiredUpdateRate(@DesiredUpdateRate)
    return true
  end
    
  def OnButtonUp(wid, event)
    @RenderWindow.SetDesiredUpdateRate(@StillUpdateRate)
    return true
  end

  def OnMouseMove(wid, event)
    return true
  end

  def OnEnter(wid, event)
    return true
  end

  def OnLeave(wid, event)
    return true
  end
    
  def OnKeyPress(wid, event)
    return true
  end

  def OnKeyRelease(wid, event)
    return true
  end
end


class GtkGLExtVTKRenderWindow < GtkGLExtVTKRenderWindowBase

  def initialize
    super

    @CurrentRenderer = nil
    @CurrentCamera = nil
    @CurrentZoom = 1.0
    @CurrentLight = nil

    @ViewportCenterX = 0
    @ViewportCenterY = 0
        
    @Picker = Vtk::CellPicker.new
    @PickedAssembly = nil
    @PickedProperty = Vtk::Property.new
    @PickedProperty.SetColor(1, 0, 0)
    @PrePickedProperty = nil
        
    @OldFocus = nil

    # these record the previous mouse position
    @LastX = 0
    @LastY = 0
  end

  def OnButtonDown(wid, event)
    @RenderWindow.SetDesiredUpdateRate(@DesiredUpdateRate)
    self.StartMotion(wid, event)
    return true
  end

  def OnButtonUp(wid, event)
    @RenderWindow.SetDesiredUpdateRate(@StillUpdateRate)
    self.EndMotion(wid, event)
    return true
  end

  def OnMouseMove(wid, event=nil)
    if (event.state & Gdk::Window::ModifierType::BUTTON1_MASK) == Gdk::Window::ModifierType::BUTTON1_MASK
      if (event.state & Gdk::Window::ModifierType::SHIFT_MASK) == Gdk::Window::ModifierType::SHIFT_MASK
        m = pointer
        self.Pan(m[0], m[1])
      else
        m = pointer
        self.Rotate(m[0], m[1])
      end
    elsif (event.state & Gdk::Window::ModifierType::BUTTON2_MASK) == Gdk::Window::ModifierType::BUTTON2_MASK
      m = pointer
      self.Pan(m[0], m[1])
    elsif (event.state & Gdk::Window::ModifierType::BUTTON3_MASK) == Gdk::Window::ModifierType::BUTTON3_MASK
      m = pointer
      self.Zoom(m[0], m[1])
    else
      return false
      end
    return true
  end

  def OnEnter(wid, event=nil)
    # a render hack because grab_focus blanks the renderwin
    grab_focus
    w = pointer
    self.UpdateRenderer(w[0], w[1])
    return true
  end
    
  def OnKeyPress(wid, event=nil)
    if (event.keyval == Gdk::Keyval.from_name("q")) ||
        (event.keyval == Gdk::Keyval.from_name("Q"))
      Gtk.main_quit
    elsif (event.keyval == Gdk::Keyval.from_name('r')) ||
        (event.keyval == Gdk::Keyval.from_name('R'))
      self.Reset
      return true
    elsif (event.keyval == Gdk::Keyval.from_name('w')) ||
        (event.keyval == Gdk::Keyval.from_name('W'))
      self.Wireframe
      return true
    elsif (event.keyval == Gdk::Keyval.from_name('s')) ||
        (event.keyval == Gdk::Keyval.from_name('S'))
      self.Surface
      return true
    elsif (event.keyval == Gdk::Keyval.from_name('p')) ||
        (event.keyval == Gdk::Keyval.from_name('P'))
      m = pointer
      self.PickActor(m[0], m[1])
      return true
    else
      return false
    end
  end

  def GetZoomFactor
    return @CurrentZoom
  end

  def SetZoomFactor(zf)
    @CurrentZoom = zf
  end

  def GetPicker
    return @Picker
  end

  def Render
    if (@CurrentLight)
      light = @CurrentLight
      light.SetPosition(*@CurrentCamera.GetPosition)
      light.SetFocalPoint(*@CurrentCamera.GetFocalPoint)
    end

    super
  end

  def UpdateRenderer(x,y)
    windowX,windowY  = window.size
#    windowX,windowY  = @widget.window.get_size

    renderers = @RenderWindow.GetRenderers
    numRenderers = renderers.GetNumberOfItems

    @CurrentRenderer = nil
    renderers.InitTraversal
    for i in 0...numRenderers
      renderer = renderers.GetNextItem
      vx,vy = [0,0]
      if (windowX > 1)
        vx = x.to_f/(windowX-1)
        if (windowY > 1)
          vy = (windowY-y.to_f-1)/(windowY-1)
          (vpxmin,vpymin,vpxmax,vpymax) = renderer.GetViewport

          if (vx >= vpxmin && vx <= vpxmax && vy >= vpymin && vy <= vpymax)
            @CurrentRenderer = renderer
            @ViewportCenterX = windowX.to_f*(vpxmax-vpxmin)/2.0 +vpxmin
            @ViewportCenterY = windowY.to_f*(vpymax-vpymin)/2.0 +vpymin
            @CurrentCamera = @CurrentRenderer.GetActiveCamera
            lights = @CurrentRenderer.GetLights
            lights.InitTraversal
            break if @CurrentLight = lights.GetNextItem
          end
        end
      end
    end

    @LastX = x
    @LastY = y
    return true
  end

  def GetCurrentRenderer
    if @CurrentRenderer.nil?
      renderers = @RenderWindow.GetRenderers
      numRenderers = renderers.GetNumberOfItems

      renderers.InitTraversal
      for i in 0...numRenderers
        break if renderer = renderers.GetNextItem
      end
      @CurrentRenderer = renderer
    end
    return @CurrentRenderer
  end

  def GetCurrentCamera
    if @CurrentCamera.nil?
      renderer = GetCurrentRenderer
      @CurrentCamera = renderer.GetActiveCamera
    end
    return @CurrentCamera
  end

  def StartMotion(wid, event=nil)
    x = event.x
    y = event.y
    self.UpdateRenderer(x,y)
    return true
  end

  def EndMotion(wid, event=nil)
    if @CurrentRenderer
      self.Render
    end
    return true
  end

  def Rotate(x,y)
    if @CurrentRenderer
            
      @CurrentCamera.Azimuth(@LastX - x)
      @CurrentCamera.Elevation(y - @LastY)
      @CurrentCamera.OrthogonalizeViewUp
            
      @LastX = x
      @LastY = y
            
      @CurrentRenderer.ResetCameraClippingRange
      self.Render
    end
  end

  def Pan(x,y)
    if @CurrentRenderer
            
      renderer = @CurrentRenderer
      camera = @CurrentCamera
      (pPoint0,pPoint1,pPoint2) = camera.GetPosition
      (fPoint0,fPoint1,fPoint2) = camera.GetFocalPoint

      if (camera.GetParallelProjection)
        renderer.SetWorldPoint(fPoint0,fPoint1,fPoint2,1.0)
        renderer.WorldToDisplay
        fx,fy,fz = renderer.GetDisplayPoint
        renderer.SetDisplayPoint(fx-x+@LastX,
                                 fy+y-@LastY,
                                 fz)
        renderer.DisplayToWorld
        fx,fy,fz,fw = renderer.GetWorldPoint
        camera.SetFocalPoint(fx,fy,fz)

        renderer.SetWorldPoint(pPoint0,pPoint1,pPoint2,1.0)
        renderer.WorldToDisplay
        fx,fy,fz = renderer.GetDisplayPoint
        renderer.SetDisplayPoint(fx-x+@LastX,
                                 fy+y-@LastY,
                                 fz)
        renderer.DisplayToWorld
        fx,fy,fz,fw = renderer.GetWorldPoint
        camera.SetPosition(fx,fy,fz)
                
      else
        fPoint0,fPoint1,fPoint2, = camera.GetFocalPoint
        # Specify a point location in world coordinates
        renderer.SetWorldPoint(fPoint0,fPoint1,fPoint2,1.0)
        renderer.WorldToDisplay
        # Convert world point coordinates to display coordinates
        dPoint = renderer.GetDisplayPoint
        focalDepth = dPoint[2]
                
        aPoint0 = @ViewportCenterX + (x - @LastX)
        aPoint1 = @ViewportCenterY - (y - @LastY)
                
        renderer.SetDisplayPoint(aPoint0,aPoint1,focalDepth)
        renderer.DisplayToWorld
                
        rPoint0,rPoint1,rPoint2,rPoint3, = renderer.GetWorldPoint
        if rPoint3 != 0.0
          rPoint0 = rPoint0/rPoint3
          rPoint1 = rPoint1/rPoint3
          rPoint2 = rPoint2/rPoint3
        end

        camera.SetFocalPoint((fPoint0 - rPoint0) + fPoint0, 
                             (fPoint1 - rPoint1) + fPoint1,
                             (fPoint2 - rPoint2) + fPoint2) 
                
        camera.SetPosition((fPoint0 - rPoint0) + pPoint0, 
                           (fPoint1 - rPoint1) + pPoint1,
                           (fPoint2 - rPoint2) + pPoint2)
      end

      @LastX = x
      @LastY = y

      self.Render
    end
  end

  def Zoom(x,y)
    if @CurrentRenderer

      renderer = @CurrentRenderer
      camera = @CurrentCamera

      zoomFactor = 1.02**(0.5*(@LastY - y))

      @CurrentZoom = @CurrentZoom * zoomFactor

      if camera.GetParallelProjection!=0
        parallelScale = camera.GetParallelScale/zoomFactor
        camera.SetParallelScale(parallelScale)
      else
        camera.Dolly(zoomFactor)
        renderer.ResetCameraClippingRange
      end

      @LastX = x
      @LastY = y

      self.Render
    end
  end

  def Reset
    if @CurrentRenderer
      @CurrentRenderer.ResetCamera
    end
    self.Render
  end

  def Wireframe
    actors = @CurrentRenderer.GetActors
    numActors = actors.GetNumberOfItems
    actors.InitTraversal
    for i in 0...numActors
      actor = actors.GetNextItem
      actor.GetProperty.SetRepresentationToWireframe
    end
    self.Render
  end
        
  def Surface
    actors = @CurrentRenderer.GetActors
    numActors = actors.GetNumberOfItems
    actors.InitTraversal
    for i in 0...numActors
      actor = actors.GetNextItem
      actor.GetProperty.SetRepresentationToSurface
    end
    self.Render
  end

  def PickActor(x,y)
    if @CurrentRenderer

      renderer = @CurrentRenderer
      picker = @Picker

      windowX,windowY  = window.size
      picker.Pick(x,(windowY - y - 1),0.0,renderer)
      assembly = picker.GetAssembly

      if (@PickedAssembly != nil) && (@PrePickedProperty != nil)
        @PickedAssembly.SetProperty(@PrePickedProperty)
        # release hold of the property
        @PrePickedProperty.UnRegister(@PrePickedProperty)
        @PrePickedProperty = nil
      end

      if (assembly != nil)
        @PickedAssembly = assembly
        @PrePickedProperty = @PickedAssembly.GetProperty
        # hold onto the property
        @PrePickedProperty.Register(@PrePickedProperty)
        @PickedAssembly.SetProperty(@PickedProperty)
      end

      self.Render
    end
  end


end



############################
if $0 == __FILE__

  Gtk.init
  Gtk::GL.init

  window = Gtk::Window.new
  window.set_title("A GtkGLExtVTKRenderWindow Demo!")
  window.signal_connect("destroy"){ Gtk.main_quit }
  window.signal_connect("delete_event"){ Gtk.main_quit }
  window.set_border_width(10)

  vtkgtk = GtkGLExtVTKRenderWindow.new
  vtkgtk.show
    
  vbox = Gtk::VBox.new(false, 3)
  vbox.show
  vbox.pack_start(vtkgtk)

  button = Gtk::Button.new('My Button')
  button.show
  vbox.pack_start(button)
  window.add(vbox)
    
  window.set_size_request(400, 400)

  # The VTK stuff.
  cone = Vtk::ConeSource.new
  cone.SetResolution(80)
  coneMapper = Vtk::PolyDataMapper.new
  coneMapper.SetInput(cone.GetOutput)
  coneActor = Vtk::Actor.new
  coneActor.SetMapper(coneMapper)    
  coneActor.GetProperty.SetColor(0.5, 0.5, 1.0)
  ren = Vtk::Renderer.new
  vtkgtk.GetRenderWindow.AddRenderer(ren)
  ren.AddActor(coneActor)

  # show the main window and start event processing.
  window.show
  Gtk.main

end
