// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libdap, A C++ implementation of the OPeNDAP Data
// Access Protocol.

// Copyright (c) 2002,2003 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//         Reza Nekovei <reza@intcomm.net>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.
 
// (c) COPYRIGHT URI/MIT 1994-1996
// Please read the full copyright statement in the file COPYRIGHT.
//
// Authors:
//      reza            Reza Nekovei (reza@intcomm.net)

// DODS-netCDF surrogate library functions for data attributes. See note
// about the symbol LOCAL in Dfile.c
// 
// ReZa 10/4/94

#include "config_nc.h"

#include <cstring>

static char rcsid[] not_used ={"$Id: Dattr.cc 18782 2008-05-23 16:39:07Z jimg $"};

#include "Dnetcdf.h"
#include "Dncx.h"

#include "nc_util.h"

/** Defined in lnetcdf/lerror. This is not included in a header (like
    lnetcdf.h because doing so would require all sources that use lnetcdf.h
    to also include Error.h 03/02/05 jhrg */
extern void set_opendap_error(const Error &);

int
nc_put_att_text(int cdfid, int varid, const char *name,
        size_t nelems, const char *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_put_att_text((*conns)[cdfid]->get_ncid(), varid, name, 
			  nelems, value);
	return rcode;
    }
#endif

    return NC_EPERM; // DODS network interface is read-only
}

int
nc_put_att_schar(int cdfid, int varid, const char *name,
        nc_type type, size_t nelems, const signed char *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_put_att_schar((*conns)[cdfid]->get_ncid(), varid, name, 
			  type, nelems, value);
	return rcode;
    }
#endif

    return NC_EPERM; // DODS network interface is read-only
}

int
nc_put_att_uchar(int cdfid, int varid, const char *name,
        nc_type type, size_t nelems, const unsigned char *value)
{ 
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_put_att_uchar((*conns)[cdfid]->get_ncid(), varid, name, 
			  type, nelems, value);
	return rcode;
    }
#endif

    return NC_EPERM; // DODS network interface is read-only
}
 
int
nc_put_att_short(int cdfid, int varid, const char *name,
        nc_type type, size_t nelems, const short *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_put_att_short((*conns)[cdfid]->get_ncid(), varid, name, 
			  type, nelems, value);
	return rcode;
    }
#endif

    return NC_EPERM; // DODS network interface is read-only
}

int
nc_put_att_int(int cdfid, int varid, const char *name,
        nc_type type, size_t nelems, const int *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_put_att_int((*conns)[cdfid]->get_ncid(), varid, name, 
			  type, nelems, value);
	return rcode;
    }
#endif

    return NC_EPERM; // DODS network interface is read-only
}

int
nc_put_att_long(int cdfid, int varid, const char *name,
        nc_type type, size_t nelems, const long *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_put_att_long((*conns)[cdfid]->get_ncid(), varid, name, 
			  type, nelems, value);
	return rcode;
    }
#endif

    return NC_EPERM; // DODS network interface is read-only
}


int
nc_put_att_float(int cdfid, int varid, const char *name,
        nc_type type, size_t nelems, const float *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_put_att_float((*conns)[cdfid]->get_ncid(), varid, name, 
			  type, nelems, value);
	return rcode;
    }
#endif

    return NC_EPERM; // DODS network interface is read-only
}

int
nc_put_att_double(int cdfid, int varid, const char *name,
        nc_type type, size_t nelems, const double *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_put_att_double((*conns)[cdfid]->get_ncid(), varid, name, 
			  type, nelems, value);
	return rcode;
    }
#endif

    return NC_EPERM; // DODS network interface is read-only
}

// Used for the netCDF 2 interface emulation. See lv2i.c. 
extern "C" int
nc_put_att(int cdfid, int varid, const char *name, nc_type type,
        size_t nelems, const void *value)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_put_att((*conns)[cdfid]->get_ncid(), varid, name, 
			  type, nelems, value);
	return rcode;
    }
#endif

    return NC_EPERM; // DODS network interface is read-only
} 


int
nc_rename_att(int cdfid, int varid, const char *name, const char *newname)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_rename_att((*conns)[cdfid]->get_ncid(), varid, name, newname);
	return rcode;
    }
#endif

    return NC_EPERM; // DODS network interface is read-only
}

int
nc_del_att(int cdfid, int varid, const char *name)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_del_att((*conns)[cdfid]->get_ncid(), varid, name);
	return rcode;
    }
#endif

    return NC_EPERM; // DODS network interface is read-only
}

int
nc_inq_attname(int cdfid, int varid, int attnum, char *name)
{
#if 0
    NCConnect &ncc = *((*conns)[cdfid]);
#endif
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_inq_attname((*conns)[cdfid]->get_ncid(), varid, attnum, name);
	return rcode;
    }
#endif

    if (attnum < 0)
      return  NC_ENOTATT;

    try {
	AttrTable &attr = (*conns)[cdfid]->get_attribute_table(varid);

	if (attr.get_size() < 1 || attnum > (int)attr.get_size() - 1)
	    return NC_ENOTATT;

	strcpy(name, attr.get_name(attr.attr_begin() + attnum).c_str());
    }
    catch (Error &e) {
        set_opendap_error(e);
	return e.get_error_code();
    }

    return NC_NOERR;
}

int 
nc_inq_attid(int cdfid, int varid, const char *name, int *attnump)
{
#if 0
    NCConnect &ncc = *((*conns)[cdfid]);
#endif

    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_inq_attid((*conns)[cdfid]->get_ncid(), varid, name, attnump);
	return rcode;
    }
#endif

    try {
	AttrTable &attr = (*conns)[cdfid]->get_attribute_table(varid);
	AttrTable::Attr_iter p = attr.attr_begin();
 
	for (int i = 0; p != attr.attr_end(); ++p, ++i) {
	    if (attr.get_name(p) == name) {
		if (attnump != 0)
		    *attnump = i;
		return NC_NOERR;
	    }
	}
    }
    catch (Error &e) {
        set_opendap_error(e);
	return e.get_error_code();
    }
    
    return NC_ENOTATT;
}

int
nc_inq_att(int cdfid, int varid, const char *name, 
	   nc_type *datatypep, size_t *lenp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_inq_att( (*conns)[cdfid]->get_ncid(), varid, 
			   name, datatypep, lenp);
	return rcode;
    }
#endif

    try {
	AttrTable &attr = (*conns)[cdfid]->get_attribute_table(varid);

	AttrTable::Attr_iter p;
	AttrTable *dummy;
	DBG(cerr << "In nc_inq_att: name: " << name << endl);
	attr.find(name, &dummy, &p);

	if (p == attr.attr_end())
	    return NC_ENOTATT;

	if (datatypep == 0 && lenp == 0)
	    return NC_NOERR;

	if (p < attr.attr_begin() || p >= attr.attr_end())
	    throw Error((string)"Bad iterator value when looking for " + name);

	DBG(cerr << "Before attr.get_attr(p) ... ");
	string attrV = attr.get_attr(p);
	DBG(cerr << "After attr.get_attr(p)" << endl);
	AttrType attrT = attr.get_attr_type(p);

	if (lenp) 
            *lenp = static_cast<size_t>(attr.get_attr_num(p));

	switch (attrT) {
	  case Attr_byte:
	    if(datatypep) *datatypep =  NC_BYTE;
	    break;
	  case Attr_int32:
	  case Attr_uint32:
	    if(datatypep) *datatypep =  NC_LONG;
	    break;
	  case Attr_int16:
	  case Attr_uint16:
	    if(datatypep) *datatypep =  NC_SHORT;
	    break;
	  case Attr_float32:
	    if(datatypep) *datatypep =  NC_FLOAT;
	    break;
	  case Attr_float64:
	    if(datatypep) *datatypep =  NC_DOUBLE;
	    break;
	  case Attr_string:
	  case Attr_url:
	    if(datatypep) *datatypep =  NC_CHAR;
	    if(lenp)
		*lenp = compute_string_attr_length(attr, p);
	    break;

	  default:
	    return NC_ENOTATT;
	}
    }
    catch (Error &e) {
        set_opendap_error(e);
	return e.get_error_code();
    }
    catch (exception &e) {
	DBG(cerr << "nc_inq_att: Exception: " << e.what() << endl);
	return NC_ENOTATT;
    }

    return NC_NOERR ;
}

int 
nc_inq_attlen(int cdfid, int varid, const char *name, size_t *lenp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_inq_attlen( (*conns)[cdfid]->get_ncid(), varid, 
			   name, lenp);
	return rcode;
    }
#endif

    if (lenp == 0)
	return NC_NOERR;

    try {
	AttrTable &attr = (*conns)[cdfid]->get_attribute_table(varid);

	AttrTable::Attr_iter p;
	AttrTable *dummy;
	attr.find(name, &dummy, &p);

	if (p == attr.attr_end())
	    return NC_ENOTATT;

	if (attr.get_attr_type(p) == Attr_string
	    || attr.get_attr_type(p) == Attr_url) {
	    *lenp = compute_string_attr_length(attr,p);
	}
	else {
	    *lenp = static_cast<size_t>(attr.get_attr_num(p));
	}
    }
    catch (Error &e) {
        set_opendap_error(e);
	return e.get_error_code();
    }
    catch (exception &e) {
	DBG(cerr << "nc_inq_attlen: Exception: " << e.what() << endl);
	return NC_ENOTATT;
    }


    return NC_NOERR;
}

int 
nc_inq_atttype(int cdfid, int varid, const char *name, nc_type *datatypep)
{
#if 0
    NCConnect &ncc = *((*conns)[cdfid]);
#endif
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_inq_atttype( (*conns)[cdfid]->get_ncid(), varid, 
			   name, datatypep);
	return rcode;
    }
#endif

    try {
	AttrTable &attr = (*conns)[cdfid]->get_attribute_table(varid);
	AttrTable::Attr_iter p;
	AttrTable *dummy;
	attr.find(name, &dummy, &p);

	if (p == attr.attr_end())
	    return NC_ENOTATT;

	if (datatypep == 0)
	    return NC_NOERR;

	AttrType attrT = attr.get_attr_type(p);

	switch (attrT) {
	  case Attr_byte:
	    *datatypep =  NC_BYTE;
	    break;
	  case Attr_int32:
	  case Attr_uint32:
	    *datatypep =  NC_LONG;
	    break;
	  case Attr_int16:
	  case Attr_uint16:
	    *datatypep =  NC_SHORT;
	    break;
	  case Attr_float32:
	    *datatypep =  NC_FLOAT;
	    break;
	  case Attr_float64:
	    *datatypep =  NC_DOUBLE;
	    break;
	  case Attr_string:
      case Attr_url:
	    *datatypep =  NC_CHAR;
        break;
	  default:
	    return NC_NOERR;	// Fails three tests w/o this. 03/09/04 jhrg
	}
    }
    catch (Error &e) {
        set_opendap_error(e);
	return e.get_error_code();
    }

    return NC_NOERR;
}

int
nc_copy_att(int incdf, int invar, const char *name, int outcdf, int outvar)
{
    // does the file id exist?
    if (!conns || incdf < 0 || incdf > MAX_NC_OPEN 
	|| outcdf < 0 || outcdf > MAX_NC_OPEN 
	|| (*conns)[incdf] == NULL || (*conns)[outcdf] == NULL) 
	return NC_EBADID;

    // case 1, both input and output files are local

#ifdef LOCAL
    if ((*conns)[incdf]->is_local() && (*conns)[outcdf]->is_local()) {
        rcode = lnc_copy_att((*conns)[incdf]->get_ncid(), invar, 
                    name, (*conns)[outcdf]->get_ncid(), outvar);
	    return rcode;
    }
#endif

    // case 2, the output file is not local, Error writing over the net.

    if (!(*conns)[outcdf]->is_local()) 
      return NC_EPERM; // DODS network interface is read-only

    // last case, input is from DODS file and output a local netcdf file

#ifdef LOCAL
    char *values = 0;         // Attribute vector to be copied,
    nc_type datatype;    // it's type, 
    size_t count;          // and it's length.

    try {
        values = (*conns)[incdf]->get_raw_values(invar, name, &count, &datatype);
    }
    catch (Error &e) {
    	delete [] values; values = 0;
        set_opendap_error(e);
    	return e.get_error_code();
    }

    // Only use the value returned by get_ncid() when calling the local (i.e.
    // real) netCDF library. When calling our functions, make sure to pass
    // the index to conns. jhrg 6/1/05
#if 0
    rcode = nc_put_att((*conns)[outcdf]->get_ncid(), outvar,  
		       name, datatype, count, (void *)values);
#endif
    DBG(cerr << "nc_copy_att" << endl);
    rcode = nc_put_att(outcdf, outvar, name, datatype, count, (void *)values);

    delete [] values; values = 0;
    return rcode;

#else

    return NC_EPERM;

#endif /* Last case LOCAL */
}

int
nc_get_att_text(int cdfid, int varid, const char *name, char *str)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_att_text((*conns)[cdfid]->get_ncid(), varid, name, str);
	return rcode;
    }
#endif

    char *values = 0;
    try {
	size_t count;
	nc_type dt;
	values  = (*conns)[cdfid]->get_raw_values(varid, name, &count, &dt);
	if (dt != NC_CHAR)
            throw Error(NC_ECHAR, "NetCDF Error");
	    // return NC_ECHAR;

	DBG(cerr << "nc_get_att_text" << endl);
#if 0
	memcpy(str, values, count);
	str[count + 1] = '\0';	// ?? Is there space for this. seems to be
				// needed. jhrg 11/10/06
	rcode = NC_NOERR;
#else
	rcode = putn_into_text((void **)&values, count, str);
#endif
	delete [] values; values = 0;
	return rcode;
    }
    catch (Error &e) {
        delete [] values; values = 0;
        set_opendap_error(e);
	return e.get_error_code();
    }
}

int
nc_get_att_schar(int cdfid, int varid, const char *name, signed char *tp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_att_schar((*conns)[cdfid]->get_ncid(), varid, name, tp);
	return rcode;
    }
#endif

    char *values = 0;
    try {
	size_t count;
	nc_type dt;
	values = (*conns)[cdfid]->get_raw_values(varid, name, &count, &dt);
	if (dt == NC_CHAR)
            throw Error(NC_ECHAR, "NetCDF Error");
	    // return NC_ECHAR;
	DBG(cerr << "nc_get_att_schar" << endl);
	rcode = putn_into_schar((void **)&values, count, tp, dt);

	delete [] values; values = 0;
	return rcode;
    }
    catch (Error &e) {
        delete [] values; values = 0;
        set_opendap_error(e);
	return e.get_error_code();
    }
}

int
nc_get_att_uchar(int cdfid, int varid, const char *name, unsigned char *tp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_att_uchar((*conns)[cdfid]->get_ncid(), varid, name, tp);
	return rcode;
    }
#endif

    char *values = 0;
    try {
	size_t count;
	nc_type dt;
	values = (*conns)[cdfid]->get_raw_values(varid, name, &count, &dt);
	if (dt == NC_CHAR)
            throw Error(NC_ECHAR, "NetCDF Error");
	    // return NC_ECHAR;

	rcode = putn_into_uchar((void **)&values, count, tp, dt);
    
	delete [] values; values = 0;
	return rcode;
    }
    catch (Error &e) {
        delete [] values; values = 0;
        set_opendap_error(e);
	return e.get_error_code();
    }
}

int
nc_get_att_short(int cdfid, int varid, const char *name, short *tp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_att_short((*conns)[cdfid]->get_ncid(), varid, name, tp);
	return rcode;
    }
#endif

    char *values = 0;
    try {
	size_t count;
	nc_type dt;
	values = (*conns)[cdfid]->get_raw_values(varid, name, &count, &dt);
	if (dt == NC_CHAR)
            throw Error(NC_ECHAR, "NetCDF Error");
	    // return NC_ECHAR;
	rcode = putn_into_short((void **)&values, count, tp, dt);

	delete [] values; values = 0;
	return rcode;
    }
    catch (Error &e) {
        delete [] values; values = 0;
        set_opendap_error(e);
	return e.get_error_code();
    }
}

int
nc_get_att_int(int cdfid, int varid, const char *name, int *tp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_att_int((*conns)[cdfid]->get_ncid(), varid, name, tp);
	return rcode;
    }
#endif

    char *values = 0;
    try {
	size_t count;
	nc_type dt;
	values = (*conns)[cdfid]->get_raw_values(varid, name, &count, &dt);
	if (dt == NC_CHAR)
            throw Error(NC_ECHAR, "NetCDF Error");
         // return NC_ECHAR;
	rcode = putn_into_int((void **)&values, count, tp, dt);

	delete [] values; values = 0;
	return rcode;
    }
    catch (Error &e) {
        delete [] values; values = 0;
        set_opendap_error(e);
	return e.get_error_code();
    }
}

int
nc_get_att_long(int cdfid, int varid, const char *name, long *tp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_att_long((*conns)[cdfid]->get_ncid(), varid, name, tp);
	return rcode;
    }
#endif

    char *values = 0;
    try {
	size_t count;
	nc_type dt;
	values = (*conns)[cdfid]->get_raw_values(varid, name, &count, &dt);
	if (dt == NC_CHAR)
            throw Error(NC_ECHAR, "NetCDF Error");
         // return NC_ECHAR;
	rcode = putn_into_long((void **)&values, count, tp, dt);

	delete [] values; values = 0;
	return rcode;
    }
    catch (Error &e) {
        delete [] values; values = 0;
        set_opendap_error(e);
	return e.get_error_code();
    }
}

int
nc_get_att_float(int cdfid, int varid, const char *name, float *tp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_att_float((*conns)[cdfid]->get_ncid(), varid, name, tp);
	return rcode;
    }
#endif

    char *values = 0;
    try {
	size_t count;
	nc_type dt;
	values = (*conns)[cdfid]->get_raw_values(varid, name, &count, &dt);
	if (dt == NC_CHAR)
            throw Error(NC_ECHAR, "NetCDF Error");
         // return NC_ECHAR;
	rcode = putn_into_float((void **)&values, count, tp, dt);

	delete [] values; values = 0;
	return rcode;
    }
    catch (Error &e) {
        delete [] values; values = 0;
        set_opendap_error(e);
	return e.get_error_code();
    }
}

int
nc_get_att_double(int cdfid, int varid, const char *name, double *tp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_att_double((*conns)[cdfid]->get_ncid(), varid, name, tp);
	return rcode;
    }
#endif

    char *values = 0;
    try {
	size_t count;
	nc_type dt;
	values = (*conns)[cdfid]->get_raw_values(varid, name, &count, &dt);
	if (dt == NC_CHAR)
            throw Error(NC_ECHAR, "NetCDF Error");
	    // return NC_ECHAR;
	rcode = putn_into_double((void **)&values, count, tp, dt);

	delete [] values; values = 0;
	return rcode;
    }
    catch (Error &e) {
        delete [] values; values = 0;
        set_opendap_error(e);
	return e.get_error_code();
    }
}

// Used for the netCDF 2 interface emulation. See lv2i.c. 
extern "C" int
nc_get_att(int cdfid, int varid, const char *name, void *values)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if ((*conns)[cdfid]->is_local()) {
	rcode = lnc_get_att((*conns)[cdfid]->get_ncid(), varid, name, values);
	return rcode;
    }
#endif

    try {
        size_t count;
        nc_type datatype;
        char *v = (*conns)[cdfid]->get_raw_values(varid, name, &count, &datatype);
	DBG(cerr << "nc_get_att" << endl);
	memcpy(values, v, count * nctypelen(datatype));
	delete[] v; v = 0;
    }
    catch (Error &e) {
        set_opendap_error(e);
	return e.get_error_code();
    }

    return NC_NOERR;
}

// $Log: Dattr.cc,v $
// Revision 1.17  2005/06/03 16:37:33  jimg
// Fixed a problem found while working on bug 773. nc_att_copy was called using
// the return value from Connect::get_ncid(). It should have been called using
// the value of cdfid (the index into conns). Only use the get_ncid() value when
// talking to the lnetcdf functions.
//
// Revision 1.16  2005/03/04 18:10:49  jimg
// At this point valgrind runs the Unidata tests for both local and remote access
// and shows no errors or leaks. There are 8 bytes still reachable from an
// exception, but that's it.
//
// Revision 1.15  2005/03/02 17:51:50  jimg
// Considerable reduction in memory leaks and fixed all errant memory
// accesses found with nc_test. OPeNDAP error codes and Error object
// message strings are now reported using the nc_strerrror() function!
//
// Revision 1.14  2004/11/30 20:44:50  jimg
// Removed old code; indented.
//
// Revision 1.13  2004/09/08 22:08:21  jimg
// More Massive changes: Code moved from the files that clone the netCDF
// function calls into NCConnect, NCAccess or nc_util.cc. Much of the
// translation functions are now methods. The netCDF type classes now
// inherit from NCAccess in addition to the DAP type classes.
//
// Revision 1.12  2004/07/26 19:10:44  jimg
// Moved netCDF CL <--> OPeNDAP server interface code to nc_util and
// NCConnect.
//
// Revision 1.11  2004/07/09 18:42:35  jimg
// Merged with release-3-4-7FCS
//
// Revision 1.10  2004/06/15 18:51:05  jimg
// Fixed bug 714: Unsigned 16 and 32 bit type attributes were not handled.
//
// Revision 1.5.4.1  2004/06/15 18:43:39  jimg
// Repaired code in void correct_att_type(). The code did not test for the
// attribute types unsigned int16 or uint32. This may not actually be a problem
// in practice with this code, but in version on the trunk the function does
// need to be fixed... See bug 714.
//
// Revision 1.9  2004/03/09 22:56:32  jimg
// Refactored so that Pix is no longer used. Some common code (in the
// glue routines) was also factored out to functions. The netCDF 2
// interface is now supplied by the lnetcdf/lv2i.c file (which is a mostly
// unaltered copy of the original v2i.c source file). See lnetcdf/README.
//
// Revision 1.8  2004/03/08 19:08:33  jimg
// This version of the code uses the Unidata netCDF 3.5.1 version of the
// netCDF 2 API emulation. This functions call our netCDF 3 API functions
// which may either interact with a DAP server r call the local netCDF 3
// functions.
//
// Revision 1.7  2004/02/25 00:47:52  jimg
// This code will translate Structures, including ones that are nested.
// Not tested much; needs work.
//
// Revision 1.6  2003/12/08 18:06:37  edavis
// Merge release-3-4 into trunk
//
// Revision 1.5  2003/01/28 07:08:24  jimg
// Merged with release-3-2-8.
//
// Revision 1.4.4.3  2002/12/18 23:40:33  pwest
// gcc3.2 compile corrections, mainly regarding using statements. Also,
// problems with multi line string literatls.
//
// Revision 1.4.4.2  2002/12/05 20:32:39  pwest
// Corrected problems with IteratorAdapter code and cleaned up file descriptors
// and memory management problems.
//
// Revision 1.4.4.1  2002/11/18 01:11:37  rmorris
// VC++ required an explicit cast to Pix in certain cases.
//
// Revision 1.4  2000/10/06 01:22:02  jimg
// Moved the CVS Log entries to the ends of files.
// Modified the read() methods to match the new definition in the dap library.
// Added exception handlers in various places to catch exceptions thrown
// by the dap library.
//
// Revision 1.3  1999/11/05 05:15:03  jimg
// Result of merge woth 3-1-0
//
// Revision 1.1.2.2  1999/10/29 05:05:19  jimg
// Reza's fixes plus the configure & Makefile update
//
// Revision 1.2  1999/10/21 13:19:05  reza
// IMAP and other bug fixed for version3.
//
// Revision 1.1.2.1  1999/10/15 19:50:55  jimg
// Changed return values and conditions for NC API entry points
//
// Revision 1.1  1999/07/28 00:22:40  jimg
// Added
//
// Revision 1.15  1999/05/07 23:45:30  jimg
// String --> string fixes
//
// Revision 1.14  1999/03/30 05:20:55  reza
// Added support for the new data types (Int16, UInt16, and Float32).
//
// Revision 1.13  1997/03/26 05:44:20  reza
// GLOBAL attribute can now be named anything with "_GLOBAL" in it (i.e.
// HDF_GLOBAL, MAT_GLOBAL, DSP_GLOBAL, etc.). This should improve interoperability
// with non-netCDF servers. Also, the DAS no longer requires attributes for all
// the variables. An internal empty attribute table (NC_BLANK) manages missing ones.
//
// Revision 1.12  1997/03/25 16:05:56  reza
// Fixed CVS duplicate log reports.
//
// Revision 1.11  1996/09/17 00:26:04  jimg
// Merged changes from a side branch which contained various changes from
// Reza and Charles.
// Removed ncdump and netexec since ncdump is now in its own directory and
// netexec is no longer used.
//
// Revision 1.10.2.1  1996/06/25 22:04:04  jimg
// Version 2.0 from Reza.
//
// Revision 1.9.2.2  1996/06/10 19:36:06  jimg
// Replaced strcpy in correct_att_type() with memcpy.
//
// Revision 1.9.2.1  1996/04/02 19:29:53  jimg
// Changed the `conns' variable so that now it is a global *pointer*, not an
// object. Each call that might add to the conns object must first check to see
// that it has been created. This was done because of link-time problems with
// static global objects.
// Fixed up some comment stuff.
// Added assert() calls for conns.
//
// Revision 1.9  1995/07/09  21:33:29  jimg
// Added copyright notice.
//
// Revision 1.8  1995/06/29  15:42:02  jimg
// Fixed instances of delete which should have included brackets
//
// Revision 1.7  1995/06/28  20:22:33  jimg
// Replaced malloc calls with calls to new (and calls to free with calls to
// delete).
//
// Revision 1.6  1995/06/23  13:59:32  reza
// Minor changes to make Efence work.
//
// Revision 1.5  1995/03/16  16:38:36  reza
// Updated byte transfer. Bytes are no longer transmitted in binary but in
// an ASCII string (0-255).
//
// Revision 1.4  1994/12/22  04:46:18  reza
// Updated to use DODS new attribute array capability.
//
// Revision 1.3  1994/12/08  19:01:27  reza
// Added support for local netCDF file access through the original netCDF code.
//
// Revision 1.2  1994/11/23  21:05:50  reza
// First working version.
//
// Revision 1.1  1994/11/03  04:35:44  reza
// Preliminary version of netCDF -> DODS library.
