require "numru/gphys/axis"

module NumRu

   class Grid

      def initialize( *axes )
	 @axes = Array.new
	 axes.each{|ag|
	    if ag.is_a?(Axis)
	       @axes.push(ag)
	    else
	       raise ArgumentError, "each argument must be an Axis"
	    end
	 }
	 @collapsed = Array.new   # Array of String
	 @rank = @axes.length
	 @axnames = Array.new
	 __check_and_set_axnames
      end

      def inspect
	 "<#{rank}D grid #{@axes.collect{|ax| ax.inspect}.join("\n\t")}>"
      end

      def __check_and_set_axnames
	 @axnames.clear
	 @axes.each{|ax| 
	    nm=ax.name
	    if @axnames.include?(nm)
	       raise "Two or more axes share a name: #{nm}"
	    end
	    @axnames.push(nm)
	 }
      end
      private :__check_and_set_axnames

      attr_reader :rank,
	          :axnames,
	          :collapsed

      def axis(i)
	 @axes[i]
      end

      def set_axis(i,ax)
	 @axes[i] = ax
      end

      def set_collapsed( collapsed )
	 @collapsed = collapsed   # Array of String
         self
      end
      def add_collapsed( collapsed )
	 @collapsed = @collapsed + collapsed   # Array of String
         self
      end

      def copy
	 # deep clone onto memory
	 out = Grid.new( *@axes.collect{|ax| ax.copy} )
	 out.set_collapsed( @collapsed.dup )
	 out
      end

      def shape
	 @axes.collect{|ax| ax.length}
      end
      alias shape_current shape

      def [] (*slicer)
	 if slicer.length == 0
	    # make a clone
	    axes = Array.new
	    (0...rank).each{ |i| axes.push( @axes[i][0..-1] ) }
	    Grid.new( *axes )
	 elsif slicer.length == rank
	    axes = Array.new
	    collapsed = Array.new
	    for i in 0...rank
	       ax = @axes[i][slicer[i]]
	       if ax.is_a?(Axis)      # else its rank became zero (collapsed)
		  axes.push( ax )
               else
		  collapsed.push( ax )
	       end
	    end
	    grid = Grid.new( *axes )
	    grid.set_collapsed( collapsed ) if collapsed.length != 0
	    grid
	 else
	    raise ArguemmentError, "# of the args does not agree with the rank"
	 end
      end

      def get_axis(dim_or_dimname)
	 ax_dim(dim_or_dimname)[0]
      end

      def dim_index(dim_or_dimname)
	 ax_dim(dim_or_dimname)[1]
      end

      def exclude(dim_or_dimname)
	 dim = dim_index(dim_or_dimname)
	 axes = @axes.dup
	 axes.delete_at(dim)
	 Grid.new( *axes )
      end

      def change_axis(dim, axis)
	 axes = @axes.dup
	 collapsed = Array.new
	 if axis.is_a?(Axis)
	    axes[dim] = axis
	 else
	    collapsed.push(axis) if axis.is_a?(String)
	    axes.delete_at(dim)
	 end
	 Grid.new( *axes ).add_collapsed( collapsed )
      end

      def change_axis!(dim, axis)
	 if axis.is_a?(Axis)
	    @axes[dim] = axis
	 else
	    @collapsed.push(axis) if axis.is_a?(String)
	    @axes.delete_at(dim)
	 end
	 @rank = @axes.length
	 __check_and_set_axnames
	 self
      end

      def insert_axis(dim, axis)
	 axes = @axes.dup
	 if axis.is_a?(Axis)
	    axes[dim+1,0] = axis    # axes.insert(dim, axis) if ruby 1.7
	 else
	    # do nothing
	 end
	 Grid.new( *axes )
      end

      def insert_axis!(dim, axis)
	 if axis == nil
	    # do nothing
	 else
	    @axes[dim+1,0] = axis    # @axes.insert(dim, axis) if ruby 1.7
	    @rank = @axes.length
	    __check_and_set_axnames
	 end
	 self
      end

      # Define operations along each axis --- The following defines
      # instance methods such as "mean" and "integ":

      def Grid.axis_operations_update
	 Axis.defined_operations(Grid).each do |method|
      	    eval <<-EOS
	    def #{method}(vary, dim_or_dimname, *extra_args)
               ax, dim = self.ax_dim(dim_or_dimname)
	       na, new_ax = ax.#{method}(vary, dim, *extra_args)

               [VArray.new(na, vary.attr), self.change_axis(dim, new_ax)]
	    end
	    EOS
         end
      end

      axis_operations_update

      ######### < protected methods > ###########

      protected

      def ax_dim(dim_or_dimname)
	 if dim_or_dimname.is_a?(Integer)
	    dim = dim_or_dimname
	    if dim < 0 || dim >= rank
	       raise ArgumentError,"rank=#{rank}: #{dim}th grid does not exist"
	    end
	 else
	    dim = @axnames.index(dim_or_dimname)
	    if !dim
	       raise ArgumentError, "Axis #{dim_or_dimname} is not contained"
	    end
	 end
	 [@axes[dim], dim]
      end


   end

end

######################################################
## < test >
if $0 == __FILE__
   include NumRu
   vx = VArray.new( NArray.float(10).indgen! + 0.5 ).rename("x")
   vy = VArray.new( NArray.float(6).indgen! ).rename("y")
   xax = Axis.new().set_pos(vx)
   yax = Axis.new(true).set_cell_guess_bounds(vy).set_pos_to_center
   xax.set_default_algorithms
   yax.set_default_algorithms
   grid = Grid.new(xax, yax)

   z = VArray.new( NArray.float(vx.length, vy.length).indgen! )
   p z.val
   p "average along x-axis:", grid.mean(z,0)[0].val, grid.mean(z,"x")[0].val
   p "average along y-axis:", grid.mean(z,1)[0].val, grid.mean(z,"y")[0].val
   p "grid set by an operation:", (g = grid.mean(z,1)[1]).rank, g.shape

   p grid.shape, grid.axis(0).pos.val, grid.axis(1).pos.val
   subgrid = grid[1..3,1..2]
   p subgrid.shape, subgrid.axis(0).pos.val, subgrid.axis(1).pos.val
   p grid[3,2].collapsed

   p grid

#scalar#   grid = Grid.new(xax)
#scalar#   p grid.mean(vx,0)

end

