#! /usr/bin/env ruby

#:title: LocaleFilter
#:main:  LocaleFilter
#
#= localefilter.rb -- Locale Filter
#
#Authors::   Michimasa Koichi <michi@ep.sci.hokudai.ac.jp>
#Copyright:: Copyright (C) Michimasa Koichi 2006-. All rights reserved.
#Document::  http://www.ep.sci.hokudai.ac.jp/~michi/pc/ruby/doc/localefilter/
#Download::  http://www.ep.sci.hokudai.ac.jp/~michi/pc/ruby/src/localefilter/
#
#== Summary
#
#ܸꤷʸɤǽϤ, Kconv Υեȥ.
#
#== Example
#
#=== Used as Object
#
#  require 'localefilter'
#
#  a = 'ƥ'
#  b = 'Ƥ'
#  lf = LocaleFilter.new
#  p lf.get_env
#  # => "Ķѿ locale"
#
#  lf.print a, "\n"
#  lf.puts b
#  # => ʸ "@locale"  "@device" ˽Ϥ
#
#  lf.locale = 'euc'
#  lf.stdout "ۤۤ\n"
#  # => ʸɤ "EUC" ꤷ "STDOUT" ˽
#
#  lf.set_device('stderr')
#  lf.puts a, b
#  # =>  "STDERR" ꤷƽ
#
#  file = File.open('/tmp/output.txt', 'w')
#  lf.device = file
#  lf.puts a, b
#  # =>  "file" ꤷƽ
#
#  lf.device = 'stdout'
#  file.close
#  # =>  "STDOUT" ᤹
#
#=== Used as Module
#
#LocaleFilter::PrintWrapper 򻲾.
#
#== Character Codes
#
#бʸɤϰʲ̤.
#(UTF-8, UTF-16  Ruby Version 1.8.2 ʹߤб)
#
#  * Shift_JIS
#  * EUC-JP
#  * JIS (ISO-2022-JP)
#  * UTF-8
#  * UTF-16
#
#== Methods
#
#=== Public Methods
#:: new
#::   print, puts
#::   locale, set_locale, locale=, device, set_device, device=
#Ѵ::   kconv
#
#=== Included from CodeConverter
#Ѵ::   txt2num, num2txt
#¾:: get_env
#
#=== Included from PrintWrapper
#::   print (-> stdout), stderr, puts (-> putsout), putserr, warn
#
#== Required Library
#
#* kconv (Standard Liblary)
#
#== Acknowlege
#
#* ͸ (rename.rb κ)
#  URL:: http://www.ep.sci.hokudai.ac.jp/~daktu32/DOC/ruby/works/ruby_works.htm
#
#*  (rename.rb Υƥ)
#  URL:: http://www.ep.sci.hokudai.ac.jp/~morikawa/
#
#==Future Plans
#* tee ɤ
#  * @device  STDOUT or STDERR Ʊ˽Ϥ᥽åɡ
#  * @tee_device ä鼫ưŪɲýϤȤ
#
#== Recent History
#
#* Ver. 2.4.3, 2007/10/05 by michi
#  * txt2num 
#
#* Ver. 2.4.2, 2007/05/19 by michi
#  * ɥȤ
#
#* Ver. 2.4.1, 2007/05/09 by michi
#  * CodeConverter βĻѹ
#
#* Ver. 2.4.0, 2006/11/16 by michi
#  * ƥ᥽åɤν°
#
#--
#
#* Ver. 2.3.0, 2006/11/10 by michi
#  * ϥ᥽åɷ⥸塼벽
#
#* Ver. 2.2.1, 2006/11/09 by michi
#  * txt2num : Kconv ʸˤϤΤޤ֤ͤѹ
#
#* Ver. 2.2.0, 2006/11/08 by michi
#  * warn ɲ
#
#++
#
class LocaleFilter
  require 'kconv'

  # LocaleFilter ΥС
  VERSION = '2.4.3'

  # LocaleFilter κǽ
  UPDATE  = '2007/10/05'

  #
  #== Summary
  #
  #ʸ̾ Kconv ʸѴ.
  #
  #== Methods
  #
  #===Module Mothods
  #Ѵ::   txt2num, num2txt
  #¾:: get_env
  #
  module CodeConverter
    #
    #ʸ̾ _txt_  Kconv ʸѴ֤.
    #
    def txt2num(txt)
      case txt.to_s.strip.chomp
      when /s(hift[-_])??jis/i, Kconv::SJIS.to_s
        num = Kconv::SJIS

      when /euc(-jp)??/i, /ujis/i, Kconv::EUC.to_s
        num = Kconv::EUC

      when /jis/i, /iso-2022-jp/i, Kconv::JIS.to_s
        num = Kconv::JIS

      when /utf(-)??8/i, Kconv::UTF8.to_s
        num = Kconv::UTF8

      when /utf(-)??16/i, Kconv::UTF16.to_s
        num = Kconv::UTF16

      when /bin(ary)??/i, Kconv::BINARY.to_s
        num = Kconv::BINARY

      when /ascii/i, Kconv::ASCII.to_s
        num = Kconv::ASCII

      else
        num = Kconv::UNKNOWN
      end
      return num
    end
    module_function :txt2num

    #
    #Kconv ʸ _num_ ʸ̾ʸѴ֤.
    #
    def num2txt(num)
      case num.to_s.to_i
      when Kconv::SJIS
        txt = 'Shift_JIS'

      when Kconv::EUC
        txt = 'EUC'

      when Kconv::JIS
        txt = 'JIS (ISO-2022-JP)'

      when Kconv::UTF8
        txt = 'UTF-8'

      when Kconv::UTF16
        txt = 'UTF-16'

      when Kconv::BINARY
        txt = 'BINARY'

      when Kconv::ASCII
        txt = 'ASCII'

      else
        txt = 'UNKNOWN'
      end
      return txt
    end
    module_function :num2txt

    #
    #Ķѿ _env_ 򸵤 Kconv ʸ֤.
    #
    def get_env(*env)
      array = env.flatten | %w(LC_ALL LC_MESSAGES LC_CTYPE LANG)
      array.each { |v|
        return txt2num(ENV[v])  if ENV[v]
      }
      return Kconv::UNKNOWN
    end
    module_function :get_env
  end

  #
  #== Summary
  #
  #ʽϥ᥽åɤ Kconv ǥե륿󥰤.
  #
  #== Description
  #
  #  include  print ʤɤνϥ᥽åɤؤ,
  #  ܸ줬üʸɤ˹碌Ѵ.
  #
  #  Object ǤȰ㤤,  STDOUT ˸ꤵ.
  #  IO#write ؤƤǤϤʤΤ
  #  STDOUT.print ʤɤˤϱƶϤʤ.
  #
  #== Example
  #
  #  require 'localefilter'
  #
  #  class Foo
  #    include LocaleFilter::PrintWrapper
  #    def bar
  #      print "ۤۤ", "\n"
  #    end
  #
  #    def baz
  #      STDOUT.print "ۤۤ", "\n"
  #    end
  #  end
  #
  #  foo = Foo.new
  #  foo.bar
  #  # => "ۤۤ" ü locale ѴƽϤ
  #
  #  foo.baz
  #  # => ʸɤѴϹԤʤ
  #
  #== Methods
  #
  #=== Public Methods
  #:: print, stderr, puts, putserr, warn
  #
  #=== Private Methods
  #Ѵ:: kconv
  #
  module PrintWrapper
    #
    #_arg_  CodeConverter#get_env ǼʸɤѴ, ֤.
    #
    def kconv(*arg)
      return false  if arg.to_s.empty?
      locale = CodeConverter.get_env
      array  = arg.flatten.map { |v|
        Kconv.kconv(v.to_s, locale)
      }
      return array
    end
    private :kconv

    #
    #_arg_  kconv Ѵ, <b>STDOUT.print</b> Ϥ.
    #
    def print(*arg)
      STDOUT.print kconv(arg)
    end

    #
    #_arg_  kconv Ѵ, <b>STDERR.print</b> Ϥ.
    #
    def stderr(*arg)
      STDERR.print kconv(arg)
    end

    #
    #_arg_  kconv Ѵ, <b>STDOUT.puts</b> Ϥ.
    #
    def puts(*arg)
      STDOUT.puts kconv(arg)
    end

    #
    #_arg_  kconv Ѵ, <b>STDERR.puts</b> Ϥ.
    #
    def putserr(*arg)
      STDERR.puts kconv(arg)
    end

    #
    #_arg_  kconv Ѵ, Ǹ˲ԤĤ <b>STDERR.print</b> Ϥ.
    #
    def warn(*arg)
      STDERR.print kconv(arg), "\n"
    end
  end

  include CodeConverter
  include PrintWrapper

  #
  #֥Ȥ. <b>@locale</b>, <b>@device</b> 
  #set_locale, set_device ǽ.
  #
  def initialize(locale = nil, device = 'stdout')
    set_locale(locale)
    set_device(device)
  end

  # CodeConverter Υ᥽åɤ Public ѹ
  public :num2txt, :txt2num, :get_env

  #
  #_arg_ ʸ <b>@locale</b> Ѵ, ֤.
  #
  def kconv(*arg)
    return false  if arg.to_s.empty?
    array = arg.flatten.map { |v|
      Kconv.kconv(v.to_s, @locale)
    }
    return array
  end

  #
  #PrintWrapper#print  alias
  #
  alias :stdout :print

  #
  #_arg_  kconv Ѵ, <b>@device</b> б print .
  #
  def print(*arg)
    @device.print kconv(arg)
  end

  #
  #PrintWrapper#puts  alias
  #
  alias :putsout :puts

  #
  #_arg_  kconv Ѵ, <b>@device</b> б puts .
  #
  def puts(*arg)
    @device.puts kconv(arg)
  end

  #
  #ʸ̾ _locale_  CodeConverter#txt2num Ѵ̤ <b>@locale</b> .
  #_locale_  <b>nil</b> ξ CodeConverter#get_env Ǽ.
  #
  def set_locale(locale = nil)
    if locale
      @locale = txt2num(locale)

    else
      @locale = get_env
    end
  end
  alias :locale= :set_locale

  #
  #<b>@locale</b>  CodeConverter#num2txt Ѵʸ֤̾.
  # set_locale ǹԤ.
  #
  def locale
    num2txt(@locale)
  end

  #
  #ϥǥХ̾ _device_  <b>@device</b> .
  #
  def set_device(device = 'stdout')
    case device
    when /(std)??out/i
      @device = STDOUT

    when /(std)??err(or)??/i
      @device = STDERR

    else
      @device = device
    end
  end
  alias :device= :set_device

  #
  #ϥǥХ̾ <b>@device</b> ֤.
  # set_device ǹԤ.
  #
  def device
    case @device
    when STDOUT
      return 'STDOUT'

    when STDERR
      return 'STDERR'

    else
      return @device
    end
  end
end
