! netcdf_io.f90 - netCDF abstract I/O
! vi: set ts=8 sw=4:
! Copyright (C) TOYODA Eizi, 2000.  All rights reserved. 

module netcdf_io

    use netcdf_slice
    implicit none

    !
    ! functions defined here return pointer to 6-dimensional array.
    !

    interface put_text
	module procedure NetcdfPutText
    end interface

    interface put_int
	module procedure NetcdfPutint
    end interface

    interface put_real
	module procedure NetcdfPutReal
    end interface

    interface get_text
	module procedure NetcdfGetText
    end interface

    interface get_int
	module procedure NetcdfGetInt
    end interface

    interface get_real
	module procedure NetcdfGetReal
    end interface

contains

    !
    ! --- OUTPUT ---
    !

    logical function NetcdfPutText(var, buf, limit) result(result)
	use dc_chars
	type(NC_VARIABLE), intent(in)::		var
	character, intent(in)::			buf(:, :, :, :, :, :)
	type(NC_LIMIT), intent(in), optional::	limit
	type(NC_LIMIT)::			mylimit
	integer::				d(NC_SLICE_DIMS), stat
    continue
	if (present(limit)) then
	    mylimit = limit
	else
	    mylimit = WholeVariable(var)
	endif
	d = count(limit)
	stat = nf_put_vars_text(var%file%id, var%id, start(mylimit), d, &
	    & stride(mylimit), char_6array_to_char(buf))
	if (stat /= 0) then
	    result = .FALSE.; return
	endif
	result = .TRUE.
    end function

    logical function NetcdfPutInt(var, buffer, limit) result(result)
	type(NC_VARIABLE), intent(in)::		var
	integer, intent(in)::			buffer(:, :, :, :, :, :)
	type(NC_LIMIT), intent(in), optional::	limit
	type(NC_LIMIT)::			mylimit
	integer::				d(NC_SLICE_DIMS), stat
    continue
	if (present(limit)) then
	    mylimit = limit
	else
	    mylimit = WholeVariable(var)
	endif
	d = count(limit)
	stat = nf_put_vars_int(var%file%id, var%id, start(mylimit), d, &
	    & stride(mylimit), reshape(buffer, (/size(mylimit)/)))
	if (stat /= 0) then
	    result = .FALSE.; return
	endif
	result = .TRUE.
    end function

    logical function NetcdfPutReal(var, buffer, limit) result(result)
	type(NC_VARIABLE), intent(in)::		var
	real, intent(in)::			buffer(:, :, :, :, :, :)
	type(NC_LIMIT), intent(in), optional::	limit
	type(NC_LIMIT)::			mylimit
	integer::				d(NC_SLICE_DIMS), stat
    continue
	if (present(limit)) then
	    mylimit = limit
	else
	    mylimit = WholeVariable(var)
	endif
	d = count(mylimit)
	stat = nf_put_vars_real(var%file%id, var%id, start(mylimit), d, &
	    & stride(mylimit), reshape(buffer, (/size(mylimit)/)))
	call NetcdfSaveError(stat, 'put_real', Fullname(var))
	if (stat /= 0) then
	    result = .FALSE.; return
	endif
	result = .TRUE.
    end function

    !
    ! --- INPUT ---
    !

    function NetcdfGetText(var, limit) result(result)
	character, pointer::			result(:, :, :, :, :, :)
	type(NC_VARIABLE), intent(in)::		var
	type(NC_LIMIT), intent(in), optional::	limit
	type(NC_LIMIT)::			mylimit
	integer::				stat
	integer, dimension(NC_SLICE_DIMS)::	d
    continue
	if (present(limit)) then
	    mylimit = limit
	else
	    mylimit = WholeVariable(var)
	endif
	d = count(mylimit)
	allocate(result(d(1), d(2), d(3), d(4), d(5), d(6)), stat=stat)
	if (stat /= 0) then
	    nullify(result); return
	endif
	call doit(Size(mylimit))
	if (stat /= 0) then
	    nullify(result); return
	endif
    contains
	subroutine doit(len)
	    use dc_chars
	    integer, intent(in)::		len
	    character(len = len)::		buffer
	continue
	    stat = nf_get_vars_text(var%file%id, var%id, start(mylimit), d, &
	    	& stride(mylimit), buffer)
	    result = char_to_char_6array(buffer, shape(result))
	end subroutine
    end function

    function NetcdfGetInt(var, limit) result(result)
	integer, pointer::			result(:, :, :, :, :, :)
	type(NC_VARIABLE), intent(in)::		var
	type(NC_LIMIT), intent(in), optional::	limit
	type(NC_LIMIT)::			mylimit
	integer::				stat
	integer, dimension(NC_SLICE_DIMS)::	d
	integer, pointer::			buffer(:)
    continue
	if (present(limit)) then
	    mylimit = limit
	else
	    mylimit = WholeVariable(var)
	endif
	d = count(mylimit)
	allocate(result(d(1), d(2), d(3), d(4), d(5), d(6)), &
	    & buffer(Size(mylimit)), stat=stat)
	if (stat /= 0) then
	    nullify(result); return
	endif
	stat = nf_get_vars_int(var%file%id, var%id, start(mylimit), d, &
	    & stride(mylimit), buffer)
	if (stat /= 0) then
	    nullify(result); return
	endif
    end function

    function NetcdfGetReal(var, limit) result(result)
	real, pointer::				result(:, :, :, :, :, :)
	type(NC_VARIABLE), intent(in)::		var
	type(NC_LIMIT), intent(in), optional::	limit
	type(NC_LIMIT)::			mylimit
	integer::				stat
	integer, dimension(NC_SLICE_DIMS)::	d, s, r
	real, pointer::				buffer(:)
    continue
	if (present(limit)) then
	    mylimit = limit
	else
	    mylimit = WholeVariable(var)
	endif
	d = count(mylimit)
	s = start(mylimit)
	r = stride(mylimit)
	allocate(result(s(1):d(1), s(2):d(2), s(3):d(3), s(4):d(4), &
	    & s(5):d(5), s(6):d(6)), buffer(size(mylimit)), stat=stat)
	if (stat /= 0) then
	    nullify(result); return
	endif
	if (size(mylimit) == 1) then
	    stat = nf_get_var1_real(var%file%id, var%id, s, &
		& result(s(1), s(2), s(3), s(4), s(5), s(6)))
	else
	    stat = nf_get_vars_real(var%file%id, var%id, s, d, r, buffer)
	    result = reshape(buffer, d)
	endif
	if (stat /= 0) then
	    call NetcdfSaveError(stat, 'GetReal', Fullname(var))
	    deallocate(result)
	    nullify(result)
	    goto 666
	endif
666	continue
	deallocate(buffer)
    end function

end module
