require "narray_miss"
require "numru/nusdas_params"
begin
  require "numru/nusdas_wrap"
rescue LoadError
  warn "NuSDaS wrap library is not loaded" if $DEBUG
end
require "date"


module NumRu

  module NuSDaSMod

    # radius of the earth
    R = 6371000 # [m]

    @@surf_suffix = "srf"

    @@ui2_max = (1<<16)-1
    @@si2_max = (1<<15)-1
    @@ui4_max = (1<<32)-1
    @@si4_max = (1<<31)-1
    @@ui8_max = (1<<64)-1
    @@si8_max = (1<<63)-1
    @@fnumber_max = 254

    private

    if [1].pack("s")[0] == 1
      def endian(nary)
        nary.hton
      end
    else
      def endian(nary)
        nary
      end
    end

    def get_sint2(str)
      if str.length == 2
        i = str.unpack("n")[0]
        return i > @@si2_max ? -1-(i^@@ui2_max) : i
      else
        len = str.length/2
        ary = Array.new(len)
        len.times{|n| ary[n] = get_sint2(str[n*2,2]) }
        return ary
      end
    end
    def get_sint4(str)
      if str.length == 4
        i = str.unpack("N")[0]
        return i > @@si4_max ? -1-(i^@@ui4_max) : i
      else
        len = str.length/4
        ary = Array.new(len)
        len.times{|n| ary[n] = get_sint4(str[n*4,4]) }
        return ary
      end
    end
    def get_sint8(str)
      if str.length == 8
        i1,i2 = str.unpack("N2")
        i = i1*@@ui4_max + i2
        return i > @@si8_max ? -1-(i^@@ui8_max) : i
      else
        len = str.length/8
        ary = Array.new(len)
        len.times{|n| ary[n] = get_sint8(str[n*8,8]) }
        return ary
      end
    end

    def get_record(io,flag_read=true)
      str = io.read(4)
      unless str.length==4
        return [nil,nil,nil]
      end
      len = str.unpack("N")[0]
      len = len - 8 if @meta[:version] == 10
      name = io.read(4)
      unless /\w\w\w\w/ =~ name
        return [nil,nil,nil]
      end
      len2 = io.read(4).unpack("N")[0]
      mtime = get_sint4(io.read(4))
      if flag_read
        str = io.read(len-12)
        str = str[0,len2-12]
#        str = str[8..len2-len-1]
        io.seek(4,IO::SEEK_CUR)
        return [name, mtime, str]
      else
        pos = io.pos
#        io.seek(len-8,IO::SEEK_CUR)
        return [name, mtime, pos, len2-12]
      end
    end

    def skip_record(io)
      len = io.read(4).unpack("N")[0]
      len = len - 8 if @meta[:version] == 10
      io.seek(len+4,IO::SEEK_CUR)
    end

    def time_from_basetime(vtime, btime, unit)
      t = vtime - btime
      case unit.upcase.rstrip
      when "MIN"
        # nothing to do
      when "MINUIT"
        # nothing to do
      when "HOUR"
        t /= 60
      when "DAY"
        t /= 1440 # 60*24
      when "MONT"
        t /= 43200 # 60*24*30
      else
        raise "sorry unit #{unit} is not supported"
      end
      return t
    end

    def time_from_zerotime(vtime, btime, unit)
      case unit.upcase.rstrip
      when "MIN"
        # nothing to do
      when "MINUIT"
        # nothing to do
      when "HOUR"
        vtime *= 60
      when "DAY"
        vtime *= 1440 # 60*24
      else
        raise "sorry unit #{unit} is not supported"
      end
      return vtime + btime
    end

  end

=begin rdoc
 handring NuSDaS data
 original method are defind as instans method
=end
  class NuSDaS

    include NuSDaSMod

    @@date0 = Date.new(1801,1,1)

    class << self

=begin rdoc
 get integer time (minutes from 00:00 1 Jan 1801)

  arguments:
   time_str: String (yyyymmddhhmm)
        e.g. "200712141630" for 16:30 14 Dec 2007
  return:
   time: Integer
=end
      def time_from_str(str)
        raise("arugment must be String") unless String === str
        raise("str length must be 12") unless str.length == 12
        year = str[0,4].to_i
        mon = str[4,2].to_i
        day = str[6,2].to_i
        hour = str[8,2].to_i
        min = str[10,2].to_i
        return ( (Date.new(year,mon,day)-@@date0).to_i*24 + hour )*60 + min
      end
=begin rdoc
 get string time

  arguments:
   time: Integer (minutes from 00:00 1 Jan 1801)
  return:
   time_str: String (yyyymmddhhmm)
        e.g. "200712141630" for 16:30 14 Dec 2007
=end
      def str_from_time(time)
        raise("arugment must be String") unless Integer === time
        date = @@date0 + time/(60*24)
        year = date.year
        mon = date.mon
        day = date.day
        i = time%(24*60)
        hour = i/60
        min = i%60
        return  "%4d%02d%02d%02d%02d"%[year,mon,day,hour,min]
      end

=begin rdoc
 judge that the path is NuSDaS root path

  arguments:
   path: String
  return:
   true or false
=end
      def is_a_NuSDaS?(path)
        File.directory?(path)
#        File.directory?(path) &&
#          ( File.exist?(File.join(path,"nusdas_def")) || File.exist?(File.join(path,"NUSDAS_DEF")) )
      end

      alias :open :new

    end

 # NuSDaS root path
    attr_reader :root

 # type1, type2, type3
    attr_reader :type1, :type2, :type3

 # debug switch
    attr_accessor :debug

=begin rdoc
 create NuSDaS object

  arguments:
   path: String (NuSDaS root directory)
   not_use_nusdas_def: boolean (optional, default=false)
  return:
   NuSDaS
=end
    def initialize(path, not_use_nusdas_def=false)
      @debug |= $DEBUG
      unless NuSDaS.is_a_NuSDaS?(path)
        raise "path must be NuSDaS root directory"
      end
      @root = path
      @meta = Hash.new
      flag = false
      unless not_use_nusdas_def
        flag = true if fnames = parse_nusdas_def
      end
      fnames = path unless flag
      unless search_dir(fnames, flag)
         warn "No NuSDaS files are found"
         @file = false
         return nil
#        raise "No NuSDaS files are found"
      end
      @file = true
      @meta[:basetimes].sort!
      @meta[:nbasetime] = @meta[:basetimes].length
      unless flag
        [:members, :validtimes, :planes, :planes2, :elements].each{|name|
          @meta[name].sort!.flatten!
        }
        @meta[:validtimes] = NArray.to_na(@meta[:validtimes])
        @meta[:nmember] = @meta[:members].length
        @meta[:nvalidtime] = @meta[:validtimes].length
        @meta[:nplane] = @meta[:planes].length
        @meta[:nelement] =  @meta[:elements].length
      end
      names = [:basetimes, :elements, :planes, :validtimes, :members]
      ind_hash = Hash.new
      @meta[:files].each_with_index{|hash,i|
        names.each{|name|
          ary = @meta[name].to_a
          v = hash[name]
          if v==ary
            hash[name] = true
          else
            unless h = ind_hash[name]
              ind_hash[name] = h = Hash.new
              ary.each_with_index{|vv,i| h[vv] = i}
            end
            if Array === v || NArray === v
              hash[name] = v.collect{|el| h[el]}
            else
              hash[name] = h[v]
            end
          end
        }
      }
      np = @meta[:nplane]
      nt = @meta[:nvalidtime]
      nm = @meta[:nmember]
      ne = @meta[:nelement]
      nb = @meta[:nbasetime]
      na = NArray.byte(ne, np, nt, nm, nb).fill(@@fnumber_max+1)
      ind = NArray.int(ne, np, nt, nm, nb).indgen
      @meta[:files].each_with_index{|h,i|
        slice = [h[:elements],h[:planes],h[:validtimes],h[:members],h[:basetimes]]
        pos = h[:record_pos]
        mask = pos.ne(0) & pos.ne(-1)
        na[ind[*slice][mask]] = i
      }
      @meta[:fnumber] = na
    end

=begin rdoc
 close all files

  return:
   nil
=end
    def close
      if @file
        @meta[:files].each{|h| h[:file].close}
      end
      @meta = nil
      return nil
    end

=begin rdoc
 inquire all variable names

 return:
  var_names: Array
=end
    def var_names
      return [] unless @file
      vars = Array.new
      if (isurf = @meta[:planes].index("SURF  "))
        ielse = NArray.sint(@meta[:nplane]).indgen.to_a
        ielse.delete(isurf)
      end
      @meta[:elements].each_with_index{|elm,i|
        fnumber = @meta[:fnumber][i,true,true,true,true]
        elm = elm.rstrip
        if isurf
          if fnumber[ielse,true,true,true].le(@@fnumber_max).count_true > 0
            vars.push elm
          end
          if fnumber[isurf,true,true,true].le(@@fnumber_max).count_true > 0
            vars.push elm+@@surf_suffix
          end
        else
          if fnumber.le(@@fnumber_max).count_true > 0
            vars.push elm
          end
        end
      }
      return vars
    end

=begin rdoc
 get NuSDaSVar object

  arguments:
   vname: String (variable name)

  return:
   var: NuSDaSVar
=end
    def var(element)
      unless String === element
        raise "vname must be String"
      end
      if /^(.*)#{@@surf_suffix}$/ =~ element
        element = $1
        surf = true
      else
        surf = false
      end
      element = element.ljust(6)
      if @meta[:elements].include?(element)
        return NuSDaSVar._new(element,@meta,self,surf)
      else
        raise "variable '#{element}' is not found"
      end
    end


=begin rdoc
 get path
=end
    def path
      @root
    end

=begin rdoc
 get infromation
=end
    def inspect
      if debug
        "#<NuSDaS: root='#{root}', type1='#{type1}', type2='#{type2}', type3='#{type3}'\n  "+@meta.collect{|k,v| "#{k}=>#{v.inspect}"}.join(",\n  ")+">"
      else
        "#<NuSDaS: root='#{root}', type1='#{type1}', type2='#{type2}', type3='#{type3}'>"
      end
    end

    private

    def parse_nusdas_def
      dname = File.join(@root, "nusdas_def")
      unless File.exist?(dname)
        dname = File.join(@root, "NUSDAS_DEF")
      end
      return nil unless File.exist?(dname)
      fname = Dir[File.join(dname,"*def")]
      if fname.length > 2
        raise "There are multiple nusdas def files"
      end
      fname = fname[0]
      params = Hash.new
      File.open(fname){|file|
        line = file.gets
        while(line)
          ary = line.chop.split
          if ary[0]
            name = ary[0].downcase.to_sym
            if params[name]
              unless Array === params[name][0]
                params[name] = [ params[name] ]
              end
              params[name].push(val=ary[1..-1])
            else
              params[name] = val = ary[1..-1]
            end
	  end
          line = file.gets
          while /^\s/ =~ line
	    val.push *line.chop.split
            line = file.gets
          end
        end
      }

      if val = params[:type1]
        @type1 = val.join.ljust(8)
        @meta[:projection] = @@projection_abbrs[val[1]]
      else
        raise 'type1 must be exist in nusdas_def'
      end

      if val = params[:type2]
        @type2 = val.join.ljust(4)
      else
        raise 'type2 must be exist in nusdas_def'
      end

      if val = params[:type3]
        @type3 = val[0].ljust(4)
      else
        raise 'type3 must be exist in nusdas_def'
      end

      if val = params[:size]
        @meta[:size] = [val[0].to_i, val[1].to_i]
      else
        raise 'size must be exist in nusdas_def'
      end

      if val = params[:distance]
        @meta[:distance] = [val[0].to_f, val[1].to_f]
      else
        @meta[:distance] = [0.0, 0.0]
      end

      if val = params[:basepoint]
        @meta[:basepoint] = [val[0].to_f, val[1].to_f]
        bp21 = /E$/ =~ val[2] ? val[2][0..-2] : val[2]
        bp20 = /E$/ =~ val[3] ? val[3][0..-2] : val[3]
        @meta[:basepoint2] = [bp20.to_f, bp21.to_f]
      else
        @meta[:basepoint] = [0.0, 0.0]
        @meta[:basepoint2] = [0.0, 0.0]
      end

      if val = params[:others]
        s1 = /E$/ =~ val[0] ? val[0][0..-2] : val[0]
        s0 = /N$/ =~ val[1] ? val[1][0..-2] : val[1]
        @meta[:others] = [s0.to_f, s1.to_f]
        s1 = /E$/ =~ val[2] ? val[2][0..-2] : val[2]
        s0 = /N$/ =~ val[3] ? val[3][0..-2] : val[3]
        @meta[:others2] = [s0.to_f, s1.to_f]
      else
        @meta[:others] = [0.0, 0.0]
        @meta[:others2] = [0.0, 0.0]
      end

      if val = params[:standard]
        s1 = /E$/ =~ val[0] ? val[0][0..-2] : val[0]
        s0 = /N$/ =~ val[1] ? val[1][0..-2] : val[1]
        @meta[:standard] = [s0.to_f, s1.to_f]
        s1 = /E$/ =~ val[2] ? val[2][0..-2] : val[2]
        s0 = /N$/ =~ val[3] ? val[3][0..-2] : val[3]
        @meta[:standard2] = [s0.to_f, s1.to_f]
      else
        @meta[:standard] = [0.0, 0.0]
        @meta[:standard2] = [0.0, 0.0]
      end

      if val = params[:creator]
        @meta[:creator] = val.join(" ")
      else
        @meta[:creator] = 'Japan Meteorological Agency, http://www.jma.go.jp/'
      end

      if val = params[:plane]
        @meta[:nplane] = val[0].to_i
      else
        raise 'plane must be exist in nusdas_def'
      end

      if val = params[:plane1]
        @meta[:planes] = val.collect{|s| s.ljust(6)}
      else
        raise 'plane1 must be exist in nusdas_def'
      end

      if val = params[:plane2]
        @meta[:planes2] = val.collect{|s| s.ljust(6)}
      else
        @meta[:planes2] = @meta[:plenes]
      end

      if val = params[:element]
        i = val[0].to_i
        @meta[:nelement] = i
        @meta[:elements] = Array.new
      else
        raise 'element must be exist in nusdas_def'
      end

      if val = params[:elementmap]
        if Array === val[0]
          len = val.length
          @meta[:elements] = ary = Array.new(len)
          len.times{|n| ary[n] = val[n][0].ljust(6) }
        else
          @meta[:elements] = [val[0].ljust(6)]
        end
      else
        raise 'elementmap must be exist in nusdas_def'
      end
      if @meta[:nelement] != @meta[:elements].length
        raise "number of elementmap is not same as element in nusdas_def"
      end

      if val = params[:information]
        if Array === val[0]
          len = val.length
          @meta[:info] = ary = Array.new
          len.times{|n| ary[n] = val[n][0] }
        else
          @meta[:info] = [val[0]]
        end
      else
        @meta[:info] = Array.new
      end
      @meta[:ninfo] = @meta[:info].length

      if val = params[:subcntl]
        @meta[:nsubc] = val[0].to_i
      else
        @meta[:nsubc] = 0
      end

      if val = params[:member]
        @meta[:nmember] = n = val[0].to_i
        params[:member_io] = val[1].upcase

        val = params[:memberlist]
        @meta[:members] = val.collect{|s| s.ljust(4)}
        if n != @meta[:members].length
          raise "number of memberlist is not same as member in nusdas_def"
        end
      else
        @meta[:nmember] = 1
        @meta[:members] = ["    "]
      end

      if val = params[:validtime]
        @meta[:nvalidtime] = val[0].to_i
        if /^(IN|OUT)$/ =~ val[1].upcase
          params[:validtime_io] = val[1]
          @meta[:validtime_unit] = val[2]
        elsif /^(IN|OUT)$/ =~ val[2].upcase
          params[:validtime_io] = val[2]
          @meta[:validtime_unit] = val[1]
        else
          raise "validtime is invalid in nusdas_def"
        end
      else
        raise 'validtime must be exist in nusdas_def'
      end

      if val = params[:validtime1]
        case val[0]
        when "ARITHMETIC"
          @meta[:validtimes] = NArray.int(@meta[:nvalidtime]).indgen(val[1].to_i,val[2].to_i)
        when "ALL_LIST"
          if @meta[:nvalidtime] != val.length-1
            raise "number of validtime1 is not same as validtime in nusdas_def"
          end
          na = NArray.int(@meta[:nvalidtime])
          val[1..-1].each_with_index{|s,i| na[i] = s.to_i}
          @meta[:validtimes] = na
        else
          raise "validtime1 is invalid in nusdas_def"
        end
      else
        raise 'validtime1 must be exist in nusdas_def'
      end

      if val = params[:validtime2]
        na = NArray.int(@meta[:nvalidtime])
        ft = val[0].to_i
        if ft > 0
          if @meta[:nvalidtime] != val.length
            raise "number of validtime2 is not same as validtime in nusdas_def"
          end
          val.each_with_index{|s,i| na[i] = s.to_i}
        else
          na.fill(-ft)
        end
        @meta[:validtimes2] = na
      else
        @meta[:validtimes2] = NArray.int(@meta[:nvalidtime]).fill(1)
      end

      if val = params[:missing]
        @meta[:missing] = val[0]
      else
        @meta[:missing] = 'NONE'
      end

      if val = params[:packing]
        @meta[:packing] = val[0]
      else
        @meta[:packing] = '2UPC'
      end

      if val = params[:value]
        @meta[:value] = val[0]
      else
        @meta[:value] = 'PVAL'
      end


      if path = params[:path]
        case path[0].upcase
        when 'NWP_PATH_S'
          fpat = File.join(@root,'_3d_name','_validtime')
        when 'NWP_PATH_BS'
          fpat = File.join(@root,'_3d_name','_basetime')
        when 'NWP_PATH_M'
          fpat = File.join(@root,'_3d_name','_member','_validtime')
        when 'NWP_PATH_VM'
          fpat = File.join(@root,'_3d_name','_member')
        when 'RELATIVE_PATH'
          fname = params[:filename]
          rpath = path[1]
          fpat = File.join(@root,rpath)
          fpat = File.join(fpat,fname[0]) if fname
        else
          raise "path is invalide in nusdas_def"
        end
      else
        fname = params[:filename]
        rpath = "_model/_attribute/_space/_time/_name/_basetime"
        rpath = File.join(rpath,"_member") if params[:member_io] == "OUT"
        rpath = File.join(rpath,"_validtime") if params[:validtime_io] == "OUT"
        fpat = File.join(@root,rpath)
        fpat = File.join(fpat,fname[0]) if fname
      end

      ary = Array.new
      fpat.sub!(/_model/,@type1[0,4])
      fpat.sub!(/_space/,@type1[-4,4])
      fpat.sub!(/_2d/,@type1[4,2])
      fpat.sub!(/_3d/,@type1[6,2])
      fpat.sub!(/_attribute/,@type2[0,2])
      fpat.sub!(/_time/,@type2[-2,2])
      fpat.sub!(/_name/,"%4.4s"%(@type3.strip))
      fpat.sub!(/_basetime/,'*')
      fpat.sub!(/_validtime/,'*')
      if /_member/ =~ fpat
        mem = @meta[:members]
        len = mem.length
        fpats = Array.new(len)
        len.times{|n| fpats[n] = fpat.sub(/_member/, mem[n]) }
      else
        fpats = [fpat]
      end

      ary = Array.new
      fpats.each{|s|
        s.gsub!(/\s/,'_')
        ary += Dir[s]
      }
      return ary
    end

    def search_dir(fnames, nusdef=false)
      fnames = Dir[File.join(fnames,'*')] unless nusdef
      flag = false
      fnames.each{|fname|
        next if File.basename(fname).upcase == "NUSDAS_DEF"
	next if /\.tar\.gz/ =~ fname
        if File.directory?(fname)
          flag = search_dir(fname) || flag
        elsif File.file?(fname)
          hash = Hash.new
          file = File.open(fname,"rb")
          hash[:file] = file
          get_version(file) unless @meta[:version] 
          name, mtime, str = get_record(file)
	  unless name == "NUSD"
            warn("NUSD record is invalid") if $DEBUG
	    file.close
	    next
	  end
          parse_nusd(str) unless  nusdef

          name, mtime, str = get_record(file)
          unless name == "CNTL"
            warn("CNTL record is invalid") if $DEBUG
            file.close
            next
          end
          hash.update parse_cntl(str, nusdef)
          name, mtime, str = get_record(file)
          case name
          when "INDX"
            hash.update parse_indx(str,hash)
          when "INDY"
            hash.update parse_indy(str,hash)
          else
            warn("INDX or INDY record is invalid") if $DEBUG
            file.close
            next
          end

          flag2 = false
          @meta[:nsubc].times{|i|
            name, mtime, str = get_record(file)
            unless name == "SUBC"
              file.close
              flag2 = true
              break
            end
            @meta[:subc] ||= Hash.new
            @meta[:subc].update parse_subc(str,hash)
          }
          next if flag2
          files_push(hash, nusdef)
          flag = true
        end
      }

      return flag
    end

    def files_push(hash, nusdef=false)
      names = [:basetimes]
      unless nusdef
        names += [:members, :planes, :planes2, :validtimes, :elements]
#        names += [:members, :planes, :planes2, :validtimes, :validtimes2, :elements]
      end
      names.each{|name|
        v = hash[name]
	flag = NArray === v
        v = v.to_a if flag
	ary = @meta[name].to_a || Array.new
        if Array === v
          if v.length > 1
            raise "BUG"
          end
          v = v[0]
        end
        ary.push(v) unless ary.include?(v)
        ary = NArray.to_na(ary) if flag
        @meta[name] = ary
      }
      [:nmember, :nplane, :nvalidtime, :nelement].each{|name|
        hash.delete(name)
      }

      a = @meta[:files] ||= Array.new
      i = a.length
      if @@fnumber_max < i
        raise "exceed maximum file number"
      end
      a.push hash
    end

    def parse_nusd(str)
      @meta[:creator] = str[0,72].gsub(/\000/,'')
      # flength_oct = get_sint8(str[72,8])
      # flength = str[84,4].unpack("N")[0]
      # nrecord = str[88,4].unpack("N")[0]
      @meta[:ninfo] = str[92,4].unpack("N")[0]
      @meta[:nsubc] = str[96,4].unpack("N")[0]
      return hash
    end

    def get_version(file)
      file.pos = 96      
      version = get_sint4(file.read(4))
      unless (version == 11 || version == 13 || version == 10)
        warn "NuSDaS vsersion is #{version}"
        warn "this version is not suported"
#        raise "this version is not suported"
      end
      @meta[:version]= version
      file.rewind
      return true
    end
      
    def parse_cntl(str,nusdef)
      hash = Hash.new
      @type1 ||= str[0,8]
      @type2 ||= str[8,4]
      @type3 ||= str[12,4]
#      hash[:basetime_str] = str[16,12]
      hash[:basetimes] = get_sint4(str[28,4])
      @meta[:validtime_unit] ||= str[32,4]
      hash[:nmember] = nm = str[36,4].unpack("N")[0]
      hash[:nvalidtime] = nt = str[40,4].unpack("N")[0]
      hash[:nplane] = np = str[44,4].unpack("N")[0]
      hash[:nelement] = ne = str[48,4].unpack("N")[0]
      unless nusdef
        @meta[:projection] ||= str[52,4]
        @meta[:size] ||= str[56,8].unpack("N2")
        @meta[:basepoint] ||= str[64,8].unpack("g2")
        @meta[:basepoint2] ||= str[72,8].unpack("g2")
        @meta[:distance] ||= str[80,8].unpack("g2")
        @meta[:standard] ||= str[88,8].unpack("g2")
        @meta[:standard2] ||= str[96,8].unpack("g2")
        @meta[:others] ||= str[104,8].unpack("g2")
        @meta[:others2] ||= str[112,8].unpack("g2")
        @meta[:value] ||=  str[120,4]
        # str[124,8]
        # str[132,24]
      end
      str = str[156..-1]
      hash[:members] = ary = Array.new(nm)
      nm.times{|i| ary[i] = str[i*4,4] }
      str = str[nm*4..-1]
      hash[:validtimes] = endian( NArray.to_na(str[0,nt*4], NArray::INT) )
      hash[:validtimes] = time_from_basetime(hash[:validtimes], hash[:basetimes], @meta[:validtime_unit])
      str = str[nt*4..-1]
#      hash[:validtimes2] = endian( NArray.to_na(str[0,nt*4], NArray::INT) )
      str = str[nt*4..-1]
      hash[:planes] = ary = Array.new(np)
      np.times{|i| ary[i] = str[i*6,6] }
      str = str[np*6..-1]
      hash[:planes2] = ary = Array.new(np)
      np.times{|i| ary[i] = str[i*6,6] }
      str = str[np*6..-1]
      hash[:elements] = ary = Array.new(ne)
      ne.times{|i| ary[i] = str[i*6,6] }
      return hash
    end

    def parse_indx(str,meta)
      ne = meta[:nelement]
      np = meta[:nplane]
      nt = meta[:nvalidtime]
      nm = meta[:nmember]
      hash = Hash.new
      na = endian( NArray.to_na(str[0,4*nm*nt*np*ne], NArray::INT, ne, np, nt, nm) )
      mask = na.lt(-1)
      if mask.count_true > 0
        na_long = NArray.object(ne, np, nt, nm)
        na_long[true,true,true,true] = na
        na_long[mask] = (1<<32)+na_long[mask]
        na = na_long
      end
      hash[:record_pos] = na
      return hash
    end

    def parse_indy(str,meta)
      ne = meta[:nelement]
      np = meta[:nplane]
      nt = meta[:nvalidtime]
      nm = meta[:nmember]
      hash = Hash.new
      size = nm*nt*np*ne
      na = endian( NArray.to_na(str[0,8*size], NArray::INT, 2, ne, np, nt, nm) )
      na_long = NArray.object(2, ne, np, nt, nm)
      mask = na.lt(0)
      na_long[true,true,true,true,true] = na
      na_long[mask] = (1<<32)+na_long[mask]
      na_long = na_long[0,false]*(1<<32) + na_long[1,false]
      mask = na_long.ge(1<<63)
      na_long[mask] = na_long[mask]-(1<<64)
      hash[:record_pos] = na_long
#      hash[:record_len] = endian( NArray.to_na(str[8*size,4*size], NArray::INT, ne, np, nt, nm) )
#      hash[:record_elmnum] = endian( NArray.to_na(str[12*size,4*size], NArray::INT, ne, np, nt, nm) )
      return hash
    end

    def parse_subc(str,meta)
      name = str[0,4]
      str = str[4..-1]
      hash = Hash.new
      case name
      when "ETA "
        hash[:nplane] = (n=str[0,4].unpack("N")[0])
        str = str[4..-1]
        hash[:a] = endian( NArray.to_na(str[0,4*(n+1)], NArray::SFLOAT) )
        str = str[4*(n+1)..-1]
        hash[:b] = endian( NArray.to_na(str[0,4*(n+1)], NArray::SFLOAT) )
        str = str[4*(n+1)..-1]
        hash[:c] = str[0,4].unpack("g")[0]
      when "TDIF"
        nmember = meta[:nmember]
        nvalidtime = meta[:nvalidtime]
        nmember = meta[:nmember]
        hash[:tdif] = endian( NArray.to_na(str[0,4*nmember*nvalidtime], NArray::INT, nvalidtime, nmember) )
        str = str[4*nmember*nvalidtime..-1]
        hash[:tint] = endian( NArray.to_na(str[0,4*nmember*nvalidtime], NArray::INT, nvalidtime, nmember) )
      when "DELT"
        hash[:delta] = get_sint4(str[0,4])
      when "ZHYB"
        hash[:nz] = (n=get_sint4(str[0,4]))
        hash[:ptrf] = str[4,4].unpack("g")[0]
        hash[:presrf] = str[8,4].unpack("g")[0]
        hash[:zrp] = endian( NArray.to_na(str[12,4*n], NArray::SFLOAT) )
        str = str[12+4*n..-1]
        hash[:zrw] = endian( NArray.to_na(str[0,4*n], NArray::SFLOAT) )
        str = str[4*n..-1]
        hash[:vctrans_p] = endian( NArray.to_na(str[0,4*n], NArray::SFLOAT) )
        str = str[4*n..-1]
        hash[:vctrans_w] = endian( NArray.to_na(str[0,4*n], NArray::SFLOAT) )
        str = str[4*n..-1]
        hash[:dvtrans_p] = endian( NArray.to_na(str[0,4*n], NArray::SFLOAT) )
        str = str[4*n..-1]
        hash[:dvtrans_w] = endian( NArray.to_na(str[0,4*n], NArray::SFLOAT) )
      when "RGAU"
        hash[:j] = get_sint4(str[0,4])
        hash[:j_start] = get_sint4(str[4,4])
        hash[:j_n] = (n=get_sint4(str[8,4]))
        hash[:i] = endian( NArray.to_na(str[12,4*n], NArray::INT) )
        str = str[12+4*n..-1]
        hash[:i_start] = endian( NArray.to_na(str[0,4*n], NArray::INT) )
        str = str[4*n..-1]
        hash[:i_n] = endian( NArray.to_na(str[0,4*n], NArray::INT) )
        str = str[4*n..-1]
        hash[:lat] = endian( NArray.to_na(str[0,4*n], NArray::SFLOAT) )
      when "RADR"
        hash[:info] ||= Array.new
        hash[:info].push endian( NArray.to_na(str[0,4*@nmember*@nvalidtime*@nplane*@nelement], NArray::INT, @nelement, @nplane, @nvalidtime, @nmember) )
      when "ISPC"
      when "RADS"
        hash[:mode] = endian( NArray.to_na(str[0,4*@nmember*@nvalidtime*@nplane*@nelementw], NArray::INT, @nelement, @nplane, @nvalidtime, @nmember) )
        str = str[4*@nmember*@nvalidtime*@nplane*@nelementw..-1]
        hash[:flag] = endian( NArray.to_na(str[0,4*@nmember*@nvalidtime*@nplane*@nelementw], NArray::INT, @nelement, @nplane, @nvalidtime, @nmember) )
        str = str[4*@nmember*@nvalidtime*@nplane*@nelementw..-1]
        hash[:n10] = endian( NArray.to_na(str[0,4*@nmember*@nvalidtime*@nplane*@nelementw], NArray::INT, @nelement, @nplane, @nvalidtime, @nmember) )
        str = str[4*@nmember*@nvalidtime*@nplane*@nelementw..-1]
        hash[:b] = endian( NArray.to_na(str[0,4*@nmember*@nvalidtime*@nplane*@nelementw], NArray::INT, @nelement, @nplane, @nvalidtime, @nmember) )
        str = str[4*@nmember*@nvalidtime*@nplane*@nelementw..-1]
        hash[:beta] = endian( NArray.to_na(str[0,4*@nmember*@nvalidtime*@nplane*@nelementw], NArray::INT, @nelement, @nplane, @nvalidtime, @nmember) )

      end
      return {name => hash}
    end

  end # class NuSDaS



  class NuSDaSVar

    include NuSDaSMod


    class << self

      alias :_new :new #:nodoc:

=begin rdoc
 create NuSDaSVar object

  arguments:
   file: NuSDaS
   vname: String

  return:
   NuSDaSVar
=end
      def new(file, vname)
        raise(ArgumentError, "file must be NuSDaS") unless NuSDaS === file
        raise(ArgumentError, "file must be NuSDaS") unless String === vname
        file.var(vname)
      end
    end

 # variable name
    attr_reader :name
 # NuSDaS
    attr_reader :nusdas
 # debug switch
    attr_accessor :debug

    def initialize(element,meta,nusdas,surf=false) #:nodoc:
      unless String===element && Hash===meta && NuSDaS===nusdas
        raise "arugument is invalid"
      end
      @debug = nusdas.debug
      meta = meta.dup
      @type1 = nusdas.type1
      @type2 = nusdas.type2
      @type3 = nusdas.type3
      @name = surf ? element.rstrip+@@surf_suffix : element.rstrip
      @meta = Hash.new
      @nusdas = nusdas
      unless (ie = meta[:elements].index(element.ljust(6)))
        warn "element is invalid"
        return nil
      end
      if (is = meta[:planes].index("SURF  "))
        if surf
          ip = is..is
        else
          ielse = NArray.sint(meta[:nplane]).indgen.to_a
          ielse.delete(is)
          ip = ielse
        end
      else
        raise("surf is true but planes does not have SURF") if surf
        ip = true
      end
      meta.each{|k,v|
        case k
        when :files
          len = v.length
          ary = Array.new(len)
          v.each_with_index{|h,i|
            ha = Hash.new
            [:file, :members, :validtimes, :basetimes].each{|name|
              ha[name] = h[name]
            }
            if ip == true # is==nil && surf==false
              ha[:planes] = h[:planes]
            else
              if surf
                ha[:planes] = (h[:planes]==true||h[:planes].include?(is)) ? [0] : nil
              else
                if h[:planes] == true
                  ha[:planes] = NArray.sint(meta[:nplane]-1).indgen.to_a
                else
                  ha[:planes] = h[:planes].collect{|ipp| ipp > is ? ipp-1 : ipp}
                end
              end
            end
            ha[:record_pos] = h[:record_pos][ie,ip,true,true]
            ary[i] = ha
          }
          @meta[:files] = ary
        when :fnumber
          @meta[k] = v[ie,ip,true,true,true]
          if @meta[k].le(@@fnumber_max).count_true == 0
            raise "#{element} has no data"
          end
        when :planes
          @meta[k] = ip == true ? v : v.values_at(*ip)
        when :nplane
          @meta[k] = ip == true ? v : surf ? 1 : v-1
        else
          if Array===v || NArray===v || Hash===v
            @meta[k] = v.dup
          else
            @meta[k] = v
          end
        end
      }

      @attr = set_attrs(@meta)
    end

=begin rdoc
 name
=end
    def name
      @name
    end

=begin rdoc
 total data array size

  return:
   total: Integer
=end
    def total
      if @meta[:projection] == "RG  "
        parm = @meta[:subc]["RGAU"]
        raise("parameters RGAU in SUBC do not exist") unless parm
        size = parm[:i_n].max * parm[:j_n]
      else
        size = @meta[:size]
        size = size[0]*size[1]
      end
      size*@meta[:nplane]*@meta[:nvalidtime]*@meta[:nmember]*@meta[:nbasetime]
    end
    alias length total

=begin rdoc
 rank of data array

  return:
   rank: Integer
=end
    def rank
      i = 2
      i += 1 if @meta[:nplane] > 1
      i += 1 if @meta[:nvalidtime] > 1
      i += 1 if @meta[:nmember] > 1
      i += 1 if @meta[:nbasetime] > 1
      return i
    end

=begin rdoc
 shape of data array

  return:
   shape: Array
=end
    def shape
      if @meta[:projection] == "RG  "
        parm = @meta[:subc]["RGAU"]
        raise("parameters RGAU in SUBC do not exist") unless parm
        ary = [parm[:i_n].max, parm[:j_n]]
      else
        ary = @meta[:size].dup
      end
      ary.push @meta[:nplane] if @meta[:nplane] > 1
      ary.push @meta[:nvalidtime] if @meta[:nvalidtime] > 1
      ary.push @meta[:nmember] if @meta[:nmember] > 1
      ary.push @meta[:nbasetime] if @meta[:nbasetime] > 1
      return ary
    end

=begin rdoc
 dimension names

  return:
   dim_anmes: Array
=end
    def dim_names
      case @meta[:projection].rstrip
      when "LL", "GS", "RG", "MER", "NPS", "SPS", "LMN", "LMS", "OL"
        ary = ["lon", "lat"]
      when "YP"
        ary = ["lat", "z"]
      else
        ary = ["x", "y"]
      end
      if @meta[:nplane] > 1
        case @type1[6,2]
        when "PP", "ET"
          ary.push "pressure"
        when "SG"
          ary.push "sigma"
        when "ZZ", "ZS"
          ary.push "z"
        when "TH"
          ary.push "theta"
        else
          ary.push "plane"
        end
      end
      ary.push "validtime" if @meta[:nvalidtime] > 1
      ary.push "member" if @meta[:nmember] > 1
      ary.push "basetime" if @meta[:nbasetime] > 1
      return ary
    end

=begin rdoc
 get array of the dimension

  arguments:
   dname: String (dimension name)
   arg: Hash
     type => symbol
       :reference: return reference value (default)
       :full: return full value
       :original: return original value
  return:
   dim: NArray  
=end
    def dim(dname)
      if Integer === dname
        dname = dim_names[dname]
      elsif String === dname
        unless dim_names.include?(dname)
          raise "#{dname} does not exist"
        end
      else
        raise "argument is invalide"
      end
      return NuSDaSDim.new(dname, @meta, @type1[6,2])
    end

=begin rdoc
 get data array

  arguments:
   index: Array[dimension length] (array of index)
  return:
   value: NArray
=end
    def get(*index)
      if @meta[:projection] == "RG  "
        parm = @meta[:subc]["RGAU"]
        raise("parameters RGAU in SUBC do not exist") unless parm
        nx = parm[:i_n].max
        ny = parm[:j_n]
      else
        nx, ny = @meta[:size]
      end
      np = @meta[:nplane]
      nt = @meta[:nvalidtime]
      ne = @meta[:nmember]
      nb = @meta[:nbasetime]
      case index.length
      when 0
        x = nx==1 ? [[(ix=0)], 1, true] : [(ix=true), nx, false]
        y = ny==1 ? [[(iy=0)], 1, true] : [(iy=true), ny, false]
        p = np==1 ? [[0], 1, true] : [0...np, np, false]
        t = nt==1 ? [[0], 1, true] : [0...nt, nt, false]
        e = ne==1 ? [[0], 1, true] : [0...ne, ne, false]
        b = nb==1 ? [[0], 1, true] : [0...nb, nb, false]
      when rank
        x = nx==1 ? [[(ix=0)], 1, true] : parse_index((ix=index.shift), nx, true)
        y = ny==1 ? [[(iy=0)], 1, true] : parse_index((iy=index.shift), ny, true)
        p = np==1 ? [[0], 1, true] : parse_index(index.shift, np)
        t = nt==1 ? [[0], 1, true] : parse_index(index.shift, nt)
        e = ne==1 ? [[0], 1, true] : parse_index(index.shift, ne)
        b = nb==1 ? [[0], 1, true] : parse_index(index.shift, nb)
      else
        raise "number of index is invalid"
      end

      if @meta[:projection] == "RG  "
        lon = dim("x")
        ind = NArray.int(nx,ny)
        if NArrayMiss === lon
          mask = lon.mask
          ind[mask] = NArray.int(mask.count_true).indgen
          mask = mask[ix,iy]
        else
          ind.indgen
          mask = nil
        end
        xorg = x
        yorg = y
        ind = ind[ix,iy]
        x = [ind[0], ind[-1]-ind[0]+1]
        y = [[0], 1]
        nx, ny = @meta[:size]
      end
      ary = NArray.byte(x[1], y[1], p[1], t[1], e[1], b[1])

      b[0].each_with_index{|l,ll|
        basetime = @meta[:basetimes][l]
        e[0].each_with_index{|m,mm|
          member = @meta[:members][m]
          t[0].each_with_index{|n,nn|
            validtime = @meta[:validtimes][n]
            p[0].each_with_index{|k,kk|
              plane = @meta[:planes][k]
              fn = @meta[:fnumber][k,n,m,l]
              if fn > @@fnumber_max
                ary = NArrayMiss.to_nam_no_dup(ary) unless NArrayMiss===ary
                ary.set_invalid(true,true,kk,nn,mm,ll)
                next
              end
              h = @meta[:files][fn]
              file = h[:file]
              kkk = h[:planes]==true ? k : h[:planes].index(k)
              nnn = h[:validtimes]==true ? n : h[:validtimes].to_a.index(n)
              mmm = h[:members]==true ? m : h[:members].index(m)
              file.pos = h[:record_pos][kkk,nnn,mmm]
              name, mtime, pos, len = get_record(file,false)
#              data = parse_data(str)
              data = parse_data(file, pos, len)
              data[:validtime] = time_from_basetime(data[:validtime], basetime, @meta[:validtime_unit])
#              p( [member, validtime, plane, [nx,ny]], [data[:member], data[:validtime], data[:plane], data[:size]])
              unless [member, validtime, plane, [nx,ny]] == [data[:member], data[:validtime], data[:plane], data[:size]]
                raise "data is not consistent"
              end
              sub = unpack_data(data[:data], x[0], x[1], y[0], y[1], data[:packing], data[:missing], data[:size])
              ary = ary.to_type(sub.typecode) if sub.typecode > ary.typecode
              if NArrayMiss===sub && !(NArrayMiss===ary)
                ary = NArrayMiss.to_nam_no_dup(ary)
              end
              ary[true,true,kk,nn,mm,ll] = sub
            }
          }
        }
      }

      if @meta[:projection] == "RG  "
        shape = [ind[0], ind[1]] + ary.shape[2..-1]
        ary2 = ary.class.new(ary.typecode, *shape)
        if mask
          if NArray === ary
            mask2 = NArray.byte(*shape)
            mask2[true,true,true,true,true,true] = mask.reshape!(shape[0],shape[1],1,1,1,1)
            ary2[mask2] = ary
            ary = NArrayMiss.to_nam(ary2, mask2)
          else
            mask2 = ary2.get_mask!
            mask2[true,true,true,true,true,true] = mask.reshape!(shape[0],shape[1],1,1,1,1)
            ary2[mask2] = ary
            ary = ary2
          end
        end
        x = xorg
        y = yorg
      end


      shape = []
      shape.push x[1] unless x[2]
      shape.push y[1] unless y[2]
      shape.push p[1] unless p[2]
      shape.push t[1] unless t[2]
      shape.push e[1] unless e[2]
      shape.push b[1] unless b[2]
      ary.reshape!(*shape)

      return ary
    end
    alias :[] :get

=begin rdoc
  attributs names

   return:
    att_names: Array
=end
    def att_names
      @attr.keys
    end

=begin rdoc
  get attribute value

   arguments:
    name: String
   return:
    value
=end
    def att(name)
      @attr[name]
    end

=begin rdoc
  get information
=end
    def inspect
      if debug
        "#<NuSDaSVar: name='#{name}', root='#{@nusdas.root}', type1='#{@type1}', type2='#{@type2}', type3='#{@type3}'\n  "+@meta.collect{|k,v| "#{k}=>#{v.inspect}"}.join(",\n  ")+">"
      else
        "#<NuSDaSVar: name='#{name}', root='#{@nusdas.root}', type1='#{@type1}', type2='#{@type2}', type3='#{@type3}'>"
      end
    end

=begin rdoc
  get NArray typecode
=end
    def typecode
      inds = Array.new(shape.length){0}
      inds[0] = 0..0
      get(*inds).typecode
    end

    private

    def parse_data(io, pos, len)
      h = Hash.new
      io.pos = pos
      str = io.read(48)
      h[:member] = str[0,4]
      h[:validtime] = get_sint4(str[4,4])
      h[:validtime2] = get_sint4(str[8,4])
      h[:plane] = str[12,6]
      h[:plane2] = str[18,6]
      h[:element] = str[24,6]
      # str[30,2]
      h[:size] = str[32,8].unpack("N2")
      h[:packing] = str[40,4]
      h[:missing] = str[44,4]
      h[:data] = [io, io.pos, len-48]
      return h
    end

    def unpack_data(io_pos, x, nx, y, ny, pakking, missing, size)
      io, pos, len = io_pos

      case pakking.rstrip
      when "1PAC", "I1"
        natype = NArray::BYTE
        byte = 1
        type = "I1"
      when "2PAC", "2UPC", "N1I2", "I2"
        natype = NArray::SINT
        byte = 2
        type = "I2"
      when "4PAC", "I4"
        natype = NArray::INT
        byte = 4
        type = "I4"
      when "R4"
        natype = NArray::SFLOAT
        byte = 4
        type = "R4"
      when "R8"
        natype = NArray::DFLOAT
        byte = 8
        type = "R8"
      when "RLEN"
        raise "sorry, not suported"
      else
        raise "pakking is invalid"
      end

      case missing
      when "UDFV"
        mstr = io.read(byte)
        len -= byte
        pos += byte
        case type
        when "I1"
          miss = mstr.unpack("C")[0]
        when "I2"
          if pakking == "2UPC"
            miss = mstr.unpack("n")[0]
          else
            miss = get_sint2(mstr)
          end
        when "I4"
          miss = get_sint4(mstr)
        when "R4"
          miss = mstr.unpack("g")[0]
        when "R8"
          miss = mstr.unpack("G")[0]
        else
          raise "bug"
        end
      when "MASK"
        total = size[0]*size[1]
        n = (total-1)/8+1
        mstr = io.read(n)
        len -= n
        pos += n
        mask = NArray.byte(total)
        mask[true] = mstr.unpack("B*")[0][0,total].unpack("c*")
        mask -= 48
      when "NONE"
        # do nothing
      else
        raise "missing is invalid"
      end

      case pakking
      when "1PAC", "2PAC", "2UPC"
        str = io.read(8)
        len -= 8
        pos += 8
        b = str[0,4].unpack("g")[0]
        a = str[4,4].unpack("g")[0]
      when "4PAC"
        str = io.read(16)
        len -= 16
        pos += 16
        b = str[0,8].unpack("G")[0]
        a = str[8,8].unpack("G")[0]
      when "RLEN"
        str = io.read(12)
        nbit = str[0,4].unpack("N")[0]
        maxv = str[4,4].unpack("N")[0]
        num = str[8,4].unpack("N")[0]
        len = (nbit*num-1)/8+1
        pos += 12
      end

      io.pos = pos

      if missing == "MASK"
        ary = endian( NArray.to_na(io.read(len), natype) )
      else
        ary = endian( NArray.to_na(slice(io,len,x,nx,y,ny,size,byte), natype) )
      end

      if missing == "UDFV"
        mask = ary.ne(miss)
      end

      case pakking
      when "1PAC", "2PAC", "2UPC"
        ary = ary.to_type(NArray::SFLOAT)
        if pakking == "2UPC"
          ma = ary.lt(0)
          ary[ma] = 65536 + ary[ma]
        end
        ary *= a
        ary += b
      when "4PAC"
        ary = ary.to_type(NArray::DFLOAT)
        ary *= a
        ary += b
      when "N1I2"
        ary = ary.to_type(NArray::SFLOAT)
        ary /= 10
      when "RLEN"
        # still undefined
        error
      end

      case missing
      when "UDFV"
        ary = NArrayMiss.to_nam_no_dup(ary, mask)
      when "MASK"
        ary2 = NArray.new(natype, size[0]*size[1])
        ary2[mask] = ary
        ary = NArrayMiss.to_nam_no_dup(ary2, mask)
        x = Integer === x ? x..(x+nx-1) : x
        y = Integer === y ? y..(y+ny-1) : y
        ary.reshape!(size[0], size[1])
        ary = ary[x,y]
      end

      ary.reshape!(nx, ny)

      return ary
    end

    def slice(io, len, x, nx, y, ny, size, byte)
      x = true if x==0 && nx==size[0]
      y = true if y==0 && ny==size[1]

      pos = io.pos

      return io.read(len) if x == true && y == true

      y = 0 if y == true

      if x == true
        if Array === y
          newstr = ""
          for j in y
            j0 = j*size[0]*byte
            io.pos = pos+j0
            newstr << io.read(size[0]*byte)
          end
          return newstr
        else
          j0 = y*size[0]*byte
          io.pos = pos+j0
          return io.read(size[0]*ny*byte)
        end
      else
        if y === true
          y = 0...size[1]
        elsif !(Array === y)
          y = y...(y+ny)
        end
        newstr = ""
        if Array === x
          for j in y
            i0 = j*size[0]*byte
            x.each{|i|
              io.pos = pos + i0+i*byte
              newstr << io.read(byte)
            }
          end
          return newstr
        else
          for j in y
            i0 = (j*size[0] + x)*byte
            io.pos = pos + i0
            newstr << io.read(nx*byte)
          end
          return newstr
        end
      end
    end

    def parse_index(arg, size, flag=false)
      shrink = false
      case arg
      when true
        d0 = flag ? true : 0
        nd = size
      when Integer
        d0 = arg < 0 ? size+arg : arg
        nd = 1
        shrink = true
      when Range
        d0 = arg.first
        d0 = size + d0 if d0 < 0
        d1 = arg.last
        d1 -= 1 if arg.exclude_end?
        d1 = size + d1 if d1 < 0
        nd = d1 - d0 + 1
      when Array
        d0 = arg
        nd = arg.length
        if arg.max > size-1
          raise "index exceed size-1"
        end
      else
        raise "index is invalid: #{arg.class.to_s}"
      end
      if nd > size
        raise "index is invalid"
      end
      if flag || Array===d0
        return [d0, nd, shrink]
      else
        return [d0..(d0+nd-1), nd, shrink]
      end
    end

    def set_attrs(meta)
      attr = Hash.new
      attr["creator"] = meta[:creator]
      attr["model_name"] = @@models[@type1[0,4]]
      attr["vertical_grid"] = @@vertical_grids[@type1[6,2]]
      attr["data_attribute"] = @@data_attributes[@type2[0,2]]
      attr["time_attribute"] = @@time_attributes[@type2[2,2]]
      attr["projection"] = @@projections[meta[:projection].rstrip]
      return attr
    end

  end # class NuSDaSVar

  class NuSDaSDim
    include NuSDaSMod
    include Math

    def initialize(name, meta, plane_type)
      @name = name
      @meta = meta
      @type = :reference
      @plane_type = plane_type

      @attr = Hash.new

      case @meta[:projection].rstrip
      when "LL", "GS", "RG", "MER", "NPS", "SPS", "LMN", "LMS", "OL"
        case name
        when "lon"
          @name2 = "x"
          @attr["units"] = "degrees_east"
          @attr["long_name"] = "longitude"
        when "lat"
          @name2 = "y"
          @attr["units"] = "degrees_north"
          @attr["long_name"] = "latitude"
        end
      when "YP"
        case name
        when "lat"
          @name2 = "x"
          @attr["units"] = "degrees_north"
          @attr["long_name"] = "latitude"
        when "z"
          @name2 = "y"
          @attr["units"] = "m"
          @attr["long_name"] = "height"
        end
      end
      if !@name2 && @meta[:nplane] > 1
        case plane_type
        when "PP", "ET"
          @name2 = "plane" if @name == "pressure"
          @attr["units"] = "hPa"
          @attr["long_name"] = "pressure_level"
        when "SG"
          @name2 = "plane" if @name == "sigma"
        when "ZS", "ZZ"
          @name2 = "plane" if @name == "z"
          @attr["units"] = "m"
          @attr["long_name"] = "height"
        when "TH"
          @name2 = "plane" if @name == "theta"
          @attr["units"] = "K"
          @attr["long_name"] = "theta_level"
        end
      end
      if @name2 == "plane"
        @attr["coordinate"] = @@vertical_grids[plane_type]
      end

      @name2 ||= @name

      if @name2 == "validtime"
        @attr["units"] = @meta[:validtime_unit].downcase
      end
      if @name2 == "basetime"
        @attr["units"] = "minutes since 1801-01-01 00:00:0.0"
      end

    end

=begin rdoc
  set type for val

   arguments:
     type: :reference, :full, or :original
=end
    def set_type(type)
      unless [:reference, :full, :original].include?(type)
        raise "type is invalid"
      end
      @type = type
    end

=begin rdoc
  dimension values

   arguments:
     type: (optional)

   return:
    val: NArray
=end
    def val(type=nil)
      if type
        unless [:reference, :full, :original].include?(type)
          raise "type is invalid"
        end
      else
        type = @type
      end
      case @name2
      when "x", "y"
        x,y = get_xy(type==:reference)
        case @name2
        when "x"
          return x
        when "y"
          return y
        end
      when "plane"
        case type
        when :reference, :full
	  if @plane_type == "ZS"
	    subc = @meta[:subc]["ZHYB"]
	    if type == :full
	      return [subc[:zrp], subc[:zrw], subc[:vctrans_p], subc[:vctrans_w], subc[:dvtrans_p], subc[:dvtrans_w]]
	    elsif type == :reference
	      return subc[:zrp]
	    end
          elsif @plane_type == "ET"
	    subc = @meta[:subc]["ETA "]
	    if type == :full
	      return [subc[:a], subc[:b], subc[:c]]
	    elsif type == :reference
	      return (subc[:a][0..-3] + subc[:a][1..-2])/2.0 + (subc[:b][0..-3]+subc[:b][1..-2])/2.0 * ( 1000.0 - subc[:c] )
	    end
	  else
            return NArray.to_na(@meta[:planes].collect{|pl| pl.to_i})
	  end
        when :original
          return NArray.to_na(@meta[:planes])
        end
      when "validtime"
        case type
        when :reference
          return @meta[:validtimes]
        when :full, :original
          nv = @meta[:nvalidtime]
          nb = @meta[:nbasetime]
          na = NArray.int(nv,nb)
          bts = @meta[:basetimes]
          vts = @meta[:validtimes]
          unit = @meta[:validtime_unit]
          nb.times{|j|
            bt = bts[j]
            nv.times{|i|
              na[i,j] = time_from_zerotime(vts[i], bt, unit)
            }
          }
          na.reshape!(nv) if nb==1
          return na
        end
      when "member"
        case type
        when :reference, :full
          return NArray.sint(@meta[:nmember]).indgen
        when :original
          return NArray.to_na(@meta[:members])
        end
      when "basetime"
        return NArray.to_na(@meta[:basetimes])
      end
    end

=begin rdoc
  attributs names

   return:
    att_names: Array
=end
    def att_names
      @attr.keys
    end

=begin rdoc
  get attribute value

   arguments:
    name: String
   return:
    value
=end
    def att(name)
      @attr[name]
    end

    def inspect
      "<NuSDaSDim: name='#{@name}'>"
    end

    private
    def get_xy(reference)
      imax, jmax = @meta[:size]
      case @meta[:projection]
      when "FG  "
      when "GS  "
        di, dj = @meta[:distance]
        bi, bj = @meta[:basepoint]
        by, bx = @meta[:basepoint2]
        dum,n = @meta[:standard]
        if n==0
          n = (180.0/dj).round
        end
        xi = NArray.sfloat(imax).indgen(1) - bi
        lon = bx + xi*di
        glat = gausslat(n)
        yj0 = ((n+1).to_f/2-bj).round
        lat = glat[yj0..(yj0+jmax-1)]
        return [lon, lat]
      when "LL  "
        di, dj = @meta[:distance]
        bi, bj = @meta[:basepoint]
        by, bx = @meta[:basepoint2]
        xi = NArray.sfloat(imax).indgen(1) - bi
        yj = NArray.sfloat(jmax).indgen(1) - bj
        lon = bx + xi*di
        lat = by - yj*dj
        return [lon, lat]
      when "LMN "
        dx, dy = @meta[:distance]
        bi, bj = @meta[:basepoint]
        by, bx = @meta[:basepoint2]
        sy,sx = @meta[:standard]
        sy2, = @meta[:standard2]
        unless bx.abs < 180
          raise "parameter for basepoint is invalide"
        end
        unless dx != 0 && dy != 0
          raise "parameter for distance is invalide"
        end
        unless sx.abs < 180
          raise "parameter for standard is invalide"
        end
        unless 0 < sy.abs && sy.abs < sy2.abs && sy2.abs < 90 && sy*sy2>0
          raise "parameter for standard is invalide"
        end
        bx = bx*PI/180
        by = by*PI/180
        sx = sx*PI/180
        sy = sy*PI/180
        sy2 = sy2*PI/180
        xi = NArray.sfloat(imax).indgen(1) - bi
        yj = NArray.sfloat(jmax).indgen(1) - bj
        x = xi*dx
        y = -yj*dy 
        n = log(cos(sy)/cos(sy2)) / log(tan(PI/4+sy2/2)/tan(PI/4+sy/2))
        f = cos(sy) * tan(PI/4+sy/2)**n * R / n
        rho0 = f / tan(PI/4+by/2)**n
        if reference
          theta = NMath::atan(x/(rho0-by))
          rho = NMath::sqrt(bx**2+(rho0-y)**2)
        else
          y.reshape!(1,jmax)
          theta = NMath::atan(x/(rho0-y))
          rho = NMath::sqrt(x**2+(rho0-y)**2)
        end
        rho = -rho if n < 0
        lon = bx + theta/n
        lat = 2*NMath::atan((f/rho)**(1.0/n))-PI/2
        lon = lon*180/PI
        lat = lat*180/PI
        return [lon,lat]
      when "LMS "
      when "MER "
        dx, dy = @meta[:distance]
        bi, bj = @meta[:basepoint]
        by, bx = @meta[:basepoint2]
        sy, = @meta[:standard]
        unless bx.abs < 180
          raise "parameter for basepoint is invalide"
        end
        xi = NArray.sfloat(imax).indgen(1) - bi
        yj = NArray.sfloat(jmax).indgen(1) - bj
        r = R*cos(sy*PI/180)
        lon = bx + xi*dx/r*180/PI
        y = -yj*dy
        lat = NMath::asin( NMath::tanh(atanh(sin(by*PI/180)) + y/r) )*180/PI
        return [lon, lat]
      when "OL  "
      when "NPS ", "SPS "
        dx, dx = @meta[:distance]
        bi, bj = @meta[:basepoint]
        by, bx = @meta[:basepoint2]
        sy, sx = @meta[:standard]
        unless bx.abs < 180
          raise "parameter for basepoint is invalide"
        end
        unless dx != 0 && dy != 0
          raise "parameter for distance is invalide"
        end
        unless sx.abs < 180
          raise "parameter for standard is invalide"
        end
        bx = bx*PI/180
        by = by*PI/180
        sx = sx*PI/180
        sy = sy*PI/180
        xi = NArray.sfloat(imax).indgen(1) - bi
        yj = NArray.sfloat(jmax).indgen(1) - bj
        r0 = 2*R*cos(sy)*tan((PI/2-by)/2)
        if reference
          lon = NMath::asin(xi*dx/r0 + sin(bx-sx)) + sx
          r = yj*dy/cos(bx-sx) + r0
          lat = PI/2 - NMath::atan(r/(2*R*cos(sy)))*2
        else
          rsin = xi*dx + r0*sin(bx-sx)
          rcos = yj*dy + r0*cos(bx-sx)
          r = NMath::sqrt(rsin**2+rcos.reshape!(1,jmax)**2)
          lon = NMath::asin(rsin/r) + sx
          lat = PI/2 - NMath::atan(r/(2*R*cos(sy)))*2
        end
        lon = lon*180/PI
        lat = lat*180/PI
        return [lon,lat]
      when "RD  "
      when "RG  "
        bi, dum = @meta[:basepoint]
        dum, bx = @meta[:basepoint2]
        parm = @meta[:subc]["RGAU"]
        raise("parameters RGAU in SUBC do not exist") unless parm
        i = parm[:i]
        i_n = parm[:i_n]
        j_n = parm[:j_n]
        i_start = parm[:i_start]
        i_n_max = i_n.max
        lon = NArrayMiss.sfloat(i_n_max,j_n)
        lat = NArrayMiss.sfloat(i_n_max,j_n)
        itotal = 0
        j_n.times{|jj|
          xi = NArray.sfloat(i_n[jj]).indgen(i_start[jj])
          lon[0...i_n[jj],jj] = (xi-bi)*360.0/i[jj] + bx
          lat[0...i_n[jj],jj] = parm[:lat][jj]
        }
        if lon.get_mask!.count_false == 0
          lon = lon.get_array!
          lat = lat.get_array!
        end
        if reference
          lon = lon.mean(1)
          lat = lat.mean(0)
        end
        return [lon,lat]
      when "RT  "
      when "SB  "
      when "ST  "
      when "XX  "
      when "YP  "
      else
        raise "projection is invalid"
      end
      warn "sorry, projection of '#{@meta[:projection]}' is not suported"
      xi = NArray.sint(imax).indgen(1)
      yj = NArray.sint(jmax).indgen(1)
      return [xi, yj]
    end

    def gausslat(n)
      # this method was written by Y.Kitamura
      # and modified by S.Nishizawa

      @@gausslat ||= Hash.new

      return @@gausslat[n] if @@gausslat[n]

      glat    = NArray.sfloat(n)
#      gweight = NArray.sfloat(n)

      eps = Float::EPSILON*8

      0.upto(n/2-1) do |i|
        y = Math::sin(Math::PI*(n+1-2*(i+1))/(2*n+1))
        tmp = 1.0
        while ((tmp/y).abs > eps)
          p0 = 0.0
          p1 = 1.0
          1.step(n-1, 2) do |j|
            p0 = ((2*j-1)*y*p1 - (j-1)*p0)/j
            p1 = ((2*j+1)*y*p0 - j*p1)/(j+1)
          end
          p2 = n*(p0 - y*p1)/(1.0 - y*y)
          tmp = p1/p2
          y = y - tmp
        end
        glat[i]      =  y
        glat[n-i-1] = -y
#        gweight[i]      = 1.0/(p2*p2*(1.0 - glat[i]*glat[i]))
#        gweight[n-i-1] = gweight[i]
      end

      glat =  NMath::asin(glat)*180/Math::PI

      @@gausslat[n] = glat

      return glat
    end

  end # class NuSDaSDim


end # module NumRu


if $0 == __FILE__


f = NumRu::NuSDaS.new("data.nus")
p f
p f.var_names
p v = f.var("_WFMLLLY/FCSV/STD1/200712051200/T")
p v.dim_names
p v.dim("x")
p v.dim("y")
p v.dim("plane")
p v.dim("t")
p v.dim("member")
p v.get(true,true,true,true,0)
p v.get(true,true,true,0,0)
p v.get(true,true,0,0,0)
p v.get(true,0..3,0,0,0)

f.close


end
