#!/usr/bin/perl
#
# dcppt.pl - PowerPoint 用 HTML 作成スクリプト
#
    @MAINTAINERS = ('Yasuhiro MORIKAWA <morikawa＠gfd-dennou.org>',
		   );
    $UPDATE      = '$Date: 2015/01/09 09:23:15 $';
    $VERSION     = '$Revision: 1.4 $';
    $URL         =
    'http://www.gfd-dennou.org/library/cc-env/dcppt/SIGEN.htm';
#
#== 概要
#
# PowerPoint から作成した PNG(などの) 画像群を見やすい
# スライド形式, およびサムネイル形式の HTML を作成するスクリプト
#
#== 使用法
#
# $ dcppt.pl -r とすること. 出力が長いのでパイプでページャに渡すと
# 良いでしょう. 例) $ dcppt.pl -r | jless
#
#
#== バグ?
#
# * Page 001, Page 002 はいらないかも
#
#== 履歴
#
# 以下のエントリは CVS のコミットログを元に, CVS によって自動的に生成
# されます. 従って, 以下の部分は基本的に手動で編集しないいように注意し
# てください.  (ログのフォーマットがずれてしまった場合には手動で修正し
# てください).
#
# $Log: dcppt.pl,v $
# Revision 1.4  2015/01/09 09:23:15  murashin
# * fix character encoding name.
#
# * 文字エンコーディング名を直した
#
# Revision 1.3  2015/01/09 09:16:46  murashin
# * use utf-8 for default character encoding instead of euc-jp
# * use Jcode and Getopt::Std instead of jcode.pl and getopts.pl
#
# * デフォルト文字コードとしてEUC-JPの代わりにUTF-8を使うようにした
# * jcode.plとgetopts.plの代わりにJcodeとGetopt::Stdを使うようにした
#
# Revision 1.2  2006/03/05 03:22:13  morikawa
# * "スライド13.png" などのファイル名は自動的に "slide13.png" に改名する.
# * デバッグメッセージの出力方式修正.
# * ファイル名の文字コードを修正するサブルーチン内で, "." や ".." をリネー
#   ムしようとするバグを修正. (今まではリネームがうまくいかないのでたま
#   たま表面化しなかった)
# * Config.dcppt を Emacs で開いた際, 自動的に cperl-mode で開くように修正.
#
# Initial 2005/08/01
#  * epppt <http://www.ep.sci.hokudai.ac.jp/~morikawa/perl/epppt/SIGEN_PUB.htm>
#    を dcppt と改名し電脳ツールとして cc-env に置く.
#

##################################################
#####                                        #####
##                 初期設定                    ###
#####                                        #####
##################################################
#
# 日本語処理のため, Jcode を組み込む.
use Jcode;

# オプション処理のため, getopts を組み込む.
use Getopt::Std;

# 情報源ディレクトリ名
$SRC_DIR = 'src';

# 設定ファイル名
$CONFIG_FILE = "$SRC_DIR/Config.dcppt";

# 画像サイズ変換コマンド
$CONVERT = 'convert';
$CONVERT_OPT = '-geometry';

# デバッグモード
$DEBUG = 0;

##################################################
#####                                        #####
##                 引数識別                    ###
#####                                        #####
##################################################

# c, d のみ引数をとる.
getopts('c:d:vVxrRmMDq');

# 引数が無い場合はヘルプ表示
unless ($opt_x || $opt_c || $opt_d ||
	$opt_v || $opt_V || $opt_r || $opt_R || $opt_m || $opt_M ||
	$opt_D || $opt_q) {
    &Help;
    exit 1;
}
# D オプションの場合は $DEBUG を真に
if ($opt_D) {
    $DEBUG = 1;
}
# q オプションの場合は $QUIET を真に
if ($opt_q) {
    $QUIET = 1;
} else {
    $QUIET = 0;
}
# 引数が v や V の場合にはバージョン情報だけ表示
if ($opt_v || $opt_V) {
    &PrintVersion;
    exit 1;
}
# 引数が r や R の場合には README.TXT をコピーしたものを出力
if ($opt_r || $opt_R) {
    &PrintReadme;
    exit 1;
}
# 引数が m や M の場合には menu.txt のテンプレートを出力
if ($opt_m || $opt_M) {
    &PrintMenuTxt("$SRC_DIR/menu.txt", $DEBUG);
    exit 1;
}
# c オプションの場合は $CONFIG_FILE 変更
if ($opt_c) {
    $CONFIG_FILE = $opt_c;
}

# d オプションの場合は $SRC_DIR 変更
if ($opt_d) {
    $SRC_DIR = $opt_d;
    # 末尾の "/" を排除しておく
    $SRC_DIR =~ s|/+$||;
}

sub Help() {
    print STDOUT <<EOF;
  dcppt.pl:
    USAGE:
      dcppt.pl [-vVrRmM] [-x] [-q] [-c configfile] [-d srcdir] [-D]

    OPTION:
      -v, -V        : バージョンナンバーを表示
      -r, -R        : README.TXT を表示
      -m, -M        : menu.txt テンプレートを出力
      -c configfile : 設定ファイルを明示して実行
      -d srcdir     : 画像ファイル置場を指定 (デフォルトは $SRC_DIR)
      -D            : デバッグモード (開発者用)
      -x            : 実行 (設定ファイルは $CONFIG_FILE)
      -q            : 問い合わせ無しで実行 (デフォルトで処理)

EOF
    &PrintVersion;
}
sub PrintVersion() {
    print STDOUT <<EOF;
  dcppt.pl Version $VERSION, Last Update: $UPDATE.

EOF
    foreach $MAINTAINER (@MAINTAINERS) {
	print STDOUT "  $MAINTAINER \n";
    }
    print STDOUT "    All Right Reserved.\n\n";
}

sub PrintMenuTxt() {
    local($menu_txt, $debug) = @_;
    print STDOUT "何ページ分の $menu_txt を出力しますか? [30]: ";
    $pagenum = <STDIN>;
    # 例外処理
    $pagenum = "30" unless ($pagenum =~ /^\d+$/);
    # 既にファイルがある場合は, 別ファイルを出力
    local($i) = 0;
    while (-f $menu_txt) {
	$i++;
	$menu_txt = "$menu_txt"."."."$i";
    }
    # $menu_txt がディレクトリも含む場合はそれらを作成する.
    local(@menu_path) = split(/\//, $menu_txt);
    local($dirpath) = "";
    local($first) = 1;
    while($#menu_path > 0){
	local($dir) = shift(@menu_path);
	$dirpath = $dir if $first;
	$dirpath .= '/' . "$dir" unless $first;
	mkdir($dirpath) unless -d $dirpath;
	print STDOUT "  mkdir $dirpath \n" if ($debug);
	$first = 0;
    }
    # ファイルオープン & 値の出力
    print STDOUT "Creating $menu_txt ...";
    open(MENUTXT, "> $menu_txt")
	|| die "エラー: $menu_txt に書き込むことが出来ませんでした. /n";
    local($j) = 1;
    for (1..${pagenum}) {
	local($outnum) = sprintf "%03d:\n", $j;
	print MENUTXT "$outnum";
	$j++;
    }
    close(MENUTXT);
    print STDOUT "done\n";
}

##################################################
#####                                        #####
##             設定ファイル確認                ###
#####                                        #####
##################################################

unless (-f $CONFIG_FILE) {
    unless ($QUIET) {
	print STDOUT "設定ファイルである $CONFIG_FILE が見当たりません.\n".
	             "  1. dcppt デフォルトの設定で処理をするなら \"y\",\n".
		     "  2. 処理を中断するなら                     \"n\",\n".
		     "  3. $CONFIG_FILE テンプレートを出力するなら\"m\"\n".
		     "を入力してください [Y/n/m]: ";
	$ans = <STDIN>;
    } else {
	$ans = "";
    }

    if ($ans =~ /^[mM].*$/) {
	print STDOUT "Creating $CONFIG_FILE ...";
	&OutputConfig($CONFIG_FILE, $VERSION, $UPDATE, $MAINTAINER, $DEBUG);
	print STDOUT "done\n";
	exit 0;
    }
    if ($ans =~ /^[nN].*$/) {
	die "エラー: 処理を中断しました.\n";
    }
}


##################################################
#####                                        #####
##             デフォルト設定値                ###
#####                                        #####
##################################################

##############################
###  ロケーション (インプット)
##
#
# ※ $SRC_DIR のデフォルトは src であり, dcppt.pl の -d
#    オプションが与えられればその引数を取る
#
# 情報ファイル
$infofile = "${SRC_DIR}/menu.txt";

# タイトルファイル
$titlefile = "${SRC_DIR}/title.txt";

##############################
###  ロケーション (アウトプット)
##
# 公開資源置場ディレクトリ
$pubdir = "pub";

# インデックスファイル
$indexfile = "${pubdir}/index.html";

# メニューファイル (イメージ)
$menufile = "${pubdir}/menu.html";
#$menufile = "";      # 画像版メニューファイルがいらない時

# メニューファイル (テキスト)
$menutxtfile = "${pubdir}/menutxt.html";
#$menutxtfile = "";   # テキスト版メニューファイルがいらない時

# サムネイルファイル名
$thumbnail_file = "${pubdir}/thumbnail.html";

# 生成した HTML ファイル置場ディレクトリ
$htmldir = "${pubdir}/html";

# メニュー画像ファイル置場ディレクトリ
$menudir = "${pubdir}/menu";

# スライド画像ファイル置場ディレクトリ
$slidedir = "${pubdir}/slide";

# サムネイル画像ファイル置場ディレクトリ
$thumbdir = "${pubdir}/thumbnail";

##############################
###  画像サイズ
##
# メニューのサイズ
$menusize = "120x90";
# スライドのサイズ
$slidesize = "640x480";
# サムネイルのサイズ
$thumbsize = "200x150";

##############################
###  色深度の設定
##
$depth = "8";      # 0: -depth オプションを利用しない

##############################
###  文字コード
##
# 出力されるファイルの日本語文字コード
# "jis", "euc", "sjis", "utf8" のどれかを選択
$jpcode = "utf8";
$system_code = "utf8";

##############################
###  スタイル設定
##
# 起動時のデフォルトのページ
# 左ページ
$defaultleft  = "$menufile";     # イメージメニュー
#$defaultleft  = "$menutxtfile";  # テキストメニュー
# 右ページ
$defaultright = "$thumbnail_file";    # サムネイル
#$defaultright = "$htmldir/001.html";  # 1 ページ目


## スタイルの設定
# 0: スタイル無し, 1: スタイルあり
$style_mode = "1";
    # テキストの色
$style_textcolor       = '#339933';
    # 背景色
$style_bgcolor         = '#FFFFFF';
    # 始めのリンク色
$style_linkcolor       = '#666600';
    # 既に参照した後のリンク色
$style_visitedcolor    = '#666633';
    # マウスを上に乗せた時のリンク色
$style_hovercolor      = '#CCFFCC';
    # リンクの... none: 下線無し, underline: 下線あり,
    #             blink: 点滅, overline:上線,
    #             line-through: 打ち消し線
$style_link_underline  = 'none';
    # リンクの... normal: 普通, bold: 太字
$style_link_fontweight = 'bold';

# 前後のページへのリンクに用いる文字列
$previouslink = '[Prev]';
$nextlink     = '[Next]';

# 各ページのリンクの上下に付けるリンク
$upperlink = '▲Parent Dirctory';
$upperurl  = '';
$lowerlink = 'ppt file▼';
$lowerurl  = '';

# 出力される各ページの末尾に付加されるメッセージを指定
#   - acktext を空にすると, メッセージそのものが出力されない
#   - acklink が指定されている場合ふたつのコロン "::" で
#     囲まれた部分は acklink へのリンクに変換
#   - $VERSION で dcppt.pl のバージョンが出力される
#   - $URL で dcppt.pl の置き場所の URL が出力される
$acktext = "This page is generated with ::dcppt:: Version $VERSION";
$acklink = "$URL";

##############################
###  サムネイル
##
# 0: サムネイル無し, 1: サムネイルあり
$thumbnail_mode    = "1";
    # サムネイルへのリンクのメッセージ
$thumbnail_link    = "[[ Thumbnail ]]";
    # サムネイルの列の数
$thumbnail_col_num = "3";


##############################
###  管理者／開発者用
##
# 画像ファイルの桁数
$len = 3;
# 以下の拡張子を持つものを画像ファイルとする.
@treat = ('gif', 'GIF', 'jpg', 'JPG', 'jpeg', 'JPEG',
	  'png', 'PNG', 'bmp', 'BMP', 'tif', 'TIF',
	  'tiff', 'TIFF');
# src/ 以下の画像名のリネームリスト
%ImgRenameList = (
		     'スライド',	'slide',
		    );


##################################################
#####                                        #####
###       ユーザ設定ファイル読み込み           ###
#####                                        #####
##################################################

# ユーザの設定ファイルが読み込める場合はそれを読み込む
if (-r $CONFIG_FILE) {
    require "$CONFIG_FILE";
}


##################################################
#####                                        #####
##                例外処理                     ###
#####                                        #####
##################################################

# 文字コードの例外処理
unless ($jpcode eq 'euc' || $jpcode eq 'sjis' || $jpcode eq 'jis' || $jpcode eq 'utf8') {
    print STDOUT "Modify \$jpcode from $jpcode to euc" if ($DEBUG);
    $jpcode = 'utf8';
}

##################################################
#####                                        #####
##             プログラム本体                  ###
#####                                        #####
##################################################


##################################################
## 標準入力またはタイトルファイルからタイトル取得
##################################################

$title = &TitleGet($titlefile, $jpcode, $QUIET, $DEBUG);

##################################################
## $SRC_DIR 内のファイル名をシステムの文字コード
## (デフォルトは utf8) にリネーム
##################################################

&RenameCurrentCode($SRC_DIR, $system_code, $DEBUG);

##################################################
## $SRC_DIR 内のファイル名が特定の日本語の場合は
## 英語にリネーム
##################################################

&RenameImgNameEtoJ($SRC_DIR, $system_code, $DEBUG);


##################################################
## pub と slide と thumbnail 用のディレクトリが
## 無い場合は新規作成. ある場合には再度生成
## するかどうか問い合わせる. 生成しない場合には
## convert はおこなわない.
##################################################

#
# 画像の convert は時間がかかるため, 既にディレクトリが
# 存在する場合には再変換しないように出来る.
#
if (-d $pubdir && -d $menudir && -d $slidedir
    && -d $thumbdir || !$thumbnail_mode){
    unless ($QUIET) {
	print STDOUT "既に $slidedir 等が存在しています.\n".
	    "画像を再度変換しますか? (変換には少々時間がかかります.) [y/N]: ";
	$ans = <STDIN>;

	if ($ans =~ /^[yY].*$/) {
	    $NoConvert = 0;
	} else {
	    $NoConvert = 1;
	}
    } else {
	$NoConvert = 1;
    }
    print STDOUT "  \$NoConvert is $NoConvert.\n" if ($DEBUG);
}

mkdir($pubdir) unless -d $pubdir;
mkdir($menudir) unless -d $menudir;
mkdir($slidedir) unless -d $slidedir;
if ($thumbnail_mode) {
    mkdir($thumbdir) unless -d $thumbdir;
}

##################################################
## $SRC_DIR 内の画像ファイルの名前のみを取得
##################################################

#
# $SRC_DIR ディレクトリ内から画像ファイルの一覧を取り出し,
# @srcimgs へ代入する
#
@srcimgs = &GetImageList($SRC_DIR, $DEBUG);


##################################################
## 適切な画像ファイル名を作成
##################################################
#
# @srcimgs の名前を適切に変更したものを @pubimgs
# として受け取る.
#
#     スライド1.PNG  ->   001.png
#      mgp00001.jpg  ->   001.jpg
#             1.GIF  ->   001.gif
#

@pubimgs = &RenameImage(@srcimgs);


##################################################
## $SRC_DIR 内の @srcimgs を $slidedir と
## $thumbdir 内に @pubimgs とリネームして移動.
## 移動の際に convert で画像サイズ変換
##################################################
if ($depth){
    $convert_depth = "-depth $depth"
} else {
    $convert_depth = ""
}

#
# 本当は @srcimsg と @pubimgs も渡したいが,
# 2つの配列を渡すのは perl の仕様上面倒なので
# サブルーチン内から直接呼ぶ
#
unless ($NoConvert){
    &ConvertImage($SRC_DIR, $menudir, $menusize,
		  $CONVERT, $convert_depth . " " . $CONVERT_OPT, $DEBUG);
    
    &ConvertImage($SRC_DIR, $slidedir, $slidesize,
		  $CONVERT, $convert_depth . " " . $CONVERT_OPT, $DEBUG);
    
    if ($thumbnail_mode){
	&ConvertImage($SRC_DIR, $thumbdir, $thumbsize,
		     $CONVERT, $convert_depth . " " . $CONVERT_OPT, $DEBUG);
    }
}

##################################################
## slide, thumbnail ディレクトリ内から
## 画像ファイルのタイプを取り出す
##################################################

$imgtype_menu  = &GetImageType($menudir, $DEBUG);
$imgtype_slide = &GetImageType($slidedir, $DEBUG);
$imgtype_thumb = &GetImageType($thumbdir, $DEBUG) if ($thumbnail_mode);

##################################################
## slide, thumbnail ディレクトリ内から
## 画像ファイルのタイプを取り出す
##################################################

#
# 画像ファイル数を $file_num へ代入.
# 
$file_num = $#pubimgs + 1;

##################################################
## 情報ファイルから, ナンバーとサブジェクトを取り出す.
##################################################

#
# 情報ファイルが無い場合は雛型作成.
#
&CreateInfo($infofile, $file_num, $DEBUG);

#
# 情報ファイルから, ナンバーとサブジェクトを取り出して @info へ
#
@info = &GetInfo($infofile, $jpcode, $DEBUG);

#
# @info に格納されるナンバーとサブジェクトを分離して
# @number、@subject、%info_hash へ
#
# - グローバル変数の @info と @number と @subject と %info_hash を
#   内部で利用 (渡したかったけど, 渡せなかったのでサボる)
#
&SplitInfo($DEBUG);

#
# 文字コードを jpcode 用から meta タグに書き込む用に変換.
#
$metajp = &JpcodeToMeta($jpcode, $DEBUG);

#
# スタイルとして書き出す文字列
#
$style_out = &StyleMake($style_mode, $style_textcolor, $style_bgcolor,
			$style_linkcolor, $style_visitedcolor,
			$style_hovercolor, $style_link_underline,
			$style_link_fontweight, $DEBUG);

$style_out_menu =
    &StyleMakeForMenu($style_mode, $style_textcolor, $style_bgcolor,
			$style_linkcolor, $style_visitedcolor,
			$style_hovercolor, $style_link_underline,
			$style_link_fontweight, $DEBUG);

#
# インデックスの作成
#

&MakeIndex($title, $style_out, $metajp,
	   $indexfile, $defaultleft, $defaultright, $DEBUG);

#
# img タグの alt 属性に「<」、「>」、「"」、「&」、「'」が入る文字を
# 書き出さないようにするため、それらが入らない文字だけを %altmsg に
# 格納。
#
#    - グローバル変数の @number と %info_hash と %altmsg を内部で利用
#      (渡したかったけど, 渡せなかったのでサボる)
#
&MakeAltMsg($DEBUG);

#
# 画像を張り込むメニューの作成
#    - グローバル変数の @number と %info_hash と %altmsg を内部で利用
#      (渡したかったけど, 渡せなかったのでサボる)

if ($menufile) {
    &MakeMenu($menudir, $htmldir, $title, $style_out_menu,
	      $jpcode, $metajp, $imgtype_menu,
	      $upperlink, $upperurl, $lowerlink, $lowerurl,
	      $thumnail_mode, $thumbnail_file, $thumbnail_link,
	      $menufile, $menutxtfile, $DEBUG);
}

#
# メニューの作成
#    - グローバル変数の @number と %info_hash を内部で利用
#      (渡したかったけど, 渡せなかったのでサボる)

if ($menutxtfile) {
    &MakeMenuTxt($title, $style_out, $jpcode, $metajp,
		 $upperlink, $upperurl, $lowerlink, $lowerurl,
		 $thumnail_mode, $thumbnail_file, $thumbnail_link,
		 $menutxtfile, $menufile, $DEBUG);
}
#
# 各 HTML の作成
#    - グローバル変数の @number と %info_hash と %altmsg を内部で利用
#      (渡したかったけど, 渡せなかったのでサボる)

&MakeHtml($htmldir, $title, $style_out,
	  $previouslink, $nextlink, $imgtype_slide,
	  $acktext, $acklink, $jpcode, $metajp, $DEBUG);

#
# サムネイルの作成
#    - グローバル変数の @number と %info_hash と %altmsg を内部で利用
#      (渡したかったけど, 渡せなかったのでサボる)

&MakeThumbnail($title, $style_out_menu, $jpcode, $metajp, $htmldir,
	       $thumnail_mode, $thumbnail_file, $thumbdir,
	       $thumbnail_link, $thumbnail_col_num,
	       $imgtype_thumb, $acktext, $acklink, $DEBUG);

#
# 終了
#
&EndMessage($DEBUG);


##################################################
#####                                        #####
##             サブルーチン群                  ###
#####                                        #####
##################################################

sub OutputConfig() {
    local($configfile, $version, $update, $maintainer, $debug) = @_;
    # $configfile がディレクトリも含む場合はそれらを作成する.
    local(@config_path) = split(/\//, $configfile);
    local($dirpath) = "";
    local($first) = 1;
    while($#config_path > 0){
	local($dir) = shift(@config_path);
	$dirpath = $dir if $first;
	$dirpath .= '/' . "$dir" unless $first;
	mkdir($dirpath) unless -d $dirpath;
	print STDOUT "  mkdir $dirpath \n" if ($debug);
	$first = 0;
    }
    open(CONFIGFILE, "> $configfile")
	|| die "エラー: $configfile に書き込むことが出来ませんでした. \n";
    print CONFIGFILE <<EOF;
#####   $configfile for dcppt.pl    #####
##
#  This is Template File for dcppt ver $VERSION
#
#  (C) Copyright by @MAINTAINERS
#  $UPDATE.  All rights reserved.
#
EOF

    print CONFIGFILE <<'EOF';
##############################
###  ロケーション (インプット)
##
#
# ※ $SRC_DIR のデフォルトは src であり, dcppt.pl の -d
#    オプションが与えられればその引数を取る
#
# 情報ファイル
$infofile = "${SRC_DIR}/menu.txt";

# タイトルファイル
$titlefile = "${SRC_DIR}/title.txt";

##############################
###  ロケーション (アウトプット)
##
# 公開資源置場ディレクトリ
$pubdir = "pub";

# インデックスファイル
$indexfile = "${pubdir}/index.html";

# メニューファイル (イメージ)
$menufile = "${pubdir}/menu.html";
#$menufile = "";      # 画像版メニューファイルがいらない時

# メニューファイル (テキスト)
$menutxtfile = "${pubdir}/menutxt.html";
#$menutxtfile = "";   # テキスト版メニューファイルがいらない時

# サムネイルファイル名
$thumbnail_file = "${pubdir}/thumbnail.html";

# 生成した HTML ファイル置場ディレクトリ
$htmldir = "${pubdir}/html";

# メニュー画像ファイル置場ディレクトリ
$menudir = "${pubdir}/menu";

# スライド画像ファイル置場ディレクトリ
$slidedir = "${pubdir}/slide";

# サムネイル画像ファイル置場ディレクトリ
$thumbdir = "${pubdir}/thumbnail";

##############################
###  画像サイズ
##
# メニューのサイズ
$menusize = "120x90";
# スライドのサイズ
$slidesize = "640x480";
# サムネイルのサイズ
$thumbsize = "200x150";

##############################
###  色深度の設定
##
$depth = "8";      # 0: -depth オプションを利用しない

##############################
###  文字コード
##
# 出力されるファイルの日本語文字コード
# "jis", "euc", "sjis", "utf8" のどれかを選択
$jpcode = "utf8";
$system_code = "utf8";

##############################
###  スタイル設定
##
# 起動時のデフォルトのページ
# 左ページ
$defaultleft  = "$menufile";     # イメージメニュー
#$defaultleft  = "$menutxtfile";  # テキストメニュー
# 右ページ
$defaultright = "$thumbnail_file";    # サムネイル
#$defaultright = "$htmldir/001.html";  # 1 ページ目


## スタイルの設定
# 0: スタイル無し, 1: スタイルあり
$style_mode = "1";
    # テキストの色
$style_textcolor       = '#339933';
    # 背景色
$style_bgcolor         = '#FFFFFF';
    # 始めのリンク色
$style_linkcolor       = '#666600';
    # 既に参照した後のリンク色
$style_visitedcolor    = '#666633';
    # マウスを上に乗せた時のリンク色
$style_hovercolor      = '#CCFFCC';
    # リンクの... none: 下線無し, underline: 下線あり,
    #             blink: 点滅, overline:上線,
    #             line-through: 打ち消し線
$style_link_underline  = 'none';
    # リンクの... normal: 普通, bold: 太字
$style_link_fontweight = 'bold';

# 前後のページへのリンクに用いる文字列
$previouslink = '[Prev]';
$nextlink     = '[Next]';

# 各ページのリンクの上下に付けるリンク
$upperlink = '▲Parent Dirctory';
$upperurl  = '';
$lowerlink = 'ppt file▼';
$lowerurl  = '';

# 出力される各ページの末尾に付加されるメッセージを指定
#   - acktext を空にすると, メッセージそのものが出力されない
#   - acklink が指定されている場合ふたつのコロン "::" で
#     囲まれた部分は acklink へのリンクに変換
#   - $VERSION で dcppt.pl のバージョンが出力される
#   - $URL で dcppt.pl の置き場所の URL が出力される
$acktext = "This page is generated with ::dcppt:: Version $VERSION";
$acklink = "$URL";

##############################
###  サムネイル
##
# 0: サムネイル無し, 1: サムネイルあり
$thumbnail_mode    = "1";
    # サムネイルへのリンクのメッセージ
$thumbnail_link    = "[[ Thumbnail ]]";
    # サムネイルの列の数
$thumbnail_col_num = "3";


##############################
###  管理者／開発者用
##
# 画像ファイルの桁数
$len = 3;
# 以下の拡張子を持つものを画像ファイルとする.
@treat = ('gif', 'GIF', 'jpg', 'JPG', 'jpeg', 'JPEG',
	  'png', 'PNG', 'bmp', 'BMP', 'tif', 'TIF',
	  'tiff', 'TIFF');
# src/ 以下の画像名のリネームリスト
%ImgRenameList = (
		     'スライド',	'slide',
		    );

EOF

    print CONFIGFILE <<EOF;
######################################################################
# 呼び出し元からみて require '${configfile}' が 1 を返すようにする.
# そうでなければファイルが見つかっていないことが判定される.
1;

######################################################################
#== Mode setting for Emacs
#Local Variables:
#mode: cperl
#End:
#
EOF

    close(CONFIGFILE);
}


# 標準入力またはタイトルファイルからタイトルを得る
#
# 引数
#	$titlefile	タイトルが書き込まれているファイル
#
# 返却値
#	ファイル内の一行目または標準入力を, 改行を取り除いて返す
#
sub TitleGet {
    local($titlefile, $jpcode, $quiet, $debug) = @_;
    local($title);
    #
    # タイトルファイルがない場合, 標準入力より入力する.
    #
    unless  (-f "$titlefile") {
	unless ($quiet) {
	    print "タイトルを記入して下さい.\n  Title []: ";
	    $input = <STDIN> ;
	    chomp($input) ;
	} else {
	    $input = "" ;
	    chomp($input) ;
	}
	&Jcode::convert(*input, "$jpcode");
	open(TITLE, "> $titlefile") ||
	    die "エラー: $titlefile に書き込むことができませんでした. \n";
	print TITLE "$input" ;
	close(TITLE);
    }
    #
    # タイトルファイルからタイトルを取り出す.
    #
    print STDOUT "Get Title form $titlefile ...";
    open(TITLEFILE, "$titlefile")
	|| die "エラー: $titlefile が見つかりませんでした. \n";
    while (<TITLEFILE>) {
	chomp;
	# 文字コード変換
	&Jcode::convert(*_, "$jpcode");
	push(@title_lists, $_);
    }
    # 一行目だけを取り出す.
    $title = $title_lists[0];
    close(TITLEFILE);
    # デバッグメッセージ
    print STDOUT "TitleGet: \$title = $title\n" if ($debug);
    # 値の返却
    print STDOUT "done.\n";
    return $title;
}

#
# $dir 内のファイル名を $code の名前に変更する
#
# 引数
#	$dir		ディレクトリ
#	$code		文字コード
#
# 返却値
#	なし
#
#
sub RenameCurrentCode() {
    local($dir, $code, $debug) = @_;
    local($srcfile, $srcfile_org, $ok);
    print STDOUT "Rename files in $dir to $code ...";
    
    opendir(SRCDIR, "$dir")
	|| die "エラー: $dir ディレクトリが存在しません. \n";
    # 文字コードが Shift-JIS である場合を考え, いったんはファイル名を
    # そのまま取得
    local(@srcfiles) = readdir(SRCDIR);
    closedir(SRCDIR);

    # ファイル名をシステムのコード (デフォルトは utf8) に変更
    foreach $srcfile (@srcfiles) {
	next if ( ($srcfile eq "..") || ($srcfile eq ".") );
	$srcfile_org = $srcfile;
	&Jcode::convert(*srcfile, "$code");
	$ok = rename("$dir/$srcfile_org", "$dir/$srcfile");
	if ($debug) {
	    print STDOUT "\n    $dir/$srcfile_org -> $dir/$srcfile";
	    print STDOUT "\t \[Failure\]" if $ok == 0;
	}
    }

    print STDOUT "\n" if ($debug);
    print STDOUT "done.\n";
}

#
# $dir 内のファイル名が %ImgRenameList のキーに一致した場合,
# その値に変更する. ファイル名の文字コードは $code であると
# 考えて一致の判定を行う.
#
# 引数
#	$dir		ディレクトリ
#
# 返却値
#	なし
#
#
sub RenameImgNameEtoJ() {
    local($dir, $code, $debug) = @_;
    local($srcfile, $from, $to, $suffix, $ok);
    print STDOUT "Rename files by \%ImgRenameList in ...";
    
    opendir(SRCDIR, "$dir")
	|| die "エラー: $dir ディレクトリが存在しません. \n";
    # ファイル名を取得
    local(@srcfiles) = readdir(SRCDIR);
    closedir(SRCDIR);

    # ファイル名が %ImgRenameList のキーにヒットする場合はその
    # 値にリネーム
    foreach $srcfile (@srcfiles) {
	foreach $from (keys %ImgRenameList) {
	    $to = $ImgRenameList{$from};
	    &Jcode::convert(*to, "$code");
	    &Jcode::convert(*from, "$code");
	    next unless ($srcfile =~ /^$from(\d+\.\w+)$/);
	    $suffix = $1;
	    $ok = rename("$dir/$srcfile", "$dir/$to$suffix");
	    if ($debug) {
		print STDOUT "\n    $dir/$srcfile -> $dir/$to$suffix";
		print STDOUT "\t \[Failure\]" if $ok == 0;
	    }
	}
    }
    
    print STDOUT "\n" if ($debug);
    print STDOUT "done.\n";
}


#
# $dir 内から画像ファイルの名前のリストを取り出す
#
# 引数
#	$dir		ディレクトリ
#
# 返却値
#	$dir 内の画像ファイル名のリストを返す
#
sub GetImageList(){
    local($dir, $debug) = @_;
    local(@treat) = @treat;
    local($srcfile, $imgtype);
    
    print STDOUT "Get Names of Image in $dir ...";
    opendir(DIR, "$dir")
	|| die "エラー: $dir ディレクトリが存在しません. \n";
    local(@srcfiles) = grep(!/^\.\.?$/, sort(readdir(DIR)));
    closedir(DIR);
    
    #
    # 画像ファイルのみ @srcfiles_img に含める.
    # 条件は, @treat の拡張子を持つことと, ファイル名に数字を含むことである.
    #
    undef local(@srcfiles_img);
    foreach $srcfile (@srcfiles){
	local($img_ok) = 0;
	local(@srcfile_parts) = split(/\./, $srcfile);
	local($srcfilename) = $srcfile_parts[$#srcfile_parts - 1];
	local($srcfile_imgtype) = $srcfile_parts[$#srcfile_parts];
	foreach $imgtype (@treat){
	    if ($srcfile_imgtype eq $imgtype){
		if ($srcfilename =~ /^.*\d+.*$/){
		    $img_ok = 1;
		}
	    }
	}
	if ($img_ok) {
	    push(@srcfiles_img, $srcfile);
	    print STDOUT "  $SRC_DIR/$srcfile is IMG file.\n" if ($debug);
	}
    }

    # ディレクトリ内に画像ファイルが無い場合はエラーを返す
    unless ($srcfiles_img[0]) {
	die "エラー: $dir 内には画像ファイルと思われるものが存在しません. \n";
    }
    print STDOUT "  done.\n";
    return @srcfiles_img;
}


# 画像ファイルリネームサブルーチン
#
# 引数
#	@org	元々の画像ファイル名
#
# 返却値
#	@org をリネームしたもの. リネームは以下のように
#       おこなわれる
#
#         - スライド1.PNG  ->   001.png
#         -  mgp00001.jpg  ->   001.jpg
#         -         1.GIF  ->   001.gif
#
sub RenameImage() {
    local(@org) = @_;
    local($debug) = $DEBUG;
    local(@rename);            # リネーム後のファイル名
    local($len)   = $len;      # リネームしたファイルの桁数
    print STDOUT "Prepare Names of image files in publish directory ...";

    undef @rename;
    local($orgimg);
    foreach $orgimg (@org){
	local(@orgimg_parts) = split(/\./, $orgimg);
	local($orgimgname) = $orgimg_parts[$#orgimg_parts - 1];
	local($orgimgtype) = $orgimg_parts[$#orgimg_parts];
	#
	# ファイル名の内, 数値でないものを取り除く.
	#
	if ($orgimgname =~ /([1-9]\d*)/){
	    $orgimgname = $1;
	} else {
	    warn "$orgimg は数字を含まないため, 画像ファイルとして識別されません.\n";
	}
	#
	# ファイルのナンバーを規定の桁数にする
	#
	if ($len < length($orgimgname)){
	    # エラーマーカー
	    local($error) = 1;
	} else {
	    # エラーマーカーの解除
	    $error = 0;
	}
	local($new_orgimgname) = sprintf('%.' . $len . 'd', $orgimgname);
	
	#
	# 拡張子を小文字にする
	#
	local($new_orgimgtype) = $orgimgtype;
	$new_orgimgtype =~ tr/A-Z/a-z/;

	#
	# ファイル名を @rename に格納.
	#
	push(@rename, "${new_orgimgname}\.${new_orgimgtype}");

	#
	# デバッグ用のメッセージ出力
	#
	if ($debug) {
	    print STDOUT "    $SRC_DIR/$orgimg -> ${new_orgimgname}\.${new_orgimgtype}";
	    print STDOUT "\t \[Irregular Rename...\]" if $error == 1 ;
	    print STDOUT "\n";
	}
    }
    print STDOUT "  done.\n";

    #
    # 名前が重複していないかチェック。
    #
    print STDOUT "Checking File Name Duplication ...";
    local($check1, $check2, $n1, $n2, @checkarray, $duplication);
    @checkarray = @rename;
    $duplication = 0;
    $n1 = 0;
    $n2 = $n1 + 1;
    while (@checkarray) {
	$check1 = shift(@checkarray);
#	print STDOUT "n1 = $n1 check1 = $check1\n" if ($debug);
	foreach $check2 (@checkarray) {
#	    print STDOUT "n2 = $n2 check2 = $check2\n" if ($debug);
	    if ($check1 eq $check2) {
		$duplication = $check1;
		last;
	    }
	    $n2++;
	}
	last if $duplication;
	$n1++;
	$n2 = $n1 + 1;
    }

#    print STDOUT "n1 = $n1\n" if ($debug);
#    print STDOUT "n2 = $n2\n" if ($debug);

    if ($duplication) {
	warn "\n       Warning!! \"$org[$n1]\" と \"$org[$n2]\" が同名の \"$duplication\" にリネームされます。\n\n";
    }
    print STDOUT "  done.\n";
    return @rename;
}


# 画像ファイルサイズ変換 & 移動サブルーチン
#
# 引数
#	$srcdir		移動元ディレクトリ
#	$todir		公開用の移動先用の画像ファイル置場
#	$imgsize	画像のサイズ (変更しようとするサイズ)
#	$convert	convert のコマンド
#	$convert_opt	convert の画像変更用オプションの文字列
#	$debug		デバッグオプション
#
# 暗黙の引数
#	@srcimgs	$SRC_DIR 内の画像ファイル一覧
#	@pubimgs	$slidedir などで使用するファイル名一覧
#
# 返却値
#	なし.
#
sub ConvertImage() {
    local($srcdir, $todir, $imgsize,
	  $convert, $convert_opt, $debug) = @_;
    local($command) = "$convert $convert_opt";
    local($n) = 0;

    print STDOUT "Convert Image files from $srcdir to $todir ...  \n";
    
    while($srcimgs[$n]){
	print STDOUT "  $command $imgsize ".
	    "$srcdir/$srcimgs[$n] $todir/$pubimgs[$n]....  ";
	
	system("$command $imgsize $srcdir/$srcimgs[$n] $todir/$pubimgs[$n]");

	print STDOUT "done.\n";
	$n++;
    }
    print STDOUT "done.\n";
}

#
# ディレクトリ内の1つ目のファイルの画像タイプを取り出す
#
# 引数
#	$dir		ディレクトリ
#
# 返却値
#	ディレクトリ内の1つ目のファイルの拡張子を返す

sub GetImageType() {
    local($dir, $debug) = @_;
    print STDOUT "Get ImageType from $dir ..." unless ($debug);
    # ディレクトリオープン
    opendir(DIR, "$dir")
	|| die "エラー: $dir ディレクトリが存在しません. \n";
    local(@imgfiles) = grep(!/^\.\.?$/, readdir(DIR));
    closedir(DIR);
    # ディレクトリ内に画像があるかチェック
    unless ($imgfiles[0]) {
	die "エラー: $dir ディレクトリ内にファイルが存在しません. \n";
    }
    # 一つ目のファイルだけを見る. (面倒いから)
    local($sampleimg) = $imgfiles[0];
    local(@imgname_parts) = split(/\./, $sampleimg);
    local($imgtype) = $imgname_parts[$#imgname_parts];

    # デバッグ用出力
    print STDOUT "GetImageType: In $dir, ImageType is $imgtype.\n" if ($debug);
    print STDOUT "done.\n" unless ($debug);

    return $imgtype;
}


# $infofile (menu.txt) 生成サブルーチン
#
# 説明
#	$infofile が存在しない場合、$file_num に合わせた $infofile
#	を作成する。$infofile が存在する場合は何もせずに返す。
#
# 引数
#	$infofile	スライド情報格納ファイル (デフォルト menu.txt)
#	$file_num	スライドの数
#	$debug		デバッグオプション
#
# 返却値
#	なし.
#
sub CreateInfo(){
    local($infofile, $file_num, $debug) = @_;
    #
    # $infofile が存在する場合は何もせずに終了。
    #
    if (-f "$infofile") {
	print STDOUT "Slide Informations file $infofile is already exist.\n";
	return;
    }
    #
    # 存在しない場合はスライド数に合わせた $infofile を作成。
    # 
    print STDOUT "Create Slide Informations file $infofile ...  ";
    print STDOUT "  In $infofile ...  \n" if ($DEBUG);
    open(INFOFILE, "> $infofile")
	|| die "$infofile に書き込むことができませんでした. \n";
    local($n) = 1 ;
    while ($n <= $file_num ) {
	local($number) = sprintf('%.' . $len . 'd', $n);
	print INFOFILE "${number}:\n";
	$n++;
	print STDOUT "    ${number}:\n" if ($debug);
    }
    close(INFOFILE);
    print STDOUT "done\n";
    return;
}

# $infofile からの @info へ情報を取得するサブルーチン
#
# 説明
#	$infofile 内のデータを @info に一行づつ格納して返す。
#	もしも書式が間違っている場合はエラーを返して終了する。
#	なお取得の際、日本語は文字コードを $jpcode に変更してしまう。
#
# 引数
#	$infofile	スライド情報格納ファイル (デフォルト menu.txt)
#	$jpcode		文字コード
#	$debug		デバッグオプション
#
# 返却値
#	@info		$infofile の各行を格納した配列。
#			文字コードは全て $jpcode へ変換。
#
sub GetInfo(){
    local($infofile, $jpcode, $debug) = @_;
    local(@info);
    print STDOUT "Get Slide Informations from $infofile ...  ";
    open(INFOFILE, "$infofile") || die "$infofile is not found.\n";
    while (<INFOFILE>) {
	# 例外処理
	unless ($_ =~ /^(\d){3}:.*$/) {
	    die "$infofile の各行の始めは 001:.., 002:.. となっている必要があります.\n"
	}
	chomp;
	# 文字コード変換
	&Jcode::convert(*_, "$jpcode");
	push(@info, $_);
	print STDOUT "    $_\n" if ($debug);
    }
    close(INFOFILE);
    print STDOUT "done\n";
    return @info;
}

#
# @info に格納されるナンバーとサブジェクトを分離して
# @number、@subject、%info_hash へ
# なお、サブジェクトが空の場合、Page 001、Page 002 などの文字を与える。
#
# 引数
#	$debug		デバッグオプション
#
# 暗黙の引数 (入力)
#	@info		各行ナンバーとサブジェクトが格納されている
#
# 暗黙の引数 (出力)
#	@number		各行のナンバーを格納
#	@subject	各行のサブジェクトを格納
#	%info_hash	ナンバーをキーに、サブジェクトを値に格納した連想配列
#
# 返却値
#	なし.
#
sub SplitInfo(){
    local($debug) = @_;
    local($key);
    print STDOUT "Split Slide Informations ...  ";
    foreach $key (@info){
	local(@info_parts) = split(/:/, $key, 2);
	print STDOUT "    Number: $info_parts[0]\t Value: " if ($debug);
	# $info_parts[1] が空の場合には "Page 001" のような
	# 文字列をいれておく
	unless ($info_parts[1] =~ /\S+/){
	    $info_parts[1] = "Page " . "$info_parts[0]";
	    print STDOUT "[Not Found. Auto Generate...] " if ($debug);
	}
	$info_hash{"$info_parts[0]"} = "$info_parts[1]";
	push(@number, $info_parts[0]);
	push(@subject, $info_parts[1]);
	print STDOUT "$info_parts[1]\n" if ($debug);
    }
    print STDOUT "done\n";
    return;
}


#
# 文字コードを jpcode 用から meta タグに書き込む用に変換するサブルーチン
#
sub JpcodeToMeta($jpcode, $debug) {
    print STDOUT "Recognize Japanese Code is $jpcode ..." unless ($debug);
    local($jpcode, $debug) = @_;
    local($metajp, %meta_jpcode_list);
    %meta_jpcode_list = (
			 'euc',		'euc-jp',
			 'sjis',	'Shift_JIS',
                         'jis',	        'iso-2022-jp',
                         'utf8',        'utf-8',
			);
    $metajp = $meta_jpcode_list{"$jpcode"};
    # デバッグ用出力
    print STDOUT "JpcodeToMeta: \$jpcode = $jpcode => \$metajp = $metajp\n" if ($debug);
    print STDOUT "done.\n" unless ($debug);
    return $metajp;
}

#
# スタイル整形サブルーチン
#
# 引数
#	$mode		0: スタイル無し, 1: スタイルあり
#	$textcolor	テキストの色
#	$bgcolor	背景色
#	$linkcolor	始めのリンク色
#	$visitedcolor	既に参照した後のリンク色
#	$hovercolor	マウスを上に乗せた時のリンク色
#	$link_underline	リンクの... none: 下線無し, underline: 下線あり,
#				    blink: 点滅, overline:上線,
#				    line-through: 打ち消し線
#	$link_fontweight リンクの... normal: 普通, bold: 太字
#
#
# 返却値
#	スタイルとして書き出す出力メッセージ
#
sub StyleMake() {
    local($mode, $textcolor, $bgcolor, $linkcolor, $visitedcolor,
	  $hovercolor, $link_underline, $link_fontweight, $debug) = @_;
    local($style_output);
    # スタイル用に書き出す文字列を出力.
    $style_output =
	'  <style type="text/css">'. "\n".
	'    <!--'. "\n".
        "      body{\n".
        "           color:$textcolor;\n".
        "           background-color:$bgcolor;\n".
        "          }\n".
        "      a:link{color:$linkcolor;}\n".
        "      a:visited{color:$visitedcolor;}\n".
        "      a:hover{color:$hovercolor;}\n".
        "      a{\n".
        "        text-decoration:$link_underline;\n".
        "        font-weight:$link_fontweight;\n".
        "      }\n".
        "    -->\n".
        "  </style>\n";

    # スタイル指定しない場合は, nul 文字を返す
    undef $style_output unless ($mode);

    # デバッグ用出力
    print STDOUT "StyleMake: \$style_output = \n $style_output\n" if ($debug);
    return $style_output;
}

#
# メニューファイル用スタイル整形サブルーチン
#
# 引数
#	$mode		0: スタイル無し, 1: スタイルあり
#	$textcolor	テキストの色
#	$bgcolor	背景色
#	$hovercolor	マウスを上に乗せた時のリンク色
#
#
# 返却値
#	メニューファイルのスタイルとして書き出す出力メッセージ
#
sub StyleMakeForMenu() {
    local($mode, $textcolor, $bgcolor, $linkcolor, $visitedcolor,
	  $hovercolor, $link_underline, $link_fontweight, $debug) = @_;
    local($style_output);
    # スタイル用に書き出す文字列を出力.
    $style_output =
	'  <style type="text/css">'. "\n".
	'    <!--'. "\n".
        "      body{\n".
        "           color:$textcolor;\n".
        "           background-color:$bgcolor;\n".
        "          }\n".
        "      a:link{color:$linkcolor;}\n".
        "      a:visited{color:$visitedcolor;}\n".
        "      a:hover{color:$hovercolor;}\n".
        "      a:link img {border-color:$bgcolor;}\n".
        "      a:visited img {border-color:$bgcolor;}\n".
        "      a:hover img {border-color:$hovercolor;}\n".
        "      a{\n".
        "        text-decoration:$link_underline;\n".
        "        font-weight:$link_fontweight;\n".
        "      }\n".
        "    -->\n".
        "  </style>\n";

    # スタイル指定しない場合は, nul 文字を返す
    undef $style_output unless ($mode);

    # デバッグ用出力
    print STDOUT "StyleMakeForMenu: \$style_output = \n $style_output\n" if ($debug);
    return $style_output;
}


#
# 各インデックスファイル作成サブルーチン
#
sub MakeIndex {
    local($title, $style_out, $metajp,
	  $indexfile, $defaultleft, $defaultright, $debug) = @_;
    print STDOUT "Creating Index File ..." unless ($debug);
    open(INDEXFILE, "> $indexfile")
	|| die "エラー: $indexfile に書き込むことが出来ませんでした. /n";
    #
    # ヘッダの書き出し
    #
    print INDEXFILE <<EOF;
<html>
<head>
  <meta http-equiv="Content-Type" content="text/html; charset=${metajp}">
  <title>$title</title>
$style_out
</head>

<frameset cols="22%,78%">
<frame src="../$defaultleft" name="menu" scrolling="yes">
<frame src="../$defaultright" name="display" scrolling="yes">
</frameset>

</html>
EOF
    print STDOUT "done.\n" unless ($debug);
    close(INDEXFILE);
}

#
# img タグの alt 属性に「<」、「>」、「"」、「&」、「'」が入る文字を
# 書き出さないようにするため、それらが入らない文字だけを %altmsg に
# 格納。
#
# 引数
#	$debug		デバッグオプション
#
# 暗黙の引数 (入力)
#	@number		各行のナンバーを格納
#	%info_hash	ナンバーをキーに、サブジェクトを値に格納した連想配列
#
# 暗黙の引数 (出力)
#	%altmsg		alt 属性に書き出す値を格納
#
# 返却値
#	なし.
#
#
sub MakeAltMsg(){
    local($debug) = @_;
    local($num);
    print STDOUT "Creating Alt Attribute ...";
    foreach $num (@number){
	# 「<」、「>」、「"」、「&」、「'」が入る文字は
	# alt 属性には書き出さない。
	if ($info_hash{$num} =~ /\<|\>|\"|\&|\'/) {
	    $altmsg{$num} = '';
	} else {
	    $altmsg{$num} = $info_hash{$num};
	}
	print STDOUT "           ${num} : $info_hash{$num}\n" if ($debug);
	print STDOUT "                    alt=\"$altmsg{$num}\"\n" if ($debug);
    }
    print STDOUT "done.\n";
    return;
}


#
# 画像を張り込んだメニューファイルの作成サブルーチン
#    - グローバル変数の @number と %info_hash と %altmsg を内部で利用
#      (渡したかったけど, 渡せなかったのでサボる)
#
sub MakeMenu() {
    local($menudir, $htmldir, $title, $style_out,
	  $jpcode, $metajp, $imgtype_menu,
	  $upperlink, $upperurl, $lowerlink, $lowerurl,
	  $thumnail_mode, $thumbnail_file, $thumbnail_link,
	  $menufile, $anotherfile, $debug) = @_;
    print STDOUT "Creating Menu File ..." unless ($debug);
    # 文字コード変換
    &Jcode::convert(*upperlink, "$jpcode");
    &Jcode::convert(*upperurl, "$jpcode");
    &Jcode::convert(*lowerlink, "$jpcode");
    &Jcode::convert(*lowerurl, "$jpcode");
    &Jcode::convert(*thumbnail_link, "$jpcode");
    open(MENUFILE, "> $menufile")
	|| die "エラー: $menufile に書き込むことが出来ませんでした. /n";
    #
    # 画像の左に書き込む数字の生成 (頭のゼロを取り除く)
    #
    local(%leftnum, $num, $left);
    undef %leftnum;
    foreach $num (@number) {
	$left = sprintf("%d", $num);
	$leftnum{$num} = $left;
    }
    #
    # ヘッダの書き出し
    #
    print MENUFILE <<EOF;
<html>
<head>
  <meta http-equiv="Content-Type" content="text/html; charset=${metajp}">
  <title> $title Menu</title>
$style_out
</head>
<body>

EOF
    #
    # 上に付けるリンク
    #
    if ($upperlink && $upperurl) {
	print MENUFILE <<EOF;
<small>
  <a href="../${upperurl}" target="_top">${upperlink}</a>
</small>

EOF
    }
    #
    # サムネイルモードへのリンク
    #
    if ($thumbnail_mode && $thumbnail_file && $thumbnail_link) {
	print MENUFILE <<EOF;
<hr noshade size="1">
<center><small>
  <a href="../${thumbnail_file}" target="display">${thumbnail_link}</a>
</small></center>
<hr noshade size="1">

EOF
    }
    #
    # イメージメニューへのリンク
    #
    if ($anotherfile) {
	print MENUFILE <<EOF;
<center><small>
  [Image] &nbsp;
  <a href="../$anotherfile" target="menu">[Text]</a>
</small></center></small></center>
<hr noshade size="1">

EOF
    }
    #
    # 画像目次リンクのヘッダ部分書き出し
    #
    print MENUFILE <<EOF;
<table cellpadding="0" cellspacing="0" width="100%" align="center">
EOF
    foreach $num (@number){
	print MENUFILE <<EOF;
<tr align="center">
  <td valign="top"><br><small>$leftnum{$num}</small></td>
  <td align="center">
    <a href="../${htmldir}/${num}.html" target="display">
    <img src="../${menudir}/${num}.${imgtype_menu}" border="15" alt="$altmsg{$num}">
    </a>
  </td>
</tr>

EOF
    }
    #
    # 目次部分のフッタの書き出し
    #
    print MENUFILE <<EOF;
</table>

EOF
    #
    # 下に付けるリンク
    #
    if ($lowerlink && $lowerurl) {
	print MENUFILE <<EOF;
<hr noshade size="1">
<small><div align="right">
  <a href="../${lowerurl}" target="_top">${lowerlink}</a>
</div></small>

EOF
    }
    #
    # フッタの書き出し
    #
    print MENUFILE <<EOF;
</body>
</html>
EOF
    close(MENUFILE);
    # デバッグ用出力
    if ($debug) {
	print STDOUT "MakeMenu:\n" ;
	foreach $num (@number){
	    print STDOUT "           ${num} : $info_hash{$num}\n";
	    print STDOUT "                    alt=\"$altmsg{$num}\"\n";
	}
    }
    print STDOUT "done.\n" unless ($debug);
}

#
# メニューファイルの作成サブルーチン
#    - グローバル変数の @number と %info_hash を内部で利用
#      (渡したかったけど, 渡せなかったのでサボる)
#
sub MakeMenuTxt() {
    local($title, $style_out, $jpcode, $metajp,
	  $upperlink, $upperurl, $lowerlink, $lowerurl,
	  $thumnail_mode, $thumbnail_file, $thumbnail_link,
	  $menufile, $anotherfile, $debug) = @_;
    print STDOUT "Creating MenuTexT File ..." unless ($debug);
    # 文字コード変換
    &Jcode::convert(*upperlink, "$jpcode");
    &Jcode::convert(*upperurl, "$jpcode");
    &Jcode::convert(*lowerlink, "$jpcode");
    &Jcode::convert(*lowerurl, "$jpcode");
    &Jcode::convert(*thumbnail_link, "$jpcode");
    open(MENUFILE, "> $menufile")
	|| die "エラー: $menufile に書き込むことが出来ませんでした. /n";
    #
    # ヘッダの書き出し
    #
    print MENUFILE <<EOF;
<html>
<head>
  <meta http-equiv="Content-Type" content="text/html; charset=${metajp}">
  <title> $title Menu</title>
$style_out
</head>
<body>

EOF
    #
    # 上に付けるリンク
    #
    if ($upperlink && $upperurl) {
	print MENUFILE <<EOF;
<small>
  <a href="../${upperurl}" target="_top">${upperlink}</a>
</small>

EOF
    }
    #
    # サムネイルモードへのリンク
    #
    if ($thumbnail_mode && $thumbnail_file && $thumbnail_link) {
	print MENUFILE <<EOF;
<hr noshade size="1">
<center><small>
  <a href="../${thumbnail_file}" target="display">${thumbnail_link}</a>
</small></center>
<hr noshade size="1">

EOF
    }
    #
    # イメージメニューへのリンク
    #
    if ($anotherfile) {
	print MENUFILE <<EOF;
<center><small>
  <a href="../$anotherfile" target="menu">[Image]</a> &nbsp;
  [Text]
</small></center></small></center>
<hr noshade size="1">

EOF
    }
    #
    # 目次リンクのヘッダ部分書き出し
    #
    print MENUFILE <<EOF;
<ol><small>
EOF
    foreach $num (@number){
	print MENUFILE <<EOF;
<li><a href="../${htmldir}/${num}.html" target="display">$info_hash{$num}</a><br>

EOF
    }
    #
    # 目次部分のフッタの書き出し
    #
    print MENUFILE <<EOF;
</small></ol>

EOF
    #
    # 下に付けるリンク
    #
    if ($lowerlink && $lowerurl) {
	print MENUFILE <<EOF;
<hr noshade size="1">
<small><div align="right">
  <a href="../${lowerurl}" target="_top">${lowerlink}</a>
</div></small>

EOF
    }
    #
    # フッタの書き出し
    #
    print MENUFILE <<EOF;
</body>
</html>
EOF
    close(MENUFILE);
    # デバッグ用出力
    if ($debug) {
	print STDOUT "MakeMenu:\n" ;
	foreach $num (@number){
	    print STDOUT "           ${num} : $info_hash{$num}\n";
	}
    }
    print STDOUT "done.\n" unless ($debug);
}


#
# 各 HTML ファイル作成サブルーチン
#    - グローバル変数の @number と %info_hash と %altmsg を内部で利用
#      (渡したかったけど, 渡せなかったのでサボる)
#
sub MakeHtml {
    local($htmldir, $title, $style_out,
	  $previouslink, $nextlink, $imgtype,
	  $acktext, $acklink, $jpcode, $metajp, $debug) = @_;
    # デバッグ用メッセージヘッダ
    print STDOUT "MakeHtml:\n" if ($debug);

    # 文字コード変換
    &Jcode::convert(*previouslink, "$jpcode");
    &Jcode::convert(*nextlink, "$jpcode");
    #
    # ディレクトリの削除と作成
    #
    print STDOUT "Removing Old HTML Files ..." unless ($debug);
    print STDOUT "  Remove ./${htmldir}, and Mkdir ${htmldir}" if ($debug);

    system("/bin/rm -rf ./${htmldir}/") if -d "./${htmldir}";
    mkdir("${htmldir}", 0755);

    print STDOUT "done.\n". "Creating HTML Files ..." unless ($debug);

    # 現在のディレクトリの場所を取得
    local($current_dir) = `pwd`;
    chomp $current_dir;
    # ディレクトリ移動
    chdir("./${htmldir}");
    print STDOUT "  Move to $htmldir ...\n" if ($debug);

    # 要素番号として $element_num を用意
    local($element_num) = 0;


    while ($number[$element_num]) {
	local($allpage_num) = $#number + 1;
	local($previous_num) = $element_num - 1;
	local($next_num) = $element_num + 1;
	local($htmlnumber) = $number[$element_num];
	local($htmlfile) = "${htmlnumber}.html";
	#
	# ヘッダの作成
	#
	open(HTMLFILE, "> $htmlfile")
	    || die "エラー: $htmlfile に書き込むことが出来ませんでした. /n";
	print HTMLFILE <<EOF;
<html>
<head>
  <meta http-equiv="Content-Type" content="text/html; charset=${metajp}">
  <title>$title [$info_hash{$htmlnumber}]</title>
$style_out
</head>
<body>
EOF
	#
	# 前後のページへのリンク, 現ページ/ページ総数, 小題の書き出し
	#
	print HTMLFILE <<EOF;

<hr noshade size="1">
<table width="90%">
  <tr>
    <td width="30%">
EOF
	# 最初のページと最後ページの場合はリンクを貼れないので
	# 条件分岐する.

	# 最初のページは [前頁] にリンクを貼らない
	if ($previous_num < 0) {
	    print HTMLFILE <<EOF;
      ${previouslink}
      <A HREF="$number["$next_num"].html">${nextlink}</A>
EOF
	}
	# 最後のページは [後頁] にリンクを貼らない
	elsif ($next_num > $#number) {
	    print HTMLFILE <<EOF;
      <A HREF="$number["$previous_num"].html">${previouslink}</A>
      ${nextlink}
EOF
	}
	# その他の頁は両方にリンク
	else {
	    print HTMLFILE <<EOF;
      <A HREF="$number["$previous_num"].html">${previouslink}</A>
      <A HREF="$number["$next_num"].html">${nextlink}</A>
EOF
	}


	print HTMLFILE <<EOF;
    </td>
    <td width="30%" align="center">
    $next_num / $allpage_num
    </td>
    <td align="right">
      $info_hash{$htmlnumber}
    </td>
  </tr>
</table>
<hr noshade size="1">

EOF
	#
	# 画像貼りつけ部分
	#
	print HTMLFILE <<EOF;
<img src="../../${slidedir}/${htmlnumber}.${imgtype}" alt="$altmsg{$htmlnumber}">

EOF
	#
	# ページ末尾のメッセージ出力
	#
	if ($acktext) {
	    print HTMLFILE <<EOF;
<hr noshade size="1">
<div align="right">
EOF
	    local($_) = $acktext;
	    if ($acklink_out = $acklink) {
		${acklink_out} = "<b><a href=\"${acklink_out}\" target=\"_top\">";
		s/::/${acklink_out}/ || s/^/${acklink_out}/;
		${acklink_out} = "</a></b>";
		s/::/${acklink_out}/ || s/$/${acklink_out}/;
	    }
	    print HTMLFILE "$_\n";
	    print HTMLFILE ": at ", &FmtTime, "</div>\n";
	}
	#
	# フッタの作成
	#
	print HTMLFILE <<EOF;
</body>
</html>
EOF
	close(HTMLFILE);
	print STDOUT "${element_num}, " unless ($debug);
	# デバッグ用出力
	if ($debug) {
	    print STDOUT "  page: ${element_num}\n".
		         "    allpage : ${allpage_num}\n".
		         "    previous: ${previous_num}\n".
		         "    next    : ${next_num}\n".
		         "    html_num: ${htmlnumber}\n".
		         "    htmlfile: ${htmlfile}\n";
	}
	# 次のループへ
	$element_num++;
    }
    # 元のディレクトリに戻る
    chdir("$current_dir");
    print STDOUT "  Back to $current_dir.\n" if ($debug);
    print STDOUT "done.\n" unless ($debug);
    return;
}

#
# 時間出力用サブルーチン
#
sub FmtTime {
	local($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst)
	 = localtime time;
	sprintf "%4d/%02d/%02d %02d:%02d:%02d",
		$year + 1900, $mon + 1, $mday, $hour, $min, $sec;
}

#
# サムネイル作成サブルーチン
#    - グローバル変数の @number と %info_hash と %altmsg を内部で利用
#      (渡したかったけど, 渡せなかったのでサボる)
#
sub MakeThumbnail() {
    local($title, $style_out, $jpcode, $metajp, $htmldir,
	  $thumnail_mode, $thumbnail_file, $thumbdir,
	  $thumbnail_link, $thumbnail_col_num,
	  $imgtype_thumb, $acktext, $acklink, $debug) = @_;
    # デフォルトの列の数
    local($thumbnail_col_num_def) = "3";

    # デバッグ用メッセージ出力
    print STDOUT "MakeThumbnail: \$thumbnail_mode = $thumbnail_mode\n" if ($debug);
    # $thumnail_mode が偽 (0) ならば何もせずに終了
    return unless ($thumbnail_mode);

    # ユーザ用, デバッグ用メッセージ出力
    print STDOUT "Creating Thumbnail File ..." unless ($debug);
    if ($debug) {
	print STDOUT "  \$thumbnail_file = $thumbnail_file\n".
	             "  \$thumbdir = $thumbdir\n".
	             "  \$thumbnail_link = $thumbnail_link\n".
	             "  \$thumbnail_col_num = $thumbnail_col_num\n".
	             "  \$thumbdir = $thumbdir\n";
    }
    # 文字コード変換
    &Jcode::convert(*thumnail_link, "$jpcode");
    # 例外処理
    unless ($thumbnail_col_num =~ /^\d+$/){
	print STDOUT "  Modify \$thumbnail_col_num ".
	    "from $thumbnail_col_num to $thumbnail_col_num_def.\n" if ($debug);
	$thumbnail_col_num = $thumbnail_col_num_def;
    }
    # ファイルオープン
    open(THUMBFILE, "> $thumbnail_file")
	|| die "エラー: $thumbnail_file に書き込むことが出来ませんでした. /n";
    #
    # ヘッダの書き出し
    #
    print THUMBFILE <<EOF;
<html>
<head>
  <meta http-equiv="Content-Type" content="text/html; charset=${metajp}">
  <title> $title Thumbnail</title>
$style_out
</head>
<body>

<hr noshade size="1">
<center>
  $title
</center>
<hr noshade size="1">


EOF
    #
    # テーブルのヘッダ部分書き出し
    #
    print THUMBFILE <<EOF;
<table border="0" cellspacing="20" cellpadding="0" align="center">
EOF

    #
    # サムネイル部分作成
    #

    # 要素番号として $element_num を用意
    local($element_num) = 0;

    while ($number[$element_num]) {
	local($thumbnum) = $number[$element_num];
	local($amari) = $thumbnum % $thumbnail_col_num;

	#
	# テーブルの配置によって <tr> のつけたしをおこなう.
	# 左端 : 3n+1 ($amari == 1)
	# 右端 : 3n   ($amari == 0)
	if ($amari == 1) {
	    print THUMBFILE <<EOF;
  <tr align="center" valign="bottom">
EOF
	}

        print THUMBFILE <<EOF;
    <td align="center" valign="top">
      <a href="../${htmldir}/${thumbnum}.html">
        <img src="../${thumbdir}/${thumbnum}.${imgtype_thumb}" border="2" alt="$altmsg{$thumbnum}">
      </a>
      <br>
      <small>
         $info_hash{$thumbnum}
      </small>
    </td>
EOF
	if ($amari == 0) {
	    print THUMBFILE <<EOF;
  </tr>
EOF
	}
	# デバッグ用出力
	if ($debug) {
	    print STDOUT "    ${thumbdir}/${thumbnum}.${imgtype_thumb} -> ".
		           "./${htmldir}/${thumbnum}.html\n";
	}
	$element_num++;
    }

    #
    # テーブルで欠けている部分を追加
    #
    local($total) = $#number + 1 ; # 画像の数を取得
    local($total_amari) = $total % $thumbnail_col_num;
    local($total_nokori) = $thumbnail_col_num - $total_amari;
    local($i);
    # デバッグ用出力
    if ($debug) {
	print STDOUT "  \$total        = $total\n".
	             "  \$total_amari  = $total_amari\n".
	             "  \$total_nokori = $total_nokori\n";
    }
    
    for ($i = $total_nokori; $i > 0; $i--) {
	print THUMBFILE <<EOF;
    <td>&nbsp;</td>
EOF
    }
    if ($total_nokori) {
	print THUMBFILE <<EOF;
  </tr>
EOF
    }

    #
    # テーブルのフッタ部分書き出し
    #
    print THUMBFILE <<EOF;
</table>
EOF

    #
    # ページ末尾のメッセージ出力
    #
    if ($acktext) {
	print THUMBFILE <<EOF;
<hr noshade size="1">
<div align="right">
EOF
	local($_) = $acktext;
	if ($acklink_out = $acklink) {
	    ${acklink_out} = "<b><a href=\"${acklink_out}\" target=\"_top\">";
	    s/::/${acklink_out}/ || s/^/${acklink_out}/;
	    ${acklink_out} = "</a></b>";
	    s/::/${acklink_out}/ || s/$/${acklink_out}/;
	}
	print THUMBFILE "$_\n";
	print THUMBFILE ": at ", &FmtTime, "</div>\n";
    }

    #
    # フッタの書き出し
    #
    print THUMBFILE <<EOF;
</body>
</html>
EOF
    close(THUMBFILE);
    
    print STDOUT "done.\n" unless ($debug);
}



# 終了メッセージの表示 (デバッグモード時)
#
sub EndMessage() {
    local($debug) = @_;
    print STDOUT "EndMessage: Execute Successfully!!\n" if ($debug);
    print STDOUT "Complete.\n" unless ($debug);
}


##################################################
#####                                        #####
##         README.TXT 出力サブルーチン         ###
#####      (内容は README.TXT のコピー)      #####
##################################################


sub PrintReadme() {
    print STDOUT <<'EOF';
☆ dcppt (GFD Dennou Club PowerPoint 整形) セット version 1 ☆

   PowerPoint ファイルを見やすい HTML 形式に変換する Perl スクリプト

■ 概要

   PowerPoint はプレゼンテーションに便利だが, それを万人に見えるような
   HTML 形式に変換するのが面倒くさい.
   (PowerPoint で「HTML形式で保存」としてもそこそこ見ることができるの
    だが, Windows でないと綺麗に見れないことがあったり, 画像によっては
    HTML 化出来ないものもある)

   このスクリプトのセットを使うことで, どのプラットフォームでも
   見ることができ, そこそこ見やすい(?)HTMLファイルを作成することが出来る.


■ 梱包物

   dcppt.tar.gz ファイルを展開することで以下のファイルができあがる
   はずである.

     ・ dcppt.pl         … HTML フォーマットを作成するPerlスクリプト
     ・ sample           … サンプル
     ・ README.TXT       … このファイル

   (開発者用ツール)
     ・ Makefile         … make 用のファイル


■ サンプル (おまけ)

   sample ディレクトリには, dcppt のサンプルが置いてある.
   (このサンプルを使えるかどうかで, システム上で dcppt が
    使えるかどうかのテストにもなるだろう)

   ・ src/             … 情報元 
          title.txt    … サンプルのタイトルファイル
          menu.txt     … サンプルのメニューファイル
          Slide*.PNG   … スライドを画像化したファイル
                          (PowerPoint の「Web ページで保存」で作成)
   ・ Makefile	       … make 用ファイル
   ・ dcppt.ppt	       … サンプルの ppt ファイル
   ・ pub/             … 公開用もの置場



   □ HTML 生成

      sample に移動し, dcppt コマンドを実行する

        $ cd sample
	$ perl ../dcppt.pl -x

      すると, pub ディレクトリ以下に, index.html 等,
      生成された HTML 等ができるはずである.

      ブラウザで実際に見てみて欲しい

      
   □ 初期化

      生成した HTML を消去し, 初期化したい場合は,

        $ make clean

      とすると良い.


■ HTML 作成手順

   ◇ Windows で行う作業

      まず, Windows 側で ppt ファイルから画像ファイルを生成する必要がある.

      PowerPoint の「ファイル」→「Webページで保存」として、
      「ファイルの種類」に gif, jpeg, png のどれかを選択する。
      「プレゼンテーションのスライドを全てエクスポートしますか？」に
      「はい」と答える。

      すると, PowerPoint のファイル名と同じな前のディレクトリに
      画像が保存されるので, それごと Linux 環境のマシン
      (Web サーバでいいでしょう) にコピーする.


   ◇ Linux (Web サーバ)で行う作業

      0. dcppt.pl が利用できるためには...

	 a. perl が使える
	    - 専攻 Web サーバなら必ず使える
	    - perl 5.6.1 で動作確認済

	    - ※ パスに関して ※
		 dcppt.pl は Debian の環境で作成したので
		 perl のパスが /usr/bin/perl となっている.
		 他の OS の場合はパスが異なる場合があるので
		 変更すること
		 (perl のパスは which perl とすると知ることができる)

		 変更は, dcppt.pl 本体の一番上の行の
		 「#!/usr/bin/perl」を書き換えればよい

	 b. Jcode が使える (日本語文字コード用ライブラリです)
	   - Debian GNU/Linux 7.0 ならば, libjcode-pm-perl パッケージを
	     インストールすることで, システムに組み込める

	 c. ImageMagic が使える (画像処理のためです)
	    - 専攻 Web サーバなら必ず使える
	    - Debian GNU/Linux 3.0 ならば, imagemagic パッケージを
	      インストールすることで, システムに組み込める

         なお, 動作確認のために sample ディレクトリが用意されている.
	 もしも下記の手順にしたがってうまく行かない場合,
	 sample 以下で以下のように動作チェックを行なって欲しい.

	    $ make clean
	    $ make

      1. 画像ファイルが入っているディレクトリを src に変更

            $ mv <ディレクトリ> src

      2. dcppt -x 実行

            $ dcppt -x

      3. 処理選択

             設定ファイルである src/Config.dcppt が見当たりません.
               1. dcppt デフォルトの設定で処理をするなら "y",
               2. 処理を中断するなら                     "n",
               3. src/Config.dcppt テンプレートを出力するなら"m"
             を入力してください [Y/n/m]: 

	 と聞かれるがそのまま Enter

      4. タイトル入力

             タイトルを記入して下さい.
	       Title []:

	 にスライドのタイトルを入力. (空白でも実行は可能)
	  
      5. 生成

         処理が開始される. (一回目は画像の処理がおこなわれる
	 ので少々時間がかかる).

	     Get Names of Image in src ...  done.
	     Prepare Names of image files in publish directory ...  done.
	     Get ImageType from pub/menu ...done.
	     Get ImageType from pub/slide ...done.
	     Get ImageType from pub/thumbnail ...done.
	     Get Slide Informations from src/menu.txt ...  done
	     Recognize Japanese Code is euc ...done.
	     Creating Index File ...done.
	     Creating Menu File ...done.
	     Creating Menu File ...done.
	     Removing Old HTML Files ...done.
	     Creating HTML Files ...0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, done.
	     Creating Thumbnail File ...done.
	     Complete.

	 最後に「Complete」がでると完了.
	
      7. できあがり確認

         pub/index.html に生成されたもののインデックスが置かれるので
	 それをみてみる.


   ◇ おまけ

      1. メニューを書き込む

         src/menu.txt に小題一覧を書き込むことで文字メニューに
	 書き出すことが出来る.
	 
	 書式は

		001:<1つめのスライドの題目>
		002:<2つめのスライドの題目>
		003:<3つめのスライドの題目>
			    :

	 となる. 具体的には

		001:はじめに
		002:目次
		003:BIOSって?
		     :

	 と書き出せば良い.

	 ※ sample 内の menu.txt を参照して欲しい.

	 書き終わったら, 先に dcppt.pl を実行した場所で

	    $ dcppt.pl -q

	 を実行すると良い. どうようにタイトルを書き換えたい場合は
	 src/title.txt を書き換えると良い.
	 
      
      2. 書式をいろいろ変更

         設定ファイル src/Config.dcppt を書き換えることで見栄え等を
	 変更することが可能である. src/Config.dcppt は

	    $ dcppt.pl -x

	 として, 

             設定ファイルである src/Config.dcppt が見当たりません.
               1. dcppt デフォルトの設定で処理をするなら "y",
               2. 処理を中断するなら                     "n",
               3. src/Config.dcppt テンプレートを出力するなら"m"
             を入力してください [Y/n/m]:
	     
         に「m」を押して Enter することで生成される.
	    
	 src/Config.dcppt を emacs や jvim などの日本語表示可能な
	 エディタで開き, 編集する.

	 ・ ロケーション

	    各種ファイルやディレクトリの設定が可能であるが, 基本的に変
	    更の必要は無い. (…というか下手に変更するとリンク切れなど
	    が起こる可能性が高いので変更しないことを強く推奨する). 変
	    更の際には, 上記で用意してきたファイルやディレクトリの名前
	    をそれ似合わせて変更する必要がある.


	 ・ メニューファイル選択

	      # メニューファイル (イメージ)
	      $menufile = "${pubdir}/menu.html";
	      #$menufile = "";      # 画像版メニューファイルがいらない時
	      
	      # メニューファイル (テキスト)
	      $menutxtfile = "${pubdir}/menutxt.html";
	      #$menutxtfile = "";   # テキスト版メニューファイルがいらない時

            メニューファイルには画像版とテキスト版が用意されているが,
	    どちらかしか必要ない場合, 上ではコメントアウトされている
	    ものの行頭の「#」を取り除くと良いだろう.

	    ただしその場合, 以下の設定もそれにあわせないと
	    メニューが表示されなくなったりするので注意.

	      # 起動時のデフォルトのページ
	      # 左ページ
	      $defaultleft  = "$menufile";     # イメージメニュー
	      #$defaultleft  = "$menutxtfile";  # テキストメニュー
	      # 右ページ
	      $defaultright = "$thumbnail_file";    # サムネイル
	      #$defaultright = "$htmldir/001.html";  # 1 ページ目

	      
	 ・ 画像サイズ

	      ##############################
	      ###  画像サイズ
	      ##
	      # メニューのサイズ
	      $menusize = "120x90";
	      # スライドのサイズ
	      $slidesize = "640x480";
	      # サムネイルのサイズ
	      $thumbsize = "200x150";

	    これを編集することで画像サイズの変更が可能である.


	 ・ 色深度の設定

	      ##############################
	      ###  色深度の設定
	      ##
              $depth = "8";      # 0: -depth オプションを利用しない

	    これを編集することで convert コマンドの -depth オプション
	    に設定する値を変更できる. 数が少ないほど画像サイズは
	    小さくなるが, 画質は荒くなる. 数字を 0 にすると, -depth
            オプションを利用せずに convert のデフォルトの処理に任せる.


	 ・ 文字コード

	      # 出力されるファイルの日本語文字コード
	      # "jis", "euc", "sjis", "utf8" のどれかを選択
	      $jpcode = "utf8";
	      $system_code = "utf8";

	    $jpcode のデフォルトは utf8 だが, Windows 環境の人などは
	    sjis でも良いのかも知れない.  (上記に該当するもの以外を入
	    力した場合は utf8 になるので注意.)

	    なお, $system_code は dcppt.pl を動作させるプラットフォー
	    ムのデフォルト文字コードに必ず合わせて欲しい.


	 ・ スタイル設定

	      # 0: スタイル無し, 1: スタイルあり
	      $style_mode = "1";
	          # テキストの色
	      $style_textcolor       = '#999900';
		          :

	    文字の色や背景色等の設定が可能である.
	    各設定変数を設定して欲しい.
	    設定が面倒な場合は $style_mode を 0 にすることで,
	    装飾無しにもできる.

	      
	      # 前後のページへのリンクに用いる文字列
	      $previouslink = '[Prev]';
	      $nextlink     = '[Next]';

	    スライドの前後のページに飛ぶ際のリンクのメッセージ
	    を設定できる.

	      $upperlink = '▲Parent Dirctory';
	      $upperurl  = '';
	      $lowerlink = 'ppt ファイル▼';
	      $lowerurl  = '';

	    左に出るメニューの上下にリンクを付けることができる.
	    $upperlink, $lowerlink で文字列の指定を,
	    $upperurl, $lowerurl が URL の指定を行なう.
	    上の $upperlink や $lowerlink は例なので,
	    変更しても構わない.
	    …が, それぞれ, どちらかでも空になっていると
	    リンクそのものができないので注意.


	 ・ サムネイル

	      # 0: サムネイル無し, 1: サムネイルあり
	      $thumbnail_mode    = "1";
	          # サムネイルへのリンクのメッセージ
	      $thumbnail_link    = "[[ Thumbnail ]]";
		  # サムネイルの列の数
	      $thumbnail_col_num = "3";

	    サムネイルの設定を行なう.

	 書き終わったら, 先に dcppt.pl を実行した場所で

	    $ dcppt.pl -q

	 を実行すると良い. どうようにタイトルを書き換えたい場合は
	 src/title.txt を書き換えると良い.

EOF
}
