#!/usr/bin/env python3
#
# Test interaction with gdb, i.e. the cysignals-CSI script
#
# NOTE: this test is known to be broken in some cases, see
# https://github.com/sagemath/cysignals/pull/52
#

import unittest

import os
import sys
from subprocess import Popen, PIPE
from tempfile import mkdtemp
from shutil import rmtree

class TestGDB(unittest.TestCase):
    def setUp(self):
        # Store crash logs in a temporary directory
        self.crash_dir = mkdtemp()
        self.env = dict(os.environ)
        self.env["CYSIGNALS_CRASH_LOGS"] = self.crash_dir

    def tearDown(self):
        rmtree(self.crash_dir)

    def test_gdb(self):
        # Run a Python subprocess which we intentionally crash to inspect the
        # crash logfile.
        p = Popen([sys.executable], stdin=PIPE, env=self.env)
        with p.stdin as stdin:
            stdin.write(b"from cysignals.tests import *\n")
            stdin.write(b"unguarded_dereference_null_pointer()\n")

        ret = p.wait()
        self.assertLess(ret, 0)

        # Check crash log
        logs = [os.path.join(self.crash_dir, fn)
                for fn in os.listdir(self.crash_dir) if fn.endswith(".log")]
        self.assertEqual(len(logs), 1)
        with open(logs[0]) as logf:
            log = logf.read()
            self.assertIn("Stack backtrace", log)
            self.assertIn("Cython backtrace", log)
            self.assertIn("unguarded_dereference_null_pointer ()", log)
            self.assertIn("cdef void dereference_null_pointer() noexcept nogil:", log)


if __name__ == '__main__':
    unittest.main()
