// @HEADER
// *****************************************************************************
//   Zoltan2: A package of combinatorial algorithms for scientific computing
//
// Copyright 2012 NTESS and the Zoltan2 contributors.
// SPDX-License-Identifier: BSD-3-Clause
// *****************************************************************************
// @HEADER

/*! \file xmlToHeaderDefinition.cpp
    \brief Builds the parameter header file required at compile time.
 */

#include <iostream>
#include <fstream>
#include <exception>
#include <string>
#include <cstring>
#include <sstream>

using std::cout;
using std::endl;
using std::string;
using std::ofstream;
using std::ifstream;
using std::ostringstream;

static string fixLine(string oldLine)
{
  string quote("\"");
  
  size_t pos = oldLine.find(quote);
  if (pos == string::npos)
    return oldLine;

  string newLine;
  string quotedQuote("\\\"");

  for (size_t i=0; i < oldLine.size(); i++){
    char c = oldLine[i];
    if (c == quote[0])
      newLine.append(quotedQuote);
    else
      newLine.append(&c,1);
  }
  return newLine;
}

int main(int argc, char  *argv[])
{
  if (argc < 3){
    cout << "Usage: " << argv[0] << " xmlFileName hppFileName" << endl;
    return 1;
  }

  char *xmlFile = argv[1];
  char *hppFile = argv[2];

  ofstream oFile;
  ifstream iFile;

  try{
    oFile.open(hppFile);
  }
  catch(std::exception &e){
    cout << "Error: " << e.what() << " " << hppFile << endl;
    return 1;
  }

  try{
    iFile.open(xmlFile);
  }
  catch(std::exception &e){
    oFile.close();
    cout << "Error: " << e.what() << " " << xmlFile << endl;
    return 1;
  }

  oFile << "// \n";
  oFile << "// This file was automatically generated by CMake\n";
  oFile << "// with the following command:\n";
  oFile << "// " << argv[0] << " " << argv[1] << " " << argv[2] << ".\n";
  oFile << "// \n";

  oFile << "#ifndef ZOLTAN2_PARAMETER_DEFINITION_HEADER\n";
  oFile << "#define ZOLTAN2_PARAMETER_DEFINITION_HEADER\n";
  oFile << "\n#define ZOLTAN2_XML_PARAMETER_STRING \"";

  char lineBuf[1024];
  iFile.clear();
  bool go=false;
  int sanity=10000;

  while (sanity--){
    iFile.getline(lineBuf, 1024);
    if (!iFile.good())
      break;
    string line(lineBuf);
    if (go && (line.size() > 1)) {
      oFile << " \\\n  " << fixLine(line);
    }
    else if (!go){
      size_t pos = line.find("ParameterList");
      if (pos == string::npos)
        continue;
      oFile << " \\\n  " << fixLine(line);
      go = true;
    }
  }

  if (go == false){
    iFile.close();
    oFile.close();
    cout << "Error: ParameterList XML definition not found." << endl;
    return 1;
  }
      
  iFile.close();

  oFile << "\"\n\n#endif  //ZOLTAN2_PARAMETER_DEFINITION_HEADER\n";

  oFile.close();

  return 0;
}

