/*
    This file is part of Msc-generator.
    Copyright (C) 2008-2022 Zoltan Turanyi
    Distributed under GNU Affero General Public License.

    Msc-generator is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Msc-generator is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with Msc-generator.  If not, see <http://www.gnu.org/licenses/>.
*/
/** @file blockcsh.h The declaration for the BlockCsh class.
* @ingroup libblock_files */

#ifndef BLOCK_CSH_H
#define BLOCK_CSH_H

#include "csh.h"
#include "blockstyle.h"

namespace block {

/** Helper structure to contain a box name and its position in the file.*/
struct CshStringWithPos
{
    std::string name;
    CshPos file_pos;
    CshStringWithPos(std::string_view n, const CshPos &r) :name(n), file_pos(r) {}
};

using CshStringWithPosList = std::vector<CshStringWithPos>;

/** Enumeration listing the potential keywords before the 'use' command.*/
enum EUseKeywords
{
    USE_KEYWORD_NONE = 0,
    USE_KEYWORD_BLOCKS = 1,
    USE_KEYWORD_ARROWS = 2,
    USE_KEYWORD_ALL = USE_KEYWORD_BLOCKS | USE_KEYWORD_ARROWS,
};

constexpr EUseKeywords operator |(EUseKeywords a, EUseKeywords b) { return EUseKeywords(unsigned(a)|unsigned(b)); }
constexpr EUseKeywords operator &(EUseKeywords a, EUseKeywords b) { return EUseKeywords(unsigned(a)&unsigned(b)); }

struct UseCommandParseHelper
{
    gsl::owner<AttributeList *> attrs;
    EUseKeywords keywords;
};

/** Additional context information for block language.*/
struct BlockCshContext
{
    std::string prefix; ///<The prefix in the current context
    bool pedantic;      ///<The value of the pedantic option
    int def_shape;      ///<The default shape in this context
    BlockCshContext() = default;
    BlockCshContext(std::string_view p, bool pe, int s) : prefix (p), pedantic(pe), def_shape(s) {}
};

struct BlockMention {
    CshPos pos;
    std::string name;    ///<If prefix is empty, this is the full name, else the name the user specified (latter user for forward references)
    std::optional<std::string> prefix; ///<The prefix the user mentioned 'name' in. If empty optional, 'name' is already a resulved, full name.
    BlockMention(const CshPos &p, std::string_view n, std::optional<std::string_view> pr = {})
        : pos(p), name(n), prefix(pr ? std::string{*pr} : std::optional<std::string>{}) {}
};

/** Coloring for Block. */
class BlockCsh : public Csh
{
    /** Pushes the parent name to the context stack duplicating the other fields.*/
    void PushContext2() { bool p = Context2.back().pedantic; int s = Context2.back().def_shape;  Context2.emplace_back(parent_name, p, s); }
public:
    std::vector<BlockCshContext> Context2;  ///<Contains a stack of additional context information
    std::string parent_name;            ///<A temporary storage of a name from the header of the block until the opening of the scope for its content.
    std::map<std::string, int> Blocks;  ///<A list of all defined entity (full name) mapping to its shape number. -1 means the 'box' shape. Only the first block of such name is stored, if there are many, 'second' is set to -2.
    std::string prefix_at_hint_loc;     ///<If addEntityNamesAtEnd is not empty, this captures the prefix of the context we hint at.
    bool hadProcReplay;                 ///<If we had a replayed procedure so far. In this case we accept all block names as valid.
    std::vector<BlockMention> BlockMentions; ///<All the COLOR_ENTITYNAME blocks, with the fullname of the entity they refer to (used for entity rename).
    BlockCsh(Csh::FileListProc proc, void *param);
    ~BlockCsh() override = default;
    std::unique_ptr<Csh> Clone() const override { return std::make_unique<BlockCsh>(*this); }

    /** Push the context stack copying what was on top.*/
    void PushContext() { Csh::PushContext(); PushContext2(); }
    /** Push the context stack  with an empty context.*/
    void PushContext(bool f, EContextParse p) { Csh::PushContext(f, p); PushContext2(); }
    void PopContext() { Csh::PopContext(); Context2.pop_back(); }  ///<Pop the context stack.

    /** Return the prefix in the current context.*/
    std::string GetCurrentPrefix() const { return Context2.size() ? Context2.back().prefix : std::string(); }

    // Inherited via Csh
    void FillNamesHints() override;
    void AddDesignOptionsToHints() override;
    void AddOptionsToHints() override;

    void AddKeywordsToHints(bool include_dir, bool include_join_around, bool include_multi_copy);
    void AddBlockKeywordsToHints() { AddKeywordsToHints(false, false, false); }
    void AddLineBeginToHints();
    void AddUseKeywordsToHints(EUseKeywords e= USE_KEYWORD_NONE);
    void AddCloneActionKeywordsToHints();
    void AddBeforeToHints();
    void AddMarkExtendToHints(bool mark, bool extend);
    void AddPortsToHints(int shape);
    void AddWhatToUpdateToHints(bool include_keywords);

    void BeforeYaccParse(const std::string& input, int cursor_p) override;
    void ParseText(const std::string& input, int cursor_p, bool pedantic) override;
    bool HandleBlockNamePlus(std::string_view aname, std::string_view avalue = {}, EHintSourceType rewrite_source = EHintSourceType::ENTITY);
    std::pair<int, std::string_view> SearchEntityName(std::string_view string) const;
    void AddEntityNamesAtTheEnd(std::string_view msg) override;
    void AfterYaccParse() override;
    /** Called when the user updates attributes of blocks, like "a" or "below x a" or "a [color=red]".
     * We will define the blocks listed if not yet defined*/
    void UpdateBlocks(const CshStringWithPosList *);
    /** Drop all block mentions in this range (assumed to be added last)*/
    void DropBlockMentions(CshPos pos);

    void AddCSH_LineBeginSoloString(CshPos pos, std::string_view string);
    void AddCSH_AfterMultiPartial(CshPos pos, std::string_view string);
    int AddCSH_BlockName(const CshPos&pos, std::string_view name, EColorSyntaxType cinstead = COLOR_ERROR);
    int AddCSH_BlockNameOrNew(const CshPos&pos, std::string_view name, EColorSyntaxType cinstead = COLOR_ERROR);
    void UpdateCSH_ArrowEndAtLineBegin(const CshPos& pos, std::string_view arrow_end);
    bool AddCSH_NewBlock(const CshPos &pos, std::string_view pname, int shape);
    bool AddCSH_NewBlock(int shape) { return AddCSH_NewBlock({}, {}, shape); }
    void AddCSH_EntityName(const CshPos&pos, std::string_view name) { _ASSERT(0); }//Use AddCSH_BlockName() instead. }
    void AddCSH_BlockNameAsAttrValue(const CshPos &pos, std::string_view name);
    void AddCSH_AttrValue_CheckAndAddEscapeHint(const CshPos &pos, std::string_view value, std::string_view name) override;
    void AddCSH_NameList(const CshStringWithPosList *l, const CshPos &pos, EColorSyntaxType cname,
                         EColorSyntaxType cbetween, EColorSyntaxType cnumber = EColorSyntaxType::COLOR_ERROR,
                         EColorSyntaxType cinstead = EColorSyntaxType::COLOR_ERROR);

    std::pair<std::string_view, size_t> EntityNameUnder(long pos) const override;
    std::string_view AskReplace(std::string_view full_entity, int pos) const override;
    std::string ReplaceEntityName(std::string_view full_entity, int pos,
                                          std::string_view replace_to,
                                          long lStart, long lEnd) const override;
};

}; //namespace

/** Yacc generated function to do the csh parsing of a graph. defined in gv_csh_lang.cpp */
extern void BlockCshParse(block::BlockCsh &csh, const char *buff, unsigned len);

#endif