/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.lazy.iterator;

import java.util.Iterator;
import java.util.NoSuchElementException;

import com.gs.collections.impl.factory.Lists;
import com.gs.collections.impl.list.Interval;
import com.gs.collections.impl.test.Verify;
import org.junit.Assert;
import org.junit.Test;

/**
 * JUnit test for {@link TakeIterator}.
 */
public class TakeIteratorTest
{
    @Test
    public void iterator()
    {
        Interval list = Interval.oneTo(5);

        Iterator<Integer> iterator1 = new TakeIterator<Integer>(list.iterator(), 2);
        assertElements(iterator1, 2);

        Iterator<Integer> iterator2 = new TakeIterator<Integer>(list, 5);
        assertElements(iterator2, 5);

        Iterator<Integer> iterator3 = new TakeIterator<Integer>(list, 10);
        assertElements(iterator3, 5);

        Iterator<Integer> iterator4 = new TakeIterator<Integer>(list, 0);
        assertElements(iterator4, 0);

        Iterator<Integer> iterator5 = new TakeIterator<Integer>(Lists.fixedSize.<Integer>of(), 0);
        assertElements(iterator5, 0);
    }

    private static void assertElements(Iterator<Integer> iterator, int count)
    {
        for (int i = 0; i < count; i++)
        {
            Assert.assertTrue(iterator.hasNext());
            Assert.assertEquals(Integer.valueOf(i + 1), iterator.next());
        }
        Assert.assertFalse(iterator.hasNext());
    }

    @Test
    public void remove()
    {
        Verify.assertThrows(UnsupportedOperationException.class, () -> new TakeIterator<Integer>(Lists.fixedSize.<Integer>of(), 0).remove());
    }

    @Test
    public void noSuchElementException()
    {
        Verify.assertThrows(NoSuchElementException.class, () -> {
            new TakeIterator<Integer>(Lists.fixedSize.<Integer>of(), 0).next();
        });

        Verify.assertThrows(NoSuchElementException.class, () -> {
            new TakeIterator<Integer>(Lists.fixedSize.of(1, 2, 3), 0).next();
        });
    }
}
