;; AMD K6/K6-2 Scheduling
;; Copyright (C) 2002-2018 Free Software Foundation, Inc.
;;
;; This file is part of GCC.
;;
;; GCC is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 3, or (at your option)
;; any later version.
;;
;; GCC is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GCC; see the file COPYING3.  If not see
;; <http://www.gnu.org/licenses/>.
;;
;; The K6 architecture is quite similar to PPro.  Important difference is
;; that there are only two decoders and they seems to be much slower than
;; any of the execution units.  So we have to pay much more attention to
;; proper scheduling for the decoders.
;; FIXME: We don't do that right now.  A good start would be to sort the
;;        instructions based on length.
;;
;; This description is based on data from the following documents:
;;
;;    "AMD-K6 Processor Data Sheet (Preliminary information)"
;;    Advanced Micro Devices, Inc., 1998.
;;
;;    "AMD-K6 Processor Code Optimization Application Note"
;;    Advanced Micro Devices, Inc., 2000.
;;
;; CPU execution units of the K6:
;;
;; store	describes the Store unit.  This unit is not modelled
;;		completely and it is only used to model lea operation.
;;		Otherwise it lies outside of any critical path.
;; load		describes the Load unit
;; alux		describes the Integer X unit
;; mm		describes the Multimedia unit, which shares a pipe
;;		with the Integer X unit.  This unit is used for MMX,
;;		which is not implemented for K6.
;; aluy		describes the Integer Y unit
;; fpu		describes the FPU unit
;; branch	describes the Branch unit
;;
;; The fp unit is not pipelined, and it can only do one operation per two
;; cycles, including fxcg.
;;
;; Generally this is a very poor description, but at least no worse than
;; the old description, and a lot easier to extend to something more
;; reasonable if anyone still cares enough about this architecture in 2004.
;;
;; ??? fxch isn't handled; not an issue until sched3 after reg-stack is real.

(define_automaton "k6_decoder,k6_load_unit,k6_store_unit,k6_integer_units,k6_fpu_unit,k6_branch_unit")

;; The K6 instruction decoding begins before the on-chip instruction cache is
;; filled.  Depending on the length of the instruction, two simple instructions
;; can be decoded in two parallel short decoders, or one complex instruction can
;; be decoded in either the long or the vector decoder.  For all practical
;; purposes, the long and vector decoder can be modelled as one decoder.
(define_cpu_unit "k6_decode_short0" "k6_decoder")
(define_cpu_unit "k6_decode_short1" "k6_decoder")
(define_cpu_unit "k6_decode_long" "k6_decoder")
(exclusion_set "k6_decode_long" "k6_decode_short0,k6_decode_short1")
(define_reservation "k6_decode_short" "k6_decode_short0|k6_decode_short1")
(define_reservation "k6_decode_vector" "k6_decode_long")

(define_cpu_unit "k6_store" "k6_store_unit")
(define_cpu_unit "k6_load" "k6_load_unit")
(define_cpu_unit "k6_alux,k6_aluy" "k6_integer_units")
(define_cpu_unit "k6_fpu" "k6_fpu_unit")
(define_cpu_unit "k6_branch" "k6_branch_unit")

;; Shift instructions and certain arithmetic are issued only on Integer X.
(define_insn_reservation "k6_alux_only" 1
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "ishift,ishift1,rotate,rotate1,alu1,negnot")
				   (eq_attr "memory" "none")))
			 "k6_decode_short,k6_alux")

(define_insn_reservation "k6_alux_only_load" 3
			 (and (eq_attr "cpu" "k6")
			       (and (eq_attr "type" "ishift,ishift1,rotate,rotate1,alu1,negnot")
				    (eq_attr "memory" "load")))
			 "k6_decode_short,k6_load,k6_alux")

(define_insn_reservation "k6_alux_only_store" 3
			 (and (eq_attr "cpu" "k6")
			       (and (eq_attr "type" "ishift,ishift1,rotate,rotate1,alu1,negnot")
				    (eq_attr "memory" "store,both,unknown")))
			 "k6_decode_long,k6_load,k6_alux,k6_store")

;; Integer divide and multiply can only be issued on Integer X, too.
(define_insn_reservation "k6_alu_imul" 2
			 (and (eq_attr "cpu" "k6")
			      (eq_attr "type" "imul"))
			 "k6_decode_vector,k6_alux*3")

(define_insn_reservation "k6_alu_imul_load" 4
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "imul")
				   (eq_attr "memory" "load")))
			 "k6_decode_vector,k6_load,k6_alux*3")

(define_insn_reservation "k6_alu_imul_store" 4
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "imul")
				   (eq_attr "memory" "store,both,unknown")))
			 "k6_decode_vector,k6_load,k6_alux*3,k6_store")

;; ??? Guessed latencies based on the old pipeline description.
(define_insn_reservation "k6_alu_idiv" 17
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "idiv")
				   (eq_attr "memory" "none")))
			 "k6_decode_vector,k6_alux*17")

(define_insn_reservation "k6_alu_idiv_mem" 19
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "idiv")
				   (eq_attr "memory" "!none")))
			 "k6_decode_vector,k6_load,k6_alux*17")

;; Basic word and doubleword ALU ops can be issued on both Integer units.
(define_insn_reservation "k6_alu" 1
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "alu,alu1,negnot,icmp,test,imovx,incdec,setcc")
				   (eq_attr "memory" "none")))
			 "k6_decode_short,k6_alux|k6_aluy")

(define_insn_reservation "k6_alu_load" 3
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "alu,alu1,negnot,icmp,test,imovx,incdec,setcc")
				   (eq_attr "memory" "load")))
			 "k6_decode_short,k6_load,k6_alux|k6_aluy")

(define_insn_reservation "k6_alu_store" 3
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "alu,alu1,negnot,icmp,test,imovx,incdec,setcc")
				   (eq_attr "memory" "store,both,unknown")))
			 "k6_decode_long,k6_load,k6_alux|k6_aluy,k6_store")

;; A "load immediate" operation does not require execution at all,
;; it is available immediately after decoding.  Special-case this.
(define_insn_reservation "k6_alu_imov" 1
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "imov")
				   (and (eq_attr "memory" "none")
					(match_operand 1 "nonimmediate_operand"))))
			 "k6_decode_short,k6_alux|k6_aluy")

(define_insn_reservation "k6_alu_imov_imm" 0
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "imov")
				   (and (eq_attr "memory" "none")
					(match_operand 1 "immediate_operand"))))
			 "k6_decode_short")

(define_insn_reservation "k6_alu_imov_load" 2
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "imov")
				   (eq_attr "memory" "load")))
			 "k6_decode_short,k6_load")

(define_insn_reservation "k6_alu_imov_store" 1
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "imov")
				   (eq_attr "memory" "store")))
			 "k6_decode_short,k6_store")

(define_insn_reservation "k6_alu_imov_both" 2
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "imov")
				   (eq_attr "memory" "both,unknown")))
			 "k6_decode_long,k6_load,k6_alux|k6_aluy")

;; The branch unit.
(define_insn_reservation "k6_branch_call" 1
			 (and (eq_attr "cpu" "k6")
			      (eq_attr "type" "call,callv"))
			 "k6_decode_vector,k6_branch")

(define_insn_reservation "k6_branch_branch" 1
			 (and (eq_attr "cpu" "k6")
			      (eq_attr "type" "ibr"))
			 "k6_decode_short,k6_branch")

;; The load and units have two pipeline stages.  The load latency is
;; two cycles.
(define_insn_reservation "k6_load_pop" 3
			 (and (eq_attr "cpu" "k6")
			      (ior (eq_attr "type" "pop")
				   (eq_attr "memory" "load,both")))
			 "k6_decode_short,k6_load")

(define_insn_reservation "k6_load_leave" 5
			 (and (eq_attr "cpu" "k6")
			      (eq_attr "type" "leave"))
			 "k6_decode_long,k6_load,(k6_alux|k6_aluy)*2")

;; ??? From the old pipeline description.  Egad!
;; ??? Apparently we take care of this reservation in adjust_cost.
(define_insn_reservation "k6_load_str" 10
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "str")
				   (eq_attr "memory" "load,both")))
			 "k6_decode_vector,k6_load*10")

;; The store unit handles lea and push.  It is otherwise unmodelled.
(define_insn_reservation "k6_store_lea" 2
			 (and (eq_attr "cpu" "k6")
			      (eq_attr "type" "lea"))
			 "k6_decode_short,k6_store,k6_alux|k6_aluy")

(define_insn_reservation "k6_store_push" 2
			 (and (eq_attr "cpu" "k6")
			      (ior (eq_attr "type" "push")
				   (eq_attr "memory" "store,both")))
			 "k6_decode_short,k6_store")

(define_insn_reservation "k6_store_str" 10
			 (and (eq_attr "cpu" "k6")
			      (eq_attr "type" "str"))
			 "k6_store*10")

;; Most FPU instructions have latency 2 and throughput 2.
(define_insn_reservation "k6_fpu" 2
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "fop,fmov,fcmp,fistp")
				   (eq_attr "memory" "none")))
			 "k6_decode_vector,k6_fpu*2")

(define_insn_reservation "k6_fpu_load" 6
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "fop,fmov,fcmp,fistp")
				   (eq_attr "memory" "load,both")))
			 "k6_decode_short,k6_load,k6_fpu*2")

(define_insn_reservation "k6_fpu_store" 6
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "fop,fmov,fcmp,fistp")
				   (eq_attr "memory" "store")))
			 "k6_decode_short,k6_store,k6_fpu*2")

(define_insn_reservation "k6_fpu_fmul" 2
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "fmul")
				   (eq_attr "memory" "none")))
			 "k6_decode_short,k6_fpu*2")

(define_insn_reservation "k6_fpu_fmul_load" 2
			 (and (eq_attr "cpu" "k6")
			      (and (eq_attr "type" "fmul")
				   (eq_attr "memory" "load,both")))
			 "k6_decode_short,k6_load,k6_fpu*2")

;; ??? Guessed latencies from the old pipeline description.
(define_insn_reservation "k6_fpu_expensive" 56
			 (and (eq_attr "cpu" "k6")
			      (eq_attr "type" "fdiv,fpspc"))
			 "k6_decode_short,k6_fpu*56")

