/* SPDX-License-Identifier: GPL-2.0 */
/*
 * ARM Mali-C55 ISP Driver - Register definitions
 *
 * Copyright (C) 2025 Ideas on Board Oy
 */

#ifndef _MALI_C55_REGISTERS_H
#define _MALI_C55_REGISTERS_H

#include <linux/bits.h>

/* ISP Common 0x00000 - 0x000ff */

#define MALI_C55_REG_API				0x00000
#define MALI_C55_REG_PRODUCT				0x00004
#define MALI_C55_REG_VERSION				0x00008
#define MALI_C55_REG_REVISION				0x0000c
#define MALI_C55_REG_PULSE_MODE				0x0003c
#define MALI_C55_REG_INPUT_MODE_REQUEST			0x0009c
#define MALI_C55_INPUT_SAFE_STOP			0x00
#define MALI_C55_INPUT_SAFE_START			0x01
#define MALI_C55_REG_MODE_STATUS			0x000a0
#define MALI_C55_REG_INTERRUPT_MASK_VECTOR		0x00030
#define MALI_C55_INTERRUPT_MASK_ALL			GENMASK(31, 0)

#define MALI_C55_REG_GLOBAL_MONITOR			0x00050

#define MALI_C55_REG_GEN_VIDEO				0x00080
#define MALI_C55_REG_GEN_VIDEO_ON_MASK			BIT(0)
#define MALI_C55_REG_GEN_VIDEO_MULTI_MASK		BIT(1)
#define MALI_C55_REG_GEN_PREFETCH_MASK			GENMASK(31, 16)

#define MALI_C55_REG_MCU_CONFIG				0x00020
#define MALI_C55_REG_MCU_CONFIG_OVERRIDE_MASK		BIT(0)
#define MALI_C55_REG_MCU_CONFIG_WRITE_MASK		BIT(1)
#define MALI_C55_MCU_CONFIG_WRITE(x)			((x) << 1)
#define MALI_C55_REG_MCU_CONFIG_WRITE_PING		BIT(1)
#define MALI_C55_REG_MCU_CONFIG_WRITE_PONG		0x00
#define MALI_C55_REG_MULTI_CONTEXT_MODE_MASK		BIT(8)
#define MALI_C55_REG_PING_PONG_READ			0x00024
#define MALI_C55_REG_PING_PONG_READ_MASK		BIT(2)

#define MALI_C55_REG_INTERRUPT_CLEAR_VECTOR		0x00034
#define MALI_C55_REG_INTERRUPT_CLEAR			0x00040
#define MALI_C55_REG_INTERRUPT_STATUS_VECTOR		0x00044

enum mali_c55_interrupts {
	MALI_C55_IRQ_ISP_START,
	MALI_C55_IRQ_ISP_DONE,
	MALI_C55_IRQ_MCM_ERROR,
	MALI_C55_IRQ_BROKEN_FRAME_ERROR,
	MALI_C55_IRQ_MET_AF_DONE,
	MALI_C55_IRQ_MET_AEXP_DONE,
	MALI_C55_IRQ_MET_AWB_DONE,
	MALI_C55_IRQ_AEXP_1024_DONE,
	MALI_C55_IRQ_IRIDIX_MET_DONE,
	MALI_C55_IRQ_LUT_INIT_DONE,
	MALI_C55_IRQ_FR_Y_DONE,
	MALI_C55_IRQ_FR_UV_DONE,
	MALI_C55_IRQ_DS_Y_DONE,
	MALI_C55_IRQ_DS_UV_DONE,
	MALI_C55_IRQ_LINEARIZATION_DONE,
	MALI_C55_IRQ_RAW_FRONTEND_DONE,
	MALI_C55_IRQ_NOISE_REDUCTION_DONE,
	MALI_C55_IRQ_IRIDIX_DONE,
	MALI_C55_IRQ_BAYER2RGB_DONE,
	MALI_C55_IRQ_WATCHDOG_TIMER,
	MALI_C55_IRQ_FRAME_COLLISION,
	MALI_C55_IRQ_UNUSED,
	MALI_C55_IRQ_DMA_ERROR,
	MALI_C55_IRQ_INPUT_STOPPED,
	MALI_C55_IRQ_MET_AWB_TARGET1_HIT,
	MALI_C55_IRQ_MET_AWB_TARGET2_HIT,
	MALI_C55_NUM_IRQ_BITS
};

#define MALI_C55_INTERRUPT_BIT(x)			BIT(x)

#define MALI_C55_REG_GLOBAL_PARAMETER_STATUS		0x00068
#define MALI_C55_GPS_PONG_FITTED			BIT(0)
#define MALI_C55_GPS_WDR_FITTED				BIT(1)
#define MALI_C55_GPS_COMPRESSION_FITTED			BIT(2)
#define MALI_C55_GPS_TEMPER_FITTED			BIT(3)
#define MALI_C55_GPS_SINTER_LITE_FITTED			BIT(4)
#define MALI_C55_GPS_SINTER_FITTED			BIT(5)
#define MALI_C55_GPS_IRIDIX_LTM_FITTED			BIT(6)
#define MALI_C55_GPS_IRIDIX_GTM_FITTED			BIT(7)
#define MALI_C55_GPS_CNR_FITTED				BIT(8)
#define MALI_C55_GPS_FRSCALER_FITTED			BIT(9)
#define MALI_C55_GPS_DS_PIPE_FITTED			BIT(10)

#define MALI_C55_REG_BLANKING				0x00084
#define MALI_C55_REG_HBLANK_MASK			GENMASK(15, 0)
#define MALI_C55_REG_VBLANK_MASK			GENMASK(31, 16)
#define MALI_C55_VBLANK(x)				((x) << 16)

#define MALI_C55_REG_HC_START				0x00088
#define MALI_C55_HC_START(h)				(((h) & 0xffff) << 16)
#define MALI_C55_REG_HC_SIZE				0x0008c
#define MALI_C55_HC_SIZE(h)				((h) & 0xffff)
#define MALI_C55_REG_VC_START_SIZE			0x00094
#define MALI_C55_VC_START(v)				((v) & 0xffff)
#define MALI_C55_VC_SIZE(v)				(((v) & 0xffff) << 16)

#define MALI_C55_REG_1024BIN_HIST			0x054a8
#define MALI_C55_1024BIN_HIST_SIZE			4096

/* Ping/Pong Configuration Space */
#define MALI_C55_REG_BASE_ADDR				0x18e88
#define MALI_C55_REG_BYPASS_0				0x18eac
#define MALI_C55_REG_BYPASS_0_VIDEO_TEST		BIT(0)
#define MALI_C55_REG_BYPASS_0_INPUT_FMT			BIT(1)
#define MALI_C55_REG_BYPASS_0_DECOMPANDER		BIT(2)
#define MALI_C55_REG_BYPASS_0_SENSOR_OFFSET_WDR		BIT(3)
#define MALI_C55_REG_BYPASS_0_GAIN_WDR			BIT(4)
#define MALI_C55_REG_BYPASS_0_FRAME_STITCH		BIT(5)
#define MALI_C55_REG_BYPASS_1				0x18eb0
#define MALI_C55_REG_BYPASS_1_DIGI_GAIN			BIT(0)
#define MALI_C55_REG_BYPASS_1_FE_SENSOR_OFFS		BIT(1)
#define MALI_C55_REG_BYPASS_1_FE_SQRT			BIT(2)
#define MALI_C55_REG_BYPASS_1_RAW_FE			BIT(3)
#define MALI_C55_REG_BYPASS_2				0x18eb8
#define MALI_C55_REG_BYPASS_2_SINTER			BIT(0)
#define MALI_C55_REG_BYPASS_2_TEMPER			BIT(1)
#define MALI_C55_REG_BYPASS_3				0x18ebc
#define MALI_C55_REG_BYPASS_3_SQUARE_BE			BIT(0)
#define MALI_C55_REG_BYPASS_3_SENSOR_OFFSET_PRE_SH	BIT(1)
#define MALI_C55_REG_BYPASS_3_MESH_SHADING		BIT(3)
#define MALI_C55_REG_BYPASS_3_WHITE_BALANCE		BIT(4)
#define MALI_C55_REG_BYPASS_3_IRIDIX			BIT(5)
#define MALI_C55_REG_BYPASS_3_IRIDIX_GAIN		BIT(6)
#define MALI_C55_REG_BYPASS_4				0x18ec0
#define MALI_C55_REG_BYPASS_4_DEMOSAIC_RGB		BIT(1)
#define MALI_C55_REG_BYPASS_4_PF_CORRECTION		BIT(3)
#define MALI_C55_REG_BYPASS_4_CCM			BIT(4)
#define MALI_C55_REG_BYPASS_4_CNR			BIT(5)
#define MALI_C55_REG_FR_BYPASS				0x18ec4
#define MALI_C55_REG_DS_BYPASS				0x18ec8
#define MALI_C55_BYPASS_CROP				BIT(0)
#define MALI_C55_BYPASS_SCALER				BIT(1)
#define MALI_C55_BYPASS_GAMMA_RGB			BIT(2)
#define MALI_C55_BYPASS_SHARPEN				BIT(3)
#define MALI_C55_BYPASS_CS_CONV				BIT(4)
#define MALI_C55_REG_ISP_RAW_BYPASS			0x18ecc
#define MALI_C55_ISP_RAW_BYPASS_BYPASS_MASK		BIT(0)
#define MALI_C55_ISP_RAW_BYPASS_FR_BYPASS_MASK		GENMASK(9, 8)
#define MALI_C55_ISP_RAW_BYPASS_RAW_FR_BYPASS		(2 << 8)
#define MALI_C55_ISP_RAW_BYPASS_RGB_FR_BYPASS		(1 << 8)
#define MALI_C55_ISP_RAW_BYPASS_DS_PIPE_DISABLE		BIT(1)
#define MALI_C55_ISP_RAW_BYPASS_RAW_BYPASS		BIT(0)

#define MALI_C55_REG_ACTIVE_WIDTH_MASK			0xffff
#define MALI_C55_REG_ACTIVE_HEIGHT_MASK			0xffff0000
#define MALI_C55_REG_BAYER_ORDER			0x18e8c
#define MALI_C55_BAYER_ORDER_MASK			GENMASK(1, 0)
#define MALI_C55_BAYER_ORDER_RGGB			0
#define MALI_C55_BAYER_ORDER_GRBG			1
#define MALI_C55_BAYER_ORDER_GBRG			2
#define MALI_C55_BAYER_ORDER_BGGR			3

#define MALI_C55_REG_METERING_CONFIG			0x18ed0
#define MALI_C55_5BIN_HIST_DISABLE_MASK			BIT(0)
#define MALI_C55_5BIN_HIST_SWITCH_MASK			GENMASK(2, 1)
#define MALI_C55_5BIN_HIST_SWITCH(x)			((x) << 1)
#define MALI_C55_AF_DISABLE_MASK			BIT(4)
#define MALI_C55_AF_SWITCH_MASK				BIT(5)
#define MALI_C55_AWB_DISABLE_MASK			BIT(8)
#define MALI_C55_AWB_SWITCH_MASK			BIT(9)
#define MALI_C55_AWB_SWITCH(x)				((x) << 9)
#define MALI_C55_AEXP_HIST_DISABLE_MASK			BIT(12)
#define MALI_C55_AEXP_HIST_DISABLE			(0x01 << 12)
#define MALI_C55_AEXP_HIST_SWITCH_MASK			GENMASK(14, 13)
#define MALI_C55_AEXP_HIST_SWITCH(x)			((x) << 13)
#define MALI_C55_AEXP_IHIST_DISABLE_MASK		BIT(16)
#define MALI_C55_AEXP_IHIST_DISABLE			(0x01 << 12)
#define MALI_C55_AEXP_SRC_MASK				BIT(24)

#define MALI_C55_REG_TPG_CH0				0x18ed8
#define MALI_C55_TEST_PATTERN_ON_OFF			BIT(0)
#define MALI_C55_TEST_PATTERN_RGB_MASK			BIT(1)
#define MALI_C55_TEST_PATTERN_RGB(x)			((x) << 1)
#define MALI_C55_REG_TPG_R_BACKGROUND			0x18ee0
#define MALI_C55_REG_TPG_G_BACKGROUND			0x18ee4
#define MALI_C55_REG_TPG_B_BACKGROUND			0x18ee8
#define MALI_C55_TPG_BACKGROUND_MAX			0xfffff
#define MALI_C55_REG_INPUT_WIDTH			0x18f98
#define MALI_C55_INPUT_WIDTH_MASK			GENMASK(18, 16)
#define MALI_C55_INPUT_WIDTH_8BIT			(0 << 16)
#define MALI_C55_INPUT_WIDTH_10BIT			(1 << 16)
#define MALI_C55_INPUT_WIDTH_12BIT			(2 << 16)
#define MALI_C55_INPUT_WIDTH_14BIT			(3 << 16)
#define MALI_C55_INPUT_WIDTH_16BIT			(4 << 16)
#define MALI_C55_INPUT_WIDTH_20BIT			(5 << 16)
#define MALI_C55_REG_SPACE_SIZE				0x4000
#define MALI_C55_REG_CONFIG_SPACES_OFFSET		0x0ab6c
#define MALI_C55_CONFIG_SPACE_SIZE			0x1231c

#define MALI_C55_REG_DIGITAL_GAIN			0x1926c
#define MALI_C55_DIGITAL_GAIN_MASK			GENMASK(12, 0)
#define MALI_C55_REG_DIGITAL_GAIN_OFFSET		0x19270
#define MALI_C55_DIGITAL_GAIN_OFFSET_MASK		GENMASK(19, 0)

#define MALI_C55_REG_SINTER_CONFIG			0x19348
#define MALI_C55_SINTER_VIEW_FILTER_MASK		GENMASK(1, 0)
#define MALI_C55_SINTER_SCALE_MODE_MASK			GENMASK(3, 2)
#define MALI_C55_SINTER_ENABLE_MASK			BIT(4)
#define MALI_C55_SINTER_FILTER_SELECT_MASK		BIT(5)
#define MALI_C55_SINTER_INT_SELECT_MASK			BIT(6)
#define MALI_C55_SINTER_RM_ENABLE_MASK			BIT(7)

/* Temper DMA */
#define MALI_C55_REG_TEMPER_DMA_IO			0x1ab78
#define MALI_C55_TEMPER_DMA_WRITE_ON			BIT(0)
#define MALI_C55_TEMPER_DMA_READ_ON			BIT(1)

/* Black Level Correction Configuration */
#define MALI_C55_REG_SENSOR_OFF_PRE_SHA_00		0x1abcc
#define MALI_C55_REG_SENSOR_OFF_PRE_SHA_01		0x1abd0
#define MALI_C55_REG_SENSOR_OFF_PRE_SHA_10		0x1abd4
#define MALI_C55_REG_SENSOR_OFF_PRE_SHA_11		0x1abd8
#define MALI_C55_SENSOR_OFF_PRE_SHA_MASK		0xfffff

/* Lens Mesh Shading Configuration */
#define MALI_C55_REG_MESH_SHADING_TABLES		0x13074
#define MALI_C55_REG_MESH_SHADING_CONFIG		0x1abfc
#define MALI_C55_MESH_SHADING_ENABLE_MASK		BIT(0)
#define MALI_C55_MESH_SHADING_MESH_SHOW_MASK		BIT(1)
#define MALI_C55_MESH_SHADING_MESH_SHOW(x)		((x) << 1)
#define MALI_C55_MESH_SHADING_SCALE_MASK		GENMASK(4, 2)
#define MALI_C55_MESH_SHADING_SCALE(x)			((x) << 2)
#define MALI_C55_MESH_SHADING_PAGE_R_MASK		GENMASK(9, 8)
#define MALI_C55_MESH_SHADING_PAGE_R(x)			((x) << 8)
#define MALI_C55_MESH_SHADING_PAGE_G_MASK		GENMASK(11, 10)
#define MALI_C55_MESH_SHADING_PAGE_G(x)			((x) << 10)
#define MALI_C55_MESH_SHADING_PAGE_B_MASK		GENMASK(13, 12)
#define MALI_C55_MESH_SHADING_PAGE_B(x)			((x) << 12)
#define MALI_C55_MESH_SHADING_MESH_WIDTH_MASK		GENMASK(21, 16)
#define MALI_C55_MESH_SHADING_MESH_WIDTH(x)		((x) << 16)
#define MALI_C55_MESH_SHADING_MESH_HEIGHT_MASK		GENMASK(29, 24)
#define MALI_C55_MESH_SHADING_MESH_HEIGHT(x)		((x) << 24)

#define MALI_C55_REG_MESH_SHADING_ALPHA_BANK		0x1ac04
#define MALI_C55_MESH_SHADING_ALPHA_BANK_R_MASK		GENMASK(2, 0)
#define MALI_C55_MESH_SHADING_ALPHA_BANK_G_MASK		GENMASK(5, 3)
#define MALI_C55_MESH_SHADING_ALPHA_BANK_G(x)		((x) << 3)
#define MALI_C55_MESH_SHADING_ALPHA_BANK_B_MASK		GENMASK(8, 6)
#define MALI_C55_MESH_SHADING_ALPHA_BANK_B(x)		((x) << 6)
#define MALI_C55_REG_MESH_SHADING_ALPHA			0x1ac08
#define MALI_C55_MESH_SHADING_ALPHA_R_MASK		GENMASK(7, 0)
#define MALI_C55_MESH_SHADING_ALPHA_G_MASK		GENMASK(15, 8)
#define MALI_C55_MESH_SHADING_ALPHA_G(x)		((x) << 8)
#define MALI_C55_MESH_SHADING_ALPHA_B_MASK		GENMASK(23, 16)
#define MALI_C55_MESH_SHADING_ALPHA_B(x)		((x) << 16)
#define MALI_C55_REG_MESH_SHADING_MESH_STRENGTH		0x1ac0c
#define MALI_c55_MESH_STRENGTH_MASK			GENMASK(15, 0)

/* AWB Gains Configuration */
#define MALI_C55_REG_AWB_GAINS1				0x1ac10
#define MALI_C55_AWB_GAIN00_MASK			GENMASK(11, 0)
#define MALI_C55_AWB_GAIN01_MASK			GENMASK(27, 16)
#define MALI_C55_AWB_GAIN01(x)				((x) << 16)
#define MALI_C55_REG_AWB_GAINS2				0x1ac14
#define MALI_C55_AWB_GAIN10_MASK			GENMASK(11, 0)
#define MALI_C55_AWB_GAIN11_MASK			GENMASK(27, 16)
#define MALI_C55_AWB_GAIN11(x)				((x) << 16)
#define MALI_C55_REG_AWB_GAINS1_AEXP			0x1ac18
#define MALI_C55_REG_AWB_GAINS2_AEXP			0x1ac1c

/* Colour Correction Matrix Configuration */
#define MALI_C55_REG_CCM_ENABLE				0x1b07c
#define MALI_C55_CCM_ENABLE_MASK			BIT(0)
#define MALI_C55_REG_CCM_COEF_R_R			0x1b080
#define MALI_C55_REG_CCM_COEF_R_G			0x1b084
#define MALI_C55_REG_CCM_COEF_R_B			0x1b088
#define MALI_C55_REG_CCM_COEF_G_R			0x1b090
#define MALI_C55_REG_CCM_COEF_G_G			0x1b094
#define MALI_C55_REG_CCM_COEF_G_B			0x1b098
#define MALI_C55_REG_CCM_COEF_B_R			0x1b0a0
#define MALI_C55_REG_CCM_COEF_B_G			0x1b0a4
#define MALI_C55_REG_CCM_COEF_B_B			0x1b0a8
#define MALI_C55_CCM_COEF_MASK				GENMASK(12, 0)
#define MALI_C55_REG_CCM_ANTIFOG_GAIN_R			0x1b0b0
#define MALI_C55_REG_CCM_ANTIFOG_GAIN_G			0x1b0b4
#define MALI_C55_REG_CCM_ANTIFOG_GAIN_B			0x1b0b8
#define MALI_C55_CCM_ANTIFOG_GAIN_MASK			GENMASK(11, 0)
#define MALI_C55_REG_CCM_ANTIFOG_OFFSET_R		0x1b0c0
#define MALI_C55_REG_CCM_ANTIFOG_OFFSET_G		0x1b0c4
#define MALI_C55_REG_CCM_ANTIFOG_OFFSET_B		0x1b0c8
#define MALI_C55_CCM_ANTIFOG_OFFSET_MASK		GENMASK(11, 0)

/* AWB Statistics Configuration */
#define MALI_C55_REG_AWB_STATS_MODE			0x1b29c
#define MALI_C55_AWB_STATS_MODE_MASK			BIT(0)
#define MALI_C55_REG_AWB_WHITE_LEVEL			0x1b2a0
#define MALI_C55_AWB_WHITE_LEVEL_MASK			GENMASK(9, 0)
#define MALI_C55_REG_AWB_BLACK_LEVEL			0x1b2a4
#define MALI_C55_AWB_BLACK_LEVEL_MASK			GENMASK(9, 0)
#define MALI_C55_REG_AWB_CR_MAX				0x1b2a8
#define MALI_C55_AWB_CR_MAX_MASK			GENMASK(11, 0)
#define MALI_C55_REG_AWB_CR_MIN				0x1b2ac
#define MALI_C55_AWB_CR_MIN_MASK			GENMASK(11, 0)
#define MALI_C55_REG_AWB_CB_MAX				0x1b2b0
#define MALI_C55_AWB_CB_MAX_MASK			GENMASK(11, 0)
#define MALI_C55_REG_AWB_CB_MIN				0x1b2b4
#define MALI_C55_AWB_CB_MIN_MASK			GENMASK(11, 0)
#define MALI_C55_REG_AWB_NODES_USED			0x1b2c4
#define MALI_C55_AWB_NODES_USED_HORIZ_MASK		GENMASK(7, 0)
#define MALI_C55_AWB_NODES_USED_VERT_MASK		GENMASK(15, 8)
#define MALI_C55_AWB_NODES_USED_VERT(x)			((x) << 8)
#define MALI_C55_REG_AWB_CR_HIGH			0x1b2c8
#define MALI_C55_AWB_CR_HIGH_MASK			GENMASK(11, 0)
#define MALI_C55_REG_AWB_CR_LOW				0x1b2cc
#define MALI_C55_AWB_CR_LOW_MASK			GENMASK(11, 0)
#define MALI_C55_REG_AWB_CB_HIGH			0x1b2d0
#define MALI_C55_AWB_CB_HIGH_MASK			GENMASK(11, 0)
#define MALI_C55_REG_AWB_CB_LOW				0x1b2d4
#define MALI_C55_AWB_CB_LOW_MASK			GENMASK(11, 0)

/* AEXP Metering Histogram Configuration */
#define MALI_C55_REG_AEXP_HIST_BASE			0x1b730
#define MALI_C55_REG_AEXP_IHIST_BASE			0x1bbac
#define MALI_C55_AEXP_HIST_SKIP_OFFSET			0
#define MALI_C55_AEXP_HIST_SKIP_X_MASK			GENMASK(2, 0)
#define MALI_C55_AEXP_HIST_SKIP_X(x)			((x) << 0)
#define MALI_C55_AEXP_HIST_OFFSET_X_MASK		BIT(3)
#define MALI_C55_AEXP_HIST_OFFSET_X(x)			((x) << 3)
#define MALI_C55_AEXP_HIST_SKIP_Y_MASK			GENMASK(6, 4)
#define MALI_C55_AEXP_HIST_SKIP_Y(x)			((x) << 4)
#define MALI_C55_AEXP_HIST_OFFSET_Y_MASK		BIT(7)
#define MALI_C55_AEXP_HIST_OFFSET_Y(x)			((x) << 7)
#define MALI_C55_AEXP_HIST_SCALE_OFFSET			4
#define MALI_C55_AEXP_HIST_SCALE_BOTTOM_MASK		GENMASK(3, 0)
#define MALI_C55_AEXP_HIST_SCALE_TOP_MASK		GENMASK(7, 4)
#define MALI_C55_AEXP_HIST_SCALE_TOP(x)			((x) << 4)
#define MALI_C55_AEXP_HIST_PLANE_MODE_OFFSET		16
#define MALI_C55_AEXP_HIST_PLANE_MODE_MASK		GENMASK(2, 0)
#define MALI_C55_AEXP_HIST_NODES_USED_OFFSET		52
#define MALI_C55_AEXP_HIST_NODES_USED_HORIZ_MASK	GENMASK(7, 0)
#define MALI_C55_AEXP_HIST_NODES_USED_VERT_MASK		GENMASK(15, 8)
#define MALI_C55_AEXP_HIST_NODES_USED_VERT(x)		((x) << 8)
#define MALI_C55_AEXP_HIST_ZONE_WEIGHTS_OFFSET		56
#define MALI_C55_AEXP_HIST_ZONE_WEIGHT_MASK		0x0f0f0f0f

/*
 * The Mali-C55 ISP has up to two output pipes; known as full resolution and
 * down scaled. The register space for these is laid out identically, but offset
 * by 372 bytes.
 */
#define MALI_C55_CAP_DEV_FR_REG_OFFSET		0x0
#define MALI_C55_CAP_DEV_DS_REG_OFFSET		0x174

#define MALI_C55_REG_CS_CONV_CONFIG			0x1c098
#define MALI_C55_CS_CONV_MATRIX_MASK			BIT(0)
#define MALI_C55_CS_CONV_FILTER_MASK			BIT(1)
#define MALI_C55_CS_CONV_HORZ_DOWNSAMPLE_MASK		BIT(2)
#define MALI_C55_CS_CONV_VERT_DOWNSAMPLE_MASK		BIT(3)
#define MALI_C55_CS_CONV_FILTER_ENABLE			(0x01 << 1)
#define MALI_C55_CS_CONV_HORZ_DOWNSAMPLE_ENABLE		(0x01 << 2)
#define MALI_C55_CS_CONV_VERT_DOWNSAMPLE_ENABLE		(0x01 << 3)
#define MALI_C55_REG_Y_WRITER_MODE			0x1c0ec
#define MALI_C55_REG_UV_WRITER_MODE			0x1c144
#define MALI_C55_WRITER_MODE_MASK			GENMASK(4, 0)
#define MALI_C55_OUTPUT_DISABLED			0
#define MALI_C55_OUTPUT_RGB32				1
#define MALI_C55_OUTPUT_A2R10G10B10			2
#define MALI_C55_OUTPUT_RGB565				3
#define MALI_C55_OUTPUT_RGB24				4
#define MALI_C55_OUTPUT_GEN32				5
#define MALI_C55_OUTPUT_RAW16				6
#define MALI_C55_OUTPUT_AYUV				8
#define MALI_C55_OUTPUT_Y410				9
#define MALI_C55_OUTPUT_YUY2				10
#define MALI_C55_OUTPUT_UYVY				11
#define MALI_C55_OUTPUT_Y210				12
#define MALI_C55_OUTPUT_NV12_21				13
#define MALI_C55_OUTPUT_YUV_420_422			17
#define MALI_C55_OUTPUT_P210_P010			19
#define MALI_C55_OUTPUT_YUV422				20
#define MALI_C55_WRITER_SUBMODE_MASK			GENMASK(7, 6)
#define MALI_C55_WRITER_SUBMODE(x)			((x) << 6)
#define MALI_C55_OUTPUT_PLANE_ALT0			0
#define MALI_C55_OUTPUT_PLANE_ALT1			1
#define MALI_C55_OUTPUT_PLANE_ALT2			2
#define MALI_C55_WRITER_FRAME_WRITE_MASK		BIT(9)
#define MALI_C55_WRITER_FRAME_WRITE_ENABLE		(0x01 << 9)
#define MALI_C55_REG_ACTIVE_OUT_Y_SIZE			0x1c0f0
#define MALI_C55_REG_ACTIVE_OUT_UV_SIZE			0x1c148
#define MALI_C55_REG_ACTIVE_OUT_SIZE_W(w)		((w) << 0)
#define MALI_C55_REG_ACTIVE_OUT_SIZE_H(h)		((h) << 16)
#define MALI_C55_REG_Y_WRITER_BANKS_BASE		0x1c0f4
#define MALI_C55_REG_Y_WRITER_BANKS_CONFIG		0x1c108
#define MALI_C55_REG_Y_WRITER_MAX_BANKS_MASK		GENMASK(2, 0)
#define MALI_C55_REG_Y_WRITER_BANKS_RESTART		BIT(3)
#define MALI_C55_REG_Y_WRITER_OFFSET			0x1c10c
#define MALI_C55_REG_UV_WRITER_BANKS_BASE		0x1c14c
#define MALI_C55_REG_UV_WRITER_BANKS_CONFIG		0x1c160
#define MALI_C55_REG_UV_WRITER_MAX_BANKS_MASK		GENMASK(2, 0)
#define MALI_C55_REG_UV_WRITER_BANKS_RESTART		BIT(3)
#define MALI_C55_REG_UV_WRITER_OFFSET			0x1c164

#define MALI_C55_REG_TEST_GEN_CH0_OFF_ON
#define MALI_C55_REG_TEST_GEN_CH0_PATTERN_TYPE		0x18edc

#define MALI_C55_REG_CROP_EN				0x1c028
#define MALI_C55_CROP_ENABLE				BIT(0)
#define MALI_C55_REG_CROP_X_START			0x1c02c
#define MALI_C55_REG_CROP_Y_START			0x1c030
#define MALI_C55_REG_CROP_X_SIZE			0x1c034
#define MALI_C55_REG_CROP_Y_SIZE			0x1c038
#define MALI_C55_REG_SCALER_TIMEOUT_EN			0x1c040
#define MALI_C55_SCALER_TIMEOUT_EN			BIT(4)
#define MALI_C55_SCALER_TIMEOUT(t)			((t) << 16)
#define MALI_C55_REG_SCALER_IN_WIDTH			0x1c044
#define MALI_C55_REG_SCALER_IN_HEIGHT			0x1c048
#define MALI_C55_REG_SCALER_OUT_WIDTH			0x1c04c
#define MALI_C55_REG_SCALER_OUT_HEIGHT			0x1c050
#define MALI_C55_REG_SCALER_HFILT_TINC			0x1c054
#define MALI_C55_REG_SCALER_HFILT_COEF			0x1c058
#define MALI_C55_REG_SCALER_VFILT_TINC			0x1c05c
#define MALI_C55_REG_SCALER_VFILT_COEF			0x1c060

#define MALI_C55_REG_GAMMA_RGB_ENABLE			0x1c064
#define MALI_C55_GAMMA_ENABLE_MASK			BIT(0)
#define MALI_C55_REG_GAMMA_GAINS_1			0x1c068
#define MALI_C55_GAMMA_GAIN_R_MASK			GENMASK(11, 0)
#define MALI_C55_GAMMA_GAIN_G_MASK			GENMASK(27, 16)
#define MALI_C55_REG_GAMMA_GAINS_2			0x1c06c
#define MALI_C55_GAMMA_GAIN_B_MASK			GENMASK(11, 0)
#define MALI_C55_REG_GAMMA_OFFSETS_1			0x1c070
#define MALI_C55_GAMMA_OFFSET_R_MASK			GENMASK(11, 0)
#define MALI_C55_GAMMA_OFFSET_G_MASK			GENMASK(27, 16)
#define MALI_C55_REG_GAMMA_OFFSETS_2			0x1c074
#define MALI_C55_GAMMA_OFFSET_B_MASK			GENMASK(11, 0)

/*
 * A re-definition of an above register. These will usually be written on a per
 * capture device basis and handled with mali_c55_cap_dev_write(), but on
 * startup is written by core.c
 */
#define MALI_C55_REG_FR_GAMMA_RGB_ENABLE		0x1c064
#define MALI_C55_REG_DS_GAMMA_RGB_ENABLE		0x1c1d8

#define MALI_C55_REG_FR_SCALER_HFILT			0x34a8
#define MALI_C55_REG_FR_SCALER_VFILT			0x44a8
#define MALI_C55_REG_DS_SCALER_HFILT			0x14a8
#define MALI_C55_REG_DS_SCALER_VFILT			0x24a8

#endif /* _MALI_C55_REGISTERS_H */
