#!/usr/bin/perl
# htroff.pl: troff-like text formatter for html output
# Copyright (C) TOYODA Eizi, 1998.  All rights reserved.
# see COPYING.TXT for terms of license.

	# number/string registers
	%NUMBER = ("font", 0, "unify", 0, "fillin", 1, "center", 0);
	%STRING = ("font", "", "lastfont", "");
	%OPTION = ();

	# environment detection
	$MSDOS = $NUMBER{"msdos"} = &MSDOS;
	$NUMBER{"jperl"} = &JPERL;
	$Japanese'NOCONV = $NUMBER{"jperl"};
	$Japanese'OUT = $MSDOS ? "Shift_JIS" : "EUC-JP";
	$RCSID = q$Id: htroff.pl,v 1.2 1999/02/17 09:32:16 toyoda Exp $;
	&initFindPath;

	# common variables and messages
	$FOREVER = 100000;
	$ME = "htroff:";
	$CO = "cannot open";
	$NF = "not found";
	$UE = "unexpected";

	# .ig support
	$IGNORE_UNTIL = undef;

	# macro support
	@REQARG = ();		# request arguments: for internal work
	@MACROARG = ();		# macro arguments: for \\$n
	$DEFINE_UNTIL = "\\.";	# failsafe
	$DEFMACRO = undef;	# if TRUE, lines are stored as macro
	%MACROS = ();		# LF separated macro lines
	%ALIAS = ('cleanup', 'nop', 'opened', 'nop');
		# request/macro aliases

	# pattern-matching request hook
	%HOOK = ('^\s*$', 'tag P');

	# conditional support
	$IFVAL = 1;		# if FALSE lines are ignored
	@IFVAL = ();		# $IFVAL of outer block is pushed
	@ELSE = ();		# if TRUE "else" lines are ignored

	# input/output streams
	@PUSHED_INPUT = ();
	@INPUT = ();
	$STRING{"<"} = $INPUT = "";
	$STRING{">"} = $OUTPUT = "STDOUT";

	# for better diag
	%LINE_COUNTER = ();
	$MACROS_DONE = 0;

	# predefined number registers
	($NUMBER{"sec"}, $NUMBER{"min"}, $NUMBER{"hour"}, $NUMBER{"dy"},
	 $NUMBER{"mo"}, $NUMBER{"yr"}, $NUMBER{"dw"}, $NUMBER{"yday"},
	 $NUMBER{"dst"}) = localtime time;
	$STRING{"month"} = ("January", "February", "March", "April",
		"May", "June", "July", "August", "September", "October",
		"November", "December")[$NUMBER{"mo"}];
	$NUMBER{"mo"}++;
	$NUMBER{"year"} = $NUMBER{"yr"} + 1900;

	&RegisterMacro("std");
	while ($_ = shift @ARGV) {
		(!/^-/ || /^-$/) && (unshift(@ARGV, $_), last);
		/^--$/ && last;
		/^-m(.*)/ && (&RegisterMacro($1), next);
		/^-o(.*)/ && (&SetOutput($1), next);
		(/^-r(.*)=/ || /^-r(.)/) && ($NUMBER{$1} = $', next);
		(/^-d(.*)=/ || /^-d(.)/) && ($STRING{$1} = $', next);
		/^-V/ && die "$RCSID\n";
		# unknown options are stored for macro use.
		/^-q(.)/ && ($OPTION{$1} = $', next);
		/^-(.)/;
		$OPTION{$1} = $';
	}
	push(@INPUT, 'end of macros');
	push(@INPUT, @ARGV);

	# event loop
	$CONTINUE = "";
	while ($_ = &getline) {
		&chop;
		$_ = "$CONTINUE$_" unless ($CONTINUE eq "");
		unless (/\\\\$/) {
			s/\\$// && ($CONTINUE = $_, next);
		}
		&ProcessLine;
		$CONTINUE = "";
	}
	&Request(".cleanup");
exit 0;

sub RegisterMacro {
	local($macroname) = @_;
	$OPTION{"-m$macroname"} = 1;
	$OPTION{"-m"} .= "-m$macroname ";
	push(@INPUT, (&FindFile("$macroname.hma") || "DATA::$macroname"));
}

# give line in $_
sub ProcessLine {
	&ExpandEscape;

	# .ig support
	if (defined $IGNORE_UNTIL && /^\.$IGNORE_UNTIL/) {
		$IGNORE_UNTIL = undef;
		return;
	}
	return if defined $IGNORE_UNTIL;

	# macro definition support
	if ($DEFMACRO && /^\0?['.]$DEFINE_UNTIL\s*$/) {	# end macro line
		$DEFMACRO = undef;
		return;
	}
	if ($DEFMACRO) {			# begin macro line
		# removing NULs.
		s/\0//g;
		$MACROS{$DEFMACRO} .= "$_\n";
		return;
	}

	# nested if support
	if (/^\0?['.]{\s/) {			# if line
		push(@IFVAL, $IFVAL);
		$IFVAL = $IFVAL && &Boolean($');
		push(@ELSE, $IFVAL);
		return;
	}
	if (/^\0?['.]}{$/ || /^\0?['.]}{\s/) {	# elsif line
		(scalar(@ELSE) == 0) && (warn("$ME $UE .}{"), return);
		$ELSE[$#ELSE] && ($IFVAL = 0, return);
		$ELSE[$#ELSE] = $IFVAL = &Boolean($');
		return;
	}
	if (/^\0?['.]}\s*$/) {			# endif line
		if (scalar @ELSE == 0) {
			warn "$ME .} without .{\n";
			return;
		}
		pop(@ELSE);
		$IFVAL = pop(@IFVAL);
		return;
	}
	$IFVAL || return;

	# normal request/macro lines
	/^['.]/ && return &Request($_);

	# pattern-matching request hook
	$STRING{"line"} = $_;
	foreach (keys %HOOK) {
		$STRING{'line'} =~ /$_/ || next;	
		$STRING{'hook'} = $_;   $STRING{'pre'} = $`;
		$STRING{'match'} = $&;  $STRING{'post'} = $';
		return &Request(".$HOOK{$_}");
	}

	&writeln($_);
	&CheckFont;
	&CheckUnify;
	&CheckCenter;
}

sub ArgSplit {
	local($_) = @_;
	local(@arg) = ();
	while (length) {
		if (/^"(([^"]|"")*)"\s*/) {
			local($match) = $1;
			$_ = $';
			$match =~ s/""/"/g;
			push(@arg, $match);
		} elsif (/^"/) {
			push(@arg, $');
			$_ = '';
		} elsif (/(\S+)\s*/) {
			push(@arg, $1);
			$_ = $';
		} else {
			last;
		}
	}
	@arg;
}

sub ExprList {
	local($rhs, $lhs);
	local(@EXPR) = (0);
	for (@_) {
		# --- numerical operators
		if ($_ eq "sub") {
			$rhs = pop(@EXPR);
			$EXPR[$#EXPR] -= $rhs;
		} elsif ($_ eq "add") {
			$rhs = pop(@EXPR);
			$EXPR[$#EXPR] += $rhs;
		} elsif ($_ eq "inc") {
			$EXPR[$#EXPR]++;
		} elsif ($_ eq "dec") {
			$EXPR[$#EXPR]--;
		} elsif ($_ eq "equiv") {
			$rhs = pop(@EXPR);
			$EXPR[$#EXPR] = ($EXPR[$#EXPR] == $rhs);
		} elsif ($_ eq "posi") {
			$EXPR[$#EXPR] = ($EXPR[$#EXPR] > 0);
		} elsif ($_ eq "nega") {
			$EXPR[$#EXPR] = ($EXPR[$#EXPR] < 0);
		# --- string operators
		} elsif ($_ eq "length") {
			$EXPR[$#EXPR] = length $EXPR[$#EXPR];
		} elsif ($_ eq "cat") {
			$rhs = pop(@EXPR);
			$EXPR[$#EXPR] .= $rhs;
		} elsif ($_ eq "eq") {
			$rhs = pop(@EXPR);
			$EXPR[$#EXPR] = ($EXPR[$#EXPR] eq $rhs);
		} elsif ($_ eq "lowercase") {
			$EXPR[$#EXPR] =~ tr/A-Z/a-z/;
		} elsif ($_ eq "uppercase") {
			$EXPR[$#EXPR] =~ tr/a-z/A-Z/;
		} elsif ($_ eq "grep" || $_ eq "subpat") {
			$rhs = pop(@EXPR);
			$rhs =~ s/\0//g;
			$EXPR[$#EXPR] =~ /$rhs/;
			$EXPR[$#EXPR] = $& if ($_ eq "grep");
			$EXPR[$#EXPR] = $1 if ($_ eq "subpat");
		} elsif ($_ eq "match") {
			$rhs = pop(@EXPR);
			$rhs =~ s/\0//g;
			$EXPR[$#EXPR] = ($EXPR[$#EXPR] =~ /$rhs/);
		} elsif ($_ eq "sub" || $_ eq "gsub") {
			$rhs = pop(@EXPR);
			$rhs =~ s/\0//g;
			$lhs = pop(@EXPR);
			$lhs =~ s/\0//g;
			$EXPR[$#EXPR] =~ s/$lhs/$rhs/ if ($_ eq "sub");
			$EXPR[$#EXPR] =~ s/$lhs/$rhs/g if ($_ eq "gsub");
		} elsif ($_ eq "file") {
			$EXPR[$#EXPR] = "" unless (-f $EXPR[$#EXPR]);
		} elsif ($_ eq "dir") {
			$EXPR[$#EXPR] = "" unless (-d $EXPR[$#EXPR]);
		} elsif ($_ eq "basename") {
			$EXPR[$#EXPR] =~ /$SUBPAT/;
			$EXPR[$#EXPR] = $1;
		} elsif ($_ eq "dirname") {
			$EXPR[$#EXPR] =~ s/$SUBPAT//;
		# --- boolean operators
		} elsif ($_ eq "not") {
			$EXPR[$#EXPR] = !$EXPR[$#EXPR];
		} elsif ($_ eq "and") {
			$rhs = pop(@EXPR);
			$EXPR[$#EXPR] = ($EXPR[$#EXPR] && $rhs);
		} elsif ($_ eq "or") {
			$rhs = pop(@EXPR);
			$EXPR[$#EXPR] = ($EXPR[$#EXPR] || $rhs);
		} elsif ($_ eq "defined") {
			$EXPR[$#EXPR] = defined $EXPR[$#EXPR];
		# --- constants
		} elsif ($_ eq "undef") {
			push(@EXPR, undef);
		} elsif ($_ eq "null") {
			push(@EXPR, "");
		} elsif ($_ eq "end") {
			push(@EXPR, 0);
		} elsif ($_ eq "begin") {
			push(@EXPR, $FOREVER);
		# --- system operators
		} elsif ($_ eq "option") {
			$EXPR[$#EXPR] = $OPTION{$EXPR[$#EXPR]};
		} elsif ($_ eq "env") {
			$EXPR[$#EXPR] = $ENV{$EXPR[$#EXPR]};
		} elsif ($_ eq "string") {
			$EXPR[$#EXPR] = $STRING{$EXPR[$#EXPR]};
		} elsif ($_ eq "number") {
			$EXPR[$#EXPR] = $NUMBER{$EXPR[$#EXPR]};
		} elsif ($_ eq "macro") {
			$EXPR[$#EXPR] = $MACROS{$EXPR[$#EXPR]};
		} elsif ($_ eq "alias") {
			$EXPR[$#EXPR] = $ALIAS{$EXPR[$#EXPR]};
		} elsif ($_ eq "hook") {
			$EXPR[$#EXPR] = $HOOK{$EXPR[$#EXPR]};
		} elsif ($_ eq "arg") {
			$EXPR[$#EXPR] = $REQARG[$EXPR[$#EXPR]];
		} elsif ($_ eq "args") {
			push(@EXPR, scalar(@REQARG));
		} elsif ($_ eq "dup") {
			push(@EXPR, $EXPR[$#EXPR]);
		} elsif ($_ eq "pop") {
			pop(@EXPR);
		} elsif ($_ eq "exch") {
			$rhs = pop(@EXPR);
			$lhs = pop(@EXPR);
			push(@EXPR, $rhs, $lhs);
		} elsif ($_ eq "debug") {
			$_ = "TOS=`$EXPR[$#EXPR]'\n";
			&Japanese'EUCToPrintable;
			warn $_;
		} elsif ($_ eq ";") {
			1;		# do nothing
		} else {
			s/^'//;
			push(@EXPR, $_);
		}
	}
	$EXPR[$#EXPR];
}

sub Boolean {
	local($argl) = @_;
	return 1 if ($argl =~ /^\s*$/);
	&ExprList(&ArgSplit($argl));
}

sub SetFont {
	local($name, $lines) = @_;
	local($ret) = "";
	if ($name eq "P") {
		$name = $STRING{"lastfont"};
		$lines = 0 if ($name eq "");
	}
	$STRING{"lastfont"} = $STRING{"font"};
	if ($lines <= 0) {
		return "" if ($STRING{"font"} eq "");
		$ret = "\\</$STRING{'font'}\\>";
		$STRING{"font"} = "";
		$NUMBER{"font"} = 0;
	} else {
		$ret = "\\</$STRING{'font'}\\>" if ($STRING{"font"} ne "");
		$ret .= "\\<$name\\>";
		$STRING{"font"} = $name;
		$NUMBER{"font"} = $lines;
	}
	$ret;
}

sub CheckFont {
	return if ($NUMBER{"font"} <= 0);
	return if (--$NUMBER{"font"} > 0);
	&tag("/$STRING{'font'}");
	$STRING{"font"} = "";
}

sub SetCenter {
	local($lines) = @_;
	&tag("DIV ALIGN=CENTER") if ($lines && !$NUMBER{"center"});
	&tag("/DIV") if (!$lines && $NUMBER{"center"});
	$NUMBER{"center"} = $lines;
}

sub CheckCenter {
	return if ($NUMBER{"center"} <= 0);
	&tag("BR");
	return if (--$NUMBER{"center"} > 0);
	&tag("/DIV");
}

sub CheckUnify {
	return if ($NUMBER{"unify"} <= 0);
	return if (--$NUMBER{"unify"} > 0);
	&writeln("");
}

sub Request {
	local($request) = @_;
	$request =~ s/^['.]\s*//;
	return if ($request eq "");
	local(@REQARG) = &ArgSplit($request);
	$reqname = shift @REQARG;
	$reqname = $ALIAS{$reqname} if ($ALIAS{$reqname});

	# troff built-in (primitive) requests
	if (defined $MACROS{$reqname}) {
		local(@macro) = split(/\n/, $MACROS{$reqname});
		local(@MACROARG) = @REQARG;
		foreach (@macro) {
			&ProcessLine;
		}
	} elsif ($reqname eq "ig") {
		$IGNORE_UNTIL = $REQARG[0] || "\\.";
	} elsif ($reqname eq "am") {
		$DEFINE_UNTIL = $REQARG[1] || "\\.";
		$DEFMACRO = $REQARG[0];
	} elsif ($reqname eq "de") {
		$DEFINE_UNTIL = $REQARG[1] || "\\.";
		$DEFMACRO = $REQARG[0];
		$MACROS{$DEFMACRO} = undef;
	} elsif ($reqname eq "rm") {
		$MACROS{$REQARG[0]} = undef;
	} elsif ($reqname eq "nr") {
		$numreg_name = shift(@REQARG);
		$NUMBER{$numreg_name} = shift(@REQARG);
		$NUMBER{"+$numreg_name"} = shift(@REQARG);
	} elsif ($reqname eq "nrexpr") {
		$numreg_name = shift(@REQARG);
		$NUMBER{$numreg_name} = &ExprList(@REQARG);
	} elsif ($reqname eq "ds") {
		$strreg_name = shift(@REQARG);
		$STRING{$strreg_name} = join(" ", @REQARG);
		$STRING{$strreg_name} =~ s/\0//g;
	} elsif ($reqname eq "dsexpr") {
		$strreg_name = shift(@REQARG);
		$STRING{$strreg_name} = &ExprList(@REQARG);
		$STRING{$strreg_name} =~ s/\0//g;
	} elsif ($reqname eq "as") {
		$strreg_name = shift(@REQARG);
		$STRING{$strreg_name} .= join(" ", @REQARG);
		$STRING{$strreg_name} =~ s/\0//g;
	} elsif ($reqname eq "asexpr") {
		$strreg_name = shift(@REQARG);
		$STRING{$strreg_name} .= &ExprList(@REQARG);
		$STRING{$strreg_name} =~ s/\0//g;
	} elsif ($reqname eq "bp") {
		&tag("HR");
	} elsif ($reqname eq "br") {
		&tag("BR");
	} elsif ($reqname eq "sp") {
		local($times) = $REQARG[0] || 1;
		for ($i = 0; $i < $times; $i++) {
			&tag("BR");
		}
	} elsif ($reqname eq "nf") {
		$NUMBER{"unify"} = 0;
		&tag("PRE") if $NUMBER{"fillin"};
		$NUMBER{"fillin"} = 0;
	} elsif ($reqname eq "fi") {
		&tag("/PRE") unless $NUMBER{"fillin"};
		$NUMBER{"fillin"} = 1;
	} elsif ($reqname eq "so") {
		push(@PUSHED_INPUT, $INPUT);
		$STRING{"<"} = $INPUT = $REQARG[0];
		open($INPUT, "<$INPUT") || warn "$ME <$INPUT $CO\n";
	} elsif ($reqname eq "ce") {
		&SetCenter($REQARG[0] || 1);
	} elsif ($reqname eq "ul") {
		$UNIFY_ONE_LINE = 1;
		&writeln(&SetFont("U", ((@REQARG > 0) ? $REQARG[0] : 1)));
	# --- htroff original requests
	} elsif ($reqname eq "it") {
		$UNIFY_ONE_LINE = 1;
		&writeln(&SetFont("I", ((@REQARG > 0) ? $REQARG[0] : 1)));
	} elsif ($reqname eq "bf") {
		$UNIFY_ONE_LINE = 1;
		&writeln(&SetFont("B", ((@REQARG > 0) ? $REQARG[0] : 1)));
	} elsif ($reqname eq "shift") {
		shift(@MACROARG);
	} elsif ($reqname eq "getline") {
		($_ = &getline) || return;
		&chop;
		@MACROARG = &ArgSplit($_);
	} elsif ($reqname eq "input") {
		unshift(@INPUT, $REQARG[0]);
	} elsif ($reqname eq "output") {
		&SetOutput($REQARG[0]);
	} elsif ($reqname eq "append") {
		&SetAppend($REQARG[0]);
	} elsif ($reqname eq "keep") {
		$STRING{"keep"} = $REQARG[0];
		delete $STRING{"keep"} if (@REQARG < 1);
	} elsif ($reqname eq "alias") {
		$ALIAS{$REQARG[0]} = $REQARG[1];
		delete $ALIAS{$REQARG[0]} if (@REQARG < 2);
	} elsif ($reqname eq "hook") {
		$REQARG[0] =~ s/\0//g;
		$HOOK{$REQARG[0]} = $REQARG[1];
		delete $HOOK{$REQARG[0]} if (@REQARG < 2);
	} elsif ($reqname eq "char") {
		$char_name = shift(@REQARG);
		$CHAR{$char_name} = shift(@REQARG);
	} elsif ($reqname eq "tag") {
		&tag(@REQARG);
	} elsif ($reqname eq "warn") {
		$_ = join(" ", @REQARG);
		&Japanese'EUCToPrintable;
		warn "$_\n";
	} elsif ($reqname eq "write") {
		$_ = join(" ", @REQARG);
		&Japanese'EUCToPrintable;
		&write($_);
	} elsif ($reqname eq "exit") {
		exit &ExprList(@REQARG);
	} elsif ($reqname eq "checkopt") {
		&CheckOpt($REQARG[0]);
	} elsif ($reqname eq "nop") {
		1;	# do nothing
	} else {
		warn "$ME undefined request $reqname.\n";
	}
}

sub CheckOpt {
	local($onechar) = @_;
	$NUMBER{"!"} = 0;
	foreach (keys %OPTION) {
		next if (length $_ == 1) && (index($onechar, $_) >= 0);
		next if /^-/;
		warn "$ME undefined option -$_ used\n";
		$NUMBER{"!"} = 1;
	}
}

sub initFindPath {
	# path delimiter
	$PATHDELIM = $MSDOS ? ";" : ":";
	@FINDPATH = split(/$PATHDELIM/, $ENV{"PATH"});
	unshift @FINDPATH, split(/$PATHDELIM/, $ENV{"FMAC"});
	if (-e $0) {
		# pattern for directory last part
		$SUBPAT = $MSDOS ? '[/\\\\]([^/\\\\]+)$' : '/([^/]+)$';
		($basedir = $0) =~ s/$SUBPAT//;
		unshift(@FINDPATH, $basedir);
	}
}

sub FindFile {
	local($fnam) = @_;
	local($try);
	return $fnam if -f $fnam;
	foreach (@FINDPATH) {
		$try = "$_/$fnam";
		return $try if (-e $try && -r $try);
	}
	undef;
}

sub SetOutput {
	$STRING{">"} = $OUTPUT = $_[0];
	$NUMBER{"!"} = 0;
	if (open($OUTPUT, ">$OUTPUT")) {
		select($OUTPUT);
	} else {
		warn "$ME >$OUTPUT $CO\n";
		$NUMBER{"!"} = $! + 0;
	}
}

sub SetAppend {
	$STRING{">"} = $OUTPUT = $_[0];
	$NUMBER{"!"} = 0;
	if (open($OUTPUT, ">>$OUTPUT")) {
		select($OUTPUT);
	} else {
		warn "$ME >>$OUTPUT $CO\n";
		$NUMBER{"!"} = $! + 0;
	}
}

sub NextFile {
	close($INPUT) unless ($INPUT eq "DATA");
	for (;;) {
		if (@PUSHED_INPUT) {
			return ($STRING{"<"} = $INPUT = pop(@PUSHED_INPUT));
		}
		return undef unless (@INPUT);
		$STRING{"<"} = $INPUT = shift(@INPUT);
		if ($INPUT eq 'end of macros') {
			$MACROS_DONE = 1;
			next;
		} elsif ($INPUT =~ /^DATA::(.*)/) {
			$macroname = $1;
			$STRING{"<"} = $INPUT = "DATA";
			# skip operation
			for (;;) {
				$line = <$INPUT> || last;
				return $INPUT if $line =~ /^DATA\s+$macroname/;
			}
			warn "$ME macro package $macroname $NF.\n";
		} else {
			open($INPUT, "<$INPUT") && do {
				&Request(".opened $INPUT") if $MACROS_DONE;
				$LINE_COUNTER{$INPUT} = 0;
				return $INPUT;
			};
			warn "$ME <$INPUT $CO\n";
		}
	}
}

sub getline {
	for (;;) {
		$getline = <$INPUT>;
		$NUMBER{"lines"} = $LINE_COUNTER{$INPUT}++;
		next if (($INPUT eq "DATA") && ($getline =~ /^DATA/));
		$getline = ""
			if (($INPUT eq "DATA") && ($getline =~ /^ENDDATA/));
		return $getline if ($getline);
		return undef unless (&NextFile);
	}
}

sub chop {
	s/\r?\n$//;		# in UNIX, MS-DOS newline becomes "\r\n"
	&Japanese'AnyToEUC;
}

sub MacroArgAll {
	local(@arg) = @MACROARG;
	foreach (@arg) {
		s/ /\\0/g;
	}
	join(" ", @arg);
}

# called in head of ProcessLine, so called twice or more for macros.
sub ExpandEscape {
	s/\\\\/\\\0/g;		# "\" guard: removed in &writeln
	s/\\".*//g;		# comment
	s/\\(['.])/\0$1/g;

	# macro argument substitution
	s/\\\$\#/scalar(@MACROARG)/ge;
	s/\\\$\*/join(" ", @MACROARG)/ge;
	s/\\\$\+/&MacroArgAll/ge;
	s/\\\$([1-9])/$MACROARG[$1-1]/g;

	# string register substitution
	s/\\\*\[([^]]+)]/$STRING{$1}/g;
	s/\\\*\((..)/$STRING{$1}/g;
	s/\\\*(.)/$STRING{$1}/g;

	# inline size change --- ignored
	s/\\s[-+]?[0-9]+/\0/g;

	# number register substitution 
	s/\\n\+\[([^]]+)]/$NUMBER{$1} += $NUMBER{"+$1"}/ge;
	s/\\n\[([^]]+)]/$NUMBER{$1}+0/ge;
	s/\\n\+\((..)/$NUMBER{$1} += $NUMBER{"+$1"}/ge;
	s/\\n\((..)/$NUMBER{$1}+0/ge;
	s/\\n\+(.)/$NUMBER{$1} += $NUMBER{"+$1"}/ge;
	s/\\n(.)/$NUMBER{$1}+0/ge;

}

sub write {
	local($_) = @_;

	# inline font change
	s/(\\f[IBUR])+\\fR/\\fR/g;	# optimization hack for .RI etc.
	s/(\\fP\\fP)+//g;		# optimization hack
	while (/\\f[IBURP]/) {
		if ($& eq "\\fI") {
			s/\\fI/&SetFont("I", $FOREVER)/e;
		} elsif ($& eq "\\fB") {
			s/\\fB/&SetFont("B", $FOREVER)/e;
		} elsif ($& eq "\\fU") {
			s/\\fU/&SetFont("U", $FOREVER)/e;
		} elsif ($& eq "\\fP") {
			s/\\fP/&SetFont("P", $FOREVER)/e;
		} else {
			s/\\fR/&SetFont("", 0)/e;
		}
	}

	# upward or downward shift
	$UPDOWN = 0;
	while (/\\[ud]/) {
		if ($& eq "\\u") {
			$UPDOWN++;
			if ($UPDOWN > 0) { s/\\u/\\QLSUP\\QG/; }
			else { s/\\u/\\QL\/SUB\\QG/; }
		} elsif ($& eq "\\d") {
			$UPDOWN--;
			if ($UPDOWN < 0) { s/\\d/\\QLSUB\\QG/; }
			else { s/\\d/\\QL\/SUP\\QG/; }
		}
	}
	while ($UPDOWN > 0) {
		s/$/\\QL\/SUP\\QG/;
		$UPDOWN--;
	}
	while ($UPDOWN < 0) {
		s/$/\\QL\/SUB\\QG/;
		$UPDOWN++;
	}

	# special one-char sequences
	s/\\\&/\0/g;
	s/\\\|/\0/g;
	s/\\\^//g;		# quite narrow space ... better than one SP?
	s/\\l/ /g;
	s/\\-/-/g;
	s/\\ /\\[nbsp]/g;
	s/\\0/\\[nbsp]/g;
	s/\\c$// && ($UNIFY_ONE_LINE = 1);	# continuation mark

	# defined character escape
	s/\\\((..)/$CHAR{$1}/g;

	# HTML's special characters converted to entity refs.
					s/\\</\\QL/g;	s/\\>/\\QG/g;
	s/&/&amp;/g;	s/"/&quot;/g;	s/</&lt;/g;	s/>/&gt;/g;
	s/\\QA/&/g;	s/\\QQ/"/g;	s/\\QL/</g;	s/\\QG/>/g;

	# SGML entity reference escape --- not compatible with groff
	s/\\\[(\w+)]/&$1;/g;

	# this must be ALL escape expansion
	s/\\e/\\/g;	

	s/\t/"&nbsp;" x 4/ge if $NUMBER{"fillin"};
	s/\0//g;		# remove backslash guard
	&Japanese'EUCToPrintable;
	print "$_";
}

sub writeln {
	local($line) = @_;
	if (defined $STRING{'keep'}) {
		$MACROS{$STRING{'keep'}} .= "$line\n";
	} else {
		&write($line);
		print "\n" unless ($NUMBER{"unify"} || $UNIFY_ONE_LINE);
		$UNIFY_ONE_LINE = undef;
	}
}

sub tag {
	local(@arg) = @_;
	&writeln("\\<". join(" ", @arg). "\\>");
}

sub MSDOS {
	# MS-DOS has no /dev/null.
	return 0 if ( -c '/dev/null' );
	# MS-DOS has /DEV/CON even in drive without /DEV.
	return 1 if ( -f '/DEV/CoN' && ! -d '/DEV' );
	# MS-DOS allows CON or NUL have extension.
	return 2 if ( -f '/coN.3b7' && -f '/NuL.j0Q' && -f '/nUl.!#$' );
	0;
}

sub JPERL {
	return 0 unless ("\xE1\xA2" =~ /^.$/);		# normal perl
	return -1 if ("\xA1\xA1" =~ /^.$/);		# EUC jperl
	1;						# SJIS jperl
}


package Japanese;
#
# Japanese conversion package
# usage:
#	1) set $Japanese'OUT to what encoding output used
#	2) set every input line to $_ and call &Japanese'AnyToEUC
#	3) input has no longer any kanji-origin metacharacter like "\*".
#	 so you can safely use it as regular expression.
#	4) set every output line to $_ and call &Japanese'EUCToPrintable
#
#	To turn off conversion, set a nonzero value to $Japanese'NOCONV.
#

sub HtoZ {
	# JIS X 0201 Katakana -> JIS X 0208 conversion
	tr/\xA1-\xFE/\x21-\x7E/;
	tr/\x21-\x25\x30\x5E\x5F/\xA3\xD6\xD7\xA2\xA6\xBC\xAB\xAC/;
	s/[\xA2-\xD7]/\xA1$&/g;
	tr/\x26-\x2F\x31-\x5D/r!\#%')cegC"\$&(*+\-\/13579;=?ADFHJ-NORUX[^-bdfhi-mos/;
	s/[\x21-\x73]/\xA5$&/g;
	tr/\x21-\x73/\xA1-\xF3/;
}

sub block_JtoE {
	if (s/^\(I//) {
		&HtoZ;
		return $_;
	}
	if (s/^\([\@-Z]//) {
		return $_;
	}
	s/^\$[\@B]// || return $_;
	tr/\x21-\x7E/\xA1-\xFE/;
	return $_;
}

sub StoE {
	local($_) = @_;
	if (/^[\xA1-\xDF]/) {
		&HtoZ;
		return $_;
	}
	local($hi, $lo) = unpack("CC", $_);
	$hi -= 0x40 if ($hi > 0x9F);
	$hi -= 0x30;
	$hi *= 2;
	if ($lo <= 0x9E) {
		$lo-- if ($lo >= 0x80);
		$lo += 0x61;
		$hi--;
	} else {
		$lo += 2;
	}
	if ($hi >= 0x115 || $hi == 0x114 && $lo >= 0xBD) {
		$hi -= 0x1B;
		$lo -= 0x1C;
		($lo += 0x5E, $hi--) if ($lo < 0xA1);
	}
	return pack("CC", $hi, $lo);
}

sub EtoS {
	local($c) = @_;
	local($hi, $lo) = unpack("CC", $c);
	if ($hi % 2) {
		$lo -= 0x61;
		$lo++ if ($lo >= 0x7F);
	} else {
		$lo -= 0x02;
	}
	$hi = int(($hi - 1) / 2) + 0x31; 
	$hi += 0x40 if ($hi >= 0xA0);
	return pack("CC", $hi, $lo);
}

sub AnyToEUC {
	return if $NOCONV;
	if (/\x1B[(\$][\@-Z]/) {
		$IN = "ISO-2022-JP";
		$OUT = "ISO-2022-JP" unless $OUT;
		s/\x0E/\x1B\(I/g;
		s/\x0F/\x1B\(B/g;
		@jblock = split(/\x1B/, $_);
		$result = shift @jblock;
		foreach (@jblock) {
			$result .= &block_JtoE;
		}
		$_ = $result;
	}
	if (/[\x81-\x9D][\x81-\xFE]/) {
		$IN = "Shift_JIS";
		$OUT = "Shift_JIS" unless $OUT;
	}
	if ($IN eq "Shift_JIS") {
		s/[\xA0-\xDF]|[\x81-\x9F\xE0-\xFC][\x40-\x7E\x80-\xFC]/&StoE($&)/ge;
	}
	$_;
}

sub block_EtoJ {
	local($_) = @_;
	tr/\xA1-\xFE/\x21-\x7E/;
	s/^/\x1B\x24B/;
	s/$/\x1B\x28B/;
}

sub EUCToPrintable {
	return if $NOCONV;
	$OUT = "(No Conversion)" unless $OUT;
	if ($OUT eq "ISO-2022-JP") {
		s/([\xA1-\xFE][\xA1-\xFE])+/&block_EtoJ($&)/ge;
	} elsif ($OUT eq "Shift_JIS") {
		s/[\xA1-\xFE][\xA1-\xFE]/&EtoS($&)/ge;
	}
}

__END__
