/*
 * zkfztbez.c --- Kanji font operator for Zeit Bezier font
 *
 * $Id: zkfztbez.c 1.7 93/02/22 07:38:19 K-Asayama Exp Locker: K-Asayama $
 */

#include "math_.h"
#include "memory_.h"
#include "ghost.h"
#include "gp.h"
#include "oper.h"
#include "errors.h"
#include "gspath.h"
#include "gsmatrix.h"
#include "gscoord.h"
#include "gsstate.h"
#include "state.h"

#define D_SIZE  512
#define D_X_CL  0       /* horizontal center of data */
#define D_Y_CL  0       /* vertical center of data */

#define B_SIZE  500
#define B_X_CL  500     /* horizontal center in BuildChar */
#define B_Y_CL  400     /* vertical center in BuildChar */

/* Forward declaration */

extern int kf_is_vchar(P1(int));
extern int kf_vmatrix(P5(int, floatp, floatp, gs_rect *, gs_matrix *));

private int hash(P1(unsigned char *));
private int draw_font(P2(char *, int));
private int outlst(P2(FILE *,size_t));

/* zkfzeitbezier */
int
zkfzeitbezier(register os_ptr op)
{
  char *buffer;
  int code, len, jis_code, wmode;
  
  check_type(op[-2], t_integer);                /* JIS Code */
  check_type(op[-1], t_integer);                /* WMode */
  check_type(op[0], t_string);          /* File Name */ 
  
  len = r_size(op);
  if((buffer = gs_malloc(len + 1, 1, "zkfztbez")) == 0)
    return e_VMerror;
  memcpy(buffer, (char *)op->value.bytes, len);
  buffer[len] = 0;
  
  jis_code = op[-2].value.intval;
  wmode = op[-1].value.intval;
  
  if(wmode && kf_is_vchar(jis_code)) {
    /* vertical kanji character */
    gs_matrix smat, vmat;
    gs_rect bbox;
    
    if((code = gs_gsave(igs)) < 0)
      return code;
    if((code = draw_font(buffer, jis_code)) < 0)
      return code;
    if((code = gs_flattenpath(igs)) < 0)
      return code;
    if((code = gs_pathbbox(igs, &bbox)) < 0)
      return code;
    if((code = gs_grestore(igs)) < 0)
      return code;
    
    gs_currentmatrix(igs, &smat);
    kf_vmatrix(jis_code, 
               (floatp)B_X_CL, (floatp)B_Y_CL, &bbox, &vmat);
    if((code = gs_concat(igs, &vmat)) < 0)
      return code;
    if((code = draw_font(buffer, jis_code)) < 0)
      return code;
    gs_setmatrix(igs, &smat);
  }
  else if((code = draw_font(buffer, jis_code)) != 0) {
    return code;
  }
  
  pop(3);
  gs_free(buffer, len + 1, 1, "zkfztbez");
  return 0;
}

/* -------- Initialization procedure -------- */

op_def zkfzeitbezier_op_defs[] = {
  {"3kfzeitbezier", zkfzeitbezier},
  op_def_end(0)
  };

/* -------- Internal routines -------- */

/*
 * Hash Index
 */
#define OFFSET_TBL_SIZE ((0x75-0x21)*94*4)
#define FNAME_MAX 256

typedef struct index_item_s {
  struct index_item_s *next;
  char str[FNAME_MAX];
  byte table[OFFSET_TBL_SIZE];
  FILE *file[3];
  size_t grif_offset[3];
  size_t grif_size[3];
} index_item;
private index_item *hash_index[256];

/* store */

private int
store(index_item *index)
{
  int key = hash((unsigned char *)(index->str));
  index->next = hash_index[key];
  hash_index[key] = index;
  return 0;
}

/* search */

private int
search(char *str,index_item **index)
{
  int key = hash((unsigned char *)str);
  index_item *item;
  item = hash_index[key];
  while(item != 0) {
    if(strcmp(item->str, str) == 0) {
      *index = item;
      return 1;
    }
    item = item->next;
  }
  return 0;
}

/*
 * Hash function
 */

private int 
hash(unsigned char *str)
{
  unsigned char hash;
  
  for(hash = 0; *str != 0; str++)
    hash ^= *str;
  
  return hash;
}

/*
 * Open font 
 */

private int
open_font(char *file,index_item **pindex)
{
  if(search(file,pindex))
    return 0;
  else {
    int code;
    int i;
    char fname_buf[256];
    index_item *index;
    static size_t s[4] = { 0, 0x30-0x21, 0x50-0x21, 0x75-0x21};
    size_t offset;
    long pos;
    byte buf[20];
    static char *fname_ext[3] = { ".fn0", ".fn1", ".fn2" };
    
    if((*pindex = (index_item *)malloc(sizeof(index_item))) == 0)
      return e_VMerror;
    index = *pindex;
    strcpy(index->str,file);
    
    for (i=0;i<3;++i) {
      strcpy(fname_buf,file);
      strcat(fname_buf,fname_ext[i]);
      
      if((index->file[i] = fopen(fname_buf,gp_fmode_rb)) == 0)
        return e_undefinedfilename;
      if (fread(buf,20,1,index->file[i]) < 1)
        return e_ioerror;
      
      if (memcmp(buf,"zfont2",6) != 0)
        return e_unknownerror;
      offset = ((uint)buf[8]+(uint)buf[9]*0x100) + 10;
      
      if (fseek(index->file[i],offset,0))
        return e_ioerror;
      if (fread(index->table+s[i]*94*4, (s[i+1]-s[i])*94*4, 1,
        index->file[i]) < 1)
        return e_ioerror;
      index->grif_offset[i] = (s[i+1]-s[i])*94*4+offset;
      if (fseek(index->file[i],0,2))
        return e_ioerror;
      pos = ftell(index->file[i]);
      if (pos == -1)
        return e_ioerror;
      index->grif_size[i] = pos - index->grif_offset[i];
    }
    
    if((code = store(index)) < 0)
      return code;
    return 0;
  }
}

#define bytes2ulong(p) \
  ( ( (ulong)(p)[0]) | (((ulong)(p)[1])<<8 ) | \
      (((ulong)(p)[2])<<16) | (((ulong)(p)[3])<<24) )

private int 
draw_font(char *font_name, int jis)
{
  int code;
  index_item *index;
  int jis_level;
  uint h,r;
  byte *p;
  size_t s,e,l;
  
  if((code = open_font(font_name,&index)) <0)
    return code;
  
  h = ((jis>>8)&0xff);
  r = (jis&0xff);
  p = index->table+(size_t)(((h-0x21)*94+r-0x21))*4;
  if (h>=0x50 && h<0x75) {
    jis_level = 2;
  }
  else if (h>=0x30 && h<0x50) {
    jis_level = 1;
  }
  else if (h>=0x21 && h<0x30) {
    jis_level = 0;
  }
  else {
    return 0; /* ignore */
  }
  
  s = bytes2ulong(p);
  if (s == 0xffffffff) {
    return 0;
  }
  e = bytes2ulong(p+4);
  l = (e == 0xffffffff || e < s) ? index->grif_size[jis_level] - s : e-s;
  
  if (fseek(index->file[jis_level],
        s+index->grif_offset[jis_level], 0))
    return e_ioerror;
  code = outlst(index->file[jis_level],l);
  
  return code;
}

#define FACTOR ((floatp)(B_SIZE)/(floatp)(D_SIZE))
#define map_x(x) \
  ((floatp)(B_X_CL) + ((floatp)(x) - (floatp)(D_X_CL)) * FACTOR)
#define map_y(y) \
  ((floatp)(B_Y_CL) + ((floatp)(y) - (floatp)(D_Y_CL)) * FACTOR)

#define moveto(x, y)    gs_moveto(igs, map_x(x), map_y(y))
#define lineto(x, y)    gs_lineto(igs, map_x(x), map_y(y))
#define curveto(x1, y1, x2, y2, x3, y3) \
  gs_curveto(igs, map_x(x1), map_y(y1), \
             map_x(x2), map_y(y2), map_x(x3), map_y(y3))
#define closepath() gs_closepath(igs)

private int
outlst(FILE *fp,size_t len)
{
#define BUF_SIZE (128*6)
  int code = 0;
  int l,p;
  int start = 1;
  int bezier = 0;
  int x, y;
  int xx[3],yy[3];
  byte *buf;
  
  buf = (byte *)gs_malloc(BUF_SIZE,1,"zeitbezier_outlst(buf)");
  if (buf == 0)
    return e_VMerror;
  
  for (start=1;start>=0&&len>0;len-=l) {
    l = min(BUF_SIZE,len);
    if (fread(buf,l,1,fp) < 1) {
      code = e_ioerror; goto bye_func;
    }
    
    for (p = 0; p < l; p+=3) {
      if (p&1) {
        x = ((uint)buf[p-1]<<4) | (buf[p+2]>>4);
        y = ((uint)buf[p+2]<<8) | buf[p+1];
      }
      else {
        x = ((uint)buf[p+1]<<4) | ((buf[p]&0xf0)>>4);
        y = ((uint)(buf[p]&0x0f)<<8) | (uint)buf[p+3];
      }
      
      if (x&0x800)
        bezier++;
      else
        bezier = 0;
      
      x = ( (x & 0x400) ? -(x & 0x3ff) : (x & 0x3ff) );
      y = ( (y & 0x400) ? -(y & 0x3ff) : (y & 0x3ff) );
      
      if (start) {                 /* begin */
        if (x == -1023 && y == -1023) {
          start = -1; break;
        }
        moveto(x,y);
        start = 0;
        bezier = 0;
      }
      else if (x == -1023 && y == -1023) {      /* end */
        closepath();
        start = 1;
        bezier = 0;
      }
      else if (bezier == 0) {
          lineto(x,y);
      }
      else if (bezier == 3 ) {
        curveto(xx[1],yy[1],xx[2],yy[2],x,y);
        bezier = 0;
      }
      else {
        xx[bezier] = x;
        yy[bezier] = y;
      }
    }
  }
  
  bye_func:
  gs_free((char *)buf,BUF_SIZE,1,"zeitbezier_outlst(buf)");
  return code;
#undef BUF_SIZE
}
