require "numru/gphys/varray"
require "numru/gphys/varraynetcdf"

module NumRu

   class Axis

      def initialize(cell=false,bare_index=false,name=nil)
	 @init_fin = false         # true/false (true if initializatn finished)
	 @name = name              # String
	 @pos = nil                # VArray (to set it can be deferred if
                                   # @cell to support mother axes)
	 @cell = cell              # true/false
	 @cell_center = nil        # VArray (defined if @cell)
         @cell_bounds = nil        # VArray (defined if @cell)
         @bare_index = bare_index  # true/false(if true @cell is meaningless)
         @aux = nil                # Hash of VArray (auxiliary quantities)
      end

      def inspect
	 "<axis pos=#{@pos.inspect}>"
      end

      def name=(nm)
	 @name=nm
      end
      def name
	 @name || "noname"
      end

      def cell?
	 @cell
      end
      def cell_center?
	 @cell && @pos.equal?(@cell_center)
      end
      def cell_bounds?
	 @cell && @pos.equal?(@cell_bounds)
      end
      def bare_index?
	 @bare_index
      end

      def flatten
	 # return VArrays contained in a flat array
	 out = Array.new
	 out.push(@pos) if @pos
	 out.push(@cell_center) if @cell_center && !cell_center?
	 out.push(@cell_bounds) if @cell_bounds && !cell_bounds?
	 if @aux
	    @aux.each{|k,v|
	       out.push(v)
	    }
	 end
	 out
      end

      def copy
	 # deep clone onto memory
	 out = Axis.new(cell?, bare_index?, name)
	 if cell?
	    out.set_cell( @cell_center.copy, @cell_bounds.copy )
	    if cell_center?
	       out.set_pos_to_center
	    elsif cell_bounds?
	       out.set_pos_to_bounds
	    end
	 else
	    out.set_pos( @pos.copy )
	 end
	 if @aux
	    @aux.each{|k,v|
	       out.set_aux(k, v.copy)
	    }
	 end
	 out
      end

      def pos=(pos)
	 if !@cell
	    if ! pos.is_a?(VArray)
	       raise ArgumentError,"arg not a VArray: #{pos.class}"  
	    end
	    if pos.rank != 1
	      raise ArgumentError,"rank of #{pos.name} (#{pos.rank}) is not 1" 
	    end

	    @pos=pos
	    @init_fin = true
	    if ! @name
	       @name = pos.name
	    end
	 else
	    raise "This method is not available for a cell axis. "+
	          "Use set_pos_to_center or set_pos_to_bounds instead."
	 end
	 pos
      end

      def set_pos(pos)
	 self.pos= pos
	 self
      end

      def pos
	 raise "pos has not been set" if !@pos
	 @pos
      end
      def cell_center
	 @cell_center
      end
      def cell_bounds
	 @cell_bounds
      end

      def length
	 if @pos
	    @pos.length
	 else
	    raise "length is not determined until pos is set"
	 end
      end

      def set_cell(center, bounds, name=nil)
	 # it is the user's obligation to ensure that center and bounds
         # have the same units etc.

	 # < error check >

	 if ! @cell; raise "method not available for a non-cell axis"; end

	 if ! center.is_a?(VArray)
	    raise ArgumentError,"1st arg not a VArray: #{center.class}"  
	 end
	 if center.rank != 1
	    raise ArgumentError,"center: rank of #{center.name} (#{center.rank}) is not 1" 
	 end

	 if ! bounds.is_a?(VArray)
	    raise ArgumentError,"2nd arg not a VArray: #{bounds.class}"  
	 end
	 if bounds.rank != 1
	    raise ArgumentError,"bounds: rank of #{bounds.name} (#{bounds.rank}) is not 1" 
	 end

	 if( center.length != bounds.length-1 )
	    raise "center.length != bounds.length-1"
	 end

	 # < do the job >

	 @cell_center = center
	 @cell_bounds = bounds
	 if name
	    @name=name
	 end
	 @init_fin = true       # To set @pos is deferred at this moment.
                                # use set_pos_to_(bounds|center) to make 
                                # the object fully available
         self
      end

      def set_aux(name,vary)
	 if !name.is_a?(String) 
	    raise ArgumentError,"1nd arg: not a String"
	 end

	 if ! vary.is_a?(VArray)
	    raise ArgumentError,"2nd arg not a VArray: #{vary.class}"  
	 end
	 if vary.rank != 1
	    raise ArgumentError,"rank of #{vary.name} (#{vary.rank}) is not 1" 
	 end

	 if !@aux; @aux = Hash.new; end

	 @aux[name] = vary
      end
      def get_aux(name)
	 @aux[name]
      end
      def aux_names
	 @aux ? @aux.keys : []
      end

      def [] (slicer)
	 case slicer
	 when Fixnum
	    # slicer is used as is
	    newax=Axis.new(@cell,@bare_index)
	 when Range, true
	    # re-organize the range to support slicing of variables
	    # with a 1 larger / smaller length.
	    if true===slicer
	       range=0..-1
	    else
	       range = slicer
	    end
	    first = range.first
	    last = range.exclude_end? ? range.last-1 : range.last
	    if first < 0
	       first += self.length  # first will be counted from the beginning
	    end
	    if last >= 0
	       last -= self.length   # last will be counted from the end
	    end
	    slicer = first..last
	    newax=Axis.new(@cell,@bare_index)
	 when Hash
	    range = slicer.keys[0]
	    step = slicer[range]
	    first = range.first
	    last = range.exclude_end? ? range.last-1 : range.last
	    if first < 0
	       first += self.length  # first will be counted from the beginning
	    end
	    if last >= 0
	       last -= self.length   # last will be counted from the end
	    end
	    slicer = { first..last, step }
	    newax=Axis.new(false,@bare_index)  # always not a cell axis
	 else
	    raise ArgumentError, "Axis slicing with #{slicer.inspect} is not available"
	 end

	 if newax.bare_index? || !newax.cell?
	    if (lc=@pos[slicer]).rank == 0
	       cval = ( lc.val.is_a?(Numeric) ? lc.val : lc.val[0] )
	       return "#{self.name}=#{cval}"
            end
	    newax.set_pos( lc )
	 elsif newax.cell?
	    center=@cell_center[slicer]
	    bounds=@cell_bounds[slicer]
	    if self.cell_bounds?
	       if bounds.rank == 0
		  b = (bounds.val.is_a?(Numeric) ? bounds.val : bounds.val[0])
		  return "#{self.name}=#{b}"
	       end
	    else
	       if center.rank == 0
		  c = (center.val.is_a?(Numeric) ? center.val : center.val[0])
		  return "#{self.name}=#{c}"
	       end
	    end
	    newax.set_cell( center, bounds )
	    if self.cell_center?
	       newax.set_pos_to_center
	    elsif self.cell_bounds?
	       newax.set_pos_to_bounds
	    end
	 end
	 if @aux
	    @aux.each{ |name, vary|
	       if (aux=vary[slicer]).rank != 0
		  newax.set_aux(name, aux)
	       else
		  print "WARNING #{__FILE__}:#{__LINE__} auxiliary VArray #{name} is eliminated\n"
	       end
	    }
	 end
	 newax.set_default_algorithms
	 newax
      end

      def set_cell_guess_bounds(center, name=nil)
	 # derive bounds with a naive assumption (should be OK for
	 # an equally separated axis).
	 if ! center.is_a?(VArray) || center.rank != 1
	    raise ArgumentError, "1st arg: not a VArray, or its rank != 1" 
	 end
	 vc = center.val
	 vb = NArray.float( vc.length + 1 )
	 vb[0] = vc[0] - (vc[1]-vc[0])/2   # Assume this!!
	 for i in 1...vb.length
	    vb[i] = 2*vc[i-1] - vb[i-1]    # from vc[i-1] = (vb[i-1]+vb[i])/2
	 end
	 bounds = VArray.new(vb, center.attr)
	 set_cell(center, bounds, name)
      end

      def set_pos_to_center
	 raise "The method is not available for a non-cell axis" if ! @cell
	 @pos = @cell_center
	 if !@name
	    @name = @cell_center.name
	 end
	 self
      end

      def set_pos_to_bounds
	 raise "The method is not available for a non-cell axis" if ! @cell
	 @pos = @cell_bounds
	 if !@name
	    @name = @cell_bounds.name
	 end
	 self
      end

      @@operations = Array.new
      @@observer_classes = Array.new

      def Axis.add_operation( method_name )
	 @@operations.push(method_name)
	 eval <<-EOS
	    def #{method_name}(ary,dim,*extra)
	       if !@algorithms || !@algorithms[\"#{method_name}\"]
	          raise "Algorithm not defined. " +
	             "Call set_default_algorithms to set a default one."
	       end
	       @algorithms[\"#{method_name}\"].call(ary,dim,*extra)
            end
	 EOS
         @@observer_classes.each{|obc| obc.axis_operations_update}
      end

      def set_algorithm(method_name, proc)
	 raise ArgumentError,"2nd arg not a Proc" if ! proc.is_a?(Proc)
	 if ! @@operations.include?(method_name)
	    raise "#{method_name} is a new operation, which has not been registered in Axis"+
               "Call Axis.add_operation(method_name) to add it."
	 end
	 @algorithms[method_name] = proc
      end

      def Axis.defined_operations( observer_class=nil )
	 # If observer_class is registered, Axis.add_operation will 
	 # notify changes by calling observer_class.axis_operations_update
	 @@observer_classes.push( observer_class ) if observer_class
	 @@operations
      end

      def set_default_algorithms

	 raise "Initialization is not completed" if ! @init_fin
	 if ! @pos
	    raise "pos has not been set. Call set_pos_to_(center|bounds)." 
	 end

	 # < define numerical integration / averaging >

	 if( @bare_index )
	    weight = NArray.int(@pos.val.length).fill!(1)
	    dist = weight.length
	 elsif ( !@cell || (@cell && @pos.equal?(@cell_bounds)) )
	    # --- for trapezoidal formula ---
	    posv = @pos.val
	    return nil if posv.length == 1  # cannot define the algorithm
	    dist = (posv[-1]-posv[0]).abs
	    weight = posv.dup
	    weight[0] = (posv[1]-posv[2]).abs/2
	    weight[-1] = (posv[-1]-posv[-2]).abs/2
	    weight[1..-2] = (posv[2..-1] - posv[0..-3]).abs/2
	 else
	    # --- assume that the center values represents the means ---
	    bd = @cell_bounds.val
	    weight = (bd[1..-1] - bd[0..-2]).abs
	    dist = (bd[-1]-bd[0]).abs
	 end

	 axrange = ":#{@pos.val.min}..#{@pos.val.max}"

	 @algorithms = Hash.new if ! @algorithms
	 # @algorithms[method_name] = aProc
         #   Here, the return value of aProc must be [result, new_axis]
	 #   (result is a NArray; new_axis==nil means that the axis is
	 #   to be eliminated (contracted) )

	 if ! @bare_index
	    @algorithms["integ"] = Proc.new{|ary,dim| 
	       sh = Array.new
	       for i in 0 ..ary.rank-1
		  sh[i] = ( (i==dim) ? weight.length : 1 )
	       end
	       [ ( ary * weight.reshape(*sh) ).val.sum(dim), 
		  "integ "+@name+axrange ]
	    }
	 end
	 @algorithms["mean"] = Proc.new{|ary,dim| 
	    sh = Array.new
	    for i in 0 ..ary.rank-1
	       sh[i] = ( (i==dim) ? weight.length : 1 )
	    end
	    wgt_ext = weight.reshape(*sh)
	    [ ( ary * weight.reshape(*sh) ).val.sum(dim)/dist,
	      "mean "+@name+axrange ]
	 }
      end

      #######

      ["integ","mean"].each{ |method|
	 add_operation(method)
      }

   end
end

###################################################
## < test >

if $0 == __FILE__
   include NumRu
   xc = VArray.new( NArray.float(10).indgen! + 0.5 ).rename("x")
   xb = VArray.new( NArray.float(11).indgen! ).rename("xb")
   axpt = Axis.new().set_pos(xc)
   axcel = Axis.new(true).set_cell(xc,xb)
   axcel_c = axcel.dup.set_pos_to_center
   axcel_b = axcel.dup.set_pos_to_bounds
   axcel2 = Axis.new(true).set_cell_guess_bounds(xc)
   p "axcel",axcel, "axcel2",axcel2
   print "########\n"
   p axpt.pos.val
   p axcel_c.pos.val
   p axcel_b.pos.val
   axpt.set_default_algorithms
   axcel_c.set_default_algorithms
   axcel_b.set_default_algorithms
   z = VArray.new( NArray.float(xc.length, 2).random! )
   w = VArray.new( NArray.float(xc.length).indgen! )
   w2 = VArray.new( NArray.float(xb.length).indgen!-0.5 )
   p z.val
   p axpt.mean(z,0), axcel_c.mean(z,0)
   p z.val.sum(0)/10
   p  axpt.mean(w,0), axcel_c.mean(w,0), 
      axpt.integ(w,0), axcel_c.integ(w,0), 
      axcel_b.integ(w2,0)
   # axcel.set_default_algorithms  # this is to fail
   print "////////\n"
   p axpt[1..3].pos.val
   p axcel_c[1..3].cell_center.val, axcel_c[1..3].cell_bounds.val
   p axcel_b[1..3].cell_center.val, axcel_b[1..3].cell_bounds.val
   p axpt[1]
   p axcel_b[5]
   p axpt
   axcel_c.set_aux('aux1',xc*4)
   p axcel_c.flatten
   p axcel_c.copy.flatten
end
