
!> A module for physical constants
!> This module is required for all components

module pp_phys_const
  use pp_vardef
  implicit none

  !> circular constant
  real(r_size), public :: pi = 3.141592653589793_r_size

  !> triple point temperature of water (K)
  real(r_size), public :: ttriple = 273.16_r_size
                               
  !> Kelvin at 0 degree Celsius (K)
  real(r_size), public :: tkelvn = 273.15_r_size
                               
  !> latent heat of vaporization (J/kg)
  real(r_size), public :: hlatnt = 2.5e+6_r_size
                               
  !> latent heat of fusion (J/kg)
  real(r_size), public :: hlf = 3.336e+5_r_size
                               
  !> individual gas constant for dry air(J/(K kg))
  real(r_size), public :: rd = 287.05e0_r_size
                               
  !> individual gas constant for wet air(J/(K kg))
  real(r_size), public :: rv = 461.5e0_r_size

  !> a coefficient in the Tetens's Formula for vapor phase
  real(r_size), public :: e0cw = 6.11e2_r_size
  !> parameter1 in the Tetens's Formula for vapor phase
  real(r_size), public :: tetn1w = 17.27e0_r_size
  !> parameter2 in the Tetens's Formula for vapor phase
  real(r_size), public :: tetn2w = 273.15e0_r_size
  !> parameter3 in the Tetens's Formula for vapor phase
  real(r_size), public :: tetn3w = 35.85e0_r_size
  !> a coefficient in the Tetens's Formula for ice
  real(r_size), public :: e0ci = 6.11e2_r_size
  !> parameter1 in the Tetens's Formula for ice phase
  real(r_size), public :: tetn1i = 21.875e0_r_size
  !> parameter2 in the Tetens's Formula for ice phase
  real(r_size), public :: tetn2i = 273.15e0_r_size
  !> parameter3 in the Tetens's Formula for ice phase
  real(r_size), public :: tetn3i = 7.65e0_r_size
  !> heat capacity of water (J/(kg K))
  real(r_size), public :: cwater = 4187.0e0_r_size
  !> heat capacity of ice (J/(kg K))
  real(r_size), public :: cice = 2093.0e0_r_size
  !> von Karman Constant
  real(r_size), public :: vkman = 0.4e0_r_size
  !> Accelaration due to gravity (m/s2)
  real(r_size), public :: grav = 9.80665e0_r_size
  !> Stephan Boltzmann Constant (\f${\rm W m^{-2}K^{-4}}\f$)
  real(r_size), public :: stb = 5.67e-8_r_size
  !> Solar Constant (\f${\rm W m^{-2}}\f$)
  real(r_size), public :: sc0 = 1.367e3_r_size

  ! derived parameters
  !> Specific heat at constant pressure
  real(r_size), public :: cp
  !> rd / cp
  real(r_size), public :: rdvcp
  !> g / cp
  real(r_size), public :: gdvcp
  !> g / rd
  real(r_size), public :: gdvrd
  !> reciprocal of triple point temperature of water
  real(r_size), public :: r_ttriple 
  !> reciprocal of specific heat at constant pressure
  real(r_size), public :: r_cp
  !> latent heat of sublimation
  real(r_size), public :: hls    ! sublimation
  !> reciprocal of latent heat of fusion
  real(r_size), public :: r_hlf
  !> hlatnt ** 2 / rv
  real(r_size), public :: chlv
  !> hls ** 2 / rv
  real(r_size), public :: chls
  !> hls * hlf / rv
  real(r_size), public :: chlslf

  !> ratio of rd to rv (nearly equal to 0.622)
  real(r_size), public :: epsilon
  !> 1 - epsilon
  real(r_size), public :: one_minus_epsilon
  ! rv / rd - 1 (nearly equal to 0.608, appearing in virtual temperature)
  real(r_size), public :: c_virtual
  !> cubic root of 6 (appearing in cloud micro physics)
  real(r_size), public :: cbrt6
  !> 1/3
  real(r_size), public :: one_third
  !> 2/3
  real(r_size), public :: two_thirds

  !> pi / 180
  real(r_size), public :: pi180

  ! constants
  !> 0.0
  real(r_size), public, parameter :: rzero = 0.0_r_size

  public :: pp_phys_const_set

contains
  !> A public subroutine to set physical constants.
  !> Users should call this subroutine before calling any other
  !> subroutines in the library.
  !> The default values are given in the head of this module.
  !> However, if you want to use your own values instead of the 
  !> default ones, you can set the values through the subroutine.
  !> Attribute of all arguments is "optional". Therefore, 
  !> you should specify only variables which you want to change.

  subroutine pp_phys_const_set( &
    & pi_in, ttriple_in, tkelvn_in, hlatnt_in, &
    & hlf_in, rd_in, rv_in, &
    & e0cw_in, tetn1w_in, tetn2w_in, tetn3w_in, &
    & e0ci_in, tetn1i_in, tetn2i_in, tetn3i_in, &
    & cwater_in, cice_in, vkman_in, grav_in, stb_in, sc0_in)

    use pp_vardef
    implicit none

    !> User input for circular constant
    real(r_size), intent(in), optional :: pi_in      
    !> User input for triple point temperature of water (K)
    real(r_size), intent(in), optional :: ttriple_in 
    !> User input for Kelvin at 0 degree Celsius (K)
    real(r_size), intent(in), optional :: tkelvn_in  
    !> User input for latent heat of vaporization (J/kg)
    real(r_size), intent(in), optional :: hlatnt_in  
    !> User input for latent heat of fusion (J/kg)
    real(r_size), intent(in), optional :: hlf_in  
    !> User input for individual gas constant for dry air(J/(K kg))
    real(r_size), intent(in), optional :: rd_in      
    !> User input for Individual gas constant for wet air(J/(K kg))
    real(r_size), intent(in), optional :: rv_in      
    !> User input for a coefficient in the Tetens's Formula for vapor phase
    real(r_size), intent(in), optional :: e0cw_in    
    !> User input for parameter1 in the Tetens's Formula for vapor phase
    real(r_size), intent(in), optional :: tetn1w_in  
    !> User input for parameter2 in the Tetens's Formula for vapor phase
    real(r_size), intent(in), optional :: tetn2w_in  
    !> User input for parameter3 in the Tetens's Formula for vapor phase
    real(r_size), intent(in), optional :: tetn3w_in  
    !> User input for a coefficient in the Tetens's Formula for ice
    real(r_size), intent(in), optional :: e0ci_in    
    !> User input for parameter1 in the Tetens's Formula for ice phase
    real(r_size), intent(in), optional :: tetn1i_in  
    !> User input for parameter2 in the Tetens's Formula for ice phase
    real(r_size), intent(in), optional :: tetn2i_in  
    !> User input for parameter3 in the Tetens's Formula for ice phase
    real(r_size), intent(in), optional :: tetn3i_in  
    !> User input for heat capacity of water (J/(kg K))
    real(r_size), intent(in), optional :: cwater_in
    !> User input for heat capacity of ice (J/(kg K))
    real(r_size), intent(in), optional :: cice_in
    !> Uset input for von Karman Constant
    real(r_size), intent(in), optional :: vkman_in
    !> User input for Accelaration due to gravity (m/s2)
    real(r_size), intent(in), optional :: grav_in
    !> User input for Stephan Boltzmann Constant (\f${\rm W m^{-2}K^{-4}}\f$)
    real(r_size), intent(in), optional :: stb_in
    !> Solar Constant (\f${\rm W m^{-2}}\f$)
    real(r_size), intent(in), optional :: sc0_in

    ! Numerical constants
    one_third = 1.0e0_r_size / 3.0e0_r_size
    two_thirds = 2.0e0_r_size / 3.0e0_r_size
    cbrt6 = 6.0_r_size ** one_third

    if (present(pi_in))        pi      =    pi_in      
    if (present(ttriple_in))   ttriple =    ttriple_in 
    if (present(tkelvn_in))    tkelvn  =    tkelvn_in
    if (present(hlatnt_in))    hlatnt  =    hlatnt_in 
    if (present(hlf_in))       hlf     =    hlf_in
    if (present(rd_in))        rd      =    rd_in      
    if (present(rv_in))        rv      =    rv_in      
    if (present(e0cw_in))      e0cw    =    e0cw_in    
    if (present(tetn1w_in))    tetn1w  =    tetn1w_in  
    if (present(tetn2w_in))   tetn2w  =    tetn2w_in  
    if (present(tetn3w_in))   tetn3w  =    tetn3w_in  
    if (present(e0ci_in))     e0ci    =    e0ci_in    
    if (present(tetn1i_in))   tetn1i  =    tetn1i_in  
    if (present(tetn2i_in))   tetn2i  =    tetn2i_in  
    if (present(tetn3i_in))   tetn3i  =    tetn3i_in  
    if (present(cwater_in))   cwater  =    cwater_in
    if (present(cice_in))     cice    =    cice_in
    if (present(vkman_in))    vkman   =    vkman_in
    if (present(grav_in))     grav    =    grav_in
    if (present(stb_in))      stb     =    stb_in
    if (present(sc0_in))      sc0     =    sc0_in

    ! derived parameters
    cp = 0.5e0_r_size * 7.0e0_r_size * rd
    r_ttriple = 1.0_r_size / ttriple
    r_cp = 1.0_r_size / cp
    rdvcp = rd * r_cp
    gdvcp = grav * r_cp
    gdvrd = grav / rd
    hls = hlatnt + hlf
    r_hlf = 1.0_r_size / hlf
        
    chlv = hlatnt ** 2 / rv
    chls = hls ** 2 / rv
    chlslf = hls * hlf / rv

    epsilon = rd / rv
    one_minus_epsilon = 1.0_r_size - epsilon
    c_virtual = 1.0_r_size / epsilon - 1.0_r_size

    pi180 = pi / 180.0_r_size

    return

  end subroutine pp_phys_const_set


end module pp_phys_const


