module pbl_mym

contains
  !
  subroutine pbl_mym_level2(dbdz, dvdzm, gm, gh, sm, sh)
    use pbl_grid, only: nz
    use pp_vardef
    use pbl_mym_const, only: ri1, ri2, ri3, ri4, shc, smc, rfc, rf1, rf2
    use pp_monit, only: pp_monit_store
    implicit none

    real(r_size), intent(in) :: dbdz(nz)
    real(r_size), intent(in) :: dvdzm(nz)

    real(r_size), intent(out) :: gm(nz)
    real(r_size), intent(out) :: gh(nz)
    real(r_size), intent(out) :: sm(nz)
    real(r_size), intent(out) :: sh(nz)

    real(r_size) :: ri, rf
    integer(4) :: kz
    
    do kz = 1, nz
      gm(kz) =   dvdzm(kz) * dvdzm(kz)
      gh(kz) = - dbdz(kz)
      !   Gradient Richardson number
      ri = - gh(kz) / max( gm(kz), 1.0e-10_r_size )
      !   Flux Richardson number
      rf = min(ri1 * (ri + ri2 - sqrt(ri ** 2 - ri3 * ri + ri4)), rfc )
      sh(kz) = shc * (rfc - rf) / (1.0 - rf)
      sm(kz) = smc * (rf1 - rf) / (rf2 - rf) * sh(kz)

    end do
!   call pp_monit_store(nz, 1, nz, sh, 'sm')
!   call pp_monit_store(nz, 1, nz, sh, 'sh')
    return
  end subroutine pbl_mym_level2
  !
  subroutine pbl_mym_length(fb_surf, r_mosurf, z_f, dz_f, dbdz, qkw, el)
    use pp_vardef
    use pbl_mym_const
    use pbl_grid, only: nz
    use pp_phys_const, only: one_third, vkman
    use pbl_mym_parm, only: my_z_limit_elb
    use pp_monit, only: pp_monit_store
    implicit none

    real(r_size), intent(in) :: fb_surf
    real(r_size), intent(in) :: r_mosurf
    real(r_size), intent(in) :: z_f(nz)
    real(r_size), intent(in) :: dz_f(nz-1)
    real(r_size), intent(in) :: dbdz(nz)
    real(r_size), intent(in) :: qkw(nz)
    real(r_size), intent(out) :: el(nz)

    integer(4) :: kz
    real(r_size) :: vsc
    real(r_size) :: qdz
    real(r_size) :: elt
    real(r_size) :: elb
    real(r_size) :: els
    real(r_size) :: rbv
    real(r_size) :: zeta

    real(r_size), parameter :: zmax = 1.0
                  ! constant used in calculating els
    real(r_size), parameter :: cns = 2.7
                  ! constant used in calculating els

    elt = 0.0
    vsc = 0.0

    do kz = 1, nz - 1
      qdz = qkw(kz) * dz_f(kz)
      elt = elt + qdz * z_f(kz)
      vsc = vsc + qdz
    end do

    elt = max(my_alpha1 * elt / (vsc + 1.e-10_r_size), elt_min)
    vsc = (elt * max(fb_surf, 0.0_r_size)) ** one_third

    do kz = 1, nz
      if (dbdz(kz) > 0.0) then
        rbv = 1.0 / sqrt(dbdz(kz))
        elb = my_alpha2 * qkw(kz) * rbv &
          &          * (1.0 + alp32 * sqrt(vsc * rbv / elt))
      else
        elb = 1.0e10
      end if

      if (z_f(kz) > my_z_limit_elb) then
        elb = min(elb, dz_f(kz))
      end if

      zeta = z_f(kz) * r_mosurf
      if (zeta > 0.0) then
        els = vkman * z_f(kz) / (1.0 + cns * min(zeta, zmax))
      else
        els = vkman * z_f(kz)                               &
          * min((1.0_r_size - my_alpha4 * zeta) ** 0.2_r_size, 2.0_r_size)
      end if
      el(kz) = elb / ( elb / elt + elb / els + 1.0)
    end do
!   call pp_monit_store(nz, 1, nz, el, 'el')
    return
  end subroutine pbl_mym_length
  !
  subroutine pbl_mym_surf_prod(r_mosurf, u_s, ftl_surf, fqw_surf, z_f1, &
    & pdk1, pdt1, pdq1, pdc1)
    use pp_vardef
    use pp_phys_const, only: vkman
    use pbl_mym_option, only: my_lowest_pd_surf, l_my_lowest_pd_surf_tqc
    use pbl_mym_phi, only: pbl_mym_phi_calc
    implicit none

    real(r_size), intent(in) :: r_mosurf
    real(r_size), intent(in) :: u_s
    real(r_size), intent(in) :: ftl_surf
    real(r_size), intent(in) :: fqw_surf
    real(r_size), intent(in) :: z_f1

    real(r_size), intent(inout) :: pdk1
    real(r_size), intent(inout) :: pdt1
    real(r_size), intent(inout) :: pdq1
    real(r_size), intent(inout) :: pdc1

    real(r_size) :: pmz
    real(r_size) :: phh
    real(r_size) :: phm

    ! Overwrite production terms by ones calculated with surface fluxes
    if (my_lowest_pd_surf > 0) then
      call pbl_mym_phi_calc(r_mosurf, z_f1, pmz, phh)

      pdk1 = 1.0 * u_s ** 3 * pmz / (vkman * z_f1)
      if (l_my_lowest_pd_surf_tqc) then
        phm  = 1.0 / u_s * phh  / (vkman * z_f1)
        pdt1 = phm * ftl_surf ** 2
        pdq1 = phm * fqw_surf ** 2
        pdc1 = phm * ftl_surf * fqw_surf
      end if  ! IF L_MY_lowest_pd_surf_tqc
    end if  ! IF MY_lowest_pd_surf
    return
  end subroutine pbl_mym_surf_prod
  !
  subroutine pbl_mym_turbulence_common(fb_surf, r_mosurf, qkw,&
    & dbdz, dvdzm, z_f, dz_f, &
    & gm, gh, sm, sh, qdiv, el, cw25)
    use pp_vardef
    use pbl_grid, only: nz
    use pp_phys_const, only: one_third
    use pbl_mym_const, only: b1, e1c, e2c, e3c, e4c, e5c, c1, a1, a2

    implicit none

    real(r_size), intent(in) :: fb_surf
    real(r_size), intent(in) :: r_mosurf
    real(r_size), intent(in) :: qkw(nz)

    real(r_size), intent(in) :: dbdz(nz)
                                ! Buoyancy gradient across layer
                                ! interface
    real(r_size), intent(in) :: dvdzm(nz)
                                ! Modulus of wind shear
    real(r_size), intent(in) :: z_f(nz)
    real(r_size), intent(in) :: dz_f(nz-1)

    real(r_size), intent(out) :: gm(nz)

    real(r_size), intent(out) :: gh(nz)
                                ! - buoyancy gradient
                                ! (a numerator of gradient Richardson number)
    real(r_size), intent(out) :: sm(nz)
    real(r_size), intent(out) :: sh(nz)
    real(r_size), intent(out) :: qdiv(nz)
    real(r_size), intent(out) :: el(nz)
    real(r_size), intent(out) :: cw25(nz)

    integer(4) :: kz

    real(r_size) :: e1
                                ! a variable denoted to E1 in the papers
    real(r_size) :: e2
                                ! a variable denoted to E2 in the papers
    real(r_size) :: e3
                                ! a variable denoted to E3 in the papers
    real(r_size) :: e4
                                ! a variable denoted to E4 in the paper
    real(r_size) :: elsq
                                ! square of mixing length
    real(r_size) :: gmel
                                ! GM times the mixing length
    real(r_size) :: ghel
                                ! GH times the mixing length
    real(r_size) :: q2sq
                                ! qke derived by level 2 scheme
    real(r_size) :: q3sq
                                ! qke derived by level 2.5 or level 3
    real(r_size) :: eden
                                ! work variable
    real(r_size) :: reden
                                ! reciprocal of eden

    call pbl_mym_level2(dbdz, dvdzm, gm, gh, sm, sh)

    call pbl_mym_length(fb_surf, r_mosurf, z_f, dz_f, dbdz, qkw, el)

    do kz = 1, nz
      elsq = el(kz) ** 2
      q2sq = b1 * elsq * (sm(kz) * gm(kz) + sh(kz) * gh(kz))
      q3sq = qkw(kz) ** 2
      gmel = gm(kz) * elsq
      ghel = gh(kz) * elsq

      ! adjust SM and SH by SQRT(q3sq / q2sq)
      if ( q3sq < q2sq ) then
        qdiv(kz) = sqrt(q3sq / q2sq)
        sm(kz) = sm(kz) * qdiv(kz)
        sh(kz) = sh(kz) * qdiv(kz)

        e1   = q3sq - e1c * ghel * qdiv(kz) ** 2
        e2   = q3sq - e2c * ghel * qdiv(kz) ** 2
        e3 = e1 + e3c * ghel * qdiv(kz) ** 2
        e4 = e1 - e4c * ghel * qdiv(kz) ** 2
        eden = e2 * e4 + e3 * e5c * gmel * qdiv(kz) ** 2
        eden = max(eden, 1.0e-20_r_size)
        reden = 1.0 / eden
      else
        e1 = q3sq - e1c * ghel
        e2 = q3sq - e2c * ghel
        e3 = e1 + e3c * ghel
        e4 = e1 - e4c * ghel
        eden = e2 * e4 + e3 * e5c * gmel
        eden = max(eden, 1.0e-20_r_size)
        reden = 1.0 / eden

        qdiv(kz) = 1.0
        sm(kz) = q3sq * a1 * (e3 - 3.0 * c1 *e4) * reden
        sh(kz) = q3sq * a2 * (e2 + 3.0 * c1 * e5c * gmel)  * reden
      end if ! test if q3sq < q2sq
      cw25(kz) =(e2 + 3.0 * c1 * e5c * gmel                           &
                           * qdiv(kz) ** 2) * one_third * reden * e1
    end do
    return

  end subroutine pbl_mym_turbulence_common
  !
  subroutine pbl_mym_turbulence_level3(              &
    & ftl_surf, fqw_surf, r_mosurf, u_s,  &
    & gm, gh, sm, sh, &
    & vt, vq, gtr, wb_ng, dtldz, dqwdz, dudz, dvdz, &
    & tsq, qsq, cov, qkw, cw25, el, z_f, rdz_f, qdiv, &
    & dfm, dfh, dfu_cg, dfv_cg, dft_cg, dfq_cg, &
    & pdk, pdt_tsq, pdt_cov, pdt, pdq_qsq, pdq_cov, pdq, &
    & pdc_tsq, pdc_qsq, pdc_cov, pdc, &
    & c2sq, smd_coef, gamv_coef)

    use pp_vardef
    use pbl_grid, only: nz
    use pp_phys_const, only: pi
    use pbl_mym_const, only: b2, e2c, e3c, e4c, e5c, cc3, a2, a1_2, &
      & coef_trbvar_diff
    use pbl_mym_option, only: imp_mode, l_my_prod_adj
    use pbl_mym_option_symbol, only: half_impl, full_impl
    use pbl_mym_parm, only:  my_prod_adj_fact
    
    implicit none

    ! Intent IN Variables
    real(r_size), intent(in) :: ftl_surf
                                ! surface heat flux
    real(r_size), intent(in) :: fqw_surf
                                ! surface moisture flux
    real(r_size), intent(in) :: r_mosurf
                                ! reciprocal of Monin-Obukhov length
    real(r_size), intent(in) :: u_s
                                ! Surface friction velocity

    real(r_size), intent(in) :: gm(nz)
    real(r_size), intent(in) :: gh(nz)
                                ! - buoyancy gradient
                                ! (a numerator of gradient Richardson number)
    real(r_size), intent(in) :: sm(nz)
                                ! Non-dimensional diffusion coefficients for
                                ! momentum derived by level 2 scheme
    real(r_size), intent(in) :: sh(nz)
                                ! Non-dimensional diffusion coefficients for
                                ! scalars derived by level 2 scheme
    real(r_size), intent(in) :: vt(nz)
                                ! A buoyancy param 
    real(r_size), intent(in) :: vq(nz)
                                ! A buoyancy param 
    real(r_size), intent(in) :: gtr(nz)
                                ! g/thetav
    real(r_size), intent(in) :: wb_ng(nz)
                                ! buoyancy flux related to the skewness
    real(r_size), intent(in) :: dtldz(nz)
                                ! gradient of TL across layer
                                ! interface 
    real(r_size), intent(in) :: dqwdz(nz)
                                ! gradient of QW across layer
                                ! interface 
    real(r_size), intent(in) :: dudz(nz)
                                ! Gradient of u
    real(r_size), intent(in) :: dvdz(nz)
                                ! Gradient of v
    
    real(r_size), intent(in) :: tsq(nz)
                         ! Self covariance of liquid potential temperature
                         ! (thetal'**2)
    real(r_size), intent(in) :: qsq(nz)
                         ! Self covariance of total water
                         ! (qw'**2)
    real(r_size), intent(in) :: cov(nz)
                         ! Correlation between thetal and qw
                         ! (thetal'qw') defined on theta levels K-1
    real(r_size), intent(in) :: qkw(nz)
                         !  sqrt(qke)
    real(r_size), intent(in) :: cw25(nz)
    real(r_size), intent(in) :: el(nz)
                                ! mixing length
    real(r_size), intent(in) :: z_f(nz)
    real(r_size), intent(in) :: rdz_f(nz)
    real(r_size), intent(inout) :: qdiv(nz)
                         ! factor for flux correction: sqrt(q3sq/q2sq)

    ! Intent OUT Variables
    real(r_size), intent(out) :: dfm(nz)
                                ! diffusion coefficient for momentum
    real(r_size), intent(out) :: dfh(nz)
                                ! diffusion coefficient for scalars

    real(r_size), intent(out) :: dfu_cg(nz)
                                ! counter gradient term for u
    real(r_size), intent(out) :: dfv_cg(nz)
                                ! counter gradient term for v
    real(r_size), intent(out) :: dft_cg(nz)
                                ! counter gradient term for TL
    real(r_size), intent(out) :: dfq_cg(nz)
                                ! counter gradient term for QW

    real(r_size), intent(out) :: pdk(nz)
    real(r_size), intent(out) :: pdt_tsq(nz)
    real(r_size), intent(out) :: pdt_cov(nz)
    real(r_size), intent(out) :: pdt(nz)
    real(r_size), intent(out) :: pdq_qsq(nz)
    real(r_size), intent(out) :: pdq_cov(nz)
    real(r_size), intent(out) :: pdq(nz)
    real(r_size), intent(out) :: pdc_tsq(nz)
    real(r_size), intent(out) :: pdc_qsq(nz)
    real(r_size), intent(out) :: pdc_cov(nz)
    real(r_size), intent(out) :: pdc(nz)

    real(r_size), intent(out) :: c2sq(nz)
    real(r_size), intent(out) :: smd_coef(nz)
    real(r_size), intent(out) :: gamv_coef(nz)

    ! Scalar
    integer(4) :: kz
                     ! Loop indexes

    real(r_size) :: e1
                                ! a variable denoted to E1 in the papers
    real(r_size) :: e2
                                ! a variable denoted to E2 in the papers
    real(r_size) :: e3
                                ! a variable denoted to E3 in the papers
    real(r_size) :: e4
                                ! a variable denoted to E4 in the paper
    real(r_size) :: q3sq
                                ! qke derived by level 2.5 or level 3
    real(r_size) :: t2sq
                                ! tsq derived by level 2 scheme
    real(r_size) :: r2sq
                                ! qsq derived by level 2 scheme
    real(r_size) :: t3sq
                                ! tsq derived by level 2.5 or 3 scheme
    real(r_size) :: r3sq
                                ! qsq derived by level 2.5 or 3 scheme
    real(r_size) :: c3sq
                                ! cov deribed by level 2.5 or 3 scheme
    real(r_size) :: wden
                                ! work variable
    real(r_size) :: eden
                                ! work variable
    real(r_size) :: reden
                                ! reciprocal of eden
    real(r_size) :: e6c
                                ! work variable
    real(r_size) :: coef
                                ! work variable
    real(r_size) :: elq
                                ! mixing length  times qkw appeared
                                ! in the production term of qke
    real(r_size) :: elh
                                ! mixing length times qkw appeared
                                ! in the production terms of tsq, qsq and cov.
    real(r_size) :: disp_coef
    real(r_size) :: clow
                            ! lower limit for difference between cov in level 3
                            ! and level 2
    real(r_size) :: cupp
                  ! upper limit for difference between cov in level 3
                  ! and level 2


    real(r_size) :: elsq
                                ! square of mixing length
    real(r_size) :: gmel
                                ! GM times the mixing length
    real(r_size) :: ghel
                                ! GH times the mixing length
    real(r_size) :: pd_res


    real(r_size) :: gamt(nz)
                             ! counter gradient term for flux of TL
                             ! gamt = gamt_tsq * tsq + gamt_cov * cov + gamt_res
    real(r_size) :: gamt_tsq(nz)
                             ! a linear part to tsq in gamt
    real(r_size) :: gamt_cov(nz)
                             ! a linear part to cov in gamt
    real(r_size) :: gamt_res(nz)
                             ! a residual part in gamt
    real(r_size) :: gamt_factor(nz)
                             ! stability factor for gamt
    real(r_size) :: gamq(nz)
                             ! counter gradient term for flux of QW
                             ! gamq = gamq_qsq * qsq + gamq_cov * cov + gamq_res
    real(r_size) :: gamq_qsq(nz)
                             ! a linear part to qsq in gamq
    real(r_size) :: gamq_cov(nz)
                             ! a linear part to cov in gamq
    real(r_size) :: gamq_res(nz)
                             ! a residual part in gamq
    real(r_size) :: gamq_factor(nz)
                             ! stability factor for gamq
    real(r_size) :: pdc_factor(nz)
                             ! stability factor for pdc
    real(r_size) :: smd(nz)
                             ! counter gradient correction for SM


    do kz = 1, nz
      elsq = el(kz) ** 2
      gmel = gm(kz) * elsq
      ghel = gh(kz) * elsq
      q3sq = qkw(kz) ** 2

      t2sq = qdiv(kz) * b2 * elsq * sh(kz) * dtldz(kz) ** 2
      r2sq = qdiv(kz) * b2 * elsq * sh(kz) * dqwdz(kz) ** 2
      c2sq(kz) = qdiv(kz) * b2 * elsq            &
        * sh(kz) * dtldz(kz) * dqwdz(kz)
      t3sq = max(tsq(kz), 0.0_r_size)
      r3sq = max(qsq(kz), 0.0_r_size)
      c3sq = cov(kz)

      c3sq = sign( min( abs(c3sq), sqrt(t3sq*r3sq) ), c3sq )

      t2sq = vt(kz) * t2sq + vq(kz) * c2sq(kz)
      r2sq = vt(kz) * c2sq(kz) + vq(kz) * r2sq
      c2sq(kz) = max(vt(kz) * t2sq + vq(kz) * r2sq,  0.0_r_size)
      t3sq = vt(kz) * t3sq + vq(kz) * c3sq
      r3sq = vt(kz) * c3sq + vq(kz) * r3sq
      c3sq = max(vt(kz) * t3sq + vq(kz) * r3sq, 0.0_r_size)

      !  Limitation on q, instead of L/q
      if ( q3sq < -gh(kz) * elsq) then
        q3sq = -elsq * gh(kz)
      end if

      ! Limitation on c3sq (0.12 =< cw =< 0.76)
      ! e2 = q^2 * phi2'
      e2   = q3sq - e2c*ghel * qdiv(kz)**2
      ! e3 = q^2 * phi3'
      e3   = q3sq + e3c*ghel * qdiv(kz)**2
      ! e4 = q^2 * phi4'
      e4   = q3sq - e4c*ghel * qdiv(kz)**2
      ! eden = q^4 D'
      eden = e2 * e4 + e3 *e5c*gmel * qdiv(kz)**2


      ! wden = numerator in the square braket in (10a) in NN2006
      !        times (1-c3) * (g/thetav)**2 * GH
      wden = cc3*gtr(kz) **2                                   &
        * elsq ** 2 / elsq                   &
        * qdiv(kz)**2                                   &
        *( e2*e4c                                   &
        - e3c*e5c*gmel * qdiv(kz)**2 )

      if ( wden /= 0.0 ) then
        clow = q3sq * ( 0.12-cw25(kz) )*eden/wden
        cupp = q3sq *( 0.76-cw25(kz) )*eden/wden

        if ( wden > 0.0 ) then
          c3sq  = min( max( c3sq, c2sq(kz) + clow), c2sq(kz) + cupp)
        else
          c3sq  = max( min( c3sq, c2sq(kz) + clow), c2sq(kz) + cupp)
        end if
      end if

      e1   = e2 + e5c*gmel * qdiv(kz) ** 2
      eden = max( eden, 1.0e-20_r_size )
      reden = 1.0 / eden

      e6c  = 3.0 * a2 *cc3 * gtr(kz) * elsq / elsq

      ! Calculate each term in  Gamma_theta
      coef = - e1 * qdiv(kz) * e6c * reden
      gamt_tsq(kz) = coef * vt(kz)
      gamt_cov(kz) = coef * vq(kz)
      gamt_res(kz) = - coef * t2sq

      ! Calculate each term in  Gamma_q
      gamq_qsq(kz) = coef * vq(kz)
      gamq_cov(kz) = coef * vt(kz)
      gamq_res(kz) = - coef * r2sq

      ! for Sm' and Sh'd(Theta_V)/dz
      smd_coef(kz)  = elsq * qdiv(kz) * e6c      &
        * gtr(kz) * reden * qdiv(kz) ** 2              &
        * (e3c + e4c) * a1_2
      gamv_coef(kz) = e1 * qdiv(kz) * e6c * gtr(kz) * reden
      smd(kz) = smd_coef(kz) * (c3sq - c2sq(kz))
      !gamv(kz) = gamv_coef(kz) * (c3sq - c2sq(kz))

      ! For elh (see below), qdiv in Level 3 is reset to 1.0.
      qdiv(kz) = 1.0

      ! Calculate diffusion coefficients
      elq = el(kz) * qkw(kz)
      dfm(kz) = elq * sm(kz)
      dfh(kz) = elq * sh(kz)


    end do

    ! Adjustment for Gamma_theta and Gamma_q
    ! After the adjustment, Gamma_theta and Gamma_q are calculated
    if (l_my_prod_adj .and.                                             &
      (imp_mode == half_impl .or. imp_mode == full_impl)) then
      do kz = 1, nz

        elq = el(kz) * qkw(kz)
        elh = elq * qdiv(kz)
        disp_coef = qkw(kz) / (b2 * el(kz))               &
          + 0.5 * coef_trbvar_diff * dfm(kz)            &
          * (2.0 * pi * my_prod_adj_fact               &
          * rdz_f(kz)) ** 2

        pdt_tsq(kz) = elh * gamt_tsq(kz) * dtldz(kz)
        if (disp_coef < pdt_tsq(kz)) then
          gamt_factor(kz) = disp_coef / pdt_tsq(kz)
        else
          gamt_factor(kz) = 1.0
        end if

        pdq_qsq(kz) = elh * gamq_qsq(kz) * dqwdz(kz)
        if (disp_coef < pdq_qsq(kz)) then
          gamq_factor(kz) = disp_coef / pdq_qsq(kz)
        else
          gamq_factor(kz) = 1.0
        end if

        gamt_tsq(kz) = gamt_factor(kz) * gamt_tsq(kz)
        gamt_cov(kz) = gamt_factor(kz) * gamt_cov(kz)
        gamt_res(kz) = gamt_factor(kz) * gamt_res(kz)

        gamq_qsq(kz) = gamq_factor(kz) * gamq_qsq(kz)
        gamq_cov(kz) = gamq_factor(kz) * gamq_cov(kz)
        gamq_res(kz) = gamq_factor(kz) * gamq_res(kz)

        pdc_cov(kz) = elh                                      &
          * (gamt_cov(kz) * dqwdz(kz)         &
          + gamq_cov(kz) * dtldz(kz)) * 0.5
        if (disp_coef < pdc_cov(kz)) then
          pdc_factor(kz) = disp_coef / pdc_cov(kz)
        else
          pdc_factor(kz) = 1.0
        end if
        gamt_tsq(kz) = pdc_factor(kz) * gamt_tsq(kz)
        gamt_cov(kz) = pdc_factor(kz) * gamt_cov(kz)
        gamt_res(kz) = pdc_factor(kz) * gamt_res(kz)

        gamq_qsq(kz) = pdc_factor(kz) * gamq_qsq(kz)
        gamq_cov(kz) = pdc_factor(kz) * gamq_cov(kz)
        gamq_res(kz) = pdc_factor(kz) * gamq_res(kz)

        gamt(kz) = gamt_tsq(kz) * tsq(kz)            &
          + gamt_cov(kz) * cov(kz)            &
          + gamt_res(kz)

        gamq(kz) = gamq_qsq(kz) * qsq(kz)            &
          + gamq_cov(kz) * cov(kz)            &
          + gamq_res(kz)

      end do
    else
      do kz = 1, nz
        gamt(kz) = gamt_tsq(kz) * tsq(kz)            &
          + gamt_cov(kz) * cov(kz)            &
          + gamt_res(kz)

        gamq(kz) = gamq_qsq(kz) * qsq(kz)            &
          + gamq_cov(kz) * cov(kz)            &
          + gamq_res(kz)

        gamt_factor(kz) = 1.0
        gamq_factor(kz) = 1.0
        pdc_factor(kz) = 1.0
      end do
    end if ! IF L_MY_PROD_ADJ

    ! Calculate production terms
    do kz = 1, nz
      elq = el(kz) * qkw(kz)
      elh = elq * qdiv(kz)

      pdk(kz) = elq * (sm(kz) * gm(kz) + sh(kz) * gh(kz))          &
        + wb_ng(kz)

      ! for tsq
      pd_res = elh * (sh(kz) * dtldz(kz) + gamt_res(kz))  &
        * dtldz(kz)
      pdt_tsq(kz) = elh * gamt_tsq(kz) * dtldz(kz)
      pdt_cov(kz) = elh * gamt_cov(kz) * dtldz(kz)

      pdt(kz) = pdt_tsq(kz) * tsq(kz) + pdt_cov(kz) * cov(kz) &
        &     + pd_res

      ! for qsq
      pd_res = elh * (sh(kz) * dqwdz(kz) + gamq_res(kz))     &
        * dqwdz(kz)
      pdq_qsq(kz) = elh * gamq_qsq(kz) * dqwdz(kz)
      pdq_cov(kz) = elh * gamq_cov(kz) * dqwdz(kz)

      pdq(kz) = pdq_qsq(kz) * qsq(kz) + pdq_cov(kz) * cov(kz) &
        &         + pd_res

      ! for cov
      pd_res = 0.5 * elh * ((sh(kz) * dtldz(kz)                   &
        + gamt_res(kz)) * dqwdz(kz)         &
        + (sh(kz) * dqwdz(kz)                 &
        + gamq_res(kz)) * dtldz(kz))

      pdc_tsq(kz) = elh * gamt_tsq(kz) * dqwdz(kz) * 0.5
      pdc_qsq(kz) = elh * gamq_qsq(kz) * dtldz(kz) * 0.5
      pdc_cov(kz) = 0.5 * elh * (gamt_cov(kz) * dqwdz(kz)        &
        + gamq_cov(kz) * dtldz(kz))

      pdc(kz) = pdc_tsq(kz) * tsq(kz) + pdc_qsq(kz) * qsq(kz) &
        &       + pdc_cov(kz) * cov(kz) + pd_res

      dfu_cg(kz) = elq * smd(kz) * dudz(kz)
      dfv_cg(kz) = elq * smd(kz) * dvdz(kz)
      dft_cg(kz) = elq * gamt(kz)
      dfq_cg(kz) = elq * gamq(kz)
    end do

    call pbl_mym_surf_prod(r_mosurf, u_s, ftl_surf, fqw_surf, z_f(1), &
      & pdk(1), pdt(1), pdq(1), pdc(1))

    return
  end subroutine pbl_mym_turbulence_level3
  !
  subroutine pbl_mym_turbulence_level25(&
    & r_mosurf, u_s, ftl_surf, fqw_surf, &
    & qkw, qdiv, &
    & sm, sh, gm, gh, wb_ng, dtldz, dqwdz, el, z_f, &
    & dfm, dfh, pdk, pdt, pdq, pdc)
    use pp_vardef
    use pbl_grid, only: nz
    implicit none

    real(r_size), intent(in) :: r_mosurf
    real(r_size), intent(in) :: u_s
    real(r_size), intent(in) :: ftl_surf
    real(r_size), intent(in) :: fqw_surf

    real(r_size), intent(in) :: qkw(nz)
    real(r_size), intent(in) :: qdiv(nz)
    real(r_size), intent(in) :: sm(nz)
    real(r_size), intent(in) :: sh(nz)
    real(r_size), intent(in) :: gm(nz)
    real(r_size), intent(in) :: gh(nz)
    real(r_size), intent(in) :: wb_ng(nz)
    real(r_size), intent(in) :: dtldz(nz)
    real(r_size), intent(in) :: dqwdz(nz)
    real(r_size), intent(in) :: el(nz)
    real(r_size), intent(in) :: z_f(nz)

    real(r_size), intent(out) :: pdk(nz)
    real(r_size), intent(out) :: pdt(nz)
    real(r_size), intent(out) :: pdq(nz)
    real(r_size), intent(out) :: pdc(nz)
    real(r_size), intent(out) :: dfm(nz)
    real(r_size), intent(out) :: dfh(nz)

    integer(4) :: kz
    real(r_size) :: elq
    real(r_size) :: elh

    do kz = 1, nz
      !     In Level 2.5, qdiv is not reset.

      elq = el(kz) * qkw(kz)
      elh = elq * qdiv(kz)

      pdk(kz) = elq * (sm(kz) * gm(kz)                      &
        + sh(kz) * gh(kz)) + wb_ng(kz)
      pdt(kz) = elh * (sh(kz) * dtldz(kz)) * dtldz(kz)
      pdq(kz) = elh * (sh(kz) * dqwdz(kz)) * dqwdz(kz)
      pdc(kz) = elh * (sh(kz) * dtldz(kz)) * dqwdz(kz) * 0.5  &
        + elh * (sh(kz) * dqwdz(kz)) * dtldz(kz) * 0.5

      dfm(kz) = elq * sm(kz)
      dfh(kz) = elq * sh(kz)
    end do

    call pbl_mym_surf_prod(r_mosurf, u_s, ftl_surf, fqw_surf, z_f(1), &
      & pdk(1), pdt(1), pdq(1), pdc(1))

    return
  end subroutine pbl_mym_turbulence_level25


  !
  subroutine pbl_mym_tend_cov(&
    & f2h_m, f2h_p, rdz_f, rdz_h, &
    & qkw, el, dfm, &
    & pdt_tsq, pdt_cov, pdt, &
    & pdq_qsq, pdq_cov, pdq, &
    & pdc_tsq, pdc_qsq, pdc_cov, pdc, &
    & tsq, qsq, cov, &
    & tend_tsq, tend_qsq, tend_cov)
    use pp_vardef
    use pbl_grid, only: nz
    use pbl_mym_const, only: b2, coef_trbvar_diff
    use pbl_mym_option, only: imp_mode
    use pbl_mym_option_symbol, only: full_impl, half_impl, expl
    use pbl_mym_simeq, only: pbl_mym_simeq_cov, pbl_mym_simeq_tend

    implicit none

    real(r_size), intent(in) :: f2h_m(nz-1)
    real(r_size), intent(in) :: f2h_p(nz-1)
    real(r_size), intent(in) :: rdz_f(nz)
    real(r_size), intent(in) :: rdz_h(nz-1)

    real(r_size), intent(in) :: qkw(nz)
    real(r_size), intent(in) :: el(nz)
    real(r_size), intent(in) :: dfm(nz)

    real(r_size), intent(in) :: pdt_tsq(nz)
    real(r_size), intent(in) :: pdt_cov(nz)
    real(r_size), intent(in) :: pdt(nz)
    real(r_size), intent(in) :: pdq_qsq(nz)
    real(r_size), intent(in) :: pdq_cov(nz)
    real(r_size), intent(in) :: pdq(nz)
    real(r_size), intent(in) :: pdc_tsq(nz)
    real(r_size), intent(in) :: pdc_qsq(nz)
    real(r_size), intent(in) :: pdc_cov(nz)
    real(r_size), intent(in) :: pdc(nz)

    real(r_size), intent(in) :: tsq(nz)
    real(r_size), intent(in) :: qsq(nz)
    real(r_size), intent(in) :: cov(nz)

    real(r_size), intent(out) :: tend_tsq(nz)
    real(r_size), intent(out) :: tend_qsq(nz)
    real(r_size), intent(out) :: tend_cov(nz)

    integer(4) :: kz

    real(r_size) :: b2l
    real(r_size) :: bp_tsq(nz)
    real(r_size) :: rp_tsq(nz)
    real(r_size) :: bp_qsq(nz)
    real(r_size) :: rp_qsq(nz)
    real(r_size) :: bp_cov(nz)
    real(r_size) :: rp_cov(nz)

    ! Integrate the covariances
    if (imp_mode == full_impl) then
      call pbl_mym_simeq_cov(                                       &
        qkw, el, dfm,       &
        f2h_m, f2h_p, rdz_f, rdz_h,                                   &
        pdt_tsq, pdt_cov, pdt,                                      &
        pdq_qsq, pdq_cov, pdq,                                      &
        pdc_cov, pdc_tsq, pdc_qsq, pdc,                             &
        tsq, qsq, cov, &
        tend_tsq, tend_qsq, tend_cov)
    else   ! half implict or explicit
      if (imp_mode == half_impl) then
        do kz = 1, nz
          b2l = 2.0 * qkw(kz) / (b2 * el(kz))

          bp_tsq(kz) = b2l - 2.0 * pdt_tsq(kz)
          rp_tsq(kz) = 2.0 * pdt(kz)

          bp_qsq(kz) = b2l - 2.0 * pdq_qsq(kz)
          rp_qsq(kz) = 2.0 * pdq(kz)

          bp_cov(kz) = b2l - 2.0 * pdc_cov(kz)
          rp_cov(kz) = 2.0 * pdc(kz)
        end do
      else if (imp_mode == expl) then
        do kz = 1, nz
          b2l = 2.0 * qkw(kz) / (b2 * el(kz))

          bp_tsq(kz) = b2l
          rp_tsq(kz) = 2.0 * pdt(kz)

          bp_qsq(kz) = b2l
          rp_qsq(kz) = 2.0 * pdq(kz)

          bp_cov(kz) = b2l
          rp_cov(kz) = 2.0 * pdc(kz)
        end do
      end if

      call pbl_mym_simeq_tend(                                   &
        & coef_trbvar_diff, dfm, rp_tsq, bp_tsq, tsq, &
        & f2h_m, f2h_p, rdz_f, rdz_h, tend_tsq)

      call pbl_mym_simeq_tend(                                   &
        & coef_trbvar_diff, dfm, rp_qsq, bp_qsq, qsq, &
        & f2h_m, f2h_p, rdz_f, rdz_h, tend_qsq)

      call pbl_mym_simeq_tend(                                   &
        & coef_trbvar_diff, dfm, rp_cov, bp_cov, cov, &
        & f2h_m, f2h_p, rdz_f, rdz_h, tend_cov)

    end if  ! if imp_mode == FULL_IMPL
    return
  end subroutine pbl_mym_tend_cov

  subroutine pbl_mym_diagnose_cov(el, pdt, pdq, pdc, qkw, tsq, qsq, cov)
    use pp_vardef
    use pbl_grid, only: nz
    use pbl_mym_const, only: b2
    implicit none

    real(r_size), intent(in) :: el(nz)
    real(r_size), intent(in) :: pdt(nz)
    real(r_size), intent(in) :: pdq(nz)
    real(r_size), intent(in) :: pdc(nz)
    real(r_size), intent(in) :: qkw(nz)

    real(r_size), intent(out) :: tsq(nz)
    real(r_size), intent(out) :: qsq(nz)
    real(r_size), intent(out) :: cov(nz)
    
    integer(4) :: kz
    real(r_size) :: b2l
    
    ! In level 2.5, tsq, qsq, cov are diagnosed assuming balance between
    ! prodcution and dissipation.
    do kz = 1, nz
      if (qkw(kz) <= 1.0e-4)  then
        b2l = 0.0
      else
        b2l = b2 * el(kz) / qkw(kz)
      end if
      tsq(kz) = b2l * 2.0 * pdt(kz)
      qsq(kz) = b2l * 2.0 * pdq(kz)
      cov(kz) = b2l * 2.0 * pdc(kz)
    end do
    return
  end subroutine pbl_mym_diagnose_cov

  subroutine pbl_mym_tend_tke_level3(qke, qkw, tsq, qsq, cov, dfm, &
    & tend_tsq, tend_qsq, tend_cov, el, vt, vq, &
    & c2sq, gm, smd_coef, gamv_coef, &
    & f2h_m, f2h_p, rdz_f, rdz_h, pdk, tend_qke)
    use pp_vardef
    use pbl_grid, only: nz
    use pbl_mym_option, only: imp_mode, my_lowest_pd_surf
    use pbl_mym_option_symbol, only: half_impl, full_impl, expl
    use pbl_mym_const, only:  coef_trbvar_diff_tke, b1
    use pbl_const, only: timestep
    use pbl_mym_simeq, only: pbl_mym_simeq_tend

    implicit none

    real(r_size), intent(in) :: qke(nz)
    real(r_size), intent(in) :: qkw(nz)
    real(r_size), intent(in) :: tsq(nz)
    real(r_size), intent(in) :: qsq(nz)
    real(r_size), intent(in) :: cov(nz)
    real(r_size), intent(in) :: dfm(nz)
    real(r_size), intent(in) :: tend_tsq(nz)
    real(r_size), intent(in) :: tend_qsq(nz)
    real(r_size), intent(in) :: tend_cov(nz)
    real(r_size), intent(in) :: el(nz)
    real(r_size), intent(in) :: vt(nz)
    real(r_size), intent(in) :: vq(nz)
    real(r_size), intent(in) :: c2sq(nz)
    real(r_size), intent(in) :: gm(nz)
    real(r_size), intent(in) :: smd_coef(nz)
    real(r_size), intent(in) :: gamv_coef(nz)
    real(r_size), intent(in) :: f2h_m(nz-1)
    real(r_size), intent(in) :: f2h_p(nz-1)
    real(r_size), intent(in) :: rdz_f(nz)
    real(r_size), intent(in) :: rdz_h(nz-1)

    real(r_size), intent(inout) :: pdk(nz)
    real(r_size), intent(out) :: tend_qke(nz)

    integer(4) :: kz
    integer(4) :: k_start_cor

    real(r_size) :: t3sq
    real(r_size) :: r3sq
    real(r_size) :: c3sq
    real(r_size) :: elq
    real(r_size) :: smd
    real(r_size) :: b1l

    real(r_size) :: rp(nz)
    real(r_size) :: bp(nz)

    ! predict qke
    if (my_lowest_pd_surf > 0) then
      k_start_cor = 2
    else
      k_start_cor = 1
    end if

    if (imp_mode == half_impl .or. imp_mode == full_impl) then
      ! add correction terms evaluated with integrated tsq, qsq and cov
      do kz = k_start_cor, nz
        t3sq = max(tsq(kz) + tend_tsq(kz) * timestep, 0.0_r_size)
        r3sq = max(qsq(kz) + tend_qsq(kz) * timestep, 0.0_r_size)
        c3sq = cov(kz) + tend_cov(kz) * timestep

        c3sq = sign( min( abs(c3sq), sqrt(t3sq*r3sq) ), c3sq )

        t3sq = vt(kz) * t3sq + vq(kz) * c3sq
        r3sq = vt(kz) * c3sq + vq(kz) * r3sq
        c3sq = max(vt(kz) * t3sq + vq(kz) * r3sq, 0.0_r_size)

        elq = el(kz) * qkw(kz)
        smd = smd_coef(kz) * (c3sq - c2sq(kz))

        pdk(kz) = pdk(kz) + elq * (smd * gm(kz)                          &
                          + gamv_coef(kz) * (c3sq- c2sq(kz)))
      end do
    else
      do kz = k_start_cor, nz
        t3sq = max(tsq(kz), 0.0_r_size)
        r3sq = max(qsq(kz), 0.0_r_size)
        c3sq = cov(kz)

        c3sq = sign( min( abs(c3sq), sqrt(t3sq*r3sq) ), c3sq )

        t3sq = vt(kz) * t3sq + vq(kz) * c3sq
        r3sq = vt(kz) * c3sq + vq(kz) * r3sq
        c3sq = max(vt(kz) * t3sq + vq(kz) * r3sq, 0.0_r_size)

        elq = el(kz) * qkw(kz)
        smd = smd_coef(kz) * (c3sq - c2sq(kz))

        pdk(kz) = pdk(kz) + elq * (smd * gm(kz)                          &
                          + gamv_coef(kz) * (c3sq- c2sq(kz)))
      end do
    end if ! if test full_impl or half_impl

    do kz = 1, nz
      b1l = b1 * el(kz)
      bp(kz) = 2.0 * qkw(kz) / b1l
      rp(kz) = 2.0 * pdk(kz)
    end do

    call pbl_mym_simeq_tend(                                   &
      & coef_trbvar_diff_tke, dfm, rp, bp, qke, &
      & f2h_m, f2h_p, rdz_f, rdz_h, tend_qke)
    return
  end subroutine pbl_mym_tend_tke_level3
!
  subroutine pbl_mym_tend_tke_level25(qke, qkw, dfm, el, &
    & f2h_m, f2h_p, rdz_f, rdz_h, pdk, tend_qke)
    use pp_vardef
    use pbl_grid, only: nz
    use pbl_mym_const, only:  coef_trbvar_diff_tke, b1
    use pbl_mym_simeq, only: pbl_mym_simeq_tend

    implicit none

    real(r_size), intent(in) :: qke(nz)
    real(r_size), intent(in) :: qkw(nz)
    real(r_size), intent(in) :: dfm(nz)
    real(r_size), intent(in) :: el(nz)

    real(r_size), intent(in) :: f2h_m(nz-1)
    real(r_size), intent(in) :: f2h_p(nz-1)
    real(r_size), intent(in) :: rdz_f(nz)
    real(r_size), intent(in) :: rdz_h(nz-1)

    real(r_size), intent(in) :: pdk(nz)
    real(r_size), intent(out) :: tend_qke(nz)

    integer(4) :: kz
    
    real(r_size) :: b1l
    real(r_size) :: rp(nz)
    real(r_size) :: bp(nz)

    ! predict qke
    do kz = 1, nz
      b1l = b1 * el(kz)
      bp(kz) = 2.0 * qkw(kz) / b1l
      rp(kz) = 2.0 * pdk(kz)
    end do

    call pbl_mym_simeq_tend(                                   &
      & coef_trbvar_diff_tke, dfm, rp, bp, qke, &
      & f2h_m, f2h_p, rdz_f, rdz_h, tend_qke)
    return
  end subroutine pbl_mym_tend_tke_level25
  !
  subroutine pbl_mym_diagnose_tke(pdk, el, qke)
    use pp_vardef
    use pbl_grid, only: nz
    use pbl_mym_const, only: b2
    use pp_phys_const, only: two_thirds
    implicit none

    real(r_size), intent(in) :: pdk(nz)
    real(r_size), intent(in) :: el(nz)
    real(r_size), intent(out) :: qke(nz)

    integer(4) :: kz
    real(r_size) :: b2l

    ! level 2
    ! diagnose qke
    do kz = 1, nz
      b2l = b2 * el(kz)
      qke(kz) = (max(b2l * 2.0_r_size * pdk(kz), 0.0_r_size)) ** two_thirds
    end do
    return
  end subroutine pbl_mym_diagnose_tke
  !
  subroutine pbl_mym_prepare(&
    & uvel, vvel, pt, qv, qc, qci, prs, qke, &
    & rdz_f, rdz_h, f2h_m, f2h_p, &
    & qkw, gtr, temp, tl, qw, &
    & dtldz, dqwdz, dudz, dvdz, dvdzm)
    use pp_vardef
    use pbl_grid, only: nz
    use pp_phys_const, only: grav, c_virtual, rdvcp, hlatnt, hls, r_cp
    use pbl_const, only: pref

    implicit none

    real(r_size), intent(in) :: uvel(nz)
    real(r_size), intent(in) :: vvel(nz)
    real(r_size), intent(in) :: pt(nz)
    real(r_size), intent(in) :: qv(nz)
    real(r_size), intent(in) :: qc(nz)
    real(r_size), intent(in) :: qci(nz)
    real(r_size), intent(in) :: prs(nz)

    real(r_size), intent(in) :: qke(nz)

    real(r_size), intent(in) :: rdz_f(nz)
    real(r_size), intent(in) :: rdz_h(nz-1)
    real(r_size), intent(in) :: f2h_m(nz-1)
    real(r_size), intent(in) :: f2h_p(nz-1)

    real(r_size), intent(out) :: qkw(nz)
    real(r_size), intent(out) :: gtr(nz)
    real(r_size), intent(out) :: temp(nz)
    real(r_size), intent(out) :: tl(nz)
    real(r_size), intent(out) :: qw(nz)
    real(r_size), intent(out) :: dtldz(nz)
    real(r_size), intent(out) :: dqwdz(nz)
    real(r_size), intent(out) :: dudz(nz)
    real(r_size), intent(out) :: dvdz(nz)
    real(r_size), intent(out) :: dvdzm(nz)

    integer(4) :: kz
    real(r_size) :: ptv
    real(r_size) :: r_exner
    real(r_size) :: tl_h(nz)
    real(r_size) :: qw_h(nz)
    real(r_size) :: u_h(nz)
    real(r_size) :: v_h(nz)

    do kz = 1, nz
      qkw(kz) = sqrt(max(qke(kz), 1.e-20_r_size))
      ptv = pt(kz) * (1.0 + c_virtual * qv(kz) - qc(kz) - qci(kz))
      gtr(kz) = grav / ptv

      r_exner = (pref / prs(kz)) ** rdvcp
      temp(kz) = pt(kz) / r_exner
      tl(kz) = pt(kz) - r_cp * r_exner * (hlatnt * qc(kz) + hls * qci(kz))
      qw(kz) = qv(kz) + qc(kz) + qci(kz)

    end do

    do kz = 1, nz - 1
      tl_h(kz) = f2h_m(kz) * tl(kz) + f2h_p(kz) * tl(kz + 1) 
      qw_h(kz) = f2h_m(kz) * qw(kz) + f2h_p(kz) * qw(kz + 1) 
      u_h(kz) = f2h_m(kz) * uvel(kz) + f2h_p(kz) * uvel(kz + 1) 
      v_h(kz) = f2h_m(kz) * vvel(kz) + f2h_p(kz) * vvel(kz + 1) 

    end do

    kz = 1
    dtldz(kz) = (tl(kz + 1) - tl(kz)) * rdz_h(kz)
    dqwdz(kz) = (qw(kz + 1) - qw(kz)) * rdz_h(kz)
    dudz(kz) = u_h(kz) * rdz_f(kz) 
    dvdz(kz) = v_h(kz) * rdz_f(kz)
    dvdzm(kz) = sqrt(dudz(kz) ** 2 + dvdz(kz) ** 2)

    do kz = 2, nz - 1
      dtldz(kz) = (tl_h(kz) - tl_h(kz - 1)) * rdz_f(kz)
      dqwdz(kz) = (qw_h(kz) - qw_h(kz - 1)) * rdz_f(kz)
      dudz(kz) = (u_h(kz) - u_h(kz - 1)) * rdz_f(kz)
      dvdz(kz) = (v_h(kz) - v_h(kz - 1)) * rdz_f(kz)

      dvdzm(kz) = sqrt(dudz(kz) ** 2 + dvdz(kz) ** 2)
    end do

    kz = nz
    dtldz(kz) = dtldz(kz - 1)
    dqwdz(kz) = dqwdz(kz - 1)
    dudz(kz) = dudz(kz - 1)
    dvdz(kz) = dvdz(kz - 1)
    dvdzm(kz) = sqrt(dudz(kz) ** 2 + dvdz(kz) ** 2)

    return

  end subroutine pbl_mym_prepare
  !
  subroutine pbl_mym_buoy( &
    & ftl_surf, fqw_surf, ustar, p_surf, &
    & z_f, u, v, temp, pt, qv, qc, qci, tl, qw, prs, exner, &
    & dtldz, dqwdz, &
    & tsq, qsq, cov, gtr, &
    & fb_surf, vt, vq, cld, ql, dbdz, wb_ng)
    use pp_vardef
    use pbl_grid, only: nz
    use pbl_mym_condensation, only: pbl_mym_condensation_run
    use pbl_shcu, only: pbl_shcu_run
    use pbl_mym_option, only: l_shcu_buoy
    implicit none

    real(r_size), intent(in) :: ftl_surf
    real(r_size), intent(in) :: fqw_surf
    real(r_size), intent(in) :: ustar
    real(r_size), intent(in) :: p_surf

    real(r_size), intent(in) :: z_f(nz)

    real(r_size), intent(in) :: u(nz)
    real(r_size), intent(in) :: v(nz)
    real(r_size), intent(in) :: temp(nz)
    real(r_size), intent(in) :: pt(nz)
    real(r_size), intent(in) :: qv(nz)
    real(r_size), intent(in) :: qc(nz)
    real(r_size), intent(in) :: qci(nz)

    real(r_size), intent(in) :: tl(nz)
    real(r_size), intent(in) :: qw(nz)
    real(r_size), intent(in) :: prs(nz)
    real(r_size), intent(in) :: exner(nz)

    real(r_size), intent(in) :: dtldz(nz)
    real(r_size), intent(in) :: dqwdz(nz)

    real(r_size), intent(in) :: tsq(nz)
    real(r_size), intent(in) :: qsq(nz)
    real(r_size), intent(in) :: cov(nz)
    real(r_size), intent(in) :: gtr(nz)

    real(r_size), intent(out) :: fb_surf
    real(r_size), intent(out) :: vt(nz)
    real(r_size), intent(out) :: vq(nz)
    real(r_size), intent(out) :: cld(nz)
    real(r_size), intent(out) :: ql(nz)
    real(r_size), intent(out) :: dbdz(nz)
    real(r_size), intent(out) :: wb_ng(nz)

    integer(4) :: kz

    real(r_size) :: zhpar

    real(r_size) :: q1(nz)
    real(r_size) :: cld_gauss(nz)

    call pbl_mym_condensation_run(temp, tl, qw, prs, tsq, qsq, cov, &
      & vt, vq, cld_gauss, ql, q1)

    fb_surf = gtr(1) * (vt(1) * ftl_surf + vq(1) * fqw_surf)
    
    do kz = 1, nz
      dbdz(kz) = gtr(kz) * (vt(kz) * dtldz(kz) + vq(kz) * dqwdz(kz))
    end do


    if (l_shcu_buoy) then
      zhpar = 3000.0_r_size   ! tentative
      call pbl_shcu_run(                                             &
                     fb_surf, ustar, p_surf,                          &
                     z_f, u, v, temp, pt, exner, prs,                 &
                     qv, qc, qci, q1, cld_gauss,                      &
                     zhpar,                                           &
                     cld, wb_ng)
    else
      do kz = 1, nz
        wb_ng(kz) = 0.0_r_size
        cld(kz) = cld_gauss(kz)
      end do
    end if

  end subroutine pbl_mym_buoy
  !
end module pbl_mym





