module rad_flux_wrapper


  ! USE statements
  !
  use vtype_module

  implicit none

  private


  ! 公開変数
  ! Public variables
  !
  logical, save :: rad_flux_wrapper_inited = .false.
                              ! 初期設定フラグ.
                              ! Initialization flag

  public :: RadFluxWrapper
  public :: RadFluxWrapperInit
  public :: RadFluxWrapperFinalize


  character(*), parameter:: module_name = 'rad_flux_wrapper'
                              ! モジュールの名称.
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: rad_Earth_V2.f90,v 1.9 2015/01/29 12:06:43 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

  !--------------------------------------------------------------------------------------

contains

  !--------------------------------------------------------------------------------------

  subroutine RadFluxWrapper(                      &
    & imax, jmax, kmax,                           & ! (in)
    & NMolNum, m_MolNum,                          & ! (in)
    & xy_AlbedoSW, xy_AlbedoLW,                   & ! (in)
    & StrFluxTOA, xy_SZADeg,                      & ! (in)
    & xyr_Press, xyr_Temp, xy_SurfTemp,           & ! (in)
    & xyrm_VMR, xyr_MeanMolWt,                    & ! (in)
    & NPtcl, a_PtclName,                          & ! (in)
    & xyza_PtclEffRad, xyza_DelPtclNum,           & ! (in)
    & xyr_RadUwFlux, xyr_RadDwFlux,               & ! (out)
    & xyra_DelRadUwFlux, xyra_DelRadDwFlux,       & ! (out)
    & xyr_RadUwFluxLW, xyr_RadDwFluxLW,           & ! (out)
    & xyr_RadUwFluxSW, xyr_RadDwFluxSW,           & ! (out)
    & xyra_DelRadUwFluxLW, xyra_DelRadDwFluxLW,   & ! (out)
    & xyra_DelRadUwFluxSW, xyra_DelRadDwFluxSW    & ! (out)
    & )


    ! USE statements
    !

    ! OpenMP
    !
    !$ use omp_lib


    use rad_flux, only : RadFluxCalcFlux


    integer , intent(in ) :: imax
    integer , intent(in ) :: jmax
    integer , intent(in ) :: kmax

    integer , intent(in ) :: NMolNum
    integer , intent(in ) :: m_MolNum(NMolNum)

    real(DP), intent(in ) :: xy_AlbedoSW     (1:imax, 1:jmax)
    real(DP), intent(in ) :: xy_AlbedoLW     (1:imax, 1:jmax)
    real(DP), intent(in ) :: StrFluxTOA
    real(DP), intent(in ) :: xy_SZADeg       (1:imax, 1:jmax)
    real(DP), intent(in ) :: xyr_Press       (1:imax, 1:jmax, 0:kmax)
    real(DP), intent(in ) :: xyr_Temp        (1:imax, 1:jmax, 0:kmax)
    real(DP), intent(in ) :: xy_SurfTemp     (1:imax, 1:jmax)
    real(DP), intent(in ) :: xyrm_VMR        (1:imax, 1:jmax, 0:kmax, 1:NMolNum)
    real(DP), intent(in ) :: xyr_MeanMolWt   (1:imax, 1:jmax, 0:kmax)

    integer          , intent(in ) :: NPtcl
    character(stdstr), intent(in ) :: a_PtclName   (NPtcl)
    real(DP)         , intent(in ) :: xyza_PtclEffRad(1:imax, 1:jmax, 1:kmax, 1:NPtcl)
    real(DP)         , intent(in ) :: xyza_DelPtclNum(1:imax, 1:jmax, 1:kmax, 1:NPtcl)

    real(DP), intent(out) :: xyr_RadUwFlux      (1:imax, 1:jmax, 0:kmax)
    real(DP), intent(out) :: xyr_RadDwFlux      (1:imax, 1:jmax, 0:kmax)
    real(DP), intent(out) :: xyra_DelRadUwFlux  (1:imax, 1:jmax, 0:kmax, -1:1)
    real(DP), intent(out) :: xyra_DelRadDwFlux  (1:imax, 1:jmax, 0:kmax, -1:1)
    real(DP), intent(out) :: xyr_RadUwFluxLW    (1:imax, 1:jmax, 0:kmax)
    real(DP), intent(out) :: xyr_RadDwFluxLW    (1:imax, 1:jmax, 0:kmax)
    real(DP), intent(out) :: xyr_RadUwFluxSW    (1:imax, 1:jmax, 0:kmax)
    real(DP), intent(out) :: xyr_RadDwFluxSW    (1:imax, 1:jmax, 0:kmax)
    real(DP), intent(out) :: xyra_DelRadUwFluxLW(1:imax, 1:jmax, 0:kmax, -1:1)
    real(DP), intent(out) :: xyra_DelRadDwFluxLW(1:imax, 1:jmax, 0:kmax, -1:1)
    real(DP), intent(out) :: xyra_DelRadUwFluxSW(1:imax, 1:jmax, 0:kmax, -1:1)
    real(DP), intent(out) :: xyra_DelRadDwFluxSW(1:imax, 1:jmax, 0:kmax, -1:1)


    ! Work variables
    !
    real(DP)          :: r_Press          (0:kmax)
    real(DP)          :: SurfTemp
    real(DP)          :: r_Temp           (0:kmax)
    real(DP)          :: rm_VMR           (0:kmax, 1:NMolNum)
    real(DP)          :: za_PtclEffRad    (1:kmax,1:NPtcl)
    real(DP)          :: za_DelPtclNum    (1:kmax,1:NPtcl)
    real(DP)          :: AlbedoSW
    real(DP)          :: AlbedoLW
    real(DP)          :: SZADeg
    real(DP)          :: r_MeanMolWt      (0:kmax)
    real(DP)          :: r_RadUwFlux      (0:kmax)
    real(DP)          :: r_RadDwFlux      (0:kmax)
    real(DP)          :: ra_DelRadUwFlux  (0:kmax,-1:1)
    real(DP)          :: ra_DelRadDwFlux  (0:kmax,-1:1)
    real(DP)          :: r_RadUwFluxLW    (0:kmax)
    real(DP)          :: r_RadDwFluxLW    (0:kmax)
    real(DP)          :: r_RadUwFluxSW    (0:kmax)
    real(DP)          :: r_RadDwFluxSW    (0:kmax)
    real(DP)          :: ra_DelRadUwFluxLW(0:kmax,-1:1)
    real(DP)          :: ra_DelRadDwFluxLW(0:kmax,-1:1)
    real(DP)          :: ra_DelRadUwFluxSW(0:kmax,-1:1)
    real(DP)          :: ra_DelRadDwFluxSW(0:kmax,-1:1)

    real(DP)          :: WNIntegStart
    real(DP)          :: WNIntegEnd

    integer              :: NThreads
    integer, allocatable :: a_ls(:)
    integer, allocatable :: a_le(:)
    integer              :: iThread

    integer  :: i
    integer  :: j
    integer  :: k
    integer  :: l
    integer  :: m
    integer  :: n


    ! 初期化確認
    ! Initialization check
    !
    if ( .not. rad_flux_wrapper_inited ) then
      write( 6, * ) 'This module has not been initialized.'
      stop
    end if


    NThreads = 1
    !$ NThreads  = omp_get_max_threads()
!    !$ write( 6, * ) "Number of processors : ", omp_get_num_procs()
!    !$ write( 6, * ) "Number of threads    : ", nthreads

    allocate( a_ls(0:NThreads-1) )
    allocate( a_le(0:NThreads-1) )

    do iThread = 0, NThreads-1
      if ( iThread == 0 ) then
        a_ls(iThread) = 1
      else
        a_ls(iThread) = a_le(iThread-1) + 1
      end if
      a_le(iThread) = a_ls(iThread) + imax*jmax / NThreads - 1
      if ( iThread + 1 <= mod( imax*jmax, NThreads ) ) then
        a_le(iThread) = a_le(iThread) + 1
      end if
    end do


    WNIntegStart = -1.0d0
    WNIntegEnd   = -1.0d0

    !$OMP PARALLEL DEFAULT(PRIVATE) &
    !$OMP SHARED( &
    !$OMP         NThreads, a_ls, a_le, &
    !$OMP         imax, jmax, kmax, NMolNum, NPtcl, &
    !$OMP         m_MolNum, a_PtclName, &
    !$OMP         WNIntegStart, WNIntegEnd, &
    !$OMP         xyr_Press, xyr_Temp, xy_SurfTemp, xyrm_VMR, &
    !$OMP         xyza_PtclEffRad, xyza_DelPtclNum, &
    !$OMP         xy_AlbedoSW, xy_AlbedoLW, &
    !$OMP         StrFluxTOA, &
    !$OMP         xy_SZADeg, xyr_MeanMolWt, &
    !$OMP         xyr_RadUwFlux, xyr_RadDwFlux, &
    !$OMP         xyra_DelRadUwFlux, xyra_DelRadDwFlux, &
    !$OMP         xyr_RadUwFluxLW, xyr_RadDwFluxLW, &
    !$OMP         xyr_RadUwFluxSW, xyr_RadDwFluxSW, &
    !$OMP         xyra_DelRadUwFluxLW, xyra_DelRadDwFluxLW, &
    !$OMP         xyra_DelRadUwFluxSW, xyra_DelRadDwFluxSW &
    !$OMP       )

    !$OMP DO
    do iThread = 0, NThreads-1

      do l = a_ls(iThread), a_le(iThread)

        i = mod( (l-1), imax ) + 1
        j = int( (l-1) / imax ) + 1

        do k = 0, kmax
          r_Press(k)       = xyr_Press(i,j,k)
          r_Temp (k)       = xyr_Temp (i,j,k)
        end do
        SurfTemp         = xy_SurfTemp(i,j)
        do m = 1, NMolNum
          do k = 0, kmax
            rm_VMR(k,m) = xyrm_VMR(i,j,k,m)
          end do
        end do
        do m = 1, NPtcl
          do k = 1, kmax
            za_PtclEffRad    (k,m) = xyza_PtclEffRad    (i,j,k,m)
            za_DelPtclNum    (k,m) = xyza_DelPtclNum    (i,j,k,m)
          end do
        end do
        AlbedoSW = xy_AlbedoSW(i,j)
        AlbedoLW = xy_AlbedoLW(i,j)
        SZADeg   = xy_SZADeg  (i,j)
        do k = 0, kmax
          r_MeanMolWt(k) = xyr_MeanMolWt(i,j,k)
        end do

        call RadFluxCalcFlux( &
          & kmax, NMolNum, &
          & m_MolNum, &
          & r_Press, &
          & SurfTemp, r_Temp, rm_VMR, &
          & NPtcl, a_PtclName, za_PtclEffRad, za_DelPtclNum, &
          & AlbedoSW, AlbedoLW, SZADeg, &
          & StrFluxTOA, &
          & r_MeanMolWt, &
          & r_RadUwFlux, r_RadDwFlux, &
          & ra_DelRadUwFlux, ra_DelRadDwFlux, &
          & r_RadUwFluxLW, r_RadDwFluxLW, &
          & r_RadUwFluxSW, r_RadDwFluxSW, &
          & ra_DelRadUwFluxLW, ra_DelRadDwFluxLW, &
          & ra_DelRadUwFluxSW, ra_DelRadDwFluxSW, &
          & WNIntegStart, WNIntegEnd &
          & )

        do k = 0, kmax
          xyr_RadUwFlux      (i,j,k) = r_RadUwFlux      (k)
          xyr_RadDwFlux      (i,j,k) = r_RadDwFlux      (k)
        end do
        do n = -1, 1
          do k = 0, kmax
            xyra_DelRadUwFlux  (i,j,k,n) = ra_DelRadUwFlux  (k,n)
            xyra_DelRadDwFlux  (i,j,k,n) = ra_DelRadDwFlux  (k,n)
          end do
        end do
        do k = 0, kmax
          xyr_RadUwFluxLW    (i,j,k) = r_RadUwFluxLW    (k)
          xyr_RadDwFluxLW    (i,j,k) = r_RadDwFluxLW    (k)
          xyr_RadUwFluxSW    (i,j,k) = r_RadUwFluxSW    (k)
          xyr_RadDwFluxSW    (i,j,k) = r_RadDwFluxSW    (k)
        end do
        do n = -1, 1
          do k = 0, kmax
            xyra_DelRadUwFluxLW(i,j,k,n) = ra_DelRadUwFluxLW(k,n)
            xyra_DelRadDwFluxLW(i,j,k,n) = ra_DelRadDwFluxLW(k,n)
            xyra_DelRadUwFluxSW(i,j,k,n) = ra_DelRadUwFluxSW(k,n)
            xyra_DelRadDwFluxSW(i,j,k,n) = ra_DelRadDwFluxSW(k,n)
          end do
        end do

      end do
    end do
    !$OMP END DO
    !$OMP END PARALLEL


  end subroutine RadFluxWrapper

  !--------------------------------------------------------------------------------------

  subroutine RadFluxWrapperInit( &
    & Grav, WaveNumAlbedoSwitch, KDOptPropNcFN &
    & )

    use rad_flux, only : RadFluxInit


    ! 宣言文 ; Declaration statements
    !
    real(DP)     :: Grav
    real(DP)     :: WaveNumAlbedoSwitch
    character(*) :: KDOptPropNcFN


    ! Local variables
    !


    if ( rad_flux_wrapper_inited ) return


    ! Initialization of modules used in this module
    !
    call RadFluxInit( &
      & Grav, &
      & WaveNumAlbedoSwitch, &
      & KDOptPropNcFN &
      & )


    rad_flux_wrapper_inited = .true.

  end subroutine RadFluxWrapperInit

  !--------------------------------------------------------------------------------------

  subroutine RadFluxWrapperFinalize

    use rad_flux, only : RadFluxFinalize


    ! 宣言文 ; Declaration statements
    !


    ! Local variables
    !


    if ( rad_flux_wrapper_inited ) return


    call RadFluxFinalize


    rad_flux_wrapper_inited = .false.

  end subroutine RadFluxWrapperFinalize

  !--------------------------------------------------------------------------------------

end module rad_flux_wrapper
