!= 土壌熱伝導係数の設定
!
!= set soil thermal diffusion coefficient
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: soil_thermdiffcoef.f90,v 1.1 2015/01/29 12:12:21 yot Exp $
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008-2009. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module soil_thermdiffcoef

  !== References
  !
!!$  !  Matthews, E.,
!!$  !    Global vegetation and land use: New high-resolution data bases for climate 
!!$  !    studies, 
!!$  !    J. Clim. Appl. Meteor., 22, 474, 1983. 
!!$  !
!!$  !  Matthews, E., 
!!$  !    Prescription of land-surface boundary conditions in GISS GCM II:
!!$  !    A simple method based on fine-resolution data bases, 
!!$  !    NASA Technical memorandum #86096, 1984.

  ! モジュール引用 ; USE statements
  !

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision.
    &                 STRING     ! 文字列.       Strings.

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public:: SetSoilThermDiffCoefSimple
  public:: SoilThermDiffCoefInit

  ! 公開変数
  ! Public variables
  !

  ! 非公開変数
  ! Private variables
  !
  logical, save :: soil_thermdiffcoef_inited = .false.
                              ! 初期設定フラグ.
                              ! Initialization flag

  real(DP), save:: SoilThermDiffCoefMin
  real(DP), save:: SoilThermDiffCoefMax

  real(DP), save:: SoilMoistCritAmnt

  character(*), parameter:: module_name = 'soil_thermdiffcoef'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: soil_thermdiffcoef.f90,v 1.1 2015/01/29 12:12:21 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

contains

  !--------------------------------------------------------------------------------------

  subroutine SetSoilThermDiffCoefSimple( &
    & xy_SurfType, xy_SoilMoist,         & ! (in )
    & xy_SoilThermDiffCoef               & ! (out)
    & )

    ! モジュール引用 ; USE statements
    !

    ! 格子点設定
    ! Grid points settings
    !
    use gridset, only: imax, & ! 経度格子点数.
                               ! Number of grid points in longitude
      &                jmax, & ! 緯度格子点数.
                               ! Number of grid points in latitude
      &                kmax    ! 鉛直層数.
                               ! Number of vertical level

    ! 宣言文 ; Declaration statements
    !
    integer , intent(in ) :: xy_SurfType         ( 0:imax-1, 1:jmax )
    real(DP), intent(in ) :: xy_SoilMoist        ( 0:imax-1, 1:jmax )
    real(DP), intent(out) :: xy_SoilThermDiffCoef( 0:imax-1, 1:jmax )


    ! 作業変数
    ! Work variables
    !
    integer:: i               ! 経度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in latitude


    ! 初期化確認
    ! Initialization check
    !
    if ( .not. soil_thermdiffcoef_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    !
    ! land
    !
    do j = 1, jmax
      do i = 0, imax-1
        if( xy_SurfType( i, j ) >= 1 ) then
          xy_SoilThermDiffCoef(i,j) =                           &
            &   ( SoilThermDiffCoefMax - SoilThermDiffCoefMin ) &
            & / ( SoilMoistCritAmnt    - 0.0_DP               ) &
            & * ( xy_SoilMoist(i,j)    - 0.0_DP               ) &
            & + SoilThermDiffCoefMin
          xy_SoilThermDiffCoef(i,j) =                           &
            & min( max( xy_SoilThermDiffCoef(i,j), SoilThermDiffCoefMin ), &
            &      SoilThermDiffCoefMax )
        else
          xy_SoilThermDiffCoef(i,j) = 1.0e100_DP
        end if
      end do
    end do


  end subroutine SetSoilThermDiffCoefSimple

  !--------------------------------------------------------------------------------------

  subroutine SoilThermDiffCoefInit( &
    & FlagSnow &
    & )

    ! モジュール引用 ; USE statements
    !

    ! 種別型パラメタ
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg

    ! バケツモデル
    ! Bucket model
    !
    use Bucket_Model, only : &
      & BucketGetSoilMoistCritAmnt, &
      & BucketModelInit


    logical, intent(in) :: FlagSnow


    ! 作業変数
    ! Work variables
    !
    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号.
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT.
                              ! IOSTAT of NAMELIST read


    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /soil_thermdiffcoef_nml/ &
      & SoilThermDiffCoefMin, &
      & SoilThermDiffCoefMax

    ! 実行文 ; Executable statement
    !

    if ( soil_thermdiffcoef_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !

    SoilThermDiffCoefMin = 0.2_DP
    SoilThermDiffCoefMax = 2.0_DP
                     !   thermal conductivity (W m-1 K-1)


    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml,                   &  ! (in)
        & nml = soil_thermdiffcoef_nml, &  ! (out)
        & iostat = iostat_nml           &  ! (out)
        & )
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
      if ( iostat_nml == 0 ) write( STDOUT, nml = soil_thermdiffcoef_nml )
    end if


    ! バケツモデル
    ! Bucket model
    !
    call BucketModelInit( &
      & FlagSnow          & ! (in)
      & )
    SoilMoistCritAmnt = BucketGetSoilMoistCritAmnt()


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, 'SoilThermDiffCoefMin = %f', d = (/SoilThermDiffCoefMin/) )
    call MessageNotify( 'M', module_name, 'SoilThermDiffCoefMax = %f', d = (/SoilThermDiffCoefMax/) )
    call MessageNotify( 'M', module_name, 'SoilMoistCritAmnt    = %f', d = (/SoilMoistCritAmnt/) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    soil_thermdiffcoef_inited = .true.

  end subroutine SoilThermDiffCoefInit

  !--------------------------------------------------------------------------------------

end module soil_thermdiffcoef
