!= 雪と海氷の定数の設定
!
!= Setting constants of snow and sea ice
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: constants_snowseaice.f90,v 1.5 2015/01/29 12:07:29 yot Exp $ 
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module constants_snowseaice
  !
  != 雪と海氷の定数の設定
  !
  != Setting constants of snow and sea ice
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! 雪と海氷の定数の設定および保管を行います. 
  ! これらの値は NAMELIST 変数群 constants_snowseaice_nml によって変更することが
  ! 可能です. 
  !
  ! Constants of snow and sea ice are set and stored. 
  ! These values can be changed by NAMELIST group name 
  ! "constants_snowseaice_nml". 
  !
  !== Procedures List
  !
  ! ConstantsSnowSeaIceInit :: 物理定数の設定
  ! ------------  :: ------------
  ! ConstantsSnowSeaIceInit :: Settings of constants of snow and sea ice
  !
  !== NAMELIST
  !
  ! NAMELIST#constants_snowseaice_nml
  !


  ! モジュール引用 ; USE statements
  !

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP     ! 倍精度実数型. Double precision. 

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public:: ConstantsSnowSeaIceInit

  ! 公開変数
  ! Public variables
  !
  logical, save :: constants_snowseaice_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag

  real(DP), public, save:: SnowThresholdForFlux
  real(DP), public, save:: SnowThresholdForAlbedo
  real(DP), public, save:: SnowAlbedo
  real(DP), public, save:: SnowThermCondCoef
                              ! Thermal conductivity of snow (W m-1 K-1)
  real(DP), public, save:: SnowVolHeatCap
                              ! Volumetric heat capacity of snow (J m-3 K-1)
  real(DP), public, save:: SnowDens
                              ! Density of snow (kg m-3)
  real(DP), public, save:: SnowMaxThermDepth
                              ! Maximum thermal depth of snow (m)
  real(DP), public, save:: TempCondWater
                        ! <Japanese>
                        ! Condensation temperature of water

  real(DP), public, save:: SeaIceVolHeatCap
                              ! Volumetric heat capacity of sea ice (J m-3 K-1)
  real(DP), public, save:: SeaIceThermCondCoef
                              ! Thermal conductivity of sea ice (W m-1 K-1)

  real(DP), public, save:: SeaIceThreshold
  real(DP), public, save:: SeaIceThickness
  real(DP), public, save:: TempBelowSeaIce

  real(DP), public, save:: SeaIceAlbedo


  real(DP), public, save:: CO2IceThreshold
  real(DP), public, save:: CO2IceAlbedoS
  real(DP), public, save:: CO2IceAlbedoN
  real(DP), public, save:: CO2IceEmisS
  real(DP), public, save:: CO2IceEmisN



  ! 非公開変数
  ! Private variables
  !

  character(*), parameter:: module_name = 'constants_snowseaice'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: constants_snowseaice.f90,v 1.5 2015/01/29 12:07:29 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

contains

  !--------------------------------------------------------------------------------------

  subroutine ConstantsSnowSeaIceInit
    !
    ! constants_snowseaice モジュールの初期化を行います. 
    ! NAMELIST#constants_snowseaice_nml の読み込みはこの手続きで行われます. 
    !
    ! "constants_snowseaice" module is initialized. 
    ! NAMELIST#constants_snowseaice_nml is loaded in this procedure. 
    !

    ! モジュール引用 ; USE statements
    !

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! 種別型パラメタ
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! メッセージ出力
    ! Message output
    !
    use dc_message, only: MessageNotify

    ! 宣言文 ; Declaration statements
    !
    implicit none

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /constants_snowseaice_nml/ &
      & SnowThresholdForAlbedo, &
      & SnowThresholdForFlux  , &
      & SnowAlbedo         , &
      & SnowThermCondCoef  , &
      & SnowVolHeatCap     , &
      & SnowDens           , &
      & SnowMaxThermDepth  , &
      & TempCondWater      , &
      & SeaIceVolHeatCap   , &
      & SeaIceThermCondCoef, &
      & SeaIceThreshold    , &
      & SeaIceThickness    , &
      & TempBelowSeaIce    , &
      & SeaIceAlbedo       , &
      & CO2IceAlbedoS      , &
      & CO2IceAlbedoN      , &
      & CO2IceEmisS        , &
      & CO2IceEmisN

          !
          ! デフォルト値については初期化手続 "constants#ConstantsInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "constants#ConstantsInit" for the default values. 
          !

    ! 実行文 ; Executable statement
    !

    if ( constants_snowseaice_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !

    SnowThresholdForAlbedo = 0.1_DP * 1.0e3_DP * 0.01e0_DP
                        ! 1 cm snow depth
                        ! 0.1 * 1000 (kg m-3; water density) x 1 cm
                        ! This value is selected arbitrarily. 
!!$    SnowAlbedo         = 0.7d0
    SnowAlbedo         = 0.75_DP
                        ! This value is selected arbitrarily. 

    SnowThresholdForFlux   = SnowThresholdForAlbedo

    ! MEMO for snow properties by Hillet
    !* snow (f=0.95) : k = 0.15e-3, C = 0.05
    !* snow (f=0.8 ) : k = 0.32e-3, C = 0.2
    !* snow (f=0.5 ) : k = 1.7e-3 , C = 0.5
    !f: porosity
    !k [cal/(cm s K)] conductivity
    !C [cal/(cm s K)] heat capacity

    SnowThermCondCoef = 0.32e-3_DP * 4.219e2_DP
                              ! Thermal conductivity of snow (W m-1 K-1)
!!$    SnowVolHeatCap     = 0.2_DP * 4.219e2_DP
    SnowVolHeatCap    = 2.0e6_DP * 200.0_DP / 1000.0_DP
                        ! This value is roughly based on ECMWF IFS value.
                        ! p.121, IFSPart4.pdf
                              ! Volumetric heat capacity of snow (J m-3 K-1)

    SnowDens          = 200.0_DP
                        ! This value is selected arbitrarily. 
                              ! Density of snow (kg m-3)

    SnowMaxThermDepth = 1.0_DP
                        ! This value is based on ECMWF IFS value.
                        ! p.121, IFSPart4.pdf
                              ! Maximum thermal depth of snow (m)


    TempCondWater      = 273.15_DP
                        ! <Japanese>
                        ! Condensation temperature of water

!!$    ! https://www.myroms.org/wiki/index.php/Sea-Ice_Model
!!$    SeaIceDens         = 900.0d0   ! no reliable reference
!!$    SeaIceSpecHeat     = 2093.0d0  ! no reliable reference
!!$    SeaIceHeatDiffCoef = 2.04d0 / ( SeaIceDens * SeaIceSpecHeat )  ! 2.04d0 (W (m K)-1),  no reliable reference

!!$    SeaIceDens         = 0.92d3
!!$    ! kg m-3
!!$    SeaIceHeatDiffCoef = 2.04d0 / ( SeaIceDens * SeaIceSpecHeat )
!!$    ! 2.04d0 (W (m K)-1),  no reliable reference
!!$    SeaIceSpecHeat     = 1.9d6 / SeaIceDens
!!$    ! no reliable reference

    SeaIceVolHeatCap    = 1.9e6_DP
    ! J m-3 K-1
    ! Value of ice in Table 12.1 by Hillel (2004). 
    SeaIceThermCondCoef = 2.2_DP
    ! W m-1 K-1
    ! Value of ice in Table 12.2 by Hillel (2004). 

    ! Reference
    !
    ! Hillet, D., 
    !   Introduction to Environmental Soil Physics, 
    !   Elsevier Academic Press, pp494, 2004.


    SeaIceThreshold    = 0.5_DP            ! arbitrarily set, no reliable reference
    SeaIceThickness    = 2.0_DP            ! arbitrarily set, no reliable reference
    TempBelowSeaIce    = 273.15_DP - 2.0_DP ! arbitrarily set, no reliable reference

!!$    SeaIceAlbedo       = 0.5_DP            ! arbitrarily set, no reliable reference
    SeaIceAlbedo       = 0.75_DP           ! arbitrarily set, no reliable reference


    CO2IceThreshold = 1.0_DP         ! No reference
    CO2IceAlbedoS   = 0.75_DP        ! No reference
    CO2IceAlbedoN   = 0.75_DP        ! No reference
    CO2IceEmisS     = 1.0_DP         ! No reference
    CO2IceEmisN     = 1.0_DP         ! No reference


    ! NAMELIST からの入力
    ! Input from NAMELIST
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml,                     &  ! (in)
        & nml = constants_snowseaice_nml, &  ! (out)
        & iostat = iostat_nml            )   ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
      if ( iostat_nml == 0 ) write( STDOUT, nml = constants_snowseaice_nml )
    end if

    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '  SnowThresholdForAlbedo = %f', d = (/ SnowThresholdForAlbedo /) )
    call MessageNotify( 'M', module_name, '  SnowAlbedo             = %f', d = (/ SnowAlbedo          /) )
    call MessageNotify( 'M', module_name, '  SnowThresholdForFlux   = %f', d = (/ SnowThresholdForFlux /) )
    call MessageNotify( 'M', module_name, '  SnowThermCondCoef   = %f', d = (/ SnowThermCondCoef   /) )
    call MessageNotify( 'M', module_name, '  SnowVolHeatCap      = %f', d = (/ SnowVolHeatCap      /) )
    call MessageNotify( 'M', module_name, '  SnowDens            = %f', d = (/ SnowDens            /) )
    call MessageNotify( 'M', module_name, '  SnowMaxThermDepth   = %f', d = (/ SnowMaxThermDepth   /) )
    call MessageNotify( 'M', module_name, '  TempCondWater       = %f', d = (/ TempCondWater       /) )
    call MessageNotify( 'M', module_name, '  SeaIceThreshold     = %f', d = (/ SeaIceThreshold     /) )
    call MessageNotify( 'M', module_name, '  SeaIceVolHeatCap    = %f', d = (/ SeaIceVolHeatCap    /) )
    call MessageNotify( 'M', module_name, '  SeaIceThermCondCoef = %f', d = (/ SeaIceThermCondCoef /) )

!!$    call MessageNotify( 'M', module_name, '  SeaIceDens         = %f', d = (/ SeaIceDens         /) )
!!$    call MessageNotify( 'M', module_name, '  SeaIceSpecHeat     = %f', d = (/ SeaIceSpecHeat     /) )
!!$    call MessageNotify( 'M', module_name, '  SeaIceHeatDiffCoef = %f', d = (/ SeaIceHeatDiffCoef /) )

    call MessageNotify( 'M', module_name, '  SeaIceThickness     = %f', d = (/ SeaIceThickness     /) )
    call MessageNotify( 'M', module_name, '  TempBelowSeaIce     = %f', d = (/ TempBelowSeaIce     /) )
    call MessageNotify( 'M', module_name, '  SeaIceAlbedo        = %f', d = (/ SeaIceAlbedo        /) )
    call MessageNotify( 'M', module_name, '  CO2IceAlbedoS       = %f', d = (/ CO2IceAlbedoS       /) )
    call MessageNotify( 'M', module_name, '  CO2IceAlbedoN       = %f', d = (/ CO2IceAlbedoN       /) )
    call MessageNotify( 'M', module_name, '  CO2IceEmisS         = %f', d = (/ CO2IceEmisS       /) )
    call MessageNotify( 'M', module_name, '  CO2IceEmisN         = %f', d = (/ CO2IceEmisN       /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    constants_snowseaice_inited = .true.

  end subroutine ConstantsSnowSeaIceInit

  !--------------------------------------------------------------------------------------

end module constants_snowseaice
