!= 物理定数設定
!
!= Physical constants settings
!
! Authors::   Yasuhiro MORIKAWA
! Version::   $Id: constants.f90,v 1.11 2014/05/07 09:39:21 murashin Exp $ 
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module constants
  !
  != 物理定数設定
  !
  != Physical constants settings
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! 物理定数の設定および保管を行います. 
  ! デフォルト値は地球大気を想定した値が設定されています. 
  ! これらの値は NAMELIST 変数群 constants_nml によって変更することが
  ! 可能です. 
  !
  ! Physical constants are set and stored. 
  ! By default, values on atmosphere of earth are set. 
  ! These values can be changed by NAMELIST group name 
  ! "constants_nml". 
  !
  !== References
  !
  ! WMO technical regulations, No. 49, Vol. I General Meteorological 
  ! Standards and Recommended Practices, 
  ! http://www.wmo.int/pages/prog/www/WIS/Publications/49%20%28Technical%20Regulations%29/49%20-%20Volume%20I/49_I_E.pdf#page=61
  !
  !== Procedures List
  !
  ! ConstantsInit :: 物理定数の設定
  ! ------------  :: ------------
  ! ConstantsInit :: Settings of physical constants
  !
  !== NAMELIST
  !
  ! NAMELIST#constants_nml
  !


  ! モジュール引用 ; USE statements
  !

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP     ! 倍精度実数型. Double precision. 

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public:: ConstantsInit

  ! 公開変数
  ! Public variables
  !
  logical, save :: constants_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag

  real(DP), save, public:: RPlanet
                              ! $ a $ [m]. 
                              ! 惑星半径. 
                              ! Radius of planet
  real(DP), save, public:: Omega
                              ! $ \Omega $ [s-1]. 
                              ! 回転角速度. 
                              ! Angular velocity
  real(DP), save, public:: Grav
                              ! $ g $ [m s-2]. 
                              ! 重力加速度. 
                              ! Gravitational acceleration
  real(DP), save, public:: MolWtDry
                              ! $ M $ [kg mol-1]. 
                              ! 乾燥大気の平均分子量. 
                              ! Mean molecular weight of dry air
  real(DP), save, public:: CpDry
                              ! $ C_p $ [J kg-1 K-1]. 
                              ! 乾燥大気の定圧比熱. 
                              ! Specific heat of air at constant pressure
  real(DP), save, public:: MolWtWet
                              ! $ M_v $ [kg mol-1]. 
                              ! 凝結成分の平均分子量. 
                              ! Mean molecular weight of condensible elements
  real(DP), save, public:: CpWet
                              ! $ C_v $ [J kg-1 K-1] . 
                              ! 凝結成分の定圧比熱. 
                              ! Specific heat of condensible elements at constant pressure
  real(DP), save, public:: LatentHeat
                              ! $ L $ [J kg-1] . 
                              ! 凝結の潜熱. 
                              ! Latent heat of condensation
  real(DP), save, public:: LatentHeatFusion
                              ! $ L $ [J kg-1] . 
                              ! 融解の潜熱. 
                              ! Latent heat of fusion

  real(DP), save, public:: GasRDry
                              ! $ R $ [J kg-1 K-1]. 
                              ! 乾燥大気の気体定数. 
                              ! Gas constant of air
  real(DP), save, public:: GasRWet
                              ! $ R_v $ [J kg-1 K-1]. 
                              ! 凝結成分の気体定数. 
                              ! Gas constant of condensible elements
  ! EpsV will be removed. 
  real(DP), save, public:: EpsV
                              ! $ \epsilon_v $ . 
                              ! 水蒸気分子量比. 
                              ! Molecular weight of water vapor

  real(DP), parameter, public:: FKarm = 0.4_DP
                              ! $ k $ .
                              ! カルマン定数. 
                              ! Karman constant


  ! 非公開変数
  ! Private variables
  !

  character(*), parameter:: module_name = 'constants'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: constants.f90,v 1.11 2014/05/07 09:39:21 murashin Exp $'
                              ! モジュールのバージョン
                              ! Module version

contains

  !--------------------------------------------------------------------------------------

  subroutine ConstantsInit
    !
    ! constants モジュールの初期化を行います. 
    ! NAMELIST#constants_nml の読み込みはこの手続きで行われます. 
    !
    ! "constants" module is initialized. 
    ! NAMELIST#constants_nml is loaded in this procedure. 
    !

    ! モジュール引用 ; USE statements
    !

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! 種別型パラメタ
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! メッセージ出力
    ! Message output
    !
    use dc_message, only: MessageNotify

    ! 物理・数学定数設定
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI,               &
      & GasRUniv

    ! 宣言文 ; Declaration statements
    !
    implicit none

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /constants_nml/ &
      & RPlanet, Omega, Grav, &
      & MolWtDry, CpDry, &
      & MolWtWet, CpWet, &
      & LatentHeat, LatentHeatFusion
          !
          ! デフォルト値については初期化手続 "constants#ConstantsInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "constants#ConstantsInit" for the default values. 
          !

    ! 実行文 ; Executable statement
    !

    if ( constants_inited ) return

    ! デフォルト値の設定
    ! Default values settings
    !
    RPlanet          = 6.371e6_DP
    Omega            = 2.0_DP * PI / ( 60.0_DP * 60.0_DP * 23.9345_DP )

!    Grav             = 9.8_DP    ! old value
    Grav             = 9.80665_DP
                       ! standard acceleration of gravity (m s-2)
                       ! WMO technical regulations (see above)

    CpDry            = 1004.6_DP

!    MolWtDry         = 28.964e-3_DP    ! old value
    MolWtDry         = 28.9644e-3_DP
                       ! WMO technical regulations (see above)

    CpWet            = 1810.0_DP

!    MolWtWet         = 18.01528e-3_DP    ! old value
    MolWtWet         = 18.0153e-3_DP
                       ! WMO technical regulations (see above)

    LatentHeat       = 2.5e6_DP 
    LatentHeatFusion = 334.0e3_DP

    ! NAMELIST からの入力
    ! Input from NAMELIST
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml, &           ! (in)
        & nml = constants_nml, &  ! (out)
        & iostat = iostat_nml )   ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
      if ( iostat_nml == 0 ) write( STDOUT, nml = constants_nml )
    end if

    GasRDry          = GasRUniv / MolWtDry
    GasRWet          = GasRUniv / MolWtWet
    EpsV             = MolWtWet / MolWtDry


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '  RPlanet          = %f', d = (/ RPlanet          /) )
    call MessageNotify( 'M', module_name, '  Omega            = %f', d = (/ Omega            /) )
    call MessageNotify( 'M', module_name, '  Grav             = %f', d = (/ Grav             /) )

    call MessageNotify( 'M', module_name, '  CpDry            = %f', d = (/ CpDry            /) )
    call MessageNotify( 'M', module_name, '  MolWtDry         = %f', d = (/ MolWtDry         /) )

    call MessageNotify( 'M', module_name, '  CpWet            = %f', d = (/ CpWet            /) )
    call MessageNotify( 'M', module_name, '  MolWtWet         = %f', d = (/ MolWtWet         /) )
    call MessageNotify( 'M', module_name, '  LatentHeat       = %f', d = (/ LatentHeat       /) )
    call MessageNotify( 'M', module_name, '  LatentHeatFusion = %f', d = (/ LatentHeatFusion /) )

    call MessageNotify( 'M', module_name, '  GasRDry          = %f', d = (/ GasRDry          /) )
    call MessageNotify( 'M', module_name, '  GasRWet          = %f', d = (/ GasRWet          /) )
    call MessageNotify( 'M', module_name, '  EpsV             = %f', d = (/ EpsV             /) )
    call MessageNotify( 'M', module_name, '  FKarm            = %f', d = (/ FKarm            /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    constants_inited = .true.

  end subroutine ConstantsInit

  !--------------------------------------------------------------------------------------

end module constants
