!= 放射関連ルーチン
!
!= Routines for radiation calculation
!
! Authors::   Yasuhiro MORIKAWA, Yukiko YAMADA, Satoshi NODA, Yoshiyuki O. TAKAHASHI
! Version::   $Id: rad_utils.f90,v 1.7 2012/10/08 12:48:45 yot Exp $ 
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module rad_utils
  !
  != 放射関連ルーチン
  !
  != Routines for radiation calculation
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! 
  !
  ! 
  !
  !== Procedures List
  !
  ! RadDTempDt        :: 放射フラックスによる温度変化の計算
  ! RadFluxOutput     :: 放射フラックスの出力
  ! ------------            :: ------------
  ! RadDTempDt        :: Calculate temperature tendency with radiation flux
  ! RadFluxOutput     :: Output radiation fluxes
  !
  !== NAMELIST
  !
  ! NAMELIST#rad_utils_nml
  !

  ! モジュール引用 ; USE statements
  !

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING, &  ! 文字列.       Strings. 
    &                 TOKEN      ! キーワード.   Keywords. 

  ! 物理・数学定数設定
  ! Physical and mathematical constants settings
  !
  use constants0, only: &
    & PI, &                 ! $ \pi $.
                            ! 円周率. Circular constant
    & StB
                            ! $ \sigma_{SB} $ .
                            ! ステファンボルツマン定数.
                            ! Stefan-Boltzmann constant

  ! 物理定数設定
  ! Physical constants settings
  !
  use constants, only: &
    & Grav, &
                            ! $ g $ [m s-2]. 
                            ! 重力加速度. 
                            ! Gravitational acceleration
    & CpDry
                            ! $ C_p $ [J kg-1 K-1]. 
                            ! 乾燥大気の定圧比熱. 
                            ! Specific heat of air at constant pressure


  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数. 
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数. 
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数. 
                             ! Number of vertical level

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public :: RadDTempDt
  public :: RadDTempDtforNHM2DWrapper
  public :: RadFluxOutput
  public :: RadUtilsInit


  ! 公開変数
  ! Public variables
  !
  logical, save, public:: rad_utils_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag


  ! 非公開変数
  ! Private variables
  !


  character(*), parameter:: module_name = 'rad_utils'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: rad_utils.f90,v 1.7 2012/10/08 12:48:45 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

contains

  !--------------------------------------------------------------------------------------

  subroutine RadDTempDt( &
    & xyr_RadLFlux, xyr_RadSFlux, xyr_Press, & ! (in)
    & xyz_DTempDtRadL, xyz_DTempDtRadS &       ! (out)
    & )
    !
    ! 放射による温度変化率を計算します. 
    ! 
    ! Temperature tendency with radiation is calculated. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & TimeN, &              ! ステップ $ t $ の時刻. Time of step $ t $. 
      & TimesetClockStart, TimesetClockStop

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut


    ! 宣言文 ; Declaration statements
    !
    implicit none
    real(DP), intent(in):: xyr_RadLFlux (0:imax-1, 1:jmax, 0:kmax)
                              ! 長波フラックス. 
                              ! Longwave flux
    real(DP), intent(in):: xyr_RadSFlux (0:imax-1, 1:jmax, 0:kmax)
                              ! 短波 (日射) フラックス. 
                              ! Shortwave (insolation) flux
    real(DP), intent(in):: xyr_Press    (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ . 気圧 (半整数レベル). 
                              ! Air pressure (half level)
    real(DP), intent(out):: xyz_DTempDtRadL (0:imax-1, 1:jmax, 1:kmax)
                              ! 長波加熱率. 
                              ! Temperature tendency with longwave
    real(DP), intent(out):: xyz_DTempDtRadS (0:imax-1, 1:jmax, 1:kmax)
                              ! 短波加熱率. 
                              ! Temperature tendency with shortwave

    ! 作業変数
    ! Work variables
    !
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction

    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. rad_utils_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    ! 放射冷却率の演算
    ! Calculate radiation cooling rate
    !
    do k = 1, kmax
      xyz_DTempDtRadL(:,:,k) = &
        & (     xyr_RadLFlux(:,:,k-1) - xyr_RadLFlux(:,:,k) ) &
        &   / ( xyr_Press(:,:,k-1)    - xyr_Press(:,:,k) ) &
        &   / CpDry * Grav

      xyz_DTempDtRadS(:,:,k) = &
        & (     xyr_RadSFlux(:,:,k-1) - xyr_RadSFlux(:,:,k) ) &
        &   / ( xyr_Press(:,:,k-1)    - xyr_Press(:,:,k) ) &
        &   / CpDry * Grav
    end do


    ! ヒストリデータ出力
    ! History data output
    !
    call HistoryAutoPut( TimeN, 'DTempDtRadL' , xyz_DTempDtRadL )
    call HistoryAutoPut( TimeN, 'DTempDtRadS' , xyz_DTempDtRadS )


    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine RadDTempDt

  !--------------------------------------------------------------------------------------

  subroutine RadDTempDtforNHM(                        &
    & xyr_RadLFlux, xyr_RadSFlux, r_Height, xyz_Dens, & ! (in)
    & xyz_DTempDtRadL, xyz_DTempDtRadS                & ! (out)
    & )
    !
    ! 非静力学系における放射による温度変化率を計算します. 
    ! 
    ! Temperature tendency with radiation is calculated for non-hydrodynamic system. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & TimeN, &              ! ステップ $ t $ の時刻. Time of step $ t $. 
      & TimesetClockStart, TimesetClockStop

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut


    ! 宣言文 ; Declaration statements
    !
    implicit none
    real(DP), intent(in):: xyr_RadLFlux (0:imax-1, 1:jmax, 0:kmax)
                              ! 長波フラックス. 
                              ! Longwave flux
    real(DP), intent(in):: xyr_RadSFlux (0:imax-1, 1:jmax, 0:kmax)
                              ! 短波 (日射) フラックス. 
                              ! Shortwave (insolation) flux
    real(DP), intent(in):: r_Height                       (0:kmax)
                              ! $ \hat{p} $ . 気圧 (半整数レベル). 
                              ! Air pressure (half level)
    real(DP), intent(in):: xyz_Dens     (0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(out):: xyz_DTempDtRadL (0:imax-1, 1:jmax, 1:kmax)
                              ! 長波加熱率. 
                              ! Temperature tendency with longwave
    real(DP), intent(out):: xyz_DTempDtRadS (0:imax-1, 1:jmax, 1:kmax)
                              ! 短波加熱率. 
                              ! Temperature tendency with shortwave

    ! 作業変数
    ! Work variables
    !
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction

    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. rad_utils_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    ! 放射冷却率の演算
    ! Calculate radiation cooling rate
    !
    do k = 1, kmax
      xyz_DTempDtRadL(:,:,k) = &
        &   - ( xyr_RadLFlux(:,:,k) - xyr_RadLFlux(:,:,k-1) ) &
        &   / (   r_Height      (k) -   r_Height      (k-1) ) &
        &   / ( CpDry * xyz_Dens(:,:,k) )

      xyz_DTempDtRadS(:,:,k) = &
        &   - ( xyr_RadSFlux(:,:,k) - xyr_RadSFlux(:,:,k-1) ) &
        &   / (   r_Height      (k) -   r_Height      (k-1) ) &
        &   / ( CpDry * xyz_Dens(:,:,k) )
    end do


    ! ヒストリデータ出力
    ! History data output
    !
    call HistoryAutoPut( TimeN, 'DTempDtRadL' , xyz_DTempDtRadL )
    call HistoryAutoPut( TimeN, 'DTempDtRadS' , xyz_DTempDtRadS )


    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine RadDTempDtforNHM

  !--------------------------------------------------------------------------------------

  subroutine RadDTempDtforNHM2DWrapper(             &
    & xr_RadLFlux, xr_RadSFlux, r_Height, xz_Dens,  & ! (in)
    & xz_DTempDtRadL, xz_DTempDtRadS                & ! (out)
    & )
    !
    ! 放射による温度変化率を計算します. 
    ! 
    ! Temperature tendency with radiation is calculated. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 宣言文 ; Declaration statements
    !
    implicit none
    real(DP), intent(in):: xr_RadLFlux (0:imax-1, 0:kmax)
                              ! 長波フラックス. 
                              ! Longwave flux
    real(DP), intent(in):: xr_RadSFlux (0:imax-1, 0:kmax)
                              ! 短波 (日射) フラックス. 
                              ! Shortwave (insolation) flux
    real(DP), intent(in):: r_Height    (0:kmax)
                              ! $ \hat{p} $ . 気圧 (半整数レベル). 
                              ! Air pressure (half level)
    real(DP), intent(in):: xz_Dens     (0:imax-1, 1:kmax)
    real(DP), intent(out):: xz_DTempDtRadL (0:imax-1, 1:kmax)
                              ! 長波加熱率. 
                              ! Temperature tendency with longwave
    real(DP), intent(out):: xz_DTempDtRadS (0:imax-1, 1:kmax)
                              ! 短波加熱率. 
                              ! Temperature tendency with shortwave

    ! 作業変数
    ! Work variables
    !
    real(DP) :: xyr_RadLFlux   (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_RadSFlux   (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyz_Dens       (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DTempDtRadL(0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DTempDtRadS(0:imax-1, 1:jmax, 1:kmax)


    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. rad_utils_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyr_RadLFlux(:,1,:) = xr_RadLFlux(:,:)
    xyr_RadSFlux(:,1,:) = xr_RadSFlux(:,:)
    xyz_Dens    (:,1,:) = xz_Dens    (:,:)

    call RadDTempDtforNHM(                              &
      & xyr_RadLFlux, xyr_RadSFlux, r_Height, xyz_Dens, & ! (in)
      & xyz_DTempDtRadL, xyz_DTempDtRadS                & ! (out)
      & )

    xz_DTempDtRadL(:,:) = xyz_DTempDtRadL(:,1,:)
    xz_DTempDtRadS(:,:) = xyz_DTempDtRadS(:,1,:)


  end subroutine RadDTempDtforNHM2DWrapper

  !-------------------------------------------------------------------

  subroutine RadFluxOutput(                          &
    & xyr_RadSUwFlux, xyr_RadSDwFlux,                & ! (in)
    & xyr_RadLUwFlux, xyr_RadLDwFlux,                & ! (in)
    & xyra_DelRadLUwFlux, xyra_DelRadLDwFlux,        & ! (in)
    & xy_DSurfTempDt, xyz_DTempDt                    & ! (in)
    & )
    !
    ! 放射フラックス (xyr_RadSFlux, xyr_RadLFlux) 
    ! について, その他の引数を用いて補正し, 出力を行う. 
    !
    ! Radiation fluxes (xyr_RadSFlux, xyr_RadLFlux) are
    ! corrected by using other arguments, and the corrected values are output.
    !

    ! モジュール引用 ; USE statements
    !

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & DelTime, &            ! $ \Delta t $ [s]
      & TimeN, &              ! ステップ $ t $ の時刻. Time of step $ t $. 
      & TimesetClockStart, TimesetClockStop

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 宣言文 ; Declaration statements
    !
    implicit none
    real(DP), intent(in):: xyr_RadSUwFlux  (0:imax-1, 1:jmax, 0:kmax)
                              ! 短波 (日射) フラックス. 
                              ! Upward shortwave (insolation) flux
    real(DP), intent(in):: xyr_RadSDwFlux  (0:imax-1, 1:jmax, 0:kmax)
                              ! 短波 (日射) フラックス. 
                              ! Downward shortwave (insolation) flux

    real(DP), intent(in):: xyr_RadLUwFlux    (0:imax-1, 1:jmax, 0:kmax)
                              ! 長波フラックス. 
                              ! Upward longwave flux
    real(DP), intent(in):: xyr_RadLDwFlux    (0:imax-1, 1:jmax, 0:kmax)
                              ! 長波フラックス. 
                              ! Downward longwave flux
    real(DP), intent(in):: xyra_DelRadLUwFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)
                              ! 長波地表温度変化. 
                              ! 
    real(DP), intent(in):: xyra_DelRadLDwFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)
                              ! 長波地表温度変化. 
                              ! 

    real(DP), intent(in):: xy_DSurfTempDt  (0:imax-1, 1:jmax)
                              ! 地表面温度変化率. 
                              ! Surface temperature tendency
    real(DP), intent(in):: xyz_DTempDt     (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \DP{T}{t} $ . 温度変化 (K s-1)
                              ! Temperature tendency (K s-1)

    ! 出力のための作業変数
    ! Work variables for output
    !
    real(DP):: xyr_RadSFlux    (0:imax-1, 1:jmax, 0:kmax)
                              ! 短波 (日射) フラックス. 
                              ! Shortwave (insolation) flux
    real(DP):: xyr_RadLFlux    (0:imax-1, 1:jmax, 0:kmax)
                              ! 長波フラックス. 
                              ! Longwave flux
    real(DP):: xyra_DelRadLFlux(0:imax-1, 1:jmax, 0:kmax, 0:1)
                              ! 長波地表温度変化. 
                              ! Surface temperature tendency with longwave
    real(DP):: xyr_RadLUwFluxCor(0:imax-1, 1:jmax, 0:kmax)
                              ! 補正された長波フラックス. 
                              ! Corrected longwave flux
    real(DP):: xyr_RadLDwFluxCor(0:imax-1, 1:jmax, 0:kmax)
                              ! 補正された長波フラックス. 
                              ! Corrected longwave flux
    real(DP):: xyr_RadLFluxCor  (0:imax-1, 1:jmax, 0:kmax)
                              ! 補正された長波フラックス. 
                              ! Corrected longwave flux

    ! 作業変数
    ! Work variables
    !
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction
    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. rad_utils_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    xyr_RadSFlux     = xyr_RadSUwFlux     - xyr_RadSDwFlux
    xyr_RadLFlux     = xyr_RadLUwFlux     - xyr_RadLDwFlux
    xyra_DelRadLFlux = xyra_DelRadLUwFlux - xyra_DelRadLDwFlux


    ! 長波フラックスの補正 ( 地表フラックス分の補正 )
    ! Correct longwave flux ( amount of surface flux )
    !
    ! Lines commented out below will be deleted soon (yot, 2010/10/31).
!!$    do k = 0, kmax
!!$      xyr_RadLFluxCor (:,:,k) = &
!!$        &     xyr_RadLFlux (:,:,k) &
!!$        &   + xyra_DelRadLFlux(:,:,k,0) * xy_DSurfTempDt (:,:) * DelTime
!!$    end do
!!$    do k = 0, kmax
!!$      xyr_RadLFluxCor(:,:,k) = &
!!$        &   xyr_RadLFlux(:,:,k) &
!!$        & + (   xy_DSurfTempDt     * xyra_DelRadLFlux(:,:,k,0)   &
!!$        &     + xyz_DTempDt(:,:,1) * xyra_DelRadLFlux(:,:,k,1) ) &
!!$        &   * DelTime
!!$    end do
    do k = 0, kmax
      xyr_RadLUwFluxCor(:,:,k) =                                   &
        &   xyr_RadLUwFlux(:,:,k)                                  &
        & + (   xy_DSurfTempDt     * xyra_DelRadLUwFlux(:,:,k,0)   &
        &     + xyz_DTempDt(:,:,1) * xyra_DelRadLUwFlux(:,:,k,1) ) &
        &   * DelTime
      xyr_RadLDwFluxCor(:,:,k) =                                   &
        &   xyr_RadLDwFlux(:,:,k)                                  &
        & + (   xy_DSurfTempDt     * xyra_DelRadLDwFlux(:,:,k,0)   &
        &     + xyz_DTempDt(:,:,1) * xyra_DelRadLDwFlux(:,:,k,1) ) &
        &   * DelTime
    end do
    xyr_RadLFluxCor = xyr_RadLUwFluxCor - xyr_RadLDwFluxCor


    ! ヒストリデータ出力
    ! History data output
    !
    call HistoryAutoPut( TimeN, 'OLR', xyr_RadLFluxCor(:,:,kmax) )
    call HistoryAutoPut( TimeN, 'SLR', xyr_RadLFluxCor(:,:,0)    )
    call HistoryAutoPut( TimeN, 'OSR', xyr_RadSFlux   (:,:,kmax) )
    call HistoryAutoPut( TimeN, 'SSR', xyr_RadSFlux   (:,:,0)    )
    call HistoryAutoPut( TimeN, 'RadLUWFLX', xyr_RadLUwFluxCor )
    call HistoryAutoPut( TimeN, 'RadLDWFLX', xyr_RadLDwFluxCor )
    call HistoryAutoPut( TimeN, 'RadSUWFLX', xyr_RadSUwFlux    )
    call HistoryAutoPut( TimeN, 'RadSDWFLX', xyr_RadSDwFlux    )


    ! ヒストリデータ出力
    ! History data output
    !
    call HistoryAutoPut( TimeN, 'OLRB', xyr_RadLFlux(:,:,kmax) )
    call HistoryAutoPut( TimeN, 'SLRB', xyr_RadLFlux(:,:,0)    )
    call HistoryAutoPut( TimeN, 'OSRB', xyr_RadSFlux(:,:,kmax) )
    call HistoryAutoPut( TimeN, 'SSRB', xyr_RadSFlux(:,:,0)    )
    call HistoryAutoPut( TimeN, 'RadLUWFLXB', xyr_RadLUwFlux )
    call HistoryAutoPut( TimeN, 'RadLDWFLXB', xyr_RadLDwFlux )
    call HistoryAutoPut( TimeN, 'RadSUWFLXB', xyr_RadSUwFlux )
    call HistoryAutoPut( TimeN, 'RadSDWFLXB', xyr_RadSDwFlux )


    ! 長波フラックスの補正 ( 地表フラックス分の補正 )
    ! Correct longwave flux ( amount of surface flux )
    !
    ! Lines commented out below will be deleted soon (yot, 2010/10/31).
!!$    do k = 0, kmax
!!$      xyr_RadLFluxCor (:,:,k) = &
!!$        &     xyr_RadLFlux (:,:,k) &
!!$        &   + xyra_DelRadLFlux(:,:,k,0) * xy_DSurfTempDt (:,:) * 2.0d0 * DelTime
!!$    end do
!!$    do k = 0, kmax
!!$      xyr_RadLFluxCor(:,:,k) = &
!!$        &   xyr_RadLFlux(:,:,k) &
!!$        & + (   xy_DSurfTempDt     * xyra_DelRadLFlux(:,:,k,0)   &
!!$        &     + xyz_DTempDt(:,:,1) * xyra_DelRadLFlux(:,:,k,1) ) &
!!$        &   * 2.0_DP * DelTime
!!$    end do
    do k = 0, kmax
      xyr_RadLUwFluxCor(:,:,k) =                                   &
        &   xyr_RadLUwFlux(:,:,k)                                  &
        & + (   xy_DSurfTempDt     * xyra_DelRadLUwFlux(:,:,k,0)   &
        &     + xyz_DTempDt(:,:,1) * xyra_DelRadLUwFlux(:,:,k,1) ) &
        &   * 2.0_DP * DelTime
      xyr_RadLDwFluxCor(:,:,k) =                                   &
        &   xyr_RadLDwFlux(:,:,k)                                  &
        & + (   xy_DSurfTempDt     * xyra_DelRadLDwFlux(:,:,k,0)   &
        &     + xyz_DTempDt(:,:,1) * xyra_DelRadLDwFlux(:,:,k,1) ) &
        &   * 2.0_DP * DelTime
    end do
    xyr_RadLFluxCor = xyr_RadLUwFluxCor - xyr_RadLDwFluxCor


    ! ヒストリデータ出力
    ! History data output
    !
    call HistoryAutoPut( TimeN, 'OLRA', xyr_RadLFluxCor(:,:,kmax) )
    call HistoryAutoPut( TimeN, 'SLRA', xyr_RadLFluxCor(:,:,0)    )
    call HistoryAutoPut( TimeN, 'OSRA', xyr_RadSFlux   (:,:,kmax) )
    call HistoryAutoPut( TimeN, 'SSRA', xyr_RadSFlux   (:,:,0)    )
    call HistoryAutoPut( TimeN, 'RadLUWFLXA', xyr_RadLUwFluxCor )
    call HistoryAutoPut( TimeN, 'RadLDWFLXA', xyr_RadLDwFluxCor )
    call HistoryAutoPut( TimeN, 'RadSUWFLXA', xyr_RadSUwFlux    )
    call HistoryAutoPut( TimeN, 'RadSDWFLXA', xyr_RadSDwFlux    )


    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine RadFluxOutput

  !-------------------------------------------------------------------

  subroutine RadUtilsInit
    !
    ! rad_utils モジュールの初期化を行います. 
    ! NAMELIST#rad_utils_nml の読み込みはこの手続きで行われます. 
    !
    ! "rad_utils" module is initialized. 
    ! "NAMELIST#rad_utils_nml" is loaded in this procedure. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 文字列操作
    ! Character handling
    !
    use dc_string, only: toChar

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! 宣言文 ; Declaration statements
    !

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
                              ! IOSTAT of NAMELIST read


    ! NAMELIST 変数群
    ! NAMELIST group name
    !
!!$    namelist /rad_utils_nml/ &
!!$      & DiffFact
!!$      & DelTimeLongValue, DelTimeLongUnit, &
!!$      & DelTimeShortValue, DelTimeShortUnit, &
!!$!
!!$      & LongBandNum, &
!!$      & LongAbsorpCoefQVap, LongAbsorpCoefDryAir, &
!!$      & LongBandWeight, LongPathLengthFact, &
!!$!
!!$      & ShortBandNum, &
!!$      & ShortAbsorpCoefQVap, ShortAbsorpCoefDryAir, &
!!$      & ShortBandWeight, ShortSecScat, &
!!$      & ShortAtmosAlbedo, &
!!$!
!!$      & RstInputFile, RstOutputFile
          !
          ! デフォルト値については初期化手続 "rad_utils#RadInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "rad_utils#RadInit" for the default values. 
          !

    ! 実行文 ; Executable statement
    !

    if ( rad_utils_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !

    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
!!$    if ( trim(namelist_filename) /= '' ) then
!!$      call FileOpen( unit_nml, &          ! (out)
!!$        & namelist_filename, mode = 'r' ) ! (in)
!!$
!!$      rewind( unit_nml )
!!$      read( unit_nml,                     & ! (in)
!!$        & nml = rad_utils_nml,            & ! (out)
!!$        & iostat = iostat_nml )             ! (out)
!!$      close( unit_nml )
!!$
!!$      call NmlutilMsg( iostat_nml, module_name ) ! (in)
!!$    end if


    ! ヒストリデータ出力のためのへの変数登録
    ! Register of variables for history data output
    !
    call HistoryAutoAddVariable( 'OLR', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'outgoing longwave', 'W m-2' )
    call HistoryAutoAddVariable( 'SLR', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface longwave', 'W m-2' )
    call HistoryAutoAddVariable( 'OSR', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'outgoing shortwave', 'W m-2' )
    call HistoryAutoAddVariable( 'SSR', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface shortwave', 'W m-2' )

    call HistoryAutoAddVariable( 'RadLUWFLX', &
      & (/ 'lon ', 'lat ', 'sigm', 'time' /), &
      & 'upward longwave flux', 'W m-2' )
    call HistoryAutoAddVariable( 'RadLDWFLX', &
      & (/ 'lon ', 'lat ', 'sigm', 'time' /), &
      & 'downward longwave flux', 'W m-2' )
    call HistoryAutoAddVariable( 'RadSUWFLX', &
      & (/ 'lon ', 'lat ', 'sigm', 'time' /), &
      & 'upward shortwave flux', 'W m-2' )
    call HistoryAutoAddVariable( 'RadSDWFLX', &
      & (/ 'lon ', 'lat ', 'sigm', 'time' /), &
      & 'downward shortwave flux', 'W m-2' )

    call HistoryAutoAddVariable( 'OLRB', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'outgoing longwave', 'W m-2' )
    call HistoryAutoAddVariable( 'SLRB', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface longwave', 'W m-2' )
    call HistoryAutoAddVariable( 'OSRB', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'outgoing shortwave', 'W m-2' )
    call HistoryAutoAddVariable( 'SSRB', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface shortwave', 'W m-2' )

    call HistoryAutoAddVariable( 'RadLUWFLXB', &
      & (/ 'lon ', 'lat ', 'sigm', 'time' /), &
      & 'upward longwave flux', 'W m-2' )
    call HistoryAutoAddVariable( 'RadLDWFLXB', &
      & (/ 'lon ', 'lat ', 'sigm', 'time' /), &
      & 'downward longwave flux', 'W m-2' )
    call HistoryAutoAddVariable( 'RadSUWFLXB', &
      & (/ 'lon ', 'lat ', 'sigm', 'time' /), &
      & 'upward shortwave flux', 'W m-2' )
    call HistoryAutoAddVariable( 'RadSDWFLXB', &
      & (/ 'lon ', 'lat ', 'sigm', 'time' /), &
      & 'downward shortwave flux', 'W m-2' )

    call HistoryAutoAddVariable( 'OLRA', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'outgoing longwave', 'W m-2' )
    call HistoryAutoAddVariable( 'SLRA', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface longwave', 'W m-2' )
    call HistoryAutoAddVariable( 'OSRA', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'outgoing shortwave', 'W m-2' )
    call HistoryAutoAddVariable( 'SSRA', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface shortwave', 'W m-2' )

    call HistoryAutoAddVariable( 'RadLUWFLXA', &
      & (/ 'lon ', 'lat ', 'sigm', 'time' /), &
      & 'upward longwave flux', 'W m-2' )
    call HistoryAutoAddVariable( 'RadLDWFLXA', &
      & (/ 'lon ', 'lat ', 'sigm', 'time' /), &
      & 'downward longwave flux', 'W m-2' )
    call HistoryAutoAddVariable( 'RadSUWFLXA', &
      & (/ 'lon ', 'lat ', 'sigm', 'time' /), &
      & 'upward shortwave flux', 'W m-2' )
    call HistoryAutoAddVariable( 'RadSDWFLXA', &
      & (/ 'lon ', 'lat ', 'sigm', 'time' /), &
      & 'downward shortwave flux', 'W m-2' )

    call HistoryAutoAddVariable( 'DTempDtRadL', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'long wave radiative heating rate', 'K s-1' )
    call HistoryAutoAddVariable( 'DTempDtRadS', &
      & (/ 'lon ', 'lat ', 'sig ', 'time' /), &
      & 'short wave radiative heating rate', 'K s-1' )


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
!!$    call MessageNotify( 'M', module_name, 'DiffFact = %f', d = (/ DiffFact /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    rad_utils_inited = .true.

  end subroutine RadUtilsInit

  !-------------------------------------------------------------------

end module rad_utils
