!= ˰漾λ
!
!= Evaluate saturation specific humidity
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: saturate.F90,v 1.7 2015/01/29 12:07:16 yot Exp $
! Tag Name::  $Name: dcpam5-20150214 $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module saturate
  !
  != ˰漾λ
  !
  != Evaluate saturation specific humidity
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! ˰漾˰漾βʬͤ򻻽Фޤ. 
  !
  ! Saturation specific humidity and temperature derivative of it are calculated. 
  !
  ! ˰漾η׻ˤϥǥեȤǤ, Dennou AGCM ѤѤ
  ! (saturate_DennouAGCM ). 
  ! ޤ, Config.mk  CPPFLAGS  -DLIB_SATURATE_NHA1992 ꤹ
  ! Nakajima et al. (1992) Ѥ (saturate_tnha1992 ). 
  !
  ! By default, a formula used by Dennou AGCM is used for calculation of 
  ! saturation specific humidity (See "saturate_DennouAGCM"). 
  ! If "-DLIB_SATURATE_NHA1992" is specified to "CPPFLAGS" in Config.mk, 
  ! Nakajima et al. (1992) is used (See "saturate_nha1992"). 
  !
  !== References
  !
  !== Procedures List
  !
  ! CalcQVapSat            :: ˰漾η׻
  ! CalcDQVapSatDTemp      :: ˰漾βʬη׻
  ! ------------  :: ------------
  ! CalcQVapSat            :: Calculate saturation specific humidity
  ! CalcDQVapSatDTemp      :: Calculate temperature derivative of saturation specific humidity
  !
  !--
  !== NAMELIST
  !
  ! NAMELIST#saturate_nml
  !++

  ! ⥸塼 ; USE statements
  !

  ! ̷ѥ᥿
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! ټ¿. Double precision. 
    &                 STRING     ! ʸ.       Strings. 

  ! å
  ! Message output
  !
  use dc_message, only: MessageNotify


#ifdef LIB_SATURATE_NHA1992
  ! Nakajima et al. (1992) Ѥ˰漾λ
  ! Evaluate saturation specific humidity with Nakajima et al. (1992)
  !
  use saturate_nha1992, only:                             &
#elif LIB_SATURATE_DENNOUAGCM
  ! Dennou AGCM μѤ˰漾λ
  ! Evaluate saturation specific humidity with method used by Dennou AGCM
  !
  use saturate_DennouAGCM, only:                          &
#else
  ! Dennou AGCM μѤ˰漾λ
  ! Evaluate saturation specific humidity with method used by Dennou AGCM
  !
  use saturate_DennouAGCM, only:                          &
#endif
    & xyz_CalcQVapSatOnLiq,       &
    & xyz_CalcDQVapSatDTempOnLiq, &
    & xyz_CalcQVapSatOnSol,       &
    & xyz_CalcDQVapSatDTempOnSol, &
    & SaturateInitCore => SaturateInit


  ! ʸ ; Declaration statements
  !
  implicit none
  private

  ! ³
  ! Public procedure
  !
  public:: SaturateWatFraction

  public:: xyz_CalcQVapSat
  public:: xyz_CalcDQVapSatDTemp
  public:: xy_CalcQVapSat
  public:: xy_CalcDQVapSatDTemp
  public:: a_CalcQVapSat
  public:: a_CalcDQVapSatDTemp
  public:: CalcQVapSat
  public:: CalcDQVapSatDTemp

  public:: xyz_CalcQVapSatOnLiq
  public:: xyz_CalcDQVapSatDTempOnLiq
  public:: xy_CalcQVapSatOnLiq
  public:: xy_CalcDQVapSatDTempOnLiq
  public:: a_CalcQVapSatOnLiq
  public:: a_CalcDQVapSatDTempOnLiq
  public:: CalcQVapSatOnLiq
  public:: CalcDQVapSatDTempOnLiq

  public:: xyz_CalcQVapSatOnSol
  public:: xyz_CalcDQVapSatDTempOnSol
  public:: xy_CalcQVapSatOnSol
  public:: xy_CalcDQVapSatDTempOnSol
  public:: a_CalcQVapSatOnSol
  public:: a_CalcDQVapSatDTempOnSol
  public:: CalcQVapSatOnSol
  public:: CalcDQVapSatDTempOnSol

  public:: xy_CalcSfcQVapSat
  public:: xy_CalcSfcDQVapSatDTemp

  public:: SaturateInit


  ! ѿ
  ! Public variables
  !
  logical, save, public:: saturate_inited = .false.
                              ! ե饰. 
                              ! Initialization flag

  ! ѿ
  ! Private variables
  !
  integer , save        :: IDWatIceFracMethod
  integer , parameter   :: IDWatIceFracMethodLin  = 1
  integer , parameter   :: IDWatIceFracMethodQuad = 2
  real(DP), save        :: TempWatLim
  real(DP), save        :: TempIceLim


  character(*), parameter:: saturate_scheme = &
#ifdef LIB_SATURATE_NHA1992
    & 'Nakajima et al. (1992)'
#elif LIB_SATURATE_DENNOUAGCM
    & 'Dennou AGCM'
#else
    & 'Dennou AGCM'
#endif
  character(*), parameter:: module_name = 'saturate'
                              ! ⥸塼̾. 
                              ! Module name

  character(*), parameter:: version = &
    & '$Name: dcpam5-20150214 $' // &
    & '$Id: saturate.F90,v 1.7 2015/01/29 12:07:16 yot Exp $'
                              ! ⥸塼ΥС
                              ! Module version

  interface SaturateWatFraction
    module procedure &
      & SaturateWatFraction3D, &
      & SaturateWatFraction0D
  end interface SaturateWatFraction

  !--------------------------------------------------------------------------------------

contains

  !--------------------------------------------------------------------------------------

  subroutine SaturateWatFraction3D( &
    & xyz_Temp,                     & ! (in )
    & xyz_WatFrac                   & ! (out)
    & )

    ! USE statements
    !

!!$    real(DP), intent(in ) :: xyz_Temp   (0:imax-1, 1:jmax, 1:kmax)
!!$    real(DP), intent(out) :: xyz_WatFrac(0:imax-1, 1:jmax, 1:kmax)
    real(DP), intent(in ) :: xyz_Temp   (:,:,:)
    real(DP), intent(out) :: xyz_WatFrac(:,:,:)


    ! ¹ʸ ; Executable statement
    !

    ! ǧ
    ! Initialization check
    !
    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


!!$    if ( FlagSnow ) then

    if ( TempWatLim == TempIceLim ) then
      xyz_WatFrac = ( sign( 1.0_DP, xyz_Temp - TempWatLim ) + 1.0_DP ) / 2.0_DP
    else
      select case ( IDWatIceFracMethod )
      case ( IDWatIceFracMethodLin  )
        xyz_WatFrac =   ( xyz_Temp - TempIceLim ) / ( TempWatLim - TempIceLim )
      case ( IDWatIceFracMethodQuad )
        xyz_WatFrac = ( max( xyz_Temp - TempIceLim, 0.0_DP ) / ( TempWatLim - TempIceLim ) )**2
      end select
      xyz_WatFrac = max( min( xyz_WatFrac, 1.0_DP ), 0.0_DP )
    end if

!!$    else
!!$
!!$      xyz_WatFrac = 1.0_DP
!!$
!!$    end if


  end subroutine SaturateWatFraction3D

  !--------------------------------------------------------------------------------------

  subroutine SaturateWatFraction0D( &
    & Temp,                       & ! (in )
    & WatFrac                     & ! (out)
    & )

    ! USE statements
    !

    real(DP), intent(in ) :: Temp
    real(DP), intent(out) :: WatFrac


    real(DP) :: xyz_Temp   (1,1,1)
    real(DP) :: xyz_WatFrac(1,1,1)

    ! ¹ʸ ; Executable statement
    !

    ! ǧ
    ! Initialization check
    !
    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp = Temp
    call SaturateWatFraction3D( &
      & xyz_Temp,               & ! (in )
      & xyz_WatFrac             & ! (out)
      & )
    WatFrac = xyz_WatFrac(1,1,1)


  end subroutine SaturateWatFraction0D

  !-------------------------------------------------------------------
  ! [Mixed phase]
  !-------------------------------------------------------------------

  function xyz_CalcQVapSat( xyz_Temp, xyz_Press ) result( xyz_QVapSat )
    !
    !  *Temp* ȵ *Press* Ѥ, 
    ! ˰漾 *QVapSat* ޤ. 
    !
    ! Calculate saturation specific humidity *QVapSat* using
    ! temperature *Temp* and air pressure *Press*. 
    !

    ! ⥸塼 ; USE statements
    !


    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xyz_Temp (:,:,:)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: xyz_Press(:,:,:)
                              ! $ p $ . . Air pressure

    real(DP):: xyz_QVapSat(1:size(xyz_Temp,1), 1:size(xyz_Temp,2), 1:size(xyz_Temp,3))
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_QVapSatOnLiq(1:size(xyz_Temp,1), 1:size(xyz_Temp,2), 1:size(xyz_Temp,3))
    real(DP):: xyz_QVapSatOnSol(1:size(xyz_Temp,1), 1:size(xyz_Temp,2), 1:size(xyz_Temp,3))
    real(DP):: xyz_WatFrac     (1:size(xyz_Temp,1), 1:size(xyz_Temp,2), 1:size(xyz_Temp,3))


    ! ¹ʸ ; Executable statement
    !

    ! ǧ
    ! Initialization check
    !

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_QVapSatOnLiq = xyz_CalcQVapSatOnLiq( xyz_Temp, xyz_Press )
    xyz_QVapSatOnSol = xyz_CalcQVapSatOnSol( xyz_Temp, xyz_Press )

    call SaturateWatFraction( &
      & xyz_Temp,             & ! (in )
      & xyz_WatFrac           & ! (out)
      & )

    xyz_QVapSat =                                     &
      &   xyz_WatFrac              * xyz_QVapSatOnLiq &
      & + ( 1.0_DP - xyz_WatFrac ) * xyz_QVapSatOnSol


  end function xyz_CalcQVapSat

  !--------------------------------------------------------------------------------------

  function xyz_CalcDQVapSatDTemp( xyz_Temp, xyz_QVapSat ) result( xyz_DQVapSatDTemp )
    !
    !  *Temp* ˰漾 *QVapSat* Ѥ, 
    ! ˰漾βʬ *DQVapSatDTemp* ޤ. 
    !
    ! Calculate temperature derivative of saturation specific humidity 
    ! *DQVapSatDTemp* using
    ! temperature *Temp* and saturation specific humidity *QVapSat*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xyz_Temp   (:,:,:)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: xyz_QVapSat(:,:,:)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: xyz_DQVapSatDTemp(size(xyz_Temp,1), size(xyz_Temp,2), size(xyz_Temp,3))
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_DQVapSatDTempOnLiq(size(xyz_Temp,1), size(xyz_Temp,2), size(xyz_Temp,3))
    real(DP):: xyz_DQVapSatDTempOnSol(size(xyz_Temp,1), size(xyz_Temp,2), size(xyz_Temp,3))
    real(DP):: xyz_WatFrac           (size(xyz_Temp,1), size(xyz_Temp,2), size(xyz_Temp,3))


    ! ¹ʸ ; Executable statement
    !

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_DQVapSatDTempOnLiq = xyz_CalcDQVapSatDTempOnLiq( xyz_Temp, xyz_QVapSat )
    xyz_DQVapSatDTempOnSol = xyz_CalcDQVapSatDTempOnSol( xyz_Temp, xyz_QVapSat )

    call SaturateWatFraction( &
      & xyz_Temp,             & ! (in )
      & xyz_WatFrac           & ! (out)
      & )

    xyz_DQVapSatDTemp =                                     &
      &   xyz_WatFrac              * xyz_DQVapSatDTempOnLiq &
      & + ( 1.0_DP - xyz_WatFrac ) * xyz_DQVapSatDTempOnSol


  end function xyz_CalcDQVapSatDTemp

  !-------------------------------------------------------------------
  ! [Mixed phase] Interfaces for 2D, 1D, and scalar variables
  !-------------------------------------------------------------------

  function xy_CalcQVapSat( xy_Temp, xy_Press ) result( xy_QVapSat )
    !
    !  *Temp* ȵ *Press* Ѥ, 
    ! ˰漾 *QVapSat* ޤ. 
    !
    ! Calculate saturation specific humidity *QVapSat* using
    ! temperature *Temp* and air pressure *Press*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xy_Temp (:,:)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: xy_Press(:,:)
                              ! $ p $ . . Air pressure

    real(DP):: xy_QVapSat(size(xy_Temp,1), size(xy_Temp,2))
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    ! ѿ
    ! Work variables
    !
    real(DP) :: xyz_Temp   (size(xy_Temp,1),size(xy_Temp,2),1)
    real(DP) :: xyz_Press  (size(xy_Temp,1),size(xy_Temp,2),1)
    real(DP) :: xyz_QVapSat(size(xy_Temp,1),size(xy_Temp,2),1)


    ! ¹ʸ ; Executable statement
    !

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp (:,:,1) = xy_Temp
    xyz_Press(:,:,1) = xy_Press

    xyz_QVapSat = xyz_CalcQVapSat( xyz_Temp, xyz_Press )

    xy_QVapSat = xyz_QVapSat(:,:,1)


  end function xy_CalcQVapSat

  !-------------------------------------------------------------------

  function xy_CalcDQVapSatDTemp( xy_Temp, xy_QVapSat ) result( xy_DQVapSatDTemp )
    !
    !  *Temp* ˰漾 *QVapSat* Ѥ, 
    ! ˰漾βʬ *DQVapSatDTemp* ޤ. 
    !
    ! Calculate temperature derivative of saturation specific humidity 
    ! *DQVapSatDTemp* using
    ! temperature *Temp* and saturation specific humidity *QVapSat*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xy_Temp   (:,:)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: xy_QVapSat(:,:)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: xy_DQVapSatDTemp(size(xy_Temp,1), size(xy_Temp,2))
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_Temp         (size(xy_Temp,1), size(xy_Temp,2), 1)
    real(DP):: xyz_QVapSat      (size(xy_Temp,1), size(xy_Temp,2), 1)
    real(DP):: xyz_DQVapSatDTemp(size(xy_Temp,1), size(xy_Temp,2), 1)

    ! ¹ʸ ; Executable statement
    !

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp   (:,:,1) = xy_Temp
    xyz_QVapSat(:,:,1) = xy_QVapSat

    xyz_DQVapSatDTemp = xyz_CalcDQVapSatDTemp( xyz_Temp, xyz_QVapSat )

    xy_DQVapSatDTemp = xyz_DQVapSatDTemp(:,:,1)


  end function xy_CalcDQVapSatDTemp

  !--------------------------------------------------------------------------------------

  function a_CalcQVapSat( a_Temp, a_Press ) result( a_QVapSat )
    !
    !  *Temp* ȵ *Press* Ѥ, 
    ! ˰漾 *QVapSat* ޤ. 
    !
    ! Calculate saturation specific humidity *QVapSat* using
    ! temperature *Temp* and air pressure *Press*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: a_Temp (:)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: a_Press(:)
                              ! $ p $ . . Air pressure

    real(DP):: a_QVapSat(size(a_Temp,1))
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_Temp   (size(a_Temp,1), 1, 1)
                              ! $ T $ . . Temperature
    real(DP):: xyz_Press  (size(a_Temp,1), 1, 1)
                              ! $ p $ . . Air pressure
    real(DP):: xyz_QVapSat(size(a_Temp,1), 1, 1)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp (:,1,1) = a_Temp
    xyz_Press(:,1,1) = a_Press

    xyz_QVapSat = xyz_CalcQVapSat( xyz_Temp, xyz_Press )

    a_QVapSat = xyz_QVapSat(:,1,1)


  end function a_CalcQVapSat

  !--------------------------------------------------------------------------------------

  function a_CalcDQVapSatDTemp( a_Temp, a_QVapSat ) result( a_DQVapSatDTemp )
    !
    !  *Temp* ˰漾 *QVapSat* Ѥ, 
    ! ˰漾βʬ *DQVapSatDTemp* ޤ. 
    !
    ! Calculate temperature derivative of saturation specific humidity 
    ! *DQVapSatDTemp* using
    ! temperature *Temp* and saturation specific humidity *QVapSat*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: a_Temp   (:)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: a_QVapSat(:)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: a_DQVapSatDTemp(size(a_Temp,1))
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_Temp   (size(a_Temp,1), 1:1, 1:1)
                              ! $ T $ . . Temperature
    real(DP):: xyz_QVapSat(size(a_Temp,1), 1:1, 1:1)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: xyz_DQVapSatDTemp(size(a_Temp,1), 1:1, 1:1)
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 


    ! ¹ʸ ; Executable statement
    !

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp   (:,1,1) = a_Temp
    xyz_QVapSat(:,1,1) = a_QVapSat

    xyz_DQVapSatDTemp = xyz_CalcDQVapSatDTemp( xyz_Temp, xyz_QVapSat )

    a_DQVapSatDTemp = xyz_DQVapSatDTemp(:,1,1)


  end function a_CalcDQVapSatDTemp

  !--------------------------------------------------------------------------------------

  function CalcQVapSat( Temp, Press ) result( QVapSat )
    !
    !  *Temp* ȵ *Press* Ѥ, 
    ! ˰漾 *QVapSat* ޤ. 
    !
    ! Calculate saturation specific humidity *QVapSat* using
    ! temperature *Temp* and air pressure *Press*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: Temp
                              ! $ T $ . . Temperature
    real(DP), intent(in):: Press
                              ! $ p $ . . Air pressure

    real(DP):: QVapSat
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_Temp   (1, 1, 1)
                              ! $ T $ . . Temperature
    real(DP):: xyz_Press  (1, 1, 1)
                              ! $ p $ . . Air pressure
    real(DP):: xyz_QVapSat(1, 1, 1)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp (:,1,1) = Temp
    xyz_Press(:,1,1) = Press

    xyz_QVapSat = xyz_CalcQVapSat( xyz_Temp, xyz_Press )

    QVapSat = xyz_QVapSat(1,1,1)


  end function CalcQVapSat

  !--------------------------------------------------------------------------------------

  function CalcDQVapSatDTemp( Temp, QVapSat ) result( DQVapSatDTemp )
    !
    !  *Temp* ˰漾 *QVapSat* Ѥ, 
    ! ˰漾βʬ *DQVapSatDTemp* ޤ. 
    !
    ! Calculate temperature derivative of saturation specific humidity 
    ! *DQVapSatDTemp* using
    ! temperature *Temp* and saturation specific humidity *QVapSat*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: Temp
                              ! $ T $ . . Temperature
    real(DP), intent(in):: QVapSat
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: DQVapSatDTemp
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_Temp   (1:1, 1:1, 1:1)
                              ! $ T $ . . Temperature
    real(DP):: xyz_QVapSat(1:1, 1:1, 1:1)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: xyz_DQVapSatDTemp(1:1, 1:1, 1:1)
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 


    ! ¹ʸ ; Executable statement
    !

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp   (1,1,1) = Temp
    xyz_QVapSat(1,1,1) = QVapSat

    xyz_DQVapSatDTemp = xyz_CalcDQVapSatDTemp( xyz_Temp, xyz_QVapSat )

    DQVapSatDTemp = xyz_DQVapSatDTemp(1,1,1)


  end function CalcDQVapSatDTemp

  !-------------------------------------------------------------------
  ! [On liquid] Interfaces for 2D, 1D, and scalar variables
  !-------------------------------------------------------------------

  function xy_CalcQVapSatOnLiq( xy_Temp, xy_Press ) result( xy_QVapSat )
    !
    !  *Temp* ȵ *Press* Ѥ, 
    ! ˰漾 *QVapSat* ޤ. 
    !
    ! Calculate saturation specific humidity *QVapSat* using
    ! temperature *Temp* and air pressure *Press*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xy_Temp (:,:)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: xy_Press(:,:)
                              ! $ p $ . . Air pressure

    real(DP):: xy_QVapSat(size(xy_Temp,1), size(xy_Temp,2))
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    ! ѿ
    ! Work variables
    !
    real(DP) :: xyz_Temp   (size(xy_Temp,1),size(xy_Temp,2),1)
    real(DP) :: xyz_Press  (size(xy_Temp,1),size(xy_Temp,2),1)
    real(DP) :: xyz_QVapSat(size(xy_Temp,1),size(xy_Temp,2),1)


    ! ¹ʸ ; Executable statement
    !

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp (:,:,1) = xy_Temp
    xyz_Press(:,:,1) = xy_Press

    xyz_QVapSat = xyz_CalcQVapSatOnLiq( xyz_Temp, xyz_Press )

    xy_QVapSat = xyz_QVapSat(:,:,1)


  end function xy_CalcQVapSatOnLiq

  !-------------------------------------------------------------------

  function xy_CalcDQVapSatDTempOnLiq( xy_Temp, xy_QVapSat ) result( xy_DQVapSatDTemp )
    !
    !  *Temp* ˰漾 *QVapSat* Ѥ, 
    ! ˰漾βʬ *DQVapSatDTemp* ޤ. 
    !
    ! Calculate temperature derivative of saturation specific humidity 
    ! *DQVapSatDTemp* using
    ! temperature *Temp* and saturation specific humidity *QVapSat*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xy_Temp   (:,:)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: xy_QVapSat(:,:)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: xy_DQVapSatDTemp(size(xy_Temp,1), size(xy_Temp,2))
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_Temp         (size(xy_Temp,1), size(xy_Temp,2), 1)
    real(DP):: xyz_QVapSat      (size(xy_Temp,1), size(xy_Temp,2), 1)
    real(DP):: xyz_DQVapSatDTemp(size(xy_Temp,1), size(xy_Temp,2), 1)

    ! ¹ʸ ; Executable statement
    !

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp   (:,:,1) = xy_Temp
    xyz_QVapSat(:,:,1) = xy_QVapSat

    xyz_DQVapSatDTemp = xyz_CalcDQVapSatDTempOnLiq( xyz_Temp, xyz_QVapSat )

    xy_DQVapSatDTemp = xyz_DQVapSatDTemp(:,:,1)


  end function xy_CalcDQVapSatDTempOnLiq

  !--------------------------------------------------------------------------------------

  function a_CalcQVapSatOnLiq( a_Temp, a_Press ) result( a_QVapSat )
    !
    !  *Temp* ȵ *Press* Ѥ, 
    ! ˰漾 *QVapSat* ޤ. 
    !
    ! Calculate saturation specific humidity *QVapSat* using
    ! temperature *Temp* and air pressure *Press*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: a_Temp (:)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: a_Press(:)
                              ! $ p $ . . Air pressure

    real(DP):: a_QVapSat(size(a_Temp,1))
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_Temp   (size(a_Temp,1), 1, 1)
                              ! $ T $ . . Temperature
    real(DP):: xyz_Press  (size(a_Temp,1), 1, 1)
                              ! $ p $ . . Air pressure
    real(DP):: xyz_QVapSat(size(a_Temp,1), 1, 1)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp (:,1,1) = a_Temp
    xyz_Press(:,1,1) = a_Press

    xyz_QVapSat = xyz_CalcQVapSatOnLiq( xyz_Temp, xyz_Press )

    a_QVapSat = xyz_QVapSat(:,1,1)


  end function a_CalcQVapSatOnLiq

  !--------------------------------------------------------------------------------------

  function a_CalcDQVapSatDTempOnLiq( a_Temp, a_QVapSat ) result( a_DQVapSatDTemp )
    !
    !  *Temp* ˰漾 *QVapSat* Ѥ, 
    ! ˰漾βʬ *DQVapSatDTemp* ޤ. 
    !
    ! Calculate temperature derivative of saturation specific humidity 
    ! *DQVapSatDTemp* using
    ! temperature *Temp* and saturation specific humidity *QVapSat*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: a_Temp   (:)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: a_QVapSat(:)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: a_DQVapSatDTemp(size(a_Temp,1))
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_Temp   (size(a_Temp,1), 1:1, 1:1)
                              ! $ T $ . . Temperature
    real(DP):: xyz_QVapSat(size(a_Temp,1), 1:1, 1:1)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: xyz_DQVapSatDTemp(size(a_Temp,1), 1:1, 1:1)
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 


    ! ¹ʸ ; Executable statement
    !

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp   (:,1,1) = a_Temp
    xyz_QVapSat(:,1,1) = a_QVapSat

    xyz_DQVapSatDTemp = xyz_CalcDQVapSatDTempOnLiq( xyz_Temp, xyz_QVapSat )

    a_DQVapSatDTemp = xyz_DQVapSatDTemp(:,1,1)


  end function a_CalcDQVapSatDTempOnLiq

  !--------------------------------------------------------------------------------------

  function CalcQVapSatOnLiq( Temp, Press ) result( QVapSat )
    !
    !  *Temp* ȵ *Press* Ѥ, 
    ! ˰漾 *QVapSat* ޤ. 
    !
    ! Calculate saturation specific humidity *QVapSat* using
    ! temperature *Temp* and air pressure *Press*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: Temp
                              ! $ T $ . . Temperature
    real(DP), intent(in):: Press
                              ! $ p $ . . Air pressure

    real(DP):: QVapSat
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_Temp   (1, 1, 1)
                              ! $ T $ . . Temperature
    real(DP):: xyz_Press  (1, 1, 1)
                              ! $ p $ . . Air pressure
    real(DP):: xyz_QVapSat(1, 1, 1)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp (:,1,1) = Temp
    xyz_Press(:,1,1) = Press

    xyz_QVapSat = xyz_CalcQVapSatOnLiq( xyz_Temp, xyz_Press )

    QVapSat = xyz_QVapSat(1,1,1)


  end function CalcQVapSatOnLiq

  !--------------------------------------------------------------------------------------

  function CalcDQVapSatDTempOnLiq( Temp, QVapSat ) result( DQVapSatDTemp )
    !
    !  *Temp* ˰漾 *QVapSat* Ѥ, 
    ! ˰漾βʬ *DQVapSatDTemp* ޤ. 
    !
    ! Calculate temperature derivative of saturation specific humidity 
    ! *DQVapSatDTemp* using
    ! temperature *Temp* and saturation specific humidity *QVapSat*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: Temp
                              ! $ T $ . . Temperature
    real(DP), intent(in):: QVapSat
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: DQVapSatDTemp
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_Temp   (1:1, 1:1, 1:1)
                              ! $ T $ . . Temperature
    real(DP):: xyz_QVapSat(1:1, 1:1, 1:1)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: xyz_DQVapSatDTemp(1:1, 1:1, 1:1)
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 


    ! ¹ʸ ; Executable statement
    !

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp   (1,1,1) = Temp
    xyz_QVapSat(1,1,1) = QVapSat

    xyz_DQVapSatDTemp = xyz_CalcDQVapSatDTempOnLiq( xyz_Temp, xyz_QVapSat )

    DQVapSatDTemp = xyz_DQVapSatDTemp(1,1,1)


  end function CalcDQVapSatDTempOnLiq

  !-------------------------------------------------------------------
  ! [On ice] Interfaces for 2D, 1D, and scalar variables
  !-------------------------------------------------------------------

  function xy_CalcQVapSatOnSol( xy_Temp, xy_Press ) result( xy_QVapSat )
    !
    !  *Temp* ȵ *Press* Ѥ, 
    ! ˰漾 *QVapSat* ޤ. 
    !
    ! Calculate saturation specific humidity *QVapSat* using
    ! temperature *Temp* and air pressure *Press*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xy_Temp (:,:)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: xy_Press(:,:)
                              ! $ p $ . . Air pressure

    real(DP):: xy_QVapSat(size(xy_Temp,1), size(xy_Temp,2))
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    ! ѿ
    ! Work variables
    !
    real(DP) :: xyz_Temp   (size(xy_Temp,1),size(xy_Temp,2),1)
    real(DP) :: xyz_Press  (size(xy_Temp,1),size(xy_Temp,2),1)
    real(DP) :: xyz_QVapSat(size(xy_Temp,1),size(xy_Temp,2),1)


    ! ¹ʸ ; Executable statement
    !

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp (:,:,1) = xy_Temp
    xyz_Press(:,:,1) = xy_Press

    xyz_QVapSat = xyz_CalcQVapSatOnSol( xyz_Temp, xyz_Press )

    xy_QVapSat = xyz_QVapSat(:,:,1)


  end function xy_CalcQVapSatOnSol

  !-------------------------------------------------------------------

  function xy_CalcDQVapSatDTempOnSol( xy_Temp, xy_QVapSat ) result( xy_DQVapSatDTemp )
    !
    !  *Temp* ˰漾 *QVapSat* Ѥ, 
    ! ˰漾βʬ *DQVapSatDTemp* ޤ. 
    !
    ! Calculate temperature derivative of saturation specific humidity 
    ! *DQVapSatDTemp* using
    ! temperature *Temp* and saturation specific humidity *QVapSat*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xy_Temp   (:,:)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: xy_QVapSat(:,:)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: xy_DQVapSatDTemp(size(xy_Temp,1), size(xy_Temp,2))
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_Temp         (size(xy_Temp,1), size(xy_Temp,2), 1)
    real(DP):: xyz_QVapSat      (size(xy_Temp,1), size(xy_Temp,2), 1)
    real(DP):: xyz_DQVapSatDTemp(size(xy_Temp,1), size(xy_Temp,2), 1)

    ! ¹ʸ ; Executable statement
    !

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp   (:,:,1) = xy_Temp
    xyz_QVapSat(:,:,1) = xy_QVapSat

    xyz_DQVapSatDTemp = xyz_CalcDQVapSatDTempOnSol( xyz_Temp, xyz_QVapSat )

    xy_DQVapSatDTemp = xyz_DQVapSatDTemp(:,:,1)


  end function xy_CalcDQVapSatDTempOnSol

  !--------------------------------------------------------------------------------------

  function a_CalcQVapSatOnSol( a_Temp, a_Press ) result( a_QVapSat )
    !
    !  *Temp* ȵ *Press* Ѥ, 
    ! ˰漾 *QVapSat* ޤ. 
    !
    ! Calculate saturation specific humidity *QVapSat* using
    ! temperature *Temp* and air pressure *Press*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: a_Temp (:)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: a_Press(:)
                              ! $ p $ . . Air pressure

    real(DP):: a_QVapSat(size(a_Temp,1))
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_Temp   (size(a_Temp,1), 1, 1)
                              ! $ T $ . . Temperature
    real(DP):: xyz_Press  (size(a_Temp,1), 1, 1)
                              ! $ p $ . . Air pressure
    real(DP):: xyz_QVapSat(size(a_Temp,1), 1, 1)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp (:,1,1) = a_Temp
    xyz_Press(:,1,1) = a_Press

    xyz_QVapSat = xyz_CalcQVapSatOnSol( xyz_Temp, xyz_Press )

    a_QVapSat = xyz_QVapSat(:,1,1)


  end function a_CalcQVapSatOnSol

  !--------------------------------------------------------------------------------------

  function a_CalcDQVapSatDTempOnSol( a_Temp, a_QVapSat ) result( a_DQVapSatDTemp )
    !
    !  *Temp* ˰漾 *QVapSat* Ѥ, 
    ! ˰漾βʬ *DQVapSatDTemp* ޤ. 
    !
    ! Calculate temperature derivative of saturation specific humidity 
    ! *DQVapSatDTemp* using
    ! temperature *Temp* and saturation specific humidity *QVapSat*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: a_Temp   (:)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: a_QVapSat(:)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: a_DQVapSatDTemp(size(a_Temp,1))
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_Temp   (size(a_Temp,1), 1:1, 1:1)
                              ! $ T $ . . Temperature
    real(DP):: xyz_QVapSat(size(a_Temp,1), 1:1, 1:1)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: xyz_DQVapSatDTemp(size(a_Temp,1), 1:1, 1:1)
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 


    ! ¹ʸ ; Executable statement
    !

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp   (:,1,1) = a_Temp
    xyz_QVapSat(:,1,1) = a_QVapSat

    xyz_DQVapSatDTemp = xyz_CalcDQVapSatDTempOnSol( xyz_Temp, xyz_QVapSat )

    a_DQVapSatDTemp = xyz_DQVapSatDTemp(:,1,1)


  end function a_CalcDQVapSatDTempOnSol

  !--------------------------------------------------------------------------------------

  function CalcQVapSatOnSol( Temp, Press ) result( QVapSat )
    !
    !  *Temp* ȵ *Press* Ѥ, 
    ! ˰漾 *QVapSat* ޤ. 
    !
    ! Calculate saturation specific humidity *QVapSat* using
    ! temperature *Temp* and air pressure *Press*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: Temp
                              ! $ T $ . . Temperature
    real(DP), intent(in):: Press
                              ! $ p $ . . Air pressure

    real(DP):: QVapSat
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_Temp   (1, 1, 1)
                              ! $ T $ . . Temperature
    real(DP):: xyz_Press  (1, 1, 1)
                              ! $ p $ . . Air pressure
    real(DP):: xyz_QVapSat(1, 1, 1)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp (:,1,1) = Temp
    xyz_Press(:,1,1) = Press

    xyz_QVapSat = xyz_CalcQVapSatOnSol( xyz_Temp, xyz_Press )

    QVapSat = xyz_QVapSat(1,1,1)


  end function CalcQVapSatOnSol

  !--------------------------------------------------------------------------------------

  function CalcDQVapSatDTempOnSol( Temp, QVapSat ) result( DQVapSatDTemp )
    !
    !  *Temp* ˰漾 *QVapSat* Ѥ, 
    ! ˰漾βʬ *DQVapSatDTemp* ޤ. 
    !
    ! Calculate temperature derivative of saturation specific humidity 
    ! *DQVapSatDTemp* using
    ! temperature *Temp* and saturation specific humidity *QVapSat*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʸ ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: Temp
                              ! $ T $ . . Temperature
    real(DP), intent(in):: QVapSat
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: DQVapSatDTemp
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_Temp   (1:1, 1:1, 1:1)
                              ! $ T $ . . Temperature
    real(DP):: xyz_QVapSat(1:1, 1:1, 1:1)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: xyz_DQVapSatDTemp(1:1, 1:1, 1:1)
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 


    ! ¹ʸ ; Executable statement
    !

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_Temp   (1,1,1) = Temp
    xyz_QVapSat(1,1,1) = QVapSat

    xyz_DQVapSatDTemp = xyz_CalcDQVapSatDTempOnSol( xyz_Temp, xyz_QVapSat )

    DQVapSatDTemp = xyz_DQVapSatDTemp(1,1,1)


  end function CalcDQVapSatDTempOnSol

  !-------------------------------------------------------------------
  ! [Mixed phase] Interfaces for 2D variables (surface)
  !-------------------------------------------------------------------

!!$  function xy_CalcSfcQVapSat( xy_FlagLand, xy_SurfSnow, xy_Temp, xy_Press ) result( xy_QVapSat )
  function xy_CalcSfcQVapSat( xy_SurfSnow, xy_Temp, xy_Press ) result( xy_QVapSat )
    !
    !  *Temp* ȵ *Press* Ѥ, 
    ! ˰漾 *QVapSat* ޤ. 
    !
    ! Calculate saturation specific humidity *QVapSat* using
    ! temperature *Temp* and air pressure *Press*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! , ɹγ
    ! snow/ice fraction
    !
    use snowice_frac, only : CalcSnowFrac


    ! ʸ ; Declaration statements
    !
    implicit none

!!$    logical , intent(in):: xy_FlagLand(:,:)
!!$                              ! Flag for land
    real(DP), intent(in):: xy_SurfSnow(:,:)
                              ! 
    real(DP), intent(in):: xy_Temp (:,:)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: xy_Press(:,:)
                              ! $ p $ . . Air pressure

    real(DP):: xy_QVapSat(1:size(xy_Temp,1), 1:size(xy_Temp,2))
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity

    ! ѿ
    ! Work variables
    !
    real(DP):: xy_QVapSatOnLiq(1:size(xy_Temp,1), 1:size(xy_Temp,2))
    real(DP):: xy_QVapSatOnSol(1:size(xy_Temp,1), 1:size(xy_Temp,2))
    real(DP):: xy_SnowFrac    (1:size(xy_Temp,1), 1:size(xy_Temp,2))


    ! ¹ʸ ; Executable statement
    !

    ! ǧ
    ! Initialization check
    !

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xy_QVapSatOnLiq = xy_CalcQVapSatOnLiq( xy_Temp, xy_Press )
    xy_QVapSatOnSol = xy_CalcQVapSatOnSol( xy_Temp, xy_Press )

    ! , ɹγ
    ! snow/ice fraction
    !
    call CalcSnowFrac(             &
!!$      & xy_FlagLand, xy_SurfSnow,  & ! (in )
      & xy_SurfSnow,               & ! (in )
      & xy_SnowFrac                & ! (out)
      & )

    xy_QVapSat =                                     &
      &   ( 1.0_DP - xy_SnowFrac ) * xy_QVapSatOnLiq &
      & + xy_SnowFrac              * xy_QVapSatOnSol


  end function xy_CalcSfcQVapSat

  !--------------------------------------------------------------------------------------

!!$  function xy_CalcSfcDQVapSatDTemp( xy_FlagLand, xy_SurfSnow, xy_Temp, xy_QVapSat ) result( xy_DQVapSatDTemp )
  function xy_CalcSfcDQVapSatDTemp( xy_SurfSnow, xy_Temp, xy_QVapSat ) result( xy_DQVapSatDTemp )
    !
    !  *Temp* ˰漾 *QVapSat* Ѥ, 
    ! ˰漾βʬ *DQVapSatDTemp* ޤ. 
    !
    ! Calculate temperature derivative of saturation specific humidity 
    ! *DQVapSatDTemp* using
    ! temperature *Temp* and saturation specific humidity *QVapSat*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! , ɹγ
    ! snow/ice fraction
    !
    use snowice_frac, only : CalcSnowFrac


    ! ʸ ; Declaration statements
    !
    implicit none

!!$    logical , intent(in):: xy_FlagLand(:,:)
!!$                              ! Flag for land
    real(DP), intent(in):: xy_SurfSnow(:,:)
                              ! 
    real(DP), intent(in):: xy_Temp   (:,:)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: xy_QVapSat(:,:)
                              ! $ q^{*} $ . ˰漾. Saturation specific humidity
    real(DP):: xy_DQVapSatDTemp(size(xy_Temp,1), size(xy_Temp,2))
                              ! $ \DP{q^{*}}{T} $ . ˰漾βʬ. 
                              ! Temperature derivative of saturation specific humidity. 

    ! ѿ
    ! Work variables
    !
    real(DP):: xy_DQVapSatDTempOnLiq(size(xy_Temp,1), size(xy_Temp,2))
    real(DP):: xy_DQVapSatDTempOnSol(size(xy_Temp,1), size(xy_Temp,2))
    real(DP):: xy_SnowFrac          (size(xy_Temp,1), size(xy_Temp,2))


    ! ¹ʸ ; Executable statement
    !

    if ( .not. saturate_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xy_DQVapSatDTempOnLiq = xy_CalcDQVapSatDTempOnLiq( xy_Temp, xy_QVapSat )
    xy_DQVapSatDTempOnSol = xy_CalcDQVapSatDTempOnSol( xy_Temp, xy_QVapSat )

    ! , ɹγ
    ! snow/ice fraction
    !
    call CalcSnowFrac(             &
!!$      & xy_FlagLand, xy_SurfSnow,  & ! (in )
      & xy_SurfSnow,               & ! (in )
      & xy_SnowFrac                & ! (out)
      & )

    xy_DQVapSatDTemp =                                     &
      &   ( 1.0_DP - xy_SnowFrac ) * xy_DQVapSatDTempOnLiq &
      & + xy_SnowFrac              * xy_DQVapSatDTempOnSol


  end function xy_CalcSfcDQVapSatDTemp

  !-------------------------------------------------------------------

  subroutine SaturateInit
    !
    ! saturate ⥸塼νԤޤ. 
    !--
    ! NAMELIST#saturate_nml ɤ߹ߤϤμ³ǹԤޤ. 
    !++
    !
    ! "saturate" module is initialized. 
    !--
    ! "NAMELIST#saturate_nml" is loaded in this procedure. 
    !++

    ! ⥸塼 ; USE statements
    !

    ! NAMELIST եϤ˴ؤ桼ƥƥ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg

    ! ե
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! ̷ѥ᥿
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! ɸϤֹ. Unit number of standard output

    ! ʸ
    ! Character handling
    !
    use dc_string, only: StoA

    ! ҥȥǡ
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! , ɹγ
    ! snow/ice fraction
    !
    use snowice_frac, only : SnowIceFracInit

    ! ʸ ; Declaration statements
    !
    implicit none

    character(STRING) :: SaturateWatIceFracType

    integer:: unit_nml        ! NAMELIST ե륪ץֹ. 
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST ɤ߹߻ IOSTAT. 
                              ! IOSTAT of NAMELIST read

    ! NAMELIST ѿ
    ! NAMELIST group name
    !
    namelist /saturate_nml/     &
      & SaturateWatIceFracType, &
      & TempWatLim,             &
      & TempIceLim
          !
          ! ǥեͤˤĤƤϽ³ "saturate#SaturateInit" 
          ! Υɤ򻲾ȤΤ. 
          !
          ! Refer to source codes in the initialization procedure
          ! "saturate#SaturateInit" for the default values. 
          !

    ! ¹ʸ ; Executable statement
    !

    if ( saturate_inited ) return

    ! ǥեͤ
    ! Default values settings
    !
    SaturateWatIceFracType = 'Lin'

!!$    TempWatLim          = 273.15_DP
!!$    TempIceLim          = 273.15_DP - 40.0_DP
    TempWatLim          = 0.0_DP
    TempIceLim          = 0.0_DP


    ! NAMELIST ɤ߹
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml, &           ! (in)
        & nml = saturate_nml, &  ! (out)
        & iostat = iostat_nml )   ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
    end if


    select case ( SaturateWatIceFracType )
    case ( 'Lin' )
      IDWatIceFracMethod = IDWatIceFracMethodLin
    case ( 'Quad' )
      IDWatIceFracMethod = IDWatIceFracMethodQuad
    case default
      call MessageNotify( 'E', module_name,         &
        & 'SaturateWatIceFracType=<%c> is not supported.', &
        & c1 = trim(SaturateWatIceFracType) )
    end select


    ! Initialization of modules used in this module
    !

    call SaturateInitCore

    ! , ɹγ
    ! snow/ice fraction
    !
    call SnowIceFracInit


    !  ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, 'Scheme of saturation = %c', &
      & c1 = saturate_scheme )
    call MessageNotify( 'M', module_name, 'SaturateWatIceFracType = %c', c1 = trim(SaturateWatIceFracType) )
    call MessageNotify( 'M', module_name, 'TempWatLim             = %f', d = (/TempWatLim/) )
    call MessageNotify( 'M', module_name, 'TempIceLim             = %f', d = (/TempIceLim/) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    saturate_inited = .true.


  end subroutine SaturateInit

  !-------------------------------------------------------------------

end module saturate
