!= ̤
!
!= Mass fixer
!
! Authors::   Yasuhiro Morikawa, Yukiko Yamada, Yoshiyuki O. Takahashi
! Version::   $Id: mass_fixer.f90,v 1.12 2013-10-06 13:37:07 yot Exp $ 
! Tag Name::  $Name: dcpam5-20140204-5 $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module mass_fixer
  !
  != ̤
  !
  != Mass fixer
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! ̤ޤ.
  !
  ! Fix masses
  !
  !== Procedures List
  !
  ! MassFixer   :: ̤
  ! ----------  :: ---------------
  ! MassFixer   :: Fix masses
  !
  !--
  !== NAMELIST
  !
  ! NAMELIST#mass_fixer_nml
  !
  !== References
  !
  ! Williamson, D. L., and J. G. Olson (1994), Climte simulations with a 
  ! semi-Lagrangian version of the NCAR community climate model, 122, 
  ! 1594-1610.
  !
  !++

  ! ⥸塼 ; USE statements
  !

  ! ʻ
  ! Grid points settings
  !
  use gridset, only: imax, & ! ٳʻ. 
                             ! Number of grid points in longitude
    &                jmax, & ! ٳʻ. 
                             ! Number of grid points in latitude
    &                kmax    ! ľؿ. 
                             ! Number of vertical level

  ! ˴ؤ
  ! Settings of array for atmospheric composition
  !
  use composition, only: &
    &                    ncmax
                              ! ʬο
                              ! Number of composition

  ! ̷ѥ᥿
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! ټ¿. Double precision. 
    &                 STRING     ! ʸ.       Strings. 

  ! å
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! ʸ ; Declaration statements
  !
  implicit none
  private

  ! ³
  ! Public procedure
  !
  public :: MassFixerWO94
  public :: MassFixer
  public :: MassFixerLayer
  public :: MassFixerColumn
  public :: MassFixerInit

  ! ѿ
  ! Public variables
  !

  ! ѿ
  ! Private variables
  !
  logical, save :: mass_fixer_inited = .false.
                              ! ե饰. 
                              ! Initialization flag

  character(*), parameter:: module_name = 'mass_fixer'
                              ! ⥸塼̾. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name: dcpam5-20140204-5 $' // &
    & '$Id: mass_fixer.f90,v 1.12 2013-10-06 13:37:07 yot Exp $'
                              ! ⥸塼ΥС
                              ! Module version

contains

  !--------------------------------------------------------------------------------------

  subroutine MassFixerWO94(    &
    & xyr_Press,               & ! (in)
    & xyzf_QMix,               & ! (inout)
    & xyr_PressRef,            & ! (in) optional
    & xyzf_QMixRef,            & ! (in) optional
    & xyzf_DQMixDt             & ! (out) optional
    & )
    !
    ! Williamson and Olson (1994) ˡ˽, ʬμ̤ޤ. 
    ! *xyzf_QMixRef* Ϳ줿ˤ, ʬͤ *xyzf_QMixRef* Τ
    ! Ʊˤʤ褦ޤ. 
    ! *xyzf_QMixRef* Ϳʤˤ, ʬͤΤ
    ! Ʊˤʤ褦ޤ. 
    ! *xyzf_DQMixDt* ˤ *xyz_QMix* Ѳ̤֤ޤ. 
    !
    ! This routine fix masses of constituents following a method proposed by 
    ! Williamson and Olson (1994).
    ! If *xyzf_QMixRef* is given, the mass is fixed to match its global 
    ! integrated value is the same as that of *xyzf_QMixRef*.
    ! If *xyzf_QMixRef* is not given, the mass is fixed to match its global 
    ! integrated value is the same as that of pre-fixed value. 
    ! Variation of *xyzf_QMix* is returned to *xyz_DQMixDt*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʪ
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI                    ! $ \pi $.
                              ! ߼Ψ. Circular constant

    ! ʪ
    ! Physical constants settings
    !
    use constants, only: Grav ! $ g $ [m s-2].
                              ! ϲ®.
                              ! Gravitational acceleration
    ! ɸǡ
    ! Axes data settings
    !
    use axesset, only: &
      & x_Lon,         &
      & y_Lat,         &
      & z_Sigma

    ! ʬʿѤ
    ! Operation for integral and average
    !
    use intavr_operate, only: IntLonLat_xy

    ! 
    ! Time control
    !
    use timeset, only: DelTime, &  ! $ \Delta t $
      & TimesetClockStart, TimesetClockStop

    ! ˴ؤ
    ! Settings of array for atmospheric composition
    !
    use composition, only: CompositionInqFlagMassFix


    ! ʸ ; Declaration statements
    !
    implicit none
    real(DP), intent(in   )          :: xyr_Press   (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ .  (Ⱦ٥). 
                              ! Air pressure (half level)
    real(DP), intent(inout)          :: xyzf_QMix   (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q $ .     漾. Specific humidity
    real(DP), intent(in   ), optional:: xyr_PressRef(0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ .  (Ⱦ٥). 
                              ! Air pressure (half level)
    real(DP), intent(in   ), optional:: xyzf_QMixRef(0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q \Delta p / g $ . ʬͤ碌ʬμ. 
                              ! Reference specific mass of constituent in a layer
    real(DP), intent(out  ), optional:: xyzf_DQMixDt(0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ \DP{q}{t} $ .  漾Ψ. 
                              ! Specific humidity correction

    ! ѿ
    ! Work variables
    !
    real(DP):: xyzf_QMixRefLV(0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q $ . ʬκ. 
                              ! Reference specific mass of constituent (local value)

    real(DP):: xyzf_QMixBefCor   (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! 漾.
                              ! Specific humidity before correction. 
    real(DP):: xyz_DelMass       (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \Delta p / g $
                              ! 
    real(DP):: xyz_DelMassRef    (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \Delta p / g $ of reference
                              ! 

    real(DP):: xy_FactAlphaNumer(0:imax-1, 1:jmax)
    real(DP):: xy_FactAlphaDenom(0:imax-1, 1:jmax)
    real(DP):: FactAlphaNumer
    real(DP):: FactAlphaDenom
    real(DP):: FactAlpha
    real(DP), parameter:: FactBeta = 1.5_DP

    real(DP):: xy_SumB(0:imax-1, 1:jmax)
    real(DP):: xy_SumA(0:imax-1, 1:jmax)
    real(DP):: SumB
    real(DP):: SumA
    real(DP):: Factor


    integer:: i               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in longitudinal direction
    integer:: j               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in latitudinal direction
    integer:: k               ! ľ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in vertical direction
    integer:: n               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in dimension of constituents

    ! ¹ʸ ; Executable statement
    !

    ! ǧ
    ! Initialization check
    !
    if ( .not. mass_fixer_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! ׻ַ¬
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    ! Check arguments
    !
    if ( present( xyr_PressRef ) .or. present( xyzf_QMixRef ) ) then
      if ( .not. ( present( xyr_PressRef ) .and. present( xyzf_QMixRef ) ) ) then
        call MessageNotify( 'E', module_name, 'If xyr_PressRef or xyzf_QMixRef is given, both have to be given.' )
      end if
    end if


    ! Backup of a variable
    if ( present( xyzf_DQMixDt ) ) then
      xyzf_QMixBefCor = xyzf_QMix
    end if

    ! Preparation of variable for reference
    if ( present( xyzf_QMixRef ) ) then
      xyzf_QMixRefLV = xyzf_QMixRef
    else
      xyzf_QMixRefLV = xyzf_QMix
    end if


    do k = 1, kmax
      xyz_DelMass(:,:,k) = ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k) ) / Grav
    end do

    if ( present( xyr_PressRef ) ) then
      do k = 1, kmax
        xyz_DelMassRef(:,:,k) = &
          & ( xyr_PressRef(:,:,k-1) - xyr_PressRef(:,:,k) ) / Grav
      end do
    else
      xyz_DelMassRef = xyz_DelMass
    end if


    ! Fill grids with negative values
    xyzf_QMix = max( xyzf_QMix, 0.0_DP )


    ! loop for constituents
    do n = 1, ncmax

      if ( CompositionInqFlagMassFix( n ) ) then

        ! 
        ! Calculation of factor, alpha
        !
        xy_FactAlphaNumer = 0.0_DP
        xy_FactAlphaDenom = 0.0_DP
        do k = kmax, 1, -1
          xy_FactAlphaNumer = xy_FactAlphaNumer                 &
            & + xyzf_QMixRefLV(:,:,k,n) * xyz_DelMassRef(:,:,k) &
            & - xyzf_QMix     (:,:,k,n) * xyz_DelMass     (:,:,k)
          xy_FactAlphaDenom = xy_FactAlphaDenom                               &
            & + xyzf_QMix   (:,:,k,n)                                         &
            & * abs( xyzf_QMix(:,:,k,n) - xyzf_QMixRefLV(:,:,k,n) )**FactBeta &
            & * xyz_DelMass(:,:,k)
        end do
        FactAlphaNumer = IntLonLat_xy( xy_FactAlphaNumer )
        FactAlphaDenom = IntLonLat_xy( xy_FactAlphaDenom )
        if ( FactAlphaDenom /= 0.0_DP ) then
          FactAlpha = FactAlphaNumer / FactAlphaDenom
        else
          FactAlpha = 0.0_DP
        end if

        xyzf_QMix(:,:,:,n) = xyzf_QMix(:,:,:,n) &
          & + FactAlpha * xyzf_QMix(:,:,:,n)    &
          &     * abs( xyzf_QMix(:,:,:,n) - xyzf_QMixRefLV(:,:,:,n) )**FactBeta

      end if

    end do

    ! 漾Ѳλ
    ! Calculate specific humidity variance
    !
    if ( present( xyzf_DQMixDt ) ) then
      xyzf_DQMixDt = ( xyzf_QMix - xyzf_QMixBefCor ) / ( 2.0_DP * DelTime )
    end if


    ! Ensure non-negative values
    ! This procedure is not included in Williams and Olson (1994).
    do n = 1, ncmax
      if ( CompositionInqFlagMassFix( n ) ) then
        xyzf_QMix(:,:,:,n) = max( xyzf_QMix(:,:,:,n), 0.0_DP )
        xy_SumB = 0.0_DP
        xy_SumA = 0.0_DP
        do k = kmax, 1, -1
          xy_SumB = xy_SumB + xyzf_QMixRefLV(:,:,k,n) * xyz_DelMassRef(:,:,k)
          xy_SumA = xy_SumA + xyzf_QMix     (:,:,k,n) * xyz_DelMass   (:,:,k)
        end do
        SumB = IntLonLat_xy( xy_SumB )
        SumA = IntLonLat_xy( xy_SumA )
        if ( SumA == 0.0_DP ) then
          Factor = 0.0_DP
        else if ( SumA < 0.0_DP ) then
          call MessageNotify( 'M', module_name,      &
            & 'WO94: n = %d, SumA is negative, %f.', &
            & i = (/ n /), d = (/ SumA /) )
          Factor = 0.0_DP
        else
          if ( SumB < 0.0_DP ) then
            call MessageNotify( 'M', module_name,      &
              & 'WO94: n = %d, SumB is negative, %f.', &
              & i = (/ n /), d = (/ SumB /) )
            Factor = 0.0_DP
          else
            Factor = SumB / SumA
          end if
        end if
        xyzf_QMix(:,:,:,n) = Factor * xyzf_QMix(:,:,:,n)
      end if
    end do


    ! This is not required.
    do n = 1, ncmax
      if ( CompositionInqFlagMassFix( n ) ) then
        do k = 1, kmax
          do j = 1, jmax
            do i = 0, imax-1
              if ( xyzf_QMix(i,j,k,n) < 0.0_DP ) then
                call MessageNotify( 'M', module_name, &
                  & 'WO94: Negative at (%f,%f,%f,%d), Val = %f.', &
                  & d = (/ x_Lon(i)*180.0_DP/PI,  y_Lat(j)*180.0_DP/PI, &
                  &        z_Sigma(k), xyzf_QMix(i,j,k,n) /), &
                  & i = (/ n /) )
              end if
            end do
          end do
        end do
      end if
    end do


    ! ׻ַ¬
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine MassFixerWO94

  !--------------------------------------------------------------------------------------

  subroutine MassFixer(           &
    & xyr_Press,                  & ! (in)
    & xyzf_QMix,                  & ! (inout)
    & xyr_PressRef,               & ! (in) optional
    & xyzf_QMixRef,               & ! (in) optional
    & xyzf_DQMixDt                & ! (out) optional
    & )
    !
    ! ʬμ̤ޤ. 
    ! *xyzf_QMixRef* Ϳ줿ˤ, ʬͤ *xyzf_QMixRef* Τ
    ! Ʊˤʤ褦ޤ. 
    ! *xyzf_QMixRef* Ϳʤˤ, ʬͤΤ
    ! Ʊˤʤ褦ޤ. 
    ! *xyzf_DQMixDt* ˤ *xyz_QMix* Ѳ̤֤ޤ. 
    !
    ! This routine fixes masses of constituents.
    ! If *xyzf_QMixRef* is given, the mass is fixed to match its global 
    ! integrated value is the same as that of *xyzf_QMixRef*.
    ! If *xyzf_QMixRef* is not given, the mass is fixed to match its global 
    ! integrated value is the same as that of pre-fixed value. 
    ! Variation of *xyzf_QMix* is returned to *xyz_DQMixDt*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʪ
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI                    ! $ \pi $.
                              ! ߼Ψ. Circular constant

    ! ʪ
    ! Physical constants settings
    !
    use constants, only: Grav ! $ g $ [m s-2].
                              ! ϲ®.
                              ! Gravitational acceleration

    ! ɸǡ
    ! Axes data settings
    !
    use axesset, only: &
      & x_Lon,         &
      & y_Lat,         &
      & z_Sigma

    ! ʬʿѤ
    ! Operation for integral and average
    !
    use intavr_operate, only: IntLonLat_xy

    ! 
    ! Time control
    !
    use timeset, only: DelTime, &  ! $ \Delta t $
      & TimesetClockStart, TimesetClockStop

    ! ˴ؤ
    ! Settings of array for atmospheric composition
    !
    use composition, only: CompositionInqFlagMassFix


    ! ʸ ; Declaration statements
    !
    implicit none
    real(DP), intent(in   )          :: xyr_Press   (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ .  (Ⱦ٥). 
                              ! Air pressure (half level)
    real(DP), intent(inout)          :: xyzf_QMix   (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q $ .     漾. Specific humidity
    real(DP), intent(in   ), optional:: xyr_PressRef(0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ .  (Ⱦ٥). 
                              ! Air pressure (half level)
    real(DP), intent(in   ), optional:: xyzf_QMixRef(0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q \Delta p / g $ . ʬͤ碌ʬμ. 
                              ! Reference specific mass of constituent in a layer
    real(DP), intent(out  ), optional:: xyzf_DQMixDt(0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ \DP{q}{t} $ .  漾Ψ. 
                              ! Specific humidity correction

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_QMixBefCor    (0:imax-1, 1:jmax, 1:kmax)
                              ! 漾.
                              ! Specific humidity before correction. 
    real(DP):: xyz_DelMass       (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \Delta p / g $
                              ! 
    real(DP):: xyz_DelMassRef    (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \Delta p / g $ of reference
                              ! 
    real(DP):: xyz_DelConsMass   (0:imax-1, 1:jmax, 1:kmax)
                              ! ʬμ.
                              ! Mass of constituents in a layer.
    real(DP):: xyz_DelConsMassRef(0:imax-1, 1:jmax, 1:kmax)
                              ! ʬͤ碌ʬμ.
                              ! Reference mass of constituents.
    real(DP):: xy_ConsMass          (0:imax-1, 1:jmax)
                              ! ʬΥ.
                              ! Mass of constituents in a layer.
    real(DP):: xy_ConsMassRef       (0:imax-1, 1:jmax)
                              ! ʬͤ碌ʬΥ.
                              ! Reference mass of constituents in a layer.
    real(DP):: ConsMass
                              ! γʬμ
                              ! Total mass of constituents
    real(DP):: ConsMassRef
                              ! ʬͤ碌γʬμ
                              ! Reference total mass of constituents.
                              !

    integer:: i               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in longitudinal direction
    integer:: j               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in latitudinal direction
    integer:: k               ! ľ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in vertical direction
    integer:: n               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in dimension of constituents

    ! ¹ʸ ; Executable statement
    !

    ! ǧ
    ! Initialization check
    !
    if ( .not. mass_fixer_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! ׻ַ¬
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    ! Check arguments
    !
    if ( present( xyr_PressRef ) .or. present( xyzf_QMixRef ) ) then
      if ( .not. ( present( xyr_PressRef ) .and. present( xyzf_QMixRef ) ) ) then
        call MessageNotify( 'E', module_name, 'If xyr_PressRef or xyzf_QMixRef is given, both have to be given.' )
      end if
    end if


    ! $ \Delta p / g $ η׻
    ! Calculate $ \Delta p / g $
    !
    do k = 1, kmax
      xyz_DelMass(:,:,k) = ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k) ) / Grav
    end do

    if ( present( xyr_PressRef ) ) then
      do k = 1, kmax
        xyz_DelMassRef(:,:,k) = ( xyr_PressRef(:,:,k-1) - xyr_PressRef(:,:,k) ) / Grav
      end do
    end if

    do n = 1, ncmax

      if ( CompositionInqFlagMassFix( n ) ) then

        ! Calculate mass of constituents
        !
        xyz_DelConsMass = xyzf_QMix(:,:,:,n) * xyz_DelMass

        if ( present( xyzf_QMixRef ) ) then
          xyz_DelConsMassRef = xyzf_QMixRef(:,:,:,n) * xyz_DelMassRef
        else
          xyz_DelConsMassRef = xyz_DelConsMass
        end if
        if ( present( xyzf_DQMixDt ) ) then
          xyz_QMixBefCor = xyzf_QMix(:,:,:,n)
        end if


        ! μ̤ľؤμ̤碌.
        ! Negative mass is removed by filling it with the mass in a layer just below.
        !
        do k = kmax, 2, -1
          do j = 1, jmax
            do i = 0, imax-1
              if ( xyz_DelConsMass(i,j,k) < 0.0_DP ) then
                xyz_DelConsMass(i,j,k-1) = xyz_DelConsMass(i,j,k-1) &
                  & + xyz_DelConsMass(i,j,k)
                xyz_DelConsMass(i,j,k  ) = 0.0_DP
              end if
            end do
          end do
        end do

        k = 1
        do j = 1, jmax
          do i = 0, imax-1
            if ( xyz_DelConsMass(i,j,k) < 0.0_DP ) then
              xyz_DelConsMass(i,j,k) = 0.0_DP
            end if
          end do
        end do


        ! Ǥ
        ! Correction in globe
        !   ¸ΤΤμ̤򸺾.
        !   Total mass is decreased to conserve mass. 
        !
        xy_ConsMass    = 0.0d0
        xy_ConsMassRef = 0.0d0
        do k = kmax, 1, -1
          xy_ConsMass    = xy_ConsMass    + xyz_DelConsMass   (:,:,k)
          xy_ConsMassRef = xy_ConsMassRef + xyz_DelConsMassRef(:,:,k)
        end do
        ConsMass    = IntLonLat_xy( xy_ConsMass    )
        ConsMassRef = IntLonLat_xy( xy_ConsMassRef )


        if ( ConsMassRef < 0.0_DP ) then 
          call MessageNotify( 'M', module_name, 'ConsMassRef is negative. ' &
            & // 'ConsMassRef is reset to zero, n = %d, ConsMassRef = %f.', &
            & i = (/ n /), d = (/ ConsMassRef /) )
          ConsMassRef = 0.0_DP
!!$        call MessageNotify( 'E', module_name, 'ConsMassRef is negative, n = %d.', i = (/ n /) )
        end if
        if ( ConsMass /= 0.0_DP ) then 
          xyz_DelConsMass = ConsMassRef / ConsMass * xyz_DelConsMass
        else
          xyz_DelConsMass = 0.0_DP
        end if

        xyzf_QMix(:,:,:,n) = xyz_DelConsMass / xyz_DelMass

        ! 漾Ѳλ
        ! Calculate specific humidity variance
        !
        if ( present( xyzf_DQMixDt ) ) then
          xyzf_DQMixDt(:,:,:,n) =     &
            &  ( xyzf_QMix(:,:,:,n) - xyz_QMixBefCor ) / ( 2.0_DP * DelTime )
        end if

      else

        if ( present( xyzf_DQMixDt ) ) then
          xyz_QMixBefCor = xyzf_QMix(:,:,:,n)
        end if

        xyzf_QMix(:,:,:,n) = max( xyzf_QMix(:,:,:,n), 0.0_DP )

        ! 漾Ѳλ
        ! Calculate specific humidity variance
        !
        if ( present( xyzf_DQMixDt ) ) then
          xyzf_DQMixDt(:,:,:,n) =     &
            &  ( xyzf_QMix(:,:,:,n) - xyz_QMixBefCor ) / ( 2.0_DP * DelTime )
        end if

      end if

    end do

    do n = 1, ncmax
      if ( CompositionInqFlagMassFix( n ) ) then
        do k = 1, kmax
          do j = 1, jmax
            do i = 0, imax-1
              if ( xyzf_QMix(i,j,k,n) < 0.0_DP ) then
                call MessageNotify( 'M', module_name, &
                  & 'Negative at (%f,%f,%f,%d), Val = %f.', &
                  & d = (/ x_Lon(i)*180.0_DP/PI, y_Lat(j)*180.0_DP/PI, &
                  &        z_Sigma(k), xyzf_QMix(i,j,k,n) /), &
                  & i = (/ n /) )
              end if
            end do
          end do
        end do
      end if
    end do


    ! ׻ַ¬
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine MassFixer

  !--------------------------------------------------------------------------------------

  subroutine MassFixerLayer(      &
    & xyr_Press,                  & ! (in)
    & xyzf_QMix,                  & ! (inout)
    & xyr_PressRef,               & ! (in) optional
    & xyzf_QMixRef,               & ! (in) optional
    & xyzf_DQMixDt                & ! (out) optional
    & )
    !
    ! ľʬμ̤ޤ. 
    ! *xyzf_QMixRef* Ϳ줿ˤ, ʬͤ *xyzf_QMixRef* Τ
    ! Ʊˤʤ褦ޤ. 
    ! *xyzf_QMixRef* Ϳʤˤ, ʬͤΤ
    ! Ʊˤʤ褦ޤ. 
    ! *xyzf_DQMixDt* ˤ *xyz_QMix* Ѳ̤֤ޤ. 
    !
    ! This routine fixes masses of constituents in each vertical layer.
    ! If *xyzf_QMixRef* is given, the mass is fixed to match its global 
    ! integrated value is the same as that of *xyzf_QMixRef*.
    ! If *xyzf_QMixRef* is not given, the mass is fixed to match its global 
    ! integrated value is the same as that of pre-fixed value. 
    ! Variation of *xyzf_QMix* is returned to *xyz_DQMixDt*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʪ
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI                    ! $ \pi $.
                              ! ߼Ψ. Circular constant

    ! ʪ
    ! Physical constants settings
    !
    use constants, only: Grav ! $ g $ [m s-2].
                              ! ϲ®.
                              ! Gravitational acceleration

    ! ɸǡ
    ! Axes data settings
    !
    use axesset, only: &
      & x_Lon,         &
      & y_Lat,         &
      & z_Sigma

    ! ʬʿѤ
    ! Operation for integral and average
    !
    use intavr_operate, only: a_IntLonLat_xya

    ! 
    ! Time control
    !
    use timeset, only: DelTime, &  ! $ \Delta t $
      & TimesetClockStart, TimesetClockStop

    ! ˴ؤ
    ! Settings of array for atmospheric composition
    !
    use composition, only: CompositionInqFlagMassFix


    ! ʸ ; Declaration statements
    !
    implicit none
    real(DP), intent(in   )          :: xyr_Press   (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ .  (Ⱦ٥). 
                              ! Air pressure (half level)
    real(DP), intent(inout)          :: xyzf_QMix   (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q $ .     漾. Specific humidity
    real(DP), intent(in   ), optional:: xyr_PressRef(0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ .  (Ⱦ٥). 
                              ! Air pressure (half level)
    real(DP), intent(in   ), optional:: xyzf_QMixRef(0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q \Delta p / g $ . ʬͤ碌ʬμ. 
                              ! Reference specific mass of constituent in a layer
    real(DP), intent(out  ), optional:: xyzf_DQMixDt(0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ \DP{q}{t} $ .  漾Ψ. 
                              ! Specific humidity correction

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_QMixBefCor    (0:imax-1, 1:jmax, 1:kmax)
                              ! 漾.
                              ! Specific humidity before correction. 
    real(DP):: xyz_DelMass       (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \Delta p / g $
                              ! 
    real(DP):: xyz_DelMassRef    (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \Delta p / g $ of reference
                              ! 
    real(DP):: xyz_DelConsMass   (0:imax-1, 1:jmax, 1:kmax)
                              ! ʬμ.
                              ! Mass of constituents in a layer.
    real(DP):: xyz_DelConsMassRef(0:imax-1, 1:jmax, 1:kmax)
                              ! ʬͤ碌ʬμ.
                              ! Reference mass of constituents.
    real(DP):: z_ConsMass   (1:kmax)
                              ! γʬμ
                              ! Total mass of constituents
    real(DP):: z_ConsMassRef(1:kmax)
                              ! ʬͤ碌γʬμ
                              ! Reference total mass of constituents.
                              !

    integer:: i               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in longitudinal direction
    integer:: j               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in latitudinal direction
    integer:: k               ! ľ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in vertical direction
    integer:: n               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in dimension of constituents

    ! ¹ʸ ; Executable statement
    !

    ! ǧ
    ! Initialization check
    !
    if ( .not. mass_fixer_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! ׻ַ¬
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    ! Check arguments
    !
    if ( present( xyr_PressRef ) .or. present( xyzf_QMixRef ) ) then
      if ( .not. ( present( xyr_PressRef ) .and. present( xyzf_QMixRef ) ) ) then
        call MessageNotify( 'E', module_name, 'If xyr_PressRef or xyzf_QMixRef is given, both have to be given.' )
      end if
    end if


    ! $ \Delta p / g $ η׻
    ! Calculate $ \Delta p / g $
    !
    do k = 1, kmax
      xyz_DelMass(:,:,k) = ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k) ) / Grav
    end do

    if ( present( xyr_PressRef ) ) then
      do k = 1, kmax
        xyz_DelMassRef(:,:,k) = ( xyr_PressRef(:,:,k-1) - xyr_PressRef(:,:,k) ) / Grav
      end do
    end if

    do n = 1, ncmax

      if ( CompositionInqFlagMassFix( n ) ) then

        ! Calculate mass of constituents
        !
        xyz_DelConsMass = xyzf_QMix(:,:,:,n) * xyz_DelMass

        if ( present( xyzf_QMixRef ) ) then
          xyz_DelConsMassRef = xyzf_QMixRef(:,:,:,n) * xyz_DelMassRef
        else
          xyz_DelConsMassRef = xyz_DelConsMass
        end if
        if ( present( xyzf_DQMixDt ) ) then
          xyz_QMixBefCor = xyzf_QMix(:,:,:,n)
        end if


        do k = 1, kmax
          do j = 1, jmax
            do i = 0, imax-1
              if ( xyz_DelConsMass(i,j,k) < 0.0_DP ) then
                xyz_DelConsMass(i,j,k) = 0.0_DP
              end if
            end do
          end do
        end do


        ! ؤǤ
        ! Correction in each layer
        !   ¸ΤΤμ̤򸺾.
        !   Total mass is decreased to conserve mass. 
        !
        z_ConsMass    = a_IntLonLat_xya( xyz_DelConsMass    )
        z_ConsMassRef = a_IntLonLat_xya( xyz_DelConsMassRef )

        do k = 1, kmax

          if ( z_ConsMassRef(k) < 0.0_DP ) then 
            call MessageNotify( 'M', module_name,                             &
              & 'z_ConsMassRef(%f) is negative. '                             &
              & // 'z_ConsMassRef is reset to zero, n = %d, z_ConsMassRef = %f.', &
              & d = (/ z_Sigma(k), z_ConsMassRef(k) /), i = (/ n /) )
            z_ConsMassRef(k) = 0.0_DP
!!$        call MessageNotify( 'E', module_name, 'ConsMassRef is negative, n = %d.', i = (/ n /) )
          end if
          if ( z_ConsMass(k) /= 0.0_DP ) then 
            xyz_DelConsMass(:,:,k) = z_ConsMassRef(k) / z_ConsMass(k) * xyz_DelConsMass(:,:,k)
          else
            xyz_DelConsMass(:,:,k) = 0.0_DP
          end if

        end do

        xyzf_QMix(:,:,:,n) = xyz_DelConsMass / xyz_DelMass

        ! 漾Ѳλ
        ! Calculate specific humidity variance
        !
        if ( present( xyzf_DQMixDt ) ) then
          xyzf_DQMixDt(:,:,:,n) =     &
            &   xyzf_DQMixDt(:,:,:,n) &
            & + ( xyzf_QMix(:,:,:,n) - xyz_QMixBefCor ) / ( 2.0_DP * DelTime )
        end if

      else

        if ( present( xyzf_DQMixDt ) ) then
          xyz_QMixBefCor = xyzf_QMix(:,:,:,n)
        end if

        xyzf_QMix(:,:,:,n) = max( xyzf_QMix(:,:,:,n), 0.0_DP )

        ! 漾Ѳλ
        ! Calculate specific humidity variance
        !
        if ( present( xyzf_DQMixDt ) ) then
          xyzf_DQMixDt(:,:,:,n) =     &
            &  ( xyzf_QMix(:,:,:,n) - xyz_QMixBefCor ) / ( 2.0_DP * DelTime )
        end if

      end if

    end do

    do n = 1, ncmax
      if ( CompositionInqFlagMassFix( n ) ) then
        do k = 1, kmax
          do j = 1, jmax
            do i = 0, imax-1
              if ( xyzf_QMix(i,j,k,n) < 0.0_DP ) then
                call MessageNotify( 'M', module_name, &
                  & 'Layer: Negative at (%f,%f,%f,%d), Val = %f.', &
                  & d = (/ x_Lon(i)*180.0_DP/PI, y_Lat(j)*180.0_DP/PI, &
                  &        z_Sigma(k), xyzf_QMix(i,j,k,n)/), &
                  & i = (/ n /) )
              end if
            end do
          end do
        end do
      end if
    end do


    ! ׻ַ¬
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine MassFixerLayer

  !--------------------------------------------------------------------------------------

  subroutine MassFixerColumn(     &
    & xyr_Press,                  & ! (in)
    & xyzf_QMix,                  & ! (inout)
    & xyr_PressRef,               & ! (in) optional
    & xyzf_QMixRef,               & ! (in) optional
    & xyzf_DQMixDt                & ! (out) optional
    & )
    !
    ! ľʬμ̤ޤ. 
    ! *xyzf_QMixRef* Ϳ줿ˤ, ʬͤ *xyzf_QMixRef* 
    ! Ʊˤʤ褦ޤ. 
    ! *xyzf_QMixRef* Ϳʤˤ, ʬͤΤ
    ! Ʊˤʤ褦ޤ. 
    ! *xyzf_DQMixDt* ˤ *xyz_QMix* Ѳ̤֤ޤ. 
    !
    ! This routine fixes masses of constituents in each vertical column.
    ! If *xyzf_QMixRef* is given, the mass is fixed to match its column 
    ! integrated value is the same as that of *xyzf_QMixRef*.
    ! If *xyzf_QMixRef* is not given, the mass is fixed to match its column 
    ! integrated value is the same as that of pre-fixed value. 
    ! Variation of *xyzf_QMix* is returned to *xyz_DQMixDt*. 
    !

    ! ⥸塼 ; USE statements
    !

    ! ʪ
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI                    ! $ \pi $.
                              ! ߼Ψ. Circular constant

    ! ʪ
    ! Physical constants settings
    !
    use constants, only: Grav ! $ g $ [m s-2].
                              ! ϲ®.
                              ! Gravitational acceleration

    ! ɸǡ
    ! Axes data settings
    !
    use axesset, only: &
      & x_Lon,         &
      & y_Lat,         &
      & z_Sigma

    ! 
    ! Time control
    !
    use timeset, only: DelTime, &  ! $ \Delta t $
      & TimesetClockStart, TimesetClockStop

    ! ˴ؤ
    ! Settings of array for atmospheric composition
    !
    use composition, only: CompositionInqFlagMassFix


    ! ʸ ; Declaration statements
    !
    implicit none
    real(DP), intent(in   )          :: xyr_Press   (0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ .  (Ⱦ٥). 
                              ! Air pressure (half level)
    real(DP), intent(inout)          :: xyzf_QMix   (0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q $ .     漾. Specific humidity
    real(DP), intent(in   ), optional:: xyr_PressRef(0:imax-1, 1:jmax, 0:kmax)
                              ! $ \hat{p} $ .  (Ⱦ٥). 
                              ! Air pressure (half level)
    real(DP), intent(in   ), optional:: xyzf_QMixRef(0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q \Delta p / g $ . ʬͤ碌ʬμ. 
                              ! Reference specific mass of constituent in a layer
    real(DP), intent(out  ), optional:: xyzf_DQMixDt(0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ \DP{q}{t} $ .  漾Ψ. 
                              ! Specific humidity correction

    ! ѿ
    ! Work variables
    !
    real(DP):: xyz_QMixBefCor    (0:imax-1, 1:jmax, 1:kmax)
                              ! 漾.
                              ! Specific humidity before correction. 
    real(DP):: xyz_DelMass       (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \Delta p / g $
                              ! 
    real(DP):: xyz_DelMassRef    (0:imax-1, 1:jmax, 1:kmax)
                              ! $ \Delta p / g $ of reference
                              ! 
    real(DP):: xyz_DelConsMass   (0:imax-1, 1:jmax, 1:kmax)
                              ! ʬμ.
                              ! Mass of constituents in a layer.
    real(DP):: xyz_DelConsMassRef(0:imax-1, 1:jmax, 1:kmax)
                              ! ʬͤ碌ʬμ.
                              ! Reference mass of constituents.
    real(DP):: xy_ConsMass          (0:imax-1, 1:jmax)
                              ! ʬΥ.
                              ! Mass of constituents in a layer.
    real(DP):: xy_ConsMassRef       (0:imax-1, 1:jmax)
                              ! ʬͤ碌ʬΥ.
                              ! Reference mass of constituents in a layer.

    integer:: i               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in longitudinal direction
    integer:: j               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in latitudinal direction
    integer:: k               ! ľ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in vertical direction
    integer:: n               ! ˲ DO 롼Ѻѿ
                              ! Work variables for DO loop in dimension of constituents

    ! ¹ʸ ; Executable statement
    !

    ! ǧ
    ! Initialization check
    !
    if ( .not. mass_fixer_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! ׻ַ¬
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    ! Check arguments
    !
    if ( present( xyr_PressRef ) .or. present( xyzf_QMixRef ) ) then
      if ( .not. ( present( xyr_PressRef ) .and. present( xyzf_QMixRef ) ) ) then
        call MessageNotify( 'E', module_name, 'If xyr_PressRef or xyzf_QMixRef is given, both have to be given.' )
      end if
    end if


    ! $ \Delta p / g $ η׻
    ! Calculate $ \Delta p / g $
    !
    do k = 1, kmax
      xyz_DelMass(:,:,k) = ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k) ) / Grav
    end do

    if ( present( xyr_PressRef ) ) then
      do k = 1, kmax
        xyz_DelMassRef(:,:,k) = ( xyr_PressRef(:,:,k-1) - xyr_PressRef(:,:,k) ) / Grav
      end do
    end if

    do n = 1, ncmax

      if ( CompositionInqFlagMassFix( n ) ) then

        ! Calculate mass of constituents
        !
        xyz_DelConsMass = xyzf_QMix(:,:,:,n) * xyz_DelMass

        if ( present( xyzf_QMixRef ) ) then
          xyz_DelConsMassRef = xyzf_QMixRef(:,:,:,n) * xyz_DelMassRef
        else
          xyz_DelConsMassRef = xyz_DelConsMass
        end if
        if ( present( xyzf_DQMixDt ) ) then
          xyz_QMixBefCor = xyzf_QMix(:,:,:,n)
        end if


        ! μ̤ľؤμ̤碌.
        ! Negative mass is removed by filling it with the mass in a layer just below.
        !
        do k = kmax, 2, -1
          do j = 1, jmax
            do i = 0, imax-1
              if ( xyz_DelConsMass(i,j,k) < 0.0_DP ) then
                xyz_DelConsMass(i,j,k-1) = xyz_DelConsMass(i,j,k-1) &
                  & + xyz_DelConsMass(i,j,k)
                xyz_DelConsMass(i,j,k  ) = 0.0_DP
              end if
            end do
          end do
        end do

        k = 1
        do j = 1, jmax
          do i = 0, imax-1
            if ( xyz_DelConsMass(i,j,k) < 0.0_DP ) then
              xyz_DelConsMass(i,j,k) = 0.0_DP
            end if
          end do
        end do


        ! Ǥ
        ! Correction in globe
        !   ¸ΤΤμ̤򸺾.
        !   Total mass is decreased to conserve mass. 
        !
        xy_ConsMass    = 0.0d0
        xy_ConsMassRef = 0.0d0
        do k = kmax, 1, -1
          xy_ConsMass    = xy_ConsMass    + xyz_DelConsMass   (:,:,k)
          xy_ConsMassRef = xy_ConsMassRef + xyz_DelConsMassRef(:,:,k)
        end do

        do j = 1, jmax
          do i = 0, imax-1

            if ( xy_ConsMassRef(i,j) < 0.0_DP ) then 
              call MessageNotify( 'M', module_name,                  &
                & 'xy_ConsMassRef(%f,%f) is negative. '              &
                & // 'The value, %f, is reset to zero, n = %d.',     &
                & d = (/ x_Lon(i)*180.0_DP/PI, y_Lat(j)*180.0_DP/PI, &
                &        xy_ConsMassRef(i,j) /),                     &
                & i = (/ n /) )
              xy_ConsMassRef(i,j) = 0.0_DP
!!$        call MessageNotify( 'E', module_name, 'ConsMassRef is negative, n = %d.', i = (/ n /) )
            end if
            if ( xy_ConsMass(i,j) /= 0.0_DP ) then 
              do k = 1, kmax
                xyz_DelConsMass(i,j,k) = xy_ConsMassRef(i,j) / xy_ConsMass(i,j) * xyz_DelConsMass(i,j,k)
              end do
            else
              do k = 1, kmax
                xyz_DelConsMass(i,j,k) = 0.0_DP
              end do
            end if

          end do
        end do

        xyzf_QMix(:,:,:,n) = xyz_DelConsMass / xyz_DelMass

        ! 漾Ѳλ
        ! Calculate specific humidity variance
        !
        if ( present( xyzf_DQMixDt ) ) then
          xyzf_DQMixDt(:,:,:,n) =     &
            &   xyzf_DQMixDt(:,:,:,n) &
            & + ( xyzf_QMix(:,:,:,n) - xyz_QMixBefCor ) / ( 2.0_DP * DelTime )
        end if

      else

        if ( present( xyzf_DQMixDt ) ) then
          xyz_QMixBefCor = xyzf_QMix(:,:,:,n)
        end if

        xyzf_QMix(:,:,:,n) = max( xyzf_QMix(:,:,:,n), 0.0_DP )

        ! 漾Ѳλ
        ! Calculate specific humidity variance
        !
        if ( present( xyzf_DQMixDt ) ) then
          xyzf_DQMixDt(:,:,:,n) =     &
            &  ( xyzf_QMix(:,:,:,n) - xyz_QMixBefCor ) / ( 2.0_DP * DelTime )
        end if

      end if

    end do

    do n = 1, ncmax
      if ( CompositionInqFlagMassFix( n ) ) then
        do k = 1, kmax
          do j = 1, jmax
            do i = 0, imax-1
              if ( xyzf_QMix(i,j,k,n) < 0.0_DP ) then
                call MessageNotify( 'M', module_name, &
                  & 'Column: Negative at (%f,%f,%f,%d), Val = %f.', &
                  & d = (/ x_Lon(i)*180.0_DP/PI, y_Lat(j)*180.0_DP/PI, &
                  &        z_Sigma(k), xyzf_QMix(i,j,k,n) /), &
                  & i = (/ n /) )
              end if
            end do
          end do
        end do
      end if
    end do


    ! ׻ַ¬
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine MassFixerColumn

  !--------------------------------------------------------------------------------------

  subroutine MassFixerInit
    !
    ! mass_fixer ⥸塼νԤޤ. 
    ! NAMELIST#mass_fixer_nml ɤ߹ߤϤμ³ǹԤޤ. 
    !
    ! "mass_fixer" module is initialized. 
    ! "NAMELIST#mass_fixer_nml" is loaded in this procedure. 
    !

    ! ⥸塼 ; USE statements
    !

    ! NAMELIST եϤ˴ؤ桼ƥƥ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg

    ! ե
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! ̷ѥ᥿
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! ɸϤֹ. Unit number of standard output

    ! ʸ
    ! Character handling
    !
    use dc_string, only: StoA

    ! ʸ ; Declaration statements
    !
    implicit none

!!$    integer:: unit_nml        ! NAMELIST ե륪ץֹ. 
!!$                              ! Unit number for NAMELIST file open
!!$    integer:: iostat_nml      ! NAMELIST ɤ߹߻ IOSTAT. 
!!$                              ! IOSTAT of NAMELIST read

    ! NAMELIST ѿ
    ! NAMELIST group name
    !
!!$    namelist /mass_fixer_nml/
          !
          ! ǥեͤˤĤƤϽ³ "mass_fixer#MassFixerInit" 
          ! Υɤ򻲾ȤΤ. 
          !
          ! Refer to source codes in the initialization procedure
          ! "mass_fixer#MassFixerInit" for the default values. 
          !

    ! ¹ʸ ; Executable statement
    !

    if ( mass_fixer_inited ) return


    ! ǥեͤ
    ! Default values settings
    !

!!$    ! NAMELIST ɤ߹
!!$    ! NAMELIST is input
!!$    !
!!$    if ( trim(namelist_filename) /= '' ) then
!!$      call FileOpen( unit_nml, &          ! (out)
!!$        & namelist_filename, mode = 'r' ) ! (in)
!!$
!!$      rewind( unit_nml )
!!$      read( unit_nml, &           ! (in)
!!$        & nml = mass_fixer_nml, &  ! (out)
!!$        & iostat = iostat_nml )   ! (out)
!!$      close( unit_nml )
!!$
!!$      call NmlutilMsg( iostat_nml, module_name ) ! (in)
!!$    end if

    !  ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    mass_fixer_inited = .true.

  end subroutine MassFixerInit

  !--------------------------------------------------------------------------------------

end module mass_fixer
