#! /usr/bin/env ruby

#:title: RecursiveUtils
#:main:  RecursiveUtils
#
#= recursiveutils.rb -- Recursive Utils
#Authors::   Michimasa Koichi <michi[at]ep.sci.hokudai.ac.jp>
#Copyright:: Copyright (C) Michimasa Koichi 2006-. All rights reserved.
#Document::  http://www.ep.sci.hokudai.ac.jp/~michi/pc/ruby/doc/recursiveutils/
#Download::  http://www.ep.sci.hokudai.ac.jp/~michi/pc/ruby/src/recursiveutils/
#
#== Summary
#Ƶǽġ뷲.
#
#== Methods
#=== Module Methods
#ե:: chmod, chown
#
#== Reference Tools
#rename.rb:: http://www.ep.sci.hokudai.ac.jp/~daktu32/DOC/ruby/works/ruby_works.htm
#rsed2.rb::  http://www.ep.sci.hokudai.ac.jp/~morikawa/ruby/rsed2/SIGEN_PUB.htm
#
#== Required Library
#===RecursiveUtils
#* localefilter ( >= Ver. 2.4.1 )
#  URL:: http://www.ep.sci.hokudai.ac.jp/~michi/pc/ruby/doc/localefilter/
#
#===RecursiveUtils::FullPathList
#* find (Standard Library)
#
#===RecursiveUtils::FrontEnd
#* optparse (Standard Library)
#
#== Acknowlege
#* ͸ (rename.rb κ)
#  URL:: http://www.ep.sci.hokudai.ac.jp/~daktu32/
#
#*  (rsed2.rb κ)
#  URL:: http://www.ep.sci.hokudai.ac.jp/~morikawa/
#
#==Future Plans
#* ƵǤʥ᥽åɤפĤɲä.
#
#== Recent History
#* Ver. 3.2.0, 2007/10/05 by michi
#  FrontEnd#rename:: إפ
#  FrontEnd#sed::    إפ
#  FrontEnd#chmod::  
#  FrontEnd#chown::  
#
#* Ver. 3.1.0, 2007/05/27 by michi
#  Rename::          force, quiet ⡼ɤνĴ, ¾
#  Sed::             @protect , ¾
#  FrontEnd#rename:: [-h] β, [-q] μ
#  FrontEnd#sed::    [-h] β, [-p] μ
#
#* Ver. 3.0.1, 2007/05/23 by michi
#  FullPathList::    ɤκŬ
#  Rename::          ɤκŬ
#  FrontEnd#rename:: إפβ
#  FrontEnd#sed::    إפβ
#
#* Ver. 3.0.0, 2007/05/09 by michi
#  FrontEnd::        
#  FrontEnd#rename:: rename.rb ܿ
#  FrontEnd#sed::    rsed.rb ܿ
#
#--
#* Ver. 2.3.2, 2006/11/21 by michi
#  Sed:: Х
#
#* Ver. 2.3.1, 2006/11/18 by michi
#  FullPathList:: Х
#  Sed::          Х
#
#* Ver. 2.3.0, 2006/11/13 by michi
#  RecursiveUtils:: ƥ饹 initialize ˥ץϤ褦
#  FullPathList::   ץ, ᥽åɤɲ
#
#* Ver. 2.2.0, 2006/11/12 by michi
#  chmod:: 
#  chown:: 
#
#* Ver. 2.1.0, 2006/11/07 by michi
#  RecursiveUtils:: FullPathList, Rename, Sed ⥸塼벽
#  FullPathList::   Rename, Sed Υѡ饹ȤƼ
#++
#
module RecursiveUtils
  require 'localefilter'

  # RecursiveUtils ΥС
  VERSION = '3.2.0'

  # RecursiveUtils κǽ
  UPDATE  = '2007/10/05'

  # RecursiveUtils θɥ쥹
  URL     = 'http://www.ep.sci.hokudai.ac.jp/~michi/pc/ruby/'

  #
  #== Summary
  #ʣ̵ΥեѥΥꥹȤ.
  #
  #== Methods
  #===Public Methods
  #:: new
  #ꥹȼ:: get, get_exist, get_dir, get_file
  #
  #===Private Methods
  #¾:: option_filter
  #
  #==Future Plans
  #* α黻ҥ᥽åɤƤߤ롩
  #
  class FullPathList
    require 'find'

    #
    #֥Ȥ. ƥץ +Attributes+ 򻲾Ȥ.
    #
    def initialize(options = {}, *path)
      @lf = LocaleFilter.new
      @recursive = options[:recursive]
      @dir       = options[:dir]
      @owner     = options[:owner]
      @readable  = options[:readable]
      @writable  = options[:writable]
      @warn      = options[:warn]
      @debug     = options[:debug]
      @pathlist  = get(path)  unless path.to_s.empty?
    end

    # ƵԤ. ǥեȤ *false*.
    attr_accessor :recursive

    # оݤ˥ǥ쥯ȥɲä. ǥեȤ *false*.
    attr_accessor :dir

    # ͭԤΥåԤ. ǥեȤ *false*.
    attr_accessor :owner

    # ɤ߹߸ΥåԤ. ǥեȤ *false*.
    attr_accessor :readable

    # 񤭹߸ΥåԤ. ǥեȤ *false*.
    attr_accessor :writable

    # ѥ¸ߤʤä˷ٹϤ. ǥեȤ *false*.
    attr_accessor :warn

    # ȯѤΥåϤ. ǥեȤ *false*.
    attr_accessor :debug

    # new  _path_ Ϳݤ˺ꥹ.
    attr_reader :pathlist

    #
    # _list_ 򥪥ץǥե륿󥰤֤.
    #
    def option_filter(list, dir = false)
      return false  if list.to_s.empty?

      unless dir || @dir
        list = list.partition { |i| File.directory?(i) }
        list[0].each { |i| @lf.warn "\"#{i}\" is directory." }  if @warn
        list = list[1]
      end

      if @owner
        list = list.partition { |i| File.owned?(i) }
        list[1].each { |i| @lf.warn "#{i}: Not owner." }  if @warn
        list = list[0]
      end

      if @readable
        list = list.partition { |i| File.readable?(i) }
        list[1].each { |i| @lf.warn "#{i}: Read permission denied." }  if @warn
        list = list[0]
      end

      if @writable
        list = list.partition { |i| File.writable?(i) }
        list[1].each { |i| @lf.warn "#{i}: Write permission denied." }  if @warn
        list = list[0]
      end

      return list
    end
    private :option_filter

    #
    #_path_ 򸵤¸ߤեΥꥹȤ֤.
    #
    def get_exist(*path)
      return false  if path.to_s.empty?
      list = Array.new

      path.flatten.uniq.each { |i|
        unless File.exist?(i)
          @lf.warn "#{i}: No such file or direcotry."  if @warn
          next
        end

        fullpath = File.expand_path(i)
        unless list.include?(fullpath)
          list << fullpath

          if @recursive && File.directory?(fullpath)
            Find.find(fullpath) { |j| list << j }
          end
        end
      }
      list.uniq
    end

    #
    #_path_ 򸵤˥ץбꥹȤ֤.
    #
    def get(*path)
      option_filter(get_exist(path))
    end

    #
    #_path_ 򸵤˥ǥ쥯ȥΥꥹȤ֤.
    #
    def get_dir(*path)
      return  unless list = get_exist(path)
      list = list.partition { |i| File.directory?(i) }
      list[1].each { |i| @lf.warn "#{i}: Not directory." }  if @warn
      option_filter(list[0], true)
    end

    #
    #_path_ 򸵤̾եΥꥹȤ֤.
    #
    def get_file(*path)
      return  unless list = get_exist(path)
      list = list.partition { |i| File.file?(i) }
      list[1].each { |i| @lf.warn "#{i}: Not regular file." }  if @warn
      option_filter(list[0], true)
    end
  end

  #
  #== Summary
  #Ƶǽ, ʸѴǽ Ruby  rename.
  #
  #== Example
  #  require 'recursiveutils'
  #
  #  obj = RecursiveUtils::Rename.new
  #  obj.rename('f..', 'bar', 'foo.txt')
  #  # => оݥե /f../  "bar" Ѵ
  #
  #  obj.recursive = true
  #  obj.upcase('/usr/local/src/')
  #  # => оݥǥ쥯ȥʲΥեʸѴ
  #
  #  obj.recursive = false
  #  obj.extention = true
  #  obj.downcase
  #  # => ȥǥ쥯ΥեγĥҤʸѴ
  #
  #  obj.extention = false
  #  obj.convert('euc', Dir.glob('./*.txt'))
  #  # => оݥեʸɤ euc Ѵ
  #
  #  obj.exec_rename(Dir.glob('./*.txt')) { |i| i.swapcase.gsub('X', '_') }
  #  # => оݥեʸȾʸѴ, "X"  "_" ִ
  #
  #== Methods
  #=== Public Methods
  #:: new
  #ե:: rename, upcase, downcase, convert, exec_rename
  #
  #=== Private Methods
  #¾:: sort_list, confirm
  #
  #== Reference Tool
  #rename.rb:: http://www.ep.sci.hokudai.ac.jp/~daktu32/DOC/ruby/works/ruby_works.htm
  #
  #==Future Plans
  #* ι⤤ե̾Ѵ᥽åɤפĤɲä.
  #
  class Rename < FullPathList
    #
    #֥Ȥ.
    #ץΰ̣ FullPathList  Rename  +Attributes+ 򻲾Ȥ.
    #
    def initialize(options = {})
      @lf = LocaleFilter.new
      @recursive = options[:recursive]
      @dir       = options[:dir]
      @owner     = options[:owner]
      @readable  = options[:readable]
      @writable  = options[:writable]
      @warn      = options[:warn]
      @debug     = options[:debug]
      @noop      = options[:noop]
      @force     = options[:force]
      @extention = options[:extention]
      @quiet     = options[:quiet]
    end

    # Ѥʤץ
    undef   :pathlist

    # Ѥʤ᥽å
    undef   :get_dir, :get_file
    private :get, :get_exist

    # ѴԤʤ. ǥեȤ *false*.
    attr_accessor :noop

    # ǧ̵ǽ¹Ԥ. ǥեȤ *false*.
    attr_accessor :force

    # оݤĥҤΤߤ˸ꤹ. ǥեȤ *false*.
    attr_accessor :extention

    # å. ǥեȤ *false*.
    attr_accessor :quiet

    #
    #Hash _list_ ʸĹĹ˥Ȥ֤.
    #
    def sort_list(list)
      list.to_a.sort { |a, b|
        (b[0].length <=> a[0].length) * 2 + (a[1] <=> b[1])
      }
    end
    private :sort_list

    #
    #ɸϤ *Yes* or *No* β.
    #
    def confirm(message)
      STDERR.print "#{message} [y/n] : "
      loop{
        case STDIN.gets.chomp
        when /^y(es)??\b/i
          return true

        when /^n(o)??\b/i
          STDERR.puts 'quit.'
          return false

        else
          STDERR.print '  Please answer yes or no. [y/n] : '
        end
      }
    end
    private :confirm

    #
    # _list_ 򸵤˺оݤ _block_ Ѵ.
    #
    def exec_rename(list, message = nil, &block)
      list = Dir.entries(Dir.pwd)  if list.to_s.empty?
      pathlist = get(list)
      raise ArgumentError,  'Error: No entry.'        unless pathlist
      raise LocalJumpError, 'Error: No block given.'  unless block
      src_list = Hash.new { |h, k| h[k] = Array.new }
      new_list = src_list.dup
      new_num  = 0

      pathlist.each { |path|
        dir, before = File.split(path)
        if @extention
          target = File.extname(before)
          after  = File.basename(before, ".*") << block.call(target)

        else
          after = block.call(before)
        end

        unless before == after
          src_list[dir] << before
          new_list[dir] << after
          new_num += 1
        end
      }

      if new_list.empty?
        STDERR.puts 'No such files which will be converted to that.'
        return false
      end
      src_list = sort_list(src_list)
      new_list = sort_list(new_list)

      unless @quiet
        src_list.each_index { |m|
          next  unless new_list[m]
          @lf.puts "\n#{src_list[m][0]}:"

          max = src_list[m][1].map{ |i| i.size }.max
          src_list[m][1].each_index { |n|
            print '    ', @lf.kconv(src_list[m][1][n]).to_s.ljust(max)
            if message
              puts message

            else
              @lf.puts "  =>  #{new_list[m][1][n]}"
            end
          }
        }
        print "\n"
      end
      return true  if @noop

      unless @force
        STDERR.print "#{new_num} files will be renamed. "  if @debug
        return false  unless confirm('Do you sure rename?')
      end

      src_list.each_index { |m|
        next  unless new_list[m]
        src_list[m][1].each_index { |n|
          src_path = File.join(src_list[m][0], src_list[m][1][n])
          new_path = File.join(src_list[m][0], new_list[m][1][n])

          if ! @force && File.exist?(new_path)
            STDERR.print %Q("#{new_path}" is already exist. )
            next  unless confirm('Overwrite now?')
          end

          File.rename(src_path, new_path)  rescue STDERR.puts $!
        }
      }
      print "\n", "done.\n"  unless @quiet
    end

    #
    #_path_ ɽ _from_ ˰פե _to_ Ѵ.
    #
    def rename(from, to, *path)
      from = /#{from}/
      exec_rename(path) { |i| i.gsub(from, to) }
    end

    #
    #_path_ ΥեʸѴ.
    #
    def upcase(*path)
      exec_rename(path) { |i| i.upcase }
    end

    #
    #_path_ ΥեʸѴ.
    #
    def downcase(*path)
      exec_rename(path) { |i| i.downcase }
    end

    #
    #_path_ Υեʸ _code_ Ѵ.
    #
    def convert(code = @lf.locale, *path)
      to = @lf.txt2num(code)

      case to
      when Kconv::ASCII, Kconv::BINARY, Kconv::UNKNOWN
        raise ArgumentError, "#{code}: Invalide character code."

      else
        puts %Q(Convert file name encode to "#{@lf.num2txt(to)}".)  unless @quiet
      end
      message = %Q[  (Convert to "#{@lf.num2txt(to)}")]

      exec_rename(path, message) { |i| Kconv.kconv(i, to) }
    end
  end

  #
  #== Summary
  #Ƶǽ Ruby  sed.
  #
  #== Example
  #  require 'recursiveutils'
  #
  #  obj = RecursiveUtils::Sed.new
  #  obj.sed('f..', 'bar', 'foo.txt')
  #  # => оݥեʸ /f../  "bar" ִ
  #
  #  obj.recursive = true
  #  obj.sed('text', 'ƥ', '/usr/local/src/')
  #  # => оݥǥ쥯ȥʲΥեƤФƼ¹Ԥ
  #
  #  obj.recursive = false
  #  obj.code = 'euc'
  #  obj.sed('ۤ', 'hoge', 'hoge.txt')
  #  # => ʸɤ "EUC" ꤷƼ¹Ԥ
  #
  #== Methods
  #=== Public Methods
  #:: new
  #ե:: sed
  #:: size=
  #
  #=== Private Methods
  #ե:: exec_sed, change_file_stat
  #¾:: set_code
  #
  #== Reference Tool
  #rsed2.rb::  http://www.ep.sci.hokudai.ac.jp/~morikawa/ruby/rsed2/SIGEN_PUB.htm
  #
  #==Future Plans
  #* ХååץեγĥҤǤ褦ˤ.
  #
  #* ʣִ¹ԤǤ褦ˤ.
  #  * Rename#exec_rename ߤʷǼΤɸ.
  #
  class Sed < FullPathList
    include LocaleFilter::CodeConverter

    # ʸȽѤɤ߹ߥν (byte).
    DEF_SIZE = 1024

    # ʸȽѤɤ߹ߥξ (byte).
    MAX_SIZE = 1024 * 1024

    # 
    SEP_EXPLAIN = ' | --- : before, +++ : after |'

    # path ֤Υѥ졼
    SEP_PATH    = ' +' << ( '-' * ( SEP_EXPLAIN.length - 3 ) ) << '+'

    # list ֤Υѥ졼
    SEP_LIST    = SEP_PATH.gsub('-', '=')

    #
    #֥Ȥ.
    #ץΰ̣ FullPathList  Sed  +Attributes+ 򻲾Ȥ.
    #
    def initialize(options = {})
      @lf = LocaleFilter.new
      @recursive = options[:recursive]
      @dir       = false
      @owner     = options[:owner]
      @readable  = true
      @writable  = true
      @warn      = options[:warn]
      @debug     = options[:debug]
      @backup    = options[:backup]
      @code      = options[:code]
      @force     = options[:force]
      @noop      = options[:noop]
      @protect   = options[:protect]
      @verbose   = options[:verbose]
      if options[:size]
        self.size=(options[:size])

      else
        @size    = DEF_SIZE
      end
    end

    # Ѥʤץ
    undef :pathlist, :dir, :dir=, :readable, :readable=, :writable, :writable=

    # Ѥʤ᥽å
    undef   :get, :get_dir, :get_env
    private :get_exist, :get_file, :txt2num, :num2txt

    #
    #ե <b>*.bk</b> Ȥ̾¸, ե񤭤.
    #ǥեȤ *false*.
    #
    attr_accessor :backup

    #
    #ʸ̾. ꤵƤʸȽ̤񤭤.
    #ǥեȤ *nil*.
    #
    attr_accessor :code

    # ե񤭤 (<b>@backup</b> ͥ褵). ǥեȤ *false*.
    attr_accessor :force

    # ѴԤʤ. ǥեȤ *false*.
    attr_accessor :noop

    #
    #եХååץե񤭤̾ǽϤ.
    #ǥեȤ *false*.
    #
    attr_accessor :protect

    # Ĺʥåɽ. ǥեȤ *false*.
    attr_accessor :verbose

    #
    #ʸȽѤɤ߹ߥ(byte).
    #ǥեȤ *DEF_SIZE*. ¤ *MAX_SIZE*.
    #
    attr_reader :size

    #
    #<b>@size</b> ѥ᥽å.
    # _size_  1 ʲ *MAX_SIZE* ʾͿ㳰 *ArgumentError* ֤.
    #
    def size=(size)
      size = size.to_i
      if size < 1
        raise ArgumentError, 'Invalid size.'

      elsif size > MAX_SIZE
        raise ArgumentError, "Too big (Max: #{MAX_SIZE} byte)."

      else
        @size = size
        puts %Q(set guess size: "#{@size}")  if @verbose
      end
    end

    #
    #_path_ (<b>@io</b>) ʸɤȽꤷ, ʸɤꤹ.
    #
    def set_code(path, code = nil)
      begin
        guess_code = Kconv.guess(@io.read(@size).to_s)
        if guess_code == Kconv::BINARY
          @lf.warn "Warning: #{path}: Binary files are not supported."  if @warn
          return false

        elsif @code
          return txt2num(@code)

        elsif guess_code == Kconv::ASCII && code
          return txt2num(code)

        else
          return guess_code
        end

      rescue
        STDERR.puts $!
        return false

      ensure
        @io.rewind
      end
    end
    private :set_code

    #
    #ե _to_  mode, owner  _from_ ͤѹ.
    #<b>@backup</b>  <b>@force</b>  *true* ξϥե̾ѹԤ.
    #
    def change_file_stat(from, to)
      stat = File::Stat.new(from)
      File.chmod(stat.mode, to)
      begin
        File.chown(stat.uid, stat.gid, to)

      rescue
        if @warn
          if @backup || @force
            file = from

          else
            file = to
          end
          @lf.warn %Q( + Warning: Owner and group of "#{file}" is not preserved.)
        end
      end

      if @force
        File.rename(to, from)
        @lf.puts ' |', " |rename: #{to} => #{File.basename(from)}"  if @verbose

      elsif @backup
        ext = '.bk'
        if @protect && File.exist?(from + ext)
          @lf.warn " + Warning: #{from}#{ext}: Already exists."  if @warn
          num = 1
          while File.exist?(from + '_' + num.to_s + ext)
            @lf.warn " + Warning: #{from}_#{num}#{ext}: Already exists."  if @warn
            num += 1
          end
          bk_name = '_' << num.to_s << ext

        else
          bk_name = ext
        end

        File.rename(from, from + bk_name)
        File.rename(to, from)
        @lf.puts ' |', " |backup: #{from} => #{File.basename(from)}#{bk_name}"  if @verbose
      end
    end
    private :change_file_stat

    #
    #_path_ (<b>@io</b>)  _from_  _to_ ִ.
    #<b>@force</b>, <b>@backup</b>, <b>@noop</b> ͤ *false* ξ
    #̾˥ղäХååץե.
    #
    def exec_sed(from, to, path, code)
      return false  unless code
      code_name = num2txt(code)
      puts "code : #{code_name}"  if @verbose

      unless @io.any? { |line| Kconv.toeuc(line) =~ from }
        @lf.warn "#{path}: Nothing to match it."  if @verbose
        return false
      end
      @io.rewind
      @lf.puts "\n#{path}: #{code_name}", SEP_LIST

      line_length = @io.read.scan(/$/).size.to_s.length
      @io.rewind

      unless @noop
        num = ''
        begin
          output_path = path + '_'
          if @protect
            while File.exist?(output_path + num.to_s)
              @lf.warn " + Warning: #{output_path}#{num}: Already exists."  if @warn
              num = num.to_i + 1
            end
            open_flag = File::WRONLY | File::CREAT | File::EXCL

          else
            open_flag = 'w'
          end
          output_io = open(output_path << num.to_s, open_flag)

        rescue Errno::EEXIST
          @lf.warn " + Warning: #{output_path}: Already exists."  if @warn
          num = num.to_i + 1
          retry

        rescue
          @lf.warn " + Warning: #{output_path}: Permission denied."  if @warn
          num = num.to_i + 1
          retry
        end
        output_lf = LocaleFilter.new(code_name, output_io)
        @lf.puts " |open: #{output_path}"  if @debug
      end

      @io.each { |line|
        line = Kconv.toeuc(line)
        n = sprintf("%*d", line_length, @io.lineno)
        if line =~ from
          @lf.puts " |#{n}: --- #{line}"
          @lf.puts " |#{n}: +++ #{line.gsub!(from, to)}"
          puts " *#{n}:#{code_name}"  if @debug

        elsif @debug
          puts " |#{n}:#{code_name}"
        end
        output_lf.print line  unless @noop
      }
      @io.close

      unless @noop
        output_io.close
        @lf.puts " |close: #{output_path}"  if @debug
        change_file_stat(path, output_path)
      end
      puts SEP_PATH
    end
    private :exec_sed

    #
    #_path_ ɽ _from_ ˰פʸ _to_ ִ.
    #̥եִ exec_sed ǹԤ.
    #
    def sed(from, to, *path)
      path     = Dir.entries(Dir.pwd)  if path.to_s.empty?
      pathlist = get_file(path)
      raise ArgumentError,  'Error: No entry.'  unless pathlist

      guess_code = Kconv.guess(to.to_s)
      case guess_code
      when Kconv::ASCII, Kconv::BINARY, Kconv::UNKNOWN
        to_code = nil
      else
        to_code = guess_code
      end
      from = /#{Kconv.toeuc(from.to_s)}/e
      to   = Kconv.toeuc(to.to_s)

      puts %Q(set locale : "#{@lf.locale}")  if @verbose
      puts SEP_PATH, SEP_EXPLAIN, SEP_PATH

      pathlist.each { |path|
        @io = open(path)
        begin
          exec_sed(from, to, path, set_code(path, to_code))

        rescue
          STDERR.puts $!
          false

        ensure
          @io.close  unless @io.closed?
        end
      }
    end
  end

  #
  #Ƶǽղä File#chmod.
  #ץμबƤ FileUtils#chmod_R ȤۤƱ.
  #
  #=== Options
  #FullPathList Υץ⻲Ȥ.
  #verbose:: ԤƤɽ.
  #noop::    ºݤνϹԤʤ.
  #
  #===Future Plans
  #* 8ʿȤäǤʡ
  #
  def chmod(mode, list, options = {})
    raise ArgumentError, 'Error: No entry.'  if list.to_s.empty?
    options[:recursive] = true
    options[:warn]      = false
    begin
      pl = FullPathList.new(options, list)
    rescue
      STDERR.puts $!
      return false
    end
    return  unless pl.pathlist
    lf = LocaleFilter.new
    message = %Q(: Change to "#{mode}".)

    pl.pathlist.each { |path|
      lf.print path, message, "\n"    if options[:verbose]
      puts File::Stat.new(path).mode  if options[:debug]

      unless options[:noop]
        File.chmod(mode, path)  rescue STDERR.puts $!
      end
    }
  end
  module_function :chmod

  #
  #Ƶǽղä File#chown.
  #ץμबƤ FileUtils#chown_R ȤۤƱ.
  #
  #=== Options
  #FullPathList Υץ⻲Ȥ.
  #verbose:: ԤƤɽ.
  #noop::    ºݤνϹԤʤ.
  #
  #===Future Plans
  #* 桼̾Ȥäǽˤ롩
  #  * Etc#getpwnam ȤФǤ
  #
  def chown(owner, group, list, options = {})
    raise ArgumentError, 'Error: No entry.'  if list.to_s.empty?
    options[:recursive] = true
    options[:warn]      = false
    begin
      pl = FullPathList.new(options, list)
    rescue
      STDERR.puts $!
      return false
    end
    return  unless pl.pathlist
    lf = LocaleFilter.new
    message = %Q(: Change to owner: "#{owner}", group: "#{group}".)

    pl.pathlist.each { |path|
      lf.print path, message, "\n"  if options[:verbose]

      unless options[:noop]
        File.chown(owner, group, path)  rescue STDERR.puts $!
      end
    }
  end
  module_function :chown

  #
  #== Summary
  #RecursiveUtils::Rename, RecursiveUtils::Sed,
  #RecursiveUtils#chmod, RecursiveUtils#chown Υեȥ
  #
  #== How to Use
  #  1) recursiveutils.rb ¸¤ղä
  #     Ruby Υ饤֥ѥ֤.
  #
  #  2) ѥ̤ä rename.rb, rsed.rb, chmod.rb, chown.rb
  #     Ȥ̾ǥܥå󥯤.
  #
  #  3) ܥå󥯤¹Ԥ̾б᥽åɤ
  #     ¹Ԥ.
  #
  #== Methods
  #===Public Methods
  #եȥ:: rename, sed
  #
  #===Private Methods
  #:: check_ver, usage, help, version, error, check_code
  #
  #===Future Plans
  #* locale 򸫤ƥإפθڤؤ
  #
  module FrontEnd
    require 'optparse'

    #
    #饤֥ΥСåѥ᥽å.
    #*now*  *req* 㤤 *false* ֤.
    #
    def check_ver(now, req)
      a_now = now.split(/[\.| ]/).map { |i| i.to_i }
      a_req = req.split(/[\.| ]/).map { |i| i.to_i }

      return true   if a_now[0] > a_req[0]
      return false  if a_now[0] < a_req[0]

      return true   if a_now[1] > a_req[1]
      return false  if a_now[1] < a_req[1]

      return true   if a_now[2] > a_req[2]
      return true   if ! a_now[3] && a_now[2] == a_req[2]

      return false
    end
    private :check_ver

    #
    #USAGE ɽѥ᥽å.
    #
    def usage(option)
      @stdout.puts "  USAGE:#{@usage}\n\n  #{option}"
      exit true
    end
    private :usage

    #
    #إɽѥ᥽å.
    #
    def help(option)
      message = "  NAME:
      #{@scriptname} - #{@summary}

  SYNOPSIS:#{@usage}

  DESCRIPTION:#{@description}

  #{option}"

      message << "\n  EXAMPLE:#{@example}\n"  if @example

      message << "\n  ENVIRONMENT:#{@env}\n"  if @env

      message << "
  VERSION:
      #{@scriptname} Version #{@version}, Last Update: #{@update}

  LIBRARY VERSION:
      RecursiveUtils Version #{RecursiveUtils::VERSION}, Last Update: #{RecursiveUtils::UPDATE}
      LocaleFilter   Version #{LocaleFilter::VERSION}, Last Update: #{LocaleFilter::UPDATE}

  DEVELOPERS:#{@developers}
        All Rights Reserved.

  URL:
      #{RecursiveUtils::URL}

  HISTORY:#{@history}"

      message << "\n\n  TODO:#{@todo}"  if @todo

      @stdout.puts message
      exit true
    end
    private :help

    #
    #Version ɽѥ᥽å.
    #
    def version
      puts "#{@scriptname} Ver. #{@version}, Last Update: #{@update}"
      exit true
    end
    private :version

    #
    #顼åɽѥ᥽å.
    #
    def error(message)
      STDERR.puts "#{@scriptname}: #{message}", "  #{@usage}",
      %Q(\nType "#{@scriptname} --help" for advanced help.)
      exit false
    end
    private :error

    #
    #ʸɥåѥ᥽å.
    #
    def check_code(code)
      num = LocaleFilter::CodeConverter.txt2num(code)
      case num
      when Kconv::ASCII, Kconv::BINARY, Kconv::UNKNOWN
        raise ArgumentError, "#{code}: Invalide character code."

      else
        return LocaleFilter::CodeConverter.num2txt(num)
      end
    end
    private :check_code

    #
    #===Summary
    #RecursiveUtils::Rename Υեȥ.
    #
    #===Synopsis
    #  rename.rb [OPTIONS] from to file [file...]
    #  rename.rb --code CODE file [file...]
    #
    #===Options
    #  -c, --code CODE    convert character code of filename
    #                     (sjis, euc, jis, utf8, utf16)
    #  -d, --down         convert to small letter
    #  -D, --debug        execute with debug mode (for developer)
    #  -e, --extension    execute only extension
    #  -f, --force        rename without confirmations
    #  -h, --usage        display usage
    #  -H, --help         display detailed help
    #  -n, --no-exec      do not rename, display converted list only
    #  -q, --quiet        execute with quiet mode
    #  -r, --recursive    execute recurrently
    #  -u, --up           convert to capital letter
    #  -v, --version      display version information
    #
    #===Future Plans
    #* ܸΥإפ
    #
    def rename(argv = ARGV)
      extend FrontEnd

      @scriptname  = 'rename.rb'

      @version     = '3.3'

      @update      = '2007/10/05'

      @summary     = 'rename multiple files'

      @description = '
      Rename multiple files at a time by regular expression.
      Convert character code of multiple filenames at a time.'

      @usage       = "
      #{@scriptname} [OPTIONS] from to file [file...]
      #{@scriptname} --code CODE file [file...]"

      @developers  = '
      Yasuhiro Morikawa  <morikawa@ep.sci.hokudai.ac.jp>
      Koichi   Michimasa <michi@ep.sci.hokudai.ac.jp>
      Daisuke  Tsukahara <daktu32@ep.sci.hokudai.ac.jp>'

      @history     = '
      2007/10/05 modified by michi,    Ver. 3.3
      2007/05/27 modified by michi,    Ver. 3.2
      2007/05/23 modified by michi,    Ver. 3.1
      2007/05/19 modified by michi,    Ver. 3.0.1
      2007/05/09 modified by michi,    Ver. 3.0
      2007/01/19 modified by michi,    Ver. 2.3.1
      2007/01/08 modified by michi,    Ver. 2.3
      2006/11/17 modified by michi,    Ver. 2.2
      2006/11/13 modified by michi,    Ver. 2.1
      2006/11/04 modified by michi,    Ver. 2.0
      2006/10/30 modified by michi,    Ver. 1.3
      2006/10/25 modified by michi,    Ver. 1.2
      2006/10/24 modified by morikawa, Ver. 1.1
      2006/10/24 modified by michi,    Ver. 1.0
      2003/12/20 modified by daktu32,  Ver. 0.2
      2003/11/18 modified by daktu32,  Ver. 0.1.1
      2003/09/?? created  by daktu32,  Ver. 0.1'

      @todo       = '
      make japanese help'

      lf_ver = '2.4.1'
      unless check_ver(LocaleFilter::VERSION, lf_ver)
        STDERR.puts %Q(error: "#{@scriptname}" required "LocaleFilter" Version #{lf_ver} or later.)
        return false
      end

      op      = OptionParser.new('OPTIONS:', 18, ' ' * 6)
      opts    = Hash.new
      @stdout = STDOUT

      op.on('-c', '--code CODE', String,
            'convert character code of filename', '(sjis, euc, jis, utf8, utf16)'
            ) { |i| opts[:code] = check_code(i) }

      op.on('-d', '--down',
            'convert to small letter'
            ) { opts[:downcase] = true }

      op.on('-D', '--debug',
            'execute with debug mode (for developer)'
            ) { opts[:debug] = true }

      op.on('-e', '--extension',
            'execute only extension'
            ) { opts[:extention] = true }

      op.on('-f', '--force',
            'rename without confirmations'
            ) { opts[:force] = true }

      op.on('-h', '--usage',
            'display usage'
            ) { usage(op.help) }

      op.on('-H', '--help',
            'display detailed help'
            ) { help(op.help) }

      op.on('-n', '--no-exec',
            'do not rename, display converted list only'
            ) { opts[:noop] = true }

      op.on('-q', '--quiet',
            'execute with quiet mode'
            ) { opts[:quiet] = true }

      op.on('-r', '--recursive',
            'execute recurrently'
            ) { opts[:recursive] = true }

      op.on('-u', '--up',
            'convert to capital letter'
            ) { opts[:upcase] = true }

      op.on('-v', '--version',
            'display version information'
            ) { version }

      op.parse!(argv)  rescue error($!)

      if opts[:code] || opts[:upcase] || opts[:downcase]
        error('No argument.')  if argv.empty?

      elsif argv.size < 3
        error('No argument.')
      end

      rf = RecursiveUtils::Rename.new(opts)
      rf.dir = true

      if opts[:debug]
        lf = LocaleFilter.new
        puts "set locale : #{lf.locale}"
      end

      begin
        if opts[:code]
          rf.convert(opts[:code], argv)

        elsif opts[:upcase]
          rf.upcase(argv)

        elsif opts[:downcase]
          rf.downcase(argv)

        else
          rf.rename(argv[0], argv[1], argv[2..-1])
        end

      rescue => message
        error(message)
      end
    end
    module_function :rename

    #
    #===Summary
    #RecursiveUtils::Sed Υեȥ.
    #
    #===Synopsis
    #   rsed.rb [OPTIONS] from to files [file...]
    #
    #===Options
    #  -b, --backup       ե *.bk Ȥ̾¸,
    #                     ꥸʥΥե񤭤.
    #  -c, --code CODE    եʸɤꤹ.
    #                     (sjis, euc, jis, utf8, utf16)
    #  -f, --force        ꥸʥΥե񤭤.
    #  -D, --debug        ǥХåѥåϤ. (ȯ)
    #  -h, --usage        Usage ȥץϤ.
    #  -H, --help         ܺ٤ʥإפϤ.
    #  -n, --no-exec      ¹Է̤ΤɸϤ˽Ϥ, ¹ԤϤʤ.
    #  -p, --protect      եХååץեκ
    #                     ¸Υե񤭤ʤ.
    #  -r, --recursive    ƵŪ˼¹Ԥ.
    #  -s, --size BYTES   եʸȽɤ߹
    #                     ե륵ꤹ.
    #                     (default : 1024 byte, max: 1048576 byte)
    #  -v, --version      СϤ.
    #  -V, --verbose      ĹʥåϤ.
    #
    #===Future Plans
    #
    #* ХååץեγĥҤǤ褦ˤ
    #* ѸΥإפ
    #
    def sed(argv = ARGV)
      extend FrontEnd

      @scriptname = 'rsed.rb'

      @version     = '5.4'

      @update      = '2007/10/05'

      @summary     = 'Ƶǽդ Ruby  sed'

      @description = %Q(
      #{@scriptname}  Ƶǽ Ruby  sed Ǥ
      default ǤϡfileפǻꤷѥΥեФ
      fromפǻꤷɽѥ򸡺toפ
      ִ塢 "file_" Τ褦˥ꥸʥΥե̾
      Ǹ˥СղäեؽϤޤ)

      @usage      = "
      #{@scriptname} [OPTIONS] from to file [file...]"

      @example    = %Q(
      #{@scriptname} -f text2002.html text2004.html ./html/200?.html

      ./html/200?.html ˳ե "text2002.html" Ȥ
      ʸ "text2004.html" Ѵޤ)

      @developers = '
      Koichi   Michimasa <michi@ep.sci.hokudai.ac.jp>
      Yasuhiro Morikawa  <morikawa@ep.sci.hokudai.ac.jp>
      Daisuke  Tsukahara <daktu32@ep.sci.hokudai.ac.jp>'

      @history    = '
      2007/10/05  ƻ  Ver. 5.4
      2007/05/27  ƻ  Ver. 5.3
      2007/05/26  ƻ  Ver. 5.2
      2007/05/23  ƻ  Ver. 5.1
      2007/05/09  ƻ  Ver. 5.0
      2007/01/19  ƻ  Ver. 4.3
      2006/11/21  ƻ  Ver. 4.2
      2006/11/17  ƻ  Ver. 4.1
      2006/11/13  ƻ  Ver. 4.0
      2006/11/09  ͸  Ver. 3.7
      2006/10/30  ƻ  Ver. 3.6
      2006/10/20  ƻ  Ver. 3.5
      2006/10/14  ƻ  Ver. 3.4
      2006/10/13  ƻ  Ver. 3.3
      2006/10/02    Ver. 3.2
      2006/10/02    Ver. 3.1
      2006/10/01  ƻ  Ver. 3.0
      2006/04/15    Ver. 2.5
      2004/12/23    
      2004/10/26    
      2004/08/06    
      2004/02/12    
      2004/02/10    
      2004/01/11  ͸  '

      @todo       = '
      ХååץեγĥҤǤ褦ˤ롣
      ѸΥإפ롣'

      lf_ver = '2.4.1'
      unless check_ver(LocaleFilter::VERSION, lf_ver)
        STDERR.puts %Q(error: "#{@scriptname}" required "LocaleFilter" Version #{lf_ver} or later.)
        return false
      end

      # ץ (banner, width, indent)
      op      = OptionParser.new('OPTIONS:', 18, ' ' * 6)
      opts    = Hash.new
      lf      = LocaleFilter.new
      @stdout = lf

      op.on('-b', '--backup',
            'ե *.bk Ȥ̾¸',
            'ꥸʥΥե񤭤롣'
            ) { opts[:backup] = true }

      op.on('-c', '--code CODE', String,
            'եʸɤꤹ롣',
            '(sjis, euc, jis, utf8, utf16)'
            ) { |i| opts[:code] = check_code(i) }

      op.on('-f', '--force',
            'ꥸʥΥե񤭤롣'
            ) { opts[:force] = true }

      op.on('-D', '--debug',
            'ǥХåѥåϤ롣 (ȯ)'
            ) { opts[:debug] = true }

      op.on('-h', '--usage',
            'Usage ȥץϤ롣'
            ) { usage(op.help) }

      op.on('-H', '--help',
            'ܺ٤ʥإפϤ롣'
            ) { help(op.help) }

      op.on('-n', '--no-exec',
            '¹Է̤ΤɸϤ˽Ϥ¹ԤϤʤ'
            ) { opts[:noop] = true }

      op.on('-p', '--protect',
            'եХååץեκ',
            '¸Υե񤭤ʤ'
            ) { opts[:protect] = true }

      op.on('-r', '--recursive',
            'ƵŪ˼¹Ԥ롣'
            ) { opts[:recursive] = true }

      op.on('-s', '--size BYTES', Integer,
            'եʸȽɤ߹',
            'ե륵ꤹ롣',
            "(default : #{Sed::DEF_SIZE} byte, max: #{Sed::MAX_SIZE} byte)"
            ) { |i| opts[:size] = i }

      op.on('-v', '--version',
            'СϤ롣'
            ) { version }

      op.on('-V', '--verbose',
            'ĹʥåϤ롣'
            ) { opts[:verbose] = true, opts[:warn] = true }

      op.parse!(argv)  rescue error($!)
      error('No argument.')  if argv.size < 3

      begin
        rs = Sed.new(opts)
        rs.sed(argv[0], argv[1], argv[2..-1])

      rescue => message
        if message.to_s =~ /Invalid size|Too big/
          message = "#{opts[:size]}: #{message}"
          @usage  = "\n"
          flag    = false
          op.to_a.each { |i|
            case i
            when /^\s*\-s/
              flag = true

            when /^\s*\-v/
              break
            end
            @usage << lf.kconv(i).to_s.chomp << "\n" if flag
          }
        end
        error(message)
      end
    end
    module_function :sed

    #
    #===Summary
    #RecursiveUtils#chmod Υեȥ.
    #
    #===Synopsis
    #  chmod.rb [OPTIONS] mode file [file...]
    #
    #===Options
    #  -d, --dir          ǥ쥯ȥоݤ˴ޤ롣
    #  -D, --debug        ǥХåѥåϤ롣 (ȯ)
    #  -h, --usage        Usage ȥץϤ롣
    #  -H, --help         ܺ٤ʥإפϤ롣
    #  -n, --no-exec      ¹Է̤ΤɸϤ˽Ϥ¹ԤϤʤ
    #  -o, --owner        ͭʤե롣
    #  -r, --readable     ɤ߹߸ʤե롣
    #  -v, --version      СϤ롣
    #  -V, --verbose      ĹʥåϤ롣
    #
    #===Future Plans
    #* 8ʿȤäǤ褦ˤ
    #* ѸΥإפ
    #
    def chmod(argv = ARGV)
      extend FrontEnd

      @scriptname  = 'chmod.rb'

      @version     = '0.1'

      @update      = '2007/10/05'

      @summary     = 'Ƶǽդ Ruby  chmod'

      @description = %Q(
      #{@scriptname}  Ƶɲä Ruby  chmod Ǥ
      File#chmod  RecursiveUtils::FullPathList 𤷤
      ƵŪ˽Ƥޤ
      FullPathList ȼΥץɲäƤ
      FileUtils#chmod_R ȤۤƱǤ
      UNIX ޥɤ chmod Ȥ mode λλ㤦Τ
      դƲ)

      @usage       = "
      #{@scriptname} [OPTIONS] mode file [file...]"

      @developers  = '
      Koichi   Michimasa <michi@ep.sci.hokudai.ac.jp>'

      @history     = '
      2007/10/05 created  by michi, Ver. 0.1'

      @todo       = '
      8ʿȤäǤ褦ˤ롣
      ѸΥإפ롣'

      lf_ver = '2.4.1'
      unless check_ver(LocaleFilter::VERSION, lf_ver)
        STDERR.puts %Q(error: "#{@scriptname}" required "LocaleFilter" Version #{lf_ver} or later.)
        return false
      end

      op      = OptionParser.new('OPTIONS:', 18, ' ' * 6)
      opts    = Hash.new
      lf      = LocaleFilter.new
      @stdout = lf

      op.on('-d', '--dir',
          'ǥ쥯ȥоݤ˴ޤ롣'
          ) { opts[:dir] = true }

      op.on('-D', '--debug',
            'ǥХåѥåϤ롣 (ȯ)'
            ) { opts[:debug] = true }

      op.on('-h', '--usage',
            'Usage ȥץϤ롣'
            ) { usage(op.help) }

      op.on('-H', '--help',
            'ܺ٤ʥإפϤ롣'
            ) { help(op.help) }

      op.on('-n', '--no-exec',
            '¹Է̤ΤɸϤ˽Ϥ¹ԤϤʤ'
            ) { opts[:noop] = true }

      op.on('-o', '--owner',
          'ͭʤե롣'
          ) { opts[:owner] = true }

      op.on('-r', '--readable',
          'ɤ߹߸ʤե롣'
          ) { opts[:readable] = true }

      op.on('-v', '--version',
            'СϤ롣'
            ) { version }

      op.on('-V', '--verbose',
            'ĹʥåϤ롣'
            ) { opts[:verbose] = true, opts[:warn] = true }

      op.parse!(argv)  rescue error($!)

      error('No argument.') if argv.size < 2

      puts "set locale : #{lf.locale}"  if opts[:debug]

      begin
        RecursiveUtils.chmod(argv[0], argv[1..-1], opts)

      rescue => message
        error(message)
      end
    end
    module_function :chmod

    #
    #===Summary
    #RecursiveUtils#chown Υեȥ.
    #
    #===Synopsis
    #  chown.rb [OPTIONS] mode file [file...]
    #
    #===Options
    #  -d, --dir          ǥ쥯ȥоݤ˴ޤ롣
    #  -D, --debug        ǥХåѥåϤ롣 (ȯ)
    #  -h, --usage        Usage ȥץϤ롣
    #  -H, --help         ܺ٤ʥإפϤ롣
    #  -n, --no-exec      ¹Է̤ΤɸϤ˽Ϥ¹ԤϤʤ
    #  -o, --owner        ͭʤե롣
    #  -r, --readable     ɤ߹߸ʤե롣
    #  -v, --version      СϤ롣
    #  -V, --verbose      ĹʥåϤ롣
    #
    #===Future Plans
    #* ѸΥإפ
    #
    def chown(argv = ARGV)
      extend FrontEnd

      @scriptname  = 'chown.rb'

      @version     = '0.1'

      @update      = '2007/10/05'

      @summary     = 'Ƶǽդ Ruby  chown'

      @description = %Q(
      #{@scriptname}  Ƶɲä Ruby  chown Ǥ
      File#chown  RecursiveUtils::FullPathList 𤷤
      ƵŪ˽Ƥޤ
      FullPathList ȼΥץɲäƤ
      FileUtils#chown_R ȤۤƱǤ
      UNIX ޥɤ chown Ȥ owner, group λλ
      㤦ΤդƲ)

      @usage       = "
      #{@scriptname} [OPTIONS] owner group file [file...]"

      @developers  = '
      Koichi   Michimasa <michi@ep.sci.hokudai.ac.jp>'

      @history     = '
      2007/10/05 created  by michi, Ver. 0.1'

      @todo       = '
      ѸΥإפ롣'

      lf_ver = '2.4.1'
      unless check_ver(LocaleFilter::VERSION, lf_ver)
        STDERR.puts %Q(error: "#{@scriptname}" required "LocaleFilter" Version #{lf_ver} or later.)
        return false
      end

      op      = OptionParser.new('OPTIONS:', 18, ' ' * 6)
      opts    = Hash.new
      lf      = LocaleFilter.new
      @stdout = lf

      op.on('-d', '--dir',
          'ǥ쥯ȥоݤ˴ޤ롣'
          ) { opts[:dir] = true }

      op.on('-D', '--debug',
            'ǥХåѥåϤ롣 (ȯ)'
            ) { opts[:debug] = true }

      op.on('-h', '--usage',
            'Usage ȥץϤ롣'
            ) { usage(op.help) }

      op.on('-H', '--help',
            'ܺ٤ʥإפϤ롣'
            ) { help(op.help) }

      op.on('-n', '--no-exec',
            '¹Է̤ΤɸϤ˽Ϥ¹ԤϤʤ'
            ) { opts[:noop] = true }

      op.on('-o', '--owner',
          'ͭʤե롣'
          ) { opts[:owner] = true }

      op.on('-r', '--readable',
          'ɤ߹߸ʤե롣'
          ) { opts[:readable] = true }

      op.on('-v', '--version',
            'СϤ롣'
            ) { version }

      op.on('-V', '--verbose',
            'ĹʥåϤ롣'
            ) { opts[:verbose] = true, opts[:warn] = true }

      op.parse!(argv)  rescue error($!)

      error('No argument.') if argv.size < 3

      puts "set locale : #{lf.locale}"  if opts[:debug]

      begin
        RecursiveUtils.chown(argv[0], argv[1], argv[2..-1], opts)

      rescue => message
        error(message)
      end
    end
    module_function :chown
  end
end

# exit  unless $0 == __FILE__
case File.basename($0)
when /rename/
  RecursiveUtils::FrontEnd.rename

when /sed/
  RecursiveUtils::FrontEnd.sed

when /chmod/
  RecursiveUtils::FrontEnd.chmod

when /chown/
  RecursiveUtils::FrontEnd.chown

else
  puts "recursiveutils.rb Ver. #{RecursiveUtils::VERSION}, Last Update: #{RecursiveUtils::UPDATE}"
end
