require 'rdoc/markup/to_html'

##
# Subclass of the RDoc::Markup::ToHtml class that supports looking up words in
# the AllReferences list. Those that are found (like AllReferences in this
# comment) will be hyperlinked

class RDoc::Markup::ToHtmlCrossref < RDoc::Markup::ToHtml

  attr_accessor :context

  ##
  # We need to record the html path of our caller so we can generate
  # correct relative paths for any hyperlinks that we find

  def initialize(from_path, context, show_hash)
    super()

    # class names, variable names, or instance variables
    @markup.add_special(/(
                           # A::B.meth(**) (for operator and assignment in Fortran 90 or 95)
                           \b\w+(::\w+)*[\.\#]\w+(\([\.\w+\*\/\+\-\=\<\>]+\))?
                           # meth(**) (for operator and assignment in Fortran 90 or 95)
                         | \#\w+(\([.\w\*\/\+\-\=\<\>]+\))?
                         | \b([A-Z]\w*(::\w+)*[.\#]\w+)  #    A::B.meth
                         | \b([A-Z]\w+(::\w+)*)          #    A::B
                         | \#\w+[!?=]?                   #    #meth_name
                         | \\?\b\w+([_\/\.]+\w+)*[!?=]?  #    meth_name
                         )/x,
                        :CROSSREF)

    # file names
    @markup.add_special(/(
                           ((\/|\.\.\/|\.\/|\w)[\w\#\/\.\-\~\:]*[!?=]?) # file_name
                         | ((\/|\.\.\/|\.\/|\w)[\w\#\/\.\-\~\:]*(\([\.\w+\*\/\+\-\=\<\>]+\))?)
                         )/x, 
                        :CROSSREFFILE)

    @from_path = from_path
    @context = context
    @show_hash = show_hash

    @seen = {}
    @seen_file = {}
  end

  ##
  # We're invoked when any text matches the CROSSREF pattern
  # (defined in MarkUp). If we fine the corresponding reference,
  # generate a hyperlink. If the name we're looking for contains
  # no punctuation, we look for it up the module/class chain. For
  # example, HyperlinkHtml is found, even without the Generator::
  # prefix, because we look for it in module Generator first.

  def handle_special_CROSSREF(special)
    name = special.text

    return @seen[name] if @seen.include? name

    if name[0,1] == '#' then
      lookup = name[1..-1]
      name = lookup unless @show_hash
    else
      lookup = name
    end

    # Find class, module, or method in class or module.
    if /([A-Z]\w*)[.\#](\w+[!?=]?)/ =~ lookup then
      container = $1
      method = $2
      ref = @context.find_symbol container, method
    elsif /([A-Za-z]\w*)[.\#](\w+(\([\.\w+\*\/\+\-\=\<\>]+\))?)/ =~ lookup then
      container = $1
      method = $2
      ref = @context.find_symbol container, method
    else
      ref = @context.find_symbol lookup
    end

    out = if lookup =~ /^\\/ then
            $'
          elsif ref and ref.document_self then
            "<a href=\"#{ref.as_href(@from_path)}\">#{name}</a>"
          else
            name
          end

    @seen[name] = out

    out
  end

  #
  # CROSSREFFILE is similar to CROSSREF. But this pattern is
  # hit to filenames or methods in files
  #
  def handle_special_CROSSREFFILE(special)
    name = special.text

    return @seen_file[name] if @seen_file.include? name

    # Find file, or method in file
    if /([\w\/\.].*\.\w+)[.\#](.*)/ =~ name
      file_name = $1
      method = $2
      ref = @context.find_file file_name, method
    else
      ref = @context.find_file name
    end

    out = if ref and ref.document_self then
            "<a href=\"#{ref.as_href(@from_path)}\">#{name}</a>"
          else
            name
          end

    @seen_file[name] = out

    out

  end

end
