!-----------------------------------------------------------------------
!     Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!-----------------------------------------------------------------------

module typhoon_analy  ! 台風感度実験用スペシャル解析モジュール

  use algebra
  use Basis
  use Derivation
  use Geometry
  use Map_Function
  use Matrix_calc
  use max_min
  use statistics
  use stdio

  implicit none

  private :: search_region_1d  ! 接線平均が可能な半径を捜索する.

interface tangent_conv_scal

  module procedure tangent_conv_scal_f, tangent_conv_scal_d

end interface tangent_conv_scal

interface tangent_mean_scal

  module procedure tangent_mean_scal_f, tangent_mean_scal_d

end interface tangent_mean_scal

interface tangent_mean_anom_scal

  module procedure tangent_mean_anom_scal_f, tangent_mean_anom_scal_d

end interface tangent_mean_anom_scal

interface tangent_mean_scal_Cart

  module procedure tangent_mean_scal_Cart_f, tangent_mean_scal_Cart_d

end interface tangent_mean_scal_Cart

interface tangent_mean_anom_scal_Cart

  module procedure tangent_mean_anom_scal_Cart_f, tangent_mean_anom_scal_Cart_d

end interface tangent_mean_anom_scal_Cart

interface tangent_median_scal

  module procedure tangent_median_scal_f, tangent_median_scal_d

end interface tangent_median_scal

interface Cart_conv_scal

  module procedure Cart_conv_scal_f, Cart_conv_scal_d

end interface Cart_conv_scal

interface Cart_mean_scal

  module procedure Cart_mean_scal_f, Cart_mean_scal_d

end interface Cart_mean_scal

interface tangent_mean_vec

  module procedure tangent_mean_vec_f, tangent_mean_vec_d

end interface tangent_mean_vec

interface tangent_mean_anom_vec

  module procedure tangent_mean_anom_vec_f, tangent_mean_anom_vec_d

end interface tangent_mean_anom_vec

interface calc_taufil

  module procedure calc_taufil_f, calc_taufil_d

end interface calc_taufil

interface hydro_grad_eqb

  module procedure hydro_grad_eqb_f!, hydro_grad_eqb_d

end interface hydro_grad_eqb

interface hydro_grad_eqb_it

  module procedure hydro_grad_eqb_it_f, hydro_grad_eqb_it_d

end interface hydro_grad_eqb_it

interface hydro_grad_eqbp

  module procedure hydro_grad_eqbp_f, hydro_grad_eqbp_d

end interface hydro_grad_eqbp

!interface hydro_grad_eqbp_it

!  module procedure hydro_grad_eqbp_it_f, hydro_grad_eqbp_it_d

!end interface hydro_grad_eqbp_it

interface grad_wind_pres

  module procedure grad_wind_pres_f, grad_wind_pres_d

end interface grad_wind_pres

interface pres_grad_wind

  module procedure pres_grad_wind_f, pres_grad_wind_d

end interface pres_grad_wind

interface SPLB_Kurihara

  module procedure SPLB_Kurihara_f, SPLB_Kurihara_d

end interface SPLB_Kurihara

interface CPS_Hart

  module procedure CPS_Hart_f, CPS_Hart_d

end interface CPS_Hart

interface DC_Braun

  module procedure DC_Braun_f, DC_Braun_d

end interface DC_Braun

interface Parallax_Himawari

  module procedure Parallax_Himawari_f, Parallax_Himawari_d

end interface Parallax_Himawari

interface Parallax_Himawari_THap

  module procedure Parallax_Himawari_THap_f, Parallax_Himawari_THap_d

end interface Parallax_Himawari_THap

interface DC_Braun_SAT

  module procedure DC_Braun_SAT_f, DC_Braun_SAT_d

end interface DC_Braun_SAT

interface DC_Sat_ZNCC

  module procedure DC_Sat_ZNCC_f, DC_Sat_ZNCC_d

end interface DC_Sat_ZNCC

interface radar_pl2rz

  module procedure radar_pl2rz_f, radar_pl2rz_d

end interface radar_pl2rz

interface radar_pz2r

  module procedure radar_pz2r_f, radar_pz2r_d

end interface radar_pz2r

interface radar_rz2p

  module procedure radar_rz2p_f, radar_rz2p_d

end interface radar_rz2p

contains

subroutine tangent_conv_scal_f( x, y, xc, yc, u, r, theta, v, undef,  &
  &                             undefg, undefgc, stdopt, axis )
  ! 任意の物理量を台風の中心から接線方向へ変換するルーチン
  ! 引数の制限として, |x(1)-xc|, |xc-x(nx)|, |y(1)-yc|, |yc-y(ny)| > r. 
  ! これ以外の場合, r で与えられる距離が領域外の範囲については
  ! undef で設定された値が代入される. undef がないときはゼロが代入される.
  ! 平均化の手順は以下のとおり.
  ! (1) nr, nt のすべての点についてそれに対応する x, y 座標値を rt_2_xy で計算.
  ! (2) その地点を含む x,y グリッドの微小領域を interpo_search_2d で検索.
  ! (3) その地点を含む 4 点が出たら, その地点でのスカラー値を 4 隅のスカラー値
  !     から, 重線形内挿 interpolation_2d で計算.
  ! 本ルーチンは平面極座標グリッド値での偏差計算を行う.
  ! 接線平均を行ったのち, デカルト座標の偏差へ落とし込むには, 
  ! tangent_mean_anom_scal_r2c を使用.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  implicit none
  real, intent(in) :: x(:)  ! 右手座標系での第一成分 [m or rad]
  real, intent(in) :: y(:)  ! 右手座標系での第二成分 [m or rad]
  real, intent(in) :: u(size(x),size(y))  ! 右手座標系での平均化する値
  real, intent(in) :: xc  ! 接線平均する際の中心 x 成分. [m or rad]
  real, intent(in) :: yc  ! 接線平均する際の中心 y 成分. [m or rad]
  real, intent(in) :: r(:)  ! (xc, yc) を中心とした極座標系動径座標 [m].
  real, intent(in) :: theta(:)  ! (xc, yc) を中心とした極座標系同位角座標 [rad].
  real, intent(inout) :: v(size(r),size(theta))  ! 平均化した u のアノマリー.
  real, intent(in), optional :: undef  ! 領域外の設定値
  real, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
                ! デフォルトは "inc".
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  character(2), intent(in), optional :: axis  ! x, y の座標系
                                           ! 'xy' = デカルト座標系 [m]
                                           ! 'll' = 球面緯度経度座標系 [rad]
                                           ! デフォルトは 'xy'.
  integer :: i, j, nx, ny, nr, nt, i_undef
  real :: r_undef, r_undefg
  real :: work(size(r),size(theta))
  real :: point(size(r),size(theta),2)
  integer :: ip(size(r),size(theta),2)
  real :: tmpx(2), tmpy(2), tmpz(2,2), inter(2)
  double precision :: tmppointd1, tmppointd2
  character(1) :: undefgcflag
  character(2) :: ax_flag
!  logical, dimension(size(r)) :: undefgc_check
  logical :: ucf, stderr

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(theta)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "tangent_conv_scal" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nt, v ),  &
  &                                     "tangent_conv_scal" )
  end if

!  undefgc_check(:)=.true.
  i_undef=0

  if(present(undef))then
    r_undef=undef
  else
    r_undef=-999.0
  end if

  if(present(undefg))then
    r_undefg=undefg
  else
    r_undefg=-999.0
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:1)
     else
        undefgcflag="i"
     end if
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(axis))then
     ax_flag=axis(1:2)
  else
     ax_flag='xy'
  end if

!-- 先の引数条件をクリアしているか確認 ---
!-- この操作は行わない.

!  call search_region_1d( (/x(1),x(nx)/), (/y(1),y(ny)/), (/xc,yc/), r, nr,  &
!  &                      stdopt=stderr, axis=ax_flag(1:2) )

!-- 先に v に undef 値を入れておく.
!  do j=1,nt
!     do i=1,nrr
!        v(i,j)=r_undef
!     end do
!  end do

!-- 過程(1) ---
  if(ax_flag(1:2)=='xy')then
     do j=1,nt
        do i=1,nr
           call rt_2_xy( r(i), theta(j), point(i,j,1), point(i,j,2) )
           point(i,j,1)=xc+point(i,j,1)
           point(i,j,2)=yc+point(i,j,2)
        end do
     end do
  else
     do j=1,nt
        do i=1,nr
           if(r(i)/=0.0)then
              call rt2ll( dble(r(i)), dble(theta(j)), dble(xc), dble(yc),  &
  &                       tmppointd1, tmppointd2 )
              point(i,j,1)=real(tmppointd1)
              point(i,j,2)=real(tmppointd2)
           else
              point(i,j,1)=xc
              point(i,j,2)=yc
           end if
        end do
     end do
  end if

!-- 過程(2) ---
  do j=1,nt
     do i=1,nr
        call interpo_search_2d( x, y, point(i,j,1), point(i,j,2),  &
      &                         ip(i,j,1), ip(i,j,2), undeff=i_undef,  &
  &                             stdopt=stderr )
     end do
  end do

!-- 過程(3) ---
  do j=1,nt
     do i=1,nr
        if(ip(i,j,1)/=i_undef.and.ip(i,j,2)/=i_undef.and.  &
  &        ip(i,j,1)/=nx.and.ip(i,j,2)/=ny)then
           tmpx(1)=x(ip(i,j,1))
           tmpx(2)=x(ip(i,j,1)+1)
           tmpy(1)=y(ip(i,j,2))
           tmpy(2)=y(ip(i,j,2)+1)
           tmpz(1,1)=u(ip(i,j,1),ip(i,j,2))
           tmpz(2,1)=u(ip(i,j,1)+1,ip(i,j,2))
           tmpz(1,2)=u(ip(i,j,1),ip(i,j,2)+1)
           tmpz(2,2)=u(ip(i,j,1)+1,ip(i,j,2)+1)
           inter(1)=point(i,j,1)
           inter(2)=point(i,j,2)

           if(present(undefg))then
              ucf=undef_checker_2df( tmpz, undefg )
              if(ucf.eqv..false.)then
                 call interpolation_2d( tmpx, tmpy, tmpz, inter, work(i,j) )
              else
                 work(i,j)=r_undefg
!                 undefgc_check(i)=.false.
              end if
           else
              call interpolation_2d( tmpx, tmpy, tmpz, inter, work(i,j) )
           end if
        else
           work(i,j)=r_undef
        end if
     end do
  end do

  do j=1,nt
     do i=1,nr
        v(i,j)=work(i,j)
     end do
  end do

end subroutine tangent_conv_scal_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_conv_scal_d( x, y, xc, yc, u, r, theta, v, undef,  &
  &                             undefg, undefgc, stdopt, axis )
  ! 任意の物理量を台風の中心から接線方向へ変換するルーチン
  ! 引数の制限として, |x(1)-xc|, |xc-x(nx)|, |y(1)-yc|, |yc-y(ny)| > r. 
  ! これ以外の場合, r で与えられる距離が領域外の範囲については
  ! undef で設定された値が代入される. undef がないときはゼロが代入される.
  ! 平均化の手順は以下のとおり.
  ! (1) nr, nt のすべての点についてそれに対応する x, y 座標値を rt_2_xy で計算.
  ! (2) その地点を含む x,y グリッドの微小領域を interpo_search_2d で検索.
  ! (3) その地点を含む 4 点が出たら, その地点でのスカラー値を 4 隅のスカラー値
  !     から, 重線形内挿 interpolation_2d で計算.
  ! 本ルーチンは平面極座標グリッド値での偏差計算を行う.
  ! 接線平均を行ったのち, デカルト座標の偏差へ落とし込むには, 
  ! tangent_mean_anom_scal_r2c を使用.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  implicit none
  double precision, intent(in) :: x(:)  ! 右手座標系での第一成分 [m or rad]
  double precision, intent(in) :: y(:)  ! 右手座標系での第二成分 [m or rad]
  double precision, intent(in) :: u(size(x),size(y))  ! 右手座標系での平均化する値
  double precision, intent(in) :: xc  ! 接線平均する際の中心 x 成分. [m or rad]
  double precision, intent(in) :: yc  ! 接線平均する際の中心 y 成分. [m or rad]
  double precision, intent(in) :: r(:)  ! (xc, yc) を中心とした極座標系動径座標 [m].
  double precision, intent(in) :: theta(:)  ! (xc, yc) を中心とした極座標系同位角座標 [rad].
  double precision, intent(inout) :: v(size(r),size(theta))  ! 平均化した u のアノマリー.
  double precision, intent(in), optional :: undef  ! 領域外の設定値
  double precision, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
                ! デフォルトは "inc".
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  character(2), intent(in), optional :: axis  ! x, y の座標系
                                           ! 'xy' = デカルト座標系 [m]
                                           ! 'll' = 球面緯度経度座標系 [rad]
                                           ! デフォルトは 'xy'.
  integer :: i, j, nx, ny, nr, nt, i_undef
  double precision :: r_undef, r_undefg
  double precision :: work(size(r),size(theta))
  double precision :: point(size(r),size(theta),2)
  integer :: ip(size(r),size(theta),2)
  double precision :: tmpx(2), tmpy(2), tmpz(2,2), inter(2)
  double precision :: tmppointd1, tmppointd2
  character(1) :: undefgcflag
  character(2) :: ax_flag
!  logical, dimension(size(r)) :: undefgc_check
  logical :: ucf, stderr

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(theta)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "tangent_conv_scal" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nt, v ),  &
  &                                     "tangent_conv_scal" )
  end if

!  undefgc_check(:)=.true.
  i_undef=0

  if(present(undef))then
    r_undef=undef
  else
    r_undef=-999.0d0
  end if

  if(present(undefg))then
    r_undefg=undefg
  else
    r_undefg=-999.0d0
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:1)
     else
        undefgcflag="i"
     end if
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(axis))then
     ax_flag=axis(1:2)
  else
     ax_flag='xy'
  end if

!-- 先の引数条件をクリアしているか確認 ---
!-- この操作は行わない.

!  call search_region_1d( (/x(1),x(nx)/), (/y(1),y(ny)/), (/xc,yc/), r, nr,  &
!  &                      stdopt=stderr, axis=ax_flag(1:2) )

!-- 先に v に undef 値を入れておく.
!  do j=1,nt
!     do i=1,nrr
!        v(i,j)=r_undef
!     end do
!  end do

!-- 過程(1) ---
  if(ax_flag(1:2)=='xy')then
     do j=1,nt
        do i=1,nr
           call rt_2_xy( r(i), theta(j), point(i,j,1), point(i,j,2) )
           point(i,j,1)=xc+point(i,j,1)
           point(i,j,2)=yc+point(i,j,2)
        end do
     end do
  else
     do j=1,nt
        do i=1,nr
           if(r(i)/=0.0d0)then
              call rt2ll( r(i), theta(j), xc, yc,  &
  &                       tmppointd1, tmppointd2 )
              point(i,j,1)=tmppointd1
              point(i,j,2)=tmppointd2
           else
              point(i,j,1)=xc
              point(i,j,2)=yc
           end if
        end do
     end do
  end if

!-- 過程(2) ---
  do j=1,nt
     do i=1,nr
        call interpo_search_2d( x, y, point(i,j,1), point(i,j,2),  &
      &                         ip(i,j,1), ip(i,j,2), undeff=i_undef,  &
  &                             stdopt=stderr )
     end do
  end do

!-- 過程(3) ---
  do j=1,nt
     do i=1,nr
        if(ip(i,j,1)/=i_undef.and.ip(i,j,2)/=i_undef.and.  &
  &        ip(i,j,1)/=nx.and.ip(i,j,2)/=ny)then
           tmpx(1)=x(ip(i,j,1))
           tmpx(2)=x(ip(i,j,1)+1)
           tmpy(1)=y(ip(i,j,2))
           tmpy(2)=y(ip(i,j,2)+1)
           tmpz(1,1)=u(ip(i,j,1),ip(i,j,2))
           tmpz(2,1)=u(ip(i,j,1)+1,ip(i,j,2))
           tmpz(1,2)=u(ip(i,j,1),ip(i,j,2)+1)
           tmpz(2,2)=u(ip(i,j,1)+1,ip(i,j,2)+1)
           inter(1)=point(i,j,1)
           inter(2)=point(i,j,2)

           if(present(undefg))then
              ucf=undef_checker_2dd( tmpz, undefg )
              if(ucf.eqv..false.)then
                 call interpolation_2d( tmpx, tmpy, tmpz, inter, work(i,j) )
              else
                 work(i,j)=r_undefg
!                 undefgc_check(i)=.false.
              end if
           else
              call interpolation_2d( tmpx, tmpy, tmpz, inter, work(i,j) )
           end if
        else
           work(i,j)=r_undef
        end if
     end do
  end do

  do j=1,nt
     do i=1,nr
        v(i,j)=work(i,j)
     end do
  end do

end subroutine tangent_conv_scal_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_mean_scal_f( x, y, xc, yc, u, r, theta, v, undef,  &
  &                             undefg, undefgc, stdopt, axis )
  ! 任意の物理量を台風の中心から接線方向へ平均するルーチン
  ! このルーチンは接線風速を平均する時には用いることはできない.
  ! 接線の平均を行う際には, 別のルーチン, tangent_mean_vec の使用が必要.
  ! 引数の制限として, |x(1)-xc|, |xc-x(nx)|, |y(1)-yc|, |yc-y(ny)| > r. 
  ! これ以外の場合, r で与えられる距離が領域外の範囲については
  ! undef で設定された値が代入される. undef がないときはゼロが代入される.
  ! 平均化の手順は以下のとおり.
  ! (1) nr, nt のすべての点についてそれに対応する x, y 座標値を rt_2_xy で計算.
  ! (2) その地点を含む x,y グリッドの微小領域を interpo_search_2d で検索.
  ! (3) その地点を含む 4 点が出たら, その地点でのスカラー値を 4 隅のスカラー値
  !     から, 重線形内挿 interpolation_2d で計算.
  ! (4) nr x nt 個の内挿スカラー値が求まったら, nt 方向に平均計算 mean_1d 使用.
  ! 以上で各 nr について平均値が得られる.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  implicit none
  real, intent(in) :: x(:)  ! 右手座標系での第一成分 [m or rad]
  real, intent(in) :: y(:)  ! 右手座標系での第二成分 [m or rad]
  real, intent(in) :: u(size(x),size(y))  ! 右手座標系での平均化する値
  real, intent(in) :: xc  ! 接線平均する際の中心 x 成分. [m or rad]
  real, intent(in) :: yc  ! 接線平均する際の中心 y 成分. [m or rad]
  real, intent(in) :: r(:)  ! (xc, yc) を中心とした極座標系動径座標 [m].
  real, intent(in) :: theta(:)  ! (xc, yc) を中心とした極座標系同位角座標 [rad].
  real, intent(inout) :: v(size(r))  ! 平均化した u の値.
  real, intent(in), optional :: undef   ! 領域外の設定値
  real, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
                ! デフォルトは "inc".
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  character(2), intent(in), optional :: axis  ! x, y の座標系
                                           ! 'xy' = デカルト座標系 [m]
                                           ! 'll' = 球面緯度経度座標系 [rad]
                                           ! デフォルトは 'xy'.
  integer :: i, nx, ny, nr, nt
  real :: r_undef, r_undefg
  real :: work(size(r),size(theta))
  character(3) :: undefgcflag
  character(2) :: ax_flag
  logical, dimension(size(r)) :: undefgc_check
  logical :: stderr

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(theta)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "tangent_mean_scal" )
     call check_array_size_dmp_message( check_array_size_1d( nr, v ),  &
  &                                     "tangent_mean_scal" )
  end if

  undefgc_check(:)=.true.

  if(present(undef))then
    r_undef=undef
  else
    r_undef=-999.0
  end if

  if(present(undefg))then
    r_undefg=undefg
  else
    r_undefg=-999.0
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:3)
     else
        undefgcflag="inc"
     end if
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(axis))then
     ax_flag=axis(1:2)
  else
     ax_flag='xy'
  end if

!-- 過程(1) - (3)

  call tangent_conv_scal_f( x, y, xc, yc, u, r, theta, work, undef=r_undef,  &
  &                         undefg=r_undefg, undefgc=undefgcflag(1:3),  &
  &                         stdopt=stderr, axis=ax_flag(1:2) )

!-- 過程(4) ---
  if(present(undefg))then
     if(undefgcflag(1:1)=='i')then
        do i=1,nr
           call Mean_1d( work(i,:), v(i), error=undefg )
        end do
     else
        do i=1,nr
           if(undefgc_check(i).eqv..true.)then
              call Mean_1d( work(i,:), v(i) )
           else
              v(i)=undefg
           end if
        end do
     end if
  else
     do i=1,nr
        call Mean_1d( work(i,:), v(i) )
     end do
  end if

end subroutine tangent_mean_scal_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_mean_scal_d( x, y, xc, yc, u, r, theta, v, undef,  &
  &                             undefg, undefgc, stdopt, axis )
  ! 任意の物理量を台風の中心から接線方向へ平均するルーチン
  ! このルーチンは接線風速を平均する時には用いることはできない.
  ! 接線の平均を行う際には, 別のルーチン, tangent_mean_vec の使用が必要.
  ! 引数の制限として, |x(1)-xc|, |xc-x(nx)|, |y(1)-yc|, |yc-y(ny)| > r. 
  ! これ以外の場合, r で与えられる距離が領域外の範囲については
  ! undef で設定された値が代入される. undef がないときはゼロが代入される.
  ! 平均化の手順は以下のとおり.
  ! (1) nr, nt のすべての点についてそれに対応する x, y 座標値を rt_2_xy で計算.
  ! (2) その地点を含む x,y グリッドの微小領域を interpo_search_2d で検索.
  ! (3) その地点を含む 4 点が出たら, その地点でのスカラー値を 4 隅のスカラー値
  !     から, 重線形内挿 interpolation_2d で計算.
  ! (4) nr x nt 個の内挿スカラー値が求まったら, nt 方向に平均計算 mean_1d 使用.
  ! 以上で各 nr について平均値が得られる.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  implicit none
  double precision, intent(in) :: x(:)  ! 右手座標系での第一成分 [m or rad]
  double precision, intent(in) :: y(:)  ! 右手座標系での第二成分 [m or rad]
  double precision, intent(in) :: u(size(x),size(y))  ! 右手座標系での平均化する値
  double precision, intent(in) :: xc  ! 接線平均する際の中心 x 成分. [m or rad]
  double precision, intent(in) :: yc  ! 接線平均する際の中心 y 成分. [m or rad]
  double precision, intent(in) :: r(:)  ! (xc, yc) を中心とした極座標系動径座標 [m].
  double precision, intent(in) :: theta(:)  ! (xc, yc) を中心とした極座標系同位角座標 [rad].
  double precision, intent(inout) :: v(size(r))  ! 平均化した u の値.
  double precision, intent(in), optional :: undef   ! 領域外の設定値
  double precision, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
                ! デフォルトは "inc".
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  character(2), intent(in), optional :: axis  ! x, y の座標系
                                           ! 'xy' = デカルト座標系 [m]
                                           ! 'll' = 球面緯度経度座標系 [rad]
                                           ! デフォルトは 'xy'.
  integer :: i, nx, ny, nr, nt
  double precision :: r_undef, r_undefg
  double precision :: work(size(r),size(theta))
  character(3) :: undefgcflag
  character(2) :: ax_flag
  logical, dimension(size(r)) :: undefgc_check
  logical :: stderr

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(theta)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "tangent_mean_scal" )
     call check_array_size_dmp_message( check_array_size_1d( nr, v ),  &
  &                                     "tangent_mean_scal" )
  end if

  undefgc_check(:)=.true.

  if(present(undef))then
    r_undef=undef
  else
    r_undef=-999.0d0
  end if

  if(present(undefg))then
    r_undefg=undefg
  else
    r_undefg=-999.0d0
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:3)
     else
        undefgcflag="inc"
     end if
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(axis))then
     ax_flag=axis(1:2)
  else
     ax_flag='xy'
  end if

!-- 過程(1) - (3)

  call tangent_conv_scal_d( x, y, xc, yc, u, r, theta, work, undef=r_undef,  &
  &                         undefg=r_undefg, undefgc=undefgcflag(1:3),  &
  &                         stdopt=stderr, axis=ax_flag(1:2) )

!-- 過程(4) ---
  if(present(undefg))then
     if(undefgcflag(1:1)=='i')then
        do i=1,nr
           call Mean_1d( work(i,:), v(i), error=undefg )
        end do
     else
        do i=1,nr
           if(undefgc_check(i).eqv..true.)then
              call Mean_1d( work(i,:), v(i) )
           else
              v(i)=undefg
           end if
        end do
     end if
  else
     do i=1,nr
        call Mean_1d( work(i,:), v(i) )
     end do
  end if

end subroutine tangent_mean_scal_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_mean_anom_scal_f( x, y, xc, yc, u, r, theta, v, undef,  &
  &                                  undefg, undefgc, stdopt, axis )
  ! 任意の物理量を台風の中心から接線方向へ平均し, そのアノマリーを計算するルーチン
  ! 引数の制限として, |x(1)-xc|, |xc-x(nx)|, |y(1)-yc|, |yc-y(ny)| > r. 
  ! これ以外の場合, r で与えられる距離が領域外の範囲については
  ! undef で設定された値が代入される. undef がないときはゼロが代入される.
  ! 平均化の手順は以下のとおり.
  ! (1) nr, nt のすべての点についてそれに対応する x, y 座標値を rt_2_xy で計算.
  ! (2) その地点を含む x,y グリッドの微小領域を interpo_search_2d で検索.
  ! (3) その地点を含む 4 点が出たら, その地点でのスカラー値を 4 隅のスカラー値
  !     から, 重線形内挿 interpolation_2d で計算.
  ! (4) nr x nt 個の内挿スカラー値が求まったら, nt 方向に偏差計算 Anomaly_1d 使用.
  ! 以上で各 nr について偏差値が得られる.
  ! 本ルーチンは平面極座標グリッド値での偏差計算を行う.
  ! 接線平均を行ったのち, デカルト座標の偏差へ落とし込むには, 
  ! tangent_mean_anom_scal_r2c を使用.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  implicit none
  real, intent(in) :: x(:)  ! 右手座標系での第一成分 [m or rad]
  real, intent(in) :: y(:)  ! 右手座標系での第二成分 [m or rad]
  real, intent(in) :: u(size(x),size(y))  ! 右手座標系での平均化する値
  real, intent(in) :: xc  ! 接線平均する際の中心 x 成分. [m or rad]
  real, intent(in) :: yc  ! 接線平均する際の中心 y 成分. [m or rad]
  real, intent(in) :: r(:)  ! (xc, yc) を中心とした極座標系動径座標 [m].
  real, intent(in) :: theta(:)  ! (xc, yc) を中心とした極座標系同位角座標 [rad].
  real, intent(inout) :: v(size(r),size(theta))  ! 平均化した u のアノマリー.
  real, intent(in), optional :: undef  ! 領域外の設定値
  real, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
                ! デフォルトは "inc".
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  character(2), intent(in), optional :: axis  ! x, y の座標系
                                           ! 'xy' = デカルト座標系 [m]
                                           ! 'll' = 球面緯度経度座標系 [rad]
                                           ! デフォルトは 'xy'.
  integer :: i, j, nx, ny, nr, nt
  real :: r_undef, r_undefg
  real :: work(size(r),size(theta))
  character(3) :: undefgcflag
  character(2) :: ax_flag
  logical, dimension(size(r)) :: undefgc_check
  logical :: stderr

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(theta)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "tangent_mean_anom_scal" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nt, v ),  &
  &                                     "tangent_mean_anom_scal" )
  end if

  undefgc_check(:)=.true.

  if(present(undef))then
    r_undef=undef
  else
    r_undef=-999.0
  end if

  if(present(undefg))then
    r_undefg=undefg
  else
    r_undefg=-999.0
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:3)
     else
        undefgcflag="inc"
     end if
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(axis))then
     ax_flag=axis(1:2)
  else
     ax_flag='xy'
  end if

!-- 過程(1) - (3) ---

  call tangent_conv_scal_f( x, y, xc, yc, u, r, theta, work, undef=r_undef,  &
  &                         undefg=r_undefg, undefgc=undefgcflag(1:3),  &
  &                         stdopt=stderr, axis=ax_flag(1:2) )

!-- 過程(4) ---
  if(present(undefg))then
     if(undefgcflag(1:1)=='i')then
        do i=1,nr
           call Anomaly_1d( work(i,:), v(i,:), error=undefg )
        end do
     else
        do i=1,nr
           if(undefgc_check(i).eqv..true.)then
              call Anomaly_1d( work(i,:), v(i,:) )
           else
              do j=1,nt
                 v(i,j)=undefg
              end do
           end if
        end do
     end if
  else
     do i=1,nr
        call Anomaly_1d( work(i,:), v(i,:) )
     end do
  end if

end subroutine tangent_mean_anom_scal_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_mean_anom_scal_d( x, y, xc, yc, u, r, theta, v, undef,  &
  &                                  undefg, undefgc, stdopt, axis )
  ! 任意の物理量を台風の中心から接線方向へ平均し, そのアノマリーを計算するルーチン
  ! 引数の制限として, |x(1)-xc|, |xc-x(nx)|, |y(1)-yc|, |yc-y(ny)| > r. 
  ! これ以外の場合, r で与えられる距離が領域外の範囲については
  ! undef で設定された値が代入される. undef がないときはゼロが代入される.
  ! 平均化の手順は以下のとおり.
  ! (1) nr, nt のすべての点についてそれに対応する x, y 座標値を rt_2_xy で計算.
  ! (2) その地点を含む x,y グリッドの微小領域を interpo_search_2d で検索.
  ! (3) その地点を含む 4 点が出たら, その地点でのスカラー値を 4 隅のスカラー値
  !     から, 重線形内挿 interpolation_2d で計算.
  ! (4) nr x nt 個の内挿スカラー値が求まったら, nt 方向に偏差計算 Anomaly_1d 使用.
  ! 以上で各 nr について偏差値が得られる.
  ! 本ルーチンは平面極座標グリッド値での偏差計算を行う.
  ! 接線平均を行ったのち, デカルト座標の偏差へ落とし込むには, 
  ! tangent_mean_anom_scal_r2c を使用.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  implicit none
  double precision, intent(in) :: x(:)  ! 右手座標系での第一成分 [m or rad]
  double precision, intent(in) :: y(:)  ! 右手座標系での第二成分 [m or rad]
  double precision, intent(in) :: u(size(x),size(y))  ! 右手座標系での平均化する値
  double precision, intent(in) :: xc  ! 接線平均する際の中心 x 成分. [m or rad]
  double precision, intent(in) :: yc  ! 接線平均する際の中心 y 成分. [m or rad]
  double precision, intent(in) :: r(:)  ! (xc, yc) を中心とした極座標系動径座標 [m].
  double precision, intent(in) :: theta(:)  ! (xc, yc) を中心とした極座標系同位角座標 [rad].
  double precision, intent(inout) :: v(size(r),size(theta))  ! 平均化した u のアノマリー.
  double precision, intent(in), optional :: undef  ! 領域外の設定値
  double precision, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
                ! デフォルトは "inc".
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  character(2), intent(in), optional :: axis  ! x, y の座標系
                                           ! 'xy' = デカルト座標系 [m]
                                           ! 'll' = 球面緯度経度座標系 [rad]
                                           ! デフォルトは 'xy'.
  integer :: i, j, nx, ny, nr, nt
  double precision :: r_undef, r_undefg
  double precision :: work(size(r),size(theta))
  character(3) :: undefgcflag
  character(2) :: ax_flag
  logical, dimension(size(r)) :: undefgc_check
  logical :: stderr

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(theta)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "tangent_mean_anom_scal" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nt, v ),  &
  &                                     "tangent_mean_anom_scal" )
  end if

  undefgc_check(:)=.true.

  if(present(undef))then
    r_undef=undef
  else
    r_undef=-999.0d0
  end if

  if(present(undefg))then
    r_undefg=undefg
  else
    r_undefg=-999.0d0
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:3)
     else
        undefgcflag="inc"
     end if
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(axis))then
     ax_flag=axis(1:2)
  else
     ax_flag='xy'
  end if

!-- 過程(1) - (3) ---

  call tangent_conv_scal_d( x, y, xc, yc, u, r, theta, work, undef=r_undef,  &
  &                         undefg=r_undefg, undefgc=undefgcflag(1:3),  &
  &                         stdopt=stderr, axis=ax_flag(1:2) )

!-- 過程(4) ---
  if(present(undefg))then
     if(undefgcflag(1:1)=='i')then
        do i=1,nr
           call Anomaly_1d( work(i,:), v(i,:), error=undefg )
        end do
     else
        do i=1,nr
           if(undefgc_check(i).eqv..true.)then
              call Anomaly_1d( work(i,:), v(i,:) )
           else
              do j=1,nt
                 v(i,j)=undefg
              end do
           end if
        end do
     end if
  else
     do i=1,nr
        call Anomaly_1d( work(i,:), v(i,:) )
     end do
  end if

end subroutine tangent_mean_anom_scal_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_mean_scal_Cart_f( x, y, xc, yc, scal, r, theta,  &
  &                                  scal_mean, undef, undefg, undefgc,  &
  &                                  stdopt, axis )
  ! 台風中心から接線平均を計算し, デカルト座標系に戻す.
  ! 接線平均ルーチンを用いて物理量を平均し, その 1 次元データを確保しておく.
  ! 同時に, デカルト系での物理量の円筒座標系での radial 位置を求める.
  ! この radial の位置における接線平均値を先の 1 次元データから内挿で求める.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  implicit none
  real, intent(in) :: x(:)  ! デカルト座標系での x 座標 [m] or lon [rad]
  real, intent(in) :: y(:)  ! デカルト座標系での y 座標 [m] or lat [rad]
  real, intent(in) :: scal(size(x),size(y))  ! デカルト座標系での平均化する値.
  real, intent(in) :: xc  ! 接線平均する際の中心 x 成分 [m] or [rad].
  real, intent(in) :: yc  ! 接線平均する際の中心 y 成分 [m] or [rad].
  real, intent(in) :: r(:)  ! 平均化したときの動径方向の座標(xc からの値を入れる).
  real, intent(in) :: theta(:)  ! 平均化するときの接線方向の座標 [rad].
  real, intent(inout) :: scal_mean(size(x),size(y))  ! デカルト系での平均値.
  real, optional :: undef  ! 内挿値が見つからないときの未定義値.
                           ! デフォルトでは dcl の未定義値
  real, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  character(2), intent(in), optional :: axis  ! x, y の座標系
                                           ! 'xy' = デカルト座標系 [m]
                                           ! 'll' = 球面緯度経度座標系 [rad]
                                           ! デフォルトは 'xy'.
  integer :: j, k, nx, ny, nr, nt, itmpr
  real :: tmp(size(r))
  real :: tmpr, tmp_anom, undeff
  character(2) :: ax
  character(3) :: undefgcflag
  logical :: stderr

  if(present(undef))then
     undeff=undef
  else
     undeff=999.0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(theta)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, scal ),  &
  &                                     "tangent_mean_scal_Cart" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, scal_mean ),  &
  &                                     "tangent_mean_scal_Cart" )
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:3)
     else
        undefgcflag="inc"
     end if
  end if

  if(present(axis))then
     ax(1:2)=axis(1:2)
  else
     ax='xy'
  end if

  if(present(undefg))then
     call tangent_mean_scal_f( x, y, xc, yc, scal, r, theta, tmp,  &
  &                    undef=undeff, undefg=undefg, undefgc=undefgcflag(1:3),  &
  &                    stdopt=stderr, axis=ax )
  else
     call tangent_mean_scal_f( x, y, xc, yc, scal, r, theta, tmp, undef=undeff,  &
  &                            stdopt=stderr, axis=ax )
  end if

!-- 接線平均値を内挿して, デカルト系での平均値を求める.
  if(present(undefg))then
     if(undefgcflag(1:3)=='err')then
        do k=1,ny
           do j=1,nx
              if(ax(1:2)=='xy')then  ! Cart.
                 tmpr=sqrt((x(j)-xc)**2+(y(k)-yc)**2)
              else  ! lon-lat
                 tmpr=real(ll2radi( dble(xc), dble(yc), dble(x(j)), dble(y(k))))
              end if
              call interpo_search_1d( r, tmpr, itmpr, stdopt=stderr )
              if(nr>itmpr)then
                 if(tmp(itmpr)/=undefg.and.tmp(itmpr+1)/=undefg)then
                    call interpolation_1d( (/r(itmpr), r(itmpr+1)/),  &
  &                       (/tmp(itmpr), tmp(itmpr+1)/), tmpr, tmp_anom )
                    scal_mean(j,k)=tmp_anom
                 else
                    scal_mean(j,k)=undefg
                 end if
              else
                 scal_mean(j,k)=undeff
              end if
           end do
        end do
     else
        do k=1,ny
           do j=1,nx
              if(ax(1:2)=='xy')then  ! Cart.
                 tmpr=sqrt((x(j)-xc)**2+(y(k)-yc)**2)
              else  ! lon-lat
                 tmpr=real(ll2radi( dble(xc), dble(yc), dble(x(j)), dble(y(k))))
              end if
              call interpo_search_1d( r, tmpr, itmpr, stdopt=stderr )
              if(nr>itmpr)then
                 if(tmp(itmpr)/=undefg.and.tmp(itmpr+1)/=undefg.and.  &
  &                 scal(j,k)/=undefg)then
                    call interpolation_1d( (/r(itmpr), r(itmpr+1)/),  &
  &                       (/tmp(itmpr), tmp(itmpr+1)/), tmpr, tmp_anom )
                    scal_mean(j,k)=tmp_anom
                 else
                    scal_mean(j,k)=undefg
                 end if
              else
                 scal_mean(j,k)=undeff
              end if
           end do
        end do
     end if
  else
     do k=1,ny
        do j=1,nx
           if(ax(1:2)=='xy')then  ! Cart.
              tmpr=sqrt((x(j)-xc)**2+(y(k)-yc)**2)
           else  ! lon-lat
              tmpr=real(ll2radi( dble(xc), dble(yc), dble(x(j)), dble(y(k)) ))
           end if
           call interpo_search_1d( r, tmpr, itmpr, stdopt=stderr )
           if(nr>itmpr)then
              call interpolation_1d( (/r(itmpr), r(itmpr+1)/),  &
  &                               (/tmp(itmpr), tmp(itmpr+1)/), tmpr, tmp_anom )
              scal_mean(j,k)=tmp_anom
           else
              scal_mean(j,k)=undeff
           end if
        end do
     end do
  end if

end subroutine tangent_mean_scal_Cart_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_mean_scal_Cart_d( x, y, xc, yc, scal, r, theta,  &
  &                                  scal_mean, undef, undefg, undefgc,  &
  &                                  stdopt, axis )
  ! 台風中心から接線平均を計算し, デカルト座標系に戻す.
  ! 接線平均ルーチンを用いて物理量を平均し, その 1 次元データを確保しておく.
  ! 同時に, デカルト系での物理量の円筒座標系での radial 位置を求める.
  ! この radial の位置における接線平均値を先の 1 次元データから内挿で求める.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  implicit none
  double precision, intent(in) :: x(:)  ! デカルト座標系での x 座標 [m] or lon [rad]
  double precision, intent(in) :: y(:)  ! デカルト座標系での y 座標 [m] or lat [rad]
  double precision, intent(in) :: scal(size(x),size(y))  ! デカルト座標系での平均化する値.
  double precision, intent(in) :: xc  ! 接線平均する際の中心 x 成分 [m] or [rad].
  double precision, intent(in) :: yc  ! 接線平均する際の中心 y 成分 [m] or [rad].
  double precision, intent(in) :: r(:)  ! 平均化したときの動径方向の座標(xc からの値を入れる).
  double precision, intent(in) :: theta(:)  ! 平均化するときの接線方向の座標 [rad].
  double precision, intent(inout) :: scal_mean(size(x),size(y))  ! デカルト系での平均値.
  double precision, optional :: undef  ! 内挿値が見つからないときの未定義値.
                           ! デフォルトでは dcl の未定義値
  double precision, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  character(2), intent(in), optional :: axis  ! x, y の座標系
                                           ! 'xy' = デカルト座標系 [m]
                                           ! 'll' = 球面緯度経度座標系 [rad]
                                           ! デフォルトは 'xy'.
  integer :: j, k, nx, ny, nr, nt, itmpr
  double precision :: tmp(size(r))
  double precision :: tmpr, tmp_anom, undeff
  character(2) :: ax
  character(3) :: undefgcflag
  logical :: stderr

  if(present(undef))then
     undeff=undef
  else
     undeff=999.0d0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(theta)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, scal ),  &
  &                                     "tangent_mean_scal_Cart" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, scal_mean ),  &
  &                                     "tangent_mean_scal_Cart" )
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:3)
     else
        undefgcflag="inc"
     end if
  end if

  if(present(axis))then
     ax(1:2)=axis(1:2)
  else
     ax='xy'
  end if

  if(present(undefg))then
     call tangent_mean_scal_d( x, y, xc, yc, scal, r, theta, tmp,  &
  &                    undef=undeff, undefg=undefg, undefgc=undefgcflag(1:3),  &
  &                    stdopt=stderr, axis=ax )
  else
     call tangent_mean_scal_d( x, y, xc, yc, scal, r, theta, tmp, undef=undeff,  &
  &                            stdopt=stderr, axis=ax )
  end if

!-- 接線平均値を内挿して, デカルト系での平均値を求める.
  if(present(undefg))then
     if(undefgcflag(1:3)=='err')then
        do k=1,ny
           do j=1,nx
              if(ax(1:2)=='xy')then  ! Cart.
                 tmpr=dsqrt((x(j)-xc)**2+(y(k)-yc)**2)
              else  ! lon-lat
                 tmpr=ll2radi( xc, yc, x(j), y(k) )
              end if
              call interpo_search_1d( r, tmpr, itmpr, stdopt=stderr )
              if(nr>itmpr)then
                 if(tmp(itmpr)/=undefg.and.tmp(itmpr+1)/=undefg)then
                    call interpolation_1d( (/r(itmpr), r(itmpr+1)/),  &
  &                       (/tmp(itmpr), tmp(itmpr+1)/), tmpr, tmp_anom )
                    scal_mean(j,k)=tmp_anom
                 else
                    scal_mean(j,k)=undefg
                 end if
              else
                 scal_mean(j,k)=undeff
              end if
           end do
        end do
     else
        do k=1,ny
           do j=1,nx
              if(ax(1:2)=='xy')then  ! Cart.
                 tmpr=dsqrt((x(j)-xc)**2+(y(k)-yc)**2)
              else  ! lon-lat
                 tmpr=ll2radi( xc, yc, x(j), y(k) )
              end if
              call interpo_search_1d( r, tmpr, itmpr, stdopt=stderr )
              if(nr>itmpr)then
                 if(tmp(itmpr)/=undefg.and.tmp(itmpr+1)/=undefg.and.  &
  &                 scal(j,k)/=undefg)then
                    call interpolation_1d( (/r(itmpr), r(itmpr+1)/),  &
  &                       (/tmp(itmpr), tmp(itmpr+1)/), tmpr, tmp_anom )
                    scal_mean(j,k)=tmp_anom
                 else
                    scal_mean(j,k)=undefg
                 end if
              else
                 scal_mean(j,k)=undeff
              end if
           end do
        end do
     end if
  else
     do k=1,ny
        do j=1,nx
           if(ax(1:2)=='xy')then  ! Cart.
              tmpr=dsqrt((x(j)-xc)**2+(y(k)-yc)**2)
           else  ! lon-lat
              tmpr=ll2radi( xc, yc, x(j), y(k) )
           end if
           call interpo_search_1d( r, tmpr, itmpr, stdopt=stderr )
           if(nr>itmpr)then
              call interpolation_1d( (/r(itmpr), r(itmpr+1)/),  &
  &                               (/tmp(itmpr), tmp(itmpr+1)/), tmpr, tmp_anom )
              scal_mean(j,k)=tmp_anom
           else
              scal_mean(j,k)=undeff
           end if
        end do
     end do
  end if

end subroutine tangent_mean_scal_Cart_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_mean_anom_scal_Cart_f( x, y, xc, yc, scal, r, theta,  &
  &                                       scal_anom, undef, undefg, undefgc,  &
  &                                       stdopt, axis )
  ! 台風中心から接線アノマリを計算し, デカルト座標系に戻す.
  ! 接線平均ルーチンを用いて物理量を平均し, その 1 次元データを確保しておく.
  ! 同時に, デカルト系での物理量の円筒座標系での radial 位置を求める.
  ! この radial の位置における接線平均値を先の 1 次元データから内挿で求める.
  ! この求めた内挿値を元のデカルトデータから引くことでアノマリとする.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  implicit none
  real, intent(in) :: x(:)  ! デカルト座標系での x 座標 [m] or lon [rad]
  real, intent(in) :: y(:)  ! デカルト座標系での y 座標 [m] or lat [rad]
  real, intent(in) :: scal(size(x),size(y))  ! デカルト座標系での平均化する値.
  real, intent(in) :: xc  ! 接線平均する際の中心 x 成分 [m] or [rad].
  real, intent(in) :: yc  ! 接線平均する際の中心 y 成分 [m] or [rad].
  real, intent(in) :: r(:)  ! 平均化したときの動径方向の座標(xc からの値を入れる).
  real, intent(in) :: theta(:)  ! 平均化するときの接線方向の座標 [rad].
  real, intent(inout) :: scal_anom(size(x),size(y))  ! デカルト系でのアノマリ.
  real, optional :: undef  ! 内挿値が見つからないときの未定義値.
                           ! デフォルトでは dcl の未定義値
  real, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  character(2), intent(in), optional :: axis  ! x, y の座標系
                                           ! 'xy' = デカルト座標系 [m]
                                           ! 'll' = 球面緯度経度座標系 [rad]
                                           ! デフォルトは 'xy'.
  integer :: j, k, nx, ny, nr, nt, itmpr
  real :: tmp(size(r))
  real :: tmpr, tmp_anom, undeff
  character(2) :: ax
  character(3) :: undefgcflag
  logical :: stderr

  if(present(undef))then
     undeff=undef
  else
     undeff=999.0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(theta)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, scal ),  &
  &                                     "tangent_mean_scal_anom_Cart" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, scal_anom ),  &
  &                                     "tangent_mean_scal_anom_Cart" )
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:3)
     else
        undefgcflag="inc"
     end if
  end if

  if(present(axis))then
     ax(1:2)=axis(1:2)
  else
     ax='xy'
  end if

  if(present(undefg))then
     call tangent_mean_scal_f( x, y, xc, yc, scal, r, theta, tmp,  &
  &                    undef=undeff, undefg=undefg, undefgc=undefgcflag(1:3),  &
  &                    stdopt=stderr, axis=ax )
  else
     call tangent_mean_scal_f( x, y, xc, yc, scal, r, theta, tmp, undef=undeff,  &
  &                            stdopt=stderr, axis=ax )
  end if

!-- 接線平均値を内挿し, その内挿値を引いてアノマリを求める.
  if(present(undefg))then
     if(undefgcflag(1:3)=='err')then
        do k=1,ny
           do j=1,nx
              if(ax(1:2)=='xy')then  ! Cart.
                 tmpr=sqrt((x(j)-xc)**2+(y(k)-yc)**2)
              else  ! lon-lat
                 tmpr=ll2radi( dble(xc), dble(yc), dble(x(j)), dble(y(k)) )
              end if
              call interpo_search_1d( r, tmpr, itmpr, stdopt=stderr )
              if(nr>itmpr)then
                 if(tmp(itmpr)/=undefg.and.tmp(itmpr+1)/=undefg)then
                    call interpolation_1d( (/r(itmpr), r(itmpr+1)/),  &
  &                       (/tmp(itmpr), tmp(itmpr+1)/), tmpr, tmp_anom )
                    scal_anom(j,k)=scal(j,k)-tmp_anom
                 else
                    scal_anom(j,k)=undefg
                 end if
              else
                 scal_anom(j,k)=undeff
              end if
           end do
        end do
     else
        do k=1,ny
           do j=1,nx
              if(ax(1:2)=='xy')then  ! Cart.
                 tmpr=sqrt((x(j)-xc)**2+(y(k)-yc)**2)
              else  ! lon-lat
                 tmpr=ll2radi( dble(xc), dble(yc), dble(x(j)), dble(y(k)) )
              end if
              call interpo_search_1d( r, tmpr, itmpr, stdopt=stderr )
              if(nr>itmpr)then
                 if(tmp(itmpr)/=undefg.and.tmp(itmpr+1)/=undefg.and.  &
  &                 scal(j,k)/=undefg)then
                    call interpolation_1d( (/r(itmpr), r(itmpr+1)/),  &
  &                       (/tmp(itmpr), tmp(itmpr+1)/), tmpr, tmp_anom )
                    scal_anom(j,k)=scal(j,k)-tmp_anom
                 else
                    scal_anom(j,k)=undefg
                 end if
              else
                 scal_anom(j,k)=undeff
              end if
           end do
        end do
     end if
  else
     do k=1,ny
        do j=1,nx
           if(ax(1:2)=='xy')then  ! Cart.
              tmpr=sqrt((x(j)-xc)**2+(y(k)-yc)**2)
           else  ! lon-lat
              tmpr=ll2radi( dble(xc), dble(yc), dble(x(j)), dble(y(k)) )
           end if
           call interpo_search_1d( r, tmpr, itmpr, stdopt=stderr )
           if(nr>itmpr)then
              call interpolation_1d( (/r(itmpr), r(itmpr+1)/),  &
  &                               (/tmp(itmpr), tmp(itmpr+1)/), tmpr, tmp_anom )
              scal_anom(j,k)=scal(j,k)-tmp_anom
           else
              scal_anom(j,k)=undeff
           end if
        end do
     end do
  end if

end subroutine tangent_mean_anom_scal_Cart_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_mean_anom_scal_Cart_d( x, y, xc, yc, scal, r, theta,  &
  &                                       scal_anom, undef, undefg, undefgc,  &
  &                                       stdopt, axis )
  ! 台風中心から接線アノマリを計算し, デカルト座標系に戻す.
  ! 接線平均ルーチンを用いて物理量を平均し, その 1 次元データを確保しておく.
  ! 同時に, デカルト系での物理量の円筒座標系での radial 位置を求める.
  ! この radial の位置における接線平均値を先の 1 次元データから内挿で求める.
  ! この求めた内挿値を元のデカルトデータから引くことでアノマリとする.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  implicit none
  double precision, intent(in) :: x(:)  ! デカルト座標系での x 座標 [m] or lon [rad]
  double precision, intent(in) :: y(:)  ! デカルト座標系での y 座標 [m] or lat [rad]
  double precision, intent(in) :: scal(size(x),size(y))  ! デカルト座標系での平均化する値.
  double precision, intent(in) :: xc  ! 接線平均する際の中心 x 成分 [m] or [rad].
  double precision, intent(in) :: yc  ! 接線平均する際の中心 y 成分 [m] or [rad].
  double precision, intent(in) :: r(:)  ! 平均化したときの動径方向の座標(xc からの値を入れる).
  double precision, intent(in) :: theta(:)  ! 平均化するときの接線方向の座標 [rad].
  double precision, intent(inout) :: scal_anom(size(x),size(y))  ! デカルト系でのアノマリ.
  double precision, optional :: undef  ! 内挿値が見つからないときの未定義値.
                           ! デフォルトでは dcl の未定義値
  double precision, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  character(2), intent(in), optional :: axis  ! x, y の座標系
                                           ! 'xy' = デカルト座標系 [m]
                                           ! 'll' = 球面緯度経度座標系 [rad]
                                           ! デフォルトは 'xy'.
  integer :: j, k, nx, ny, nr, nt, itmpr
  double precision :: tmp(size(r))
  double precision :: tmpr, tmp_anom, undeff
  character(2) :: ax
  character(3) :: undefgcflag
  logical :: stderr

  if(present(undef))then
     undeff=undef
  else
     undeff=999.0d0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(theta)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, scal ),  &
  &                                     "tangent_mean_scal_anom_Cart" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, scal_anom ),  &
  &                                     "tangent_mean_scal_anom_Cart" )
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:3)
     else
        undefgcflag="inc"
     end if
  end if

  if(present(axis))then
     ax(1:2)=axis(1:2)
  else
     ax='xy'
  end if

  if(present(undefg))then
     call tangent_mean_scal_d( x, y, xc, yc, scal, r, theta, tmp,  &
  &                    undef=undeff, undefg=undefg, undefgc=undefgcflag(1:3),  &
  &                    stdopt=stderr, axis=ax )
  else
     call tangent_mean_scal_d( x, y, xc, yc, scal, r, theta, tmp, undef=undeff,  &
  &                            stdopt=stderr, axis=ax )
  end if

!-- 接線平均値を内挿し, その内挿値を引いてアノマリを求める.
  if(present(undefg))then
     if(undefgcflag(1:3)=='err')then
        do k=1,ny
           do j=1,nx
              if(ax(1:2)=='xy')then  ! Cart.
                 tmpr=dsqrt((x(j)-xc)**2+(y(k)-yc)**2)
              else  ! lon-lat
                 tmpr=ll2radi( xc, yc, x(j), y(k) )
              end if
              call interpo_search_1d( r, tmpr, itmpr, stdopt=stderr )
              if(nr>itmpr)then
                 if(tmp(itmpr)/=undefg.and.tmp(itmpr+1)/=undefg)then
                    call interpolation_1d( (/r(itmpr), r(itmpr+1)/),  &
  &                       (/tmp(itmpr), tmp(itmpr+1)/), tmpr, tmp_anom )
                    scal_anom(j,k)=scal(j,k)-tmp_anom
                 else
                    scal_anom(j,k)=undefg
                 end if
              else
                 scal_anom(j,k)=undeff
              end if
           end do
        end do
     else
        do k=1,ny
           do j=1,nx
              if(ax(1:2)=='xy')then  ! Cart.
                 tmpr=dsqrt((x(j)-xc)**2+(y(k)-yc)**2)
              else  ! lon-lat
                 tmpr=ll2radi( xc, yc, x(j), y(k) )
              end if
              call interpo_search_1d( r, tmpr, itmpr, stdopt=stderr )
              if(nr>itmpr)then
                 if(tmp(itmpr)/=undefg.and.tmp(itmpr+1)/=undefg.and.  &
  &                 scal(j,k)/=undefg)then
                    call interpolation_1d( (/r(itmpr), r(itmpr+1)/),  &
  &                       (/tmp(itmpr), tmp(itmpr+1)/), tmpr, tmp_anom )
                    scal_anom(j,k)=scal(j,k)-tmp_anom
                 else
                    scal_anom(j,k)=undefg
                 end if
              else
                 scal_anom(j,k)=undeff
              end if
           end do
        end do
     end if
  else
     do k=1,ny
        do j=1,nx
           if(ax(1:2)=='xy')then  ! Cart.
              tmpr=dsqrt((x(j)-xc)**2+(y(k)-yc)**2)
           else  ! lon-lat
              tmpr=ll2radi( xc, yc, x(j), y(k) )
           end if
           call interpo_search_1d( r, tmpr, itmpr, stdopt=stderr )
           if(nr>itmpr)then
              call interpolation_1d( (/r(itmpr), r(itmpr+1)/),  &
  &                               (/tmp(itmpr), tmp(itmpr+1)/), tmpr, tmp_anom )
              scal_anom(j,k)=scal(j,k)-tmp_anom
           else
              scal_anom(j,k)=undeff
           end if
        end do
     end do
  end if

end subroutine tangent_mean_anom_scal_Cart_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_median_scal_f( x, y, xc, yc, u, r, theta, v, undef,  &
  &                               undefg, undefgc, stdopt, axis )
  ! 任意の物理量を台風の中心から接線方向へ座標変換し, 中央値を求めるルーチン
  ! このルーチンは接線風速の中央値を求める時には用いることはできない.
  ! 引数の制限として, |x(1)-xc|, |xc-x(nx)|, |y(1)-yc|, |yc-y(ny)| > r. 
  ! これ以外の場合, r で与えられる距離が領域外の範囲については
  ! undef で設定された値が代入される. undef がないときはゼロが代入される.
  ! 中央値を求める手順は以下のとおり.
  ! (1) nr, nt のすべての点についてそれに対応する x, y 座標値を rt_2_xy で計算.
  ! (2) その地点を含む x,y グリッドの微小領域を interpo_search_2d で検索.
  ! (3) その地点を含む 4 点が出たら, その地点でのスカラー値を 4 隅のスカラー値
  !     から, 重線形内挿 interpolation_2d で計算.
  ! (4) nr x nt 個の内挿スカラー値が求まったら, nt 方向にmedian_1d 使用.
  ! 以上で各 nr について中央値が得られる.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  implicit none
  real, intent(in) :: x(:)  ! 右手座標系での第一成分 [m or rad]
  real, intent(in) :: y(:)  ! 右手座標系での第二成分 [m or rad]
  real, intent(in) :: u(size(x),size(y))  ! 右手座標系での中央値を求める値
  real, intent(in) :: xc  ! 接線平均する際の中心 x 成分. [m or rad]
  real, intent(in) :: yc  ! 接線平均する際の中心 y 成分. [m or rad]
  real, intent(in) :: r(:)  ! (xc, yc) を中心とした極座標系動径座標 [m].
  real, intent(in) :: theta(:)  ! (xc, yc) を中心とした極座標系同位角座標 [rad].
  real, intent(inout) :: v(size(r))  ! u の中央値.
  real, intent(in), optional :: undef   ! 領域外の設定値
  real, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
                ! デフォルトは "inc".
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  character(2), intent(in), optional :: axis  ! x, y の座標系
                                           ! 'xy' = デカルト座標系 [m]
                                           ! 'll' = 球面緯度経度座標系 [rad]
                                           ! デフォルトは 'xy'.
  integer :: i, nx, ny, nr, nt
  real :: r_undef, r_undefg
  real :: work(size(r),size(theta))
  character(3) :: undefgcflag
  character(2) :: ax_flag
  logical, dimension(size(r)) :: undefgc_check
  logical :: stderr

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(theta)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "tangent_mean_scal" )
     call check_array_size_dmp_message( check_array_size_1d( nr, v ),  &
  &                                     "tangent_mean_scal" )
  end if

  undefgc_check(:)=.true.

  if(present(undef))then
    r_undef=undef
  else
    r_undef=-999.0
  end if

  if(present(undefg))then
    r_undefg=undefg
  else
    r_undefg=-999.0
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:3)
     else
        undefgcflag="inc"
     end if
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(axis))then
     ax_flag=axis(1:2)
  else
     ax_flag='xy'
  end if

!-- 過程(1) - (3)

  call tangent_conv_scal_f( x, y, xc, yc, u, r, theta, work, undef=r_undef,  &
  &                         undefg=r_undefg, undefgc=undefgcflag(1:3),  &
  &                         stdopt=stderr, axis=ax_flag(1:2) )

!-- 過程(4) ---
  if(present(undefg))then
     if(undefgcflag(1:1)=='i')then
        do i=1,nr
           call Median_1d( work(i,:), v(i), error=undefg )
        end do
     else
        do i=1,nr
           if(undefgc_check(i).eqv..true.)then
              call Median_1d( work(i,:), v(i) )
           else
              v(i)=undefg
           end if
        end do
     end if
  else
     do i=1,nr
        call Median_1d( work(i,:), v(i) )
     end do
  end if

end subroutine tangent_median_scal_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_median_scal_d( x, y, xc, yc, u, r, theta, v, undef,  &
  &                               undefg, undefgc, stdopt, axis )
  ! 任意の物理量を台風の中心から接線方向へ座標変換し, 中央値を求めるルーチン
  ! このルーチンは接線風速の中央値を求める時には用いることはできない.
  ! 引数の制限として, |x(1)-xc|, |xc-x(nx)|, |y(1)-yc|, |yc-y(ny)| > r. 
  ! これ以外の場合, r で与えられる距離が領域外の範囲については
  ! undef で設定された値が代入される. undef がないときはゼロが代入される.
  ! 中央値を求める手順は以下のとおり.
  ! (1) nr, nt のすべての点についてそれに対応する x, y 座標値を rt_2_xy で計算.
  ! (2) その地点を含む x,y グリッドの微小領域を interpo_search_2d で検索.
  ! (3) その地点を含む 4 点が出たら, その地点でのスカラー値を 4 隅のスカラー値
  !     から, 重線形内挿 interpolation_2d で計算.
  ! (4) nr x nt 個の内挿スカラー値が求まったら, nt 方向にmedian_1d 使用.
  ! 以上で各 nr について中央値が得られる.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  implicit none
  double precision, intent(in) :: x(:)  ! 右手座標系での第一成分 [m or rad]
  double precision, intent(in) :: y(:)  ! 右手座標系での第二成分 [m or rad]
  double precision, intent(in) :: u(size(x),size(y))  ! 右手座標系での中央値を求める値
  double precision, intent(in) :: xc  ! 接線平均する際の中心 x 成分. [m or rad]
  double precision, intent(in) :: yc  ! 接線平均する際の中心 y 成分. [m or rad]
  double precision, intent(in) :: r(:)  ! (xc, yc) を中心とした極座標系動径座標 [m].
  double precision, intent(in) :: theta(:)  ! (xc, yc) を中心とした極座標系同位角座標 [rad].
  double precision, intent(inout) :: v(size(r))  ! u の中央値.
  double precision, intent(in), optional :: undef   ! 領域外の設定値
  double precision, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
                ! デフォルトは "inc".
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  character(2), intent(in), optional :: axis  ! x, y の座標系
                                           ! 'xy' = デカルト座標系 [m]
                                           ! 'll' = 球面緯度経度座標系 [rad]
                                           ! デフォルトは 'xy'.
  integer :: i, nx, ny, nr, nt
  double precision :: r_undef, r_undefg
  double precision :: work(size(r),size(theta))
  character(3) :: undefgcflag
  character(2) :: ax_flag
  logical, dimension(size(r)) :: undefgc_check
  logical :: stderr

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(theta)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "tangent_mean_scal" )
     call check_array_size_dmp_message( check_array_size_1d( nr, v ),  &
  &                                     "tangent_mean_scal" )
  end if

  undefgc_check(:)=.true.

  if(present(undef))then
    r_undef=undef
  else
    r_undef=-999.0d0
  end if

  if(present(undefg))then
    r_undefg=undefg
  else
    r_undefg=-999.0d0
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:3)
     else
        undefgcflag="inc"
     end if
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(axis))then
     ax_flag=axis(1:2)
  else
     ax_flag='xy'
  end if

!-- 過程(1) - (3)

  call tangent_conv_scal_d( x, y, xc, yc, u, r, theta, work, undef=r_undef,  &
  &                         undefg=r_undefg, undefgc=undefgcflag(1:3),  &
  &                         stdopt=stderr, axis=ax_flag(1:2) )

!-- 過程(4) ---
  if(present(undefg))then
     if(undefgcflag(1:1)=='i')then
        do i=1,nr
           call Median_1d( work(i,:), v(i), error=undefg )
        end do
     else
        do i=1,nr
           if(undefgc_check(i).eqv..true.)then
              call Median_1d( work(i,:), v(i) )
           else
              v(i)=undefg
           end if
        end do
     end if
  else
     do i=1,nr
        call Median_1d( work(i,:), v(i) )
     end do
  end if

end subroutine tangent_median_scal_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine Cart_conv_scal_f( r, theta, v, x, y, xc, yc, u, undef,  &
  &                          undefg, undefgc, stdopt, axis )
  ! 任意の物理量を台風を中心とした平面極座標からデカルト座標へ変換するルーチン
  ! 引数の制限として, |x(1)-xc|, |xc-x(nx)|, |y(1)-yc|, |yc-y(ny)| > r. 
  ! これ以外の場合, r で与えられる距離が領域外の範囲については
  ! undef で設定された値が代入される. undef がないときはゼロが代入される.
  ! 平均化の手順は以下のとおり.
  ! (1) nx, ny のすべての点についてそれに対応する r, t 座標値を xy_2_rt で計算.
  ! (2) その地点を含む r,t グリッドの微小領域を interpo_search_2d で検索.
  ! (3) その地点を含む 4 点が出たら, その地点でのスカラー値を 4 隅のスカラー値
  !     から, 重線形内挿 interpolation_2d で計算.
  ! 本ルーチンは平面極座標グリッド値での偏差計算を行う.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  use Math_Const
  implicit none
  real, intent(in) :: r(:)  ! (xc, yc) を中心とした極座標系動径座標 [m].
  real, intent(in) :: theta(:)  ! (xc, yc) を中心とした極座標系同位角座標 [rad].
  real, intent(in) :: v(size(r),size(theta))  ! 平面極座標上に定義された変数.
  real, intent(in) :: x(:)  ! 右手座標系での第一成分 [m or rad]
  real, intent(in) :: y(:)  ! 右手座標系での第二成分 [m or rad]
  real, intent(in) :: xc  ! 接線平均する際の中心 x 成分. [m or rad]
  real, intent(in) :: yc  ! 接線平均する際の中心 y 成分. [m or rad]
  real, intent(inout) :: u(size(x),size(y))  ! デカルト座標上に定義される変数.
  real, intent(in), optional :: undef  ! 領域外の設定値
  real, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
                ! デフォルトは "inc".
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  character(2), intent(in), optional :: axis  ! x, y の座標系
                                           ! 'xy' = デカルト座標系 [m]
                                           ! 'll' = 球面緯度経度座標系 [rad]
                                           ! デフォルトは 'xy'.
  integer :: i, j, nx, ny, nr, nt, i_undef
  real :: r_undef, r_undefg
  real :: work(size(x),size(y))
  real :: point(size(x),size(y),2)
  integer :: ip(size(x),size(y),2)
  real :: tmpx(2), tmpy(2), tmpz(2,2), inter(2)
  double precision :: tmppoint1, tmppoint2
  character(1) :: undefgcflag
  character(2) :: ax_flag
!  logical, dimension(size(r)) :: undefgc_check
  logical :: ucf, stderr

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(theta)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "tangent_conv_scal" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nt, v ),  &
  &                                     "tangent_conv_scal" )
  end if

!  undefgc_check(:)=.true.
  i_undef=0

  if(present(undef))then
    r_undef=undef
  else
    r_undef=-999.0
  end if

  if(present(undefg))then
    r_undefg=undefg
  else
    r_undefg=-999.0
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:1)
     else
        undefgcflag="i"
     end if
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(axis))then
     ax_flag=axis(1:2)
  else
     ax_flag='xy'
  end if

!-- 先の引数条件をクリアしているか確認 ---
!-- この操作は行わない.

!  call search_region_1d( (/x(1),x(nx)/), (/y(1),y(ny)/), (/xc,yc/), r, nr,  &
!  &                      stdopt=stderr, axis=ax_flag(1:2) )

!-- 先に v に undef 値を入れておく.
!  do j=1,nt
!     do i=1,nrr
!        v(i,j)=r_undef
!     end do
!  end do

!-- 過程(1) ---
  if(ax_flag(1:2)=='xy')then
     do j=1,ny
        do i=1,nx
           call xy_2_rt( x(i), y(j), xc, yc, point(i,j,1), point(i,j,2) )
!           if(point(i,j,2)<0.0)then
!              point(i,j,2)=point(i,j,2)+2.0*pi
!           end if
           if(point(i,j,2)<theta(1))then  ! +2pi
              do while(point(i,j,2)<theta(1))
                 point(i,j,2)=point(i,j,2)+2.0*pi
              end do
           else if(point(i,j,2)>theta(nt))then  ! -2pi
              do while(point(i,j,2)>theta(nt))
                 point(i,j,2)=point(i,j,2)-2.0*pi
              end do
           end if
        end do
     end do
  else if(ax_flag(1:2)=='ll')then
     do j=1,ny
        do i=1,nx
           call ll2rt( dble(xc), dble(yc), dble(x(i)), dble(y(j)),  &
  &                    tmppoint1, tmppoint2 )
           point(i,j,1)=real(tmppoint1)
           point(i,j,2)=real(tmppoint2)
!           if(point(i,j,2)<0.0)then
!              point(i,j,2)=point(i,j,2)+2.0*pi
!           end if
           if(point(i,j,2)<theta(1))then  ! +2pi
              do while(point(i,j,2)<theta(1))
                 point(i,j,2)=point(i,j,2)+2.0*pi
              end do
           else if(point(i,j,2)>theta(nt))then  ! -2pi
              do while(point(i,j,2)>theta(nt))
                 point(i,j,2)=point(i,j,2)-2.0*pi
              end do
           end if
        end do
     end do
  end if

!-- 過程(2) ---
  do j=1,ny
     do i=1,nx
        call interpo_search_2d( r, theta, point(i,j,1), point(i,j,2),  &
      &                         ip(i,j,1), ip(i,j,2), undeff=i_undef,  &
  &                             stdopt=stderr )
     end do
  end do

!-- 過程(3) ---
  do j=1,ny
     do i=1,nx
        if(ip(i,j,1)/=i_undef.and.ip(i,j,2)/=i_undef.and.  &
  &        ip(i,j,1)/=nr.and.ip(i,j,2)/=nt)then
           tmpx(1)=r(ip(i,j,1))
           tmpx(2)=r(ip(i,j,1)+1)
           tmpy(1)=theta(ip(i,j,2))
           tmpy(2)=theta(ip(i,j,2)+1)
           tmpz(1,1)=v(ip(i,j,1),ip(i,j,2))
           tmpz(2,1)=v(ip(i,j,1)+1,ip(i,j,2))
           tmpz(1,2)=v(ip(i,j,1),ip(i,j,2)+1)
           tmpz(2,2)=v(ip(i,j,1)+1,ip(i,j,2)+1)
           inter(1)=point(i,j,1)
           inter(2)=point(i,j,2)

           if(present(undefg))then
              ucf=undef_checker_2df( tmpz, undefg )
              if(ucf.eqv..false.)then
                 call interpolation_2d( tmpx, tmpy, tmpz, inter, work(i,j) )
              else
                 work(i,j)=r_undefg
!                 undefgc_check(i)=.false.
              end if
           else
              call interpolation_2d( tmpx, tmpy, tmpz, inter, work(i,j) )
           end if
        else
           work(i,j)=r_undef
        end if
     end do
  end do

  do j=1,ny
     do i=1,nx
        u(i,j)=work(i,j)
     end do
  end do

end subroutine Cart_conv_scal_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine Cart_conv_scal_d( r, theta, v, x, y, xc, yc, u, undef,  &
  &                          undefg, undefgc, stdopt, axis )
  ! 任意の物理量を台風を中心とした平面極座標からデカルト座標へ変換するルーチン
  ! 引数の制限として, |x(1)-xc|, |xc-x(nx)|, |y(1)-yc|, |yc-y(ny)| > r. 
  ! これ以外の場合, r で与えられる距離が領域外の範囲については
  ! undef で設定された値が代入される. undef がないときはゼロが代入される.
  ! 平均化の手順は以下のとおり.
  ! (1) nx, ny のすべての点についてそれに対応する r, t 座標値を xy_2_rt で計算.
  ! (2) その地点を含む r,t グリッドの微小領域を interpo_search_2d で検索.
  ! (3) その地点を含む 4 点が出たら, その地点でのスカラー値を 4 隅のスカラー値
  !     から, 重線形内挿 interpolation_2d で計算.
  ! 本ルーチンは平面極座標グリッド値での偏差計算を行う.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  use Math_Const
  implicit none
  double precision, intent(in) :: r(:)  ! (xc, yc) を中心とした極座標系動径座標 [m].
  double precision, intent(in) :: theta(:)  ! (xc, yc) を中心とした極座標系同位角座標 [rad].
  double precision, intent(in) :: v(size(r),size(theta))  ! 平面極座標上に定義された変数.
  double precision, intent(in) :: x(:)  ! 右手座標系での第一成分 [m or rad]
  double precision, intent(in) :: y(:)  ! 右手座標系での第二成分 [m or rad]
  double precision, intent(in) :: xc  ! 接線平均する際の中心 x 成分. [m or rad]
  double precision, intent(in) :: yc  ! 接線平均する際の中心 y 成分. [m or rad]
  double precision, intent(inout) :: u(size(x),size(y))  ! デカルト座標上に定義される変数.
  double precision, intent(in), optional :: undef  ! 領域外の設定値
  double precision, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
                ! デフォルトは "inc".
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  character(2), intent(in), optional :: axis  ! x, y の座標系
                                           ! 'xy' = デカルト座標系 [m]
                                           ! 'll' = 球面緯度経度座標系 [rad]
                                           ! デフォルトは 'xy'.
  integer :: i, j, nx, ny, nr, nt, i_undef
  double precision :: r_undef, r_undefg
  double precision :: work(size(x),size(y))
  double precision :: point(size(x),size(y),2)
  integer :: ip(size(x),size(y),2)
  double precision :: tmpx(2), tmpy(2), tmpz(2,2), inter(2)
  double precision :: tmppoint1, tmppoint2
  character(1) :: undefgcflag
  character(2) :: ax_flag
!  logical, dimension(size(r)) :: undefgc_check
  logical :: ucf, stderr

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(theta)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "tangent_conv_scal" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nt, v ),  &
  &                                     "tangent_conv_scal" )
  end if

!  undefgc_check(:)=.true.
  i_undef=0

  if(present(undef))then
    r_undef=undef
  else
    r_undef=-999.0d0
  end if

  if(present(undefg))then
    r_undefg=undefg
  else
    r_undefg=-999.0d0
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:1)
     else
        undefgcflag="i"
     end if
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(axis))then
     ax_flag=axis(1:2)
  else
     ax_flag='xy'
  end if

!-- 先の引数条件をクリアしているか確認 ---
!-- この操作は行わない.

!  call search_region_1d( (/x(1),x(nx)/), (/y(1),y(ny)/), (/xc,yc/), r, nr,  &
!  &                      stdopt=stderr, axis=ax_flag(1:2) )

!-- 先に v に undef 値を入れておく.
!  do j=1,nt
!     do i=1,nrr
!        v(i,j)=r_undef
!     end do
!  end do

!-- 過程(1) ---
  if(ax_flag(1:2)=='xy')then
     do j=1,ny
        do i=1,nx
           call xy_2_rt( x(i), y(j), xc, yc, point(i,j,1), point(i,j,2) )
!           if(point(i,j,2)<0.0d0)then
!              point(i,j,2)=point(i,j,2)+2.0d0*pi_dp
!           end if
           if(point(i,j,2)<theta(1))then  ! +2pi
              do while(point(i,j,2)<theta(1))
                 point(i,j,2)=point(i,j,2)+2.0d0*pi_dp
              end do
           else if(point(i,j,2)>theta(nt))then  ! -2pi
              do while(point(i,j,2)>theta(nt))
                 point(i,j,2)=point(i,j,2)-2.0d0*pi_dp
              end do
           end if
        end do
     end do
  else if(ax_flag(1:2)=='ll')then
     do j=1,ny
        do i=1,nx
           call ll2rt( xc, yc, x(i), y(j),  &
  &                    tmppoint1, tmppoint2 )
           point(i,j,1)=tmppoint1
           point(i,j,2)=tmppoint2
!           if(point(i,j,2)<0.0d0)then
!              point(i,j,2)=point(i,j,2)+2.0d0*pi_dp
!           end if
           if(point(i,j,2)<theta(1))then  ! +2pi
              do while(point(i,j,2)<theta(1))
                 point(i,j,2)=point(i,j,2)+2.0d0*pi_dp
              end do
           else if(point(i,j,2)>theta(nt))then  ! -2pi
              do while(point(i,j,2)>theta(nt))
                 point(i,j,2)=point(i,j,2)-2.0d0*pi_dp
              end do
           end if
        end do
     end do
  end if

!-- 過程(2) ---
  do j=1,ny
     do i=1,nx
        call interpo_search_2d( r, theta, point(i,j,1), point(i,j,2),  &
      &                         ip(i,j,1), ip(i,j,2), undeff=i_undef,  &
  &                             stdopt=stderr )
     end do
  end do

!-- 過程(3) ---
  do j=1,ny
     do i=1,nx
        if(ip(i,j,1)/=i_undef.and.ip(i,j,2)/=i_undef.and.  &
  &        ip(i,j,1)/=nr.and.ip(i,j,2)/=nt)then
           tmpx(1)=r(ip(i,j,1))
           tmpx(2)=r(ip(i,j,1)+1)
           tmpy(1)=theta(ip(i,j,2))
           tmpy(2)=theta(ip(i,j,2)+1)
           tmpz(1,1)=v(ip(i,j,1),ip(i,j,2))
           tmpz(2,1)=v(ip(i,j,1)+1,ip(i,j,2))
           tmpz(1,2)=v(ip(i,j,1),ip(i,j,2)+1)
           tmpz(2,2)=v(ip(i,j,1)+1,ip(i,j,2)+1)
           inter(1)=point(i,j,1)
           inter(2)=point(i,j,2)

           if(present(undefg))then
              ucf=undef_checker_2dd( tmpz, undefg )
              if(ucf.eqv..false.)then
                 call interpolation_2d( tmpx, tmpy, tmpz, inter, work(i,j) )
              else
                 work(i,j)=r_undefg
!                 undefgc_check(i)=.false.
              end if
           else
              call interpolation_2d( tmpx, tmpy, tmpz, inter, work(i,j) )
           end if
        else
           work(i,j)=r_undef
        end if
     end do
  end do

  do j=1,ny
     do i=1,nx
        u(i,j)=work(i,j)
     end do
  end do

end subroutine Cart_conv_scal_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine Cart_mean_scal_f( r, v, x, y, xc, yc, u, undef,  &
  &                          undefg, undefgc, stdopt, forcer0, axis )
  ! 任意の物理量を台風を中心とした平面極座標動径分布からデカルト座標へ
  ! 変換するルーチン.
  ! 引数の制限として, |x(1)-xc|, |xc-x(nx)|, |y(1)-yc|, |yc-y(ny)| > r. 
  ! これ以外の場合, r で与えられる距離が領域外の範囲については
  ! undef で設定された値が代入される. undef がないときはゼロが代入される.
  ! 平均化の手順は以下のとおり.
  ! (1) nx, ny のすべての点についてそれに対応する r 座標値を計算.
  ! (2) その地点を含む r グリッドの微小領域を interpo_search_2d で検索.
  ! (3) その地点を含む 2 点が出たら, その地点でのスカラー値を 2 隅のスカラー値
  !     から, 線形内挿 interpolation_1d で計算.
  ! 本ルーチンは平面極座標グリッド値での偏差計算を行う.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  use Math_Const
  implicit none
  real, intent(in) :: r(:)  ! (xc, yc) を中心とした極座標系動径座標 [m].
  real, intent(in) :: v(size(r))  ! 平均化した任意の物理量.
  real, intent(in) :: x(:)  ! 右手座標系での第一成分 [m or rad]
  real, intent(in) :: y(:)  ! 右手座標系での第二成分 [m or rad]
  real, intent(in) :: xc  ! 接線平均する際の中心 x 成分. [m or rad]
  real, intent(in) :: yc  ! 接線平均する際の中心 y 成分. [m or rad]
  real, intent(inout) :: u(size(x),size(y))  ! 右手座標系での平均化する値
  real, intent(in), optional :: undef  ! 領域外の設定値
  real, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
                ! デフォルトは "inc".
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  logical, intent(in), optional :: forcer0  ! 中心を隣接点と同じ値で埋める.
                                           ! .true. = 埋める, 
                                           ! .false. = 埋めない.
                                           ! デフォルト: 埋めない.
  character(2), intent(in), optional :: axis  ! x, y の座標系
                                           ! 'xy' = デカルト座標系 [m]
                                           ! 'll' = 球面緯度経度座標系 [rad]
                                           ! デフォルトは 'xy'.
  integer :: i, j, nx, ny, nr, i_undef
  real :: r_undef, r_undefg
  real :: work(size(x),size(y))
  real :: point(size(x),size(y))
  integer :: ip(size(x),size(y))
  character(1) :: undefgcflag
  character(2) :: ax_flag
!  logical, dimension(size(r)) :: undefgc_check
  logical :: ucf, stderr, forcer0_flag

  nx=size(x)
  ny=size(y)
  nr=size(r)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "tangent_conv_scal" )
     call check_array_size_dmp_message( check_array_size_1d( nr, v ),  &
  &                                     "tangent_conv_scal" )
  end if

!  undefgc_check(:)=.true.
  i_undef=0

  if(present(undef))then
    r_undef=undef
  else
    r_undef=-999.0
  end if

  if(present(undefg))then
    r_undefg=undefg
  else
    r_undefg=-999.0
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:1)
     else
        undefgcflag="i"
     end if
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(axis))then
     ax_flag=axis(1:2)
  else
     ax_flag='xy'
  end if

  if(present(forcer0))then
     forcer0_flag=forcer0
  else
     forcer0_flag=.false.
  end if

!-- 先の引数条件をクリアしているか確認 ---
!-- この操作は行わない.

!  call search_region_1d( (/x(1),x(nx)/), (/y(1),y(ny)/), (/xc,yc/), r, nr,  &
!  &                      stdopt=stderr, axis=ax_flag(1:2) )

!-- 先に v に undef 値を入れておく.
!  do j=1,nt
!     do i=1,nrr
!        v(i,j)=r_undef
!     end do
!  end do

!-- 過程(1) ---
  if(ax_flag(1:2)=='xy')then
     do j=1,ny
        do i=1,nx
           point(i,j)=sqrt((x(i)-xc)**2+(y(j)-yc)**2)
        end do
     end do
  else if(ax_flag(1:2)=='ll')then
     do j=1,ny
        do i=1,nx
           point(i,j)=ll2radi( dble(xc), dble(yc), dble(x(i)), dble(y(j)) )
        end do
     end do
  end if

!-- 過程(2) ---
  do j=1,ny
     do i=1,nx
        call interpo_search_1d( r, point(i,j), ip(i,j), undeff=i_undef,  &
  &                             stdopt=stderr )
     end do
  end do

!-- 過程(3) ---
  do j=1,ny
     do i=1,nx
        if(ip(i,j)/=i_undef.and.ip(i,j)/=nr)then
           if(present(undefg))then
              ucf=undef_checker_1df( v(ip(i,j):ip(i,j)+1), undefg )
              if(ucf.eqv..false.)then
                 call interpolation_1d( r(ip(i,j):ip(i,j)+1),  &
  &                                     v(ip(i,j):ip(i,j)+1),  &
  &                                     point(i,j), work(i,j) )
              else
                 work(i,j)=r_undefg
!                 undefgc_check(i)=.false.
              end if
           else
              call interpolation_1d( r(ip(i,j):ip(i,j)+1),  &
  &                                  v(ip(i,j):ip(i,j)+1),  &
  &                                  point(i,j), work(i,j) )
           end if
           if(point(i,j)<r(2))then
              if(forcer0_flag.eqv..true.)then
                 work(i,j)=v(2)  ! 中心は r(1), r(2) が隣接点.
              end if
           end if
        else
           work(i,j)=r_undef
           if(point(i,j)<r(1))then
              if(forcer0_flag.eqv..true.)then
                 work(i,j)=v(1)  ! i_undef で来ているので, 中心は r(1) より内.
              end if
           end if
        end if
     end do
  end do

  do j=1,ny
     do i=1,nx
        u(i,j)=work(i,j)
     end do
  end do

end subroutine Cart_mean_scal_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine Cart_mean_scal_d( r, v, x, y, xc, yc, u, undef,  &
  &                          undefg, undefgc, stdopt, forcer0, axis )
  ! 任意の物理量を台風を中心とした平面極座標動径分布からデカルト座標へ
  ! 変換するルーチン.
  ! 引数の制限として, |x(1)-xc|, |xc-x(nx)|, |y(1)-yc|, |yc-y(ny)| > r. 
  ! これ以外の場合, r で与えられる距離が領域外の範囲については
  ! undef で設定された値が代入される. undef がないときはゼロが代入される.
  ! 平均化の手順は以下のとおり.
  ! (1) nx, ny のすべての点についてそれに対応する r 座標値を計算.
  ! (2) その地点を含む r グリッドの微小領域を interpo_search_2d で検索.
  ! (3) その地点を含む 2 点が出たら, その地点でのスカラー値を 2 隅のスカラー値
  !     から, 線形内挿 interpolation_1d で計算.
  ! 本ルーチンは平面極座標グリッド値での偏差計算を行う.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  use Math_Const
  implicit none
  double precision, intent(in) :: r(:)  ! (xc, yc) を中心とした極座標系動径座標 [m].
  double precision, intent(in) :: v(size(r))  ! 平均化した任意の物理量.
  double precision, intent(in) :: x(:)  ! 右手座標系での第一成分 [m or rad]
  double precision, intent(in) :: y(:)  ! 右手座標系での第二成分 [m or rad]
  double precision, intent(in) :: xc  ! 接線平均する際の中心 x 成分. [m or rad]
  double precision, intent(in) :: yc  ! 接線平均する際の中心 y 成分. [m or rad]
  double precision, intent(inout) :: u(size(x),size(y))  ! 右手座標系での平均化する値
  double precision, intent(in), optional :: undef  ! 領域外の設定値
  double precision, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
                ! デフォルトは "inc".
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  logical, intent(in), optional :: forcer0  ! 中心を隣接点と同じ値で埋める.
                                           ! .true. = 埋める, 
                                           ! .false. = 埋めない.
                                           ! デフォルト: 埋めない.
  character(2), intent(in), optional :: axis  ! x, y の座標系
                                           ! 'xy' = デカルト座標系 [m]
                                           ! 'll' = 球面緯度経度座標系 [rad]
                                           ! デフォルトは 'xy'.
  integer :: i, j, nx, ny, nr, i_undef
  double precision :: r_undef, r_undefg
  double precision :: work(size(x),size(y))
  double precision :: point(size(x),size(y))
  integer :: ip(size(x),size(y))
  character(1) :: undefgcflag
  character(2) :: ax_flag
!  logical, dimension(size(r)) :: undefgc_check
  logical :: ucf, stderr, forcer0_flag

  nx=size(x)
  ny=size(y)
  nr=size(r)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "tangent_conv_scal" )
     call check_array_size_dmp_message( check_array_size_1d( nr, v ),  &
  &                                     "tangent_conv_scal" )
  end if

!  undefgc_check(:)=.true.
  i_undef=0

  if(present(undef))then
    r_undef=undef
  else
    r_undef=-999.0d0
  end if

  if(present(undefg))then
    r_undefg=undefg
  else
    r_undefg=-999.0d0
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:1)
     else
        undefgcflag="i"
     end if
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(axis))then
     ax_flag=axis(1:2)
  else
     ax_flag='xy'
  end if

  if(present(forcer0))then
     forcer0_flag=forcer0
  else
     forcer0_flag=.false.
  end if

!-- 先の引数条件をクリアしているか確認 ---
!-- この操作は行わない.

!  call search_region_1d( (/x(1),x(nx)/), (/y(1),y(ny)/), (/xc,yc/), r, nr,  &
!  &                      stdopt=stderr, axis=ax_flag(1:2) )

!-- 先に v に undef 値を入れておく.
!  do j=1,nt
!     do i=1,nrr
!        v(i,j)=r_undef
!     end do
!  end do

!-- 過程(1) ---
  if(ax_flag(1:2)=='xy')then
     do j=1,ny
        do i=1,nx
           point(i,j)=dsqrt((x(i)-xc)**2+(y(j)-yc)**2)
        end do
     end do
  else if(ax_flag(1:2)=='ll')then
     do j=1,ny
        do i=1,nx
           point(i,j)=ll2radi( xc, yc, x(i), y(j) )
        end do
     end do
  end if

!-- 過程(2) ---
  do j=1,ny
     do i=1,nx
        call interpo_search_1d( r, point(i,j), ip(i,j), undeff=i_undef,  &
  &                             stdopt=stderr )
     end do
  end do

!-- 過程(3) ---
  do j=1,ny
     do i=1,nx
        if(ip(i,j)/=i_undef.and.ip(i,j)/=nr)then
           if(present(undefg))then
              ucf=undef_checker_1dd( v(ip(i,j):ip(i,j)+1), undefg )
              if(ucf.eqv..false.)then
                 call interpolation_1d( r(ip(i,j):ip(i,j)+1),  &
  &                                     v(ip(i,j):ip(i,j)+1),  &
  &                                     point(i,j), work(i,j) )
              else
                 work(i,j)=r_undefg
!                 undefgc_check(i)=.false.
              end if
           else
              call interpolation_1d( r(ip(i,j):ip(i,j)+1),  &
  &                                  v(ip(i,j):ip(i,j)+1),  &
  &                                  point(i,j), work(i,j) )
           end if
           if(point(i,j)<r(2))then
              if(forcer0_flag.eqv..true.)then
                 work(i,j)=v(2)  ! 中心は r(1), r(2) が隣接点.
              end if
           end if
        else
           work(i,j)=r_undef
           if(point(i,j)<r(1))then
              if(forcer0_flag.eqv..true.)then
                 work(i,j)=v(1)  ! i_undef で来ているので, 中心は r(1) より内.
              end if
           end if
        end if
     end do
  end do

  do j=1,ny
     do i=1,nx
        u(i,j)=work(i,j)
     end do
  end do

end subroutine Cart_mean_scal_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_mean_vec_f( charc, x, y, xc, yc, u1, u2, r, theta, v,  &
  &                            undef, undefg, undefgc, stdopt )
  ! 任意の物理量を台風の中心から接線方向へ平均するルーチン
  ! 接線風速平均用.
  ! 引数の制限として, |x(1)-xc|, |xc-x(nx)|, |y(1)-yc|, |yc-y(ny)| > r.
  ! これ以外の場合, r で与えられる距離が領域外の範囲については
  ! undef で設定された値が代入される. undef がないときはゼロが代入される.
  ! 平均化の手順は以下のとおり.
  ! (1) nr, nt のすべての点についてそれに対応する x, y 座標値を rt_2_xy で計算.
  ! (2) その地点を含む x,y グリッドの微小領域を interpo_search_2d で検索.
  ! (3) その地点を含む 4 点が出たら, その地点でのスカラー値を 4 隅のスカラー値
  !     から, 重線形内挿 interpolation_2d で計算.
  !     これと同じループ内で, 2 つのベクトル成分の平均値が得られるので,
  !     これらを用いて vec_prod_2d によって中心からの位置ベクトルとの外積を計算
  !     . 同じループ内で中心からの距離で割って v の接線成分のみ抽出.
  ! (4) nr x nt 個の内挿接線成分値が求まったら, nt 方向に平均計算 mean_1d 使用.
  ! 以上で各 nr について平均値が得られる.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  ! なお, 本ルーチンは可読性のため, 上の過程を tangent_mean_scal ルーチンに
  ! 丸投げしている. 本ルーチン内で行われるのは, charc に合わせて,
  ! 各デカルト座標点における中心点に対する接線風, 動径風を計算するのみである.
  implicit none
  character(6), intent(in) :: charc  ! 動径成分か接線成分かの判別, vector = 接線, scalar = 動径成分.
  real, intent(in) :: x(:)  ! デカルト座標系での x 座標
  real, intent(in) :: y(:)  ! デカルト座標系での y 座標
  real, intent(in) :: u1(size(x),size(y))  ! デカルト座標系での平均化する値 1
  real, intent(in) :: u2(size(x),size(y))  ! デカルト座標系での平均化する値 2
  real, intent(in) :: xc  ! 接線平均する際の中心 x 成分.
  real, intent(in) :: yc  ! 接線平均する際の中心 y 成分.
  real, intent(in) :: r(:)  ! 平均化したときの動径方向の座標(xc からの値を入れる).
  real, intent(in) :: theta(:)  ! 平均化するときの接線方向の座標 [rad].
  real, intent(inout) :: v(size(r))  ! 平均化した u の値.
  real, intent(in), optional :: undef  ! 領域外の設定値
  real, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  integer :: i, j, nx, ny, nr, z_count
  real, dimension(size(x),size(y)) :: posx, posy, abpos, vecz
  logical :: stderr

  nx=size(x)
  ny=size(y)
  nr=size(r)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u1 ),  &
  &                                     "tangent_mean_vec" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u2 ),  &
  &                                     "tangent_mean_vec" )
     call check_array_size_dmp_message( check_array_size_1d( nr, v ),  &
  &                                     "tangent_mean_vec" )
  end if

  z_count=0

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

!-- まず, charc に合わせて, 水平風ベクトルを動径・接線風に変換する.
!-- 変換すると, その値はスカラー値として評価されるので,
!-- tangent_mean_scal に投げる.

!-- 中心点に対する各デカルト座標点の水平位置ベクトルを計算.

  do j=1,ny
     do i=1,nx
        posx(i,j)=x(i)-xc
        posy(i,j)=y(j)-yc
     end do
  end do

  select case (trim(charc))
  case ('vector')
     if(present(undefg))then
        call vec_prod_2d( posx, posy, u1, u2, vecz, undeff=undefg )
     else
        call vec_prod_2d( posx, posy, u1, u2, vecz )
     end if
  case ('scalar')
     if(present(undefg))then
        call dot_prod_2d( posx, posy, u1, u2, vecz, undeff=undefg )
     else
        call dot_prod_2d( posx, posy, u1, u2, vecz )
     end if
  case default
     write(*,*) "error : bad character. select 'vector', or 'scalar'."
     stop
  end select

  call abst_2d( posx, posy, abpos )

  if(present(undefg))then
     do j=1,ny
        do i=1,nx   
           if(vecz(i,j)/=undefg.and.abpos(i,j)/=0.0)then
              vecz(i,j)=vecz(i,j)/abpos(i,j)
           else if(abpos(i,j)==0.0)then
              z_count=z_count+1
              vecz(i,j)=0.0
           end if
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           if(abpos(i,j)/=0.0)then
              vecz(i,j)=vecz(i,j)/abpos(i,j)
           else
              z_count=z_count+1
              vecz(i,j)=0.0
           end if
        end do
     end do
  end if

  if(present(undef))then
     if(present(undefg))then
        call tangent_mean_scal_f( x, y, xc, yc, vecz, r, theta, v,  &
  &                               undef=undef, undefg=undefg,  &
  &                               undefgc=trim(undefgc), stdopt=stderr )
     else
        call tangent_mean_scal_f( x, y, xc, yc, vecz, r, theta, v,  &
  &                               undef=undef, stdopt=stderr )
     end if
  else if(present(undefg))then
     call tangent_mean_scal_f( x, y, xc, yc, vecz, r, theta, v,  &
  &                            undefg=undefg, undefgc=trim(undefgc),  &
  &                            stdopt=stderr )
  else
     call tangent_mean_scal_f( x, y, xc, yc, vecz, r, theta, v, stdopt=stderr )
  end if

end subroutine tangent_mean_vec_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_mean_vec_d( charc, x, y, xc, yc, u1, u2, r, theta, v,  &
  &                            undef, undefg, undefgc, stdopt )
  ! 任意の物理量を台風の中心から接線方向へ平均するルーチン
  ! 接線風速平均用.
  ! 引数の制限として, |x(1)-xc|, |xc-x(nx)|, |y(1)-yc|, |yc-y(ny)| > r.
  ! これ以外の場合, r で与えられる距離が領域外の範囲については
  ! undef で設定された値が代入される. undef がないときはゼロが代入される.
  ! 平均化の手順は以下のとおり.
  ! (1) nr, nt のすべての点についてそれに対応する x, y 座標値を rt_2_xy で計算.
  ! (2) その地点を含む x,y グリッドの微小領域を interpo_search_2d で検索.
  ! (3) その地点を含む 4 点が出たら, その地点でのスカラー値を 4 隅のスカラー値
  !     から, 重線形内挿 interpolation_2d で計算.
  !     これと同じループ内で, 2 つのベクトル成分の平均値が得られるので,
  !     これらを用いて vec_prod_2d によって中心からの位置ベクトルとの外積を計算
  !     . 同じループ内で中心からの距離で割って v の接線成分のみ抽出.
  ! (4) nr x nt 個の内挿接線成分値が求まったら, nt 方向に平均計算 mean_1d 使用.
  ! 以上で各 nr について平均値が得られる.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  ! なお, 本ルーチンは可読性のため, 上の過程を tangent_mean_scal ルーチンに
  ! 丸投げしている. 本ルーチン内で行われるのは, charc に合わせて,
  ! 各デカルト座標点における中心点に対する接線風, 動径風を計算するのみである.
  implicit none
  character(6), intent(in) :: charc  ! 動径成分か接線成分かの判別, vector = 接線, scalar = 動径成分.
  double precision, intent(in) :: x(:)  ! デカルト座標系での x 座標
  double precision, intent(in) :: y(:)  ! デカルト座標系での y 座標
  double precision, intent(in) :: u1(size(x),size(y))  ! デカルト座標系での平均化する値 1
  double precision, intent(in) :: u2(size(x),size(y))  ! デカルト座標系での平均化する値 2
  double precision, intent(in) :: xc  ! 接線平均する際の中心 x 成分.
  double precision, intent(in) :: yc  ! 接線平均する際の中心 y 成分.
  double precision, intent(in) :: r(:)  ! 平均化したときの動径方向の座標(xc からの値を入れる).
  double precision, intent(in) :: theta(:)  ! 平均化するときの接線方向の座標 [rad].
  double precision, intent(inout) :: v(size(r))  ! 平均化した u の値.
  double precision, intent(in), optional :: undef  ! 領域外の設定値
  double precision, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  integer :: i, j, nx, ny, nr, z_count
  double precision, dimension(size(x),size(y)) :: posx, posy, abpos, vecz
  logical :: stderr

  nx=size(x)
  ny=size(y)
  nr=size(r)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u1 ),  &
  &                                     "tangent_mean_vec" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u2 ),  &
  &                                     "tangent_mean_vec" )
     call check_array_size_dmp_message( check_array_size_1d( nr, v ),  &
  &                                     "tangent_mean_vec" )
  end if

  z_count=0

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

!-- まず, charc に合わせて, 水平風ベクトルを動径・接線風に変換する.
!-- 変換すると, その値はスカラー値として評価されるので,
!-- tangent_mean_scal に投げる.

!-- 中心点に対する各デカルト座標点の水平位置ベクトルを計算.

  do j=1,ny
     do i=1,nx
        posx(i,j)=x(i)-xc
        posy(i,j)=y(j)-yc
     end do
  end do

  select case (trim(charc))
  case ('vector')
     if(present(undefg))then
        call vec_prod_2d( posx, posy, u1, u2, vecz, undeff=undefg )
     else
        call vec_prod_2d( posx, posy, u1, u2, vecz )
     end if
  case ('scalar')
     if(present(undefg))then
        call dot_prod_2d( posx, posy, u1, u2, vecz, undeff=undefg )
     else
        call dot_prod_2d( posx, posy, u1, u2, vecz )
     end if
  case default
     write(*,*) "error : bad character. select 'vector', or 'scalar'."
     stop
  end select

  call abst_2d( posx, posy, abpos )

  if(present(undefg))then
     do j=1,ny
        do i=1,nx   
           if(vecz(i,j)/=undefg.and.abpos(i,j)/=0.0d0)then
              vecz(i,j)=vecz(i,j)/abpos(i,j)
           else if(abpos(i,j)==0.0d0)then
              z_count=z_count+1
              vecz(i,j)=0.0d0
           end if
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           if(abpos(i,j)/=0.0d0)then
              vecz(i,j)=vecz(i,j)/abpos(i,j)
           else
              z_count=z_count+1
              vecz(i,j)=0.0d0
           end if
        end do
     end do
  end if

  if(present(undef))then
     if(present(undefg))then
        call tangent_mean_scal_d( x, y, xc, yc, vecz, r, theta, v,  &
  &                               undef=undef, undefg=undefg,  &
  &                               undefgc=trim(undefgc), stdopt=stderr )
     else
        call tangent_mean_scal_d( x, y, xc, yc, vecz, r, theta, v,  &
  &                               undef=undef, stdopt=stderr )
     end if
  else if(present(undefg))then
     call tangent_mean_scal_d( x, y, xc, yc, vecz, r, theta, v,  &
  &                            undefg=undefg, undefgc=trim(undefgc),  &
  &                            stdopt=stderr )
  else
     call tangent_mean_scal_d( x, y, xc, yc, vecz, r, theta, v, stdopt=stderr )
  end if

end subroutine tangent_mean_vec_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_mean_anom_vec_f( charc, x, y, xc, yc, u1, u2, r, theta, v,  &
  &                                 undef, undefg, undefgc, stdopt )
  ! 任意の物理量を台風の中心から接線方向へ平均アノマリを計算するルーチン
  ! 接線風速平均用.
  ! 引数の制限として, |x(1)-xc|, |xc-x(nx)|, |y(1)-yc|, |yc-y(ny)| > r. 
  ! これ以外の場合, r で与えられる距離が領域外の範囲については
  ! undef で設定された値が代入される. undef がないときはゼロが代入される.
  ! 平均化の手順は以下のとおり.
  ! (1) nr, nt のすべての点についてそれに対応する x, y 座標値を rt_2_xy で計算.
  ! (2) その地点を含む x,y グリッドの微小領域を interpo_search_2d で検索.
  ! (3) その地点を含む 4 点が出たら, その地点でのスカラー値を 4 隅のスカラー値
  !     から, 重線形内挿 interpolation_2d で計算.
  !     これと同じループ内で, 2 つのベクトル成分の平均値が得られるので,
  !     これらを用いて vec_prod_2d によって中心からの位置ベクトルとの外積を計算
  !     . 同じループ内で中心からの距離で割って v の接線成分のみ抽出.
  ! (4) nr x nt 個の内挿接線成分値が求まったら, nt 方向に平均計算 mean_1d 使用.
  ! 以上で各 nr について平均値が得られる.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  ! なお, 本ルーチンは可読性のため, 上の過程を tangent_mean_scal ルーチンに
  ! 丸投げしている. 本ルーチン内で行われるのは, charc に合わせて,
  ! 各デカルト座標点における中心点に対する接線風, 動径風を計算するのみである.
  implicit none
  character(6), intent(in) :: charc  ! 動径成分か接線成分かの判別, vector = 接線, scalar = 動径成分.
  real, intent(in) :: x(:)  ! デカルト座標系での x 座標
  real, intent(in) :: y(:)  ! デカルト座標系での y 座標
  real, intent(in) :: u1(size(x),size(y))  ! デカルト座標系での平均化する値 1
  real, intent(in) :: u2(size(x),size(y))  ! デカルト座標系での平均化する値 2
  real, intent(in) :: xc  ! 接線平均する際の中心 x 成分.
  real, intent(in) :: yc  ! 接線平均する際の中心 y 成分.
  real, intent(in) :: r(:)  ! 平均化したときの動径方向の座標(xc からの値を入れる).
  real, intent(in) :: theta(:)  ! 平均化するときの接線方向の座標 [rad].
  real, intent(inout) :: v(size(r),size(theta))  ! アノマリの u の値.
  real, intent(in), optional :: undef  ! 領域外の設定値
  real, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  integer :: i, j, nx, ny, nr, nt, z_count
  real, dimension(size(x),size(y)) :: posx, posy, abpos, vecz
  logical :: stderr

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(theta)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u1 ),  &
  &                                     "tangent_mean_anom_vec" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u2 ),  &
  &                                     "tangent_mean_anom_vec" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nt, v ),  &
  &                                     "tangent_mean_anom_vec" )
  end if

  z_count=0

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

!-- まず, charc に合わせて, 水平風ベクトルを動径・接線風に変換する.
!-- 変換すると, その値はスカラー値として評価されるので,
!-- tangent_mean_scal に投げる.

!-- 中心点に対する各デカルト座標点の水平位置ベクトルを計算.

  do j=1,ny
     do i=1,nx
        posx(i,j)=x(i)-xc
        posy(i,j)=y(j)-yc
     end do
  end do

  select case (trim(charc))
  case ('vector')
     if(present(undefg))then
        call vec_prod_2d( posx, posy, u1, u2, vecz, undeff=undefg )
     else
        call vec_prod_2d( posx, posy, u1, u2, vecz )
     end if
  case ('scalar')
     if(present(undefg))then
        call dot_prod_2d( posx, posy, u1, u2, vecz, undeff=undefg )
     else
        call dot_prod_2d( posx, posy, u1, u2, vecz )
     end if
  case default
     write(*,*) "error : bad character. select 'vector', or 'scalar'."
     stop
  end select

  call abst_2d( posx, posy, abpos )

  if(present(undefg))then
     do j=1,ny
        do i=1,nx   
           if(vecz(i,j)/=undefg.and.abpos(i,j)/=0.0)then
              vecz(i,j)=vecz(i,j)/abpos(i,j)
           else if(abpos(i,j)==0.0)then
              z_count=z_count+1
              vecz(i,j)=0.0
           end if
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           if(abpos(i,j)/=0.0)then
              vecz(i,j)=vecz(i,j)/abpos(i,j)
           else
              z_count=z_count+1
              vecz(i,j)=0.0
           end if
        end do
     end do
  end if

  if(present(undef))then
     if(present(undefg))then
        call tangent_mean_anom_scal_f( x, y, xc, yc, vecz, r, theta, v,  &
  &                                    undef=undef, undefg=undefg,  &
  &                                    undefgc=trim(undefgc), stdopt=stderr )
     else
        call tangent_mean_anom_scal_f( x, y, xc, yc, vecz, r, theta, v,  &
  &                                    undef=undef, stdopt=stderr )
     end if
  else if(present(undefg))then
     call tangent_mean_anom_scal_f( x, y, xc, yc, vecz, r, theta, v,  &
  &                                 undefg=undefg, undefgc=trim(undefgc),  &
  &                                 stdopt=stderr )
  else
     call tangent_mean_anom_scal_f( x, y, xc, yc, vecz, r, theta, v,  &
  &                                 stdopt=stderr )
  end if

end subroutine tangent_mean_anom_vec_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_mean_anom_vec_d( charc, x, y, xc, yc, u1, u2, r, theta, v,  &
  &                                 undef, undefg, undefgc, stdopt )
  ! 任意の物理量を台風の中心から接線方向へ平均アノマリを計算するルーチン
  ! 接線風速平均用.
  ! 引数の制限として, |x(1)-xc|, |xc-x(nx)|, |y(1)-yc|, |yc-y(ny)| > r. 
  ! これ以外の場合, r で与えられる距離が領域外の範囲については
  ! undef で設定された値が代入される. undef がないときはゼロが代入される.
  ! 平均化の手順は以下のとおり.
  ! (1) nr, nt のすべての点についてそれに対応する x, y 座標値を rt_2_xy で計算.
  ! (2) その地点を含む x,y グリッドの微小領域を interpo_search_2d で検索.
  ! (3) その地点を含む 4 点が出たら, その地点でのスカラー値を 4 隅のスカラー値
  !     から, 重線形内挿 interpolation_2d で計算.
  !     これと同じループ内で, 2 つのベクトル成分の平均値が得られるので,
  !     これらを用いて vec_prod_2d によって中心からの位置ベクトルとの外積を計算
  !     . 同じループ内で中心からの距離で割って v の接線成分のみ抽出.
  ! (4) nr x nt 個の内挿接線成分値が求まったら, nt 方向に平均計算 mean_1d 使用.
  ! 以上で各 nr について平均値が得られる.
  ! 以下, 処理の都合で所々に present(undefg) が入っているが,
  ! 純粋な処理には関係ないので, ソースを読む場合は, present(undefg) の else
  ! の箇所を参照されたい.
  ! なお, 本ルーチンは可読性のため, 上の過程を tangent_mean_scal ルーチンに
  ! 丸投げしている. 本ルーチン内で行われるのは, charc に合わせて,
  ! 各デカルト座標点における中心点に対する接線風, 動径風を計算するのみである.
  implicit none
  character(6), intent(in) :: charc  ! 動径成分か接線成分かの判別, vector = 接線, scalar = 動径成分.
  double precision, intent(in) :: x(:)  ! デカルト座標系での x 座標
  double precision, intent(in) :: y(:)  ! デカルト座標系での y 座標
  double precision, intent(in) :: u1(size(x),size(y))  ! デカルト座標系での平均化する値 1
  double precision, intent(in) :: u2(size(x),size(y))  ! デカルト座標系での平均化する値 2
  double precision, intent(in) :: xc  ! 接線平均する際の中心 x 成分.
  double precision, intent(in) :: yc  ! 接線平均する際の中心 y 成分.
  double precision, intent(in) :: r(:)  ! 平均化したときの動径方向の座標(xc からの値を入れる).
  double precision, intent(in) :: theta(:)  ! 平均化するときの接線方向の座標 [rad].
  double precision, intent(inout) :: v(size(r),size(theta))  ! アノマリの u の値.
  double precision, intent(in), optional :: undef  ! 領域外の設定値
  double precision, intent(in), optional :: undefg  ! 格子点に欠損がある場合の内挿未定義値
  character(3), intent(in), optional :: undefgc  ! undefg がある場合の処理
                ! "inc" = その格子点を参照値として内挿する点のみ平均操作時に除外して計算.
                ! "err" = その格子点を参照値として内挿する点を平均操作時に含む場合, 平均値そのものを未定義として計算. この場合, 未定義値は undefg となる.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  integer :: i, j, nx, ny, nr, nt, z_count
  double precision, dimension(size(x),size(y)) :: posx, posy, abpos, vecz
  logical :: stderr

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(theta)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u1 ),  &
  &                                     "tangent_mean_anom_vec" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u2 ),  &
  &                                     "tangent_mean_anom_vec" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nt, v ),  &
  &                                     "tangent_mean_anom_vec" )
  end if

  z_count=0

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

!-- まず, charc に合わせて, 水平風ベクトルを動径・接線風に変換する.
!-- 変換すると, その値はスカラー値として評価されるので,
!-- tangent_mean_scal に投げる.

!-- 中心点に対する各デカルト座標点の水平位置ベクトルを計算.

  do j=1,ny
     do i=1,nx
        posx(i,j)=x(i)-xc
        posy(i,j)=y(j)-yc
     end do
  end do

  select case (trim(charc))
  case ('vector')
     if(present(undefg))then
        call vec_prod_2d( posx, posy, u1, u2, vecz, undeff=undefg )
     else
        call vec_prod_2d( posx, posy, u1, u2, vecz )
     end if
  case ('scalar')
     if(present(undefg))then
        call dot_prod_2d( posx, posy, u1, u2, vecz, undeff=undefg )
     else
        call dot_prod_2d( posx, posy, u1, u2, vecz )
     end if
  case default
     write(*,*) "error : bad character. select 'vector', or 'scalar'."
     stop
  end select

  call abst_2d( posx, posy, abpos )

  if(present(undefg))then
     do j=1,ny
        do i=1,nx   
           if(vecz(i,j)/=undefg.and.abpos(i,j)/=0.0d0)then
              vecz(i,j)=vecz(i,j)/abpos(i,j)
           else if(abpos(i,j)==0.0d0)then
              z_count=z_count+1
              vecz(i,j)=0.0d0
           end if
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           if(abpos(i,j)/=0.0d0)then
              vecz(i,j)=vecz(i,j)/abpos(i,j)
           else
              z_count=z_count+1
              vecz(i,j)=0.0d0
           end if
        end do
     end do
  end if

  if(present(undef))then
     if(present(undefg))then
        call tangent_mean_anom_scal_d( x, y, xc, yc, vecz, r, theta, v,  &
  &                                    undef=undef, undefg=undefg,  &
  &                                    undefgc=trim(undefgc), stdopt=stderr )
     else
        call tangent_mean_anom_scal_d( x, y, xc, yc, vecz, r, theta, v,  &
  &                                    undef=undef, stdopt=stderr )
     end if
  else if(present(undefg))then
     call tangent_mean_anom_scal_d( x, y, xc, yc, vecz, r, theta, v,  &
  &                                 undefg=undefg, undefgc=trim(undefgc),  &
  &                                 stdopt=stderr )
  else
     call tangent_mean_anom_scal_d( x, y, xc, yc, vecz, r, theta, v,  &
  &                                 stdopt=stderr )
  end if

end subroutine tangent_mean_anom_vec_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_mean_turb( signal, r, z, u, v, w, rho, nuh, nuv, val, undef, sfctau )
!  接線平均した乱流フラックスを計算する.
!  接線平均しているので, tau_{*2} 成分 (\theta 微分の成分) は含まれない.
  implicit none
  character(1) :: signal  ! 円筒座標系の何番目の乱流成分かを判定する.
                  ! [1] = 円筒座標における radial 座標成分 (方程式 vr 成分)
                  ! [2] = 円筒座標における tangential 座標成分 (方程式 vt 成分)
                  ! [3] = 円筒座標における vertical 座標成分 (方程式 w 成分)
  real, intent(in) :: r(:)  ! 動径方向の位置座標 [m]
  real, intent(in) :: z(:)  ! 鉛直方向の位置座標 [m]
  real, intent(in) :: u(size(r),size(z))  ! x に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: v(size(r),size(z))  ! y に対応する方向の 2 次元ベクトル成分
  real, intent(in) :: w(size(r),size(z))  ! y に対応する方向の 2 次元ベ>クトル成分
  real, intent(in) :: rho(size(z))  ! 水平面に平均した基本場の密度 [kg/m^3]
  real, intent(in) :: nuh(size(r),size(z))  ! 水平渦粘性係数
  real, intent(in) :: nuv(size(r),size(z))  ! 鉛直渦粘性係数
  real, intent(inout) :: val(size(r),size(z))  ! 乱流フラックス
  real, intent(in), optional :: undef
  real, intent(in), optional :: sfctau(size(r))  ! 地表面からのフラックス
                 ! これが与えられれば, 最下層の応力はこれで置き換える.
  integer :: i   ! イタレーション用添字
!  integer :: j   ! イタレーション用添字
  integer :: k   ! イタレーション用添字
  integer :: id   ! イタレーション用添字
  integer :: nr  ! 空間配列要素数 1 次元目
  integer :: nz  ! 空間配列要素数 2 次元目
  real :: dr  ! 1 次元目を微分する格子間隔 [m]
  real :: dz  ! 2 次元目を微分する格子間隔 [m]
  character(1) :: signaltau(3)
  real, dimension(size(r),size(z),3) :: tau  ! signal 方向に
              ! 作用する 1,2,3 面に垂直な応力
  real, dimension(size(r),size(z)) :: tmp
  real, dimension(size(r)) :: stau

  signaltau=(/ '1', '2', '3' /)

  dr=r(2)-r(1)
  dz=z(2)-z(1)
  nr=size(r)
  nz=size(z) 

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, u ),  &
  &                                     "tangent_mean_turb" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, v ),  &
  &                                     "tangent_mean_turb" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, w ),  &
  &                                     "tangent_mean_turb" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, nuh ),  &
  &                                     "tangent_mean_turb" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, nuv ),  &
  &                                     "tangent_mean_turb" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, val ),  &
  &                                     "tangent_mean_turb" )
     call check_array_size_dmp_message( check_array_size_1d( nz, rho ),  &
  &                                     "tangent_mean_turb" )
     call check_array_size_dmp_message( check_array_size_1d( nr, sfctau ),  &
  &                                     "tangent_mean_turb" )
  end if

  val=0.0

  do id=1,3
     if(id/=2)then  ! tau_{*2} 成分はゼロなので, 計算しない.
        if(present(sfctau))then
           stau(:)=sfctau(:)
           call tangent_mean_Reynolds( signal//signaltau(id),  &
  &             r, z, u, v, w, rho, nuh, nuv, tau(:,:,id), sfctau=stau )
        else
           call tangent_mean_Reynolds( signal//signaltau(id),  &
  &             r, z, u, v, w, rho, nuh, nuv, tau(:,:,id) )
        end if
     end if
  end do

!-- (signal, 1) 成分の計算
  do k=1,nz
     call grad_1d( r, tau(:,k,1), tmp(:,k))
     do i=1,nr
        if(r(i)/=0.0)then
           val(i,k)=tmp(i,k)+val(i,k)+tau(i,k,1)/r(i)
        else
           val(i,k)=tmp(i,k)+val(i,k)
        end if
     end do
  end do

!-- (signal, 3) 成分の計算
  do i=1,nr
     call grad_1d( z, tau(i,:,3), tmp(i,:))
     do k=1,nz
        val(i,k)=tmp(i,k)+val(i,k)
     end do
  end do



end subroutine

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_mean_Reynolds( signal, r, z, u, v, w, rho, nuh, nuv, val, undef, sfctau )
! 円筒座標系におけるレイノルズ応力テンソルを計算する.
  implicit none
  character(2) :: signal  ! 計算するテンソル成分.
                  ! ['11', '22', '33'] = それぞれ対角テンソル成分
                  ! ['12', '13', '21', '23', '31', '32'] = それぞれ非対角
                  ! テンソル成分. ただし, 対称テンソルであるため, '12'='21' を
                  ! 計算していることに注意.
  real, intent(in) :: r(:)  ! radial 方向の空間座標 [m]
  real, intent(in) :: z(:)  ! vertical 方向の空間座標 [m]
  real, intent(in) :: u(size(r),size(z))  ! radial に対応する方向の 3 次元ベクトル成分
  real, intent(in) :: v(size(r),size(z))  ! tangential に対応する方向の 3 次元ベクトル成分
  real, intent(in) :: w(size(r),size(z))  ! vertical に対応する方向の 3 次元ベクトル成分
  real, intent(in) :: rho(size(z))  ! 水平面に平均した基本場の密度 [kg/m^3]
  real, intent(in) :: nuh(size(r),size(z))  ! 水平渦粘性係数
  real, intent(in) :: nuv(size(r),size(z))  ! 鉛直渦粘性係数
  real, intent(inout) :: val(size(r),size(z))  ! 計算されたテンソル成分
! 現在, 以下のオプションは使用していない.
  real, intent(in), optional :: undef
  real, intent(in), optional :: sfctau(size(r))  ! 地表面からのフラックス
                 ! これが与えられれば, 最下層の応力はこれで置き換える.
  integer :: i   ! イタレーション用添字
!  integer :: j   ! イタレーション用添字
  integer :: k   ! イタレーション用添字
  integer :: nr  ! 空間配列要素数 1 次元目
  integer :: nz  ! 空間配列要素数 3 次元目
  real :: dr  ! 1 次元目を微分する格子間隔 [m]
  real :: dz  ! 3 次元目を微分する格子間隔 [m]
  real :: sxx(size(r),size(z)), nu(size(r),size(z))
  real :: stau(size(r))

  dr=r(2)-r(1)
  dz=z(2)-z(1)
  nr=size(r)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, u ),  &
  &                                     "tangent_mean_Reynolds" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, v ),  &
  &                                     "tangent_mean_Reynolds" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, w ),  &
  &                                     "tangent_mean_Reynolds" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, nuh ),  &
  &                                     "tangent_mean_Reynolds" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, nuv ),  &
  &                                     "tangent_mean_Reynolds" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, val ),  &
  &                                     "tangent_mean_Reynolds" )
     call check_array_size_dmp_message( check_array_size_1d( nz, rho ),  &
  &                                     "tangent_mean_Reynolds" )
     call check_array_size_dmp_message( check_array_size_1d( nr, sfctau ),  &
  &                                     "tangent_mean_Reynolds" )
  end if

  val=0.0
  stau=0.0

  if(present(sfctau))then
     if(signal(2:2)=='3'.and.signal(1:1)/='3')then
        stau(:)=sfctau(:)
     end if
  end if

!-- [NOTE]
!-- 以下, 文字で case の or ができないため, 
!-- if 文の入れ子ではなく, if 文の並列表記で case と同じように見せかける.
!-- これはもちろん, 上から順に if をたどるが, どの場合も 2 種類以上の if に
!-- 合致しないことが既知であるために可能となる書き方であり,
!-- 並列表記した if の 2 パターン以上に合致してしまうような条件文では,
!-- case の代用には用いることができないことに注意.
!-- 本ライブラリでこのような紛らわしい表記をしている場合は必ず NOTE が入る.

  if(signal(1:2)=='12'.or.signal(1:2)=='21')then
     call tangent_mean_deform( signal, r, z, u, v, w, sxx )

     do k=1,nz
        do i=1,nr
           nu(i,k)=nuh(i,k)
        end do
     end do
  end if

  if(signal(1:2)=='23'.or.signal(1:2)=='32')then
     call tangent_mean_deform( signal, r, z, u, v, w, sxx )

     if(signal(2:2)=='3')then
        do k=1,nz
           do i=1,nr
              nu(i,k)=nuv(i,k)
           end do
        end do
     else
        do k=1,nz
           do i=1,nr
              nu(i,k)=nuh(i,k)
           end do
        end do
     end if
  end if

  if(signal(1:2)=='13'.or.signal(1:2)=='31')then
     call tangent_mean_deform( signal, r, z, u, v, w, sxx )

     if(signal(2:2)=='3')then
        do k=1,nz
           do i=1,nr
              nu(i,k)=nuv(i,k)
           end do
        end do
     else
        do k=1,nz
           do i=1,nr
              nu(i,k)=nuh(i,k)
           end do
        end do
     end if
  end if

  if(signal(1:2)=='11')then
     call tangent_mean_deform( signal, r, z, u, v, w, sxx )
     call div( r, z, u, w, val )
     do k=1,nz
        do i=1,nr
           if(r(i)/=0.0)then
              val(i,k)=val(i,k)+u(i,k)/r(i)
           end if
        end do
     end do

     do k=1,nz
        do i=1,nr
           nu(i,k)=nuh(i,k)
        end do
     end do
  end if

  if(signal(1:2)=='22')then
     call tangent_mean_deform( signal, r, z, u, v, w, sxx )
     call div( r, z, u, w, val )
     do k=1,nz
        do i=1,nr
           if(r(i)/=0.0)then
              val(i,k)=val(i,k)+u(i,k)/r(i)
           end if
        end do
     end do

     do k=1,nz
        do i=1,nr
           nu(i,k)=nuh(i,k)
        end do
     end do

  end if

  if(signal(1:2)=='33')then
     call tangent_mean_deform( signal, r, z, u, v, w, sxx )
     call div( r, z, u, w, val )
     do k=1,nz
        do i=1,nr
           if(r(i)/=0.0)then
              val(i,k)=val(i,k)+u(i,k)/r(i)
           end if
        end do
     end do

     do k=1,nz
        do i=1,nr
           nu(i,k)=nuh(i,k)
        end do
     end do

  end if

!-- 以下の式は, 最初 val = 0 で if 文の中で計算されているものとされていない
!-- ものに分かれるので, 統一の式で評価ができる.
!-- 計算されていないものについてはそもそもゼロである.

!-- 以下, 最下層は地表面フラックスを代入するかどうかのオプションのため, 別ループ

  if(present(sfctau))then
     do i=1,nr
        val(i,1)=stau(i)
     end do
  else
     do i=1,nr
        val(i,1)=rho(1)*nu(i,1)*(sxx(i,1)-(2.0/3.0)*val(i,1))
     end do
  end if

  do k=2,nz
     do i=1,nr
        val(i,k)=rho(k)*nu(i,k)*(sxx(i,k)-(2.0/3.0)*val(i,k))
     end do
  end do

end subroutine


!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine calc_taufil_f( x, y, u, v, val, undef, sx, sy )
! Rozoff et al. (2006) で定義される filamentation time を計算する.
  implicit none
  real, intent(in) :: x(:)  ! 空間座標第一成分 [任意]
  real, intent(in) :: y(:)  ! 空間座標第二成分 [任意]
  real, intent(in) :: u(size(x),size(y))  ! x 方向 2 次元風ベクトル成分
  real, intent(in) :: v(size(x),size(y))  ! x 方向 2 次元風ベクトル成分
  real, intent(inout) :: val(size(x),size(y))  ! 計算された tau_fil
  real, intent(in), optional :: undef
  real, intent(in), optional :: sx(size(x),size(y))  ! スケール因子 x 成分
  real, intent(in), optional :: sy(size(x),size(y))  ! スケール因子 y 成分
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  real, dimension(size(x),size(y)) :: hx, hy  ! sx, sy
  real, dimension(size(x),size(y)) :: s1, s2, s3  ! S1, S2, zeta
  real, dimension(size(x),size(y)) :: dudx, dudy, dvdx, dvdy
  real, dimension(size(x),size(y)) :: ds1dx, ds1dy, ds2dx, ds2dy
  logical :: sx_flag, sy_flag

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "calc_taufil" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, v ),  &
  &                                     "calc_taufil" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, val ),  &
  &                                     "calc_taufil" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, sx ),  &
  &                                     "calc_taufil" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, sy ),  &
  &                                     "calc_taufil" )
  end if

  if(present(sx))then
     sx_flag=.true.
     hx=sx
  else
     sx_flag=.false.
     ds1dx=0.0
     ds1dy=0.0
     hx=1.0
  end if
  if(present(sy))then
     sy_flag=.true.
     hy=sy
  else
     sy_flag=.false.
     ds2dx=0.0
     ds2dy=0.0
     hy=1.0
  end if

  if(present(undef))then

  !-- calculating essential variables
     call grad_2d( x, y, u, dudx, dudy, undeff=undef )
     call grad_2d( x, y, v, dvdx, dvdy, undeff=undef )
  !-- calculating scale factor variables
     if(sx_flag.eqv..true.)then
        call grad_2d( x, y, hx, ds1dx, ds1dy, undeff=undef )
     end if
     if(sy_flag.eqv..true.)then
        call grad_2d( x, y, hy, ds2dx, ds2dy, undeff=undef )
     end if
  !-- calculating each value
     do j=1,ny
        do i=1,nx
           if(dudx(i,j)/=undef.and.dudy(i,j)/=undef.and.  &
  &           dvdx(i,j)/=undef.and.dvdy(i,j)/=undef)then
              s1(i,j)=dudx(i,j)/hx(i,j)-dvdy(i,j)/hy(i,j)  &
  &                   +(v(i,j)*ds1dy(i,j)-u(i,j)*ds2dx(i,j))/(hx(i,j)*hy(i,j))
              s2(i,j)=dvdx(i,j)/hx(i,j)+dudy(i,j)/hy(i,j)  &
  &                   -(u(i,j)*ds2dx(i,j)+v(i,j)*ds1dy(i,j))/(hx(i,j)*hy(i,j))
              s3(i,j)=dvdx(i,j)/hx(i,j)-dudy(i,j)/hy(i,j)  &
  &                   +(v(i,j)*ds2dx(i,j)-u(i,j)*ds1dy(i,j))/(hx(i,j)*hy(i,j))
              val(i,j)=s1(i,j)**2+s2(i,j)**2-s3(i,j)**2
              if(val(i,j)>0.0)then
                 val(i,j)=2.0/sqrt(val(i,j))
              else
                 val(i,j)=undef
              end if
           else
              val(i,j)=undef
           end if
        end do
     end do

  else
     
  !-- calculating essential variables
     call grad_2d( x, y, u, dudx, dudy )
     call grad_2d( x, y, v, dvdx, dvdy )
  !-- calculating scale factor variables
     if(sx_flag.eqv..true.)then
        call grad_2d( x, y, hx, ds1dx, ds1dy )
     end if
     if(sy_flag.eqv..true.)then
        call grad_2d( x, y, hy, ds2dx, ds2dy )
     end if
  !-- calculating each value
     do j=1,ny
        do i=1,nx
           s1(i,j)=dudx(i,j)/hx(i,j)-dvdy(i,j)/hy(i,j)  &
  &                +(v(i,j)*ds1dy(i,j)-u(i,j)*ds2dx(i,j))/(hx(i,j)*hy(i,j))
           s2(i,j)=dvdx(i,j)/hx(i,j)+dudy(i,j)/hy(i,j)  &
  &                -(u(i,j)*ds2dx(i,j)+v(i,j)*ds1dy(i,j))/(hx(i,j)*hy(i,j))
           s3(i,j)=dvdx(i,j)/hx(i,j)-dudy(i,j)/hy(i,j)  &
  &                +(v(i,j)*ds2dx(i,j)-u(i,j)*ds1dy(i,j))/(hx(i,j)*hy(i,j))
           val(i,j)=s1(i,j)**2+s2(i,j)**2-s3(i,j)**2
           if(val(i,j)>0.0)then
              val(i,j)=2.0/sqrt(val(i,j))
           else
              val(i,j)=0.0
           end if
        end do
     end do

  end if

end subroutine calc_taufil_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine calc_taufil_d( x, y, u, v, val, undef, sx, sy )
! Rozoff et al. (2006) で定義される filamentation time を計算する.
  implicit none
  double precision, intent(in) :: x(:)  ! 空間座標第一成分 [任意]
  double precision, intent(in) :: y(:)  ! 空間座標第二成分 [任意]
  double precision, intent(in) :: u(size(x),size(y))  ! x 方向 2 次元風ベクトル成分
  double precision, intent(in) :: v(size(x),size(y))  ! x 方向 2 次元風ベクトル成分
  double precision, intent(inout) :: val(size(x),size(y))  ! 計算された tau_fil
  double precision, intent(in), optional :: undef
  double precision, intent(in), optional :: sx(size(x),size(y))  ! スケール因子 x 成分
  double precision, intent(in), optional :: sy(size(x),size(y))  ! スケール因子 y 成分
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: nx  ! 空間配列要素数 1 次元目
  integer :: ny  ! 空間配列要素数 2 次元目
  double precision, dimension(size(x),size(y)) :: hx, hy  ! sx, sy
  double precision, dimension(size(x),size(y)) :: s1, s2, s3  ! S1, S2, zeta
  double precision, dimension(size(x),size(y)) :: dudx, dudy, dvdx, dvdy
  double precision, dimension(size(x),size(y)) :: ds1dx, ds1dy, ds2dx, ds2dy
  logical :: sx_flag, sy_flag

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "calc_taufil" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, v ),  &
  &                                     "calc_taufil" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, val ),  &
  &                                     "calc_taufil" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, sx ),  &
  &                                     "calc_taufil" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, sy ),  &
  &                                     "calc_taufil" )
  end if

  if(present(sx))then
     sx_flag=.true.
     hx=sx
  else
     sx_flag=.false.
     ds1dx=0.0d0
     ds1dy=0.0d0
     hx=1.0d0
  end if
  if(present(sy))then
     sy_flag=.true.
     hy=sy
  else
     sy_flag=.false.
     ds2dx=0.0d0
     ds2dy=0.0d0
     hy=1.0d0
  end if

  if(present(undef))then

  !-- calculating essential variables
     call grad_2d( x, y, u, dudx, dudy, undeff=undef )
     call grad_2d( x, y, v, dvdx, dvdy, undeff=undef )
  !-- calculating scale factor variables
     if(sx_flag.eqv..true.)then
        call grad_2d( x, y, hx, ds1dx, ds1dy, undeff=undef )
     end if
     if(sy_flag.eqv..true.)then
        call grad_2d( x, y, hy, ds2dx, ds2dy, undeff=undef )
     end if
  !-- calculating each value
     do j=1,ny
        do i=1,nx
           if(dudx(i,j)/=undef.and.dudy(i,j)/=undef.and.  &
  &           dvdx(i,j)/=undef.and.dvdy(i,j)/=undef)then
              s1(i,j)=dudx(i,j)/hx(i,j)-dvdy(i,j)/hy(i,j)  &
  &                   +(v(i,j)*ds1dy(i,j)-u(i,j)*ds2dx(i,j))/(hx(i,j)*hy(i,j))
              s2(i,j)=dvdx(i,j)/hx(i,j)+dudy(i,j)/hy(i,j)  &
  &                   -(u(i,j)*ds2dx(i,j)+v(i,j)*ds1dy(i,j))/(hx(i,j)*hy(i,j))
              s3(i,j)=dvdx(i,j)/hx(i,j)-dudy(i,j)/hy(i,j)  &
  &                   +(v(i,j)*ds2dx(i,j)-u(i,j)*ds1dy(i,j))/(hx(i,j)*hy(i,j))
              val(i,j)=s1(i,j)**2+s2(i,j)**2-s3(i,j)**2
              if(val(i,j)>0.0d0)then
                 val(i,j)=2.0d0/sqrt(val(i,j))
              else
                 val(i,j)=undef
              end if
           else
              val(i,j)=undef
           end if
        end do
     end do

  else
     
  !-- calculating essential variables
     call grad_2d( x, y, u, dudx, dudy )
     call grad_2d( x, y, v, dvdx, dvdy )
  !-- calculating scale factor variables
     if(sx_flag.eqv..true.)then
        call grad_2d( x, y, hx, ds1dx, ds1dy )
     end if
     if(sy_flag.eqv..true.)then
        call grad_2d( x, y, hy, ds2dx, ds2dy )
     end if
  !-- calculating each value
     do j=1,ny
        do i=1,nx
           s1(i,j)=dudx(i,j)/hx(i,j)-dvdy(i,j)/hy(i,j)  &
  &                +(v(i,j)*ds1dy(i,j)-u(i,j)*ds2dx(i,j))/(hx(i,j)*hy(i,j))
           s2(i,j)=dvdx(i,j)/hx(i,j)+dudy(i,j)/hy(i,j)  &
  &                -(u(i,j)*ds2dx(i,j)+v(i,j)*ds1dy(i,j))/(hx(i,j)*hy(i,j))
           s3(i,j)=dvdx(i,j)/hx(i,j)-dudy(i,j)/hy(i,j)  &
  &                +(v(i,j)*ds2dx(i,j)-u(i,j)*ds1dy(i,j))/(hx(i,j)*hy(i,j))
           val(i,j)=s1(i,j)**2+s2(i,j)**2-s3(i,j)**2
           if(val(i,j)>0.0d0)then
              val(i,j)=2.0d0/dsqrt(val(i,j))
           else
              val(i,j)=0.0d0
           end if
        end do
     end do

  end if

end subroutine calc_taufil_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine tangent_mean_deform( signal, r, z, u, v, w, val, undef )
! デカルト座標系における変形速度テンソルを計算する.
  implicit none
  character(2) :: signal  ! 計算するテンソル成分.
                  ! ['11', '22', '33'] = それぞれ対角テンソル成分
                  ! ['12', '13', '21', '23', '31', '32'] = それぞれ非対角
                  ! テンソル成分. ただし, 対称テンソルであるため, '12'='21' を
                  ! 計算していることに注意.
  real, intent(in) :: r(:)  ! radial 方向の空間座標 [m]
  real, intent(in) :: z(:)  ! vertical 方向の空間座標 [m]
  real, intent(in) :: u(size(r),size(z))  ! radial に対応する方向の 3 次元ベクトル成分
  real, intent(in) :: v(size(r),size(z))  ! tangential に対応する方向の 3 次元ベクトル成分
  real, intent(in) :: w(size(r),size(z))  ! vertical に対応する方向の 3 次元ベクトル成分
  real, intent(inout) :: val(size(r),size(z))  ! 計算されたテンソル成分
! 現在, 以下のオプションは使用していない.
  real, intent(in), optional :: undef
  integer :: i   ! イタレーション用添字
  integer :: j   ! イタレーション用添字
  integer :: k   ! イタレーション用添字
  integer :: nr  ! 空間配列要素数 1 次元目
  integer :: nz  ! 空間配列要素数 2 次元目
  real :: dr  ! 1 次元目を微分する格子間隔 [m]
  real :: dz  ! 2 次元目を微分する格子間隔 [m]

  dr=r(2)-r(1)
  dz=z(2)-z(1)
  nr=size(r)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, u ),  &
  &                                     "tangent_mean_deform" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, v ),  &
  &                                     "tangent_mean_deform" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, w ),  &
  &                                     "tangent_mean_deform" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, val ),  &
  &                                     "tangent_mean_deform" )
  end if

!-- [NOTE]
!-- 以下, 文字で case の or ができないため, 
!-- if 文の入れ子ではなく, if 文の並列表記で case と同じように見せかける.
!-- これはもちろん, 上から順に if をたどるが, どの場合も 2 種類以上の if に
!-- 合致しないことが既知であるために可能となる書き方であり,
!-- 並列表記した if の 2 パターン以上に合致してしまうような条件文では,
!-- case の代用には用いることができないことに注意.
!-- 本ライブラリでこのような紛らわしい表記をしている場合は必ず NOTE が入る.

  if(signal(1:2)=='12'.or.signal(1:2)=='21')then
     do k=1,nz
        call grad_1d( r, v(:,k), val(:,k) )
        do i=1,nr
           if(r(i)/=0.0)then
              val(i,k)=val(i,k)-v(i,k)/r(i)
           end if
        end do
     end do
  end if

  if(signal(1:2)=='23'.or.signal(1:2)=='32')then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nr
        call grad_1d( z, v(k,:), val(k,:) )
     end do
!$omp end do
!$omp end parallel
  end if

  if(signal(1:2)=='13'.or.signal(1:2)=='31')then
     call div( r, z, w, u, val )
  end if

  if(signal(1:2)=='11')then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        call grad_1d( r, u(:,k), val(:,k) )
        val(:,k)=2.0*val(:,k)
     end do
!$omp end do
!$omp end parallel
  end if

  if(signal(1:2)=='22')then
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(j,k)
     do k=1,nz
        do j=1,nr
           if(r(j)/=0.0)then
              val(j,k)=2.0*u(j,k)/r(j)
           else
              val(j,k)=0.0
           end if
        end do
     end do
!$omp end do
!$omp end parallel
  end if

  if(signal(1:2)=='33')then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(j)
     do j=1,nr
        call grad_1d( z, w(j,:), val(j,:) )
        val(j,:)=2.0*val(j,:)
     end do
!$omp end do
!$omp end parallel
  end if

end subroutine

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine hydro_grad_eqb_it_f( r, z, coril, v, pres_s, rho_s,  &
  &                             pres, rho, error, bound, pres_zb, rho_zb, dl )
!  サウンディングと軸対称流から静力学・傾度風平衡場の計算.
  use Thermo_Const
  use Phys_Const
  implicit none
  real, intent(in) :: r(:)  ! 動径座標 [m]
  real, intent(in) :: z(:)  ! 鉛直座標 [m]
  real, intent(in) :: coril(size(r),size(z))  ! コリオリパラメータ [/s]
  real, intent(in) :: v(size(r),size(z))  ! 軸対称流 [m/s]
  real, intent(in) :: pres_s(size(z))  ! サウンディングの気圧 [Pa]
  real, intent(in) :: rho_s(size(z))  ! サウンディングの密度 [kg/m^3]
  real, intent(in), optional :: error  ! イタレーションの収束条件
                    ! default = 1.0e-5
  real, intent(inout) :: pres(size(r),size(z))  ! 平衡場の気圧 [Pa]
  real, intent(inout) :: rho(size(r),size(z))  ! 平衡場の密度 [kg/m^3]
  real, intent(in), optional :: pres_zb(size(r))  ! 気圧の鉛直境界条件 [Pa]
  real, intent(in), optional :: rho_zb(size(r))  ! 密度の鉛直境界条件 [kg/m^3]
  character(2), intent(in), optional :: bound  ! 境界条件の設定場所
                         ! 1 文字目: r -> "i" or "o"
                         ! 2 文字目: z -> "b" or "t",
                         ! デフォルト: "ot"
  integer, intent(in), optional :: dl  ! デバッグレベル
  real :: dr(size(r)), dz(size(z))
  real :: pres_zbound(size(r)), rho_zbound(size(r))
  real, dimension(size(r),size(z)) :: old_pres, old_rho, tmprho
  real, dimension(size(r),size(z)) :: force
  integer :: nr, nz
  integer :: i, j, nrb, nzb, nri, nro
  real :: err, err_tmp, err_max, tmpp, tmpc, tmpr
  character(2) :: bc

  nr=size(r)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, coril ),  &
  &                                     "hydro_grad_eqb_it" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, v ),  &
  &                                     "hydro_grad_eqb_it" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, pres ),  &
  &                                     "hydro_grad_eqb_it" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, rho ),  &
  &                                     "hydro_grad_eqb_it" )
     call check_array_size_dmp_message( check_array_size_1d( nz, pres_s ),  &
  &                                     "hydro_grad_eqb_it" )
     call check_array_size_dmp_message( check_array_size_1d( nz, rho_s ),  &
  &                                     "hydro_grad_eqb_it" )
  end if

  if(present(error))then
     err_max=error
  else
     err_max=1.0e-5
  end if

  if(present(bound))then
     bc=bound(1:2)
  else
     bc="ot"
  end if

  select case (bc(1:1))
  case ("i")
     nrb=1
  case ("o")
     nrb=nr
  end select

  select case (bc(2:2))
  case ("b")
     nzb=1
  case ("t")
     nzb=nz
  end select

  if(present(pres_zb))then
     pres_zbound=pres_zb
  else
     pres_zbound=pres_s(nzb)
  end if

  if(present(rho_zb))then
     rho_zbound=rho_zb
  else
     rho_zbound=rho_s(nzb)
  end if

!-- 以下で各高度において, 密度は一定であるとして傾度風平衡から気圧を計算,
!-- その値を用いて静力学平衡から密度を修正. eps 以下になるまで繰り返す.
!-- 外縁で 2 次元場とサウンディングを一致.
  do j=1,nz
     do i=1,nr
        old_pres(i,j)=pres_s(j)
     end do
  end do
!-- 密度については, 水平面一様で設定
  do j=1,nz
     do i=1,nr
        old_rho(i,j)=log(rho_s(j))
     end do
  end do

  select case (bc(1:1))
  case ("i")
     do i=2,nr
        dr(i)=r(i)-r(i-1)
     end do
  case ("o")
     do i=1,nr-1
        dr(i)=r(i+1)-r(i)
     end do
  end select
  select case (bc(2:2))
  case ("b")
     do j=2,nz
        dz(j)=z(j)-z(j-1)
     end do
  case ("t")
     do j=1,nz-1
        dz(j)=z(j+1)-z(j)
     end do
  end select

!-- 以下でイタレーション開始.
  err=err_max

  do while(err>=err_max)
     err=0.0

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j)

     do j=1,nz
        do i=1,nr
           if(r(i)/=0.0)then
              force(i,j)=coril(i,j)*v(i,j)+v(i,j)*v(i,j)/r(i)
           else
              force(i,j)=0.5*coril(i,j)*(v(i,j)+v(i+1,j))  &
  &                     +0.25*(v(i,j)+v(i+1,j))**2/(0.5*r(i+1))  ! tentative manner
              !force(i,j)=0.0
           end if
        end do
     end do

!$omp end do
!$omp end parallel

!-- 境界条件の位置に応じて反復の範囲を変更
     select case (bc(1:2))
     case("ot")  ! nr, nz が境界

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,tmpp,tmpc,tmpr)

        do j=1,nz-1
           do i=1,nr-1
              tmpp=old_pres(i+1,j)*dz(j)+force(i,j)*dr(i)*old_pres(i,j+1)/g
              tmpc=dz(j)+dr(i)*force(i,j)/g
              tmpr=old_rho(i+1,j)*dz(j)+force(i,j)*dr(i)*old_rho(i,j+1)/g  &
  &                +(force(i,j+1)-force(i,j))*dr(i)/g
              pres(i,j)=tmpp/tmpc
              tmprho(i,j)=tmpr/tmpc
           end do
        end do

!$omp end do
!$omp end parallel
     case("ob")  ! nr, 1 が境界

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,tmpp,tmpc,tmpr)

        do j=2,nz
           do i=1,nr-1
              tmpp=old_pres(i+1,j)*dz(j)-force(i,j)*dr(i)*old_pres(i,j-1)/g
              tmpc=dz(j)-dr(i)*force(i,j)/g
              tmpr=old_rho(i+1,j)*dz(j)-force(i,j)*dr(i)*old_rho(i,j-1)/g  &
  &                +(force(i,j)-force(i,j-1))*dr(i)/g
              pres(i,j)=tmpp/tmpc
              tmprho(i,j)=tmpr/tmpc
           end do
        end do

!$omp end do
!$omp end parallel
     case("it")  ! 1, nz が境界

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,tmpp,tmpc,tmpr)

        do j=1,nz-1
           do i=2,nr
              tmpp=old_pres(i-1,j)*dz(j)-force(i,j)*dr(i)*old_pres(i,j+1)/g
              tmpc=dz(j)-dr(i)*force(i,j)/g
              tmpr=old_rho(i-1,j)*dz(j)-force(i,j)*dr(i)*old_rho(i,j+1)/g  &
  &                -(force(i,j+1)-force(i,j))*dr(i)/g
              pres(i,j)=tmpp/tmpc
              tmprho(i,j)=tmpr/tmpc
           end do
        end do

!$omp end do
!$omp end parallel
     case("ib")  ! 1, 1 が境界

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,tmpp,tmpc,tmpr)

        do j=2,nz
           do i=2,nr
              tmpp=old_pres(i-1,j)*dz(j)+force(i,j)*dr(i)*old_pres(i,j-1)/g
              tmpc=dz(j)+dr(i)*force(i,j)/g
              tmpr=old_rho(i-1,j)*dz(j)+force(i,j)*dr(i)*old_rho(i,j-1)/g  &
  &                -(force(i,j)-force(i,j-1))*dr(i)/g
              pres(i,j)=tmpp/tmpc
              tmprho(i,j)=tmpr/tmpc
           end do
        end do

!$omp end do
!$omp end parallel

     end select

!-- 境界値の設定
     do i=1,nr
        pres(i,nzb)=pres_zbound(i)
        tmprho(i,nzb)=log(rho_zbound(i))
     end do
     do j=1,nz
        pres(nrb,j)=pres_s(j)
        tmprho(nrb,j)=log(rho_s(j))
     end do

!-- 密度場の収束を計算
     do j=1,nz
        do i=1,nr
!ORG           if(tmprho(i,j)==0.0)then
!ORG              err_tmp=abs(exp(old_rho(i,j))-exp(tmprho(i,j)))/abs(exp(old_rho(i,j)))
!ORG           else
!ORG              err_tmp=abs(exp(old_rho(i,j))-exp(tmprho(i,j)))/abs(exp(tmprho(i,j)))
!ORG           end if
           if(pres(i,j)==0.0)then
              err_tmp=abs(old_pres(i,j)-pres(i,j))/abs(old_pres(i,j))
           else
              err_tmp=abs(old_pres(i,j)-pres(i,j))/abs(pres(i,j))
           end if

!-- 最大誤差の更新
           if(err<=err_tmp)then
              err=err_tmp
           end if

           old_rho(i,j)=tmprho(i,j)
           old_pres(i,j)=pres(i,j)

        end do
     end do

  end do

!ORG  do j=1,nz
!ORG     do i=1,nr
!ORG        rho(i,j)=exp(tmprho(i,j))
!ORG     end do
!ORG  end do

!-- 密度は圧力の鉛直勾配から計算
  select case (bc(1:1))
  case ("i")
     nri=2
     nro=nr
     rho(1,1:nz)=rho_s(1:nz)
  case ("o")
     nri=1
     nro=nr-1
     rho(nr,1:nz)=rho_s(1:nz)
  end select

!  do i=nri,nro
!     call grad_1d( z(1:nz), pres(i,1:nz), tmprho(i,1:nz) )
!  end do

  select case (bc(2:2))
  case ("t")
     do j=1,nz-1
        do i=nri,nro
           rho(i,j)=-(1.0/g)*(pres(i,j+1)-pres(i,j))/dz(j)
        end do
     end do
     rho(1:nr,nz)=rho_zbound(1:nr)
  case ("b")
     do j=2,nz
        do i=nri,nro
           rho(i,j)=-tmprho(i,j)/g
           rho(i,j)=-(1.0/g)*(pres(i,j)-pres(i,j-1))/dz(j)
        end do
     end do
     rho(1:nr,1)=rho_zbound(1:nr)
  end select

  if(present(dl))then
     do j=1,nz
        call debug_flag_r( dl, 'typhoon_analy', 'hydro_grad_eqb (pres)',  &
  &                        pres(1,j), 'Pa' )
     end do
  end if

end subroutine hydro_grad_eqb_it_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine hydro_grad_eqb_it_d( r, z, coril, v, pres_s, rho_s,  &
  &                             pres, rho, error, bound, pres_zb, rho_zb, dl )
!  サウンディングと軸対称流から静力学・傾度風平衡場の計算.
  use Thermo_Const
  use Phys_Const
  implicit none
  double precision, intent(in) :: r(:)  ! 動径座標 [m]
  double precision, intent(in) :: z(:)  ! 鉛直座標 [m]
  double precision, intent(in) :: coril(size(r),size(z))  ! コリオリパラメータ [/s]
  double precision, intent(in) :: v(size(r),size(z))  ! 軸対称流 [m/s]
  double precision, intent(in) :: pres_s(size(z))  ! サウンディングの気圧 [Pa]
  double precision, intent(in) :: rho_s(size(z))  ! サウンディングの密度 [kg/m^3]
  double precision, intent(in), optional :: error  ! イタレーションの収束条件
                    ! default = 1.0e-5
  double precision, intent(inout) :: pres(size(r),size(z))  ! 平衡場の気圧 [Pa]
  double precision, intent(inout) :: rho(size(r),size(z))  ! 平衡場の密度 [kg/m^3]
  double precision, intent(in), optional :: pres_zb(size(r))  ! 気圧の鉛直境界条件 [Pa]
  double precision, intent(in), optional :: rho_zb(size(r))  ! 密度の鉛直境界条件 [kg/m^3]
  character(2), intent(in), optional :: bound  ! 境界条件の設定場所
                         ! 1 文字目: r -> "i" or "o"
                         ! 2 文字目: z -> "b" or "t",
                         ! デフォルト: "ot"
  integer, intent(in), optional :: dl  ! デバッグレベル
  double precision :: dr(size(r)), dz(size(z))
  double precision :: pres_zbound(size(r)), rho_zbound(size(r))
  double precision :: old_pres(size(r),size(z)), old_rho(size(r),size(z))
  double precision :: N2(size(r),size(z)), tmprho(size(r),size(z)), force(size(r),size(z))
  integer :: nr, nz
  integer :: i, j, nrb, nzb, nri, nro
  double precision :: err, err_tmp, err_max, tmpp, tmpc, tmpr
  character(2) :: bc

  nr=size(r)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, coril ),  &
  &                                     "hydro_grad_eqb_it" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, v ),  &
  &                                     "hydro_grad_eqb_it" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, pres ),  &
  &                                     "hydro_grad_eqb_it" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, rho ),  &
  &                                     "hydro_grad_eqb_it" )
     call check_array_size_dmp_message( check_array_size_1d( nz, pres_s ),  &
  &                                     "hydro_grad_eqb_it" )
     call check_array_size_dmp_message( check_array_size_1d( nz, rho_s ),  &
  &                                     "hydro_grad_eqb_it" )
  end if

  if(present(error))then
     err_max=error
  else
     err_max=1.0d-5
  end if

  if(present(bound))then
     bc=bound(1:2)
  else
     bc="ot"
  end if

  select case (bc(1:1))
  case ("i")
     nrb=1
  case ("o")
     nrb=nr
  end select

  select case (bc(2:2))
  case ("b")
     nzb=1
  case ("t")
     nzb=nz
  end select

  if(present(pres_zb))then
     pres_zbound=pres_zb
  else
     pres_zbound=pres_s(nzb)
  end if

  if(present(rho_zb))then
     rho_zbound=rho_zb
  else
     rho_zbound=rho_s(nzb)
  end if

!-- 以下で各高度において, 密度は一定であるとして傾度風平衡から気圧を計算,
!-- その値を用いて静力学平衡から密度を修正. eps 以下になるまで繰り返す.
!-- 外縁で 2 次元場とサウンディングを一致.
  do j=1,nz
     do i=1,nr
        old_pres(i,j)=pres_s(j)
     end do
  end do
!-- 密度については, 水平面一様で設定
  do j=1,nz
     do i=1,nr
        old_rho(i,j)=dlog(rho_s(j))
     end do
  end do

  select case (bc(1:1))
  case ("i")
     do i=2,nr
        dr(i)=r(i)-r(i-1)
     end do
  case ("o")
     do i=1,nr-1
        dr(i)=r(i+1)-r(i)
     end do
  end select
  select case (bc(2:2))
  case ("b")
     do j=2,nz
        dz(j)=z(j)-z(j-1)
     end do
  case ("t")
     do j=1,nz-1
        dz(j)=z(j+1)-z(j)
     end do
  end select

!-- 以下でイタレーション開始.
  err=err_max

  do while(err>=err_max)
     err=0.0d0

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j)

     do j=1,nz
        do i=1,nr
           if(r(i)/=0.0d0)then
              force(i,j)=coril(i,j)*v(i,j)+v(i,j)*v(i,j)/r(i)
           else
              force(i,j)=0.0d0
           end if
        end do
     end do

!$omp end do
!$omp end parallel

!-- 境界条件の位置に応じて反復の範囲を変更
     select case (bc(1:2))
     case("ot")  ! nr, nz が境界

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,tmpp,tmpc,tmpr)

        do j=1,nz-1
           do i=1,nr-1
              tmpp=old_pres(i+1,j)*dz(j)+force(i,j)*dr(i)*old_pres(i,j+1)/g_dp
              tmpc=dz(j)+dr(i)*force(i,j)/g_dp
              if(force(i,j+1)>force(i,j))then
                 tmpr=old_rho(i+1,j)*dz(j)+force(i,j)*dr(i)*old_rho(i,j+1)/g_dp
              else
                 tmpr=old_rho(i+1,j)*dz(j)+force(i,j)*dr(i)*old_rho(i,j+1)/g_dp  &
  &                   +(force(i,j+1)-force(i,j))*dr(i)/g_dp
              end if
              pres(i,j)=tmpp/tmpc
              tmprho(i,j)=tmpr/tmpc
           end do
        end do

!$omp end do
!$omp end parallel
     case("ob")  ! nr, 1 が境界

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,tmpp,tmpc,tmpr)

        do j=2,nz
           do i=1,nr-1
              tmpp=old_pres(i+1,j)*dz(j)-force(i,j)*dr(i)*old_pres(i,j-1)/g_dp
              tmpc=dz(j)-dr(i)*force(i,j)/g_dp
              tmpr=old_rho(i+1,j)*dz(j)-force(i,j)*dr(i)*old_rho(i,j-1)/g_dp  &
  &                +(force(i,j)-force(i,j-1))*dr(i)/g_dp
              pres(i,j)=tmpp/tmpc
              tmprho(i,j)=tmpr/tmpc
           end do
        end do

!$omp end do
!$omp end parallel
     case("it")  ! 1, nz が境界

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,tmpp,tmpc,tmpr)

        do j=1,nz-1
           do i=2,nr
              tmpp=old_pres(i-1,j)*dz(j)-force(i,j)*dr(i)*old_pres(i,j+1)/g_dp
              tmpc=dz(j)-dr(i)*force(i,j)/g_dp
              tmpr=old_rho(i-1,j)*dz(j)-force(i,j)*dr(i)*old_rho(i,j+1)/g_dp  &
  &                -(force(i,j+1)-force(i,j))*dr(i)/g_dp
              pres(i,j)=tmpp/tmpc
              tmprho(i,j)=tmpr/tmpc
           end do
        end do

!$omp end do
!$omp end parallel
     case("ib")  ! 1, 1 が境界

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,tmpp,tmpc,tmpr)

        do j=2,nz
           do i=2,nr
              tmpp=old_pres(i-1,j)*dz(j)+force(i,j)*dr(i)*old_pres(i,j-1)/g_dp
              tmpc=dz(j)+dr(i)*force(i,j)/g_dp
              tmpr=old_rho(i-1,j)*dz(j)+force(i,j)*dr(i)*old_rho(i,j-1)/g_dp  &
  &                -(force(i,j)-force(i,j-1))*dr(i)/g_dp
              pres(i,j)=tmpp/tmpc
              tmprho(i,j)=tmpr/tmpc
           end do
        end do

!$omp end do
!$omp end parallel

     end select

!-- 境界値の設定
     do i=1,nr
        pres(i,nzb)=pres_zbound(i)
        tmprho(i,nzb)=dlog(rho_zbound(i))
     end do
     do j=1,nz
        pres(nrb,j)=pres_s(j)
        tmprho(nrb,j)=dlog(rho_s(j))
     end do

!-- 密度場の収束を計算
     do j=1,nz
        do i=1,nr
!ORG           if(tmprho(i,j)==0.0d0)then
!ORG              err_tmp=dabs(exp(old_rho(i,j))-exp(tmprho(i,j)))/dabs(exp(old_rho(i,j)))
!ORG           else
!ORG              err_tmp=dabs(exp(old_rho(i,j))-exp(tmprho(i,j)))/dabs(exp(tmprho(i,j)))
!ORG           end if
           if(pres(i,j)==0.0d0)then
              err_tmp=dabs(old_pres(i,j)-pres(i,j))/dabs(old_pres(i,j))
           else
              err_tmp=dabs(old_pres(i,j)-pres(i,j))/dabs(pres(i,j))
           end if

!-- 最大誤差の更新
           if(err<=err_tmp)then
              err=err_tmp
           end if

           old_rho(i,j)=tmprho(i,j)
           old_pres(i,j)=pres(i,j)

        end do
     end do

  end do

!ORG  do j=1,nz
!ORG     do i=1,nr
!ORG        rho(i,j)=exp(tmprho(i,j))
!ORG     end do
!ORG  end do

!-- 密度は圧力の鉛直勾配から計算
  select case (bc(1:1))
  case ("i")
     nri=2
     nro=nr
     rho(1,1:nz)=rho_s(1:nz)
  case ("o")
     nri=1
     nro=nr-1
     rho(nr,1:nz)=rho_s(1:nz)
  end select

  do i=nri,nro
     call grad_1d( z(1:nz), pres(i,1:nz), tmprho(i,1:nz) )
  end do

  select case (bc(2:2))
  case ("t")
     do j=1,nz-1
        do i=nri,nro
           rho(i,j)=-tmprho(i,j)/g_dp
        end do
     end do
     rho(1:nr,nz)=rho_zbound(1:nr)
  case ("b")
     do j=2,nz
        do i=nri,nro
           rho(i,j)=-tmprho(i,j)/g_dp
        end do
     end do
     rho(1:nr,1)=rho_zbound(1:nr)
  end select

  if(present(dl))then
     do j=1,nz
        call debug_flag_d( dl, 'typhoon_analy', 'hydro_grad_eqb (pres)',  &
  &                        pres(1,j), 'Pa' )
     end do
  end if

end subroutine hydro_grad_eqb_it_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine hydro_grad_eqb_f( r, z, coril, v, pres_rb, rho_rb, pres_zb, rho_zb,  &
  &                          pres, rho, dl, bound )
!  サウンディングと軸対称流から静力学・傾度風平衡場の計算.
!  反復法を用いずに計算する.
  use Thermo_Const
  use Phys_Const
  implicit none
  real, intent(in) :: r(:)  ! 動径座標 [m]
  real, intent(in) :: z(:)  ! 鉛直座標 [m]
  real, intent(in) :: coril(size(r),size(z))  ! コリオリパラメータ [/s]
  real, intent(in) :: v(size(r),size(z))  ! 軸対称流 [m/s]
  real, intent(in) :: pres_rb(size(z))  ! 気圧の側面境界条件 [Pa]
  real, intent(in) :: rho_rb(size(z))  ! 密度の側面境界条件 [kg/m^3]
  real, intent(in) :: pres_zb(size(r))  ! 気圧の鉛直境界条件 [Pa]
  real, intent(in) :: rho_zb(size(r))  ! 密度の鉛直境界条件 [kg/m^3]
  real, intent(out) :: pres(size(r),size(z))  ! 平衡場の気圧 [Pa]
  real, intent(out) :: rho(size(r),size(z))  ! 平衡場の密度 [kg/m^3]
  integer, intent(in), optional :: dl  ! デバッグレベル
  character(2), intent(in), optional :: bound  ! 境界条件の設定場所
                         ! 1 文字目: r -> "i" or "o"
                         ! 2 文字目: z -> "b" or "t",
                         ! デフォルト: "ot"
  integer, allocatable, dimension(:,:,:) :: k, l
  integer :: nkr(2), nkz(2), nrr(2), nzz(2)
  real :: dr(size(r)), dz(size(z))
  real :: force(size(r),size(z))
  real :: a(2*(size(r)-1)*(size(z)-1),2*(size(r)-1)*(size(z)-1))
  real :: b(2*(size(r)-1)*(size(z)-1)), x(2*(size(r)-1)*(size(z)-1))
  integer :: i, j, id, nr, nz, nk, ierr
  character(2) :: bc

  nr=size(r)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, coril ),  &
  &                                     "hydro_grad_eqb" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, v ),  &
  &                                     "hydro_grad_eqb" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, pres ),  &
  &                                     "hydro_grad_eqb" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, rho ),  &
  &                                     "hydro_grad_eqb" )
     call check_array_size_dmp_message( check_array_size_1d( nz, pres_rb ),  &
  &                                     "hydro_grad_eqb" )
     call check_array_size_dmp_message( check_array_size_1d( nz, rho_rb ),  &
  &                                     "hydro_grad_eqb" )
     call check_array_size_dmp_message( check_array_size_1d( nr, pres_zb ),  &
  &                                     "hydro_grad_eqb" )
     call check_array_size_dmp_message( check_array_size_1d( nr, rho_zb ),  &
  &                                     "hydro_grad_eqb" )
  end if

  if(present(bound))then
     bc=bound
  else
     bc="ot"
  end if

  !-- 境界の範囲を格子点で指定
  !-- 未知変数は 1->nr,1->nz の範囲, 方程式はその中点で 1->nr-1,1->nz-1 の範囲
  !-- nkr,nkz: 未知変数の範囲, nrr,nzz: 境界に隣接する点を除いた方程式の範囲
  !-- r 方向の微分方程式は 1->nr-1, z 方向の微分方程式は 1->nz-1 で固定
  select case (bc(1:1))
  case ("i")
     nkr=(/2,nr/)
     nrr=(/2,nr-1/)  ! r 方向の微分方程式について (半格子ずれ)
  case ("o")
     nkr=(/1,nr-1/)
     nrr=(/1,nr-2/)
  end select

  select case (bc(2:2))
  case ("b")
     nkz=(/2,nz/)
     nzz=(/2,nz-1/)  ! z 方向の微分方程式 (半格子ずれ)
  case ("t")
     nkz=(/1,nz-1/)
     nzz=(/1,nz-2/)
  end select

  allocate(k(nr,nz,2),stat=ierr)  ! 一部利用しない要素番号あり
  allocate(l(nr,nz,2),stat=ierr)  ! 一部利用しない要素番号あり

  k=0
  l=0

  ! k,l で pres/rho (2 次元配列) を 1 次元にマップ
  nk=0
  !-- z 方向の微分方程式について
  do j=1,nz-1
     do i=nkr(1),nkr(2)
        nk=nk+1
        k(i,j,1)=nk
     end do
  end do
  !-- r 方向の微分方程式について
  do j=nkz(1),nkz(2)
     do i=1,nr-1
        nk=nk+1
        k(i,j,2)=nk
     end do
  end do

  nk=0
  do id=1,2
     do j=nkz(1),nkz(2)
        do i=nkr(1),nkr(2)
           nk=nk+1
           l(i,j,id)=nk
        end do
     end do
  end do

  do i=1,nr-1
     dr(i)=r(i+1)-r(i)
  end do
  do j=1,nz-1
     dz(j)=z(j+1)-z(j)
  end do

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j)

  do j=1,nz
     do i=1,nr
        if(r(i)/=0.0)then
           force(i,j)=coril(i,j)*v(i,j)+v(i,j)*v(i,j)/r(i)
        else
           force(i,j)=0.0
        end if
     end do
  end do

!$omp end do
!$omp end parallel

  a=0.0  ! a の 1 要素目は行列の行 (解く方程式の格子点番号)
         ! a の 2 要素目は行列の列 (解く変数の番号)
         ! a(k(i,j),l(i,j)): 方程式の点 (i,j) での点 (i,j) での変数に対応する係数
  b=0.0  ! b の要素は解く方程式の格子点番号

  !-- 1. 境界に隣接した格子点での方程式以外
  !-- 1.1 z 方向の微分 (z 方向の範囲は境界隣接含めて 1->nz-1) = k(:,:,1) の範囲
  do j=nzz(1),nzz(2)  ! 境界隣接点を除いている
     do i=nkr(1),nkr(2)
        !-- 変数 p (= l(:,:,1)) について
        a(k(i,j,1),l(i,j,1))=-1.0/dz(j)
        a(k(i,j,1),l(i,j+1,1))=1.0/dz(j)
        !-- 変数 rho (= l(:,:,2)) について
        a(k(i,j,1),l(i,j,2))=0.5*g
        a(k(i,j,1),l(i,j+1,2))=0.5*g
     end do
  end do

  !-- 1.2 r 方向の微分 (r 方向の範囲は境界隣接含めて 1->nr-1) = k(:,:,2) の範囲
  do j=nkz(1),nkz(2)
     do i=nrr(1),nrr(2)  ! 境界隣接点を除いている
        !-- 変数 p (= l(:,:,1)) について
        a(k(i,j,2),l(i,j,1))=-1.0/dr(i)
        a(k(i,j,2),l(i+1,j,1))=1.0/dr(i)
        !-- 変数 rho (= l(:,:,2)) について
        a(k(i,j,2),l(i,j,2))=0.5*(force(i+1,j)+force(i,j))
        a(k(i,j,2),l(i+1,j,2))=a(k(i,j,2),l(i,j,2))
        !-- その他
        b(k(i,j,2))=0.5*(force(i+1,j)+force(i,j))*rho_rb(j)
     end do
  end do

  !-- 2. 境界に隣接した格子点での方程式
  !-- 2.1 z 微分方程式 (z 方向境界, k(:,:,1))
  select case (bc(2:2))
  case ("b")
     do i=nkr(1),nkr(2)
        a(k(i,1,1),l(i,2,1))=1.0/dz(1)
        a(k(i,1,1),l(i,2,2))=0.5*g
        ! z の境界条件として与えられている p, rho
        ! は静力学からの偏差が引かれていないので, ここで引いて b に入れる.
        b(k(i,1,1))=(pres_zb(i)-pres_rb(1))/dz(1)-0.5*g*(rho_zb(i)-rho_rb(1))
     end do
  case ("t")
     do i=nkr(1),nkr(2)
        a(k(i,nz-1,1),l(i,nz-1,1))=-1.0/dz(nz-1)
        a(k(i,nz-1,1),l(i,nz-1,2))=0.5*g
        ! z の境界条件として与えられている p, rho
        ! は静力学からの偏差が引かれていないので, ここで引いて b に入れる.
        b(k(i,nz-1,1))=-(pres_zb(i)-pres_rb(nz))/dz(nz-1)-0.5*g*(rho_zb(i)-rho_rb(nz))
     end do
  case default
     write(*,*) "*** ERROR (typhoon_analy:hydro_grad_eqb) ***: bc is invalid"
     stop
  end select

  !-- 2.2 r 微分方程式 (r 方向境界, k(:,:,2))
  select case (bc(1:1))
  case ("i")
     do j=nkz(1),nkz(2)
        a(k(1,j,2),l(2,j,1))=1.0/dr(1)
        a(k(1,j,2),l(2,j,2))=-0.5*(force(2,j)+force(1,j))
        b(k(1,j,2))=0.5*(force(2,j)+force(1,j))*rho_rb(j)
     end do
  case ("o")
     do j=nkz(1),nkz(2)
        a(k(nr-1,j,2),l(nr-1,j,1))=-1.0/dr(nr-1)
        a(k(nr-1,j,2),l(nr-1,j,2))=0.5*(force(nr,j)+force(nr-1,j))
        b(k(nr-1,j,2))=0.5*(force(nr,j)+force(nr-1,j))*rho_rb(j)
     end do
  case default
     write(*,*) "*** ERROR (typhoon_analy:hydro_grad_eqb) ***: bc is invalid"
     stop
  end select

  !-- ガウスの掃き出し法
  call fp_gauss( a, b, x )

  !-- 解いた x を出力変数に割り当てる
  !-- p: l(:,:,1), rho: l(:,:,2)
  do j=nkz(1),nkz(2)
     do i=nkr(1),nkr(2)
        ! 静力学平衡分を足し合わせる.
        pres(i,j)=x(l(i,j,1))!+pres_rb(j)
        rho(i,j)=x(l(i,j,2))!+rho_rb(j)
     end do
  end do

  select case (bc(1:1))
  case ("i")
     pres(1,1:nz)=pres_rb(1:nz)
     rho(1,1:nz)=rho_rb(1:nz)
  case ("o")
     pres(nr,1:nz)=pres_rb(1:nz)
     rho(nr,1:nz)=rho_rb(1:nz)
  end select

  select case (bc(2:2))
  case ("b")
     pres(1:nr,1)=pres_zb(1:nr)
     rho(1:nr,1)=rho_zb(1:nr)
  case ("t")
     pres(1:nr,nz)=pres_zb(1:nr)
     rho(1:nr,nz)=rho_zb(1:nr)
  end select

  if(present(dl))then
     do j=1,nz
        call debug_flag_r( dl, 'typhoon_analy', 'hydro_grad_eqb (pres)',  &
  &                        pres(1,j), 'Pa' )
     end do
  end if

end subroutine hydro_grad_eqb_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine hydro_grad_eqbp_f( r, p, coril, v, zph_s, temp_s, zph, temp, error, dl )
!  サウンディングと軸対称流から静力学・傾度風平衡場の計算.
!  鉛直座標が圧力の場合の計算
  use Thermo_Const
  use Phys_Const
  implicit none
  real, intent(in) :: r(:)  ! 動径座標 [m]
  real, intent(in) :: p(:)  ! 鉛直座標 [Pa]
  real, intent(in) :: coril(size(r),size(p))  ! コリオリパラメータ [/s]
  real, intent(in) :: v(size(r),size(p))  ! 軸対称流 [m/s]
  real, intent(in) :: zph_s(size(p))  ! サウンディングのジオポテンシャル高度 [m]
  real, intent(in) :: temp_s(size(p))  ! サウンディングの温度 [K]
  real, intent(in), optional :: error  ! イタレーションの収束条件
                    ! default = 1.0e-5
  real, intent(inout) :: zph(size(r),size(p))  ! 平衡場のジオポテンシャル高度 [m]
  real, intent(inout) :: temp(size(r),size(p))  ! 平衡場の温度 [K]
  integer, intent(in), optional :: dl  ! デバッグレベル
  real :: dr(size(r)), dp(size(p))
  real :: old_zph(size(r),size(p)), old_temp(size(r),size(p))
  real :: tmptemp(size(r),size(p)), force(size(r),size(p))
  integer :: nr, nz
  integer :: i, j
  real :: err, err_tmp, err_max, tmpp, tmpc, tmpr

  nr=size(r)
  nz=size(p)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, coril ),  &
  &                                     "hydro_grad_eqbp" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, v ),  &
  &                                     "hydro_grad_eqbp" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, zph ),  &
  &                                     "hydro_grad_eqbp" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, temp ),  &
  &                                     "hydro_grad_eqbp" )
     call check_array_size_dmp_message( check_array_size_1d( nz, zph_s ),  &
  &                                     "hydro_grad_eqbp" )
     call check_array_size_dmp_message( check_array_size_1d( nz, temp_s ),  &
  &                                     "hydro_grad_eqbp" )
  end if

  if(present(error))then
     err_max=error
  else
     err_max=1.0e-5
  end if

!-- 以下で各高度において,
!-- 温度は一定であるとして傾度風平衡からジオポテンシャル高度を計算,
!-- その値を用いて静力学平衡から温度を修正. eps 以下になるまで繰り返す.
!-- 外縁で 2 次元場とサウンディングを一致.
  do j=1,nz
     do i=1,nr
        old_zph(i,j)=zph_s(j)
     end do
  end do
!-- 温度については, 水平面一様で設定
  do j=1,nz
     do i=1,nr
        old_temp(i,j)=temp_s(j)
     end do
  end do

  do i=1,nr-1
     dr(i)=r(i+1)-r(i)
  end do
  do j=1,nz-1
     dp(j)=log(p(j+1)/p(j))
  end do

!-- 以下でイタレーション開始.
  err=err_max

  do while(err>=err_max)
     err=0.0

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j)

     do j=1,nz
        do i=1,nr
           if(r(i)/=0.0)then
              force(i,j)=coril(i,j)*v(i,j)+v(i,j)*v(i,j)/r(i)
           else
              force(i,j)=0.0
           end if
        end do
     end do

!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,tmpc,tmpr)

     do j=1,nz-1
        do i=1,nr-1
           zph(i,j)=old_zph(i+1,j)-force(i,j)*dr(i)/g
           tmptemp(i,j)=old_temp(i+1,j)+(force(i,j+1)-force(i,j))*dr(i)/(dp(j)*Rd)
        end do
     end do

!$omp end do
!$omp end parallel

!-- 境界値の設定
     do i=1,nr
        zph(i,nz)=zph_s(nz)
        tmptemp(i,nz)=temp_s(nz)
     end do
     do j=1,nz
        zph(nr,j)=zph_s(j)
        tmptemp(nr,j)=temp_s(j)
     end do

!-- 密度場の収束を計算
     do j=1,nz
        do i=1,nr
           if(tmptemp(i,j)==0.0)then
              err_tmp=abs(old_temp(i,j)-tmptemp(i,j))/abs(old_temp(i,j))
           else
              err_tmp=abs(old_temp(i,j)-tmptemp(i,j))/abs(tmptemp(i,j))
           end if

!-- 最大誤差の更新
           if(err<=err_tmp)then
              err=err_tmp
           end if

           old_temp(i,j)=tmptemp(i,j)
           old_zph(i,j)=zph(i,j)

        end do
     end do

  end do

  do j=1,nz
     do i=1,nr
        temp(i,j)=tmptemp(i,j)
     end do
  end do

  if(present(dl))then
     do j=1,nz
        call debug_flag_r( dl, 'typhoon_analy', 'hydro_grad_eqbp (zph)',  &
  &                        zph(1,j), 'Pa' )
     end do
  end if

end subroutine hydro_grad_eqbp_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine hydro_grad_eqbp_d( r, p, coril, v, zph_s, temp_s, zph, temp, error, dl )
!  サウンディングと軸対称流から静力学・傾度風平衡場の計算.
!  鉛直座標が圧力の場合の計算
  use Thermo_Const
  use Phys_Const
  implicit none
  double precision, intent(in) :: r(:)  ! 動径座標 [m]
  double precision, intent(in) :: p(:)  ! 鉛直座標 [Pa]
  double precision, intent(in) :: coril(size(r),size(p))  ! コリオリパラメータ [/s]
  double precision, intent(in) :: v(size(r),size(p))  ! 軸対称流 [m/s]
  double precision, intent(in) :: zph_s(size(p))  ! サウンディングのジオポテンシャル高度 [m]
  double precision, intent(in) :: temp_s(size(p))  ! サウンディングの温度 [K]
  double precision, intent(in), optional :: error  ! イタレーションの収束条件
                    ! default = 1.0e-5
  double precision, intent(inout) :: zph(size(r),size(p))  ! 平衡場のジオポテンシャル高度 [m]
  double precision, intent(inout) :: temp(size(r),size(p))  ! 平衡場の温度 [K]
  integer, intent(in), optional :: dl  ! デバッグレベル
  double precision :: dr(size(r)), dp(size(p))
  double precision :: old_zph(size(r),size(p)), old_temp(size(r),size(p))
  double precision :: tmptemp(size(r),size(p)), force(size(r),size(p))
  integer :: nr, nz
  integer :: i, j
  double precision :: err, err_tmp, err_max, tmpp, tmpc, tmpr

  nr=size(r)
  nz=size(p)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, coril ),  &
  &                                     "hydro_grad_eqbp" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, v ),  &
  &                                     "hydro_grad_eqbp" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, zph ),  &
  &                                     "hydro_grad_eqbp" )
     call check_array_size_dmp_message( check_array_size_2d( nr, nz, temp ),  &
  &                                     "hydro_grad_eqbp" )
     call check_array_size_dmp_message( check_array_size_1d( nz, zph_s ),  &
  &                                     "hydro_grad_eqbp" )
     call check_array_size_dmp_message( check_array_size_1d( nz, temp_s ),  &
  &                                     "hydro_grad_eqbp" )
  end if

  if(present(error))then
     err_max=error
  else
     err_max=1.0d-5
  end if

!-- 以下で各高度において,
!-- 温度は一定であるとして傾度風平衡からジオポテンシャル高度を計算,
!-- その値を用いて静力学平衡から温度を修正. eps 以下になるまで繰り返す.
!-- 外縁で 2 次元場とサウンディングを一致.
  do j=1,nz
     do i=1,nr
        old_zph(i,j)=zph_s(j)
     end do
  end do
!-- 温度については, 水平面一様で設定
  do j=1,nz
     do i=1,nr
        old_temp(i,j)=temp_s(j)
     end do
  end do

  do i=1,nr-1
     dr(i)=r(i+1)-r(i)
  end do
  do j=1,nz-1
     dp(j)=dlog(p(j+1)/p(j))
  end do

!-- 以下でイタレーション開始.
  err=err_max

  do while(err>=err_max)
     err=0.0d0

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j)

     do j=1,nz
        do i=1,nr
           if(r(i)/=0.0d0)then
              force(i,j)=coril(i,j)*v(i,j)+v(i,j)*v(i,j)/r(i)
           else
              force(i,j)=0.0d0
           end if
        end do
     end do

!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,tmpc,tmpr)

     do j=1,nz-1
        do i=1,nr-1
           zph(i,j)=old_zph(i+1,j)-force(i,j)*dr(i)/g_dp
           tmptemp(i,j)=old_temp(i+1,j)+(force(i,j+1)-force(i,j))*dr(i)/(dp(j)*Rd_dp)
        end do
     end do

!$omp end do
!$omp end parallel

!-- 境界値の設定
     do i=1,nr
        zph(i,nz)=zph_s(nz)
        tmptemp(i,nz)=temp_s(nz)
     end do
     do j=1,nz
        zph(nr,j)=zph_s(j)
        tmptemp(nr,j)=temp_s(j)
     end do

!-- 密度場の収束を計算
     do j=1,nz
        do i=1,nr
           if(tmptemp(i,j)==0.0d0)then
              err_tmp=dabs(old_temp(i,j)-tmptemp(i,j))/dabs(old_temp(i,j))
           else
              err_tmp=dabs(old_temp(i,j)-tmptemp(i,j))/dabs(tmptemp(i,j))
           end if

!-- 最大誤差の更新
           if(err<=err_tmp)then
              err=err_tmp
           end if

           old_temp(i,j)=tmptemp(i,j)
           old_zph(i,j)=zph(i,j)

        end do
     end do

  end do

  do j=1,nz
     do i=1,nr
        temp(i,j)=tmptemp(i,j)
     end do
  end do

  if(present(dl))then
     do j=1,nz
        call debug_flag_d( dl, 'typhoon_analy', 'hydro_grad_eqbp (zph)',  &
  &                        zph(1,j), 'Pa' )
     end do
  end if

end subroutine hydro_grad_eqbp_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine grad_wind_pres_f( r, coril, v, rho, r_ref, p_ref, pres )
!  傾度風平衡場を満たす気圧場を計算する.
  implicit none
  real, intent(in) :: r(:)  ! r 方向の位置座標 [m]
  real, intent(in) :: coril(size(r))  ! コリオリパラメータ [/s]
  real, intent(in) :: v(size(r))  ! r 方向の位置座標 [m]
  real, intent(in) :: rho(size(r))  ! 密度 [kg/m^3]
  real, intent(in) :: r_ref  ! 積分定数となる位置座標 [m]
  real, intent(in) :: p_ref  ! r_ref での気圧 (積分定数) [Pa]
  real, intent(inout) :: pres(size(r))  ! 傾度風平衡での気圧 [Pa]
  integer :: i, nr
  real :: grad(size(r))

  nr=size(r)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nr, coril ),  &
  &                                     "grad_wind_pres" )
     call check_array_size_dmp_message( check_array_size_1d( nr, v ),  &
  &                                     "grad_wind_pres" )
     call check_array_size_dmp_message( check_array_size_1d( nr, rho ),  &
  &                                     "grad_wind_pres" )
     call check_array_size_dmp_message( check_array_size_1d( nr, pres ),  &
  &                                     "grad_wind_pres" )
  end if

  do i=1,nr
     if(r(i)/=0.0)then
        grad(i)=rho(i)*(v(i)*v(i)/r(i)+coril(i)*v(i))
     else
        grad(i)=0.0
     end if
  end do

  do i=1,nr
     if(r(i)<r_ref)then
        call rectangle_int( r, grad, r(i), r_ref, pres(i) )
        pres(i)=p_ref-pres(i)
     else if(r(i)>r_ref)then
        call rectangle_int( r, grad, r_ref, r(i), pres(i) )
        pres(i)=p_ref+pres(i)
     else
        pres(i)=p_ref
     end if
  end do

end subroutine grad_wind_pres_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine grad_wind_pres_d( r, coril, v, rho, r_ref, p_ref, pres )
!  傾度風平衡場を満たす気圧場を計算する.
  implicit none
  double precision, intent(in) :: r(:)  ! r 方向の位置座標 [m]
  double precision, intent(in) :: coril(size(r))  ! コリオリパラメータ [/s]
  double precision, intent(in) :: v(size(r))  ! r 方向の位置座標 [m]
  double precision, intent(in) :: rho(size(r))  ! 密度 [kg/m^3]
  double precision, intent(in) :: r_ref  ! 積分定数となる位置座標 [m]
  double precision, intent(in) :: p_ref  ! r_ref での気圧 (積分定数) [Pa]
  double precision, intent(inout) :: pres(size(r))  ! 傾度風平衡での気圧 [Pa]
  integer :: i, nr
  double precision :: grad(size(r))

  nr=size(r)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nr, coril ),  &
  &                                     "grad_wind_pres" )
     call check_array_size_dmp_message( check_array_size_1d( nr, v ),  &
  &                                     "grad_wind_pres" )
     call check_array_size_dmp_message( check_array_size_1d( nr, rho ),  &
  &                                     "grad_wind_pres" )
     call check_array_size_dmp_message( check_array_size_1d( nr, pres ),  &
  &                                     "grad_wind_pres" )
  end if

  do i=1,nr
     if(r(i)/=0.0d0)then
        grad(i)=rho(i)*(v(i)*v(i)/r(i)+coril(i)*v(i))
     else
        grad(i)=0.0d0
     end if
  end do

  do i=1,nr
     if(r(i)<r_ref)then
        call rectangle_int( r, grad, r(i), r_ref, pres(i) )
        pres(i)=p_ref-pres(i)
     else if(r(i)>r_ref)then
        call rectangle_int( r, grad, r_ref, r(i), pres(i) )
        pres(i)=p_ref+pres(i)
     else
        pres(i)=p_ref
     end if
  end do

end subroutine grad_wind_pres_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine pres_grad_wind_f( r, coril, pres, rho, v )
!  傾度風を計算する.
  implicit none
  real, intent(in) :: r(:)  ! r 方向の位置座標 [m]
  real, intent(in) :: coril(size(r))  ! コリオリパラメータ [/s]
  real, intent(in) :: pres(size(r))  ! 傾度風平衡での気圧 [Pa]
  real, intent(in) :: rho(size(r))  ! 密度 [kg/m^3]
  real, intent(inout) :: v(size(r))  ! r 方向の位置座標 [m]
  integer :: i, nr
  real :: grad(size(r)), tmp(size(r))

  nr=size(r)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nr, coril ),  &
  &                                     "pres_grad_wind" )
     call check_array_size_dmp_message( check_array_size_1d( nr, v ),  &
  &                                     "pres_grad_wind" )
     call check_array_size_dmp_message( check_array_size_1d( nr, rho ),  &
  &                                     "pres_grad_wind" )
     call check_array_size_dmp_message( check_array_size_1d( nr, pres ),  &
  &                                     "pres_grad_wind" )
  end if

  call grad_1d( r, pres, grad )

  do i=1,nr
     if(r(i)/=0.0)then
        tmp(i)=4.0*grad(i)/(rho(i)*coril(i)*coril(i)*r(i))
        v(i)=0.5*coril(i)*r(i)*(-1.0+sqrt(1.0+tmp(i)))
     else
        v(i)=0.0
     end if
  end do

end subroutine pres_grad_wind_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine pres_grad_wind_d( r, coril, pres, rho, v )
!  傾度風を計算する.
  implicit none
  double precision, intent(in) :: r(:)  ! r 方向の位置座標 [m]
  double precision, intent(in) :: coril(size(r))  ! コリオリパラメータ [/s]
  double precision, intent(in) :: pres(size(r))  ! 傾度風平衡での気圧 [Pa]
  double precision, intent(in) :: rho(size(r))  ! 密度 [kg/m^3]
  double precision, intent(inout) :: v(size(r))  ! r 方向の位置座標 [m]
  integer :: i, nr
  double precision :: grad(size(r)), tmp(size(r))

  nr=size(r)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nr, coril ),  &
  &                                     "pres_grad_wind" )
     call check_array_size_dmp_message( check_array_size_1d( nr, v ),  &
  &                                     "pres_grad_wind" )
     call check_array_size_dmp_message( check_array_size_1d( nr, rho ),  &
  &                                     "pres_grad_wind" )
     call check_array_size_dmp_message( check_array_size_1d( nr, pres ),  &
  &                                     "pres_grad_wind" )
  end if

  call grad_1d( r, pres, grad )

  do i=1,nr
     if(r(i)/=0.0d0)then
        tmp(i)=4.0d0*grad(i)/(rho(i)*coril(i)*coril(i)*r(i))
        v(i)=0.5d0*coril(i)*r(i)*(-1.0d0+dsqrt(1.0d0+tmp(i)))
     else
        v(i)=0.0d0
     end if
  end do

end subroutine pres_grad_wind_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine SPLB_Kurihara_f( axopt, phi0, x, y, Lx, Ly, ival, oval,  &
  &                         optm, optnx, optny )
! Kurihara etal. (1990) における場の物理量に含まれる任意の波数成分を
! フィルタアウトするルーチン.
! ただし, optn を与えると, Nguyen and Chen (2011) による高解像度
! データ対応の修正版で処理される.
! 用いる式は Nguyen and Chen (2011) の (3), (4), (5) 式.
! n=1 の場合, Kurihara etal. (1990) の手法に一致する.

  use Math_Const
  use Phys_Const

  implicit none

  character(2), intent(in) :: axopt     ! 計算する座標 'xy' or 'll'
                                        ! dx, dy の単位に影響する.
  real, intent(in) :: phi0     ! 基準緯度 [rad]
  real, intent(in) :: x(:)     ! ival 第一要素の格子点 [m] or [rad]
  real, intent(in) :: y(:)     ! ival 第二要素の格子点 [m] or [rad]
  real, intent(in) :: Lx(2)    ! フィルタする x 領域 [m] or [rad]
  real, intent(in) :: Ly(2)    ! フィルタする y 領域 [m] or [rad]
  real, intent(in) :: ival(size(x),size(y))    ! フィルタアウトする場の変数
  real, intent(inout) :: oval(size(x),size(y)) ! フィルタアウトされた場の変数
  integer, intent(in), optional :: optm(:)  ! フィルタアウトする波数
                                  ! デフォルトは 8 波数, 11 要素
                                  ! 2,3,4,2,5,6,7,2,8,9,2
  integer, intent(in), optional :: optnx(:)  ! 高解像度版の格子点
  integer, intent(in), optional :: optny(:)  ! 高解像度版の格子点

  integer :: nx, ny, m, n, nnx, nny
  integer :: i, j, k, l
  integer, dimension(2) :: ilx, ily
  integer, allocatable, dimension(:) :: qnx, qny
  real :: dx, dy
  real, allocatable, dimension(:) :: Km
  real :: tmpval(size(x),size(y))

  nx=size(x)
  ny=size(y)

!-- dx, dy を m 単位で設定

  if(axopt(1:2)=='xy')then
     dx=x(2)-x(1)
     dy=y(2)-y(1)
  else if(axopt(1:2)=='ll')then
     dx=radius*cos(phi0)*(x(2)-x(1))
     dy=radius*(y(2)-y(1))
  else
     write(*,*) "*** ERROR (SPLB_Kurihara) *** : axopt is invalid."
     stop
  end if

write(*,*) "dx", dx, dy
!-- フィルター係数の計算

  if(present(optm))then
     m=size(optm)
     allocate(Km(m))
     do i=1,m
        Km(i)=0.5/(1.0-cos(2.0*pi/real(optm(i))))
     end do
  else
     allocate(Km(11))
     Km=(/0.25,                        &  ! m=2
  &       1.0/3.0,                     &  ! m=3
  &       0.5,                         &  ! m=4
  &       0.25,                        &  ! m=2
  &       0.5/(1.0-cos(0.4*pi)),       &  ! m=5
  &       1.0,                         &  ! m=6
  &       0.5/(1.0-cos(2.0*pi/7.0)),   &  ! m=7
  &       0.25,                        &  ! m=2
  &       0.5/(1.0-cos(0.25*pi)),      &  ! m=8
  &       0.5/(1.0-cos(2.0*pi/9.0)),   &  ! m=9
  &       0.25/)                          ! m=2
     m=11
  end if

write(*,*) "Km", Km
!-- 高解像度版での 3 点格子の位置を計算.

  if(present(optnx))then
     nnx=size(optnx)
  else   ! Original filterring according to Kurihara et al. (1990)
     nnx=1
  end if

  if(present(optny))then
     nny=size(optny)
  else   ! Original filterring according to Kurihara et al. (1990)
     nny=1
  end if

  allocate(qnx(nnx))
  allocate(qny(nny))


  if(nnx>1)then
     do i=1,nnx
        qnx(i)=int(radius*(pi/180.0)*cos(phi0)/(real(optnx(i))*dx))
     end do
  else
     qnx(1)=int(radius*(pi/180.0)*cos(phi0)/dx)
  end if

  if(qnx(1)<1)then
     qnx(1)=1
  end if

  if(nny>1)then
     do i=1,nny
        qny(i)=int(radius*(pi/180.0)/(real(optny(i))*dy))
     end do
  else
     qny(1)=int(radius*(pi/180.0)/dy)
  end if

  if(qny(1)<1)then
     qny(1)=1
  end if

write(*,*) "nnx", nnx, nny, qnx, qny
!-- フィルタリング領域の格子点番号を検索

  call interpo_search_1d( x, Lx(1), ilx(1), stdopt=.true. )
  call interpo_search_1d( x, Lx(2), ilx(2), stdopt=.true. )
  call interpo_search_1d( y, Ly(1), ily(1), stdopt=.true. )
  call interpo_search_1d( y, Ly(2), ily(2), stdopt=.true. )

  if(ilx(1)-qnx(1)<1)then
     write(*,*) "*** ERROR (SPLB_Kurihara) *** : Lx(1) is out of range."
     write(*,*) "Over number = ", ilx(1)-qnx(1)
     stop
  end if
  if(ilx(2)+qnx(1)>nx)then
     write(*,*) "*** ERROR (SPLB_Kurihara) *** : Lx(2) is out of range."
     write(*,*) "Over number = ", ilx(2)+qnx(1)
     stop
  end if
  if(ily(1)-qny(1)<1)then
     write(*,*) "*** ERROR (SPLB_Kurihara) *** : Ly(1) is out of range."
     write(*,*) "Over number = ", ily(1)-qny(1)
     stop
  end if
  if(ily(2)+qny(1)>ny)then
     write(*,*) "*** ERROR (SPLB_Kurihara) *** : Ly(2) is out of range."
     write(*,*) "Over number = ", ily(2)+qny(1)
     stop
  end if

write(*,*) "ilx", ilx, ily, Lx, Ly, x(ilx(1)), x(ilx(2)), y(ily(1)), y(ily(2))
  tmpval=ival
  oval=ival

!-- 配列第一成分フィルター開始

  do l=1,nnx
     do k=1,m

        do j=ily(1),ily(2)
           do i=ilx(1),ilx(2)
              oval(i,j)=tmpval(i,j)  &
  &                    +Km(k)*(tmpval(i-qnx(l),j)+tmpval(i+qnx(l),j)  &
  &                            -2.0*tmpval(i,j))
           end do
        end do

        tmpval=oval

     end do
  end do

!-- 配列第二成分フィルター開始

  do l=1,nny
     do k=1,m

        do j=ily(1),ily(2)
           do i=ilx(1),ilx(2)
              oval(i,j)=tmpval(i,j)  &
  &                    +Km(k)*(tmpval(i,j-qny(l))+tmpval(i,j+qny(l))  &
  &                            -2.0*tmpval(i,j))
           end do
        end do

        tmpval=oval

     end do
  end do

end subroutine SPLB_Kurihara_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine SPLB_Kurihara_d( axopt, phi0, x, y, Lx, Ly, ival, oval,  &
  &                         optm, optnx, optny )
! Kurihara etal. (1990) における場の物理量に含まれる任意の波数成分を
! フィルタアウトするルーチン.
! ただし, optn を与えると, Nguyen and Chen (2011) による高解像度
! データ対応の修正版で処理される.
! 用いる式は Nguyen and Chen (2011) の (3), (4), (5) 式.
! n=1 の場合, Kurihara etal. (1990) の手法に一致する.

  use Math_Const
  use Phys_Const

  implicit none

  character(2), intent(in) :: axopt     ! 計算する座標 'xy' or 'll'
                                        ! dx, dy の単位に影響する.
  double precision, intent(in) :: phi0     ! 基準緯度 [rad]
  double precision, intent(in) :: x(:)     ! ival 第一要素の格子点 [m] or [rad]
  double precision, intent(in) :: y(:)     ! ival 第二要素の格子点 [m] or [rad]
  double precision, intent(in) :: Lx(2)    ! フィルタする x 領域 [m] or [rad]
  double precision, intent(in) :: Ly(2)    ! フィルタする y 領域 [m] or [rad]
  double precision, intent(in) :: ival(size(x),size(y))    ! フィルタアウトする場の変数
  double precision, intent(inout) :: oval(size(x),size(y)) ! フィルタアウトされた場の変数
  integer, intent(in), optional :: optm(:)  ! フィルタアウトする波数
                                  ! デフォルトは 8 波数, 11 要素
                                  ! 2,3,4,2,5,6,7,2,8,9,2
  integer, intent(in), optional :: optnx(:)  ! 高解像度版の格子点
  integer, intent(in), optional :: optny(:)  ! 高解像度版の格子点

  integer :: nx, ny, m, n, nnx, nny
  integer :: i, j, k, l
  integer, dimension(2) :: ilx, ily
  integer, allocatable, dimension(:) :: qnx, qny
  double precision :: dx, dy
  double precision, allocatable, dimension(:) :: Km
  double precision :: tmpval(size(x),size(y))

  nx=size(x)
  ny=size(y)

!-- dx, dy を m 単位で設定

  if(axopt(1:2)=='xy')then
     dx=x(2)-x(1)
     dy=y(2)-y(1)
  else if(axopt(1:2)=='ll')then
     dx=radius_dp*dcos(phi0)*(x(2)-x(1))
     dy=radius_dp*(y(2)-y(1))
  else
     write(*,*) "*** ERROR (SPLB_Kurihara) *** : axopt is invalid."
     stop
  end if

write(*,*) "dx", dx, dy
!-- フィルター係数の計算

  if(present(optm))then
     m=size(optm)
     allocate(Km(m))
     do i=1,m
        Km(i)=0.5d0/(1.0d0-dcos(2.0d0*pi_dp/dble(optm(i))))
     end do
  else
     allocate(Km(11))
     Km=(/0.25d0,                           &  ! m=2
  &       1.0d0/3.0d0,                      &  ! m=3
  &       0.5d0,                            &  ! m=4
  &       0.25d0,                           &  ! m=2
  &       0.5d0/(1.0d0-dcos(0.4d0*pi_dp)),  &  ! m=5
  &       1.0d0,                            &  ! m=6
  &       0.5d0/(1.0d0-dcos(2.0d0*pi_dp/7.0d0)),  &  ! m=7
  &       0.25d0,                           &  ! m=2
  &       0.5d0/(1.0d0-dcos(0.25d0*pi_dp)), &  ! m=8
  &       0.5d0/(1.0d0-dcos(2.0d0*pi_dp/9.0d0)),  &  ! m=9
  &       0.25d0/)                             ! m=2
     m=11
  end if

write(*,*) "Km", Km
!-- 高解像度版での 3 点格子の位置を計算.

  if(present(optnx))then
     nnx=size(optnx)
  else   ! Original filterring according to Kurihara et al. (1990)
     nnx=1
  end if

  if(present(optny))then
     nny=size(optny)
  else   ! Original filterring according to Kurihara et al. (1990)
     nny=1
  end if

  allocate(qnx(nnx))
  allocate(qny(nny))


  if(nnx>1)then
     do i=1,nnx
        qnx(i)=int(radius_dp*(pi_dp/180.0d0)*dcos(phi0)/(dble(optnx(i))*dx))
     end do
  else
     qnx(1)=int(radius_dp*(pi_dp/180.0d0)*dcos(phi0)/dx)
  end if

  if(qnx(1)<1)then
     qnx(1)=1
  end if

  if(nny>1)then
     do i=1,nny
        qny(i)=int(radius_dp*(pi/180.0d0)/(dble(optny(i))*dy))
     end do
  else
     qny(1)=int(radius_dp*(pi_dp/180.0d0)/dy)
  end if

  if(qny(1)<1)then
     qny(1)=1
  end if

write(*,*) "nnx", nnx, nny, qnx, qny
!-- フィルタリング領域の格子点番号を検索

  call interpo_search_1d( x, Lx(1), ilx(1), stdopt=.true. )
  call interpo_search_1d( x, Lx(2), ilx(2), stdopt=.true. )
  call interpo_search_1d( y, Ly(1), ily(1), stdopt=.true. )
  call interpo_search_1d( y, Ly(2), ily(2), stdopt=.true. )

  if(ilx(1)-qnx(1)<1)then
     write(*,*) "*** ERROR (SPLB_Kurihara) *** : Lx(1) is out of range."
     write(*,*) "Over number = ", ilx(1)-qnx(1)
     stop
  end if
  if(ilx(2)+qnx(1)>nx)then
     write(*,*) "*** ERROR (SPLB_Kurihara) *** : Lx(2) is out of range."
     write(*,*) "Over number = ", ilx(2)+qnx(1)
     stop
  end if
  if(ily(1)-qny(1)<1)then
     write(*,*) "*** ERROR (SPLB_Kurihara) *** : Ly(1) is out of range."
     write(*,*) "Over number = ", ily(1)-qny(1)
     stop
  end if
  if(ily(2)+qny(1)>ny)then
     write(*,*) "*** ERROR (SPLB_Kurihara) *** : Ly(2) is out of range."
     write(*,*) "Over number = ", ily(2)+qny(1)
     stop
  end if

write(*,*) "ilx", ilx, ily, Lx, Ly, x(ilx(1)), x(ilx(2)), y(ily(1)), y(ily(2))
  tmpval=ival
  oval=ival

!-- 配列第一成分フィルター開始

  do l=1,nnx
     do k=1,m

        do j=ily(1),ily(2)
           do i=ilx(1),ilx(2)
              oval(i,j)=tmpval(i,j)  &
  &                    +Km(k)*(tmpval(i-qnx(l),j)+tmpval(i+qnx(l),j)  &
  &                            -2.0d0*tmpval(i,j))
           end do
        end do

        tmpval=oval

     end do
  end do

!-- 配列第二成分フィルター開始

  do l=1,nny
     do k=1,m

        do j=ily(1),ily(2)
           do i=ilx(1),ilx(2)
              oval(i,j)=tmpval(i,j)  &
  &                    +Km(k)*(tmpval(i,j-qny(l))+tmpval(i,j+qny(l))  &
  &                            -2.0d0*tmpval(i,j))
           end do
        end do

        tmpval=oval

     end do
  end do

end subroutine SPLB_Kurihara_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine CPS_Hart_f( x, y, center, z300, z600, z900, mv, B, VTL, VTU )
! 北畠 (2011; 天気) に基づき, Hart (2003a) の低気圧位相空間パラメータ
! の各成分を計算する.
! 計算手順は以下.
! (1) 移動方向 (角度) の決定 : atan(mv(2)/mv(1))
! (2) 500 km 範囲内での平均化.
!     移動方向を (1) で rad 変換しているので, tangen_mean ルーチンで
!     任意角度 rad 内のみの平均化を行う.
! (3) 500 km 範囲内での最大最小値の計算.
!     (1) で得られている rad 情報から, 500 km 以遠の格子に関しては全て
!     undef を付与しておいて, max_min モジュールを使用する.
! [注意] : 値は北半球を想定して計算されているので,
!          南半球に適用する際は, 得られる B に負号を付与すればよい.
!          また, 本ルーチンを用いる際は p 座標系で行われることを想定している.

  use Math_Const

  implicit none

  real, intent(in) :: x(:)  ! x 方向の座標 [m]
  real, intent(in) :: y(:)  ! y 方向の座標 [m]
  real, intent(in) :: center(2)
                         ! 低気圧の中心座標 [m].
                         ! それぞれ x, y 成分の座標に対応する.
  real, intent(in) :: z300(size(x),size(y))
                      ! 300 hPa 高度面でのジオポテンシャル高度 [m]
  real, intent(in) :: z600(size(x),size(y))
                      ! 600 hPa 高度面でのジオポテンシャル高度 [m]
  real, intent(in) :: z900(size(x),size(y))
                      ! 900 hPa 高度面でのジオポテンシャル高度 [m]
  real, intent(in) :: mv(2)   ! 移動速度の x, y 各成分 [m/s].
  real, intent(inout) :: B    ! 気温差パラメータ
  real, intent(inout) :: VTL  ! 下層温度核構造
  real, intent(inout) :: VTU  ! 上層温度核構造
  integer :: nx, ny, i, j, nr, nt
  integer :: irad, itmp
  real, parameter :: rcalc=500.0e3   ! 計算範囲 [m].
  real, parameter :: undef=-1.0e6
  real :: radt, dzmax3, dzmax6, dzmax9, dzmin3, dzmin6, dzmin9, rb, lb
  real, dimension(size(x),size(y)) :: thick, tmpz3, tmpz6, tmpz9
  real, allocatable, dimension(:) :: theta   ! 角度 [rad].
  real, allocatable, dimension(:) :: r       ! 中心からの距離 [m].
  real, allocatable, dimension(:) :: tmpr, tmpl

  nx=size(x)
  ny=size(y)
  nr=nx

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, z300 ),  &
  &                                     "CPS_Hart" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, z600 ),  &
  &                                     "CPS_Hart" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, z900 ),  &
  &                                     "CPS_Hart" )
  end if

  allocate(r(nr))
  allocate(tmpr(nr))
  allocate(tmpl(nr))
  r=(/(((x(2)-x(1))*real(i-1)),i=1,nr)/)
  nt=int(2.0*pi*rcalc/(x(2)-x(1)))
  write(*,*) "*** MESSAGE (CPS_Hart:typhoon_analy) ***"
  write(*,*) "theta number is set of ", nt
  allocate(theta(nt))

  !-- (1) 移動方向の計算
  if(mv(1)>=0.0.and.mv(2)>=0.0)then        ! 第一象限
     if(mv(1)==0.0.and.mv(2)==0.0)then
        radt=0.0
     else if(mv(1)==0.0)then
        radt=0.5*pi
     else if(mv(2)==0.0)then
        radt=0.0
     else
        radt=atan(mv(2)/mv(1))
     end if
  else if(mv(1)<0.0.and.mv(2)>=0.0)then    ! 第二象限
     radt=acos(mv(1)/sqrt(mv(1)**2+mv(2)**2))
  else if(mv(1)<0.0.and.mv(2)<0.0)then     ! 第三象限
     radt=pi+atan(abs(mv(2))/abs(mv(1)))
  else if(mv(1)>=0.0.and.mv(2)<0.0)then    ! 第四象限
     radt=asin(mv(2)/sqrt(mv(1)**2+mv(2)**2))
  end if

  if(0.0>radt)then
     radt=radt+2.0*pi
  else if(2.0*pi<radt)then
     radt=radt-2.0*pi
  end if

  theta=(/((radt+2.0*pi*real(i-1)/real(nt)),i=1,nt)/)

  call interpo_search_1d( r, rcalc, irad )
  irad=irad+1

  !-- 500 km 以遠を全て undef で埋める.
  thick=undef
  tmpz3=undef
  tmpz6=undef
  tmpz9=undef

  do j=1,ny
     do i=1,nx
        thick(i,j)=z600(i,j)-z900(i,j)
        if(sqrt((x(i)-center(1))**2+(y(j)-center(2))**2)<=rcalc)then
           tmpz3(i,j)=z300(i,j)
           tmpz6(i,j)=z600(i,j)
           tmpz9(i,j)=z900(i,j)
        end if
     end do
  end do

  !-- (2) 500 km 以内での平均
  call tangent_mean_scal_f( x, y, center(1), center(2), thick,  &
  &                         r(1:irad), theta(1:nt/2), tmpl(1:irad),  &
  &                         undef=undef, undefg=undef, undefgc='inc' )
  call tangent_mean_scal_f( x, y, center(1), center(2), thick,  &
  &                         r(1:irad), theta(nt/2+1:nt), tmpr(1:irad),  &
  &                         undef=undef, undefg=undef, undefgc='inc' )

  do i=1,irad
     tmpr(i)=tmpr(i)*r(i)
     tmpl(i)=tmpl(i)*r(i)
  end do

  call rectangle_int( r, tmpr, r(1), rcalc, rb )
  call rectangle_int( r, tmpl, r(1), rcalc, lb )

  rb=rb*(2.0/(rcalc**2-r(1)**2))
  lb=lb*(2.0/(rcalc**2-r(1)**2))

  !-- (3) 500 km 以内での最大, 最小値
  call max_val_2d( tmpz3, itmp, itmp, dzmax3, undef=undef )
  call max_val_2d( tmpz6, itmp, itmp, dzmax6, undef=undef )
  call max_val_2d( tmpz9, itmp, itmp, dzmax9, undef=undef )
  call min_val_2d( tmpz3, itmp, itmp, dzmin3, undef=undef )
  call min_val_2d( tmpz6, itmp, itmp, dzmin6, undef=undef )
  call min_val_2d( tmpz9, itmp, itmp, dzmin9, undef=undef )

  B=(rb-lb)
  VTL=((dzmax6-dzmin6)-(dzmax9-dzmin9))/(log(2.0/3.0))
  VTU=((dzmax3-dzmin3)-(dzmax6-dzmin6))/(log(0.5))

end subroutine CPS_Hart_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine CPS_Hart_d( x, y, center, z300, z600, z900, mv, B, VTL, VTU )
! 北畠 (2011; 天気) に基づき, Hart (2003a) の低気圧位相空間パラメータ
! の各成分を計算する.
! 計算手順は以下.
! (1) 移動方向 (角度) の決定 : atan(mv(2)/mv(1))
! (2) 500 km 範囲内での平均化.
!     移動方向を (1) で rad 変換しているので, tangen_mean ルーチンで
!     任意角度 rad 内のみの平均化を行う.
! (3) 500 km 範囲内での最大最小値の計算.
!     (1) で得られている rad 情報から, 500 km 以遠の格子に関しては全て
!     undef を付与しておいて, max_min モジュールを使用する.
! [注意] : 値は北半球を想定して計算されているので,
!          南半球に適用する際は, 得られる B に負号を付与すればよい.
!          また, 本ルーチンを用いる際は p 座標系で行われることを想定している.

  use Math_Const

  implicit none

  double precision, intent(in) :: x(:)  ! x 方向の座標 [m]
  double precision, intent(in) :: y(:)  ! y 方向の座標 [m]
  double precision, intent(in) :: center(2)
                         ! 低気圧の中心座標 [m].
                         ! それぞれ x, y 成分の座標に対応する.
  double precision, intent(in) :: z300(size(x),size(y))
                      ! 300 hPa 高度面でのジオポテンシャル高度 [m]
  double precision, intent(in) :: z600(size(x),size(y))
                      ! 600 hPa 高度面でのジオポテンシャル高度 [m]
  double precision, intent(in) :: z900(size(x),size(y))
                      ! 900 hPa 高度面でのジオポテンシャル高度 [m]
  double precision, intent(in) :: mv(2)   ! 移動速度の x, y 各成分 [m/s].
  double precision, intent(inout) :: B    ! 気温差パラメータ
  double precision, intent(inout) :: VTL  ! 下層温度核構造
  double precision, intent(inout) :: VTU  ! 上層温度核構造
  integer :: nx, ny, i, j, nr, nt
  integer :: irad, itmp
  double precision, parameter :: rcalc=500.0d3   ! 計算範囲 [m].
  double precision, parameter :: undef=-1.0d6
  double precision :: radt, dzmax3, dzmax6, dzmax9, dzmin3, dzmin6, dzmin9, rb, lb
  double precision, dimension(size(x),size(y)) :: thick, tmpz3, tmpz6, tmpz9
  double precision, allocatable, dimension(:) :: theta   ! 角度 [rad].
  double precision, allocatable, dimension(:) :: r       ! 中心からの距離 [m].
  double precision, allocatable, dimension(:) :: tmpr, tmpl

  nx=size(x)
  ny=size(y)
  nr=nx

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, z300 ),  &
  &                                     "CPS_Hart" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, z600 ),  &
  &                                     "CPS_Hart" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, z900 ),  &
  &                                     "CPS_Hart" )
  end if

  allocate(r(nr))
  allocate(tmpr(nr))
  allocate(tmpl(nr))
  r=(/(((x(2)-x(1))*dble(i-1)),i=1,nr)/)
  nt=int(2.0d0*pi_dp*rcalc/(x(2)-x(1)))
  write(*,*) "*** MESSAGE (CPS_Hart:typhoon_analy) ***"
  write(*,*) "theta number is set of ", nt
  allocate(theta(nt))

  !-- (1) 移動方向の計算
  if(mv(1)>=0.0d0.and.mv(2)>=0.0d0)then        ! 第一象限
     if(mv(1)==0.0d0.and.mv(2)==0.0d0)then
        radt=0.0d0
     else if(mv(1)==0.0d0)then
        radt=0.5d0*pi_dp
     else if(mv(2)==0.0d0)then
        radt=0.0d0
     else
        radt=datan(mv(2)/mv(1))
     end if
  else if(mv(1)<0.0d0.and.mv(2)>=0.0d0)then    ! 第二象限
     radt=dacos(mv(1)/dsqrt(mv(1)**2+mv(2)**2))
  else if(mv(1)<0.0d0.and.mv(2)<0.0d0)then     ! 第三象限
     radt=pi_dp+datan(dabs(mv(2))/dabs(mv(1)))
  else if(mv(1)>=0.0d0.and.mv(2)<0.0d0)then    ! 第四象限
     radt=dasin(mv(2)/dsqrt(mv(1)**2+mv(2)**2))
  end if

  if(0.0d0>radt)then
     radt=radt+2.0d0*pi_dp
  else if(2.0d0*pi_dp<radt)then
     radt=radt-2.0d0*pi_dp
  end if

  theta=(/((radt+2.0d0*pi_dp*dble(i-1)/dble(nt)),i=1,nt)/)

  call interpo_search_1d( r, rcalc, irad )
  irad=irad+1

  !-- 500 km 以遠を全て undef で埋める.
  thick=undef
  tmpz3=undef
  tmpz6=undef
  tmpz9=undef

  do j=1,ny
     do i=1,nx
        if(dsqrt((x(i)-center(1))**2+(y(j)-center(2))**2)<=rcalc)then
           thick(i,j)=z600(i,j)-z900(i,j)
           tmpz3(i,j)=z300(i,j)
           tmpz6(i,j)=z600(i,j)
           tmpz9(i,j)=z900(i,j)
        end if
     end do
  end do

  !-- (2) 500 km 以内での平均
  call tangent_mean_scal_d( x, y, center(1), center(2), thick,  &
  &                         r(1:irad), theta(1:nt/2), tmpl(1:irad),  &
  &                         undef=undef, undefg=undef, undefgc='inc' )
  call tangent_mean_scal_d( x, y, center(1), center(2), thick,  &
  &                         r(1:irad), theta(nt/2+1:nt), tmpr(1:irad),  &
  &                         undef=undef, undefg=undef, undefgc='inc' )

  do i=1,irad
     tmpr(i)=tmpr(i)*r(i)
     tmpl(i)=tmpl(i)*r(i)
  end do

  call rectangle_int( r, tmpr, r(1), rcalc, rb )
  call rectangle_int( r, tmpl, r(1), rcalc, lb )

  rb=rb*(2.0d0/(rcalc**2-r(1)**2))
  lb=lb*(2.0d0/(rcalc**2-r(1)**2))

  !-- (3) 500 km 以内での最大, 最小値
  call max_val_2d( tmpz3, itmp, itmp, dzmax3, undef=undef )
  call max_val_2d( tmpz6, itmp, itmp, dzmax6, undef=undef )
  call max_val_2d( tmpz9, itmp, itmp, dzmax9, undef=undef )
  call min_val_2d( tmpz3, itmp, itmp, dzmin3, undef=undef )
  call min_val_2d( tmpz6, itmp, itmp, dzmin6, undef=undef )
  call min_val_2d( tmpz9, itmp, itmp, dzmin9, undef=undef )

  B=(rb-lb)
  VTL=((dzmax6-dzmin6)-(dzmax9-dzmin9))/(dlog(2.0d0/3.0d0))
  VTU=((dzmax3-dzmin3)-(dzmax6-dzmin6))/(dlog(0.5d0))

end subroutine CPS_Hart_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine DC_Braun_f( x, y, fg, pres, search_dis, var_dis, center, undef,  &
  &                    stdopt )
! Braun (2002) の方法を基に台風の中心を推定する.
  use Math_Const
  implicit none
  real, intent(in) :: x(:)  ! x 方向の座標 [m]
  real, intent(in) :: y(:)  ! y 方向の座標 [m]
  integer, intent(in) :: fg(2)
                         ! 中心点の第一推定値 (通常気圧の最低値等から得る)
                         ! fg(1) = x 方向の要素番号, fg(2) = y 方向の要素番号
  real, intent(in) :: pres(size(x),size(y))
                         ! ある高度での気圧 (地表面気圧でもよい.)
                         ! ただし, 地表面気圧の場合は, 海面校正しておくこと.
  real, intent(in) :: search_dis  ! 検索する領域 (fg の位置を中心に)
                         ! 例えば, 100000.0 なら, fg を中心に縦横 100 km 四方
  real, intent(in) :: var_dis  ! 推定中心位置から偏差を計算する半径 [m]
  integer, intent(inout) :: center(2)  ! 求めた中心点の各要素数
  real, intent(in), optional :: undef  ! 気圧に未定義値がある場合, その未定義値.
                             ! 本ルーチンでは, 未定義値がある場合, 
                             ! その格子点のみ偏差計算に使用しない.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)

  integer :: i, j, ix, jy, nx, ny, nr, nt, nxgmin, nxgmax, nygmin, nygmax
  integer :: ompnum, tmp_o_num
  real :: dr, dtheta, undeff, tmpmin, tmp_anom, tmp_counter
  real, allocatable, dimension(:) :: rad, theta
  real, allocatable, dimension(:,:) :: anom_check
  real, allocatable, dimension(:,:,:) :: apres
  logical :: stderr

!-- OpenMP 用整数関数
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, pres ),  &
  &                                     "DC_Braun" )
  end if

  if(present(undef))then
     undeff=undef
  else
     undeff=-999.0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

!-- 検索格子点範囲を規程

  call interpo_search_2d( x, y, x(fg(1))-0.5*search_dis,  &
  &                             y(fg(2))-0.5*search_dis,  &
  &                       nxgmin, nygmin, undeff=int(undeff),  &
  &                       stdopt=stderr )

  call interpo_search_2d( x, y, x(fg(1))+0.5*search_dis,  &
  &                             y(fg(2))+0.5*search_dis,  &
  &                       nxgmax, nygmax, undeff=int(undeff),  &
  &                       stdopt=stderr )

  if(nxgmin==int(undeff))then  ! 領域外参照の場合の処理
     nxgmin=1
  end if
  if(nygmin==int(undeff))then  ! 領域外参照の場合の処理
     nygmin=1
  end if
  if(nxgmax==int(undeff))then  ! 領域外参照の場合の処理
     nxgmax=1
  end if
  if(nygmax==int(undeff))then  ! 領域外参照の場合の処理
     nygmax=1
  end if

!  nxgmin=fg(1)-(search_dis-1)/2
!  nxgmax=fg(1)+(search_dis-1)/2
!  nygmin=fg(2)-(search_dis-1)/2
!  nygmax=fg(2)+(search_dis-1)/2

  allocate(anom_check(nx,ny))

!-- openmp での条件付きコンパイル
!-- 接線平均アノマリの箇所を openmp 並列したいが,
!-- apres が inout 属性なので, private 属性を指定しないと
!-- thread ごとに apres の値が上書きされてしまう.
!-- そこで, threads number を参照した 3 次元配列にして,
!-- thread ごとに別の配列を使うように変更.

   ompnum=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP が有効の場合はここも有効.

  allocate(apres(nx,ny,ompnum))

!-- 円筒系への変換の際には, var_dis での接線解像度が x, y に等しくなるように
!-- 設定する.

  nr=int(var_dis/(x(2)-x(1)))+1
  nt=int(2.0*pi*var_dis/(x(2)-x(1)))+1
!  nt=4
  dr=x(2)-x(1)
  dtheta=2.0*pi/real(nt-1)

  allocate(rad(nr))
  allocate(theta(nt))

  rad=(/((dr*real(i-1)),i=1,nr)/)
  theta=(/((dtheta*real(i-1)),i=1,nt)/)

!-- 各探索格子点について, 接線平均偏差をとり,
!-- 各格子点における偏差の合計を計算する.

  anom_check=undeff  ! 探索範囲外の格子点にはすべて undeff を入れる.
                     ! 探索範囲内の格子にはあとでゼロが入れられ初期化する.

  tmp_o_num=1  ! OpenMP が有効でない場合, この値が apres の 3 次元目へ

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,tmp_anom,tmp_o_num,tmp_counter)

  do j=nygmin,nygmax
     do i=nxgmin,nxgmax

        if(pres(i,j)/=undeff)then
!$         tmp_o_num=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合, この値が apres の 3 次元目へ

           apres(:,:,tmp_o_num)=0.0
           tmp_counter=0

           call tangent_mean_anom_scal_Cart_f( x, y, x(i), y(j), pres, rad, theta,  &
  &                                            apres(:,:,tmp_o_num), undef=undeff,  &
  &                                            undefg=undeff, undefgc='inc',  &
  &                                            stdopt=stderr )

           tmp_anom=0.0

           do jy=1,ny
              do ix=1,nx
                 if(apres(ix,jy,tmp_o_num)/=undeff)then
                    tmp_counter=tmp_counter+1
                    tmp_anom=tmp_anom+apres(ix,jy,tmp_o_num)*apres(ix,jy,tmp_o_num)
                 end if
              end do
           end do

           if(tmp_counter>0)then  ! 平均したが undef しかないときは更新しない.
              anom_check(i,j)=tmp_anom
           end if
        end if
     end do
  end do

!$omp end do
!$omp end parallel

!-- 計算した偏差の合計値のうち, 最小となる格子点を求める.

  call min_val_2d( anom_check, center(1), center(2), tmpmin, undef=undeff )

  if(stderr.eqv..false.)then
     if(center(1)==nx+1.or.center(2)==ny+1)then
        write(*,*) "*** WARNING *** : DC_Braun (typhoon_analy)"
        write(*,*) "Setting the undef only point."
     end if
  end if

  deallocate(rad)
  deallocate(theta)
  deallocate(anom_check)
  deallocate(apres)

end subroutine DC_Braun_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine DC_Braun_d( x, y, fg, pres, search_dis, var_dis, center, undef,  &
  &                    stdopt )
! Braun (2002) の方法を基に台風の中心を推定する.
  use Math_Const
  implicit none
  double precision, intent(in) :: x(:)  ! x 方向の座標 [m]
  double precision, intent(in) :: y(:)  ! y 方向の座標 [m]
  integer, intent(in) :: fg(2)
                         ! 中心点の第一推定値 (通常気圧の最低値等から得る)
                         ! fg(1) = x 方向の要素番号, fg(2) = y 方向の要素番号
  double precision, intent(in) :: pres(size(x),size(y))
                         ! ある高度での気圧 (地表面気圧でもよい.)
                         ! ただし, 地表面気圧の場合は, 海面校正しておくこと.
  double precision, intent(in) :: search_dis  ! 検索する領域 (fg の位置を中心に)
                         ! 例えば, 100000.0 なら, fg を中心に縦横 100 km 四方
  double precision, intent(in) :: var_dis  ! 推定中心位置から偏差を計算する半径 [m]
  integer, intent(inout) :: center(2)  ! 求めた中心点の各要素数
  double precision, intent(in), optional :: undef  ! 気圧に未定義値がある場合, その未定義値.
                             ! 本ルーチンでは, 未定義値がある場合, 
                             ! その格子点のみ偏差計算に使用しない.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)

  integer :: i, j, ix, jy, nx, ny, nr, nt, nxgmin, nxgmax, nygmin, nygmax
  integer :: ompnum, tmp_o_num
  double precision :: dr, dtheta, undeff, tmpmin, tmp_anom, tmp_counter
  double precision, allocatable, dimension(:) :: rad, theta
  double precision, allocatable, dimension(:,:) :: anom_check
  double precision, allocatable, dimension(:,:,:) :: apres
  logical :: stderr

!-- OpenMP 用整数関数
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, pres ),  &
  &                                     "DC_Braun" )
  end if

  if(present(undef))then
     undeff=undef
  else
     undeff=-999.0d0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

!-- 検索格子点範囲を規程

  call interpo_search_2d( x, y, x(fg(1))-0.5d0*search_dis,  &
  &                             y(fg(2))-0.5d0*search_dis,  &
  &                       nxgmin, nygmin, undeff=int(undeff),  &
  &                       stdopt=stderr )

  call interpo_search_2d( x, y, x(fg(1))+0.5d0*search_dis,  &
  &                             y(fg(2))+0.5d0*search_dis,  &
  &                       nxgmax, nygmax, undeff=int(undeff),  &
  &                       stdopt=stderr )

  if(nxgmin==int(undeff))then  ! 領域外参照の場合の処理
     nxgmin=1
  end if
  if(nygmin==int(undeff))then  ! 領域外参照の場合の処理
     nygmin=1
  end if
  if(nxgmax==int(undeff))then  ! 領域外参照の場合の処理
     nxgmax=1
  end if
  if(nygmax==int(undeff))then  ! 領域外参照の場合の処理
     nygmax=1
  end if

!  nxgmin=fg(1)-(search_dis-1)/2
!  nxgmax=fg(1)+(search_dis-1)/2
!  nygmin=fg(2)-(search_dis-1)/2
!  nygmax=fg(2)+(search_dis-1)/2

  allocate(anom_check(nx,ny))

!-- openmp での条件付きコンパイル
!-- 接線平均アノマリの箇所を openmp 並列したいが,
!-- apres が inout 属性なので, private 属性を指定しないと
!-- thread ごとに apres の値が上書きされてしまう.
!-- そこで, threads number を参照した 3 次元配列にして,
!-- thread ごとに別の配列を使うように変更.

   ompnum=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP が有効の場合はここも有効.

  allocate(apres(nx,ny,ompnum))

!-- 円筒系への変換の際には, var_dis での接線解像度が x, y に等しくなるように
!-- 設定する.

  nr=int(var_dis/(x(2)-x(1)))+1
  nt=int(2.0d0*pi_dp*var_dis/(x(2)-x(1)))+1
!  nt=4
  dr=x(2)-x(1)
  dtheta=2.0d0*pi_dp/dble(nt-1)

  allocate(rad(nr))
  allocate(theta(nt))

  rad=(/((dr*dble(i-1)),i=1,nr)/)
  theta=(/((dtheta*dble(i-1)),i=1,nt)/)

!-- 各探索格子点について, 接線平均偏差をとり,
!-- 各格子点における偏差の合計を計算する.

  anom_check=undeff  ! 探索範囲外の格子点にはすべて undeff を入れる.
                     ! 探索範囲内の格子にはあとでゼロが入れられ初期化する.

  tmp_o_num=1  ! OpenMP が有効でない場合, この値が apres の 3 次元目へ

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,tmp_anom,tmp_o_num,tmp_counter)

  do j=nygmin,nygmax
     do i=nxgmin,nxgmax

        if(pres(i,j)/=undeff)then
!$         tmp_o_num=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合, この値が apres の 3 次元目へ

           apres(:,:,tmp_o_num)=0.0d0
           tmp_counter=0

           call tangent_mean_anom_scal_Cart_d( x, y, x(i), y(j), pres, rad, theta,  &
  &                                            apres(:,:,tmp_o_num), undef=undeff,  &
  &                                            undefg=undeff, undefgc='inc',  &
  &                                            stdopt=stderr )

           tmp_anom=0.0d0

           do jy=1,ny
              do ix=1,nx
                 if(apres(ix,jy,tmp_o_num)/=undeff)then
                    tmp_counter=tmp_counter+1
                    tmp_anom=tmp_anom+apres(ix,jy,tmp_o_num)*apres(ix,jy,tmp_o_num)
                 end if
              end do
           end do

           if(tmp_counter>0)then  ! 平均したが undef しかないときは更新しない.
              anom_check(i,j)=tmp_anom
           end if
        end if
     end do
  end do

!$omp end do
!$omp end parallel

!-- 計算した偏差の合計値のうち, 最小となる格子点を求める.

  call min_val_2d( anom_check, center(1), center(2), tmpmin, undef=undeff )

  if(stderr.eqv..false.)then
     if(center(1)==nx+1.or.center(2)==ny+1)then
        write(*,*) "*** WARNING *** : DC_Braun (typhoon_analy)"
        write(*,*) "Setting the undef only point."
     end if
  end if

  deallocate(rad)
  deallocate(theta)
  deallocate(anom_check)
  deallocate(apres)

end subroutine DC_Braun_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine DC_Braun_SAT_f( x, y, fg, tbb, search_dis, var_dis, fg_canr,  &
  &                        center, undef, stdopt )
! Braun (2002) の方法を応用して, 静止衛星での放射輝度温度から
! 台風の中心を推定する.
  use Math_Const
  implicit none
  real, intent(in) :: x(:)  ! x 方向の座標 [m]
  real, intent(in) :: y(:)  ! y 方向の座標 [m]
  integer, intent(in) :: fg(2)
                         ! 中心点の第一推定値 (通常気圧の最低値等から得る)
                         ! fg(1) = x 方向の要素番号, fg(2) = y 方向の要素番号
  real, intent(in) :: tbb(size(x),size(y))
                         ! 放射輝度温度 [K]
  real, intent(in) :: search_dis  ! 検索する領域 (fg の位置を中心に)
                         ! 例えば, 100000.0 なら, fg を中心に縦横 100 km 四方
  real, intent(in) :: var_dis  ! 推定中心位置から偏差を計算する半径 [m]
  real, intent(in) :: fg_canr  ! fg の点から放射輝度の最大最小値を求める半径 [m]
                               ! この範囲内での最大最小からの平均値以上の点が
                               ! 接線平均偏差の候補点となる (search_dis の矩形内).
  integer, intent(inout) :: center(2)  ! 求めた中心点の各要素数
  real, intent(in), optional :: undef  ! 気圧に未定義値がある場合, その未定義値.
                             ! 本ルーチンでは, 未定義値がある場合, 
                             ! その格子点のみ偏差計算に使用しない.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)

  integer :: i, j, ix, jy, nx, ny, nr, nt, nxgmin, nxgmax, nygmin, nygmax
  integer :: ompnum, tmp_o_num, itmpx, itmpy
  real :: dr, dtheta, undeff, tmpmin, tmp_anom, tmp_counter, tmp_rad
  real :: tbb_fg_max, tbb_fg_min, tbb_fg_mean
  real, allocatable, dimension(:) :: rad, theta
  real, allocatable, dimension(:,:) :: anom_check
  real, allocatable, dimension(:,:,:) :: atbb
  logical :: calc_flag(size(x),size(y))  ! 中心点候補点フラグ
  logical :: stderr

!-- OpenMP 用整数関数
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, tbb ),  &
  &                                     "DC_Braun" )
  end if

  if(present(undef))then
     undeff=undef
  else
     undeff=-999.0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  calc_flag=.false.

!-- 検索格子点範囲を規程

  call interpo_search_2d( x, y, x(fg(1))-0.5*search_dis,  &
  &                             y(fg(2))-0.5*search_dis,  &
  &                       nxgmin, nygmin, undeff=int(undeff),  &
  &                       stdopt=stderr )

  call interpo_search_2d( x, y, x(fg(1))+0.5*search_dis,  &
  &                             y(fg(2))+0.5*search_dis,  &
  &                       nxgmax, nygmax, undeff=int(undeff),  &
  &                       stdopt=stderr )

  if(nxgmin==int(undeff))then  ! 領域外参照の場合の処理
     nxgmin=1
  end if
  if(nygmin==int(undeff))then  ! 領域外参照の場合の処理
     nygmin=1
  end if
  if(nxgmax==int(undeff))then  ! 領域外参照の場合の処理
     nxgmax=1
  end if
  if(nygmax==int(undeff))then  ! 領域外参照の場合の処理
     nygmax=1
  end if

!  nxgmin=fg(1)-(search_dis-1)/2
!  nxgmax=fg(1)+(search_dis-1)/2
!  nygmin=fg(2)-(search_dis-1)/2
!  nygmax=fg(2)+(search_dis-1)/2

  allocate(anom_check(nx,ny))

!-- 第一推定点から fg_canr 範囲内の放射輝度温度の最大最小
!-- を求め, その平均値を求める. (anom_check を使い回す 1)

  anom_check=undeff

  do j=1,ny
     do i=1,nx
        tmp_rad=sqrt((x(i)-x(fg(1)))**2+(y(j)-y(fg(2)))**2)
        if(tmp_rad<=fg_canr)then
           anom_check(i,j)=tbb(i,j)
        end if
     end do
  end do

  call max_val_2d( anom_check, itmpx, itmpy, tbb_fg_max, undef=undeff )  
  call min_val_2d( anom_check, itmpx, itmpy, tbb_fg_min, undef=undeff )  
  tbb_fg_mean=0.5*(tbb_fg_max+tbb_fg_min)

!-- openmp での条件付きコンパイル
!-- 接線平均アノマリの箇所を openmp 並列したいが,
!-- atbb が inout 属性なので, private 属性を指定しないと
!-- thread ごとに atbb の値が上書きされてしまう.
!-- そこで, threads number を参照した 3 次元配列にして,
!-- thread ごとに別の配列を使うように変更.

  ompnum=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP が有効の場合はここも有効.

  allocate(atbb(nx,ny,ompnum))

!-- 円筒系への変換の際には, var_dis での接線解像度が x, y に等しくなるように
!-- 設定する.

  nr=int(var_dis/(x(2)-x(1)))+1
  nt=int(2.0*pi*var_dis/(x(2)-x(1)))+1
!  nt=4
  dr=x(2)-x(1)
  dtheta=2.0*pi/real(nt-1)

  allocate(rad(nr))
  allocate(theta(nt))

  rad=(/((dr*real(i-1)),i=1,nr)/)
  theta=(/((dtheta*real(i-1)),i=1,nt)/)

!-- 各探索格子点について, 接線平均偏差をとり,
!-- 各格子点における偏差の合計を計算する.

  anom_check=undeff  ! 探索範囲外の格子点にはすべて undeff を入れる.
                     ! 探索範囲内の格子にはあとでゼロが入れられ初期化する.

  tmp_o_num=1  ! OpenMP が有効でない場合, この値が atbb の 3 次元目へ

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,tmp_anom,tmp_o_num,tmp_counter)

  do j=nygmin,nygmax
     do i=nxgmin,nxgmax

        if(tbb(i,j)/=undeff.and.tbb(i,j)>=tbb_fg_mean)then
           ! 候補点が上で求めた閾値以上の放射輝度温度を持っているか
!$         tmp_o_num=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合, この値が atbb の 3 次元目へ

           atbb(:,:,tmp_o_num)=0.0
           tmp_counter=0

           call tangent_mean_anom_scal_Cart_f( x, y, x(i), y(j), tbb, rad, theta,  &
  &                                            atbb(:,:,tmp_o_num), undef=undeff,  &
  &                                            undefg=undeff, undefgc='inc',  &
  &                                            stdopt=stderr )

           tmp_anom=0.0

           do jy=1,ny
              do ix=1,nx
                 if(atbb(ix,jy,tmp_o_num)/=undeff)then
                    tmp_counter=tmp_counter+1
                    tmp_anom=tmp_anom+atbb(ix,jy,tmp_o_num)*atbb(ix,jy,tmp_o_num)
                 end if
              end do
           end do

           if(tmp_counter>0)then  ! 平均したが undef しかないときは更新しない.
              anom_check(i,j)=tmp_anom
           end if
        end if
     end do
  end do

!$omp end do
!$omp end parallel

!-- 計算した偏差の合計値のうち, 最小となる格子点を求める.

  call min_val_2d( anom_check, center(1), center(2), tmpmin, undef=undeff )

  if(stderr.eqv..false.)then
     if(center(1)==nx+1.or.center(2)==ny+1)then
        write(*,*) "*** WARNING *** : DC_Braun (typhoon_analy)"
        write(*,*) "Setting the undef only point."
     end if
  end if

  deallocate(rad)
  deallocate(theta)
  deallocate(anom_check)
  deallocate(atbb)

end subroutine DC_Braun_SAT_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine DC_Braun_SAT_d( x, y, fg, tbb, search_dis, var_dis, fg_canr,  &
  &                        center, undef, stdopt )
! Braun (2002) の方法を応用して, 静止衛星での放射輝度温度から
! 台風の中心を推定する.
  use Math_Const
  implicit none
  double precision, intent(in) :: x(:)  ! x 方向の座標 [m]
  double precision, intent(in) :: y(:)  ! y 方向の座標 [m]
  integer, intent(in) :: fg(2)
                         ! 中心点の第一推定値 (通常気圧の最低値等から得る)
                         ! fg(1) = x 方向の要素番号, fg(2) = y 方向の要素番号
  double precision, intent(in) :: tbb(size(x),size(y))
                         ! 放射輝度温度 [K]
  double precision, intent(in) :: search_dis  ! 検索する領域 (fg の位置を中心に)
                         ! 例えば, 100000.0 なら, fg を中心に縦横 100 km 四方
  double precision, intent(in) :: var_dis  ! 推定中心位置から偏差を計算する半径 [m]
  double precision, intent(in) :: fg_canr  ! fg の点から放射輝度の最大最小値を求める半径 [m]
                               ! この範囲内での最大最小からの平均値以上の点が
                               ! 接線平均偏差の候補点となる (search_dis の矩形内).
  integer, intent(inout) :: center(2)  ! 求めた中心点の各要素数
  double precision, intent(in), optional :: undef  ! 気圧に未定義値がある場合, その未定義値.
                             ! 本ルーチンでは, 未定義値がある場合, 
                             ! その格子点のみ偏差計算に使用しない.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)

  integer :: i, j, ix, jy, nx, ny, nr, nt, nxgmin, nxgmax, nygmin, nygmax
  integer :: ompnum, tmp_o_num, itmpx, itmpy
  double precision :: dr, dtheta, undeff, tmpmin, tmp_anom, tmp_counter, tmp_rad
  double precision :: tbb_fg_max, tbb_fg_min, tbb_fg_mean
  double precision, allocatable, dimension(:) :: rad, theta
  double precision, allocatable, dimension(:,:) :: anom_check
  double precision, allocatable, dimension(:,:,:) :: atbb
  logical :: calc_flag(size(x),size(y))  ! 中心点候補点フラグ
  logical :: stderr

!-- OpenMP 用整数関数
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, tbb ),  &
  &                                     "DC_Braun" )
  end if

  if(present(undef))then
     undeff=undef
  else
     undeff=-999.0d0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  calc_flag=.false.

!-- 検索格子点範囲を規程

  call interpo_search_2d( x, y, x(fg(1))-0.5d0*search_dis,  &
  &                             y(fg(2))-0.5d0*search_dis,  &
  &                       nxgmin, nygmin, undeff=int(undeff),  &
  &                       stdopt=stderr )

  call interpo_search_2d( x, y, x(fg(1))+0.5d0*search_dis,  &
  &                             y(fg(2))+0.5d0*search_dis,  &
  &                       nxgmax, nygmax, undeff=int(undeff),  &
  &                       stdopt=stderr )

  if(nxgmin==int(undeff))then  ! 領域外参照の場合の処理
     nxgmin=1
  end if
  if(nygmin==int(undeff))then  ! 領域外参照の場合の処理
     nygmin=1
  end if
  if(nxgmax==int(undeff))then  ! 領域外参照の場合の処理
     nxgmax=1
  end if
  if(nygmax==int(undeff))then  ! 領域外参照の場合の処理
     nygmax=1
  end if

!  nxgmin=fg(1)-(search_dis-1)/2
!  nxgmax=fg(1)+(search_dis-1)/2
!  nygmin=fg(2)-(search_dis-1)/2
!  nygmax=fg(2)+(search_dis-1)/2

  allocate(anom_check(nx,ny))

!-- 第一推定点から fg_canr 範囲内の放射輝度温度の最大最小
!-- を求め, その平均値を求める. (anom_check を使い回す 1)

  anom_check=undeff

  do j=1,ny
     do i=1,nx
        tmp_rad=dsqrt((x(i)-x(fg(1)))**2+(y(j)-y(fg(2)))**2)
        if(tmp_rad<=fg_canr)then
           anom_check(i,j)=tbb(i,j)
        end if
     end do
  end do

  call max_val_2d( anom_check, itmpx, itmpy, tbb_fg_max, undef=undeff )  
  call min_val_2d( anom_check, itmpx, itmpy, tbb_fg_min, undef=undeff )  
  tbb_fg_mean=0.5d0*(tbb_fg_max+tbb_fg_min)

!-- openmp での条件付きコンパイル
!-- 接線平均アノマリの箇所を openmp 並列したいが,
!-- atbb が inout 属性なので, private 属性を指定しないと
!-- thread ごとに atbb の値が上書きされてしまう.
!-- そこで, threads number を参照した 3 次元配列にして,
!-- thread ごとに別の配列を使うように変更.

  ompnum=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP が有効の場合はここも有効.

  allocate(atbb(nx,ny,ompnum))

!-- 円筒系への変換の際には, var_dis での接線解像度が x, y に等しくなるように
!-- 設定する.

  nr=int(var_dis/(x(2)-x(1)))+1
  nt=int(2.0d0*pi_dp*var_dis/(x(2)-x(1)))+1
!  nt=4
  dr=x(2)-x(1)
  dtheta=2.0d0*pi_dp/dble(nt-1)

  allocate(rad(nr))
  allocate(theta(nt))

  rad=(/((dr*dble(i-1)),i=1,nr)/)
  theta=(/((dtheta*dble(i-1)),i=1,nt)/)

!-- 各探索格子点について, 接線平均偏差をとり,
!-- 各格子点における偏差の合計を計算する.

  anom_check=undeff  ! 探索範囲外の格子点にはすべて undeff を入れる.
                     ! 探索範囲内の格子にはあとでゼロが入れられ初期化する.

  tmp_o_num=1  ! OpenMP が有効でない場合, この値が atbb の 3 次元目へ

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,tmp_anom,tmp_o_num,tmp_counter)

  do j=nygmin,nygmax
     do i=nxgmin,nxgmax

        if(tbb(i,j)/=undeff.and.tbb(i,j)>=tbb_fg_mean)then
           ! 候補点が上で求めた閾値以上の放射輝度温度を持っているか
!$         tmp_o_num=OMP_GET_THREAD_NUM()+1  ! OpenMP が有効の場合, この値が atbb の 3 次元目へ

           atbb(:,:,tmp_o_num)=0.0d0
           tmp_counter=0

           call tangent_mean_anom_scal_Cart_d( x, y, x(i), y(j), tbb, rad, theta,  &
  &                                            atbb(:,:,tmp_o_num), undef=undeff,  &
  &                                            undefg=undeff, undefgc='inc',  &
  &                                            stdopt=stderr )

           tmp_anom=0.0d0

           do jy=1,ny
              do ix=1,nx
                 if(atbb(ix,jy,tmp_o_num)/=undeff)then
                    tmp_counter=tmp_counter+1
                    tmp_anom=tmp_anom+atbb(ix,jy,tmp_o_num)*atbb(ix,jy,tmp_o_num)
                 end if
              end do
           end do

           if(tmp_counter>0)then  ! 平均したが undef しかないときは更新しない.
              anom_check(i,j)=tmp_anom
           end if
        end if
     end do
  end do

!$omp end do
!$omp end parallel

!-- 計算した偏差の合計値のうち, 最小となる格子点を求める.

  call min_val_2d( anom_check, center(1), center(2), tmpmin, undef=undeff )

  if(stderr.eqv..false.)then
     if(center(1)==nx+1.or.center(2)==ny+1)then
        write(*,*) "*** WARNING *** : DC_Braun (typhoon_analy)"
        write(*,*) "Setting the undef only point."
     end if
  end if

  deallocate(rad)
  deallocate(theta)
  deallocate(anom_check)
  deallocate(atbb)

end subroutine DC_Braun_SAT_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine Parallax_Himawari_f( lambda_cld, phi_cld, h_cld,  &
  &                             lambda_cor, phi_cor, undef )
! ひまわり 8 号の輝度温度データ (から換算した雲頂高度データ) をもとに
! lambda_cld, phi_cld での緯度経度を視差補正して lambda_cor, phi_cor に変換する.
  use Math_Const
  implicit none
  real, intent(in) :: lambda_cld(:,:)  ! ひまわり格子点経度 [rad]
  real, intent(in) :: phi_cld(size(lambda_cld,1),size(lambda_cld,2))
                                       ! ひまわり格子点緯度 [rad]
  real, intent(in) :: h_cld(size(lambda_cld,1),size(lambda_cld,2))
                                       ! ひまわり格子点での雲頂高度 [m]
  real, intent(inout) :: lambda_cor(size(lambda_cld,1),size(lambda_cld,2))
                                       ! 視差補正後格子点経度 [rad]
  real, intent(inout) :: phi_cor(size(lambda_cld,1),size(lambda_cld,2))
                                       ! 視差補正後格子点緯度 [rad]
  real, intent(in), optional :: undef  ! 雲頂高度の未定義値

  !-- ひまわり 8 号固有パラメータ
  real :: re=6378.1370e3
  real :: rp=6356.7523e3
  real :: rrat=1.00336409e0
  real :: hsat=42164.0e3
  real :: psat=0.0e0
  real :: lsat=140.7e0

  real :: x0, y0, z0, xs, ys, zs
  real :: h, lcld, pcld, lcldr, pcldr, lcldc, pcldc
  real :: xx0, yy0, zz0, xxs, yys, zzs, xys0, x20, y20, z20, onemr
  real :: d2r, r2d, xyz0, gamcoe, gamcoe2, tparm
  real :: ztmp, zeps, parfunc, pardfunc, ztmpa, ztmpb, ztmpc
  real :: xa, ya, za, phia, lambdaa, rundef
  integer :: i, j, nx, ny

  d2r=pi/180.0e0
!  r2d=180.0e0/pi

  rrat=re/rp
  onemr=1.0e0-rrat**2

  nx=size(lambda_cld,1)
  ny=size(lambda_cld,2)

  if(present(undef))then
     rundef=undef
  else
     rundef=-100.0e0
  end if

  do j=1,ny
     do i=1,nx

        pcldr=atan2(tan(phi_cld(i,j)),rrat**2)  ! geogra -> geocen
        lcldr=lambda_cld(i,j)
        h=h_cld(i,j)

        x0=rrat*sin(0.5e0*pi-pcldr)*cos(lcldr)
        y0=rrat*sin(0.5e0*pi-pcldr)*sin(lcldr)
        z0=cos(0.5e0*pi-pcldr)
        xs=(hsat/rp)*cos(lsat*d2r)
        ys=(hsat/rp)*sin(lsat*d2r)
        zs=0.0e0
        h=h/rp

        xx0=xs-x0
        yy0=ys-y0
        zz0=zs-z0
        xxs=x0*zs-z0*xs
        yys=y0*zs-z0*ys
        x20=xx0**2
        y20=yy0**2
        z20=zz0**2
        xys0=xxs*xx0+yys*yy0
        xyz0=x20+y20+z20

        if(h/=rundef.and.z0/=0.0e0)then

           ztmp=z0
           ztmpa=ztmp
           ztmpb=0.8e0*ztmp
           ztmpc=0.5e0*(ztmpa+ztmpb)
           parfunc=(((xxs+ztmpa*xx0)**2+(yys+ztmpa*yy0)**2  &
  &                 +(zz0*rrat*ztmpa)**2-(zz0*rrat)**2)  &
  &                *(1.0e0-onemr*(ztmpa**2))  &
  &                +((ztmpa*h)**2)*((rrat**2)*(x20+y20)+z20)-(h*zz0)**2)**2  &
  &               -4.0e0*((xys0*ztmpa+xyz0*(ztmpa**2)-z20)**2)  &
  &                *(1.0e0-onemr*(ztmpa**2))*((h*rrat)**2)
           pardfunc=(((xxs+ztmpc*xx0)**2+(yys+ztmpc*yy0)**2  &
  &                  +(zz0*rrat*ztmpc)**2-(zz0*rrat)**2)  &
  &                 *(1.0e0-onemr*(ztmpc**2))  &
  &                 +((ztmpc*h)**2)*((rrat**2)*(x20+y20)+z20)-(h*zz0)**2)**2  &
  &                -4.0e0*((xys0*ztmpc+xyz0*(ztmpc**2)-z20)**2)  &
  &                 *(1.0e0-onemr*(ztmpc**2))*((h*rrat)**2)
           do while (abs(ztmpa-ztmpb)>1.0e-6)
              if(parfunc*pardfunc<0.0e0)then
                 ztmpb=ztmpc
                 ztmpc=0.5e0*(ztmpa+ztmpb)
                 pardfunc=(((xxs+ztmpc*xx0)**2+(yys+ztmpc*yy0)**2  &
  &                        +(zz0*rrat*ztmpc)**2-(zz0*rrat)**2)  &
  &                       *(1.0e0-onemr*(ztmpc**2))  &
  &                       +((ztmpc*h)**2)*((rrat**2)*(x20+y20)+z20)-(h*zz0)**2)**2  &
  &                      -4.0e0*((xys0*ztmpc+xyz0*(ztmpc**2)-z20)**2)  &
  &                       *(1.0e0-onemr*(ztmpc**2))*((h*rrat)**2)
              else
                 ztmpa=ztmpc
                 ztmpc=0.5e0*(ztmpa+ztmpb)
                 parfunc=pardfunc
                 pardfunc=(((xxs+ztmpc*xx0)**2+(yys+ztmpc*yy0)**2  &
  &                        +(zz0*rrat*ztmpc)**2-(zz0*rrat)**2)  &
  &                       *(1.0e0-onemr*(ztmpc**2))  &
  &                       +((ztmpc*h)**2)*((rrat**2)*(x20+y20)+z20)-(h*zz0)**2)**2  &
  &                      -4.0e0*((xys0*ztmpc+xyz0*(ztmpc**2)-z20)**2)  &
  &                       *(1.0e0-onemr*(ztmpc**2))*((h*rrat)**2)
              end if
           end do
           ztmp=ztmpc

           za=ztmp
           gamcoe=1.0e0/sqrt(1.0e0-onemr*(za**2))
           gamcoe2=((rrat+h*gamcoe)/(1.0e0+h*rrat*gamcoe))**2
           tparm=((x0*xx0+y0*yy0+z0*zz0*gamcoe2)/(x20+y20+z20*gamcoe2))  &
  &             *(-1.0e0+sqrt(1.0e0+((x20+y20+z20*gamcoe2)  &
  &                                  /((x0*xx0+y0*yy0+z0*zz0*gamcoe2)**2))  &
  &                                  *((2.0e0+gamcoe*h*rrat)*gamcoe*h*rrat  &
  &                                   -z0*z0*(gamcoe2-rrat**2))))
           if(tparm<0.0e0.or.tparm>1.0e0)then
              tparm=((x0*xx0+y0*yy0+z0*zz0*gamcoe2)/(x20+y20+z20*gamcoe2))  &
  &                *(-1.0e0-sqrt(1.0e0+((x20+y20+z20*gamcoe2)  &
  &                                     /((x0*xx0+y0*yy0+z0*zz0*gamcoe2)**2))  &
  &                                     *((2.0e0+gamcoe*h*rrat)*gamcoe*h*rrat  &
  &                                      -z0*z0*(gamcoe2-rrat**2))))
              if(tparm<0.0e0.or.tparm>1.0e0)then
                 tparm=rundef
                 lambda_cor(i,j)=rundef
                 phi_cor(i,j)=rundef
                 cycle
!                 write(*,*) "*** WARNING (Parallax_Himawari) ***: tparm is invalid."
              end if
           end if

!        if(zs-z0/=0.0e0)then
!           write(*,*) "tparm check", tparm, (-z0+za+(h*rrat*za)*gamcoe)/(zs-z0)
!           tparm=(-z0+za+(h*rrat*za)*gamcoe)/(zs-z0)
!        else
!           tparm=0.0e0
!        end if
           xa=((1.0e0-tparm)*x0+tparm*xs)/(1.0e0+(h/rrat)*gamcoe)
           ya=((1.0e0-tparm)*y0+tparm*ys)/(1.0e0+(h/rrat)*gamcoe)

           phia=atan2(tan(0.5e0*pi-acos(za)),1.0e0/(rrat**2))  ! geocen -> geogra
           lambdaa=atan2(ya,xa)!*r2d

           lambda_cor(i,j)=lambdaa
           phi_cor(i,j)=phia

        else if(h/=rundef.and.z0==0.0e0)then

           tparm=((xys0-x20-y20)/((xs-x0)**2+(ys-y0)**2))  &
  &             *(-1.0e0+sqrt(1.0e0+((xs-x0)**2+(ys-y0)**2)  &
  &                                  *(2.0*rrat*h+h**2)/((xys0-x20-y20)**2)))
           if(tparm<0.0e0.or.tparm>1.0e0)then
              tparm=((xys0-x20-y20)/((xs-x0)**2+(ys-y0)**2))  &
  &                *(-1.0e0-sqrt(1.0e0+((xs-x0)**2+(ys-y0)**2)  &
  &                                     *(2.0*rrat*h+h**2)/((xys0-x20-y20)**2)))
              if(tparm<0.0e0.or.tparm>1.0e0)then
                 tparm=rundef
                 lambda_cor(i,j)=rundef
                 phi_cor(i,j)=rundef
                 cycle
!                 write(*,*) "*** ERROR (Parallax_Himawari) ***: tparm is invalid."
              end if
           end if

           xa=((1.0e0-tparm)*x0+tparm*xs)/(1.0e0+(h/rrat))
           ya=((1.0e0-tparm)*y0+tparm*ys)/(1.0e0+(h/rrat))

           lambdaa=atan2(ya,xa)!*r2d

           lambda_cor(i,j)=lambdaa
           phi_cor(i,j)=pcldr

        else

           tparm=rundef
           lambda_cor(i,j)=rundef
           phi_cor(i,j)=rundef

        end if

     end do
  end do

end subroutine Parallax_Himawari_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine Parallax_Himawari_d( lambda_cld, phi_cld, h_cld,  &
  &                             lambda_cor, phi_cor, undef )
! ひまわり 8 号の輝度温度データ (から換算した雲頂高度データ) をもとに
! lambda_cld, phi_cld での緯度経度を視差補正して lambda_cor, phi_cor に変換する.
  use Math_Const
  implicit none
  double precision, intent(in) :: lambda_cld(:,:)  ! ひまわり格子点経度 [rad]
  double precision, intent(in) :: phi_cld(size(lambda_cld,1),size(lambda_cld,2))
                                                   ! ひまわり格子点緯度 [rad]
  double precision, intent(in) :: h_cld(size(lambda_cld,1),size(lambda_cld,2))
                                                   ! ひまわり格子点での雲頂高度 [m]
  double precision, intent(inout) :: lambda_cor(size(lambda_cld,1),size(lambda_cld,2))
                                                   ! 視差補正後格子点経度 [rad]
  double precision, intent(inout) :: phi_cor(size(lambda_cld,1),size(lambda_cld,2))
                                                   ! 視差補正後格子点緯度 [rad]
  double precision, intent(in), optional :: undef  ! 雲頂高度の未定義値

  !-- ひまわり 8 号固有パラメータ
  double precision :: re=6378.1370d3
  double precision :: rp=6356.7523d3
  double precision :: rrat=1.00336409d0
  double precision :: hsat=42164.0d3
  double precision :: psat=0.0d0
  double precision :: lsat=140.7d0

  double precision :: x0, y0, z0, xs, ys, zs
  double precision :: h, lcld, pcld, lcldr, pcldr, lcldc, pcldc
  double precision :: xx0, yy0, zz0, xxs, yys, zzs, xys0, x20, y20, z20, onemr
  double precision :: d2r, r2d, xyz0, gamcoe, gamcoe2, tparm
  double precision :: ztmp, zeps, parfunc, pardfunc, ztmpa, ztmpb, ztmpc
  double precision :: xa, ya, za, phia, lambdaa, rundef
  integer :: i, j, nx, ny

  d2r=pi_dp/180.0d0
!  r2d=180.0d0/pi_dp

  rrat=re/rp
  onemr=1.0d0-rrat**2

  nx=size(lambda_cld,1)
  ny=size(lambda_cld,2)

  if(present(undef))then
     rundef=undef
  else
     rundef=-100.0d0
  end if

  do j=1,ny
     do i=1,nx

        pcldr=datan2(dtan(phi_cld(i,j)),rrat**2)  ! geogra -> geocen
        lcldr=lambda_cld(i,j)
        h=h_cld(i,j)

        x0=rrat*dsin(0.5d0*pi_dp-pcldr)*dcos(lcldr)
        y0=rrat*dsin(0.5d0*pi_dp-pcldr)*dsin(lcldr)
        z0=dcos(0.5d0*pi_dp-pcldr)
        xs=(hsat/rp)*dcos(lsat*d2r)
        ys=(hsat/rp)*dsin(lsat*d2r)
        zs=0.0d0
        h=h/rp

        xx0=xs-x0
        yy0=ys-y0
        zz0=zs-z0
        xxs=x0*zs-z0*xs
        yys=y0*zs-z0*ys
        x20=xx0**2
        y20=yy0**2
        z20=zz0**2
        xys0=xxs*xx0+yys*yy0
        xyz0=x20+y20+z20

        if(h/=rundef.and.z0/=0.0d0)then

           ztmp=z0
           ztmpa=ztmp
           ztmpb=0.8d0*ztmp
           ztmpc=0.5d0*(ztmpa+ztmpb)
           parfunc=(((xxs+ztmpa*xx0)**2+(yys+ztmpa*yy0)**2  &
  &                 +(zz0*rrat*ztmpa)**2-(zz0*rrat)**2)  &
  &                *(1.0d0-onemr*(ztmpa**2))  &
  &                +((ztmpa*h)**2)*((rrat**2)*(x20+y20)+z20)-(h*zz0)**2)**2  &
  &               -4.0d0*((xys0*ztmpa+xyz0*(ztmpa**2)-z20)**2)  &
  &                *(1.0d0-onemr*(ztmpa**2))*((h*rrat)**2)
           pardfunc=(((xxs+ztmpc*xx0)**2+(yys+ztmpc*yy0)**2  &
  &                  +(zz0*rrat*ztmpc)**2-(zz0*rrat)**2)  &
  &                 *(1.0d0-onemr*(ztmpc**2))  &
  &                 +((ztmpc*h)**2)*((rrat**2)*(x20+y20)+z20)-(h*zz0)**2)**2  &
  &                -4.0d0*((xys0*ztmpc+xyz0*(ztmpc**2)-z20)**2)  &
  &                 *(1.0d0-onemr*(ztmpc**2))*((h*rrat)**2)
           do while (abs(ztmpa-ztmpb)>1.0d-12)
              if(parfunc*pardfunc<0.0d0)then
                 ztmpb=ztmpc
                 ztmpc=0.5d0*(ztmpa+ztmpb)
                 pardfunc=(((xxs+ztmpc*xx0)**2+(yys+ztmpc*yy0)**2  &
  &                        +(zz0*rrat*ztmpc)**2-(zz0*rrat)**2)  &
  &                       *(1.0d0-onemr*(ztmpc**2))  &
  &                       +((ztmpc*h)**2)*((rrat**2)*(x20+y20)+z20)-(h*zz0)**2)**2  &
  &                      -4.0d0*((xys0*ztmpc+xyz0*(ztmpc**2)-z20)**2)  &
  &                       *(1.0d0-onemr*(ztmpc**2))*((h*rrat)**2)
              else
                 ztmpa=ztmpc
                 ztmpc=0.5d0*(ztmpa+ztmpb)
                 parfunc=pardfunc
                 pardfunc=(((xxs+ztmpc*xx0)**2+(yys+ztmpc*yy0)**2  &
  &                        +(zz0*rrat*ztmpc)**2-(zz0*rrat)**2)  &
  &                       *(1.0d0-onemr*(ztmpc**2))  &
  &                       +((ztmpc*h)**2)*((rrat**2)*(x20+y20)+z20)-(h*zz0)**2)**2  &
  &                      -4.0d0*((xys0*ztmpc+xyz0*(ztmpc**2)-z20)**2)  &
  &                       *(1.0d0-onemr*(ztmpc**2))*((h*rrat)**2)
              end if
           end do
           ztmp=ztmpc

           za=ztmp
           gamcoe=1.0d0/dsqrt(1.0d0-onemr*(za**2))
           gamcoe2=((rrat+h*gamcoe)/(1.0d0+h*rrat*gamcoe))**2
           tparm=((x0*xx0+y0*yy0+z0*zz0*gamcoe2)/(x20+y20+z20*gamcoe2))  &
  &             *(-1.0d0+dsqrt(1.0d0+((x20+y20+z20*gamcoe2)  &
  &                                  /((x0*xx0+y0*yy0+z0*zz0*gamcoe2)**2))  &
  &                                  *((2.0d0+gamcoe*h*rrat)*gamcoe*h*rrat  &
  &                                   -z0*z0*(gamcoe2-rrat**2))))
           if(tparm<0.0d0.or.tparm>1.0d0)then
              tparm=((x0*xx0+y0*yy0+z0*zz0*gamcoe2)/(x20+y20+z20*gamcoe2))  &
  &                *(-1.0d0-dsqrt(1.0d0+((x20+y20+z20*gamcoe2)  &
  &                                     /((x0*xx0+y0*yy0+z0*zz0*gamcoe2)**2))  &
  &                                     *((2.0d0+gamcoe*h*rrat)*gamcoe*h*rrat  &
  &                                      -z0*z0*(gamcoe2-rrat**2))))
              if(tparm<0.0d0.or.tparm>1.0d0)then
                 tparm=rundef
                 lambda_cor(i,j)=rundef
                 phi_cor(i,j)=rundef
                 cycle
!                 write(*,*) "*** ERROR (Parallax_Himawari) ***: tparm is invalid."
              end if
           end if

!        if(zs-z0/=0.0d0)then
!           write(*,*) "tparm check", tparm, (-z0+za+(h*rrat*za)*gamcoe)/(zs-z0)
!           tparm=(-z0+za+(h*rrat*za)*gamcoe)/(zs-z0)
!        else
!           tparm=0.0d0
!        end if
           xa=((1.0d0-tparm)*x0+tparm*xs)/(1.0d0+(h/rrat)*gamcoe)
           ya=((1.0d0-tparm)*y0+tparm*ys)/(1.0d0+(h/rrat)*gamcoe)

           phia=datan2(dtan(0.5d0*pi_dp-dacos(za)),1.0d0/(rrat**2))  ! geocen -> geogra
           lambdaa=datan2(ya,xa)!*r2d

           lambda_cor(i,j)=lambdaa
           phi_cor(i,j)=phia

        else if(h/=rundef.and.z0==0.0d0)then

           tparm=((xys0-x20-y20)/((xs-x0)**2+(ys-y0)**2))  &
  &             *(-1.0d0+dsqrt(1.0d0+((xs-x0)**2+(ys-y0)**2)  &
  &                                  *(2.0*rrat*h+h**2)/((xys0-x20-y20)**2)))
           if(tparm<0.0d0.or.tparm>1.0d0)then
              tparm=((xys0-x20-y20)/((xs-x0)**2+(ys-y0)**2))  &
  &                *(-1.0d0-dsqrt(1.0d0+((xs-x0)**2+(ys-y0)**2)  &
  &                                     *(2.0*rrat*h+h**2)/((xys0-x20-y20)**2)))
              if(tparm<0.0d0.or.tparm>1.0d0)then
                 tparm=rundef
                 lambda_cor(i,j)=rundef
                 phi_cor(i,j)=rundef
                 cycle
!                 write(*,*) "*** ERROR (Parallax_Himawari) ***: tparm is invalid."
              end if
           end if

           xa=((1.0d0-tparm)*x0+tparm*xs)/(1.0d0+(h/rrat))
           ya=((1.0d0-tparm)*y0+tparm*ys)/(1.0d0+(h/rrat))

           lambdaa=datan2(ya,xa)!*r2d

           lambda_cor(i,j)=lambdaa
           phi_cor(i,j)=pcldr

        else

           tparm=rundef
           lambda_cor(i,j)=rundef
           phi_cor(i,j)=rundef

        end if

     end do
  end do

end subroutine Parallax_Himawari_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine Parallax_Himawari_THap_f( lambda_cld, phi_cld, h_cld,  &
  &                                  lambda_cor, phi_cor, undef )
! ひまわり 8 号の輝度温度データ (から換算した雲頂高度データ) をもとに
! lambda_cld, phi_cld での緯度経度を視差補正して lambda_cor, phi_cor に変換する.
! Tsukada and Horinouchi (2020) で用いられている近似版
  use Math_Const
  implicit none
  real, intent(in) :: lambda_cld(:,:)  ! ひまわり格子点経度 [rad]
  real, intent(in) :: phi_cld(size(lambda_cld,1),size(lambda_cld,2))
                                                   ! ひまわり格子点緯度 [rad]
  real, intent(in) :: h_cld(size(lambda_cld,1),size(lambda_cld,2))
                                                   ! ひまわり格子点での雲頂高度 [m]
  real, intent(inout) :: lambda_cor(size(lambda_cld,1),size(lambda_cld,2))
                                                   ! 視差補正後格子点経度 [rad]
  real, intent(inout) :: phi_cor(size(lambda_cld,1),size(lambda_cld,2))
                                                   ! 視差補正後格子点緯度 [rad]
  real, intent(in), optional :: undef  ! 雲頂高度の未定義値

  !-- ひまわり 8 号固有パラメータ
  real :: re=6378.1370e3
  real :: rp=6356.7523e3
  real :: rrat=1.00336409e0
  real :: hsat=42164.0e3
  real :: psat=0.0e0
  real :: lsat=140.7e0

  real :: x0, y0, z0, xs, ys, zs
  real :: h, lcld, pcld, lcldr, pcldr, lcldc, pcldc
  real :: d2r, r2d, e1, e2, e3, rl, cd
  real :: eps, xc, yc, zc, xd, yd, zd, rlr, ra
  real :: xa, ya, za, phia, lambdaa, rundef
  integer :: i, j, nx, ny

  d2r=pi/180.0e0
!  r2d=180.0e0/pi

  rrat=re/rp

  nx=size(lambda_cld,1)
  ny=size(lambda_cld,2)

  if(present(undef))then
     rundef=undef
  else
     rundef=-100.0e0
  end if

  do j=1,ny
     do i=1,nx

        pcldr=phi_cld(i,j)
        lcldr=lambda_cld(i,j)
        h=h_cld(i,j)
        rl=re/sqrt(cos(pcldr)**2+(rrat*sin(pcldr))**2)

        x0=rl*cos(pcldr)*sin(lcldr)
        y0=rl*sin(pcldr)
        z0=rl*cos(pcldr)*cos(lcldr)
        xs=hsat*cos(psat*d2r)*sin(lsat*d2r)
        ys=hsat*sin(psat*d2r)
        zs=hsat*cos(psat*d2r)*cos(lsat*d2r)

        xd=xs-x0
        yd=ys-y0
        zd=zs-z0
        rlr=((re+h)/(rp+h))**2
        e1=xd**2+rlr*yd**2+zd**2
        e2=x0*xd+rlr*y0*yd+z0*zd
        e3=x0**2+rlr*y0**2+z0**2-(re+h)**2
        cd=(e2/e1)*(-1.0e0+sqrt(1.0e0-e1*e3/(e2**2)))

        xa=x0+cd*xd
        ya=y0+cd*yd
        za=z0+cd*zd
        eps=abs(za/xa)
        phia=atan2(ya*rlr/sqrt(xa**2+za**2),rrat**2)

        ra=re/sqrt((cos(phia))**2+(rrat*sin(phia))**2)
        xc=(xa/abs(xa))*ra*cos(phia)/sqrt(1.0+eps**2)
        zc=(za/abs(za))*eps*abs(xc)

        lambdaa=atan2(xc,zc)

        lambda_cor(i,j)=lambdaa
        phi_cor(i,j)=phia

     end do
  end do

end subroutine Parallax_Himawari_THap_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine Parallax_Himawari_THap_d( lambda_cld, phi_cld, h_cld,  &
  &                                  lambda_cor, phi_cor, undef )
! ひまわり 8 号の輝度温度データ (から換算した雲頂高度データ) をもとに
! lambda_cld, phi_cld での緯度経度を視差補正して lambda_cor, phi_cor に変換する.
! Tsukada and Horinouchi (2020) で用いられている近似版
  use Math_Const
  implicit none
  double precision, intent(in) :: lambda_cld(:,:)  ! ひまわり格子点経度 [rad]
  double precision, intent(in) :: phi_cld(size(lambda_cld,1),size(lambda_cld,2))
                                                   ! ひまわり格子点緯度 [rad]
  double precision, intent(in) :: h_cld(size(lambda_cld,1),size(lambda_cld,2))
                                                   ! ひまわり格子点での雲頂高度 [m]
  double precision, intent(inout) :: lambda_cor(size(lambda_cld,1),size(lambda_cld,2))
                                                   ! 視差補正後格子点経度 [rad]
  double precision, intent(inout) :: phi_cor(size(lambda_cld,1),size(lambda_cld,2))
                                                   ! 視差補正後格子点緯度 [rad]
  double precision, intent(in), optional :: undef  ! 雲頂高度の未定義値

  !-- ひまわり 8 号固有パラメータ
  double precision :: re=6378.1370d3
  double precision :: rp=6356.7523d3
  double precision :: rrat=1.00336409d0
  double precision :: hsat=42164.0d3
  double precision :: psat=0.0d0
  double precision :: lsat=140.7d0

  double precision :: x0, y0, z0, xs, ys, zs
  double precision :: h, lcld, pcld, lcldr, pcldr, lcldc, pcldc
  double precision :: d2r, r2d, e1, e2, e3, rl, cd
  double precision :: eps, xc, yc, zc, xd, yd, zd, rlr, ra
  double precision :: xa, ya, za, phia, lambdaa, rundef
  integer :: i, j, nx, ny

  d2r=pi_dp/180.0d0
!  r2d=180.0d0/pi_dp

  rrat=re/rp

  nx=size(lambda_cld,1)
  ny=size(lambda_cld,2)

  if(present(undef))then
     rundef=undef
  else
     rundef=-100.0d0
  end if

  do j=1,ny
     do i=1,nx

        pcldr=phi_cld(i,j)
        lcldr=lambda_cld(i,j)
        h=h_cld(i,j)
        rl=re/dsqrt(dcos(pcldr)**2+(rrat*dsin(pcldr))**2)

        x0=rl*dcos(pcldr)*dsin(lcldr)
        y0=rl*dsin(pcldr)
        z0=rl*dcos(pcldr)*dcos(lcldr)
        xs=hsat*dcos(psat*d2r)*dsin(lsat*d2r)
        ys=hsat*dsin(psat*d2r)
        zs=hsat*dcos(psat*d2r)*dcos(lsat*d2r)

        xd=xs-x0
        yd=ys-y0
        zd=zs-z0
        rlr=((re+h)/(rp+h))**2
        e1=xd**2+rlr*yd**2+zd**2
        e2=x0*xd+rlr*y0*yd+z0*zd
        e3=x0**2+rlr*y0**2+z0**2-(re+h)**2
        cd=(e2/e1)*(-1.0d0+dsqrt(1.0d0-e1*e3/(e2**2)))

        xa=x0+cd*xd
        ya=y0+cd*yd
        za=z0+cd*zd
        eps=dabs(za/xa)
        phia=datan2(ya*rlr/dsqrt(xa**2+za**2),rrat**2)

        ra=re/dsqrt((dcos(phia))**2+(rrat*dsin(phia))**2)
        xc=(xa/dabs(xa))*ra*dcos(phia)/dsqrt(1.0+eps**2)
        zc=(za/dabs(za))*eps*abs(xc)

        lambdaa=datan2(xc,zc)

        lambda_cor(i,j)=lambdaa
        phi_cor(i,j)=phia

     end do
  end do

end subroutine Parallax_Himawari_THap_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine Parallax_GOES16_d( lambda_cld, phi_cld, h_cld,  &
  &                           lambda_cor, phi_cor, undef )
! ひまわり 8 号の輝度温度データ (から換算した雲頂高度データ) をもとに
! lambda_cld, phi_cld での緯度経度を視差補正して lambda_cor, phi_cor に変換する.
  use Math_Const
  implicit none
  double precision, intent(in) :: lambda_cld(:,:)  ! ひまわり格子点経度 [rad]
  double precision, intent(in) :: phi_cld(size(lambda_cld,1),size(lambda_cld,2))
                                                   ! ひまわり格子点緯度 [rad]
  double precision, intent(in) :: h_cld(size(lambda_cld,1),size(lambda_cld,2))
                                                   ! ひまわり格子点での雲頂高度 [m]
  double precision, intent(inout) :: lambda_cor(size(lambda_cld,1),size(lambda_cld,2))
                                                   ! 視差補正後格子点経度 [rad]
  double precision, intent(inout) :: phi_cor(size(lambda_cld,1),size(lambda_cld,2))
                                                   ! 視差補正後格子点緯度 [rad]
  double precision, intent(in), optional :: undef  ! 雲頂高度の未定義値

  !-- ひまわり 8 号固有パラメータ
  double precision :: re=6378.1370d3
  double precision :: rp=6356.7523d3
  double precision :: rrat=1.00336409d0
  double precision :: hsat=42164.0d3
  double precision :: psat=0.0d0
  double precision :: lsat=-75.2d0

  double precision :: x0, y0, z0, xs, ys, zs
  double precision :: h, lcld, pcld, lcldr, pcldr, lcldc, pcldc
  double precision :: xx0, yy0, zz0, xxs, yys, zzs, xys0, x20, y20, z20, onemr
  double precision :: d2r, r2d, xyz0, gamcoe, gamcoe2, tparm
  double precision :: ztmp, zeps, parfunc, pardfunc, ztmpa, ztmpb, ztmpc
  double precision :: xa, ya, za, phia, lambdaa, rundef
  integer :: i, j, nx, ny

  d2r=pi_dp/180.0d0
!  r2d=180.0d0/pi_dp

  rrat=re/rp
  onemr=1.0d0-rrat**2

  nx=size(lambda_cld,1)
  ny=size(lambda_cld,2)

  if(present(undef))then
     rundef=undef
  else
     rundef=-100.0d0
  end if

  do j=1,ny
     do i=1,nx

        pcldr=datan2(dtan(phi_cld(i,j)),rrat**2)  ! geogra -> geocen
        lcldr=lambda_cld(i,j)
        h=h_cld(i,j)

        x0=rrat*dsin(0.5d0*pi_dp-pcldr)*dcos(lcldr)
        y0=rrat*dsin(0.5d0*pi_dp-pcldr)*dsin(lcldr)
        z0=dcos(0.5d0*pi_dp-pcldr)
        xs=(hsat/rp)*dcos(lsat*d2r)
        ys=(hsat/rp)*dsin(lsat*d2r)
        zs=0.0d0
        h=h/rp

        xx0=xs-x0
        yy0=ys-y0
        zz0=zs-z0
        xxs=x0*zs-z0*xs
        yys=y0*zs-z0*ys
        x20=xx0**2
        y20=yy0**2
        z20=zz0**2
        xys0=xxs*xx0+yys*yy0
        xyz0=x20+y20+z20

        if(h/=rundef.and.z0/=0.0d0)then

           ztmp=z0
           ztmpa=ztmp
           ztmpb=0.8d0*ztmp
           ztmpc=0.5d0*(ztmpa+ztmpb)
           parfunc=(((xxs+ztmpa*xx0)**2+(yys+ztmpa*yy0)**2  &
  &                 +(zz0*rrat*ztmpa)**2-(zz0*rrat)**2)  &
  &                *(1.0d0-onemr*(ztmpa**2))  &
  &                +((ztmpa*h)**2)*((rrat**2)*(x20+y20)+z20)-(h*zz0)**2)**2  &
  &               -4.0d0*((xys0*ztmpa+xyz0*(ztmpa**2)-z20)**2)  &
  &                *(1.0d0-onemr*(ztmpa**2))*((h*rrat)**2)
           pardfunc=(((xxs+ztmpc*xx0)**2+(yys+ztmpc*yy0)**2  &
  &                  +(zz0*rrat*ztmpc)**2-(zz0*rrat)**2)  &
  &                 *(1.0d0-onemr*(ztmpc**2))  &
  &                 +((ztmpc*h)**2)*((rrat**2)*(x20+y20)+z20)-(h*zz0)**2)**2  &
  &                -4.0d0*((xys0*ztmpc+xyz0*(ztmpc**2)-z20)**2)  &
  &                 *(1.0d0-onemr*(ztmpc**2))*((h*rrat)**2)
           do while (abs(ztmpa-ztmpb)>1.0d-12)
              if(parfunc*pardfunc<0.0d0)then
                 ztmpb=ztmpc
                 ztmpc=0.5d0*(ztmpa+ztmpb)
                 pardfunc=(((xxs+ztmpc*xx0)**2+(yys+ztmpc*yy0)**2  &
  &                        +(zz0*rrat*ztmpc)**2-(zz0*rrat)**2)  &
  &                       *(1.0d0-onemr*(ztmpc**2))  &
  &                       +((ztmpc*h)**2)*((rrat**2)*(x20+y20)+z20)-(h*zz0)**2)**2  &
  &                      -4.0d0*((xys0*ztmpc+xyz0*(ztmpc**2)-z20)**2)  &
  &                       *(1.0d0-onemr*(ztmpc**2))*((h*rrat)**2)
              else
                 ztmpa=ztmpc
                 ztmpc=0.5d0*(ztmpa+ztmpb)
                 parfunc=pardfunc
                 pardfunc=(((xxs+ztmpc*xx0)**2+(yys+ztmpc*yy0)**2  &
  &                        +(zz0*rrat*ztmpc)**2-(zz0*rrat)**2)  &
  &                       *(1.0d0-onemr*(ztmpc**2))  &
  &                       +((ztmpc*h)**2)*((rrat**2)*(x20+y20)+z20)-(h*zz0)**2)**2  &
  &                      -4.0d0*((xys0*ztmpc+xyz0*(ztmpc**2)-z20)**2)  &
  &                       *(1.0d0-onemr*(ztmpc**2))*((h*rrat)**2)
              end if
           end do
           ztmp=ztmpc

           za=ztmp
           gamcoe=1.0d0/dsqrt(1.0d0-onemr*(za**2))
           gamcoe2=((rrat+h*gamcoe)/(1.0d0+h*rrat*gamcoe))**2
           tparm=((x0*xx0+y0*yy0+z0*zz0*gamcoe2)/(x20+y20+z20*gamcoe2))  &
  &             *(-1.0d0+dsqrt(1.0d0+((x20+y20+z20*gamcoe2)  &
  &                                  /((x0*xx0+y0*yy0+z0*zz0*gamcoe2)**2))  &
  &                                  *((2.0d0+gamcoe*h*rrat)*gamcoe*h*rrat  &
  &                                   -z0*z0*(gamcoe2-rrat**2))))
           if(tparm<0.0d0.or.tparm>1.0d0)then
              tparm=((x0*xx0+y0*yy0+z0*zz0*gamcoe2)/(x20+y20+z20*gamcoe2))  &
  &                *(-1.0d0-dsqrt(1.0d0+((x20+y20+z20*gamcoe2)  &
  &                                     /((x0*xx0+y0*yy0+z0*zz0*gamcoe2)**2))  &
  &                                     *((2.0d0+gamcoe*h*rrat)*gamcoe*h*rrat  &
  &                                      -z0*z0*(gamcoe2-rrat**2))))
              if(tparm<0.0d0.or.tparm>1.0d0)then
                 tparm=rundef
                 lambda_cor(i,j)=rundef
                 phi_cor(i,j)=rundef
                 cycle
!                 write(*,*) "*** ERROR (Parallax_Himawari) ***: tparm is invalid."
              end if
           end if

!        if(zs-z0/=0.0d0)then
!           write(*,*) "tparm check", tparm, (-z0+za+(h*rrat*za)*gamcoe)/(zs-z0)
!           tparm=(-z0+za+(h*rrat*za)*gamcoe)/(zs-z0)
!        else
!           tparm=0.0d0
!        end if
           xa=((1.0d0-tparm)*x0+tparm*xs)/(1.0d0+(h/rrat)*gamcoe)
           ya=((1.0d0-tparm)*y0+tparm*ys)/(1.0d0+(h/rrat)*gamcoe)

           phia=datan2(dtan(0.5d0*pi_dp-dacos(za)),1.0d0/(rrat**2))  ! geocen -> geogra
           lambdaa=datan2(ya,xa)!*r2d

           lambda_cor(i,j)=lambdaa
           phi_cor(i,j)=phia

        else if(h/=rundef.and.z0==0.0d0)then

           tparm=((xys0-x20-y20)/((xs-x0)**2+(ys-y0)**2))  &
  &             *(-1.0d0+dsqrt(1.0d0+((xs-x0)**2+(ys-y0)**2)  &
  &                                  *(2.0*rrat*h+h**2)/((xys0-x20-y20)**2)))
           if(tparm<0.0d0.or.tparm>1.0d0)then
              tparm=((xys0-x20-y20)/((xs-x0)**2+(ys-y0)**2))  &
  &                *(-1.0d0-dsqrt(1.0d0+((xs-x0)**2+(ys-y0)**2)  &
  &                                     *(2.0*rrat*h+h**2)/((xys0-x20-y20)**2)))
              if(tparm<0.0d0.or.tparm>1.0d0)then
                 tparm=rundef
                 lambda_cor(i,j)=rundef
                 phi_cor(i,j)=rundef
                 cycle
!                 write(*,*) "*** ERROR (Parallax_Himawari) ***: tparm is invalid."
              end if
           end if

           xa=((1.0d0-tparm)*x0+tparm*xs)/(1.0d0+(h/rrat))
           ya=((1.0d0-tparm)*y0+tparm*ys)/(1.0d0+(h/rrat))

           lambdaa=datan2(ya,xa)!*r2d

           lambda_cor(i,j)=lambdaa
           phi_cor(i,j)=pcldr

        else

           tparm=rundef
           lambda_cor(i,j)=rundef
           phi_cor(i,j)=rundef

        end if

     end do
  end do

end subroutine Parallax_GOES16_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine DC_Satellite( x, y, fg, temp, search_dis, center, undef,  &
  &                      smon, sig, stdopt )
! Jaiswal and Kishtawal (2011,IEEE) に基づき, 衛星赤外放射輝度温度の水平分布
! をもとに, 台風中心を求めるルーチン.
  use Math_Const
  implicit none
  real, intent(in) :: x(:)  ! x 方向の座標 [m,deg]
  real, intent(in) :: y(:)  ! y 方向の座標 [m,deg]
  integer, intent(in) :: fg(2)
                         ! 中心点の第一推定値 (通常気圧の最低値等から得る)
                         ! fg(1) = x 方向の要素番号, fg(2) = y 方向の要素番号
  real, intent(in) :: temp(size(x),size(y))
                         ! 衛星観測された赤外放射輝度温度 [K]
  real, intent(in) :: search_dis  ! 検索する領域 (fg の位置を中心に) [m,deg]
                         ! 例えば, 100000.0 なら, fg を中心に縦横 100 km 四方
                         ! ここで与える数値の単位は x,y の単位と一致させること.
  integer, intent(inout) :: center(2)  ! 求めた中心点の各要素数
  real, intent(in), optional :: undef  ! 温度に未定義値がある場合, その未定義値.
                             ! 本ルーチンでは, 未定義値がある場合, 
                             ! その格子点のみ偏差計算に使用しない.
  integer, intent(in), optional :: smon  ! 温度偏差を求めるときの平均を
                                         ! とる領域格子数 (奇数設定).
                         ! default はオリジナル 20 x 20 km^2 相当の格子数.
  integer, intent(in), optional :: sig(2)  ! 原論文での中心修正用フィルタの sigma1, 2.
                             ! この値が指定されると, 原論文での最終修正処理を
                             ! 施す. デフォルトでは施さない.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)

  integer :: i, j, k, nx, ny, nxgmin, nxgmax, nygmin, nygmax
  integer :: ompnum, tmp_o_num, smnum, hsmnum, hsig, tmpi
  real, allocatable, dimension(:,:) :: dval, ddval
  integer, allocatable, dimension(:,:,:) :: dm
  real :: undeff, tmpa, tmpmin, tbmean, tbgaumean, tb2mean, gaut2mean
  real :: coe1, coe2, tempmin, tempmax
  real, allocatable, dimension(:,:) :: dtdx, dtdy, anom_temp
  real, allocatable, dimension(:,:) :: temp2, gaut, tbgau, gaut2
  logical :: stderr

!-- OpenMP 用整数関数
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, temp ),  &
  &                                     "DC_Satellite" )
  end if

  if(present(undef))then
     undeff=undef
  else
     undeff=-999.0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(present(smon))then
     smnum=smon
  else
     smnum=int(20.0e3/(x(2)-x(1)))
  end if
  if(mod(smnum,2)==0)then
     smnum=smnum+1
  end if
  hsmnum=(smnum-1)/2

!-- 検索格子点範囲を規程

  call interpo_search_2d( x, y, x(fg(1))-0.5*search_dis,  &
  &                             y(fg(2))-0.5*search_dis,  &
  &                       nxgmin, nygmin, undeff=int(undeff),  &
  &                       stdopt=stderr )

  call interpo_search_2d( x, y, x(fg(1))+0.5*search_dis,  &
  &                             y(fg(2))+0.5*search_dis,  &
  &                       nxgmax, nygmax, undeff=int(undeff),  &
  &                       stdopt=stderr )

  if(nxgmin==int(undeff).or.nxgmin<hsmnum+1)then  ! 領域外参照の場合の処理
     nxgmin=hsmnum+1
  end if
  if(nygmin==int(undeff).or.nygmin<hsmnum+1)then  ! 領域外参照の場合の処理
     nygmin=hsmnum+1
  end if
  if(nxgmax==int(undeff).or.nxgmax>nx-hsmnum-1)then  ! 領域外参照の場合の処理
     nxgmax=nx-hsmnum-1
  end if
  if(nygmax==int(undeff).or.nygmax>ny-hsmnum-1)then  ! 領域外参照の場合の処理
     nygmax=ny-hsmnum-1
  end if

  allocate(dtdx(nx,ny))
  allocate(dtdy(nx,ny))
  allocate(anom_temp(nx,ny))

  dtdx=0.0
  dtdy=0.0
  anom_temp=0.0

!-- openmp での条件付きコンパイル
!-- 接線平均アノマリの箇所を openmp 並列したいが,
!-- apres が inout 属性なので, private 属性を指定しないと
!-- thread ごとに apres の値が上書きされてしまう.
!-- そこで, threads number を参照した 3 次元配列にして,
!-- thread ごとに別の配列を使うように変更.

   ompnum=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP が有効の場合はここも有効.

  allocate(dval(nx,ny))
  allocate(ddval(nx,ny))
  allocate(dm(nx,ny,ompnum))

  dval=0.0
  dm=0

  tmp_o_num=1  ! OpenMP が有効でない場合, この値が apres の 3 次元目へ

!-- [1] 温度の偏差を求める.
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,tmpa)

  do j=nygmin-hsmnum,nygmax+hsmnum
     do i=hsmnum+1,nx-hsmnum
        call Mean_2d( temp(i-hsmnum:i+hsmnum,j-hsmnum:j+hsmnum), tmpa,  &
  &                   error=undeff )
        anom_temp(i,j)=abs(temp(i,j)-tmpa)
     end do
  end do

!$omp end do

!$omp barrier

!-- [2] 温度偏差から空間勾配を求める.

!$omp do schedule(runtime) private(i,j)

  do j=nygmin,nygmax
     do i=nxgmin,nxgmax
        dtdx(i,j)=anom_temp(i+hsmnum,j)-anom_temp(i-hsmnum,j)
        dtdy(i,j)=anom_temp(i,j+hsmnum)-anom_temp(i,j-hsmnum)
     end do
  end do

!$omp end do

!$omp barrier

!-- [3] 密度マトリクスを求める.

!$omp do schedule(runtime) private(i,j,tmp_o_num)

  do j=nygmin,nygmax
     do i=nxgmin,nxgmax
!$      tmp_o_num=OMP_GET_THREAD_NUM()+1
        ! OpenMP が有効の場合, この値が apres の 3 次元目へ
        call cross_line( x, y, dtdx(i,j), dtdy(i,j), x(i), y(j),  &
  &                      dm(:,:,tmp_o_num), stdopt=stderr )
     end do
  end do

!$omp end do
!$omp end parallel

  do j=nygmin,nygmax
     do i=nxgmin,nxgmax
        do k=1,ompnum
           dval(i,j)=dval(i,j)+real(dm(i,j,k))
        end do
     end do
  end do

!-- neighboring mean

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

  do j=nygmin+hsmnum,nygmax-hsmnum
     do i=nxgmin+hsmnum,nxgmax-hsmnum
        call Mean_2d( dval(i-hsmnum:i+hsmnum,j-hsmnum:j+hsmnum), ddval(i,j) )
     end do
  end do

!$omp end do
!$omp end parallel

!-- 計算した点のうち, 最大となる格子点を求める.
  call max_val_2d( ddval, center(1), center(2), tmpmin, undef=undeff )
  ddval=0.0

  if(present(sig))then
     hsig=(sig(1)+sig(2))/2
     coe1=(1.0/real(sig(1)))**2
     coe2=(1.0/real(sig(2)))**2

     allocate(gaut(2*hsig+1,2*hsig+1))
     allocate(temp2(nx,ny))
     allocate(tbgau(nx,ny))

     call max_val_2d( temp(center(1)-hsig:center(1)+hsig,center(2)-hsig:center(2)+hsig),  &
  &                   tmpi, tmpi, tempmax, undef=undeff )
     call min_val_2d( temp(center(1)-hsig:center(1)+hsig,center(2)-hsig:center(2)+hsig),  &
  &                   tmpi, tmpi, tempmin, undef=undeff )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=1,ny
        do i=1,nx
           if(temp(i,j)/=undeff)then
              temp2(i,j)=temp(i,j)**2
           else
              temp2(i,j)=undeff
           end if
        end do
     end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

     do j=1,2*hsig+1
        do i=1,2*hsig+1
           gaut(i,j)=tempmin+(tempmax-tempmin)  &
  &                 *exp(-real((i-hsig)**2+(j-hsig)**2)*coe1)  &
  &                  -(sig(1)**2)*coe2*exp(-0.5*real((i-hsig)**2+(j-hsig)**2)*coe2)
        end do
     end do

!$omp end do
!$omp end parallel

     ddval=undeff

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,tbmean,tb2mean,tbgaumean)

     do j=center(2)-hsig,center(2)+hsig
        do i=center(1)-hsig,center(1)+hsig
           call prod_priv_2d( temp(i-hsig:i+hsig,j-hsig:j+hsig),  &
  &                           gaut(1:2*hsig+1,1:2*hsig+1),  &
  &                           tbgau(i-hsig:i+hsig,j-hsig:j+hsig),  &
  &                           undef=undeff )
           call Mean_2d( temp(i-hsig:i+hsig,j-hsig:j+hsig), tbmean,  &
  &                      error=undeff )
           call Mean_2d( temp2(i-hsig:i+hsig,j-hsig:j+hsig), tb2mean,  &
  &                      error=undeff )
           call Mean_2d( tbgau(i-hsig:i+hsig,j-hsig:j+hsig), tbgaumean,  &
  &                      error=undeff )
           ddval(i,j)=1.0-tbgaumean/(tb2mean-tbmean**2)
if(ddval(i,j)>1.0)then
write(*,*) "Detect over 1.", tb2mean, tbgaumean
end if
        end do
     end do

!$omp end do
!$omp end parallel

write(*,*) "check center before", center(1:2), ddval(center(1),center(2))
!-- 計算した点のうち, 最大となる格子点を求める.
     call max_val_2d( ddval, center(1), center(2), tmpmin, undef=undeff )
write(*,*) "check center after", center(1:2), ddval(center(1),center(2))
  end if

  if(stderr.eqv..false.)then
     if(center(1)==nx+1.or.center(2)==ny+1)then
        write(*,*) "*** WARNING *** : DC_Braun (typhoon_analy)"
        write(*,*) "Setting the undef only point."
     end if
  end if

  deallocate(dtdx)
  deallocate(dtdy)
  deallocate(anom_temp)
  deallocate(dm)

end subroutine DC_Satellite

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine DCE_Yang( x, y, xc, yc, r, t, tbb, ljudge, tbbr, sig,  &
  &                  dmoat, dinner, douter, lcri1, lcri2, lcri3, lcri4,  &
  &                  undef, undefg, undefgc, stdopt, axis )
! Judge whether CE or not for the storm based on satellite images (Yang et al). 
  use Math_Const
  implicit none
  real, intent(in) :: x(:)  ! lon. or x coordinate [rad] or [m]
  real, intent(in) :: y(:)  ! lat. or x coordinate [rad] or [m]
  real, intent(in) :: r(:)  ! radial coordinate [m]
  real, intent(in) :: t(:)  ! azimuthal coordinate [rad]
  real, intent(in) :: xc    ! center position in x [m] or [rad].
  real, intent(in) :: yc    ! center position in y [m] or [rad].
  real, intent(in) :: tbb(size(x),size(y))
                            ! Tbb value on x-y space
  logical, intent(out) :: ljudge  ! .true. = CE, .false. = Not CE
  real, intent(inout), optional :: tbbr(size(r),8)
                            ! radial profiles of Tbb in all sections. 
  real, intent(inout), optional :: sig(size(r),8)  ! sigma of Tbb in all sections.
  real, intent(inout), optional :: dmoat        ! width of the moat
  real, intent(inout), optional :: dinner       ! distance of the inner eyewall
  real, intent(inout), optional :: douter       ! width of the outer eyewall
  logical, intent(inout), optional :: lcri1(8)  ! judge in Criterion (1)
  logical, intent(inout), optional :: lcri2(8)  ! judge in Criterion (2)
  logical, intent(inout), optional :: lcri3(8)  ! judge in Criterion (3)
  logical, intent(inout), optional :: lcri4     ! judge in Criterion (4)
  real, intent(in), optional :: undef  ! undefined value
  real, intent(in), optional :: undefg ! undefined value for insufficient area.
  character(3), intent(in), optional :: undefgc
                            ! methods in the presence of undefg. 
                            ! 'inc' = still averaged in the insuff. area. 
                            ! 'err' = just set undefg
  logical, intent(in), optional :: stdopt
                            ! stdout flag for undef (default: .false.). 
  character(2), intent(in), optional :: axis
                            ! type of the horizontal coordinates. 
                            ! 'xy' = Cartesian [m] (default)
                            ! 'll' = Lat-Lon [rad]

  integer, parameter :: isecn=8  ! total section number
  real, parameter :: runmean=10.0e3  ! Running mean distance [m]
  real, parameter :: r_crit=200.0e3  ! maximum radius in Criteria (1).
  real, parameter :: Tbb_crit=230.0  ! minimum Tbb in Criteria (3).
  integer, parameter :: Tsec_crit=5  ! minimum section number in Criteria (4).
  real, parameter :: di_crit=50.0e3  ! minimum distance in Criteria (5).

  integer :: nx, ny, nr, nt, ii, jj, ll, mm, nnr, irunmean
  integer :: icrit4
  integer, dimension(isecn) :: it_strt, it_endt, secmn, secmx
  integer, dimension(isecn) :: itbb_inmin, itbb_outmin, itbb_max
  integer, dimension(size(r),isecn) :: isecmn, isecmx
  real, dimension(size(r),size(t)) :: tmptbb, tmptbbrun
  real :: tmpt(size(t))
  real, dimension(size(r),isecn) :: tbbsecm, tbbsig
  real :: tmpr, tmp_anom, undeff, d2r, rmax, rmin, tmin, tmou
  character(2) :: ax
  character(3) :: undefgcflag
  logical :: stderr, lcrit4, lcrit5
  logical, dimension(isecn) :: lcrit1, lcrit2, lcrit3, lmin_tbb, lmax_tbb

  nx=size(x)
  ny=size(y)
  nr=size(r)
  nt=size(t)
  d2r=pi/180.0
  ljudge=.false.
  irunmean=int(runmean/(r(2)-r(1)))+1

!-- Check the outermost radius of 150 km

  call interpo_search_1d( r, r_crit, nnr )

  if(present(undef))then
     undeff=undef
  else
     undeff=999.0
  end if

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, tbb ),  &
  &                                     "DCE_Yang" )
  end if

  if(present(undefg))then
     if(present(undefgc))then
        undefgcflag=undefgc(1:3)
     else
        undefgcflag="inc"
     end if
  end if

  if(present(axis))then
     ax(1:2)=axis(1:2)
  else
     ax='xy'
  end if

  if(present(dmoat))then
     dmoat=undeff
  end if

  if(present(dinner))then
     dinner=undeff
  end if

  if(present(douter))then
     douter=undeff
  end if

!-- set 8 section and shift angle
  call interpo_search_1d( t, 360.0*(1.0-0.5/real(isecn))*d2r, it_strt(1) )
  call interpo_search_1d( t, 0.5*360.0/real(isecn)*d2r, it_endt(1) )

  do ii=it_strt(1),nt
     tmpt(ii-it_strt(1)+1)=t(ii)-2.0*pi
  end do
  do ii=1,it_strt(1)-1
     tmpt(ii+nt-it_strt(1)+1)=t(ii)
  end do

  it_strt(1:isecn)=(/(((ii-1)*(nt/isecn)+1),ii=1,isecn)/)
  it_endt(1:isecn-1)=it_strt(2:isecn)
  it_endt(isecn)=nt

!-- convert Cartesian Tbb to Annular Tbb.
!-- In option, do running mean of Tbb over runmean

  if(present(undefg))then
     call tangent_conv_scal_f( x, y, xc, yc, tbb, r, tmpt,  &
  &                            tmptbb, undef=undeff,  &
  &                            undefg=undefg, undefgc=undefgcflag(1:3),  &
  &                            stdopt=stderr, axis=ax )
     
  else
     call tangent_conv_scal_f( x, y, xc, yc, tbb, r, tmpt,  &
  &                            tmptbb, undef=undeff, stdopt=stderr, axis=ax )
  end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(ii)
  do ii=1,nt
     call Move_ave( tmptbb(1:nr,ii), irunmean, tmptbbrun(1:nr,ii), error=undeff )
  end do
!$omp end do
!$omp end parallel

  !-- irunmean 分オフセットになっているのを元に戻す.
  tmptbb(1:nr-irunmean+1,1:nt)=tmptbbrun(irunmean:nr,1:nt)

!-- Criteria (1) check local max and min of section-averaged Tbb in each section
!-- And, calculate standard deviation in each sec and radial bin.

  lcrit1=.false.

!$omp parallel default(shared)
!$omp do schedule(runtime) private(ii,jj)
  do ii=1,isecn
     do jj=1,nnr
        call Mean_1d( tmptbb(jj,it_strt(ii):it_endt(ii)), tbbsecm(jj,ii),  &
  &                   error=undeff )
        call stand_devi_1d( tmptbb(jj,it_strt(ii):it_endt(ii)), tbbsig(jj,ii),  &
  &                         error=undeff )
     end do
  end do
!$omp end do
!$omp end parallel

  do ii=1,isecn
     lmin_tbb(ii)=local_peak_checker_1df( tbbsecm(1:nnr,ii), '-',  &
  &                                       np=secmn(ii),  &
  &                                       ip=isecmn(1:nnr,ii),  &
  &                                       undef=undeff )
     lmax_tbb(ii)=local_peak_checker_1df( tbbsecm(1:nnr,ii), '+',  &
  &                                       np=secmx(ii),  &
  &                                       ip=isecmx(1:nnr,ii),  &
  &                                       undef=undeff )

     if(lmin_tbb(ii).eqv..true.)then
        if(secmn(ii)>1)then
           do jj=2,secmn(ii)
              if(lcrit1(ii).eqv..false.)then
                 do ll=1,secmx(ii)
                    if(isecmn(jj-1,ii)<isecmx(ll,ii).and.  &
  &                    isecmn(jj,ii)>isecmx(ll,ii))then
                       lcrit1(ii)=.true.
                       itbb_inmin(ii)=isecmn(jj-1,ii)
                       itbb_outmin(ii)=isecmn(jj,ii)
                       itbb_max(ii)=isecmx(ll,ii)
write(*,*) "local mn", itbb_inmin(ii), itbb_outmin(ii), tbbsecm(itbb_inmin(ii),ii), tbbsecm(itbb_outmin(ii),ii)
write(*,*) "local mx", itbb_max(ii), tbbsecm(itbb_max(ii),ii)
                       exit
                    end if
                 end do
              else
                 exit
              end if
           end do
        end if
     end if
  end do

!-- Criteria (2) check Tbb(max)>=Tbb(outmin)+sig(outmin), and
!                      Tbb(max)>=Tbb(inmin)+sig(inmin) in each section

  lcrit2=.false.

  do ii=1,isecn
     if(lcrit1(ii).eqv..true.)then
        tmin=tbbsecm(itbb_inmin(ii),ii)+tbbsig(itbb_inmin(ii),ii)
        tmou=tbbsecm(itbb_outmin(ii),ii)+tbbsig(itbb_outmin(ii),ii)
        if(tbbsecm(itbb_max(ii),ii)>=tmin.and.  &
  &        tbbsecm(itbb_max(ii),ii)>=tmou)then
           lcrit2(ii)=.true.
        end if
     end if
  end do

!-- Criteria (3) check Tbb(outmin)<=230K in true sections for (1) and (2)

  lcrit3=.false.

  do ii=1,isecn
     if((lcrit1(ii).eqv..true.).and.(lcrit2(ii).eqv..true.))then
        if(tbbsecm(itbb_outmin(ii),ii)<=Tbb_crit)then
           lcrit3(ii)=.true.
        end if
     end if
  end do

!-- Criteria (4) count number in true sections for (1), (2), and (3)

  lcrit4=.false.
  icrit4=0

  do ii=1,isecn
     if((lcrit1(ii).eqv..true.).and.(lcrit2(ii).eqv..true.).and.  &
  &     (lcrit3(ii).eqv..true.))then
        icrit4=icrit4+1
     end if
  end do

  if(icrit4>=Tsec_crit)then
     lcrit4=.true.
  end if

!-- Criteria (5) check distances outer Tbb min radii between two sections. 

  lcrit5=.false.

  if(lcrit4.eqv..true.)then
     do ii=1,isecn-1
        if((lcrit1(ii).eqv..true.).and.(lcrit2(ii).eqv..true.).and.  &
  &        (lcrit3(ii).eqv..true.))then
           do jj=ii+1,isecn
              if((lcrit1(jj).eqv..true.).and.(lcrit2(jj).eqv..true.).and.  &
  &              (lcrit3(jj).eqv..true.))then
                 if(abs(r(itbb_outmin(ii))-r(itbb_outmin(jj)))>di_crit)then
                    return
                 end if
              end if
           end do
        end if
     end do
     lcrit5=.true.
  end if

  if(lcrit5.eqv..true.)then

     if(present(dmoat))then
        dmoat=0.0
        do ii=1,isecn-1
           if((lcrit1(ii).eqv..true.).and.(lcrit2(ii).eqv..true.).and.  &
  &           (lcrit3(ii).eqv..true.))then
              do jj=itbb_max(ii),1,-1
                 if(tbbsecm(jj,ii)<0.5*tbbsig(itbb_inmin(ii),ii)  &
  &                               +tbbsecm(itbb_inmin(ii),ii))then
                    rmin=r(jj)
                    exit
                 end if
              end do
              do jj=itbb_max(ii),nnr
                 if(tbbsecm(jj,ii)<0.5*tbbsig(itbb_outmin(ii),ii)  &
  &                               +tbbsecm(itbb_outmin(ii),ii))then
                    rmax=r(jj)
                    exit
                 end if
              end do
              dmoat=dmoat+(rmax-rmin)
           end if
        end do
        dmoat=dmoat/real(icrit4)
     end if

     if(present(douter))then
        douter=0.0
        do ii=1,isecn-1
           if((lcrit1(ii).eqv..true.).and.(lcrit2(ii).eqv..true.).and.  &
  &           (lcrit3(ii).eqv..true.))then
              do jj=itbb_outmin(ii),1,-1
                 if(tbbsecm(jj,ii)>=0.5*tbbsig(itbb_outmin(ii),ii)  &
  &                               +tbbsecm(itbb_outmin(ii),ii))then
                    rmin=r(jj)
                    exit
                 end if
              end do
              do jj=itbb_outmin(ii),nnr
                 if(tbbsecm(jj,ii)>=0.5*tbbsig(itbb_outmin(ii),ii)  &
  &                               +tbbsecm(itbb_outmin(ii),ii))then
                    rmax=r(jj)
                    exit
                 end if
              end do
              douter=douter+(rmax-rmin)
           end if
        end do
        douter=douter/real(icrit4)
     end if

     if(present(dinner))then
        dinner=0.0
        do ii=1,isecn-1
           if((lcrit1(ii).eqv..true.).and.(lcrit2(ii).eqv..true.).and.  &
  &           (lcrit3(ii).eqv..true.))then
              do jj=2,nnr
                 if(tbbsecm(jj,ii)<=0.5*tbbsig(itbb_inmin(ii),ii)  &
  &                               +tbbsecm(itbb_inmin(ii),ii))then
                    rmax=r(jj)
                    exit
                 end if
              end do
              dinner=dinner+rmax
           end if
        end do
        dinner=dinner/real(icrit4)
     end if

     ljudge=.true.

  end if

  if(present(tbbr))then
     tbbr=tbbsecm
  end if

  if(present(tbbr))then
     sig=tbbsig
  end if

  if(present(lcri1))then
     lcri1=lcrit1
  end if

  if(present(lcri2))then
     lcri2=lcrit2
  end if

  if(present(lcri3))then
     lcri3=lcrit3
  end if

  if(present(lcri4))then
     lcri4=lcrit4
  end if

end subroutine DCE_Yang

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine DC_Sat_ZNCC_f( ref_img, img, swx, swy, undef_ref, undef_img )
! ゼロ平均正規化相互相関というテンプレートマッチング法を用いて,
! 衛星画像における台風中心を求める.
! 予め中心位置が判明している画像 (A) とそこから少し時間が変化した画像 (B) では
! 台風の水平構造はほぼ同じであると仮定する. すると, 画像 A と画像 B との違いは
! 時間変化によって位置が移動したこと以外はない. したがって, 画像 A を
! 参照画像としたテンプレートマッチング法により, 画像 B から画像 A と
! 最も類似する部分を抽出できる. この抽出された画像の南西端と画像 A の
! 南西端の緯度経度の差が画像 A, B 間の時間で台風が移動した距離と考えられる.
! [注意] この手法は, 画像 A, B 間での台風の水平構造が変化しないと仮定している
!        ため, 与える 2 画像間の時間変化量が可能な限り小さいことが望ましい.
!        ひまわり 8 号の機動観測 (2.5 分解像度) で用いることを想定している. 
! 具体的には, 
! 1. 中心位置が与えられている衛星画像を ref_img として与える.
!    データは放射輝度温度でもアルベド値でも構わない.
! 2. ref_img で与えられた衛星 2 次元分布と最も類似する img の南西端格子番号を
!    swx, swy として出力する.
! 3. ref_img の東西南北格子点数が nrx, nry とすると, 
!    img における (swx, swy) から (swx+nrx-1,swy+nry-1) までが,
!    ref_img と最も類似度の高い画像部分となる.
! 4. ref_img の南西端緯度経度と img の swx, swy の緯度経度の差が
!    ref_img と img の間での台風の移動ベクトルとなるので, ref_img の
!    台風中心の緯度経度にこの移動ベクトル分を足し合わせると, 
!    img での台風の中心緯度経度となる. 

  implicit none
  real, intent(in) :: ref_img(:,:)  ! 参照画像の任意数値 (K, 1, etc.)
  real, intent(in) :: img(:,:)      ! 探索画像の任意数値 (K, 1, etc.)
  integer, intent(inout) :: swx     ! ref_img と最も類似する img の西端格子
  integer, intent(inout) :: swy     ! ref_img と最も類似する img の南端格子
  real, intent(in), optional :: undef_ref  ! ref_img での未定義値
  real, intent(in), optional :: undef_img  ! img での未定義値

  integer :: i, j, k, nrx, nry, nix, niy, nx, ny
  real :: RCC(size(img,1)-size(ref_img,1)+1,size(img,2)-size(ref_img,2)+1)
  real :: iref_img(size(ref_img,1),size(ref_img,2))
  real :: tmpv, undef
  logical :: undeflag

  nrx=size(ref_img,1)
  nry=size(ref_img,2)
  nix=size(img,1)
  niy=size(img,2)
  nx=nix-nrx+1
  ny=niy-nry+1
  undef=-999.0

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nrx, nry, ref_img ),  &
  &                                     "DC_Sat_ZNCC" )
     call check_array_size_dmp_message( check_array_size_2d( nix, niy, img ),  &
  &                                     "DC_Sat_ZNCC" )
  end if

  if(nix<=nrx.or.niy<=nry)then
     write(*,*) "*** ERROR (DC_Sat_ZNCC) *** : img must be larger than ref_img."
     write(*,*) "Stop."
     stop
  end if

  do j=1,nry
     do i=1,nrx
        iref_img(i,j)=ref_img(i,j)
     end do
  end do

!-- 両者の未定義値を 1 つに統一する.
  undeflag=.true.
  if(present(undef_img))then
     undef=undef_img
     if(present(undef_ref))then
        do j=1,nry
           do i=1,nrx
              if(iref_img(i,j)==undef_ref)then
                 iref_img(i,j)=undef
              end if
           end do
        end do
     end if
  else
     if(present(undef_ref))then
        undef=undef_ref
     else
        undeflag=.false.   ! 未定義を持っていない.
     end if
  end if

!-- 本計算開始

!$omp parallel default(shared)

  if(undeflag.eqv..true.)then

!$omp do schedule(runtime) private(i,j)

     do j=1,ny
        do i=1,nx
           call Cor_Coe_2d( iref_img(1:nrx,1:nry), img(i:nrx+i-1,j:nry+j-1),  &
  &                         RCC(i,j), error=undef )
        end do
     end do

!$omp end do

  else

!$omp do schedule(runtime) private(i,j)

     do j=1,ny
        do i=1,nx
           call Cor_Coe_2d( iref_img(1:nrx,1:nry), img(i:nrx+i-1,j:nry+j-1),  &
  &                         RCC(i,j) )
        end do
     end do

!$omp end do

  end if

!$omp end parallel

  if(undeflag.eqv..true.)then
     call max_val_2d( RCC, swx, swy, tmpv, undef=undef )
  else
     call max_val_2d( RCC, swx, swy, tmpv )
  end if

end subroutine DC_Sat_ZNCC_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine DC_Sat_ZNCC_d( ref_img, img, swx, swy, undef_ref, undef_img )
! ゼロ平均正規化相互相関というテンプレートマッチング法を用いて,
! 衛星画像における台風中心を求める.
! 予め中心位置が判明している画像 (A) とそこから少し時間が変化した画像 (B) では
! 台風の水平構造はほぼ同じであると仮定する. すると, 画像 A と画像 B との違いは
! 時間変化によって位置が移動したこと以外はない. したがって, 画像 A を
! 参照画像としたテンプレートマッチング法により, 画像 B から画像 A と
! 最も類似する部分を抽出できる. この抽出された画像の南西端と画像 A の
! 南西端の緯度経度の差が画像 A, B 間の時間で台風が移動した距離と考えられる.
! [注意] この手法は, 画像 A, B 間での台風の水平構造が変化しないと仮定している
!        ため, 与える 2 画像間の時間変化量が可能な限り小さいことが望ましい.
!        ひまわり 8 号の機動観測 (2.5 分解像度) で用いることを想定している. 
! 具体的には, 
! 1. 中心位置が与えられている衛星画像を ref_img として与える.
!    データは放射輝度温度でもアルベド値でも構わない.
! 2. ref_img で与えられた衛星 2 次元分布と最も類似する img の南西端格子番号を
!    swx, swy として出力する.
! 3. ref_img の東西南北格子点数が nrx, nry とすると, 
!    img における (swx, swy) から (swx+nrx-1,swy+nry-1) までが,
!    ref_img と最も類似度の高い画像部分となる.
! 4. ref_img の南西端緯度経度と img の swx, swy の緯度経度の差が
!    ref_img と img の間での台風の移動ベクトルとなるので, ref_img の
!    台風中心の緯度経度にこの移動ベクトル分を足し合わせると, 
!    img での台風の中心緯度経度となる. 

  implicit none
  double precision, intent(in) :: ref_img(:,:)  ! 参照画像の任意数値 (K, 1, etc.)
  double precision, intent(in) :: img(:,:)      ! 探索画像の任意数値 (K, 1, etc.)
  integer, intent(inout) :: swx     ! ref_img と最も類似する img の西端格子
  integer, intent(inout) :: swy     ! ref_img と最も類似する img の南端格子
  double precision, intent(in), optional :: undef_ref  ! ref_img での未定義値
  double precision, intent(in), optional :: undef_img  ! img での未定義値

  integer :: i, j, k, nrx, nry, nix, niy, nx, ny
  double precision :: RCC(size(img,1)-size(ref_img,1)+1,size(img,2)-size(ref_img,2)+1)
  double precision :: iref_img(size(ref_img,1),size(ref_img,2))
  double precision :: tmpv, undef
  logical :: undeflag

  nrx=size(ref_img,1)
  nry=size(ref_img,2)
  nix=size(img,1)
  niy=size(img,2)
  nx=nix-nrx+1
  ny=niy-nry+1
  undef=-999.0d0

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nrx, nry, ref_img ),  &
  &                                     "DC_Sat_ZNCC" )
     call check_array_size_dmp_message( check_array_size_2d( nix, niy, img ),  &
  &                                     "DC_Sat_ZNCC" )
  end if

  if(nix<=nrx.or.niy<=nry)then
     write(*,*) "*** ERROR (DC_Sat_ZNCC) *** : img must be larger than ref_img."
     write(*,*) "Stop."
     stop
  end if

  iref_img=ref_img

!-- 両者の未定義値を 1 つに統一する.
  undeflag=.true.
  if(present(undef_img))then
     undef=undef_img
     if(present(undef_ref))then
        do j=1,nry
           do i=1,nrx
              if(iref_img(i,j)==undef_ref)then
                 iref_img(i,j)=undef
              end if
           end do
        end do
     end if
  else
     if(present(undef_ref))then
        undef=undef_ref
     else
        undeflag=.false.   ! 未定義を持っていない.
     end if
  end if

!-- 本計算開始

!$omp parallel default(shared)

  if(undeflag.eqv..true.)then

!$omp do schedule(runtime) private(i,j)

     do j=1,ny
        do i=1,nx
           call Cor_Coe_2d( iref_img(1:nrx,1:nry), img(i:nrx+i-1,j:nry+j-1),  &
  &                         RCC(i,j), error=undef )
        end do
     end do

!$omp end do

  else

!$omp do schedule(runtime) private(i,j)

     do j=1,ny
        do i=1,nx
           call Cor_Coe_2d( iref_img(1:nrx,1:nry), img(i:nrx+i-1,j:nry+j-1),  &
  &                         RCC(i,j) )
        end do
     end do

!$omp end do

  end if

!$omp end parallel

  if(undeflag.eqv..true.)then
     call max_val_2d( RCC, swx, swy, tmpv, undef=undef )
  else
     call max_val_2d( RCC, swx, swy, tmpv )
  end if

end subroutine DC_Sat_ZNCC_d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine radar_pl2rz_f( phi_e, h, Rl, r, z )
! 与えられた仰角, レーダービーム距離からレーダーからの距離と高度を計算する.
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: phi_e  ! 仰角 [rad]
  real, intent(in) :: h     ! レーダー設置高度 [m]
  real, intent(in) :: Rl    ! レーダービームの距離 [m]
  real, intent(out) :: r    ! レーダーからの距離 [m]
  real, intent(out) :: z    ! レーダーからの高度 [m]
  real :: x, y, theta, phid, cost

  phid=0.5*pi-phi_e
  x=radius+h+Rl*cos(phid)
  y=Rl*sin(phid)
  theta=atan2(y,x)
  cost=cos(theta)
  r=x*theta/cost
  z=x/cost-radius

end subroutine radar_pl2rz_f

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine radar_pl2rz_d( phi_e, h, Rl, r, z )
! 与えられた仰角, レーダービーム距離からレーダーからの距離と高度を計算する.
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: phi_e  ! 仰角 [rad]
  double precision, intent(in) :: h     ! レーダー設置高度 [m]
  double precision, intent(in) :: Rl    ! レーダービームの距離 [m]
  double precision, intent(out) :: r    ! レーダーからの距離 [m]
  double precision, intent(out) :: z    ! レーダーからの高度 [m]
  double precision :: x, y, theta, phid, cost

  phid=0.5d0*pi_dp-phi_e
  x=radius_dp+h+Rl*dcos(phid)
  y=Rl*dsin(phid)
  theta=datan2(y,x)
  cost=dcos(theta)
  r=x*theta/cost
  z=x/cost-radius_dp

end subroutine radar_pl2rz_d

!--------------------------------------------------------------
!--------------------------------------------------------------

real function radar_pz2r_f( phi_e, h, z )
! 与えられた仰角, 高度からレーダーからの距離を計算する.
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: phi_e  ! 仰角 [rad]
  real, intent(in) :: h     ! レーダー設置高度 [m]
  real, intent(in) :: z     ! レーダーからの高度 [m]
  real :: theta, phid

  phid=0.5*pi-phi_e
  theta=phid+asin(-((radius+h)/(radius+z))*sin(phid))
  radar_pz2r_f=(radius+z)*theta

  return
end function radar_pz2r_f

!--------------------------------------------------------------
!--------------------------------------------------------------

double precision function radar_pz2r_d( phi_e, h, z )
! 与えられた仰角, 高度からレーダーからの距離を計算する.
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: phi_e  ! 仰角 [rad]
  double precision, intent(in) :: h     ! レーダー設置高度 [m]
  double precision, intent(in) :: z     ! レーダーからの高度 [m]
  double precision :: theta, phid

  phid=0.5d0*pi_dp-phi_e
  theta=phid+dasin(-((radius_dp+h)/(radius_dp+z))*dsin(phid))
  radar_pz2r_d=(radius_dp+z)*theta

  return
end function radar_pz2r_d

!--------------------------------------------------------------
!--------------------------------------------------------------

real function radar_rz2p_f( h, r, z )
! 与えられた距離, 高度からレーダービームの仰角を計算する.
  use Phys_Const
  use Math_Const
  implicit none
  real, intent(in) :: h     ! レーダー設置高度 [m]
  real, intent(in) :: r     ! レーダーからの距離 [m]
  real, intent(in) :: z     ! レーダーからの高度 [m]
  real :: theta

  theta=r/(radius+z)
  if(sin(theta)/=0.0)then
     radar_rz2p_f=atan((cos(theta)-(radius+h)/(radius+z))/sin(theta))
  else
     radar_rz2p_f=0.5*pi
  end if

  return
end function radar_rz2p_f

!--------------------------------------------------------------
!--------------------------------------------------------------

double precision function radar_rz2p_d( h, r, z )
! 与えられた距離, 高度からレーダービームの仰角を計算する.
  use Phys_Const
  use Math_Const
  implicit none
  double precision, intent(in) :: h     ! レーダー設置高度 [m]
  double precision, intent(in) :: r     ! レーダーからの距離 [m]
  double precision, intent(in) :: z     ! レーダーからの高度 [m]
  double precision :: theta

  theta=r/(radius_dp+z)
  if(dsin(theta)/=0.0d0)then
     radar_rz2p_d=datan((dcos(theta)-(radius_dp+h)/(radius_dp+z))/dsin(theta))
  else
     radar_rz2p_d=0.5d0*pi_dp
  end if

  return
end function radar_rz2p_d

!--------------------------------------------------------------
!  以下, private ルーチン
!--------------------------------------------------------------

subroutine search_region_1d( x, y, c, r, nr, stdopt )
! 接線平均可能な半径を計算するルーチン
  implicit none
  real, intent(in) :: x(2)  ! x 方向の両端座標 [m]
  real, intent(in) :: y(2)  ! y 方向の両端座標 [m]
  real, intent(in) :: c(2)  ! 円筒中心の位置座標 (x,y) [m]
  real, intent(in) :: r(:)  ! 動径方向の位置座標 [m]
  integer, intent(inout) :: nr  ! 平均可能半径の要素番号 (r(nr) が可能半径)
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  integer :: nrr, tmp_nr
  real :: xc, yc
  logical :: stderr

  nrr=size(r)
  xc=c(1)
  yc=c(2)
  nr=nrr

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  if(abs(x(1)-xc) < r(nrr))then
     if(stderr.eqv..false.)then
        write(*,*) "typhoon_analy WARNING :"
        write(*,*) "|x(1)-xc| >= rmax. "
        write(*,*) "undef value is substituted out of region."
     end if
     call interpo_search_1d( r, abs(x(1)-xc), tmp_nr, stdopt=stderr )
     nr=tmp_nr+1  ! interpo_search は abs の値より小さい r の要素番号が入るため.
                  ! 以下も同じ理由
  else
     if(abs(x(2)-xc) < r(nrr))then
        if(stderr.eqv..false.)then
           write(*,*) "typhoon_analy WARNING :"
           write(*,*) "|x(nx)-xc| >= rmax. "
           write(*,*) "undef value is substituted out of region."
        end if
        call interpo_search_1d( r, abs(x(2)-xc), tmp_nr, stdopt=stderr )
        if(tmp_nr+1<nr)then
           nr=tmp_nr+1
        end if
     else
        if(abs(y(1)-yc) < r(nrr))then
           if(stderr.eqv..false.)then
              write(*,*) "typhoon_analy WARNING :"
              write(*,*) "|y(1)-yc| >= rmax. "
              write(*,*) "undef value is substituted out of region."
           end if
           call interpo_search_1d( r, abs(y(1)-yc), tmp_nr, stdopt=stderr )
           if(tmp_nr+1<nr)then
              nr=tmp_nr+1
           end if
        else
           if(abs(y(2)-yc) < r(nrr))then
              if(stderr.eqv..false.)then
                 write(*,*) "typhoon_analy WARNING :"
                 write(*,*) "|y(ny)-yc| >= rmax. "
                 write(*,*) "undef value is substituted out of region."
              end if
              call interpo_search_1d( r, abs(y(2)-yc), tmp_nr, stdopt=stderr )
              if(tmp_nr+1<nr)then
                 nr=tmp_nr+1
              end if
           end if
        end if
     end if
  end if

end subroutine search_region_1d

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine cross_line( x, y, dx, dy, x0, y0, counter, stdopt )
! 任意のベクトルに平行な直線がどの格子点上を通過しているかをカウントする.
  implicit none
  real, dimension(:), intent(in) :: x  ! x 方向座標
  real, dimension(:), intent(in) :: y  ! y 方向座標
  real, intent(in) :: dx  ! x 方向の傾き
  real, intent(in) :: dy  ! y 方向の傾き
  real, intent(in) :: x0  ! 直線の開始点 x 座標
  real, intent(in) :: y0  ! 直線の開始点 y 座標
  integer, dimension(size(x),size(y)), intent(inout) :: counter
                    ! 計算には dy/dx という相対値が用いられるので,
                    ! dx, dy の単位は気にしなくてよい.
                    ! x0, y0 は両端に直線を伸ばす.
  logical, intent(in), optional :: stdopt  ! 探索範囲が見つからない旨の標準出力を表示させないようにする.
                                           ! default では .false. (表示させる)
  integer :: i, j, nx, ny, itmp, jtmp, ix, iy
  integer, dimension(size(x),size(y)) :: dummy
  real :: dt, tmpx, tmpy
  logical :: stderr

  nx=size(x)
  ny=size(y)
  dummy=0

  if(present(stdopt))then
     stderr=stdopt
  else
     stderr=.false.
  end if

  call nearest_search_2d( x, y, x0, y0, ix, iy )

  if(dx/=0.0.or.dy/=0.0)then
     if(dx==0.0)then
        counter(ix,1:ny)=1
     else if(dy==0.0)then
        counter(1:nx,iy)=1
     else
        dt=dy/dx
     !-- まず, x 方向に 1 格子点ずつ直線を伸ばし, その各 x 点のときの y 点に
     !-- 対応する点をカウントしていく.
        do i=1,nx
           tmpy=y0+dt*(x(i)-x0)
           call interpo_search_1d( y, tmpy, jtmp, stdopt=stderr )
           if(jtmp>0)then
              if(tmpy/=y(jtmp).and.jtmp<ny)then  ! 2 格子点間に存在したときは
                 ! 隣接 2 格子点を両方カウントする.
                 dummy(i,jtmp)=1
                 dummy(i,jtmp+1)=1
                 counter(i,jtmp)=counter(i,jtmp)+1
                 counter(i,jtmp+1)=counter(i,jtmp+1)+1
              else
                 counter(i,jtmp)=counter(i,jtmp)+1
              end if
           end if
        end do
        dt=dx/dy
     !-- 次に, y 方向に 1 格子点ずつ直線を伸ばし, その各 y 点のときの x 点に
     !-- 対応する点をカウントしていく.
     !-- ダブルカウントをしないために, counter = 1 の場合はカウントしない.
        do j=1,ny
           tmpx=x0+dt*(y(j)-y0)
           call interpo_search_1d( x, tmpx, itmp, stdopt=stderr )
           if(itmp>0)then
              if(dummy(itmp,j)==0)then
                 if(tmpx/=x(itmp).and.itmp<nx)then
                    counter(itmp,j)=counter(itmp,j)+1
                    if(dummy(itmp+1,j)==0)then
                       counter(itmp+1,j)=counter(itmp+1,j)+1
                    end if
                 else
                    counter(itmp,j)=counter(itmp,j)+1
                 end if
              end if
           end if
        end do
     end if
  end if

end subroutine cross_line

!--------------------------------------------------------------
!--------------------------------------------------------------

subroutine prod_priv_2d( ival1, ival2, oval, undef )
! 2 次元配列の各成分について, 差の絶対値 2 乗を返す.
  implicit none
  real, dimension(:,:), intent(in) :: ival1  ! 変数 1
  real, dimension(size(ival1,1),size(ival1,2)), intent(in) :: ival2  ! 変数 2
  real, dimension(size(ival1,1),size(ival1,2)), intent(inout) :: oval   ! 出力結果
  real, intent(in), optional :: undef
  integer :: nx, ny, i, j

  nx=size(ival1,1)
  ny=size(ival1,2)

  if(present(undef))then
     do j=1,ny
        do i=1,nx
           if(ival1(i,j)/=undef.and.ival2(i,j)/=undef)then
              oval(i,j)=(ival1(i,j)-ival2(i,j))**2
           else
              oval(i,j)=undef
           end if
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           oval(i,j)=(ival1(i,j)-ival2(i,j))**2
        end do
     end do
  end if

end subroutine prod_priv_2d

!--------------------------------------------------------------
!--------------------------------------------------------------

logical function undef_checker_1df( val, undef )
! 任意配列 val について, すべての要素について undef という値が入っているか
! どうかをチェックする. 1 つでも undef が入っていれば .true. を返す.
  implicit none
  real, dimension(:), intent(in) :: val  ! チェックする配列
  real, intent(in) :: undef  ! チェックする変数値
  integer :: i, nx
  logical :: checker

  nx=size(val)
  checker=.false.

  do i=1,nx
     if(val(i)==undef)then
        checker=.true.
        exit
     end if
  end do

  undef_checker_1df=checker

  return
end function undef_checker_1df

!--------------------------------------------------------------
!--------------------------------------------------------------

logical function undef_checker_1dd( val, undef )
! 任意配列 val について, すべての要素について undef という値が入っているか
! どうかをチェックする. 1 つでも undef が入っていれば .true. を返す.
  implicit none
  double precision, dimension(:), intent(in) :: val  ! チェックする配列
  double precision, intent(in) :: undef  ! チェックする変数値
  integer :: i, nx
  logical :: checker

  nx=size(val)
  checker=.false.

  do i=1,nx
     if(val(i)==undef)then
        checker=.true.
        exit
     end if
  end do

  undef_checker_1dd=checker

  return
end function undef_checker_1dd

!--------------------------------------------------------------
!--------------------------------------------------------------

logical function undef_checker_2df( val, undef )
! 任意配列 val について, すべての要素について undef という値が入っているか
! どうかをチェックする. 1 つでも undef が入っていれば .true. を返す.
  implicit none
  real, dimension(:,:), intent(in) :: val  ! チェックする配列
  real, intent(in) :: undef  ! チェックする変数値
  integer :: i, nx
  logical :: checker

  nx=size(val,2)
  checker=.false.

  do i=1,nx
     checker=undef_checker_1df( val(:,i), undef )
     if(checker.eqv..true.)then
        exit
     end if
  end do

  undef_checker_2df=checker

  return
end function undef_checker_2df

!--------------------------------------------------------------
!--------------------------------------------------------------

logical function undef_checker_2dd( val, undef )
! 任意配列 val について, すべての要素について undef という値が入っているか
! どうかをチェックする. 1 つでも undef が入っていれば .true. を返す.
  implicit none
  double precision, dimension(:,:), intent(in) :: val  ! チェックする配列
  double precision, intent(in) :: undef  ! チェックする変数値
  integer :: i, nx
  logical :: checker

  nx=size(val,2)
  checker=.false.

  do i=1,nx
     checker=undef_checker_1dd( val(:,i), undef )
     if(checker.eqv..true.)then
        exit
     end if
  end do

  undef_checker_2dd=checker

  return
end function undef_checker_2dd

!--------------------------------------------------------------
!--------------------------------------------------------------

logical function undef_checker_3df( val, undef )
! 任意配列 val について, すべての要素について undef という値が入っているか
! どうかをチェックする. 1 つでも undef が入っていれば .true. を返す.
  implicit none
  real, dimension(:,:,:), intent(in) :: val  ! チェックする配列
  real, intent(in) :: undef  ! チェックする変数値
  integer :: i, nx
  logical :: checker

  nx=size(val,3)
  checker=.false.

  do i=1,nx
     checker=undef_checker_2df( val(:,:,i), undef )
     if(checker.eqv..true.)then
        exit
     end if
  end do

  undef_checker_3df=checker

  return
end function undef_checker_3df

!--------------------------------------------------------------
!--------------------------------------------------------------

logical function undef_checker_3dd( val, undef )
! 任意配列 val について, すべての要素について undef という値が入っているか
! どうかをチェックする. 1 つでも undef が入っていれば .true. を返す.
  implicit none
  double precision, dimension(:,:,:), intent(in) :: val  ! チェックする配列
  double precision, intent(in) :: undef  ! チェックする変数値
  integer :: i, nx
  logical :: checker

  nx=size(val,3)
  checker=.false.

  do i=1,nx
     checker=undef_checker_2dd( val(:,:,i), undef )
     if(checker.eqv..true.)then
        exit
     end if
  end do

  undef_checker_3dd=checker

  return
end function undef_checker_3dd

!--------------------------------------------------------------
!--------------------------------------------------------------

!subroutine Rangular_moment(xp,yp,x,y,u,v,mome)
! 任意の点まわりの相対角運動量を計算するルーチン
!
! 本当は 3 次元ベクトルで計算されるが, 気象学では 3 次元量はあまり需要がない
! であろうという判断から, ある回転軸まわりの角運動量成分のみを
! 計算することにしている.
!
! 主目的は台風の中心軸を中心に鉛直軸まわりの角運動量を計算することを目的とする.
!
! $$M=rv,\quad r=中心軸からの距離, \quad v=風速の同位角成分$$
!
! 位置と風速に緯度の変換を与えれば, 全球での自転軸まわりの角運動量を
! 計算することも可能.
! ベクトルの外積計算ルーチン vec_prod_2d を用いることで極座標でも計算可能.
!  use algebra
!  implicit none
!  real, intent(in) :: x(:)  ! x 方向の位置座標
!  real, intent(in) :: y(:)  ! y 方向の位置座標
!  real, intent(in) :: xp  ! 回転軸の x 位置座標
!  real, intent(in) :: yp  ! 回転軸の y 位置座標
!  real, intent(in) :: u(size(x),size(y))  ! 位置 i,j での風速の 1 成分
!  real, intent(in) :: v(size(x),size(y))  ! 位置 i,j での風速の 1 成分
!  real, intent(inout) :: mome(size(x),size(y))  ! 回転軸まわりの相対角運動量
!  real :: xxp(size(x),size(y),1)  ! x,y,xp,yp から計算した軸中心からの位置ベクトル x 成分
!  real :: yyp(size(x),size(y),1)  ! x,y,xp,yp から計算した軸中心からの位置ベクトル y 成分
!  integer :: i, j, nx, ny
!  real :: tmp(size(x),size(y),1)
! 計算することも可能.
! ベクトルの外積計算ルーチン vec_prod_2d を用いることで極座標でも計算可能.
!  use algebra
!  implicit none
!  real, intent(in) :: x(:)  ! x 方向の位置座標
!  real, intent(in) :: y(:)  ! y 方向の位置座標
!  real, intent(in) :: xp  ! 回転軸の x 位置座標
!  real, intent(in) :: yp  ! 回転軸の y 位置座標
!  real, intent(in) :: u(size(x),size(y))  ! 位置 i,j での風速の 1 成分
!  real, intent(in) :: v(size(x),size(y))  ! 位置 i,j での風速の 1 成分
!  real, intent(inout) :: mome(size(x),size(y))  ! 回転軸まわりの相対角運動量
!  real :: xxp(size(x),size(y),1)  ! x,y,xp,yp から計算した軸中心からの位置ベクトル x 成分
!  real :: yyp(size(x),size(y),1)  ! x,y,xp,yp から計算した軸中心からの位置ベクトル y 成分
!  integer :: i, j, nx, ny
!  real :: tmp(size(x),size(y),1)
!
!  nx=size(x)
!  ny=size(y)
!
!!$omp parallel do shared(xxp,yyp,x,y,xp,yp) private(i,j)
!  do j=1,ny
!     do i=1,nx
!        xxp(i,j,1)=x(i)-xp
!        yyp(i,j,1)=y(j)-yp
!     end do
!  end do
!!$omp end parallel do
!
!  tmp=0.0
!  call vec_prod_2d(xxp,yyp,tmp,u,v,tmp,tmp,tmp,mome)
!
!end subroutine Rangular_moment
!
!
!subroutine Aangular_moment(xp,yp,x,y,u,v,f,mome)
!! 任意の点まわりの絶対角運動量を計算するルーチン
!!
!! 主目的は台風の中心軸を中心に鉛直軸まわりの角運動量を計算することを目的とする.
!!
!! $$M=rv+\dfrac{fr^2}{2} ,\quad r=中心軸からの距離, \quad v=風速の同位角成分$$
!!
!! 位置と風速に緯度の変換を与えれば, 全球での自転軸まわりの角運動量を
!! 計算することも可能.
!!
!! ベクトルの外積計算ルーチン vec_prod_2d を用いることで極座標でも計算可能.
!  use algebra
!  implicit none
!  real, intent(in) :: x(:)  ! x 方向の位置座標
!  real, intent(in) :: y(:)  ! y 方向の位置座標
!  real, intent(in) :: xp  ! 回転軸の x 位置座標
!  real, intent(in) :: yp  ! 回転軸の y 位置座標
!  real, intent(in) :: u(size(x),size(y))  ! 位置 i,j での風速の 1 成分
!  real, intent(in) :: v(size(x),size(y))  ! 位置 i,j での風速の 1 成分
!  real, intent(in) :: f(size(x),size(y))  ! 位置 i,j でのコリオリパラメータ
!  real, intent(inout) :: mome(size(x),size(y))  ! 回転軸まわりの相対角運動量
!  real :: xxp(size(x),size(y),1)  ! x,y,xp,yp から計算した軸中心からの位置ベクトル x 成分
!  real :: yyp(size(x),size(y),1)  ! x,y,xp,yp から計算した軸中心からの位置ベクトル y 成分
!  integer :: i, j, nx, ny
!  real :: tmp(size(x),size(y),1), rp(size(x),size(y)), tmp1(1)
!
!  nx=size(x)
!  ny=size(y)
!
!!$omp parallel do shared(xxp,yyp,x,y,xp,yp) private(i,j)
!  do j=1,ny
!     do i=1,nx
!        xxp(i,j,1)=x(i)-xp
!        yyp(i,j,1)=y(j)-yp
!     end do
!  end do
!!$omp end parallel do
!
!  tmp=0.0
!  call vec_prod_2d(xxp,yyp,tmp,u,v,tmp,tmp,tmp,mome)
!  call radius(xp,yp,0.0,x,y,tmp1,rp)
!
!!$omp parallel do shared(mome,f,rp) private(i,j)
!  do j=1,ny
!     do i=1,nx
!        mome(i,j)=mome(i,j)+0.5*f(i,j)*rp(i,j)**2
!     end do
!  end do
!!$omp end parallel do
!
!end subroutine Aangular_moment

!-----------------------------------------
!-----------------------------------------

!-----------------------------------------
!-----------------------------------------

end module typhoon_analy
