module sub_mod

contains

!subroutine type_convert_r2d( ival, oval )
!  implicit none
!  real, intent(in) :: ival(:,:)
!  double precision, intent(out) :: oval(size(ival,1),size(ival,2))
!  integer :: ix, jy, ii, jj
!
!  ix=size(ival,1)
!  jy=size(ival,2)
!
!  do jj=1,jy
!     do ii=1,ix
!        oval(ii,jj)=dble(ival(ii,jj))
!     end do
!  end do
!
!end subroutine type_convert_r2d

!subroutine type_convert_r2cd( ival, oval, imval )
!  implicit none
!  real, intent(in) :: ival(:,:)
!  complex(kind(0d0)), intent(out) :: oval(size(ival,1),size(ival,2))
!  real, intent(in), optional :: imval(size(ival,1),size(ival,2))
!  integer :: ix, jy, ii, jj
!  complex(kind(0d0)) :: img_unit
!
!  ix=size(ival,1)
!  jy=size(ival,2)
!
!  img_unit=(0.0d0,1.0d0)
!
!  if(present(imval))then
!     do jj=1,jy
!        do ii=1,ix
!           oval(ii,jj)=dble(ival(ii,jj))+img_unit*dble(imval(ii,jj))
!        end do
!     end do
!  else
!     do jj=1,jy
!        do ii=1,ix
!           oval(ii,jj)=dble(ival(ii,jj))
!        end do
!     end do
!  end if
!
!end subroutine type_convert_r2cd

!subroutine type_convert_d2cd( ival, oval, imval )
!  implicit none
!  double precision, intent(in) :: ival(:,:)
!  complex(kind(0d0)), intent(out) :: oval(size(ival,1),size(ival,2))
!  double precision, intent(in), optional :: imval(size(ival,1),size(ival,2))
!  integer :: ix, jy, ii, jj
!  complex(kind(0d0)) :: img_unit
!
!  ix=size(ival,1)
!  jy=size(ival,2)
!
!  img_unit=(0.0d0,1.0d0)
!
!  if(present(imval))then
!     do jj=1,jy
!        do ii=1,ix
!           oval(ii,jj)=ival(ii,jj)+img_unit*imval(ii,jj)
!        end do
!     end do
!  else
!     do jj=1,jy
!        do ii=1,ix
!           oval(ii,jj)=ival(ii,jj)
!        end do
!     end do
!  end if
!
!end subroutine type_convert_d2cd

subroutine type_convert_d2r( ival, oval )
  implicit none
  double precision, intent(in) :: ival(:,:)
  real, intent(out) :: oval(size(ival,1),size(ival,2))
  integer :: ix, jy, ii, jj

  ix=size(ival,1)
  jy=size(ival,2)

  do jj=1,jy
     do ii=1,ix
        oval(ii,jj)=real(ival(ii,jj))
     end do
  end do

end subroutine type_convert_d2r

subroutine rearrange_rxy2ryx( ix, jy, rvalxy, rvalyx )
!-- rearranging real(nx,ny) array to real(ny,nx)
  implicit none
  integer, intent(in) :: ix
  integer, intent(in) :: jy
  double precision, intent(in) :: rvalxy(ix,jy)
  double precision, intent(inout) :: rvalyx(jy,ix)
  integer :: j

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j)

  do j=1,jy
     rvalyx(j,1:ix)=rvalxy(1:ix,j)
  end do

!$omp end do
!$omp end parallel

end subroutine rearrange_rxy2ryx


subroutine rearrange_ryx2rxy( ix, jy, rvalyx, rvalxy )
!-- rearranging real(ny,nx) array to real(nx,ny)
  implicit none
  integer, intent(in) :: ix
  integer, intent(in) :: jy
  double precision, intent(in) :: rvalyx(jy,ix)
  double precision, intent(inout) :: rvalxy(ix,jy)
  integer :: j

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j)

  do j=1,jy
     rvalxy(1:ix,j)=rvalyx(j,1:ix)
  end do

!$omp end do
!$omp end parallel

end subroutine rearrange_ryx2rxy


subroutine rearrange_ryx2cxy( ix, jy, rvalyx, cvalxy )
!-- rearranging real(-ny:ny,-nx:nx) array 
!--          to complex(1:2*nx+1,1:2*ny+1)
  use Math_Const
  implicit none
  integer, intent(in) :: ix
  integer, intent(in) :: jy
  double precision, intent(in) :: rvalyx(-jy:jy,-ix:ix)
  complex(kind(0d0)), intent(inout) :: cvalxy(1:2*ix+1,1:2*jy+1)
  integer :: i, j

  cvalxy(1,1)=rvalyx(0,0)

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

  do i=1,ix
     cvalxy(i+1,1)=rvalyx(0,i)+img_cdp*rvalyx(0,-i)
     cvalxy(2*ix-i+2,1)=dconjg(cvalxy(i+1,1))
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(j)

  do j=1,jy
     cvalxy(1,j+1)=rvalyx(j,0)+img_cdp*rvalyx(-j,0)
     cvalxy(1,2*jy-j+2)=dconjg(cvalxy(1,j+1))
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

  do j=1,jy
     do i=1,ix
        cvalxy(i+1,j+1)=rvalyx(j,i)+img_cdp*rvalyx(-j,-i)
        cvalxy(2*ix-i+2,2*jy-j+2)=dconjg(cvalxy(i+1,j+1))
        cvalxy(i+1,2*jy-j+2)=rvalyx(-j,i)+img_cdp*rvalyx(j,-i)
        cvalxy(2*ix-i+2,j+1)=dconjg(cvalxy(i+1,2*jy-j+2))
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine rearrange_ryx2cxy


subroutine rearrange_cxy2ryx( ix, jy, cvalxy, rvalyx )
!-- rearranging complex(1:2*nx+1,1:2*ny+1) array 
!--          to real(-ny:ny,-nx:nx)
  implicit none
  integer, intent(in) :: ix
  integer, intent(in) :: jy
  complex(kind(0d0)), intent(in) :: cvalxy(1:2*ix+1,1:2*jy+1)
  double precision, intent(inout) :: rvalyx(-jy:jy,-ix:ix)
  integer :: i, j

  rvalyx(0,0)=cvalxy(1,1)

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

  do i=1,ix
     rvalyx(0,i)=dble(cvalxy(i+1,1))
     rvalyx(0,-i)=dimag(cvalxy(i+1,1))
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(j)

  do j=1,jy
     rvalyx(j,0)=dble(cvalxy(1,j+1))
     rvalyx(-j,0)=dimag(cvalxy(1,j+1))
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

  do j=1,jy
     do i=1,ix
        rvalyx(j,i)=dble(cvalxy(i+1,j+1))
        rvalyx(-j,-i)=dimag(cvalxy(i+1,j+1))
        rvalyx(-j,i)=dble(cvalxy(i+1,2*jy-j+2))
        rvalyx(j,-i)=dimag(cvalxy(i+1,2*jy-j+2))
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine rearrange_cxy2ryx


subroutine all_clear()
  use savegloval_define
  use saveval_define
  implicit none

  ITJR=0
  ITIR=0
  ITJJ=0
  ITIJ=0
  TJR=0.0d0
  TIR=0.0d0
  TJJ=0.0d0
  TIJ=0.0d0

  zi=0.0d0

!  psid=0.0
!  ud=0.0
!  vd=0.0
!  zd=0.0
  tmpr=0.0d0
  tmpi=0.0d0
  
!  ur=0.0d0
!  vr=0.0d0
  zor=0.0d0
!  psior=0.0d0
  psiko=0.0d0
  zko=0.0d0
  psikn=0.0d0
  pk=0.0d0
  uko_sbl=0.0d0
  vko_sbl=0.0d0
  ukn_sbl=0.0d0
  vkn_sbl=0.0d0
  nbm_opt=0.0d0
  sbl_optu=0.0d0
  sbl_optv=0.0d0

  xi=0.0d0
  yi=0.0d0
  xd=0.0
  yd=0.0
  x=0.0d0
  y=0.0d0
  xj=0.0d0
  yj=0.0d0
!ORG  t=0.0d0

  u_nbm_isp=0.0d0
  v_nbm_isp=0.0d0
!  zor_isp=0.0d0
!  psior_isp=0.0d0
!  ur_isp=0.0d0
!  vr_isp=0.0d0
!  zko_isp=0.0d0
!  psiko_isp=0.0d0
!  uk_isp=0.0d0
!  vk_isp=0.0d0
!  tmp_work=0.0d0

end subroutine all_clear

end module sub_mod
