!---------------------------------------------------------------
! Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!---------------------------------------------------------------

module Basis  ! 基本的な計算関数を実行するモジュール

  integer, save :: check_array_size_iflag=0

  integer, parameter :: i8=selected_int_kind(16)

  type dtime  ! 開始日の日付
     integer :: year_d  ! 西暦
     integer :: month_d  ! 月
     integer :: day_d  ! 日
     integer :: hour_d  ! 時
     integer :: min_d  ! 分
     integer :: sec_d  ! 秒
  end type dtime

  type dptime  ! 開始日の日付
     integer(kind=i8) :: year_d  ! 西暦
     integer(kind=i8) :: month_d  ! 月
     integer(kind=i8) :: day_d  ! 日
     integer(kind=i8) :: hour_d  ! 時
     integer(kind=i8) :: min_d  ! 分
     integer(kind=i8) :: sec_d  ! 秒
  end type dptime

  public  :: check_array_size_1d,  &
  &          check_array_size_i1_1d,  &
  &          check_array_size_i2_1d,  &
  &          check_array_size_i4_1d,  &
  &          check_array_size_f4_1d,  &
  &          check_array_size_d8_1d

  public  :: check_array_size_2d,  &
  &          check_array_size_i1_2d,  &
  &          check_array_size_i2_2d,  &
  &          check_array_size_i4_2d,  &
  &          check_array_size_f4_2d,  &
  &          check_array_size_d8_2d

  public  :: check_array_size_3d,  &
  &          check_array_size_i1_3d,  &
  &          check_array_size_i2_3d,  &
  &          check_array_size_i4_3d,  &
  &          check_array_size_f4_3d,  &
  &          check_array_size_d8_3d

  interface check_array_size_1d
     module procedure check_array_size_i1_1d,  &
  &                   check_array_size_i2_1d,  &
  &                   check_array_size_i4_1d,  &
  &                   check_array_size_f4_1d,  &
  &                   check_array_size_d8_1d
  end interface check_array_size_1d

  interface check_array_size_2d
     module procedure check_array_size_i1_2d,  &
  &                   check_array_size_i2_2d,  &
  &                   check_array_size_i4_2d,  &
  &                   check_array_size_f4_2d,  &
  &                   check_array_size_d8_2d
  end interface check_array_size_2d

  interface check_array_size_3d
     module procedure check_array_size_i1_3d,  &
  &                   check_array_size_i2_3d,  &
  &                   check_array_size_i4_3d,  &
  &                   check_array_size_f4_3d,  &
  &                   check_array_size_d8_3d
  end interface check_array_size_3d

  interface counter_day
     module procedure counter_day_i4,  &
  &                   counter_day_i8
  end interface counter_day

  interface counter_sec
     module procedure counter_sec_i4,  &
  &                   counter_sec_i8
  end interface counter_sec

  interface time_zone_convert
     module procedure time_zone_convert_i4,  &
  &                   time_zone_convert_i8
  end interface time_zone_convert

  interface sec_convert
     module procedure sec_convert_i4,  &
  &                   sec_convert_i8
  end interface sec_convert

  interface check_leap_year
     module procedure check_leap_year_i4,  &
  &                   check_leap_year_i8
  end interface check_leap_year

contains

subroutine rand_make(seed,L,output)
  ! 任意の桁数で擬似乱数を生成するサブルーチン
  ! 混合合同法という乱数生成アルゴリズムを用いて擬似乱数を生成
  ! $x_{n+1}=a\times x_{n}+b (mod \; L)$
  implicit none
  integer, intent(in) :: seed  ! 生成する乱数の種初期値
  integer, intent(in) :: L  ! 出力する最大桁数 + 1 の数値
  integer, intent(inout) :: output  ! 出力される乱数
  integer :: a, b, x0, i, input
!  integer, external :: time

  input=seed
!  input=time()
  input=mod(input,L)
  a=11
  b=12
  x0=input

  do i=1,10
     x0=a*x0+b
     x0=mod(x0,L)
  end do

  output=x0

end subroutine

!-----------------------------------
!-----------------------------------

subroutine multi_rand_make(seed,L,n,interval,output)
  ! rand_make を用いて任意の個数の乱数を生成する.
  implicit none
  integer, intent(in) :: seed  ! 生成する乱数の種初期値
  integer, intent(in) :: L  ! 出力する最大桁数 + 1 の数値
  integer, intent(in) :: n  ! 生成する乱数の個数
  integer, intent(in) :: interval  ! 乱数の種の間隔
  integer, intent(inout) :: output(n)  ! 出力される乱数
  integer :: a, b, x0, i, j, input, itmp
!  integer, external :: time

  itmp=0

  do j=1,n
     if(j==1)then
        itmp=seed
!        itmp=time()
     else
        itmp=itmp+interval
     end if

     input=mod(itmp,L)
     a=11
     b=12
     x0=input

     do i=1,10
        x0=a*x0+b
        x0=mod(x0,L)
     end do

     output(j)=x0

  end do

end subroutine

!-----------------------------------
!-----------------------------------

character(100) function r2c_convert( rval, forma )
! 実数型を文字型に変換する
  implicit none
  real, intent(in) :: rval  ! 変換する実数
  character(*), intent(in), optional :: forma  ! 指定するフォーマット
  character(100) :: tmp

  if(present(forma))then
     write(tmp,trim(forma)) rval
  else
     write(tmp,*) rval
  end if

  r2c_convert=tmp

  return
end function

!-----------------------------------
!-----------------------------------

real function c2r_convert( cval )
! 文字型を実数型に変換する
  implicit none
  character(*), intent(in) :: cval  ! 変換する文字

  read(cval,*) c2r_convert

  return
end function

!-----------------------------------
!-----------------------------------

double precision function c2d_convert( cval )
! 文字型を倍精度実数型に変換する
  implicit none
  character(*), intent(in) :: cval  ! 変換する文字

  read(cval,*) c2d_convert

  return
end function

!-----------------------------------
!-----------------------------------

character(100) function i2c_convert( ival, forma )
! 実数型を文字型に変換する
  implicit none
  integer, intent(in) :: ival  ! 変換する整数
  character(*), intent(in), optional :: forma  ! 指定するフォーマット
  character(100) :: tmp

  if(present(forma))then
     write(tmp,trim(forma)) ival
  else
     write(tmp,*) ival
  end if

  i2c_convert=tmp

  return
end function

!-----------------------------------
!-----------------------------------

integer function c2i_convert( cval )
! 文字型を実数型に変換する
  implicit none
  character(*), intent(in) :: cval  ! 変換する文字

  read(cval,*) c2i_convert

  return
end function

!-----------------------------------
!-----------------------------------

integer function split_num( cval, split_str )
! split_str で指定された文字列を区分け記号として, 文字列 cval を分割したときの
! 分割個数を返す. split_str が指定されない場合, 空白文字を分割記号として処理.
! 現在, 分割記号は 1 文字のみ対応.
  implicit none
  character(*), intent(in) :: cval  ! 分割したい文字列
  character(1), intent(in), optional :: split_str  ! 分割記号
  character(1) :: split
  integer :: nc, counter, i
  logical :: double_flag

  if(present(split_str))then
     split=split_str
  else
     split=' '
  end if

  nc=len_trim(adjustl(cval))
  counter=0
  double_flag=.false.

  do i=1,nc
     if(cval(i:i)==split)then
        counter=counter+1
     end if
  end do

  split_num=counter+1

  return
end function

!-----------------------------------
!-----------------------------------

subroutine splitting( cval, num, cval_ar, split_str )
! cval を split_str を区分け文字として cval_ar という配列に分割する.
! 現在, split_str は 1 文字にのみ対応している. デフォルトでは半角スペースが
! 対応している.
  implicit none
  character(*), intent(in) :: cval  ! 分割したい文字列
  integer, intent(in) :: num        ! num 個に分割.
                                    ! この値は先に split_num 関数で調べておく.
  character(*), dimension(num), intent(inout) :: cval_ar
                                    ! 分割された文字列が格納される.
  character(1), intent(in), optional :: split_str
  character(1) :: split
  integer :: nc, counter, i
  integer, dimension(num) :: isnum, ienum
  logical :: double_flag

  if(present(split_str))then
     split=split_str
  else
     split=' '
  end if

  nc=len(cval)
  double_flag=.false.

  if(cval(1:1)==split)then
     isnum(1)=-1
     ienum(1)=0
     isnum(2)=2
     counter=2
  else
     isnum(1)=1
     counter=1
  end if

  do i=2,nc-1  ! 文字列の最初と最後に分割文字が入っていてもそれは無視できるため.
     if(cval(i:i)==split)then
        ienum(counter)=i-1
        counter=counter+1
        isnum(counter)=i+1
     end if
  end do

  if(counter>num)then
     write(*,*) "*** ERROR (basis:splitting) *** : counter exceeds num. Stop."
     stop
  end if

  if(cval(nc:nc)==split)then
     ienum(counter)=nc-1
  else
     ienum(num)=nc
  end if

  do i=1,num
     if(isnum(i)>ienum(i))then
        cval_ar(i)=''
     else
        cval_ar(i)=cval(isnum(i):ienum(i))
     end if
  end do

end subroutine splitting

!-----------------------------------
!-----------------------------------

subroutine gsub( cval, oval, bc, ac, mode )
! cval の中における, 文字列 bc を文字列 ac に置き換える.
! ac と bc の文字数が一致している必要はないが, bc > ac の場合は左詰めで
! cval の文字数が少なくなる. 逆に ac > bc の場合は文字数が多くなる.
! 引数 mode は置き換え時のオプション設定.
  implicit none
  character(*), intent(in) :: cval ! 置き換えたい文字列
  character(*), intent(inout) :: oval ! 置き換えた結果の文字列
  character(*), intent(in) :: bc    ! 置き換える前の文字列
  character(*), intent(in) :: ac    ! 置き換えた後の文字列
  integer, intent(in), optional :: mode  ! 1 = デフォルト (1 回置き換え)
                                         ! 2 = 置き換え後の全文字列に置き換え
                                         ! 前の文字列がまだ残っている場合, 再度
                                         ! 置き換えを行い, 全くなくなるまで
                                         ! 処理を繰り返す.
  character(10000) :: tmpc
  integer :: nc, bcounter, acounter, nbc, nac, noc, imode
  logical :: mode_flag, loopflag

  if(present(mode))then
     imode=mode
  else
     imode=1
  end if
  tmpc=''

  nc=len(cval)
  nbc=len(bc)
  nac=len(ac)

  oval=""
  bcounter=1
  acounter=1

  do while (bcounter<=nc)
     if(bcounter+nbc-1<=nc)then  ! 末尾が元の文字列を超えているか
        if(cval(bcounter:bcounter+nbc-1)==bc(1:nbc))then
           oval(acounter:acounter+nac-1)=ac(1:nac)
           acounter=acounter+nac
           bcounter=bcounter+nbc
        else
           oval(acounter:acounter)=cval(bcounter:bcounter)
           acounter=acounter+1
           bcounter=bcounter+1
        end if
     else  ! 超えていればその時点以降に置き換え文字は存在しない.
        oval(acounter:acounter+nc-bcounter+1)=cval(bcounter:nc)
        acounter=acounter+nc-bcounter+2
        bcounter=nc+1
        noc=acounter-1
        exit
     end if
  end do

  select case (imode)
  case (2)   ! 複数回チェックしてなくなるまで置き換え

     tmpc=''
     loopflag=.true.

     do while (loopflag.eqv..true.)

        nc=noc
        nbc=len(bc)
        nac=len(ac)

        tmpc=""
        acounter=1
        bcounter=1

        do while (bcounter<=nc)
           if(bcounter+nbc-1<=nc)then  ! 末尾が元の文字列を超えているか
              if(oval(bcounter:bcounter+nbc-1)==bc(1:nbc))then
                 tmpc(acounter:acounter+nac-1)=ac(1:nac)
                 acounter=acounter+nac
                 bcounter=bcounter+nbc
              else
                 tmpc(acounter:acounter)=oval(bcounter:bcounter)
                 acounter=acounter+1
                 bcounter=bcounter+1
              end if
           else  ! 超えていればその時点以降に置き換え文字は存在しない.
              tmpc(acounter:acounter+nc-bcounter+1)=oval(bcounter:nc)
              acounter=acounter+nc-bcounter+2
              bcounter=nc+1
              noc=acounter-1
              exit
           end if
        end do

        if(oval(1:len(oval))==tmpc(1:len(tmpc)))then
           loopflag=.false.
        end if

        oval=tmpc(1:noc)

     end do

  end select

  return

end subroutine gsub

!-----------------------------------
!-----------------------------------

integer function counter_day_i4( stime, etime )
! 開始日から終了日までの日数をカウントする.
  implicit none
  type(dtime), intent(in) :: stime  ! 開始日
  type(dtime), intent(in) :: etime  ! 終了日
  integer, parameter, dimension(13) :: month=(/31,28,31,30,31,30,  &
  &                                            31,31,30,31,30,31,29/)
  integer :: nt, nm, days, i, year_tmp, year_fact
  integer :: nsy, nsm, nsd, ney, nem, ned

  nt=etime%year_d-stime%year_d
  nsy=stime%year_d
  ney=etime%year_d

  nm=etime%month_d-stime%month_d
  days=0
  nsm=stime%month_d
  nsd=stime%day_d
  nem=etime%month_d
  ned=etime%day_d

!-- etime%year_d と stime%year_d の間に 1 年以上間が空く場合,
!-- その間は年単位で日数を足す. (閏年で場合分けする.)
  if(nt>1)then
     do i=nsy+1,ney-1
        if(check_leap_year(i).eqv..true.)then
           year_fact=366
        else
           year_fact=365
        end if
        days=days+year_fact
     end do
  end if

!-- 上の処理を行っているので, あとは終了年の日数を足し,
!-- 開始年で開始日までをひくのみ.
!-- まず, 1 ケ月まるまる存在する月を足し合わせる.
!-- ただし, 開始日が 12 月, 終了日が 1 月の場合は, 日を足す過程で足されるので,
!-- ここではカウントしない.
  if(nsy<ney)then  ! 年がまたがる場合
     ! 終了日の足し合わせ
     if(nem>1)then  ! 終了日が 1 月ではない場合.
        do i=1,nem-1
           if((i==2).and.(check_leap_year(ney).eqv..true.))then
              days=days+month(13)
           else
              days=days+month(i)
           end if
        end do
     end if

     do i=nsm,12
        if((i==2).and.(check_leap_year(nsy).eqv..true.))then
           days=days+month(13)
        else
           days=days+month(i)
        end if
     end do

  else

     if(nm>0)then  ! 年がまたがらず, 月がまたがる場合.
        do i=nsm,nem-1
           if((i==2).and.(check_leap_year(ney).eqv..true.))then
              days=days+month(13)
           else
              days=days+month(i)
           end if
        end do
     end if
  end if

  days=days+ned-nsd

  days=days+1
  counter_day_i4=days

  return
end function

!-----------------------------------
!-----------------------------------

integer(kind=i8) function counter_day_i8( stime, etime )
! 開始日から終了日までの日数をカウントする.
  implicit none
  type(dptime), intent(in) :: stime  ! 開始日
  type(dptime), intent(in) :: etime  ! 終了日
  integer(kind=i8), parameter, dimension(13) :: month=(/31,28,31,30,  &
  &                                               31,30,31,31,  &
  &                                               30,31,30,31,  &
  &                                               29/)
  integer(kind=i8) :: nt, nm, days, i, year_tmp, year_fact
  integer(kind=i8) :: nsy, nsm, nsd, ney, nem, ned

  nt=etime%year_d-stime%year_d
  nsy=stime%year_d
  ney=etime%year_d

  nm=etime%month_d-stime%month_d
  days=0
  nsm=stime%month_d
  nsd=stime%day_d
  nem=etime%month_d
  ned=etime%day_d

!-- etime%year_d と stime%year_d の間に 1 年以上間が空く場合,
!-- その間は年単位で日数を足す. (閏年で場合分けする.)
  if(nt>1)then
     do i=nsy+1,ney-1
        if(check_leap_year(i).eqv..true.)then
           year_fact=366
        else
           year_fact=365
        end if
        days=days+year_fact
     end do
  end if

!-- 上の処理を行っているので, あとは終了年の日数を足し,
!-- 開始年で開始日までをひくのみ.
!-- まず, 1 ケ月まるまる存在する月を足し合わせる.
!-- ただし, 開始日が 12 月, 終了日が 1 月の場合は, 日を足す過程で足されるので,
!-- ここではカウントしない.
  if(nsy<ney)then  ! 年がまたがる場合
     ! 終了日の足し合わせ
     if(nem>1)then  ! 終了日が 1 月ではない場合.
        do i=1,nem-1
           if((i==2).and.(check_leap_year(ney).eqv..true.))then
              days=days+month(13)
           else
              days=days+month(i)
           end if
        end do
     end if

     do i=nsm,12
        if((i==2).and.(check_leap_year(nsy).eqv..true.))then
           days=days+month(13)
        else
           days=days+month(i)
        end if
     end do

  else

     if(nm>0)then  ! 年がまたがらず, 月がまたがる場合.
        do i=nsm,nem-1
           if((i==2).and.(check_leap_year(ney).eqv..true.))then
              days=days+month(13)
           else
              days=days+month(i)
           end if
        end do
     end if
  end if

  days=days+ned-nsd

  days=days+1
  counter_day_i8=days

  return
end function

!-----------------------------------
!-----------------------------------

integer function counter_sec_i4( stime, etime )
! 開始日から終了日までの秒数をカウントする.
  implicit none
  type(dtime), intent(in) :: stime  ! 開始日時
  type(dtime), intent(in) :: etime  ! 終了日時
  integer :: nday, tmp_sec
  integer :: nsh, nsm, nss, neh, nem, nes

  tmp_sec=0

!-- まず, 日数をカウントする.

  nday=counter_day_i4( stime, etime )

!-- 日数を元に, 日分の秒をカウント.
! counter_day は 1 日から計算されているから
! 同一日では counter_day = 1, 24 時間以内で日をまたぐ場合は counter_day = 2.

  if(nday>2)then
     tmp_sec=(nday-2)*86400
  end if

!-- stime から etime までの hour, minite, sec を計算して, トータルの秒を返す.

  nsh=stime%hour_d
  nsm=stime%min_d
  nss=stime%sec_d
  neh=etime%hour_d
  nem=etime%min_d
  nes=etime%sec_d

  if(nday==1)then  ! 日数が 1 日以内に存在する場合, hour からのみ計算.
     tmp_sec=tmp_sec+(neh-nsh)*3600+(nem-nsm)*60+(nes-nss)
  else  ! 日数がまたがっている場合は, stime の日を 86400 s でカウントしてから
        ! nsh, nsm, nss を引けばその日の秒が計算できる.
     tmp_sec=tmp_sec+86400  ! stime の 1 日分
     tmp_sec=tmp_sec+neh*3600+nem*60+nes  ! etime の日の秒を足す
     tmp_sec=tmp_sec-nsh*3600-nsm*60-nss  ! stime の日の秒を引く
  end if

  counter_sec_i4=tmp_sec

  return
end function

!-----------------------------------
!-----------------------------------

integer(kind=i8) function counter_sec_i8( stime, etime )
! 開始日から終了日までの秒数をカウントする.
  implicit none
  type(dptime), intent(in) :: stime  ! 開始日時
  type(dptime), intent(in) :: etime  ! 終了日時
  integer(kind=i8) :: nday, tmp_sec
  integer(kind=i8) :: nsh, nsm, nss, neh, nem, nes

  tmp_sec=0

!-- まず, 日数をカウントする.

  nday=counter_day_i8( stime, etime )

!-- 日数を元に, 日分の秒をカウント.
! counter_day は 1 日から計算されているから
! 同一日では counter_day = 1, 24 時間以内で日をまたぐ場合は counter_day = 2.

  if(nday>2)then
     tmp_sec=(nday-2)*86400
  end if

!-- stime から etime までの hour, minite, sec を計算して, トータルの秒を返す.

  nsh=stime%hour_d
  nsm=stime%min_d
  nss=stime%sec_d
  neh=etime%hour_d
  nem=etime%min_d
  nes=etime%sec_d

  if(nday==1)then  ! 日数が 1 日以内に存在する場合, hour からのみ計算.
     tmp_sec=tmp_sec+(neh-nsh)*3600+(nem-nsm)*60+(nes-nss)
  else  ! 日数がまたがっている場合は, stime の日を 86400 s でカウントしてから
        ! nsh, nsm, nss を引けばその日の秒が計算できる.
     tmp_sec=tmp_sec+86400  ! stime の 1 日分
     tmp_sec=tmp_sec+neh*3600+nem*60+nes  ! etime の日の秒を足す
     tmp_sec=tmp_sec-nsh*3600-nsm*60-nss  ! stime の日の秒を引く
  end if

  counter_sec_i8=tmp_sec

  return
end function

!-----------------------------------
!-----------------------------------

subroutine time_zone_convert_i4( factor, itime, ctime )
! タイムゾーンの変換を行うルーチン
! 例えば, JST を UTC に変換する場合, factor = -9 とすればよい.
  implicit none
  integer, intent(in) :: factor        ! 何時間変化するか.
  type(dtime), intent(in) :: itime     ! 変換前の時刻
  type(dtime), intent(inout) :: ctime  ! 変換後の時刻
  integer :: iyear, imonth, iday, ihour
  integer, parameter, dimension(13) :: month=(/31,28,31,30,31,30,  &
  &                                            31,31,30,31,30,31,29/)

  iyear=itime%year_d
  imonth=itime%month_d
  iday=itime%day_d
  ihour=itime%hour_d

  ihour=ihour+factor

  if(ihour<0)then
     do while(ihour<0)
        iday=iday-1
        ihour=ihour+24
     end do
  else if(ihour>=24)then
     do while(ihour>=24)
        iday=iday+1
        ihour=ihour-24
     end do
  end if

  if(iday<1)then
     do while(iday<1)
        if((check_leap_year(iyear).eqv..true.).and.(imonth==2))then
           iday=iday+month(13)
        else
           iday=iday+month(imonth)
        end if

        imonth=imonth-1

        if(imonth<=0)then
           iyear=iyear-1
           imonth=12
        end if
     end do
  else if(iday>month(imonth))then
     do while(iday>month(imonth))
        if((check_leap_year(iyear).eqv..true.).and.(imonth==2))then
           iday=iday-month(13)
        else
           iday=iday-month(imonth)
        end if
        imonth=imonth+1
        if(imonth>12)then
           iyear=iyear+1
           imonth=imonth-12
        end if
     end do
  end if

  ctime%year_d=iyear
  ctime%month_d=imonth
  ctime%day_d=iday
  ctime%hour_d=ihour
  ctime%min_d=itime%min_d
  ctime%sec_d=itime%sec_d

end subroutine

!-----------------------------------
!-----------------------------------

subroutine time_zone_convert_i8( factor, itime, ctime )
! タイムゾーンの変換を行うルーチン
! 例えば, JST を UTC に変換する場合, factor = -9 とすればよい.
  implicit none
  integer(kind=i8), intent(in) :: factor        ! 何時間変化するか.
  type(dptime), intent(in) :: itime     ! 変換前の時刻
  type(dptime), intent(inout) :: ctime  ! 変換後の時刻
  integer(kind=i8) :: iyear, imonth, iday, ihour
  integer(kind=i8), parameter, dimension(13) :: month=(/31,28,31,30,  &
  &                                               31,30,31,31,  &
  &                                               30,31,30,31,  &
  &                                               29/)

  iyear=itime%year_d
  imonth=itime%month_d
  iday=itime%day_d
  ihour=itime%hour_d

  ihour=ihour+factor

  if(ihour<0)then
     do while(ihour<0)
        iday=iday-1
        ihour=ihour+24
     end do
  else if(ihour>=24)then
     do while(ihour>=24)
        iday=iday+1
        ihour=ihour-24
     end do
  end if

  if(iday<1)then
     do while(iday<1)
        if((check_leap_year(iyear).eqv..true.).and.(imonth==2))then
           iday=iday+month(13)
        else
           iday=iday+month(imonth)
        end if

        imonth=imonth-1

        if(imonth<=0)then
           iyear=iyear-1
           imonth=12
        end if
     end do
  else if(iday>month(imonth))then
     do while(iday>month(imonth))
        if((check_leap_year(iyear).eqv..true.).and.(imonth==2))then
           iday=iday-month(13)
        else
           iday=iday-month(imonth)
        end if
        imonth=imonth+1
        if(imonth>12)then
           iyear=iyear+1
           imonth=imonth-12
        end if
     end do
  end if

  ctime%year_d=iyear
  ctime%month_d=imonth
  ctime%day_d=iday
  ctime%hour_d=ihour
  ctime%min_d=itime%min_d
  ctime%sec_d=itime%sec_d

end subroutine

!-----------------------------------
!-----------------------------------

subroutine sec_convert_i4( factor, itime, ctime )
! factor 秒分, 加減算を行うルーチン
  implicit none
  integer, intent(in) :: factor        ! 加減算を行う秒数.
  type(dtime), intent(in) :: itime     ! 変換前の時刻
  type(dtime), intent(inout) :: ctime  ! 変換後の時刻
  type(dtime) :: ttime
  integer :: ifact, ofact, fhour, fmin, fsec

  fhour=factor/3600
  fmin=(factor-fhour*3600)/60  ! factor is both of positive and negative.
  fsec=factor-fhour*3600-fmin*60

  !-- sec
  ttime%sec_d=itime%sec_d+fsec
  ttime%min_d=itime%min_d+fmin
  ttime%hour_d=itime%hour_d+fhour
  ttime%day_d=itime%day_d
  ttime%month_d=itime%month_d
  ttime%year_d=itime%year_d

  do while(ttime%sec_d<0)
     ttime%min_d=ttime%min_d-1
     ttime%sec_d=ttime%sec_d+60
  end do

  do while(ttime%sec_d>=60)
     ttime%min_d=ttime%min_d+1
     ttime%sec_d=ttime%sec_d-60
  end do

  do while(ttime%min_d<0)
     ttime%hour_d=ttime%hour_d-1
     ttime%min_d=ttime%min_d+60
  end do

  do while(ttime%min_d>=60)
     ttime%hour_d=ttime%hour_d+1
     ttime%min_d=ttime%min_d-60
  end do

  call time_zone_convert_i4( 0, ttime, ctime )

end subroutine

!-----------------------------------
!-----------------------------------

subroutine sec_convert_i8( factor, itime, ctime )
! factor 秒分, 加減算を行うルーチン
  implicit none
  integer(kind=i8), intent(in) :: factor        ! 加減算を行う秒数.
  type(dptime), intent(in) :: itime     ! 変換前の時刻
  type(dptime), intent(inout) :: ctime  ! 変換後の時刻
  type(dptime) :: ttime
  integer(kind=i8) :: ifact, ofact, fhour, fmin, fsec

  fhour=factor/3600
  fmin=(factor-fhour*3600)/60  ! factor is both of positive and negative.
  fsec=factor-fhour*3600-fmin*60

  !-- sec
  ttime%sec_d=itime%sec_d+fsec
  ttime%min_d=itime%min_d+fmin
  ttime%hour_d=itime%hour_d+fhour
  ttime%day_d=itime%day_d
  ttime%month_d=itime%month_d
  ttime%year_d=itime%year_d

  do while(ttime%sec_d<0)
     ttime%min_d=ttime%min_d-1
     ttime%sec_d=ttime%sec_d+60
  end do

  do while(ttime%sec_d>=60)
     ttime%min_d=ttime%min_d+1
     ttime%sec_d=ttime%sec_d-60
  end do

  do while(ttime%min_d<0)
     ttime%hour_d=ttime%hour_d-1
     ttime%min_d=ttime%min_d+60
  end do

  do while(ttime%min_d>=60)
     ttime%hour_d=ttime%hour_d+1
     ttime%min_d=ttime%min_d-60
  end do

  call time_zone_convert_i8( 0_i8, ttime, ctime )

end subroutine

integer function count_cmax( cval )
! 与えられた文字型配列から要素の中での最大文字数を返す.
  implicit none
  character(*), intent(in) :: cval(:)
  integer :: ii, ix

  ix=size(cval)
  count_cmax=0

  do ii=1,ix
     count_cmax=max(count_cmax,len_trim(adjustl(cval(ii))))
  end do

  return

end function

!-----------------------------------
! Private functions
!-----------------------------------

integer function check_array_size_i1_1d( n1, a )
  implicit none
  integer, intent(in) :: n1  ! reference array size
  integer(1), intent(in) :: a(:)

  if(size(a)/=n1)then
     check_array_size_i1_1d=1
  else
     check_array_size_i1_1d=0
  end if

  return

end function check_array_size_i1_1d

integer function check_array_size_i2_1d( n1, a )
  implicit none
  integer, intent(in) :: n1  ! reference array size
  integer(2), intent(in) :: a(:)

  if(size(a)/=n1)then
     check_array_size_i2_1d=1
  else
     check_array_size_i2_1d=0
  end if

  return

end function check_array_size_i2_1d

integer function check_array_size_i4_1d( n1, a )
  implicit none
  integer, intent(in) :: n1  ! reference array size
  integer, intent(in) :: a(:)

  if(size(a)/=n1)then
     check_array_size_i4_1d=1
  else
     check_array_size_i4_1d=0
  end if

  return

end function check_array_size_i4_1d

integer function check_array_size_f4_1d( n1, a )
  implicit none
  integer, intent(in) :: n1  ! reference array size
  real, intent(in) :: a(:)

  if(size(a)/=n1)then
     check_array_size_f4_1d=1
  else
     check_array_size_f4_1d=0
  end if

  return

end function check_array_size_f4_1d

integer function check_array_size_d8_1d( n1, a )
  implicit none
  integer, intent(in) :: n1  ! reference array size
  double precision, intent(in) :: a(:)

  if(size(a)/=n1)then
     check_array_size_d8_1d=1
  else
     check_array_size_d8_1d=0
  end if

  return

end function check_array_size_d8_1d

integer function check_array_size_i1_2d( n1, n2, a )
  implicit none
  integer, intent(in) :: n1  ! reference array size
  integer, intent(in) :: n2  ! reference array size
  integer(1), intent(in) :: a(:,:)

  if(size(a,1)/=n1)then
     check_array_size_i1_2d=1
  else
     check_array_size_i1_2d=0
  end if

  if(size(a,2)/=n2)then
     check_array_size_i1_2d=check_array_size_i1_2d+2
  end if

  return

end function check_array_size_i1_2d

integer function check_array_size_i2_2d( n1, n2, a )
  implicit none
  integer, intent(in) :: n1  ! reference array size
  integer, intent(in) :: n2  ! reference array size
  integer(2), intent(in) :: a(:,:)

  if(size(a,1)/=n1)then
     check_array_size_i2_2d=1
  else
     check_array_size_i2_2d=0
  end if

  if(size(a,2)/=n2)then
     check_array_size_i2_2d=check_array_size_i2_2d+2
  end if

  return

end function check_array_size_i2_2d

integer function check_array_size_i4_2d( n1, n2, a )
  implicit none
  integer, intent(in) :: n1  ! reference array size
  integer, intent(in) :: n2  ! reference array size
  integer, intent(in) :: a(:,:)

  if(size(a,1)/=n1)then
     check_array_size_i4_2d=1
  else
     check_array_size_i4_2d=0
  end if

  if(size(a,2)/=n2)then
     check_array_size_i4_2d=check_array_size_i4_2d+2
  end if

  return

end function check_array_size_i4_2d

integer function check_array_size_f4_2d( n1, n2, a )
  implicit none
  integer, intent(in) :: n1  ! reference array size
  integer, intent(in) :: n2  ! reference array size
  real, intent(in) :: a(:,:)

  if(size(a,1)/=n1)then
     check_array_size_f4_2d=1
  else
     check_array_size_f4_2d=0
  end if

  if(size(a,2)/=n2)then
     check_array_size_f4_2d=check_array_size_f4_2d+2
  end if

  return

end function check_array_size_f4_2d

integer function check_array_size_d8_2d( n1, n2, a )
  implicit none
  integer, intent(in) :: n1  ! reference array size
  integer, intent(in) :: n2  ! reference array size
  double precision, intent(in) :: a(:,:)

  if(size(a,1)/=n1)then
     check_array_size_d8_2d=1
  else
     check_array_size_d8_2d=0
  end if

  if(size(a,2)/=n2)then
     check_array_size_d8_2d=check_array_size_d8_2d+2
  end if

  return

end function check_array_size_d8_2d

integer function check_array_size_i1_3d( n1, n2, n3, a )
  implicit none
  integer, intent(in) :: n1  ! reference array size
  integer, intent(in) :: n2  ! reference array size
  integer, intent(in) :: n3  ! reference array size
  integer(1), intent(in) :: a(:,:,:)

  if(size(a,1)/=n1)then
     check_array_size_i1_3d=1
  else
     check_array_size_i1_3d=0
  end if

  if(size(a,2)/=n2)then
     check_array_size_i1_3d=check_array_size_i1_3d+2
  end if

  if(size(a,3)/=n3)then
     check_array_size_i1_3d=check_array_size_i1_3d+4
  end if

  return

end function check_array_size_i1_3d

integer function check_array_size_i2_3d( n1, n2, n3, a )
  implicit none
  integer, intent(in) :: n1  ! reference array size
  integer, intent(in) :: n2  ! reference array size
  integer, intent(in) :: n3  ! reference array size
  integer(2), intent(in) :: a(:,:,:)

  if(size(a,1)/=n1)then
     check_array_size_i2_3d=1
  else
     check_array_size_i2_3d=0
  end if

  if(size(a,2)/=n2)then
     check_array_size_i2_3d=check_array_size_i2_3d+2
  end if

  if(size(a,3)/=n3)then
     check_array_size_i2_3d=check_array_size_i2_3d+4
  end if

  return

end function check_array_size_i2_3d

integer function check_array_size_i4_3d( n1, n2, n3, a )
  implicit none
  integer, intent(in) :: n1  ! reference array size
  integer, intent(in) :: n2  ! reference array size
  integer, intent(in) :: n3  ! reference array size
  integer, intent(in) :: a(:,:,:)

  if(size(a,1)/=n1)then
     check_array_size_i4_3d=1
  else
     check_array_size_i4_3d=0
  end if

  if(size(a,2)/=n2)then
     check_array_size_i4_3d=check_array_size_i4_3d+2
  end if

  if(size(a,3)/=n3)then
     check_array_size_i4_3d=check_array_size_i4_3d+4
  end if

  return

end function check_array_size_i4_3d

integer function check_array_size_f4_3d( n1, n2, n3, a )
  implicit none
  integer, intent(in) :: n1  ! reference array size
  integer, intent(in) :: n2  ! reference array size
  integer, intent(in) :: n3  ! reference array size
  real, intent(in) :: a(:,:,:)

  if(size(a,1)/=n1)then
     check_array_size_f4_3d=1
  else
     check_array_size_f4_3d=0
  end if

  if(size(a,2)/=n2)then
     check_array_size_f4_3d=check_array_size_f4_3d+2
  end if

  if(size(a,3)/=n3)then
     check_array_size_f4_3d=check_array_size_f4_3d+4
  end if

  return

end function check_array_size_f4_3d

integer function check_array_size_d8_3d( n1, n2, n3, a )
  implicit none
  integer, intent(in) :: n1  ! reference array size
  integer, intent(in) :: n2  ! reference array size
  integer, intent(in) :: n3  ! reference array size
  double precision, intent(in) :: a(:,:,:)

  if(size(a,1)/=n1)then
     check_array_size_d8_3d=1
  else
     check_array_size_d8_3d=0
  end if

  if(size(a,2)/=n2)then
     check_array_size_d8_3d=check_array_size_d8_3d+2
  end if

  if(size(a,3)/=n3)then
     check_array_size_d8_3d=check_array_size_d8_3d+4
  end if

  return

end function check_array_size_d8_3d

subroutine check_array_size_dmp_message( stat, routine_name )
  implicit none
  integer, intent(in) :: stat
  character(*), intent(in) :: routine_name
  character(100) :: mes, err_num

  select case (check_array_size_iflag)
  case (1)
     mes='Message'
  case (2)
     mes='Warning'
  case (3)
     mes='Error'
  end select

  if(stat>0)then
     select case (stat)
     case (1)
        err_num='1'
     case (2)
        err_num='2'
     case (3)
        err_num='1, 2'
     case (4)
        err_num='3'
     case (5)
        err_num='1, 3'
     case (6)
        err_num='2, 3'
     case (7)
        err_num='1, 2, 3'
     end select

     write(*,*) "*** "//trim(adjustl(mes))//" ("  &
  &             //trim(adjustl(routine_name))//") *** : "  &
  &             //"Invalid array size = "//trim(adjustl(err_num))//'.'
     if(check_array_size_iflag==3)then
        stop
     end if
  end if

end subroutine check_array_size_dmp_message


logical function check_leap_year_i4( year )
! 閏年かどうかを判断して, 閏年なら .true. を返す.

  implicit none

  integer, intent(in) :: year    ! 西暦
  logical :: tmpl

  tmpl=.false.

  if(mod(year,4)==0)then
     if(mod(year,100)==0)then
        if(mod(year,400)==0)then
           tmpl=.true.
        end if
     else
        tmpl=.true.
     end if
  end if

  check_leap_year_i4=tmpl

  return

end function check_leap_year_i4

logical function check_leap_year_i8( year )
! 閏年かどうかを判断して, 閏年なら .true. を返す.

  implicit none

  integer(kind=i8), intent(in) :: year    ! 西暦
  logical :: tmpl

  tmpl=.false.

  if(mod(year,4)==0)then
     if(mod(year,100)==0)then
        if(mod(year,400)==0)then
           tmpl=.true.
        end if
     else
        tmpl=.true.
     end if
  end if

  check_leap_year_i8=tmpl

  return

end function check_leap_year_i8

end module
