!-----------------------------------------------------------------------
!     Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!-----------------------------------------------------------------------

module Thermo_Function  ! Ǯϳؤ˴طؿ
  ! Ǯϳѿ֤Ѵؿξ, "..2.."Ȥˤʤ.
  ! ξ, 2 ƤΤ 2 θˤΤѴȤ
  ! Ȥ̣Ƥ.
use Thermo_Const
use Phys_Const
use stdio
use Algebra
use Statistics

contains

real function tetens( T, dl )  ! ƥƥμ¸Ѥ˰¿׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  ! 絤β [K]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real, parameter :: a=7.5, b=237.7, c=9.5, d=265.5

  if(t<=t0)then
     tetens=e0*10.0**(c*(t-t0)/(t-t0+d))
  else
     tetens=e0*10.0**(a*(t-t0)/(t-t0+b))
  end if

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'tetens', tetens, 'Pa' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function goff_gratch(T, dl)  ! goff-gratch μѤ˰¿׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  ! 絤β [K]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real, parameter :: a=-7.90298, b=5.02808, c=-1.3816e-7, d=8.1328e-3
  real, parameter :: pa=11.344, pb=-3.49149
  real, parameter :: tst=373.15
  real, parameter :: est=1.01325e5
  real :: term

  term=a*(tst/T-1.0)+b*log10(tst/T)+c*(10.0**(pa*(1.0-T/tst))-1.0)+d*(10.0**(pb*(tst/T-1.0))-1.0)
  goff_gratch=est*10.0**(term)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'goff_gratch', goff_gratch, 'Pa' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function goff_gratch_i(T, dl)  ! goff-gratch μѤ˰¿ (ɹ˰) ׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  ! 絤β [K]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real, parameter :: a=-9.09718, b=-3.56654, c=0.876793
  real, parameter :: est=6.1173e2
  real :: term

  term=a*(ti0/T-1.0)+b*log10(ti0/T)+c*(1.0-t/ti0)
  goff_gratch_i=ei0*10.0**(term)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'goff_gratch_i', goff_gratch_i, 'Pa' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function es_Bolton(T, dl)  ! Bolton(1980) μˡѤ˰¿׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  ! 絤β [K]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real, parameter :: a=17.67, c=29.65

  es_Bolton=e0*exp(a*((T-t0)/(T-c)))

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'es_Bolton', es_Bolton, 'Pa' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function es_TD(e, dl)  ! es_Bolton Ѥ˰¿η׻εջ
! Ϫ٤׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: e  ! 絤ο [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real, parameter :: a=17.67, c=29.65
  real, parameter :: emin=1.0e-3

  if(e>0.0)then
     es_TD=c+(a*(t0-c))/(a-log(e/e0))
  else
     es_TD=c+(a*(t0-c))/(a-log(emin/e0))
  end if

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'es_TD', es_TD, 'Pa' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function LH(T, dl)  !  T ˤǮη׻
  ! ܴؿ, ǮȤη̤Ȥۤ礭ʺޤʤ.
  ! ܷ׻, ǮβѲ˴ؤ֥ҥۥåդμפѤ.
  ! ޤ, κݤɬפʱο갵Ǯӿ갵Ǯ
  ! ٰ¸ʤΤȲꤷ, 줾
  ! $C_l=4190,\; C_{pv}=1870$ȤͤѤƳФطǤ.
  ! ä, ѤǮͤ䤽βٰ¸θȷѲǽ.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  ! 絤β [K]
  integer, intent(in), optional :: dl  ! ǥХå٥

  LH=LH0-2.32e3*(T-t0)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'LH', LH, 'J kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function eP_2_qv(e,P, dl)  ! 麮׻
  use Thermo_Const
  implicit none
  real, intent(in) :: e  !  [Pa]
  real, intent(in) :: P  ! 絤 [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: eps

  eps=Rd/Rv
  eP_2_qv=eps*e/(P-e)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'eP_2_qv', eP_2_qv, 'kg kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function TP_2_qvs(T,P, dl)  ! ٤˰º׻
  ! Ǥ, es_Bolton Ѥ˰¿׻,
  ! eP_2_qv ѤƺѴ뤳Ȥ˰º׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: P  ! 絤 [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: eps
  real :: es

  eps=Rd/Rv
  es=es_Bolton(T)
  TP_2_qvs=eps*es/(P-es)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'TP_2_qvs', TP_2_qvs, 'kg kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function qvP_2_e(qv,P, dl)  ! ׻
  use Thermo_Const
  implicit none
  real, intent(in) :: qv  !  [kg / kg]
  real, intent(in) :: P  !  [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: eps

  eps=Rd/Rv
  qvP_2_e=P*qv/(eps+qv)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'qvP_2_e', qvP_2_e, 'Pa' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function theta_dry(T,P, dl)  ! 絤ˤ벹̤׻
  ! , 絤ˤƤ, ¬ P ȤƷ׻뤳ȤǤ
  ! η̴̤ؿ theta_moist η̤ȤۤѤʤ.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: P  ! 絤ε(⤷, 絤) [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: kappa

  kappa=Rd/Cpd
  theta_dry=T*(p0/P)**kappa

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'theta_dry', theta_dry, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function theta_moist(T,P,qv, dl)  ! 絤ˤ벹̤׻
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: P  ! 絤 [Pa]
  real, intent(in) :: qv  !  [kg / kg]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: eps, kappa, CR

  eps=Rd/Rv
  kappa=Rd/Cpd
  CR=Cpv/Cpd

  kappa=kappa*((1.0+qv*CR)/(1.0+qv/eps))  ! kappa ͤ夫񤭤Ƥ뤳Ȥ
  theta_moist=T*(p0/P)**kappa

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'theta_moist', theta_moist, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function thetaP_2_T(theta,P, dl)  ! , Ϥ鲹٤׻(絤ȤƷ׻)
  use Thermo_Const
  implicit none
  real, intent(in) :: theta  !  [K]
  real, intent(in) :: P  ! 絤 [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: kappa

  kappa=Rd/Cpd

  thetaP_2_T=theta*(P/p0)**kappa

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'thetaP_2_T', thetaP_2_T, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function thetaT_2_P(theta,T, dl)  ! , ٤鰵Ϥ׻(絤ȤƷ׻)
  use Thermo_Const
  implicit none
  real, intent(in) :: theta  !  [K]
  real, intent(in) :: T  !  [T]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: kappa

  kappa=Cpd/Rd

  thetaT_2_P=p0*(T/theta)**kappa

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'thetaT_2_P', thetaT_2_P, 'Pa' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function TqvP_2_TLCL(T,qv,P, dl)  !! ٤Ⱥ T_LCL ׻
  ! 椫,  T_LCL ׻
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: qv  !  [kg / kg]
  real, intent(in) :: P  !  [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real, parameter :: coe=2840.0, a=3.5, b=4.805, c=55.0
  real, parameter :: emin=1.0e-3
  real :: e

  e=qvP_2_e(qv,P)
  e=e*1.0e-2
  if(e>0.0)then
     TqvP_2_TLCL=coe/(a*log(T)-log(e)-b)+55.0
  else
     TqvP_2_TLCL=coe/(a*log(T)-log(emin)-b)+55.0
  end if

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'TqvP_2_TLCL', TqvP_2_TLCL, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function thetae_Bolton(T,qv,P, dl)
  ! Bolton(1980) ˤˡѤ̤׻.
  ! ̤ϵǮǤ̤Ǥ.
  ! T_LCL ѤΤ, ΤδؿѤ.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: qv  !  [kg / kg]
  real, intent(in) :: P  !  [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: T_LCL, qvs
  real, parameter :: a=0.2854, b=0.28, c=3376.0, d=0.81

  T_LCL=TqvP_2_TLCL(T,qv,P)
  qvs=TP_2_qvs(T_LCL,P)
  thetae_Bolton=T*((p0/P)**(a*(1.0-b*qvs)))  &
   &            *exp((c/T_LCL-2.54)*qvs*(1.0+d*qvs))

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'thetae_Bolton', thetae_Bolton, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function thetaes_Bolton(T,P, dl)  ! Bolton(1980) ˤˡѤ˰̤׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: P  !  [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: qvs
  real, parameter :: a=0.2854, b=0.28, c=3376.0, d=0.81

  qvs=TP_2_qvs(T,P)
  thetaes_Bolton=T*((p0/P)**(a*(1.0-b*qvs)))  &
   &            *exp((c/T-2.54)*qvs*(1.0+d*qvs))

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'thetaes_Bolton', thetaes_Bolton, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function TqvP_2_thetae(T,qv,P, dl)  ! , , ̤׻.
  ! T_LCL ѤΤ, ΤδؿѤ.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: qv  !  [kg / kg]
  real, intent(in) :: P  !  [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: T_LCL, kappa, theta_d, e, qvs
  real, parameter :: a=0.2854, b=0.28, c=3376.0, d=0.81

  kappa=Rd/Cpd
  e=qvP_2_e(qv,P)
  T_LCL=TqvP_2_TLCL(T,qv,P)
  theta_d=T*(p0/(P-e))**kappa
  qvs=TP_2_qvs(T_LCL,P)
  TqvP_2_thetae=theta_d*exp(LH(T_LCL)*qvs/(Cpd*T_LCL))

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'TqvP_2_thetae', TqvP_2_thetae, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function TqvP_2_thetaes(T,P, dl)  ! , , ˰̤׻.
  ! T_LCL ѤΤ, ΤδؿѤ.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
!  real, intent(in) :: qv  !  [kg / kg]
  real, intent(in) :: P  !  [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: kappa, theta_d, qvs !,e
  real, parameter :: a=0.2854, b=0.28, c=3376.0, d=0.81

  kappa=Rd/Cpd
!  e=qvP_2_e(qv,P)
!  theta_d=T*(p0/(P-e))**kappa
  theta_d=T*(p0/P)**kappa
  qvs=TP_2_qvs(T,P)
  TqvP_2_thetaes=theta_d*exp(LH(T)*qvs/(Cpd*T))

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'TqvP_2_thetaes', TqvP_2_thetaes, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function RHT_2_e(RH,T, dl)  ! м٤Ȳ٤׻
  ! $RH=(e/es)\times 100$ Ȥ׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: RH  ! м [%]
  real, intent(in) :: T  !  [K]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: es

  es=es_Bolton(T)
  RHT_2_e=RH*es*1.0e-2

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'RHT_2_e', RHT_2_e, 'Pa' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function eT_2_RH(e,T, dl)  ! Ȳ٤м٤׻
  ! $RH=(e/es)\times 100$ Ȥ׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: e  !  [Pa]
  real, intent(in) :: T  !  [K]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: es

  es=es_Bolton(T)
  eT_2_RH=100.0*e/es

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'eT_2_RH', eT_2_RH, '%' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function RHTP_2_qv(RH,T,P, dl)  ! м٤Ȳ٤麮׻
  ! RHT_2_e ׻, eP_2_qv 麮׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: RH  ! м [%]
  real, intent(in) :: T  !  [K]
  real, intent(in) :: P  !  [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: e

  e=RHT_2_e(RH,T)
  RHTP_2_qv=eP_2_qv(e,P)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'RHTP_2_qv', RHTP_2_qv, 'kg kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function qvTP_2_RH(qv,T,P, dl)  ! Ȳ٤м٤׻.
  ! qvP_2_e ׻, м٤Ѥ.
  use Thermo_Const
  implicit none
  real, intent(in) :: qv  ! м [kg / kg]
  real, intent(in) :: T  !  [K]
  real, intent(in) :: P  !  [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: e

  e=qvP_2_e(qv,P)
  qvTP_2_RH=eT_2_RH(e,T)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'qvTP_2_RH', qvTP_2_RH, '%' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function qvT_2_Tv( qv, T, dl)  ! ٤ȿ椫鲾٤׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: qv  !  [kg / kg]
  real, intent(in) :: T   !  [K]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: eps

  eps=Rd/Rv
  qvT_2_Tv=T*(1.0+qv/eps)/(1.0+qv)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'qvT_2_Tv', qvT_2_Tv, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function qvTv_2_T( qv, Tv, dl )  ! Ȳ٤鲹٤׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: qv  !  [kg / kg]
  real, intent(in) :: Tv  !  [K]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: eps

  eps=Rd/Rv
  qvTv_2_T=Tv*(1.0+qv)/(1.0+qv/eps)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'qvTv_2_T', qvTv_2_T, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function TvT_2_qv( Tv, T, dl)  ! ٤Ȳ٤׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: Tv  !  [K]
  real, intent(in) :: T   !  [K]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: eps

  eps=Rd/Rv

  if(Tv/=T)then
     TvT_2_qv=eps*((T-Tv)/(eps*Tv-T))
  else
     TvT_2_qv=0.0
  end if

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'TvT_2_qv', TvT_2_qv, 'kg kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function TqvP_2_thetav( T, qv, P, dl )  ! , , Ϥ鲾̤׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: qv  !  [kg / kg]
  real, intent(in) :: T   !  [K]
  real, intent(in) :: P   !  [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: kappa, Tv

  kappa=Rd/cpd
  Tv=qvT_2_Tv(qv,T)
  TqvP_2_thetav=theta_dry(Tv,P)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'TqvP_2_thetav', TqvP_2_thetav, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function thetavqvP_2_T( ptv, qv, P, dl )  ! , , Ϥ鲹٤׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: ptv !  [K]
  real, intent(in) :: qv  !  [kg / kg]
  real, intent(in) :: P   !  [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: kappa, Tv

  kappa=Rd/cpd
  Tv=ptv*exner_func_dry( P )
  thetavqvP_2_T=qvTv_2_T( qv, Tv )

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'thetavqvP_2_T', thetavqvP_2_T, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function TthetavP_2_qv( T, thetav, P, dl )  ! , , Ϥ׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: T       !  [K]
  real, intent(in) :: thetav  !  [K]
  real, intent(in) :: P       !  [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: Tv, eps, exn

  eps=Rd/Rv

  exn=exner_func_dry( P )

  if(T/=exn*thetav)then
     TthetavP_2_qv=eps*((T-exn*thetav)/(eps*exn*thetav-T))
  else
     TthetavP_2_qv=0.0
  end if

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'TthetavP_2_qv', TthetavP_2_qv, 'kg kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function exner_func_dry(P, dl)  ! 絤ˤĤƤΥʡؿ׻
  use Thermo_Const
  implicit none
  real, intent(in) :: P  !  [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: kappa

  kappa=Rd/Cpd
  exner_func_dry=(P/p0)**kappa

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'exner_func_dry', exner_func_dry, '1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function hypsometric_form(p,z,T,z_t, dl)  ! ٤ȵͿƤ٤ε.
!  ιˤήˤɸ絤βٸΨǤ 6.5 [K/km] Ƿ׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: p  ! Ȥʤ٤ˤ밵 [Pa]
  real, intent(in) :: z  ! Ȥʤ [m]
  real, intent(in) :: T  ! Ǥβ [K]
  real, intent(in), optional :: z_t  !  [m] : ǥեȤǤ 0 m.
  integer, intent(in), optional :: dl  ! ǥХå٥
  real, parameter :: gam = 6.5e-3  ! ɸ絤βٸΨ [K/m]
  real :: z_calc, p_tmp

!write(*,*) "hypsometric, g=", g

  if(present(z_t))then
     z_calc=z_t
  else
     z_calc=0.0
  end if

  p_tmp=p*((T+gam*z)/(T+gam*z_calc))**(g/(gam*Rd))
  hypsometric_form=p_tmp

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'hypsometric_form', hypsometric_form, 'Pa' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function z_hypsometric_form(p,z,T,p_t, dl)
!  ٤ȵͿƤ뵤ι٤ (¬).
!  ٤ιˤήˤɸ絤βٸΨǤ 6.5 [K/km] Ƿ׻.
  use Thermo_Const
  use Phys_Const
  implicit none
  real, intent(in) :: p  ! Ȥʤ밵 [Pa]
  real, intent(in) :: z  ! Ȥʤ밵ϤǤι [m]
  real, intent(in) :: T  ! Ǥβ [K]
  real, intent(in) :: p_t  ! 밵 [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real, parameter :: gam = 6.5e-3  ! ɸ絤βٸΨ [K/m]
  real :: alpha

  alpha=Rd*gam/g
  z_hypsometric_form=z-(T/gam)*((p_t/p)**alpha-1.0)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'z_hypsometric_form', z_hypsometric_form, 'm' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function qv_2_sh( qv, dl )
! 椫漾׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: qv  !  [kg/kg]
  integer, intent(in), optional :: dl  ! ǥХå٥

  qv_2_sh=qv/(1.0+qv)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'qv_2_sh', qv_2_sh, 'kg kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function sh_2_qv( sh, dl )
! 漾׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: sh  ! 漾 [kg/kg]
  integer, intent(in), optional :: dl  ! ǥХå٥

  sh_2_qv=sh/(1.0-sh)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'sh_2_qv', sh_2_qv, 'kg kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function Cefp( qv, dl )
! 椫ͭ갵Ǯ׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: qv  !  [kg/kg]
  integer, intent(in), optional :: dl  ! ǥХå٥

  Cefp=(Cpd+qv*Cpv)/(1.0+qv)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'Cefp', Cefp, 'J K-1 kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function Cefv( qv, dl )
! 椫ͭǮ׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: qv  !  [kg/kg]
  integer, intent(in), optional :: dl  ! ǥХå٥

  Cefv=(Cvd+qv*Cvv)/(1.0+qv)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'Cefv', Cefv, 'J K-1 kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function Reff( qv, dl )
! 椫ͭ׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: qv  !  [kg/kg]
  integer, intent(in), optional :: dl  ! ǥХå٥

  Reff=(Rd+qv*Rv)/(1.0+qv)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'Reff', Reff, 'J K-1 kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function Cl( T, dl )
! տǮ׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real, parameter :: c1=4190.0, c2=4770.0
  real :: val_rat, tref

  val_rat=(c1-c2)/40.0
  tref=273.15-40.0

  if(T>=273.15)then
     Cl=c1
  else
     Cl=c2+val_rat*(T-tref)
  end if

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'Cl', Cl, 'J K-1 kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function Tq_2_Trho( T, qv, qo, dl )
! ٤ȿʪ̩ٲ٤׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: qv  !  [kg/kg]
  real, intent(in), optional :: qo(:)  ! ʳοŷʪ [kg/kg]
                   ! ŷʪΥƥϤǤ⹽ʤ.
  integer, intent(in), optional :: dl  ! ǥХå٥
  integer :: i, n
  real :: tmpq

  if(present(qo))then
     n=size(qo)
     tmpq=qv
     do i=1,n
        tmpq=tmpq+qo(i)
     end do
  else
     tmpq=qv
  end if

  Tq_2_Trho=T*(1.0+qv*Rv/Rd)/(1.0+tmpq)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'Tq_2_Trho', Tq_2_Trho, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function moist_enthal( T, qv, qo, dl )
! ٤ȿʪ鼾ᥨ󥿥ԡ׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: qv  !  [kg/kg]
  real, intent(in), optional :: qo(:)  ! ʳοŷʪ [kg/kg]
                   ! ŷʪΥƥϤǤ⹽ʤ.
  integer, intent(in), optional :: dl  ! ǥХå٥
  integer :: i, n
  real :: tmpq

  if(present(qo))then
     n=size(qo)
     tmpq=qv
     do i=1,n
        tmpq=tmpq+qo(i)
     end do
  else
     tmpq=qv
  end if

  moist_enthal=(Cpd+tmpq*Cl( T ))*T+LH( T )*qv

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'moist_enthal', moist_enthal, 'J kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function liquid_enthal( T, ql, qo, dl )
! ٤ȿʪտ奨󥿥ԡ׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: ql  ! ο庮 [kg/kg]
  real, intent(in), optional :: qo(:)  ! οʳοʪ [kg/kg]
                   ! ʪΥƥϤǤ⹽ʤʿޤ.
  integer, intent(in), optional :: dl  ! ǥХå٥
  integer :: i, n
  real :: tmpq

  if(present(qo))then
     n=size(qo)
     tmpq=ql
     do i=1,n
        tmpq=tmpq+qo(i)
     end do
  else
     tmpq=ql
  end if

  liquid_enthal=(Cpd+tmpq*Cpv)*T-LH( T )*ql

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'liquid_enthal', liquid_enthal, 'J kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function esi_Emanuel( T, dl )
! Emanuel (1994)  (4.4.15) ɹ˰¾׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real, parameter :: a=23.33086, b=6111.72784, c=0.15215

  esi_Emanuel=a-b/T+c*log(T)
  esi_Emanuel=exp(esi_Emanuel)*100.0

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'esi_Emanuel', esi_Emanuel, 'Pa' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function thetae_Emanuel( T, qv, pres, qo, dl )
! Emanuel (1994)  (4.5.11) ̤׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: qv  !  [kg/kg]
  real, intent(in) :: pres  !  [Pa]
  real, intent(in), optional :: qo(:)  ! ʳοʪ [kg/kg]
  integer, intent(in), optional :: dl  ! ǥХå٥
  integer :: i, n
  real :: tmpq, calH, pdry, pow1, pow2, pow3, coe1, coe2, coe3, Cltmp, LHtmp

  if(present(qo))then
     n=size(qo)
     tmpq=qv
     do i=1,n
        tmpq=tmpq+qo(i)
     end do
  else
     tmpq=qv
  end if

  pdry=pres-qvP_2_e( qv, pres )
  calH=qvTP_2_RH( qv, T, pres )*0.01
  Cltmp=Cl( T )
  LHtmp=LH( T )

  pow1=Rd/(Cpd+Cltmp*tmpq)
  pow2=qv*Rv/(Cpd+Cltmp*tmpq)
  pow3=LHtmp*qv/((Cpd+Cltmp*tmpq)*T)
  coe1=(p0/pdry)**pow1
  coe2=calH**pow2
  coe3=exp(pow3)

  thetae_Emanuel=T*coe1*coe2*coe3

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'thetae_Emanuel', thetae_Emanuel, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function thetal_Emanuel( T, ql, pres, qo, dl )
! Emanuel (1994)  (4.5.15) տ岹̤׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: ql  ! տ庮 [kg/kg]
  real, intent(in) :: pres  !  [Pa]
  real, intent(in), optional :: qo(:)  ! տʳοʪʿޤ [kg/kg]
  integer, intent(in), optional :: dl  ! ǥХå٥
  integer :: i, n
  real :: tmpq, pow1, pow2, pow3, coe1, coe2, coe3, epsi

  epsi=Rd/Rv

  if(present(qo))then
     n=size(qo)
     tmpq=ql
     do i=1,n
        tmpq=tmpq+qo(i)
     end do
  else
     tmpq=ql
  end if

  pow1=(Rd+tmpq*Rv)/(Cpd+tmpq*Cpv)
  pow2=(tmpq*Rv)/(Cpd+tmpq*Cpv)
  pow3=-LH( T )*ql/((Cpd+tmpq*Cpv)*T)
  coe1=((p0/pres)*(1.0-ql/(epsi+tmpq)))**pow1
  coe2=(tmpq/(tmpq-ql))**pow2
  coe3=exp(pow3)

  thetal_Emanuel=T*coe1*coe2*coe3

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'thetal_Emanuel', thetal_Emanuel, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function thetalv_Emanuel( T, ql, qv, pres, qo, dl )
! Emanuel (1994)  (4.5.18) տ岾̤׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: ql  ! տ庮 [kg/kg]
  real, intent(in) :: qv  !  [kg/kg]
  real, intent(in) :: pres  !  [Pa]
  real, intent(in), optional :: qo(:)  ! տ, ʳοʪ [kg/kg]
  integer, intent(in), optional :: dl  ! ǥХå٥
  integer :: i, n
  real :: tmpq, pow1, pow2, pow3, coe1, coe2, coe3, epsi, tmpv, tmpcoe

  epsi=Rd/Rv
  tmpv=qvT_2_Tv( qv, T )

  if(present(qo))then
     n=size(qo)
     tmpq=ql+qv
     do i=1,n
        tmpq=tmpq+qo(i)
     end do
  else
     tmpq=ql+qv
  end if

  pow1=(Rd+tmpq*Rv)/(Cpd+tmpq*Cpv)
  pow2=(tmpq*Rv)/(Cpd+tmpq*Cpv)
  pow3=-LH( T )*ql/((Cpd+tmpq*Cpv)*T)
  tmpcoe=(p0/pres)**pow1
  coe1=(1.0-ql/(epsi+tmpq))**(pow1-1.0)
  coe2=(tmpq/(tmpq-ql))**pow2
  coe3=exp(pow3)

  thetalv_Emanuel=tmpv*tmpcoe*coe1*coe2*coe3

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'thetalv_Emanuel', thetalv_Emanuel, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function thetaw_Emanuel( T, qv, pres, eps, dl )
! Emanuel (1994)  (4.7.10) 뼾岹̤׻.
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: qv  !  [kg/kg]
  real, intent(in) :: pres  !  [Pa]
  integer, intent(in), optional :: eps  ! « (ǥե = 1.e-6)
  integer, intent(in), optional :: dl  ! ǥХå٥
  integer :: i, n
  real, parameter :: a=0.81, b=3376.0, c=2.54, theta1=100.0, theta2=600.0
  real :: tmpa, tmpb, tmpc, fa, fb, fc, tmpthetae, thetaa, thetab, thetac
  real :: rast, err_max, tmp_err

  if(present(eps))then
     err_max=eps
  else
     err_max=1.0e-6
  end if

  tmpthetae=thetae_Bolton( T, qv, pres )
  rast=TP_2_qvs( T, pres )

  tmp_err=err_max

  tmpa=rast*(1.0+a*rast)*(b/theta1-c)-log(tmpthetae/theta1)
  tmpb=rast*(1.0+a*rast)*(b/theta2-c)-log(tmpthetae/theta2)
  thetaa=theta1
  thetab=theta2

  do while (err_max<=tmp_err)
     thetac=0.5*(thetab+thetaa)
     tmpc=rast*(1.0+a*rast)*(b/thetac-c)-log(tmpthetae/thetac)

     if(tmpa*tmpc>0.0)then
        tmp_err=abs(tmpa-tmpc)
        tmpa=tmpc
        thetaa=thetac
     else
        tmp_err=abs(tmpb-tmpc)
        tmpb=tmpc
        thetab=thetac
     end if
  end do

  thetaw_Emanuel=thetac

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'thetaw_Emanuel', thetaw_Emanuel, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function MSE_Emanuel( T, qv, z, qo, dl )
! Emanuel (1994)  (4.5.23) 뼾Ūͥ륮׻.
  use Thermo_Const
  use Phys_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: qv  !  [kg/kg]
  real, intent(in) :: z  ! ɽ̹ [m]
  real, intent(in), optional :: qo(:)  ! ʳοʪ [kg/kg]
  integer, intent(in), optional :: dl  ! ǥХå٥
  integer :: i, n
  real :: tmpq

  if(present(qo))then
     n=size(qo)
     tmpq=qv
     do i=1,n
        tmpq=tmpq+qo(i)
     end do
  else
     tmpq=qv
  end if

  MSE_Emanuel=(Cpd+tmpq*Cl( T ))*T+LH( T )*qv+(1.0+tmpq)*g*z

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'MSE_Emanuel', MSE_Emanuel, 'J kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function DSE_Emanuel( T, qv, z, dl )
! Emanuel (1994)  (4.5.24) 봥Ūͥ륮׻.
  use Thermo_Const
  use Phys_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: qv  !  [kg/kg]
  real, intent(in) :: z  ! ɽ̹ [m]
  integer, intent(in), optional :: dl  ! ǥХå٥
  integer :: i, n

  DSE_Emanuel=(Cpd+qv*Cpv)*T+(1.0+qv)*g*z

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'DSE_Emanuel', DSE_Emanuel, 'J kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function WSE_Emanuel( T, ql, z, qo, dl )
! Emanuel (1994)  (4.5.25) տŪͥ륮׻.
  use Thermo_Const
  use Phys_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: ql  ! տ庮 [kg/kg]
  real, intent(in) :: z  ! ɽ̹ [m]
  real, intent(in), optional :: qo(:)  ! տʳοʪʿޤ [kg/kg]
  integer, intent(in), optional :: dl  ! ǥХå٥
  integer :: i, n
  real :: tmpq

  if(present(qo))then
     n=size(qo)
     tmpq=ql
     do i=1,n
        tmpq=tmpq+qo(i)
     end do
  else
     tmpq=ql
  end if

  WSE_Emanuel=(Cpd+tmpq*Cpv)*T-LH( T )*ql+(1.0+tmpq)*g*z

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'WSE_Emanuel', WSE_Emanuel, 'J kg-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

!-- 餷Ф餯Ѵ -----------------

real function rhoT_2_P( rho, T, dl )
! 絤ξ, ̩٤Ȳ٤ͿƵ.
  use Thermo_Const
  implicit none
  real, intent(in) :: rho  ! 絤̩ [kg/m^3]
  real, intent(in) :: T    ! 絤β [K]
  integer, intent(in), optional :: dl  ! ǥХå٥

  rhoT_2_P=rho*Rd*T

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'rhoT_2_P', rhoT_2_P, 'Pa' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function rhoP_2_T( rho, P, dl )
! 絤ξ, ̩٤ȵͿƲ٤.
  use Thermo_Const
  implicit none
  real, intent(in) :: rho  ! 絤̩ [kg/m^3]
  real, intent(in) :: P    ! 絤ΰ [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥

  rhoP_2_T=P/(Rd*rho)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'rhoP_2_T', rhoP_2_T, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function TP_2_rho( T, P, dl )
! 絤ξ, ٤ȵͿ̩٤.
  use Thermo_Const
  implicit none
  real, intent(in) :: T    ! 絤β [K]
  real, intent(in) :: P    ! 絤ΰ [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥

  TP_2_rho=p/(Rd*T)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'TP_2_rho', TP_2_rho, 'kg m-3' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------


!-----------------------------------------------------
!-----------------------------------------------------

real function get_gamma_d( dl )  ! ǮΨƤִؿ(δؿɬפ)
  use Thermo_Const
  use Phys_Const
  implicit none
  integer, intent(in), optional :: dl  ! ǥХå٥

  get_gamma_d=-g/Cpd

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'get_gamma_d', get_gamma_d, 'K m-1' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function moist_laps_temp( p_stan, T_stan, p, dl )
! Ǯˤ뼾ǮΨ˽,  p ˤ뵤ؿ.
! ϥѡε.
  use Thermo_Const
  implicit none
  real, intent(in) :: p_stan  ! ൤ɸ [Pa]
  real, intent(in) :: T_stan  ! ಹ [K]
  real, intent(in) :: p  ! ٤ˤ뵤 [Pa]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: seqpt, t_temp, seqpt_temp, dt, seqpt_temp2
  integer, parameter :: ilim=20
  real, parameter :: limit=2.0e-4

  seqpt=thetaes_Bolton( T_stan, p_stan )
  seqpt_temp=thetaes_Bolton( t_stan, p )
  if(p>p_stan)then
     dt=1.0
  else
     dt=-1.0
  end if
  t_temp=t_stan+dt
  seqpt_temp2=thetaes_Bolton( t_temp, p )

  do while ( (seqpt_temp2-seqpt)*(seqpt_temp-seqpt) >0.0 )
     t_temp=t_temp+dt
     seqpt_temp=seqpt_temp2
     seqpt_temp2=thetaes_Bolton( t_temp, p )
  end do

  seqpt_temp=seqpt_temp2
  dt=1.0

  do while ( abs(seqpt_temp-seqpt) > limit )

     dt=0.5*dt

     if( (seqpt_temp-seqpt)>0.0 )then
        t_temp=t_temp-dt
     else
        t_temp=t_temp+dt
     end if

     seqpt_temp=thetaes_Bolton( t_temp, p )

  end do

  moist_laps_temp=t_temp

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'moist_laps_temp', moist_laps_temp, 'K' )
  end if

  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function TTd_2_RH_Bolton( T, Td, dl )
  ! ٤Ϫ٤м٤׻.
  ! Bolton μѤƿ˰¿׻. 
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: Td ! Ϫ [K]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: e,es

  e=es_Bolton(Td)
  es=es_Bolton(T)

  TTd_2_RH_Bolton=(e/es)*100.0

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'TTd_2_RH_Bolton', TTd_2_RH_Bolton, '%' )
  end if
  
  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function TTd_2_RH_tetens( T, Td, dl )
  ! ٤Ϫ٤м٤׻.
  ! tetens μѤƿ˰¿׻. 
  use Thermo_Const
  implicit none
  real, intent(in) :: T  !  [K]
  real, intent(in) :: Td ! Ϫ [K]
  integer, intent(in), optional :: dl  ! ǥХå٥
  real :: e,es

  e=tetens(Td)
  es=tetens(T)

  TTd_2_RH_tetens=(e/es)*100.0

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'TTd_2_RH_tetens', TTd_2_RH_tetens, '%' )
  end if
  
  return
end function

!-----------------------------------------------------
!-----------------------------------------------------

real function rho_ocean( pres, temp, sal, dl )
! UNESCO (1981) ξѤ, ̩٤׻.
! Ǥη׻ˤϰʲΤ褦ʥɺνޤޤƤ뤳Ȥդ.
!    (a**x)*(b**y)=exp(x*ln(a)+y*ln(b))
! ʤʤ, 
!    (a**x)*(b**y)=z
! => ln{(a**x)*(b**y)}=ln(z)
! => x*ln(a)+y*ln(b)=ln(z)
! => exp(x*ln(a)+y*ln(b))=z.
! Ȥ, c*(b**y) Ȥξ ( x ѿ٤) :
! c*(b**y)=exp(ln(c)+y*ln(b)) ɽ.
  implicit none
  real, intent(in) :: pres    !  [Bar]
  real, intent(in) :: temp    !  [degC]
  real, intent(in) :: sal     ! ʬǻ [PSU]
  integer, intent(in), optional :: dl  ! ǥХå٥

  real :: rhoz0, rhowc, Kw, Kz0, Kz, tmptemp

  if(temp>0.0)then
     tmptemp=temp
     Kw=19652.21+148.4206*tmptemp-2.327105*tmptemp*tmptemp  &
     &  +exp(log(1.360477e-2)+3.0*log(tmptemp))  &
     &  -exp(log(5.155288e-5)+4.0*log(tmptemp))
     Kz0=Kw+sal*(54.6746-0.603459*tmptemp  &
     &          +exp(log(1.09987e-2)+2.0*log(tmptemp))  &
     &          -exp(log(6.167e-5)+3.0*log(tmptemp)))  &
     &     +(sal**1.5)*(7.944e-2+1.6483e-2*tmptemp  &
     &          -exp(log(5.3009e-4)+2.0*log(tmptemp)))

     Kz=Kz0+pres*(3.239908+1.43713e-3*tmptemp  &
     &           +exp(log(1.16092e-4)+2.0*log(tmptemp))  &
     &           -exp(log(5.77905e-7)+3.0*log(tmptemp)))  &
     &     +pres*sal*(2.2838e-3-1.0981e-5*tmptemp  &
     &           -exp(log(1.6078e-6)+2.0*log(tmptemp)))  &
     &     +pres*(sal**1.5)*1.91075e-4  &
     &     +(pres**2)*(8.50935e-5-6.12293e-6*tmptemp  &
     &           +exp(log(5.2787e-8)+2.0*log(tmptemp)))  &
     &     +(pres**2)*sal*(-9.9348e-7+2.0816e-8*tmptemp  &
     &           +exp(log(9.1697e-10)+2.0*log(tmptemp)))

     rhowc=999.842594+6.793952e-2*tmptemp  &
     &               -exp(log(9.09529e-3)+2.0*log(tmptemp))  &
     &               +exp(log(1.001685e-4)+3.0*log(tmptemp))  &
     &               -exp(log(1.120083e-6)+4.0*log(tmptemp))  &
     &               +exp(log(6.536332e-9)+5.0*log(tmptemp))

     rhoz0=rhowc  &
     &     +sal*(0.824493-4.0899e-3*tmptemp  &
     &          +exp(log(7.6438e-5)+2.0*log(tmptemp))  &
     &          -exp(log(8.2467e-7)+3.0*log(tmptemp))  &
     &          +exp(log(5.3875e-9)+4.0*log(tmptemp)))  &
     &     +(sal**1.5)*(-5.72466e-3+1.0227e-4*tmptemp  &
     &          -exp(log(1.6546e-6)+2.0*log(tmptemp)))  &
     &     +(sal**2)*4.8314e-4

  else if(temp<0.0)then

     tmptemp=-temp
     Kw=19652.21-148.4206*tmptemp-2.327105*tmptemp*tmptemp  &
     &  -exp(log(1.360477e-2)+3.0*log(tmptemp))  &
     &  -exp(log(5.155288e-5)+4.0*log(tmptemp))

     Kz0=Kw+sal*(54.6746+0.603459*tmptemp  &
     &          +exp(log(1.09987e-2)+2.0*log(tmptemp))  &
     &          +exp(log(6.167e-5)+3.0*log(tmptemp)))  &
     &     +(sal**1.5)*(7.944e-2-1.6483e-2*tmptemp  &
     &          -exp(log(5.3009e-4)+2.0*log(tmptemp)))

     Kz=Kz0+pres*(3.239908-1.43713e-3*tmptemp  &
     &           +exp(log(1.16092e-4)+2.0*log(tmptemp))  &
     &           +exp(log(5.77905e-7)+3.0*log(tmptemp)))  &
     &     +pres*sal*(2.2838e-3+1.0981e-5*tmptemp  &
     &           -exp(log(1.6078e-6)+2.0*log(tmptemp)))  &
     &     +pres*(sal**1.5)*1.91075e-4  &
     &     +(pres**2)*(8.50935e-5+6.12293e-6*tmptemp  &
     &           +exp(log(5.2787e-8)+2.0*log(tmptemp)))  &
     &     +(pres**2)*sal*(-9.9348e-7-2.0816e-8*tmptemp  &
     &           +exp(log(9.1697e-10)+2.0*log(tmptemp)))

     rhowc=999.842594-6.793952e-2*tmptemp  &
     &               -exp(log(9.09529e-3)+2.0*log(tmptemp))  &
     &               -exp(log(1.001685e-4)+3.0*log(tmptemp))  &
     &               -exp(log(1.120083e-6)+4.0*log(tmptemp))  &
     &               -exp(log(6.536332e-9)+5.0*log(tmptemp))

     rhoz0=rhowc  &
     &     +sal*(0.824493+4.0899e-3*tmptemp  &
     &          +exp(log(7.6438e-5)+2.0*log(tmptemp))  &
     &          +exp(log(8.2467e-7)+3.0*log(tmptemp))  &
     &          +exp(log(5.3875e-9)+4.0*log(tmptemp)))  &
     &     +(sal**1.5)*(-5.72466e-3-1.0227e-4*tmptemp  &
     &          -exp(log(1.6546e-6)+2.0*log(tmptemp)))  &
     &     +(sal**2)*4.8314e-4

  else

     Kw=19652.21
     Kz0=Kw+sal*(54.6746  &
     &     +(sal**1.5)*7.944e-2)
     Kz=Kz0+pres*3.239908  &
     &     +pres*sal*2.2838e-3  &
     &     +pres*(sal**1.5)*1.91075e-4  &
     &     +(pres**2)*8.50935e-5  &
     &     +(pres**2)*sal*(-9.9348e-7)

     rhowc=999.842594

     rhoz0=rhowc  &
     &     +sal*0.824493  &
     &     -5.72466e-3*(sal**1.5)  &
     &     +4.8314e-4*(sal**2)

  end if

  rho_ocean=rhoz0/(1.0-pres/Kz)

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'rho_ocean', rho_ocean, 'kg m-3' )
  end if

  return
end function rho_ocean

!-----------------------------------------------------
!-----------------------------------------------------

real function PTS_2_theta( pres, temp, sal, dl )
! Vallis (2006)  (1.153) Ѥ, β٤򲹰 [degC] Ѵ.
! theta = T exp(f),
! f(p,T,S) = - k*(1+gams*p/2+b*(T-to0))
! k = a0 * bt * p / cpo = 1.67 * pres * 10^(-4+5-3-3) / 1.027 / 3.986 / alph,
! b = bts / bt = 0.1 / 1.67
! cpo = cpo0 * (1 + bss * (S-so0) ) = 3.986e3 * alph
  implicit none
  real, intent(in) :: pres    !  [Bar ~ x10 m]
  real, intent(in) :: temp    !  [degC]
  real, intent(in) :: sal     ! ʬǻ [PSU]
  integer, intent(in), optional :: dl  ! ǥХå٥

  real :: a0=9.738e-4    ! [m3 kg-1]
  real :: to0=283.0      ! [K]
  real :: so0=35.0       ! [psu]
  real :: bt=1.67e-4     ! [K-1]
  real :: bts=1.0e-5     ! [K-2]
  real :: bs=7.8e-4      ! [psu-1]
  real :: bss=1.5e-3     ! [psu-1]
  real :: cpo0=3986.0    ! [J K-1 kg-1]
  real :: gams=1.1e-3    ! [Bar-1]

  real :: k, b, alph, f, TK

  TK=temp+t0
  b=0.1/1.67
  alph=1.0+bss*(sal-so0)
  k=1.67e-5*pres/(1.027*3.986*alph)
  f=-k*(1.0+0.5*gams*pres+b*(TK-to0))

  PTS_2_theta=TK*exp(f)-t0

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'PTS_2_theta', PTS_2_theta, 'degC' )
  end if

end function PTS_2_theta

!-----------------------------------------------------
!-----------------------------------------------------

real function OHC( pres, temp, undef, dep_thres, neg, dl )
! Lin et al. (2008) Ѥ, Ǯ̤׻.
! ֤ͤ kJ/cm2 ñ̤֤.
  implicit none
  real, intent(in) :: pres(:)             !  [dBar]
  real, intent(in) :: temp(size(pres))    !  [degC]
  real, intent(in), optional :: undef     ! ̤
  real, intent(in), optional :: dep_thres ! ׻оݤȤǡϿ.
                                          ! ǥե = Ǿؤ.
  logical, intent(in), optional :: neg    ! d26 ޤǤ 26 degC ʲ
                                          ! ʤäƤޤؤΰ.
                                          ! .true. = ؤʬ׻ʤ.
                                          ! .false. = ؤʬ׻.
                                          ! default = .true.
                                          ! .false. ξ, OHC < 0 Ȥʤ
                                          ! ǽ⤢.
  integer, intent(in), optional :: dl  ! ǥХå٥

  real, parameter :: Cpsea=4178.0
  real, parameter :: rhocu=1026.0
  real, parameter :: TOHC=26.0
  integer :: i, j, nz
  real :: d26, dtop, dthres
  real, dimension(size(pres)) :: ttemp
  logical :: neg_flag

  nz=size(pres)

  if(present(dep_thres))then
     dthres=dep_thres
  else
     dthres=0.0
  end if

  if(present(neg))then
     neg_flag=neg
  else
     neg_flag=.true.
  end if

  if(present(undef))then

     !-- determining dtop
     do i=1,nz
        if(temp(i)/=undef)then
           dtop=pres(i)
           exit
        end if
     end do
     if(dthres/=0.0.and.dtop>=dthres)then
        OHC=undef
        return
     end if

     !-- determining d26
     do i=nz,2,-1
        if(temp(i)/=undef.and.temp(i-1)/=undef.and.  &
  &        temp(i)<=TOHC.and.temp(i-1)>=TOHC)then
           d26=pres(i-1)
           exit
        end if
     end do
     if(temp(nz)>=TOHC.and.temp(nz)/=undef)then
        d26=pres(nz)
     end if

     !-- calculating delta T
     do i=1,nz
        if(temp(i)/=undef)then
           ttemp(i)=temp(i)-TOHC
           if(neg_flag.eqv..true.)then
              if(ttemp(i)<0.0)then
                 ttemp(i)=0.0
              end if
           end if
        end if
     end do

     if(d26<=dtop)then
        OHC=undef
        return
     end if

     !-- calculating integration
     call rectangle_int( pres(1:nz), ttemp(1:nz), dtop, d26, OHC, undeff=undef )

     !-- adjusting unit
     if(OHC/=undef)then
        OHC=OHC*Cpsea*rhocu*1.0e-7
     end if

  else

     !-- determining dtop
     dtop=pres(1)
     if(dthres/=0.0.and.dtop>=dthres)then
        OHC=0.0
        return
     end if

     !-- determining d26
     do i=nz,2,-1
        if(temp(i)<=TOHC.and.temp(i-1)>=TOHC)then
           d26=pres(i-1)
           exit
        end if
     end do
     if(temp(nz)>=TOHC)then
        d26=pres(nz)
     end if

     !-- calculating delta T
     do i=1,nz
        ttemp(i)=temp(i)-TOHC
        if(neg_flag.eqv..true.)then
           if(ttemp(i)<0.0)then
              ttemp(i)=0.0
           end if
        end if
     end do

     if(d26<=dtop)then
        OHC=0.0
        return
     end if

     !-- calculating integration
     call rectangle_int( pres(1:nz), ttemp(1:nz), dtop, d26, OHC )

     !-- adjusting unit
     OHC=OHC*Cpsea*rhocu*1.0e-7

  end if

  if(present(dl))then
     call debug_flag_r( dl, 'Thermo_Function', 'OHC', OHC, 'kJ cm-2' )
  end if

  return
end function OHC

!-----------------------------------------------------
!-----------------------------------------------------

!real function




!  return
!end function

!-----------------------------------------------------
!-----------------------------------------------------




end module
