# MathML饤֥
# $Id: x-math.rb 114 2005-08-13 15:04:36Z hiraku $
#
# Copyright (C) 2005, KURODA Hiraku <hiraku@hinet.mydns.jp>
# You can redistribute it and/or modify it under GPL2.
#

module MathML
	class Error < StandardError; end

	# MathMLδ쥯饹
	class Base
		# ̾
		def ename
			''
		end

		# Ǥ°°򵭽Ҥݤϡ" name='value'"Ȥ(Ƭζ)
		def attribute
			''
		end

		# ʸؤѴ
		def to_s
			''
		end
	end

	# Ƥʤδ쥯饹
	class TagOnly < Base
		def to_s
			"<#{ename}#{attribute}/>"
		end
	end

	# <br>ǥ饹
	class Break < TagOnly
		def ename
			'br'
		end

		def attribute
			' xmlns="http://www.w3.org/1999/xhtml"'
		end
	end

	# <none>ǥ饹
	class None < TagOnly
		def ename
			'none'
		end
	end

	# <mspace>ǥ饹
	class Space < TagOnly
		def initialize(space)
			@space = space
		end

		def ename
			'mspace'
		end

		def attribute
			" width='#{@space}'"
		end
	end

	# Ƥĥδ쥯饹
	class WithContent < Base
		# ꤷʸñ̤1ʸǥȤ֤
		def indent(text)
			result = ''
			text.to_s.each_line do |l|
				result << ' '+l
			end
			return result
		end

		# ǤγϥȽλ
		def otag
			"<#{ename}#{attribute}>"
		end

		def ctag
			"</#{ename}>"
		end
	end

	# ʣǤƤȤƻƤ(<math>, <mrow>)δ쥯饹
	class BlockBase < WithContent
		def initialize
			@contents=Array.new
			super
		end

		def <<( content )
			@contents << content if content
			self
		end

		def last
			@contents.last
		end

		def pop
			@contents.pop
		end

		# Ƥ򥤥ǥȤʸˤ֤
		def contents
			indent(@contents.join("\n"))
		end

		def to_s
			@contents.nitems>0 ? %Q[#{otag}\n#{contents}\n#{ctag}] : %Q[#{otag}#{ctag}]
		end

		def pop
			@contents.pop
		end
	end

	# 1ġޤʸƤȤ(<mi>, <mo>)δ쥯饹
	class InlineBase < WithContent
		def <<(c)
			@content = c
			self
		end

		def content
			@content.to_s
		end

		# otag=<hoge>, ctag=</hoge>Ԥ
		def to_s
			c = content
			l = c.count("\n")
			l += 1 if /[^\n]\z/ =~ c
			if l>1 then
				%Q[#{otag}\n#{indent(c)}\n#{ctag}]
			else
				"#{otag}#{c}#{ctag}"
			end
		end
	end

	module Variant
		NON = -1
		NORMAL = 0
		BOLD = 1

		attr_accessor :variant

		def initialize
			super
			@variant = NON
		end

		def attribute
			case @variant
			when NORMAL
				v = 'normal'
			when BOLD
				v = 'bold'
			else
				return ''
			end
			%Q[ mathvariant='#{v}']
		end
	end

	module Align
		CENTER=0
		LEFT=1
		RIGHT=2
	end

	# <math>ǥ饹
	class Math < BlockBase
		def initialize(displaystyle=false)
			super()
			@ds = displaystyle
		end

		def style
			if @ds then
				'block'
			else
				'inline'
			end
		end

		def ename
			'math'
		end

		def attribute
			%Q[ xmlns='http://www.w3.org/1998/Math/MathML' display='#{style}']
		end
	end

	# <mrow>ǥ饹
	class Row < BlockBase
		def ename
			'mrow'
		end
	end

	class Fenced < BlockBase
		attr_accessor :open, :close
		def ename
			'mfenced'
		end

		def attribute
			open = @open ? @open : '.'
			open='{' if open=='\{'
			open='' if open=='.'

			close = @close ? @close : '.'
			close='}' if close=='\}'
			close='' if close=='.'

			r = ""
			r << %Q[ open='#{open}'] 
			r << %Q[ close='#{close}']
		end
	end

	# <mfrac>ǥ饹
	class Frac < InlineBase
		attr_accessor :numerator

		def initialize
			super
			@numerator=nil
		end

		def ename
			'mfrac'
		end

		def content
			r = @numerator ? @numerator.to_s : None.new.to_s
			r << "\n" << @content.to_s
		end
	end

	# <mover>ǥ饹
	class Over < InlineBase
		attr_accessor :over

		def initialize
			super
			@over=nil
		end

		def ename
			'mover'
		end

		def content
			r = %Q[#{@content.to_s}\n#{@over.to_s}]
		end
	end

	# <munder>ǥ饹
	class Under < InlineBase
		attr_accessor :under

		def initialize
			super
			@under=nil
		end

		def ename
			'munder'
		end

		def content
			r = %Q[#{@content.to_s}\n#{@under.to_s}]
		end
	end

	# <msubsup>ǥ饹(<msub>, <msup>ͤ)
	class SubSup < InlineBase
		attr_accessor :sub, :sup

		def initialize
			super
			@sub=nil
			@sup=nil
		end

		def ename_sub
			'sub'
		end

		def ename_sup
			'sup'
		end

		def content
			r = @content.to_s
			r = None.new.to_s if r==''
			r << ("\n" << @sub.to_s) if @sub
			r << ("\n" << @sup.to_s) if @sup
			return r
		end

		def ename
			if @sub && @sup then
				"m#{ename_sub}#{ename_sup}"
			elsif @sub then
				"m#{ename_sub}"
			elsif @sup
				"m#{ename_sup}"
			else
				raise Error.new('No sub&sup.')
			end
		end

		def to_s
			if @sub || @sup then
				super
			else
				@content.to_s
			end
		end
	end

	# <munderover>ǥ饹
	class UnderOver < SubSup
		def ename_sub
			'under'
		end

		def ename_sup
			'over'
		end
	end

	# <mn>ǥ饹
	class Number < InlineBase
		include Variant
		def ename
			"mn"
		end
	end

	# <mi>ǥ饹
	class Identifier < InlineBase
		include Variant
		def ename
			"mi"
		end
	end
	
	# <mo>ǥ饹
	class Operator < InlineBase
		def ename
			"mo"
		end
	end

	# <mtext>ǥ饹
	class Text < InlineBase
		def ename
			"mtext"
		end
	end

	# <msqrt>ǥ饹
	class Sqrt < InlineBase
		def ename
			"msqrt"
		end
	end

	# <mtable>ǥ饹
	class Table < BlockBase
		include Align
		attr_accessor :aligns, :vlines, :hlines
		def initialize
			@aligns = []
			@vlines = []
			@hlines = []
			super
		end

		def ename
			'mtable'
		end

		def attribute
			attr = ""
			astr = ""
			@aligns.each do |i|
				case i
				when CENTER
					astr << " center"
				when LEFT
					astr << " left"
				when RIGHT
					astr << " right"
				end
			end
			attr << " columnalign='#{astr.strip}'" unless astr=~/^( center)*$/

			astr = ""
			@vlines.each do |i|
				astr << (i ? " solid" : " none")
			end
			attr << " columnlines='#{astr.strip}'" unless astr=~/^( none)*$/

			astr = ""
			@hlines.each do |i|
				astr << (i ? " solid" : " none")
			end
			attr << " rowlines='#{astr.strip}'" unless astr=~/^( none)*$/

			attr
		end
	end

	# <mtr>ǥ饹
	class Tr < BlockBase
		def ename
			'mtr'
		end
	end

	# <mtd>ǥ饹
	class Td < BlockBase
		include Align
		attr :align, true

		def initialize
			@align = CENTER
			super
		end

		def ename
			'mtd'
		end

		def attribute
			r = super
			case @align
			when CENTER
				r << ""
			when LEFT
				r << " columnalign='left'"
			when RIGHT
				r << " columnalign='right'"
			end
		end
	end

	# ʲMathMLؤѴѥ饹ʤ
	module LaTeX
		MBEC = /\\.|[^\\]/m

		REG_WHITESPACE = /\A\s+/
		REG_NUMERICS = /\A\s*((\.\d+)|(\d+(\.\d+)?))/
		REG_OPERATORS = /\A\s*([,\.\+\-\*=\/\(\)\[\]<>"'|;:!])/
		REG_ALPHABETS = /\A\s*([a-zA-Z])/
		REG_BLOCK_OPEN = /\A\s*\{/
		REG_BLOCK_CLOSE = /\A\s*\}/
		REG_COMMANDS = /\A\s*\\([a-zA-Z]+|[^a-zA-Z])/
		REG_PARAM = /\A\s*\{\s*(.*?)\s*\}/
		REG_WBSLASH = /\A\s*\\\\/
		REG_BRACES = /\A\s*([.|\[\]\(\)<>])/

		OVERS = {'hat'=>'circ', 'breve'=>'smile', 'grave'=>'grave',
			'acute'=>'acute', 'dot'=>'sdot', 'ddot'=>'nldr', 'tilde'=>'tilde',
			'bar'=>'macr', 'vec'=>'rightarrow', 'check'=>'vee', 'widehat'=>'circ',
			'overline'=>'macr', 'widetilde'=>'tilde', 'overbrace'=>'OverBrace'}
		UNDERS = {'underbrace'=>'UnderBrace', 'underline'=>'macr'}

		# ե̾
		module Fonts
			NORMAL = 0
			BOLD = 1
			BLACKBOLD = 2
			SCRIPT = 3
			FRAKTUR = 4
			ROMAN = 5
		end

		module Misc
			class BraceNotClosed < StandardError; end
			class NotEnvironment < StandardError; end
			class EnvironmentNotEnd < StandardError; end

			def slice_environment(src)
				raise NotEnvironment unless src[REG_COMMANDS, 1]=="begin"
				s = src.dup
				r = slice_block!(s)
				until s=~/\A\s*\z/ || s[REG_COMMANDS, 1]=="end"
					r << slice_block!(s, true)
				end
				raise EnvironmentNotEnd unless s[REG_COMMANDS, 1]=="end"
				r << slice_block!(s) << slice_block!(s)
				r
			end

			def slice_environment!(src)
				env = slice_environment(src)
				src.slice!(0, env.size)
				env
			end

			def slice_block(src, slice_env=false)
				case src
				when /\A\s*\z/
					src.slice(/\s*/)
				when REG_COMMANDS
					if slice_env && src[REG_COMMANDS, 1]=="begin"
						slice_environment(src)
					else
						src.slice(/(#{REG_COMMANDS})/, 1)
					end
				when REG_BLOCK_OPEN
					nest=0
					block=""
					src.scan(/\G(#{MBEC}*?)([\{\}])/) do |m|
						case m[1]
						when "{"
							nest+=1
						when "}"
							nest-=1
							if nest==0
								block = $`+m[0]+m[1]
								break
							end
						end
					end
					raise BraceNotClosed if nest>0
					block
				else
					src.slice(/\A(\s*.)/)
				end
			end

			def slice_block!(src, slice_env=false)
				block = slice_block(src, slice_env)
				src.slice!(0, block.size)
				block
			end

			REG_BRACED = /\A\s*\{(.*?)\}\s*\z/
			def slice_brace!(src)
				src.sub!(REG_BRACED){$1} if src=~REG_BRACED
				src
			end
		end

		class ParseError < StandardError
			attr_accessor :correct, :remain
			def initialize(message, remain = "", correct = "")
				@correct = correct
				@remain = remain
				super(message)
			end

			def inspect
				"#{message} : '#{@correct}' / '#{@remain}'\n"+backtrace[0..5].join("\n")
			end
		end

		class Parser
			include LaTeX
			include LaTeX::Misc

			attr_accessor :unsecure_entity
			def initialize
				@unsecure_entity = false
				@entity_list = Hash.new
			end

			def add_entity_list(list)
				list.each do |i|
					@entity_list[i] = true
				end
			end

			def parse(src, displaystyle=false)
				@src = src
				@ds = displaystyle
				begin
					original = @src.clone
					@src = @src.gsub(/\\\\/){"\001"}.
						gsub(/\\%/){"\002"}.
						gsub(/%.*$/){''}.
						gsub(/\002/){'\\%'}.
						gsub(/\001/){'\\\\'}
					m = Math.new(@ds)
					until @src =~ /\A\s*\z/
						m << private_parse(@src, Fonts::NORMAL)
					end
					m.to_s
				rescue ParseError => e
					e.correct = original[0...(original.size-e.remain.size)]
					raise
				end
			end

			private
			def private_parse(src, font=nil)
				orig = [@src, @elist, @font]
				@src = src
				@elist = Array.new
				@font = font if font
				begin
					until @src=~/\A\s*\z/
						@elist << parse_to_element
					end
					@elist
				rescue BraceNotClosed => e
					raise ParseError.new("Brace not closed.", @src)
				rescue NotEnvironment => e
					raise ParseError.new("Not environment.", @src)
				rescue EnvironmentNotEnd => e
					raise ParseError.new("Environment not end.", @src)
				rescue ParseError => e
					e.remain += @src
					raise
				ensure
					if orig
						@src, @elist, @font = orig
					end
				end
			end

			def parse_block(message="Syntax error.", font=nil)
				block = slice_block!(@src)
				raise ParseError.new(message) if block.size==0
				r = private_parse(block, font)[0]
			end

			def parse_to_element
				case @src
				when REG_NUMERICS
					parse_num
				when REG_ALPHABETS
					parse_char
				when REG_OPERATORS
					parse_operator
				when REG_BLOCK_OPEN
					begin
						Row.new << private_parse(slice_block!(@src)[/\A\{(.*)\}\z/, 1])
					rescue BraceNotClosed
						raise ParseError.new("Brace not closed.")
					rescue ParseError => e
						e.remain = "}"+e.remain
						raise
					end
				when /\A\s*_/
					parse_sub
				when /\A\s*\^/
					parse_sup
				when REG_COMMANDS
					parse_command
				else
					raise ParseError.new('Syntax error.')
				end
			end

			def parse_num
				n = Number.new
				n.variant = (@font==Fonts::BOLD) ? Variant::BOLD : Variant::NON
				n << @src.slice!(REG_NUMERICS).slice(REG_NUMERICS, 1)
			end

			def parse_char
				c = @src.slice!(REG_ALPHABETS).slice(REG_ALPHABETS, 1)
				case @font
				when Fonts::ROMAN
					v = Variant::NORMAL
				when Fonts::BOLD
					v = Variant::BOLD
				when Fonts::BLACKBOLD
					c = %Q[&#{c}opf;]
				when Fonts::SCRIPT
					c = %Q[&#{c}scr;]
				when Fonts::FRAKTUR
					c = %Q[&#{c}fr;]
				else
					v = Variant::NON
				end
				i = Identifier.new
				i.variant = v
				i << c
			end

			def parse_operator
				o = @src.slice!(REG_OPERATORS).slice(REG_OPERATORS,1)
				case o
				when '<'
					o ='&lt;'
				when '>'
					o = '&gt;'
				end
				Operator.new << o
			end

			def parse_sub
				if @elist.last.is_a?(SubSup)
					ss = @elist.pop
				else
					e = @elist.pop
					ss = SubSup.new
					ss << e
				end

				raise ParseError.new("Double subscript.") if ss.sub
				@src.slice!(/\A\s*\_/)
				ss.sub = parse_block("Subscript not exist.")
				ss
			end

			def parse_sup
				if @elist.last.is_a?(SubSup) then
					ss = @elist.pop
				else
					l = @elist.pop
					ss = SubSup.new
					ss << l
				end

				raise ParseError.new("Double superscript.") if ss.sup != nil
				@src.slice!(/\A\s*\^/)
				ss.sup = parse_block("Superscript not exist.")
				ss
			end

			def entity(str)
				str.sub(/^(.*)$/){"&#{$1};"}
			end

			def parse_symbol_command(com, plain=false)
				unless SymbolCommands.include?(com)
					@src = "\\"+com+@src
					raise ParseError.new("Undefined command.") 
				end
				data = SymbolCommands[com]
				return nil unless data

				su = data[0]
				el = data[1]
				el = :o unless el
				s = data[2]
				s = com.to_sym unless s
				s = com if s.class==String && s.length==0

				case el
				when :I
					el = Identifier.new
				when :i
					el = Identifier.new
					el.variant = Variant::NORMAL unless s.class==String&&s.length>1
				when :o
					el = Operator.new
				when :n
					el = Number.new
				else
					raise ParseError.new("Inner data broken.")
				end

				case s
				when String
				when Fixnum
					s = "&\#x#{s.to_s(16)};"
				when Symbol
					s = "&#{s.to_s};"
				end

				return s if plain
				el << s

				case su
				when :s
					SubSup.new << el
				when :u
					UnderOver.new << el
				else
					raise ParseError.new("Inner data broken.")
				end
			end

			def parse_command
				com = @src.slice!(REG_COMMANDS).slice(REG_COMMANDS, 1)

				if OVERS.key?(com)
					o = Over.new
					o.over = (Operator.new << entity(OVERS[com]))
					o << parse_block
					return o
				end

				if UNDERS.key?(com)
					u = Under.new
					u.under = (Operator.new << entity(UNDERS[com]))
					u << parse_block
					return u
				end

				case com
				when '\\'
					@ds ? nil : Break.new
				when 'entity'
					parse_user_entity
				when 'stackrel'
					o = Over.new
					o.over = parse_block
					o << parse_block
				when 'quad'
					Space.new("1em")
				when 'qquad'
					Space.new("2em")
				when ','
					Space.new("0.167em")
				when ':'
					Space.new("0.222em")
				when ';'
					Space.new("0.278em")
				when 'it'
					@font = Fonts::NORMAL
					nil
				when 'rm'
					@font = Fonts::ROMAN
					nil
				when 'bf'
					@font = Fonts::BOLD
					nil
				when 'mathit'
					Row.new << parse_block("Syntax error.", Fonts::NORMAL)
				when 'mathrm'
					Row.new << parse_block("Syntax error.", Fonts::ROMAN)
				when 'mathbf'
					Row.new << parse_block("Syntax error.", Fonts::BOLD)
				when 'mathbb'
					Row.new << parse_block("Syntax error.", Fonts::BLACKBOLD)
				when 'mathscr'
					Row.new << parse_block("Syntax error.", Fonts::SCRIPT)
				when 'mathfrak'
					Row.new << parse_block("Syntax error.", Fonts::FRAKTUR)
				when 'frac'
					f = Frac.new
					f.numerator = parse_block
					f << parse_block
				when 'sqrt'
					s = Sqrt.new
					s << parse_block
				when 'mbox'
					parse_mbox
				when 'left'
					parse_left_right("right")
				when 'bigg'
					parse_left_right("bigg")
				when 'begin'
					parse_environment
				else
					parse_symbol_command(com)
				end
			end

			def parse_user_entity
				raise ParseError.new("Need parameter.") unless @src=~REG_PARAM
				param = slice_block!(@src)[/\A\{\s*(.*)\s*\}\z/, 1]
				unless @unsecure_entity || @entity_list[param]
					@src = param+"}"+@src
					raise ParseError.new("Unregistered entity.")
				end
				Operator.new << entity(param)
			end

			def parse_mbox
				Text.new << slice_block!(@src)[/\A\{(.*)\}\z/, 1]
			end

			def parse_left_right(right)
				f = Fenced.new
				o = slice_block!(@src)
				raise ParseError.new('Need brace here.') unless o=~REG_BRACES || Delimiters.include?(o[REG_COMMANDS, 1])

				f.open = (o=~REG_BRACES ? o : parse_symbol_command(o[REG_COMMANDS, 1], true))
				body = ""
				until @src=~/\A\s*\\#{right}/
					body << slice_block!(@src, true)
					raise ParseError.new('Brace unclosed.') if @src.size==0
				end
				f << (Row.new << private_parse(body))
				@src.slice!(/\A\s*\\#{right}/)
				c = slice_block!(@src)
				raise ParseError.new('Need brace here.') unless c=~REG_BRACES || Delimiters.include?(c[REG_COMMANDS, 1])
				f.close = (c=~REG_BRACES ? c : parse_symbol_command(c[REG_COMMANDS, 1], true))
				f
			end

			def parse_environment
				font_orig = @font
				begin
					en = slice_block!(@src).sub(/\A\{(.*)\}\z/){$1}
					raise ParseError.new('Environment name not exist.') if en==""

					case en
					when 'array'
						parse_array
					else
						raise ParseError.new("Undefined environment.")
					end
				ensure
					@font = font_orig
				end
			end

			def parse_array
				layout = slice_block!(@src)
				l = slice_brace!(layout.dup)
				t = Table.new
				vlined=(l=~/\A\s*\|/)
				columned=false
				until l=~/\A\s*\z/
					c = slice_block!(l).strip
					raise ParseError.new("Syntax error.", layout.sub(/\A\s*\{.*?\s*\}/, c+l+"}")) unless c=~/[clr\|@]/

					if c=='|'
						t.aligns << Align::CENTER if vlined
						t.vlines << true
						vline = true
						columned=false
					else
						t.vlines << false if columned
						vline=false
						columned=true
						case c
						when 'l'
							t.aligns << Align::LEFT
						when 'c'
							t.aligns << Align::CENTER
						when 'r'
							t.aligns << Align::RIGHT
						when '@'
							t.aligns << Align::CENTER
							slice_block!(l)
						end
					end
				end
				slice_brace!(layout)
				raise ParseError.new('Need parameter here.') if layout==""

				row_parsed = false
				hlined = false
				until @src[REG_COMMANDS, 1]=="end"
					raise ParseError.new('Matching \end not exist.') if @src=~/\A\s*\z/
					if @src[REG_COMMANDS, 1]=="hline"
						slice_block!(@src)
						t << Tr.new unless row_parsed
						t.hlines << true
						row_parsed = false
						hlined = true
					else
						t.hlines << false if row_parsed
						t << parse_table_row(layout.dup)
						@src.slice!(REG_WBSLASH)
						row_parsed = true
						hlined = false
					end
				end
				if hlined
					tr = Tr.new
					(0..t.vlines.size).each {|i| tr << Td.new}
					t << tr
				end
				raise ParseError.new("Need \\end{array}.") unless slice_block(@src)=~/\A\s*\\end\z/
				slice_block!(@src)
				raise ParseError.new("Environment mismatched.") unless slice_block(@src)=~/\A\s*\{\s*array\s*\}/
				slice_block!(@src)
				t
			end

			def parse_table_row(layout)
				r = Tr.new
				first_column = true
				vlined = (slice_block(layout).strip=='|')
				until layout=~/\A\s*\z/
					c = slice_block!(layout).strip
					if c=='|'
						r << Td.new if vlined
						vlined = true
						next
					else
						vlined = false

						case c
						when 'r', 'l', 'c'
						when '@'
							r << (Td.new << private_parse(slice_block!(layout)))
							next
						end
						if first_column
							first_column = false
						else
							unless (b=slice_block!(@src))=~/\A\s*&/
								raise ParseError.new("Need more column.", b)
							end
						end
						col = ""
						until @src[REG_COMMANDS, 1]=="end" || @src=~/\A\s*(&|\\\\)/ || @src=~/\A\s*\z/
							col << slice_block!(@src, true)
						end
						r << (Td.new << private_parse(col))
					end
				end
				r << Td.new if vlined
				raise ParseError.new("Too many column.") if @src=~/\A\s*&/
				r
			end
		end
	end
end

# Automatically generated constants
module MathML
	module LaTeX
SymbolCommands={
"{"=>[:s,:o,""],
"}"=>[:s,:o,""],
"#"=>[:s,:o,""],
"$"=>[:s,:o,""],
"&"=>[:s,:o,:amp],
"_"=>[:s,:o,""],
"%"=>[:s,:o,""],
","=>nil,
"varepsilon"=>[:s,:I],
"mathdollar"=>[:s,:o,"$"],
"lbrace"=>[:s],
"rbrace"=>[:s],
"P"=>[:s,:o,:para],
"mathparagraph"=>[:s,:o,:para],
"S"=>[:s,:o,:sect],
"mathsection"=>[:s,:o,:sect],
"dag"=>[:s,:o,:dagger],
"dagger"=>[:s],
"ddag"=>[:s,:o,:ddagger],
"ddagger"=>[:s],
"copyright"=>[:s,:o,:copy],
"pounds"=>[:s,:o,:pound],
"mathsterling"=>[:s,:o,:pound],
"dots"=>[:s,:o,:mldr],
"mathellipsis"=>[:s,:o,:mldr],
"ldots"=>[:s,:o,:mldr],
"ensuremath"=>nil,
"|"=>[:s,:o,:Vert],
"mho"=>[:s],
"Join"=>[:s,:o,:bowtie],
"Box"=>[:s,:o,:square],
"Diamond"=>[:s],
"leadsto"=>[:s,:o,:zigrarr],
"sqsubset"=>[:s],
"sqsupset"=>[:s],
"lhd"=>[:s,:o,:vltri],
"unlhd"=>[:s,:o,:ltrie],
"rhd"=>[:s,:o,:vrtri],
"unrhd"=>[:s,:o,:rtrie],
"log"=>[:s,:i,""],
"lg"=>[:s,:i,""],
"ln"=>[:s,:i,""],
"lim"=>[:u,:i,""],
"limsup"=>[:u,:i,"lim sup"],
"liminf"=>[:u,:i,"lim inf"],
"sin"=>[:s,:i,""],
"arcsin"=>[:s,:i,""],
"sinh"=>[:s,:i,""],
"cos"=>[:s,:i,""],
"arccos"=>[:s,:i,""],
"cosh"=>[:s,:i,""],
"tan"=>[:s,:i,""],
"arctan"=>[:s,:i,""],
"tanh"=>[:s,:i,""],
"cot"=>[:s,:i,""],
"coth"=>[:s,:i,""],
"sec"=>[:s,:i,""],
"csc"=>[:s,:i,""],
"max"=>[:u,:i,""],
"min"=>[:u,:i,""],
"sup"=>[:u,:i,""],
"inf"=>[:u,:i,""],
"arg"=>[:s,:i,""],
"ker"=>[:s,:i,""],
"dim"=>[:s,:i,""],
"hom"=>[:s,:i,""],
"det"=>[:u,:i,""],
"exp"=>[:s,:i,""],
"Pr"=>[:u,:i,""],
"gcd"=>[:u,:i,""],
"deg"=>[:s,:i,""],
"prime"=>[:s],
"alpha"=>[:s,:I],
"beta"=>[:s,:I],
"gamma"=>[:s,:I],
"delta"=>[:s,:I],
"epsilon"=>[:s,:I],
"zeta"=>[:s,:I],
"eta"=>[:s,:I],
"theta"=>[:s,:I],
"iota"=>[:s,:I],
"kappa"=>[:s,:I],
"lambda"=>[:s,:I],
"mu"=>[:s,:I],
"nu"=>[:s,:I],
"xi"=>[:s,:I],
"pi"=>[:s,:I],
"rho"=>[:s,:I],
"sigma"=>[:s,:I],
"tau"=>[:s,:I],
"upsilon"=>[:s,:I],
"phi"=>[:s,:I],
"chi"=>[:s,:I],
"psi"=>[:s,:I],
"omega"=>[:s,:I],
"vartheta"=>[:s,:I],
"varpi"=>[:s,:I],
"varrho"=>[:s,:I],
"varsigma"=>[:s,:I],
"varphi"=>[:s,:I],
"Gamma"=>[:s,:i],
"Delta"=>[:s,:i],
"Theta"=>[:s,:i],
"Lambda"=>[:s,:i],
"Xi"=>[:s,:i],
"Pi"=>[:s,:i],
"Sigma"=>[:s,:i],
"Upsilon"=>[:s,:i,:Upsi],
"Phi"=>[:s,:i],
"Psi"=>[:s,:i],
"Omega"=>[:s,:i],
"aleph"=>[:s,:i],
"hbar"=>[:s,:i,:hslash],
"imath"=>[:s,:i],
"jmath"=>[:s,:i],
"ell"=>[:s],
"wp"=>[:s],
"Re"=>[:s,:i],
"Im"=>[:s,:i],
"partial"=>[:s,:o,:part],
"infty"=>[:s,:n,:infin],
"emptyset"=>[:s,:i,:empty],
"nabla"=>[:s,:i],
"surd"=>[:s,:o,:Sqrt],
"top"=>[:s],
"bot"=>[:s],
"angle"=>[:s],
"not"=>[:s],
"triangle"=>[:s],
"forall"=>[:s],
"exists"=>[:s,:o,:exist],
"neg"=>[:s,:o,:not],
"lnot"=>[:s,:o,:not],
"flat"=>[:s],
"natural"=>[:s],
"sharp"=>[:s],
"clubsuit"=>[:s],
"diamondsuit"=>[:s],
"heartsuit"=>[:s],
"spadesuit"=>[:s],
"coprod"=>[:u],
"bigvee"=>[:u],
"bigwedge"=>[:u],
"biguplus"=>[:u],
"bigcap"=>[:u],
"bigcup"=>[:u],
"intop"=>[:u,:o,:int],
"int"=>[:s,:o],
"prod"=>[:u],
"sum"=>[:u],
"bigotimes"=>[:u],
"bigoplus"=>[:u],
"bigodot"=>[:u],
"ointop"=>[:u,:o,:oint],
"oint"=>[:s],
"bigsqcup"=>[:u],
"smallint"=>[:u,:o,:int],
"triangleleft"=>[:s],
"triangleright"=>[:s],
"bigtriangleup"=>[:s],
"bigtriangledown"=>[:s],
"wedge"=>[:s],
"land"=>[:s,:o,:wedge],
"vee"=>[:s],
"lor"=>[:s,:o,:vee],
"cap"=>[:s],
"cup"=>[:s],
"sqcap"=>[:s],
"sqcup"=>[:s],
"uplus"=>[:s],
"amalg"=>[:s],
"diamond"=>[:s],
"bullet"=>[:s],
"wr"=>[:s],
"div"=>[:s],
"odot"=>[:s],
"oslash"=>[:s],
"otimes"=>[:s],
"ominus"=>[:s],
"oplus"=>[:s],
"mp"=>[:s],
"pm"=>[:s],
"circ"=>[:s,:o,:cir],
"bigcirc"=>[:s],
"setminus"=>[:s],
"cdot"=>[:s],
"ast"=>[:s],
"times"=>[:s],
"star"=>[:s],
"propto"=>[:s],
"sqsubseteq"=>[:s],
"sqsupseteq"=>[:s],
"parallel"=>[:s],
"mid"=>[:s],
"dashv"=>[:s],
"vdash"=>[:s],
"nearrow"=>[:s],
"searrow"=>[:s],
"nwarrow"=>[:s],
"swarrow"=>[:s],
"Leftrightarrow"=>[:s],
"Leftarrow"=>[:s],
"Rightarrow"=>[:s],
"neq"=>[:s,:o,:ne],
"ne"=>[:s],
"leq"=>[:s],
"le"=>[:s],
"geq"=>[:s],
"ge"=>[:s],
"succ"=>[:s],
"prec"=>[:s],
"approx"=>[:s],
"succeq"=>[:s,:o,:sccue],
"preceq"=>[:s,:o,:prcue],
"supset"=>[:s],
"subset"=>[:s],
"supseteq"=>[:s],
"subseteq"=>[:s],
"in"=>[:s],
"ni"=>[:s],
"owns"=>[:s,:o,:ni],
"gg"=>[:s],
"ll"=>[:s],
"leftrightarrow"=>[:s],
"leftarrow"=>[:s],
"gets"=>[:s,:o,:leftarrow],
"rightarrow"=>[:s],
"to"=>[:s,:o,:rightarrow],
"mapstochar"=>[:s,:o,:vdash],
"mapsto"=>[:s],
"sim"=>[:s],
"simeq"=>[:s],
"perp"=>[:s],
"equiv"=>[:s],
"asymp"=>[:s],
"smile"=>[:s],
"frown"=>[:s],
"leftharpoonup"=>[:s],
"leftharpoondown"=>[:s],
"rightharpoonup"=>[:s],
"rightharpoondown"=>[:s],
"cong"=>[:s],
"notin"=>[:s],
"rightleftharpoons"=>[:s],
"doteq"=>[:s],
"joinrel"=>nil,
"relbar"=>[:s,:o,"-"],
"Relbar"=>[:s,:o,"="],
"lhook"=>[:s,:o,:sub],
"hookrightarrow"=>[:s],
"rhook"=>[:s,:o,:sup],
"hookleftarrow"=>[:s],
"bowtie"=>[:s],
"models"=>[:s],
"Longrightarrow"=>[:s],
"longrightarrow"=>[:s],
"longleftarrow"=>[:s],
"Longleftarrow"=>[:s],
"longmapsto"=>[:s,:o,:mapsto],
"longleftrightarrow"=>[:s],
"Longleftrightarrow"=>[:s],
"iff"=>[:s],
"ldotp"=>[:s,:o,"."],
"cdotp"=>[:s,:o,:cdot],
"colon"=>[:s],
"cdots"=>[:s,:o,:ctdot],
"vdots"=>[:s,:o,:vellip],
"ddots"=>[:s,:o,:dtdot],
"braceld"=>[:s,:o,0x25dc],
"bracerd"=>[:s,:o,0x25dd],
"bracelu"=>[:s,:o,0x25df],
"braceru"=>[:s,:o,0x25de],
"lmoustache"=>[:s],
"rmoustache"=>[:s],
"arrowvert"=>[:s,:o,:vert],
"Arrowvert"=>[:s,:o,:Vert],
"Vert"=>[:s],
"vert"=>[:s],
"uparrow"=>[:s],
"downarrow"=>[:s],
"updownarrow"=>[:s],
"Uparrow"=>[:s],
"Downarrow"=>[:s],
"Updownarrow"=>[:s],
"backslash"=>[:s,:o,"\\"],
"rangle"=>[:s],
"langle"=>[:s],
"rceil"=>[:s],
"lceil"=>[:s],
"rfloor"=>[:s],
"lfloor"=>[:s],
"lgroup"=>[:s,:o,0x2570],
"rgroup"=>[:s,:o,0x256f],
"bracevert"=>[:s,:o,:vert],
"mathunderscore"=>[:s,:o,"_"],
"square"=>[:s],
"rightsquigarrow"=>[:s],
"lozenge"=>[:s],
"vartriangleright"=>[:s],
"vartriangleleft"=>[:s],
"trianglerighteq"=>[:s],
"trianglelefteq"=>[:s],
"boxdot"=>[:s,:o,:dotsquare],
"boxplus"=>[:s],
"boxtimes"=>[:s],
"blacksquare"=>[:s],
"centerdot"=>[:s],
"blacklozenge"=>[:s],
"circlearrowright"=>[:s],
"circlearrowleft"=>[:s],
"leftrightharpoons"=>[:s],
"boxminus"=>[:s],
"Vdash"=>[:s],
"Vvdash"=>[:s],
"vDash"=>[:s],
"twoheadrightarrow"=>[:s],
"twoheadleftarrow"=>[:s],
"leftleftarrows"=>[:s],
"rightrightarrows"=>[:s],
"upuparrows"=>[:s],
"downdownarrows"=>[:s],
"upharpoonright"=>[:s],
"restriction"=>[:s,:o,:upharpoonright],
"downharpoonright"=>[:s],
"upharpoonleft"=>[:s],
"downharpoonleft"=>[:s],
"rightarrowtail"=>[:s],
"leftarrowtail"=>[:s],
"leftrightarrows"=>[:s],
"rightleftarrows"=>[:s],
"Lsh"=>[:s],
"Rsh"=>[:s],
"leftrightsquigarrow"=>[:s],
"looparrowleft"=>[:s],
"looparrowright"=>[:s],
"circeq"=>[:s],
"succsim"=>[:s],
"gtrsim"=>[:s],
"gtrapprox"=>[:s],
"multimap"=>[:s],
"therefore"=>[:s],
"because"=>[:s],
"doteqdot"=>[:s],
"Doteq"=>[:s,:o,:doteqdot],
"triangleq"=>[:s],
"precsim"=>[:s],
"lesssim"=>[:s],
"lessapprox"=>[:s],
"eqslantless"=>[:s],
"eqslantgtr"=>[:s],
"curlyeqprec"=>[:s],
"curlyeqsucc"=>[:s],
"preccurlyeq"=>[:s],
"leqq"=>[:s],
"leqslant"=>[:s,:o,:leq],
"lessgtr"=>[:s],
"backprime"=>[:s],
"risingdotseq"=>[:s],
"fallingdotseq"=>[:s],
"succcurlyeq"=>[:s],
"geqq"=>[:s],
"geqslant"=>[:s,:o,:geq],
"gtrless"=>[:s],
"bigstar"=>[:s],
"between"=>[:s],
"blacktriangledown"=>[:s],
"blacktriangleright"=>[:s],
"blacktriangleleft"=>[:s],
"vartriangle"=>[:s,:o,:triangle],
"blacktriangle"=>[:s],
"triangledown"=>[:s],
"eqcirc"=>[:s],
"lesseqgtr"=>[:s],
"gtreqless"=>[:s],
"lesseqqgtr"=>[:s],
"gtreqqless"=>[:s],
"Rrightarrow"=>[:s],
"Lleftarrow"=>[:s],
"veebar"=>[:s],
"barwedge"=>[:s],
"doublebarwedge"=>[:s],
"measuredangle"=>[:s],
"sphericalangle"=>[:s,:o,:angsph],
"varpropto"=>[:s],
"smallsmile"=>[:s,:o,:smile],
"smallfrown"=>[:s,:o,:frown],
"Subset"=>[:s],
"Supset"=>[:s],
"Cup"=>[:s],
"doublecup"=>[:s,:o,:Cup],
"Cap"=>[:s],
"doublecap"=>[:s,:o,:Cap],
"curlywedge"=>[:s],
"curlyvee"=>[:s],
"leftthreetimes"=>[:s],
"rightthreetimes"=>[:s],
"subseteqq"=>[:s],
"supseteqq"=>[:s],
"bumpeq"=>[:s],
"Bumpeq"=>[:s],
"lll"=>[:s,:o,:Ll],
"llless"=>[:s,:o,:Ll],
"ggg"=>[:s],
"gggtr"=>[:s,:o,:ggg],
"circledS"=>[:s],
"pitchfork"=>[:s],
"dotplus"=>[:s],
"backsim"=>[:s],
"backsimeq"=>[:s],
"complement"=>[:s],
"intercal"=>[:s],
"circledcirc"=>[:s],
"circledast"=>[:s],
"circleddash"=>[:s],
"lvertneqq"=>[:s,:o,:lneqq],
"gvertneqq"=>[:s,:o,:gneqq],
"nleq"=>[:s,:o,0x2270],
"ngeq"=>[:s,:o,0x2271],
"nless"=>[:s],
"ngtr"=>[:s],
"nprec"=>[:s],
"nsucc"=>[:s],
"lneqq"=>[:s],
"gneqq"=>[:s],
"nleqslant"=>[:s],
"ngeqslant"=>[:s],
"lneq"=>[:s],
"gneq"=>[:s],
"npreceq"=>[:s,:o,:nprcue],
"nsucceq"=>[:s,:o,:nsccue],
"precnsim"=>[:s],
"succnsim"=>[:s],
"lnsim"=>[:s],
"gnsim"=>[:s],
"nleqq"=>[:s],
"ngeqq"=>[:s],
"precneqq"=>[:s,:o,"<mfrac mathsize='1%' linethickness='0'><mo>&prec;</mo><mo>&ne;</mo></mfrac>"],
"succneqq"=>[:s,:o,"<mfrac mathsize='1%' linethickness='0'><mo>&succ;</mo><mo>&ne;</mo></mfrac>"],
"precnapprox"=>[:s],
"succnapprox"=>[:s],
"lnapprox"=>[:s,:o,"<mfrac mathsize='1%' linethickness='0'><mo>&lt;</mo><mo>&nap;</mo></mfrac>"],
"gnapprox"=>[:s,:o,"<mfrac mathsize='1%' linethickness='0'><mo>&gt;</mo><mo>&nap;</mo></mfrac>"],
"nsim"=>[:s],
"ncong"=>[:s],
"diagup"=>[:s,:o,0x2571],
"diagdown"=>[:s,:o,0x2572],
"varsubsetneq"=>[:s,:o,:subsetneq],
"varsupsetneq"=>[:s,:o,:supsetneq],
"nsubseteqq"=>[:s],
"nsupseteqq"=>[:s],
"subsetneqq"=>[:s],
"supsetneqq"=>[:s],
"varsubsetneqq"=>[:s,:o,:subsetneqq],
"varsupsetneqq"=>[:s,:o,:supsetneqq],
"subsetneq"=>[:s],
"supsetneq"=>[:s],
"nsubseteq"=>[:s],
"nsupseteq"=>[:s],
"nparallel"=>[:s],
"nmid"=>[:s],
"nshortmid"=>[:s,:o,:nmid],
"nshortparallel"=>[:s,:o,:nparallel],
"nvdash"=>[:s],
"nVdash"=>[:s],
"nvDash"=>[:s],
"nVDash"=>[:s],
"ntrianglerighteq"=>[:s],
"ntrianglelefteq"=>[:s],
"ntriangleleft"=>[:s],
"ntriangleright"=>[:s],
"nleftarrow"=>[:s],
"nrightarrow"=>[:s],
"nLeftarrow"=>[:s],
"nRightarrow"=>[:s],
"nLeftrightarrow"=>[:s],
"nleftrightarrow"=>[:s],
"divideontimes"=>[:s],
"varnothing"=>[:s],
"nexists"=>[:s],
"Finv"=>[:s,:o,0x2132],
"Game"=>[:s,:o,"G"],
"eth"=>[:s],
"eqsim"=>[:s],
"beth"=>[:s],
"gimel"=>[:s],
"daleth"=>[:s],
"lessdot"=>[:s],
"gtrdot"=>[:s],
"ltimes"=>[:s],
"rtimes"=>[:s],
"shortmid"=>[:s,:o,:mid],
"shortparallel"=>[:s],
"smallsetminus"=>[:s,:o,:setminus],
"thicksim"=>[:s,:o,:sim],
"thickapprox"=>[:s,:o,:approx],
"approxeq"=>[:s],
"succapprox"=>[:s],
"precapprox"=>[:s],
"curvearrowleft"=>[:s],
"curvearrowright"=>[:s],
"digamma"=>[:s],
"varkappa"=>[:s],
"Bbbk"=>[:s,:i,:kopf],
"hslash"=>[:s],
"backepsilon"=>[:s],
"ulcorner"=>[:s,:o,:boxdr],
"urcorner"=>[:s,:o,:boxdl],
"llcorner"=>[:s,:o,:boxur],
"lrcorner"=>[:s,:o,:boxul],
}
Delimiters=[
"lmoustache",
"rmoustache",
"arrowvert",
"Arrowvert",
"Vert",
"vert",
"uparrow",
"downarrow",
"updownarrow",
"Uparrow",
"Downarrow",
"Updownarrow",
"backslash",
"rangle",
"langle",
"rbrace",
"lbrace",
"rceil",
"lceil",
"rfloor",
"lfloor",
"lgroup",
"rgroup",
"bracevert",
"ulcorner",
"urcorner",
"llcorner",
"lrcorner",
"{",
"|",
"}",
]
	end
end
