
require "numru/ganalysis/planet"
require "numru/ganalysis/met"

module NumRu
  module GAnalysis

    # Library to handle the sigma coordinate of atmospheric GCMs
    # 
    module SigmaCoord

      module_function

      # Find the sigma coordinate in a GPhys object (empirically)
      # 
      # ARGUMENT
      # * gphys [GPhys]
      # * error [nil/false or true] change the behavior if a
      #   pressure coordinate is not found. Default: returns nil;
      #   if error is true, an exception is raised.
      # RETURN VALUE
      # * Integer to indicate the dimension of the sigma coordinate,
      #   or nil if not found (by default; see above)
      def find_sigma_d(gphys, error=nil)
        un0 = Units.new("1")
        (gphys.rank-1).downto(0) do |d|
          crd = gphys.coord(d)
          if crd.units =~ un0 and 
             ( crd.get_att('standard_name')=="atmosphere_sigma_coordinate" or
               /sigma/i =~ crd.long_name or /^sig/i =~ crd.name )
            # then, it's assumed that this is the sigma coordinate
            return(d)
          end
        end
        if error
          raise("Could not find a pressure coordinate.")
        else
          nil
        end
      end

      # Derive pressure from sigma and surface pressure
      # 
      # ARGUMENTS
      # * ps [GPhys or VArray or NArray] : surface pressure (multi-D)
      # * sig [GPhys or VArray or NArray] : sigma (1D)
      # * zdim [Integer] : dimension at which the vertical dim is incorpolated
      # RETURN VALUE
      # * pressure [GPhys or VArray or NArray depending the class of ps]
      def sig_ps2p(ps, sig, zdim)
        if sig.is_a?(VArray) or sig.is_a?(GPhys)
          sigval = sig.val
        else
          sigval = sig
        end
        if ps.is_a?(VArray) or ps.is_a?(GPhys)
          psval = ps.val
        else
          psval = ps
        end
        psval = psval.newdim(zdim)
        zdim.times{sigval = sigval.newdim(0)}
        (ps.rank-zdim).times{sigval = sigval.newdim(-1)}
        pval = psval * sigval
        case ps
        when VArray, GPhys
          p = VArray.new(pval, 
                         {"long_name"=>"pressure","units"=>ps.units.to_s}, "p")
          if ps.is_a?(GPhys)
            case sig
            when GPhys
              sigax = sig.axis(0)
            when VArray
              sigax =  Axis.new().set_pos(sig)
            else
              sigc = VArray.new(sig,{"units"=>"","long_name"=>"sigma"},"sigma")
              sigax =  Axis.new().set_pos(sigc)
            end
            grid = ps.grid.insert_axis(zdim,sigax)
            p = GPhys.new(grid, p)
          end
          p
        else
          pval
        end
      end

    end
  end
end
