/*
 * File:
 *
 *	option.c
 *
 * Author:
 *
 *	Brad Rullman
 *	Department of Computer Science  FR-35
 *	University of Washington
 *	Seattle, Washington  98155
 *	email: ecola@cs.washington.edu
 *
 *	Copyright @ March, 1987 - This program, or any subset of the functions
 *	herein, is not to be redistributed, or used for personal or commercial
 *	gain in the form of fame, fortune, or shorter working hours, without
 *	the consent of the author.
 *
 * Function:
 *
 *	A file of functions which implement the interface to the
 *	Option subwindow.
 *
 * Contents:
 *
 *   Public:
 *
 *	InitOptionSW		Creates the panel and panel items.
 *	FixOptionSW		Resets the size and position of OptionSW.
 *	UpdateViewFields	Updates the "View Width" and "Height" fields.
 *
 *   Private:
 *
 *	openFont		Opens a font's pxl file.
 *	clearCharField		Clears char field that is not being typed in.
 *	loadChar		Loads a char from the current font's pxl file.
 *	saveChar		Saves a char in the current font's pxl file.
 *	applyViewResize		Let the user resize the ViewSW.
 *	selectMagnification	Called to change the screen Magnification.
 *	selectCellType		Called to change the type of PaintSW cell.
 *	selectMouseMode		Called to change drawing mode of the mouse.
 *	clear			Clears the subwindows and associated pixrects.
 */

#include "global.h"
#include "option.h"

static Panel_item	openFontItem;
static Panel_item	fontFileItem;
static Panel_item	loadCharItem;
static Panel_item	saveCharItem;
static Panel_item	charNameItem;
static Panel_item	charNumberItem;
static Panel_item	magnificationItem;
static Panel_item	viewWidthItem;
static Panel_item	viewHeightItem;
static Panel_item	applyItem;
static Panel_item	cellTypeItem;
static Panel_item	mouseModeItem;
static Panel_item	undoItem;
static Panel_item	clearItem;



/*
 * InitOptionSW
 *
 * Input:
 *	none.
 * Output:
 *	none.
 * Action:
 *	Initializes the Option subwindow (panel) and its panel items.
 */
void
InitOptionSW()
{
    int col1 = 5;		int row1 = 8;
    int col2 = col1 + 130;	int row2 = row1 + 23;
    int col3 = col2 + 155;	int row3 = row2 + 23;
    int col4 = col3 + 81;	int row4 = row3 + 15;
    				int row5 = row4 + 12;
				int row6 = row5 + 18;
				int row7 = row6 + 18;
    struct pixfont *optionSWFont, *optionSWBoldFont;
    static void openFont(), loadChar(), saveChar(), applyViewResize(), 
		selectMagnification(),selectCellType(), selectMouseMode(), 
		clear();
    Panel_setting clearCharField();
    void UpdateViewFields();

    optionSWFont = pf_open(FONTFILE);
    if (optionSWFont == NULL) {
	AbortWithError("InitOptionSW: can't find FONTFILE\n");
    }
    optionSWBoldFont = pf_open(BOLDFONTFILE);
    if (optionSWBoldFont == NULL) {
	AbortWithError("InitOptionSW: can't find BOLDFONTFILE\n");
    }

    OptionSW = window_create(FontToolFrame, PANEL,
			WIN_FONT,	optionSWFont,
			WIN_CURSOR,	MainCursor,
			0);

    fontFileItem = panel_create_item(OptionSW, PANEL_TEXT,
	PANEL_LABEL_X,			col1,
	PANEL_LABEL_Y,			row1,
	PANEL_VALUE_FONT,		optionSWBoldFont,
	PANEL_LABEL_STRING,		"PXL File:",
	PANEL_VALUE_DISPLAY_LENGTH,	26,
	0);
    openFontItem = panel_create_item(OptionSW, PANEL_BUTTON,
	PANEL_LABEL_X,	    col3,
	PANEL_LABEL_Y,	    row1-3,
	PANEL_LABEL_IMAGE,  panel_button_image(OptionSW, "Open Font", 9,NULL),
	PANEL_NOTIFY_PROC,  openFont,
	0);
    charNameItem = panel_create_item(OptionSW, PANEL_TEXT,
	PANEL_LABEL_X,			col1,
	PANEL_LABEL_Y,			row2,
	PANEL_VALUE_FONT,		optionSWBoldFont,
	PANEL_LABEL_STRING,		"Char Name:",
	PANEL_VALUE_STORED_LENGTH,	1,
	PANEL_VALUE_DISPLAY_LENGTH,	1,
	PANEL_NOTIFY_LEVEL,		PANEL_ALL,
	PANEL_NOTIFY_PROC,		clearCharField,
	0);
    charNumberItem = panel_create_item(OptionSW, PANEL_TEXT,
	PANEL_LABEL_X,			col2,
	PANEL_LABEL_Y,			row2,
	PANEL_VALUE_FONT,		optionSWBoldFont,
	PANEL_LABEL_STRING,		"or Char Number:",
	PANEL_VALUE_STORED_LENGTH,	3,
	PANEL_VALUE_DISPLAY_LENGTH,	3,
	PANEL_NOTIFY_LEVEL,		PANEL_ALL,
	PANEL_NOTIFY_PROC,		clearCharField,
	0);
    loadCharItem = panel_create_item(OptionSW, PANEL_BUTTON,
	PANEL_LABEL_X,      col3,
	PANEL_LABEL_Y,	    row2-3,
	PANEL_LABEL_IMAGE,  panel_button_image(OptionSW, "Load Char", 9, NULL),
	PANEL_NOTIFY_PROC,  loadChar,
	0);
    saveCharItem = panel_create_item(OptionSW, PANEL_BUTTON,
	PANEL_LABEL_X,      col4,
	PANEL_LABEL_Y,	    row2-3,
	PANEL_LABEL_IMAGE,  panel_button_image(OptionSW, "Save Char", 9, NULL),
	PANEL_NOTIFY_PROC,  saveChar,
	0);
    viewWidthItem = panel_create_item(OptionSW, PANEL_TEXT,
	PANEL_LABEL_X,			col1,
	PANEL_LABEL_Y,			row3,
	PANEL_VALUE_FONT,		optionSWBoldFont,
	PANEL_LABEL_STRING,		"View Width:",
	PANEL_VALUE_STORED_LENGTH,	3,
	PANEL_VALUE_DISPLAY_LENGTH,	3,
	0);
    viewHeightItem = panel_create_item(OptionSW, PANEL_TEXT,
	PANEL_LABEL_X,			col2,
	PANEL_LABEL_Y,			row3,
	PANEL_VALUE_FONT,		optionSWBoldFont,
	PANEL_LABEL_STRING,		"View Height:",
	PANEL_VALUE_STORED_LENGTH,	3,
	PANEL_VALUE_DISPLAY_LENGTH,	3,
	0);
    applyItem = panel_create_item(OptionSW, PANEL_BUTTON,
	PANEL_LABEL_X,      col3,
	PANEL_LABEL_Y,	    row3-3,
	PANEL_LABEL_IMAGE,  panel_button_image(OptionSW, "Apply", 9, NULL),
	PANEL_NOTIFY_PROC,  applyViewResize,
	0);
    panel_create_item(OptionSW, PANEL_MESSAGE,
	PANEL_LABEL_X,      col1,
	PANEL_LABEL_Y,	    row4,
	PANEL_LABEL_STRING,	"..........................................................................",
	0);
    magnificationItem = panel_create_item(OptionSW, PANEL_CHOICE,
	PANEL_LABEL_X,		col1,
	PANEL_LABEL_Y,		row5,
	PANEL_LABEL_STRING,	"Magnification:",
	PANEL_FEEDBACK, 	PANEL_INVERTED,
	PANEL_CHOICE_STRINGS,	"1", "2", "4", "6", "8", "10", 0,
	PANEL_CHOICE_YS,	row5+5, 0,
	PANEL_VALUE,		MAG10,
	PANEL_NOTIFY_PROC,	selectMagnification,
	0);
    cellTypeItem = panel_create_item(OptionSW, PANEL_CHOICE,
	PANEL_LABEL_X,		col1,
	PANEL_LABEL_Y,		row6,
	PANEL_LABEL_STRING,	"Cell Type:",
	PANEL_FEEDBACK, 	PANEL_INVERTED,
	PANEL_CHOICE_STRINGS,	"Normal", "Write-Black", "Write-White", 0,
	PANEL_CHOICE_YS,	row6+5, 0,
	PANEL_VALUE,		CELLNORMAL,
	PANEL_NOTIFY_PROC,	selectCellType,
	0);
    mouseModeItem = panel_create_item(OptionSW, PANEL_CHOICE,
	PANEL_LABEL_X,		col1,
	PANEL_LABEL_Y,		row7,
	PANEL_LABEL_STRING,	"Mouse Mode:",
	PANEL_FEEDBACK, 	PANEL_INVERTED,
	PANEL_CHOICE_STRINGS,	"Points", "Lines", "Fill", "Move", 0,
	PANEL_CHOICE_YS,	row7+5, 0,
	PANEL_VALUE,		MODEPOINTS,
	PANEL_NOTIFY_PROC,	selectMouseMode,
	0);
    clearItem = panel_create_item(OptionSW, PANEL_BUTTON,
	PANEL_LABEL_X,      col4,
	PANEL_LABEL_Y,	    row6+2,
	PANEL_LABEL_IMAGE,  panel_button_image(OptionSW, "Clear", 9, NULL),
	PANEL_NOTIFY_PROC,  clear,
	0);
    window_fit_height(OptionSW);
    UpdateViewFields();
}



/*
 * FixOptionSW
 *
 * Input:
 *	none.
 * Output:
 *	none.
 * Action:
 *	Makes sure the OptionSW is positioned and sized correctly in case
 *	the user has resized the main tool window, or in case the ViewSW
 *	has changed in size.
 */
void
FixOptionSW()
{
    Rect r;

    r = *((Rect *) window_get(FontToolFrame, WIN_RECT));
    OptionRect.r_left = ViewRect.r_width + SWSPACING;
    OptionRect.r_width = r.r_width - 2*FRAMEMARGIN - ViewRect.r_width
				- SWSPACING;
    OptionRect.r_height = MIN(
	OptionSWHeight,
	r.r_height - (int) window_get(FontToolFrame, WIN_TOP_MARGIN)
	    - MessageRect.r_height - SWSPACING - FRAMEMARGIN);
    r = OptionRect;
    window_set(OptionSW, WIN_RECT, &r, 0);
}



/*
 * openFont
 *
 * Input:
 *	item : The item in the OptionSW which has received input.
 *	event: The event that occurred.
 * Output:
 *	none.
 * Action:
 *	openFont is called by the SunView package whenever the "Open
 *	Font" command option is clicked in the Option subwindow.  The
 *	action is to read the "PXL File" text field and try to open
 *	that file.
 */
static void
openFont(item, event)
Panel_item item;
Event *event;
{
    char nameBuf[MAXFILENAMLEN];
    int result;

    strcpy(nameBuf, (char *)panel_get_value(fontFileItem));
    if (strlen(nameBuf) <= 0) {
	ShowMsg("***** Type a file name in the PXL File field \
		before opening. *****");
	return;
    }
    result = OpenPxlFile(nameBuf);
    switch (result) {
	case -1:
	    ShowMsg("***** Couldn't open file. *****");
	    break;
	case 0:
	    ShowMsg("***** Not a TeX pxl file! *****");
	    break;
	default:
	    ClearMsg();
	    break;
    }
}



/*
 * clearCharField
 *
 * Input:
 *	item : The item in the OptionSW which has received input.
 *	event: The event that occurred.
 * Output:
 *	The proper setting for the item in which the event took place.
 * Action:
 *	clearCharField is called by the SunView package whenever
 *	the text in the "Char Name" or "Char Number" fields in the
 *	Option subwindow is changed.  This routine ensures that when
 *	the user is typing in one field, the other field is clear.
 */
Panel_setting
clearCharField(item, event)
Panel_item item;
Event *event;
{
    panel_set_value((item==charNameItem) ? charNumberItem : charNameItem,"");
    return(panel_text_notify(item, event));
}



/*
 * loadChar
 *
 * Input:
 *	item : The item in the OptionSW which has received input.
 *	event: The event that occurred.
 * Output:
 *	none.
 * Action:
 *	loadChar is called by the SunView package whenever the "Load
 *	Char" command option is clicked in the Option subwindow.  It
 *	reads the text in the Option subwindow's "Char Name" or
 *	"Char Number" field and tries to load that character from the
 *	current pxl file.
 */
static void
loadChar(item, event)
Panel_item item;
Event *event;
{
    char charBuf[4];
    int c;

    /*
     * Find out which character field has some text in it, read the field,
     * then load the character from the current pxl file.
     */
    strcpy(charBuf, (char *)panel_get_value(charNameItem));
    if (strlen(charBuf) > 0) {
	ReadCharacter(charBuf[0]);
	return;
    }
    strcpy(charBuf, (char *)panel_get_value(charNumberItem));
    if (strlen(charBuf) > 0) {
	c = atoi(charBuf);
	if ( (c >= 0) && (c < 128) ) {
	    ReadCharacter(c);
	}
	else {
	    ShowMsg("***** Invalid ASCII character number \
			entered; please retype. *****");
	}
	return;
    }
    ShowMsg("***** Make an entry in one of the Character \
		fields before loading. *****");
}



/*
 * saveChar
 *
 * Input:
 *	item : The item in the OptionSW which has received input.
 *	event: The event that occurred.
 * Output:
 * Action:
 *	saveChar is called by the SunView package whenever the "Save
 *	Char" command option is clicked in the Option subwindow.  It
 *	reads the text in the Option subwindow's "Char Name" or
 *	"Char Number" field and tries to write that character to the
 *	current pxl file.
 */
static void
saveChar(item, event)
Panel_item item;
Event *event;
{
    char charBuf[4];
    int c;

    /*
     * Find out which character field has some text in it, read the field,
     * then save the character in the current pxl file.
     */
    strcpy(charBuf, (char *)panel_get_value(charNameItem));
    if (strlen(charBuf) > 0) {
	WriteCharacter(charBuf[0]);
	return;
    }
    strcpy(charBuf, (char *)panel_get_value(charNumberItem));
    if (strlen(charBuf) > 0) {
	c = atoi(charBuf);
	if ( (c >= 0) && (c < 128) ) {
	    WriteCharacter(c);
	}
	else {
	    ShowMsg("***** Invalid ASCII character number \
			entered; please retype. *****");
	}
	return;
    }
    ShowMsg("***** Make an entry in one of the Character \
		fields before saving. *****");
}



/*
 * UpdateViewFields
 *
 * Input:
 *	none.
 * Output:
 *	none.
 * Action:
 *	Displays the current pixel dimensions of the View subwindow in the
 *	"View Width" and "View Height" fields of the Option subwindow.
 */
void
UpdateViewFields()
{
    char charBuf[4];

    sprintf(charBuf, "%d", ViewPR->pr_size.x);
    panel_set_value(viewWidthItem, charBuf);
    sprintf(charBuf, "%d", ViewPR->pr_size.y);
    panel_set_value(viewHeightItem, charBuf);
}



/*
 * applyViewResize
 *
 * Input:
 *	item : The item in the OptionSW which has received input.
 *	event: The event that occurred.
 * Output:
 *	none.
 * Action:
 *	applyViewResize is called by the SunView package whenever the 
 *	"Apply" command option is clicked in the Option subwindow.  It
 *	reads the text in the Option subwindow's "View Width" and
 *	"View Height" fields and resizes the ViewSW and related structures
 *	based on the contents of these fields.  The View Width is kept
 *	within the range MINVIEWSWWIDTH...MAXVIEWSWWIDTH; the View Height
 *	is kept within the range MINVIEWSWHEIGHT...MAXVIEWSWHEIGHT.
 */
static void
applyViewResize(item, event)
Panel_item item;
Event *event;
{
    char charBuf[4];
    int w = 0, h = 0;

    strcpy(charBuf, (char *)panel_get_value(viewWidthItem));
    if (strlen(charBuf) > 0) {
	w = atoi(charBuf);
    }
    w = MIN(MAXVIEWSWWIDTH, MAX(MINVIEWSWWIDTH, w));

    strcpy(charBuf, (char *)panel_get_value(viewHeightItem));
    if (strlen(charBuf) > 0) {
	h = atoi(charBuf);
    }
    h = MIN(MAXVIEWSWHEIGHT, MAX(MINVIEWSWHEIGHT, h));

    if ( (ViewPR->pr_size.x != w) || (ViewPR->pr_size.y != h) ) {
	SETCURSORWATCH;
	ResizePRs(w, h, 1);
	DisplayPR(0, 0, ViewPR->pr_size.x, ViewPR->pr_size.y);
	SETCURSORNORMAL;
    }
}



/*
 * selectMagnification
 *
 * Input:
 *	item : The item in the OptionSW which has received input.
 *	value: The new ordinal value of the item.
 *	event: The event that occurred.
 * Output:
 *	none.
 * Action:
 *	Sets the global Magnification variable based on value, which
 *	should be supplied by the OptionSW whenever the Magnification field
 *	is changed.  Then the contents of the PaintPR are re-scaled and
 *	re-displayed.
 */
static void
selectMagnification(item, value, event)
Panel_item item;
int value;
Event *event;
{
    SETCURSORWATCH;
    ReferenceXY.x = SCALEDOWN(ReferenceXY.x);
    ReferenceXY.y = SCALEDOWN(ReferenceXY.y);
    switch (value) {
	case MAG1:
	    Magnification = 1;
	    break;
	case MAG2:
	    Magnification = 2;
	    break;
	case MAG4:
	    Magnification = 4;
	    CellMask = &WriteBlackCell4;
	    CellOffset = 1;
	    break;
	case MAG6:
	    Magnification = 6;
	    CellMask = &WriteBlackCell6;
	    CellOffset = 1;
	    break;
	case MAG8:
	    Magnification = 8;
	    CellMask = &WriteBlackCell8;
	    CellOffset = 2;
	    break;
	case MAG10:
	    Magnification = 10;
	    CellMask = &WriteBlackCell10;
	    CellOffset = 2;
	    break;
	default:
	    Magnification = MAXMAG;
	    CellMask = &WriteBlackCell10;
	    CellOffset = 2;
	    break;
    }
    ReferenceXY.x = SCALEUP(ReferenceXY.x);
    ReferenceXY.y = SCALEUP(ReferenceXY.y);
    FixPaintSW();
    CLEARPR(PaintPR);
    DisplayPR(0, 0, ViewPR->pr_size.x, ViewPR->pr_size.y);
    RedisplayFrame();
    SETCURSORNORMAL;
}



/*
 * selectCellType
 *
 * Input:
 *	item : The item in the OptionSW which has received input.
 *	value: The new ordinal value of the item.
 *	event: The event that occurred.
 * Output:
 *	none.
 * Action:
 *	Sets the global CellType variable based on value, which
 *	should be supplied by the OptionSW whenever the Cell Type field
 *	is changed.  Then the contents of the PaintPR are re-displayed.
 */
static void
selectCellType(item, value, event)
Panel_item item;
int value;
Event *event;
{
    SETCURSORWATCH;
    CellType = value;
    CLEARPR(PaintPR);
    DisplayPR(0, 0, ViewPR->pr_size.x, ViewPR->pr_size.y);
    SETCURSORNORMAL;
}



/*
 * selectMouseMode
 *
 * Input:
 *	item : The item in the OptionSW which has received input.
 *	value: The new ordinal value of the item.
 *	event: The event that occurred.
 * Output:
 *	none.
 * Action:
 *	Sets the global MouseMode variable based on value, which
 *	should be supplied by the OptionSW whenever the Mouse Mode field
 *	is changed.
 */
static void
selectMouseMode(item, value, event)
Panel_item item;
int value;
Event *event;
{
    MouseMode = value;
    switch (MouseMode) {
	case MODEPOINTS:
	    ShowMsg("Left mouse button paints, middle erases, right undoes.");
	    break;
	case MODELINES:
	    ShowMsg("Left mouse button draws black lines, middle \
			draws white, right undoes.");
	    break;
	case MODEFILL:
	    ShowMsg("Left mouse button fills with black, middle with white, \
			right undoes.");
	    break;
	case MODEMOVE:
	    ShowMsg("Left or middle mouse button moves the reference point, \
			right undoes.");
	    break;
	default:
	    break;
    }
}



/*
 * clear
 *
 * Input:
 *	none.
 * Output:
 *	none.
 * Action:
 *	Clears the contents of the ViewPR, PaintPR, ViewSW, and PaintSW.
 */
static void
clear()
{
    ResetRect(&UndoRect, 0, 0, ViewPR->pr_size.x, ViewPR->pr_size.y);
    COPYPR(ViewPR, UndoPR);
    CLEARPR(ViewPR);
    CLEARPR(PaintPR);
    RedisplayPRs();
    CLEARUNDOREFPT;
}
