#! /bin/sh
#
# conf - dynamically configure a makefile
#
# uses lines of the form "# conf" ... "# endconf" to figure out what
# to do.
#
# This script uses the file `Makefile' as a base and creates a
# new `Makefile.local'.  It is assumed that the first thing the
# original makefile does is run make on the local Makefile.

PATH=/bin:/usr/bin:/usr/ucb:"$PATH"; export PATH

# saved configuration file
conf=Config

# temp files
t1=/tmp/conf1.$$
t2=/tmp/conf2.$$

# clean up if interrupted
trap "rm -f $t1 $t2; exit 1" 1 2 3 15

# Step 1: find the "# conf" lines, gripe if none.  (But if $1 is
# "update", we will just read the saved configuration.)
case $1 in
update)
	if [ ! -f $conf ]; then
		echo "Your saved configuration file \`$conf' is gone!"
		echo "(Try \`make conf' again.)"
		exit 1
	fi;;
*)
	grep "^#[ 	][ 	]*conf[ 	][ 	]*" Makefile |
		sort | uniq >$t1
	case $? in
	0)	;;
	*)	echo "Something is seriously wrong with \`Makefile': there are
no \`# conf' lines in it.  You'll have to fix it by hand (sorry)."
		rm -f $t1; exit 1;;
	esac;;
esac

# Step 2: collect list of entries, and remember for later.
# If $1 is "update", take the remembered entries; if "all", take all entries.
case "$1" in
update)
	# N.B.: The following depends on the fact that the first two
	# lines of $conf are comments, the third the configuration.
	list="`awk 'NR==3 {print}' $conf`";;
all)
	list=`awk '{if ($3 != "none") printf ("%s ", $3)}' <$t1`
	echo "# This file contains the default \`all' configuration.
# Do not edit it yourself: use \`make conf' instead.
$list" >$conf;;
*)
	list=
	for i in `awk '{if ($3 != "none") printf ("%s ", $3)}' <$t1`; do
		echo -n "Do you have a(n) $i? "
		read answer
		case "$answer" in
		y|Y|yes|Yes|YES) list="$list $i";;
		esac;
	done
	echo "# This file contains your configuration.
# Do not edit it yourself: use \`make conf' instead.
$list" >$conf;;
esac

# Step 3: edit the Makefile so that the things that are configured
# in are on, and those that are not are commented out.  Put the result
# in Makefile.local.
# This `if' is for 4.3beta machines, where `mv -f' is botched
if [ -f Makefile.local ]; then
	mv -f Makefile.local Makefile.l.bak
fi
awk "
BEGIN {
	want = 0;			# 0 => normal
					# 1 => uncomment, -1 => comment
	split(\"$list\", list);
}
/^#[ 	][ 	]*conf[ 	][ 	]*/ {	# a conf item
	want = -1;			# assume we do not want it
	for (i in list)
		if (list[i] == \$3)
			want = 1;	# we do want it after all
	print;				# in any case, echo it
	next;
}
/^#[ 	][ 	]*endconf/ {		# end of conf item
	want = 0;			# back to normal
}
/^# / {					# something that was commented out
	if (want <= 0) {		# do not really want it
		print;
		next;
	}
	for (i = 2; i < NF; i++)	# want it: uncomment
		printf(\"%s \", \$i);
	print \$NF;
	next;
}
{					# anything else
	if (want >= 0)			# take it as is
		print;
	else				# comment it out
		printf(\"# %s\n\", \$0);
}" <Makefile >$t2
(trap '' 1 2 3 15; mv -f $t2 Makefile.local)

# Clean up and exit
rm -f $t1 $t2; exit 0
