/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */

#include <stdio.h>
#include <strings.h>
#include <sys/file.h>
#include "printpdr.h"
#include "units.h"
#include "defs.h"
#include "extfil.h"
#include "fontd.h"
#include "pdr.h"
#include "char.h"

extern char *getenv();
extern char *Malloc();
extern char *HandleFileNameExtension();
extern char *LocateFileWithPath();
extern char *StrcpyAlloc();
extern char *GetBytes();
extern char *texfonts_default;
extern char *tmpdir_default;

void Usage();
char * PdrFileName;
char * PdrFileNameExpanded; /* Full path name. */
char * ProgName;

int TableWidth = 4; /* Default table width */

/* SelfContained? If so a complete LaTeX file results from running this program.
   Just run LaTeX on it and you get your encoding vector tables. */
int SelfContained = TRUE;

char *PdrSearchPath; /* -P option path */

/*
 * main
 * ****
 * A program to read in a pdr file printing the encoding vector
 * and some other information. The output is a .tex file which
 * is printed with the help of LaTeX.
 */
main(argc, argv)
     int argc;
     char *argv[];
{
  extern char * optarg;
  extern int optind;
  int index;
  EX_FILES ex_f;
  int i;
  int le; /* Length of basename. */
  PDR_P p;
  int tfmw[256]; /* tfm width */
  int st_l[256]; /* String lengths */
  char * string[256]; /* Name of character or the PS code of it. */
  int tc[256]; /* Type of character, CT_...*/
  int execps_t[256]; /* Type of character if CT_EXECPS */
  int DviNum;
  int DviDen;
  double conv;
  int Resolution;
  int HConvUnMag;
  int c;
  int Slitex;

  ProgName = argv[0];
  InitProgName (ProgName, tmpdir_default);
  FExInit();
  PdrSearchPath = NULL;

  /* Usage message */
  if (argc == 1) {
    Usage();
    exit (1);
  }

  Resolution = 300;

  /* Initializations. */
  for (i=0; i<256; i++) {
    string[i] = NULL;
    tfmw[i] = -1;
    tc[i] = CT_NONE;
    execps_t[i] = -1;
  }

  /* Option processing. */
  index = 1;
  while ((c=getopt(argc, argv, "w:iP:")) != EOF) {
    switch (c) {
      case 'w':
	if (sscanf (optarg, "%d", &TableWidth) != 1)
	  Fatal ("main(): -w option.");
	break;
      case 'i':
	SelfContained = FALSE;
	break;
      case 'P':
	PdrSearchPath = StrcpyAlloc(optarg);
	break;
      case '?':
	Fatal ("main(): Illegal option");
      default:
	Fatal ("main(): default.");
      }
  } /* while */

  /* Locate pdr file, open it, check on correct file id, allocate
   * structure for pdr file data and insert into
   * font structure, read basename, append '\0' to it. */

  PdrFileName = HandleFileNameExtension (1, argv[optind], "pdr");
  if (Strlen(PdrSearchPath) == 0)
    PdrFileNameExpanded = LocateFileWithPath (PdrFileName, getenv("TEXFONTS"),
					      texfonts_default, TRUE);
  else
    PdrFileNameExpanded = LocateFileWithPath (PdrFileName, PdrSearchPath, NULL, TRUE);

  if (Strlen(PdrFileNameExpanded) == 0)
    Fatal2 ("main(): cannot find \"%s\"", PdrFileName);

  FExOpen (&ex_f, EFT_READ, 0, PdrFileNameExpanded, NULL);

  if (NoSignExtend(EX_FP(ex_f), 2) != PDR_FILE_ID)
    Fatal ("main(): Illegal file id in pdr file");
  p = (PDR_P) Malloc(sizeof(PDR));
  p->p_same = NULL;

  p->p_basename = Malloc((le=NoSignExtend(EX_FP(ex_f),1))+1);
  for (i=0; i<le; i++)
    p->p_basename[i] = NoSignExtend (EX_FP(ex_f), 1);
  p->p_basename[le] = '\0';

  /* Prepend some stuff to make output a self contained LaTeX program if so desired. */
  if (SelfContained) {
    printf ("\\documentstyle{article}\n");
    printf ("\\input inputd.tex\n");
    printf ("\\InputD{macros.latex}\n");
    printf ("\\InputD{book-verb.tex}\n");
    printf ("\\InputD{fonts-la.tex}\n");
    printf ("\\begin{document}\n");
  }

  printf ("\\par\\noindent{\\bf Info {\\tt PDR} file %s}\n",
	   PdrFileName);
  printf ("\n\n");
  printf ("{\\bf\\noindent Preamble information:}\n\n");
  printf ("\\vspace{12pt}\n");
  printf ("\\begin{tabular}{ll}\n");
  printf ("Font file:& %s\\\\\n", PdrFileName);
  p->p_cs = NoSignExtend(EX_FP(ex_f), 4);
  printf ("Checksum (hex):& %x\\\\\n", p->p_cs);
  p->p_hcc = NoSignExtend(EX_FP(ex_f), 2); /* Highest character code. */
  p->p_ds = NoSignExtend(EX_FP(ex_f), 4); /* Design size. */
  p->p_sp = NoSignExtend(EX_FP(ex_f), 4); /* Space factor. */
  p->p_a = FIXES_TO_FLOAT(SignExtend(EX_FP(ex_f), 4)); /* Matrix elements. */
  p->p_b = FIXES_TO_FLOAT(SignExtend(EX_FP(ex_f), 4));
  p->p_c = FIXES_TO_FLOAT(SignExtend(EX_FP(ex_f), 4));
  p->p_d = FIXES_TO_FLOAT(SignExtend(EX_FP(ex_f), 4));
  p->p_tx = SignExtend(EX_FP(ex_f), 4);
  p->p_ty = SignExtend(EX_FP(ex_f), 4);
  p->p_o = SignExtend(EX_FP(ex_f), 4);
  p->p_mc = NoSignExtend (EX_FP(ex_f), 4);
  p->p_emul = NoSignExtend (EX_FP(ex_f), 1);
  Slitex = NoSignExtend (EX_FP(ex_f), 1);
  p->p_length = NoSignExtend (EX_FP(ex_f), 1);

  printf ("Font matrix:&");
  printf ("%6.4lf %6.4lf\\\\& %6.4lf %6.4lf\\\\\n",
	   p->p_a, p->p_b, p->p_c, p->p_d);

  /* Print design size and use size in pt */
  printf ("Basename:& \\tt %s\\\\\n", p->p_basename);
  printf ("Number of characters in the font:& %d\n", p->p_length);

  /* The following code is from "conv.c" of the driver source. */
  DviNum = 25400000;
  DviDen = 473628672;
  conv =  (double)DviNum/(double)DviDen/100000.0/2.54*Resolution;
  HConvUnMag = (int) (1.0 / conv  + 0.5);
/* The following is wrong. 
  printf ("Design size:& %6.3lf pt\\\\\n",
	  (double) (double)(p->p_ds)/HConvUnMag/Resolution*72.27);
*/
  printf ("\\end{tabular}\n");
  printf ("\n\n");
  printf ("\\noindent End of information extracted from the {\\tt PDR} file\n");
  printf ("\\noindent of this font.\n");
  printf ("The encoding vector can be found in Figure~\\ref{t-%s}\n",
	   PdrFileName);
  printf ("on \\page{t-%s}.\n", PdrFileName);
  printf ("\n\n");

  /* Read in the character info from the pfd file. The output goes into
   * LaTeX figure. */
  for (i=0; i<p->p_length; i++) {
    index = NoSignExtend(EX_FP(ex_f), 1); /* Character index. */
    if (index < 0 || index >= 256)
      Fatal2 ("main(): illegal character index in pdr file: index '%o", index);
    tc[index] = NoSignExtend(EX_FP(ex_f), 1); /* Type of character. */
    execps_t[index] = NoSignExtend(EX_FP(ex_f), 1); /* Execps class. */
    tfmw[index] = NoSignExtend(EX_FP(ex_f), 4); /* Tfm width. */
    st_l[index] = NoSignExtend (EX_FP(ex_f), 2); /* Length of name of character. */
    if (st_l[index] <= 0)
      Fatal3 ("main(): reading pdr file: index '%o, illegal string length %d",
	      index, st_l[index]);
    string[index] = GetBytes(EX_FP(ex_f), st_l[index]);
  }

  FExClose (&ex_f);

  /* Generate TeX code for the beginning of the figure. */
  printf ("\\begin{figure}\n");
  printf ("\\baselineskip = 7pt\n");
  printf ("$$\\vbox\\bgroup\n");
  printf ("\\tabskip = 5pt\n\
      \\halign\\bgroup &\\tiny\\tt#\\hfil\\cr\n");

#ifdef DEBUG
  /* First a short form of the name figure. */
  for (i=0; i<256; i++) {
    switch (tc[i]) {
      case CT_AFM:
	printf ("%% '%3o: %s\n", i, string[i]);
        break;
      case CT_ASS:
	printf ("%% '%3o: %s *\n", i, string[i]);
	break;
      case CT_EXECPS:
	printf ("%% '%3o [PS]: -->%s<--\n", i, string[i]);
	break;
      case CT_WIDTH_ONLY:
	printf ("%% '%3o [WO]\n", i);
	break;
      case CT_NONE:
	printf ("%% '%3o none\n", i);
	break;
      default:
	Fatal2 ("main(): default-1: i = %d", i);
    }
  }
#endif

  /* Now print the encoding vector as an \halign figure. */
  for (i=0; i<256; i++) {
    if (i%TableWidth == 0) {
      printf ("'%03o& ", i);
    }
    /* Depending on the character.... */
    switch (tc[i]) {
      case CT_AFM:
	printf ("/%s", string[i]);
        break;
      case CT_ASS:
	printf ("/%s $^{*}$", string[i]);
	break;
      case CT_EXECPS:
	printf ("$<$**$>$");
	break;
      case CT_WIDTH_ONLY:
	printf ("$<$++$>$");
	break;
      case CT_NONE:
	break;
      default:
	Fatal ("main(): default-2");
    }

    if (i%TableWidth == (TableWidth-1))
      printf ("\\cr\n");
    else {
      printf ("& ");
      if (i%4 == 3)
	printf ("\n");
    }
  }
  printf ("\\egroup\\egroup$$\n");
  printf ("\\caption{Encoding vector of {\\tt %s}}\n",
	   PdrFileName);
  printf ("\\label{t-%s}\n", PdrFileName);
  printf ("\\end{figure}\n");

  for (i=0; i<256; i++) {
    switch (tc[i]) {
      case CT_AFM:
      case CT_ASS:
        break;
      case CT_EXECPS:
	printf ("\\bigskip\n\\noindent{\\bf ExecPsCode:: '%o}\n",
		 i);
	printf ("\\beginverbatim\n");
	printf ("%s\n", string[i]);
	printf ("\\endverbatim\n");
	break;
      case CT_NONE:
	break;
      case CT_WIDTH_ONLY:
	break;
      default:
	Fatal ("main(): default-3");
    } /* switch */
  } /* for */

  if (SelfContained)
    printf ("\\end{document}\n");

  exit(0);
} /* main */

/*
 * Usage
 * *****
 */
void
Usage()
{
  fprintf (stderr, "Usage: %s\n\
        [-i]\n\
        [-w #-of-columns] (number of columns of generated table)\n\
        [-P path] (Path to locate .pdr files)\n\
              pdr-file-name\n", ProgName);
}
