/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */

/*
 * This version functionally identical to the System V getopt() described
 * below.
 * Written by Eric Kostelich, Center for Nonlinear Dynamics,
 * Dept. of Physics, University of Texas, Austin, Texas  78712.

      GETOPT(3C)                  (UNIX 5.0)                 GETOPT(3C)

      NAME
           getopt - get option letter from argument vector

      SYNTAX
           int getopt (argc, argv, optstring)
           int argc;
           char **argv;
           char *optstring;

           extern char *optarg;
           extern int optind;

      DESCRIPTION
           Getopt returns the next option letter in argv that matches a
           letter in optstring.  Optstring is a string of recognized
           option letters; if a letter is followed by a colon, the
           option is expected to have an argument that may or may not
           be separated from it by white space.  Optarg is set to point
           to the start of the option argument on return from getopt.

           Getopt places in optind the argv index of the next argument
           to be processed.  Because optind is external, it is normally
           initialized to zero automatically before the first call to
           getopt.

           When all options have been processed (i.e., up to the first
           non-option argument), getopt returns EOF.  The special
           option -- may be used to delimit the end of the options; EOF
           will be returned, and -- will be skipped.

      DIAGNOSTICS
           Getopt prints an error message on stderr and returns a
           question mark (?) when it encounters an option letter not
           included in optstring.

      WARNING
           The above routine uses <stdio.h>, which causes it to
           increase the size of programs, not otherwise using standard
           I/O, more than might be expected.

      EXAMPLE
           The following code fragment shows how one might process the
           arguments for a command that can take the mutually exclusive
           options a and bbbb, and the options ffff and o, both of which
           require arguments:

                main (argc, argv)
                int argc;
                char **argv;
                {
                     int c;
                     extern int optind;
                     extern char *optarg;
                       .
                       .
                       .
                     while ((c = getopt (argc, argv, "abf:o:")) != EOF)
                          switch (c) {
                          case 'a':
                               if (bflg)
                                    errflg++;
                               else
                                    aflg++;
                               break;
                          case 'b':
                               if (aflg)
                                    errflg++;
                               else
                                    bproc( );
                               break;
                          case 'f':
                               ifile = optarg;
                               break;
                          case 'o':
                               ofile = optarg;
                               bufsiza = 512;
                               break;
                          case '?':
                               errflg++;
                          }
                     if (errflg) {
                          fprintf (stderr, "usage: . . . ");
                          exit  2);
                     }
                     for ( ; optind < argc; optind++) {
                          if (access (argv[optind], 4)) {
                       .
                       .
                       .
                }

      SEE ALSO
           getopt(1).
 */

#include <stdio.h>
#include <string.h>

/*  External variables */

int	optind = 1;	/* index into argv[] of next string to be processed */
char	*optarg = NULL;	/* pointer to option argument, if any */

/*
 * getopt
 * ******
 * argc: argument counter.
 * argv: argument values.
 * control: option string.
 * RET: EOF when done
 *      Otherwise the option character and argument if any.
 *      '?' for an illegal option.
 */
int getopt(argc, argv, control)
     int	argc;
     char	**argv;
     char	*control;
{
  static int argidx = 1; /* index within a string in argv[] to check for
			  * next option letter, since options without
			  * arguments can be concatenated. */
  char	*cidx; /* ptr into control string of option letter */
  int	value; /* return code */

  optarg = NULL; /* Just to be sure. */

  if (Strlen(argv[optind]) == 0)
    return (EOF);

  if (argv[optind][0] != '-') /* No '-': so there is no option! */
    return (EOF);
  /* If the 'option' is '--' it means stop reading options! Can be used to
     have non-option data following which starts with '--' */
  if (argv[optind][1] == '-')
    return(EOF);
  /* If you have a '-' by itself, return '\0' */
  if (argv[optind][1] == '\0') {
    optind++;
    return (0);
  }

  /* Find position of option letter in the control string.  NULL means
   * it's not there, so it's an invalid option. */
  if((cidx = index(control, argv[optind][argidx])) == NULL)  {
    (void) fprintf(stderr, "%s: option \"%c\" is illegal.\n", argv[0],
		   argv[optind][argidx]);
    value = '?'; /* Indicates illegal option */
    /* Advance so next option is read next time. */
    if(argv[optind][++argidx] == '\0')  {  
      /* No more characters in argv[optind], go to argv[optind+1] */
      optind++;
      argidx = 1;
    }  
  }
  /* Found the option character in control. */
  else  {
    value = (int) *cidx; /* Option character */
    argidx++; /* Next option, beginning of argument, or \0. */
    if(cidx[1] == ':')  {   /* Option takes argument. */
      if(argv[optind][argidx] != '\0') /* Argument of option follows option imm. */
	optarg = argv[optind++] + argidx;
      else /* Argument of option is next argv (argv[optind+1]) */
	if (++optind >= argc) {
	  (void) fprintf(stderr, "%s: option \"%c\" requires an argument.\n",
			 argv[0], argv[optind-1][argidx-1]);
	} else
	  optarg = argv[optind++];
      argidx = 1;
    }
    /* Option with no argument. */
    else  {
      /* Check whether next option appended to the same string or whether
	 we have to switch to the next argv[optind]. */
      if(argv[optind][argidx] == '\0') { /* Next argv[optind] */
	argidx = 1;
	optind++;
      }
    }
  }
  return(value);
}
