IMPLEMENTATION MODULE vduinterface;

(* Author:         Andrew Trevorrow
   Implementation: Modula-2 under VAX/UNIX 4.2 BSD
   Date Started:   June, 1986

   Description:
   InitVDUInterface initializes the generic VDU routines and parameters
   according to the vdu value set by InitSysInterface (in main module).
   Note that new VDUs can be added without having to recompile
   the main module (search for "XXX" below).
*)

FROM screenio IMPORT
   Write, WriteString, WriteLn, RestoreTerminal;

FROM sysinterface IMPORT
   vdu;

FROM aed512vdu IMPORT
   screenht,       (* AED screen can be 512 or 483 pixels high *)
   InitAED512;     (* assigns AED512 specific values to generic VDU variables *)

FROM ansivdu IMPORT
   InitANSI;       (* assigns ANSI specific values to generic VDU variables   *)

FROM regisvdu IMPORT
   InitREGIS;      (* assigns REGIS specific values to generic VDU variables  *)

FROM vis500vdu IMPORT
   InitVIS500;     (* assigns VIS500 specific values to generic VDU variables *)

FROM vis550vdu IMPORT
   InitVIS550;     (* assigns VIS550 specific values to generic VDU variables *)

FROM vt640vdu IMPORT
   InitVT640;      (* assigns VT640 specific values to generic VDU variables  *)

FROM vt220vdu IMPORT
   InitVT220;      (* assigns VT220 specific values to generic VDU variables  *)

(* Add other VDU specific modules as more terminals are implemented.
FROM xxxvdu IMPORT
   InitXXX;
*)

(******************************************************************************)

PROCEDURE InitTeXtoASCII;

(* Initialize TeXtoASCII array used in specific ShowChar/Rectangle routines
   to map a given TeX char into a similar, displayable ASCII char.
*)

VAR ch : CHAR;

BEGIN
FOR ch :=   0C TO  12C DO TeXtoASCII[ch] := '?' END; (* Greek letters *)
FOR ch :=  13C TO  17C DO TeXtoASCII[ch] := '?' END; (* ligatures *)
TeXtoASCII[20C] := 'i';                              (* dotless i *)
TeXtoASCII[21C] := 'j';                              (* dotless j *)
TeXtoASCII[22C] := '`';                              (* grave accent *)
TeXtoASCII[23C] := "'";                              (* acute accent *)
FOR ch :=  24C TO  27C DO TeXtoASCII[ch] := '~' END; (* high accents *)
TeXtoASCII[30C] := ',';                              (* cedilla *)
FOR ch :=  31C TO  40C DO TeXtoASCII[ch] := '?' END; (* diphthongs, foreigns *)
FOR ch :=  41C TO 133C DO TeXtoASCII[ch] := ch  END; (* same: !..0..9..A..[ *)
TeXtoASCII[134C] := '"';                             (* open double quote *)
TeXtoASCII[135C] := ']';                             (* same *)
FOR ch := 136C TO 137C DO TeXtoASCII[ch] := '^' END; (* more high accents *)
FOR ch := 140C TO 172C DO TeXtoASCII[ch] := ch  END; (* same: `..z *)
FOR ch := 173C TO 174C DO TeXtoASCII[ch] := '-' END; (* en dash, em dash *)
FOR ch := 175C TO 177C DO TeXtoASCII[ch] := '~' END; (* more high accents *)
END InitTeXtoASCII;

(******************************************************************************)

PROCEDURE StringsEqual (s1, s2 : ARRAY OF CHAR) : BOOLEAN;

(* Returns TRUE if uppercase s1 = uppercase s2.
   Comparison is based on the underlying order of character codes.
   E.g., for the ASCII code, ' ' < '0'..'9' < 'A'..'Z'.
*)

CONST NULL = 0C;

VAR i : CARDINAL;

BEGIN
(* SYSDEP: we assume end of string is first NULL, or string is full *)
i := 0;
LOOP
   IF CAP(s1[i]) <> CAP(s2[i]) THEN
      RETURN FALSE;
   ELSIF s1[i] = NULL THEN                    (* = s2[i] *)
      RETURN TRUE;                            (* both NULL terminated *)
   END;
   INC(i);
   IF i > HIGH(s1) THEN                       (* s1 is full *)
      IF i > HIGH(s2) THEN RETURN TRUE END;   (* so is s2 *)
      RETURN s2[i] = NULL;                    (* s2 NULL terminated? *)
   ELSIF i > HIGH(s2) THEN                    (* s2 is full *)
      RETURN s1[i] = NULL;                    (* s1 NULL terminated? *)
   END;
END;
END StringsEqual;

(******************************************************************************)

PROCEDURE InitVDUInterface;

(* We assume DVItoVDU has already called InitSysInterface to
   initialize the vdu value.
*)

BEGIN
InitTeXtoASCII;
IF StringsEqual(vdu,'AED483') THEN
   screenht := 483;   (* override initial implementation value of 512 *)
   InitAED512;        (* uses screenht, so must be called after above line *)
ELSIF StringsEqual(vdu,'AED512') THEN
   InitAED512;
ELSIF StringsEqual(vdu,'ANSI')  OR
      StringsEqual(vdu,'VT100') THEN
   InitANSI;
ELSIF StringsEqual(vdu,'REGIS') OR
      StringsEqual(vdu,'VT240') OR
      StringsEqual(vdu,'GIGI')  OR
      StringsEqual(vdu,'VK100') OR
      StringsEqual(vdu,'VT125') THEN
   InitREGIS;
ELSIF StringsEqual(vdu,'VT100132') THEN
   InitANSI;          (* it is up to user to set VDU to 132 columns *)
   windowwd := 132;   (* just increase window width *)
ELSIF StringsEqual(vdu,'VIS240') OR
      StringsEqual(vdu,'VIS241') THEN
   InitREGIS;
   (* the VIS240/241 has more text lines and a bigger window region: *)
   bottoml := 29;
   windowwd := 800;
   windowht := 580 - windowv;
ELSIF StringsEqual(vdu,'VIS500') THEN
   InitVIS500;
ELSIF StringsEqual(vdu,'VIS550') THEN
   InitVIS550;
ELSIF StringsEqual(vdu,'VT220') THEN
   InitVT220;
ELSIF StringsEqual(vdu,'VT640') THEN
   InitVT640;
(* add a new VDU here (keep order alphabetical)
ELSIF StringsEqual(vdu,'XXX') THEN
   InitXXX;
*)
ELSE
   WriteString('Bad -v value! (='); WriteString(vdu); Write(')'); WriteLn;
   WriteString('Specify one of the following:'); WriteLn;
   WriteString('   AED483    (AED 512 with 512 by 483 screen)');
   WriteLn;
   WriteString('   AED512    (AED 512 with 512 by 512 screen)');
   WriteLn;
   WriteString('   ANSI      (any ANSI compatible VDU; synonym = VT100)');
   WriteLn;
   WriteString('   REGIS     (any ReGIS compatible VDU; ');
   WriteString('synonyms = GIGI, VK100, VT125, VT240)');
   WriteLn;
   WriteString('   VIS240    (VISUAL 240; synonym = VIS241)');
   WriteLn;
   WriteString('   VIS500    (VISUAL 500)');
   WriteLn;
   WriteString('   VIS550    (VISUAL 550)');
   WriteLn;
   WriteString('   VT100132  (any VT100 compatible VDU in 132 column mode)');
   WriteLn;
   WriteString('   VT220     (VT220 using down-loaded chunky graphics)');
   WriteLn;
   WriteString('   VT640     (VT100 with Retro-Graphics)');
   WriteLn;
   (* add a new VDU here (keep order alphabetical)
   WriteString('   XXX');
   WriteLn;
   *)
   RestoreTerminal; HALT;
END;
END InitVDUInterface;

END vduinterface.
