;;;	     mew.el :: Message interface to Emacs Window
;;;
;;;	  Copyright (C) 1994, 1995, 1996  Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: March 22, 1994
;;; Revised: March 30, 1996
;;;

(defconst mew-version "Mew version 1.05+")

;;;  Minimum setup.
;;;
;;; (autoload 'mew "mew" nil t)
;;; (autoload 'mew-send "mew" nil t)

(defvar mew-profile (or (getenv "MH") (expand-file-name "~/.mh_profile")))

(defvar mew-debug nil)
;(setq mew-debug nil)
;(setq mew-debug t)

(defvar mew-emacs18-p (string-match "^18" emacs-version))
(defvar mew-emacs19-p (string< "19" emacs-version))
(defvar mew-mule-p    (boundp 'MULE))
(defvar mew-xemacs-p  (string-match "XEmacs\\|Lucid" emacs-version))
(defvar mew-os2-p     nil)

(provide 'mew)
(require 'mew-refile)
(require 'mew-url)
(require 'mew-pem)
(require 'mew-pgp)
(require 'mew-fib)

;;;
;;; user customizable variables
;;;

(defvar mew-demo t
  "*Mew demo displays at boot time when non-nil.")

(defvar mew-analysis t
  "*MIME syntax analysis and cache when non-nil.
   Skip syntax analysis when nil.")

(defvar mew-cache-size 10
  "*Number of buffer for message cache.")

(defvar mew-summary-cache-file ".mew-cache"
  "*Cache file for summary mode contents.")

(defvar mew-summary-cache-use t
  "*If non-nil, insert mew-summary-cache-file into summary mode buffer 
when the summary mode is visited and it does not exist in buffer. 
mew-summary-cache-file is automatically saved on inc and scan.")

(defvar mew-window-use-full nil
  "*Dynamic window configuration when nil. This variable will be obsolated
because mew will support mew-window-percent(e.g 50, 100, etc.).")

(defvar mew-summary-show-direction 'down
  "*Direction for SPC at end of message. 
'up 'down 'next(current direction) 'stop")

(defvar mew-mime-compose-folder-delete 'ask
 "*Action to delete MIME draft directory after sending a multipart letter.
Proper value is 'ask', 'delete', and 'retain'.")


(defvar mew-mail-domain-list nil
  "*Your e-mail address domain list like 
(\"is.aist-nara.ac.jp\" \"csce.kyushu-u.ac.jp\").
They are used for mail-domain completion in draft mode(C-cTAB)")

(defvar mew-mail-domain (or (car mew-mail-domain-list) "")
  "*Your e-mail address domain.")

(defvar mew-mail-address-list nil
  "*The addresses included in this list never appear on the Cc:
field on a draft buffer.")


(defvar mew-redist-needs-full-contents nil)
  ; if you are using MH/post that speaks SMTP, leave this as "nil".
  ; if you are using MH/post that spawns sendmail and passes message
  ; with PIPE, change this as t.

(defvar mew-eof-string nil
  "*A value inserted on the end of message buffer if non-nil.")

(defvar mew-loop-depth 20
  "*A value to decide loop depth for header field syntax analysis")
(defvar mew-header-max-length 50
  "*If the length of a header exceeds this value, it is not arranged nor
MIME decoded.")

;; Draft fields

(defvar mew-reply-to nil
  "*A value inserted into Reply-To: field in draft mode if non-nil.")
(defvar mew-fcc nil
  "*A value inserted into Fcc: field in draft mode if non-nil.")
(defvar mew-cc nil
  "*A value inserted into Cc: field in draft mode if non-nil.")
(defvar mew-dcc nil
  "*A value inserted into Dcc: field in draft mode if non-nil.")
(defvar mew-from nil
  "*A value inserted into From: field in draft mode if non-nil.")
(defvar mew-x-mailer
  (concat mew-version " on Emacs " emacs-version
	  (and mew-mule-p
	       (string-match "^[0-9.]*" mule-version)
	       (concat ", Mule " (substring mule-version
					    (match-beginning 0) 
					    (match-end 0)))))
  "*A value inserted into From: field in draft mode if non-nil.")

(defvar mew-header-alist nil
  "*alist of key and value pair for header field to be inserted on draft.
'((\"X-fingerprint:\" . \"6B 63 38 88 67 5E 96 8E  CE A4 62 73 3F 11 64 94\")
  (\"X-URL:\" . \"http://uranus.aist-nara.ac.jp/~kazu/\"))"
  )

(defvar mew-auto-add-content-type nil
  "*Automatically adds MIME Content-Type: header if non-nil, asks otherwise.")
(defvar mew-ask-cc t
  "*Prompt user for CC: field if non-nil")
(defvar mew-ask-subject nil
  "*Prompt user for Subject: field when send if non-nil")

(defvar mew-emacs-y-or-n-p-use nil
  "*SPC is y and RET is n if t. SPC is n and RET is y, otherwise.")

;; Overlays for mouse highlight and cursor underline

(defvar mew-highlight-lines-use nil
  "*Use highlight line in summary mode.")
(defvar mew-underline-lines-use mew-emacs19-p
  "*Use underline line in summary mode.")

(cond (mew-xemacs-p
       (or (find-face 'underline)
	   (progn (make-face 'underline)
		  (set-face-underline-p 'underline t)))
       (fset 'mew-overlayp (symbol-function 'extentp))
       (fset 'mew-make-overlay (symbol-function 'make-extent))
       (defun mew-move-overlay (overlay beg end &optional buffer)
	 (set-extent-endpoints overlay beg end))
       (defun mew-overlay-put (overlay prop value)
	 (set-extent-property overlay prop value)))
      (mew-emacs19-p
       (fset 'mew-overlayp (symbol-function 'overlayp))
       (fset 'mew-make-overlay (symbol-function 'make-overlay))
       (fset 'mew-move-overlay (symbol-function 'move-overlay))
       (fset 'mew-overlay-put (symbol-function 'overlay-put))))

;; xxx 
;; Currently only one signature file is supported. 
;; But you can use multiple signature files in the future.
(defvar mew-signature-file "~/.signature")

(defvar mew-file-append-p nil
  "If non-nil, a message or a part is appended to the existing file
on mew-summary-save(\"y\"). Otherwise overwrited.")

;;;
;;; Citation
;;;

(defvar mew-cite-prefix "> "
  "*Prefix of citation.")
(defvar mew-cite-hook nil
  "*If you want to use super-cite, (setq mew-cite-hook 'sc-cite-original).")
(defvar mail-citation-hook nil) ;; not to require it

(defvar mew-cite-prefix-function nil)
;(defvar mew-cite-prefix-function 'mew-cite-prefix-username)

(defvar mew-cite-prefix-confirmp nil)
(defun mew-cite-prefix-username ()
  (format "%s> " (mew-header-delete-at
		  (mew-header-extract-addr
		   (mew-field-get-value "From:"))))
  )

(defvar mew-cite-fields '("From:" "Subject:" "Date:")
  "*The fields that you want to extract as citation label. 
If you change this valuable, you must change mew-cite-format.
The value of the first field becomes the first argment for mew-cite-format.
(e.g. The default first argment is a value of From: field.)
The value of the second field becomes the second argment for mew-cite-format.
....")

(defvar mew-cite-format "From: %s\nSubject: %s\nDate: %s\n\n")

;;;
;;; Hooks
;;;

(defvar mew-init-hook nil
  "*Hook called at initialize time.")
(defvar mew-summary-mode-hook nil
  "*Hook called in summary mode.")
(defvar mew-draft-mode-hook nil
  "*Hook called in draft mode.")
(defvar mew-message-mode-hook nil
  "*Hook called in message mode.")
(defvar mew-message-hook nil
  "*Hook called whenever message displayed.")
(defvar mew-send-hook nil
  "*Hook called before sending a message")
(defvar mew-quit-hook nil
  "*Hook called on quit.")
(defvar mew-summary-inc-sentinel-hook nil
  "*Hook called when inc finished.")
(defvar mew-summary-scan-sentinel-hook nil
  "*Hook called when scan finished.")
(defvar mew-summary-exec-hook nil
  "*Hook called when mew-summary-exec finished.")

(defvar mew-summary-recenter-p t)
(defvar mew-break-pages t)
(defvar mew-page-delimiter "^\^L")

;; Character code

(cond
 (mew-mule-p
  (defvar mew-mule-charset-local     *iso-2022-jp*) ;; Set your local charset
  (defvar mew-mule-charset-integrity *iso-2022-jp*)
  (defvar mew-mule-charset-virtual   *iso-2022-jp*unix)
  ;; include ^M as it is.
  )
 )

;;
;; BBDB
;;

(defvar mew-use-bbdb nil
  "*Use Insidious Big Brother Database if t")


(defvar mew-inbox "+inbox")
(defvar mew-folders-file ".folders")
(defvar mew-folders-ignore '("from"))
(defvar mew-folders-default-folder "from")

(defvar mew-print-function (function lpr-buffer))

(defvar mew-default-boundary "--%s(%s)--")

(defvar mew-x-mew "X-Mew:")

(cond 
 ((string= mew-mail-domain "")
  (defvar mew-mail-address (user-login-name)))
 (t
  (defvar mew-mail-address (concat (user-login-name) "@" mew-mail-domain)))
 )

;;
;; It's nice to tell current message if lines of summary window is odd.
;;
(defvar mew-window-configuration 
  (if mew-emacs19-p
      '((summary (1  0))
	(message (8 31)))
    '((summary (1  0))
      (message (4 19)))
    )
  "*Ratio of windows")

(defvar mew-summary-message-regex "^ *\\([0-9]+\\)")
(defvar mew-summary-part-regex "^\t+\\([0-9.]+\\)")
(defvar mew-address-separator ":, \t\n")
(defvar mew-summary-edit-again-regex
  "----- Original message follows -----\\|----- Unsent message follows -----")

;; X face

(defvar mew-x-face-filter '("uncompface" "ikon2xbm"))
(defvar mew-x-face-prog "xv")
(defvar mew-x-face-args nil)
(defvar mew-x-face-file "~/.xface"
  "*If non-nil and the file exists, X-Face: fields is inserted.")

;;
;; current solution to decide To: when reply
;;

(defvar mew-noreplyto-to-list '("From:")
  "To: field list picked at mew-summary-reply if Reply-To: doesn't exist.")

(defvar mew-noreplyto-cc-list '("To:" "Cc:" "Apparently-To:")
  "Cc: field list picked at mew-summary-reply if Reply-To: doesn't exist.")

(defvar mew-replyto-to-list '("Reply-To:" "From:")
  "To: field list picked at mew-summary-reply if Reply-To: exists.")

(defvar mew-replyto-cc-list '("To:" "Cc:" "Apparently-To:")
  "Cc: field list picked at mew-summary-reply if Reply-To: exists.")

;;
;;

(defvar mew-draft-mode-syntax-table nil
  "*Syntax table used while in mew letter mode.")

(defvar mew-fields
  '("To:" "Cc:" "Subject:" "Dcc:" "Fcc:" "Bcc:"
    "Reply-To:" "Followup-To:")
  )

(defvar mew-field-completion-switch
  '(("To:" . mew-draft-address-comp)
    ("Cc:" . mew-draft-address-comp)
    ("Dcc:" . mew-draft-address-comp)
    ("Bcc:" . mew-draft-address-comp)
    ("Reply-To:" . mew-draft-address-comp)
    ("Fcc:" . mew-draft-folder-comp))
  )

(defvar mew-field-visible
  '("Subject:" "From:" "To:" "Apparently-To:"
    "Cc:" "Newsgroups:" "Date:" "Reply-To:"
    "Resent-From:" "Resent-To:" "Resent-Cc:"
    "Mime-Version:" "Content-Type:")
  "*Visible fields for mail header.")

(defvar mew-field-invisible
  '("Received:" "Return-Path:" "Sender:" "Errors-To:"
    "Resent-Date:" "Message-Id:" "Resent-Message-Id:"
    "Resent-Sender:" "Resent-Reply-To:" "Delivery-Date:")
  "*Invisible fields for mail header.")

(defvar mew-field-delete
  '("Date:" "Message-Id:" "Return-Path:" "Received:"
    "From:" "Sender:" "Delivery-Date:" "X-Mailer:"))

(defvar mew-field-resent
  '("Resent-From:" "Resent-Sender:" "Resent-To:"
    "Resent-Cc:" "Resent-Date:" "Resent-Message-Id:" "Resent-Reply-To:"))

;; xxx If you change the order or add new fields, change mew-part-*.
;; (Key . header-syntax-p)
(defconst mew-mime-fields
  (list '("Content-Type:" . t)
	'("Content-Transfer-Encoding:" . t)
	'("Content-ID:" . nil) ;; xxx
	'("Content-Description:" . nil)
	(cons mew-x-mew nil)))

(defvar mew-mime-fields-number (length mew-mime-fields))
(defconst mew-part-magic (+ (length [type beg end]) (length mew-mime-fields)))

(defvar mew-prog-inc         "inc")
(defvar mew-prog-scan        "scan")
(defvar mew-prog-vscan       "vscan")
(defvar mew-prog-ali         "ali")
(defvar mew-prog-folders     "folders")
(defvar mew-prog-folder      "folder")
(defvar mew-prog-pack        "folder")
(defvar mew-prog-rmm         "rmm")
(defvar mew-prog-refile      "refile")
(defvar mew-prog-send        "send")
(defvar mew-prog-sortm       "sortm")
(defvar mew-prog-pick        "pick")
(defvar mew-prog-mime-encode "mewencode")
(defvar mew-prog-mime-decode "mewdecode")

(defvar mew-prog-unshar   "unshar")
(defvar mew-prog-uumerge  "uumerge")
(defvar mew-prog-tar      "tar")
(defvar mew-prog-compress "compress")
(defvar mew-prog-gzip     "gzip")

(defvar mew-prog-mime-encode-quoted '("-q"))
(defvar mew-prog-mime-decode-quoted '("-q"))

(defvar mew-prog-mime-encode-base64 '("-b"))
(defvar mew-prog-mime-decode-base64 '("-b"))
(defvar mew-prog-mime-encode-base64-text '("-b" "-t"))
(defvar mew-prog-mime-decode-base64-text '("-b" "-t"))

(defvar mew-prog-mime-encode-gzip64 '("-g"))
(defvar mew-prog-mime-decode-gzip64 '("-g"))
(defvar mew-prog-mime-encode-gzip64-text '("-g" "-t"))
(defvar mew-prog-mime-decode-gzip64-text '("-g" "-t"))

(defvar mew-prog-pick-default-arg "-from")

(defvar mew-prog-pick-argalist
  '(("-from") ("-date") ("-cc") ("-to") ("-search") ("-before")
    ("-after") ("-subject") ("-not") ("-or") ("-and")
    ("-lbrace") ("-rbrace")))

(defvar mew-prog-pick-arglogic-unary
  '("-not"))

(defvar mew-prog-pick-arglogic-binary
  '("-or" "-and"))

(defvar mew-prog-pick-argdate
  '("-before" "-after"))

(defvar mew-prog-pick-argdatealist
  '(("yesterday") ("today") ("tomorrow")
    ("sunday") ("monday") ("tuesday") ("wednesday")
    ("thursday") ("friday") ("saturday")))

(defvar mew-pick-macro-alist nil
  "mew-pick-define-macro uses this variable.")

(defvar mew-prog-scan-default-range "update"
  "*A value to decide default range of scan.")

(defvar mew-prog-scan-default-range-alist '(("+drafts" . "all"))
  "*alist of folder and default range pair for scan."
  )

(defvar mew-mark-rmm    ?D)
(defvar mew-mark-mark   ?@)
(defvar mew-mark-hop    ?*)
(defvar mew-mark-refile ?o) ;; do you like "^"?
(defvar mew-mark-tmp    ?%) ;; temporary use only.

(defvar mew-temp-file nil
  "Temporary file for Mew. This must be unreadable from others.
If not, it may become a security hole.")

(defvar mew-mark-switch
  (list 
   (cons mew-mark-rmm    'mew-delete-line)
   (cons mew-mark-refile 'mew-delete-line)
   (cons mew-mark-mark   'mew-summary-mark-letitbe)
   (cons mew-mark-hop    'mew-summary-mark-letitbe)
   )
  )

(defvar mew-undo-switch
  (list 
   (cons mew-mark-rmm    'mew-summary-unmark)
   (cons mew-mark-refile 'mew-summary-refile-undo)
   (cons mew-mark-mark   'mew-summary-unmark)
   (cons mew-mark-hop    'mew-summary-unmark)
   )
  )

;(defvar mew-draft-mime-folder (format "+%s/mime" mew-draft-folder))
(defvar mew-draft-mime-folder "+drafts/mime")
(defvar mew-draft-multipart-begin "---- multipart --")
(defvar mew-draft-multipart-end   "---- multipart ----")

(defvar mew-summary-mode-map    nil)
(defvar mew-virtual-mode-map    nil)
(defvar mew-message-mode-map    nil)
(defvar mew-draft-mode-map      nil)
(defvar mew-draft-header-map    nil)
(defvar mew-draft-body-map      nil)
(defvar mew-draft-multipart-map nil)
(defvar mew-minibuffer-map      nil)
(defvar mew-pick-map            nil)
(defvar mew-folder-map          nil)

;;;
;;; filename "^$" never matches anything
;;;

(defvar 
 mew-mime-content-type
 '(
   ("audio/basic"
    "\\.au$"	"base64"	("sh" ("-c" "cat - > /dev/audio") nil))
;   ("audio/basic"
;    "\\.au$"	"base64"	("aplay" () t))
;   ("audio/basic"
;    "\\.au$"	"base64"	("play_au" () t)) ;; BSD/OS
   ("image/gif"
    "\\.gif$" 	"base64"	("xv" ("-geometry" "+0+0") t))
   ("image/jpeg"
    "\\.jpe?g$"	"base64"	("xv" ("-geometry" "+0+0") t))
   ("image/x-xwd"
    "\\.xwd$"	"base64"	("xv" ("-geometry" "+0+0") t))
   ("image/x-xbm"
    "\\.xbm$"	"base64"	("xv" ("-geometry" "+0+0") t))
   ("image/x-bmp"
    "\\.bmp$"	"base64"	("xv" ("-geometry" "+0+0") t))
   ("image/.*"
    "^$"        "base64"	("xv" ("-geometry" "+0+0") t))
   ("video/mpeg"
    "\\.mpe?g$"	"base64"	("mpeg_play" () t))
   ("application/postscript"
    "\\.ps$"	"quoted-printable"	("ghostview" ("-geom" "+0+0") t))
   ("application/octet-stream"
    "\\.tar\\.?g?z?$" "base64"	(mew-mime-application/octet-strem () nil))
   ("application/octet-stream"
    "\\.gz$" "base64"           (mew-mime-application/octet-strem () nil))
   ("application/octet-stream"
    "\\.Z$" "base64"	        (mew-mime-application/octet-strem () nil))
   ("application/x-graph"
    "\\.xg$"	"quoted-printable"	("xgraph" () t))
   ("message/external-body" 
    "\\.extb$"	nil		(mew-mime-external-body () nil))
   ("message/rfc822"
    "^[0-9]+$"	nil		(mew-mime-message/rfc822 () nil))
;   ("text/html"
;    "\\.html$"	nil		(mew-mime-text/html () nil))
   ("text/richtext"
    "\\.rtf$"	nil		(mew-mime-text/plain () nil))
   ("text/plain"
	".*"	nil		(mew-mime-text/plain () nil))
   ("text/.*"
	"^$"    nil             (mew-mime-text/plain () nil))
   (".*" 
        "^$"    nil             (mew-mime-application/octet-strem () nil))
   )
 "(content-type filename encoding (program (options ...) async))"
 )

(defvar mew-mime-content-type-list
  '("Text/Plain" "Text/html"
    "Message/Rfc822" "Message/External-body" ;; "Message/Partial"
    "Application/Postscript" "Application/Octet-Stream"
    "Image/Jpeg" "Image/Gif"
    "Audio/Basic" "Video/Mpeg"
    "Application/X-Graph")
  "Candidate of Content-Type: when CT: is changed in draft buffer."
  )

(defvar mew-mime-content-type-multipart-list
  '("Multipart/Mixed" "Multipart/Alternative"
    ;;"Multipart/Digest" "Multipart/Parallel"
    )
  "Candidate of 'Content-Type: Multipart' when CT: is changed in draft buffer."
  )

(defvar mew-mime-content-type-binary
  '("Application/Octet-Stream"
    "Image/Jpeg" "Image/Gif" "Image/x-xwd" "Image/x-xbm" "Image/x-bmp"
    "Audio/Basic" "Video/Mpeg"
    )
  "Content-Type: list to save as binary file in mew-summary-save on Mule"
  )
     
;;
(defmacro mew-file-attr (filename alist)
  (` (mew-assoc2 (, filename) (, alist) 1 t)))

(defmacro mew-type-attr (filename alist)
  (` (mew-assoc2 (, filename) (, alist) 0 nil)))

;; for mapcar
(defun mew-file-content (attr)
  (car attr))

(defmacro mew-file-encoding (attr)
  (` (car (cdr (cdr (, attr))))))

;;
(defmacro mew-content-attr (content alist)
  (` (mew-assoc2 (, content) (, alist) 0 t)))

(defmacro mew-content-program (attr)
  (` (car (car (cdr (cdr (cdr (, attr))))))))

(defmacro mew-content-options (attr)
  (` (car (cdr (car (cdr (cdr (cdr (, attr)))))))))


(defmacro mew-content-async (attr)
  (` (car (cdr (cdr (car (cdr (cdr (cdr (, attr))))))))))

;;;
;;;
;;;

(if mew-mule-p
    (defvar mew-mule-character-set
      (list 
       (if (boundp '*iso-8859-1*)	;; Latin-1
	   (list lc-ltn1	*iso-8859-1*		"iso-8859-1")
	 (if (boundp '*ctext*) ;; for old mule
	     (list lc-ltn1	*ctext*			"iso-8859-1")))
       (if (boundp '*iso-8859-2*)	;; Latin-2
	   (list lc-ltn2	*iso-8859-2*		"iso-8859-2"))
       (if (boundp '*iso-8859-3*)	;; Latin-3
	   (list lc-ltn3	*iso-8859-3*		"iso-8859-3"))
       (if (boundp '*iso-8859-4*)	;; Latin-4
	   (list lc-ltn4	*iso-8859-4*		"iso-8859-4"))
       (if (boundp '*iso-8859-5*)	;; Cyrillic
	   (list lc-crl		*iso-8859-5*		"iso-8859-5"))
       (if (boundp '*iso-8859-6*)	;; Arabic
	   (list lc-arb		*iso-8859-6*		"iso-8859-6"))
       (if (boundp '*iso-8859-7*)	;; Greek
	   (list lc-grk		*iso-8859-7*		"iso-8859-7"))
       (if (boundp '*iso-8859-8*)	;; Hebrew
	   (list lc-hbw		*iso-8859-8*		"iso-8859-8"))
       (if (boundp '*iso-8859-9*)	;; Latin-5
	   (list lc-ltn5	*iso-8859-9*		"iso-8859-9"))
       (if (boundp '*iso-2022-jp*)	;; Japanese
	   (list lc-jp		*iso-2022-jp*		"iso-2022-jp"))
       (if (boundp '*iso-2022-kr*)	;; Korean
	   (list lc-kr		*iso-2022-kr*		"iso-2022-kr"))
;;       (if (boundp '*euc-korea*)
;;	   (list lc-kr		*euc-korea*		"euc-korea"))
       (if (boundp '*iso-2022-ss2-7*)
	   (list t		*iso-2022-ss2-7*	"iso-2022-jp-2"))
       )
      )
  )

(defmacro mew-mule-lc-attr (lc alist)
  (` (mew-assoc (, lc) (, alist) 0 nil)))

(defmacro mew-mule-lc-content (attr)
  (` (car (cdr (cdr (, attr))))))

(defmacro mew-mule-lc-symbol (attr)
  (` (car (cdr (, attr)))))

(defmacro mew-mule-content-attr (content alist)
  (` (mew-assoc2 (, content) (, alist) 2 t)))

(defmacro mew-mule-content-coding (attr)
  (` (car (cdr (, attr)))))

;;;
;;; constant
;;;

(defconst mew-buffer-message     "*Mew message*")
(defconst mew-buffer-mime       " *mew mime*")
(defconst mew-buffer-hello      " *mew hello*")
(defconst mew-buffer-cache      " *mew cache*")
(defconst mew-buffer-watch       "*Mew watch*")
(defconst mew-buffer-tmp        " *mew tmp*")
(defconst mew-buffer-completions "*Mew completions*")
(defconst mew-buffer-whom        "*Mew whom*")
(defconst mew-mime-version "1.0")

;; set here due to mew-buffer-tmp
(defvar mew-prog-audio (list "cat" "/dev/audio" mew-buffer-tmp nil))
;(defvar mew-prog-audio '("arecord" nil nil nil "-silentlevel" "-100"))

(defun mew-buffer-message ()
  (if (and window-system mew-emacs19-p)
      (concat
       mew-buffer-message
       (int-to-string
	(if mew-xemacs-p
	    (mew-member-match 
	     (cdr (assq
		   'window-id
		   (screen-parameters (window-screen (selected-window)))))
	     (sort
	      (mapcar
	       (function 
		(lambda (frame)
		  (cdr (assq
			'window-id
			(screen-parameters frame)))))
	       (screen-list))
	      (function string<)))
	  (mew-member-match 
	   (cdr (assq
		 'window-id
		 (frame-parameters (window-frame (selected-window)))))
	   (sort
	    (mapcar
	     (function 
	      (lambda (frame)
		(cdr (assq
		      'window-id
		      (frame-parameters frame)))))
	     (frame-list))
	    (function string<))))))
    mew-buffer-message
    ))


(defconst mew-hello-message 
"

Welcome to Mew world.

Mew -- Message interface to Emacs Window

%s

Copyright (C) 1994, 1995, 1996  Kazuhiko Yamamoto

Please send comments to kazu@is.aist-nara.ac.jp.

/\\\\ - \\\\/"
)

(defconst mew-error-message 
"

    Sorry, your configuration is *NOT* appropriate for Mew. 
    Please check the following items.

        (1) Does \"~/.mh_profile\" exist?
        (2) Does \"~/.mh_profile\" contain \"Path: Mail\" field?
        (3) Does \"~/.mh_profile\" contain \"Draft-Folder: drafts\" field?
        (4) Is the path for MH included in exec-path?
              Put (setq exec-path (cons \"/usr/local/bin/mh\" exec-path))
              in .emacs.
")

;;;
;;; initialize global variables
;;;

(setplist 'mew-current-state 
	  (list
	   'message    nil 
	   'cache      nil
	   'part       nil
	   'window     nil
	   ))

(defun mew-current-get (key)
  (get 'mew-current-state key))

(defun mew-current-set (key value)
  (put 'mew-current-state key value))

(defvar mew-current-marker (make-marker))
(defvar mew-current-marker2 (make-marker))
(defvar mew-decode-marker (make-marker))

(defvar mew-path nil) ;; should be nil

(setq mew-draft-folder nil)
(setq mew-alias-alist  ())
(setq mew-folder-list  ())
(setq mew-folder-alist ())
(setq mew-refile-alist ())
(setq mew-folders (list mew-inbox))

(setq mew-message-citation nil)
(setq mew-message-citation-buffer nil) ;; should be nil
(defvar mew-last-shell-command "")

(setq mew-process-file-alist nil)

(setq mail-header-separator "----")
(setq mew-decode-message nil)

;;
;; buffer local
;;

(defvar mew-summary-buffer-process nil)
(defvar mew-summary-buffer-string nil)
(defvar mew-summary-buffer-direction nil)
(defvar mew-summary-buffer-folder-cache-time nil)
(defvar mew-watch-buffer-process nil)
(defvar mew-draft-buffer-header nil)
(defvar mew-draft-buffer-multipart nil)
(defvar mew-mime-syntax ()
  "Variable to save MIME syntax list")
(defvar mew-summary-buffer-refile nil)
(defvar mew-cache-attribute nil)
(defvar mew-summary-buffer-disp-msg t)
(defvar mew-summary-buffer-left-msgs "-")
(defvar mew-multi-syntax nil)
(defvar mew-underline-overlay nil)

(mapcar
 (function make-variable-buffer-local)
 (list 'mew-summary-buffer-process
       'mew-summary-buffer-string
       'mew-summary-buffer-direction
       'mew-summary-buffer-folder-cache-time
       'mew-mime-syntax
       'mew-summary-buffer-refile
       'mew-cache-attribute
       'mew-watch-buffer-process
       'mew-draft-buffer-header
       'mew-draft-buffer-multipart
       'mew-summary-buffer-disp-msg
       'mew-summary-buffer-left-msgs
       'mew-multi-syntax
       'mew-underline-overlay
       ))

;;;
;;; environment setting
;;;

(cond 
 (mew-emacs18-p
  (defun mew-menu-define (symbol maps doc menu))
  (fset 'mew-buffer-substring 'buffer-substring))
 (mew-xemacs-p
  (defmacro mew-menu-define (symbol maps doc menu)
    (` (progn
	 (defvar (, symbol) nil (, doc))
	 (mew-menu-do-define
	  (quote (, symbol)) (, maps) (, doc) (, menu)))))
  (defun mew-menu-do-define (symbol maps doc menu)
    (set symbol menu)
    (fset symbol (list 'lambda '(e)
		       doc
		       '(interactive "@e")
		       '(run-hooks 'activate-menubar-hook)
		       '(setq zmacs-region-stays 't)
		       (list 'popup-menu symbol)))
    (define-key maps 'button3 symbol))
  (fset 'mew-buffer-substring 'buffer-substring))
 (t ;; mew-emacs19-p not xemacs
  (require 'easymenu)
  (fset 'mew-menu-define 'easy-menu-define)
  (if (fboundp 'buffer-substring-no-properties)
      (fset 'mew-buffer-substring 'buffer-substring-no-properties)
    (defun mew-buffer-substring (beg end)
      "Return the text from BEG to END, without text properties, as a string."
      (let ((string (buffer-substring beg end)))
	(set-text-properties 0 (length string) nil string)
	string)))
  )
 )

(if (< max-lisp-eval-depth 1000) (setq max-lisp-eval-depth 1000))

(if mew-summary-mode-map
    ()
  (setq mew-summary-mode-map (make-sparse-keymap))
  (define-key mew-summary-mode-map " "    'mew-summary-show)
  (define-key mew-summary-mode-map "."    'mew-summary-display)
  (define-key mew-summary-mode-map "<"    'mew-summary-display-top)
  (define-key mew-summary-mode-map ">"    'mew-summary-display-bottom)
  (define-key mew-summary-mode-map "\177" 'mew-summary-prev-page)
  (define-key mew-summary-mode-map "\r"   'mew-summary-scroll-up)
  (define-key mew-summary-mode-map "\e\r" 'mew-summary-scroll-down)
  (define-key mew-summary-mode-map "g"    'mew-summary-goto-folder)
  (define-key mew-summary-mode-map "j"    'mew-summary-jump-message)
  (define-key mew-summary-mode-map "o"    'mew-summary-refile)
  (define-key mew-summary-mode-map "O"    'mew-summary-pack)
  (define-key mew-summary-mode-map "i"    'mew-summary-inc)
  (define-key mew-summary-mode-map "s"    'mew-summary-scan)
  (define-key mew-summary-mode-map "S"    'mew-summary-sort)
  (define-key mew-summary-mode-map "d"    'mew-summary-rmm)
  (define-key mew-summary-mode-map "x"    'mew-summary-exec)
  (define-key mew-summary-mode-map "X"    'mew-summary-exec-current)
  (define-key mew-summary-mode-map "a"    'mew-summary-reply)
  (define-key mew-summary-mode-map "A"    'mew-summary-reply-with-citation)
  (define-key mew-summary-mode-map "E"    'mew-summary-reedit)
  (define-key mew-summary-mode-map "\ee"  'mew-summary-edit-again)
  (define-key mew-summary-mode-map "V"    'mew-summary-virtual)
  (define-key mew-summary-mode-map "f"    'mew-summary-forward)
  (define-key mew-summary-mode-map "F"    'mew-summary-multi-forward)
  (define-key mew-summary-mode-map "r"    'mew-summary-redist)
  (define-key mew-summary-mode-map "@"    'mew-summary-mark-mark)
  (define-key mew-summary-mode-map "*"    'mew-summary-mark-hop)
  (define-key mew-summary-mode-map "y"    'mew-summary-save)
  (define-key mew-summary-mode-map "u"    'mew-summary-undo)
  (define-key mew-summary-mode-map "U"    'mew-summary-undo-all)
  (define-key mew-summary-mode-map "n"    'mew-summary-display-down)
  (define-key mew-summary-mode-map "p"    'mew-summary-display-up)
  (define-key mew-summary-mode-map "N"    'mew-summary-display-hop-down)
  (define-key mew-summary-mode-map "P"    'mew-summary-display-hop-up)
  (define-key mew-summary-mode-map "w"    'mew-summary-send)
  (define-key mew-summary-mode-map "B"    'mew-summary-burst)
  (define-key mew-summary-mode-map "Z"    'mew-status-update)
  (define-key mew-summary-mode-map "/"    'mew-summary-search)
  (define-key mew-summary-mode-map "?"    'mew-summary-search-mark)
  (define-key mew-summary-mode-map "!"    'mew-summary-refile-again)
  (define-key mew-summary-mode-map "#"    'mew-summary-print)
  (define-key mew-summary-mode-map "|"    'mew-summary-pipe-message)
  (define-key mew-summary-mode-map "q"    'mew-summary-suspend)
  (define-key mew-summary-mode-map "Q"    'mew-summary-quit)
  (define-key mew-summary-mode-map "m"	  (make-sparse-keymap))
  (define-key mew-summary-mode-map "mo"   'mew-summary-mark-refile)
  (define-key mew-summary-mode-map "md"   'mew-summary-mark-rmm)
  (define-key mew-summary-mode-map "m@"   'mew-summary-mark-at)
  (define-key mew-summary-mode-map "m*"   'mew-summary-mark-asterisk)
  (define-key mew-summary-mode-map "ms"   'mew-summary-swap-mark<->hop)
  (define-key mew-summary-mode-map "mr"   'mew-summary-mark-regexp)
  (define-key mew-summary-mode-map "ma"   'mew-summary-mark-all)
  (define-key mew-summary-mode-map "mu"   'mew-summary-mark-undo-all)
  (define-key mew-summary-mode-map "\C-c\C-s" 'mew-summary-isearch-forward)
  (define-key mew-summary-mode-map "\C-c\C-r" 'mew-summary-isearch-backward)
  (define-key mew-summary-mode-map "\C-c\C-o" 'mew-summary-jump-to-draft-buffer)
  (define-key mew-summary-mode-map "\el"  'mew-summary-recenter)
  (define-key mew-summary-mode-map "\et"  'mew-summary-uudecode)
  (define-key mew-summary-mode-map "\C-c\C-d" 'mew-pem-decrypt-letter)
  (define-key mew-summary-mode-map "\C-cd"    'mew-pgp-decode-letter)
  (define-key mew-summary-mode-map "\C-ca"    'mew-pgp-add-key)
  (define-key mew-summary-mode-map "\C-c\C-l"
    'mew-summary-convert-local-charset)
  (define-key mew-summary-mode-map "\C-c\177" 'mew-summary-convert-header)
  (define-key mew-summary-mode-map "\es"  'mew-summary-unshar)
  (define-key mew-summary-mode-map "v"    'mew-summary-toggle-disp-msg)
  (define-key mew-summary-mode-map "\ea"  'mew-summary-toggle-analysis)
  (define-key mew-summary-mode-map "\C-c\C-x" 'mew-summary-x-face)
  (define-key mew-summary-mode-map "\C-c\C-q" 'mew-kill-buffer)
  (cond
   (mew-emacs19-p
    (mew-menu-define mew-summary-mode-menu
		     mew-summary-mode-map
		     "Menu used in Mew summary mode."
		     '("Mew"
		       ["Show" mew-summary-show t]
		       ["Next part" mew-summary-display-down t]
		       ["Previous part" mew-summary-display-up t]
		       ["Top" mew-summary-display-top t]
		       ["Bottom" mew-summary-display-bottom t]
		       ["Jump" mew-summary-jump-message t]
		       "----"
		       ["Delete" mew-summary-rmm t]
		       ["Refile" mew-summary-refile t]
		       ["Mark mark" mew-summary-mark-mark t]
		       ["Mark hop" mew-summary-mark-hop t]
		       ["Undo" mew-summary-undo t]
		       ["Undo all" mew-summary-undo-all t]
		       ["Execute" mew-summary-exec t]
		       ["Suspend" mew-summary-suspend t]
		       ["Quit" mew-summary-quit t]
		       "----"
		       ("Manipulate folder"
			["Inc" mew-summary-inc t]
			["Scan" mew-summary-scan t]
			["Pack" mew-summary-pack t]
			["Sort" mew-summary-sort t]
			["Burst" mew-summary-burst t]
			["Go to folder" mew-summary-goto-folder t]
			)
		       ("Manipulate file"
			["Save" mew-summary-save t]
			["PEM decrypt" mew-pem-decrypt-letter t]
			["PGP decode" mew-pgp-decode-letter t]
			["Convert to local charset" mew-summary-convert-local-charset t]
			["Display X-Face" mew-summary-x-face t]
			)
		       ("Write/Reply/Forward"
			["Write a mail" mew-summary-send t]
			["Reedit" mew-summary-reedit t]
			["Reply" mew-summary-reply t]
			["Reply with citation" mew-summary-reply-with-citation t]
			["Forward" mew-summary-forward t]
			["Multi forward" mew-summary-multi-forward t]
			)
		       ("Misc"
			["Recenter" mew-summary-recenter t]
			["Uudecode" mew-summary-uudecode t]
			["Unshar" mew-summary-unshar t]
			["Print" mew-summary-print t]
		        ["Pipe message" mew-summary-pipe-message t]
			["Isearch forward" mew-summary-isearch-forward t]
			["Isearch backward" mew-summary-isearch-backward t]
			["PGP add key" mew-pgp-add-key t]
			["Toggle disp msg" mew-summary-toggle-disp-msg t]
                        ["Toggle analysis" mew-summary-toggle-analysis t]
			["Convert header" mew-summary-convert-header t]
			)
		       ))
    (if mew-xemacs-p
	(define-key mew-summary-mode-map 'button2 'mew-summary-mouse-show)
      (define-key mew-summary-mode-map [mouse-2] 'mew-summary-mouse-show))
    )
   )
  )


(if mew-virtual-mode-map
    ()
  (setq mew-virtual-mode-map (make-sparse-keymap))
  (define-key mew-virtual-mode-map " "    'mew-summary-show)
  (define-key mew-virtual-mode-map "."    'mew-summary-display)
  (define-key mew-virtual-mode-map "<"    'mew-summary-display-top)
  (define-key mew-virtual-mode-map ">"    'mew-summary-display-bottom)
  (define-key mew-virtual-mode-map "\177" 'mew-summary-prev-page)
  (define-key mew-virtual-mode-map "\r"   'mew-summary-scroll-up)
  (define-key mew-virtual-mode-map "\e\r" 'mew-summary-scroll-down)
  (define-key mew-virtual-mode-map "g"    'mew-summary-goto-folder)
  (define-key mew-virtual-mode-map "j"    'mew-summary-jump-message)
  (define-key mew-virtual-mode-map "i"    'mew-summary-inc)
  (define-key mew-virtual-mode-map "a"    'mew-summary-reply)
  (define-key mew-virtual-mode-map "A"    'mew-summary-reply-with-citation)
  (define-key mew-virtual-mode-map "E"    'mew-summary-reedit)
  (define-key mew-virtual-mode-map "\ee"  'mew-summary-edit-again)
  (define-key mew-virtual-mode-map "f"    'mew-summary-forward)
  (define-key mew-virtual-mode-map "F"    'mew-summary-multi-forward)
  (define-key mew-virtual-mode-map "r"    'mew-summary-redist)
  (define-key mew-virtual-mode-map "@"    'mew-summary-mark-mark)
  (define-key mew-virtual-mode-map "*"    'mew-summary-mark-hop)
  (define-key mew-virtual-mode-map "y"    'mew-summary-save)
  (define-key mew-virtual-mode-map "u"    'mew-summary-undo)
  (define-key mew-virtual-mode-map "U"    'mew-summary-undo-all)
  (define-key mew-virtual-mode-map "n"    'mew-summary-display-down)
  (define-key mew-virtual-mode-map "p"    'mew-summary-display-up)
  (define-key mew-virtual-mode-map "N"    'mew-summary-display-hop-down)
  (define-key mew-virtual-mode-map "P"    'mew-summary-display-hop-up)
  (define-key mew-virtual-mode-map "w"    'mew-summary-send)
  (define-key mew-virtual-mode-map "B"    'mew-summary-burst)
  (define-key mew-virtual-mode-map "Z"    'mew-status-update)
  (define-key mew-virtual-mode-map "#"    'mew-summary-print)
  (define-key mew-virtual-mode-map "|"    'mew-summary-pipe-message)
  (define-key mew-virtual-mode-map "q"    'mew-summary-suspend)
  (define-key mew-virtual-mode-map "Q"    'mew-summary-quit)
  (define-key mew-virtual-mode-map "\C-c\C-s" 'mew-summary-isearch-forward)
  (define-key mew-virtual-mode-map "\C-c\C-r" 'mew-summary-isearch-backward)
  (define-key mew-virtual-mode-map "\el"  'mew-summary-recenter)
  (define-key mew-virtual-mode-map "\et"  'mew-summary-uudecode)
  (define-key mew-virtual-mode-map "\C-c\C-d" 'mew-pem-decrypt-letter)
  (define-key mew-virtual-mode-map "\C-cd"    'mew-pgp-decode-letter)
  (define-key mew-virtual-mode-map "\C-ca"    'mew-pgp-add-key)
  (define-key mew-virtual-mode-map "\C-c\C-l"
    'mew-summary-convert-local-charset)
  (define-key mew-virtual-mode-map "\C-c\177" 'mew-summary-convert-header)
  (define-key mew-virtual-mode-map "\es"  'mew-summary-unshar)
  (define-key mew-virtual-mode-map "v"    'mew-summary-toggle-disp-msg)
  (define-key mew-virtual-mode-map "\ea"  'mew-summary-toggle-analysis)
  (define-key mew-virtual-mode-map "\C-c\C-x" 'mew-summary-x-face)
  (define-key mew-virtual-mode-map "\C-c\C-q" 'mew-kill-buffer)
  )

(if mew-message-mode-map
    ()
  (setq mew-message-mode-map (make-sparse-keymap))
  (define-key mew-message-mode-map " "    'mew-message-next-page)
  (define-key mew-message-mode-map "\177" 'mew-message-prev-page)
  (define-key mew-message-mode-map "n"    'mew-message-next-msg)
  (define-key mew-message-mode-map "p"    'mew-message-prev-msg)
  (define-key mew-message-mode-map "h"    'mew-message-goto-summary)
  (define-key mew-message-mode-map "a"    'mew-message-reply)
  (define-key mew-message-mode-map "A"    'mew-message-reply-with-citation)
  (define-key mew-message-mode-map "f"    'mew-message-forward)
  (define-key mew-message-mode-map "r"    'mew-message-redist)
  (define-key mew-message-mode-map "d"    'mew-message-rmm)
  (define-key mew-message-mode-map "o"    'mew-message-refile)
  (cond
   (mew-emacs19-p
    (mew-menu-define mew-message-mode-menu
  		     mew-message-mode-map
  		     "Menu used in Mew message mode."
  		     '("Mew/message"
  		       ["Next part" mew-message-next-msg t]
  		       ["Prev part" mew-message-prev-msg t]
  		       ["Next page" mew-message-next-page t]
  		       ["Prev page" mew-message-prev-page t]
 		       ["Goto summary" mew-message-goto-summary t]
		       "---"
		       ("Reply/Forward"
			["Reply"               mew-message-reply t]
			["Reply with citation" mew-message-reply-with-citation t]
			["Forward"             mew-message-forward t]
			["Redistribute"        mew-message-redist t])
		       ["Delete"              mew-message-rmm t]
		       ["Refile"              mew-message-refile t]))
    )))

(if mew-draft-mode-map
    ()
  (setq mew-draft-mode-map (make-sparse-keymap))
  (let ((begin ?\40) (end ?\177))
    (while (<= begin end)
      (define-key mew-draft-mode-map 
	(char-to-string begin) 'mew-draft-keyswitch)
      (setq begin (1+ begin))))
  (define-key mew-draft-mode-map "\C-l"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\t"       'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-c\t"   'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-d"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-o"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-q"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-t"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-w"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-k"     'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\r"       'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\n"       'mew-draft-keyswitch)
  (define-key mew-draft-mode-map "\C-c\C-y" 'mew-draft-cite)
  (define-key mew-draft-mode-map "\C-cy"    'mew-draft-yank)
  (define-key mew-draft-mode-map "\C-c\C-w" 'mew-draft-check-whom)
  (define-key mew-draft-mode-map "\C-c\C-m" 'mew-draft-make-mime)
  (define-key mew-draft-mode-map "\C-c\C-u" 'mew-draft-undo)
  (define-key mew-draft-mode-map "\C-c\C-c" 'mew-draft-send-letter)
  (define-key mew-draft-mode-map "\C-c\C-s" 'mew-pem-sign-letter)
  (define-key mew-draft-mode-map "\C-c\C-e" 'mew-pem-encrypt-letter)
  (define-key mew-draft-mode-map "\C-cs"    'mew-pgp-sign-letter)
  (define-key mew-draft-mode-map "\C-ce"    'mew-pgp-encrypt-letter)
  (define-key mew-draft-mode-map "\C-cb"    'mew-pgp-sign-encrypt-letter)
  (define-key mew-draft-mode-map "\C-cp"    'mew-pgp-insert-public-key)
  (define-key mew-draft-mode-map "\C-c\C-q" 'mew-draft-kill)
  (define-key mew-draft-mode-map "\C-cM"    'mew-draft-prepare-multipart)
  (define-key mew-draft-mode-map "\C-c\C-n" 'mew-fib-next-item)
  (define-key mew-draft-mode-map "\C-c\C-p" 'mew-fib-previous-item)
  (define-key mew-draft-mode-map "\C-cu"    'mew-fib-flush-input)
  (define-key mew-draft-mode-map "\C-c\C-f" 'mew-fib-fill-default)
  (define-key mew-draft-mode-map "\C-c\C-k" 'mew-fib-delete-frame)
  (define-key mew-draft-mode-map "\C-c\177" 'mew-header-mime-decode)
  (cond
   (mew-emacs19-p
    (mew-menu-define
     mew-draft-mode-menu
     mew-draft-mode-map
     "Menu used in Mew draft mode."
     '("Mew/draft"
       ["Citation" mew-draft-cite t]
       ["Yank" mew-draft-yank t]
       ("Multipart"
	["Prepare multi part message"         mew-draft-prepare-multipart (not (mew-multi-p))  ]
	"----"
	["Insert a file as a part by copying" mew-multi-copy              (>= (mew-multi-line) 3)]
	["Insert a file as a part by linking" mew-multi-link              (>= (mew-multi-line) 3)]
	["Insert audio part"                  mew-multi-audio             (>= (mew-multi-line) 3)]
	["Insert an external reference"       mew-multi-external-body     (>= (mew-multi-line) 3)]
	["Insert a sub-multipart"             mew-multi-multipart         (>= (mew-multi-line) 3)]
	"----"
	["Delete this part"                   mew-multi-delete            (and (>= (mew-multi-line) 3) (mew-multi-part-number))]
	["Describe this part"                 mew-multi-description       (and (>= (mew-multi-line) 1) (mew-multi-part-number))]
	["Read this file into a buffer"       mew-multi-find-file         (and (>= (mew-multi-line) 3) (mew-multi-part-number))]
	["Read a new file into a buffer"      mew-multi-find-mew-file     (>= (mew-multi-line) 3)]
	["Rename this part"                   mew-multi-rename            (and (>= (mew-multi-line) 3) (mew-multi-part-number))]
	["Change the type of this part"       mew-multi-type              (and (>= (mew-multi-line) 1) (mew-multi-part-number))]
	["Gzip64 this part"                   mew-multi-gzip64            (and (>= (mew-multi-line) 1) (mew-multi-part-number))]
	["Encode this part with base64"       mew-multi-base64            (and (>= (mew-multi-line) 1) (mew-multi-part-number))]
	["Encode with quoted-printable"       mew-multi-quoted-printable  (and (>= (mew-multi-line) 1) (mew-multi-part-number))]
	("PGP"
	 ["Sign this part"                    mew-multi-pgp-sign          (and (>= (mew-multi-line) 1) (mew-multi-part-number))]
	 ["Encrypt this part"                 mew-multi-pgp-enc           (and (>= (mew-multi-line) 1) (mew-multi-part-number))]
	 ["Sign and encrypt this part"        mew-multi-pgp-sign-enc      (and (>= (mew-multi-line) 1) (mew-multi-part-number))]
	 )
	["Undo encoding"                      mew-multi-undo              (and (>= (mew-multi-line) 1) (mew-multi-part-number))]
	)
       ["Make mime message" mew-draft-make-mime t]
       ["Prepare multi part message" mew-draft-prepare-multipart t]
       ["Make mime message" mew-draft-make-mime t]
       ["Insert signature" mew-draft-insert-signature t]
       ["Send letter" mew-draft-send-letter t]
       ["Kill draft" mew-draft-kill t]
       "----"
       ("PEM"
	["PEM sign" mew-pem-sign-letter t]
	["PEM encrypt" mew-pem-encrypt-letter t]
	)

       ("PGP"
	["PGP sign" mew-pgp-sign-letter t]
	["PGP encrypt" mew-pgp-encrypt-letter t]
	["PGP sign then encrypt" mew-pgp-sign-encrypt-letter t]
	["PGP insert public key" mew-pgp-insert-public-key t]
	)

       ("FIB"
	["FIB next item" mew-fib-next-item t]
	["FIB previous item" mew-fib-previous-item t]
	["FIB flush input" mew-fib-flush-input t]
	["FIB fill default" mew-fib-fill-default t]
	["FIB delete frame" mew-fib-delete-frame t]
	)
       ))
    ))
  )

(if mew-draft-header-map
    ()
  (setq mew-draft-header-map (make-sparse-keymap))
  (define-key mew-draft-header-map "\t"     'mew-draft-header-comp)
  (define-key mew-draft-header-map "\C-c\t" 'mew-draft-header-domain)
  (if mew-use-bbdb
      (define-key mew-draft-header-map "\e\t" 'bbdb-complete-name))
  )

(if mew-draft-body-map
    ()
  (setq mew-draft-body-map (make-sparse-keymap))
  (define-key mew-draft-body-map "\t"     'tab-to-tab-stop)
  (define-key mew-draft-body-map "\C-c\t" 'mew-draft-insert-signature)
  )

(if mew-draft-multipart-map
    ()
  (setq mew-draft-multipart-map (make-keymap))
  (let ((begin ?\0) (end ?\177))
    (while (<= begin end)
      (define-key mew-draft-multipart-map 
	(char-to-string begin) 'mew-draft-null-function)
      (setq begin (1+ begin))))
  (define-key mew-draft-multipart-map "a"    'mew-multi-audio)
  (define-key mew-draft-multipart-map "c"    'mew-multi-copy)
  (define-key mew-draft-multipart-map "d"    'mew-multi-delete)
  (define-key mew-draft-multipart-map "D"    'mew-multi-description)
  (define-key mew-draft-multipart-map "e"    'mew-multi-external-body)
  (define-key mew-draft-multipart-map "f"    'mew-multi-find-file)
  (define-key mew-draft-multipart-map "F"    'mew-multi-find-mew-file)
  (define-key mew-draft-multipart-map "l"    'mew-multi-link)
  (define-key mew-draft-multipart-map "m"    'mew-multi-multipart)
  (define-key mew-draft-multipart-map "r"    'mew-multi-rename)
  (define-key mew-draft-multipart-map "T"    'mew-multi-type)
  (define-key mew-draft-multipart-map "G"    'mew-multi-gzip64)
  (define-key mew-draft-multipart-map "B"    'mew-multi-base64)
  (define-key mew-draft-multipart-map "Q"    'mew-multi-quoted-printable)
  (define-key mew-draft-multipart-map "S"    'mew-multi-pgp-sign)
  (define-key mew-draft-multipart-map "E"    'mew-multi-pgp-enc)
  (define-key mew-draft-multipart-map "P"    'mew-multi-pgp-sign-enc)
  (define-key mew-draft-multipart-map "U"    'mew-multi-undo)
  )

(if mew-minibuffer-map
    ()
  (setq mew-minibuffer-map (make-sparse-keymap))
  (define-key mew-minibuffer-map "\t"     'mew-minibuffer-alias)
  (define-key mew-minibuffer-map "\C-c\t" 'mew-minibuffer-domain)
  (define-key mew-minibuffer-map "\r"     'exit-minibuffer)
  (define-key mew-minibuffer-map "\n"     'exit-minibuffer)
  (define-key mew-minibuffer-map "\C-g"   'abort-recursive-edit)
  (if mew-use-bbdb
      (define-key mew-minibuffer-map "\e\t"     'bbdb-complete-name))
  )

(if mew-pick-map
    ()
  (setq mew-pick-map (make-sparse-keymap))
  (define-key mew-pick-map "\t"   'mew-pick-complete)
  (define-key mew-pick-map "\r"   'exit-minibuffer)
  (define-key mew-pick-map "\n"   'exit-minibuffer)
  (define-key mew-pick-map "\C-g" 'abort-recursive-edit)
  )

(if mew-folder-map
    ()
  (setq mew-folder-map (make-sparse-keymap))
  (define-key mew-folder-map "\t"     'mew-minibuffer-folders)
  (define-key mew-folder-map "\r"     'exit-minibuffer)
  (define-key mew-folder-map "\n"     'exit-minibuffer)
  (define-key mew-folder-map "\C-g"   'abort-recursive-edit)
  )

(cond 
 (mew-mule-p
  (define-program-coding-system nil mew-prog-inc  mew-mule-charset-local)
  (define-program-coding-system nil mew-prog-scan mew-mule-charset-local)
  (define-program-coding-system nil mew-prog-send mew-mule-charset-local)
  (define-program-coding-system nil mew-prog-pick mew-mule-charset-local)
  (define-program-coding-system nil mew-prog-mime-encode *noconv*)
  (define-program-coding-system nil mew-prog-mime-decode *noconv*)
  (define-program-coding-system nil mew-prog-vscan mew-mule-charset-virtual)
  )
 )

(cond 
 ((null mew-draft-mode-syntax-table)
  (setq mew-draft-mode-syntax-table
	(make-syntax-table text-mode-syntax-table))
  (set-syntax-table mew-draft-mode-syntax-table)
  (modify-syntax-entry ?% "." mew-draft-mode-syntax-table))
 )

;;;
;;; Basic functions
;;;

(defun mew-version-show ()
  (interactive)
  (message "%s" mew-version))

(defun mew-replace-character (string from to)
  (let ((len (length string))
	(cnt 0))
    (while (< cnt len)
      (if (equal (aref string cnt) from)
	  (aset string cnt to))
      (setq cnt (1+ cnt)))
    string
    ))

(defun mew-y-or-n-p (string)
  (interactive)
  (if mew-emacs-y-or-n-p-use
      (y-or-n-p string)
    (let ((char nil) (ociea cursor-in-echo-area))
      (unwind-protect
	  (progn
	    (setq cursor-in-echo-area t)
	    (catch 'break
	      (while t
		(message (concat string "(y=RET or n=SPC) "))
		(setq char (read-char))
		(cond
		 ((equal char ?y)
		  (setq char t)
		  (throw 'break nil))
		 ((equal char ?\r)
		  (setq char t)
		  (throw 'break nil))
		 ((equal char ?n)
		  (setq char nil)
		  (throw 'break nil))
		 ((equal char 32)
		  (setq char nil)
		  (throw 'break nil))
		 (t (ding))
		 )
		))
	    )
	(setq cursor-in-echo-area ociea)
	)
      char ;; return value
      ))
  )

(defun mew-member-match (str list)
  (let ((n 0))
    (catch 'member
      (while list
	(if (equal (downcase (car list)) (downcase str))
	    (throw 'member n))
	(setq list (cdr list))
	(setq n (1+ n))
	))
    ))

(defun mew-delq (key list)
  (let* ((pointer (cons nil list))
	 (top pointer))
    (while (cdr pointer)
      (if (equal key (car (cdr pointer))) 
	  (progn
	    (setcdr pointer (cdr (cdr pointer)))
	    (setq pointer (cons nil nil)))
	(setq pointer (cdr pointer))))
    (cdr top)))

(defun mew-folder-to-dir (folder)
  (if (equal (aref folder 0) ?+)
      (substring folder 1 nil)
    folder)
  )

(defun mew-dir-to-folder (dir)
  (if (equal (aref dir 0) ?+)
      dir
    (concat "+" dir))
  )

(defmacro mew-draft-to-mime (draft)
  (` (concat mew-draft-mime-folder "/" (file-name-nondirectory (, draft)))))

(defun mew-make-folder (folder)
  (mew-make-directory (mew-folder-to-dir folder)))

(defun mew-make-directory (path)
  (let ((parent (directory-file-name (file-name-directory path))))
  (if (null (file-directory-p parent))
    (mew-make-directory parent))
  (if (and (file-exists-p path) (not (file-directory-p path)))
      (delete-file path))
  (call-process "mkdir" nil nil nil path)
  ))

;; I don't like such a procedural programming, but for max depth safety.
(defun mew-rassoq (a alist)
  (catch 'conscell
    (while alist
      (if (equal (cdr (car alist)) a)
	  (throw 'conscell (car alist)))
      (setq alist (cdr alist)))))

(defun mew-match (pos &optional string)
  (cond 
   ((stringp string)
    (substring string (match-beginning pos) (match-end pos)))
   (t 
    (mew-buffer-substring (match-beginning pos) (match-end pos)))
   ))

(defun mew-assoc (key alist nth match)
  (let ((case-fold-search t))
    (if match
	(mew-assoc-match key alist nth)
      (mew-assoc-equal key alist nth))))

(defun mew-assoc-match (key alist nth)
  (if (null alist) 
      ()
    (let* ((car (car alist))
	   (nthobj (nth nth car)))
      (if (or (and (stringp nthobj)
		   (let ((case-fold-search t))
		     (string-match key nthobj))) ;;; here here
	      (equal nthobj key)
	      (eq nthobj t))
	  car
	(mew-assoc-match key (cdr alist) nth)))))

(defun mew-assoc2 (key alist nth match)
  (let ((case-fold-search t))
    (if match
	(mew-assoc-match2 key alist nth)
      (mew-assoc-equal key alist nth))))

(defun mew-assoc-match2 (key alist nth)
  (if (null alist) 
      ()
    (let* ((car (car alist))
	   (nthobj (nth nth car)))
      (if (or (and (stringp nthobj)
		   (let ((case-fold-search t))
		     (string-match nthobj key))) ;;; here here
	      (equal nthobj key)
	      (eq nthobj t))
	  car
	(mew-assoc-match2 key (cdr alist) nth)))))

(defun mew-assoc-equal (key alist nth)
  (if (null alist) 
      ()
    (let* ((car (car alist))
	   (nthobj (nth nth car)))
      (if (or (equal nthobj key)
	      (eq nthobj t))
	  car
	(mew-assoc-equal key (cdr alist) nth)))))

;;(defun mew-assoc2 (key alist nth exact)
;;  (let ((case-fold-search t))
;;    (cond
;;     ((null alist) ())
;;     ((and exact 
;;	   (stringp (nth nth (car alist)))
;;	   (equal (downcase key) (downcase (nth nth (car alist)))))
;;      (car alist))
;;     ((equal (nth nth (car alist)) key)
;;      (car alist))
;;     ((eq (nth nth (car alist)) t)
;;      (car alist))
;;     (t (mew-assoc key (cdr alist) nth exact))
;;     )
;;    ))

(defmacro mew-directory-empty-p (dir)
  (` (null (car (cdr (cdr (directory-files (, dir)))))))
  )

(defun mew-which(file path)
  (catch 'loop
    (while path
      (if (file-exists-p (expand-file-name file (car path)))
	  (throw 'loop (expand-file-name file (car path)))
	(setq path (cdr path)))
      )
    ))

;;;
;;; Window configuration stack
;;;

(defvar mew-window-stack nil)

(defun mew-window-push ()
  (interactive)
  (let* ((key (cond
	       ((and mew-emacs19-p (not mew-xemacs-p))
		(selected-frame))
	       (t 'dummy)
	       ))
	 (stack (mew-alist-search mew-window-stack key)))
    (setq stack (cons (current-window-configuration) stack))
    (setq mew-window-stack
	  (mew-alist-add mew-window-stack key stack))
    ))

(defun mew-window-pop ()
  (interactive)
  (let* ((key (cond
	       ((and mew-emacs19-p (not mew-xemacs-p))
		(selected-frame))
	       (t 'dummy)
	       ))
	 (stack (mew-alist-search mew-window-stack key)))
    (if stack
	(set-window-configuration (car-safe stack))
      (if (and mew-emacs19-p
	       (not mew-xemacs-p)
	       (nth 1 (frame-list))
	       (mew-y-or-n-p "Delete this frame? "))
	  (delete-frame)
	(message "Window Stack is empty.")))
    (setq stack (cdr-safe stack))
    (setq mew-window-stack
	  (mew-alist-add mew-window-stack key stack))
    ))

;;;
;;; Bootstrap --- mew and mew-send
;;;

(defun mew (&optional arg)
  (interactive "P")
  (mew-window-push)
  (if (null mew-path) (mew-init))
  (if arg
      (mew-summary-goto-folder) ;; C-u
    (let ((folder mew-inbox))
      (if (get-buffer folder)
	  (switch-to-buffer folder)
	(mew-summary-folder-create folder))
      (mew-summary-inc)
      )
    ))

(defun mew-send (&optional to cc subject)
  (interactive)
;  (mew-window-push) ;; xxx
  (mew-current-set 'window (current-window-configuration))
  (if (null mew-path) (mew-init))
  (mew-summary-send t to cc subject)
  )

;;
;; Functions for boot time
;;

(defun mew-init ()
  (mew-hello)
  (if mew-demo
      (mew-demo mew-demo-string))
  (message "Setting mew world ...")
  (mew-set-environment)
  (if mew-use-bbdb (require 'bbdb-com)) ;; bbdb is implicitly required
  (run-hooks 'mew-init-hook)
  (message "Setting mew world ... done")
  )

(defun mew-center-line ()
  (save-excursion
    (end-of-line)
    (let ((ll (current-column)))
      (beginning-of-line)
      (indent-to (/ (- (window-width) ll) 2))
      )
    ))

(defun mew-center-region (from to)
  (interactive "r")
  (if (> from to)
      (let ((tem to))
        (setq to from from tem)))
  (save-excursion
    (save-restriction
      (narrow-to-region from to)
      (goto-char from)
      (while (not (eobp))
        (mew-center-line)
        (forward-line 1)))))

(defun mew-hello ()
  (let ((nbuf (get-buffer-create mew-buffer-hello)))
    (mew-window-configure nbuf '(1 0))
    (erase-buffer)
    (insert (format mew-hello-message mew-version))
    (mew-center-region (point-min) (point-max))
    (end-of-line)
    (insert (make-string (1- (- (window-width) (current-column))) 32))
    (sit-for 0) ;; to redraw
    ))

(defun mew-set-environment ()
  (let ((xerr nil))
    (condition-case err
	(progn
	  (set-buffer (get-buffer-create mew-buffer-tmp))
	  (erase-buffer)
	  (if (file-exists-p mew-profile)
	      (insert-file-contents mew-profile)
	    (setq xerr (format "No %s" mew-profile))
	    (error ""))
	  (if (setq mew-path (mew-field-get-value "Path:"))
	      (setq mew-path
		    (expand-file-name 
		     (mew-header-delete-ws mew-path)
		     (expand-file-name "~")))
	    (setq xerr "No Path:")
	    (error ""))
	  (if (setq mew-draft-folder (mew-field-get-value "Draft-Folder:"))
	      (progn
		(setq mew-draft-folder
		      (format "+%s" (mew-folder-to-dir
				     (mew-header-delete-ws		  
				      mew-draft-folder))))
		(setq mew-temp-file
		      (mew-expand-file-name "temp" mew-draft-folder))
		)
	    (setq xerr "No Draft-Folder:")
	    (error ""))
	    ;;
	    (if (null mew-alias-alist)
		(setq mew-alias-alist (mew-alias-make-alist)))
	    ;;
	    (setq mew-folders-file 
		  (expand-file-name mew-folders-file mew-path))
	    (cond 
	     ((null mew-folder-list)
	      (setq mew-folder-list (mew-folder-make-list))
	      (setq mew-folder-alist (mew-folder-make-alist mew-folder-list))
	      (setq mew-refile-alist (mew-refile-make-alist mew-folder-list))
	      ))
	    )
      (error (set-buffer mew-buffer-hello)
	     (insert mew-error-message)
	     (set-buffer-modified-p nil)
	     (setq buffer-read-only t)
	     (if xerr (error xerr)
	       (error "If you have ~/Mail/.mew-from-alist, remove it. Or maybe exec-path error")))
      )
    ))

;;;
;;; window configuration
;;;

(defun mew-window-configure (nbuf action)
;;; action : summary, message or list
;;; list for action (1 0)  for Summary only
;;; list for action (3 10) for Summary and Message
  (if (equal action 'summary)
      (mew-current-set 'message nil))
  (let* ((windows
	  (if (listp action) 
	      action
	    (car (cdr (assq action mew-window-configuration)))))
	 (obufwin (get-buffer-window (current-buffer)))
	 (msgwin  (get-buffer-window (mew-buffer-message)))
	 (height nil) (winsum nil) (sumheight 0) (msgheight 0))
    (cond 
     (mew-window-use-full
      ;; Delete other windows and use full emacs window.
      (delete-other-windows)
      (setq height (window-height (selected-window))))
     (t
      (setq height
	    (+ (if obufwin (window-height obufwin) 0)
	       (if msgwin  (window-height msgwin)  0)
	       ))))
    (if (get-buffer (mew-buffer-message))
	(delete-windows-on (mew-buffer-message))
      (save-excursion
	(set-buffer (get-buffer-create (mew-buffer-message)))
	(mew-message-mode)))
    (setq winsum (apply (function +) windows))
    (if (not (zerop (nth 0 windows)))
	(setq sumheight (max window-min-height
			     (/ (* height (nth 0 windows)) winsum))))
    (if (equal (% sumheight 2) 1) (setq sumheight (1+ sumheight)))
    (if (not (zerop (nth 1 windows)))
	(setq msgheight (max window-min-height
			     (- height sumheight))))
    (setq height (+ sumheight msgheight))

    (if (or (not mew-xemacs-p)
	    ;; xemacs doesn't like to enlarge the only window:
	    (> (length (window-list)) 2))
	(enlarge-window (max 0 (- height (window-height (selected-window))))))

    (if (null (zerop sumheight))
	(switch-to-buffer nbuf 'norecord))
    (if (zerop msgheight)
	()
      (split-window nil sumheight)
      (other-window 1)
      (switch-to-buffer (mew-buffer-message) 'norecord))
    ))

;;;
;;; Input method
;;;

(defun mew-input-folder (default)
  (let ((folder))
    (setq folder (completing-read (format "Folder name (%s): " default)
				  mew-folder-alist
				  nil
				  nil  ;; not require match
				  "+"))
    (directory-file-name 
     (if (or (string= folder "") (string= folder "+"))
 	 default
       folder))))

(defun mew-input-folders (default &optional string)
  (let ((folders))
    (setq folders
	  (read-from-minibuffer (format "Folder name (%s): " default)
				(or string "+")
				mew-folder-map))
    (if (or (string= folders "") (string= folders "+"))
	(setq folders default))
    (mapcar (function directory-file-name) 
	    (mew-header-split
	     (mew-header-delete-ws folders) ?,))
    ))

(defun mew-input-range (folder)
  "Return (range erase-update)."
  (mew-summary-multipart-delete)
  ;; for the case when parts are expanded in the bottom of the folder
  (let*
      ((pair (assoc folder mew-prog-scan-default-range-alist))
       (default (or (cdr pair) mew-prog-scan-default-range))
       (range (read-string (format "Range (%s): " default) "")))
    (if (string= range "")
	(setq range default))
    (cond
     ;; range is other than "update"
     ((not (string= range "update"))
      (list range 'erase)) ;; non-update, erase it
     ;; update
     ((get-buffer folder)
      (save-excursion
	(set-buffer folder)
	(goto-char (point-max))
	(if (bobp)
	    (list "all" 'update) ;; buffer is empty. no need to erase
	  (forward-line -1)
	  (list 
	   (concat
	    (int-to-string (1+ (string-to-int (mew-summary-message-number))))
	    "-" 
	    "last")
	   'update) ;; this is update!
	  )))
     ;; update but folder doesn't exist in Emacs. 
     (t (list "all" 'update)) ;; no need to erase
     )
    ))

(defun mew-input-address (prompt) ;; prompt="To:"
  (read-from-minibuffer (concat prompt " ") "" mew-minibuffer-map nil)
  )

(defun mew-input-filename (&optional prompt)
  (expand-file-name (read-file-name (or prompt "File : ") "~/")))

(defun mew-input-directory-name ()
  (let ((dir (expand-file-name (read-file-name "Directory : " nil "~" t))))
    (if (file-directory-p dir)
	dir
      (mew-input-directory-name)
      )
    ))

(defun mew-input-string (prompt subdir default)
  (let ((input (read-string (format prompt subdir default) "")))
    (if (string= input "") default input))
  )

(defun mew-input-type (filename type-list default)
  (let ((completion-ignore-case t)
	(type))
    (setq type (completing-read
		(format "Type for %s (%s): " filename default)
		(mapcar (function (lambda (x) (cons x x)))
			type-list)
		nil
		t  ;; not require match
		""))
    (if (string= type "") default type)
    ))

;;;
;;; Misc
;;;

(defun mew-boundary-get (&optional string)
  (if (null string) (setq string "Next_Part"))
  (format mew-default-boundary
	  string
	  (mew-replace-character (current-time-string) 32 ?_) ; 32 == " "
	  )
  )

(defun mew-expand-file-name (msg &optional buffer-or-name)
  (if (bufferp buffer-or-name)
      (setq buffer-or-name (buffer-name buffer-or-name)))
  (and buffer-or-name 
       (setq buffer-or-name (directory-file-name buffer-or-name)))
  (if buffer-or-name
      (expand-file-name 
       (concat (mew-folder-to-dir buffer-or-name) "/" (mew-folder-to-dir msg))
       mew-path)
    (expand-file-name (mew-folder-to-dir msg) mew-path))
  )

;;;
;;; Message header oparation
;;;

(defun mew-field-get-value (field)
  "currently, when no match, it returns nil."
  (let ((case-fold-search t)
	(start nil)
	(key nil)
	(match nil)
	(count 0)
	(ret nil)) ;; (concat nil "foo") -> "foo"
    (save-excursion
      (goto-char (point-min))
      (catch 'header
	(while (re-search-forward (format "^-*$\\|^%s[ \t]*" field) nil t)
	  (if (>= count mew-loop-depth)
	      (throw 'header ret)) ;; xxx warn to users?
	  (setq count (1+ count))
	  (setq key (mew-match 0))
	  (setq start (match-end 0))
	  (if (string-match "^-*$" key) (throw 'header ret))
	  (forward-line 1)
	  (while (looking-at "[ \t]")
	    (forward-line 1))
	  (setq match (mew-buffer-substring start (1- (point))))
	  (if (null ret)
	      (if (null (string= "" match))
		  (setq ret match))
	    (if (null (string= "" match))
		(setq ret (concat ret "," match))))
	)))
    ret
    ))
	  
(defun mew-field-get-line (field)
;; return  with ^J
;; currently, when no match, it returns nil.
  (let ((case-fold-search t)
	(start nil)
	(key nil)
	(count 0)
	(ret nil)) ;; (concat nil "foo") -> "foo"
    (save-excursion
      (goto-char (point-min))
      (catch 'header
	(while (re-search-forward (format "^-*$\\|^%s" field) nil t)
	  (if (>= count mew-loop-depth)
	      (throw 'header ret)) ;; xxx warn to users?
	  (setq count (1+ count))
	  (setq key (mew-match 0))
	  (setq start (match-beginning 0))
	  (if (string-match "^-*$" key) (throw 'header ret))
	  (forward-line 1)
	  (while (looking-at "[ \t]")
	    (forward-line 1))
	  (setq ret (concat ret (mew-buffer-substring start (point))))
	  ))
      )
    ret
    ))

(defun mew-field-delete-lines (fields)
  (let ((case-fold-search t)
	(start nil)
	(key nil)
	(regex (mew-make-field-regex fields)))
    (save-excursion
      (goto-char (point-min))
      (catch 'header
	(while (re-search-forward (concat "^-*$\\|" regex) nil t)
	  (setq key (mew-match 0))
	  (setq start (match-beginning 0))
	  (if (string-match "^-*$" key) (throw 'header nil))
	  (forward-line 1)
	  (while (looking-at "[ \t]")
	    (forward-line 1))
	  (delete-region start (point))
	  ))
      )))

(defun mew-field-resent-lines (fields)
  (let ((case-fold-search t)
	(start nil)
	(key nil)
	(regex (mew-make-field-regex fields)))
    (save-excursion
      (goto-char (point-min))
      (catch 'header
	(while (re-search-forward (concat "^-*$\\|" regex) nil t)
	  (setq key (mew-match 0))
	  (setq start (match-beginning 0))
	  (if (string-match "^-*$" key) (throw 'header nil))
	  (beginning-of-line)
	  (insert "Prev-")
	  ))
      )))


(defun mew-field-insert-last (field value)
  (save-excursion
    (let ((case-fold-search t))
      (if (string= value "")
	  ()
	(goto-char (point-min))
	(re-search-forward 
	 (format "^-*$\\|^%s" field))
	(let ((match (mew-match 0)))
	  (if (string= (downcase match) (downcase field))
	      (error "Field %s already exists." field)
	    (beginning-of-line)
	    (insert-before-markers (format "%s %s\n" field value)))
	  )
	))))

(defun mew-field-insert-here (field &optional value can)
  (if can
      (if value
	  (insert (format "%s %s\r\n" field value))
	(insert (format "%s\r\n" field)))
    (if value
	(insert (format "%s %s\n" field value))
      (insert (format "%s\n" field)))
    ))

;;;
;;; field parser
;;;

(defun mew-header-syntax (string)
  (mew-header-split
   (mew-header-concat-list
    (mew-header-delete-ws-list 
     (mew-header-separate-quote string)))
   ?;))

(defun mew-header-split (string sep)
 (let ((quote nil)
       (len (length string))
       (ret nil)
       (start 0)
       (n 0))
   (while (< n len)
     (cond
      ((equal (aref string n) ?\")
       (setq quote (not quote)))
      ((equal (aref string n) sep)
       (if (null quote)
	   (progn
	     (setq ret (cons (substring string start n) ret))
	     (setq start (1+ n))
	     )
	 ))
      )
     (setq n (1+ n))
     )
   (setq ret (cons (substring string start n) ret))
   (nreverse ret)
   ))

(defmacro mew-header-concat-list (list)
  (` (mapconcat (function (lambda (x) x)) (, list) "")))

(defmacro mew-header-delete-ws-list (list)
  (` (mapcar (function mew-header-delete-ws) (, list))))

(defmacro mew-header-delete-ws2-list (list)
  (` (mapcar (function mew-header-delete-ws2) (, list))))

(defun mew-header-delete-ws (string)
  (if (eq (aref string 0) ?\")
      (substring string 1 (1- (length string)))
    (while (string-match "[ \t]+\\|\n[ \t]+\\|([^()]*)" string)
      ;; regular expression cannot express recursive paretheses.
      ;; delete () again and again.
      (setq string (concat (substring string 0 (match-beginning 0))
			   (substring string 
				      (match-end 0)
				      (length string)))))
    string))

(defun mew-header-delete-ws2 (string)
  (if (eq (aref string 0) ?\")
      string
    (while (string-match "[ \t]+\\|\n[ \t]+\\|([^()]*)" string)
      (setq string (concat (substring string 0 (match-beginning 0))
			   (substring string 
				      (match-end 0)
				      (length string)))))
    string))

(defun mew-header-separate-quote (string)
  (let ((ret nil) begin (end 0) (lastend 0))
    (while (string-match "\"[^\"]*\"" string end)
      (setq lastend end)
      (setq begin (match-beginning 0))
      (setq end (match-end 0))
      (if (equal begin lastend)
	  (setq ret (cons (substring string lastend end) ret))
	(setq ret (cons 
		   (substring string begin end)
		   (cons 
		    (substring string lastend begin)
		    ret))))
      )
    (if (equal (length string) end)
	nil
      (setq ret (cons (substring string end nil) ret)))
    (nreverse ret)
    ))

;;;
;;; extract address
;;;

(defmacro mew-header-extract-addr-list (list)
  (` (mapcar (function mew-header-extract-addr) (, list))))

(defun mew-header-extract-addr (str)
  "Extracts a real e-mail address from STR and returns it.
e.g. \"Mine Sakurai <m-sakura@ccs.mt.nec.co.jp>\"
  ->  \"m-sakura@ccs.mt.nec.co.jp\".
e.g. \"m-sakura@ccs.mt.nec.co.jp (Mine Sakurai)\"
  ->  \"m-sakura@ccs.mt.nec.co.jp\"."
  (cond ((string-match ".*<\\([^>]*\\)>" str) ;; .* to extract last <>
         (mew-match 1 str))
        ((string-match "\\([^ \t]*@[^ \t]*\\)" str)
         (mew-match 1 str))
        (t str)
    )
  )

(defun mew-header-delete-nullstring-list (list)
  (if (null list)
      ()
    (cond 
     ((string= (car list) "") 
      (mew-header-delete-nullstring-list (cdr list)))
     (t 
      (cons (car list) (mew-header-delete-nullstring-list (cdr list)))))))

(defun mew-header-user-collect (list &optional func)
  (let ((fields (mapcar (function mew-field-get-value) list))
	(users "") (count 0) (begin 0))
    (while fields
      (if (car fields) 
	  (setq users (concat users "," (car fields))))
      (setq fields (cdr fields)))
    (while (and (< count mew-loop-depth) 
		(string-match "," users begin))
      (setq begin (match-end 0))
      (setq count (1+ count))
      )
    (if (equal count mew-loop-depth) (setq users (substring users 0 begin)))
    (let ((list
	   (mew-header-delete-nullstring-list
	    (mew-header-delete-at-list
	     (mew-header-extract-addr-list
	      (mew-header-split
	       (mew-header-concat-list
		(mew-header-delete-ws2-list
		 (mew-header-separate-quote users)))
	       ?,))))))
      (if func
	  (mapcar func list)
	list))
    ))

(defun mew-header-address-collect (list)
  (let ((fields (mapcar (function mew-field-get-value) list))
	(users "") (count 0) (begin 0))
    (while fields
      (if (car fields) 
	  (setq users (concat users "," (car fields))))
      (setq fields (cdr fields)))
    (while (and (< count mew-loop-depth) 
		(string-match "," users begin))
      (setq begin (match-end 0))
      (setq count (1+ count))
      )
    (if (>= count mew-loop-depth)
	(progn
	  (message "Too many addresses, truncated after %d" mew-loop-depth)
	  (ding)
	  (sit-for 3) ;; enough time to read?
	  ))
    (if (equal count mew-loop-depth) (setq users (substring users 0 begin)))
    (mew-header-delete-nullstring-list
     (mew-header-extract-addr-list
      (mew-header-split
       (mew-header-concat-list
	(mew-header-delete-ws2-list
	 (mew-header-separate-quote users)))
       ?,)))
    ))


(defmacro mew-header-delete-at-list (list)
  (` (mapcar (function mew-header-delete-at) (, list))))

;; never macro for mapcar
;; by Motonori NAKAMURA <motonori@cs.ritsumei.ac.jp> on 17 Aug 1994
(defun mew-header-delete-at (string)
  (if (string-match "@.*:" string)
      (setq string (substring string (match-end 0) (length string)))
    (setq string (substring string 0 (string-match "%" string)))
    (setq string (substring string 0 (string-match "@" string)))
    (substring string 0 (string-match ":;" string))
    ))

;;
;; e-mail address canonical form 
;;

(defmacro mew-header-canform-list (list)
  (` (mapcar (function mew-header-canform) (, list))))

(defun mew-header-canform (string)
  "Complete STRING with mew-mail-domain."
  (if (string= mew-mail-domain "")
      string
    (if (string-match "^.*@\\(.*\\)$" string)
	(let ((last-dom nil)
	      (dom (mew-match 1 string))
	      (rest-dom nil)
	      (len (length mew-mail-domain)))
	  (if (string-match "^.*\\.\\([^.]*\\)$" dom)
	      (setq last-dom (mew-match 1 dom))
	    (setq last-dom dom))
	  (if (string-match (concat "\\(" last-dom "\\)") mew-mail-domain)
	      ;; \\(?
	      (setq rest-dom (substring mew-mail-domain (match-end 0) len)))
	  (if rest-dom
	      (concat string rest-dom)
	    string))
      (concat string "@" mew-mail-domain))))

;;
;; e-mail address alias
;;

(defun mew-alias-expand (alias)
  "The function of MH alias expansion. If ALIAS can be expanded, it 
returns the expansion. Otherwise, it returns ALIAS itself."
  (let ((expn (cdr (assoc alias mew-alias-alist))))
    ;assoc returns nil if alist is nil
    (if expn expn alias)))

(defun mew-alias-make-alist ()
  "Make alias alist with association of (alias . expantion) from
MH \"ali\" command. Currently, only \"user: user@domain\" syntax
is supported."
  (save-excursion
    (let ((case-fold-search t)
	  (alias nil)
	  (expn nil)
	  (alist nil)
	  (buf (get-buffer-create mew-buffer-tmp)))
      (set-buffer buf)
      (erase-buffer)
      (call-process mew-prog-ali nil t nil "-nolist" "-nonormalize" "-nouser")
      ;; concat separated lines by comma
      (goto-char (point-min))
      (while (re-search-forward ",$" nil t)
	(end-of-line)
	(forward-char 1)
 	(delete-backward-char 1))
      ;;
      (goto-char (point-min))
      (while (re-search-forward "^\\([^:]+\\):\\s-+\\(.*\\)$" nil t)
	(setq alias (mew-match 1)
	      expn (mew-match 2))
	;; append for first assoc comes first
	(setq alist (cons (cons alias expn) alist)))
      ;; load from-alist
      (if (not mew-from-alist)
	  (setq mew-from-alist
		(mew-alist-load mew-from-file-name)))
      (mapcar
       '(lambda (arg)
	  (if (and (car arg)
		   (not (assoc (mew-header-delete-at (car arg)) alist)))
	      (setq alist 
		    (cons (cons (mew-header-delete-at (car arg)) 
				(car arg))
			  alist))))
       mew-from-alist)
      (nreverse alist) ; return value
      )))

(defun mew-header-expand-alias-list (list)
  "The function of MH alias expansion. One \"ali\" command is executed
   for all the members of LIST. And, list is returned."
  (let ((field-value nil)
	(multi-line nil))
    (if (null list)
	()
      (save-excursion
	;; alias extraction
        (set-buffer mew-buffer-tmp)
        (erase-buffer)
	(apply 'call-process 
	       mew-prog-ali nil mew-buffer-tmp nil "-list" list)
	(goto-char (point-min))
        (setq list nil)
        (while (not (eobp))
          (let ((p (point))
                (theline nil))
            (end-of-line)
            (setq theline (mew-buffer-substring p (point)))
            (delete-region p (point))
	    (setq list (cons theline list))
            (forward-line)))
        (nreverse list)))))

;;
;; folders
;;

(defmacro mew-folder-make-alist (list)
  (` (mapcar (function mew-folder-pair) (, list))))

;; never defmacro for (mapcar)
(defun mew-folder-pair (string)
  (cons (format "+%s" string) (format "+%s" string)))

(defmacro mew-refile-make-alist (list)
  (` (mapcar (function mew-refile-pair) (, list))))


(defun mew-refile-pair (string)
  (if (mew-folders-ignore-p string)
      nil
    (cons (downcase (file-name-nondirectory (directory-file-name string)))
	  (format "+%s" (directory-file-name string)))

    ))

(defun mew-folders-ignore-p (string)
  (let ((ilist mew-folders-ignore))
    (catch 'ignore
      ;; while always returns nil
      (while ilist
	(if (string-match (concat "^" (car ilist))
			  string)
	    (throw 'ignore t))
	(setq ilist (cdr ilist))))))

(defun mew-folder-make-list ()
  (save-excursion
    (let ((case-fold-search t)
	  (folders ())
	  (folder nil)
	  (start nil))
      (set-buffer (get-buffer-create mew-buffer-tmp))
      (erase-buffer)
      (if (file-exists-p mew-folders-file)
	  (insert-file-contents mew-folders-file)
	(call-process mew-prog-folders nil t nil "-fast" "-recurse")
	)
      (goto-char (point-min))
      (while (not (eobp))
        (setq start (point))
        (forward-line 1)
	(setq folder (mew-buffer-substring start (1- (point))))
	(if (string-match (concat "^" (car folders) "/") folder)
	    (setq folders 
		  (cons folder 
			(cons (concat (car folders) "/"	)
			      (cdr folders))))
	  (setq folders (cons folder folders)))
	)
      folders ;; return value
      )))

;;;
;;; Summary mode
;;;

(defun mew-summary-mode ()
  "Major mode for reading messages.
The keys that are defined for this mode are:

SPC	Read through messages. See mew-summary-show-direction to set 
	'up,'down,'next(current direction) or 'stop. Default is 'down.
DEL	Back scroll this message.
	Unnecessary header fields are hidden over the window. Type DEL
	to see them when message are displayed.
.	Display this message or part. 
	If without MIME analysis, force to analyze this message.

RET	1 line scroll up this message.
ESC RET 1 line scroll down this message.

C-n	Go to the next line.
C-p	Go to the previous line.
n	Display below message or part.
p	Display above message or part.
N	Jump to below * marked message or display below message
	around multipart.
P	Jump to above * marked message or display above message
	around multipart.
j	Jump to a message according to inputed number.

i	Incorporate +inbox asynchronously.
s	Scan this folder asynchronously. 
g	Go to inputed folder.

w	Prepare draft to send a message.
a	Reply to this message or this part.
A	Reply to this message and insert it in draft buffer.
f	Forward this message as MIME format.
F	Forward @ marked messages as MIME format.
E	Edit this message again to send. Or edit this rfc822 part
	typically included MIME-encapsulated error message.
	In a draft folder, it just edits the message. Otherwise, 
	copy the message to draft folder, then edit.
ESC e	Edit an old fashioned error mail in which the original message 
	is encapsulated after \"----- Original message follows -----\".
r	Re-distribute this message with Resent-To:. It is strongly 
	discouraged to use this command since beginners are always 
	confused. Please use \"f\" instead.

v	Toggle summary window only and summary & message windows.
	If you select summary window only, \"D\" never displays the 
	next message. So, you can set D marks quickly.
ESC a	Toggle with/out MIME analysis.
ESC l	Recenter this folder.

o	Mark this message or this part's message with refile
	mark(default is \"o\"). If already marked with \"o\", it
	prints to which folder this message is refiled.
	This can overlay other marks. When it overlays, the cursor stays
	on the message. If it marks newly, displays the next message.
!	Refile this message or part to the previous refile folder.
d	Mark this message with delete mark(default is \"D\").
	This can overlay other marks. When it overlays, the cursor stays
	on the message. If it marks newly, displays the next message.
*	Mark this message with \"*\". Use N or P to jump to * marked message.
	It can overlay \"@\". The cursor stays always.
	See also 'mo', 'md', 'mr', and 'ma'.
@	Mark this message with \"@\" for F, ESC s, and ESC t.
	It can overlay \"*\". The cursor stays always.
u	Cancel mark on this message.
U	Cancel all inputed mark.
x	Process marked messages. To cancel \"*\" mark, use u or U.

mo	Mark all messages whose marks are \"*\" with refile mark(\"o\").
	This is very convenient to refile all messages picked by '?'.
md	Mark all messages whose marks are \"*\" with delete mark(\"D\").
mr	Mark all messages matched with inputed regular expression with \"*\".
ma	Mark all messages with \"*\" which are not marked.
m*	Change mark \"@\" into mark \"*\".
m@	Change mark \"*\" into mark \"@\".
ms	Swap mark \"@\" and \"*\".

C-cC-s	Incremental search forward on the message buffer.
C-cC-r	Incremental search backward on the message buffer.

ESC s	Apply unshar on @ marked messages.
ESC t	Apply uudecode on @ marked messages.

/	Pick messages according to inputed condition, then scan.
?	Pick messages according to inputed condition, then mark *.
V	Goto virtual mode which gives a single view to picked messages
	from multiple folders. Enter virtual folder name, comma-separated
	folders, and pick pattern.

y	Copy this message or save this part as inputed file name.
#	Print this message or this part.
|	Pipe this message.

S	Sort messages in this folder.
O	Pack messages in this folder.

B	Burst messages encapsulated in MIME.

q	Switch to other buffer.
Q	Exit mew.
C-cC-q	Kill this buffer.

C-cC-l	Localize the transformed charset.
C-cd	Manual PGP decryption/vrfy.
C-ca	Add PGP public key in this message to pubring.pgp.
C-cC-d	Manual PEM decryption/vrfy.
C-cC-x	Display xface.

Z	Update aliases and folders list. If you create a new alias or folder
	with native MH, try this.

Range means as follows;
	<num1>-<num2>, <num>:+N, <num>:-N
	first:N, prev:N, next:N, last:N, update

Use 'all' to flush the summary buffer.  'update' means the range
between the last message included in summary mode + 1 and the real last
message on the folder.

Pick condition means as follews;
	<cond1> -and <cond2>, <cond1> -or <cond2>
	-from <pat>, -to <pat>, -search <pat> 
	-date <date>, -before <date>, -after <date>
	--xxx <pat>
	-lbrace or \"(\", -rbrace or \")\"
"
  (interactive)
  (setq major-mode 'mew-summary-mode)
  (setq mode-name "Summary")
  (use-local-map mew-summary-mode-map)
  (setq buffer-read-only t)
  (setq truncate-lines t)
  ;; xxx
  (if (equal (nth (- (length mode-line-format) 2) mode-line-format)
	     '(-3 . "%p"))
      (setq mode-line-format
	    (let ((mlf (copy-sequence mode-line-format))
		  (l (length mode-line-format)))
	      (setcdr (nthcdr (- l 3) mlf)
		      '("[" mew-summary-buffer-left-msgs " more]" "-%-"))
	      mlf)
	    ))
  (if mew-highlight-lines-use
      (if mew-xemacs-p
	  (setq mode-motion-hook 'mode-motion-highlight-line)
	(if (save-excursion 
	      (goto-char (point-min))
	      (not (overlays-at (point))))
	    (mew-summary-highlight-lines-region (point-min) (point-max)))))
  (run-hooks 'mew-summary-mode-hook)
  )

(defun mew-summary-folder-name ()
  (cond 
   ((equal major-mode 'mew-summary-mode)
    (buffer-name))
   ((equal major-mode 'mew-virtual-mode)
    (save-excursion
      (beginning-of-line)
      (if (looking-at ".*\r \\(\\+.*\\) \\(.*\\)\n")
	  (mew-match 1)
	nil
	)))
   (t nil)
   )
  )
    
(defun mew-summary-message-number ()
  (cond 
   ((equal major-mode 'mew-summary-mode)
    (save-excursion
      (beginning-of-line)
      (if (looking-at mew-summary-message-regex)
	  (mew-match 1)
	nil)))
   ((equal major-mode 'mew-virtual-mode)
    (save-excursion
      (beginning-of-line)
      (if (looking-at ".*\r \\(\\+.*\\) \\(.*\\)\n")
	  (mew-match 2)
	nil)))
   (t nil)
   ))


(defun mew-summary-part-number ()
  (save-excursion
    (beginning-of-line)
    (if (looking-at mew-summary-part-regex)
	(mew-match 1)
      nil
      )
    ))

;;
;; process control
;;

;; mew-summary-buffer-process is a key to see if exclusive

(defun mew-summary-exclusive-p ()
  (cond
   ((processp mew-summary-buffer-process)
    (message "%sing now. Try again later."
	     (process-name mew-summary-buffer-process))
    nil) ;; not exclusive
   (t t)) ;; exclusive
  )

;;
;;
;;

(defun mew-summary-goto-folder ()
  (interactive)
  (let* ((folder (mew-input-folder mew-inbox))
	 (dir (mew-expand-file-name folder)))
    (if (null (file-directory-p dir))
	(message "No such folder %s" folder)
      (if (get-buffer folder)
	  (switch-to-buffer folder)
	(mew-summary-folder-create folder))
      (mew-summary-scan)
      )))

(defun mew-summary-folder-create (folder)
  (let ((cache (mew-expand-file-name mew-summary-cache-file folder)))
    (switch-to-buffer (get-buffer-create folder))
    (cond
     ((file-exists-p cache)
      (insert-file-contents cache)
      (setq mew-summary-buffer-folder-cache-time 
	    (nth 5 (file-attributes cache))))
     )
    (mew-summary-mode)
    ))

(defun mew-summary-folder-mark-exec ()
  (if (mew-summary-mark-exist-p (list mew-mark-rmm mew-mark-refile))
      (if (mew-y-or-n-p "Marked messages exist. Process mark before scan? ")
	  (mew-summary-exec)
	)))

(defun mew-file-chase-links (filename)
  "Chase links in FILENAME until a name that is not a link.
Does not examine containing directories for links,
unlike `file-truename'."
  (let (tem (count 100) (newname filename))
    (while (setq tem (file-symlink-p newname))
      (if (= count 0)
	  (error "Apparent cycle of symbolic links for %s" filename))
      ;; In the context of a link, `//' doesn't mean what Emacs thinks.
      (while (string-match "//+" tem)
	(setq tem (concat (substring tem 0 (1+ (match-beginning 0)))
			  (substring tem (match-end 0)))))
      ;; Handle `..' by hand, since it needs to work in the
      ;; target of any directory symlink.
      ;; This code is not quite complete; it does not handle
      ;; embedded .. in some cases such as ./../foo and foo/bar/../../../lose.
      (while (string-match "\\`\\.\\./" tem)
	(setq tem (substring tem 3))
	(setq newname (file-name-as-directory
		       ;; Do the .. by hand.
		       (directory-file-name
			(file-name-directory
			 ;; Chase links in the default dir of the symlink.
			 (mew-file-chase-links
			  (directory-file-name
			   (file-name-directory newname))))))))
      (setq newname (expand-file-name tem (file-name-directory newname)))
      (setq count (1- count)))
    newname))

(defun mew-summary-folder-dir-newp ()
  ;; buffer switched
  (let* ((dir (mew-file-chase-links (mew-expand-file-name (buffer-name))))
	 (tdir (nth 5 (file-attributes dir)))
	 (da (car tdir))
	 (db (car (cdr tdir)))
	 (cache (expand-file-name mew-summary-cache-file dir))
	 (tcache (nth 5 (file-attributes cache)))
	 (fa (car tcache))
	 (fb (car (cdr tcache))))
    (cond
     ((null tdir) nil)
     ((null tcache) t) ;; no cache, do update!
     ((> da fa) t)
     ((= da fa) (if (> db fb) t nil)) ;; nil if same 
     (t nil)
     )
    ))

(defun mew-summary-folder-cache-oldp ()
  ;; buffer switched
  (let* ((tfile (nth 5 (file-attributes 
			(mew-expand-file-name mew-summary-cache-file 
					      (buffer-name)))))
	 (fa (car tfile))
	 (fb (car (cdr tfile)))
	 (tbuf mew-summary-buffer-folder-cache-time)
	 (ba (car tbuf))
	 (bb (car (cdr tbuf))))
    (cond
     ((or (null tfile) (null tbuf)) nil) ;; xxx
     ((> fa ba) t)
     ((= fa ba) (if (> fb bb) t nil)) ;; nil if same 
     (t nil)
     )
    ))

(defun mew-summary-folder-cache-manage (folder)
  (switch-to-buffer folder)
  (let ((cache (mew-expand-file-name mew-summary-cache-file folder)))
    (if mew-summary-cache-use
	(if (and (file-exists-p cache) (mew-summary-folder-cache-oldp))
	    (let ((buffer-read-only nil))
	      (erase-buffer)
	      (insert-file-contents cache)
	      (setq mew-summary-buffer-folder-cache-time 
		    (nth 5 (file-attributes cache)))
	      ))))
  (if (not (equal major-mode 'mew-summary-mode)) (mew-summary-mode))
  )


(defun mew-summary-folder-cache-save ()
  (let ((cache (mew-expand-file-name mew-summary-cache-file (buffer-name))))
    (write-region (point-min) (point-max) cache)
    (setq mew-summary-buffer-folder-cache-time
	  (nth 5 (file-attributes cache)))
    ))

;;
;; inc
;; 

(defvar mew-summary-inc-start nil)

(defun mew-summary-inc ()
  (interactive)
  (mew-summary-folder-cache-manage mew-inbox)
  (if (and mew-summary-cache-use (mew-summary-folder-dir-newp))
      (progn
	(mew-summary-scan-body mew-inbox (mew-input-range mew-inbox))
	;; scan the gap
	(while mew-summary-buffer-process (sit-for 1))
	))
  (mew-summary-inc-body)
  )

;; drop folder is meaningless at present.
(defun mew-summary-inc-body ()
  (set-buffer (get-buffer mew-inbox)) ;; for safety
  (if (not (equal major-mode 'mew-summary-mode)) (mew-summary-mode))
  (mew-window-configure (current-buffer) 'summary)
  (mew-current-set 'message nil)
  (mew-current-set 'part nil)
  (mew-current-set 'cache nil)
  (setq mew-summary-buffer-direction 'down)
  (mew-summary-multipart-delete)
  (if (null (mew-summary-exclusive-p))
      ()
    (condition-case err
	(progn
	  (message "Incing ...")
	  (goto-char (point-max))
	  (setq mew-summary-inc-start (point))
	  (setq mew-summary-buffer-process
		(start-process "Inc" (current-buffer) 
			       mew-prog-inc
			       "-width"
			       (if (< (window-width) 80)
				   "80"
				 (int-to-string (window-width)))
			       ))
	  (set-process-filter mew-summary-buffer-process
			      'mew-summary-inc-filter)
	  (set-process-sentinel mew-summary-buffer-process
				'mew-summary-inc-sentinel)
	  (process-kill-without-query mew-summary-buffer-process)
	  )
      (quit
       (set-process-sentinel mew-summary-buffer-process nil)
       (setq mew-summary-inc-start nil)
       ;; xxx buffer local so set-buffer ???
       (setq mew-summary-buffer-process nil)
       (setq mew-summary-buffer-string nil)
       )
      )
    ))

(defun mew-summary-inc-filter (process string)
  (save-excursion
    (set-buffer (process-buffer process)) ;; for safety
    (setq mew-summary-buffer-string 
	  (concat mew-summary-buffer-string string)) ;; nil can concat
    (let ((buffer-read-only nil)
	  (po (point-max)))
      (if (string-match "^Incorpo.*\n\n" mew-summary-buffer-string)
	  (setq mew-summary-buffer-string
		(substring mew-summary-buffer-string (match-end 0))))
      (while (string-match "^ *[0-9]+\\(.\\).*\n" mew-summary-buffer-string)
	;; leave mew-summary-buffer-string if "inc: no messages"
	(goto-char (point-max))
	(if (not (equal "+" (mew-match 1 mew-summary-buffer-string)))
	    ;; never delete the character after message number
	    (insert (substring mew-summary-buffer-string 0 (match-end 0)))
	  ;; message number
	  (insert (substring mew-summary-buffer-string 0 (match-beginning 1)))
	  ;; delete "+"
	  (insert " ")
	  ;; the rest of line
	  (insert (substring mew-summary-buffer-string
			     (match-end 1)
			     (match-end 0)))
	  )
	(setq mew-summary-buffer-string
	      (substring mew-summary-buffer-string (match-end 0)))
	)
      (mew-summary-highlight-lines-region po (point))
      )))

(defun mew-summary-inc-sentinel (process event)
  (save-excursion
    (set-buffer (process-buffer process)) ;; for safety
    (message "Incing ... done") ;; error message overrides
    (if mew-summary-buffer-string
	(cond 
	 ((string-match "^inc: no mail" mew-summary-buffer-string)
	  (message "No new mail"))
	 ((string-match "^inc: " mew-summary-buffer-string)
	  (message "Inc error"))
	 (t 
	  ;; save cache only when success
	  (if mew-summary-cache-use (mew-summary-folder-cache-save))
	  ;; again to override write buffer message
	  (message "Incing ... done") ;; error message overrides
	  (let ((buffer-read-only nil))
	    (goto-char (point-max))
	    (insert mew-summary-buffer-string)
	    (goto-char mew-summary-inc-start)
	    (keep-lines (concat mew-summary-message-regex
				"\\|"
				mew-summary-part-regex)))
	  )
	 ))
    (set-buffer-modified-p nil)
    (set-process-sentinel mew-summary-buffer-process nil)
    (setq mew-summary-inc-start nil)
    (setq mew-summary-buffer-process nil)
    (setq mew-summary-buffer-string nil)
    (run-hooks 'mew-summary-inc-sentinel-hook)
    ))

;;
;; scan
;;

(defun mew-summary-scan ()
  (interactive)
  (let* ((folder (buffer-name)))
    (mew-summary-folder-cache-manage folder)
    (mew-summary-folder-mark-exec)
    (goto-char (point-max))
    (if (null (mew-member folder mew-folders))
	(setq mew-folders (cons folder mew-folders)))
    (if (or (interactive-p)
	    (and mew-summary-cache-use
		 (mew-summary-folder-dir-newp)))
	(mew-summary-scan-body folder (mew-input-range folder)))
    ))

(defun mew-summary-scan-body (folder range-erase)
  (let ((range (car range-erase)) 
	(erase (car (cdr range-erase))))
    (set-buffer (get-buffer folder)) ;; for safety
    (if (not (equal major-mode 'mew-summary-mode)) (mew-summary-mode))
    (mew-window-configure (current-buffer) 'summary)
    (mew-current-set 'message nil)
    (mew-current-set 'part nil)
    (mew-current-set 'cache nil)
    (setq mew-summary-buffer-direction 'down)
    (if (null (mew-summary-exclusive-p))
	()
      (condition-case err
	  (progn
	    (message "Scanning %s ..." folder)
	    (let ((buffer-read-only nil))
	      (if (equal erase 'erase)
		  (erase-buffer)
		(goto-char (point-max))))
	    (if (null (listp range)) (setq range (list range)))
	    (setq mew-summary-buffer-process
		  (apply (function start-process) 
			 "Scann" 
			 (current-buffer) 
			 mew-prog-scan
			 folder
			 "-noclear" "-noheader"
			 "-width"
			 (if (< (window-width) 80)
			     "80"
			   (int-to-string (window-width)))
			 range
			 ))
	    (set-process-filter mew-summary-buffer-process
				'mew-summary-scan-filter)
	    (set-process-sentinel mew-summary-buffer-process
				  'mew-summary-scan-sentinel)
	    (process-kill-without-query mew-summary-buffer-process)
	    )
	(quit
	 (set-process-sentinel mew-summary-buffer-process nil)
	 (setq mew-summary-buffer-process nil)
	 (setq mew-summary-buffer-string nil)
	 ))
      )
    ))

(defun mew-summary-scan-filter (process string)
  (save-excursion
    (set-buffer (process-buffer process)) ;; for safety
    (setq mew-summary-buffer-string 
	  (concat mew-summary-buffer-string string)) ;; nil can concat
    (let ((buffer-read-only nil)
	  (po (point-max)))
      (while (string-match "^ *[0-9]+\\(.\\).*\n" 
			   mew-summary-buffer-string)
	;; leave mew-summary-buffer-string if "scan: no messages"
	(goto-char (point-max))
	(if (not (equal "+" (mew-match 1 mew-summary-buffer-string)))
	    ;; never delete the character after message number
	    (insert (substring mew-summary-buffer-string 0 (match-end 0)))
	  ;; message number
	  (insert (substring mew-summary-buffer-string 0 (match-beginning 1)))
	  ;; delete "+"
	  (insert " ")
	  ;; the rest of line
	  (insert (substring mew-summary-buffer-string
			     (match-end 1)
			     (match-end 0)))
	  )
	(setq mew-summary-buffer-string
	      (substring mew-summary-buffer-string (match-end 0)))
	)
      (mew-summary-highlight-lines-region po (point))
      )))

(defun mew-summary-scan-sentinel (process event)
  (save-excursion
    (set-buffer (process-buffer process)) ;; for safety
    (message "Scanning %s ... done" (buffer-name))
    (if mew-summary-buffer-string
	(cond
	 ((string-match "^scan: no messages" mew-summary-buffer-string)
	  (message "No messages in %s" (buffer-name)))
	 (t 
	  ;; save only when success
	  (if mew-summary-cache-use (mew-summary-folder-cache-save))
	  ;; again to override write buffer message
	  (message "Scanning %s ... done" (buffer-name)) 
	  (let ((buffer-read-only nil))
	    (goto-char (point-max))
	    (insert mew-summary-buffer-string)
	    (goto-char (point-min))
	    (keep-lines mew-summary-message-regex))
	  )
	 ))
    (set-buffer-modified-p nil)
    (set-process-sentinel mew-summary-buffer-process nil)
    (setq mew-summary-buffer-process nil)
    (setq mew-summary-buffer-string nil)
    (run-hooks 'mew-summary-scan-sentinel-hook)
    ))

;;
;; Pick
;;

(defun mew-pick-define-macro (str1 str2)
  (interactive (list
		(concat "-" (read-string "Pick pattern: -"))
		(read-string "Macro Body: ")))
  (let* ((list (mew-header-split str1 32))
	 (assoc (assoc (car list) mew-pick-macro-alist))
	 (body (mew-header-split str2 32))
	 (info (cons (cdr list) body)))
    (if assoc
	(setcdr assoc info)
      (setq mew-pick-macro-alist
	    (cons (cons (car list) info)
		  mew-pick-macro-alist)))))

(defun mew-pick-macro-expand (patlist)
  (cond ((null patlist) nil)
	((assoc (car patlist) mew-prog-pick-argalist)
	 (let ((first (car patlist))
	       (rest (cdr patlist)))
	   (if (or (mew-member first mew-prog-pick-arglogic-binary)
		   (mew-member first mew-prog-pick-arglogic-unary)
		   (mew-member first '("-lbrace" "-rbrace")))
	       (cons first
		     (mew-pick-macro-expand rest))
	     (cons first
		   (cons (car rest)
			 (mew-pick-macro-expand (cdr rest)))))))
	((mew-member (car patlist) mew-prog-pick-argdate)
	 (cons (car patlist)
	       (cons (nth 1 patlist)
		     (mew-pick-macro-expand (nthcdr 2 patlist)))))
	(t
	 (let ((assoc (assoc (car patlist)
			     mew-pick-macro-alist)))
	   (if (not assoc)
	       (cons (car patlist)
		     (mew-pick-macro-expand (cdr patlist)))
	     (let ((args (nth 1 assoc))
		   (body (copy-sequence (nthcdr 2 assoc)))
		   arg find replc (replclist (cdr patlist)))
	       (while (and args replclist)
		 (setq arg (car args))
		 (setq args (cdr args))
		 (setq replc (car replclist))
		 (setq replclist (cdr replclist))
		 (setq find (member arg body))
		 (while find 
		   (setcar find replc)
		   (setq find
			 (member arg body))))
	       (mew-pick-macro-expand
		(append body replclist))))))))

(defun mew-pick-complete ()
  (interactive)
  (let ((word nil) (comp nil) (all nil)
	(alist (append mew-pick-macro-alist
		       mew-prog-pick-argalist)))
    (if (null (setq word (mew-delete-backward-char)))
	()
      (setq comp (try-completion word alist))
      (setq all (all-completions word alist))
      (cond
       ((eq comp t)
	(insert (car (assoc word alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions)))
       ((stringp comp)
	(insert comp)
	(and (> (length all) 1)
	     (with-output-to-temp-buffer mew-buffer-completions
	       (display-completion-list all))))
       (t (insert word)
	  (ding))
       )
      )
    ))

(defun mew-pick-input-pattern (&optional prompt)
  (mew-pick-macro-expand
   (mew-pick-input-pattern-raw prompt)))

(defun mew-pick-input-pattern-raw (&optional prompt)
  (let ((pat nil))
    (setq pat (read-from-minibuffer 
	       (format "Pattern%s (%s) : "
		       (if prompt prompt "")
		       mew-prog-pick-default-arg)
	       "-" mew-pick-map nil))
    (setq pat (downcase pat))
    (cond
     ((string-match " " pat)
      (mapcar (function mew-summary-search-blace)
	      (mew-header-delete-nullstring-list
	       (mew-header-split pat 32))));; 32 is " "
     (t 
      (if (or (string= pat "") (string= pat "-"))
	  (setq pat mew-prog-pick-default-arg))
      (cond
       ((mew-member pat mew-prog-pick-arglogic-unary)
	(cons pat (mew-pick-input-pattern-raw (format " for %s" pat))))
       ((mew-member pat mew-prog-pick-arglogic-binary)
	(mew-pick-input-logic pat))
       ((mew-member pat mew-prog-pick-argdate)
	(cons pat (mew-pick-input-value (format "Date for %s" pat)
					mew-prog-pick-argdatealist)))
       ((assoc pat mew-pick-macro-alist)
	(let* ((assoc (assoc pat mew-pick-macro-alist))
	       (args (nth 1 assoc)) arg n datep res)
	  (setq res (list pat))
	  (while args
	    (setq arg (car args))
	    (setq args (cdr args))
	    (setq n (and (string-match "[0-9]+" arg) (mew-match 0 arg)))
	    (setq datep (string-match "d" arg))
	    (setq res
		  (nconc
		   (mew-pick-input-value 
		    (cond ((and n datep)
			   (format "Date arg %s for %s" n pat))
			  (n
			   (format "Arg %s for %s" n pat))
			  (datep
			   (format "Date for %s" pat))
			  (t
			   (format "Value for %s" pat)))
		    (and datep mew-prog-pick-argdatealist))
		   res)))
	  (nreverse res)))
       (t (cons pat (mew-pick-input-value (format "Value for %s" pat))))
       )
      )
    )))

;; pattern = key value | logic

(defun mew-pick-input-value (prompt &optional alist)
  (let ((ret nil))
    (cond
     (alist (setq ret (completing-read (format "%s : " prompt)
				       alist
				       nil
				       nil  ;; not require match
				       "")))
     (t (setq ret (read-string (format "%s : " prompt) "")))
     )
    (list ret)
    ))

(defun mew-pick-input-logic (logic)
  (append (mew-pick-input-pattern-raw (format "1 for %s" logic))
	  (cons logic 
		(mew-pick-input-pattern-raw (format "2 for %s" logic))))
  )

(defun mew-summary-search-blace (string)
  (cond
   ((equal string "(") "-lbrace")
   ((equal string ")") "-rbrace")
   (t string)))

(defun mew-summary-search ()
  (interactive)
  (let ((folder (mew-input-folder (buffer-name)))
	(pattern nil)
	(range nil))
    (if (null (file-directory-p (mew-expand-file-name folder)))
	(message "No such folder %s" folder)
      (setq pattern (mew-pick-input-pattern))
      (message "Picking messages in %s ..." folder)
      (setq range (mew-summary-pick folder pattern))
      (message "Picking messages in %s ... done" folder)
      (if (get-buffer folder)
	  (switch-to-buffer folder)
	(mew-summary-folder-create folder))
      (if range (mew-summary-scan-body folder (list range 'erase)))
      )
    ))

(defun mew-summary-search-mark ()
  (interactive)
  (if (equal (point-min) (point-max))
      (message "No messages in this buffer.")
    (let ((folder (buffer-name))
	  (pattern nil)
	  (first nil)
	  (last nil)
	  (range nil))
      (setq pattern (mew-pick-input-pattern))
      (message "Picking messages in %s ..." folder)
      (goto-char (point-min))
      (setq first (mew-summary-message-number))
      (goto-char (point-max))
      (forward-line -1)
      (setq last (mew-summary-message-number))
      (setq range (mew-summary-pick folder pattern (concat first "-" last)))
      (message "Picking messages in %s ... done" folder)
      (if (null range)
	  ()
	(message "Marking messages ... ")
	(goto-char (point-min))
	(while (not (eobp))
	  (if (and (null (mew-summary-marked-p))
		   (mew-member-del (mew-summary-message-number) range))
	      (mew-summary-mark mew-mark-hop))
	  (forward-line))
	(message "Marking messages ... done")
	)
      )))

(defun mew-member-del (a list)
  (let ((pointer (cons nil list)))
    (if (stringp a)
	(let ((b (concat a "/")))
	  (catch 'member
	    (while list
	      (if (or (equal (car list) a)
		      (equal (car list) b))
		  (progn
		    (setcdr pointer (cdr list))
		    (throw 'member t))
		(setq list (cdr list))
		(setq pointer (cdr pointer))
		))))
      (catch 'member
	(while list
	  (if (equal (car list) a)
	      (progn
		(setcdr pointer (cdr list))
		(throw 'member t))
	    (setq list (cdr list))
	    (setq pointer (cdr pointer))
	    )))))
  )

(defun mew-summary-pick (folder pattern &optional range)
  (let ((start nil)
	(point nil)
	(msgs nil))
    (setq range (or range "all"))
    (save-excursion
      (set-buffer (get-buffer-create mew-buffer-tmp))
      (erase-buffer)
      (apply (function call-process)
	     mew-prog-pick nil t nil "-list" folder range pattern)
      (goto-char (point-min))
      (cond 
       ((looking-at "pick: no messages") (message "No such messages") nil)
       ((looking-at "pick: ") (message "Illegal pattern") nil)
       (t 
	(while (not (eobp))
	  (setq start (point))
	  (forward-line 1)
	  (setq msgs (cons (mew-buffer-substring start (1- (point))) msgs))
	  )
	(nreverse msgs))
       )
      )))

;;
;; show
;;

(defun mew-summary-highlight-lines-region (beg end &optional move)
  (if (and mew-highlight-lines-use (not mew-emacs19-p))
      (let ((p (point))
	    (buffer-read-only nil))
	(goto-char beg)
	(if move
	    (let ((ovls (overlays-at (point))))
	      (mapcar 
	       (function 
		(lambda (ovl) (move-overlay ovl end
					    (overlay-end ovl))))
	       ovls)))
	(while (> end (point))
	  (overlay-put
	   (make-overlay (point) (progn (end-of-line) (point)))
	   'mouse-face 'highlight)
	  (forward-line 1))
	(goto-char p))
    )
  )

(defun mew-summary-underline-lines ()
  (if mew-underline-lines-use
      (progn
	(if (mew-overlayp mew-underline-overlay)
	    (mew-move-overlay mew-underline-overlay
			      (save-excursion (beginning-of-line) (point))
			      (save-excursion (end-of-line) (point)))
	  (setq mew-underline-overlay
		(mew-make-overlay 
		 (save-excursion (beginning-of-line) (point))
		 (save-excursion (end-of-line) (point)))))
	(mew-overlay-put mew-underline-overlay 'face 'underline))))

(defun mew-summary-toggle-analysis ()
  (interactive)
  (cond 
   (mew-analysis
    (message "Skip MIME analysis")
    (setq mew-analysis nil))
   (t
    (message "Display message with MIME analysis")
    (setq mew-analysis t))
   ))

(defun mew-summary-show ()
  (interactive)
  (let* ((fld (mew-summary-folder-name))
	 (msg (mew-summary-message-number))
	 (ofld-msg (mew-current-get 'message))
	 (part (mew-summary-part-number))
	 (opart (mew-current-get 'part))
	 (buf (buffer-name))
	 (next nil))
    (if (not (file-exists-p (mew-expand-file-name (concat fld "/" msg))))
	(message "File not exists.")
      (mew-summary-toggle-disp-msg 'on)
      (unwind-protect
	  (progn
	    ;; message buffer
	    (cond 
	     (msg 
	      (mew-window-configure buf 'message)
	      (if (or (null ofld-msg)
		      (null (equal (cons fld msg) ofld-msg))
		      (and (equal (cons fld msg) ofld-msg) 
			   (null (equal opart part))))
		  (mew-summary-display-message fld msg buf)
		(if (mew-message-next-page)
		    (setq next t)))
	      )
	     (part
	      (mew-window-configure buf 'message)
	      (if (or (null opart) (null (equal opart part)))
		  (mew-summary-display-part 
		   (mew-cache-mime-syntax (mew-cache-hit ofld-msg)) part)
		(if (mew-message-next-page)
		    (setq next t)))
	      )
	     (t
	      (message "No message")
	      nil
	      )
	     ))
	(if (or msg part)
	    (progn
	      (pop-to-buffer buf)
	      (mew-summary-recenter)
	      (mew-summary-underline-lines)
	      (if next
		  (cond 
		   ((equal mew-summary-show-direction 'down)
		    (mew-summary-display-down))
		   ((equal mew-summary-show-direction 'up)
		    (mew-summary-display-up))
		   ((equal mew-summary-show-direction 'next)
		    (mew-summary-display-next))
		   (t ())
		   )
		)
	      )
	  )
	))
    )
  (set-buffer-modified-p nil)
  )

(defun mew-summary-toggle-disp-msg (&optional arg)
  (interactive)
  (cond 
   ((equal arg 'on)
    (setq mew-summary-buffer-disp-msg t))
   ((equal arg 'off)
    (setq mew-summary-buffer-disp-msg nil))
   (t
    (setq mew-summary-buffer-disp-msg (not mew-summary-buffer-disp-msg))
    (if mew-summary-buffer-disp-msg
	(mew-summary-display)
      (mew-summary-multipart-delete) ;; xxx 
      (mew-window-configure (current-buffer) 'summary)))))

(defun mew-summary-display (&optional notforce)
  (interactive)
  (let* ((fld (mew-summary-folder-name))
	 (msg (mew-summary-message-number))
	 (fld-msg (cons fld msg))
	 (ofld-msg (mew-current-get 'message))
	 (part (mew-summary-part-number))
	 (opart (mew-current-get 'part))
	 (buf (buffer-name))
	 (cache nil))
    (cond 
     ((and (null msg) (null part))
      (message "No message"))
     ((not (file-exists-p (mew-expand-file-name (concat fld "/" msg))))
      (message "File does not exist"))
     (t
      (unwind-protect
	  (progn
	    (if (or (interactive-p) mew-summary-buffer-disp-msg)
		(progn 
		  (mew-summary-toggle-disp-msg 'on)
		  (mew-window-configure buf 'message))
	      (mew-window-configure (current-buffer) 'summary)
	      (set-buffer (mew-buffer-message)))
	    (cond
	     (msg
	      (if (or (null notforce)
		      (null ofld-msg)
		      (null (equal (cons fld msg) ofld-msg)))
		  (if (or (interactive-p) mew-analysis)
		      (setq cache
			    (mew-summary-display-message
			     fld msg buf 'analysis))
		    (setq cache
			  (mew-summary-display-message fld msg buf))))
	      )
	     (part
	      (if (or (null opart) (null (equal opart part)))
		  (mew-summary-display-part 
		   (mew-cache-mime-syntax (mew-cache-hit ofld-msg)) part)))
	     (t (message "No message"))
	     ))
	(progn
	  (pop-to-buffer buf)
	  (if (null notforce) (mew-summary-recenter))
	  (mew-summary-underline-lines)
	  )
	)
      (set-buffer-modified-p nil)
      (or cache (mew-cache-hit ofld-msg)) ;; return value
      )
     )
    ))

(defun mew-summary-multipart-delete ()
  (let ((buf (current-buffer)))
    (if (null (marker-position mew-current-marker))
	()
      (set-buffer (marker-buffer mew-current-marker))
      (set-marker mew-current-marker2 (point))
      (goto-char (marker-position mew-current-marker))
      (let ((start (point))
	    (buffer-read-only nil))
	(while (looking-at "^\t")
	  (forward-line))
	(delete-region start (point)))
      (set-marker mew-current-marker nil)
      (goto-char (marker-position mew-current-marker2))
      (set-buffer buf)
      )
    ))

(defun mew-summary-multipart-print (fld syntax)
  (save-excursion
    (set-buffer fld)
    ;; summary buffer
    (forward-line 1)
    (set-marker mew-current-marker (point) (current-buffer))
    (let ((buffer-read-only nil))
      (save-excursion (mew-mime-syntax-print syntax)))
    (forward-line -1)
    (set-buffer-modified-p nil) ;; xxx
    ))

(defun mew-summary-mode-line (buf)
  (save-excursion
    (set-buffer buf)
    ;; if not running process in this buffer
    ;; display how many messages are unread
    (if (null mew-summary-buffer-process)
	(setq mew-summary-buffer-left-msgs ;; local variable
	      (int-to-string (1- (count-lines (point) (point-max)))))
      )
    ))

(defun mew-summary-display-message (fld msg buf &optional analysis)
  ;; message buffer
  (let ((hit nil)
	(zmacs-regions nil)) ;; for XEmacs
    (let ((buffer-read-only nil))
      (erase-buffer) ;; for PGP pass phase input
      (set-buffer-modified-p nil))
    (set-marker (mark-marker) nil) ;; kill mark for cite
    (if (or mew-analysis analysis)
	(progn
	  (setq hit (mew-cache-message (cons fld msg)))
	  ;; copy local variable
	  (setq mew-mime-syntax (mew-cache-mime-syntax hit))
	  (mew-current-set 'cache hit)
	  )
      (setq mew-mime-syntax nil)
      (mew-current-set 'cache nil)
      )
    (mew-current-set 'message (cons fld msg))
    (mew-current-set 'part nil)
    ;;
    (mew-summary-multipart-delete)
    (mew-summary-mode-line buf)
    ;; message buffer
    (setq mew-decode-message nil)
    (let ((buffer-read-only nil))
      (if (equal fld mew-draft-folder)
	  (insert-file-contents
	   (mew-expand-file-name (concat fld "/" msg)))
	(if (null (or mew-analysis analysis))
	    (mew-header-insert-from-file
	     (mew-expand-file-name (concat fld "/" msg)))
	  (if (mew-message-multipart-p mew-mime-syntax)
	      (mew-summary-multipart-print buf mew-mime-syntax))
	  (mew-mime-message/rfc822 mew-mime-syntax)
	  )
	)
      )
    (setq mew-decode-message nil)
    (run-hooks 'mew-message-hook)
    (set-buffer-modified-p nil) ;; xxx message buffer 
    hit ;; return value
    ))

(defun mew-summary-display-part (fullpart num &optional non-erase)
;; called in message buffer
;; return t to next part
  (let* ((part    (mew-mime-part-syntax fullpart num))
	 (begin   (mew-part-get-begin  part))
	 (end     (mew-part-get-end    part))
	 (type    (or (mew-part-get-type part) "text/plain"))
	 (params  (mew-part-get-params part))
	 (attr    (mew-content-attr type mew-mime-content-type))
	 (program (mew-content-program attr))
	 (options (mew-content-options attr))
	 (async   (mew-content-async   attr))
	 (zmacs-regions nil))  ;; for XEmacs
    (if (null non-erase)
	(let ((buffer-read-only nil))
	  (erase-buffer)
	  (set-buffer-modified-p nil))) ;; xxx
    (set-marker (mark-marker) nil) ;; kill mark for cite
    (setq mew-message-citation 'noheader)
    ;;
    (if (symbolp program)
	(if (fboundp program)
	    (if (equal program 'mew-mime-message/rfc822)
		(funcall program part) ;; for recursive MIME
	      (funcall program begin end params)))
      (save-excursion
	(set-buffer (mew-current-get 'cache))
	(let ((selective-display nil)
	      (mc-flag nil))
	  (if async
	      (mew-mime-start-process begin end program options)
	    (mew-mime-call-process begin end program options)
	    ))))
    (mew-current-set 'part num) ;; should be after funcall
    (run-hooks 'mew-message-hook)
    (set-buffer-modified-p nil) ;; xxx message buffer
    ))

(defun mew-summary-recenter ()
  (interactive)
  (if (or mew-summary-recenter-p
	  (interactive-p))
      (recenter (/ (- (window-height) 2) 2))
    ))

;;
;;
;;

(defun mew-mime-start-process (begin end program options)
  (if (mew-which program exec-path)
      (if (mew-y-or-n-p (format "Start %s? " program))
	  (let ((pro nil)
		(tmp (make-temp-name mew-temp-file))
		(mc-flag nil)
		(file-coding-system (if mew-mule-p *noconv*)))
	    (write-region begin end tmp)
	    (message "Starting %s ..." program)
	    (setq pro (apply (function start-process)
			     (format "*mew %s*" program)
			     mew-buffer-tmp ;; xxx
			     program
			     (append options (list tmp))))
	    (set-process-sentinel pro 'mew-mime-start-process-sentinel)
	    (message "Sending %s ... done" program)
	    (setq mew-process-file-alist
		  (cons (cons pro tmp) mew-process-file-alist))
	    ))
    (message "Program %s is not found" program))
  t ;; to next part
  )

(defun mew-mime-start-process-sentinel (process event)
  (let ((al (assoc process mew-process-file-alist)))
    (if (cdr al) (delete-file (cdr al)))
    (setq mew-process-file-alist (mew-delq al mew-process-file-alist))
    ))

(defun mew-mime-call-process (begin end program options)
  (if (mew-which program exec-path)
      (if (mew-y-or-n-p (format "Call %s? " program))
	  (let ((mc-flag nil)
		(file-coding-system (if mew-mule-p *noconv*)))
	    (message "Calling %s ..." program)
	    (apply (function call-process-region) 
		   begin end program nil nil nil options)
	    (message "Calling %s ... done" program)
	    ))
    (message "Program %s is not found" program))
  t ;; to next part
  )

(defun mew-mime-prev-page ()
  (interactive)
  (let ((buf (current-buffer)))
    (unwind-protect
	(progn
	  (pop-to-buffer (mew-buffer-message))
	  (scroll-down))
      (pop-to-buffer buf)))
  )

(defun mew-message-header-end (begin end)
  (save-excursion
    (set-buffer (mew-current-get 'cache))
    (narrow-to-region begin end)  
    (goto-char (point-min))
    (re-search-forward "^-*$")
    (save-restriction ;; for no body
      (forward-line 1))
    (prog1 
	(point)
      (widen))
    ))

;; change
(defun mew-mime-text/plain (begin end &optional params)
  (save-excursion
    (set-buffer (mew-buffer-message))
    (let ((buffer-read-only nil))
      (insert-buffer-substring (mew-current-get 'cache) begin end)
      (if mew-eof-string
	  (progn
	    (goto-char (point-max)) ;; necessary?
	    (insert mew-eof-string)))
      )
    (cond (mew-break-pages 
	   (goto-char (point-min))
	   (mew-message-narrow-to-page)
	   ))
    (set-buffer-modified-p nil) ;; xxx
    ))

(fset 'mew-mime-application/octet-strem 'mew-mime-text/plain)

(defun mew-mime-message/rfc822 (part)
  (let* ((hbegin (mew-message-get-begin part))
	 (hend   (mew-message-get-end   part))
	 (cache  (mew-current-get 'cache))
	 (buffer-read-only nil))
    (mew-header-insert-from-cache cache hbegin hend)
    (if (mew-message-singlepart-p part)
	(mew-summary-display-part part nil 'non-erase)) ;; nil is single
    ;; display-part sets citation noheader
    (setq mew-message-citation 'header)
    (set-buffer-modified-p nil) ;; xxx
    ))

;;
;;
;;

(defun mew-summary-display-top ()
  (interactive)
  (goto-char (point-min))
  (mew-summary-display))

(defun mew-summary-display-bottom ()
  (interactive)
  (goto-char (point-max))
  (if (not (bobp)) (forward-line -1))
  (mew-summary-display))


(defun mew-summary-mouse-show (e)
  (interactive "e")
  (mouse-set-point e)
  (beginning-of-line)
  (mew-summary-show))

;;
;; direction
;;

(defun mew-summary-next ()
  (if (equal mew-summary-buffer-direction 'up)
      (mew-summary-up)
    (mew-summary-down))
  )

(defun mew-summary-message-next ()
  (if (equal mew-summary-buffer-direction 'up)
      (mew-summary-message-up)
    (mew-summary-message-down))
  )

(defun mew-summary-down ()
  (forward-line)
  (cond 
   ((re-search-forward 
     (concat mew-summary-message-regex 
	     "[ " (char-to-string mew-mark-hop) "]\\|"
	     mew-summary-part-regex)
     nil t nil)
    (beginning-of-line)
    (setq mew-summary-buffer-direction 'down)
    t)
   (t 
    (mew-summary-multipart-delete)
    (mew-current-set 'message nil)
    (forward-line -1)
    (mew-window-configure (current-buffer) 'summary)
    (message "No more message")
    nil)
   )
  )

(defun mew-summary-message-down ()
  (forward-line)
  (cond 
   ((re-search-forward
     (concat mew-summary-message-regex "[ " (char-to-string mew-mark-hop) "]")
     nil t nil)
    (beginning-of-line)
    (setq mew-summary-buffer-direction 'down)
    t)
   (t 
    (message "No more message")
    nil)
   )
  )

(defun mew-summary-up ()
  (interactive)
  (cond 
   ((re-search-backward
     (concat mew-summary-message-regex 
	     "[ " (char-to-string mew-mark-hop) "]\\|"
	     mew-summary-part-regex)
     nil t nil)
    (setq mew-summary-buffer-direction 'up)
    t)
   (t 
    (mew-summary-multipart-delete)
    (mew-window-configure (current-buffer) 'summary)
    (message "No more message")
    nil)
   ))

(defun mew-summary-message-up (&optional nochange mark)
  (interactive)
  (let ((regex (concat " "
		       (char-to-string mew-mark-hop)
		       (if mark (char-to-string mark)))))
    (setq regex (concat mew-summary-message-regex "[" regex "]"))
    (cond 
     ((re-search-backward regex nil t nil)
      (if (null nochange)
	  (setq mew-summary-buffer-direction 'up))
      t)
     (t 
      (message "No more message")
      nil)
     ))
  )

(defun mew-summary-display-next ()
  (interactive)
  (cond
   ((and (mew-summary-next) mew-summary-buffer-disp-msg)
    (mew-summary-display)
    ))
  )

(defun mew-summary-display-down ()
  (interactive)
  (if (and (mew-summary-down) mew-summary-buffer-disp-msg)
      (mew-summary-display))
  )

(defun mew-summary-down-mark (mark)
  (interactive)
  (forward-line)
  (cond 
   ((re-search-forward (concat mew-summary-message-regex
			       (regexp-quote (char-to-string mark)))
		       nil t nil)
    (beginning-of-line)
    t)
   (t 
    (forward-line -1)
    (message "No more marked message")
    nil)))

(defun mew-summary-display-hop-down ()
  (interactive)
  (let ((part nil))
    (if (looking-at mew-summary-part-regex)
	(setq part t)
      (save-excursion
	(forward-line)
	(if (looking-at mew-summary-part-regex)
	    (setq part t))))
    (if (null part)
	(if (mew-summary-down-mark mew-mark-hop)
	    (if mew-summary-buffer-disp-msg ;; just seed up!
		(mew-summary-display)))
      (mew-summary-message-down)
      (if mew-summary-buffer-disp-msg ;; just seed up!
	  (mew-summary-display)))
    ))

(defun mew-summary-display-up ()
  (interactive)
  (if (and (mew-summary-up) mew-summary-buffer-disp-msg)
      (mew-summary-display))
  )

(defun mew-summary-up-mark (mark)
  (interactive)
  (cond 
   ((re-search-backward (concat mew-summary-message-regex
			       (regexp-quote (char-to-string mark)))
			nil t nil)
    t)
   (t 
    (message "No more marked message")
    nil)
   ))

(defun mew-summary-display-hop-up ()
  (interactive)
  (let ((part nil))
    (if (looking-at mew-summary-part-regex)
	(setq part t))
    (if (null part)
	(if (mew-summary-up-mark mew-mark-hop)
	    (if mew-summary-buffer-disp-msg ;; just seed up!
		(mew-summary-display)))
      (mew-summary-message-up)
      (if mew-summary-buffer-disp-msg ;; just seed up!
	  (mew-summary-display)))
    ))

(defun mew-summary-prev-page ()
  (interactive)
  (let ((fld (mew-summary-folder-name))
	(msg (mew-summary-message-number))
	(part (mew-summary-part-number)))
    (cond
     ((not (or msg part))
      (message "No message or part here"))
     ((and msg (not (equal (cons fld msg) (mew-current-get 'message))))
      (message "Please show the current message first"))
     ;; part exist only when msg = current msg
     ((and part (not (equal part (mew-current-get 'part)))) 
      (message "Please show the current part first"))
     (t 
      (let ((buf (current-buffer)))
	(unwind-protect
	    (progn
	      (pop-to-buffer (mew-buffer-message))
	      (mew-message-prev-page))
	  (pop-to-buffer buf))))
     )))

(defun mew-summary-scroll-up ()
  (interactive)
  (let ((buf (current-buffer))
	(msg (mew-summary-message-number))
	(ofld-msg (mew-current-get 'message))
	(part (mew-summary-part-number))
	(opart (mew-current-get 'part)))
    (cond ((or (and msg (null part) (string= msg (cdr ofld-msg)))
	       (and part (string= part opart)))
	   (unwind-protect
	       (progn
		 (mew-window-configure buf 'message)
		 (mew-message-next-page 1))
	     (pop-to-buffer buf)))
	  ((or msg part)
	   (mew-summary-show))
	  (t
	   (message "No message or part here")))))

(defun mew-summary-scroll-down ()
  (interactive)
  (if (or (mew-summary-message-number) (mew-summary-part-number))
      (let ((buf (current-buffer)))
	(unwind-protect
	    (progn
	      (mew-window-configure buf 'message)
	      (mew-message-prev-page 1)
	      )
	  (pop-to-buffer buf))
	)
    (message "No message or part here")
    ))

;;
;;
;;

(defun mew-summary-rmm (count)
  (interactive "P")
  (if (mew-summary-part-number)
      ;; go onto the current message anyway
      ;; mew-summary-multipart-delete saves excursion...
      (re-search-backward mew-summary-message-regex nil t nil))
  (mew-summary-multipart-delete)
  (if (not (numberp count))
      (mew-summary-rmm-one)
    (while (> count 0)
      (mew-summary-mark mew-mark-rmm)
      (mew-summary-multipart-delete)
      (forward-line)
      (setq count (1- count)))
    (while (< count 0)
      (mew-summary-mark mew-mark-rmm)
      (mew-summary-multipart-delete)
      (forward-line -1)
      (setq count (1+ count)))
    ))

(defun mew-summary-rmm-one ()
  (let ((mark (mew-summary-get-mark))
	(msg (mew-summary-message-number)))
    (cond
     ((null msg) (message "No message"))
     ((equal mark mew-mark-rmm) (message "Already marked as rmm"))
     ((equal mark mew-mark-refile)
      (if (mew-y-or-n-p "Already marked as refile. Delete it? ")
	  (progn
	    (mew-summary-undo-one)
	    (save-excursion (mew-summary-mark mew-mark-rmm))
	    )))
     ((or (equal mark mew-mark-hop) (equal mark mew-mark-mark))
      (mew-summary-undo-one)
      (save-excursion (mew-summary-mark mew-mark-rmm)))
     (t ;; no mark
      (save-excursion (mew-summary-mark mew-mark-rmm))
      (mew-summary-display-next)
      )
     )))

(defun mew-summary-exec-current ()
  (interactive)
  (let (beg end)
    (save-excursion
      (setq beg (progn (beginning-of-line) (point)))
      (setq end (progn (end-of-line) (point))))
    (mew-summary-exec beg end)))

(defun mew-summary-exec-region (r1 r2)
  (interactive "r")
  (mew-summary-exec r1 r2))

(defun mew-summary-exec (&optional r1 r2)
  (interactive)
  (message "Collecting marks ...")
  (let* ((begin (or r1 (point-min)))
	 (end (or r2 (point-max)))
	 (msgs (mew-summary-mark-collect mew-mark-refile begin end))
	 (dels (mew-summary-mark-collect mew-mark-rmm begin end))
	 (src (buffer-name))
	 refal folder folder-list tmp msg)
    (if (not (or msgs dels))
	(message "No marks")
      (mew-window-configure (current-buffer) 'summary)
      (setq mew-summary-buffer-process t)
      (message "Refiling and deleting ...")
      (while msgs
	(setq msg (car msgs))
	(setq msgs (cdr msgs))
	(if (setq folder-list 
		  (car (cdr (assoc msg mew-summary-buffer-refile))))
	    ;; first, remove current folder from folder-list
	    (if (mew-member src folder-list)
		(save-excursion
		  (setq folder-list (delete src folder-list))
		  (goto-char (point-min))
		  (re-search-forward (concat "^ *" msg) nil t)
		  (mew-summary-unmark)
		  ;; if folder-list still have folders, refile to the
		  ;; folders using "-link" option
		  (if (> (length folder-list) 0)
		      (apply (function call-process)
			     mew-prog-refile nil nil nil
			     "-src" src
			     (append (list "-link") folder-list (list msg)))))
	      ;; if multi-refile, I must refile one by one. -- nom
	      (if (nth 1 folder-list)
		  (apply (function call-process)
			 mew-prog-refile nil nil nil
			 "-src" src
			 (append folder-list (list msg)))
		;; else (folder-list has only one folder)
		(setq folder (car folder-list))
		(if (setq tmp (assoc folder refal))
		    (setq refal (cons (append tmp (list msg))
				      (mew-delq tmp refal)))
		  (setq refal (cons (list folder msg) refal))
		  )
		)
	      )
	  )
	)
      (while refal
	(apply (function call-process)
	       mew-prog-refile nil nil nil
	       "-src" src
	       (car (car refal))
	       (cdr (car refal)))
	(setq refal (cdr refal)))
      (setq mew-summary-buffer-refile nil)
      (if dels
	  (progn
	    (apply (function call-process)
		   mew-prog-rmm
		   nil
		   nil
		   nil
		   (buffer-name)
		   dels)))
      (mew-current-set 'message nil) ;;; hack
      (mew-refile-folder-guess-save)
      (let ((next t))
	(if (mew-summary-marked-p)
	    (mew-summary-message-down)) ;; point moves
	(if (mew-summary-message-number)
	    (setq next (mew-summary-message-number)))
	(mew-summary-multipart-delete)
	(mew-summary-mark-process begin end)
	(mew-summary-jump-message next)
	)
      (if mew-summary-cache-use (mew-summary-folder-cache-save))
      (setq mew-summary-buffer-process nil)
      (run-hooks mew-summary-exec-hook)
      (set-buffer-modified-p nil)
      (message "Refiling and deleting ... done")
      )))

(defun mew-summary-reply ()
  (interactive)
  (cond 
   ((or (mew-summary-message-number)
	(mew-summary-part-number))
    (mew-current-set 'window (current-window-configuration))
    (let ((buf (buffer-name))
	  (file (mew-draft-get-new))
	  from to cc subject in-reply-to references
	  cbuf cache
	  num folder)
      (unwind-protect
	  (progn
	    (if (get-buffer (mew-buffer-message))
		(delete-windows-on (mew-buffer-message)))
	    (if (< (window-height) 25) (delete-other-windows))
	    (split-window-vertically)
	    (switch-to-buffer-other-window (find-file-noselect file))
	    (mew-draft-rename file)
	    (setq cbuf (current-buffer)) ;; draft
	    (pop-to-buffer buf) 
	    ;; summary or virtual
	    (setq num (mew-summary-message-number))
	    (setq folder (mew-summary-folder-name))
	    ;;
	    (setq cache (mew-summary-display)) ;; force to display
	    (if cache
		(set-buffer cache)
	      (set-buffer (mew-buffer-message)))
	    ;; now cache buffer
	    (setq from (mew-header-address-collect '("From:")))
	    (cond 
	     ;; This message was sent by me.
	     ;; So, maintain To: and Cc:.
	     ((and from (string-match (concat "^" mew-mail-address "$")
				      (car from)))
	      (setq to (mew-header-address-collect '("To:" "Apparently-To:")))
 	      (if (and (null (cdr to))
		       (car to) ;; not null
		       (string-match ":;" (car to)))
		  (setq to (or (mew-header-address-collect '("Reply-To:"))
			       from)))
	      (setq cc (mew-header-address-collect '("Cc:"))))
	     ;;
	     (t
	      (cond 
	       ((mew-field-get-value "Reply-To:")
		(setq to (mew-header-address-collect mew-replyto-to-list))
		(setq cc (mew-header-address-collect mew-replyto-cc-list)))
	       (t
		(setq to (mew-header-address-collect mew-noreplyto-to-list))
		(setq cc (mew-header-address-collect mew-noreplyto-cc-list)))
	       )
	      )
	     )
	    (setq subject (mew-field-get-value "Subject:"))
	    (if (and subject (not (string-match "^Re:" subject)))
		(setq subject (concat "Re: " subject)))
	    (setq in-reply-to (mew-field-get-value "Date:"))
	    (setq references (mew-field-get-value "Message-ID:"))
	    ;;
	    (pop-to-buffer cbuf) ;; draft
	    (mew-draft-header subject nil to cc in-reply-to references)
	    (mew-draft-mode) ;; for hilight
	    )
	(save-buffer))) ;; to make sure not to use this draft again
    (message "Draft is prepared")
    t ) ;; preparation is succeeded.
   (t 
    (message "No message")
    nil) ;; preparation is failed.
   )
  )

(defun mew-summary-reply-with-citation ()
  "Reply to current message and insert it in draft-buffer."
  (interactive)
  (if (mew-summary-reply)
      (mew-draft-cite)
    ))

(defun mew-summary-reedit ()
;;
;; reedit a message in a folder or a message encapsulated in MIME
;;
  (interactive)
  (let* ((msg (mew-summary-message-number)) ;; must get msg here
	(part (mew-summary-part-number))
	(fld (mew-summary-folder-name))
	(rename nil)
	(beg)
	(cache (if part (mew-current-get 'cache)))
	(syntax (if part (mew-mime-part-syntax
			  (mew-cache-mime-syntax cache) part)))
	(type (if syntax (mew-part-get-type syntax))))
    (cond
     ((and (null msg) (null part))
      (message "No message"))
     ((and part (not (equal "message/rfc822" (downcase type))))
      (message "Can't reedit %s" type))
     (t
      (mew-current-set 'window (current-window-configuration))
      (mew-window-configure (current-buffer) 'summary)
      (unwind-protect
	  (progn
	    (cond
	     ((or part (not (equal fld mew-draft-folder)))
	      (setq rename (mew-draft-get-new)))
	     (t 
	      (setq rename msg))
	     )
	    ;; prepare draft file
	    (switch-to-buffer 
	     (find-file-noselect 
	      (mew-expand-file-name rename mew-draft-folder)))
	    (mew-draft-mode) ;; for charset
	    (cond
	     (part
	      (insert-buffer-substring
	       cache
	       (mew-part-get-begin syntax)
	       (mew-part-get-end (mew-message-get-part syntax))))
	     ((not (equal fld mew-draft-folder))
	      (insert-file-contents (mew-expand-file-name msg fld))
	      )
	     ;; if fld equal mew-draft-folder, message already exists.
	     )
	    (mew-draft-rename rename)
	    (mew-field-delete-lines mew-field-delete)
	    ;; delimiter setup
	    ;; Both "^$" and "----" are OK.
	    (goto-char (point-min))
	    (re-search-forward "^-*$" nil t)
	    (beginning-of-line)
	    (setq beg (point))
	    (forward-line)
	    (delete-region beg (point))
	    ;; Dcc or Fcc:
	    (if (and mew-fcc (not (mew-field-get-value "Fcc:")))
		(mew-field-insert-here "Fcc:" mew-fcc))
	    (if (and mew-dcc (not (mew-field-get-value "Dcc:"))) 
		(mew-field-insert-here "Dcc:" mew-dcc))
 	    (if (and mew-from (not (mew-field-get-value "From:")))
 		(mew-field-insert-here "From:" mew-from))
 	    (if (and mew-reply-to (not (mew-field-get-value "Reply-To:")))
 		(mew-field-insert-here "Reply-To:" mew-reply-to))
	    (if (and mew-x-mailer (not (mew-field-get-value "X-Mailer:")))
		(mew-field-insert-here "X-Mailer:" mew-x-mailer))
	    (setq mew-draft-buffer-header (point-marker))
	    (insert "----\n")
	    )
	(save-buffer))
      (message "Draft is prepared"))
     )
    ))

(defun mew-summary-edit-again ()
  (interactive)
  (let ((msg (mew-summary-message-number)) ;; must get msg here
	(fld (mew-summary-folder-name))
	(rename nil)
	beg)
    (cond 
     ((eobp) (message "No message"))
     ((null msg)
      (message "Please use this command on a message"))
     (t
      (mew-current-set 'window (current-window-configuration))
      (mew-window-configure (current-buffer) 'summary)
      (unwind-protect
	  (progn
	    (cond
	     ((not (equal fld mew-draft-folder))
	      (setq rename (mew-draft-get-new)))
	     (t 
	      (setq rename msg))
	     )
	    ;; prepare draft file
	    (switch-to-buffer 
	     (find-file-noselect 
	      (mew-expand-file-name rename mew-draft-folder)))
	    (mew-draft-mode) ;; charset
	    (cond
	     ((not (equal fld mew-draft-folder))
	      (insert-file-contents (mew-expand-file-name msg fld))
	      )
	     ;; if fld equal mew-draft-folder, message already exists.
	     )
	    (mew-draft-rename rename)
	    (goto-char (point-min))
	    (if (re-search-forward mew-summary-edit-again-regex nil t)
		(progn
		  (forward-line 1)
		  (delete-region (point-min) (point))
		  ))
	    (mew-field-delete-lines mew-field-delete)
	    ;; delimiter setup
	    ;; Both "^$" and "----" are OK.
	    (goto-char (point-min))
	    (re-search-forward "^-*$" nil t)
	    (beginning-of-line)
	    (setq beg (point))
	    (forward-line)
	    (delete-region beg (point))
	    ;; Dcc or Fcc:
	    (if (and mew-fcc (not (mew-field-get-value "Fcc:")))
		(mew-field-insert-here "Fcc:" mew-fcc))
	    (if (and mew-dcc (not (mew-field-get-value "Dcc:"))) 
		(mew-field-insert-here "Dcc:" mew-dcc))
 	    (if (and mew-from (not (mew-field-get-value "From:")))
 		(mew-field-insert-here "From:" mew-from))
 	    (if (and mew-reply-to (not (mew-field-get-value "Reply-To:")))
 		(mew-field-insert-here "Reply-To:" mew-reply-to))
	    (if (and mew-x-mailer (not (mew-field-get-value "X-Mailer:")))
		(mew-field-insert-here "X-Mailer:" mew-x-mailer))
	    (setq mew-draft-buffer-header (point-marker))
	    (insert "----\n")
	    )
	(save-buffer))
      (message "Draft is prepared"))
     )
    ))

(defun mew-summary-sort ()    
  (interactive)
  (let ((folder (buffer-name)))
    (if (null (mew-summary-exclusive-p))
	()
      (if (mew-summary-mark-exist-p (list mew-mark-rmm mew-mark-refile))
	  (if (mew-y-or-n-p
	       "Marked messages exist. Process mark before scan? ")
	      (mew-summary-exec)))
      (if (null (mew-y-or-n-p (format "Sort %s ? " folder)))
	  ()
	(setq mew-summary-buffer-process t)
	(message "Sorting %s ... " folder)
	(call-process mew-prog-sortm nil nil nil folder)
	(message "Sorting %s ... done" folder)
	(setq mew-summary-buffer-process nil)
	(let ((buffer-read-only nil)) (erase-buffer))  ;; for update
	(mew-summary-scan-body folder (mew-input-range folder))
	)
      )
    ))

(defun mew-summary-pack ()    
  (interactive)
  (let ((folder (buffer-name)))
    (if (null (mew-summary-exclusive-p))
	()
      (if (mew-summary-mark-exist-p (list mew-mark-rmm mew-mark-refile))
	  (if (mew-y-or-n-p
	       "Marked messages exist. Process mark before scan? ")
	      (mew-summary-exec)))
      (if (null (mew-y-or-n-p (format "Pack %s? " folder)))
	  ()
	(setq mew-summary-buffer-process t)
	(message "Packing %s ... " folder)
	(call-process mew-prog-pack nil nil nil folder "-pack")
	(message "Packing %s ... done" folder)
	(setq mew-summary-buffer-process nil)
	(let ((buffer-read-only nil)) (erase-buffer)) ;; for update
	(mew-summary-scan-body folder (mew-input-range folder))
	)
      )
    ))

(defun mew-summary-forward ()
  (interactive)
  ;; need to make this elegant
  (cond 
   ((or (mew-summary-message-number)
	(mew-summary-part-number))
    (mew-current-set 'window (current-window-configuration))
    (cond 
     ((mew-summary-part-number)
      (mew-summary-message-up t) ;; no direction change
      (mew-summary-display))
     )
    (let* ((buf (buffer-name))
	   (fld (mew-summary-folder-name))
	   (msg (mew-summary-message-number))
	   (cbuf nil)
	   (subject "")
	   (file (mew-draft-get-new))
	   (dirname (file-name-nondirectory file)))
      (unwind-protect
	  (progn
	    (delete-other-windows)
	    (split-window-vertically)
	    (switch-to-buffer-other-window (find-file-noselect file))
	    (mew-draft-rename file)
	    (setq cbuf (current-buffer)) ;; draft
	    (pop-to-buffer buf)
	    ;;
	    (set-buffer (mew-summary-display)) ;; force to display
	    ;; now cache buffer
	    (setq subject (concat "Forward: "
				  (mew-field-get-value "Subject:")))
	    (pop-to-buffer cbuf) ;;; draft
	    ;;
	    (mew-draft-header subject 'nl)
	    (mew-draft-mode)
	    (mew-draft-multi-copy file (list (cons fld msg)))
	    (setq mew-multi-syntax (mew-multi-multi-syntax dirname 1))
	    (save-excursion
	      (mew-draft-prepare-multipart)
	      ;; status saved
	      ))
	(save-buffer))) ;; to make sure not to use this draft again
    (message "Draft is prepared"))
   (t (message "No message")))
  )

(defun mew-summary-multi-forward ()
  (interactive)
  (let ((fld-msg (mew-summary-mark-collect2 mew-mark-mark)))
    (cond
     (fld-msg
      (mew-current-set 'window (current-window-configuration))
      (let* ((fld (buffer-name))
	     (file (mew-draft-get-new))
	     (dirname (file-name-nondirectory file)))
	(unwind-protect
	    (progn
	      (delete-other-windows)
	      (split-window-vertically)
	      (switch-to-buffer-other-window (find-file-noselect file))
	      (mew-draft-rename file)
	      (mew-draft-header nil 'nl)
	      (mew-draft-mode)
	      (mew-draft-multi-copy file fld-msg)
	      (setq mew-multi-syntax
		    (mew-multi-multi-syntax dirname (length fld-msg)))
	      (save-excursion
		(mew-draft-prepare-multipart)
		))
	  (save-buffer)) ;; to make sure not to use this draft again
	(message "Draft is prepared"))
      )
     (t (message "No marks"))
     )
    ))

(defun mew-summary-redist ()
  (interactive)
  (cond 
   ((mew-summary-message-number)
    (mew-current-set 'window (current-window-configuration))
    (let* ((file (mew-expand-file-name (mew-summary-message-number)
				       (mew-summary-folder-name)))
	   (to (mew-input-address "Resent-To:"))
	   (cc (if mew-ask-cc (mew-input-address "Resent-Cc:")))
	   (draft (mew-expand-file-name (mew-draft-get-new) mew-draft-folder)))
      (save-excursion
	(set-buffer (find-file-noselect draft))
	(if mew-redist-needs-full-contents
	  (progn
	    (insert-file-contents file)
	    (mew-field-resent-lines mew-field-resent)
	    (mew-field-delete-lines mew-field-delete)
	    (goto-char (point-min))))
	(if (or (null to) (equal to ""))
	    ()
	  (mew-field-insert-here "Resent-To:" to))
	(if (or (null cc) (equal cc ""))
	    ()
	  (mew-field-insert-here "Resent-Cc:" cc))
	(save-buffer) 
	(message "Redistributing ... ")
	(if mew-redist-needs-full-contents
	  (call-process "/bin/sh" nil nil nil "-c"
		         (format "mhdist=1 %s %s"
				  mew-prog-send
				  (buffer-file-name)))
	  (call-process "/bin/sh" nil nil nil "-c"
		         (format "mhdist=1 mhaltmsg=%s %s %s"
				  file
				  mew-prog-send
				  (buffer-file-name))))
	(message "Redistributing ... done")
	(kill-buffer (current-buffer))
	))
    )
   (t (message "No message"))
   )
  )

(defun mew-draft-multi-copy (draft fld-msg-list)
  (let* ((mimefolder (mew-draft-to-mime draft))
	 (mimedir (mew-expand-file-name mimefolder)))
    (if (null (file-directory-p mimedir))
	(mew-make-directory mimedir)
      (if (null (mew-directory-empty-p mimedir))
	  (if (mew-y-or-n-p (format "Mime folder %s is not empty. Delete it? "
				mimefolder))
	      (progn
		(call-process "rm" nil nil nil "-rf" mimedir)
		(mew-make-directory mimedir))
	    )))
    (while fld-msg-list
      (if mew-os2-p
	  (call-process "cp" nil nil nil
			(mew-expand-file-name
			 (cdr (car fld-msg-list)) ;; message
			 (car (car fld-msg-list))) ;; folder
			(mew-draft-get-new mimefolder))
	(call-process "ln" nil nil nil "-s"
		      (mew-expand-file-name
		       (cdr (car fld-msg-list)) ;; message
		       (car (car fld-msg-list))) ;; folder
		      (mew-draft-get-new mimefolder))
	)
      (setq fld-msg-list (cdr fld-msg-list)))
    ))

(defun mew-summary-send (&optional unconfig to cc subject)
  (interactive)
  (let ((file (mew-draft-get-new)))
    (if (null unconfig)
	(mew-current-set 'window (current-window-configuration)))
    (mew-window-configure (current-buffer) 'summary)
    (unwind-protect
	(progn
	  (switch-to-buffer (find-file-noselect file))
	  (mew-draft-rename file)
	  (mew-draft-header subject nil to nil)
	  (mew-draft-mode) ;; hack hack mew-draft-buffer-header
	  )
      (save-buffer))) ;; to make sure not to use this draft again
  (message "Draft is prepared")
  )

(defun mew-summary-save ()
  (interactive)
  (let ((msg (mew-summary-message-number))
	(part (mew-summary-part-number))
	(action "Save")
	(append-p nil)
	(cbuf (mew-summary-folder-name))
	file)
    (if (not (or msg part))
	(message "No message or part here")
      (setq file (mew-input-filename))
      (if (file-exists-p file)
	  (if (null mew-file-append-p)
	      (setq action "Overwrite")
	    (setq action "Append")
	    (setq append-p t)))
      (cond
       (msg
	(if (mew-y-or-n-p (format "%s message %s to %s? " action msg file))
	    (save-excursion
	      (set-buffer (get-buffer-create mew-buffer-tmp))
	      (widen)
	      (erase-buffer)
	      (let ((mc-flag nil)
		    (file-coding-system-for-read (if mew-mule-p *noconv*))
		    (file-coding-system (if mew-mule-p *noconv*)))
		(insert-file-contents (mew-expand-file-name msg cbuf))
		(write-region (point-min) (point-max) file append-p))
	      (message "Wrote to %s" file)
	      )
	  )
	)
       (part
	(if (mew-y-or-n-p (format "%s part %s to %s? " action part file))
	    (save-excursion
	      (set-buffer (mew-current-get 'cache))
	      (let* ((case-fold-search t)
		     (syntax (mew-mime-part-syntax
			      (mew-cache-mime-syntax
			       (mew-current-get 'cache)) part))
		     (begin (mew-part-get-begin syntax))
		     (end (mew-part-get-end syntax))
		     (type (mew-part-get-type syntax)))
		(if (string-match "message/rfc822" type)
		    (setq end (mew-part-get-end
			       (mew-message-get-part syntax))))
		(if (mew-member-match type mew-mime-content-type-binary)
		    (let ((mc-flag nil)
			  (file-coding-system (if mew-mule-p *noconv*)))
		      (write-region begin end file append-p))
		  (let ((file-coding-system
			(if mew-mule-p mew-mule-charset-local)))
		    (write-region begin end file append-p)
		    ))
		)
	      (message "Wrote to %s" file)
	      ))
	)
       )
      )
    ))

(defun mew-summary-burst ()
  (interactive)
  (let (target dir)
    (save-excursion
      (cond 
       ((mew-summary-part-number)
	(mew-summary-message-up t)
	(mew-summary-display)
	)
       )
      (if (null (mew-summary-message-number))
	  (message "No message")
	(setq target (mew-expand-file-name 
		      (mew-summary-message-number)
		      (mew-summary-folder-name)))
	(setq dir (mew-input-directory-name))
	(message "Bursting ... ")
	(set-buffer mew-buffer-tmp)
	(widen)
	(erase-buffer)
	(let ((selective-display nil)
	      (mc-flag nil)
	      (file-coding-system-for-read (if mew-mule-p *noconv*))
	      (file-coding-system (if mew-mule-p *noconv*))
	      (syntax) (n 1))
	  (insert-file-contents target)
	  (setq syntax (mew-mime-syntax))
	  (if (mew-message-singlepart-p syntax)
	      ()
	    (setq syntax (mew-multi-get-parts (mew-part-get-part syntax)))
	    (while syntax
	      (if (null (mew-part-message-p (car syntax)))
		  ()
		(write-region (mew-part-get-begin (car syntax))
			      (mew-part-get-end (mew-part-get-part (car syntax)))
			      ;; xxx
			      (expand-file-name (int-to-string n) dir))
		(setq n (1+ n)))
	      (setq syntax (cdr syntax))
	      ))
	  )
	  (message "Bursting ... done")
	  )
      )
    ))
      
(defun mew-summary-print ()
  (interactive)
  (mew-summary-display)
  (save-excursion
    (set-buffer (mew-buffer-message))
    (save-restriction
      (widen)
      (if (mew-y-or-n-p "Print this message? ")
	  (funcall mew-print-function))
      )
    ))

(defun mew-summary-unshar ()
  (interactive)
  (let ((fld-msg (nreverse (mew-summary-mark-collect2 mew-mark-mark)))
	(files nil)
	(dir nil))
    (if (null fld-msg)
	(message "No marked messages")
      (if (null (mew-y-or-n-p (format "Execute unshar for these messages? "
				      mew-prog-unshar)))
	  ()
	(setq dir (mew-input-directory-name))
	(while fld-msg
	  (setq files (cons (mew-expand-file-name
			     (cdr (car fld-msg)) ;; message
			     (car (car fld-msg))) ;; folder
			    files))
	  (setq fld-msg (cdr fld-msg))
	  )
	(message "Executing %s ... " mew-prog-unshar)
	(apply 'call-process mew-prog-unshar nil nil nil "-d" dir files)
	(message "Executing %s ... done" mew-prog-unshar)
	))
    ))

(defun mew-summary-uudecode ()
  (interactive)
  (let ((fld-msg (nreverse (mew-summary-mark-collect2 mew-mark-mark)))
	(files nil)
	(dir nil)
	(tarfile nil)
	(case-fold-search nil))
    (if (null fld-msg)
	(message "No marked messages")
      (if (null (mew-y-or-n-p (format "Execute %s for these messages? "
				   mew-prog-uumerge)))
	  ()
	(setq dir (mew-input-directory-name))
	(while fld-msg
	  (setq files (cons (mew-expand-file-name
			     (cdr (car fld-msg)) ;; message
			     (car (car fld-msg))) ;; folder
			    files))
	  (setq fld-msg (cdr fld-msg))
	  )
	(save-excursion
	  (set-buffer mew-buffer-tmp)
	  (let ((buffer-read-only nil)) (erase-buffer))
	  (message "Executing %s ..." mew-prog-uumerge)
	  (apply 'call-process mew-prog-uumerge
		 nil t nil "-d" dir files)
	  (message "Executing %s ... done" mew-prog-uumerge)
	  (goto-char (point-min))
	  (if (looking-at "^uumerge:")
	      (message "Failed to executing %s" mew-prog-uumerge)
	    (forward-line)
	    (setq tarfile (mew-buffer-substring (point-min) (1- (point))))
	    (setq tarfile (mew-summary-prog-exec
			   mew-prog-compress "-df" "Z" dir tarfile))
	    (setq tarfile (mew-summary-prog-exec
			   mew-prog-gzip "-df" "gz" dir tarfile))
	    (if (string-match "^\\(.*\\)\\.tar$" tarfile)
		(if (null (mew-y-or-n-p (format "Execute %s for %s? " 
					    mew-prog-tar tarfile)))
		    ()
		  (message (format "Executing %s for %s ... "
				    mew-prog-tar tarfile))
		  (call-process "sh" nil nil nil "-c"
				(concat "cd " dir " ; cat " 
					tarfile " | tar -xf -"))
		  (message (format "Executing %s for %s ... done"
				    mew-prog-tar tarfile))
		  ))
	    ))
	))
    ))

(defun mew-summary-prog-exec (prog opts suf dir tarfile)
  (if (string-match (format "^\\(.*\\)\\.%s$" suf) tarfile)
      (if (null (mew-y-or-n-p (format "Execute %s for %s? " prog tarfile)))
	  tarfile
	(message (format "Executing %s for %s ... " prog tarfile))
	(call-process prog nil nil nil opts (expand-file-name tarfile dir))
	(message (format "Executing %s for %s ... done" prog tarfile))
	(mew-match 1 tarfile))
    tarfile
    ))

(defun mew-summary-jump-message (&optional msg)
;; if msg is t, jump last
  (interactive)
  (let ((here (point)))
    (if (null msg) (setq msg (read-string "Message No. : " "")))
    (cond 
     ((equal msg "") ())
     ((equal msg t)
      (goto-char (point-max))) ;; (forward-line -1)
     (t 
      (goto-char (point-min))
      (if (re-search-forward (format "^[ ]*%s[^0-9]+" msg) nil t)
	  (beginning-of-line)
	(goto-char here))))
    ))

(defun mew-summary-jump-to-draft-buffer ()
  (interactive)
  (let ((num)
	(x (buffer-list)))
    (while x
      (if (string-match "^\\+drafts/" (buffer-name (car x)))
	  (setq num
		(cons (substring (buffer-name (car x)) (match-end 0)) num)))
      (setq x (cdr x)))
    (cond ((null num)
	   (error "No drafts buffer exist!"))
	  (t
	   (switch-to-buffer
	    (concat "+drafts/"
		     (car (sort num
				(function (lambda(a b)
					    (not (string< a b))))))))))))

;;
;; mark
;;

(defun mew-summary-marked-p ()
  (save-excursion
      (beginning-of-line)
      (cond 
       ((looking-at (concat mew-summary-message-regex " \\|^$")) nil) ;; zzz
       (t t))))

(defun mew-summary-mark-exist-p (marklist)
  (let ((regex 
	 (concat
	  mew-summary-message-regex
	  "["
	  (mapconcat (function char-to-string) marklist "\\|")
	  "]"
	   )))
    (save-excursion
      (goto-char (point-min))
      (re-search-forward regex nil t)
      )
    ))

(defun mew-summary-mark (mark)
  (let ((buffer-read-only nil))
    (cond 
     ((null (mew-summary-marked-p))
      (beginning-of-line)
      (re-search-forward mew-summary-message-regex)
      (delete-char 1)
      (insert (char-to-string mark)))
     (t (message "Already marked")))
    ))

(defun mew-summary-unmark ()
  (save-excursion
    (let ((buffer-read-only nil))
      (beginning-of-line)
      (re-search-forward mew-summary-message-regex)
      (delete-char 1)
      (insert " ")
      )))

(defun mew-summary-mark-letitbe ()
  () ;; do nothing
  )

(defun mew-summary-mark-process (begin end)
  (let ((buffer-read-only nil)
	(m (make-marker)))
    (set-marker m end)
    (goto-char begin)
    (while (re-search-forward 
	    (concat mew-summary-message-regex "\\([^ +0-9]\\)") m t)
      (funcall (cdr (assoc (string-to-char (mew-match 2)) mew-mark-switch)))
      )
    (set-buffer-modified-p nil)
    ))

(defun mew-summary-mark-weak (mark)
  (let ((cmark (mew-summary-get-mark))
	(msg (mew-summary-message-number)))
    (cond 
     ((null msg) (message "No message"))
     ((or (equal cmark mew-mark-refile) (equal cmark mew-mark-rmm))
      (message "Already strongly marked as %s" (char-to-string cmark)))
     ((equal cmark mark)
      (message "Already marked as %s" (char-to-string cmark)))
     (t ;; no mark
      (save-excursion
	(if cmark (mew-summary-undo-one))
	(mew-summary-mark mark)))
     ))
  (set-buffer-modified-p nil)
  )

(defun mew-summary-mark-mark ()
  (interactive)
  (mew-summary-mark-weak mew-mark-mark)
  )

(defun mew-summary-mark-hop ()
  (interactive)
  (mew-summary-mark-weak mew-mark-hop)
  )

(defun mew-summary-mark-collect (mark &optional begin end)
  (save-excursion
    (let ((re (format (concat mew-summary-message-regex "%s") 
		      (regexp-quote (char-to-string mark))))
	  (msglist nil))
      (goto-char (if begin begin (point-min)))
      (while (re-search-forward re end t)
	(setq msglist (cons (mew-summary-message-number) msglist)))
      (nreverse msglist))))

(defun mew-summary-mark-collect2 (mark)
  (save-excursion
    (let ((re (format (concat mew-summary-message-regex "%s") 
		      (regexp-quote (char-to-string mark))))
	  (msglist nil))
      (goto-char (point-min))
      (while (re-search-forward re nil t)
	(setq msglist (cons 
		       (cons
			(mew-summary-folder-name)
			(mew-summary-message-number))
		       msglist)))
      (nreverse msglist) ;; return
      )))

(defun mew-summary-get-mark ()
  (let (mark)
    (save-excursion
      (beginning-of-line)
      (if (not (looking-at (concat mew-summary-message-regex "\\(.\\)")))
	  nil
	(setq mark (mew-match 2)) ;; regex includes \( \)
	(cond 
	 ((string-equal " " mark) nil)
	 (t (string-to-char mark))
	 )
	)
      )))
	
;;
;; From: Hiroshi ENOMOTO <enomoto@apt.fxis.fujixerox.co.jp>
;;

(defun mew-summary-mark-all ()
  (interactive)
  (let ((regex (concat mew-summary-message-regex " ")))
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward regex nil t)
	(mew-summary-mark mew-mark-hop))
      ))
  )

(defun mew-summary-mark-undo-all ()
  (interactive)
  (let ((regex (concat mew-summary-message-regex "[\\*oD]")))
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward regex nil t)
	(mew-summary-undo-one))
      ))
  )

(defun mew-summary-mark-regexp (regex)
  (interactive "sRegexp: ")
  (let ()
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward regex nil t)
	(if (not (mew-summary-marked-p))
	    (mew-summary-mark mew-mark-hop))
	)
      )
    )
  )

(defun mew-summary-refile-mark (folder &optional msg notmark)
  (let (msg tmp)
    (setq msg (or msg (mew-summary-message-number)))
    (setq mew-summary-buffer-refile
	  (if (setq tmp (assoc msg mew-summary-buffer-refile))
	      (if (mew-member folder (car (cdr tmp)))
		  mew-summary-buffer-refile
		  (cons 
		   (list msg (append (car (cdr tmp)) (list folder)))
		   (mew-alist-del mew-summary-buffer-refile msg)))
	    ; make new entry in mew-summary-buffer-refile
	    (cons (list msg (list folder))
		  mew-summary-buffer-refile)))
    (if notmark 
	()
      (mew-summary-mark mew-mark-refile))
    )
  )

(defun mew-decide-folder ()
  (interactive)
  (let (learn-info msg-buffer (tofolder nil) todir)
    (setq msg-buffer (or (mew-summary-display t)
			 (mew-buffer-message)))
    (save-excursion
      (set-buffer msg-buffer)
      (setq learn-info (mew-refile-folder-guess)))
    (setq tofolder (mew-input-folder (car learn-info)))
    (setq todir (mew-folder-to-dir tofolder))
    (if (mew-member todir mew-folder-list)
	tofolder
      (if (or (file-exists-p (concat mew-path "/" todir))
	      (mew-y-or-n-p 
	       (format "No folder %s exists. Create it? " tofolder)))
	  (progn
	    (setq 
	     mew-refile-alist (cons (mew-refile-pair todir) mew-refile-alist)
	     mew-folder-alist (cons (mew-folder-pair todir) mew-folder-alist)
	     mew-folder-list (cons todir  mew-folder-list))
	    (if (file-readable-p mew-folders-file)
		(save-excursion
		  (set-buffer (get-buffer-create mew-buffer-tmp))
		  (erase-buffer)
		  (insert todir)
		  (insert "\n")
		  (append-to-file (point-min) (point-max) mew-folders-file)))
	    tofolder ;; return value
	    )
	nil) ;; input is n
      ))
  )

(defun mew-summary-mark-refile ()
  (interactive)
  (let ((regex (concat mew-summary-message-regex
		       (regexp-quote (char-to-string mew-mark-hop))))
	(mew-summary-buffer-disp-msg nil)
	)
    (save-excursion
      (goto-char (point-min))
      (if (not (re-search-forward regex nil t))
	  (message "No marked messages")
	(beginning-of-line)
	(let ((mew-summary-refile-last-destination nil)
	      (mew-analysis nil)
	      folder)
	  (if (setq folder (mew-decide-folder))
	      (while (re-search-forward regex nil t)
		(let ((buffer-read-only nil));; mew-summary-unmark
		  (delete-backward-char 1)
		  (insert " "))
		(mew-summary-refile-mark folder)
		(forward-line 1))
	    ))
	)
      ))
  )

(defun mew-summary-exchange-mark (old-mark new-mark)
  (let ((regex (concat mew-summary-message-regex
		       (regexp-quote (char-to-string old-mark))))
	)
    (save-excursion
      (goto-char (point-min))
      (if (not (re-search-forward regex nil t))
	  (message "No marked messages")
	(beginning-of-line)
	(while (re-search-forward regex nil t)
	  (let ((buffer-read-only nil))
	    (delete-backward-char 1)
	    (insert (char-to-string new-mark)))
	  ))
      ))
  )
   
(defun mew-summary-mark-rmm ()		; * -> D
  (interactive)
  (mew-summary-exchange-mark mew-mark-hop mew-mark-rmm))

(defun mew-summary-mark-at ()		; * -> @
  (interactive)
  (mew-summary-exchange-mark mew-mark-hop mew-mark-mark))

(defun mew-summary-mark-asterisk ()	; @ -> *
  (interactive)
  (mew-summary-exchange-mark mew-mark-mark mew-mark-hop))

(defun mew-summary-swap-mark<->hop ()	; @ <-> *
  (interactive)
  (mew-summary-exchange-mark mew-mark-mark mew-mark-tmp)
  (mew-summary-exchange-mark mew-mark-hop mew-mark-mark)
  (mew-summary-exchange-mark mew-mark-tmp mew-mark-hop)
  )

;;
;; undo
;;

(defun mew-summary-undo (count)
  (interactive "P")
  (if (mew-summary-part-number)
      ;; go onto the current message anyway
      ;; mew-summary-multipart-delete saves excursion...
      (re-search-backward mew-summary-message-regex nil t nil))
  (mew-summary-multipart-delete)
  (save-excursion
    (if (not (numberp count))
	(mew-summary-undo-one)
      (while (> count 0)
	(mew-summary-undo-one)
	(forward-line)
	(setq count (1- count)))
      (while (< count 0)
	(mew-summary-undo-one)
	(forward-line -1)
      (setq count (1+ count)))
      )
    )
  (set-buffer-modified-p nil)
  )

(defun mew-summary-undo-one ()
  (interactive)
  (cond 
   ((mew-summary-marked-p)
    (beginning-of-line)
    (looking-at (concat mew-summary-message-regex "\\([^ +0-9]\\)"))
    (funcall (cdr (assoc (string-to-char (mew-match 2)) mew-undo-switch))))
   ((eobp) (message "No message"))
   (t (message "No mark"))
   )
  )

(defun mew-summary-undo-all ()
  (interactive)
  (let ((char nil) (ociea cursor-in-echo-area) (regex nil))
    (unwind-protect
	(progn
	  (message "Input mark : ")
	  (setq cursor-in-echo-area t)
	  (setq char (read-char))
	  (message "Input mark : %s" (char-to-string char))
	  )
      (setq cursor-in-echo-area ociea))
    (if (null (assoc char mew-mark-switch))
	(message "Mark %s is not supported" (char-to-string char))
      (mew-summary-multipart-delete)
      (let ((buffer-read-only nil))
	(message "Undoing %s ..." (char-to-string char))
	(save-excursion
	  (goto-char (point-min))
	  (setq regex (concat mew-summary-message-regex
			      (regexp-quote (char-to-string char))))
	  (while (re-search-forward regex nil t)
	    (delete-char -1)
	    (insert " "))
	  )
	(message "Undoing %s ... done" (char-to-string char))
	))
    )
  (set-buffer-modified-p nil)
  )

(defun mew-summary-refile-undo ()
  (setq mew-summary-buffer-refile
	(mew-delq (assoc (mew-summary-message-number) 
			 mew-summary-buffer-refile)
		  mew-summary-buffer-refile))
  (mew-summary-unmark)
  )

(defun mew-summary-convert-local-charset ()
  (interactive)
  (if mew-mule-p
      (save-excursion
	(set-buffer (mew-buffer-message))
	(goto-char (point-min))
	(if (eq mew-message-citation 'header)
	    (progn
	      (re-search-forward "^-*$" nil t)
	      (forward-line 1)
	      ))
	(let ((buffer-read-only nil))
	  (code-convert-region (point) (point-max) 
			       mew-mule-charset-local *internal*)
	  )
	))
  )

(defun mew-summary-convert-header ()
  (interactive)
  (if mew-mule-p
      (save-excursion
	(set-buffer (mew-buffer-message))
	(goto-char (point-min))
	(let ((buffer-read-only nil))
	  (mew-header-mime-decode)
	  )
	)
    )
  )

;;
;;
;;

(defun mew-summary-isearch-forward ()
  (interactive)
  (let ((cwin (get-buffer-window (current-buffer)))
	(mwin (get-buffer-window (mew-buffer-message))))
    (if mwin 
	(progn
	  (select-window mwin)
	  (unwind-protect
	      (isearch-forward)
	    (select-window cwin))
	  ))
    ))

(defun mew-summary-isearch-backward ()
  (interactive)
  (let ((cwin (get-buffer-window (current-buffer)))
	(mwin (get-buffer-window (mew-buffer-message))))
    (if mwin 
	(progn
	  (select-window mwin)
	  (unwind-protect
	      (isearch-backward)
	    (select-window cwin))
	  ))
    ))

;;
;;
;;

(defun mew-summary-x-face ()
  (interactive)
  (save-excursion
    (let (xface)
      (set-buffer (mew-buffer-message))
      (if (null (setq xface (mew-field-get-value "X-Face:")))
	  ()
	(set-buffer mew-buffer-tmp)
	(widen)
	(erase-buffer)
	(insert xface)
	(let ((selective-display nil)
	      (mc-flag nil)
	      (call-process-hook nil)
	      (default-process-coding-system 
		(if mew-mule-p (cons *noconv* *noconv*)))
	      (filters mew-x-face-filter))
	  (while filters
	    (call-process-region (point-min) (point-max) 
				 (car filters)
				 'delete t nil)
	    (setq filters (cdr filters))
	    )
	  (mew-mime-start-process (point-min) (point-max)
				  mew-x-face-prog
				mew-x-face-args)
	  )
	))
    ))

;;
;; quiting
;;

(defun mew-summary-suspend ()
  (interactive)
  (mew-window-pop))

(defun mew-kill-buffer (&optional buf)
  (interactive)
  (if buf
      (if (get-buffer buf) (kill-buffer buf))
    (kill-buffer (current-buffer)))
  )

;;
;; XXX ++virtual??
(defun mew-summary-quit ()
  "Exit summary mode."
  (interactive)
  (if (null (mew-y-or-n-p "Quit mew? "))
      ()
    (setq mew-path nil)
    (setq mew-draft-folder nil)
    (setq mew-alias-alist ())
    (setq mew-folder-list ())
    (setq mew-folder-alist ())
    (setq mew-refile-alist ())
    (setq mew-cache nil)
    (if (and window-system mew-emacs19-p)
	(let ((x (buffer-list)))
	  (while x
	    (if (and (buffer-name (car x))
		     (string-match mew-buffer-message (buffer-name (car x))))
		(mew-kill-buffer (car x))
	      (setq x (cdr x))
	      )))
      (mew-kill-buffer (mew-buffer-message)))
    (mew-kill-buffer mew-buffer-hello)
    (mew-kill-buffer mew-buffer-mime)
    (mew-kill-buffer mew-buffer-tmp)
    (let ((x (buffer-list))
	  (regex (concat "^"
;;			 (regexp-quote mew-buffer-watch) ;; self terminate
;;			 "\\|^"
			 (regexp-quote mew-buffer-cache))))
      (while x
	(if (and (buffer-name (car x))
		 (string-match regex (buffer-name (car x))))
	    (mew-kill-buffer (car x))
	  (setq x (cdr x))
	  )))
    (while mew-folders
      (mew-kill-buffer (car mew-folders))
      (setq mew-folders (cdr mew-folders))
      )
    (setq mew-folders (list mew-inbox))
    (run-hooks 'mew-quit-hook)
    )
  )

;; kill-emacs-hook should not be a list of functions but a single function
;; since kill-emacs does not use run-hooks but c subroutine.
;; This means that you can't add-hook kill-emacs-hook.
;;
;;So, set
;; (setq mew-init-hook
;;       (function
;;        (lambda ()
;; 	 (setq kill-emacs-hook (function mew-mark-process-all-folders)))))
;;
;;or
;;
;; (setq mew-init-hook
;;       (function
;;        (lambda ()
;;         (your job1)
;;         (your job2)
;;	   ...
;;	   (mew-mark-process-all-folders))))

(defun mew-mark-process-all-folders ()
  (let ((folders mew-folders))
    (save-excursion
      (while folders
	(if (bufferp (get-buffer (car folders))) ;; 
	    (progn
	      (set-buffer (car folders))
	      (if (mew-summary-mark-exist-p
		   (list mew-mark-rmm mew-mark-refile))
		  (if (mew-y-or-n-p
		       (format "Marks exists in %s. Process marks? "
			       (car folders)))
		      (mew-summary-exec)))
	      ))
	(setq folders (cdr folders))
	))
    ))

;;
;;
;;

(defun mew-summary-pipe-message (prefix command)
  (interactive
   (list current-prefix-arg 
	 (read-string "Shell command on message: " mew-last-shell-command)))
  (mew-summary-display)
  (save-excursion
    (set-buffer (mew-buffer-message))
    (widen)
    (if (string-equal command "")
	(setq command mew-last-shell-command))
    (goto-char (point-min)) ; perhaps this line won't be necessary
    (if prefix
	(search-forward "\n\n"))
    (shell-command-on-region (point) (point-max) command nil)
    (setq mew-last-shell-command command))
  (mew-message-narrow-to-page)
  )

;;;
;;; Virtual mode
;;;

(defun mew-virtual-mode ()
  "Major mode for reading picked messages over multiple folders.
The keys that are defined for this mode are:

SPC	Read through messages. See mew-summary-show-direction to set 
	'up,'down,'next(current direction) or 'stop. Default is 'down.
DEL	Back scroll this message.
	Unnecessary header fields are hidden over the window. Type DEL
	to see them when message are displayed.
.	Display this message or part. 
	If without MIME analysis, force to analyze this message.

RET	1 line scroll up this message.
ESC RET 1 line scroll down this message.

C-n	Go to the next line.
C-p	Go to the previous line.
n	Display below message or part.
p	Display above message or part.
N	Jump to below * marked message or display below message
	around multipart.
P	Jump to above * marked message or display above message
	around multipart.
j	Jump to a message according to inputed number.

i	Incorporate +inbox asynchronously.
g	Go to inputed folder.

w	Prepare draft to send a message.
a	Reply to this message or this part.
A	Reply to this message and insert it in draft buffer.
f	Forward this message as MIME format.
F	Forward @ marked messages as MIME format.
E	Edit this message again to send. In a draft folder, it just edit
	the message. Otherwise, copy the message to draft folder, then
	edit.
r	Re-distribute this message with Resent-To: and Resent-Cc:.

v	Toggle summary window only and summary & message windows.
	If you select summary window only, \"D\" never displays the 
	next message. So, you can set D marks quickly.
ESC a	Toggle with/out MIME analysis.
ESC l	Recenter this folder.

*	Mark this message with \"*\". Use N or P to jump to * marked message.
@	Mark this message with \"@\" for F, ESC s, and ESC t.
u	Cancel mark on this message.
U	Cancel all inputed mark.

C-cC-s	Incremental search forward on the message buffer.
C-cC-r	Incremental search backward on the message buffer.

ESC s	Apply unshar on @ marked messages.
ESC t	Apply uudecode on @ marked messages.

y	Copy this message or save this part as inputed file name.
#	Print this message or this part.
|	Pipe this message.

S	Sort messages in this folder.
O	Pack messages in this folder.

B	Burst messages encapsulated in MIME.

q	Switch to other buffer.
Q	Exit mew.
C-cC-q	Kill this buffer.

C-cC-l	Localize the transformed charset.
C-cd	Manual PGP decryption/vrfy.
C-ca	Add PGP public key in this message to pubring.pgp.
C-cC-d	Manual PEM decryption/vrfy.
C-cC-x	Display xface.

Z	Update aliases and folders list. If you create a new alias or folder
	with native MH, try this.

"
  (interactive)
  (setq major-mode 'mew-virtual-mode)
  (setq mode-name "Virtual")
  (use-local-map mew-virtual-mode-map)
  (setq buffer-read-only t)
  (setq truncate-lines t)
  (setq selective-display t)
  (setq selective-display-ellipses nil)
  (if (equal (nth (- (length mode-line-format) 2) mode-line-format)
	     '(-3 . "%p"))
      (setq mode-line-format
	    (let ((mlf (copy-sequence mode-line-format))
		  (l (length mode-line-format)))
	      (setcdr (nthcdr (- l 3) mlf)
		      '("[" mew-summary-buffer-left-msgs " more]" "-%-"))
	      mlf)
	    ))
  (if mew-highlight-lines-use
      (if mew-xemacs-p
	  (setq mode-motion-hook 'mode-motion-highlight-line)
	(if (save-excursion 
	      (goto-char (point-min))
	      (not (overlays-at (point))))
	    (mew-summary-highlight-lines-region (point-min) (point-max)))))
  (run-hooks 'mew-virtual-mode-hook)
  )

(defun mew-virtual-folder-message ()
  (interactive)
  (looking-at "^ *\\([0-9]+\\).*\r\\(.*\\)\n")
  (message "%s" (mew-match 2))
  )

(defun mew-virtual-scan (vfolder pickargs folders)
  (set-buffer (get-buffer-create vfolder))
  (mew-virtual-mode)
  (mew-window-configure (current-buffer) 'summary)
  (mew-current-set 'message nil)
  (mew-current-set 'part nil)
  (mew-current-set 'cache nil)
  (setq mew-summary-buffer-direction 'down)
  (if (null (mew-summary-exclusive-p))
      ()
    (condition-case err
	(progn
	  (message "Scanning %s ..." vfolder)
	  (let ((buffer-read-only nil))
	    (erase-buffer))
	  (goto-char (point-max))
	  (setq mew-summary-buffer-process
		(apply (function start-process)
		       "Scann" 
		       (current-buffer) 
		       mew-prog-vscan
		       "--scan"
		       "-noclear" "-noheader"
		       "-width"
		       (if (< (window-width) 80)
			   "80"
			 (int-to-string (window-width)))
		       (if pickargs 
			   (cons "--pick" (append pickargs folders))
			 folders
			 )))
	  (set-process-filter mew-summary-buffer-process
			      'mew-summary-scan-filter)
	  (set-process-sentinel mew-summary-buffer-process
				'mew-virtual-scan-sentinel)
	  (process-kill-without-query mew-summary-buffer-process)
	  )
      (quit
       (set-process-sentinel mew-summary-buffer-process nil)
       (setq mew-summary-buffer-process nil)
       (setq mew-summary-buffer-string nil)
       ))
    ))

(defun mew-summary-virtual ()
  (interactive)
  (let ((name (concat 
	       "++" 
	       (mew-input-string "Virtual folder name %s(%s): " 
				 "" ;; dummy
				 "virtual")))
	(folders (mew-input-folders (buffer-name)))
	(pattern (mew-pick-input-pattern)))
    (mew-virtual-scan name pattern folders)
    )
  )

(defun mew-virtual-scan-sentinel (process event)
  (save-excursion
    (set-buffer (process-buffer process)) ;; for safety
    (set-buffer-modified-p nil)
    (message "Scanning %s ... done" (buffer-name))
    (if mew-summary-buffer-string
	(if (string-match "^scan: no messages" mew-summary-buffer-string)
	    (message "No messages in %s" (buffer-name))
	  ;; need to hack
	  (let ((buffer-read-only nil))
	    (goto-char (point-max))
	    (insert mew-summary-buffer-string)
	    (goto-char (point-min))
	    (keep-lines mew-summary-message-regex))
	  ))
    (set-process-sentinel mew-summary-buffer-process nil)
    (setq mew-summary-buffer-process nil)
    (setq mew-summary-buffer-string nil)
    ))

;;;
;;; Message mode
;;;

(defun mew-message-mode ()
  "Major mode for display a message.
The keys that are defined for this mode are:

SPC	Scroll up this message.
DEL	Back scroll this message.
n	Display below message or part.
p	Display above message or part.
h	Get back to summary mode.
a	Reply to this message.
A	Reply to this message and insert it in draft buffer.
f	Forward this message as MIME format.
r	Re-distribute this message with Resent-To: and Resent-Cc:.
d	Mark this message with delete mark(default is \"D\").
o	Mark this message or this part's message with refile
	mark(default is \"o\"). If already marked with \"o\", it
	prints to which folder this message is refiled.
!	
"
  (interactive)
  (setq major-mode 'mew-message-mode)
  (setq mode-name "Message")
  (use-local-map mew-message-mode-map)
  (setq buffer-read-only t)
  (setq mew-message-citation 'header)
  (make-local-variable 'page-delimiter)
  (setq page-delimiter mew-page-delimiter)
  (run-hooks 'mew-message-mode-hook)
  )

(defun mew-message-next-page (&optional lines)
;; return t   if no more pages
;;        nil if more pages
  (interactive)
  (move-to-window-line -1)
  (if (save-excursion
        (end-of-line)
        (and (pos-visible-in-window-p)
             (eobp)))
      ;; Nothing in this page.
      (if (or (null mew-break-pages)
	      (save-excursion
		(save-restriction
		  (widen) (forward-line 1) (eobp)))) ;Real end-of-buffer?
;;	  (progn (message "End of message") t) ;Nothing more.
	  t
	(mew-message-narrow-to-page 1)		;Go to next page.
	nil
	)
    ;; More in this page.
    (condition-case ()
	(scroll-up lines)
      (end-of-buffer
       ;; Long lines may cause an end-of-buffer error.
       (goto-char (point-max))))
    nil
    ))

(defun mew-message-narrow-to-page (&optional arg)
  (interactive "P")
  (setq arg (if arg (prefix-numeric-value arg) 0))
  (save-excursion
    (forward-page -1) ;;Beginning of current page.
    (forward-char 1)  ;; for compatibility with emacs-19.28 and emacs-19.29
    (widen)
    (cond
     ((> arg 0)	(forward-page arg))
     ((< arg 0) (forward-page (1- arg)))
     )
    (forward-page)
    (narrow-to-region (point)
		      (progn
			(forward-page -1)
			(if (and (eolp) (not (bobp)))
			    (forward-line 1))
			(point)))
    ))

(defun mew-message-prev-page (&optional lines)
  (interactive)
  (move-to-window-line 0)
  (if (and mew-break-pages
	   (bobp)
	   (not (save-restriction (widen) (bobp)))) ;Real beginning-of-buffer?
      (progn
	(mew-message-narrow-to-page -1) ;Go to previous page.
	(goto-char (point-max))
	(recenter -1))
    (scroll-down lines)))

(defun mew-message-goto-summary ()
  (interactive)
  (pop-to-buffer (car (mew-current-get 'message)))
  (if (cdr (mew-current-get 'message))
      (mew-summary-jump-message (cdr (mew-current-get 'message)))))

(defun mew-message-reply ()
  (interactive)
  (mew-message-goto-summary)
  (call-interactively 'mew-summary-reply))

(defun mew-message-reply-with-citation ()
  (interactive)
  (mew-message-goto-summary)
  (call-interactively 'mew-summary-reply-with-citation))

(defun mew-message-forward ()
  (interactive)
  (mew-message-goto-summary)
  (call-interactively 'mew-summary-forward))

(defun mew-message-redist ()
  (interactive)
  (mew-message-goto-summary)
  (call-interactively 'mew-summary-redist))
 
(defun mew-message-rmm ()
  (interactive)
  (mew-message-goto-summary)
  (call-interactively 'mew-summary-rmm))

(defun mew-message-refile ()
  (interactive)
  (mew-message-goto-summary)
  (call-interactively 'mew-summary-refile))

;;; hack hack 
;;; Now folder name and buffer name is different

(defun mew-message-next-msg (&optional arg)
  (interactive "p")
  (let* ((obuf (current-buffer))
	 (buf (window-buffer (previous-window)))
	 (buffer-read-only nil)
	 (mew-summary-buffer-disp-msg-current t)
	 msg part)
    (pop-to-buffer buf) ;; for the next forward-line
    (if (not (or (eq major-mode 'mew-summary-mode)
		 (eq major-mode 'mew-virtual-mode)))
	()
      (forward-line arg) ;; minus arg results in prev
      (mew-summary-display))
    (pop-to-buffer obuf) ;; for window config
    ))
      
(defun mew-message-prev-msg (&optional arg)
  (interactive "p")
  (mew-message-next-msg (- arg)))

(defmacro mew-cache-mime-syntax (buf)
  (` (save-excursion
       (set-buffer (, buf))
       mew-mime-syntax)))

;;
;;  Prepare new message --- caching
;;

(defvar mew-cache nil
      "Mail cache. (old ... new) order alist with association
 ((\"+folder\" . \"message\") . cache-buffer)"
      )

(defmacro mew-cache-buffer-get (entry)
  (` (cdr (, entry))))

(defmacro mew-cache-folder-get (entry)
  (` (car (car (, entry)))))

(defmacro mew-cache-message-get (entry)
  (` (cdr (car (, entry)))))

(defmacro mew-cache-entry-make (fld-msg buf)
  (` (cons (, fld-msg) (, buf))))

(defmacro mew-cache-hit (fld-msg)
  "Return value assosiated with key."
  (` (mew-cache-buffer-get (assoc (, fld-msg) mew-cache))))

;;
;; UMEMURA Akihiro <akihiro@theory.brl.ntt.jp>
;;; mew-cache-sort
;;;   remove ENTRY from mew-cache if exists, and 
;;;   add entry at end of mew-cache.

(defun mew-cache-sort (entry)
  (let* ((pointer (cons nil mew-cache))
	 (top pointer))
    (while (cdr pointer)
      (if (equal (car (cdr pointer)) entry)
	  (setcdr pointer (cdr (cdr pointer)))
	(setq pointer (cdr pointer))))
    (setcdr pointer (list entry))
    (setq mew-cache (cdr top))))

(defun mew-cache-add (fld-msg)
  "Rutern associating buffer"
  (let ((len (length mew-cache))
	(buf nil))
    (if (>= len mew-cache-size)
	(prog1
	    (setq buf (mew-cache-buffer-get (car mew-cache)))
	  (setq mew-cache (nconc (cdr mew-cache)
				  (list (mew-cache-entry-make fld-msg buf))))
	  )
      (prog1
	  (setq buf (get-buffer-create 
		     (concat mew-buffer-cache (int-to-string len))))
	(setq mew-cache (nconc mew-cache
				(list (mew-cache-entry-make fld-msg buf))))
	))
    ))

(defun mew-cache-attribute-get (file)
  ;; 5th is modified time
  ;; 7th is file size
  (let ((attr (file-attributes file)))
    (list (nth 5 attr)
	  (nth 7 attr))))
       
(defun mew-cache-message (fld-msg)
  (let ((hit (mew-cache-hit fld-msg))
	(file (mew-expand-file-name (concat (car fld-msg) "/" (cdr fld-msg)))))
    (save-excursion
      (if hit
	  (progn
	    (mew-cache-sort (mew-cache-entry-make fld-msg hit))
	    (set-buffer hit)
	    ;; in cache buffer
	    (if (null (equal mew-cache-attribute 
			     (mew-cache-attribute-get file)))
		;; cache is invalid
		(mew-cache-message-new file)))
	;; no cache hit
	(setq hit (mew-cache-add fld-msg))
	(set-buffer hit)
	;; in cache buffer
	(mew-cache-message-new file)
	)
      hit ;; retrun value
      )))

(defun mew-cache-message-new (file)
  ;; in cache buffer
  (widen)
  (erase-buffer)
  (if mew-mule-p
      (let ((file-coding-system-for-read *noconv*))
	(insert-file-contents file))
    (insert-file-contents file))
  (if mew-debug
      (let ((debug-on-error t))
	(mew-mime-localform)
	(setq mew-mime-syntax (mew-mime-syntax-spice (mew-mime-syntax)))
	)
    (condition-case err
	(progn
	  (mew-mime-localform)
	  ;; buffer local variables
	  (setq mew-mime-syntax (mew-mime-syntax-spice (mew-mime-syntax)))
	  )
      (error
       (widen)
       (goto-char (point-min))
       (re-search-forward "^-*$")
       (beginning-of-line)
       (setq mew-mime-syntax (mew-rfc822-default-type))
       )
      ))
  (setq mew-cache-attribute (mew-cache-attribute-get file))
  )

;; use to debug
(defun mew-cache-flush ()
  (interactive)
  (mew-summary-multipart-delete)
  (let ((n 0))
    (while (< n mew-cache-size)
      (mew-kill-buffer (concat mew-buffer-cache (int-to-string n)))
      (setq n (1+ n))
      ))
  (mew-current-set 'cache nil)
  (setq mew-cache nil)
  )

(defun mew-status-update (arg)
  (interactive "P")
  (cond
   (arg ;; C-u
    (message "Updating aliases and folders ... ")
    (setq mew-alias-alist (mew-alias-make-alist))
    (setq mew-folder-list (mew-folder-make-list))
    (setq mew-folder-alist (mew-folder-make-alist mew-folder-list))
    (setq mew-refile-alist (mew-refile-make-alist mew-folder-list))
    (message "Updating aliases and folders ... done")
    )
   (t
    (message "Updating aliases ... ")
    (setq mew-alias-alist (mew-alias-make-alist))
    (setq mew-refile-alist (mew-refile-make-alist mew-folder-list))
    (message "Updating aliases ... done"))
   )
  )

;;
;; Cache header coordination
;;

(defun mew-cache-delete-field (field begin)
  (let ((case-fold-search t)
	(start nil))
    (save-excursion
      (goto-char (or begin (point-min)))
      (if (re-search-forward (format "^$\\|^%s" field) nil t)
	  (if (not (string= (downcase (mew-match 0)) (downcase field)))
	      nil
	    (setq start (match-beginning 0))
	    (forward-line 1)
	    (while (looking-at "[ \t]")
	      (forward-line 1))
	    (prog1
		(mew-buffer-substring start (point))
	      (delete-region start (point)))
	    )
	nil)
      )))

(defun mew-cache-delete-content-fields (&optional begin mime)
  (let ((case-fold-search t)
	(start nil)
	key)
    (save-excursion
      (goto-char (or begin (point-min)))
      (catch 'loop
	(while (re-search-forward "^$\\|^Content-" nil t)
	  (setq key (mew-match 0))
	  (setq start (match-beginning 0))
	  (if (string-match "^$" key) 
	      (if (not mime)
		  (throw 'loop nil)
		(forward-line 1)
		(delete-region start (point))
		(throw 'loop nil)
		))
	  (forward-line 1)
	  (while (looking-at "[ \t]")
	    (forward-line 1))
	  (delete-region start (point))
	  ))
      )))

;;
;; MIME local form creator
;;

(defun mew-charset-7bit-p (charset)
  (cond 
   ((string-match "iso-8859" charset) nil)
   ((string-match "iso-2022" charset) t)
   ((string-match "us-ascii" charset) t)
   (t t) ;; not safe
   )
  )

(defun mew-mime-decode (boundary encode charset &optional text)
  ;; encode never be nil
  ;; charset may be nil if type is not text/plain
  (let ((beg (point))
	(end (mew-mime-boundary-end boundary))
	(option nil)
	(selective-display nil)
	(mc-flag nil)
	(encode (downcase encode)))
    (if charset (setq charset (downcase charset)))
    (cond 
     ((string= encode "quoted-printable") 
      (setq option mew-prog-mime-decode-quoted))
     ((string= encode "base64")
      (if text
	  (setq option mew-prog-mime-decode-base64-text)
	(setq option mew-prog-mime-decode-base64)))
     ((string= encode "x-gzip64")
      (if text
	  (setq option mew-prog-mime-decode-gzip64-text)
	(setq option mew-prog-mime-decode-gzip64)))
     )
    (if boundary (set-marker mew-decode-marker (1+ end)))
    ;; boundary exists, so set marker after region
    (if (null option)
	();; 7bit, 8bit, and binary
      (apply 'call-process-region beg end
	     mew-prog-mime-decode t t nil option)
      )
    (if (and mew-mule-p charset (not (string-match "us-ascii" charset)))
	(code-convert-region 
	 beg
	 (if boundary
	     (1- (marker-position mew-decode-marker)) ;; tricky
	   (point-max)) ;; singlepart stuff
	 (mew-mule-content-coding
	  (mew-mule-content-attr charset mew-mule-character-set))
	 *internal*
	 )
      )
    ))

(defun mew-mime-boundary-end (boundary)
  (if (null boundary)
      (point-max)
    (save-excursion
      (re-search-forward (concat "^--" boundary))
      (beginning-of-line)
      (1- (point)) ;; boundary includes preceding CRLF
      )
    ))

(defun mew-mime-localform ()
  (if (null (mew-field-get-value "Mime-Version:"))
      (if mew-mule-p
	  (progn
	    (code-convert-region (point-min) (point-max)
				 mew-mule-charset-local *internal*)
	    (goto-char (point-min))
	    (mew-header-mime-decode))) ;; RFC1522
    (goto-char (point-min))
    ;; message/rfc822
    (mew-mime-localform-singlepart))
  )

(defun mew-mime-localform-singlepart (&optional boundary digest)
  ;;Called just after boundary. So, just before Content-Type:.
  (mew-header-mime-decode)
  (let* ((case-fold-search t)
	 (begin  (point)) ;; xxx
	 (part   (mew-part-make digest));; move point after content-*
	 (type   (mew-part-get-type   part))
	 (encode (mew-part-get-encode part))
	 (charset (mew-part-get-params-member
		   "charset"
		   (mew-part-get-params part)))
	 (format))
    (if (null encode) (setq encode "7bit"))
    (cond 
     ;; message/rfc822,  no decoding is required
     ((string-match "message/rfc822" type)
      (while (looking-at "^$") (forward-line)) ;; skip null mime header
      (save-restriction
	(narrow-to-region (point) (mew-mime-boundary-end boundary))
	(mew-mime-localform)))
     ;; multipart, no decoding is required
     ((string-match "^Multipart/\\(.*\\)$" type)
      (if (string-match "digest" (mew-match 1 type))
	  (mew-mime-localform-multipart part 'digest)
	(mew-mime-localform-multipart part)))
     ;; External body local and rfc822 -- nom
     ((and (string-match "message/external-body" type)
	   (looking-at "content-type:  *message/rfc822")
	   (equal "local-file"
		  (mew-part-get-params-member 
		   "access-type" (mew-part-get-params part))))
      (let ((file-name 
	     (mew-part-get-params-member "name" (mew-part-get-params part)))
	    (file-coding-system-for-read *noconv*))
	(if boundary
	    (mew-cache-delete-content-fields begin 'mime)
	  (delete-region (point-min) (point-max)))
	(goto-char (mew-mime-boundary-end boundary))
	(insert-file-contents file-name))
      (save-restriction
	(narrow-to-region (point) (mew-mime-boundary-end boundary))
	(mew-mime-localform))
      )
     ;; PEM
     ((string-match mew-pem-content-type type)
      (mew-cache-delete-field "Content-Type:" begin)
      (delete-char -1) ;; xxx delete unnecessary \n
      (mew-pem-decrypt-region (point) (mew-mime-boundary-end boundary))
      (if (null (looking-at "Content-"))
	  (insert "\n"))
      ;; xxx eval single part again after decode
      )
     ;; PGP here
     ((string-match (concat "^" mew-pgp-content-type "$") type)
      (if (not (mew-which mew-prog-pgp exec-path))
	  ()
	(setq format (mew-part-get-params-member 
		      "format"
		      (mew-part-get-params part)))
	(if (and format (equal "mime" (downcase format)))
	    (progn
	      (mew-cache-delete-content-fields begin 'mime) ;; delete "\n"
	      (mew-pgp-decode-region (point)
				     (mew-mime-boundary-end boundary)
				     'mime)
	      (mew-mime-localform-singlepart boundary)
	      )
	  ;; treat as local charset
	  (mew-cache-delete-content-fields begin nil) ;; leave "\n"
	  (mew-pgp-decode-region (point) (mew-mime-boundary-end boundary))
	  )))
     (t ;; charset is "us-ascii" if it is nil.
      (if (string-match "text/" type) ;; text/*
	  (mew-mime-decode boundary encode charset 'text)
	(mew-mime-decode boundary encode charset)))
     )
    ))

(defun mew-mime-localform-multipart (part &optional digest)
  (let ((case-fold-search nil) ;; boundary is case sensitive
	(boundary nil))
    (if (null (setq boundary (mew-part-get-params-member 
			      "boundary" (mew-part-get-params part))))
	(error "No boundary parameter for multipart"))
    (re-search-forward (concat "^--" boundary ))
    (forward-line) 
    (mew-mime-localform-singlepart boundary digest)
    (catch 'multipart
      (while (re-search-forward (concat "^--" boundary "-?-?$"))
	(forward-line)
	(if (string= (regexp-quote (mew-match 0)) (concat "--" boundary "--"))
	    (throw 'multipart nil)
	  )
	(mew-mime-localform-singlepart boundary digest)
	))
    ))

;;;
;;; MIME local form syntax analizer
;;;

;; <message> = [ 'message hbeg hend
;;                ("message/rfc822") (CTE:) (CI:) (CD:) (Mew:) <part> ]
;; <single>  = [ 'single beg end (CT:) (CTE:) (CI:) (CD:) (Mew:) ]
;; <multi>   = [ 'multi nil nil
;;               ("multipart/mixed" |
;;                "multipart/digest" |
;;                "multipart/alternative" |
;;                "multipart/parallel " )
;;               (CTE:) (CI:) (CD:) (Mew:)
;;                1*<part> ]
;; <part>    = <message>|<single>|<multi>

(defmacro mew-message-singlepart-p (message)
  (` (eq (aref (aref (, message) mew-part-magic) 0) 'single)))

(defmacro mew-message-multipart-p (message)
  (` (eq (aref (aref (, message) mew-part-magic) 0) 'multi)))

(defmacro mew-message-get-begin (message)
  (` (aref (, message) 1)))

(defmacro mew-message-set-begin (begin message)
  (` (progn (aset (, message) 1 (, begin)) (, message))))

(defmacro mew-message-get-end (message)
  (` (aref (, message) 2)))

(defmacro mew-message-set-end (end message)
  (` (progn (aset (, message) 2 (, end)) (, message))))

(defmacro mew-message-get-part (message)
  (` (aref (, message) mew-part-magic)))

(defmacro mew-message-set-part (body message)
  (` (progn (aset (, message) mew-part-magic (, body)) (, message))))

(defmacro mew-message-part-set-end (end message)
  (` (progn (aset (aref (, message) mew-part-magic) 2 (, end)) (, message))))

;;

(defmacro mew-part-message-p (part)
  (` (eq (aref (, part) 0) 'message)))

(defmacro mew-part-singlepart-p (part)
  (` (eq (aref (, part) 0) 'single)))

(defmacro mew-part-multipart-p (part)
  (` (eq (aref (, part) 0) 'multi)))

(defmacro mew-part-get-begin (part)
  (` (aref (, part) 1)))

(defmacro mew-part-set-begin (begin part)
  (` (progn (aset (, part) 1 (, begin)) (, part))))

(defmacro mew-part-get-end (part)
  (` (aref (, part) 2)))

(defmacro mew-part-set-end (end part)
  (` (progn (aset (, part) 2 (, end)) (, part))))

(defmacro mew-part-get-type (part)
  (` (car (aref (, part) 3))))

(defmacro mew-part-get-params (part)
  (` (cdr (aref (, part) 3))))

(defmacro mew-part-get-param1 (part)
  (` (car (cdr (aref (, part) 3)))))

(defun mew-part-get-params-member (parameter params)
  (let ((case-fold-search t)
	(ret))
    (catch 'loop
      (while params
	(if (string-match (concat "^" parameter "=\\(.*\\)$") (car params))
	    (throw 'loop (regexp-quote (mew-match 1 (car params)))))
	(setq params (cdr params))
	)
      )
    ))

(defmacro mew-part-get-encode (part)
  (` (car (aref (, part) 4))))

(defmacro mew-part-get-id (part)
  (` (car (aref (, part) 5))))

(defmacro mew-part-get-desc (part)
  (` (car (aref (, part) 6))))

(defmacro mew-part-get-mew (part)
  (` (car (aref (, part) 7))))

(defmacro mew-part-get-part (part)
  (` (aref (, part) mew-part-magic )))

(defun mew-multi-get-parts (multi) 
  (let ((ret nil)
	(n (1- (length multi)))
	(sentinel (1- mew-part-magic)))
    (while (> n  sentinel)
      (setq ret (cons (aref multi n) ret))
      (setq n (1- n))
      )
    ret
    ))

(defmacro mew-part-to-multi (part)
  (` (progn (aset (, part) 0 'multi) (, part))))

;; mew-mime-fields
(defmacro mew-part-get-head (part)
  (` (vector (aref (, part) 0) (aref (, part) 1) (aref (, part) 2)
	     (aref (, part) 3) (aref (, part) 4)
	     (aref (, part) 5) (aref (, part) 6)
	     (aref (, part) 7))))

(defmacro mew-part-only-one-part-p (multi)
  (` (equal (1+ mew-part-magic) (length (, multi)))))

(defmacro mew-single-make (fields)
  (` (vconcat (vector 'single (point) nil) (, fields))))

(defmacro mew-syntax-cat (parts part)
  (` (vconcat (, parts) (vector (, part)))))

;;
;; If you change mew-mime-fields, change here.
;;

(defun mew-mime-default-type (&optional type)
  (forward-line) ;; skip null line due to "^-*$"
  (vector 'single (point) (point-max) ;; end is dummy
	  (if (null type)
	      '("text/plain" "charset=us-ascii")
	    (list type))
	  nil nil nil nil)
  )

(defun mew-rfc822-default-type (&optional part)
  (let ((encode (if part (list (mew-part-get-encode part))))
	(id     (if part (list (mew-part-get-id part))))
	(desc   (if part (list (mew-part-get-desc part))))
	(mew    (if part (list (mew-part-get-mew part)))))
    ;; mew-part-get-* decapsulate the list, so quote with list.
    (vector 'message (point-min) (point) ;; end is dummy
	    (list "message/rfc822") encode id desc mew (mew-mime-default-type))
    ))

;;

(defun mew-part-make (digest)
  (let ((fields (mew-fields-get-as-list mew-mime-fields)))
    ;; point moves after content-* and \n
    (if (car fields)
	(mew-single-make fields)
      (if digest
	  (mew-single-make (cons '("message/rfc822") (cdr fields)))
	(mew-single-make (cons '("text/plain" "charset=us-ascii") (cdr fields)))))
    ))

(defun mew-position (list a)
  (let ((n 0)
	(ret nil))
    (catch 'loop 
      (while list 
	(if (equal (downcase (car (car list))) (downcase a))
	    (throw 'loop (setq ret n)))
	(setq n (1+ n))
	(setq list (cdr list))
	)
      )
    ret
    ))

(defun mew-fields-get-as-list (fields)
  (let* ((case-fold-search t)
	 (start nil)
	 (end nil)
	 (key nil)
	 (position nil)
	 (len (length fields))
	 (vec (make-vector len nil))
	 (n (1- len))
	 (ret))
    (catch 'loop
      (while t 
	;; Emacs 19's regex is not line based, so \n is necessary.
	;; \n is not harmless to Emacs 18.
	(if (looking-at "^-*$") (throw 'loop (forward-line)))
	(re-search-forward "^\\([^:\n]+:\\)[ \t]*" nil t)
	(setq key (mew-match 1))
	(setq start (match-end 0))
	(forward-line 1)
	(while (looking-at "[ \t]")
	  (forward-line 1))
	(setq end (1- (point)))
	(if (setq position (mew-position fields key))
	    (aset vec position (mew-buffer-substring start end)))
	))
    (while (> n -1)
      (if (aref vec n)
	  (if (cdr (nth n fields)) ;; header-syntax is necessary?
	      (setq ret (cons (mew-header-syntax (aref vec n)) ret))
	    (setq ret (cons (list (aref vec n)) ret)))
	(setq ret (cons nil ret)))
      (setq n (1- n))
      )
    ret
    ))

;;
;;
;;

(defun mew-mime-syntax (&optional part)
  (let* ((ver (mew-field-get-value "Mime-Version:"))
	 (mime (and ver (mew-header-delete-ws ver)))
	 (case-fold-search t))
    (goto-char (point-min))
    (re-search-forward "^-*$")
    (beginning-of-line)
    (cond
     ((not mime)  ;; RFC822
      (mew-rfc822-default-type part) ;; min, point, point+1, point-max
      )
     (t ;; MIME
      (if (null (string-match (concat "^" mew-mime-version) mime))
	  (error "Unsupport MIME version %s." ver)
	(let ((skel (mew-rfc822-default-type part))) ;; cursor after "^-*$"
	  ;; virtual message/rfc822 header
	  ;; (mew-mime-default-type) in (mew-rfc822-default-type)
	  ;; is meaningless
	  ;; min, point, point+1, point-max
	  (goto-char (point-min))
	  (mew-message-set-part 
	   (mew-mime-syntax-singlepart nil 'toplevel) skel)
	  )
	)
      )
     )
    ))

(defun mew-mime-syntax-singlepart (&optional digest toplevel)
  ;;called just after boundary.
  (let* ((case-fold-search t)
	 (part (mew-part-make digest));; move point after content-* and \n
	 (type (mew-part-get-type part))
	 (subpart))
    (cond
     ((string-match "^Multipart/\\(.*\\)$" type)
      (if (string-match "digest" (mew-match 1 type))
	  (mew-mime-syntax-multipart part 'digest)
	(mew-mime-syntax-multipart part)))
     ((string-match "^message/rfc822" type)
      (save-restriction
	(narrow-to-region (point) (point-max))
	(mew-mime-syntax part)))
     (t
      (if toplevel (mew-part-set-end (point-max) part) part))
     )
    ))

(defun mew-mime-syntax-multipart (part &optional digest)
  (let ((case-fold-search nil) ;; boundary is case sensitive
	(boundary nil)
	(last nil)
	(multi [ ]))
    (if (null (setq boundary (mew-part-get-params-member 
			      "boundary" (mew-part-get-params part))))
	(error "No boundary parameter for multipart"))
    (re-search-forward (concat "^--" boundary "$"))
    (forward-line) 
    (setq last (mew-mime-syntax-singlepart digest))
    (catch 'multipart
      (while (re-search-forward (concat "^--" boundary "-?-?$"))
	(beginning-of-line) ;; for end
	;; boundary includes preceding CRLF, so (1- )
	(if (mew-part-message-p last)
	    (if (mew-message-multipart-p last)
		(setq multi (mew-syntax-cat multi last))
	      (setq multi (mew-syntax-cat multi 
					  (mew-message-part-set-end
					   (1- (point)) last))))
	  (setq multi (mew-syntax-cat multi
				      (mew-part-set-end (1- (point)) last))))
	(forward-line) ;; skip boundary
	(if (string= (regexp-quote (mew-match 0)) (concat "--" boundary "--"))
	    (throw 'multipart
		   (vconcat (mew-part-set-end
			     (point) (mew-part-to-multi part)) multi))
	  (setq last (mew-mime-syntax-singlepart digest)))
	))
    ))

;;;
;;; Make mew-mime-syntax appropriate to Mew MIME viewer
;;;

;; mew-mime-fields
(defconst mew-multi-skel [multi nil nil ("multipart/mixed") nil nil nil nil])

(defun mew-mime-syntax-spice (syntax)
  (cond 
   ((mew-part-singlepart-p syntax) syntax)
   ((mew-part-message-p syntax)
    (let* ((head (mew-part-get-head syntax))
	   (part (mew-message-get-part syntax))
	   (hotpart (mew-mime-syntax-spice part)))
      (if (mew-part-message-p part)
	  (mew-syntax-cat head (mew-syntax-cat mew-multi-skel hotpart))
	(mew-syntax-cat head hotpart))
      )
    )
   ((mew-part-multipart-p syntax)
    (let* ((head (mew-part-get-head syntax))
	   (parts (mew-multi-get-parts syntax))
	   (len (length parts))
	   (vec (make-vector len nil))
	   (n 0))
      (while (< n len)
	(aset vec n (mew-mime-syntax-spice (nth n parts)))
	(setq n (1+ n))
	)
      (vconcat head vec)
      ))
   )
  )

;;
;;
;;

(defun mew-mime-syntax-print-loop (parts pref)
  (let ((po (point)))
    (cond
     ((mew-part-message-p parts)
      (insert (format "%s\t%s\t%s\n" 
		      pref
		      (mew-part-get-type parts)
		      (or (mew-part-get-desc parts) "")))
      (mew-summary-highlight-lines-region po (point) t)
      (if (mew-part-get-mew parts)
	  (setq mew-decode-message
		(concat mew-decode-message
			(format (concat mew-x-mew " <%s> ")
				(progn (string-match "[1-9.]+" pref)
				       (mew-match 0 pref)))
			(mew-part-get-mew parts)
			"\n")))
      (if (mew-part-multipart-p (mew-message-get-part parts))
	  (mew-mime-syntax-print-loop
	   (mew-message-get-part parts)
	   (concat "\t" pref))))
     ((mew-part-singlepart-p parts)
      (insert (format "%s\t%s\t%s\n" 
		      pref
		      (mew-part-get-type parts)
		      (or (mew-part-get-desc parts) "")))
      (mew-summary-highlight-lines-region po (point) t)
      (if (mew-part-get-mew parts)
	  (setq mew-decode-message
		(concat mew-decode-message
			(format (concat mew-x-mew " <%s> ")
				(progn (string-match "[1-9.]+" pref)
				       (mew-match 0 pref)))
			(mew-part-get-mew parts)
			"\n"))))
     ((mew-part-multipart-p parts)
      (if (mew-part-get-mew parts)
	  (setq mew-decode-message
		(concat mew-decode-message
			(format (concat mew-x-mew " <%s> ")
				(progn (string-match "[1-9.]+" pref)
				       (mew-match 0 pref)))
			(mew-part-get-mew parts)
			"\n")))
      (setq parts (mew-multi-get-parts parts))
      (let ((count 1))
	(while parts
	  (mew-mime-syntax-print-loop 
	   (car parts)
	   (concat pref "." (int-to-string count)))
	  (setq count (1+ count))
	  (setq parts (cdr parts))
	  ))
      )
     )
    )
  )

(defun mew-mime-syntax-print (parts)
  ;; message is always multipart.
  (setq parts (mew-multi-get-parts (mew-message-get-part parts)))
  (let ((pref "\t")
	(count 1))
    (while parts
      (mew-mime-syntax-print-loop 
       (car parts)
       (concat pref (int-to-string count)))
      (setq count (1+ count))
      (setq parts (cdr parts))
      )
    ))

(defun mew-mime-part-syntax (syntax num)
  (if (null num)
      (mew-message-get-part syntax)
    (mew-mime-part-syntax-loop syntax (mew-multi-split-number num))
    )
  )

(defun mew-mime-part-syntax-loop (syntax nums)
  (cond 
   ((null nums)	syntax) ;; singlepart mutch here
   ((mew-part-message-p syntax)
    (mew-mime-part-syntax-loop (mew-message-get-part syntax) nums))
   ((mew-part-multipart-p syntax)
    (mew-mime-part-syntax-loop
     (nth (1- (car nums)) (mew-multi-get-parts syntax)) (cdr nums)))
   )
  )

;;;
;;;
;;;

(defun mew-header-insert-from-file (file)
  (insert-file-contents file) ;; whole message
  ;;(insert "\n")
  (mew-header-arrange)
  )

(defun mew-header-insert-from-cache (cache begin end)
  (insert-buffer-substring cache begin end)
  (mew-header-arrange)
  (if mew-decode-message (insert mew-decode-message))
  (insert "\n")
  )

(defun mew-make-field-regex (fields)
  (mapconcat (function (lambda (x) (concat "^" x))) fields "\\|")
  )

(defvar mew-field-visible-regex nil)
(defvar mew-field-invisible-regex nil)
(defvar mew-field-visible-length 0)

(defun mew-make-regexes ()
  (setq mew-field-visible-regex 
	(mew-make-field-regex mew-field-visible))
  (setq mew-field-invisible-regex 
	(mew-make-field-regex mew-field-invisible))
  (setq mew-field-visible-length
	(length mew-field-visible))
  )

(defun mew-header-arrange ()
  (goto-char (point-min))
  (re-search-forward "^-*$")
  (if (> (count-lines (point-min) (point)) mew-header-max-length)
      (message "Header is too long")
    (if (not mew-field-visible-regex)
	(mew-make-regexes))
    (let* ((len mew-field-visible-length)
	   (visible-regex mew-field-visible-regex)
	   (invisible-regex mew-field-invisible-regex)
	   (visible (make-vector len nil))
	   (others nil) (key "") (line "") (start nil) (n nil))
      (goto-char (point-min))
      (catch 'header
	(while (re-search-forward "^-*$\\|^[^ \t\n]*:")
	  (setq key (mew-match 0))
	  (setq start (match-beginning 0))
	  (if (string-match "^-*$" key)
	      (throw 'header (and 
			      (get-buffer-window (mew-buffer-message))
			      (recenter 0))))
	  (forward-line 1)
	  (while (looking-at "[ \t]")
	    (forward-line 1))
	  (if (string-match invisible-regex key)
	      ()
	    (setq line (mew-buffer-substring start (point)))
	    (delete-region start (point))
	    (if (null (string-match visible-regex key))
		(setq others (concat others line))
	      (setq n (mew-member-match key mew-field-visible))
	      (aset visible n (concat line (aref visible n)))
	      ))
	  ))
      (setq n 0)
      (while (< n len)
	(if (aref visible n) (insert (aref visible n)))
	(setq n (1+ n))
	)
      (and others (insert others))
      )
    ))


;;;
;;; Draft mode
;;;

(defvar mew-draft-window-config nil)

(defun mew-draft-mode ()
  "Major mode for composing MIME message.
Key action is different in each region -- Header, Body, Multipart.

To send a draft, type C-cC-m and C-cC-c.
To make multipart, type C-cM, edit multipart, and type C-cC-m and C-cC-c.

*Whole buffer key assignment:

C-cC-y	Cite from *mew message* buffer.
C-cC-m	Make MIME message. Charset guess, maps directory structure to 
	multipart, and so on.
C-cC-c	Send this message. If you skipped C-cC-m, Content-Type: is added
	to header and you are asked, 
	\"Content-Type: was automatically added. Send this mail? \".
	Type y or n. 
	Mew sends the message in background. So, when you exit Emacs, you
	may be asked, 
	\"Active processes exist; kill them and exit anyway? (yes or no)\".
	In this case, check if *mew watch* buffer exist. If so, never 
	exit Emacs because Mew is still sending the message.
C-cC-u	Undo C-cC-m on multipart.

C-cC-q	Kill this draft.

(Following functions automatically insert the appropriate Content-Type:. 
So, C-cC-m is not necessary.)
C-cC-e	Encrypt this message by PEM. Currently only FJPEM is supported.
C-cC-s	Sign this message by PEM. Currently only FJPEM is supported.
C-ce	Encrypt this message by PGP.
C-cs	Sign this message by PGP.
C-cb	Sign then encrypt this message by PGP.
C-cp	Insert your PGP public key here.

C-cM	Prepare multipart region.

*Header region key assignment:

TAB	Complete field keys.
	Complete and expand email address aliases.
	Complete folder names.
C-cTAB	Complete your mail domain.


*Body region key assignment:

C-cTAB	Insert signature file.


*Multipart region Key assignment:

c	Copy a file in +drafts/mime/<num>/.
d	Delete this file.	
l	Link inputed file from +drafts/mime/<num>/.
r	Rename this file.
m	Make sub-multipart.
f	Read this file into a buffer.
F	Read a new file into a buffer.
e	Input external-body.
a	Sampling voice and insert as audio file.
T	Change its Content-Type:.
D	Input its Content-Description:.
B	Mark as Base64(B).
Q	Mark as Quoted-Printable(Q).
G	Mark as Gzip64(G).
S	Mark as PGP sign(PS).
E	Mark as PGP encrypt. Input to whom you encrypt this part(PE).
P	Mark as PGP sign then encrypt. Input to whom you encrypt this 
	part(PSE).
U	Get mark back.


* Fill blanks

Prepare ~/.mew-fib like;

	name:  Kazuhiko Yamamoto
	email: kazu@is.aist-nara.ac.jp

If you receive a message like;

	Your name : |>name<|
	Your e-mail address: |>email<|

Type a (in summary mode), C-cy, C-cC-f, and C-cC-k makes following
draft.

	Your name : Kazuhiko Yamamoto
	Your e-mail address: kazu@is.aist-nara.ac.jp

In this way, mew-fil fills up items quoted like |> <| from .mew-fib.

C-cy	Yank from *mew message* buffer without citation label.
C-cC-f	Fill |>item<| from .mew-fib.
C-cC-k	Delete all quotations, i.e. |> <|.
C-cC-n	Jump to the next fib item.
C-cC-p	Jump to the previous fib item.
C-cu	Flush input from .mew-fib.

Moreover, mew-fib supports aliases like;

	email: kazu@is.aist-nara.ac.jp
	e-mail:

"
  (interactive)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "^[ \t]*[-_][-_][-_]+$\\|" paragraph-start))
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate
        (concat "^[ \t]*[-_][-_][-_]+$\\|" paragraph-separate))
  (setq major-mode 'mew-draft-mode)
  (setq mode-name "Draft")
  (use-local-map mew-draft-mode-map)
  (set-syntax-table mew-draft-mode-syntax-table)
  (cd (expand-file-name "~"))
  (run-hooks 'text-mode-hook 'mew-draft-mode-hook)
  (if mew-emacs19-p
      (if (featurep 'hilit19)
          (hilit-rehighlight-buffer)
        (redraw-display))
    (redraw-display))
  ;; xxx how about xemacs ?
  (cond
   (mew-mule-p
    (make-local-variable 'file-coding-system)
    (make-local-variable 'file-coding-system-for-read)
    (make-local-variable 'default-process-coding-system)
    (make-local-variable 'mew-draft-window-config)
    (setq file-coding-system mew-mule-charset-integrity)
    (setq file-coding-system-for-read *autoconv*)
    (setq default-process-coding-system 
	  (cons mew-mule-charset-integrity mew-mule-charset-integrity))
    )
   )
  (cond
   (mew-emacs18-p ;; never use (and )
    (if auto-fill-hook 
	(progn
	  (make-local-variable 'auto-fill-hook)
	  (setq auto-fill-hook (function mew-draft-auto-fill))
	  )
      )
    )
   ((not mew-xemacs-p)
    (if auto-fill-function
	(progn
	  (make-local-variable 'auto-fill-function)
	  (setq auto-fill-function (function mew-draft-auto-fill))
	  )
      )
    )
   )
  )

(defun mew-draft-auto-fill ()
  (do-auto-fill)
  (if (mew-draft-in-header-p)
      (save-excursion
	(beginning-of-line)
	(if (not (looking-at "[^ \t]+:\\|[ \t]"))
	    (insert "\t"))
	)))

(defun mew-draft-get-new (&optional folder)
  (save-excursion
    (set-buffer mew-buffer-tmp)
    (erase-buffer)
    (call-process "mhpath" nil (current-buffer) nil
		   (or folder mew-draft-folder)
		   "new")
    (goto-char (point-min))
    (forward-line)
    (mew-buffer-substring (point-min) (1- (point)))
    ))

(defun mew-draft-rename (file)
  (rename-buffer (concat mew-draft-folder 
			 "/" 
			 (file-name-nondirectory file)))
  )

(defmacro mew-to-cc-magic ()
  (` (or (catch (quote match)
	   (car (mapcar
		 (function (lambda (arg)
			     (and (string-match arg c)
				  (throw (quote match) t))))
		     del)))
	 (string-match ":;" c))))

(defun mew-draft-header (&optional subject nl to cc in-reply-to references)
;; to -- string or list
;; cc -- string or list
;; nl -- one empty line under "----", which is necessary if
;;      multipart region is prepared
  (let ((del (cons (concat "^" (regexp-quote (user-login-name)) "$")
		   (cons (concat "^" mew-mail-address "$")
			 mew-mail-address-list)))
	;; deleting list for Cc:
	(c nil)
	(back nil)
	(tf to)) ;; to is used in last
    (goto-char (point-min))
    ;; Insert To: at first.
    ;; All addresses inserted on To: are appended to del.
    (cond
     ((null tf) (insert "To: \n"))
     ;; To: inputed from the mini-buffer.
     ((stringp tf)
      ;; Cc: is also string
      ;; We believe that user never specifies the same address of To: to Cc:
      ;; So, not add to to del.
      (mew-field-insert-here "To:" tf)) 
     ;; To: collected by reply
     ((listp tf)
      (insert (format "To: %s" (car tf)))
      (setq del (cons (car tf) del))
      (while tf
	(setq c (car tf) tf (cdr tf))
	(if (mew-to-cc-magic)
	    ()
	  (insert (format ",\n\t%s" c))
	  (setq del (cons c del))
	  ))
	(insert "\n"))
     )
    (cond
     ((null cc) ()) ; do nothing ;(insert "Cc: \n"))
     ;; Cc: inputed from the mini-buffer.
     ((stringp cc)
      (mew-field-insert-here "Cc:" cc))
     ;; Cc: collected by reply.
     ((listp cc)
      (insert "Cc: ")
      ;; find the first Cc: value since this is preceding by Cc:
      (catch 'first
	(while cc
	  (setq c (car cc) cc (cdr cc))
	  (if (mew-to-cc-magic)
	      ()
	    (insert c)
	    (setq del (cons c del))
	    (throw 'first nil))
	  ))
      ;; insert the second or more Cc: preceding ",\n\t"
      (while cc
	(setq c (car cc) cc (cdr cc))
	(if (mew-to-cc-magic)
	    ()
	  (insert (format ",\n\t%s" c))
	  (setq del (cons c del))
	  ))
	(insert "\n"))
     ))
  (and mew-cc   (mew-field-insert-here "Cc:" mew-cc))
  (mew-field-insert-here "Subject:" (if subject subject "")) ;; tricky
  (and mew-from (mew-field-insert-here "From:" mew-from))
  (and mew-fcc  (mew-field-insert-here "Fcc:" mew-fcc))
  (and mew-dcc  (mew-field-insert-here "Dcc:" mew-dcc))
  (and mew-reply-to (mew-field-insert-here "Reply-To:" mew-reply-to))
  ;; xxx date to in-reply-to ?
  (and in-reply-to (mew-field-insert-here 
		    "In-Reply-To:"
		    (concat "Your message of \"" in-reply-to "\"")))
  (and references (mew-field-insert-here "References:" references))
  (if (and mew-x-face-file
	   (file-exists-p (expand-file-name mew-x-face-file)))
      (let ((xface))
	(save-excursion
	  (set-buffer mew-buffer-tmp)
	  (erase-buffer)
	  (insert-file-contents (expand-file-name mew-x-face-file))
	  (setq xface (mew-buffer-substring (point-min) (max (buffer-size) 1)))
	  )
	(mew-field-insert-here "X-Face:" xface)
	))
  (and mew-x-mailer (mew-field-insert-here "X-Mailer:" mew-x-mailer))
  (let ((halist mew-header-alist))
    (while halist
      (and (stringp (car (car halist))) (stringp (cdr (car halist)))
	   (mew-field-insert-here (car (car halist)) (cdr (car halist))))
      (setq halist (cdr halist)))
    )
  (mew-field-insert-here "Mime-Version:" mew-mime-version)
  (setq mew-draft-buffer-header (point-marker))
  (insert "----\n")
  (if nl (insert "\n"))
  (if to ;; there is no To:
      ()
    (goto-char (point-min))
    (end-of-line))
  )

(defun mew-draft-send-letter (&optional arg)
  "Send this message. If arg is specified, leave the draft as it is."
  (interactive "P")
  (if mew-ask-subject
      (save-excursion
	(goto-char (point-min))
	(re-search-forward "^\\(Subject:.*\\|-+\\)$" nil t 1)
	(if (string-match "^Subject: +$"
			  (buffer-substring (match-beginning 0) (match-end 0)))
	    (insert (read-from-minibuffer "Subject: " nil nil)))))
  (run-hooks 'mew-send-hook)
  (save-excursion
    (if (mew-field-get-value "Content-Type:")
	(mew-draft-real-send-letter arg)
      (message "Making a MIME letter ...")
      (mew-draft-make-mime)
      (message "Making a MIME letter ... done")
      (condition-case err
	  (if (or mew-auto-add-content-type
		  (mew-y-or-n-p
		   "Content-Type: was automatically added. Send this mail? "))
	      (mew-draft-real-send-letter arg)
	    (mew-draft-delete-content-type))
	(quit
	 (mew-draft-delete-content-type)))
      )
    ))

(defun mew-draft-delete-content-type ()
  (goto-char (point-min))
  (save-restriction
    (narrow-to-region (point) (marker-position mew-draft-buffer-header))
    (let ((case-fold-search t)
	  (begin nil))
      (re-search-forward "^Content-Type:")
      (beginning-of-line) ;; for safty
      (setq begin (point))
      (if (looking-at "Content-Type: multipart")
	  ()
	(forward-line)
	(while (looking-at "[ \t]")
	  (forward-line 1))
	(delete-region begin (point))
	)
      )
    ))

(defun mew-draft-real-send-letter (&optional arg)
  (let* ((mimefolder (mew-draft-to-mime (buffer-name)))
	 (mimedir (mew-expand-file-name mimefolder))
	 (msg (file-name-nondirectory (buffer-file-name)))
	 (fcc nil)
	 (delete nil))
    (set-buffer-modified-p t)		; Make sure buffer is written
    (save-buffer)
    ;; make backup folder(s)
    ;; Fcc: fields may have multiple values.
    (setq fcc (mew-header-address-collect '("Fcc:")))
    (while fcc
      (if (file-exists-p (mew-expand-file-name (car fcc)))
	  () ;; do nothing
	(call-process mew-prog-folder nil nil nil
		      "-create"
		      (mew-dir-to-folder (car fcc))))
      (setq fcc (cdr fcc)))
    (if arg
	() ;; leave the draft
      (kill-buffer (current-buffer))
      ;;
      (if (mew-current-get 'window)
	  (progn
	    (set-window-configuration (mew-current-get 'window))
	    (mew-current-set 'window nil))))
    (set-buffer (generate-new-buffer mew-buffer-watch))
    ;; watch buffer
    (setq mew-watch-buffer-process
	  (start-process "Send" (current-buffer)
			 mew-prog-send
			 "-draftfolder" mew-draft-folder
			 "-draftmessage" msg
			 "-watch" "-verbose"))
    (set-process-sentinel mew-watch-buffer-process
			  'mew-watch-sentinel)
    (message "Sending a message in background")
    (if (null (file-directory-p mimedir))
	()
      (cond
       ((equal mew-mime-compose-folder-delete 'ask)
	(setq delete (mew-y-or-n-p
		      (format "Folder %s exists. Remove it? " mimefolder))))
       ((equal mew-mime-compose-folder-delete 'delete)
	(setq delete t))
       ((equal mew-mime-compose-folder-delete 'retain)
	(setq delete nil))
       (t
	(setq delete (mew-y-or-n-p
		      (format "Folder %s exists. Remove it? " mimefolder))))
       )
      (if (null delete )
	  (format "Folder %s remains" mimefolder)
	(message "Removing folder %s ... " mimefolder)
	(call-process "rm" nil nil nil "-rf" mimedir)
	(message "Removing folder %s ... done" mimefolder))
      )
    ))

(defun mew-watch-sentinel (process event)
  (let ((cbuf (current-buffer)) (kbuf (process-buffer process)))
    (set-buffer kbuf)
    (goto-char (point-min))
    (if (null (re-search-forward "^send:" nil t))
	(progn
	  (set-buffer cbuf)  ;; to avoid cursor-in-echo-area bug
	  (kill-buffer kbuf) ;; set-buffer before kill-buffer
	  )
      (ding)
      (message "Send failed")
      (beginning-of-line)
      (switch-to-buffer (process-buffer process))
      (local-set-key "\C-c\C-q" 'mew-kill-buffer)
      )
    ))

(defun mew-draft-yank ()
  (interactive)
  (let ((cite))
    (save-excursion
      (set-buffer (mew-buffer-message))
      (save-restriction
	(widen)
	(setq cite
	      (cond
	       ((if mew-xemacs-p 
		    (mark t) ;;; Niclas Andersson <nican@ida.liu.se>
		  (marker-position (mark-marker))) ;;; (mark)
		(mew-buffer-substring (region-beginning)
				      (region-end)))
	       ((equal mew-message-citation 'noheader)
		(mew-buffer-substring (point-min) (point-max)))
	       (t 
		(goto-char (point-min))
		(re-search-forward (format "^-*$"))
		(forward-line)
		(mew-buffer-substring (point) (point-max)))
	       )
	      )
	))
    (insert-string cite)
    ))

(defun mew-cite-strings ()
  (let ((fields (mapcar (function mew-field-get-value) mew-cite-fields)))
    (setq fields (mapcar (function (lambda (x) (if (null x) "" x))) fields))
    (apply (function format) mew-cite-format fields)
    ))

(defun mew-draft-cite (&optional arg)
  (interactive "P")
  (let ((cite "")
	(noheader (eq mew-message-citation 'noheader))
        (nonmewbuf mew-message-citation-buffer)) ; buffer local
    (save-excursion
      (set-buffer (or nonmewbuf (mew-buffer-message)))
      (save-restriction
        ;; first prepare "cite"
        (widen)
        (setq cite
              ;; Niclas Andersson <nican@ida.liu.se>
              (if (or (and mew-xemacs-p (mark t))
                      (marker-position (mark-marker))) ; (mark)
		  (progn
		    (setq noheader t)
		    ;; -> cite
		    (mew-buffer-substring (region-beginning) (region-end))
		    )
		;; with header
                (mew-buffer-substring (point-min) (point-max))
		))
        (if noheader
            (progn
              (set-buffer (or nonmewbuf
                              ;; header exists only in cache if multipart
                              (mew-cache-hit (mew-current-get 'message))
                              (mew-buffer-message)))
              (goto-char (point-min))
              (re-search-forward "^$" nil t)
              (setq cite (concat (mew-buffer-substring (point-min) (point))
                                 "\n" cite))
              ))
        ))
    (save-restriction
      (narrow-to-region (point)(point)) ;; for (goto-char (point-min))
      (insert cite)
      (push-mark (point))
      (goto-char (point-min)))
    (cond
     (mew-cite-hook
      (run-hooks 'mew-cite-hook))
     (mail-citation-hook
      (run-hooks 'mail-citation-hook))
     (t (mew-cite-original arg))
     )
    ))

(defun mew-cite-original (&optional arg)
  (if (< (marker-position (mark-marker)) (point))
      (exchange-point-and-mark))
  (let ((beg (point)) (end (marker-position (mark-marker)))
        label prefix)
    (save-restriction
      (narrow-to-region beg end)
      (condition-case nil
          (setq label (mew-cite-strings))
        (error "Syntax of mew-cite-format was changed. Read explanation of mew-cite-fields"))
      (if (null mew-cite-prefix-function)
          (setq prefix mew-cite-prefix)
        (setq prefix (funcall mew-cite-prefix-function)))
      (if mew-cite-prefix-confirmp
          (let ((ask (read-string 
                      (format "Prefix (\"%s\"): " prefix) "")))
            (if (not (string= ask "")) (setq prefix ask))))
      ;; C-u C-c C-y cites body with header.
      (if (eq arg nil) 
	  ;; header has been already cited. So, delete it.
	  (delete-region beg
			 (progn
			   (re-search-forward "^-*$" nil t)
			   (forward-line 1)
			   (point))))
      (insert label)
      (set-mark (point)) ;; for C-x C-x
      (and (bolp) (insert prefix))
      (while (equal 0 (forward-line 1))
	(or (equal (point) (point-max))
	    (insert prefix)))
      ) ;; restriction
    )
  )

;(defun mew-cite-strings ()
;  (let ((fields (mapcar (function mew-field-get-value) mew-cite-fields)))
;    (eval-expression (` (format (, mew-cite-format) (,@ fields))))
;    ))

;(defun mew-draft-cite (&optional arg)
;  (interactive "P")
;  (condition-case err
;      (let ((label "") (cite "") (fields "")
;	    (nonmewbuf mew-message-citation-buffer)) ;; buffer local
;	(save-excursion
;	  (set-buffer (or nonmewbuf (mew-buffer-message)))
;	  (save-restriction
;	    ;; first prepare "cite"
;	    (widen)
;	    (setq cite
;		  (cond
;		   (arg  ;; C-u C-c C-y cites body with header.
;		    (mew-buffer-substring (point-min) (point-max)))
;		   ((if mew-xemacs-p 
;			(mark t) ;;; Niclas Andersson <nican@ida.liu.se>
;		      (marker-position (mark-marker))) ;;; (mark)
;		    (mew-buffer-substring (region-beginning)
;					  (region-end)))
;		   ((equal mew-message-citation 'noheader)
;		    (mew-buffer-substring (point-min) (point-max)))
;		   (t 
;		    (goto-char (point-min))
;		    (re-search-forward (format "^-*$"))
;		    (forward-line)
;		    (mew-buffer-substring (point) (point-max)))
;		   )
;		  )
;	    ))
;	(if (null mew-cite-hook)
;	    (let (prefix)
;	      ;; use mew original cite
;	      (save-excursion
;		(set-buffer (or nonmewbuf
;				;; header exists only in cache if multipart
;				(mew-cache-hit (mew-current-get 'message))
;				(mew-buffer-message)))
;		(setq label (mew-cite-strings))
;		(if (null mew-cite-prefix-function)
;		    (setq prefix mew-cite-prefix)
;		  (setq prefix (if mew-cite-prefix-function
;				   (funcall mew-cite-prefix-function)))
;		  (if mew-cite-prefix-confirmp
;		      (let ((ask (read-string 
;				  (format "Prefix (\"%s\"): " prefix) "")))
;			(if (not (string= ask "")) (setq prefix ask))))
;		  ))
;	      (insert label)
;	      (narrow-to-region (point) (point))
;	      (set-mark (point)) ;; for C-x C-x
;	      (insert cite)
;	      (goto-char (point-min))
;	      (and (bolp) (insert prefix))
;	      (while (equal 0 (forward-line 1))
;		(or (equal (point) (point-max))
;		    (insert prefix)))
;	      )
;	  ;; use super-cite
;	  (narrow-to-region (point) (point))
;	  (save-excursion
;	    ;; get fields for super-cite
;	    (set-buffer (or nonmewbuf
;			    ;; header exists only in cache if multipart
;			    (mew-cache-hit (mew-current-get 'message))
;			    (mew-buffer-message)))
;	    (beginning-of-buffer)
;	    (re-search-forward "^$")
;	    (setq fields (mew-buffer-substring (point-min) (point)))
;	    )
;	  (insert fields)
;	  (insert "\n")
;	  (insert cite)
;	  (beginning-of-buffer)
;	  (push-mark nil)
;	  (end-of-buffer)
;	  (run-hooks 'mew-cite-hook)
;	  (pop-mark)
;	  )
;	(widen)
;	(message "") ;; prevent echo
;	)
;    (error (message "Syntax of mew-cite-format is changed. Read explanation of mew-cite-fields"))
;    )
;  )

(defun mew-draft-check-whom ()
  "Verify recipients of the draft."
   (interactive)
  (let* ((to-cc (mew-header-address-collect
		'("To:" "Cc:" "Bcc:" "Apparently-To:")))
	 (exp-to-cc (mew-header-expand-alias-list to-cc))
	 (buf (current-buffer)))
    (message "Checking recipients ... ")
    (get-buffer-create mew-buffer-whom)
    (switch-to-buffer-other-window mew-buffer-whom)
    (erase-buffer)
    (mapcar
     (function (lambda (x) (insert (format "%s\n" x))))
     exp-to-cc
     )
    (pop-to-buffer buf)
    (message "Checking recipients ... done")
    ))

(defun mew-draft-kill ()
  (interactive)
  (let* ((mimefolder (mew-draft-to-mime (buffer-name))) ;; buffer will kill
	 (mimedir (mew-expand-file-name mimefolder))
	 (delete nil))
    (if (mew-y-or-n-p "Kill draft message? ")
	(progn
	  (set-buffer-modified-p t) ;; for delete recovery file
	  (save-buffer)
	  (if (file-exists-p (buffer-file-name))
	      (delete-file (buffer-file-name)))
	  (set-buffer-modified-p nil)
	  (and (get-buffer (mew-draft-to-mime (buffer-name)))
	       (kill-buffer (mew-draft-to-mime (buffer-name))))
	  (kill-buffer (buffer-name))
	  (if (mew-current-get 'window)
	      (progn
		(set-window-configuration (mew-current-get 'window))
		(mew-current-set 'window nil)))
	  (message "Draft was killed")
	  (if (null (file-directory-p mimedir))
	      ()
	    (cond
	     ((equal mew-mime-compose-folder-delete 'ask)
	      (setq delete (mew-y-or-n-p
			    (format "Folder %s exists. Remove it? "
				    mimefolder))))
	     ((equal mew-mime-compose-folder-delete 'delete)
	      (setq delete t))
	     ((equal mew-mime-compose-folder-delete 'retain)
	      (setq delete nil))
	     (t
	      (setq delete (mew-y-or-n-p
			    (format "Folder %s exists. Remove it? "
				    mimefolder))))
	     )
	    (if (null delete )
		(format "Folder %s remains" mimefolder)
	      (message "Removing folder %s ... " mimefolder)
	      (call-process "rm" nil nil nil "-rf" mimedir)
	      (message "Removing folder %s ... done" mimefolder))
	    )
	  )
      (message "Draft was not killed"))
    ))

(defun mew-draft-insert-signature ()
  (interactive)
  (insert-file-contents (expand-file-name mew-signature-file))
  )

;;;
;;; draft mime operation
;;;

(defun mew-draft-make-mime ()
  (interactive)
  (if (null (mew-field-get-value "Content-Type:"))
      (if (mew-multi-p)
	  (mew-draft-make-multi)
	(mew-draft-make-single))
    (ding)
    (message "Content-Type: already exists!")
    )
  )

(defun mew-draft-make-single ()
  (let ((case-fold-search t)
	(charset (mew-draft-charset-guess-letter))
	file)
    (goto-char (marker-position mew-draft-buffer-header))
    ;; the beginning of "----"
    (forward-line 1)
    ;; cursor is just after ----
    (mew-field-insert-last "Content-Type:"
			   (format "Text/Plain; charset=%s" charset))
    ;; cursor is just after ----
    (if (mew-charset-7bit-p charset) ;;; xxxx hack qp is wrong
	(forward-line -2) ;; go to content-type
      (save-excursion
	;; Quoted-printable is necessary.
	;; Delete-region may be overhead but makes code simple ^^;
	;; I don't care of coding-system, so this forces
	;; input in ASCII!!
	(setq file (make-temp-name mew-temp-file))
	(write-region (point) (point-max) file)
	(delete-region (point) (point-max))
	(apply 'call-process
	       mew-prog-mime-encode file t nil mew-prog-mime-encode-quoted)
	(if (file-exists-p file) (delete-file file))
	(mew-field-insert-last "Content-Transfer-Encoding:"
			       "quoted-printable")
	(forward-line -3)) ;; go to content-type
      )
    ))

(defvar mew-draft-backup-file ".mew-backup")
(defvar mew-draft-syntax-file ".mew-syntax")

(defun mew-draft-make-multi ()
  (goto-char (marker-position mew-draft-buffer-multipart))
  (delete-region (point) (point-max))
  ;; disable marker
  (set-marker mew-draft-buffer-multipart nil)
  (let* ((path (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	 (backup-file (concat path "/" mew-draft-backup-file))
	 (syntax-file (concat path "/" mew-draft-syntax-file))
	 (syntax mew-multi-syntax))
    (write-region (point-min) (point-max) backup-file)
    (save-excursion
      (set-buffer (find-file-noselect syntax-file))
      (erase-buffer)
      (prin1 syntax (current-buffer))
      (save-buffer)
      (message "")
      (kill-buffer (current-buffer))
      ))
  (setq mew-draft-buffer-multipart nil)
  ;; delete delimiter
  (goto-char (marker-position mew-draft-buffer-header))
  (let ((beg (point))
	(syntax mew-multi-syntax))
    (forward-line 1)
    (delete-region beg (point))
    ;; disable marker
    (set-marker mew-draft-buffer-header nil)
    (setq mew-draft-buffer-header nil)
    ;; just after the header == beg
    ;; see if cover page is empty or not
    (while (and (looking-at "^$") (not (eobp)))
      (forward-line))
    (if (not (eobp))
	;; cover page exists, save it.
	;; On Mule, local charset may be chosen.
	;; This code may be overhead but makes code simple.
	(write-region beg (point-max)
		      (concat 
		       (mew-expand-file-name (mew-draft-to-mime (buffer-name)))
		       "/" mew-draft-coverpage))
      ;; cover page doesn't exist
      ;; remove the cover page entry
      (setq syntax (mew-multi-remove '(1) syntax)))
    (delete-region beg (point-max))
    (mew-draft-make-multipart syntax 
			      (mew-expand-file-name mew-draft-mime-folder)
			      0)
    (goto-char (point-min))
    (re-search-forward "^-*$")
    (setq mew-draft-buffer-header (point-marker))
    )
  ;; move cursor to content-type
  (goto-char (point-min))
  (re-search-forward "^Content-Type:")
  (beginning-of-line)
  (sit-for 0) ;; to flush display
  )

(defun mew-draft-make-multipart (syntax path depth)
  (let* ((boundary
	  (mew-boundary-get ;; 0 is nil for Next_Part
	   (if (> depth 0) (format "BOUNDARY%s" (int-to-string depth)))))
	 (mul (car syntax))
	 (fullname (mew-expand-file-name (mew-multi-get-name mul) path))
	 (type (mew-multi-get-type mul))
	 (desc (mew-multi-get-desc mul))
	 (encode (mew-multi-get-encode mul))
	 (param (mew-multi-get-param mul))
	 (start (point))
	 (tmp))
    (mew-field-insert-here "Content-Type:" 
			   (concat type ";\n\tboundary=\"" boundary "\""))
    (setq syntax (cdr syntax))
    (while syntax
      (insert (concat "\n--" boundary "\n"))
      (if (listp (car (car syntax)))
	  (mew-draft-make-multipart (car syntax) fullname (1+ depth))
	(mew-insert-file
	 ;; file
	 (mew-expand-file-name 
	  (mew-multi-get-name (car syntax)) fullname)
	 ;; type
	 (mew-multi-get-type (car syntax))
	 ;; description
	 (mew-multi-get-desc (car syntax))
	 ;; encoding
	 (mew-multi-get-encode (car syntax))
	 ;;
	 (mew-multi-get-param (car syntax))
	 )
	)
      (setq syntax (cdr syntax))
      )
    (insert (concat "\n--" boundary "--\n"))
    (if (not encode)
	()
      (setq tmp (make-temp-name mew-temp-file))
      (write-region start (point) tmp)
      (delete-region start (point))
      (mew-insert-pgp-encoded-file tmp type desc encode param)
      (goto-char (point-max))
      ) ;; tmp must be deleted
    ))

(defun mew-insert-file (file type desc encoding param)
  (if (equal desc "")
      (setq desc nil)
    (setq desc (mew-header-encode-string desc "B")))
  (cond
   ;; even if encoding == gzip64, we should guess charset
   ((and encoding (or (equal encoding 'pgp-sign)
		      (equal encoding 'pgp-enc)
		      (equal encoding 'pgp-sign-enc)))
    (mew-insert-pgp-encoded-file file type desc encoding param))
   ((string= (downcase type) "text/plain")
    (mew-insert-text-file file type desc encoding))
   ((and encoding (stringp encoding))
    (mew-insert-encoded-file file type desc encoding))
   ((string= (downcase type) "message/external-body")
    (insert-file-contents file)
    (if desc (mew-field-insert-here "Content-Description:"
				    (concat "\"" desc "\""))))
   (t
    (mew-field-insert-here "Content-Type:" type)
    (if desc (mew-field-insert-here "Content-Description:"
				     (concat "\"" desc "\"")))
    (insert "\n")
    (insert-file-contents file))
   )
  (goto-char (point-max))
  )

(defun mew-insert-encoded-file (file type desc encoding)
  (let (option)
    (mew-field-insert-here "Content-Type:" type)
    (if desc (mew-field-insert-here "Content-Description:"
				    (concat "\"" desc "\"")))
    (mew-field-insert-here "Content-Transfer-Encoding:" encoding)
    (insert "\n")
    (if (or (equal (downcase type) "text/plain")
	    (equal (downcase type) "application/postscript"))
	(cond
	 ((string= encoding "quoted-printable") 
	  (setq option mew-prog-mime-encode-quoted))
	 ((string= encoding "base64")
	(setq option mew-prog-mime-encode-base64-text))
	 ((string= encoding "x-gzip64")
	  (setq option mew-prog-mime-encode-gzip64-text))
	 )
      (cond
       ((string= encoding "quoted-printable") 
	(setq option mew-prog-mime-encode-quoted))
       ((string= encoding "base64")
	(setq option mew-prog-mime-encode-base64))
       ((string= encoding "x-gzip64")
	(setq option mew-prog-mime-encode-gzip64))
       )
      )
    (apply 'call-process mew-prog-mime-encode file t nil option)
    ))

(defun mew-insert-text-file (file type desc encoding)
  (let (charset option)
    ;; <<<Important experience>>>
    ;; Strictly speaking, *noconv* is right for canonical(transmit)
    ;; form. But it is very inconvenient to check the contents
    ;; just before sending this message. As a compromise, we include
    ;; the files as local charset to let users see the contents.
    ;; On UNIX file system, how is the charset guessed? There are few
    ;; chances to use charset other than local charset and US-ASCII.
    (let ((file-coding-system-for-read
	   (if mew-mule-p mew-mule-charset-local))) 
      (insert-file-contents file)
      )
    (setq charset (mew-draft-charset-guess-region (point) (point-max)))
    ;; If encoding is not nil, we should encode it.
    ;; If charset is not 7bit, we should encode it with pq.
    (if (or encoding (not (mew-charset-7bit-p charset)))
	(save-excursion
	  ;; Quoted-printable is necessary.
	  ;; Delete-region may be overhead but makes code simple ^^;
	  ;; I don't care of coding-system, so this forces
	  ;; input in ASCII!!
	  (delete-region (point) (point-max))
	  (cond
	   ((string= encoding "base64")
	    (setq option mew-prog-mime-encode-base64-text))
	   ((string= encoding "x-gzip64")
	    (setq option mew-prog-mime-encode-gzip64-text))
	   (t 
	    (setq encoding "quoted-printable")
	    (setq option mew-prog-mime-encode-quoted))
	   )
	  (apply 'call-process mew-prog-mime-encode file t nil option))
      )
    (mew-field-insert-here "Content-Type:"
			   (format "Text/Plain; charset=%s" charset))
    (if desc
	(mew-field-insert-here "Content-Description:" (concat "\"" desc "\"")))
    (if encoding
	(mew-field-insert-here "Content-Transfer-Encoding:" encoding))
    (insert "\n")
    ;; need not to goto-char point-max
    ))

;;
;; undo
;;

(defun mew-draft-undo ()
  (interactive)
  (let* ((path (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	 (backup-file (concat path "/" mew-draft-backup-file))
	 (syntax-file (concat path "/" mew-draft-syntax-file))
	 (syntax nil))
    (if (not (and (file-exists-p backup-file) (file-exists-p syntax-file)))
	(message "Can't undo")
      (erase-buffer)
      (insert-file-contents backup-file)
      (goto-char (point-min))
      (re-search-forward "^-*$")
      (beginning-of-line)
      (setq mew-draft-buffer-header (point-marker))
      (save-excursion
	(if (not (find-file-read-only syntax-file))
	    ()
	  (goto-char (point-min))
	  (setq syntax (read (current-buffer)))
	  (kill-buffer (current-buffer))
	  ))
      (setq mew-multi-syntax syntax)
      (mew-draft-prepare-multipart)
      (delete-file backup-file)
      (delete-file syntax-file)
      )
    ))

;; 
;; This code will be obsolated
;;
;; MIME encode region.
;; Buffer is *noconv*; Output from process is also *noconv*(ASCII).
(defun mew-mime-encode-region (beg end encode &optional text)
  (let ((option nil)
	(selective-display nil)
	(mc-flag nil))
    (setq encode (downcase encode))
    (cond
     ((string= encode "quoted-printable") 
      (setq option mew-prog-mime-encode-quoted))
     ((string= encode "base64")
      (if text
	  (setq option mew-prog-mime-encode-base64-text)
	(setq option mew-prog-mime-encode-base64)))
     ((string= encode "x-gzip64")
      (if text 
	  (setq option mew-prog-mime-encode-gzip64-text)
	(setq option mew-prog-mime-encode-gzip64)))
     )
    (if (null option)
	();; maybe 7bit encoding
      (message "MIME encoding ...")
      ;; Buffer is assumed to be included as *noconv*
      ;; We have already done; 
      ;; (define-program-coding-system nil mew-prog-mime-encode *noconv*)
      (apply 'call-process-region beg end
	     mew-prog-mime-encode t t nil option)
      (message "MIME encoding ... done")
      )
    ))

(defun mew-draft-charset-guess-letter ()
  (interactive)
  (mew-draft-charset-guess-region 
   (if (markerp mew-draft-buffer-header)
       (marker-position  mew-draft-buffer-header)
     (point-min))
   (point-max)
   ))

(defun mew-draft-charset-guess-region (beg end)
  (interactive "r")
  "Guess minimum character set name."
  (cond 
   (mew-mule-p
    (let ((lc (car (find-charset-region beg end))))
      (if (null lc)
	  "us-ascii"
	(mew-mule-lc-content (mew-mule-lc-attr lc mew-mule-character-set))
	)))
   (mew-emacs19-p
    (save-excursion
      (goto-char beg)
      (if (re-search-forward "[\200-\377]" end t)
	  "iso-8859-1"
	"us-ascii")))
   (t "us-ascii") ;; Emacs
   ))

;;;
;;; Draft multipart stuff
;;;

;; 3 for mark
;; 1 for space
;; 4 for number
;; 16 for filename (chopped)
;; 1 for space
;; 26 for type (chopped)
;; 1 for space
;; 26 + 2 for despcription (chopped)
;; chopped ".."
;;
;;---- multipart --
;;    0   1/               Multipart/Mixed           
;;PE  1     00CoverPage      Text/Plain              "Hello World"
;;MES 2     file2            Text/Plain              "0123456789012345678901"
;;    3.0   dir3/            Multipart/Mixed
;;    3.1     822              Message/Rfc822        "0123456789012345678901"
;;G   4     file4            Text/Plain
;;---- multipart  ----

;(setq syntax
;      '(("1/" "Multipart/Mixed" "" nil "")
;	("00CoverPage" "Text/Plain" "Hello World" 'pgp-enc "")
;	("file2" "Text/Plain" "0123456789012345678901" 'moss-enc-sign "kazu")
;	(("dir3/" "Multipart/Mixed" "" nil "")
;	 ("822" "Message/Rfc822" "0123456789012345678901xxxx" ""))
;	("file4" "Text/Plain" "" nil "gzip64" "")
;	)
;      )

;; G    -- 'x-gzip64
;; PE   -- 'pgp-enc
;; PS   -- 'pgp-sign
;; PSE  -- 'pgp-sign-enc
;; ME   -- 'moss-enc
;; MS   -- 'moss-sign
;; MES  -- 'moss-enc-sign
;; MOSS -- 'moss-sign-enc

;;
;; prepare multipart region
;;

(defvar mew-draft-coverpage "00CoverPage")

(defun mew-multi-initial-syntax (dir)
  (list (list (concat dir "/") "Multipart/Mixed" "" nil "")
	(list mew-draft-coverpage "Text/Plain" "" nil ""))
  )

(defun mew-multi-multi-syntax (dir num)
  (let (ret)
    (while (not (equal num 0))
      (setq ret
	    (cons (list (int-to-string num) "Message/rfc822" "" nil "") ret))
      (setq num (1- num))
      )
    (append (mew-multi-initial-syntax dir) ret)
    ))

(defun mew-draft-prepare-multipart ()
  ;; syntax should contain coverpage
  (interactive)
  (save-excursion
    (let* ((fullname (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (cp (concat fullname (concat "/" mew-draft-coverpage)))
	   (dir (file-name-nondirectory fullname))
	   ;;(dir (file-name-nondirectory (buffer-name))) is also OK.
	   (path (file-name-directory fullname)))
      (if (mew-multi-p)
	  ()
	(mew-multi-insert-boundary)
	(mew-make-directory fullname)
	(if (null (file-exists-p cp))
	    (call-process "touch" nil nil nil cp))
	(if (null mew-multi-syntax)
	    (setq mew-multi-syntax (mew-multi-initial-syntax dir)))
	(mew-multi-print "" 0 mew-multi-syntax)
	))
    )
  (goto-char (point-max))
  (beginning-of-line)
  )

(defun mew-multi-insert-boundary ()
   (goto-char (point-max))
   (if (null (bolp))(insert "\n"))
   (if (markerp mew-draft-buffer-multipart)
       (set-marker mew-draft-buffer-multipart (point))
     (setq mew-draft-buffer-multipart (point-marker)))
   (insert mew-draft-multipart-begin)
   (insert "\n")
   (insert mew-draft-multipart-end)
   (beginning-of-line)
   )

;;
;; print stuff
;;

(defun mew-multi-print (number level syntax)
  (let ((buffer-read-only nil)
	(count 0))
    (while syntax
      (if (listp (car (car syntax)))
	  (mew-multi-print (concat number (int-to-string count) ".")
			   (1+ level)
			   (car syntax))
	(insert 
	 (mew-multi-format (concat number (int-to-string count))
			   level 
			   (mew-multi-get-name (car syntax))
			   (mew-multi-get-type (car syntax))
			   (mew-multi-get-desc (car syntax))
			   (mew-multi-get-encode (car syntax))
			   (mew-multi-get-param (car syntax)))
	 ))
      (setq syntax (cdr syntax))
      (setq count (1+ count))
      )
    ))

(defun mew-multi-format (number level name type desc encode param)
  ;; level starts 1
  (let* ((marks "    ") (space " ") 
	 (quote "\"") (space-char 32)
	 (cnt "..") (lcnt (length cnt))
	 (ln 16) (lnc (- ln lcnt))
	 (lt 22) (ltc (- lt lcnt))
	 (ld 24) (ldc (- ld lcnt))
	 (dirp (string-match "/$" name))
	 (desc-or-param (if (equal param "") desc param)))
    (if encode
	(cond
	 ((equal encode "base64") (setq marks "B   "))
	 ((equal encode "quoted-printable") (setq marks "Q   "))
	 ((equal encode "x-gzip64") (setq marks "G   "))
	 ((equal encode 'pgp-sign) (setq marks "PS  "))
	 ((equal encode 'pgp-enc) (setq marks "PE  "))
	 ((equal encode 'pgp-sign-enc) (setq marks "PSE "))
	 )
      )
    (concat
     marks
     number
     space
     (make-string (if dirp 2 4) space-char)

     (if (< ln (length name))
	 (concat (substring name 0 lnc) cnt)
       (concat name (make-string (- ln (length name)) space-char)))
     space

     (let* ((tmp (* 2 (if dirp (- level 2) (1- level))))
	    (vlt (- lt tmp))
	    (vltc (- ltc tmp))
	    (ltype (length type)))
       (if (< vlt ltype)
	   (concat (substring type 0 vltc) cnt)
	 (concat type (make-string (- vlt ltype) space-char))))
     space

     (if (equal 0 (length desc-or-param))
	 ""
	(if (< ld (length desc-or-param))
	    (concat quote (substring desc-or-param 0 ldc) cnt quote)
	  (concat quote desc-or-param  quote)))
     "\n"
     )
    ))

(defun mew-multi-display ()
  (interactive)
  (let ((end nil)
	(stay (mew-multi-line))
	(i 1))
    (cond
     ((mew-multi-p)
      (goto-char (point-max))
      (re-search-backward (concat "^" mew-draft-multipart-end "$"))
      (setq end (point))
      (re-search-backward (concat "^" mew-draft-multipart-begin "$"))   
      (forward-line)
      (delete-region (point) end)
      (save-excursion
	(mew-multi-print "" 1 mew-multi-syntax)
	)
      (while (< i stay)
	(forward-line)
	(setq i (1+ i))
	)
      ))
    ))

;;
;; some goodies
;;

(defmacro mew-multi-p ()
  (` (markerp mew-draft-buffer-multipart))
  )

(defun mew-multi-part-number ()
  (save-excursion
    (beginning-of-line)
    (if (looking-at "^....\\([\.0-9]+\\) *")
	(mew-match 1)
      nil
      )
    ))

(defun mew-multi-line ()
  (save-excursion
    (beginning-of-line)
    (count-lines
     (marker-position mew-draft-buffer-multipart)
     (point))
    ))

(defmacro mew-multi-get-name (list)
 (` (car (, list))))

(defmacro mew-multi-get-type (list)
 (` (car (cdr (, list)))))

(defmacro mew-multi-get-desc (list)
 (` (car (cdr (cdr (, list))))))

(defmacro mew-multi-get-encode (list)
 (` (car (cdr (cdr (cdr (, list)))))))

(defmacro mew-multi-get-param (list)
 (` (car (cdr (cdr (cdr (cdr (, list))))))))

(defmacro mew-multi-set-name (name list)
  (` (mew-setnth 0 (, name) (, list))))

(defmacro mew-multi-set-type (type list)
  (` (mew-setnth 1 (, type) (, list))))

(defmacro mew-multi-set-desc (desc list)
  (` (mew-setnth 2 (, desc) (, list))))

(defmacro mew-multi-set-encode (mark list)
  (` (mew-setnth 3 (, mark) (, list))))

(defmacro mew-multi-set-param (mark list)
  (` (mew-setnth 4 (, mark) (, list))))

(defun mew-multi-split-number (num-str &optional plus1)
  (let ((start 0)
	(end 0)
	(nums nil))
    (while (setq end (string-match "\\." num-str start))
      (setq nums (cons (string-to-int (substring num-str start end)) nums))
      (setq start (1+ end))
      )
    (if (null plus1)
	(nreverse (cons (string-to-int (substring num-str start nil)) nums))
      (nreverse (cons (1+ (string-to-int (substring num-str start nil))) nums))
      )
    ))

(defun mew-setnth (num value list)
  (let ((n (1- num))
	(ret nil))
    (while (< -1 n)
      (setq ret (cons (nth n list) ret))
      (setq n (1- n))
      )
    (append ret (list value) (nthcdr (1+ num) list))
    ))

;; ignore "1/"
(defun mew-multi-expand-path (nums syntax)
  (let ((path ""))
    (while (cdr nums)
      (setq syntax (nth (car nums) syntax))
      (setq path (concat path (mew-multi-get-name (car syntax))))
      (setq nums (cdr nums))
      )
    path
    ))

;;
;; Get the entry
;;

(defun mew-multi-get (nums syntax)
  "Get the entry for NUMS from SYNTAX.
e.g. entry == (name type desc mark)"
  (while nums
    (setq syntax (nth (car nums) syntax))
    (setq nums (cdr nums))
    )
  syntax
  )

;;
;; Set entry, entry should be prepared before calling this.
;;

(defun mew-multi-set (nums entry syntax)
  (if (null nums)
      entry
    (let ((n (1- (car nums)))
	  (ret nil))
      (while (< -1 n)
	(setq ret (cons (nth n syntax) ret))
	(setq n (1- n))
	)
      (append ret
	      (list (mew-multi-set (cdr nums)
				   entry
				   (nth (car nums) syntax)))
	      (nthcdr (1+ (car nums)) syntax))
      )
    ))

;;
;; Insert entry
;;

(defun mew-multi-insert (nums entry syntax)
  (let ((n (1- (car nums)))
	(ret nil))
    (cond 
     ;; insert diretory
     ((and (null (cdr (cdr nums))) (equal (car (cdr nums)) 0))
      (while (< -1 n)
	(setq ret (cons (nth n syntax) ret))
	(setq n (1- n))
	)
      (append ret (list (list entry)) (nthcdr (car nums) syntax))
      )
     ;; insert file
     ((null (cdr nums))
      (while (< -1 n)
	(setq ret (cons (nth n syntax) ret))
	(setq n (1- n))
	)
      (append ret (list entry) (nthcdr (car nums) syntax))
      )
     (t
      (while (< -1 n)
	(setq ret (cons (nth n syntax) ret))
	(setq n (1- n))
	)
      (append ret
	      (list (mew-multi-insert (cdr nums)
				      entry
				      (nth (car nums) syntax)))
	      (nthcdr (1+ (car nums)) syntax))
      )
     )
    ))

;;
;; remove
;;

(defun mew-multi-remove (nums syntax)
  (let ((n (1- (car nums)))
	(ret nil))
    (cond 
     ;; insert diretory
     ((and (null (cdr (cdr nums))) (equal (car (cdr nums)) 0))
      (while (< -1 n)
	(setq ret (cons (nth n syntax) ret))
	(setq n (1- n))
	)
      (append ret (nthcdr (1+ (car nums)) syntax))
      )
     ;; insert file
     ((null (cdr nums))
      (while (< -1 n)
	(setq ret (cons (nth n syntax) ret))
	(setq n (1- n))
	)
      (append ret (nthcdr (1+ (car nums)) syntax))
      )
     (t
      (while (< -1 n)
	(setq ret (cons (nth n syntax) ret))
	(setq n (1- n))
	)
      (append ret
	      (list (mew-multi-remove (cdr nums)
				      (nth (car nums) syntax)))
	      (nthcdr (1+ (car nums)) syntax))
      )
     )
    ))

;;
;; commands
;;

(defun mew-multi-nums (&optional previous plus1)
  (save-excursion
    (if previous (forward-line -1))
    (if (mew-multi-part-number)
	(mew-multi-split-number (mew-multi-part-number) plus1)
      nil)
    ))

(defun mew-multi-copy ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't copy here")
    (let* ((nums (mew-multi-nums 'previous 'plus1))
	   (subdir (mew-multi-expand-path nums mew-multi-syntax))
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (fromfile (mew-input-filename "Copy from : "))
	   (tofile (mew-input-string "Copy to %s(%s): " 
				     subdir
				     (file-name-nondirectory fromfile)))
	   (efile nil))
      (setq efile (if (equal subdir "") tofile (concat subdir tofile)))
      (if (and (file-exists-p (expand-file-name efile mimedir))
	       (null (mew-y-or-n-p (message "File %s exists. Overwrite it? " 
					    efile))))
	  ()
	(save-excursion
	  (set-buffer (get-buffer-create mew-buffer-tmp))
	  (widen)
	  (erase-buffer)
	  (let ((mc-flag nil)
		(file-coding-system-for-read (if mew-mule-p *noconv*))
		(file-coding-system (if mew-mule-p *noconv*)))
	    (insert-file-contents fromfile)
	    (write-region (point-min) (point-max)
			  (expand-file-name efile mimedir))
	    ))
;;	(call-process "cp" nil nil nil
;;		      fromfile 
;;		      (expand-file-name efile mimedir))
	(setq mew-multi-syntax
	      (mew-multi-insert 
	       nums
	       (list tofile
		     (mew-file-content 
		      (mew-file-attr tofile mew-mime-content-type))
		     ""
		     (mew-file-encoding
		      (mew-file-attr tofile mew-mime-content-type))
		     ""
		     )
	       mew-multi-syntax))
	(mew-multi-display)
	)
      )
    ))

(defun mew-multi-link ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't link here.")
    (let* ((doit t)
	   (nums (mew-multi-nums 'previous 'plus1))
	   (subdir (mew-multi-expand-path nums mew-multi-syntax))
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (fromfile (mew-input-filename "Link from : "))
	   (tofile (mew-input-string "link to %s(%s): " 
				     subdir
				     (file-name-nondirectory fromfile)))
	   (efile nil))
      (setq efile (if (equal subdir "") tofile (concat subdir tofile)))
      (if (file-exists-p (expand-file-name efile mimedir))
	  (if (null (mew-y-or-n-p (message "File %s exists. Overwrite it? "
					   efile)))
	      (setq doit nil)
	    (call-process "rm" nil nil nil 
			  (expand-file-name efile mimedir))))
      (if (null doit)
	  ()
	(if mew-os2-p
	    (call-process "cp" nil nil nil fromfile 
			  (expand-file-name efile mimedir))
	  (call-process "ln" nil nil nil "-s" fromfile 
			(expand-file-name efile mimedir)))
	(setq mew-multi-syntax
	      (mew-multi-insert 
	       nums
	       (list tofile
		     (mew-file-content 
		      (mew-file-attr tofile mew-mime-content-type))
		     ""
		     (mew-file-encoding
		      (mew-file-attr tofile mew-mime-content-type))
		     ""
		     )
	       mew-multi-syntax))
	(mew-multi-display)
	))
    ))

(defun mew-multi-rename ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't rename here.")
    (if (null (mew-multi-part-number))
	(message "No part here.")
      (let* ((insert-default-directory nil)
	     (nums (mew-multi-nums))
	     (subdir (mew-multi-expand-path nums mew-multi-syntax))
	     (mimedir (mew-expand-file-name
		       (mew-draft-to-mime (buffer-name))))
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (fromfile (mew-multi-get-name entry))
	     (efrom (concat subdir fromfile))
	     (tofile nil)
	     (efile nil))
	(setq tofile
	      (read-file-name 
	       (format "From %s%s to (%s): " subdir fromfile fromfile)
	       (expand-file-name subdir mimedir)
	       nil nil))
	(if (equal tofile "") (setq tofile fromfile))
	(setq efile (concat subdir tofile))
	(if (and (file-exists-p (expand-file-name efile mimedir))
		 (null (mew-y-or-n-p
			(message "File %s exists. Overwrite it? " efile))))
	    ()
	  (call-process "mv" nil nil nil 
			(expand-file-name efrom mimedir)
			(expand-file-name efile mimedir))
	(setq mew-multi-syntax
	      (mew-multi-set nums
			     (mew-multi-set-name tofile entry)
			     mew-multi-syntax))
	(mew-multi-display)
	))
      )))
	
(defun mew-multi-delete ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't delete here.")
    (if (null (mew-multi-part-number))
	(message "No part here.")
      (let* ((nums (mew-multi-nums))
	     (subdir (mew-multi-expand-path nums mew-multi-syntax))
	     (mimedir (mew-expand-file-name
		       (mew-draft-to-mime (buffer-name))))
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (name (mew-multi-get-name entry))
	     (ename nil)
	     (doit nil))
	(cond
	 ((string-match ".*/$" name)
	  (setq ename subdir)
	  (if (mew-y-or-n-p
	       (format "Delete %s with contained files? " ename))
	      (setq doit t)
	    (setq doit nil)))
	 (t 
	  (setq ename (if (equal subdir "") name (concat subdir name)))
	  (if (mew-y-or-n-p (format "Delete %s? " ename))
	      (setq doit t)
	    (setq doit nil)))
	 )
	(if (null doit)
	    ()
	  (message "Deleting %s ... " ename)
	  (call-process "rm" nil nil nil
			"-rf"
			(expand-file-name ename mimedir))
	  (message "Deleting %s ... done" ename)
	  (setq mew-multi-syntax (mew-multi-remove nums mew-multi-syntax))
	  (mew-multi-display)
	  )
	))
    ))

;; subpart
(defun mew-multi-multipart ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't create sub-multipart here.")
    (let* ((nums (mew-multi-nums 'previous 'plus1))
	   (subdir (mew-multi-expand-path nums mew-multi-syntax))
	   (insert-default-directory nil)
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (dir (read-file-name 
		 (format "Directory name (%s): " subdir) mimedir nil nil));; xxx
	   (edir nil))
      ;; mimedir{/subdir}/dir
      ;; edir = {/subdir}/dir
      (setq edir (if (equal subdir "") dir (concat subdir dir)))
      (if (and (file-exists-p (expand-file-name edir mimedir))
	       (null (mew-y-or-n-p
		      (message 
		       "File or directory %s exists. Overwrite it? " 
		       edir))))
	  ()
	(mew-make-directory (expand-file-name edir mimedir))
	(setq mew-multi-syntax
	      (mew-multi-insert 
	       (append nums (list 0)) ;; directory
	       (list 
		(if (string-match "/$" dir) dir (concat dir "/"))
		"Multipart/Mixed"
		"" ;; desc
		nil ;; CTE: is nil
		"") ;; param
	       mew-multi-syntax))
	(mew-multi-display)
	))
    ))

(defun mew-multi-find-file ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't find a file here.")
    (if (null (mew-multi-part-number))
	(message "No part here.")
      (let* ((nums (mew-multi-nums))
	     (subdir (mew-multi-expand-path nums mew-multi-syntax))
	     (mimedir (mew-expand-file-name
		       (mew-draft-to-mime (buffer-name))))
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (file (mew-multi-get-name entry))
	     (efile nil))
	(setq efile (if (equal subdir "") file (concat subdir file)))
	(find-file (expand-file-name efile mimedir))
	;; buffer switched
	(local-set-key "\C-c\C-q" 'mew-kill-buffer)
	)
      )))

(defun mew-multi-find-mew-file ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't find a new file here.")
    (let* ((nums (mew-multi-nums 'previous 'plus1))
	   (subdir (mew-multi-expand-path nums mew-multi-syntax))
	   (mimedir (mew-expand-file-name
		     (mew-draft-to-mime (buffer-name))))
	   (file (mew-input-string "Open file as %s(%s): " 
				    subdir
				    "a.txt"))
	   (efile nil))
      (setq efile (if (equal subdir "") file (concat subdir file)))
      (setq mew-multi-syntax
	    (mew-multi-insert 
	     nums
	     (list file "Text/Plain" "" nil "")
	     mew-multi-syntax))
      (mew-multi-display)
      (find-file (expand-file-name efile mimedir))
      ;; buffer switched
      (local-set-key "\C-c\C-q" 'mew-kill-buffer)
      )
    ))

(defun mew-multi-description ()
  (interactive)
  (if (< (mew-multi-line) 1)
      (message "Can't tag description here")
    (if (null (mew-multi-part-number))
	(message "Can't tag description here")
      (let* ((part (mew-multi-part-number))
	     (nums (mew-multi-split-number part))
	     (description)
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (name (mew-multi-get-name entry)))
	(if (string-match "/$" name)
	    (message "Can't tag description to Multipart.")
	  (setq description
		(read-string (format "Description (%s): " name) ""))
	  (if (equal description "") (setq description name))
	  (setq mew-multi-syntax
		(mew-multi-set nums
			       (mew-multi-set-desc description entry)
			       mew-multi-syntax))
	  (mew-multi-display)
	  ))
      )
    ))

(defmacro mew-multi-directory-p (nums)
  (` (equal 0 (nth (1- (length (, nums))) (, nums)))))

(defun mew-multi-type ()
  (interactive)
  (if (< (mew-multi-line) 1)
      (message "Can't change Content-Type: here")
    (if (null (mew-multi-part-number))
	(message "Can't change Content-Type: here")
      (let* ((part (mew-multi-part-number))
	     (nums (mew-multi-split-number part))
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (name (mew-multi-get-name entry))
	     (type (mew-multi-get-type entry)))
	(if (mew-multi-directory-p nums)
	    (setq type
		  (mew-input-type
		   name mew-mime-content-type-multipart-list type))
	  (setq type (mew-input-type name mew-mime-content-type-list type)))
	(setq mew-multi-syntax
	      (mew-multi-set nums
			     (mew-multi-set-type type entry)
			     mew-multi-syntax))
	(mew-multi-display)
	)
      )
    ))

(defun mew-multi-base64 ()
  (interactive)
  (if (< (mew-multi-line) 1)
      (message "Can't encode with base64 here")
    (if (null (mew-multi-part-number))
	(message "Can't encode with base64 here")
      (let* ((part (mew-multi-part-number))
	     (nums (mew-multi-split-number part))
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (encode (mew-multi-get-encode entry))
	     (name (mew-multi-get-name entry))
	     (type (mew-multi-get-type entry))
	     (case-fold-search t))
	(if (or (string-match "/$" name)
		(equal "^Message/" type)
		(equal "^Multipart/" type))
	    (message "Can't encode with base64 here")
	  (setq mew-multi-syntax
		(mew-multi-set 
		 nums
		 (mew-multi-set-param "" (mew-multi-set-encode "base64" entry))
		 mew-multi-syntax))
	  (mew-multi-display)
	  ))
      ))
  )

(defun mew-multi-quoted-printable ()
  (interactive)
  (if (< (mew-multi-line) 1)
      (message "Can't encode with quoted-printable here")
    (if (null (mew-multi-part-number))
	(message "Can't encode with quoted-printable here")
      (let* ((part (mew-multi-part-number))
	     (nums (mew-multi-split-number part))
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (encode (mew-multi-get-encode entry))
	     (name (mew-multi-get-name entry))
	     (type (mew-multi-get-type entry))
	     (case-fold-search t))
	(if (or (string-match "/$" name)
		(equal "^Message/" type)
		(equal "^Multipart/" type))
	    (message "Can't encode with quoted-printable here")
	  (setq mew-multi-syntax
		(mew-multi-set 
		 nums
		 (mew-multi-set-param 
		  "" (mew-multi-set-encode "quoted-printable" entry))
		 mew-multi-syntax))
	  (mew-multi-display)
	  ))
      ))
  )

(defun mew-multi-gzip64 ()
  (interactive)
  (if (< (mew-multi-line) 1)
      (message "Can't apply gzip here")
    (if (null (mew-multi-part-number))
	(message "Can't apply gzip here")
      (let* ((part (mew-multi-part-number))
	     (nums (mew-multi-split-number part))
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (encode (mew-multi-get-encode entry))
	     (name (mew-multi-get-name entry))
	     (type (mew-multi-get-type entry)))
	(if (string-match "/$" name)
	    (message "Can't apply gzip to multipart.")
	  (if (or (equal (downcase type) "text/plain")
		  (equal (downcase type) "application/postscript"))
	      ;; never use compress to types other than text/plain or 
	      ;; application/postscript. Indeed, compression is not
	      ;; effective to compressed data such as jpeg.
	      (progn
		(setq mew-multi-syntax
		      (mew-multi-set 
		       nums
		       (mew-multi-set-param 
			"" (mew-multi-set-encode "x-gzip64" entry))
		       mew-multi-syntax))
		(mew-multi-display))
	    (message "Can't apply gzip to %s" type))
	  ))
      ))
  )

(defun mew-multi-pgp-sign ()
  (interactive)
  (if (< (mew-multi-line) 1)
      (message "Can't PGP sign here")
    (if (null (mew-multi-part-number))
	(message "Can't PGP sign here")
      (let* ((part (mew-multi-part-number))
	     (nums (mew-multi-split-number part))
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (encode (mew-multi-get-encode entry))
	     (name (mew-multi-get-name entry))
	     (type (mew-multi-get-type entry)))
	(setq mew-multi-syntax
	      (mew-multi-set 
	       nums
	       (mew-multi-set-param 
		"" (mew-multi-set-encode 'pgp-sign entry))
		mew-multi-syntax))
	(mew-multi-display)
	)
      )
    ))

(defun mew-multi-pgp-enc ()
  (interactive)
  (if (< (mew-multi-line) 1)
      (message "Can't PGP encrypt here")
    (if (null (mew-multi-part-number))
	(message "Can't PGP encrypt here")
      (let* ((part (mew-multi-part-number))
	     (nums (mew-multi-split-number part))
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (encode (mew-multi-get-encode entry))
	     (name (mew-multi-get-name entry))
	     (type (mew-multi-get-type entry))
	     (towhom))
	(setq towhom (mew-input-address "To:"))
	(setq mew-multi-syntax
	      (mew-multi-set nums
			     (mew-multi-set-param
			      towhom
			      (mew-multi-set-encode 'pgp-enc entry))
			     mew-multi-syntax))
	(mew-multi-display)
	))
    ))

(defun mew-multi-pgp-sign-enc ()
  (interactive)
  (if (< (mew-multi-line) 1)
      (message "Can't PGP sign then encrypt here")
    (if (null (mew-multi-part-number))
	(message "Can't PGP sign then encrypt here")
      (let* ((part (mew-multi-part-number))
	     (nums (mew-multi-split-number part))
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (encode (mew-multi-get-encode entry))
	     (name (mew-multi-get-name entry))
	     (type (mew-multi-get-type entry))
	     (towhom))
	(setq towhom (mew-input-address "To:"))
	(setq mew-multi-syntax
	      (mew-multi-set nums
			     (mew-multi-set-param
			      towhom
			      (mew-multi-set-encode
			       'pgp-sign-enc entry))
			     mew-multi-syntax))
	(mew-multi-display)
	))
    ))

(defun mew-multi-undo ()
  (interactive)
  (if (< (mew-multi-line) 1)
      (message "Can't undo encoding here")
    (if (null (mew-multi-part-number))
	(message "Can't undo encoding here")
      (let* ((part (mew-multi-part-number))
	     (nums (mew-multi-split-number part))
	     (entry (mew-multi-get nums mew-multi-syntax))
	     (type (mew-multi-get-type entry))
	     (name (mew-multi-get-name entry)))
	;;; Multipart/Mixed?
	(setq mew-multi-syntax
	      (mew-multi-set
	       nums
	       (mew-multi-set-param
		""
		(mew-multi-set-encode 
		 (mew-file-encoding
		  (mew-content-attr type mew-mime-content-type))
		 entry))
		mew-multi-syntax))
	(mew-multi-display)
	))
    ))

(defun mew-multi-audio ()
  (interactive)
  (if (< (mew-multi-line) 3)
      (message "Can't insert voice here.")
    (let* ((nums (mew-multi-nums 'previous 'plus1))
	   (subdir (mew-multi-expand-path nums mew-multi-syntax))
	   (mimedir (mew-expand-file-name (mew-draft-to-mime (buffer-name))))
	   (file (mew-input-string "Create as %s(%s): " 
				    subdir
				    "a.au"))
	   (efile nil)
	   (buf nil))
      (setq efile (if (equal subdir "") file (concat subdir file)))
      (save-excursion
	(set-buffer mew-buffer-tmp)
	(erase-buffer)
	(condition-case errorcode
	    (let ((selective-display nil)
		  (mc-flag nil)
		  (call-process-hook nil)
		  (default-process-coding-system 
		    (if mew-mule-p (cons *noconv* *noconv*))))
	      (while (not (mew-y-or-n-p "Are you ready? ")) ())
	      (message "Type C-g to finish recording...")
	      (apply 'call-process mew-prog-audio)
	      )
	  (quit (message "Type C-g to finish recording... done.")))
	(let ((mc-flag nil)
	      (file-coding-system (if mew-mule-p *noconv* nil)))
	  (write-region (point-min) (point-max)
			(expand-file-name efile mimedir)))
	)
      (kill-buffer mew-buffer-tmp)
      (setq mew-multi-syntax
	    (mew-multi-insert 
	     nums
	     (list file "Audio/Basic" "" "base64" "")
	     mew-multi-syntax))
      (mew-multi-display)
      )))

;;;
;;; Draft magic functions
;;;

(defun mew-draft-in-header-p ()
  (if (markerp mew-draft-buffer-header)
      (<= (point) (marker-position mew-draft-buffer-header))
    nil)
  )

(defun mew-draft-in-multipart-p ()
  (if (markerp mew-draft-buffer-multipart)
      (>= (point) (marker-position mew-draft-buffer-multipart))
    nil)
  )

(defun mew-command-key-sequence (key command)
  "Remove initial sequence of universal arguments from KEY and return
the real key sequence that invoked COMMAND.  If COMMAND is invoked by
`execute-extended-command', nil is returned."
  (while (not (or (zerop (length key))
                  (eq command (key-binding key))))
    (setq key (substring key 1)))
  (and (not (zerop (length key))) key))

(defun mew-draft-keyswitch ()
  (interactive)
  (let ((func) (key (this-command-keys)))
    (cond
     ((and (markerp mew-draft-buffer-multipart) (mew-draft-in-multipart-p))
      (setq func (lookup-key mew-draft-multipart-map key)))
     ((and (markerp mew-draft-buffer-header) (mew-draft-in-header-p))
      (setq func (lookup-key mew-draft-header-map key)))
     (t 
      (setq func (lookup-key mew-draft-body-map key)) ;; \e\t -> 1
      (if (not (symbolp func)) (setq func nil)))
     )
    (if (null func)
	(setq func (lookup-key (current-global-map)
			       (mew-command-key-sequence key 'mew-draft-keyswitch))))
    (if func
	(call-interactively func)
      (insert key)) ;; for safety
    ))

(defun mew-draft-null-function ()
  (interactive)
  ())

(defun mew-draft-header-comp ()
  (interactive)
  (let ((func nil))
    (if (mew-draft-on-key-p)
	(mew-draft-key-comp)
      (if (setq func (mew-draft-on-value-p))
	  (funcall (cdr func))
	(tab-to-tab-stop))) ;; default keybinding
    ))

(defun mew-draft-on-key-p ()
  (if (bolp)
      (if (bobp) 
	  t
	(save-excursion
	  (forward-line -1)
	  (if (looking-at ".*,[ \t]?$") nil t)
	  )
	)
    (let ((pos (point)))
      (save-excursion
	(beginning-of-line)
	(if (looking-at "[ \t]") 
	    nil
	  (if (re-search-forward ":" pos t) nil t))
	))
    ))
      
(defun mew-draft-on-value-p ()
  (save-excursion
    (beginning-of-line)
    (while (and (< (point-min) (point))	(looking-at "[ \t]"))
      (forward-line -1)
      )
    (if (looking-at "\\([^:]*:\\)")
	(mew-assoc-match (mew-match 1) mew-field-completion-switch 0)
      nil) ;; what a case reach here?
    ))
      
(defun mew-draft-address-comp ()
  (let ((word nil) (comp nil) (all nil) (len nil))
    (if (null (setq word (mew-delete-backward-char)))
	(tab-to-tab-stop)
      (setq comp (try-completion word mew-alias-alist))
      (setq all (all-completions word mew-alias-alist))
      (setq len (length word))
      (cond
       ((eq comp t)
	(insert (cdr (assoc word mew-alias-alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions))
	(if mew-draft-window-config
	    (progn
	      (set-window-configuration mew-draft-window-config)
	      (setq mew-draft-window-config nil)
	      ))
	)
       ((and (equal (aref word (1- len)) ?@)
	     (assoc (substring word 0 (1- len)) mew-alias-alist))
	(insert (cdr (assoc (substring word 0 (1- len)) mew-alias-alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions))
	(if mew-draft-window-config
	    (progn
	      (set-window-configuration mew-draft-window-config)
	      (setq mew-draft-window-config nil)
	      ))
	)
       ((stringp comp)
	(insert comp)
	(if (> (length all) 1)
	    (progn
	      (or mew-draft-window-config
		  (setq mew-draft-window-config
			(current-window-configuration)))
	      (with-output-to-temp-buffer mew-buffer-completions
		(display-completion-list all))
	      )
	  )
	)
       (t (insert word)
	  (message "No matching alias"))
       )
      )))

(defun mew-draft-folder-comp ()
  (let ((word nil) (comp nil) (all nil) (len nil))
    (if (null (setq word (mew-delete-backward-char)))
	(insert "+")
      (setq comp (try-completion word mew-folder-alist))
      (setq all (all-completions word mew-folder-alist))
      (setq len (length word))
      (cond
       ((eq comp t)
	(insert (cdr (assoc word mew-folder-alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions))
	(if mew-draft-window-config
	    (progn
	      (set-window-configuration mew-draft-window-config)
	      (setq mew-draft-window-config nil)
	      ))
	)
       ((stringp comp)
	(insert comp)
	(if (> (length all) 1)
	    (progn
	      (or mew-draft-window-config
		  (setq mew-draft-window-config
			(current-window-configuration)))
	      (with-output-to-temp-buffer mew-buffer-completions
		(display-completion-list all))
	      )
	  )
	)
       (t (insert word)
	  (message "No matching folder"))
       )
      )))

(defun mew-delete-backward-char ()
  "Delete appropriate preceeding word and return it."
  (let ((case-fold-search t)
	(start nil)
	(end (point))
	;; mew-address-separator == ":, \t\n"
	(regex (concat "[^" mew-address-separator "]")))
    (save-excursion
      ;; find the position of separator.
      (while (and (not (bobp))
		  (string-match regex (mew-buffer-substring 
				       (1- (point)) (point))))
	(forward-char -1))
      (setq start (point))
      (if (= start end)
	  nil ;; cursor is just after separator
	(prog1
	    (mew-buffer-substring start end) ;; return the word
	  (delete-region start end)))
      )
    ))

(defun mew-draft-key-comp ()
  (let ((word nil) (comp nil) (all nil) (len nil)
	(alist nil))
    (if (null (setq word (mew-delete-key)))
	(progn
	  (or mew-draft-window-config
	      (setq mew-draft-window-config
		    (current-window-configuration)))
	  (with-output-to-temp-buffer mew-buffer-completions
	    (display-completion-list mew-fields))
	  )
      (setq alist (mew-fiels-make-alist mew-fields)) ;; to make dynamically
      (setq comp (try-completion word alist))
      (setq all (all-completions word alist))
      (setq len (length word))
      (cond
       ((eq comp t)
	(insert (cdr (assoc word alist)))
	(insert " ")
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions))
	(if mew-draft-window-config
	    (progn
	      (set-window-configuration mew-draft-window-config)
	      (setq mew-draft-window-config nil)
	      ))
	)
       ((stringp comp)
	(if (equal (length all) 1)
	    (progn
	      (insert (cdr (assoc comp alist)))
	      (insert " ")
	      (and (get-buffer mew-buffer-completions)
		   (kill-buffer mew-buffer-completions))
	      (if mew-draft-window-config
		  (progn
		    (set-window-configuration mew-draft-window-config)
		    (setq mew-draft-window-config nil)
		    )))
	  (insert (capitalize comp))
	  (or mew-draft-window-config
	      (setq mew-draft-window-config
		    (current-window-configuration)))
	  (with-output-to-temp-buffer mew-buffer-completions
	    (display-completion-list all))
	  ))
       (t (insert word)
	  (message "No matching key"))
       ))))

(defun mew-fiels-make-alist (list)
  (mapcar (function 
	   (lambda (x) 
	     (if (string-match "^\\(.*\\):$" x)
		 (cons (capitalize (mew-match 1 x)) (capitalize x)))))
	  list)
  )

(defun mew-delete-key ()
  (let ((pos (point)))
    (beginning-of-line)
    (prog1
	(capitalize (mew-buffer-substring (point) pos))
      (delete-region (point) pos)
      )
    ))

(defun mew-draft-header-domain ()
  (interactive)
  (let ((field (mew-draft-on-value-p)) (word nil))
    (if (or (null field) (not (equal (cdr field) 'mew-draft-address-comp)))
	(lisp-complete-symbol) ;; default key binding
      (setq word (mew-delete-backward-char-at))
      ;; word is strings between @ and cursor
      (cond
       ((equal word nil) ;; @ doesn't exist.
	(if (null mew-mail-domain-list)
	    ()
	  (insert "@")
	  (insert (car mew-mail-domain-list))
	  (and (get-buffer mew-buffer-completions)
	       (kill-buffer mew-buffer-completions))
	  (if mew-draft-window-config
	      (progn
		(set-window-configuration mew-draft-window-config)
		(setq mew-draft-window-config nil)
		))))
       ((equal word t) ;; just after @
	(if mew-mail-domain-list
	    (progn
	      (insert (car mew-mail-domain-list))
	      (and (get-buffer mew-buffer-completions)
		   (kill-buffer mew-buffer-completions))
	      (if mew-draft-window-config
		  (progn
		    (set-window-configuration mew-draft-window-config)
		    (setq mew-draft-window-config nil)
		    )))))
       (t
	(let ((comp nil) (all nil) (len nil)
	      (dalist (mew-slide-pair mew-mail-domain-list)))
	  (setq comp (try-completion word dalist))
	  (setq all (all-completions word dalist))
	  (setq len (length word))
	  (cond
	   ;; already completed
	   ((eq comp t) 
	    (insert (cdr (assoc word dalist))) ;; next candidate
	    (and (get-buffer mew-buffer-completions)
		 (kill-buffer mew-buffer-completions))
	    (if mew-draft-window-config
		(progn
		  (set-window-configuration mew-draft-window-config)
		  (setq mew-draft-window-config nil)
		  )))
	   ;; just one candidate
	   ((equal 1 (length all))
	    (insert (car all))
	    (and (get-buffer mew-buffer-completions)
		 (kill-buffer mew-buffer-completions))
	    (if mew-draft-window-config
		(progn
		  (set-window-configuration mew-draft-window-config)
		  (setq mew-draft-window-config nil)
		  ))
	    (message "Sole completion"))
	   ;; one or more candidate
	   ((stringp comp)
	    (insert comp)
	    (if (> (length all) 1)
		(progn
		  (or mew-draft-window-config
		      (setq mew-draft-window-config
			    (current-window-configuration)))
		  (with-output-to-temp-buffer mew-buffer-completions
		    (display-completion-list all))
		  )))
	   ;; no candidate
	   (t (insert word)
	      (message "No matching alias"))
	   )
	  ))
       ))
    ))

(defun mew-delete-backward-char-at ()
  (interactive)
  (let ((case-fold-search t)
	(start nil)
	(end (point))
	(regex (concat "[^" mew-address-separator "]")))
    (save-excursion
      (while (and (not (bobp))
		  (string-match regex (mew-buffer-substring (1- (point)) (point))))
	(forward-char -1))
      (if (null (re-search-forward "@" end t))
	  nil ;; @ doesn't exist.
	(setq start (point))
	(if (= start end)
	    t ;; just after @
	  (prog1
	      (mew-buffer-substring start end)
	    (delete-region start end)))
	)
      )
    ))

(defun mew-slide-pair (x)
  (let ((ret nil)
	(len (length x))
	(first (car x)))
    (cond 
     ((eq x 0) nil)
     ((eq x 1) (cons first first))
     (t
      (while (cdr x)
	(setq ret (cons (cons (nth 0 x) (nth 1 x)) ret))
	(setq x (cdr x))
	)
      (setq ret (cons (cons (car x) first) ret))
      (nreverse ret)
      )
     )
    )
  )

;;;
;;; mini buffer
;;;

(defun mew-temp-minibuffer-message (m)
  "A Lisp version of `temp_minibuffer_message' from minibuf.c."
  (if (fboundp 'temp-minibuffer-message)
      (temp-minibuffer-message m)
    (let ((savemax (point-max)))
      (save-excursion
	(goto-char (point-max))
	(insert (format " (%s)" m)))
      (let ((inhibit-quit t))
	(sit-for 2)
	(delete-region savemax (point-max))
	(if quit-flag
	    (setq quit-flag nil
		  unread-command-char 7))
	))
    ))

(defun mew-minibuffer-alias ()
  (interactive)
  (let ((word nil) (comp nil) (all nil) (len nil))
    (if (null (setq word (mew-delete-backward-char)))
	(tab-to-tab-stop) ;; default key binding
      (setq comp (try-completion word mew-alias-alist))
      (setq all (all-completions word mew-alias-alist))
      (setq len (length word))
      (cond
       ((eq comp t)
	(insert (cdr (assoc word mew-alias-alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions)
	     (mew-window-pop)))
       ((and (equal (aref word (1- len)) ?@)
	     (assoc (substring word 0 (1- len)) mew-alias-alist))
	(insert (cdr (assoc (substring word 0 (1- len)) mew-alias-alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions)
	     (mew-window-pop)))
       ((stringp comp)
	(insert comp)
	(and (> (length all) 1)
	     (mew-window-push)
	     (with-output-to-temp-buffer mew-buffer-completions
	       (display-completion-list all))))
       (t (insert word)
	  (mew-temp-minibuffer-message "No matching alias"))
       )
      )
    ))

(defun mew-minibuffer-domain ()
  (interactive)
  (let ((word (mew-delete-backward-char-at)))
;;;;    (if (null (setq word (mew-delete-backward-char-at)))
;;;;	(lisp-complete-symbol) ;; default key binding)
    ;; word is strings between @ and cursor
    (cond
     ((equal word nil) ;; @ doesn't exist.
      (if (null mew-mail-domain-list)
	  ()
	(insert "@")
	(with-output-to-temp-buffer mew-buffer-completions
	  (display-completion-list mew-mail-domain-list))))
     ((equal word t) ;; just after @
      (if mew-mail-domain-list
	  (with-output-to-temp-buffer mew-buffer-completions
	    (display-completion-list mew-mail-domain-list)))
      )
     (t
      (let ((comp nil) (all nil) (len nil)
	    (dalist (mew-slide-pair mew-mail-domain-list)))
	(setq comp (try-completion word dalist))
	(setq all (all-completions word dalist))
	(setq len (length word))
	(cond
	 ;; already completed
	 ((eq comp t) 
	  (insert (cdr (assoc word dalist))) ;; next candidate
	  (and (get-buffer mew-buffer-completions)
	       (kill-buffer mew-buffer-completions))
	  )
	 ;; just one candidate
	 ((equal 1 (length all))
	  (insert (car all))
	  (and (get-buffer mew-buffer-completions)
	       (kill-buffer mew-buffer-completions))
	  (mew-temp-minibuffer-message "Sole completion")
	  )
	 ;; one or more candidate
	 ((stringp comp)
	  (insert comp)
	  (and (> (length all) 1)
	       (with-output-to-temp-buffer mew-buffer-completions
		 (display-completion-list all))))
	 ;; no candidate
	 (t (insert word)
	    (mew-temp-minibuffer-message "No matching domain"))
	 ))
      )
     )
    ))

;;
;; folder completion
;;

(defun mew-minibuffer-folders ()
  (interactive)
  (let ((word nil) (comp nil) (all nil) (len nil))
    (if (null (setq word (mew-delete-backward-char)))
	(tab-to-tab-stop) ;; default key binding
      (setq comp (try-completion word mew-folder-alist))
      (setq all (all-completions word mew-folder-alist))
      (setq len (length word))
      (cond
       ((eq comp t) 
	;; insert completed word
	(insert (cdr (assoc word mew-folder-alist)))
	(and (get-buffer mew-buffer-completions)
	     (kill-buffer mew-buffer-completions)))
       ((stringp comp)
	(insert comp) ;; insert substring
	(and (> (length all) 1)
	     (with-output-to-temp-buffer mew-buffer-completions
	       (display-completion-list all))))
       (t (insert word)
	  (mew-temp-minibuffer-message "No matching alias"))
       )
      )
    ))

;;;
;;; MIME encoding and decoding
;;;

(defconst mew-base64-mask 63)
(defconst mew-base64-boundary1 26)
(defconst mew-base64-boundary2 52)
(defconst mew-base64-boundary3 62)
(defconst mew-base64-boundary4 63)

(defvar mew-encode-switch
  (list
   (cons "B" 'mew-base64-encode-string)
   (cons "Q" 'mew-quoted-printable-encode-string)
   )
  )

(defvar mew-decode-switch
  (list
   (cons "B" 'mew-base64-decode-string)
   (cons "Q" 'mew-quoted-printable-decode-string)
   )
  )

(defmacro mew-decode-func-attr (encode alist)
  (` (mew-assoc (, encode) (, alist) 0 t)))

(defmacro mew-decode-func-func (attr)
  (` (cdr (, attr))))

(defvar mew-decode-header
   '(("To:" . nil)
     ("Cc:" . nil)
     ("From:" . t)
     ("Subject:" . t)
     ("Content-Description:" . t))
   )

(defmacro mew-decode-attr (key alist)
  (` (mew-assoc (, key) (, alist) 0 t)))

(defmacro mew-decode-delp (attr)
  (` (cdr (, attr))))

;;;
;;; kick off routine
;;;

(defun mew-header-mime-decode ()
  (interactive)
  (let ((case-fold-search t)
	(key   "")
	(start nil)
	(value nil)
	(attr  nil)
	end)
    (save-excursion
      (re-search-forward "^-*$")
      (setq end (point)))
    (if (> (count-lines (point) end) mew-header-max-length)
	() ;; skip 
      (save-excursion
	(catch 'header
	  (while (re-search-forward "^-*$\\|^[^ \t\n]*:")
	    (setq key (mew-match 0))
	    (setq start (match-end 0))
	    (if (string-match "^-*$" key) (throw 'header nil))
	    (forward-line 1)
	    (while (looking-at "[ \t]")
	      (forward-line 1))
	    (if (setq attr (mew-decode-attr key mew-decode-header))
		(progn
		  (setq value (mew-buffer-substring start (1- (point))))
		  (delete-region start (point))
		  (insert (mew-header-decode value (mew-decode-delp attr)))
		  (insert "\n")
		  ))
	    )))
      )))

(defun mew-header-decode (str &optional del-ret)
  (if del-ret
      (while (string-match "\n[ \t]*" str)
	(setq str (concat (substring str 0 (match-beginning 0))
			  (substring str (match-end 0) (length str)))
	      )
	)
    )
  (while (string-match "=\\?\\([^?]+\\)\\?\\(.\\)\\?\\([^?]+\\)\\?= ?" str)
    ;; last " ?" is for mhn. (I don't like it.)
    (let ((charset (mew-match 1 str))
	  (encode (mew-match 2 str))
	  (enstr (mew-match 3 str))
	  (head (substring str 0 (match-beginning 0)))
	  (tail (substring str (match-end 0) (length str)))
	  (destr ""))
      (setq destr 
	    (funcall 
	     (mew-decode-func-func
	      (mew-decode-func-attr encode mew-decode-switch))
	     enstr))
      (if mew-mule-p
	  (setq destr
		(code-convert-string
		 destr
		 (mew-mule-content-coding
		  (mew-mule-content-attr charset mew-mule-character-set))
		 *internal*
		 ))
	)
      (setq str (concat head destr tail))
      ))
  str
  )

;;
;; Base 64 encode
;;

(defun mew-base64-encode-string (str256)
  (let* ((rest (% (length str256) 3))
	 (count 0)
	 zpad epad len end
	 (ret ""))
    (cond 
     ((= rest 0) (setq zpad 0 epad 0 end nil))
     ((= rest 1) (setq zpad 2 epad 2 end -2))
     ((= rest 2) (setq zpad 1 epad 1 end -1))
     )
    (setq str256 (concat str256 (make-string zpad 0)))
    (setq len (length str256))
    (while (< count len)
      (let ((char0 (aref str256 count))
	    (char1 (aref str256 (1+ count)))
	    (char2 (aref str256 (+ 2 count))))
	(setq ret
	      (concat
	       ret
	       (char-to-string (mew-base64-char64 (lsh char0 -2)))
	       (char-to-string
		(mew-base64-char64 (logior (lsh (logand char0 3) 4)
					   (lsh char1 -4))))
	       (char-to-string
		(mew-base64-char64 (logior (lsh (logand char1 15) 2)
					   (lsh char2 -6))))
	       (char-to-string (mew-base64-char64 (logand char2 63))) ;; xxx
	       )
	      )
	(setq count (+ count 3))
	))
    (concat (substring ret 0 end) (make-string epad ?=))
    ))
	      
(defun mew-base64-char256 (ch64)
  (cond
   ((null ch64) 0)
   ((and (<= ?A ch64) (<= ch64 ?Z)) (- ch64 ?A))
   ((and (<= ?a ch64) (<= ch64 ?z)) (+ (- ch64 ?a) mew-base64-boundary1))
   ((and (<= ?0 ch64) (<= ch64 ?9)) (+ (- ch64 ?0) mew-base64-boundary2))
   ((= ch64 ?+) mew-base64-boundary3)
   ((= ch64 ?/) mew-base64-boundary4)
   ((= ch64 ?=) 0)
   )
  )

;;
;; Base 64 decode
;;

(defun mew-base64-decode-string (str64)
  (let* ((len (length str64))
	 (count 0)
	 (ret ""))
    (while (< count len)
      (let ((char0 (mew-base64-char256 (aref str64 count)))
            (char1 (mew-base64-char256 (aref str64 (1+ count))))
            (char2 (mew-base64-char256 (aref str64 (+ 2 count))))
            (char3 (mew-base64-char256 (aref str64 (+ 3 count)))))
	(setq ret
	      (concat
	       ret
	       (char-to-string
		(logand (logior (lsh char0 2) (lsh char1 -4)) 255))
	       (char-to-string 
		(logand (logior (lsh char1 4) (lsh char2 -2)) 255))
	       (char-to-string
		(logand (logior (lsh char2 6) char3) 255))
	       )
	      )
	(setq count (+ count 4))
	))
    (if (string-match "=+$" str64)
	(substring ret 0 (- (match-beginning 0) (match-end 0)))
      ret
      )
    ))

(defun mew-base64-char64 (ch256)
  (aref "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/" ch256))

;;
;; Quoted-printable decode
;;

(defun mew-hexstring-to-int (hex)
  (let ((len (length hex))
	(count 0)
	(ch nil)
	(ret 0))
    (while (< count len)
      (setq ch (aref hex count))
      (cond
       ((and (<= ?0 ch) (<= ch ?9))
	(setq ret (+ (* ret 16) (- ch ?0))))
       ((and (<= ?A ch) (<= ch ?F))
	(setq ret (+ (* ret 16) (+ (- ch ?A) 10))))
       ((and (<= ?a ch) (<= ch ?f))
	(setq ret (+ (* ret 16) (+ (- ch ?a) 10))))
       )
      (setq count (1+ count))
      )
    ret
    ))

(defun mew-quoted-printable-decode-string (qpstr)
  (let ((start) (end))
    (while (string-match "_" qpstr)
      (aset qpstr (match-beginning 0) 32)) ;; 32 = space
    (while (string-match "=[0-9A-Z][0-9A-Z]" qpstr)
      (setq start (match-beginning 0))
      (setq end (match-end 0))
      (setq qpstr
	    (concat (substring qpstr 0 start)
		    (char-to-string (mew-hexstring-to-int 
				     (substring qpstr (1+ start) end)))
		    (substring qpstr end nil))))
    qpstr
    ))

(defun mew-char-to-qhex (char)
  (concat "="
	  (char-to-string (aref "0123456789ABCDEF" (lsh char -4)))
	  (char-to-string (aref "0123456789ABCDEF" (logand char 15)))))

(defun mew-quoted-printable-encode-string (str)
  (let ((len (length str))
	(count 0)
	(ch nil)
	(ret))
    (while (< count len)
      (setq ch (aref str count))
      (cond
       ((and (> ch 32)
	     (< ch 126)
	     (not (equal ch ?=))
	     (not (equal ch ??))
	     (not (equal ch ?_)) ;; space
	     )
	(setq ret (concat ret (char-to-string ch)))
	)
       ((equal ch 32)
	(setq ret (concat ret "_"))
	)
       (t 
	(setq ret (concat ret (mew-char-to-qhex ch)))
	)
       )
      (setq count (1+ count))
      )
    ret
    ))

(defun mew-header-encode-string (str encode)
  (let ((fun (mew-decode-func-func
	      (mew-decode-func-attr encode mew-encode-switch)))
	lc)
    (cond
     (mew-mule-p
      (if (setq lc (car (find-charset-string str)))
	  (let* ((attr (mew-mule-lc-attr lc mew-mule-character-set))
		 (charset (mew-mule-lc-content attr))
		 (symbol (mew-mule-lc-symbol attr))
		 (txstr (code-convert-string str *internal* symbol)))
	    (concat "=?" charset "?" encode "?" (funcall fun txstr) "?=")
	    )
	str
	)
      )
     (mew-emacs19-p
      (if (string-match "[\200-\377]" str)
	  (concat "=?iso-8859-1?" encode "?" (funcall fun str) "?=")
	str)
      )
     (t str)
     )
    ))

;;;
;;; Demo
;;;

(defun mew-demo (list)
  ;; 9 is "/\\ - \\/"
  (let* ((wl (window-width))
	 (pl (/ (- wl 9) 2))
	 (pre (make-string pl 32))
	 (suf (make-string (1- (- (- wl pl) 9)) 32)))
    (save-window-excursion
      (select-window (get-buffer-window (get-buffer mew-buffer-hello)))
      (while list
	(mew-demo-print (car list) pre suf)
	(mew-demo-loop)
	(setq list (cdr list)))
      )))

(setq mew-demo-string
'(
 "/\\\\ - \\\\/"

 "-\\\\ - \\\\/" "\\\\\\ - \\\\/" "|\\\\ - \\\\/" "/\\\\ - \\\\/"
 "-\\\\ - \\\\/" "\\\\\\ - \\\\/" "|\\\\ - \\\\/" "/\\\\ - \\\\/"

 "/|\\ - \\\\/"  "//\\ - \\\\/" "/-\\ - \\\\/" "/\\\\ - \\\\/"
 "/|\\ - \\\\/"  "//\\ - \\\\/"  "/-\\ - \\\\/" "/\\\\ - \\\\/"

 "/\\| - \\\\/" "/\\/ - \\\\/" "/\\- - \\\\/" "/\\\\ - \\\\/" 
 "/\\| - \\\\/" "/\\/ - \\\\/" "/\\- - \\\\/" "/\\\\ - \\\\/"

 "/\\\\ - |\\/" "/\\\\ - /\\/" "/\\\\ - -\\/" "/\\\\ - \\\\/"
 "/\\\\ - |\\/" "/\\\\ - /\\/" "/\\\\ - -\\/" "/\\\\ - \\\\/"

 "/\\\\ - \\|/" "/\\\\ - \\//" "/\\\\ - \\-/" "/\\\\ - \\\\/"
 "/\\\\ - \\|/" "/\\\\ - \\//" "/\\\\ - \\-/" "/\\\\ - \\\\/"

 "/\\\\ - \\\\-" "/\\\\ - \\\\\\" "/\\\\ - \\\\|" "/\\\\ - \\\\/"
 "/\\\\ - \\\\-" "/\\\\ - \\\\\\" "/\\\\ - \\\\|" "/\\\\ - \\\\/"
 )
)

(defun mew-delete-line ()
  (beginning-of-line)
  (let ((start (point)))
    (forward-line)
    (delete-region start (point))
    ))

(defun mew-demo-print (string prefix suffix)
  (goto-char (point-max))
  (mew-delete-line)
  (insert (concat prefix string suffix))
  )

(cond
 (mew-emacs18-p
  (defun mew-demo-loop ()
    (let ((i 30))
      (while (> i 1)
	(sit-for 0)
	(message "")
	(setq i (1- i))))))
 (t
  (defun mew-demo-loop ()
    (sit-for (string-to-int "0.02"))))
 )

;;;
;;; End of Mew
;;;
